<?php
/**
 * This file is part of OpenMediaVault.
 *
 * @license   http://www.gnu.org/licenses/gpl.html GPL Version 3
 * @author    Volker Theile <volker.theile@openmediavault.org>
 * @copyright Copyright (c) 2009-2016 Volker Theile
 *
 * OpenMediaVault is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * OpenMediaVault is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenMediaVault. If not, see <http://www.gnu.org/licenses/>.
 */
class OMVRpcServiceHealth extends \OMV\Rpc\ServiceAbstract {
	/**
	 * Get the RPC service name.
	 */
	public function getName() {
		return "Health";
	}

	/**
	 * Initialize the RPC service.
	 */
	public function initialize() {
		$this->registerMethod("noop");
		$this->registerMethod("getHealthInfo");
	}

	/**
	 * This function can be used to check if the communication between
	 * WebGUI and server exists.
	 * @param params The method parameters.
	 * @param context The context of the caller.
	 * @return A quotation from the Dune novel, which is a string.
	 */
	public function noop($params, $context) {
		// Validate the RPC caller context.
		$this->validateMethodContext($context, [
			"role" => OMV_ROLE_ADMINISTRATOR
		]);
		return array_rand_value(\OMV\Environment::get("OMV_DUNE_QUOTES"));
	}

	/**
	 * Get the list of running processes and a system summary information
	 * as well.
	 * @param params The method parameters.
	 * @param context The context of the caller.
	 * @return The 'top' console command output.
	 */
	public function getTopHealthInfo($params, $context) {
		// Validate the RPC caller context.
		$this->validateMethodContext($context, [
			"role" => OMV_ROLE_ADMINISTRATOR
		]);
		// Get task overview.
		$cmd = new \OMV\System\Process("top", "-b", "-n 1");
		$cmd->setRedirect2to1();
		$cmd->execute($output);
		return implode("\n", $output);
	}


	/**
	 * Get various system information.
	 * @param params The method parameters.
	 * @param context The context of the caller.
	 * @return An array of system information.
	 */
	public function getHealthInfo($params, $context) {
		// Validate the RPC caller context.
		$this->validateMethodContext($context, [
			"role" => OMV_ROLE_ADMINISTRATOR
		]);
		// Get various system informations.
		$prd = new \OMV\ProductInfo();
		$uname = posix_uname();
		$memStats = \OMV\System\System::getMemoryStats();
		$memStats['wobufferscache']['percentage'] = round(
		  ($memStats['wobufferscache']['used'] * 100) /
		  $memStats['mem']['total']);
		$cpuStats = \OMV\System\System::getCpuStats();
		$index = 0;
		return [
			[
				"name" => gettext("CPU temperature"),
				"value" => \OMV\System\Health\Health::getCpuTemp(),
				"type" => "string",
				"index" => $index++
			],[
				"name" => gettext("Fan speed"),
				"value" => \OMV\System\Health\Health::getFanSpeed(),
				"type" => "string",
				"index" => $index++
			]
		];
	}
}
