#!/bin/sh
# Based on raspi-config
# Part of raspi-config https://github.com/RPi-Distro/raspi-config
#
# See LICENSE file for copyright and license details

INTERACTIVE=True
ASK_TO_REBOOT=0
GLOBAL_ROOT_IS_DEV=True
GLOBAL_ROOT_FS=/dev/null

get_init_sys() {
  if command -v systemctl > /dev/null && systemctl | grep -q '\-\.mount'; then
    SYSTEMD=1
  elif [ -f /etc/init.d/cron ] && [ ! -h /etc/init.d/cron ]; then
    SYSTEMD=0
  else
    echo "Unrecognised init system"
    return 1
  fi
}

do_get_root_part() {
for x in $(cat /proc/cmdline); do
    case "$x" in
        root=*)
            GLOBAL_ROOT_IS_DEV=False
            GLOBAL_ROOT_FS="${x#root=}"
            rootdev="${x#root=}"
            if [ "${rootdev:0:4}" = "/dev" ] ; then
                GLOBAL_ROOT_IS_DEV=True
                echo ${rootdev:5}
            elif [ "${rootdev:0:4}" = "UUID" ] ; then
                findfs ${rootdev} | tail -1 | sed 's|/dev/||g'
            elif [ "${rootdev:0:4}" = "LABE" ] ; then
                findfs ${rootdev} | tail -1 | sed 's|/dev/||g'
            elif [ "${rootdev:0:4}" = "PART" ] ; then
                if [ "${rootdev:4:4}" = "UUID" ] ; then
                    findfs ${rootdev} | tail -1 | sed 's|/dev/||g'
                elif [ "${rootdev:4:4}" = "LABE" ] ; then
                    findfs ${rootdev} | tail -1 | sed 's|/dev/||g'
                fi
            fi
    esac
done
}

get_can_expand() {
  get_init_sys
  do_get_root_part
  if [ $SYSTEMD -eq 1 ]; then
    ROOT_PART=$(mount | sed -n 's|^/dev/\(.*\) on / .*|\1|p')
  else
    ROOT_PART=$(do_get_root_part)

    if [ "x${ROOT_PART}" = "x" ]; then
      echo 1
      exit
    fi
    ROOT_PART=$(readlink /dev/root)
  fi

  ROOT_DISK=`echo ${ROOT_PART} | sed s/'[0-9]*'//g`
  PART_NUM=`echo ${ROOT_PART} | sed s/'mmcblk.p'/''/g`
  if [ "$PART_NUM" = "$ROOT_PART" ]; then
    PART_NUM=`echo ${ROOT_PART} | sed s/'sd[a-z]'/''/g`
  else
    ROOT_DISK=${ROOT_PART:0:7}
  fi

  if [ "$PART_NUM" = "$ROOT_PART" ]; then
    echo 1
    exit
  fi

  if [ "$PART_NUM" -ne 2 ]; then
    echo 1
    exit
  fi

  LAST_PART_NUM=$(parted /dev/${ROOT_DISK} -ms unit s p | tail -n 1 | cut -f 1 -d:)
  if [ $LAST_PART_NUM -ne $PART_NUM ]; then
    echo 1
    exit
  fi
  echo 0
}

do_expand_rootfs() {
  get_init_sys
  do_get_root_part
  if [ $SYSTEMD -eq 1 ]; then
    ROOT_PART=$(mount | sed -n 's|^/dev/\(.*\) on / .*|\1|p')
  else
    ROOT_PART=$(do_get_root_part)

    if [ "x${ROOT_PART}" = "x" ]; then

      whiptail --msgbox "/dev/root does not exist or is not a symlink. Don't know how to expand" 20 60 2
      return 0
    fi
    ROOT_PART=$(readlink /dev/root)
  fi

  ROOT_DISK=`echo ${ROOT_PART} | sed s/'[0-9]*'//g`
  PART_NUM=`echo ${ROOT_PART} | sed s/'mmcblk.p'/''/g`
  if [ "$PART_NUM" = "$ROOT_PART" ]; then
    PART_NUM=`echo ${ROOT_PART} | sed s/'sd[a-z]'/''/g`
  else
    ROOT_DISK=${ROOT_PART:0:7}
  fi

  if [ "$PART_NUM" = "$ROOT_PART" ]; then
    whiptail --msgbox "$ROOT_PART is not an SD card. Don't know how to expand" 20 60 2
    return 0
  fi

  # NOTE: the NOOBS partition layout confuses parted. For now, let's only 
  # agree to work with a sufficiently simple partition layout
  if [ "$PART_NUM" -ne 2 ]; then
    whiptail --msgbox "Your partition layout is not currently supported by this tool. You are probably using NOOBS, in which case your root filesystem is already expanded anyway." 20 60 2
    return 0
  fi

  LAST_PART_NUM=$(parted /dev/${ROOT_DISK} -ms unit s p | tail -n 1 | cut -f 1 -d:)
  if [ $LAST_PART_NUM -ne $PART_NUM ]; then
    whiptail --msgbox "$ROOT_PART is not the last partition. Don't know how to expand" 20 60 2
    return 0
  fi

  # Get the starting offset of the root partition
  PART_START=$(parted /dev/${ROOT_DISK} -ms unit s print | grep "^${PART_NUM}" | cut -f 2 -d: | sed s/'s'/''/g)
  [ "$PART_START" ] || return 1
  # Return value will likely be error for fdisk/gdisk as it fails to reload the
  # partition table because the root fs is mounted

  # convert to or re-align GPT
  gdisk /dev/${ROOT_DISK} <<EOF
p
w
y
y
EOF

echo ""

  PARTUUID=`lsblk /dev/$ROOT_PART -o PARTUUID | tail -1`
  PARTLABEL=`lsblk /dev/$ROOT_PART -o PARTLABEL | tail -1`

  gdisk /dev/${ROOT_DISK} <<EOF
p
d
$PART_NUM
n
$PART_NUM
$PART_START


p
w
y
EOF

echo ""

  # restore uuid and label
  gdisk /dev/${ROOT_DISK} <<EOF
c
$PART_NUM
$PARTLABEL
x
c
$PART_NUM
$PARTUUID
p
w
y
EOF
  ASK_TO_REBOOT=1

  # now set up an init.d script
cat <<EOFA > /etc/init.d/resize2fs_once &&
#!/bin/sh
### BEGIN INIT INFO
# Provides:          resize2fs_once
# Required-Start:
# Required-Stop:
# Default-Start: 3
# Default-Stop:
# Short-Description: Resize the root filesystem to fill partition
# Description:
### END INIT INFO

. /lib/lsb/init-functions

case "\$1" in
  start)
    log_daemon_msg "Starting resize2fs_once" &&
EOFA

if [ "$GLOBAL_ROOT_IS_DEV" = True ]; then
  cat <<EOFB >> /etc/init.d/resize2fs_once
    resize2fs /dev/$ROOT_PART &&
EOFB
else
  cat <<EOFC >> /etc/init.d/resize2fs_once
    ROOT_FS_DEV=\`findfs ${GLOBAL_ROOT_FS} | tail -1\`
    resize2fs \$ROOT_FS_DEV &&
EOFC
fi

cat <<EOFZ >> /etc/init.d/resize2fs_once &&
    update-rc.d resize2fs_once remove &&
    rm /etc/init.d/resize2fs_once &&
    log_end_msg \$?
    ;;
  *)
    echo "Usage: \$0 start" >&2
    exit 3
    ;;
esac
EOFZ
  sed -i 's|\${ROOT_PART}|'${ROOT_PART}'|g' /etc/init.d/resize2fs_once
  chmod +x /etc/init.d/resize2fs_once &&
  update-rc.d resize2fs_once defaults &&
  if [ "$INTERACTIVE" = True ]; then
    whiptail --msgbox "Root partition has been resized.\nThe filesystem will be enlarged upon the next reboot" 20 60 2
  fi
}

get_hostname() {
    cat /etc/hostname | tr -d " \t\n\r"
}

do_hostname() {
  if [ "$INTERACTIVE" = True ]; then
    whiptail --msgbox "\
Please note: RFCs mandate that a hostname's labels \
may contain only the ASCII letters 'a' through 'z' (case-insensitive), 
the digits '0' through '9', and the hyphen.
Hostname labels cannot begin or end with a hyphen. 
No other symbols, punctuation characters, or blank spaces are permitted.\
" 20 70 1
  fi
  CURRENT_HOSTNAME=`cat /etc/hostname | tr -d " \t\n\r"`
  if [ "$INTERACTIVE" = True ]; then
    NEW_HOSTNAME=$(whiptail --inputbox "Please enter a hostname" 20 60 "$CURRENT_HOSTNAME" 3>&1 1>&2 2>&3)
  else
    NEW_HOSTNAME=$1
    true
  fi
  if [ $? -eq 0 ]; then
    echo $NEW_HOSTNAME > /etc/hostname
    sed -i "s/127.0.1.1.*$CURRENT_HOSTNAME/127.0.1.1\t$NEW_HOSTNAME/g" /etc/hosts
    ASK_TO_REBOOT=1
  fi
}

do_finish() {
  #disable_raspi_config_at_boot
  if [ $ASK_TO_REBOOT -eq 1 ]; then
    whiptail --yesno "Would you like to reboot now?" 20 60 2
    if [ $? -eq 0 ]; then # yes
      sync
      reboot
    fi
  fi
  exit 0
}

# $1 = filename, $2 = key name
get_json_string_val() {
  sed -n -e "s/^[[:space:]]*\"$2\"[[:space:]]*:[[:space:]]*\"\(.*\)\"[[:space:]]*,$/\1/p" $1
}

nonint() {
    $*
}

#
# Command line options for non-interactive use
#
for i in $*
do
  case $i in
  --get-rootfs)
    INTERACTIVE=False
    do_get_root_part
    exit 0
    ;;
  --expand-rootfs)
    INTERACTIVE=False
    do_expand_rootfs
    printf "Please reboot\n"
    exit 0
    ;;
  nonint)
    INTERACTIVE=False
    $@
    ;;
  *)
    # unknown option
    ;;
  esac
done

#if [ "GET" = "${OPT_MEMORY_SPLIT:-}" ]; then
#  set -u # Fail on unset variables
#  get_current_memory_split
#  echo $CURRENT_MEMSPLIT
#  exit 0
#fi

# Everything else needs to be run as root
if [ $(id -u) -ne 0 ]; then
  printf "Script must be run as root. Try 'sudo dru-config'\n"
  exit 1
fi

do_expand_rootfs
