#ifndef MY_ABC_HERE
#define MY_ABC_HERE
#endif
 
#include <linux/time.h>
#include <linux/fs.h>
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
#include <linux/fsnotify.h>
#endif  
#include <linux/jbd2.h>
#include <linux/mount.h>
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
#include <linux/swap.h>
#endif  
#include <linux/path.h>
#include <linux/aio.h>
#include <linux/quotaops.h>
#include <linux/pagevec.h>
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
#include <linux/socket.h>
#include <linux/net.h>
#include <net/sock.h>
#endif  
#include "ext4.h"
#include "ext4_jbd2.h"
#include "xattr.h"
#include "acl.h"

static int ext4_release_file(struct inode *inode, struct file *filp)
{
	if (ext4_test_inode_state(inode, EXT4_STATE_DA_ALLOC_CLOSE)) {
		ext4_alloc_da_blocks(inode);
		ext4_clear_inode_state(inode, EXT4_STATE_DA_ALLOC_CLOSE);
	}
	 
	if ((filp->f_mode & FMODE_WRITE) &&
			(atomic_read(&inode->i_writecount) == 1) &&
		        !EXT4_I(inode)->i_reserved_data_blocks)
	{
		down_write(&EXT4_I(inode)->i_data_sem);
		ext4_discard_preallocations(inode);
		up_write(&EXT4_I(inode)->i_data_sem);
	}
	if (is_dx(inode) && filp->private_data)
		ext4_htree_free_dir_info(filp->private_data);

	return 0;
}

void ext4_unwritten_wait(struct inode *inode)
{
	wait_queue_head_t *wq = ext4_ioend_wq(inode);

	wait_event(*wq, (atomic_read(&EXT4_I(inode)->i_unwritten) == 0));
}

static int
ext4_unaligned_aio(struct inode *inode, const struct iovec *iov,
		   unsigned long nr_segs, loff_t pos)
{
	struct super_block *sb = inode->i_sb;
	int blockmask = sb->s_blocksize - 1;
	size_t count = iov_length(iov, nr_segs);
	loff_t final_size = pos + count;

	if (pos >= i_size_read(inode))
		return 0;

	if ((pos & blockmask) || (final_size & blockmask))
		return 1;

	return 0;
}

static ssize_t
ext4_file_dio_write(struct kiocb *iocb, const struct iovec *iov,
		    unsigned long nr_segs, loff_t pos)
{
	struct file *file = iocb->ki_filp;
	struct inode *inode = file->f_mapping->host;
	struct blk_plug plug;
	int unaligned_aio = 0;
	ssize_t ret;
	int *overwrite = iocb->private;
	size_t length = iov_length(iov, nr_segs);

	if (ext4_test_inode_flag(inode, EXT4_INODE_EXTENTS) &&
	    !is_sync_kiocb(iocb))
		unaligned_aio = ext4_unaligned_aio(inode, iov, nr_segs, pos);

	if (unaligned_aio) {
		mutex_lock(ext4_aio_mutex(inode));
		ext4_unwritten_wait(inode);
	}

	BUG_ON(iocb->ki_pos != pos);

	mutex_lock(&inode->i_mutex);
	blk_start_plug(&plug);

	if (ext4_should_dioread_nolock(inode) && !unaligned_aio &&
	    !file->f_mapping->nrpages && pos + length <= i_size_read(inode)) {
		struct ext4_map_blocks map;
		unsigned int blkbits = inode->i_blkbits;
		int err, len;

		map.m_lblk = pos >> blkbits;
		map.m_len = (EXT4_BLOCK_ALIGN(pos + length, blkbits) >> blkbits)
			- map.m_lblk;
		len = map.m_len;

		err = ext4_map_blocks(NULL, inode, &map, 0);
		 
		if (err == len && (map.m_flags & EXT4_MAP_MAPPED))
			*overwrite = 1;
	}

	ret = __generic_file_aio_write(iocb, iov, nr_segs, &iocb->ki_pos);
	mutex_unlock(&inode->i_mutex);

	if (ret > 0 || ret == -EIOCBQUEUED) {
		ssize_t err;

		err = generic_write_sync(file, pos, ret);
		if (err < 0 && ret > 0)
			ret = err;
	}
	blk_finish_plug(&plug);

	if (unaligned_aio)
		mutex_unlock(ext4_aio_mutex(inode));

	return ret;
}

static ssize_t
ext4_file_write(struct kiocb *iocb, const struct iovec *iov,
		unsigned long nr_segs, loff_t pos)
{
	struct inode *inode = file_inode(iocb->ki_filp);
	ssize_t ret;
	int overwrite = 0;

	if (!(ext4_test_inode_flag(inode, EXT4_INODE_EXTENTS))) {
		struct ext4_sb_info *sbi = EXT4_SB(inode->i_sb);
		size_t length = iov_length(iov, nr_segs);

		if ((pos > sbi->s_bitmap_maxbytes ||
		    (pos == sbi->s_bitmap_maxbytes && length > 0)))
			return -EFBIG;

		if (pos + length > sbi->s_bitmap_maxbytes) {
			nr_segs = iov_shorten((struct iovec *)iov, nr_segs,
					      sbi->s_bitmap_maxbytes - pos);
		}
	}

	iocb->private = &overwrite;
	if (unlikely(iocb->ki_filp->f_flags & O_DIRECT))
		ret = ext4_file_dio_write(iocb, iov, nr_segs, pos);
	else
		ret = generic_file_aio_write(iocb, iov, nr_segs, pos);

	return ret;
}

static const struct vm_operations_struct ext4_file_vm_ops = {
	.fault		= filemap_fault,
	.page_mkwrite   = ext4_page_mkwrite,
	.remap_pages	= generic_file_remap_pages,
};

static int ext4_file_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct address_space *mapping = file->f_mapping;

	if (!mapping->a_ops->readpage)
		return -ENOEXEC;
	file_accessed(file);
	vma->vm_ops = &ext4_file_vm_ops;
	return 0;
}

static int ext4_file_open(struct inode * inode, struct file * filp)
{
	struct super_block *sb = inode->i_sb;
	struct ext4_sb_info *sbi = EXT4_SB(inode->i_sb);
	struct ext4_inode_info *ei = EXT4_I(inode);
	struct vfsmount *mnt = filp->f_path.mnt;
	struct path path;
	char buf[64], *cp;

	if (unlikely(!(sbi->s_mount_flags & EXT4_MF_MNTDIR_SAMPLED) &&
		     !(sb->s_flags & MS_RDONLY))) {
		sbi->s_mount_flags |= EXT4_MF_MNTDIR_SAMPLED;
		 
		memset(buf, 0, sizeof(buf));
		path.mnt = mnt;
		path.dentry = mnt->mnt_root;
		cp = d_path(&path, buf, sizeof(buf));
		if (!IS_ERR(cp)) {
			handle_t *handle;
			int err;

			handle = ext4_journal_start_sb(sb, EXT4_HT_MISC, 1);
			if (IS_ERR(handle))
				return PTR_ERR(handle);
			err = ext4_journal_get_write_access(handle, sbi->s_sbh);
			if (err) {
				ext4_journal_stop(handle);
				return err;
			}
			strlcpy(sbi->s_es->s_last_mounted, cp,
				sizeof(sbi->s_es->s_last_mounted));
			ext4_handle_dirty_super(handle, sb);
			ext4_journal_stop(handle);
		}
	}
	 
	if (sbi->s_journal && !ei->jinode && (filp->f_mode & FMODE_WRITE)) {
		struct jbd2_inode *jinode = jbd2_alloc_inode(GFP_KERNEL);

		spin_lock(&inode->i_lock);
		if (!ei->jinode) {
			if (!jinode) {
				spin_unlock(&inode->i_lock);
				return -ENOMEM;
			}
			ei->jinode = jinode;
			jbd2_journal_init_jbd_inode(ei->jinode, inode);
			jinode = NULL;
		}
		spin_unlock(&inode->i_lock);
		if (unlikely(jinode != NULL))
			jbd2_free_inode(jinode);
	}
	return dquot_file_open(inode, filp);
}

#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
ssize_t ext4_splice_from_socket(struct file *file, struct socket *sock,
				loff_t __user *ppos, size_t count_req)
{
	struct address_space *mapping = file->f_mapping;
	const struct address_space_operations *a_ops = mapping->a_ops;
	struct inode *inode = mapping->host;
	loff_t pos;
	int err = 0, remaining;
	struct kvec iov;
	struct msghdr msg = { 0 };
	size_t written = 0, verified_sz;

	if (unlikely(file->f_flags & O_DIRECT))
		return -EINVAL;

	if (copy_from_user(&pos, ppos, sizeof(loff_t)))
		return -EFAULT;

	err = rw_verify_area(WRITE, file, &file->f_pos, count_req);
	if (err < 0) {
		pr_debug("%s: rw_verify_area, err %d\n", __func__, err);
		return err;
	}

	verified_sz = err;

	file_start_write(file);

	mutex_lock(&inode->i_mutex);

	if (!(ext4_test_inode_flag(inode, EXT4_INODE_EXTENTS))) {
		struct ext4_sb_info *sbi = EXT4_SB(inode->i_sb);

		if ((pos > sbi->s_bitmap_maxbytes) ||
		    (pos == sbi->s_bitmap_maxbytes && verified_sz > 0)) {
			mutex_unlock(&inode->i_mutex);
			file_end_write(file);
			return -EFBIG;
		}

		if (pos + verified_sz > sbi->s_bitmap_maxbytes)
			verified_sz = sbi->s_bitmap_maxbytes - pos;
	}

	current->backing_dev_info = mapping->backing_dev_info;

	err = generic_write_checks(file, &pos, &verified_sz,
				   S_ISBLK(inode->i_mode));
	if (err) {
		pr_debug("%s: generic_write_checks err %d\n", __func__, err);
		goto cleanup;
	}

	if (verified_sz == 0) {
		pr_debug("%s: generic_write_checks err, verified_sz %d\n",
			 __func__, verified_sz);
		goto cleanup;
	}

	err = file_remove_suid(file);
	if (err) {
		pr_debug("%s: file_remove_suid, err %d\n", __func__, err);
		goto cleanup;
	}

	err = file_update_time(file);
	if (err) {
		pr_debug("%s: file_update_time, err %d\n", __func__, err);
		goto cleanup;
	}

	remaining = verified_sz;

	while (remaining > 0) {
		unsigned long offset;	 
		unsigned long bytes;	 
		int copied;		 
		struct page *page;
		void *fsdata;
		long rcvtimeo;
		char *paddr;

		offset = (pos & (PAGE_CACHE_SIZE - 1));
		bytes = min_t(unsigned long, PAGE_CACHE_SIZE - offset,
			      remaining);

		err = a_ops->write_begin(file, mapping, pos, bytes,
					 AOP_FLAG_UNINTERRUPTIBLE,
					 &page, &fsdata);
		if (unlikely(err)) {
			pr_debug("%s: write_begin err %d\n", __func__, err);
			break;
		}

		if (mapping_writably_mapped(mapping))
			flush_dcache_page(page);

		paddr = kmap(page) + offset;
		iov.iov_base = paddr;
		iov.iov_len = bytes;

		rcvtimeo = sock->sk->sk_rcvtimeo;
		sock->sk->sk_rcvtimeo = 5 * HZ;

		copied = kernel_recvmsg(sock, &msg, &iov, 1,
					bytes, MSG_WAITALL);

		sock->sk->sk_rcvtimeo = rcvtimeo;

		if (unlikely(copied <= 0)) {
			kunmap(page);

			err = copied;

			pr_debug("%s: kernel_recvmsg err %d\n", __func__, err);

			a_ops->write_end(file, mapping, pos,
					 bytes, 0, page, fsdata);
			break;
		}

		if (unlikely(copied != bytes)) {
			char *kaddr;
			char *buff;

			pr_debug("%s: partial bytes %ld copied %d\n",
				 __func__, bytes, copied);

			buff = kmalloc(copied, GFP_KERNEL);
			if (unlikely(!buff)) {
				err = -ENOMEM;
				break;
			}
			 
			memcpy(buff, paddr, copied);

			kunmap(page);
			err = a_ops->write_end(file, mapping, pos,
					       bytes, 0, page, fsdata);
			if (unlikely(err < 0)) {
				kfree(buff);
				pr_debug("%s: write_end partial, err %d\n",
					 __func__, err);
				break;
			}

			err = a_ops->write_begin(file, mapping, pos, copied,
						 AOP_FLAG_UNINTERRUPTIBLE,
						 &page, &fsdata);
			if (unlikely(err)) {
				kfree(buff);
				pr_debug("%s: write_begin partial, err %d\n",
					 __func__, err);
				break;
			}

			if (mapping_writably_mapped(mapping))
				flush_dcache_page(page);

			kaddr = kmap_atomic(page) + offset;
			memcpy(kaddr, buff, copied);

			kfree(buff);
			kunmap_atomic(kaddr);

			mark_page_accessed(page);
			err = a_ops->write_end(file, mapping, pos, copied,
					       copied, page, fsdata);
			if (unlikely(err < 0)) {
				pr_debug("%s: write_end partial, err %d\n",
					 __func__, err);
				break;
			}

			BUG_ON(copied != err);

			pos += copied;
			written += copied;

			break;
		}

		kunmap(page);

		mark_page_accessed(page);
		err = a_ops->write_end(file, mapping, pos, bytes,
				       copied, page, fsdata);

		if (unlikely(err < 0)) {
			pr_debug("%s: write_end, err %d\n", __func__, err);
			break;
		}

		BUG_ON(copied != err);

		remaining -= copied;
		pos += copied;
		written += copied;
	}

	if (written > 0)
		balance_dirty_pages_ratelimited(mapping);

cleanup:
	current->backing_dev_info = NULL;

	mutex_unlock(&inode->i_mutex);

	if (written > 0) {
		err = generic_write_sync(file, pos - written, written);
		if (err < 0) {
			written = 0;
			goto done;
		}
		fsnotify_modify(file);

		if (copy_to_user(ppos, &pos, sizeof(loff_t))) {
			written = 0;
			err = -EFAULT;
		}
	}
done:
	file_end_write(file);

	return written ? written : err;
}
#endif  

static int ext4_find_unwritten_pgoff(struct inode *inode,
				     int whence,
				     struct ext4_map_blocks *map,
				     loff_t *offset)
{
	struct pagevec pvec;
	unsigned int blkbits;
	pgoff_t index;
	pgoff_t end;
	loff_t endoff;
	loff_t startoff;
	loff_t lastoff;
	int found = 0;

	blkbits = inode->i_sb->s_blocksize_bits;
	startoff = *offset;
	lastoff = startoff;
	endoff = (loff_t)(map->m_lblk + map->m_len) << blkbits;

	index = startoff >> PAGE_CACHE_SHIFT;
	end = endoff >> PAGE_CACHE_SHIFT;

	pagevec_init(&pvec, 0);
	do {
		int i, num;
		unsigned long nr_pages;

		num = min_t(pgoff_t, end - index, PAGEVEC_SIZE);
		nr_pages = pagevec_lookup(&pvec, inode->i_mapping, index,
					  (pgoff_t)num);
		if (nr_pages == 0) {
			if (whence == SEEK_DATA)
				break;

			BUG_ON(whence != SEEK_HOLE);
			 
			if (lastoff == startoff || lastoff < endoff)
				found = 1;
			break;
		}

		if (lastoff == startoff && whence == SEEK_HOLE &&
		    lastoff < page_offset(pvec.pages[0])) {
			found = 1;
			break;
		}

		for (i = 0; i < nr_pages; i++) {
			struct page *page = pvec.pages[i];
			struct buffer_head *bh, *head;

			if (lastoff < endoff && whence == SEEK_HOLE &&
			    page->index > end) {
				found = 1;
				*offset = lastoff;
				goto out;
			}

			lock_page(page);

			if (unlikely(page->mapping != inode->i_mapping)) {
				unlock_page(page);
				continue;
			}

			if (!page_has_buffers(page)) {
				unlock_page(page);
				continue;
			}

			if (page_has_buffers(page)) {
				lastoff = page_offset(page);
				bh = head = page_buffers(page);
				do {
					if (buffer_uptodate(bh) ||
					    buffer_unwritten(bh)) {
						if (whence == SEEK_DATA)
							found = 1;
					} else {
						if (whence == SEEK_HOLE)
							found = 1;
					}
					if (found) {
						*offset = max_t(loff_t,
							startoff, lastoff);
						unlock_page(page);
						goto out;
					}
					lastoff += bh->b_size;
					bh = bh->b_this_page;
				} while (bh != head);
			}

			lastoff = page_offset(page) + PAGE_SIZE;
			unlock_page(page);
		}

		if (nr_pages < num && whence == SEEK_HOLE) {
			found = 1;
			*offset = lastoff;
			break;
		}

		index = pvec.pages[i - 1]->index + 1;
		pagevec_release(&pvec);
	} while (index <= end);

out:
	pagevec_release(&pvec);
	return found;
}

static loff_t ext4_seek_data(struct file *file, loff_t offset, loff_t maxsize)
{
	struct inode *inode = file->f_mapping->host;
	struct ext4_map_blocks map;
	struct extent_status es;
	ext4_lblk_t start, last, end;
	loff_t dataoff, isize;
	int blkbits;
	int ret = 0;

	mutex_lock(&inode->i_mutex);

	isize = i_size_read(inode);
	if (offset >= isize) {
		mutex_unlock(&inode->i_mutex);
		return -ENXIO;
	}

	blkbits = inode->i_sb->s_blocksize_bits;
	start = offset >> blkbits;
	last = start;
	end = isize >> blkbits;
	dataoff = offset;

	do {
		map.m_lblk = last;
		map.m_len = end - last + 1;
		ret = ext4_map_blocks(NULL, inode, &map, 0);
		if (ret > 0 && !(map.m_flags & EXT4_MAP_UNWRITTEN)) {
			if (last != start)
				dataoff = (loff_t)last << blkbits;
			break;
		}

		ext4_es_find_delayed_extent_range(inode, last, last, &es);
		if (es.es_len != 0 && in_range(last, es.es_lblk, es.es_len)) {
			if (last != start)
				dataoff = (loff_t)last << blkbits;
			break;
		}

		if (map.m_flags & EXT4_MAP_UNWRITTEN) {
			int unwritten;
			unwritten = ext4_find_unwritten_pgoff(inode, SEEK_DATA,
							      &map, &dataoff);
			if (unwritten)
				break;
		}

		last++;
		dataoff = (loff_t)last << blkbits;
	} while (last <= end);

	mutex_unlock(&inode->i_mutex);

	if (dataoff > isize)
		return -ENXIO;

	if (dataoff < 0 && !(file->f_mode & FMODE_UNSIGNED_OFFSET))
		return -EINVAL;
	if (dataoff > maxsize)
		return -EINVAL;

	if (dataoff != file->f_pos) {
		file->f_pos = dataoff;
		file->f_version = 0;
	}

	return dataoff;
}

static loff_t ext4_seek_hole(struct file *file, loff_t offset, loff_t maxsize)
{
	struct inode *inode = file->f_mapping->host;
	struct ext4_map_blocks map;
	struct extent_status es;
	ext4_lblk_t start, last, end;
	loff_t holeoff, isize;
	int blkbits;
	int ret = 0;

	mutex_lock(&inode->i_mutex);

	isize = i_size_read(inode);
	if (offset >= isize) {
		mutex_unlock(&inode->i_mutex);
		return -ENXIO;
	}

	blkbits = inode->i_sb->s_blocksize_bits;
	start = offset >> blkbits;
	last = start;
	end = isize >> blkbits;
	holeoff = offset;

	do {
		map.m_lblk = last;
		map.m_len = end - last + 1;
		ret = ext4_map_blocks(NULL, inode, &map, 0);
		if (ret > 0 && !(map.m_flags & EXT4_MAP_UNWRITTEN)) {
			last += ret;
			holeoff = (loff_t)last << blkbits;
			continue;
		}

		ext4_es_find_delayed_extent_range(inode, last, last, &es);
		if (es.es_len != 0 && in_range(last, es.es_lblk, es.es_len)) {
			last = es.es_lblk + es.es_len;
			holeoff = (loff_t)last << blkbits;
			continue;
		}

		if (map.m_flags & EXT4_MAP_UNWRITTEN) {
			int unwritten;
			unwritten = ext4_find_unwritten_pgoff(inode, SEEK_HOLE,
							      &map, &holeoff);
			if (!unwritten) {
				last += ret;
				holeoff = (loff_t)last << blkbits;
				continue;
			}
		}

		break;
	} while (last <= end);

	mutex_unlock(&inode->i_mutex);

	if (holeoff > isize)
		holeoff = isize;

	if (holeoff < 0 && !(file->f_mode & FMODE_UNSIGNED_OFFSET))
		return -EINVAL;
	if (holeoff > maxsize)
		return -EINVAL;

	if (holeoff != file->f_pos) {
		file->f_pos = holeoff;
		file->f_version = 0;
	}

	return holeoff;
}

loff_t ext4_llseek(struct file *file, loff_t offset, int whence)
{
	struct inode *inode = file->f_mapping->host;
	loff_t maxbytes;

	if (!(ext4_test_inode_flag(inode, EXT4_INODE_EXTENTS)))
		maxbytes = EXT4_SB(inode->i_sb)->s_bitmap_maxbytes;
	else
		maxbytes = inode->i_sb->s_maxbytes;

	switch (whence) {
	case SEEK_SET:
	case SEEK_CUR:
	case SEEK_END:
		return generic_file_llseek_size(file, offset, whence,
						maxbytes, i_size_read(inode));
	case SEEK_DATA:
		return ext4_seek_data(file, offset, maxbytes);
	case SEEK_HOLE:
		return ext4_seek_hole(file, offset, maxbytes);
	}

	return -EINVAL;
}

const struct file_operations ext4_file_operations = {
	.llseek		= ext4_llseek,
	.read		= do_sync_read,
	.write		= do_sync_write,
	.aio_read	= generic_file_aio_read,
	.aio_write	= ext4_file_write,
	.unlocked_ioctl = ext4_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl	= ext4_compat_ioctl,
#endif
	.mmap		= ext4_file_mmap,
	.open		= ext4_file_open,
	.release	= ext4_release_file,
	.fsync		= ext4_sync_file,
	.splice_read	= generic_file_splice_read,
	.splice_write	= generic_file_splice_write,
#if defined(CONFIG_SYNO_LSP_ARMADA)
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
	.splice_from_socket = ext4_splice_from_socket,
#else
	.splice_from_socket = generic_splice_from_socket,
#endif  
#endif  
	.fallocate	= ext4_fallocate,
};

const struct inode_operations ext4_file_inode_operations = {
#ifdef MY_ABC_HERE
	.syno_getattr	= ext4_syno_getattr,
#endif
#ifdef MY_ABC_HERE
	.syno_get_archive_ver	= ext4_syno_get_archive_ver,
	.syno_set_archive_ver	= ext4_syno_set_archive_ver,
#endif
	.setattr	= ext4_setattr,
	.getattr	= ext4_getattr,
	.setxattr	= generic_setxattr,
	.getxattr	= generic_getxattr,
	.listxattr	= ext4_listxattr,
	.removexattr	= generic_removexattr,
	.get_acl	= ext4_get_acl,
	.fiemap		= ext4_fiemap,
};
