 
#include <linux/err.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/gpio.h>
#include <linux/mmc/card.h>
#include <linux/mmc/host.h>
#include <linux/mmc/slot-gpio.h>
#include <linux/platform_data/pxa_sdhci.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/pm.h>
#include <linux/pm_runtime.h>
#if defined(CONFIG_SYNO_LSP_ARMADA)
#include <linux/mbus.h>
#endif  

#include "sdhci.h"
#include "sdhci-pltfm.h"

#define PXAV3_RPM_DELAY_MS     50

#define SD_CLOCK_BURST_SIZE_SETUP		0x10A
#define SDCLK_SEL	0x100
#define SDCLK_DELAY_SHIFT	9
#define SDCLK_DELAY_MASK	0x1f

#if defined(CONFIG_SYNO_LSP_ARMADA)
#define SD_EXTRA_PARAM_REG	0x100
#else  
#define SD_CFG_FIFO_PARAM       0x100
#endif  
#define SDCFG_GEN_PAD_CLK_ON	(1<<6)
#define SDCFG_GEN_PAD_CLK_CNT_MASK	0xFF
#define SDCFG_GEN_PAD_CLK_CNT_SHIFT	24

#if defined(CONFIG_SYNO_LSP_ARMADA)
#define SD_FIFO_PARAM_REG	0x104
#define SD_USE_DAT3		BIT(7)
#define SD_OVRRD_CLK_OEN	BIT(11)
#define SD_FORCE_CLK_ON		BIT(12)
#endif  

#define SD_SPI_MODE          0x108
#define SD_CE_ATA_1          0x10C
#if defined(CONFIG_SYNO_LSP_ARMADA)
#define SDCE_MMC_CARD		BIT(28)
#endif  

#define SD_CE_ATA_2          0x10E
#define SDCE_MISC_INT		(1<<2)
#define SDCE_MISC_INT_EN	(1<<1)

#if defined(CONFIG_SYNO_LSP_ARMADA)
 
#define SDHCI_WINDOW_CTRL(i)	(0x80 + ((i) << 3))
#define SDHCI_WINDOW_BASE(i)	(0x84 + ((i) << 3))
#define SDHCI_MAX_WIN_NUM	8

#define SDIO3_CONF_CLK_INV	BIT(0)
#define SDIO3_CONF_SD_FB_CLK	BIT(2)

static int mv_conf_mbus_windows(struct device *dev, void __iomem *regs,
				const struct mbus_dram_target_info *dram)
{
	int i;

	if (!dram) {
		dev_err(dev, "no mbus dram info\n");
		return -EINVAL;
	}

	for (i = 0; i < SDHCI_MAX_WIN_NUM; i++) {
		writel(0, regs + SDHCI_WINDOW_CTRL(i));
		writel(0, regs + SDHCI_WINDOW_BASE(i));
	}

	for (i = 0; i < dram->num_cs; i++) {
		const struct mbus_dram_window *cs = dram->cs + i;

		writel(((cs->size - 1) & 0xffff0000) |
			(cs->mbus_attr << 8) |
			(dram->mbus_dram_target_id << 4) | 1,
			regs + SDHCI_WINDOW_CTRL(i));
		 
		writel(cs->base, regs + SDHCI_WINDOW_BASE(i));
	}

	return 0;
}
#endif  

static void pxav3_set_private_registers(struct sdhci_host *host, u8 mask)
{
	struct platform_device *pdev = to_platform_device(mmc_dev(host->mmc));
	struct sdhci_pxa_platdata *pdata = pdev->dev.platform_data;
#if defined(CONFIG_SYNO_LSP_ARMADA)
	struct device_node *np = pdev->dev.of_node;
	u32 reg_val;
#endif  

	if (mask == SDHCI_RESET_ALL) {
		 
		if (pdata && 0 != pdata->clk_delay_cycles) {
			u16 tmp;

			tmp = readw(host->ioaddr + SD_CLOCK_BURST_SIZE_SETUP);
			tmp |= (pdata->clk_delay_cycles & SDCLK_DELAY_MASK)
				<< SDCLK_DELAY_SHIFT;
			tmp |= SDCLK_SEL;
			writew(tmp, host->ioaddr + SD_CLOCK_BURST_SIZE_SETUP);
		}

#if defined(CONFIG_SYNO_LSP_ARMADA)
		if (of_device_is_compatible(np, "marvell,armada-380-sdhci") &&
		    host->quirks2 & SDHCI_QUIRK2_KEEP_INT_CLK_ON) {
			reg_val = sdhci_readl(host, SD_FIFO_PARAM_REG);
			reg_val |= SD_USE_DAT3 | SD_OVRRD_CLK_OEN |
				   SD_FORCE_CLK_ON;
			sdhci_writel(host, reg_val, SD_FIFO_PARAM_REG);

			reg_val = sdhci_readl(host, SDHCI_CLOCK_CONTROL);
			reg_val |= SDHCI_CLOCK_INT_EN;
			sdhci_writel(host, reg_val, SDHCI_CLOCK_CONTROL);
		}
#endif  
	}
}

#if defined(CONFIG_SYNO_LSP_ARMADA)
static void pxav3_init_card(struct sdhci_host *host, struct mmc_card *card)
{
	struct platform_device *pdev = to_platform_device(mmc_dev(host->mmc));
	struct device_node *np = pdev->dev.of_node;
	u32 reg_val;

	if (of_device_is_compatible(np, "marvell,armada-380-sdhci")) {
		reg_val = sdhci_readl(host, SD_CE_ATA_1);
		if (mmc_card_mmc(card))
			reg_val |= SDCE_MMC_CARD;
		else
			reg_val &= ~SDCE_MMC_CARD;
		sdhci_writel(host, reg_val, SD_CE_ATA_1);
	}
}
#endif  

#define MAX_WAIT_COUNT 5
static void pxav3_gen_init_74_clocks(struct sdhci_host *host, u8 power_mode)
{
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct sdhci_pxa *pxa = pltfm_host->priv;
	u16 tmp;
	int count;

	if (pxa->power_mode == MMC_POWER_UP
			&& power_mode == MMC_POWER_ON) {

		dev_dbg(mmc_dev(host->mmc),
				"%s: slot->power_mode = %d,"
				"ios->power_mode = %d\n",
				__func__,
				pxa->power_mode,
				power_mode);

		tmp = readw(host->ioaddr + SD_CE_ATA_2);
		tmp |= SDCE_MISC_INT_EN;
		writew(tmp, host->ioaddr + SD_CE_ATA_2);

#if defined(CONFIG_SYNO_LSP_ARMADA)
		tmp = readw(host->ioaddr + SD_EXTRA_PARAM_REG);
		tmp |= SDCFG_GEN_PAD_CLK_ON;
		writew(tmp, host->ioaddr + SD_EXTRA_PARAM_REG);
#else  
		tmp = readw(host->ioaddr + SD_CFG_FIFO_PARAM);
		tmp |= SDCFG_GEN_PAD_CLK_ON;
		writew(tmp, host->ioaddr + SD_CFG_FIFO_PARAM);
#endif  

		udelay(740);
		count = 0;

		while (count++ < MAX_WAIT_COUNT) {
			if ((readw(host->ioaddr + SD_CE_ATA_2)
						& SDCE_MISC_INT) == 0)
				break;
			udelay(10);
		}

		if (count == MAX_WAIT_COUNT)
			dev_warn(mmc_dev(host->mmc), "74 clock interrupt not cleared\n");

		tmp = readw(host->ioaddr + SD_CE_ATA_2);
		tmp |= SDCE_MISC_INT;
		writew(tmp, host->ioaddr + SD_CE_ATA_2);
	}
	pxa->power_mode = power_mode;
}

static int pxav3_set_uhs_signaling(struct sdhci_host *host, unsigned int uhs)
{
#if defined(CONFIG_SYNO_LSP_ARMADA)
	struct platform_device *pdev = to_platform_device(mmc_dev(host->mmc));
	struct device_node *np = pdev->dev.of_node;
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct sdhci_pxa *pxa = pltfm_host->priv;
	u16 ctrl_2;
	u8 reg_val;
#else  
	u16 ctrl_2;
#endif  

	ctrl_2 = sdhci_readw(host, SDHCI_HOST_CONTROL2);

	ctrl_2 &= ~SDHCI_CTRL_UHS_MASK;
	switch (uhs) {
	case MMC_TIMING_UHS_SDR12:
		ctrl_2 |= SDHCI_CTRL_UHS_SDR12;
		break;
	case MMC_TIMING_UHS_SDR25:
		ctrl_2 |= SDHCI_CTRL_UHS_SDR25;
		break;
	case MMC_TIMING_UHS_SDR50:
		ctrl_2 |= SDHCI_CTRL_UHS_SDR50 | SDHCI_CTRL_VDD_180;
		break;
	case MMC_TIMING_UHS_SDR104:
		ctrl_2 |= SDHCI_CTRL_UHS_SDR104 | SDHCI_CTRL_VDD_180;
		break;
	case MMC_TIMING_UHS_DDR50:
		ctrl_2 |= SDHCI_CTRL_UHS_DDR50 | SDHCI_CTRL_VDD_180;
		break;
	}

#if defined(CONFIG_SYNO_LSP_ARMADA)
	 
	if (of_device_is_compatible(np, "marvell,armada-380-sdhci")) {
		reg_val = readb(pxa->sdio3_conf_reg);
		if (uhs == MMC_TIMING_UHS_SDR50 ||
		    uhs == MMC_TIMING_UHS_DDR50) {
			reg_val &= ~SDIO3_CONF_CLK_INV;
			reg_val |= SDIO3_CONF_SD_FB_CLK;
		} else if (uhs == MMC_TIMING_MMC_HS ||
		           uhs == MMC_TIMING_SD_HS) {
			reg_val &= ~SDIO3_CONF_CLK_INV;
			reg_val &= ~SDIO3_CONF_SD_FB_CLK;
		} else {
			reg_val |= SDIO3_CONF_CLK_INV;
			reg_val &= ~SDIO3_CONF_SD_FB_CLK;
		}
		writeb(reg_val, pxa->sdio3_conf_reg);
	}
#endif  

	sdhci_writew(host, ctrl_2, SDHCI_HOST_CONTROL2);
	dev_dbg(mmc_dev(host->mmc),
		"%s uhs = %d, ctrl_2 = %04X\n",
		__func__, uhs, ctrl_2);

	return 0;
}

static const struct sdhci_ops pxav3_sdhci_ops = {
	.platform_reset_exit = pxav3_set_private_registers,
	.set_uhs_signaling = pxav3_set_uhs_signaling,
	.platform_send_init_74_clocks = pxav3_gen_init_74_clocks,
	.get_max_clock = sdhci_pltfm_clk_get_max_clock,
#if defined(CONFIG_SYNO_LSP_ARMADA)
	.init_card = pxav3_init_card,
#endif  
};

static struct sdhci_pltfm_data sdhci_pxav3_pdata = {
	.quirks = SDHCI_QUIRK_DATA_TIMEOUT_USES_SDCLK
		| SDHCI_QUIRK_NO_ENDATTR_IN_NOPDESC
		| SDHCI_QUIRK_32BIT_ADMA_SIZE
#if defined(CONFIG_SYNO_LSP_ARMADA)
 
		| SDHCI_QUIRK_MISSING_CAPS,
#else  
		| SDHCI_QUIRK_CAP_CLOCK_BASE_BROKEN,
#endif  
	.ops = &pxav3_sdhci_ops,
};

#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
static struct sdhci_pltfm_data sdhci_armada_380_pdata = {
	.quirks = SDHCI_QUIRK_DATA_TIMEOUT_USES_SDCLK
		| SDHCI_QUIRK_NO_ENDATTR_IN_NOPDESC
		| SDHCI_QUIRK_32BIT_ADMA_SIZE
		| SDHCI_QUIRK_MISSING_CAPS,
	.ops = &pxav3_sdhci_ops,
};
#endif  

#ifdef CONFIG_OF
static const struct of_device_id sdhci_pxav3_of_match[] = {
	{
		.compatible = "mrvl,pxav3-mmc",
#if defined(CONFIG_SYNO_LSP_ARMADA)
		.data = &sdhci_pxav3_pdata,
	},
	{
		.compatible = "marvell,armada-380-sdhci",
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
		.data = &sdhci_armada_380_pdata,
#else
		.data = &sdhci_pxav3_pdata,
#endif  
#endif  
	},
	{},
};
MODULE_DEVICE_TABLE(of, sdhci_pxav3_of_match);

static struct sdhci_pxa_platdata *pxav3_get_mmc_pdata(struct device *dev)
{
	struct sdhci_pxa_platdata *pdata;
	struct device_node *np = dev->of_node;
	u32 clk_delay_cycles;

	pdata = devm_kzalloc(dev, sizeof(*pdata), GFP_KERNEL);
	if (!pdata)
		return NULL;

	if (!of_property_read_u32(np, "mrvl,clk-delay-cycles",
				  &clk_delay_cycles))
		pdata->clk_delay_cycles = clk_delay_cycles;

	return pdata;
}
#else
static inline struct sdhci_pxa_platdata *pxav3_get_mmc_pdata(struct device *dev)
{
	return NULL;
}
#endif

static int sdhci_pxav3_probe(struct platform_device *pdev)
{
	struct sdhci_pltfm_host *pltfm_host;
	struct sdhci_pxa_platdata *pdata = pdev->dev.platform_data;
	struct device *dev = &pdev->dev;
#if defined(CONFIG_SYNO_LSP_ARMADA)
	struct device_node *np = pdev->dev.of_node;
#endif  
	struct sdhci_host *host = NULL;
	struct sdhci_pxa *pxa = NULL;
#if defined(CONFIG_SYNO_LSP_ARMADA)
	struct resource *res;
#endif  
	const struct of_device_id *match;
#if defined(CONFIG_SYNO_LSP_ARMADA)
	const struct sdhci_pltfm_data *sdhci_pltfm_data;
#endif  

	int ret;
	struct clk *clk;

	pxa = kzalloc(sizeof(struct sdhci_pxa), GFP_KERNEL);
	if (!pxa)
		return -ENOMEM;

#if defined(CONFIG_SYNO_LSP_ARMADA)
	match = of_match_device(of_match_ptr(sdhci_pxav3_of_match), &pdev->dev);

	if (match)
		sdhci_pltfm_data = match->data;
	else
		sdhci_pltfm_data = &sdhci_pxav3_pdata;

	host = sdhci_pltfm_init(pdev, sdhci_pltfm_data);
#else  
	host = sdhci_pltfm_init(pdev, &sdhci_pxav3_pdata);
#endif  
	if (IS_ERR(host)) {
		kfree(pxa);
		return PTR_ERR(host);
	}

#if defined(CONFIG_SYNO_LSP_ARMADA)
	if (of_device_is_compatible(np, "marvell,armada-380-sdhci")) {
		res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
		pxa->mbus_win_regs = devm_ioremap_resource(&pdev->dev, res);
		if (IS_ERR(pxa->mbus_win_regs)) {
			ret = PTR_ERR(pxa->mbus_win_regs);
			goto err_clk_get;
		}

		res = platform_get_resource(pdev, IORESOURCE_MEM, 2);
		pxa->sdio3_conf_reg = devm_ioremap_resource(&pdev->dev, res);
		if (IS_ERR(pxa->sdio3_conf_reg)) {
			ret = PTR_ERR(pxa->sdio3_conf_reg);
			goto err_clk_get;
		}

		ret = mv_conf_mbus_windows(&pdev->dev, pxa->mbus_win_regs,
					   mv_mbus_dram_info());
		if (ret < 0)
			goto err_clk_get;
	}
#endif  

	pltfm_host = sdhci_priv(host);
	pltfm_host->priv = pxa;

	clk = clk_get(dev, NULL);
	if (IS_ERR(clk)) {
		dev_err(dev, "failed to get io clock\n");
		ret = PTR_ERR(clk);
		goto err_clk_get;
	}
	pltfm_host->clk = clk;
	clk_prepare_enable(clk);

	host->mmc->caps |= MMC_CAP_1_8V_DDR;

#if defined(CONFIG_SYNO_LSP_ARMADA)
	 
#else  
	match = of_match_device(of_match_ptr(sdhci_pxav3_of_match), &pdev->dev);
#endif  
	if (match) {
		mmc_of_parse(host->mmc);
		sdhci_get_of_property(pdev);
		pdata = pxav3_get_mmc_pdata(dev);
		pdev->dev.platform_data = pdata;
#if defined(CONFIG_SYNO_LSP_ARMADA)
		host->caps = sdhci_readl(host, SDHCI_CAPABILITIES);
		host->caps1 = sdhci_readl(host, SDHCI_CAPABILITIES_1);

		if (of_device_is_compatible(np, "marvell,armada-380-sdhci")) {
			if (of_get_property(np, "no-1-8-v", NULL)) {
				host->caps &= ~SDHCI_CAN_VDD_180;
				host->mmc->caps &= ~MMC_CAP_1_8V_DDR;
			} else
				host->caps &= ~SDHCI_CAN_VDD_330;

			host->caps1 &= ~(SDHCI_SUPPORT_SDR104 |
					 SDHCI_USE_SDR50_TUNING);

			if (of_property_read_bool(np, "dat3-cd") &&
			    !of_property_read_bool(np, "broken-cd"))
				host->quirks2 |= SDHCI_QUIRK2_KEEP_INT_CLK_ON;
		}
#endif  
	} else if (pdata) {
		 
		if (pdata->flags & PXA_FLAG_CARD_PERMANENT)
			host->mmc->caps |= MMC_CAP_NONREMOVABLE;

		if (pdata->flags & PXA_FLAG_SD_8_BIT_CAPABLE_SLOT)
			host->mmc->caps |= MMC_CAP_8_BIT_DATA;

		if (pdata->quirks)
			host->quirks |= pdata->quirks;
		if (pdata->quirks2)
			host->quirks2 |= pdata->quirks2;
		if (pdata->host_caps)
			host->mmc->caps |= pdata->host_caps;
		if (pdata->host_caps2)
			host->mmc->caps2 |= pdata->host_caps2;
		if (pdata->pm_caps)
			host->mmc->pm_caps |= pdata->pm_caps;

		if (gpio_is_valid(pdata->ext_cd_gpio)) {
			ret = mmc_gpio_request_cd(host->mmc, pdata->ext_cd_gpio);
			if (ret) {
				dev_err(mmc_dev(host->mmc),
					"failed to allocate card detect gpio\n");
				goto err_cd_req;
			}
		}
	}

	pm_runtime_set_active(&pdev->dev);
	pm_runtime_enable(&pdev->dev);
	pm_runtime_set_autosuspend_delay(&pdev->dev, PXAV3_RPM_DELAY_MS);
	pm_runtime_use_autosuspend(&pdev->dev);
	pm_suspend_ignore_children(&pdev->dev, 1);
	pm_runtime_get_noresume(&pdev->dev);

	ret = sdhci_add_host(host);
	if (ret) {
		dev_err(&pdev->dev, "failed to add host\n");
		pm_runtime_forbid(&pdev->dev);
		pm_runtime_disable(&pdev->dev);
		goto err_add_host;
	}

	platform_set_drvdata(pdev, host);

	if (host->mmc->pm_caps & MMC_PM_KEEP_POWER) {
		device_init_wakeup(&pdev->dev, 1);
		host->mmc->pm_flags |= MMC_PM_WAKE_SDIO_IRQ;
	} else {
		device_init_wakeup(&pdev->dev, 0);
	}

	pm_runtime_put_autosuspend(&pdev->dev);

	return 0;

err_add_host:
	clk_disable_unprepare(clk);
	clk_put(clk);
err_cd_req:
err_clk_get:
	sdhci_pltfm_free(pdev);
	kfree(pxa);
	return ret;
}

static int sdhci_pxav3_remove(struct platform_device *pdev)
{
	struct sdhci_host *host = platform_get_drvdata(pdev);
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct sdhci_pxa *pxa = pltfm_host->priv;

	pm_runtime_get_sync(&pdev->dev);
	sdhci_remove_host(host, 1);
	pm_runtime_disable(&pdev->dev);

	clk_disable_unprepare(pltfm_host->clk);
	clk_put(pltfm_host->clk);

	sdhci_pltfm_free(pdev);
	kfree(pxa);

	platform_set_drvdata(pdev, NULL);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int sdhci_pxav3_suspend(struct device *dev)
{
	int ret;
	struct sdhci_host *host = dev_get_drvdata(dev);

	pm_runtime_get_sync(dev);
	ret = sdhci_suspend_host(host);
	pm_runtime_mark_last_busy(dev);
	pm_runtime_put_autosuspend(dev);

	return ret;
}

static int sdhci_pxav3_resume(struct device *dev)
{
	int ret;
	struct sdhci_host *host = dev_get_drvdata(dev);
#if defined(CONFIG_SYNO_LSP_ARMADA)
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	struct sdhci_pxa *pxa = pltfm_host->priv;
	struct device_node *np = dev->of_node;

	if (of_device_is_compatible(np, "marvell,armada-380-sdhci"))
		ret = mv_conf_mbus_windows(dev, pxa->mbus_win_regs,
					   mv_mbus_dram_info());
#endif  

	pm_runtime_get_sync(dev);
	ret = sdhci_resume_host(host);
	pm_runtime_mark_last_busy(dev);
	pm_runtime_put_autosuspend(dev);

	return ret;
}
#endif

#ifdef CONFIG_PM_RUNTIME
static int sdhci_pxav3_runtime_suspend(struct device *dev)
{
	struct sdhci_host *host = dev_get_drvdata(dev);
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	unsigned long flags;

	if (pltfm_host->clk) {
		spin_lock_irqsave(&host->lock, flags);
		host->runtime_suspended = true;
		spin_unlock_irqrestore(&host->lock, flags);

		clk_disable_unprepare(pltfm_host->clk);
	}

	return 0;
}

static int sdhci_pxav3_runtime_resume(struct device *dev)
{
	struct sdhci_host *host = dev_get_drvdata(dev);
	struct sdhci_pltfm_host *pltfm_host = sdhci_priv(host);
	unsigned long flags;

	if (pltfm_host->clk) {
		clk_prepare_enable(pltfm_host->clk);

		spin_lock_irqsave(&host->lock, flags);
		host->runtime_suspended = false;
		spin_unlock_irqrestore(&host->lock, flags);
	}

	return 0;
}
#endif

#ifdef CONFIG_PM
static const struct dev_pm_ops sdhci_pxav3_pmops = {
	SET_SYSTEM_SLEEP_PM_OPS(sdhci_pxav3_suspend, sdhci_pxav3_resume)
	SET_RUNTIME_PM_OPS(sdhci_pxav3_runtime_suspend,
		sdhci_pxav3_runtime_resume, NULL)
};

#define SDHCI_PXAV3_PMOPS (&sdhci_pxav3_pmops)

#else
#define SDHCI_PXAV3_PMOPS NULL
#endif

static struct platform_driver sdhci_pxav3_driver = {
	.driver		= {
		.name	= "sdhci-pxav3",
#ifdef CONFIG_OF
		.of_match_table = sdhci_pxav3_of_match,
#endif
		.owner	= THIS_MODULE,
		.pm	= SDHCI_PXAV3_PMOPS,
	},
	.probe		= sdhci_pxav3_probe,
	.remove		= sdhci_pxav3_remove,
};

module_platform_driver(sdhci_pxav3_driver);

MODULE_DESCRIPTION("SDHCI driver for pxav3");
MODULE_AUTHOR("Marvell International Ltd.");
MODULE_LICENSE("GPL v2");
