 
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#if defined(CONFIG_SYNO_LSP_ARMADA)
#include <linux/irqchip/chained_irq.h>
#endif  
#include <linux/io.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#if defined(CONFIG_SYNO_LSP_ARMADA)
#include <linux/of_pci.h>
#endif  
#include <linux/irqdomain.h>
#if defined(CONFIG_SYNO_LSP_ARMADA)
#include <linux/slab.h>
#include <linux/syscore_ops.h>
#include <linux/msi.h>
#endif  
#include <asm/mach/arch.h>
#include <asm/exception.h>
#include <asm/smp_plat.h>
#include <asm/mach/irq.h>

#include "irqchip.h"

#define ARMADA_370_XP_INT_SET_MASK_OFFS		(0x48)
#define ARMADA_370_XP_INT_CLEAR_MASK_OFFS	(0x4C)
#if defined(CONFIG_SYNO_LSP_ARMADA)
#define ARMADA_370_XP_INT_CPU_SUBSYS_MASK_OFFS	(0x54)
#endif  

#define ARMADA_370_XP_INT_CONTROL		(0x00)
#define ARMADA_370_XP_INT_SET_ENABLE_OFFS	(0x30)
#define ARMADA_370_XP_INT_CLEAR_ENABLE_OFFS	(0x34)
#define ARMADA_370_XP_INT_SOURCE_CTL(irq)	(0x100 + irq*4)

#define ARMADA_370_XP_CPU_INTACK_OFFS		(0x44)
#if defined(CONFIG_SYNO_LSP_ARMADA)
#define ARMADA_375_PPI_CAUSE			(0x10)
#endif  

#define ARMADA_370_XP_SW_TRIG_INT_OFFS           (0x4)
#define ARMADA_370_XP_IN_DRBEL_MSK_OFFS          (0xc)
#define ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS        (0x8)

#define ARMADA_370_XP_MAX_PER_CPU_IRQS		(28)

#define ARMADA_370_XP_TIMER0_PER_CPU_IRQ	(5)
#if defined(CONFIG_SYNO_LSP_ARMADA)
#define ARMADA_370_XP_CPU_SUBSYS_PERF_CNT	(3)

#define ARMADA_370_XP_GBE0_PER_CPU_IRQ		(8)
#define ARMADA_370_XP_GBE3_PER_CPU_IRQ		(15)
#endif  

#define IPI_DOORBELL_START                      (0)
#define IPI_DOORBELL_END                        (8)
#define IPI_DOORBELL_MASK                       0xFF
#if defined(CONFIG_SYNO_LSP_ARMADA)
#define PCI_MSI_DOORBELL_START                  (16)
#define PCI_MSI_DOORBELL_NR                     (16)
#define PCI_MSI_DOORBELL_END                    (32)
#define PCI_MSI_DOORBELL_MASK                   0xFFFF0000

#ifdef CONFIG_SMP
static DEFINE_RAW_SPINLOCK(irq_controller_lock);
#endif

static void __iomem *cpus_int_base;
#else  
static DEFINE_RAW_SPINLOCK(irq_controller_lock);
#endif  
static void __iomem *per_cpu_int_base;
static void __iomem *main_int_base;
static struct irq_domain *armada_370_xp_mpic_domain;
#if defined(CONFIG_SYNO_LSP_ARMADA)
static u32 doorbell_mask_reg;
#ifdef CONFIG_PCI_MSI
static struct irq_domain *armada_370_xp_msi_domain;
static DECLARE_BITMAP(msi_used, PCI_MSI_DOORBELL_NR);
static DEFINE_MUTEX(msi_used_lock);
static phys_addr_t msi_doorbell_addr;
#endif
#endif  

static void armada_370_xp_irq_mask(struct irq_data *d)
{
	irq_hw_number_t hwirq = irqd_to_hwirq(d);

#if defined(CONFIG_SYNO_LSP_ARMADA) && defined(CONFIG_SMP)
	int cpu;

	if (hwirq > ARMADA_370_XP_MAX_PER_CPU_IRQS)
#else
	if (hwirq != ARMADA_370_XP_TIMER0_PER_CPU_IRQ)
#endif
		writel(hwirq, main_int_base +
				ARMADA_370_XP_INT_CLEAR_ENABLE_OFFS);
#if defined(CONFIG_SYNO_LSP_ARMADA) && defined(CONFIG_SMP)
	 
	else if ((hwirq >= ARMADA_370_XP_GBE0_PER_CPU_IRQ) &&
		 (hwirq <= ARMADA_370_XP_GBE3_PER_CPU_IRQ) &&
		 (nr_cpu_ids > 1)) {
		for_each_possible_cpu(cpu) {
			if (cpumask_test_cpu(cpu, d->affinity))
				writel(hwirq, cpus_int_base + 0x100 * cpu + ARMADA_370_XP_INT_SET_MASK_OFFS);
		}
	}
#endif  
	else
		writel(hwirq, per_cpu_int_base +
				ARMADA_370_XP_INT_SET_MASK_OFFS);
}

static void armada_370_xp_irq_unmask(struct irq_data *d)
{
	irq_hw_number_t hwirq = irqd_to_hwirq(d);

#if defined(CONFIG_SYNO_LSP_ARMADA) && defined(CONFIG_SMP)
	int cpu;

	if (hwirq > ARMADA_370_XP_MAX_PER_CPU_IRQS)
#else
	if (hwirq != ARMADA_370_XP_TIMER0_PER_CPU_IRQ)
#endif
		writel(hwirq, main_int_base +
				ARMADA_370_XP_INT_SET_ENABLE_OFFS);
#if defined(CONFIG_SYNO_LSP_ARMADA) && defined(CONFIG_SMP)
	 
	else if ((hwirq >= ARMADA_370_XP_GBE0_PER_CPU_IRQ) &&
		 (hwirq <= ARMADA_370_XP_GBE3_PER_CPU_IRQ) &&
		 (nr_cpu_ids > 1)) {
		for_each_possible_cpu(cpu) {
			if (cpumask_test_cpu(cpu, d->affinity))
				writel(hwirq, cpus_int_base + 0x100 * cpu + ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
		}
	}
#endif  
	else
		writel(hwirq, per_cpu_int_base +
				ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
}

#if defined(CONFIG_SYNO_LSP_ARMADA)
#ifdef CONFIG_PCI_MSI

static int armada_370_xp_alloc_msi(void)
{
	int hwirq;

	mutex_lock(&msi_used_lock);
	hwirq = find_first_zero_bit(&msi_used, PCI_MSI_DOORBELL_NR);
	if (hwirq >= PCI_MSI_DOORBELL_NR)
		hwirq = -ENOSPC;
	else
		set_bit(hwirq, msi_used);
	mutex_unlock(&msi_used_lock);

	return hwirq;
}

static void armada_370_xp_free_msi(int hwirq)
{
	mutex_lock(&msi_used_lock);
	if (!test_bit(hwirq, msi_used))
		pr_err("trying to free unused MSI#%d\n", hwirq);
	else
		clear_bit(hwirq, msi_used);
	mutex_unlock(&msi_used_lock);
}

static int armada_370_xp_setup_msi_irq(struct msi_chip *chip,
				       struct pci_dev *pdev,
				       struct msi_desc *desc)
{
	struct msi_msg msg;
	irq_hw_number_t hwirq;
	int virq;

	hwirq = armada_370_xp_alloc_msi();
	if (hwirq < 0)
		return hwirq;

	virq = irq_create_mapping(armada_370_xp_msi_domain, hwirq);
	if (!virq) {
		armada_370_xp_free_msi(hwirq);
		return -EINVAL;
	}

	irq_set_msi_desc(virq, desc);

	msg.address_lo = msi_doorbell_addr;
	msg.address_hi = 0;
	msg.data = 0xf00 | (hwirq + 16);

	write_msi_msg(virq, &msg);
	return 0;
}

static void armada_370_xp_teardown_msi_irq(struct msi_chip *chip,
					   unsigned int irq)
{
	struct irq_data *d = irq_get_irq_data(irq);
	irq_dispose_mapping(irq);
	armada_370_xp_free_msi(d->hwirq);
}

static struct irq_chip armada_370_xp_msi_irq_chip = {
	.name = "armada_370_xp_msi_irq",
	.irq_enable = unmask_msi_irq,
	.irq_disable = mask_msi_irq,
	.irq_mask = mask_msi_irq,
	.irq_unmask = unmask_msi_irq,
};

static int armada_370_xp_msi_map(struct irq_domain *domain, unsigned int virq,
				 irq_hw_number_t hw)
{
	irq_set_chip_and_handler(virq, &armada_370_xp_msi_irq_chip,
				 handle_simple_irq);
	set_irq_flags(virq, IRQF_VALID);

	return 0;
}

static const struct irq_domain_ops armada_370_xp_msi_irq_ops = {
	.map = armada_370_xp_msi_map,
};

static int armada_370_xp_msi_init(struct device_node *node,
				  phys_addr_t main_int_phys_base)
{
	struct msi_chip *msi_chip;
	u32 reg;
	int ret;

	msi_doorbell_addr = main_int_phys_base +
		ARMADA_370_XP_SW_TRIG_INT_OFFS;

	msi_chip = kzalloc(sizeof(*msi_chip), GFP_KERNEL);
	if (!msi_chip)
		return -ENOMEM;

	msi_chip->setup_irq = armada_370_xp_setup_msi_irq;
	msi_chip->teardown_irq = armada_370_xp_teardown_msi_irq;
	msi_chip->of_node = node;

	armada_370_xp_msi_domain =
		irq_domain_add_linear(NULL, PCI_MSI_DOORBELL_NR,
				      &armada_370_xp_msi_irq_ops,
				      NULL);
	if (!armada_370_xp_msi_domain) {
		kfree(msi_chip);
		return -ENOMEM;
	}

	ret = of_pci_msi_chip_add(msi_chip);
	if (ret < 0) {
		irq_domain_remove(armada_370_xp_msi_domain);
		kfree(msi_chip);
		return ret;
	}

	reg = readl(per_cpu_int_base + ARMADA_370_XP_IN_DRBEL_MSK_OFFS)
		| PCI_MSI_DOORBELL_MASK;

	writel(reg, per_cpu_int_base +
	       ARMADA_370_XP_IN_DRBEL_MSK_OFFS);

	writel(1, per_cpu_int_base + ARMADA_370_XP_INT_CLEAR_MASK_OFFS);

	return 0;
}
#else
static inline int armada_370_xp_msi_init(struct device_node *node,
					 phys_addr_t main_int_phys_base)
{
	return 0;
}
#endif
#endif  

#ifdef CONFIG_SMP
static int armada_xp_set_affinity(struct irq_data *d,
				  const struct cpumask *mask_val, bool force)
{
	unsigned long reg;
	unsigned long new_mask = 0;
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
	unsigned long present_mask = 0;
#else
	unsigned long online_mask = 0;
#endif  
	unsigned long count = 0;
	irq_hw_number_t hwirq = irqd_to_hwirq(d);
	int cpu;

	for_each_cpu(cpu, mask_val) {
		new_mask |= 1 << cpu_logical_map(cpu);
		count++;
	}

#if defined(CONFIG_SYNO_LSP_ARMADA)
	 
	if (count > 1 && (hwirq < ARMADA_370_XP_GBE0_PER_CPU_IRQ ||
			  hwirq > ARMADA_370_XP_GBE3_PER_CPU_IRQ))
#else  
	if (count > 1)
#endif  
		return -EINVAL;
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
	for_each_cpu(cpu, cpu_present_mask)
		present_mask |= 1 << cpu_logical_map(cpu);
#else
	for_each_cpu(cpu, cpu_online_mask)
		online_mask |= 1 << cpu_logical_map(cpu);
#endif  

	raw_spin_lock(&irq_controller_lock);

	reg = readl(main_int_base + ARMADA_370_XP_INT_SOURCE_CTL(hwirq));
#if defined(CONFIG_SYNO_LSP_ARMADA_2015_T1_1p7)
	reg = (reg & (~present_mask)) | new_mask;
#else
	reg = (reg & (~online_mask)) | new_mask;
#endif  
	writel(reg, main_int_base + ARMADA_370_XP_INT_SOURCE_CTL(hwirq));

	raw_spin_unlock(&irq_controller_lock);

	return 0;
}
#endif

static struct irq_chip armada_370_xp_irq_chip = {
	.name		= "armada_370_xp_irq",
	.irq_mask       = armada_370_xp_irq_mask,
	.irq_mask_ack   = armada_370_xp_irq_mask,
	.irq_unmask     = armada_370_xp_irq_unmask,
#ifdef CONFIG_SMP
	.irq_set_affinity = armada_xp_set_affinity,
#endif
};

#if defined(CONFIG_SYNO_LSP_ARMADA)
static void armada_370_xp_enable_pmu_irq(void *data)
{
	unsigned long cpuid = smp_processor_id();
	writel(1 << cpuid, per_cpu_int_base +
				ARMADA_370_XP_INT_CPU_SUBSYS_MASK_OFFS);
}
#endif  

static int armada_370_xp_mpic_irq_map(struct irq_domain *h,
				      unsigned int virq, irq_hw_number_t hw)
{
	armada_370_xp_irq_mask(irq_get_irq_data(virq));
#if defined(CONFIG_SYNO_LSP_ARMADA) && defined(CONFIG_SMP)
	if (hw > ARMADA_370_XP_MAX_PER_CPU_IRQS)
#else
	if (hw != ARMADA_370_XP_TIMER0_PER_CPU_IRQ)
#endif
		writel(hw, per_cpu_int_base +
			ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
	else
		writel(hw, main_int_base + ARMADA_370_XP_INT_SET_ENABLE_OFFS);
	irq_set_status_flags(virq, IRQ_LEVEL);

#if defined(CONFIG_SYNO_LSP_ARMADA)
	 
	if (hw == ARMADA_370_XP_CPU_SUBSYS_PERF_CNT)
		on_each_cpu_mask(cpu_online_mask,
			armada_370_xp_enable_pmu_irq, NULL, 1);

	if ((hw == ARMADA_370_XP_TIMER0_PER_CPU_IRQ) ||
				(hw == ARMADA_370_XP_CPU_SUBSYS_PERF_CNT)) {
#else  
	if (hw == ARMADA_370_XP_TIMER0_PER_CPU_IRQ) {
#endif  
		irq_set_percpu_devid(virq);
		irq_set_chip_and_handler(virq, &armada_370_xp_irq_chip,
					handle_percpu_devid_irq);

	} else {
		irq_set_chip_and_handler(virq, &armada_370_xp_irq_chip,
					handle_level_irq);
	}
	set_irq_flags(virq, IRQF_VALID | IRQF_PROBE);

	return 0;
}

#ifdef CONFIG_SMP
void armada_mpic_send_doorbell(const struct cpumask *mask, unsigned int irq)
{
	int cpu;
	unsigned long map = 0;

	for_each_cpu(cpu, mask)
		map |= 1 << cpu_logical_map(cpu);

	dsb();

	writel((map << 8) | irq, main_int_base +
		ARMADA_370_XP_SW_TRIG_INT_OFFS);
}

void armada_xp_mpic_smp_cpu_init(void)
{
	 
	writel(0, per_cpu_int_base + ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS);

	writel(IPI_DOORBELL_MASK, per_cpu_int_base +
		ARMADA_370_XP_IN_DRBEL_MSK_OFFS);

	writel(0, per_cpu_int_base + ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
}
#endif  

static struct irq_domain_ops armada_370_xp_mpic_irq_ops = {
	.map = armada_370_xp_mpic_irq_map,
	.xlate = irq_domain_xlate_onecell,
};

#if defined(CONFIG_SYNO_LSP_ARMADA)
#ifdef CONFIG_PCI_MSI
 
static void armada_370_xp_mpic_handle_cascade_msi(void)
{
	u32 msimask, msinr;

	msimask = readl_relaxed(per_cpu_int_base +
				ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS)
		& PCI_MSI_DOORBELL_MASK;

	writel(~msimask, per_cpu_int_base +
	       ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS);

	for (msinr = PCI_MSI_DOORBELL_START;
	     msinr < PCI_MSI_DOORBELL_END; msinr++) {
		int irq;

		if (!(msimask & BIT(msinr)))
			continue;

		irq = irq_find_mapping(armada_370_xp_msi_domain,
				       msinr - 16);
		generic_handle_irq(irq);
	}
}
#else
static void armada_370_xp_mpic_handle_cascade_msi(void) { }
#endif

static void armada_370_xp_mpic_handle_cascade_irq(unsigned int irq,
						  struct irq_desc *desc)
{
	struct irq_chip *chip = irq_get_chip(irq);
	unsigned long irqmap, irqn, cpuid, irqsrc;
	unsigned int cascade_irq;
#ifdef CONFIG_SMP
	struct irq_data *irqd;
#endif

	chained_irq_enter(chip, desc);

	cpuid = raw_smp_processor_id();
	irqmap = readl_relaxed(per_cpu_int_base + ARMADA_375_PPI_CAUSE);
	for_each_set_bit(irqn, &irqmap, BITS_PER_LONG) {

		irqsrc = readl_relaxed(main_int_base + ARMADA_370_XP_INT_SOURCE_CTL(irqn));
		 
		if ((irqsrc & (0x101 << cpuid)) == 0)
			continue;

		if (irqn == 1) {
			armada_370_xp_mpic_handle_cascade_msi();
		} else {
			cascade_irq = irq_find_mapping(armada_370_xp_mpic_domain, irqn);
#ifdef CONFIG_SMP
			irqd = irq_get_irq_data(cascade_irq);
			if (cpumask_test_cpu(cpuid, irqd->affinity))
				generic_handle_irq(cascade_irq);
#else
			generic_handle_irq(cascade_irq);
#endif
		}
	}

	chained_irq_exit(chip, desc);
}
#endif  

static asmlinkage void __exception_irq_entry
armada_370_xp_handle_irq(struct pt_regs *regs)
{
	u32 irqstat, irqnr;

	do {
		irqstat = readl_relaxed(per_cpu_int_base +
					ARMADA_370_XP_CPU_INTACK_OFFS);
		irqnr = irqstat & 0x3FF;

		if (irqnr > 1022)
			break;

#if defined(CONFIG_SYNO_LSP_ARMADA)
		if (irqnr > 1) {
#else  
		if (irqnr > 0) {
#endif  
			irqnr =	irq_find_mapping(armada_370_xp_mpic_domain,
					irqnr);
			handle_IRQ(irqnr, regs);
			continue;
		}

#if defined(CONFIG_SYNO_LSP_ARMADA) && defined(CONFIG_PCI_MSI)
		 
		if (irqnr == 1) {
			u32 msimask, msinr;

			msimask = readl_relaxed(per_cpu_int_base +
						ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS)
				& PCI_MSI_DOORBELL_MASK;

			writel(~msimask, per_cpu_int_base +
			       ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS);

			for (msinr = PCI_MSI_DOORBELL_START;
			     msinr < PCI_MSI_DOORBELL_END; msinr++) {
				int irq;

				if (!(msimask & BIT(msinr)))
					continue;

				irq = irq_find_mapping(armada_370_xp_msi_domain,
						       msinr - 16);
				handle_IRQ(irq, regs);
			}
		}
#endif  

#ifdef CONFIG_SMP
		 
		if (irqnr == 0) {
			u32 ipimask, ipinr;

			ipimask = readl_relaxed(per_cpu_int_base +
						ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS)
				& IPI_DOORBELL_MASK;

			writel(~ipimask, per_cpu_int_base +
				ARMADA_370_XP_IN_DRBEL_CAUSE_OFFS);

			for (ipinr = IPI_DOORBELL_START;
			     ipinr < IPI_DOORBELL_END; ipinr++) {
				if (ipimask & (0x1 << ipinr))
					handle_IPI(ipinr, regs);
			}
			continue;
		}
#endif

	} while (1);
}

#if defined(CONFIG_SYNO_LSP_ARMADA)
static int armada_370_xp_mpic_suspend(void)
{
	doorbell_mask_reg = readl(per_cpu_int_base +
				  ARMADA_370_XP_IN_DRBEL_MSK_OFFS);
	return 0;
}

static void armada_370_xp_mpic_resume(void)
{
	int nirqs;
	irq_hw_number_t irq;

	nirqs = (readl(main_int_base + ARMADA_370_XP_INT_CONTROL) >> 2) & 0x3ff;
	for (irq = 0; irq < nirqs; irq++) {
		struct irq_data *data;
		int virq;

		virq = irq_linear_revmap(armada_370_xp_mpic_domain, irq);
		if (virq == 0)
			continue;

		if (irq != ARMADA_370_XP_TIMER0_PER_CPU_IRQ)
			writel(irq, per_cpu_int_base +
			       ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
		else
			writel(irq, main_int_base +
			       ARMADA_370_XP_INT_SET_ENABLE_OFFS);

		data = irq_get_irq_data(virq);
		if (!irqd_irq_disabled(data))
			armada_370_xp_irq_unmask(data);
	}

	writel(doorbell_mask_reg,
	       per_cpu_int_base + ARMADA_370_XP_IN_DRBEL_MSK_OFFS);
	if (doorbell_mask_reg & IPI_DOORBELL_MASK)
		writel(0, per_cpu_int_base + ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
	if (doorbell_mask_reg & PCI_MSI_DOORBELL_MASK)
		writel(1, per_cpu_int_base + ARMADA_370_XP_INT_CLEAR_MASK_OFFS);
}

struct syscore_ops armada_370_xp_mpic_syscore_ops = {
	.suspend	= armada_370_xp_mpic_suspend,
	.resume		= armada_370_xp_mpic_resume,
};
#endif  

static int __init armada_370_xp_mpic_of_init(struct device_node *node,
					     struct device_node *parent)
{
#if defined(CONFIG_SYNO_LSP_ARMADA)
	struct resource main_int_res, per_cpu_int_res, cpus_int_res;
	int parent_irq;
	u32 control;

	BUG_ON(of_address_to_resource(node, 0, &main_int_res));
	BUG_ON(of_address_to_resource(node, 1, &per_cpu_int_res));

	BUG_ON(!request_mem_region(main_int_res.start,
				   resource_size(&main_int_res),
				   node->full_name));
	BUG_ON(!request_mem_region(per_cpu_int_res.start,
				   resource_size(&per_cpu_int_res),
				   node->full_name));

	main_int_base = ioremap(main_int_res.start,
				resource_size(&main_int_res));
	BUG_ON(!main_int_base);

	per_cpu_int_base = ioremap(per_cpu_int_res.start,
				   resource_size(&per_cpu_int_res));
	BUG_ON(!per_cpu_int_base);

	if (nr_cpu_ids > 1) {
		BUG_ON(of_address_to_resource(node, 2, &cpus_int_res));
		BUG_ON(!request_mem_region(cpus_int_res.start,
					   resource_size(&cpus_int_res),
					   node->full_name));
		cpus_int_base = ioremap(cpus_int_res.start,
					resource_size(&cpus_int_res));
		BUG_ON(!cpus_int_base);
	}
#else  
	u32 control;

	main_int_base = of_iomap(node, 0);
	per_cpu_int_base = of_iomap(node, 1);

	BUG_ON(!main_int_base);
	BUG_ON(!per_cpu_int_base);
#endif  

	control = readl(main_int_base + ARMADA_370_XP_INT_CONTROL);

	armada_370_xp_mpic_domain =
		irq_domain_add_linear(node, (control >> 2) & 0x3ff,
				&armada_370_xp_mpic_irq_ops, NULL);

#if defined(CONFIG_SYNO_LSP_ARMADA)
	BUG_ON(!armada_370_xp_mpic_domain);
#else  
	if (!armada_370_xp_mpic_domain)
		panic("Unable to add Armada_370_Xp MPIC irq domain (DT)\n");

	irq_set_default_host(armada_370_xp_mpic_domain);
#endif  

#ifdef CONFIG_SMP
	armada_xp_mpic_smp_cpu_init();

	cpumask_clear(irq_default_affinity);
	cpumask_set_cpu(smp_processor_id(), irq_default_affinity);

#endif

#if defined(CONFIG_SYNO_LSP_ARMADA)
	armada_370_xp_msi_init(node, main_int_res.start);

	parent_irq = irq_of_parse_and_map(node, 0);
	if (parent_irq <= 0) {
		irq_set_default_host(armada_370_xp_mpic_domain);
		set_handle_irq(armada_370_xp_handle_irq);
	} else {
		irq_set_chained_handler(parent_irq,
					armada_370_xp_mpic_handle_cascade_irq);
	}

	register_syscore_ops(&armada_370_xp_mpic_syscore_ops);
#else  
	set_handle_irq(armada_370_xp_handle_irq);
#endif  

	return 0;
}

IRQCHIP_DECLARE(armada_370_xp_mpic, "marvell,mpic", armada_370_xp_mpic_of_init);
