/*******************************************************************************
Copyright (C) 2013 Annapurna Labs Ltd.

This file may be licensed under the terms of the Annapurna Labs Commercial
License Agreement.

Alternatively, this file can be distributed under the terms of the GNU General
Public License V2 or V3 as published by the Free Software Foundation and can be
found at http://www.gnu.org/licenses/gpl-2.0.html

Alternatively, redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following conditions are
met:

    *     Redistributions of source code must retain the above copyright notice,
	  this list of conditions and the following disclaimer.

    *     Redistributions in binary form must reproduce the above copyright
	  notice, this list of conditions and the following disclaimer in
	  the documentation and/or other materials provided with the
	  distribution.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

/**
 *  @{
 * @file   al_hal_eth_ec_regs.h
 *
 * @brief Ethernet controller registers
 *
 */

#ifndef __AL_HAL_EC_REG_H
#define __AL_HAL_EC_REG_H

#ifdef __cplusplus
extern "C" {
#endif
/*
* Unit Registers
*/

struct al_ec_gen {
	/* [0x0] Ethernet controller Version */
	uint32_t version;
	/* [0x4] Enable modules operation. */
	uint32_t en;
	/* [0x8] Enable FIFO operation on the EC side. */
	uint32_t fifo_en;
	/* [0xc] General L2 configuration for the Ethernet controlle ... */
	uint32_t l2;
	/* [0x10] Configure protocol index values */
	uint32_t cfg_i;
	/* [0x14] Configure protocol index values (extended protocols ... */
	uint32_t cfg_i_ext;
	/* [0x18] Enable modules operation (extended operations). */
	uint32_t en_ext;
	uint32_t rsrvd[9];
};
struct al_ec_mac {
	/* [0x0] General configuration of the MAC side of the Ethern ... */
	uint32_t gen;
	/* [0x4] Minimum packet size  */
	uint32_t min_pkt;
	/* [0x8] Maximum packet size  */
	uint32_t max_pkt;
	uint32_t rsrvd[13];
};
struct al_ec_rxf {
	/* [0x0] Rx FIFO input controller configuration 1 */
	uint32_t cfg_1;
	/* [0x4] Rx FIFO input controller configuration 2 */
	uint32_t cfg_2;
	/* [0x8] Threshold to start reading packet from the Rx FIFO */
	uint32_t rd_fifo;
	/* [0xc] Threshold to stop writing packet to the Rx FIFO */
	uint32_t wr_fifo;
	/* [0x10] Threshold to stop writing packet to the loopback FI ... */
	uint32_t lb_fifo;
	/* [0x14] Rx FIFO input controller loopback FIFO configuratio ... */
	uint32_t cfg_lb;
	/* [0x18] Configuration for dropping packet at the FIFO outpu ... */
	uint32_t out_drop;
	uint32_t rsrvd[25];
};
struct al_ec_epe {
	/* [0x0] Ethernet parsing engine configuration 1 */
	uint32_t parse_cfg;
	/* [0x4] Protocol index action table address */
	uint32_t act_table_addr;
	/* [0x8] Protocol index action table data */
	uint32_t act_table_data_1;
	/* [0xc] Protocol index action table data */
	uint32_t act_table_data_2;
	/* [0x10] Protocol index action table data */
	uint32_t act_table_data_3;
	/* [0x14] Protocol index action table data */
	uint32_t act_table_data_4;
	/* [0x18] Protocol index action table data */
	uint32_t act_table_data_5;
	/* [0x1c] Protocol index action table data */
	uint32_t act_table_data_6;
	/* [0x20] Input result vector, default values for parser inpu ... */
	uint32_t res_def;
	/* [0x24] Result input vector selection */
	uint32_t res_in;
	uint32_t rsrvd[6];
};
struct al_ec_epe_res {
	/* [0x0] Parser result vector pointer */
	uint32_t p1;
	/* [0x4] Parser result vector pointer */
	uint32_t p2;
	/* [0x8] Parser result vector pointer */
	uint32_t p3;
	/* [0xc] Parser result vector pointer */
	uint32_t p4;
	/* [0x10] Parser result vector pointer */
	uint32_t p5;
	/* [0x14] Parser result vector pointer */
	uint32_t p6;
	/* [0x18] Parser result vector pointer */
	uint32_t p7;
	/* [0x1c] Parser result vector pointer */
	uint32_t p8;
	/* [0x20] Parser result vector pointer */
	uint32_t p9;
	/* [0x24] Parser result vector pointer */
	uint32_t p10;
	/* [0x28] Parser result vector pointer */
	uint32_t p11;
	/* [0x2c] Parser result vector pointer */
	uint32_t p12;
	/* [0x30] Parser result vector pointer */
	uint32_t p13;
	/* [0x34] Parser result vector pointer */
	uint32_t p14;
	/* [0x38] Parser result vector pointer */
	uint32_t p15;
	/* [0x3c] Parser result vector pointer */
	uint32_t p16;
	/* [0x40] Parser result vector pointer */
	uint32_t p17;
	/* [0x44] Parser result vector pointer */
	uint32_t p18;
	/* [0x48] Parser result vector pointer */
	uint32_t p19;
	/* [0x4c] Parser result vector pointer */
	uint32_t p20;
	uint32_t rsrvd[12];
};
struct al_ec_epe_h {
	/* [0x0] Header length, support for header length table for  ... */
	uint32_t hdr_len;
};
struct al_ec_epe_p {
	/* [0x0] Data  for comparison */
	uint32_t comp_data;
	/* [0x4] Mask for comparison */
	uint32_t comp_mask;
	/* [0x8] Compare control */
	uint32_t comp_ctrl;
	uint32_t rsrvd[4];
};
struct al_ec_epe_a {
	/* [0x0] Protocol index action register */
	uint32_t prot_act;
};
struct al_ec_rfw {
	/* [0x0] Tuple (4/2) Hash configuration */
	uint32_t thash_cfg_1;
	/* [0x4] Tuple (4/2) Hash configuration */
	uint32_t thash_cfg_2;
	/* [0x8] MAC Hash configuration */
	uint32_t mhash_cfg_1;
	/* [0xc] MAC Hash configuration */
	uint32_t mhash_cfg_2;
	/* [0x10] MAC Hash configuration */
	uint32_t hdr_split;
	/* [0x14] Masking the errors described in  register rxf_drop  ... */
	uint32_t meta_err;
	/* [0x18] Configuration for generating the MetaData for the R ... */
	uint32_t meta;
	/* [0x1c] Configuration for generating the MetaData for the R ... */
	uint32_t filter;
	/* [0x20] 4 tupple hash table address */
	uint32_t thash_table_addr;
	/* [0x24] 4 tupple hash table data */
	uint32_t thash_table_data;
	/* [0x28] MAC hash table address */
	uint32_t mhash_table_addr;
	/* [0x2c] MAC hash table data */
	uint32_t mhash_table_data;
	/* [0x30] VLAN table address */
	uint32_t vid_table_addr;
	/* [0x34] VLAN table data */
	uint32_t vid_table_data;
	/* [0x38] VLAN p-bits table address */
	uint32_t pbits_table_addr;
	/* [0x3c] VLAN p-bits table data */
	uint32_t pbits_table_data;
	/* [0x40] DSCP table address */
	uint32_t dscp_table_addr;
	/* [0x44] DSCP table data */
	uint32_t dscp_table_data;
	/* [0x48] TC table address */
	uint32_t tc_table_addr;
	/* [0x4c] TC table data */
	uint32_t tc_table_data;
	/* [0x50] Control table address */
	uint32_t ctrl_table_addr;
	/* [0x54] Control table data */
	uint32_t ctrl_table_data;
	/* [0x58] Forwarding output configuration */
	uint32_t out_cfg;
	/* [0x5c] Flow steering mechanism, 
Table address */
	uint32_t fsm_table_addr;
	/* [0x60] Flow steering mechanism,  
Table data */
	uint32_t fsm_table_data;
	/* [0x64] Selection of data to be used in packet forwarding0  ... */
	uint32_t ctrl_sel;
	/* [0x68] Default VLAN data, used for untagged packets */
	uint32_t default_vlan;
	/* [0x6c] Default HASH output values */
	uint32_t default_hash;
	/* [0x70] Default override values, in case a packet was filte ... */
	uint32_t default_or;
	/* [0x74] Latched information when a drop condition occurred */
	uint32_t drop_latch;
	/* [0x78] Check sum calculation configuration */
	uint32_t checksum;
	/* [0x7c] LRO offload engine configuration register */
	uint32_t lro_cfg_1;
	/* [0x80] LRO offload engine Check rules configurations for I ... */
	uint32_t lro_check_ipv4;
	/* [0x84] LRO offload engine IPv4 values configuration */
	uint32_t lro_ipv4;
	/* [0x88] LRO offload engine Check rules configurations for I ... */
	uint32_t lro_check_ipv6;
	/* [0x8c] LRO offload engine IPv6 values configuration */
	uint32_t lro_ipv6;
	/* [0x90] LRO offload engine Check rules configurations for T ... */
	uint32_t lro_check_tcp;
	/* [0x94] LRO offload engine IPv6 values configuration */
	uint32_t lro_tcp;
	/* [0x98] LRO offload engine Check rules configurations for U ... */
	uint32_t lro_check_udp;
	/* [0x9c] LRO offload engine Check rules configurations for U ... */
	uint32_t lro_check_l2;
	/* [0xa0] LRO offload engine Check rules configurations for U ... */
	uint32_t lro_check_gen;
	/* [0xa4] Rules for storing packet information into the cache ... */
	uint32_t lro_store;
	/* [0xa8] VLAN table default */
	uint32_t vid_table_def;
	/* [0xac] Control table default */
	uint32_t ctrl_table_def;
	/* [0xb0] Additional configuration 0 */
	uint32_t cfg_a_0;
	/* [0xb4] Tuple (4/2) Hash configuration (extended for RoCE a ... */
	uint32_t thash_cfg_3;
	/* [0xb8] Tuple (4/2) Hash configuration , mask for the input ... */
	uint32_t thash_mask_outer_ipv6;
	/* [0xbc] Tuple (4/2) Hash configuration , mask for the input ... */
	uint32_t thash_mask_outer;
	/* [0xc0] Tuple (4/2) Hash configuration , mask for the input ... */
	uint32_t thash_mask_inner_ipv6;
	/* [0xc4] Tuple (4/2) Hash configuration , mask for the input ... */
	uint32_t thash_mask_inner;
	uint32_t rsrvd[10];
};
struct al_ec_rfw_udma {
	/* [0x0] Per UDMA default configuration */
	uint32_t def_cfg;
};
struct al_ec_rfw_hash {
	/* [0x0] key configuration (320 bits */
	uint32_t key;
};
struct al_ec_rfw_priority {
	/* [0x0] Priority to queue mapping configuration */
	uint32_t queue;
};
struct al_ec_rfw_default {
	/* [0x0] Default forwarding configuration options */
	uint32_t opt_1;
};
struct al_ec_fwd_mac {
	/* [0x0] MAC address data [31:0] */
	uint32_t data_l;
	/* [0x4] MAC address data [15:0] */
	uint32_t data_h;
	/* [0x8] MAC address mask [31:0] */
	uint32_t mask_l;
	/* [0xc] MAC address mask [15:0] */
	uint32_t mask_h;
	/* [0x10] MAC compare control */
	uint32_t ctrl;
};
struct al_ec_msw {
	/* [0x0] Configuration for unicast packets */
	uint32_t uc;
	/* [0x4] Configuration for multicast packets */
	uint32_t mc;
	/* [0x8] Configuration for broadcast packets */
	uint32_t bc;
	uint32_t rsrvd[3];
};
struct al_ec_tso {
	/* [0x0] Input configuration */
	uint32_t in_cfg;
	/* [0x4] MetaData default cache table address */
	uint32_t cache_table_addr;
	/* [0x8] MetaData default cache table data */
	uint32_t cache_table_data_1;
	/* [0xc] MetaData default cache table data */
	uint32_t cache_table_data_2;
	/* [0x10] MetaData default cache table data */
	uint32_t cache_table_data_3;
	/* [0x14] MetaData default cache table data */
	uint32_t cache_table_data_4;
	/* [0x18] TCP control bit operation for first segment */
	uint32_t ctrl_first;
	/* [0x1c] TCP control bit operation for middle segments  */
	uint32_t ctrl_middle;
	/* [0x20] TCP control bit operation for last segment */
	uint32_t ctrl_last;
	/* [0x24] Additional TSO configurations */
	uint32_t cfg_add_0;
	/* [0x28] TSO configuration for tunneled packets */
	uint32_t cfg_tunnel;
	uint32_t rsrvd[13];
};
struct al_ec_tso_sel {
	/* [0x0] MSS value */
	uint32_t mss;
};
struct al_ec_tpe {
	/* [0x0] Parsing configuration */
	uint32_t parse;
	uint32_t rsrvd[15];
};
struct al_ec_tpm_udma {
	/* [0x0] Default VLAN data */
	uint32_t vlan_data;
	/* [0x4] UDMA MAC SA information for spoofing */
	uint32_t mac_sa_1;
	/* [0x8] UDMA MAC SA information for spoofing */
	uint32_t mac_sa_2;
};
struct al_ec_tpm_sel {
	/* [0x0] Ethertype values for VLAN modification */
	uint32_t etype;
};
struct al_ec_tfw {
	/* [0x0] Tx FIFO Wr configuration */
	uint32_t tx_wr_fifo;
	/* [0x4] VLAN table address */
	uint32_t tx_vid_table_addr;
	/* [0x8] VLAN table data */
	uint32_t tx_vid_table_data;
	/* [0xc] Tx FIFO Rd configuration */
	uint32_t tx_rd_fifo;
	/* [0x10] Tx FIFO rRd configuration, checksum insertion */
	uint32_t tx_checksum;
	/* [0x14] Tx forwarding general configuration register */
	uint32_t tx_gen;
	/* [0x18] Tx spoofing configuration */
	uint32_t tx_spf;
	/* [0x1c] TX data FIFO status */
	uint32_t data_fifo;
	/* [0x20] Tx control FIFO status */
	uint32_t ctrl_fifo;
	/* [0x24] Tx header FIFO status */
	uint32_t hdr_fifo;
	uint32_t rsrvd[14];
};
struct al_ec_tfw_udma {
	/* [0x0] Default GMDA output bitmap for unicast packet */
	uint32_t uc_udma;
	/* [0x4] Default GMDA output bitmap for multicast packet */
	uint32_t mc_udma;
	/* [0x8] Default GMDA output bitmap for broadcast packet */
	uint32_t bc_udma;
	/* [0xc] Tx spoofing configuration */
	uint32_t spf_cmd;
	/* [0x10] Forwarding decision control */
	uint32_t fwd_dec;
	uint32_t rsrvd;
};
struct al_ec_tmi {
	/* [0x0] Forward packets back to the Rx data path for local  ... */
	uint32_t tx_cfg;
	uint32_t rsrvd[3];
};
struct al_ec_efc {
	/* [0x0] Mask of pause_on  [7:0] for the Ethernet controller ... */
	uint32_t ec_pause;
	/* [0x4] Mask of Ethernet controller "almost full" indicatio ... */
	uint32_t ec_xoff;
	/* [0x8] Mask for generating XON indication pulse */
	uint32_t xon;
	/* [0xc] Mask for generating GPIO output XOFF indication fro ... */
	uint32_t gpio;
	/* [0x10] Rx FIFO threshold for generating the almost full in ... */
	uint32_t rx_fifo_af;
	/* [0x14] Rx FIFO threshold for generating the almost full in ... */
	uint32_t rx_fifo_hyst;
	/* [0x18] Rx FIFO threshold for generating the almost full in ... */
	uint32_t stat;
	/* [0x1c] XOFF timer for the 1G MACSets the interval (in SB_C ... */
	uint32_t xoff_timer_1g;
	/* [0x20] PFC force flow control generation */
	uint32_t ec_pfc;
	uint32_t rsrvd[3];
};
struct al_ec_fc_udma {
	/* [0x0] Mask of "pause_on"  [0] for all queues */
	uint32_t q_pause_0;
	/* [0x4] Mask of "pause_on"  [1] for all queues */
	uint32_t q_pause_1;
	/* [0x8] Mask of "pause_on"  [2] for all queues */
	uint32_t q_pause_2;
	/* [0xc] Mask of "pause_on"  [3] for all queues */
	uint32_t q_pause_3;
	/* [0x10] Mask of "pause_on"  [4] for all queues */
	uint32_t q_pause_4;
	/* [0x14] Mask of "pause_on"  [5] for all queues */
	uint32_t q_pause_5;
	/* [0x18] Mask of "pause_on"  [6] for all queues */
	uint32_t q_pause_6;
	/* [0x1c] Mask of "pause_on"  [7] for all queues */
	uint32_t q_pause_7;
	/* [0x20] Mask of external GPIO input pause [0] for all queue ... */
	uint32_t q_gpio_0;
	/* [0x24] Mask of external GPIO input pause [1] for all queue ... */
	uint32_t q_gpio_1;
	/* [0x28] Mask of external GPIO input pause [2] for all queue ... */
	uint32_t q_gpio_2;
	/* [0x2c] Mask of external GPIO input pause [3] for all queue ... */
	uint32_t q_gpio_3;
	/* [0x30] Mask of external GPIO input [4] for all queues */
	uint32_t q_gpio_4;
	/* [0x34] Mask of external GPIO input [5] for all queues */
	uint32_t q_gpio_5;
	/* [0x38] Mask of external GPIO input [6] for all queues */
	uint32_t q_gpio_6;
	/* [0x3c] Mask of external GPIO input [7] for all queues */
	uint32_t q_gpio_7;
	/* [0x40] Mask of "pause_on"  [7:0] for the UDMA stream inter ... */
	uint32_t s_pause;
	/* [0x44] Mask of Rx “almost full” indication for generating  ... */
	uint32_t q_xoff_0;
	/* [0x48] Mask of RXx "almost full" indication for generating ... */
	uint32_t q_xoff_1;
	/* [0x4c] Mask of Rx "almost full" indication for generating  ... */
	uint32_t q_xoff_2;
	/* [0x50] Mask of Rx "almost full" indication for generating  ... */
	uint32_t q_xoff_3;
	/* [0x54] Mask of Rx "almost full" indication for generating  ... */
	uint32_t q_xoff_4;
	/* [0x58] Mask of Rx "almost full" indication for generating  ... */
	uint32_t q_xoff_5;
	/* [0x5c] Mask of Rx "almost full" indication for generating  ... */
	uint32_t q_xoff_6;
	/* [0x60] Mask of Rx almost full" indication for generating X ... */
	uint32_t q_xoff_7;
	uint32_t rsrvd[7];
};
struct al_ec_tpg_rpa_res {
	/* [0x0] NOT used */
	uint32_t not_used;
	uint32_t rsrvd[63];
};
struct al_ec_eee {
	/* [0x0] EEE configuration */
	uint32_t cfg_e;
	/* [0x4] Number of clocks to get into EEE mode. */
	uint32_t pre_cnt;
	/* [0x8] Number of clocks to stop MAC EEE mode after getting ... */
	uint32_t post_cnt;
	/* [0xc] Number of clocks to stop the TX MAC interface after ... */
	uint32_t stop_cnt;
	/* [0x10] EEE status */
	uint32_t stat_eee;
	uint32_t rsrvd[59];
};
struct al_ec_stat {
	/* [0x0] Rx Frequency adjust FIFO input  packets */
	uint32_t faf_in_rx_pkt;
	/* [0x4] Rx Frequency adjust FIFO input  short error packets ... */
	uint32_t faf_in_rx_short;
	/* [0x8] Rx Frequency adjust FIFO input  long error packets */
	uint32_t faf_in_rx_long;
	/* [0xc] Rx Frequency adjust FIFO output  packets */
	uint32_t faf_out_rx_pkt;
	/* [0x10] Rx Frequency adjust FIFO output  short error packet ... */
	uint32_t faf_out_rx_short;
	/* [0x14] Rx Frequency adjust FIFO output  long error packets ... */
	uint32_t faf_out_rx_long;
	/* [0x18] Rx Frequency adjust FIFO output  drop packets */
	uint32_t faf_out_drop;
	/* [0x1c] Number of packets written into the Rx FIFO (without ... */
	uint32_t rxf_in_rx_pkt;
	/* [0x20] Number of error packets written into the Rx FIFO (w ... */
	uint32_t rxf_in_fifo_err;
	/* [0x24] Number of packets written into the loopback  FIFO ( ... */
	uint32_t lbf_in_rx_pkt;
	/* [0x28] Number of error packets written into the loopback F ... */
	uint32_t lbf_in_fifo_err;
	/* [0x2c] Number of packets read from Rx FIFO 1 */
	uint32_t rxf_out_rx_1_pkt;
	/* [0x30] Number of packets read from Rx FIFO 2 (loopback FIF ... */
	uint32_t rxf_out_rx_2_pkt;
	/* [0x34] Rx FIFO output drop packets from FIFO 1 */
	uint32_t rxf_out_drop_1_pkt;
	/* [0x38] Rx FIFO output drop packets from FIFO 2 (loop back) ... */
	uint32_t rxf_out_drop_2_pkt;
	/* [0x3c] Rx Parser 1, input packet counter */
	uint32_t rpe_1_in_rx_pkt;
	/* [0x40] Rx Parser 1, output packet counter */
	uint32_t rpe_1_out_rx_pkt;
	/* [0x44] Rx Parser 2, input packet counter */
	uint32_t rpe_2_in_rx_pkt;
	/* [0x48] Rx Parser 2, output packet counter */
	uint32_t rpe_2_out_rx_pkt;
	/* [0x4c] Rx Parser 3 (MACsec), input packet counter */
	uint32_t rpe_3_in_rx_pkt;
	/* [0x50] Rx Parser 3 (MACsec), output packet counter */
	uint32_t rpe_3_out_rx_pkt;
	/* [0x54] Tx parser, input packet counter */
	uint32_t tpe_in_tx_pkt;
	/* [0x58] Tx parser, output packet counter */
	uint32_t tpe_out_tx_pkt;
	/* [0x5c] Tx packet modification, input packet counter */
	uint32_t tpm_tx_pkt;
	/* [0x60] Tx forwarding input packet counter */
	uint32_t tfw_in_tx_pkt;
	/* [0x64] Tx forwarding input packet counter */
	uint32_t tfw_out_tx_pkt;
	/* [0x68] Rx forwarding input packet counter */
	uint32_t rfw_in_rx_pkt;
	/* [0x6c] Rx Forwarding, packet with VLAN command drop indica ... */
	uint32_t rfw_in_vlan_drop;
	/* [0x70] Rx Forwarding, packets with parse drop indication */
	uint32_t rfw_in_parse_drop;
	/* [0x74] Rx Forwarding, multicast packets */
	uint32_t rfw_in_mc;
	/* [0x78] Rx Forwarding, broadcast packets */
	uint32_t rfw_in_bc;
	/* [0x7c] Rx Forwarding, tagged packets */
	uint32_t rfw_in_vlan_exist;
	/* [0x80] Rx Forwarding, untagged packets */
	uint32_t rfw_in_vlan_nexist;
	/* [0x84] Rx Forwarding, packets with MAC address drop indica ... */
	uint32_t rfw_in_mac_drop;
	/* [0x88] Rx Forwarding, packets with undetected MAC address */
	uint32_t rfw_in_mac_ndet_drop;
	/* [0x8c] Rx Forwarding, packets with drop indication from th ... */
	uint32_t rfw_in_ctrl_drop;
	/* [0x90] Rx Forwarding, packets with L3_protocol_index drop  ... */
	uint32_t rfw_in_prot_i_drop;
	/* [0x94] EEE, number of times the system went into EEE state ... */
	uint32_t eee_in;
	uint32_t rsrvd[90];
};
struct al_ec_stat_udma {
	/* [0x0] Rx forwarding output packet counter */
	uint32_t rfw_out_rx_pkt;
	/* [0x4] Rx forwarding output drop packet counter */
	uint32_t rfw_out_drop;
	/* [0x8] Multi-stream write, number of Rx packets */
	uint32_t msw_in_rx_pkt;
	/* [0xc] Multi-stream write, number of dropped packets at SO ... */
	uint32_t msw_drop_q_full;
	/* [0x10] Multi-stream write, number of dropped packets at SO ... */
	uint32_t msw_drop_sop;
	/* [0x14] Multi-stream write, number of dropped packets at EO ... */
	uint32_t msw_drop_eop;
	/* [0x18] Multi-stream write, number of packets written to th ... */
	uint32_t msw_wr_eop;
	/* [0x1c] Multi-stream write, number of packets read from the ... */
	uint32_t msw_out_rx_pkt;
	/* [0x20] Number of transmitted packets without TSO enabled */
	uint32_t tso_no_tso_pkt;
	/* [0x24] Number of transmitted packets with TSO enabled */
	uint32_t tso_tso_pkt;
	/* [0x28] Number of TSO segments that were generated */
	uint32_t tso_seg_pkt;
	/* [0x2c] Number of TSO segments that required padding */
	uint32_t tso_pad_pkt;
	/* [0x30] Tx Packet modification, MAC SA spoof error  */
	uint32_t tpm_tx_spoof;
	/* [0x34] Tx MAC interface, input packet counter */
	uint32_t tmi_in_tx_pkt;
	/* [0x38] Tx MAC interface, number of packets forwarded to th ... */
	uint32_t tmi_out_to_mac;
	/* [0x3c] Tx MAC interface, number of packets forwarded to th ... */
	uint32_t tmi_out_to_rx;
	/* [0x40] Tx MAC interface, number of transmitted bytes */
	uint32_t tx_q0_bytes;
	/* [0x44] Tx MAC interface, number of transmitted bytes */
	uint32_t tx_q1_bytes;
	/* [0x48] Tx MAC interface, number of transmitted bytes */
	uint32_t tx_q2_bytes;
	/* [0x4c] Tx MAC interface, number of transmitted bytes */
	uint32_t tx_q3_bytes;
	/* [0x50] Tx MAC interface, number of transmitted packets */
	uint32_t tx_q0_pkts;
	/* [0x54] Tx MAC interface, number of transmitted packets */
	uint32_t tx_q1_pkts;
	/* [0x58] Tx MAC interface, number of transmitted packets */
	uint32_t tx_q2_pkts;
	/* [0x5c] Tx MAC interface, number of transmitted packets */
	uint32_t tx_q3_pkts;
	uint32_t rsrvd[40];
};
struct al_ec_msp {
	/* [0x0] Ethernet parsing engine configuration 1 */
	uint32_t p_parse_cfg;
	/* [0x4] Protocol index action table address */
	uint32_t p_act_table_addr;
	/* [0x8] Protocol index action table data */
	uint32_t p_act_table_data_1;
	/* [0xc] Protocol index action table data */
	uint32_t p_act_table_data_2;
	/* [0x10] Protocol index action table data */
	uint32_t p_act_table_data_3;
	/* [0x14] Protocol index action table data */
	uint32_t p_act_table_data_4;
	/* [0x18] Protocol index action table data */
	uint32_t p_act_table_data_5;
	/* [0x1c] Protocol index action table data */
	uint32_t p_act_table_data_6;
	/* [0x20] Input result vector, default values for parser inpu ... */
	uint32_t p_res_def;
	/* [0x24] Result input vector selection */
	uint32_t p_res_in;
	uint32_t rsrvd[6];
};
struct al_ec_msp_p {
	/* [0x0] Header length, support for header length table for  ... */
	uint32_t h_hdr_len;
};
struct al_ec_msp_c {
	/* [0x0] Data  for comparison */
	uint32_t p_comp_data;
	/* [0x4] Mask for comparison */
	uint32_t p_comp_mask;
	/* [0x8] Compare control */
	uint32_t p_comp_ctrl;
	uint32_t rsrvd[4];
};
struct al_ec_crce {
	/* [0x0] RoCE CRC init value */
	uint32_t roce_crc_init;
	/* [0x4] FCoE CRC init value */
	uint32_t fcoe_crc_init;
	/* [0x8] Packet header / trailer size - bytes not included i ... */
	uint32_t pkt_cfg;
	/* [0xc] Bit Byte Swap enable */
	uint32_t swap_en;
	/* [0x10] RoCE mask data word 0 */
	uint32_t roce_word_0;
	/* [0x14] RoCE mask data word 1 */
	uint32_t roce_word_1;
	/* [0x18] RoCE mask data word 2 */
	uint32_t roce_word_2;
	/* [0x1c] RoCE mask data word 3 */
	uint32_t roce_word_3;
	/* [0x20] General CRC engine configuration */
	uint32_t gen_cfg;
	uint32_t rsrvd[7];
};
struct al_ec_wol {
	/* [0x0] WoL enable configuration,Packet forwarding and inte ... */
	uint32_t wol_en;
	/* [0x4] Password for magic_password packet detection - bits ... */
	uint32_t magic_pswd_l;
	/* [0x8] Password for magic+password packet detection -  47: ... */
	uint32_t magic_pswd_h;
	/* [0xc] Configured L3 Destination IP address for WoL Ipv6 p ... */
	uint32_t ipv6_dip_word0;
	/* [0x10] Configured L3 Destination IP address for WoL Ipv6 p ... */
	uint32_t ipv6_dip_word1;
	/* [0x14] Configured L3 Destination IP address for WoL Ipv6 p ... */
	uint32_t ipv6_dip_word2;
	/* [0x18] Configured L3 Destination IP address for WoL Ipv6 p ... */
	uint32_t ipv6_dip_word3;
	/* [0x1c] Configured L3 Destination IP address for WoL Ipv4 p ... */
	uint32_t ipv4_dip;
	/* [0x20] Configured EtherType for WoL EtherType_da/EtherType ... */
	uint32_t ethertype;
	uint32_t rsrvd[7];
};
struct al_ec_pth {
	/* [0x0] system time counter (Time of Day) */
	uint32_t system_time_seconds;
	/* [0x4] system time subseconds in a seconds (resolution wit ... */
	uint32_t system_time_subseconds_msb;
	/* [0x8] system time subseconds in a seconds (resolution wit ... */
	uint32_t system_time_subseconds_lsb;
	/* [0xc] clock period in femtoseconds */
	uint32_t clock_period_msb;
	/* [0x10] clock period in femtoseconds */
	uint32_t clock_period_lsb;
	/* [0x14] control for internally updating the system time reg ... */
	uint32_t int_update_ctrl;
	/* [0x18] value to update system time with, on next internal  ... */
	uint32_t int_update_seconds;
	/* [0x1c] value to update system time with, on next internal  ... */
	uint32_t int_update_subseconds_msb;
	/* [0x20] value to update system time with, on next internal  ... */
	uint32_t int_update_subseconds_lsb;
	/* [0x24] control for internally updating the system time reg ... */
	uint32_t ext_update_ctrl;
	/* [0x28] value to update system time with, on next internal  ... */
	uint32_t ext_update_seconds;
	/* [0x2c] value to update system time with, on next internal  ... */
	uint32_t ext_update_subseconds_msb;
	/* [0x30] value to update system time with, on next internal  ... */
	uint32_t ext_update_subseconds_lsb;
	/* [0x34] value to be added to system_time upon SW read, to c ... */
	uint32_t read_compensation_subseconds_msb;
	/* [0x38] value to be added to system_time upon SW read, to c ... */
	uint32_t read_compensation_subseconds_lsb;
	/* [0x3c] value to be added to system_time upon SW write to c ... */
	uint32_t int_write_compensation_subseconds_msb;
	/* [0x40] value to be added to system_time upon SW write to c ... */
	uint32_t int_write_compensation_subseconds_lsb;
	/* [0x44] value to be added to system_time upon system_time u ... */
	uint32_t ext_write_compensation_subseconds_msb;
	/* [0x48] value to be added to system_time upon system_time u ... */
	uint32_t ext_write_compensation_subseconds_lsb;
	/* [0x4c] value to be added to system_time before transferrin ... */
	uint32_t sync_compensation_subseconds_msb;
	/* [0x50] value to be added to system_time before transferrin ... */
	uint32_t sync_compensation_subseconds_lsb;
	uint32_t rsrvd[11];
};
struct al_ec_pth_egress {
	/* [0x0] control register for egress trigger #0 */
	uint32_t trigger_ctrl;
	/* [0x4] next egress trigger (#0) */
	uint32_t trigger_seconds;
	/* [0x8] next egress trigger (#0) */
	uint32_t trigger_subseconds_msb;
	/* [0xc] next egress trigger (#0) */
	uint32_t trigger_subseconds_lsb;
	/* [0x10] External output pulse width */
	uint32_t pulse_width_subseconds_msb;
	/* [0x14] External output pulse width */
	uint32_t pulse_width_subseconds_lsb;
	uint32_t rsrvd[2];
};
struct al_ec_pth_db {
	/* [0x0] timestamp[0], in resolution of 2^18 femtosec =~ 0 */
	uint32_t ts;
	uint32_t rsrvd[5];
};
struct al_ec_roce {
	/* [0x0] RoCE configuration  */
	uint32_t roce_gen_cfg_1;
	/* [0x4] READ_RESPONSE  state error */
	uint32_t rr_err;
	/* [0x8] READ_RESPONSE qualification error mask */
	uint32_t rr_qual;
	/* [0xc] READ_RESPONSE packet counter mask */
	uint32_t rr_err_cnt;
	/* [0x10] READ_RESPONSE interrupt generation  error mask */
	uint32_t rr_err_int;
	/* [0x14] WRITE REQUEST  state error */
	uint32_t wr_err;
	/* [0x18] WRITE REQUEST qualification error mask */
	uint32_t wr_qual;
	/* [0x1c] WRITE REQUEST packet counter mask */
	uint32_t wr_err_cnt;
	/* [0x20] WRITE REQUEST interrupt generation  error mask */
	uint32_t wr_err_int;
	/* [0x24] RoCE configuration  */
	uint32_t roce_gen_en_1;
	/* [0x28] RoCE GRH header fields configuration  */
	uint32_t roce_hdr;
	/* [0x2c] READ RESPONSE ERROR packets */
	uint32_t rr_error;
	/* [0x30] READ RESPONSE correct packets */
	uint32_t rr_ok;
	/* [0x34] WRITE REQUEST ERROR packets */
	uint32_t wr_error;
	/* [0x38] WRITE REQUEST correct packets */
	uint32_t wr_ok;
	uint32_t rsrvd[49];
};
struct al_ec_roce_qp_db {
	/* [0x0] QP value for comparing to the received DEST_QP */
	uint32_t qp_val;
	/* [0x4] RoCE configuration  */
	uint32_t roce_qp_cfg_1;
	/* [0x8] RoCE configuration  */
	uint32_t roce_qp_cfg_2;
	/* [0xc] READ_RESPONSE Control */
	uint32_t rr_ctrl;
	/* [0x10] WRITE REQUEST Control */
	uint32_t wr_ctrl;
	/* [0x14] WRITE REQUEST PSN control */
	uint32_t wr_psn;
	/* [0x18] WRITE REQUEST, LAST PSN value that the SW sampled */
	uint32_t wr_psn_last_sw;
	/* [0x1c] WRITE REQUEST, LAST PSN value that the HW sampled ( ... */
	uint32_t wr_psn_last_hw;
	/* [0x20] WRITE REQUEST, LAST PSN value that the HW sampled ( ... */
	uint32_t wr_psn_hw;
	/* [0x24] Address translation offset high */
	uint32_t addr_offset_h;
	/* [0x28] Address translation offset low */
	uint32_t addr_offset_l;
	/* [0x2c] Address translation mask high */
	uint32_t addr_mask_h;
	/* [0x30] Address translation mask low */
	uint32_t addr_mask_l;
	/* [0x34] PKEY */
	uint32_t p_key;
	/* [0x38] RKEY */
	uint32_t r_key;
	/* [0x3c] RoCE status information */
	uint32_t roce_status;
	/* [0x40] READ REQUEST FIFO */
	uint32_t rr_fifo_1;
	/* [0x44] READ REQUEST FIFO */
	uint32_t rr_fifo_2;
	/* [0x48] READ REQUEST FIFO */
	uint32_t rr_fifo_3;
	/* [0x4c] READ REQUEST FIFO */
	uint32_t rr_fifo_4;
	/* [0x50] READ REQUEST FIFO status */
	uint32_t rr_fifo_status;
	/* [0x54] READ REQUEST FIFO control */
	uint32_t rr_fifo_ctrl;
};

struct al_ec_regs {
	uint32_t rsrvd_0[32];
	struct al_ec_gen gen;                                   /* [0x80] */
	struct al_ec_mac mac;                                   /* [0xc0] */
	struct al_ec_rxf rxf;                                   /* [0x100] */
	struct al_ec_epe epe[2];                                /* [0x180] */
	struct al_ec_epe_res epe_res;                           /* [0x200] */
	struct al_ec_epe_h epe_h[32];                           /* [0x280] */
	struct al_ec_epe_p epe_p[32];                           /* [0x300] */
	struct al_ec_epe_a epe_a[32];                           /* [0x680] */
	struct al_ec_rfw rfw;                                   /* [0x700] */
	struct al_ec_rfw_udma rfw_udma[4];                      /* [0x7f0] */
	struct al_ec_rfw_hash rfw_hash[10];                     /* [0x800] */
	struct al_ec_rfw_priority rfw_priority[8];              /* [0x828] */
	struct al_ec_rfw_default rfw_default[8];                /* [0x848] */
	struct al_ec_fwd_mac fwd_mac[32];                       /* [0x868] */
	struct al_ec_msw msw;                                   /* [0xae8] */
	struct al_ec_tso tso;                                   /* [0xb00] */
	struct al_ec_tso_sel tso_sel[8];                        /* [0xb60] */
	struct al_ec_tpe tpe;                                   /* [0xb80] */
	struct al_ec_tpm_udma tpm_udma[4];                      /* [0xbc0] */
	struct al_ec_tpm_sel tpm_sel[4];                        /* [0xbf0] */
	struct al_ec_tfw tfw;                                   /* [0xc00] */
	struct al_ec_tfw_udma tfw_udma[4];                      /* [0xc60] */
	struct al_ec_tmi tmi;                                   /* [0xcc0] */
	struct al_ec_efc efc;                                   /* [0xcd0] */
	struct al_ec_fc_udma fc_udma[4];                        /* [0xd00] */
	struct al_ec_tpg_rpa_res tpg_rpa_res;                   /* [0xf00] */
	struct al_ec_eee eee;                                   /* [0x1000] */
	struct al_ec_stat stat;                                 /* [0x1100] */
	struct al_ec_stat_udma stat_udma[4];                    /* [0x1300] */
	struct al_ec_msp msp;                                   /* [0x1700] */
	struct al_ec_msp_p msp_p[32];                           /* [0x1740] */
	struct al_ec_msp_c msp_c[32];                           /* [0x17c0] */
	struct al_ec_crce crce;                                 /* [0x1b40] */
	struct al_ec_wol wol;                                   /* [0x1b80] */
	uint32_t rsrvd_1[80];
	struct al_ec_pth pth;                                   /* [0x1d00] */
	struct al_ec_pth_egress pth_egress[8];                  /* [0x1d80] */
	struct al_ec_pth_db pth_db[16];                         /* [0x1e80] */
	struct al_ec_roce roce;                                 /* [0x2000] */
	struct al_ec_roce_qp_db roce_qp_db[16];                 /* [0x2100] */
};

/*
* Registers Fields
*/

/**** version register ****/
/* Revision number (Minor) */
#define EC_GEN_VERSION_RELEASE_NUM_MINOR_MASK 0x000000FF
#define EC_GEN_VERSION_RELEASE_NUM_MINOR_SHIFT 0
/* Revision number (Major) */
#define EC_GEN_VERSION_RELEASE_NUM_MAJOR_MASK 0x0000FF00
#define EC_GEN_VERSION_RELEASE_NUM_MAJOR_SHIFT 8
/* Day of release */
#define EC_GEN_VERSION_DATE_DAY_MASK     0x001F0000
#define EC_GEN_VERSION_DATE_DAY_SHIFT    16
/* Month of release */
#define EC_GEN_VERSION_DATA_MONTH_MASK   0x01E00000
#define EC_GEN_VERSION_DATA_MONTH_SHIFT  21
/* Year of release (starting from 2000) */
#define EC_GEN_VERSION_DATE_YEAR_MASK    0x3E000000
#define EC_GEN_VERSION_DATE_YEAR_SHIFT   25
/* Reserved */
#define EC_GEN_VERSION_RESERVED_MASK     0xC0000000
#define EC_GEN_VERSION_RESERVED_SHIFT    30

/**** en register ****/
/* Enable Frequency adjust FIFO input controller operation. */
#define EC_GEN_EN_FAF_IN                 (1 << 0)
/* Enable Frequency adjust FIFO output controller operation. */
#define EC_GEN_EN_FAF_OUT                (1 << 1)
/* Enable Rx FIFO input controller 1 operation. */
#define EC_GEN_EN_RXF_IN                 (1 << 2)
/* Enable Rx FIFO output controller  operation. */
#define EC_GEN_EN_RXF_OUT                (1 << 3)
/* Enable Rx forwarding input controller operation. */
#define EC_GEN_EN_RFW_IN                 (1 << 4)
/* Enable Rx forwarding output controller operation. */
#define EC_GEN_EN_RFW_OUT                (1 << 5)
/* Enable Rx multi-stream write controller operation. */
#define EC_GEN_EN_MSW_IN                 (1 << 6)
/* Enable Rx first parsing engine output operation. */
#define EC_GEN_EN_RPE_1_OUT              (1 << 7)
/* Enable Rx first parsing engine input operation. */
#define EC_GEN_EN_RPE_1_IN               (1 << 8)
/* Enable Rx second parsing engine output operation. */
#define EC_GEN_EN_RPE_2_OUT              (1 << 9)
/* Enable Rx second parsing engine input operation. */
#define EC_GEN_EN_RPE_2_IN               (1 << 10)
/* Enable Rx MACsec parsing engine output operation. */
#define EC_GEN_EN_RPE_3_OUT              (1 << 11)
/* Enable Rx MACsec parsing engine input operation. */
#define EC_GEN_EN_RPE_3_IN               (1 << 12)
/* Enable Loopback FIFO input controller 1 operation. */
#define EC_GEN_EN_LBF_IN                 (1 << 13)
/* Enable Rx packet analyzer operation. */
#define EC_GEN_EN_RPA                    (1 << 14)

#define EC_GEN_EN_RESERVED_15            (1 << 15)
/* Enable Tx stream interface operation. */
#define EC_GEN_EN_TSO                    (1 << 16)
/* Enable Tx parser input controller operation. */
#define EC_GEN_EN_TPE_IN                 (1 << 17)
/* Enable Tx parser output controller operation. */
#define EC_GEN_EN_TPE_OUT                (1 << 18)
/* Enable Tx packet modification operation. */
#define EC_GEN_EN_TPM                    (1 << 19)
/* Enable Tx forwarding input controller operation. */
#define EC_GEN_EN_TFW_IN                 (1 << 20)
/* Enable Tx forwarding output controller operation. */
#define EC_GEN_EN_TFW_OUT                (1 << 21)
/* Enable Tx MAC interface controller operation. */
#define EC_GEN_EN_TMI                    (1 << 22)
/* Enable Tx packet generator operation. */
#define EC_GEN_EN_TPG                    (1 << 23)

#define EC_GEN_EN_RESERVED_31_MASK       0xFF000000
#define EC_GEN_EN_RESERVED_31_SHIFT      24

/**** fifo_en register ****/
/* Enable Frequency adjust FIFO operation (input). */
#define EC_GEN_FIFO_EN_FAF_IN            (1 << 0)
/* Enable Frequency adjust FIFO operation (output). */
#define EC_GEN_FIFO_EN_FAF_OUT           (1 << 1)
/* Enable Rx FIFO operation. */
#define EC_GEN_FIFO_EN_RX_FIFO           (1 << 2)
/* Enable Rx forwarding FIFO operation. */
#define EC_GEN_FIFO_EN_RFW_FIFO          (1 << 3)
/* Enable Rx multistream write FIFO operation */
#define EC_GEN_FIFO_EN_MSW_FIFO          (1 << 4)
/* Enable Rx first parser FIFO operation. */
#define EC_GEN_FIFO_EN_RPE_1_FIFO        (1 << 5)
/* Enable Rx second parser FIFO operation. */
#define EC_GEN_FIFO_EN_RPE_2_FIFO        (1 << 6)
/* Enable Rx MACsec parser FIFO operation. */
#define EC_GEN_FIFO_EN_RPE_3_FIFO        (1 << 7)
/* Enable Loopback FIFO operation. */
#define EC_GEN_FIFO_EN_LB_FIFO           (1 << 8)

#define EC_GEN_FIFO_EN_RESERVED_15_9_MASK 0x0000FE00
#define EC_GEN_FIFO_EN_RESERVED_15_9_SHIFT 9
/* Enable Tx parser FIFO operation. */
#define EC_GEN_FIFO_EN_TPE_FIFO          (1 << 16)
/* Enable Tx forwarding FIFO operation. */
#define EC_GEN_FIFO_EN_TFW_FIFO          (1 << 17)

#define EC_GEN_FIFO_EN_RESERVED_31_18_MASK 0xFFFC0000
#define EC_GEN_FIFO_EN_RESERVED_31_18_SHIFT 18

/**** l2 register ****/
/* Size of a 802.3 Ethernet header (DA+SA) */
#define EC_GEN_L2_SIZE_802_3_MASK        0x0000003F
#define EC_GEN_L2_SIZE_802_3_SHIFT       0
/* Size of a 802.3 + MACsec 8 byte header */
#define EC_GEN_L2_SIZE_802_3_MS_8_MASK   0x00003F00
#define EC_GEN_L2_SIZE_802_3_MS_8_SHIFT  8
/* Offset of the L2 header from the beginning of the packet. */
#define EC_GEN_L2_OFFSET_MASK            0x7F000000
#define EC_GEN_L2_OFFSET_SHIFT           24

/**** cfg_i register ****/
/* IPv4 protocol index */
#define EC_GEN_CFG_I_IPV4_INDEX_MASK     0x0000001F
#define EC_GEN_CFG_I_IPV4_INDEX_SHIFT    0
/* IPv6 protocol index */
#define EC_GEN_CFG_I_IPV6_INDEX_MASK     0x000003E0
#define EC_GEN_CFG_I_IPV6_INDEX_SHIFT    5
/* TCP protocol index */
#define EC_GEN_CFG_I_TCP_INDEX_MASK      0x00007C00
#define EC_GEN_CFG_I_TCP_INDEX_SHIFT     10
/* UDP protocol index */
#define EC_GEN_CFG_I_UDP_INDEX_MASK      0x000F8000
#define EC_GEN_CFG_I_UDP_INDEX_SHIFT     15
/* MACsec with 8 bytes SecTAG */
#define EC_GEN_CFG_I_MACSEC_8_INDEX_MASK 0x01F00000
#define EC_GEN_CFG_I_MACSEC_8_INDEX_SHIFT 20
/* MACsec with 16 bytes SecTAG */
#define EC_GEN_CFG_I_MACSEC_16_INDEX_MASK 0x3E000000
#define EC_GEN_CFG_I_MACSEC_16_INDEX_SHIFT 25

/**** cfg_i_ext register ****/
/* FcoE protocol index */
#define EC_GEN_CFG_I_EXT_FCOE_INDEX_MASK 0x0000001F
#define EC_GEN_CFG_I_EXT_FCOE_INDEX_SHIFT 0
/* RoCE protocol index */
#define EC_GEN_CFG_I_EXT_ROCE_INDEX_L3_1_MASK 0x000003E0
#define EC_GEN_CFG_I_EXT_ROCE_INDEX_L3_1_SHIFT 5
/* RoCE protocol index */
#define EC_GEN_CFG_I_EXT_ROCE_INDEX_L3_2_MASK 0x00007C00
#define EC_GEN_CFG_I_EXT_ROCE_INDEX_L3_2_SHIFT 10
/* RoCE protocol index */
#define EC_GEN_CFG_I_EXT_ROCE_INDEX_L4_MASK 0x000F8000
#define EC_GEN_CFG_I_EXT_ROCE_INDEX_L4_SHIFT 15

/**** en_ext register ****/
/* Enable Usage of Ethernet port memories for testing */
#define EC_GEN_EN_EXT_MEM_FOR_TEST       (1 << 0)
/* Enable MAC loop back (RX --> TX, after MAC layer) for 802 */
#define EC_GEN_EN_EXT_MAC_LB             (1 << 4)
/* CRC forward value for the MAC TX when working in loop back mo ... */
#define EC_GEN_EN_EXT_MAC_LB_CRC_FWD     (1 << 5)
/* Ready signal configuration when in loop back mode:00 – ready  ... */
#define EC_GEN_EN_EXT_MAC_LB_READY_CFG_MASK 0x000000C0
#define EC_GEN_EN_EXT_MAC_LB_READY_CFG_SHIFT 6
/* Bypass the PTH completion update */
#define EC_GEN_EN_EXT_PTH_COMPLETION_BYPASS (1 << 16)
/* selection between the 1G and 10G MAC:
0 – 1G
1 – 10G */
#define EC_GEN_EN_EXT_PTH_1_10_SEL       (1 << 17)
/* selection between descriptor caching option (WORD selection)0 ... */
#define EC_GEN_EN_EXT_CACHE_WORD_SPLIT   (1 << 20)

/**** gen register ****/
/* Enable swap of input byte order */
#define EC_MAC_GEN_SWAP_IN_BYTE          (1 << 0)

/**** min_pkt register ****/
/* Minimum packet size  */
#define EC_MAC_MIN_PKT_SIZE_MASK         0x000FFFFF
#define EC_MAC_MIN_PKT_SIZE_SHIFT        0

/**** max_pkt register ****/
/* Maximum packet size  */
#define EC_MAC_MAX_PKT_SIZE_MASK         0x000FFFFF
#define EC_MAC_MAX_PKT_SIZE_SHIFT        0

/**** cfg_1 register ****/
/* Drop packet at the ingress0 - Packets are not dropped at the  ... */
#define EC_RXF_CFG_1_DROP_AT_INGRESS     (1 << 0)
/* Accept packet criteria at start of packet indication */
#define EC_RXF_CFG_1_SOP_ACCEPT          (1 << 1)
/* Select the arbiter between Rx packets and Tx packets (packets ... */
#define EC_RXF_CFG_1_ARB_SEL             (1 << 2)
/* Arbiter priority when strict priority is selected in arb_sel0 ... */
#define EC_RXF_CFG_1_ARB_P               (1 << 3)
/* Force loopback operation */
#define EC_RXF_CFG_1_FORCE_LB            (1 << 4)
/* Forwarding selection between Rx path and/or packet analyzer */
#define EC_RXF_CFG_1_FWD_SEL_MASK        0x00000300
#define EC_RXF_CFG_1_FWD_SEL_SHIFT       8

/**** cfg_2 register ****/
/* FIFO USED threshold for accepting new packets, low threshold  ... */
#define EC_RXF_CFG_2_FIFO_USED_TH_L_MASK 0x0000FFFF
#define EC_RXF_CFG_2_FIFO_USED_TH_L_SHIFT 0
/* FIFO USED threshold for accepting new packets, high threshold ... */
#define EC_RXF_CFG_2_FIFO_USED_TH_H_MASK 0xFFFF0000
#define EC_RXF_CFG_2_FIFO_USED_TH_H_SHIFT 16

/**** rd_fifo register ****/
/* Minimum number of entries in the data FIFO to start reading p ... */
#define EC_RXF_RD_FIFO_TH_DATA_MASK      0x0000FFFF
#define EC_RXF_RD_FIFO_TH_DATA_SHIFT     0
/* Enable cut through operation */
#define EC_RXF_RD_FIFO_EN_CUT_TH         (1 << 16)

/**** wr_fifo register ****/

#define EC_RXF_WR_FIFO_TH_DATA_MASK      0x0000FFFF
#define EC_RXF_WR_FIFO_TH_DATA_SHIFT     0

#define EC_RXF_WR_FIFO_TH_INFO_MASK      0xFFFF0000
#define EC_RXF_WR_FIFO_TH_INFO_SHIFT     16

/**** lb_fifo register ****/

#define EC_RXF_LB_FIFO_TH_DATA_MASK      0x0000FFFF
#define EC_RXF_LB_FIFO_TH_DATA_SHIFT     0

#define EC_RXF_LB_FIFO_TH_INFO_MASK      0xFFFF0000
#define EC_RXF_LB_FIFO_TH_INFO_SHIFT     16

/**** cfg_lb register ****/
/* FIFO USED threshold for accepting new packets */
#define EC_RXF_CFG_LB_FIFO_USED_TH_INT_MASK 0x0000FFFF
#define EC_RXF_CFG_LB_FIFO_USED_TH_INT_SHIFT 0
/* FIFO USED threshold for generating ready for the Tx path */
#define EC_RXF_CFG_LB_FIFO_USED_TH_EXT_MASK 0xFFFF0000
#define EC_RXF_CFG_LB_FIFO_USED_TH_EXT_SHIFT 16

/**** out_drop register ****/

#define EC_RXF_OUT_DROP_MAC_ERR          (1 << 0)

#define EC_RXF_OUT_DROP_MAC_COL          (1 << 1)

#define EC_RXF_OUT_DROP_MAC_DEC          (1 << 2)

#define EC_RXF_OUT_DROP_MAC_LEN          (1 << 3)

#define EC_RXF_OUT_DROP_MAC_PHY          (1 << 4)

#define EC_RXF_OUT_DROP_MAC_FIFO         (1 << 5)

#define EC_RXF_OUT_DROP_MAC_FCS          (1 << 6)

#define EC_RXF_OUT_DROP_MAC_ETYPE        (1 << 7)

#define EC_RXF_OUT_DROP_EC_LEN           (1 << 8)

#define EC_RXF_OUT_DROP_EC_FIFO          (1 << 9)

/**** parse_cfg register ****/
/* MAX number of beats for packet parsing */
#define EC_EPE_PARSE_CFG_MAX_BEATS_MASK  0x000000FF
#define EC_EPE_PARSE_CFG_MAX_BEATS_SHIFT 0
/* MAX number of parsing iterations for packet parsing */
#define EC_EPE_PARSE_CFG_MAX_ITER_MASK   0x0000FF00
#define EC_EPE_PARSE_CFG_MAX_ITER_SHIFT  8

/**** act_table_addr register ****/
/* Address for accessing the table */
#define EC_EPE_ACT_TABLE_ADDR_VAL_MASK   0x0000001F
#define EC_EPE_ACT_TABLE_ADDR_VAL_SHIFT  0

/**** act_table_data_1 register ****/
/* Table data[5:0] - Offset to next protocol [bytes][6] - Next p ... */
#define EC_EPE_ACT_TABLE_DATA_1_VAL_MASK 0x03FFFFFF
#define EC_EPE_ACT_TABLE_DATA_1_VAL_SHIFT 0

/**** act_table_data_2 register ****/
/* Table Data [8:0] - Offset to data in the packet [bits][17:9]  ... */
#define EC_EPE_ACT_TABLE_DATA_2_VAL_MASK 0x1FFFFFFF
#define EC_EPE_ACT_TABLE_DATA_2_VAL_SHIFT 0

/**** act_table_data_3 register ****/
/* Table Data  [8:0] - Offset to data in the packet [bits] [17:9 ... */
#define EC_EPE_ACT_TABLE_DATA_3_VAL_MASK 0x1FFFFFFF
#define EC_EPE_ACT_TABLE_DATA_3_VAL_SHIFT 0

/**** act_table_data_4 register ****/
/* Table data[7:0] - Offset to header length location in the pac ... */
#define EC_EPE_ACT_TABLE_DATA_4_VAL_MASK 0x0FFFFFFF
#define EC_EPE_ACT_TABLE_DATA_4_VAL_SHIFT 0

/**** act_table_data_6 register ****/
/* Table data[0] - WR header length[10:1] - Write header length  ... */
#define EC_EPE_ACT_TABLE_DATA_6_VAL_MASK 0x007FFFFF
#define EC_EPE_ACT_TABLE_DATA_6_VAL_SHIFT 0

/**** res_in register ****/
/* Selector for input parse_en0 – Input vector1 – Default value  ... */
#define EC_EPE_RES_IN_SEL_PARSE_EN       (1 << 0)
/* Selector for input protocol_index 0 - Input vector 1 - Defaul ... */
#define EC_EPE_RES_IN_SEL_PROT_INDEX     (1 << 1)
/* Selector for input hdr_offset 0 - Input vector 1 - Default va ... */
#define EC_EPE_RES_IN_SEL_HDR_OFFSET     (1 << 2)

/**** p1 register ****/
/* Location of the input protocol index in the parser result vec ... */
#define EC_EPE_RES_P1_IN_PROT_INDEX_MASK 0x000003FF
#define EC_EPE_RES_P1_IN_PROT_INDEX_SHIFT 0

/**** p2 register ****/
/* Location of the input offset in the parser result vector */
#define EC_EPE_RES_P2_IN_OFFSET_MASK     0x000003FF
#define EC_EPE_RES_P2_IN_OFFSET_SHIFT    0

/**** p3 register ****/
/* Location of the input parse enable in the parser result vecto ... */
#define EC_EPE_RES_P3_IN_PARSE_EN_MASK   0x000003FF
#define EC_EPE_RES_P3_IN_PARSE_EN_SHIFT  0

/**** p4 register ****/
/* Location of the control bits in the parser result vector */
#define EC_EPE_RES_P4_CTRL_BITS_MASK     0x000003FF
#define EC_EPE_RES_P4_CTRL_BITS_SHIFT    0

/**** p5 register ****/
/* Location of the MAC DA in the parser result vector */
#define EC_EPE_RES_P5_DA_MASK            0x000003FF
#define EC_EPE_RES_P5_DA_SHIFT           0

/**** p6 register ****/
/* Location of the MAC SA in the parser result vector */
#define EC_EPE_RES_P6_SA_MASK            0x000003FF
#define EC_EPE_RES_P6_SA_SHIFT           0

/**** p7 register ****/
/* Location of the first VLAN in the parser result vector */
#define EC_EPE_RES_P7_VLAN_1_MASK        0x000003FF
#define EC_EPE_RES_P7_VLAN_1_SHIFT       0

/**** p8 register ****/
/* Location of the second VLAN in the parser result vector */
#define EC_EPE_RES_P8_VLAN_2_MASK        0x000003FF
#define EC_EPE_RES_P8_VLAN_2_SHIFT       0

/**** p9 register ****/
/* Location of the L3 protocol index in the parser result vector ... */
#define EC_EPE_RES_P9_L3_PROT_INDEX_MASK 0x000003FF
#define EC_EPE_RES_P9_L3_PROT_INDEX_SHIFT 0

/**** p10 register ****/
/* Location of the L3 offset in the parser result vector */
#define EC_EPE_RES_P10_L3_OFFSET_MASK    0x000003FF
#define EC_EPE_RES_P10_L3_OFFSET_SHIFT   0

/**** p11 register ****/
/* Location of the L3 SIP in the parser result vector */
#define EC_EPE_RES_P11_L3_SIP_MASK       0x000003FF
#define EC_EPE_RES_P11_L3_SIP_SHIFT      0

/**** p12 register ****/
/* Location of the L3 DIP in the parser result vector */
#define EC_EPE_RES_P12_L3_DIP_MASK       0x000003FF
#define EC_EPE_RES_P12_L3_DIP_SHIFT      0

/**** p13 register ****/
/* Location of the L3 priority in the parser result vector */
#define EC_EPE_RES_P13_L3_PRIORITY_MASK  0x000003FF
#define EC_EPE_RES_P13_L3_PRIORITY_SHIFT 0

/**** p14 register ****/
/* Location of the L3 header length in the parser result vector */
#define EC_EPE_RES_P14_L3_HDR_LEN_MASK   0x000003FF
#define EC_EPE_RES_P14_L3_HDR_LEN_SHIFT  0

/**** p15 register ****/
/* Location of the L4 protocol index in the parser result vector ... */
#define EC_EPE_RES_P15_L4_PROT_INDEX_MASK 0x000003FF
#define EC_EPE_RES_P15_L4_PROT_INDEX_SHIFT 0

/**** p16 register ****/
/* Location of the L4 source port in the parser result vector */
#define EC_EPE_RES_P16_L4_SRC_PORT_MASK  0x000003FF
#define EC_EPE_RES_P16_L4_SRC_PORT_SHIFT 0

/**** p17 register ****/
/* Location of the L4 destination port in the parser result vect ... */
#define EC_EPE_RES_P17_L4_DST_PORT_MASK  0x000003FF
#define EC_EPE_RES_P17_L4_DST_PORT_SHIFT 0

/**** p18 register ****/
/* Location of the L4 offset in the parser result vector */
#define EC_EPE_RES_P18_L4_OFFSET_MASK    0x000003FF
#define EC_EPE_RES_P18_L4_OFFSET_SHIFT   0

/**** p19 register ****/
/* Location of the Ether type in the parser result vector when w ... */
#define EC_EPE_RES_P19_WOL_ETYPE_MASK    0x000003FF
#define EC_EPE_RES_P19_WOL_ETYPE_SHIFT   0

/**** p20 register ****/
/* Location of the RoCE QP number field in the parser result vec ... */
#define EC_EPE_RES_P20_ROCE_QPN_MASK     0x000003FF
#define EC_EPE_RES_P20_ROCE_QPN_SHIFT    0

/**** hdr_len register ****/
/* Value for selecting table 1 */
#define EC_EPE_H_HDR_LEN_TABLE_1_MASK    0x000000FF
#define EC_EPE_H_HDR_LEN_TABLE_1_SHIFT   0
/* Value for selecting table 2 */
#define EC_EPE_H_HDR_LEN_TABLE_2_MASK    0x00FF0000
#define EC_EPE_H_HDR_LEN_TABLE_2_SHIFT   16

/**** comp_data register ****/
/* Data 1 for comparison */
#define EC_EPE_P_COMP_DATA_DATA_1_MASK   0x0000FFFF
#define EC_EPE_P_COMP_DATA_DATA_1_SHIFT  0
/* Data 2 for comparison  
[18:16] - Stage 
[24:19] - Branch ID */
#define EC_EPE_P_COMP_DATA_DATA_2_MASK   0x01FF0000
#define EC_EPE_P_COMP_DATA_DATA_2_SHIFT  16

/**** comp_mask register ****/
/* Data 1 for comparison */
#define EC_EPE_P_COMP_MASK_DATA_1_MASK   0x0000FFFF
#define EC_EPE_P_COMP_MASK_DATA_1_SHIFT  0
/* Data 2 for comparison 
[18:16] – Stage
[24:19] – Branch ID */
#define EC_EPE_P_COMP_MASK_DATA_2_MASK   0x01FF0000
#define EC_EPE_P_COMP_MASK_DATA_2_SHIFT  16

/**** comp_ctrl register ****/
/* Output result value */
#define EC_EPE_P_COMP_CTRL_RES_MASK      0x0000001F
#define EC_EPE_P_COMP_CTRL_RES_SHIFT     0
/* Compare command for the data_1 field00 - Compare01 - <=10 - > ... */
#define EC_EPE_P_COMP_CTRL_CMD_1_MASK    0x00030000
#define EC_EPE_P_COMP_CTRL_CMD_1_SHIFT   16
/* Compare command for the data_2 field 00 - Compare 01 - <= 10  ... */
#define EC_EPE_P_COMP_CTRL_CMD_2_MASK    0x000C0000
#define EC_EPE_P_COMP_CTRL_CMD_2_SHIFT   18
/* Entry is valid */
#define EC_EPE_P_COMP_CTRL_VALID         (1 << 31)

/**** prot_act register ****/
/* Drop indication for the selected protocol index */
#define EC_EPE_A_PROT_ACT_DROP           (1 << 0)
/* Mapping value Used when mapping the entire protocol index ran ... */
#define EC_EPE_A_PROT_ACT_MAP_MASK       0x00000F00
#define EC_EPE_A_PROT_ACT_MAP_SHIFT      8

/**** thash_cfg_1 register ****/
/* Hash function output selection:000 - [7:0]001 - [15:8]010 - [ ... */
#define EC_RFW_THASH_CFG_1_OUT_SEL_MASK  0x00000007
#define EC_RFW_THASH_CFG_1_OUT_SEL_SHIFT 0
/* Selects between hash functions00 - toeplitz01 - CRC-3210 – 0x ... */
#define EC_RFW_THASH_CFG_1_FUNC_SEL_MASK 0x00000300
#define EC_RFW_THASH_CFG_1_FUNC_SEL_SHIFT 8
/* Enable SIP/DIP swap if SIP<DIP */
#define EC_RFW_THASH_CFG_1_ENABLE_IP_SWAP (1 << 16)
/* Enable PORT swap if SPORT<DPORT */
#define EC_RFW_THASH_CFG_1_ENABLE_PORT_SWAP (1 << 17)

/**** mhash_cfg_1 register ****/
/* Hash function output selection:000 - [7:0]001 - [15:8]010 - [ ... */
#define EC_RFW_MHASH_CFG_1_OUT_SEL_MASK  0x00000007
#define EC_RFW_MHASH_CFG_1_OUT_SEL_SHIFT 0
/* Selects the input to the MAC hash fiunction0 - DA1 - DA + SA ... */
#define EC_RFW_MHASH_CFG_1_INPUT_SEL     (1 << 4)
/* Selects between hash functions00 - toeplitz01 - CRC-3210 – 0x ... */
#define EC_RFW_MHASH_CFG_1_FUNC_SEL_MASK 0x00000300
#define EC_RFW_MHASH_CFG_1_FUNC_SEL_SHIFT 8

/**** hdr_split register ****/
/* Default header length for header split */
#define EC_RFW_HDR_SPLIT_DEF_LEN_MASK    0x0000FFFF
#define EC_RFW_HDR_SPLIT_DEF_LEN_SHIFT   0
/* Enable header split operation */
#define EC_RFW_HDR_SPLIT_EN              (1 << 16)

/**** meta_err register ****/
/* Mask for error 1 in the Rx descriptor */
#define EC_RFW_META_ERR_MASK_1_MASK      0x000003FF
#define EC_RFW_META_ERR_MASK_1_SHIFT     0
/* Mask for error 2 in the Rx descriptor */
#define EC_RFW_META_ERR_MASK_2_MASK      0x03FF0000
#define EC_RFW_META_ERR_MASK_2_SHIFT     16

/**** meta register ****/
/* Selection of the L3 offset source: 1 - Inner packet 0 - Outer ... */
#define EC_RFW_META_L3_LEN_SEL           (1 << 0)
/* Selection of the L3 offset source:1 - Inner packet0 - Outer p ... */
#define EC_RFW_META_L3_OFFSET_SEL        (1 << 1)
/* selection of the l3 protocol index source: 1 - inner packet 0 ... */
#define EC_RFW_META_L3_PROT_SEL          (1 << 2)
/* selection of the l4 protocol index source:  1 - inner packet  ... */
#define EC_RFW_META_L4_PROT_SEL          (1 << 3)
/* Selects how to calculate the L3 header length when L3 is IpPv ... */
#define EC_RFW_META_L3_LEN_CALC          (1 << 4)
/* Selection of the Ipv4 fragment indication source:  1 - Inner  ... */
#define EC_RFW_META_FRAG_SEL             (1 << 5)
/* Selection of the L4 offset source:1 - Inner packet0 - Outer p ... */
#define EC_RFW_META_L4_OFFSET_SEL        (1 << 6)

/**** filter register ****/
/* Filter undetected MAC DA */
#define EC_RFW_FILTER_UNDET_MAC          (1 << 0)
/* Filter specific MAC DA based on MAC table output. */
#define EC_RFW_FILTER_DET_MAC            (1 << 1)
/* Filter all tagged. */
#define EC_RFW_FILTER_TAGGED             (1 << 2)
/* Filter all untagged. */
#define EC_RFW_FILTER_UNTAGGED           (1 << 3)
/* Filter all broadcast. */
#define EC_RFW_FILTER_BC                 (1 << 4)
/* Filter all multicast. */
#define EC_RFW_FILTER_MC                 (1 << 5)
/* Filter based on parsing output (used to drop selected protoco ... */
#define EC_RFW_FILTER_PARSE              (1 << 6)
/* Filter packet based on VLAN table output. */
#define EC_RFW_FILTER_VLAN_VID           (1 << 7)
/* Filter packet based on control table output. */
#define EC_RFW_FILTER_CTRL_TABLE         (1 << 8)
/* Filter packet based on protocol index action register. */
#define EC_RFW_FILTER_PROT_INDEX         (1 << 9)
/* Filter packet based on WoL decision */
#define EC_RFW_FILTER_WOL                (1 << 10)
/* Override filter decision and forward to default UDMA/queue;dr ... */
#define EC_RFW_FILTER_OR_UNDET_MAC       (1 << 16)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_DET_MAC         (1 << 17)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_TAGGED          (1 << 18)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_UNTAGGED        (1 << 19)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_BC              (1 << 20)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_MC              (1 << 21)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_PARSE           (1 << 22)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_VLAN_VID        (1 << 23)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_CTRL_TABLE      (1 << 24)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_PROT_INDEX      (1 << 25)
/* Override filter decision and forward to default UDMA/queue;Dr ... */
#define EC_RFW_FILTER_OR_WOL             (1 << 26)

/**** thash_table_addr register ****/
/* Address for accessing the table */
#define EC_RFW_THASH_TABLE_ADDR_VAL_MASK 0x000000FF
#define EC_RFW_THASH_TABLE_ADDR_VAL_SHIFT 0

/**** thash_table_data register ****/
/* Table data (valid only after configuring the table address re ... */
#define EC_RFW_THASH_TABLE_DATA_VAL_MASK 0x00003FFF
#define EC_RFW_THASH_TABLE_DATA_VAL_SHIFT 0

/**** mhash_table_addr register ****/
/* Address for accessing the table */
#define EC_RFW_MHASH_TABLE_ADDR_VAL_MASK 0x000000FF
#define EC_RFW_MHASH_TABLE_ADDR_VAL_SHIFT 0

/**** mhash_table_data register ****/
/* Table data (valid only after configuring the table address re ... */
#define EC_RFW_MHASH_TABLE_DATA_VAL_MASK 0x0000003F
#define EC_RFW_MHASH_TABLE_DATA_VAL_SHIFT 0

/**** vid_table_addr register ****/
/* Address for accessing the table */
#define EC_RFW_VID_TABLE_ADDR_VAL_MASK   0x00000FFF
#define EC_RFW_VID_TABLE_ADDR_VAL_SHIFT  0

/**** vid_table_data register ****/
/* Table data (valid only after configuring the table address re ... */
#define EC_RFW_VID_TABLE_DATA_VAL_MASK   0x0000003F
#define EC_RFW_VID_TABLE_DATA_VAL_SHIFT  0

/**** pbits_table_addr register ****/
/* Address for accessing the table */
#define EC_RFW_PBITS_TABLE_ADDR_VAL_MASK 0x00000007
#define EC_RFW_PBITS_TABLE_ADDR_VAL_SHIFT 0

/**** pbits_table_data register ****/
/* VLAN P-bits to internal priority mapping */
#define EC_RFW_PBITS_TABLE_DATA_VAL_MASK 0x00000007
#define EC_RFW_PBITS_TABLE_DATA_VAL_SHIFT 0

/**** dscp_table_addr register ****/
/* Address for accessing the table */
#define EC_RFW_DSCP_TABLE_ADDR_VAL_MASK  0x000000FF
#define EC_RFW_DSCP_TABLE_ADDR_VAL_SHIFT 0

/**** dscp_table_data register ****/
/* IPv4 DSCP to internal priority mapping */
#define EC_RFW_DSCP_TABLE_DATA_VAL_MASK  0x00000007
#define EC_RFW_DSCP_TABLE_DATA_VAL_SHIFT 0

/**** tc_table_addr register ****/
/* Address for accessing the table */
#define EC_RFW_TC_TABLE_ADDR_VAL_MASK    0x000000FF
#define EC_RFW_TC_TABLE_ADDR_VAL_SHIFT   0

/**** tc_table_data register ****/
/* IPv6 TC to internal priority mapping */
#define EC_RFW_TC_TABLE_DATA_VAL_MASK    0x00000007
#define EC_RFW_TC_TABLE_DATA_VAL_SHIFT   0

/**** ctrl_table_addr register ****/
/* Address for accessing the table[0] – VLAN table control out[1 ... */
#define EC_RFW_CTRL_TABLE_ADDR_VAL_MASK  0x000007FF
#define EC_RFW_CTRL_TABLE_ADDR_VAL_SHIFT 0

/**** ctrl_table_data register ****/
/* Control table output for selecting the forwarding MUXs[3:0] - ... */
#define EC_RFW_CTRL_TABLE_DATA_VAL_MASK  0x000FFFFF
#define EC_RFW_CTRL_TABLE_DATA_VAL_SHIFT 0

/**** out_cfg register ****/
/* Number of MetaData at the end of the packet1 - One MetaData b ... */
#define EC_RFW_OUT_CFG_META_CNT_MASK     0x00000003
#define EC_RFW_OUT_CFG_META_CNT_SHIFT    0
/* Enable packet drop */
#define EC_RFW_OUT_CFG_DROP_EN           (1 << 2)
/* Swap output byte order */
#define EC_RFW_OUT_CFG_SWAP_OUT_BYTE     (1 << 3)
/* Enable the insertion of the MACSEC decoding result into the M ... */
#define EC_RFW_OUT_CFG_EN_MACSEC_DEC     (1 << 4)
/* Sample time of the time stamp:0 – SOP (for 10G MAC)1 – EOP (f ... */
#define EC_RFW_OUT_CFG_TIMESTAMP_SAMPLE  (1 << 5)

/**** fsm_table_addr register ****/
/* Address for accessing the table :[2:0] - Outer header control ... */
#define EC_RFW_FSM_TABLE_ADDR_VAL_MASK   0x0000007F
#define EC_RFW_FSM_TABLE_ADDR_VAL_SHIFT  0

/**** fsm_table_data register ****/
/* Flow steering mechanism output selectors:[1:0] – Input select ... */
#define EC_RFW_FSM_TABLE_DATA_VAL_MASK   0x00000007
#define EC_RFW_FSM_TABLE_DATA_VAL_SHIFT  0

/**** ctrl_sel register ****/
/* Packet type (UC/MC/BC) for the control table */
#define EC_RFW_CTRL_SEL_PKT_TYPE         (1 << 0)
/* L3 protocol index for the control table */
#define EC_RFW_CTRL_SEL_L3_PROTOCOL      (1 << 1)
/* Selects the content and structure of the control table addres ... */
#define EC_RFW_CTRL_SEL_ADDR_MASK        0x0000000C
#define EC_RFW_CTRL_SEL_ADDR_SHIFT       2

/**** default_vlan register ****/
/* Default VLAN data, used for untagged packets */
#define EC_RFW_DEFAULT_VLAN_DATA_MASK    0x0000FFFF
#define EC_RFW_DEFAULT_VLAN_DATA_SHIFT   0

/**** default_hash register ****/
/* Default UDMA */
#define EC_RFW_DEFAULT_HASH_UDMA_MASK    0x0000000F
#define EC_RFW_DEFAULT_HASH_UDMA_SHIFT   0
/* Default queue */
#define EC_RFW_DEFAULT_HASH_QUEUE_MASK   0x00030000
#define EC_RFW_DEFAULT_HASH_QUEUE_SHIFT  16

/**** default_or register ****/
/* Default UDMA */
#define EC_RFW_DEFAULT_OR_UDMA_MASK      0x0000000F
#define EC_RFW_DEFAULT_OR_UDMA_SHIFT     0
/* Default queue */
#define EC_RFW_DEFAULT_OR_QUEUE_MASK     0x00030000
#define EC_RFW_DEFAULT_OR_QUEUE_SHIFT    16

/**** checksum register ****/
/* Check that the length in the UDP header matches the length in ... */
#define EC_RFW_CHECKSUM_UDP_LEN          (1 << 0)
/* Select the header that will be used for the checksum when a t ... */
#define EC_RFW_CHECKSUM_HDR_SEL          (1 << 1)
/* Enable L4 checksum when L3 fragmentation is detected */
#define EC_RFW_CHECKSUM_L4_FRAG_EN       (1 << 2)
/* L3 Checksum result selection for the Metadata descriptor0 – o ... */
#define EC_RFW_CHECKSUM_L3_CKS_SEL       (1 << 4)
/* L4 Checksum result selection for the Metadata descriptor0 – o ... */
#define EC_RFW_CHECKSUM_L4_CKS_SEL       (1 << 5)

/**** lro_cfg_1 register ****/
/* Select the header that will be used for the LRO offload engin ... */
#define EC_RFW_LRO_CFG_1_HDR_SEL         (1 << 0)
/* Select the L2 header that will be used for the LRO offload en ... */
#define EC_RFW_LRO_CFG_1_HDR_L2_SEL      (1 << 1)

/**** lro_check_ipv4 register ****/
/* Check version field. */
#define EC_RFW_LRO_CHECK_IPV4_VER        (1 << 0)
/* Check IHL field == 5. */
#define EC_RFW_LRO_CHECK_IPV4_IHL_0      (1 << 1)
/* Check IHL field >= 5. */
#define EC_RFW_LRO_CHECK_IPV4_IHL_1      (1 << 2)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_IPV4_IHL_2      (1 << 3)
/* Compare DSCP to previous packet. */
#define EC_RFW_LRO_CHECK_IPV4_DSCP       (1 << 4)
/* Check that Total length >= lro_ipv4_tlen_val. */
#define EC_RFW_LRO_CHECK_IPV4_TLEN       (1 << 5)
/* Compare to previous packet value +1. */
#define EC_RFW_LRO_CHECK_IPV4_ID         (1 << 6)
/* Compare to lro_ipv4_flags_val with lro_ipv4_flags_mask_0. */
#define EC_RFW_LRO_CHECK_IPV4_FLAGS_0    (1 << 7)
/* Compare to previous packet flags with lro_ipv4_flags_mask_1. */
#define EC_RFW_LRO_CHECK_IPV4_FLAGS_1    (1 << 8)
/* Verify that the fragment offset field is 0. */
#define EC_RFW_LRO_CHECK_IPV4_FRAG       (1 << 9)
/* Verify that the TTL value >0. */
#define EC_RFW_LRO_CHECK_IPV4_TTL_0      (1 << 10)
/* Compare TTL value to previous packet. */
#define EC_RFW_LRO_CHECK_IPV4_TTL_1      (1 << 11)
/* Compare to previous packet protocol field. */
#define EC_RFW_LRO_CHECK_IPV4_PROT_0     (1 << 12)
/* Verify that the protocol is TCP or UDP. */
#define EC_RFW_LRO_CHECK_IPV4_PROT_1     (1 << 13)
/* Verify that the check sum is correct. */
#define EC_RFW_LRO_CHECK_IPV4_CHECKSUM   (1 << 14)
/* Compare SIP to previous packet. */
#define EC_RFW_LRO_CHECK_IPV4_SIP        (1 << 15)
/* Compare DIP to previous packet. */
#define EC_RFW_LRO_CHECK_IPV4_DIP        (1 << 16)

/**** lro_ipv4 register ****/
/* Total length minimum value */
#define EC_RFW_LRO_IPV4_TLEN_VAL_MASK    0x0000FFFF
#define EC_RFW_LRO_IPV4_TLEN_VAL_SHIFT   0
/* Flags value  */
#define EC_RFW_LRO_IPV4_FLAGS_VAL_MASK   0x00070000
#define EC_RFW_LRO_IPV4_FLAGS_VAL_SHIFT  16
/* Flags mask */
#define EC_RFW_LRO_IPV4_FLAGS_MASK_0_MASK 0x00380000
#define EC_RFW_LRO_IPV4_FLAGS_MASK_0_SHIFT 19
/* Flags mask */
#define EC_RFW_LRO_IPV4_FLAGS_MASK_1_MASK 0x01C00000
#define EC_RFW_LRO_IPV4_FLAGS_MASK_1_SHIFT 22
/* Version value */
#define EC_RFW_LRO_IPV4_VER_MASK         0xF0000000
#define EC_RFW_LRO_IPV4_VER_SHIFT        28

/**** lro_check_ipv6 register ****/
/* Check version field */
#define EC_RFW_LRO_CHECK_IPV6_VER        (1 << 0)
/* Compare TC to previous packet. */
#define EC_RFW_LRO_CHECK_IPV6_TC         (1 << 1)
/* Compare flow label field to previous packet. */
#define EC_RFW_LRO_CHECK_IPV6_FLOW       (1 << 2)
/* Check that Total length >= lro_ipv6_pen_val. */
#define EC_RFW_LRO_CHECK_IPV6_PLEN       (1 << 3)
/* Compare to previous packet next header field. */
#define EC_RFW_LRO_CHECK_IPV6_NEXT_0     (1 << 4)
/* Verify that the next header is TCP or UDP. */
#define EC_RFW_LRO_CHECK_IPV6_NEXT_1     (1 << 5)
/* Verify that hop limit is >0. */
#define EC_RFW_LRO_CHECK_IPV6_HOP_0      (1 << 6)
/* Compare hop limit to previous packet. */
#define EC_RFW_LRO_CHECK_IPV6_HOP_1      (1 << 7)
/* Compare SIP to previous packet. */
#define EC_RFW_LRO_CHECK_IPV6_SIP        (1 << 8)
/* Compare DIP to previous packet. */
#define EC_RFW_LRO_CHECK_IPV6_DIP        (1 << 9)

/**** lro_ipv6 register ****/
/* Payload length minimum value */
#define EC_RFW_LRO_IPV6_PLEN_VAL_MASK    0x0000FFFF
#define EC_RFW_LRO_IPV6_PLEN_VAL_SHIFT   0
/* Version value */
#define EC_RFW_LRO_IPV6_VER_MASK         0x0F000000
#define EC_RFW_LRO_IPV6_VER_SHIFT        24

/**** lro_check_tcp register ****/
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_TCP_SRC_PORT    (1 << 0)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_TCP_DST_PORT    (1 << 1)
/* If (SYN == 1), don't check  */
#define EC_RFW_LRO_CHECK_TCP_SN          (1 << 2)
/* Check data offset field == 5. */
#define EC_RFW_LRO_CHECK_TCP_OFFSET_0    (1 << 3)
/* Check data offset field >= 5. */
#define EC_RFW_LRO_CHECK_TCP_OFFSET_1    (1 << 4)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_TCP_OFFSET_2    (1 << 5)
/* Compare reserved field to lro_tcp_res. */
#define EC_RFW_LRO_CHECK_TCP_RES         (1 << 6)
/* Compare to lro_tcp_ecn_val and lro_tcp_ecn_mask_0. */
#define EC_RFW_LRO_CHECK_TCP_ECN_0       (1 << 7)
/* Compare to previous packet ECN field with lro_tcp_ecn_mask_1 */
#define EC_RFW_LRO_CHECK_TCP_ECN_1       (1 << 8)
/* Compare to lro_tcp_ctrl_val and lro_tcp_ctrl_mask_0. */
#define EC_RFW_LRO_CHECK_TCP_CTRL_0      (1 << 9)
/* Compare to previous packet ECN field with lro_tcp_ctrl_mask_1 */
#define EC_RFW_LRO_CHECK_TCP_CTRL_1      (1 << 10)
/* Verify that check sum is correct. */
#define EC_RFW_LRO_CHECK_TCP_CHECKSUM    (1 << 11)

/**** lro_tcp register ****/
/* Reserved field default value */
#define EC_RFW_LRO_TCP_RES_MASK          0x00000007
#define EC_RFW_LRO_TCP_RES_SHIFT         0
/* ECN field value */
#define EC_RFW_LRO_TCP_ECN_VAL_MASK      0x00000038
#define EC_RFW_LRO_TCP_ECN_VAL_SHIFT     3
/* ECN field mask */
#define EC_RFW_LRO_TCP_ECN_MASK_0_MASK   0x000001C0
#define EC_RFW_LRO_TCP_ECN_MASK_0_SHIFT  6
/* ECN field mask */
#define EC_RFW_LRO_TCP_ECN_MASK_1_MASK   0x00000E00
#define EC_RFW_LRO_TCP_ECN_MASK_1_SHIFT  9
/* Control field value */
#define EC_RFW_LRO_TCP_CTRL_VAL_MASK     0x0003F000
#define EC_RFW_LRO_TCP_CTRL_VAL_SHIFT    12
/* Control field mask */
#define EC_RFW_LRO_TCP_CTRL_MASK_0_MASK  0x00FC0000
#define EC_RFW_LRO_TCP_CTRL_MASK_0_SHIFT 18
/* Control field mask */
#define EC_RFW_LRO_TCP_CTRL_MASK_1_MASK  0x3F000000
#define EC_RFW_LRO_TCP_CTRL_MASK_1_SHIFT 24

/**** lro_check_udp register ****/
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_UDP_SRC_PORT    (1 << 0)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_UDP_DST_PORT    (1 << 1)
/* Verify that check sum is correct. */
#define EC_RFW_LRO_CHECK_UDP_CHECKSUM    (1 << 2)

/**** lro_check_l2 register ****/
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_MAC_DA       (1 << 0)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_MAC_SA       (1 << 1)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_1_EXIST (1 << 2)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_1_VID   (1 << 3)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_1_CFI   (1 << 4)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_1_PBITS (1 << 5)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_2_EXIST (1 << 6)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_2_VID   (1 << 7)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_2_CFI   (1 << 8)
/* Compare to previous packet. */
#define EC_RFW_LRO_CHECK_L2_VLAN_2_PBITS (1 << 9)
/* Verify that the FCS is correct. */
#define EC_RFW_LRO_CHECK_L2_FCS          (1 << 10)

/**** lro_check_gen register ****/
/* Compare to previous packet */
#define EC_RFW_LRO_CHECK_GEN_UDMA        (1 << 0)
/* Compare to previous packet */
#define EC_RFW_LRO_CHECK_GEN_QUEUE       (1 << 1)

/**** lro_store register ****/
/* Store packet information if protocol match. */
#define EC_RFW_LRO_STORE_IPV4            (1 << 0)
/* Store packet information if protocol match. */
#define EC_RFW_LRO_STORE_IPV6            (1 << 1)
/* Store packet information if protocol match. */
#define EC_RFW_LRO_STORE_TCP             (1 << 2)
/* Store packet information if protocol match. */
#define EC_RFW_LRO_STORE_UDP             (1 << 3)
/* Store packet if Ipv4 flags match the register value with mask */
#define EC_RFW_LRO_STORE_IPV4_FLAGS_VAL_MASK 0x00000070
#define EC_RFW_LRO_STORE_IPV4_FLAGS_VAL_SHIFT 4
/* Mask for Ipv4 flags */
#define EC_RFW_LRO_STORE_IPV4_FLAGS_MASK_MASK 0x00000380
#define EC_RFW_LRO_STORE_IPV4_FLAGS_MASK_SHIFT 7
/* Store packet if TCP control and ECN match the register value  ... */
#define EC_RFW_LRO_STORE_TCP_CTRL_VAL_MASK 0x0007FC00
#define EC_RFW_LRO_STORE_TCP_CTRL_VAL_SHIFT 10
/* Mask for TCP control */
#define EC_RFW_LRO_STORE_TCP_CTRL_MASK_MASK 0x0FF80000
#define EC_RFW_LRO_STORE_TCP_CTRL_MASK_SHIFT 19

/**** vid_table_def register ****/
/* Table default data (valid only after configuring the table ad ... */
#define EC_RFW_VID_TABLE_DEF_VAL_MASK    0x0000003F
#define EC_RFW_VID_TABLE_DEF_VAL_SHIFT   0
/* Default data selection
0 – Default value
1 – Table data out */
#define EC_RFW_VID_TABLE_DEF_SEL         (1 << 6)

/**** ctrl_table_def register ****/
/* Control table output for selecting the forwarding MUXs [3:0]  ... */
#define EC_RFW_CTRL_TABLE_DEF_VAL_MASK   0x000FFFFF
#define EC_RFW_CTRL_TABLE_DEF_VAL_SHIFT  0
/* Default data selection 0 - Default value 1 - Table data out ... */
#define EC_RFW_CTRL_TABLE_DEF_SEL        (1 << 20)

/**** cfg_a_0 register ****/
/* selection of the L3 checksum result in the Metadata00 – L3 ch ... */
#define EC_RFW_CFG_A_0_META_L3_CHK_RES_SEL_MASK 0x00000003
#define EC_RFW_CFG_A_0_META_L3_CHK_RES_SEL_SHIFT 0
/* selection of the L4 checksum result in the Metadata0 – L4 che ... */
#define EC_RFW_CFG_A_0_META_L4_CHK_RES_SEL (1 << 2)
/* selection of the LRO_contet_value result in the Metadata0 – L ... */
#define EC_RFW_CFG_A_0_LRO_CONTEXT_SEL   (1 << 4)

/**** thash_cfg_3 register ****/
/* enable Hash value for RoCE packets in outer packet */
#define EC_RFW_THASH_CFG_3_ENABLE_OUTER_ROCE (1 << 0)
/* enable Hash value for RoCE packets in inner packet */
#define EC_RFW_THASH_CFG_3_ENABLE_INNER_ROCE (1 << 1)
/* enable Hash value for FcoE packets in outer packet */
#define EC_RFW_THASH_CFG_3_ENABLE_OUTER_FCOE (1 << 2)
/* enable Hash value for FcoE packets in inner packet */
#define EC_RFW_THASH_CFG_3_ENABLE_INNER_FCOE (1 << 3)

/**** thash_mask_outer_ipv6 register ****/
/* Ipv6 source IP address */
#define EC_RFW_THASH_MASK_OUTER_IPV6_SRC_MASK 0x0000FFFF
#define EC_RFW_THASH_MASK_OUTER_IPV6_SRC_SHIFT 0
/* Ipv6 destination IP address */
#define EC_RFW_THASH_MASK_OUTER_IPV6_DST_MASK 0xFFFF0000
#define EC_RFW_THASH_MASK_OUTER_IPV6_DST_SHIFT 16

/**** thash_mask_outer register ****/
/* Ipv4 source IP address */
#define EC_RFW_THASH_MASK_OUTER_IPV4_SRC_MASK 0x0000000F
#define EC_RFW_THASH_MASK_OUTER_IPV4_SRC_SHIFT 0
/* Ipv4 destination IP address */
#define EC_RFW_THASH_MASK_OUTER_IPV4_DST_MASK 0x000000F0
#define EC_RFW_THASH_MASK_OUTER_IPV4_DST_SHIFT 4
/* TCP source port */
#define EC_RFW_THASH_MASK_OUTER_TCP_SRC_PORT_MASK 0x00000300
#define EC_RFW_THASH_MASK_OUTER_TCP_SRC_PORT_SHIFT 8
/* TCP destination port */
#define EC_RFW_THASH_MASK_OUTER_TCP_DST_PORT_MASK 0x00000C00
#define EC_RFW_THASH_MASK_OUTER_TCP_DST_PORT_SHIFT 10
/* UDP source port */
#define EC_RFW_THASH_MASK_OUTER_UDP_SRC_PORT_MASK 0x00003000
#define EC_RFW_THASH_MASK_OUTER_UDP_SRC_PORT_SHIFT 12
/* UDP destination port */
#define EC_RFW_THASH_MASK_OUTER_UDP_DST_PORT_MASK 0x0000C000
#define EC_RFW_THASH_MASK_OUTER_UDP_DST_PORT_SHIFT 14

/**** thash_mask_inner_ipv6 register ****/
/* Ipv6 source IP address */
#define EC_RFW_THASH_MASK_INNER_IPV6_SRC_MASK 0x0000FFFF
#define EC_RFW_THASH_MASK_INNER_IPV6_SRC_SHIFT 0
/* Ipv6 destination IP address */
#define EC_RFW_THASH_MASK_INNER_IPV6_DST_MASK 0xFFFF0000
#define EC_RFW_THASH_MASK_INNER_IPV6_DST_SHIFT 16

/**** thash_mask_inner register ****/
/* Ipv4 source IP address */
#define EC_RFW_THASH_MASK_INNER_IPV4_SRC_MASK 0x0000000F
#define EC_RFW_THASH_MASK_INNER_IPV4_SRC_SHIFT 0
/* Ipv4 destination IP address */
#define EC_RFW_THASH_MASK_INNER_IPV4_DST_MASK 0x000000F0
#define EC_RFW_THASH_MASK_INNER_IPV4_DST_SHIFT 4
/* TCP source port */
#define EC_RFW_THASH_MASK_INNER_TCP_SRC_PORT_MASK 0x00000300
#define EC_RFW_THASH_MASK_INNER_TCP_SRC_PORT_SHIFT 8
/* TCP destination port */
#define EC_RFW_THASH_MASK_INNER_TCP_DST_PORT_MASK 0x00000C00
#define EC_RFW_THASH_MASK_INNER_TCP_DST_PORT_SHIFT 10
/* UDP source port */
#define EC_RFW_THASH_MASK_INNER_UDP_SRC_PORT_MASK 0x00003000
#define EC_RFW_THASH_MASK_INNER_UDP_SRC_PORT_SHIFT 12
/* UDP destination port */
#define EC_RFW_THASH_MASK_INNER_UDP_DST_PORT_MASK 0x0000C000
#define EC_RFW_THASH_MASK_INNER_UDP_DST_PORT_SHIFT 14

/**** def_cfg register ****/
/* Number of padding bytes to add at the beginning of each Ether ... */
#define EC_RFW_UDMA_DEF_CFG_RX_PAD_MASK  0x0000003F
#define EC_RFW_UDMA_DEF_CFG_RX_PAD_SHIFT 0

/**** queue register ****/
/* Mapping between priority and queue number */
#define EC_RFW_PRIORITY_QUEUE_MAP_MASK   0x00000003
#define EC_RFW_PRIORITY_QUEUE_MAP_SHIFT  0

/**** opt_1 register ****/
/* Default UDMA for forwarding  */
#define EC_RFW_DEFAULT_OPT_1_UDMA_MASK   0x0000000F
#define EC_RFW_DEFAULT_OPT_1_UDMA_SHIFT  0
/* Default priority for forwarding */
#define EC_RFW_DEFAULT_OPT_1_PRIORITY_MASK 0x00000700
#define EC_RFW_DEFAULT_OPT_1_PRIORITY_SHIFT 8
/* Default queue for forwarding */
#define EC_RFW_DEFAULT_OPT_1_QUEUE_MASK  0x00030000
#define EC_RFW_DEFAULT_OPT_1_QUEUE_SHIFT 16

/**** data_h register ****/
/* MAC address data  */
#define EC_FWD_MAC_DATA_H_VAL_MASK       0x0000FFFF
#define EC_FWD_MAC_DATA_H_VAL_SHIFT      0

/**** mask_h register ****/
/* MAC address mask  */
#define EC_FWD_MAC_MASK_H_VAL_MASK       0x0000FFFF
#define EC_FWD_MAC_MASK_H_VAL_SHIFT      0

/**** ctrl register ****/
/* Control value for Rx forwarding engine[0] - Drop indication[2 ... */
#define EC_FWD_MAC_CTRL_RX_VAL_MASK      0x000001FF
#define EC_FWD_MAC_CTRL_RX_VAL_SHIFT     0

/* Drop indication */
#define EC_FWD_MAC_CTRL_RX_VAL_DROP		(1 << 0)

/* control table command input */
#define EC_FWD_MAC_CTRL_RX_VAL_CTRL_CMD_MASK	0x00000006
#define EC_FWD_MAC_CTRL_RX_VAL_CTRL_CMD_SHIFT	1

/* UDMA selection */
#define EC_FWD_MAC_CTRL_RX_VAL_UDMA_MASK	0x000000078
#define EC_FWD_MAC_CTRL_RX_VAL_UDMA_SHIFT	3

/* queue number */
#define EC_FWD_MAC_CTRL_RX_VAL_QID_MASK		0x00000180
#define EC_FWD_MAC_CTRL_RX_VAL_QID_SHIFT	7

/* Entry is valid for Rx forwarding engine. */
#define EC_FWD_MAC_CTRL_RX_VALID		(1 << 15)
/* Control value for Tx forwarding engine */
#define EC_FWD_MAC_CTRL_TX_VAL_MASK		0x001F0000
#define EC_FWD_MAC_CTRL_TX_VAL_SHIFT		16
/* Entry is valid for Tx forwarding engine. */
#define EC_FWD_MAC_CTRL_TX_VALID		(1 << 31)

/**** uc register ****/
/* timer max value for waiting for a stream to be ready to accep ... */
#define EC_MSW_UC_TIMER_MASK             0x0000FFFF
#define EC_MSW_UC_TIMER_SHIFT            0
/* drop packet if target queue in the UDMA is full */
#define EC_MSW_UC_Q_FULL_DROP_MASK       0x000F0000
#define EC_MSW_UC_Q_FULL_DROP_SHIFT      16
/* Drop packet if timer expires */
#define EC_MSW_UC_TIMER_DROP_MASK        0x0F000000
#define EC_MSW_UC_TIMER_DROP_SHIFT       24

/**** mc register ****/
/* Timer max value for waiting for a stream to be ready to accep ... */
#define EC_MSW_MC_TIMER_MASK             0x0000FFFF
#define EC_MSW_MC_TIMER_SHIFT            0
/* Drop packet if target queue in UDMA is full. */
#define EC_MSW_MC_Q_FULL_DROP_MASK       0x000F0000
#define EC_MSW_MC_Q_FULL_DROP_SHIFT      16
/* Drop packet if timer expires. */
#define EC_MSW_MC_TIMER_DROP_MASK        0x0F000000
#define EC_MSW_MC_TIMER_DROP_SHIFT       24

/**** bc register ****/
/* Timer max value for waiting for a stream to be ready to accep ... */
#define EC_MSW_BC_TIMER_MASK             0x0000FFFF
#define EC_MSW_BC_TIMER_SHIFT            0
/* Drop packet if target queue in UDMA is full. */
#define EC_MSW_BC_Q_FULL_DROP_MASK       0x000F0000
#define EC_MSW_BC_Q_FULL_DROP_SHIFT      16
/* Drop packet if timer expires. */
#define EC_MSW_BC_TIMER_DROP_MASK        0x0F000000
#define EC_MSW_BC_TIMER_DROP_SHIFT       24

/**** in_cfg register ****/
/* Swap input bytes order */
#define EC_TSO_IN_CFG_SWAP_BYTES         (1 << 0)
/* Selects strict priority or round robin scheduling between GDM ... */
#define EC_TSO_IN_CFG_SEL_SP_RR          (1 << 1)
/* Selects scheduler numbering direction */
#define EC_TSO_IN_CFG_SEL_SCH_DIR        (1 << 2)
/* Minimum L2 packet size (not including FCS) */
#define EC_TSO_IN_CFG_L2_MIN_SIZE_MASK   0x00007F00
#define EC_TSO_IN_CFG_L2_MIN_SIZE_SHIFT  8
/* Swap input bytes order */
#define EC_TSO_IN_CFG_SP_INIT_VAL_MASK   0x000F0000
#define EC_TSO_IN_CFG_SP_INIT_VAL_SHIFT  16

/**** cache_table_addr register ****/
/* Address for accessing the table */
#define EC_TSO_CACHE_TABLE_ADDR_VAL_MASK 0x0000000F
#define EC_TSO_CACHE_TABLE_ADDR_VAL_SHIFT 0

/**** ctrl_first register ****/
/* Data to be written into the control BIS. */
#define EC_TSO_CTRL_FIRST_DATA_MASK      0x000001FF
#define EC_TSO_CTRL_FIRST_DATA_SHIFT     0
/* Mask for control bits,[0] – Use original packet header */
#define EC_TSO_CTRL_FIRST_MASK_MASK      0x01FF0000
#define EC_TSO_CTRL_FIRST_MASK_SHIFT     16

/**** ctrl_middle register ****/
/* Data to be written into the control BIS. */
#define EC_TSO_CTRL_MIDDLE_DATA_MASK     0x000001FF
#define EC_TSO_CTRL_MIDDLE_DATA_SHIFT    0
/* Mask for the control bits[0] - Use original packet header */
#define EC_TSO_CTRL_MIDDLE_MASK_MASK     0x01FF0000
#define EC_TSO_CTRL_MIDDLE_MASK_SHIFT    16

/**** ctrl_last register ****/
/* Data to be written into the control BIS. */
#define EC_TSO_CTRL_LAST_DATA_MASK       0x000001FF
#define EC_TSO_CTRL_LAST_DATA_SHIFT      0
/* Mask for the control bits[0] - Use original packet header */
#define EC_TSO_CTRL_LAST_MASK_MASK       0x01FF0000
#define EC_TSO_CTRL_LAST_MASK_SHIFT      16

/**** cfg_add_0 register ****/
/* MSS selection option:0 – MSS value is selected using MSS_sel  ... */
#define EC_TSO_CFG_ADD_0_MSS_SEL         (1 << 0)

/**** cfg_tunnel register ****/
/* enable TSO with tunneling */
#define EC_TSO_CFG_TUNNEL_EN_TUNNEL_TSO  (1 << 0)
/* Enable outer UDP checksum update */
#define EC_TSO_CFG_TUNNEL_EN_UDP_CHKSUM  (1 << 8)
/* Enable outer UDP length update */
#define EC_TSO_CFG_TUNNEL_EN_UDP_LEN     (1 << 9)
/* Enable outer Ip6  length update */
#define EC_TSO_CFG_TUNNEL_EN_IPV6_PLEN   (1 << 10)
/* Enable outer Ipv4 checksum update */
#define EC_TSO_CFG_TUNNEL_EN_IPV4_CHKSUM (1 << 11)
/* Enable outer Ipv4 Identification update */
#define EC_TSO_CFG_TUNNEL_EN_IPV4_IDEN   (1 << 12)
/* Enable outer Ipv4 length update */
#define EC_TSO_CFG_TUNNEL_EN_IPV4_TLEN   (1 << 13)

/**** mss register ****/
/* MSS value */
#define EC_TSO_SEL_MSS_VAL_MASK          0x000FFFFF
#define EC_TSO_SEL_MSS_VAL_SHIFT         0

/**** parse register ****/
/* Max number of bus beats for parsing */
#define EC_TPE_PARSE_MAX_BEATS_MASK      0x0000FFFF
#define EC_TPE_PARSE_MAX_BEATS_SHIFT     0

/**** vlan_data register ****/
/* UDMA default VLAN 1 data */
#define EC_TPM_UDMA_VLAN_DATA_DEF_1_MASK 0x0000FFFF
#define EC_TPM_UDMA_VLAN_DATA_DEF_1_SHIFT 0
/* UDMA default VLAN 2 data */
#define EC_TPM_UDMA_VLAN_DATA_DEF_2_MASK 0xFFFF0000
#define EC_TPM_UDMA_VLAN_DATA_DEF_2_SHIFT 16

/**** mac_sa_2 register ****/
/* MAC source address data [47:32] */
#define EC_TPM_UDMA_MAC_SA_2_H_VAL_MASK  0x0000FFFF
#define EC_TPM_UDMA_MAC_SA_2_H_VAL_SHIFT 0
/* Drop indication for MAC SA spoofing0 – Don't drop */
#define EC_TPM_UDMA_MAC_SA_2_DROP        (1 << 16)
/* Replace indication for MAC SA spoofing 0 - Don't replace */
#define EC_TPM_UDMA_MAC_SA_2_REPLACE     (1 << 17)

/**** etype register ****/
/* Ether type value  */
#define EC_TPM_SEL_ETYPE_VAL_MASK        0x0000FFFF
#define EC_TPM_SEL_ETYPE_VAL_SHIFT       0

/**** tx_wr_fifo register ****/
/* Max data beats that can be used in the Tx FIFO */
#define EC_TFW_TX_WR_FIFO_DATA_TH_MASK   0x0000FFFF
#define EC_TFW_TX_WR_FIFO_DATA_TH_SHIFT  0
/* Max packets that can be stored in the Tx FIFO */
#define EC_TFW_TX_WR_FIFO_INFO_TH_MASK   0xFFFF0000
#define EC_TFW_TX_WR_FIFO_INFO_TH_SHIFT  16

/**** tx_vid_table_addr register ****/
/* Address for accessing the table */
#define EC_TFW_TX_VID_TABLE_ADDR_VAL_MASK 0x00000FFF
#define EC_TFW_TX_VID_TABLE_ADDR_VAL_SHIFT 0

/**** tx_vid_table_data register ****/
/* Table data (valid only after configuring the table address re ... */
#define EC_TFW_TX_VID_TABLE_DATA_VAL_MASK 0x0000001F
#define EC_TFW_TX_VID_TABLE_DATA_VAL_SHIFT 0

/**** tx_rd_fifo register ****/
/* Read data thheshold when cut through mode is enabled. */
#define EC_TFW_TX_RD_FIFO_READ_TH_MASK   0x0000FFFF
#define EC_TFW_TX_RD_FIFO_READ_TH_SHIFT  0
/* Enable cut through operation of the Tx FIFO. */
#define EC_TFW_TX_RD_FIFO_EN_CUT_THROUGH (1 << 16)

/**** tx_checksum register ****/
/* Enable L3 checksum insertion. */
#define EC_TFW_TX_CHECKSUM_L3_EN         (1 << 0)
/* Enable L4 checksum insertion. */
#define EC_TFW_TX_CHECKSUM_L4_EN         (1 << 1)
/* Enable L4 checksum when L3 fragmentation is detected. */
#define EC_TFW_TX_CHECKSUM_L4_FRAG_EN    (1 << 2)

/**** tx_gen register ****/
/* Force forward of all Tx packets to MAC. */
#define EC_TFW_TX_GEN_FWD_ALL_TO_MAC     (1 << 0)
/* Select the Packet generator as the source of Tx packets0 – Tx ... */
#define EC_TFW_TX_GEN_SELECT_PKT_GEN     (1 << 1)

/**** tx_spf register ****/
/* Select the VID for spoofing check:[0] – Packet VID[1] – Forwa ... */
#define EC_TFW_TX_SPF_VID_SEL            (1 << 0)

/**** data_fifo register ****/
/* FIFO used value (number of entries) */
#define EC_TFW_DATA_FIFO_USED_MASK       0x0000FFFF
#define EC_TFW_DATA_FIFO_USED_SHIFT      0
/* FIFO FULL status */
#define EC_TFW_DATA_FIFO_FULL            (1 << 16)
/* FIFO EMPTY status */
#define EC_TFW_DATA_FIFO_EMPTY           (1 << 17)

/**** ctrl_fifo register ****/
/* FIFO used value (number of entries) */
#define EC_TFW_CTRL_FIFO_USED_MASK       0x0000FFFF
#define EC_TFW_CTRL_FIFO_USED_SHIFT      0
/* FIFO FULL status */
#define EC_TFW_CTRL_FIFO_FULL            (1 << 16)
/* FIFO EMPTY status */
#define EC_TFW_CTRL_FIFO_EMPTY           (1 << 17)

/**** hdr_fifo register ****/
/* FIFO used value (number of entries) */
#define EC_TFW_HDR_FIFO_USED_MASK        0x0000FFFF
#define EC_TFW_HDR_FIFO_USED_SHIFT       0
/* FIFO FULL status */
#define EC_TFW_HDR_FIFO_FULL             (1 << 16)
/* FIFO EMPTY status */
#define EC_TFW_HDR_FIFO_EMPTY            (1 << 17)

/**** uc_udma register ****/
/* Default UDMA bitmap
(MSB represents physical port) */
#define EC_TFW_UDMA_UC_UDMA_DEF_MASK     0x0000001F
#define EC_TFW_UDMA_UC_UDMA_DEF_SHIFT    0

/**** mc_udma register ****/
/* Default UDMA bitmap (MSB represents physical port.) */
#define EC_TFW_UDMA_MC_UDMA_DEF_MASK     0x0000001F
#define EC_TFW_UDMA_MC_UDMA_DEF_SHIFT    0

/**** bc_udma register ****/
/* Default UDMA bitmap (MSB represents physical port.) */
#define EC_TFW_UDMA_BC_UDMA_DEF_MASK     0x0000001F
#define EC_TFW_UDMA_BC_UDMA_DEF_SHIFT    0

/**** spf_cmd register ****/
/* Command for the VLAN spoofing00 – Ignore  mismatch */
#define EC_TFW_UDMA_SPF_CMD_VID_MASK     0x00000003
#define EC_TFW_UDMA_SPF_CMD_VID_SHIFT    0
/* Command for VLAN spoofing 00 - Ignore  mismatch */
#define EC_TFW_UDMA_SPF_CMD_MAC_MASK     0x0000000C
#define EC_TFW_UDMA_SPF_CMD_MAC_SHIFT    2

/**** fwd_dec register ****/
/* Forwarding decision control:[0] – Enable internal switch */
#define EC_TFW_UDMA_FWD_DEC_CTRL_MASK    0x000003FF
#define EC_TFW_UDMA_FWD_DEC_CTRL_SHIFT   0

/**** tx_cfg register ****/
/* Swap output byte order */
#define EC_TMI_TX_CFG_SWAP_BYTES         (1 << 0)
/* Enable forwarding to the Rx data path. */
#define EC_TMI_TX_CFG_EN_FWD_TO_RX       (1 << 1)
/* Force forwarding all packets to the MAC. */
#define EC_TMI_TX_CFG_FORCE_FWD_MAC      (1 << 2)
/* Force forwarding all packets to the MAC. */
#define EC_TMI_TX_CFG_FORCE_FWD_RX       (1 << 3)
/* Force loop back operation */
#define EC_TMI_TX_CFG_FORCE_LB           (1 << 4)

/**** ec_pause register ****/
/* Mask of pause_on [7:0] */
#define EC_EFC_EC_PAUSE_MASK_MAC_MASK    0x000000FF
#define EC_EFC_EC_PAUSE_MASK_MAC_SHIFT   0
/* Mask of gpio input [7:0] */
#define EC_EFC_EC_PAUSE_MASK_GPIO_MASK   0x0000FF00
#define EC_EFC_EC_PAUSE_MASK_GPIO_SHIFT  8

/**** ec_xoff register ****/
/* Mask 1 for XOFF [7:0]
Mask 1 for almost full indication, */
#define EC_EFC_EC_XOFF_MASK_1_MASK       0x000000FF
#define EC_EFC_EC_XOFF_MASK_1_SHIFT      0
/* Mask 2 for XOFF [7:0] Mask 2 for sampled almost full indicati ... */
#define EC_EFC_EC_XOFF_MASK_2_MASK       0x0000FF00
#define EC_EFC_EC_XOFF_MASK_2_SHIFT      8

/**** xon register ****/
/* Mask 1 for generating XON pulse, masking XOFF [0] */
#define EC_EFC_XON_MASK_1                (1 << 0)
/* Mask 2 for generating XON pulse, masking “almost full” indica ... */
#define EC_EFC_XON_MASK_2                (1 << 1)

/**** gpio register ****/
/* Mask for generating GPIO output XOFF indication from XOFF[0] */
#define EC_EFC_GPIO_MASK_1               (1 << 0)

/**** rx_fifo_af register ****/
/* Threshold */
#define EC_EFC_RX_FIFO_AF_TH_MASK        0x0000FFFF
#define EC_EFC_RX_FIFO_AF_TH_SHIFT       0

/**** rx_fifo_hyst register ****/
/* Threshold low */
#define EC_EFC_RX_FIFO_HYST_TH_LOW_MASK  0x0000FFFF
#define EC_EFC_RX_FIFO_HYST_TH_LOW_SHIFT 0
/* Threshold high */
#define EC_EFC_RX_FIFO_HYST_TH_HIGH_MASK 0xFFFF0000
#define EC_EFC_RX_FIFO_HYST_TH_HIGH_SHIFT 16

/**** stat register ****/
/* 10G MAC PFC mode, input from the 10 MAC */
#define EC_EFC_STAT_PFC_MODE             (1 << 0)

/**** ec_pfc register ****/
/* Force PFC flow control */
#define EC_EFC_EC_PFC_FORCE_MASK         0x000000FF
#define EC_EFC_EC_PFC_FORCE_SHIFT        0

/**** q_pause_0 register ****/
/* [i] – Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_0_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_0_MASK_SHIFT  0

/**** q_pause_1 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_1_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_1_MASK_SHIFT  0

/**** q_pause_2 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_2_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_2_MASK_SHIFT  0

/**** q_pause_3 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_3_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_3_MASK_SHIFT  0

/**** q_pause_4 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_4_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_4_MASK_SHIFT  0

/**** q_pause_5 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_5_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_5_MASK_SHIFT  0

/**** q_pause_6 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_6_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_6_MASK_SHIFT  0

/**** q_pause_7 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_PAUSE_7_MASK_MASK   0x0000000F
#define EC_FC_UDMA_Q_PAUSE_7_MASK_SHIFT  0

/**** q_gpio_0 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_0_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_0_MASK_SHIFT   0

/**** q_gpio_1 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_1_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_1_MASK_SHIFT   0

/**** q_gpio_2 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_2_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_2_MASK_SHIFT   0

/**** q_gpio_3 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_3_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_3_MASK_SHIFT   0

/**** q_gpio_4 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_4_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_4_MASK_SHIFT   0

/**** q_gpio_5 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_5_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_5_MASK_SHIFT   0

/**** q_gpio_6 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_6_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_6_MASK_SHIFT   0

/**** q_gpio_7 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_GPIO_7_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_GPIO_7_MASK_SHIFT   0

/**** s_pause register ****/
/* Mask of pause_on [7:0] */
#define EC_FC_UDMA_S_PAUSE_MASK_MAC_MASK 0x000000FF
#define EC_FC_UDMA_S_PAUSE_MASK_MAC_SHIFT 0
/* Mask of GPIO input  [7:0] */
#define EC_FC_UDMA_S_PAUSE_MASK_GPIO_MASK 0x0000FF00
#define EC_FC_UDMA_S_PAUSE_MASK_GPIO_SHIFT 8

/**** q_xoff_0 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_0_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_0_MASK_SHIFT   0

/**** q_xoff_1 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_1_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_1_MASK_SHIFT   0

/**** q_xoff_2 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_2_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_2_MASK_SHIFT   0

/**** q_xoff_3 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_3_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_3_MASK_SHIFT   0

/**** q_xoff_4 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_4_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_4_MASK_SHIFT   0

/**** q_xoff_5 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_5_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_5_MASK_SHIFT   0

/**** q_xoff_6 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_6_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_6_MASK_SHIFT   0

/**** q_xoff_7 register ****/
/* [i] - Mask for Q[i] */
#define EC_FC_UDMA_Q_XOFF_7_MASK_MASK    0x0000000F
#define EC_FC_UDMA_Q_XOFF_7_MASK_SHIFT   0

/**** cfg_e register ****/
/* Use MAC Tx FIFO empty status for EEE control. */
#define EC_EEE_CFG_E_USE_MAC_TX_FIFO     (1 << 0)
/* Use MAC Rx FIFO empty status for EEE control. */
#define EC_EEE_CFG_E_USE_MAC_RX_FIFO     (1 << 1)
/* Use Ethernet controller Tx FIFO empty status for EEE control */
#define EC_EEE_CFG_E_USE_EC_TX_FIFO      (1 << 2)
/* Use Ethernet controller Rx FIFO empty status for EEE control */
#define EC_EEE_CFG_E_USE_EC_RX_FIFO      (1 << 3)
/* Enable Low power signaling. */
#define EC_EEE_CFG_E_ENABLE              (1 << 4)
/* Mask output to MAC.  */
#define EC_EEE_CFG_E_MASK_MAC_EEE        (1 << 8)
/* Mask output to stop MAC interface. */
#define EC_EEE_CFG_E_MASK_EC_TMI_STOP    (1 << 9)

/**** stat_eee register ****/
/* EEE state */
#define EC_EEE_STAT_EEE_STATE_MASK       0x0000000F
#define EC_EEE_STAT_EEE_STATE_SHIFT      0
/* EEE detected */
#define EC_EEE_STAT_EEE_DET              (1 << 4)

/**** p_parse_cfg register ****/
/* MAX number of beats for packet parsing */
#define EC_MSP_P_PARSE_CFG_MAX_BEATS_MASK 0x000000FF
#define EC_MSP_P_PARSE_CFG_MAX_BEATS_SHIFT 0
/* MAX number of parsing iterations for packet parsing */
#define EC_MSP_P_PARSE_CFG_MAX_ITER_MASK 0x0000FF00
#define EC_MSP_P_PARSE_CFG_MAX_ITER_SHIFT 8

/**** p_act_table_addr register ****/
/* Address for accessing the table */
#define EC_MSP_P_ACT_TABLE_ADDR_VAL_MASK 0x0000001F
#define EC_MSP_P_ACT_TABLE_ADDR_VAL_SHIFT 0

/**** p_act_table_data_1 register ****/
/* Table data[5:0] - Offset to next protocol [bytes] [6] - Next  ... */
#define EC_MSP_P_ACT_TABLE_DATA_1_VAL_MASK 0x03FFFFFF
#define EC_MSP_P_ACT_TABLE_DATA_1_VAL_SHIFT 0

/**** p_act_table_data_2 register ****/
/* table data  [8:0] - offset to data in the packet [bits] [17:9 ... */
#define EC_MSP_P_ACT_TABLE_DATA_2_VAL_MASK 0x1FFFFFFF
#define EC_MSP_P_ACT_TABLE_DATA_2_VAL_SHIFT 0

/**** p_act_table_data_3 register ****/
/* Table data   [8:0] - Offset to data in the packet [bits]  [17 ... */
#define EC_MSP_P_ACT_TABLE_DATA_3_VAL_MASK 0x1FFFFFFF
#define EC_MSP_P_ACT_TABLE_DATA_3_VAL_SHIFT 0

/**** p_act_table_data_4 register ****/
/* Table data [7:0] - Offset to the header length location in th ... */
#define EC_MSP_P_ACT_TABLE_DATA_4_VAL_MASK 0x0FFFFFFF
#define EC_MSP_P_ACT_TABLE_DATA_4_VAL_SHIFT 0

/**** p_act_table_data_6 register ****/
/* Table data [0] - Wr header length [10:1] - Write header lengt ... */
#define EC_MSP_P_ACT_TABLE_DATA_6_VAL_MASK 0x007FFFFF
#define EC_MSP_P_ACT_TABLE_DATA_6_VAL_SHIFT 0

/**** p_res_in register ****/
/* Selector for input parse_en 0 - Input vector 1 - Default valu ... */
#define EC_MSP_P_RES_IN_SEL_PARSE_EN     (1 << 0)
/* Selector for input protocol_index  0 - Input vector  1 - Defa ... */
#define EC_MSP_P_RES_IN_SEL_PROT_INDEX   (1 << 1)
/* Selector for input hdr_offset  0 - input vector  1 - Default  ... */
#define EC_MSP_P_RES_IN_SEL_HDR_OFFSET   (1 << 2)

/**** h_hdr_len register ****/
/* Value for selecting table 1 */
#define EC_MSP_P_H_HDR_LEN_TABLE_1_MASK  0x000000FF
#define EC_MSP_P_H_HDR_LEN_TABLE_1_SHIFT 0
/* Value for selecting table 2 */
#define EC_MSP_P_H_HDR_LEN_TABLE_2_MASK  0x00FF0000
#define EC_MSP_P_H_HDR_LEN_TABLE_2_SHIFT 16

/**** p_comp_data register ****/
/* Data 1 for comparison */
#define EC_MSP_C_P_COMP_DATA_DATA_1_MASK 0x0000FFFF
#define EC_MSP_C_P_COMP_DATA_DATA_1_SHIFT 0
/* Data 2 for comparison 
[18:16] - Stage 
[24:19] - Branch ID */
#define EC_MSP_C_P_COMP_DATA_DATA_2_MASK 0x01FF0000
#define EC_MSP_C_P_COMP_DATA_DATA_2_SHIFT 16

/**** p_comp_mask register ****/
/* Data 1 for comparison */
#define EC_MSP_C_P_COMP_MASK_DATA_1_MASK 0x0000FFFF
#define EC_MSP_C_P_COMP_MASK_DATA_1_SHIFT 0
/* Data 2 for comparison  
[18:16] - Stage 
[24:19] - Branch ID */
#define EC_MSP_C_P_COMP_MASK_DATA_2_MASK 0x01FF0000
#define EC_MSP_C_P_COMP_MASK_DATA_2_SHIFT 16

/**** p_comp_ctrl register ****/
/* Output result value */
#define EC_MSP_C_P_COMP_CTRL_RES_MASK    0x0000001F
#define EC_MSP_C_P_COMP_CTRL_RES_SHIFT   0
/* Compare command for the data_1 field 00 - Compare 01 - <= 10  ... */
#define EC_MSP_C_P_COMP_CTRL_CMD_1_MASK  0x00030000
#define EC_MSP_C_P_COMP_CTRL_CMD_1_SHIFT 16
/* Compare command for the data_2 field 00 - Compare 01 - <= 10  ... */
#define EC_MSP_C_P_COMP_CTRL_CMD_2_MASK  0x000C0000
#define EC_MSP_C_P_COMP_CTRL_CMD_2_SHIFT 18
/* Entry is valid */
#define EC_MSP_C_P_COMP_CTRL_VALID       (1 << 31)

/**** pkt_cfg register ****/
/* Number of bytes in RoCE packet LRH header */
#define EC_CRCE_PKT_CFG_ROCE_HEAD_SIZE_MASK 0x000000FF
#define EC_CRCE_PKT_CFG_ROCE_HEAD_SIZE_SHIFT 0
/* Number of bytes in FCoE packet header */
#define EC_CRCE_PKT_CFG_FCOE_HEAD_SIZE_MASK 0x0000FF00
#define EC_CRCE_PKT_CFG_FCOE_HEAD_SIZE_SHIFT 8
/* Number of bytes to ignore in the end of RoCE packet */
#define EC_CRCE_PKT_CFG_ROCE_TRAIL_SIZE_MASK 0x000F0000
#define EC_CRCE_PKT_CFG_ROCE_TRAIL_SIZE_SHIFT 16
/* Number of bytes to ignore in the end of FCoE packet */
#define EC_CRCE_PKT_CFG_FCOE_TRAIL_SIZE_MASK 0x00F00000
#define EC_CRCE_PKT_CFG_FCOE_TRAIL_SIZE_SHIFT 20
/* Select the header that will be used for the CRC engine when a ... */
#define EC_CRCE_PKT_CFG_HDR_SEL          (1 << 24)

/**** swap_en register ****/
/* Enable byte swap on input data of RoCE packet */
#define EC_CRCE_SWAP_EN_ROCE_DATA_BYTE_SWAP (1 << 0)
/* Enable bit swap (within each byte) on input data of RoCE pack ... */
#define EC_CRCE_SWAP_EN_ROCE_DATA_BIT_SWAP (1 << 1)
/* Enable byte swap on crc32 output of RoCE packet */
#define EC_CRCE_SWAP_EN_ROCE_CRC32_BYTE_SWAP (1 << 2)
/* Enable bit swap (within each byte) on crc32 output of RoCE pa ... */
#define EC_CRCE_SWAP_EN_ROCE_CRC32_BIT_SWAP (1 << 3)
/* Enable bit complement on crc32 output of RoCE packet */
#define EC_CRCE_SWAP_EN_ROCE_CRC32_BIT_COMP (1 << 4)
/* Enable byte swap on input data of FCoE packet */
#define EC_CRCE_SWAP_EN_FCOE_DATA_BYTE_SWAP (1 << 5)
/* Enable bit swap (within each byte) on input data of FCoE pack ... */
#define EC_CRCE_SWAP_EN_FCOE_DATA_BIT_SWAP (1 << 6)
/* Enable byte swap on crc32 output of FCoE packet */
#define EC_CRCE_SWAP_EN_FCOE_CRC32_BYTE_SWAP (1 << 7)
/* Enable bit swap (within each byte) on crc32 output of FCoE pa ... */
#define EC_CRCE_SWAP_EN_FCOE_CRC32_BIT_SWAP (1 << 8)
/* Enable bit complement on crc32 output of FCoE packet */
#define EC_CRCE_SWAP_EN_FCOE_CRC32_BIT_COMP (1 << 9)

/**** gen_cfg register ****/
/* default result for undetected protocols. */
#define EC_CRCE_GEN_CFG_RES_DEF          (1 << 0)
/* Enable CRC (RoCE/FCoE) insertion in the TX data path(override ... */
#define EC_CRCE_GEN_CFG_EN_TX_CRC_INS    (1 << 4)
/* Enable RoCE insertion in the TX data path */
#define EC_CRCE_GEN_CFG_EN_TX_ROCE_INS   (1 << 5)
/* Enable FcoE insertion in the TX data path */
#define EC_CRCE_GEN_CFG_EN_TX_FCOE_INS   (1 << 6)

/**** wol_en register ****/
/* Interrupt enable WoL MAC DA Unicast detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_UNICAST  (1 << 0)
/* Interrupt enable WoL L2 Multicast detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_MULTICAST (1 << 1)
/* Interrupt enable WoL L2 Broadcast detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_BROADCAST (1 << 2)
/* Interrupt enable WoL IPv4 detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_IPV4     (1 << 3)
/* Interrupt enable WoL IPv6 detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_IPV6     (1 << 4)
/* Interrupt enable WoL EtherType+MAC DA detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_ETHERTYPE_DA (1 << 5)
/* Interrupt enable WoL EtherType+L2 Broadcast detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_ETHERTYPE_BC (1 << 6)
/* Interrupt enable WoL parser detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_PARSER   (1 << 7)
/* Interrupt enable WoL magic detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_MAGIC    (1 << 8)
/* Interrupt enable WoL magic+password detected  packet */
#define EC_WOL_WOL_EN_INTRPT_EN_MAGIC_PSWD (1 << 9)
/* Forward enable WoL MAC DA Unicast detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_UNICAST    (1 << 16)
/* Forward enable WoL L2 Multicast detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_MULTICAST  (1 << 17)
/* Forward enable WoL L2 Broadcast detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_BROADCAST  (1 << 18)
/* Forward enable WoL IPv4 detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_IPV4       (1 << 19)
/* Forward enable WoL IPv6 detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_IPV6       (1 << 20)
/* Forward enable WoL EtherType+MAC DA detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_ETHERTYPE_DA (1 << 21)
/* Forward enable WoL EtherType+L2 Broadcast detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_ETHERTYPE_BC (1 << 22)
/* Forward enable WoL parser detected  packet */
#define EC_WOL_WOL_EN_FWRD_EN_PARSER     (1 << 23)

/**** magic_pswd_h register ****/
/* Password for magic_password packet detection - bits 47:32 */
#define EC_WOL_MAGIC_PSWD_H_VAL_MASK     0x0000FFFF
#define EC_WOL_MAGIC_PSWD_H_VAL_SHIFT    0

/**** ethertype register ****/
/* Configured EtherType 1 for WoL EtherType_da/EtherType_bc pack ... */
#define EC_WOL_ETHERTYPE_VAL_1_MASK      0x0000FFFF
#define EC_WOL_ETHERTYPE_VAL_1_SHIFT     0
/* Configured EtherType 2 for WoL EtherType_da/EtherType_bc pack ... */
#define EC_WOL_ETHERTYPE_VAL_2_MASK      0xFFFF0000
#define EC_WOL_ETHERTYPE_VAL_2_SHIFT     16

#define EC_PTH_SYSTEM_TIME_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_SYSTEM_TIME_SUBSECONDS_LSB_VAL_SHIFT 14

#define EC_PTH_CLOCK_PERIOD_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_CLOCK_PERIOD_LSB_VAL_SHIFT 14

/**** int_update_ctrl register ****/
/* 1'b0 - update on next ingress trigger1'b1 - update on next wr ... */
#define EC_PTH_INT_UPDATE_CTRL_UPDATE_TRIG (1 << 0)
/* 3’b000 – set system time according to the value in SYSTEM_INT ... */
#define EC_PTH_INT_UPDATE_CTRL_UPDATE_METHOD_MASK 0x0000000E
#define EC_PTH_INT_UPDATE_CTRL_UPDATE_METHOD_SHIFT 1
/* 1'b1 - next update writes subseconds field1'b0 - next update  ... */
#define EC_PTH_INT_UPDATE_CTRL_SUBSECOND_MASK (1 << 4)
/* 1'b1 - next update writes subseconds1'b0 - next update doesn' ... */
#define EC_PTH_INT_UPDATE_CTRL_SECOND_MASK (1 << 5)

#define EC_PTH_INT_UPDATE_CTRL_INT_TRIG_EN (1 << 16)

#define EC_PTH_INT_UPDATE_CTRL_PULSE_LEVEL_N (1 << 17)

#define EC_PTH_INT_UPDATE_CTRL_POLARITY  (1 << 18)

/**** int_update_subseconds_lsb register ****/

#define EC_PTH_INT_UPDATE_SUBSECONDS_LSB_RESERVED_13_0_MASK 0x00003FFF
#define EC_PTH_INT_UPDATE_SUBSECONDS_LSB_RESERVED_13_0_SHIFT 0

#define EC_PTH_INT_UPDATE_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_INT_UPDATE_SUBSECONDS_LSB_VAL_SHIFT 14
/* 3’b000 – set system time according to the value in SYSTEM_INT ... */
#define EC_PTH_EXT_UPDATE_CTRL_UPDATE_METHOD_MASK 0x0000000E
#define EC_PTH_EXT_UPDATE_CTRL_UPDATE_METHOD_SHIFT 1
/* 1'b1 - next update writes subseconds field1'b0 - next update  ... */
#define EC_PTH_EXT_UPDATE_CTRL_SUBSECOND_MASK (1 << 4)
/* 1'b1 - next update writes subseconds1'b0 - next update doesn' ... */
#define EC_PTH_EXT_UPDATE_CTRL_SECOND_MASK (1 << 5)

#define EC_PTH_EXT_UPDATE_CTRL_EXT_TRIG_EN_MASK 0x00001F00
#define EC_PTH_EXT_UPDATE_CTRL_EXT_TRIG_EN_SHIFT 8

#define EC_PTH_EXT_UPDATE_CTRL_PULSE_LEVEL_N_MASK 0x001F0000
#define EC_PTH_EXT_UPDATE_CTRL_PULSE_LEVEL_N_SHIFT 16

#define EC_PTH_EXT_UPDATE_CTRL_POLARITY_MASK 0x1F000000
#define EC_PTH_EXT_UPDATE_CTRL_POLARITY_SHIFT 24

/**** ext_update_subseconds_lsb register ****/

#define EC_PTH_EXT_UPDATE_SUBSECONDS_LSB_RESERVED_13_0_MASK 0x00003FFF
#define EC_PTH_EXT_UPDATE_SUBSECONDS_LSB_RESERVED_13_0_SHIFT 0

#define EC_PTH_EXT_UPDATE_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_EXT_UPDATE_SUBSECONDS_LSB_VAL_SHIFT 14

#define EC_PTH_READ_COMPENSATION_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_READ_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT 14

#define EC_PTH_INT_WRITE_COMPENSATION_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_INT_WRITE_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT 14

#define EC_PTH_EXT_WRITE_COMPENSATION_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_EXT_WRITE_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT 14

#define EC_PTH_SYNC_COMPENSATION_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_SYNC_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT 14

/**** trigger_ctrl register ****/
/* 1'b1 - trigger enabled
1'b0 - trigger disabled */
#define EC_PTH_EGRESS_TRIGGER_CTRL_EN    (1 << 0)
/* 1'b1 - periodic trigger
1'b0 - single event trigger */
#define EC_PTH_EGRESS_TRIGGER_CTRL_PERIODIC (1 << 1)
/* 1'b0 - regular polarity
1'b1 - negative polarity */
#define EC_PTH_EGRESS_TRIGGER_CTRL_POLARITY (1 << 2)
/* periodic pulse period (miliseconds) */
#define EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SUBSEC_MASK 0x00FFFFF0
#define EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SUBSEC_SHIFT 4
/* periodic pulse period (seconds) */
#define EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SEC_MASK 0xFF000000
#define EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SEC_SHIFT 24

/**** trigger_subseconds_lsb register ****/

#define EC_PTH_EGRESS_TRIGGER_SUBSECONDS_LSB_RESERVED_13_0_MASK 0x00003FFF
#define EC_PTH_EGRESS_TRIGGER_SUBSECONDS_LSB_RESERVED_13_0_SHIFT 0

#define EC_PTH_EGRESS_TRIGGER_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_EGRESS_TRIGGER_SUBSECONDS_LSB_VAL_SHIFT 14

/**** pulse_width_subseconds_lsb register ****/

#define EC_PTH_EGRESS_PULSE_WIDTH_SUBSECONDS_LSB_RESERVED_13_0_MASK 0x00003FFF
#define EC_PTH_EGRESS_PULSE_WIDTH_SUBSECONDS_LSB_RESERVED_13_0_SHIFT 0

#define EC_PTH_EGRESS_PULSE_WIDTH_SUBSECONDS_LSB_VAL_MASK 0xFFFFC000
#define EC_PTH_EGRESS_PULSE_WIDTH_SUBSECONDS_LSB_VAL_SHIFT 14

/**** roce_gen_cfg_1 register ****/
/* Ignore number of segments check for Middle packets */
#define EC_ROCE_ROCE_GEN_CFG_1_NUM_SEG_MIDDLE_IGNORE (1 << 0)
/* Ignore number of segments check for Last packets */
#define EC_ROCE_ROCE_GEN_CFG_1_NUM_SEG_LAST_IGNORE (1 << 1)
/* Ignore TVER field check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_TVER (1 << 4)
/* Ignore NXTHDR field check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_NXTHDR (1 << 5)
/* Ignore IPVER field check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_IPVER (1 << 6)
/* Ignore ROCE L3 protocol index check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_L3_INDEX_1_MASK 0x00000700
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_L3_INDEX_1_SHIFT 8
/* Ignore ROCE L3 protocol index check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_L3_INDEX_2_MASK 0x00007000
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_L3_INDEX_2_SHIFT 12
/* Ignore ROCE L4 protocol index check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_L4_INDEX_MASK 0x00070000
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_L4_INDEX_SHIFT 16
/* Ignore ROCE NXTHDR check */
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_NXTHDR_MASK 0x00700000
#define EC_ROCE_ROCE_GEN_CFG_1_IGNORE_ROCE_NXTHDR_SHIFT 20
/* Enable each one of the 3 RoCE packet types */
#define EC_ROCE_ROCE_GEN_CFG_1_EN_ROCE_PKT_TYPE_MASK 0x07000000
#define EC_ROCE_ROCE_GEN_CFG_1_EN_ROCE_PKT_TYPE_SHIFT 24
/* Select the header that will be used for the RoCE acceleration ... */
#define EC_ROCE_ROCE_GEN_CFG_1_HDR_SEL   (1 << 28)

/**** rr_err register ****/
/* Mask error for FSM state transition to IDLE */
#define EC_ROCE_RR_ERR_MASK_MASK         0x003FFFFF
#define EC_ROCE_RR_ERR_MASK_SHIFT        0

/**** rr_qual register ****/
/* Mask error for packet qualification */
#define EC_ROCE_RR_QUAL_MASK_MASK        0x003FFFFF
#define EC_ROCE_RR_QUAL_MASK_SHIFT       0

/**** rr_err_cnt register ****/
/* Mask error for error counter */
#define EC_ROCE_RR_ERR_CNT_MASK_MASK     0x003FFFFF
#define EC_ROCE_RR_ERR_CNT_MASK_SHIFT    0

/**** rr_err_int register ****/
/* Mask error for interrupt generation */
#define EC_ROCE_RR_ERR_INT_MASK_MASK     0x003FFFFF
#define EC_ROCE_RR_ERR_INT_MASK_SHIFT    0

/**** roce_gen_en_1 register ****/
/* Enable READ RESPONSE acceleration */
#define EC_ROCE_ROCE_GEN_EN_1_EN_RR      (1 << 0)
/* Enable WRITE REQUEST acceleration */
#define EC_ROCE_ROCE_GEN_EN_1_EN_WR      (1 << 1)
/* Enable UDMA selection by RoCE engine */
#define EC_ROCE_ROCE_GEN_EN_1_EN_UDMA_SEL (1 << 4)
/* Enable QUEUE selection by RoCE engine */
#define EC_ROCE_ROCE_GEN_EN_1_EN_QUEUE_SEL (1 << 5)
/* Enable header split */
#define EC_ROCE_ROCE_GEN_EN_1_EN_HEADER_SPLIT (1 << 6)
/* Enable direct data placement */
#define EC_ROCE_ROCE_GEN_EN_1_EN_DIRECT_DATA_PLACEMENT (1 << 7)
/* Enable EOP data trimming (ICRC) */
#define EC_ROCE_ROCE_GEN_EN_1_EN_EOP_DATA_TRIM (1 << 8)
/* Enable the RR FIFO */
#define EC_ROCE_ROCE_GEN_EN_1_RR_FIFO_EN (1 << 16)
/* Load the size of the RR FIFO (the size of each QP FIFO) */
#define EC_ROCE_ROCE_GEN_EN_1_RR_FIFO_SIZE_LOAD (1 << 17)

/**** roce_hdr register ****/
/* NXTHDR field of the GRH */
#define EC_ROCE_ROCE_HDR_NXTHDR_MASK     0x000000FF
#define EC_ROCE_ROCE_HDR_NXTHDR_SHIFT    0
/* TVER field of the BTH */
#define EC_ROCE_ROCE_HDR_IPVER_MASK      0x00000F00
#define EC_ROCE_ROCE_HDR_IPVER_SHIFT     8
/* IPVER field of the GRH */
#define EC_ROCE_ROCE_HDR_TVER_MASK       0x0000F000
#define EC_ROCE_ROCE_HDR_TVER_SHIFT      12

/**** qp_val register ****/
/* DATA for comparison */
#define EC_ROCE_QP_DB_QP_VAL_DATA_MASK   0x00FFFFFF
#define EC_ROCE_QP_DB_QP_VAL_DATA_SHIFT  0
/* Entry is valid */
#define EC_ROCE_QP_DB_QP_VAL_VALID       (1 << 31)

/**** roce_qp_cfg_1 register ****/
/* QP service type:[3] – ignore service type[2:0] – service type ... */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_SERVICE_TYPE_MASK 0x0000000F
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_SERVICE_TYPE_SHIFT 0
/* Enable RR address translation for this QP */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_EN_RR_TRANSLATION (1 << 4)
/* Enable RR address translation for this QP */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_EN_WR_TRANSLATION (1 << 5)
/* UDMA value */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_UDMA_MASK 0x000F0000
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_UDMA_SHIFT 16
/* Queue value */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_QUEUE_MASK 0x00300000
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_QUEUE_SHIFT 20
/* UDMA selector */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_UDMA_SEL_MASK 0x03000000
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_UDMA_SEL_SHIFT 24
/* Queue selector */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_QUEUE_SEL_MASK 0x0C000000
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_QUEUE_SEL_SHIFT 26
/* MTU shift for read request segment count calculation */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_MTU_SHIFT_MASK 0xF0000000
#define EC_ROCE_QP_DB_ROCE_QP_CFG_1_MTU_SHIFT_SHIFT 28

/**** roce_qp_cfg_2 register ****/
/* Ignore Syndrome check result */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_2_IGNORE_SYNDROME (1 << 0)
/* Ignore RKEY check result */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_2_IGNORE_RKEY (1 << 1)
/* Ignore PKEY check result */
#define EC_ROCE_QP_DB_ROCE_QP_CFG_2_IGNORE_PKEY (1 << 2)

/**** rr_ctrl register ****/
/* Activate the READ RESPONSE state machine */
#define EC_ROCE_QP_DB_RR_CTRL_ACTIVATE   (1 << 0)
/* Force the READ RESPONSE state machine to change from ACTIVE s ... */
#define EC_ROCE_QP_DB_RR_CTRL_DEACTIVATE_FROM_ACTIVE (1 << 4)
/* Force the READ RESPONSE state machine to change from MESSAGE  ... */
#define EC_ROCE_QP_DB_RR_CTRL_DEACTIVATE_FROM_MESSAGE (1 << 8)
/* Enable READ RESPONSE acceleration */
#define EC_ROCE_QP_DB_RR_CTRL_ENABLE     (1 << 16)
/* Enable READ RESPONSE direct data placement */
#define EC_ROCE_QP_DB_RR_CTRL_ENABLE_DDP (1 << 17)
/* Enable READ RESPONSE direct UDMA override */
#define EC_ROCE_QP_DB_RR_CTRL_ENABLE_UDMA_OR (1 << 18)
/* Enable READ RESPONSE direct Queue override */
#define EC_ROCE_QP_DB_RR_CTRL_ENABLE_QUEUE_OR (1 << 19)

/**** wr_ctrl register ****/
/* Activate the READ RESPONSE state machine */
#define EC_ROCE_QP_DB_WR_CTRL_ACTIVATE   (1 << 0)
/* Enable the transition from IDLE to SYNC */
#define EC_ROCE_QP_DB_WR_CTRL_SYNC       (1 << 1)
/* Force the READ RESPONSE state machine to change from ACTIVE s ... */
#define EC_ROCE_QP_DB_WR_CTRL_DEACTIVATE_FROM_ACTIVE (1 << 4)
/* Force the READ RESPONSE state machine to change from MESSAGE  ... */
#define EC_ROCE_QP_DB_WR_CTRL_DEACTIVATE_FROM_MESSAGE (1 << 8)
/* Force the READ RESPONSE state machine to change from SYNC sta ... */
#define EC_ROCE_QP_DB_WR_CTRL_DEACTIVATE_FROM_SYNC (1 << 12)
/* Enable WRITE REQUEST acceleration */
#define EC_ROCE_QP_DB_WR_CTRL_ENABLE     (1 << 16)
/* Enable WRITE REQUEST direct data placement */
#define EC_ROCE_QP_DB_WR_CTRL_ENABLE_DDP (1 << 17)
/* Enable WRITE REQUESTE direct UDMA override */
#define EC_ROCE_QP_DB_WR_CTRL_ENABLE_UDMA_OR (1 << 18)
/* Enable WRITE REQUEST direct Queue override */
#define EC_ROCE_QP_DB_WR_CTRL_ENABLE_QUEUE_OR (1 << 19)

/**** wr_psn register ****/
/* PSN value
(write only when the QP WR is in IDLE state */
#define EC_ROCE_QP_DB_WR_PSN_VAL_MASK    0x00FFFFFF
#define EC_ROCE_QP_DB_WR_PSN_VAL_SHIFT   0

/**** wr_psn_last_sw register ****/
/* PSN value */
#define EC_ROCE_QP_DB_WR_PSN_LAST_SW_VAL_MASK 0x00FFFFFF
#define EC_ROCE_QP_DB_WR_PSN_LAST_SW_VAL_SHIFT 0

/**** wr_psn_last_hw register ****/
/* PSN value */
#define EC_ROCE_QP_DB_WR_PSN_LAST_HW_VAL_MASK 0x00FFFFFF
#define EC_ROCE_QP_DB_WR_PSN_LAST_HW_VAL_SHIFT 0

/**** wr_psn_hw register ****/
/* PSN value */
#define EC_ROCE_QP_DB_WR_PSN_HW_VAL_MASK 0x00FFFFFF
#define EC_ROCE_QP_DB_WR_PSN_HW_VAL_SHIFT 0

/**** p_key register ****/
/* value */
#define EC_ROCE_QP_DB_P_KEY_VAL_MASK     0x0000FFFF
#define EC_ROCE_QP_DB_P_KEY_VAL_SHIFT    0

/**** roce_status register ****/
/* READ RESPONSE state */
#define EC_ROCE_QP_DB_ROCE_STATUS_RR_STATE_MASK 0x00000007
#define EC_ROCE_QP_DB_ROCE_STATUS_RR_STATE_SHIFT 0
/* WRITE REQUEST state */
#define EC_ROCE_QP_DB_ROCE_STATUS_WR_STATE_MASK 0x000000F0
#define EC_ROCE_QP_DB_ROCE_STATUS_WR_STATE_SHIFT 4

/**** rr_fifo_3 register ****/
/* Expected READ RESPONSE PSN */
#define EC_ROCE_QP_DB_RR_FIFO_3_RR_PSN_MASK 0x00FFFFFF
#define EC_ROCE_QP_DB_RR_FIFO_3_RR_PSN_SHIFT 0
/* Address requires translation */
#define EC_ROCE_QP_DB_RR_FIFO_3_NEED_TRANSLATION (1 << 31)

/**** rr_fifo_4 register ****/
/* Number of segments for this READ REQUEST */
#define EC_ROCE_QP_DB_RR_FIFO_4_NUM_OF_SEG_MASK 0x00FFFFFF
#define EC_ROCE_QP_DB_RR_FIFO_4_NUM_OF_SEG_SHIFT 0

/**** rr_fifo_status register ****/
/* FIFO status – used */
#define EC_ROCE_QP_DB_RR_FIFO_STATUS_USED_MASK 0x0000007F
#define EC_ROCE_QP_DB_RR_FIFO_STATUS_USED_SHIFT 0
/* FIFO status – empty */
#define EC_ROCE_QP_DB_RR_FIFO_STATUS_EMPTY (1 << 16)
/* FIFO status – full */
#define EC_ROCE_QP_DB_RR_FIFO_STATUS_FULL (1 << 20)

/**** rr_fifo_ctrl register ****/
/* FIFO reset */
#define EC_ROCE_QP_DB_RR_FIFO_CTRL_RESET (1 << 0)
/* FIFO reset */
#define EC_ROCE_QP_DB_RR_FIFO_CTRL_SIZE_MASK 0x00007F00
#define EC_ROCE_QP_DB_RR_FIFO_CTRL_SIZE_SHIFT 8

#ifdef __cplusplus
}
#endif

#endif /* __AL_HAL_EC_REG_H */

/** @} end of ... group */
