#ifndef MY_ABC_HERE
#define MY_ABC_HERE
#endif
 
#include "al_hal_eth.h"
#include "al_hal_udma_iofic.h"
#include "al_hal_udma_config.h"
#include "al_hal_eth_ec_regs.h"
#include "al_hal_eth_mac_regs.h"
#include "al_hal_unit_adapter_regs.h"

#define AL_ETH_EEE_STOP_CNT		100000

#define AL_ETH_TX_PKT_UDMA_FLAGS	(AL_ETH_TX_FLAGS_NO_SNOOP | \
					 AL_ETH_TX_FLAGS_INT)

#define AL_ETH_TX_PKT_META_FLAGS	(AL_ETH_TX_FLAGS_IPV4_L3_CSUM | \
					 AL_ETH_TX_FLAGS_L4_CSUM |	\
					 AL_ETH_TX_FLAGS_L4_PARTIAL_CSUM |	\
					 AL_ETH_TX_FLAGS_L2_MACSEC_PKT | \
					 AL_ETH_TX_FLAGS_L2_DIS_FCS |\
					 AL_ETH_TX_FLAGS_TSO |\
					 AL_ETH_TX_FLAGS_TS)

#define AL_ETH_TX_SRC_VLAN_CNT_MASK	3
#define AL_ETH_TX_SRC_VLAN_CNT_SHIFT	5
#define AL_ETH_TX_L4_PROTO_IDX_MASK	0x1F
#define AL_ETH_TX_L4_PROTO_IDX_SHIFT	8
#define AL_ETH_TX_TUNNEL_MODE_SHIFT		18
#define AL_ETH_TX_OUTER_L3_PROTO_SHIFT		20
#define AL_ETH_TX_VLAN_MOD_ADD_SHIFT		22
#define AL_ETH_TX_VLAN_MOD_DEL_SHIFT		24
#define AL_ETH_TX_VLAN_MOD_E_SEL_SHIFT		26
#define AL_ETH_TX_VLAN_MOD_VID_SEL_SHIFT	28
#define AL_ETH_TX_VLAN_MOD_PBIT_SEL_SHIFT	30

#define AL_ETH_TX_META_STORE			(1 << 21)
#define AL_ETH_TX_META_L3_LEN_MASK		0xff
#define AL_ETH_TX_META_L3_OFF_MASK		0xff
#define AL_ETH_TX_META_L3_OFF_SHIFT		8
#define AL_ETH_TX_META_MSS_LSB_VAL_SHIFT	22
#define AL_ETH_TX_META_MSS_MSB_TS_VAL_SHIFT	16
#define AL_ETH_TX_META_OUTER_L3_LEN_MASK	0x1f
#define AL_ETH_TX_META_OUTER_L3_LEN_SHIFT	24
#define AL_ETH_TX_META_OUTER_L3_OFF_HIGH_MASK	0x18
#define AL_ETH_TX_META_OUTER_L3_OFF_HIGH_SHIFT	10
#define AL_ETH_TX_META_OUTER_L3_OFF_LOW_MASK	0x07
#define AL_ETH_TX_META_OUTER_L3_OFF_LOW_SHIFT	29

#define AL_ETH_TX_MACSEC_SIGN_SHIFT			  0		 
#define AL_ETH_TX_MACSEC_ENCRYPT_SHIFT			  1		 
#define AL_ETH_TX_MACSEC_AN_LSB_SHIFT			  2		 
#define AL_ETH_TX_MACSEC_AN_MSB_SHIFT			  3
#define AL_ETH_TX_MACSEC_SC_LSB_SHIFT			  4		 
#define AL_ETH_TX_MACSEC_SC_MSB_SHIFT			  9
#define AL_ETH_TX_MACSEC_SECURED_PYLD_LEN_LSB_SHIFT	 10		 
#define AL_ETH_TX_MACSEC_SECURED_PYLD_LEN_MSB_SHIFT	 23

#define AL_ETH_RX_L3_PROTO_IDX_MASK	0x1F
#define AL_ETH_RX_SRC_VLAN_CNT_MASK	3
#define AL_ETH_RX_SRC_VLAN_CNT_SHIFT	5
#define AL_ETH_RX_L4_PROTO_IDX_MASK	0x1F
#define AL_ETH_RX_L4_PROTO_IDX_SHIFT	8

#define AL_ETH_RX_HASH_SHIFT		16
#define AL_ETH_RX_HASH_MASK		(0xffff << AL_ETH_RX_HASH_SHIFT)
#define AL_S2M_DESC_LEN2_SHIFT		16
#define AL_S2M_DESC_LEN2_MASK		(0xff << AL_S2M_DESC_LEN2_SHIFT)

#define ETH_MAC_GEN_LED_CFG_BLINK_TIMER_VAL 5
#define ETH_MAC_GEN_LED_CFG_ACT_TIMER_VAL 7

typedef struct {
	uint32_t ctrl_meta;
	uint32_t len;
	uint32_t word2;
	uint32_t word3;
} al_eth_rx_cdesc;

struct al_eth_epe_p_reg_entry {
	uint32_t data;
	uint32_t mask;
	uint32_t ctrl;
};

#define AL_ETH_MDIO_DELAY_PERIOD	1  
#define AL_ETH_MDIO_DELAY_COUNT		150  

#define AL_ETH_EPE_ENTRIES_NUM 26
static struct al_eth_epe_p_reg_entry al_eth_epe_p_regs[AL_ETH_EPE_ENTRIES_NUM] = {
	{ 0x0, 0x0, 0x0 },
	{ 0x0, 0x0, 0x1 },
	{ 0x0, 0x0, 0x2 },
	{ 0x0, 0x0, 0x3 },
	{ 0x18100, 0xFFFFF, 0x80000004 },
	{ 0x188A8, 0xFFFFF, 0x80000005 },
	{ 0x99100, 0xFFFFF, 0x80000006 },
	{ 0x98100, 0xFFFFF, 0x80000007 },
	{ 0x10800, 0x7FFFF, 0x80000008 },
	{ 0x20000, 0x73FFF, 0x80000009 },
	{ 0x20000, 0x70000, 0x8000000A },
	{ 0x186DD, 0x7FFFF, 0x8000000B },
	{ 0x30600, 0x7FF00, 0x8000000C },
	{ 0x31100, 0x7FF00, 0x8000000D },
	{ 0x32F00, 0x7FF00, 0x8000000E },
	{ 0x32900, 0x7FF00, 0x8000000F },
	{ 0x105DC, 0x7FFFF, 0x80010010 },
	{ 0x188E5, 0x7FFFF, 0x80000011 },
	{ 0x72000, 0x72000, 0x80000012 },
	{ 0x70000, 0x72000, 0x80000013 },
	{ 0x46558, 0x7FFFF, 0x80000001 },
	{ 0x18906, 0x7FFFF, 0x80000015 },
	{ 0x18915, 0x7FFFF, 0x80000016 },
	{ 0x31B00, 0x7FF00, 0x80000017 },
	{ 0x30400, 0x7FF00, 0x80000018 },
	{ 0x0, 0x0, 0x8000001F }
};

struct al_eth_epe_control_entry {
	uint32_t data[6];
};

static struct al_eth_epe_control_entry al_eth_epe_control_table[AL_ETH_EPE_ENTRIES_NUM] = {
	{{ 0x2800000, 0x0, 0x0, 0x0, 0x1, 0x400000 }},
	{{ 0x280004C, 0x746000, 0xA46030, 0xE00000, 0x2, 0x400000 }},
	{{ 0x2800054, 0x746000, 0xA46030, 0x1600000, 0x2, 0x400000 }},
	{{ 0x280005C, 0x746000, 0xA46030, 0x1E00000, 0x2, 0x400000 }},
	{{ 0x2800042, 0xD42000, 0x0, 0x400000, 0x1010412, 0x400000 }},
	{{ 0x2800042, 0xD42000, 0x0, 0x400000, 0x1010412, 0x400000 }},
	{{ 0x2800042, 0xE42000, 0x0, 0x400000, 0x2020002, 0x400000 }},
	{{ 0x2800042, 0xE42000, 0x0, 0x400000, 0x2020002, 0x400000 }},
	{{ 0x280B046, 0x0, 0x6C1008, 0x0, 0x4, 0x406800 }},
	{{ 0x2800049, 0xF44060, 0x1744080, 0x14404, 0x6, 0x400011 }},
	{{ 0x2015049, 0xF44060, 0x1744080, 0x14404, 0x8080007, 0x400011 }},
	{{ 0x280B046, 0xF60040, 0x6C1004, 0x2800000, 0x6, 0x406811 }},
	{{ 0x2815042, 0x1F42000, 0x2042010, 0x1414460, 0x10100009, 0x40B800 }},
	{{ 0x2815042, 0x1F42000, 0x2042010, 0x800000, 0x10100009, 0x40B800 }},
	{{ 0x280B042, 0x0, 0x0, 0x430400, 0x4040009, 0x0 }},
	{{ 0x2815580, 0x0, 0x0, 0x0, 0x4040005, 0x0 }},
	{{ 0x280B000, 0x0, 0x0, 0x0, 0x1, 0x400000 }},
	{{ 0x2800040, 0x174E000, 0x0, 0x0, 0xE, 0x406800 }},
	{{ 0x280B000, 0x0, 0x0, 0x600000, 0x1, 0x406800 }},
	{{ 0x280B000, 0x0, 0x0, 0xE00000, 0x1, 0x406800 }},
	{{ 0x2800000, 0x0, 0x0, 0x0, 0x1, 0x400000 }},
	{{ 0x280B046, 0x0, 0x0, 0x2800000, 0x7, 0x400000 }},
	{{ 0x280B046, 0xF60040, 0x6C1004, 0x2800000, 0x6, 0x406811 }},
	{{ 0x2815042, 0x1F43028, 0x2000000, 0xC00000, 0x10100009, 0x40B800 }},
	{{ 0x2815400, 0x0, 0x0, 0x0, 0x4040005, 0x0 }},
	{{ 0x2800000, 0x0, 0x0, 0x0, 0x1, 0x400000 }}
};

#define AL_ETH_IS_1G_MAC(mac_mode) (((mac_mode) == AL_ETH_MAC_MODE_RGMII) || ((mac_mode) == AL_ETH_MAC_MODE_SGMII))

static int al_udma_state_set_wait(struct al_udma *dma, enum al_udma_state new_state)
{
	enum al_udma_state state;
	enum al_udma_state expected_state = new_state;
	int count = 1000;
	int rc;

	rc = al_udma_state_set(dma, new_state);
	if (rc != 0) {
		al_warn("[%s] warn: failed to change state, error %d\n", dma->name, rc);
		return rc;
	}

	if ((new_state == UDMA_NORMAL) || (new_state == UDMA_DISABLE))
		expected_state = UDMA_IDLE;

	do {
		state = al_udma_state_get(dma);
		if (count-- == 0) {
			al_warn("[%s] warn: dma state didn't change to %s\n",
				 dma->name, al_udma_states_name[new_state]);
			return -ETIMEDOUT;
		}
	} while (state != expected_state);
	return 0;
}

static void al_eth_epe_init(struct al_hal_eth_adapter *adapter)
{
	int idx;

	if (adapter->enable_rx_parser == 0) {
		al_dbg("eth [%s]: disable rx parser\n", adapter->name);

		al_reg_write32(&adapter->ec_regs_base->epe[0].res_def, 0x08000000);
		al_reg_write32(&adapter->ec_regs_base->epe[0].res_in, 0x7);

		al_reg_write32(&adapter->ec_regs_base->epe[1].res_def, 0x08000000);
		al_reg_write32(&adapter->ec_regs_base->epe[1].res_in, 0x7);

		al_reg_write32(&adapter->ec_regs_base->msp.p_res_def, 0x08000080);
		al_reg_write32(&adapter->ec_regs_base->msp.p_res_in, 0x7);

		return;
	}
	al_dbg("eth [%s]: enable rx parser\n", adapter->name);
	for (idx = 0; idx < AL_ETH_EPE_ENTRIES_NUM; idx++) {
		al_reg_write32(&adapter->ec_regs_base->epe_p[idx].comp_data, al_eth_epe_p_regs[idx].data);
		al_reg_write32(&adapter->ec_regs_base->epe_p[idx].comp_mask, al_eth_epe_p_regs[idx].mask);
		al_reg_write32(&adapter->ec_regs_base->epe_p[idx].comp_ctrl, al_eth_epe_p_regs[idx].ctrl);

		al_reg_write32(&adapter->ec_regs_base->msp_c[idx].p_comp_data, al_eth_epe_p_regs[idx].data);
		al_reg_write32(&adapter->ec_regs_base->msp_c[idx].p_comp_mask, al_eth_epe_p_regs[idx].mask);
		al_reg_write32(&adapter->ec_regs_base->msp_c[idx].p_comp_ctrl, al_eth_epe_p_regs[idx].ctrl);

		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_addr, idx);
		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_data_6,
						al_eth_epe_control_table[idx].data[5]);
		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_data_2,
						al_eth_epe_control_table[idx].data[1]);
		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_data_3,
						al_eth_epe_control_table[idx].data[2]);
		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_data_4,
						al_eth_epe_control_table[idx].data[3]);
		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_data_5,
						al_eth_epe_control_table[idx].data[4]);
		al_reg_write32(&adapter->ec_regs_base->epe[0].act_table_data_1,
						al_eth_epe_control_table[idx].data[0]);

		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_addr, idx);
		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_data_6,
						al_eth_epe_control_table[idx].data[5]);
		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_data_2,
						al_eth_epe_control_table[idx].data[1]);
		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_data_3,
						al_eth_epe_control_table[idx].data[2]);
		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_data_4,
						al_eth_epe_control_table[idx].data[3]);
		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_data_5,
						al_eth_epe_control_table[idx].data[4]);
		al_reg_write32(&adapter->ec_regs_base->epe[1].act_table_data_1,
						al_eth_epe_control_table[idx].data[0]);

		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_addr, idx);
		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_data_6,
						al_eth_epe_control_table[idx].data[5]);
		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_data_2,
						al_eth_epe_control_table[idx].data[1]);
		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_data_3,
						al_eth_epe_control_table[idx].data[2]);
		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_data_4,
						al_eth_epe_control_table[idx].data[3]);
		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_data_5,
						al_eth_epe_control_table[idx].data[4]);
		al_reg_write32(&adapter->ec_regs_base->msp.p_act_table_data_1,
						al_eth_epe_control_table[idx].data[0]);
	}

	al_reg_write32(&adapter->ec_regs_base->epe[0].res_def, 0x08000080);
	al_reg_write32(&adapter->ec_regs_base->epe[0].res_in, 0x7);

	al_reg_write32(&adapter->ec_regs_base->epe[1].res_def, 0x08000080);
	al_reg_write32(&adapter->ec_regs_base->epe[1].res_in, 0);

	al_reg_write32(&adapter->ec_regs_base->msp.p_res_def, 0x08000080);
	al_reg_write32(&adapter->ec_regs_base->msp.p_res_in, 0x7);

	al_reg_write32(&adapter->ec_regs_base->epe_h[8].hdr_len, (4 << 16) | 4);

	al_reg_write32(&adapter->ec_regs_base->rfw.meta, 0x3f);

	al_reg_write32(&adapter->ec_regs_base->rfw.checksum, EC_RFW_CHECKSUM_HDR_SEL);
}

int al_eth_adapter_init(struct al_hal_eth_adapter *adapter, struct al_eth_adapter_params *params)
{
	struct al_udma_params udma_params;
	struct al_udma_m2s_pkt_len_conf conf;
	int i;
	uint32_t reg;
	int rc;

	al_dbg("eth [%s]: initialize controller's UDMA. id = %d\n", params->name, params->udma_id);
	al_dbg("eth [%s]: UDMA base regs: %p\n", params->name, params->udma_regs_base);
	al_dbg("eth [%s]: EC base regs: %p\n", params->name, params->ec_regs_base);
	al_dbg("eth [%s]: MAC base regs: %p\n", params->name, params->mac_regs_base);
	al_dbg("eth [%s]: enable_rx_parser: %x\n", params->name, params->enable_rx_parser);

	adapter->name = params->name;
	adapter->dev_id = params->dev_id;
	adapter->rev_id = params->rev_id;
	adapter->udma_id = params->udma_id;
	adapter->udma_regs_base = params->udma_regs_base;
	adapter->ec_regs_base = (struct al_ec_regs __iomem*)params->ec_regs_base;
	adapter->mac_regs_base = (struct al_eth_mac_regs __iomem*)params->mac_regs_base;
	adapter->unit_regs = (struct unit_regs __iomem *)params->udma_regs_base;
	adapter->enable_rx_parser = params->enable_rx_parser;
	adapter->ec_ints_base = ((void __iomem *)adapter->ec_regs_base) + 0x1c00;
	adapter->mac_ints_base = ((void __iomem *)adapter->mac_regs_base) + 0x800;

	udma_params.udma_reg = (union udma_regs __iomem *)&adapter->unit_regs->m2s;
	udma_params.type = UDMA_TX;
	udma_params.num_of_queues = AL_ETH_UDMA_TX_QUEUES;
	udma_params.name = "eth tx";
	rc = al_udma_init(&adapter->tx_udma, &udma_params);

	if (rc != 0) {
		al_err("failed to initialize %s, error %d\n",
			 udma_params.name, rc);
		return rc;
	}
	rc = al_udma_state_set_wait(&adapter->tx_udma, UDMA_NORMAL);
	if (rc != 0) {
		al_err("[%s]: failed to change state, error %d\n",
			 udma_params.name, rc);
		return rc;
	}
	 
	udma_params.udma_reg = (union udma_regs __iomem *)&adapter->unit_regs->s2m;
	udma_params.type = UDMA_RX;
	udma_params.num_of_queues = AL_ETH_UDMA_RX_QUEUES;
	udma_params.name = "eth rx";
	rc = al_udma_init(&adapter->rx_udma, &udma_params);

	if (rc != 0) {
		al_err("failed to initialize %s, error %d\n",
			 udma_params.name, rc);
		return rc;
	}

	rc = al_udma_state_set_wait(&adapter->rx_udma, UDMA_NORMAL);
	if (rc != 0) {
		al_err("[%s]: failed to change state, error %d\n",
			 udma_params.name, rc);
		return rc;
	}
	al_dbg("eth [%s]: controller's UDMA successfully initialized\n",
		 params->name);

	conf.encode_64k_as_zero = AL_TRUE;
	conf.max_pkt_size = 0xfffff;
	al_udma_m2s_packet_size_cfg_set(&adapter->tx_udma, &conf);

	al_udma_m2s_max_descs_set(&adapter->tx_udma, AL_ETH_PKT_MAX_BUFS + 1);

	if (adapter->ec_regs_base != NULL) {
		 
		for (i = 0; i < 4; i++) {
			al_reg_write32(&adapter->ec_regs_base->tso.cache_table_addr, i + (adapter->udma_id * 4));
			al_reg_write32(&adapter->ec_regs_base->tso.cache_table_data_1, 0x00000000);
			al_reg_write32(&adapter->ec_regs_base->tso.cache_table_data_2, 0x00000000);
			al_reg_write32(&adapter->ec_regs_base->tso.cache_table_data_3, 0x00000000);
			al_reg_write32(&adapter->ec_regs_base->tso.cache_table_data_4, 0x00000000);
		}
	}
	 
	if (adapter->udma_id != 0) {
		return 0;
	}
	 
	al_reg_write32(&adapter->ec_regs_base->gen.en, 0xffffffff);
	al_reg_write32(&adapter->ec_regs_base->gen.fifo_en, 0xffffffff);

	if (adapter->rev_id > AL_ETH_REV_ID_0) {
		 
		al_reg_write32_masked(&adapter->ec_regs_base->gen.en_ext,
				      EC_GEN_EN_EXT_CACHE_WORD_SPLIT,
				      EC_GEN_EN_EXT_CACHE_WORD_SPLIT);

		al_reg_write32(&adapter->ec_regs_base->tso.cfg_add_0,
						EC_TSO_CFG_ADD_0_MSS_SEL);

		al_reg_write32(&adapter->ec_regs_base->tso.cfg_tunnel,
						(EC_TSO_CFG_TUNNEL_EN_TUNNEL_TSO |
						 EC_TSO_CFG_TUNNEL_EN_UDP_CHKSUM |
						 EC_TSO_CFG_TUNNEL_EN_UDP_LEN |
						 EC_TSO_CFG_TUNNEL_EN_IPV6_PLEN |
						 EC_TSO_CFG_TUNNEL_EN_IPV4_CHKSUM |
						 EC_TSO_CFG_TUNNEL_EN_IPV4_IDEN |
						 EC_TSO_CFG_TUNNEL_EN_IPV4_TLEN));
	}

	al_reg_write32(&adapter->ec_regs_base->mac.gen, 0x00000001);
	 
	al_reg_write32(&adapter->ec_regs_base->tmi.tx_cfg, EC_TMI_TX_CFG_EN_FWD_TO_RX|EC_TMI_TX_CFG_SWAP_BYTES);

	al_reg_write32(&adapter->ec_regs_base->tfw_udma[0].fwd_dec, 0x000003fb);

	al_reg_write32(&adapter->ec_regs_base->rfw_default[0].opt_1, 0x00000001);

	al_reg_write32(&adapter->ec_regs_base->rfw.vid_table_addr, 0x00000000);
	 
	al_reg_write32(&adapter->ec_regs_base->rfw.vid_table_data, 0x00000000);
	 
	al_reg_write32(&adapter->ec_regs_base->rfw.thash_cfg_1, 0x00000000);

	al_eth_epe_init(adapter);

	reg = al_reg_read32(&adapter->ec_regs_base->tso.in_cfg);
	reg &= ~0x7F00;  
	al_reg_write32(&adapter->ec_regs_base->tso.in_cfg, reg);

	return 0;
}

int al_eth_ec_mac_ints_config(struct al_hal_eth_adapter *adapter)
{

	al_dbg("eth [%s]: enable ethernet and mac interrupts\n", adapter->name);

	if (adapter->udma_id != 0)
		return -EPERM;

	al_iofic_config(adapter->ec_ints_base, AL_INT_GROUP_A,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);
	al_iofic_config(adapter->ec_ints_base, AL_INT_GROUP_B,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);
	al_iofic_config(adapter->ec_ints_base, AL_INT_GROUP_C,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);
	al_iofic_config(adapter->ec_ints_base, AL_INT_GROUP_D,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);

	al_iofic_unmask(adapter->ec_ints_base, AL_INT_GROUP_A, 1);

	al_iofic_config(adapter->mac_ints_base, AL_INT_GROUP_A,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);
	al_iofic_config(adapter->mac_ints_base, AL_INT_GROUP_B,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);
	al_iofic_config(adapter->mac_ints_base, AL_INT_GROUP_C,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);
	al_iofic_config(adapter->mac_ints_base, AL_INT_GROUP_D,
		INT_CONTROL_GRP_SET_ON_POSEDGE | INT_CONTROL_GRP_CLEAR_ON_READ);

	al_iofic_unmask(adapter->mac_ints_base, AL_INT_GROUP_A, AL_BIT(12));

	al_iofic_unmask(adapter->unit_regs, AL_INT_GROUP_D, AL_BIT(11));
	return 0;
}

int al_eth_ec_mac_isr(struct al_hal_eth_adapter *adapter)
{
	uint32_t cause;
	al_dbg("[%s]: ethernet interrupts handler\n", adapter->name);

	if (adapter->udma_id != 0)
		return -EPERM;

	cause = al_iofic_read_cause(adapter->ec_ints_base, AL_INT_GROUP_A);
	al_dbg("[%s]: ethernet group A cause 0x%08x\n", adapter->name, cause);
	if (cause & 1)
	{
		cause = al_iofic_read_cause(adapter->mac_ints_base, AL_INT_GROUP_A);
		al_dbg("[%s]: mac group A cause 0x%08x\n", adapter->name, cause);

		cause = al_iofic_read_cause(adapter->mac_ints_base, AL_INT_GROUP_B);
		al_dbg("[%s]: mac group B cause 0x%08x\n", adapter->name, cause);

		cause = al_iofic_read_cause(adapter->mac_ints_base, AL_INT_GROUP_C);
		al_dbg("[%s]: mac group C cause 0x%08x\n", adapter->name, cause);

		cause = al_iofic_read_cause(adapter->mac_ints_base, AL_INT_GROUP_D);
		al_dbg("[%s]: mac group D cause 0x%08x\n", adapter->name, cause);
	}
	cause = al_iofic_read_cause(adapter->ec_ints_base, AL_INT_GROUP_B);
	al_dbg("[%s]: ethernet group B cause 0x%08x\n", adapter->name, cause);
	cause = al_iofic_read_cause(adapter->ec_ints_base, AL_INT_GROUP_C);
	al_dbg("[%s]: ethernet group C cause 0x%08x\n", adapter->name, cause);
	cause = al_iofic_read_cause(adapter->ec_ints_base, AL_INT_GROUP_D);
	al_dbg("[%s]: ethernet group D cause 0x%08x\n", adapter->name, cause);

	return 0;
}

int al_eth_adapter_stop(struct al_hal_eth_adapter *adapter)
{
	int rc;

	al_dbg("eth [%s]: stop controller's UDMA\n", adapter->name);

	rc = al_udma_state_set_wait(&adapter->tx_udma, UDMA_DISABLE);
	if (rc != 0) {
		al_warn("[%s] warn: failed to change state, error %d\n",
			 adapter->tx_udma.name, rc);
		return rc;
	}

	al_dbg("eth [%s]: controller's TX UDMA stopped\n",
		 adapter->name);
	 
	rc = al_udma_state_set_wait(&adapter->rx_udma, UDMA_DISABLE);
	if (rc != 0) {
		al_warn("[%s] warn: failed to change state, error %d\n",
			 adapter->rx_udma.name, rc);
		return rc;
	}

	al_dbg("eth [%s]: controller's RX UDMA stopped\n",
		 adapter->name);
	return 0;
}

int al_eth_adapter_reset(struct al_hal_eth_adapter *adapter)
{
	al_dbg("eth [%s]: reset controller's UDMA\n", adapter->name);

	return -EPERM;
}

int al_eth_queue_config(struct al_hal_eth_adapter *adapter, enum al_udma_type type, uint32_t qid,
			     struct al_udma_q_params *q_params)
{
	struct al_udma *udma;
	int rc;

	al_dbg("eth [%s]: config UDMA %s queue %d\n", adapter->name,
		 type == UDMA_TX ? "Tx" : "Rx", qid);

	if (type == UDMA_TX) {
		udma = &adapter->tx_udma;
	} else {
		udma = &adapter->rx_udma;
	}

	q_params->dev_id = adapter->dev_id;
	q_params->rev_id = adapter->rev_id;

	rc = al_udma_q_init(udma, qid, q_params);

	return rc;
}

int al_eth_queue_enable(struct al_hal_eth_adapter *adapter __attribute__((__unused__)),
			enum al_udma_type type __attribute__((__unused__)),
			uint32_t qid __attribute__((__unused__)))
{
	return -EPERM;
}
int al_eth_queue_disable(struct al_hal_eth_adapter *adapter __attribute__((__unused__)),
			 enum al_udma_type type __attribute__((__unused__)),
			 uint32_t qid __attribute__((__unused__)))
{
	return -EPERM;
}

int al_eth_rx_pkt_limit_config(struct al_hal_eth_adapter *adapter, uint32_t min_rx_len, uint32_t max_rx_len)
{
	al_assert(max_rx_len <= AL_ETH_MAX_FRAME_LEN);

	al_reg_write32(&adapter->ec_regs_base->mac.min_pkt, min_rx_len);
	 
	al_reg_write32(&adapter->ec_regs_base->mac.max_pkt, max_rx_len);

	if (AL_ETH_IS_1G_MAC(adapter->mac_mode)) {
		al_reg_write32(&adapter->mac_regs_base->mac_1g.frm_len, max_rx_len + 8);
	} else {
		al_reg_write32(&adapter->mac_regs_base->mac_10g.frm_len, max_rx_len + 8);
	}
	return 0;
}

int al_eth_mac_config(struct al_hal_eth_adapter *adapter, enum al_eth_mac_mode mode)
{
	char * mode_str;

	switch(mode) {
	case AL_ETH_MAC_MODE_RGMII:
		mode_str = "RGMII";
		al_reg_write32(&adapter->mac_regs_base->gen.clk_cfg, 0x40003210);

		al_reg_write32(&adapter->mac_regs_base->mac_1g.cmd_cfg, 0x01800010);

		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_section_empty, 0x00000000);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_section_full, 0x0000000c);  
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_almost_empty, 0x00000008);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_almost_full, 0x00000008);

		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_section_empty, 0x00000008);  
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_section_full, 0x0000000c);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_almost_empty, 0x00000008);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_almost_full, 0x00000008);

		al_reg_write32(&adapter->mac_regs_base->gen.cfg, 0x00000000);

		al_reg_write32(&adapter->mac_regs_base->gen.mac_1g_cfg, 0x00000002);
		 
		al_reg_write32_masked(&adapter->mac_regs_base->gen.mux_sel, ~ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, 0x00063910);
		al_reg_write32(&adapter->mac_regs_base->gen.rgmii_sel, 0xF);
		break;
	case AL_ETH_MAC_MODE_SGMII:
		mode_str = "SGMII";
		al_reg_write32(&adapter->mac_regs_base->gen.clk_cfg, 0x40053210);

		al_reg_write32(&adapter->mac_regs_base->mac_1g.cmd_cfg, 0x01800010);

		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_section_empty, 0x00000000);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_section_full, 0x0000000c);  
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_almost_empty, 0x00000008);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.rx_almost_full, 0x00000008);

		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_section_empty, 0x00000008);  
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_section_full, 0x0000000c);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_almost_empty, 0x00000008);
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.tx_almost_full, 0x00000008);

		al_reg_write32(&adapter->mac_regs_base->gen.cfg, 0x000000c0);

		al_reg_write32(&adapter->mac_regs_base->gen.mac_1g_cfg, 0x00000002);
		 
		al_reg_write32_masked(&adapter->mac_regs_base->gen.mux_sel, ~ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, 0x00063910);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x000004f0);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x00000401);

#ifdef AL_HAL_ETH_FAST_AN
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr, 0x00000012);
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_data, 0x00000040);
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr, 0x00000013);
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_data, 0x00000000);
#endif

		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr, 0x00000014);
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_data, 0x0000000b);

		al_reg_write32_masked(&adapter->mac_regs_base->gen.led_cfg,
				      ETH_MAC_GEN_LED_CFG_SEL_MASK,
				      ETH_MAC_GEN_LED_CFG_SEL_DEFAULT_REG);
		break;

	case AL_ETH_MAC_MODE_XAUI:
		mode_str = "XAUI";
		 
		al_reg_write32(&adapter->mac_regs_base->mac_10g.cmd_cfg, 0x01022830);
		 
		al_reg_write32(&adapter->mac_regs_base->gen.cfg, 0x00000003);

		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_32_64, 0x00000401);
 
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_64_32, 0x00000401);
 
		al_reg_write32_masked(&adapter->mac_regs_base->gen.mux_sel, ~ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, 0x06883910);
		al_reg_write32(&adapter->mac_regs_base->gen.clk_cfg, 0x10003210);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x0000040f);
		break;
	case AL_ETH_MAC_MODE_RXAUI:
		mode_str = "RXAUI";
		 
		al_reg_write32(&adapter->mac_regs_base->mac_10g.cmd_cfg, 0x01022830);
		 
		al_reg_write32(&adapter->mac_regs_base->gen.cfg, 0x0000000b);
		 
		al_reg_write32(&adapter->mac_regs_base->gen.rxaui_cfg, 0x00000005);
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_32_64, 0x00000401);
 
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_64_32, 0x00000401);
 
		al_reg_write32_masked(&adapter->mac_regs_base->gen.mux_sel, ~ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, 0x002c3910);
		al_reg_write32(&adapter->mac_regs_base->gen.clk_cfg, 0x0003210);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x00000403);
		break;

	case AL_ETH_MAC_MODE_10GbE_Serial:
		mode_str = "KR";

		al_reg_write32(&adapter->mac_regs_base->mac_10g.cmd_cfg, 0x01022830);
		 
		al_reg_write32(&adapter->mac_regs_base->gen.cfg, 0x00000005);
		 
		al_reg_write32(&adapter->mac_regs_base->gen.rxaui_cfg, 0x00000007);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_cfg, 0x000001F1);
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_32_64, 0x00000401);
 
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_64_32, 0x00000401);
 
		al_reg_write32_masked(&adapter->mac_regs_base->gen.mux_sel, ~ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, 0x00073910);
		al_reg_write32(&adapter->mac_regs_base->gen.clk_cfg, 0x10003210);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x000004f0);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x00000401);

		al_reg_write32_masked(&adapter->mac_regs_base->gen.led_cfg,
				      ETH_MAC_GEN_LED_CFG_SEL_MASK,
				      ETH_MAC_GEN_LED_CFG_SEL_DEFAULT_REG);
		break;
	case AL_ETH_MAC_MODE_10G_SGMII:
		mode_str = "10G_SGMII";

		al_reg_write32(&adapter->mac_regs_base->mac_10g.cmd_cfg, 0x01022830);

		al_reg_write32(&adapter->mac_regs_base->gen.cfg, 0x00000001);

		al_reg_write32(&adapter->mac_regs_base->mac_10g.if_mode, 0x0000002b);
		al_reg_write32(&adapter->mac_regs_base->mac_10g.control, 0x00009140);
		 
#ifdef AL_HAL_ETH_FAST_AN
		al_reg_write32(&adapter->mac_regs_base->mac_10g.link_timer_lo, 0x00000040);
		al_reg_write32(&adapter->mac_regs_base->mac_10g.link_timer_hi, 0x00000000);
#endif

		al_reg_write32(&adapter->mac_regs_base->gen.rxaui_cfg, 0x00000007);
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_32_64, 0x00000401);
 
		al_reg_write32(&adapter->mac_regs_base->gen.xgmii_dfifo_64_32, 0x00000401);
 
		al_reg_write32_masked(&adapter->mac_regs_base->gen.mux_sel, ~ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, 0x00063910);
		al_reg_write32(&adapter->mac_regs_base->gen.clk_cfg, 0x40003210);
		al_reg_write32(&adapter->mac_regs_base->gen.sd_fifo_ctrl, 0x00000401);

		al_reg_write32_masked(&adapter->mac_regs_base->gen.led_cfg,
				      ETH_MAC_GEN_LED_CFG_SEL_MASK,
				      ETH_MAC_GEN_LED_CFG_SEL_DEFAULT_REG);
		break;

	default:
		al_err("Eth: unsupported MAC mode %d", mode);
		return -EPERM;
	}
	adapter->mac_mode = mode;
	al_info("configured MAC to %s mode:\n", mode_str);

	return 0;
}

int al_eth_mac_start(struct al_hal_eth_adapter *adapter)
{
	if (AL_ETH_IS_1G_MAC(adapter->mac_mode))
		 
		al_reg_write32_masked(&adapter->mac_regs_base->mac_1g.cmd_cfg, 0x3, 0x3);
	else	 
		al_reg_write32_masked(&adapter->mac_regs_base->mac_10g.cmd_cfg, 0x3, 0x3);

	return 0;
}

int al_eth_mac_stop(struct al_hal_eth_adapter *adapter)
{
	if (AL_ETH_IS_1G_MAC(adapter->mac_mode))
		 
		al_reg_write32(&adapter->mac_regs_base->mac_1g.cmd_cfg, 0x0);
	else	 
		al_reg_write32(&adapter->mac_regs_base->mac_10g.cmd_cfg, 0x0);

	return 0;
}

int al_eth_capabilities_get(struct al_hal_eth_adapter *adapter, struct al_eth_capabilities *caps)
{
	al_assert(caps);

	caps->speed_10_HD = AL_FALSE;
	caps->speed_10_FD = AL_FALSE;
	caps->speed_100_HD = AL_FALSE;
	caps->speed_100_FD = AL_FALSE;
	caps->speed_1000_HD = AL_FALSE;
	caps->speed_1000_FD = AL_FALSE;
	caps->speed_10000_HD = AL_FALSE;
	caps->speed_10000_FD = AL_FALSE;
	caps->pfc = AL_FALSE;
	caps->eee = AL_FALSE;

	switch (adapter->mac_mode) {
		case AL_ETH_MAC_MODE_RGMII:
		case AL_ETH_MAC_MODE_SGMII:
			caps->speed_10_HD = AL_TRUE;
			caps->speed_10_FD = AL_TRUE;
			caps->speed_100_HD = AL_TRUE;
			caps->speed_100_FD = AL_TRUE;
			caps->speed_1000_FD = AL_TRUE;
			caps->eee = AL_TRUE;
			break;
		case AL_ETH_MAC_MODE_10GbE_Serial:
			caps->speed_10000_FD = AL_TRUE;
			caps->pfc = AL_TRUE;
			break;
		default:
		al_err("Eth: unsupported MAC mode %d", adapter->mac_mode);
		return -EPERM;
	}
	return 0;
}

#ifdef CONFIG_SYNO_ALPINE_A0
int al_eth_mac_link_config(struct al_hal_eth_adapter *adapter,
			   al_bool force_1000_base_x,
			   al_bool an_enable,
			   uint32_t speed,
			   al_bool full_duplex)
#else
int al_eth_mac_link_config(struct al_hal_eth_adapter *adapter, uint32_t speed, al_bool full_duplex)
#endif
{
	uint32_t mac_ctrl;
#ifdef CONFIG_SYNO_ALPINE_A0
	uint32_t sgmii_ctrl = 0;
	uint32_t sgmii_if_mode = 0;
#endif

	if (!AL_ETH_IS_1G_MAC(adapter->mac_mode)) {
		al_err("eth [%s]: this function not supported in this mac mode.\n",
			       adapter->name);
		return -EINVAL;
	}
#ifdef CONFIG_SYNO_ALPINE_A0
	if ((adapter != AL_ETH_MAC_MODE_RGMII) && (an_enable)) {
		 
		al_info("eth [%s]: set auto negotiation to enable\n", adapter->name);
	} else {
#endif
		al_info("eth [%s]: set link speed to %dMbps. %s duplex.\n", adapter->name,
			speed, full_duplex == AL_TRUE ? "full" : "half");
		if ((speed != 10) && (speed != 100) && (speed != 1000)) {
			al_err("eth [%s]: bad speed parameter (%d).\n",
				       adapter->name, speed);
			return -EINVAL;
		}
		if ((speed == 1000) && (full_duplex == AL_FALSE)) {
			al_err("eth [%s]: half duplex in 1Gbps is not supported.\n",
				       adapter->name);
			return -EINVAL;
		}
#ifdef CONFIG_SYNO_ALPINE_A0
	}
#endif

	mac_ctrl = al_reg_read32(&adapter->mac_regs_base->mac_1g.cmd_cfg);
#ifdef CONFIG_SYNO_ALPINE_A0
	if (adapter->mac_mode == AL_ETH_MAC_MODE_SGMII) {
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr,
			       ETH_MAC_SGMII_REG_ADDR_CTRL_REG);
		sgmii_ctrl = al_reg_read32(&adapter->mac_regs_base->sgmii.reg_data);
		 
		if (force_1000_base_x == AL_FALSE)
			sgmii_if_mode = ETH_MAC_SGMII_REG_DATA_IF_MODE_SGMII_EN;

		if (an_enable == AL_TRUE) {
			sgmii_if_mode |= ETH_MAC_SGMII_REG_DATA_IF_MODE_SGMII_AN;
			sgmii_ctrl |= ETH_MAC_SGMII_REG_DATA_CTRL_AN_ENABLE;
		} else {
			sgmii_ctrl &= ~(ETH_MAC_SGMII_REG_DATA_CTRL_AN_ENABLE);
		}
	}

	if (full_duplex == AL_TRUE) {
		AL_REG_MASK_CLEAR(mac_ctrl, AL_ETH_1G_MAC_CTRL_HD_EN);
	} else {
		AL_REG_MASK_SET(mac_ctrl, AL_ETH_1G_MAC_CTRL_HD_EN);
		sgmii_if_mode |= ETH_MAC_SGMII_REG_DATA_IF_MODE_SGMII_DUPLEX;
	}

	if (speed == 1000) {
		AL_REG_MASK_SET(mac_ctrl, AL_ETH_1G_MAC_CTRL_1G_SPD);
		sgmii_if_mode |= ETH_MAC_SGMII_REG_DATA_IF_MODE_SGMII_SPEED_1000;
	} else {
		AL_REG_MASK_CLEAR(mac_ctrl, AL_ETH_1G_MAC_CTRL_1G_SPD);
		if (speed == 10) {
			AL_REG_MASK_SET(mac_ctrl, AL_ETH_1G_MAC_CTRL_10M_SPD);
		} else {
			sgmii_if_mode |= ETH_MAC_SGMII_REG_DATA_IF_MODE_SGMII_SPEED_100;
			AL_REG_MASK_CLEAR(mac_ctrl, AL_ETH_1G_MAC_CTRL_10M_SPD);
		}
	}

	if (adapter->mac_mode == AL_ETH_MAC_MODE_SGMII) {
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr,
			       ETH_MAC_SGMII_REG_ADDR_IF_MODE_REG);
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_data,
			       sgmii_if_mode);

		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr,
			       ETH_MAC_SGMII_REG_ADDR_CTRL_REG);
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_data,
			       sgmii_ctrl);
	}
#else  
	if (full_duplex == AL_TRUE)
		AL_REG_MASK_CLEAR(mac_ctrl, AL_ETH_1G_MAC_CTRL_HD_EN);
	else
		AL_REG_MASK_SET(mac_ctrl, AL_ETH_1G_MAC_CTRL_HD_EN);

	if (speed == 1000)
		AL_REG_MASK_SET(mac_ctrl, AL_ETH_1G_MAC_CTRL_1G_SPD);
	else {
		AL_REG_MASK_CLEAR(mac_ctrl, AL_ETH_1G_MAC_CTRL_1G_SPD);
		if (speed == 10)
			AL_REG_MASK_SET(mac_ctrl, AL_ETH_1G_MAC_CTRL_10M_SPD);
		else
			AL_REG_MASK_CLEAR(mac_ctrl, AL_ETH_1G_MAC_CTRL_10M_SPD);
	}
#endif  

	al_reg_write32(&adapter->mac_regs_base->mac_1g.cmd_cfg, mac_ctrl);
	return 0;
}

int al_eth_mac_loopback_config(struct al_hal_eth_adapter *adapter, int enable)
{
	const char *state = (enable) ? "enable" : "disable";

	al_dbg("eth [%s]: loopback %s\n", adapter->name, state);
	if (AL_ETH_IS_1G_MAC(adapter->mac_mode)) {
		uint32_t reg;
		reg = al_reg_read32(&adapter->mac_regs_base->mac_1g.cmd_cfg);
		if (enable)
			reg |= AL_BIT(15);
		else
			reg &= ~AL_BIT(15);
		al_reg_write32(&adapter->mac_regs_base->mac_1g.cmd_cfg, reg);
	} else {
		al_err("Eth: mac loopback not supported in this mode %d", adapter->mac_mode);
		return -EPERM;
	}
	return 0;
}

int al_eth_mdio_config(
	struct al_hal_eth_adapter	*adapter,
	enum al_eth_mdio_type		mdio_type,
	al_bool				shared_mdio_if,
	enum al_eth_ref_clk_freq	ref_clk_freq,
	unsigned int			mdio_clk_freq_khz)
{
	enum al_eth_mdio_if mdio_if = AL_ETH_MDIO_IF_10G_MAC;
	const char *if_name = (mdio_if == AL_ETH_MDIO_IF_1G_MAC) ? "10/100/1G MAC" : "10G MAC";
	const char *type_name = (mdio_type == AL_ETH_MDIO_TYPE_CLAUSE_22) ? "Clause 22" : "Clause 45";
	const char *shared_name = (shared_mdio_if == AL_TRUE) ? "Yes" : "No";

	unsigned int ref_clk_freq_khz;
	uint32_t val;

	al_dbg("eth [%s]: mdio config: interface %s. type %s. shared: %s\n", adapter->name, if_name, type_name, shared_name);
	adapter->shared_mdio_if = shared_mdio_if;

	val = al_reg_read32(&adapter->mac_regs_base->gen.cfg);
	al_dbg("eth [%s]: mdio config: 10G mac \n", adapter->name);

	switch(mdio_if)
	{
		case AL_ETH_MDIO_IF_1G_MAC:
			val &= ~AL_BIT(10);
			break;
		case AL_ETH_MDIO_IF_10G_MAC:
			val |= AL_BIT(10);
			break;
	}
	al_reg_write32(&adapter->mac_regs_base->gen.cfg, val);
	adapter->mdio_if = mdio_if;

	if (mdio_if == AL_ETH_MDIO_IF_10G_MAC)
	{
		val = al_reg_read32(&adapter->mac_regs_base->mac_10g.mdio_cfg_status);
		switch(mdio_type)
		{
			case AL_ETH_MDIO_TYPE_CLAUSE_22:
				val &= ~AL_BIT(6);
				break;
			case AL_ETH_MDIO_TYPE_CLAUSE_45:
				val |= AL_BIT(6);
				break;
		}

		switch (ref_clk_freq) {
		default:
			al_err("eth [%s]: %s: invalid reference clock frequency"
				" (%d)\n",
				adapter->name, __func__, ref_clk_freq);
		case AL_ETH_REF_FREQ_375_MHZ:
			ref_clk_freq_khz = 375000;
			break;
		case AL_ETH_REF_FREQ_187_5_MHZ:
			ref_clk_freq_khz = 187500;
			break;
		case AL_ETH_REF_FREQ_250_MHZ:
			ref_clk_freq_khz = 250000;
			break;
		case AL_ETH_REF_FREQ_500_MHZ:
			ref_clk_freq_khz = 500000;
			break;
		};

		val &= ~(0x1FF << 7);
		val |= (ref_clk_freq_khz / (2 * mdio_clk_freq_khz)) << 7;
		al_reg_write32(&adapter->mac_regs_base->mac_10g.mdio_cfg_status, val);
	}else{
		if(mdio_type != AL_ETH_MDIO_TYPE_CLAUSE_22) {
			al_err("eth [%s] mdio type not supported for this interface\n",
				 adapter->name);
			return -EINVAL;
		}
	}
	adapter->mdio_type = mdio_type;

	return 0;
}

int al_eth_mdio_1g_mac_read(struct al_hal_eth_adapter *adapter,
			    uint32_t phy_addr __attribute__((__unused__)),
			    uint32_t reg, uint16_t *val)
{
	*val = al_reg_read32(
		&adapter->mac_regs_base->mac_1g.phy_regs_base + reg);
	return 0;
}

int al_eth_mdio_1g_mac_write(struct al_hal_eth_adapter *adapter,
			     uint32_t phy_addr __attribute__((__unused__)),
			     uint32_t reg, uint16_t val)
{
	al_reg_write32(
		&adapter->mac_regs_base->mac_1g.phy_regs_base + reg, val);
	return 0;
}

int al_eth_mdio_10g_mac_wait_busy(struct al_hal_eth_adapter *adapter)
{
	int	count = 0;
	uint32_t mdio_cfg_status;

	do {
		mdio_cfg_status = al_reg_read32(&adapter->mac_regs_base->mac_10g.mdio_cfg_status);
 
		if (mdio_cfg_status & AL_BIT(0)){
			if (count > 0)
				al_dbg("eth [%s] mdio: still busy!\n", adapter->name);
		}else{
			return 0;
		}
		al_udelay(AL_ETH_MDIO_DELAY_PERIOD);
	}while(count++ < AL_ETH_MDIO_DELAY_COUNT);

	return -ETIMEDOUT;
}

int al_eth_mdio_10g_mac_type22(struct al_hal_eth_adapter *adapter, int read, uint32_t phy_addr, uint32_t reg, uint16_t *val)
{
	int rc;
	const char *op = (read == 1) ? "read":"write";
	uint32_t mdio_cfg_status;
	uint16_t mdio_cmd;

	rc = al_eth_mdio_10g_mac_wait_busy(adapter);
	if (rc) {
		al_err(" eth [%s] mdio %s failed. HW is busy\n", adapter->name, op);
		return rc;
	}

	mdio_cmd = (uint16_t)(0x1F & reg);
	mdio_cmd |= (0x1F & phy_addr) << 5;

	if (read)
		mdio_cmd |= AL_BIT(15);  

	al_reg_write16(&adapter->mac_regs_base->mac_10g.mdio_cmd,
			mdio_cmd);
	if (!read)
		al_reg_write16(&adapter->mac_regs_base->mac_10g.mdio_data,
				*val);

	rc = al_eth_mdio_10g_mac_wait_busy(adapter);
	if (rc != 0) {
		al_err(" %s mdio %s failed on timeout\n", adapter->name, op);
		return -ETIMEDOUT;
	}

	mdio_cfg_status = al_reg_read32(&adapter->mac_regs_base->mac_10g.mdio_cfg_status);

	if (mdio_cfg_status & AL_BIT(1)){  
		al_err(" %s mdio %s failed on error. phy_addr 0x%x reg 0x%x\n",
			adapter->name, op, phy_addr, reg);
			return -EIO;
	}
	if (read)
		*val = al_reg_read16(
			(uint16_t *)&adapter->mac_regs_base->mac_10g.mdio_data);
	return 0;
}

int al_eth_mdio_10g_mac_type45(struct al_hal_eth_adapter *adapter, int read, uint32_t port_addr, uint32_t device, uint32_t reg, uint16_t *val)
{
	int rc;
	const char *op = (read == 1) ? "read":"write";
	uint32_t mdio_cfg_status;
	uint16_t mdio_cmd;

	rc = al_eth_mdio_10g_mac_wait_busy(adapter);
	if (rc) {
		al_err(" %s mdio %s failed. HW is busy\n", adapter->name, op);
		return rc;
	}
	 
	mdio_cmd = (uint16_t)(0x1F & device);
	mdio_cmd |= (0x1F & port_addr) << 5;
	al_reg_write16(&adapter->mac_regs_base->mac_10g.mdio_cmd,
			mdio_cmd);

	al_reg_write16(&adapter->mac_regs_base->mac_10g.mdio_regaddr, reg);
	 
	rc = al_eth_mdio_10g_mac_wait_busy(adapter);
	if (rc) {
		al_err(" %s mdio %s (address frame) failed on timeout\n", adapter->name, op);
		return rc;
	}

	if (read) {
		mdio_cmd |= AL_BIT(15);  
		al_reg_write16(
			(uint16_t *)&adapter->mac_regs_base->mac_10g.mdio_cmd,
			mdio_cmd);
	} else {
		al_reg_write16(
			(uint16_t *)&adapter->mac_regs_base->mac_10g.mdio_data,
			*val);
	}
	 
	rc = al_eth_mdio_10g_mac_wait_busy(adapter);
	if (rc) {
		al_err(" %s mdio %s failed on timeout\n", adapter->name, op);
		return rc;
	}

	mdio_cfg_status = al_reg_read32(&adapter->mac_regs_base->mac_10g.mdio_cfg_status);

	if (mdio_cfg_status & AL_BIT(1)){  
		al_err(" %s mdio %s failed on error. port 0x%x, device 0x%x reg 0x%x\n",
			adapter->name, op, port_addr, device, reg);
			return -EIO;
	}
	if (read)
		*val = al_reg_read16(
			(uint16_t *)&adapter->mac_regs_base->mac_10g.mdio_data);
	return 0;
}

static int al_eth_mdio_lock(struct al_hal_eth_adapter *adapter)
{
	int	count = 0;
	uint32_t mdio_ctrl_1;

	if (adapter->shared_mdio_if == AL_FALSE)
		return 0;  

	do {
		mdio_ctrl_1 = al_reg_read32(&adapter->mac_regs_base->gen.mdio_ctrl_1);
 
		if (mdio_ctrl_1 & AL_BIT(0)){
			if (count > 0)
				al_dbg("eth %s mdio interface still busy!\n", adapter->name);
		}else{
			return 0;
		}
		al_udelay(AL_ETH_MDIO_DELAY_PERIOD);
	}while(count++ < (AL_ETH_MDIO_DELAY_COUNT * 4));
#ifdef MY_DEF_HERE
	al_dbg(" %s mdio failed to take ownership. MDIO info reg: 0x%08x\n",
#else
	al_err(" %s mdio failed to take ownership. MDIO info reg: 0x%08x\n",
#endif
		adapter->name, al_reg_read32(&adapter->mac_regs_base->gen.mdio_1));

	return -ETIMEDOUT;
}

static int al_eth_mdio_free(struct al_hal_eth_adapter *adapter)
{
	if (adapter->shared_mdio_if == AL_FALSE)
		return 0;  

	al_reg_write32(&adapter->mac_regs_base->gen.mdio_ctrl_1, 0);

	al_udelay(2 * AL_ETH_MDIO_DELAY_PERIOD);

	return 0;
}

int al_eth_mdio_read(struct al_hal_eth_adapter *adapter, uint32_t phy_addr, uint32_t device, uint32_t reg, uint16_t *val)
{
	int rc;
	rc = al_eth_mdio_lock(adapter);

	if (rc)
		return rc;

	if (adapter->mdio_if == AL_ETH_MDIO_IF_1G_MAC)
		rc = al_eth_mdio_1g_mac_read(adapter, phy_addr, reg, val);
	else
		if (adapter->mdio_type == AL_ETH_MDIO_TYPE_CLAUSE_22)
			rc = al_eth_mdio_10g_mac_type22(adapter, 1, phy_addr, reg, val);
		else
			rc = al_eth_mdio_10g_mac_type45(adapter, 1, phy_addr, device, reg, val);

	al_eth_mdio_free(adapter);
	al_dbg("eth mdio read: phy_addr %x, device %x, reg %x val %x\n", phy_addr, device, reg, *val);
	return rc;
}

int al_eth_mdio_write(struct al_hal_eth_adapter *adapter, uint32_t phy_addr, uint32_t device, uint32_t reg, uint16_t val)
{
	int rc;
	al_dbg("eth mdio write: phy_addr %x, device %x, reg %x, val %x\n", phy_addr, device, reg, val);
	rc = al_eth_mdio_lock(adapter);
	 
	if (rc)
		return rc;

	if (adapter->mdio_if == AL_ETH_MDIO_IF_1G_MAC)
		rc = al_eth_mdio_1g_mac_write(adapter, phy_addr, reg, val);
	else
		if (adapter->mdio_type == AL_ETH_MDIO_TYPE_CLAUSE_22)
			rc = al_eth_mdio_10g_mac_type22(adapter, 0, phy_addr, reg, &val);
		else
			rc = al_eth_mdio_10g_mac_type45(adapter, 0, phy_addr, device, reg, &val);

	al_eth_mdio_free(adapter);
	return rc;
}

void al_dump_tx_desc(union al_udma_desc *tx_desc)
{
	uint32_t *ptr = (uint32_t *)tx_desc;
	al_dbg("eth tx desc:\n");
	al_dbg("0x%08x\n", *(ptr++));
	al_dbg("0x%08x\n", *(ptr++));
	al_dbg("0x%08x\n", *(ptr++));
	al_dbg("0x%08x\n", *(ptr++));
}

static void
al_dump_tx_pkt(struct al_udma_q *tx_dma_q, struct al_eth_pkt *pkt)
{
	const char *tso = (pkt->flags & AL_ETH_TX_FLAGS_TSO) ? "TSO" : "";
	const char *l3_csum = (pkt->flags & AL_ETH_TX_FLAGS_IPV4_L3_CSUM) ? "L3 CSUM" : "";
	const char *l4_csum = (pkt->flags & AL_ETH_TX_FLAGS_L4_CSUM) ?
	  ((pkt->flags & AL_ETH_TX_FLAGS_L4_PARTIAL_CSUM) ? "L4 PARTIAL CSUM" : "L4 FULL CSUM") : "";
	const char *fcs = (pkt->flags & AL_ETH_TX_FLAGS_L2_DIS_FCS) ? "Disable FCS" : "";
	char *l3_proto_name = "unknown";
	char *l4_proto_name = "unknown";
	uint32_t total_len = 0;
	int i;

	al_dbg("[%s %d]: flags: %s %s %s %s\n", tx_dma_q->udma->name, tx_dma_q->qid,
		tso, l3_csum, l4_csum, fcs);

	switch (pkt->l3_proto_idx) {
	case AL_ETH_PROTO_ID_IPv4:
		l3_proto_name = "IPv4";
		break;
	case AL_ETH_PROTO_ID_IPv6:
		l3_proto_name = "IPv6";
		break;
	default:
		l3_proto_name = "unknown";
		break;
	}
	switch (pkt->l4_proto_idx) {
	case AL_ETH_PROTO_ID_TCP:
		l4_proto_name = "TCP";
		break;
	case AL_ETH_PROTO_ID_UDP:
		l4_proto_name = "UDP";
		break;
	default:
		l4_proto_name = "unknown";
		break;
	}

	al_dbg("[%s %d]: L3 proto: %d (%s). L4 proto: %d (%s). vlan source count %d. mod add %d. mod del %d\n",
			tx_dma_q->udma->name, tx_dma_q->qid, pkt->l3_proto_idx,
			l3_proto_name, pkt->l4_proto_idx, l4_proto_name,
			pkt->source_vlan_count, pkt->vlan_mod_add_count,
			pkt->vlan_mod_del_count);

	if (pkt->meta) {
		const char * store = pkt->meta->store ? "Yes" : "No";

		al_dbg("[%s %d]: tx pkt with meta data. words valid %x\n",
			tx_dma_q->udma->name, tx_dma_q->qid,
			pkt->meta->words_valid);
		if (tx_dma_q->rev_id == AL_ETH_REV_ID_0)
			al_dbg("[%s %d]: meta: store to cache %s. l3 hdr len %d. "
			       "l3 hdr offset %d. l4 hdr len %d. mss sel %d\n",
				tx_dma_q->udma->name, tx_dma_q->qid, store,
				pkt->meta->l3_header_len, pkt->meta->l3_header_offset,
				pkt->meta->l4_header_len,
				pkt->meta->mss_idx_sel);
		else
			al_dbg("[%s %d]: meta: store to cache %s. l3 hdr len %d. "
			       "l3 hdr offset %d. l4 hdr len %d. mss val %d\n",
				tx_dma_q->udma->name, tx_dma_q->qid, store,
				pkt->meta->l3_header_len, pkt->meta->l3_header_offset,
				pkt->meta->l4_header_len,
				pkt->meta->mss_val);
	}

	al_dbg("[%s %d]: num of bufs: %d\n", tx_dma_q->udma->name, tx_dma_q->qid,
		pkt->num_of_bufs);
	for (i = 0; i < pkt->num_of_bufs; i++) {
		al_dbg("eth [%s %d]: buf[%d]: len 0x%08x. address 0x%016llx\n", tx_dma_q->udma->name, tx_dma_q->qid,
			i, pkt->bufs[i].len, (unsigned long long)pkt->bufs[i].addr);
		total_len += pkt->bufs[i].len;
	}
	al_dbg("[%s %d]: total len: 0x%08x\n", tx_dma_q->udma->name, tx_dma_q->qid, total_len);

}
 
int al_eth_tx_pkt_prepare(struct al_udma_q *tx_dma_q, struct al_eth_pkt *pkt)
{
	union al_udma_desc *tx_desc;
	uint32_t tx_descs;
	uint32_t flags = AL_M2S_DESC_FIRST | AL_M2S_DESC_CONCAT;
	uint64_t vmid = ((uint64_t)pkt->vmid) << AL_UDMA_DESC_VMID_SHIFT;
	uint32_t meta_ctrl;
	uint32_t ring_id;
	int buf_idx;

	al_dbg("[%s %d]: new tx pkt\n", tx_dma_q->udma->name, tx_dma_q->qid);

	al_dump_tx_pkt(tx_dma_q, pkt);

	tx_descs = pkt->num_of_bufs;
	if (pkt->meta) {
		tx_descs += 1;
	}
	if (unlikely(al_udma_available_get(tx_dma_q) < tx_descs)) {
		al_dbg("[%s %d]: failed to allocate (%d) descriptors",
			 tx_dma_q->udma->name, tx_dma_q->qid, tx_descs);
		return 0;
	}
	if (pkt->meta) {
		uint32_t meta_word_0 = pkt->flags & AL_ETH_TX_FLAGS_INT;
		uint32_t meta_word_1 = 0;
		uint32_t meta_word_2 = 0;
		uint32_t meta_word_3 = 0;

		meta_word_0 |= AL_M2S_DESC_FIRST | AL_M2S_DESC_META_DATA;
		flags &= ~(AL_M2S_DESC_FIRST | AL_ETH_TX_FLAGS_INT);

		tx_desc = al_udma_desc_get(tx_dma_q);
		 
		ring_id = al_udma_ring_id_get(tx_dma_q) <<
			AL_M2S_DESC_RING_ID_SHIFT;

		meta_word_0 |= ring_id;
		meta_word_0 |= pkt->meta->words_valid << 12;

		if (pkt->meta->store)
			meta_word_0 |= AL_ETH_TX_META_STORE;

		if (pkt->meta->words_valid & 1) {
			meta_word_0 |= pkt->meta->vlan1_cfi_sel;
			meta_word_0 |= pkt->meta->vlan2_vid_sel << 2;
			meta_word_0 |= pkt->meta->vlan2_cfi_sel << 4;
			meta_word_0 |= pkt->meta->vlan2_pbits_sel << 6;
			meta_word_0 |= pkt->meta->vlan2_ether_sel << 8;
		}

		if (pkt->meta->words_valid & 2) {
			meta_word_1 = pkt->meta->vlan1_new_vid;
			meta_word_1 |= pkt->meta->vlan1_new_cfi << 12;
			meta_word_1 |= pkt->meta->vlan1_new_pbits << 13;
			meta_word_1 |= pkt->meta->vlan2_new_vid << 16;
			meta_word_1 |= pkt->meta->vlan2_new_cfi << 28;
			meta_word_1 |= pkt->meta->vlan2_new_pbits << 29;
		}

		if (pkt->meta->words_valid & 4) {
			meta_word_2 = pkt->meta->l3_header_len & AL_ETH_TX_META_L3_LEN_MASK;
			meta_word_2 |= (pkt->meta->l3_header_offset & AL_ETH_TX_META_L3_OFF_MASK) <<
				AL_ETH_TX_META_L3_OFF_SHIFT;
			meta_word_2 |= (pkt->meta->l4_header_len & 0x3f) << 16;

			if (tx_dma_q->rev_id == AL_ETH_REV_ID_0) {
				meta_word_2 |= (pkt->meta->mss_idx_sel & 7) << 24;
			} else {
				uint32_t l3_offset;

				if (unlikely(pkt->flags & AL_ETH_TX_FLAGS_TS))
					meta_word_0 |= pkt->meta->ts_index << AL_ETH_TX_META_MSS_MSB_TS_VAL_SHIFT;
				else
					meta_word_0 |= (((pkt->meta->mss_val & 0x3c00) >> 10)
							<< AL_ETH_TX_META_MSS_MSB_TS_VAL_SHIFT);
				meta_word_2 |= ((pkt->meta->mss_val & 0x03ff)
						<< AL_ETH_TX_META_MSS_LSB_VAL_SHIFT);

				l3_offset = (pkt->meta->outer_l3_offset >> 1);

				meta_word_0 |=
					(((l3_offset &
					   AL_ETH_TX_META_OUTER_L3_OFF_HIGH_MASK) >> 3)
					   << AL_ETH_TX_META_OUTER_L3_OFF_HIGH_SHIFT);

				meta_word_3 |=
					((l3_offset &
					   AL_ETH_TX_META_OUTER_L3_OFF_LOW_MASK)
					   << AL_ETH_TX_META_OUTER_L3_OFF_LOW_SHIFT);

				meta_word_3 |=
					(((pkt->meta->outer_l3_len >> 2) &
					   AL_ETH_TX_META_OUTER_L3_LEN_MASK)
					   << AL_ETH_TX_META_OUTER_L3_LEN_SHIFT);
			}
		}

		if (pkt->meta->words_valid & 8) {
		 
		if (pkt->flags & AL_ETH_TX_FLAGS_L2_MACSEC_PKT) {  
				meta_word_3 |= pkt->macsec_sign			<< AL_ETH_TX_MACSEC_SIGN_SHIFT;
				meta_word_3 |= pkt->macsec_encrypt		<< AL_ETH_TX_MACSEC_ENCRYPT_SHIFT;
				meta_word_3 |= pkt->macsec_association_number	<< AL_ETH_TX_MACSEC_AN_LSB_SHIFT;
				meta_word_3 |= pkt->macsec_secure_channel	<< AL_ETH_TX_MACSEC_SC_LSB_SHIFT;
				meta_word_3 |= pkt->macsec_secured_pyld_len	<< AL_ETH_TX_MACSEC_SECURED_PYLD_LEN_LSB_SHIFT;
			}
		}

		tx_desc->tx_meta.len_ctrl = swap32_to_le(meta_word_0);
		tx_desc->tx_meta.meta_ctrl = swap32_to_le(meta_word_1);
		tx_desc->tx_meta.meta1 = swap32_to_le(meta_word_2);
		tx_desc->tx_meta.meta2 = swap32_to_le(meta_word_3);
		al_dump_tx_desc(tx_desc);
	}

	meta_ctrl = pkt->flags & AL_ETH_TX_PKT_META_FLAGS;

	al_assert((pkt->flags & (AL_ETH_TX_FLAGS_L4_CSUM|AL_ETH_TX_FLAGS_L4_PARTIAL_CSUM)) !=
		  AL_ETH_TX_FLAGS_L4_PARTIAL_CSUM);

	al_assert((pkt->flags & (AL_ETH_TX_FLAGS_TSO|AL_ETH_TX_FLAGS_TS)) !=
		  (AL_ETH_TX_FLAGS_TSO|AL_ETH_TX_FLAGS_TS));

	meta_ctrl |= pkt->l3_proto_idx;
	meta_ctrl |= pkt->l4_proto_idx << AL_ETH_TX_L4_PROTO_IDX_SHIFT;
	meta_ctrl |= pkt->source_vlan_count << AL_ETH_TX_SRC_VLAN_CNT_SHIFT;
	meta_ctrl |= pkt->vlan_mod_add_count << AL_ETH_TX_VLAN_MOD_ADD_SHIFT;
	meta_ctrl |= pkt->vlan_mod_del_count << AL_ETH_TX_VLAN_MOD_DEL_SHIFT;
	meta_ctrl |= pkt->vlan_mod_v1_ether_sel << AL_ETH_TX_VLAN_MOD_E_SEL_SHIFT;
	meta_ctrl |= pkt->vlan_mod_v1_vid_sel << AL_ETH_TX_VLAN_MOD_VID_SEL_SHIFT;
	meta_ctrl |= pkt->vlan_mod_v1_pbits_sel << AL_ETH_TX_VLAN_MOD_PBIT_SEL_SHIFT;

	if (tx_dma_q->rev_id > AL_ETH_REV_ID_0) {
		meta_ctrl |= pkt->tunnel_mode << AL_ETH_TX_TUNNEL_MODE_SHIFT;
		if (pkt->outer_l3_proto_idx == AL_ETH_PROTO_ID_IPv6)
			meta_ctrl |= 1 << AL_ETH_TX_OUTER_L3_PROTO_SHIFT;
	}

	flags |= pkt->flags & AL_ETH_TX_PKT_UDMA_FLAGS;
	for(buf_idx = 0; buf_idx < pkt->num_of_bufs; buf_idx++ ) {
		uint32_t flags_len = flags;
				tx_desc = al_udma_desc_get(tx_dma_q);
		 
		ring_id = al_udma_ring_id_get(tx_dma_q) <<
			AL_M2S_DESC_RING_ID_SHIFT;

		flags_len |= ring_id;

		if (buf_idx == (pkt->num_of_bufs - 1))
			flags_len |= AL_M2S_DESC_LAST;

		flags &= AL_ETH_TX_FLAGS_NO_SNOOP;
		flags |= AL_M2S_DESC_CONCAT;

		flags_len |= pkt->bufs[buf_idx].len & AL_M2S_DESC_LEN_MASK;
		tx_desc->tx.len_ctrl = swap32_to_le(flags_len);
		if (buf_idx == 0)
			tx_desc->tx.meta_ctrl = swap32_to_le(meta_ctrl);
		tx_desc->tx.buf_ptr = swap64_to_le(
			pkt->bufs[buf_idx].addr | vmid);
		al_dump_tx_desc(tx_desc);
	}

	al_dbg("[%s %d]: pkt descriptors written into the tx queue. descs num (%d)\n",
		tx_dma_q->udma->name, tx_dma_q->qid, tx_descs);

	return tx_descs;
}

void al_eth_tx_dma_action(struct al_udma_q *tx_dma_q, uint32_t tx_descs)
{
	 
	al_udma_desc_action_add(tx_dma_q, tx_descs);
}

int al_eth_comp_tx_get(struct al_udma_q *tx_dma_q)
{
#ifdef CONFIG_SYNO_ALPINE_V2_5_3
 
#else
	union al_udma_cdesc *cdesc;
#endif
	int rc;

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	rc = al_udma_cdesc_get_all(tx_dma_q, NULL);
#else
	rc = al_udma_cdesc_get_all(tx_dma_q, &cdesc);
#endif
	if (rc != 0) {
		al_udma_cdesc_ack(tx_dma_q, rc);
		al_dbg("[%s %d]: tx completion: descs (%d)\n",
			 tx_dma_q->udma->name, tx_dma_q->qid, rc);
	}

	return rc;
}

int al_eth_tso_mss_config(struct al_hal_eth_adapter *adapter, uint8_t idx, uint32_t mss_val)
{

	al_assert(idx <= 8);  
	al_assert(mss_val <= AL_ETH_TSO_MSS_MAX_VAL);  
	al_assert(mss_val >= AL_ETH_TSO_MSS_MIN_VAL);  

	al_reg_write32(&adapter->ec_regs_base->tso_sel[idx].mss, mss_val);
	return 0;
}

void al_eth_rx_desc_config(
			struct al_hal_eth_adapter *adapter,
			enum al_eth_rx_desc_lro_context_val_res lro_sel,
			enum al_eth_rx_desc_l4_chk_res_sel l4_sel,
			enum al_eth_rx_desc_l3_chk_res_sel l3_sel)
{
	uint32_t reg_val = 0;

	reg_val |= (lro_sel == AL_ETH_L4_OFFSET) ?
			EC_RFW_CFG_A_0_LRO_CONTEXT_SEL : 0;

	reg_val |= (l4_sel == AL_ETH_l4_INNER_OUTER_CHK) ?
			EC_RFW_CFG_A_0_META_L4_CHK_RES_SEL : 0;

	reg_val |= l3_sel << EC_RFW_CFG_A_0_META_L3_CHK_RES_SEL_SHIFT;

	al_reg_write32(&adapter->ec_regs_base->rfw.cfg_a_0, reg_val);
}

int al_eth_rx_buffer_add(struct al_udma_q *rx_dma_q,
			      struct al_buf *buf, uint32_t flags,
			      struct al_buf *header_buf)
{
	uint64_t vmid = ((uint64_t)flags & AL_ETH_RX_FLAGS_VMID_MASK) <<
		AL_UDMA_DESC_VMID_SHIFT;
	uint32_t flags_len = flags & ~AL_ETH_RX_FLAGS_VMID_MASK;
	union al_udma_desc *rx_desc;

	al_dbg("[%s %d]: add rx buffer.\n", rx_dma_q->udma->name, rx_dma_q->qid);

#if 1
	if (unlikely(al_udma_available_get(rx_dma_q) < 1)) {
		al_dbg("[%s]: rx q (%d) has no enough free descriptor",
			 rx_dma_q->udma->name, rx_dma_q->qid);
		return -ENOSPC;
	}
#endif
	rx_desc = al_udma_desc_get(rx_dma_q);

	flags_len |= al_udma_ring_id_get(rx_dma_q) << AL_M2S_DESC_RING_ID_SHIFT;
	flags_len |= buf->len & AL_M2S_DESC_LEN_MASK;

	if (flags & AL_ETH_RX_FLAGS_DUAL_BUF) {
		al_assert(header_buf != NULL);  
		al_assert(AL_ADDR_HIGH(buf->addr) == AL_ADDR_HIGH(header_buf->addr));  

		flags_len |= (header_buf->len << AL_M2S_DESC_LEN_SHIFT) &
			AL_M2S_DESC_LEN_MASK;
		rx_desc->rx.buf2_ptr_lo = swap32_to_le(AL_ADDR_LOW(header_buf->addr));
	}
	rx_desc->rx.len_ctrl = swap32_to_le(flags_len);
	rx_desc->rx.buf1_ptr = swap64_to_le(buf->addr | vmid);

	return 0;
}

void al_eth_rx_buffer_action(struct al_udma_q *rx_dma_q, uint32_t descs_num)
{
	al_dbg("[%s]: update the rx engine tail pointer: queue %d. descs %d\n",
		 rx_dma_q->udma->name, rx_dma_q->qid, descs_num);

	al_udma_desc_action_add(rx_dma_q, descs_num);
}

uint32_t al_eth_pkt_rx(struct al_udma_q *rx_dma_q, struct al_eth_pkt *pkt)
{
#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	volatile union al_udma_cdesc *cdesc;
#else
	union al_udma_cdesc *cdesc;
#endif
	al_eth_rx_cdesc *rx_desc;
	uint32_t i;
	uint32_t rc;

	rc = al_udma_cdesc_packet_get(rx_dma_q, &cdesc);
	if (rc == 0)
		return 0;

	al_assert(rc < AL_ETH_PKT_MAX_BUFS);

	al_dbg("[%s]: fetch rx packet: queue %d.\n",
		 rx_dma_q->udma->name, rx_dma_q->qid);

	for (i = 0; i < rc; i++) {
		uint32_t len;

		rx_desc = (al_eth_rx_cdesc *)al_cdesc_next(rx_dma_q, cdesc, i);

		len = swap32_from_le(rx_desc->len);
#if 0
		if (i == 0)
			pkt->rx_header_len = (len & AL_S2M_DESC_LEN2_MASK) >>
				AL_S2M_DESC_LEN2_SHIFT;
#endif
		pkt->bufs[i].len = len & AL_M2S_DESC_LEN_MASK;
	}
	 
	pkt->flags = swap32_from_le(rx_desc->ctrl_meta);
#ifdef AL_ETH_RX_DESC_RAW_GET
	pkt->rx_desc_raw[0] = pkt->flags;
	pkt->rx_desc_raw[1] = swap32_from_le(rx_desc->len);
	pkt->rx_desc_raw[2] = swap32_from_le(rx_desc->word2);
	pkt->rx_desc_raw[3] = swap32_from_le(rx_desc->word3);
#endif
	 
	pkt->l3_proto_idx = pkt->flags & AL_ETH_RX_L3_PROTO_IDX_MASK;
	pkt->l4_proto_idx = (pkt->flags >> AL_ETH_RX_L4_PROTO_IDX_SHIFT) & AL_ETH_RX_L4_PROTO_IDX_MASK;
	pkt->rxhash = (swap32_from_le(rx_desc->len) & AL_ETH_RX_HASH_MASK) >>
		AL_ETH_RX_HASH_SHIFT;

	pkt->macsec_rx_flags = (swap32_from_le(rx_desc->word3) & AL_ETH_MACSEC_RX_FLAGS_MASK) >>
						AL_ETH_MACSEC_RX_FLAGS_LSB_SHIFT;
	if (pkt->macsec_rx_flags & AL_ETH_MACSEC_RX_FLAGS_IS_MACSEC) {
		pkt->macsec_association_number	= (pkt->macsec_rx_flags & AL_ETH_MACSEC_RX_FLAGS_AN_MASK) >>
						AL_ETH_MACSEC_RX_FLAGS_AN_LSB_SHIFT;
		pkt->macsec_secured_pyld_len	= (pkt->macsec_rx_flags & AL_ETH_MACSEC_RX_FLAGS_SL_MASK) >>
							AL_ETH_MACSEC_RX_FLAGS_SL_LSB_SHIFT;
		pkt->macsec_encrypt		= (pkt->macsec_rx_flags & AL_ETH_MACSEC_RX_FLAGS_IS_ENCRYPTED_MASK) >>
							AL_ETH_MACSEC_RX_FLAGS_IS_ENCRYPTED_SHIFT;
		pkt->macsec_sign		= (pkt->macsec_rx_flags & AL_ETH_MACSEC_RX_FLAGS_IS_SIGNED_MASK) >>
							AL_ETH_MACSEC_RX_FLAGS_IS_SIGNED_SHIFT;
	}

	al_udma_cdesc_ack(rx_dma_q, rc);
	return rc;
}

int al_eth_thash_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint32_t entry)
{

	al_assert(idx < AL_ETH_RX_THASH_TABLE_SIZE);  

	al_reg_write32(&adapter->ec_regs_base->rfw.thash_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.thash_table_data, entry);
	return 0;
}

int al_eth_fsm_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint32_t entry)
{

	al_assert(idx < AL_ETH_RX_FSM_TABLE_SIZE);  

	al_reg_write32(&adapter->ec_regs_base->rfw.fsm_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.fsm_table_data, entry);
	return 0;
}

static uint32_t	al_eth_fwd_ctrl_entry_to_val(struct al_eth_fwd_ctrl_table_entry *entry)
{
	uint32_t val = 0;
	AL_REG_FIELD_SET(val,  AL_FIELD_MASK(3,0), 0, entry->prio_sel);
	AL_REG_FIELD_SET(val,  AL_FIELD_MASK(7,4), 4, entry->queue_sel_1);
	AL_REG_FIELD_SET(val,  AL_FIELD_MASK(9,8), 8, entry->queue_sel_2);
	AL_REG_FIELD_SET(val,  AL_FIELD_MASK(13,10), 10, entry->udma_sel);
	AL_REG_BIT_VAL_SET(val, 19, !!(entry->filter == AL_TRUE));

	return val;
}

static int al_eth_ctrl_index_match(struct al_eth_fwd_ctrl_table_index *index, uint32_t i) {
	if ((index->vlan_table_out != AL_ETH_FWD_CTRL_IDX_VLAN_TABLE_OUT_ANY)
		&& (index->vlan_table_out != AL_REG_BIT_GET(i, 0)))
		return 0;
	if ((index->tunnel_exist != AL_ETH_FWD_CTRL_IDX_TUNNEL_ANY)
		&& (index->tunnel_exist != AL_REG_BIT_GET(i, 1)))
		return 0;
	if ((index->vlan_exist != AL_ETH_FWD_CTRL_IDX_VLAN_ANY)
		&& (index->vlan_exist != AL_REG_BIT_GET(i, 2)))
		return 0;
	if ((index->mac_table_match != AL_ETH_FWD_CTRL_IDX_MAC_TABLE_ANY)
		&& (index->mac_table_match != AL_REG_BIT_GET(i, 3)))
		return 0;
	if ((index->protocol_id != AL_ETH_PROTO_ID_ANY)
		&& (index->protocol_id != AL_REG_FIELD_GET(i, AL_FIELD_MASK(8,4),4)))
		return 0;
	if ((index->mac_type != AL_ETH_FWD_CTRL_IDX_MAC_DA_TYPE_ANY)
		&& (index->mac_type != AL_REG_FIELD_GET(i, AL_FIELD_MASK(10,9),9)))
		return 0;
	return 1;
}

int al_eth_ctrl_table_set(struct al_hal_eth_adapter *adapter,
			  struct al_eth_fwd_ctrl_table_index *index,
			  struct al_eth_fwd_ctrl_table_entry *entry)
{
	uint32_t val = al_eth_fwd_ctrl_entry_to_val(entry);
	uint32_t i;

	for (i = 0; i < AL_ETH_RX_CTRL_TABLE_SIZE; i++) {
		if (al_eth_ctrl_index_match(index, i)) {
			al_reg_write32(&adapter->ec_regs_base->rfw.ctrl_table_addr, i);
			al_reg_write32(&adapter->ec_regs_base->rfw.ctrl_table_data, val);
		}
	}
	return 0;
}

int al_eth_ctrl_table_def_set(struct al_hal_eth_adapter *adapter,
			      al_bool use_table,
			      struct al_eth_fwd_ctrl_table_entry *entry)
{
	uint32_t val = al_eth_fwd_ctrl_entry_to_val(entry);

	if (use_table)
		val |= EC_RFW_CTRL_TABLE_DEF_SEL;

	al_reg_write32(&adapter->ec_regs_base->rfw.ctrl_table_def, val);

	return 0;
}

int al_eth_ctrl_table_raw_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint32_t entry)
{

	al_assert(idx < AL_ETH_RX_CTRL_TABLE_SIZE);  

	al_reg_write32(&adapter->ec_regs_base->rfw.ctrl_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.ctrl_table_data, entry);
	return 0;
}

int al_eth_ctrl_table_def_raw_set(struct al_hal_eth_adapter *adapter, uint32_t val)
{
	al_reg_write32(&adapter->ec_regs_base->rfw.ctrl_table_def, val);

	return 0;
}

int al_eth_hash_key_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint32_t val)
{

	al_assert(idx < AL_ETH_RX_HASH_KEY_NUM);  

	al_reg_write32(&adapter->ec_regs_base->rfw_hash[idx].key, val);

	return 0;
}

static uint32_t	al_eth_fwd_mac_table_entry_to_val(struct al_eth_fwd_mac_table_entry *entry)
{
	uint32_t val = 0;

	val |= (entry->filter == AL_TRUE) ? EC_FWD_MAC_CTRL_RX_VAL_DROP : 0;
	val |= ((entry->udma_mask << EC_FWD_MAC_CTRL_RX_VAL_UDMA_SHIFT) &
					EC_FWD_MAC_CTRL_RX_VAL_UDMA_MASK);

	val |= ((entry->qid << EC_FWD_MAC_CTRL_RX_VAL_QID_SHIFT) &
					EC_FWD_MAC_CTRL_RX_VAL_QID_MASK);

	val |= (entry->rx_valid == AL_TRUE) ? EC_FWD_MAC_CTRL_RX_VALID : 0;

	val |= ((entry->tx_target << EC_FWD_MAC_CTRL_TX_VAL_SHIFT) &
					EC_FWD_MAC_CTRL_TX_VAL_MASK);

	val |= (entry->tx_valid == AL_TRUE) ? EC_FWD_MAC_CTRL_TX_VALID : 0;

	return val;
}

int al_eth_fwd_mac_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx,
			     struct al_eth_fwd_mac_table_entry *entry)
{
	uint32_t val;

	al_assert(idx < AL_ETH_FWD_MAC_NUM);  

	val = (entry->addr[2] << 24) | (entry->addr[3] << 16) |
	      (entry->addr[4] << 8) | entry->addr[5];
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].data_l, val);
	val = (entry->addr[0] << 8) | entry->addr[1];
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].data_h, val);
	val = (entry->mask[2] << 24) | (entry->mask[3] << 16) |
	      (entry->mask[4] << 8) | entry->mask[5];
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].mask_l, val);
	val = (entry->mask[0] << 8) | entry->mask[1];
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].mask_h, val);

	val = al_eth_fwd_mac_table_entry_to_val(entry);
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].ctrl, val);
	return 0;
}

int al_eth_fwd_mac_addr_raw_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint32_t addr_lo, uint32_t addr_hi, uint32_t mask_lo, uint32_t mask_hi)
{
	al_assert(idx < AL_ETH_FWD_MAC_NUM);  

	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].data_l, addr_lo);
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].data_h, addr_hi);
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].mask_l, mask_lo);
	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].mask_h, mask_hi);

	return 0;
}

int al_eth_fwd_mac_ctrl_raw_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint32_t ctrl)
{
	al_assert(idx < AL_ETH_FWD_MAC_NUM);  

	al_reg_write32(&adapter->ec_regs_base->fwd_mac[idx].ctrl, ctrl);

	return 0;
}

int al_eth_mac_addr_store(void * __iomem ec_base, uint32_t idx, uint8_t *addr)
{
	struct al_ec_regs __iomem *ec_regs_base = (struct al_ec_regs __iomem*)ec_base;
	uint32_t val;

	al_assert(idx < AL_ETH_FWD_MAC_NUM);  

	val = (addr[2] << 24) | (addr[3] << 16) | (addr[4] << 8) | addr[5];
	al_reg_write32(&ec_regs_base->fwd_mac[idx].data_l, val);
	val = (addr[0] << 8) | addr[1];
	al_reg_write32(&ec_regs_base->fwd_mac[idx].data_h, val);
	return 0;
}

int al_eth_mac_addr_read(void * __iomem ec_base, uint32_t idx, uint8_t *addr)
{
	struct al_ec_regs __iomem *ec_regs_base = (struct al_ec_regs __iomem*)ec_base;
	uint32_t addr_lo = al_reg_read32(&ec_regs_base->fwd_mac[idx].data_l);
	uint16_t addr_hi = al_reg_read32(&ec_regs_base->fwd_mac[idx].data_h);

	addr[5] = addr_lo & 0xff;
	addr[4] = (addr_lo >> 8) & 0xff;
	addr[3] = (addr_lo >> 16) & 0xff;
	addr[2] = (addr_lo >> 24) & 0xff;
	addr[1] = addr_hi & 0xff;
	addr[0] = (addr_hi >> 8) & 0xff;
	return 0;
}

int al_eth_fwd_mhash_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint8_t udma_mask, uint8_t qid)
{
	uint32_t val = 0;
	al_assert(idx < AL_ETH_FWD_MAC_HASH_NUM);  

	AL_REG_FIELD_SET(val,  AL_FIELD_MASK(3,0), 0, udma_mask);
	AL_REG_FIELD_SET(val,  AL_FIELD_MASK(5,4), 4, qid);

	al_reg_write32(&adapter->ec_regs_base->rfw.mhash_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.mhash_table_data, val);
	return 0;
}
static uint32_t	al_eth_fwd_vid_entry_to_val(struct al_eth_fwd_vid_table_entry *entry)
{
	uint32_t val = 0;
	AL_REG_BIT_VAL_SET(val, 0, entry->control);
	AL_REG_BIT_VAL_SET(val, 1, entry->filter);
	AL_REG_FIELD_SET(val, AL_FIELD_MASK(5,2), 2, entry->udma_mask);

	return val;
}

int al_eth_fwd_vid_config_set(struct al_hal_eth_adapter *adapter, al_bool use_table,
			      struct al_eth_fwd_vid_table_entry *default_entry,
			      uint32_t default_vlan)
{
	uint32_t reg;

	reg = al_eth_fwd_vid_entry_to_val(default_entry);
	if (use_table)
		reg |= EC_RFW_VID_TABLE_DEF_SEL;
	else
		reg &= ~EC_RFW_VID_TABLE_DEF_SEL;
	al_reg_write32(&adapter->ec_regs_base->rfw.vid_table_def, reg);
	al_reg_write32(&adapter->ec_regs_base->rfw.default_vlan, default_vlan);

	return 0;
}

int al_eth_fwd_vid_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx,
			     struct al_eth_fwd_vid_table_entry *entry)
{
	uint32_t val;
	al_assert(idx < AL_ETH_FWD_VID_TABLE_NUM);  

	val = al_eth_fwd_vid_entry_to_val(entry);
	al_reg_write32(&adapter->ec_regs_base->rfw.vid_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.vid_table_data, val);
	return 0;
}

int al_eth_fwd_pbits_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint8_t prio)
{

	al_assert(idx < AL_ETH_FWD_PBITS_TABLE_NUM);  
	al_assert(prio < AL_ETH_FWD_PRIO_TABLE_NUM);  
	al_reg_write32(&adapter->ec_regs_base->rfw.pbits_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.pbits_table_data, prio);
	return 0;
}

int al_eth_fwd_priority_table_set(struct al_hal_eth_adapter *adapter, uint8_t prio, uint8_t qid)
{
	al_assert(prio < AL_ETH_FWD_PRIO_TABLE_NUM);  

	al_reg_write32(&adapter->ec_regs_base->rfw_priority[prio].queue, qid);
	return 0;
}

int al_eth_fwd_dscp_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint8_t prio)
{

	al_assert(idx < AL_ETH_FWD_DSCP_TABLE_NUM);  

	al_reg_write32(&adapter->ec_regs_base->rfw.dscp_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.dscp_table_data, prio);
	return 0;
}

int al_eth_fwd_tc_table_set(struct al_hal_eth_adapter *adapter, uint32_t idx, uint8_t prio)
{

	al_assert(idx < AL_ETH_FWD_DSCP_TABLE_NUM);  

	al_reg_write32(&adapter->ec_regs_base->rfw.tc_table_addr, idx);
	al_reg_write32(&adapter->ec_regs_base->rfw.tc_table_data, prio);
	return 0;
}

int al_eth_fwd_default_udma_config(struct al_hal_eth_adapter *adapter, uint32_t idx,
				   uint8_t udma_mask)
{
	al_reg_write32_masked(&adapter->ec_regs_base->rfw_default[idx].opt_1,
			       EC_RFW_DEFAULT_OPT_1_UDMA_MASK,
			       udma_mask << EC_RFW_DEFAULT_OPT_1_UDMA_SHIFT);
	return 0;
}

int al_eth_fwd_default_queue_config(struct al_hal_eth_adapter *adapter, uint32_t idx,
				   uint8_t qid)
{
	al_reg_write32_masked(&adapter->ec_regs_base->rfw_default[idx].opt_1,
			       EC_RFW_DEFAULT_OPT_1_QUEUE_MASK,
			       qid << EC_RFW_DEFAULT_OPT_1_QUEUE_SHIFT);
	return 0;
}

int al_eth_fwd_default_priority_config(struct al_hal_eth_adapter *adapter, uint32_t idx,
				   uint8_t prio)
{
	al_reg_write32_masked(&adapter->ec_regs_base->rfw_default[idx].opt_1,
			       EC_RFW_DEFAULT_OPT_1_PRIORITY_MASK,
			       prio << EC_RFW_DEFAULT_OPT_1_PRIORITY_SHIFT);
	return 0;
}

int al_eth_switching_config_set(struct al_hal_eth_adapter *adapter, uint8_t udma_id, uint8_t forward_all_to_mac, uint8_t enable_int_switching,
					enum al_eth_tx_switch_vid_sel_type vid_sel_type,
					enum al_eth_tx_switch_dec_type uc_dec,
					enum al_eth_tx_switch_dec_type mc_dec,
					enum al_eth_tx_switch_dec_type bc_dec)
{
	uint32_t reg;

	if (udma_id == 0) {
		reg = al_reg_read32(&adapter->ec_regs_base->tfw.tx_gen);
		if (forward_all_to_mac)
			reg |= EC_TFW_TX_GEN_FWD_ALL_TO_MAC;
		else
			reg &= ~EC_TFW_TX_GEN_FWD_ALL_TO_MAC;
		al_reg_write32(&adapter->ec_regs_base->tfw.tx_gen, reg);
	}

	reg = enable_int_switching;
	reg |= (vid_sel_type & 7) << 1;
	reg |= (bc_dec & 3) << 4;
	reg |= (mc_dec & 3) << 6;
	reg |= (uc_dec & 3) << 8;
	al_reg_write32(&adapter->ec_regs_base->tfw_udma[udma_id].fwd_dec, reg);

	return 0;
}

#define AL_ETH_RFW_FILTER_SUPPORTED(rev_id)	\
	(AL_ETH_RFW_FILTER_UNDET_MAC | \
	AL_ETH_RFW_FILTER_DET_MAC | \
	AL_ETH_RFW_FILTER_TAGGED | \
	AL_ETH_RFW_FILTER_UNTAGGED | \
	AL_ETH_RFW_FILTER_BC | \
	AL_ETH_RFW_FILTER_MC | \
	AL_ETH_RFW_FILTER_VLAN_VID | \
	AL_ETH_RFW_FILTER_CTRL_TABLE | \
	AL_ETH_RFW_FILTER_PROT_INDEX | \
	((rev_id > AL_ETH_REV_ID_0) ? (AL_ETH_RFW_FILTER_WOL) : 0))

int al_eth_filter_config(struct al_hal_eth_adapter *adapter, struct al_eth_filter_params *params)
{
	uint32_t reg;

	al_assert(params);  

	if (params->filters & ~(AL_ETH_RFW_FILTER_SUPPORTED(adapter->rev_id))) {
		al_err("[%s]: unsupported filter options (0x%08x)\n", adapter->name, params->filters);
		return -EINVAL;
	}

	reg = al_reg_read32(&adapter->ec_regs_base->rfw.out_cfg);
	if (params->enable == AL_TRUE)
		AL_REG_MASK_SET(reg, EC_RFW_OUT_CFG_DROP_EN);
	else
		AL_REG_MASK_CLEAR(reg, EC_RFW_OUT_CFG_DROP_EN);
	al_reg_write32(&adapter->ec_regs_base->rfw.out_cfg, reg);

	al_reg_write32_masked(
		&adapter->ec_regs_base->rfw.filter,
		AL_ETH_RFW_FILTER_SUPPORTED(adapter->rev_id),
		params->filters);
	if (params->filters & AL_ETH_RFW_FILTER_PROT_INDEX) {
		int i;
		for (i = 0; i < AL_ETH_PROTOCOLS_NUM; i++) {
			reg = al_reg_read32(&adapter->ec_regs_base->epe_a[i].prot_act);
			if (params->filter_proto[i] == AL_TRUE)
				AL_REG_MASK_SET(reg, EC_EPE_A_PROT_ACT_DROP);
			else
				AL_REG_MASK_CLEAR(reg, EC_EPE_A_PROT_ACT_DROP);
			al_reg_write32(&adapter->ec_regs_base->epe_a[i].prot_act, reg);
		}
	}
	return 0;
}

int al_eth_filter_override_config(struct al_hal_eth_adapter *adapter,
				  struct al_eth_filter_override_params *params)
{
	uint32_t reg;

	al_assert(params);  

	if (params->filters & ~(AL_ETH_RFW_FILTER_SUPPORTED(adapter->rev_id))) {
		al_err("[%s]: unsupported override filter options (0x%08x)\n", adapter->name, params->filters);
		return -EINVAL;
	}

	al_reg_write32_masked(
		&adapter->ec_regs_base->rfw.filter,
		AL_ETH_RFW_FILTER_SUPPORTED(adapter->rev_id) << 16,
		params->filters << 16);

	reg = al_reg_read32(&adapter->ec_regs_base->rfw.default_or);
	AL_REG_FIELD_SET(reg, EC_RFW_DEFAULT_OR_UDMA_MASK, EC_RFW_DEFAULT_OR_UDMA_SHIFT, params->udma);
	AL_REG_FIELD_SET(reg, EC_RFW_DEFAULT_OR_QUEUE_MASK, EC_RFW_DEFAULT_OR_QUEUE_SHIFT, params->qid);
	al_reg_write32(&adapter->ec_regs_base->rfw.default_or, reg);
	return 0;
}

int al_eth_switching_default_bitmap_set(struct al_hal_eth_adapter *adapter, uint8_t udma_id, uint8_t udma_uc_bitmask,
						uint8_t udma_mc_bitmask,uint8_t udma_bc_bitmask)
{
	al_reg_write32(&adapter->ec_regs_base->tfw_udma[udma_id].uc_udma, udma_uc_bitmask);
	al_reg_write32(&adapter->ec_regs_base->tfw_udma[udma_id].mc_udma, udma_mc_bitmask);
	al_reg_write32(&adapter->ec_regs_base->tfw_udma[udma_id].bc_udma, udma_bc_bitmask);

	return 0;
}

int al_eth_flow_control_config(struct al_hal_eth_adapter *adapter, struct al_eth_flow_control_params *params)
{
	uint32_t reg;
	int i;
	al_assert(params);  

	switch(params->type){
	case AL_ETH_FLOW_CONTROL_TYPE_LINK_PAUSE:
		al_dbg("[%s]: config flow control to link pause mode.\n", adapter->name);
		if (params->obay_enable == AL_TRUE)
			 
			al_reg_write32(&adapter->ec_regs_base->efc.ec_pause, 1);
		else
			al_reg_write32(&adapter->ec_regs_base->efc.ec_pause, 0);

		if (params->gen_enable == AL_TRUE)
			 
			al_reg_write32(&adapter->ec_regs_base->efc.ec_xoff, 1 << EC_EFC_EC_XOFF_MASK_2_SHIFT);
		else
			al_reg_write32(&adapter->ec_regs_base->efc.ec_xoff, 0);

		if (AL_ETH_IS_1G_MAC(adapter->mac_mode))
			 
			al_reg_write32(&adapter->ec_regs_base->efc.xon, EC_EFC_XON_MASK_2 | EC_EFC_XON_MASK_1);

		al_reg_write32(&adapter->ec_regs_base->efc.rx_fifo_hyst, params->rx_fifo_th_low | (params->rx_fifo_th_high << EC_EFC_RX_FIFO_HYST_TH_HIGH_SHIFT));

		for (i = 0; i < 4; i++) {
			if (params->obay_enable == AL_TRUE)
				 
				al_reg_write32(&adapter->ec_regs_base->fc_udma[i].q_pause_0,
						params->prio_q_map[i][0]);
			else
				al_reg_write32(&adapter->ec_regs_base->fc_udma[i].q_pause_0, 0);

			if (params->gen_enable == AL_TRUE)
				 
				al_reg_write32(&adapter->ec_regs_base->fc_udma[i].q_xoff_0, params->prio_q_map[i][0]);
			else
				al_reg_write32(&adapter->ec_regs_base->fc_udma[i].q_xoff_0, 0);
		}

		if (AL_ETH_IS_1G_MAC(adapter->mac_mode)) {
			 
			al_reg_write32(
				&adapter->mac_regs_base->mac_1g.pause_quant,
				params->quanta);
			al_reg_write32(
				&adapter->ec_regs_base->efc.xoff_timer_1g,
				params->quanta_th);

		} else {
			 
			al_reg_write32(
				&adapter->mac_regs_base->mac_10g.cl01_pause_quanta,
				params->quanta);
			 
			al_reg_write32(
				&adapter->mac_regs_base->mac_10g.cl01_quanta_thresh,
				params->quanta_th);
		}
	break;
	case AL_ETH_FLOW_CONTROL_TYPE_PFC:
		al_dbg("[%s]: config flow control to PFC mode.\n", adapter->name);
		al_assert(!AL_ETH_IS_1G_MAC(adapter->mac_mode));  ;

		for (i = 0; i < 4; i++) {
			int prio;
			for (prio = 0; prio < 8; prio++) {
				if (params->obay_enable == AL_TRUE)
					 
					al_reg_write32(&adapter->ec_regs_base->fc_udma[i].q_pause_0 + prio,
							params->prio_q_map[i][0]);
				else
					al_reg_write32(&adapter->ec_regs_base->fc_udma[i].q_pause_0 + prio,
							0);

				if (params->gen_enable == AL_TRUE)
					al_reg_write32(&adapter->ec_regs_base->fc_udma[0].q_xoff_0 + prio,
							params->prio_q_map[i][0]);
				else
					al_reg_write32(&adapter->ec_regs_base->fc_udma[0].q_xoff_0 + prio,
							0);
			}
		}

		if (params->gen_enable == AL_TRUE)
			al_reg_write32(&adapter->ec_regs_base->efc.ec_xoff, 0xFF << EC_EFC_EC_XOFF_MASK_2_SHIFT);
		else
			al_reg_write32(&adapter->ec_regs_base->efc.ec_xoff, 0);

		al_reg_write32(&adapter->ec_regs_base->efc.rx_fifo_hyst, params->rx_fifo_th_low | (params->rx_fifo_th_high << EC_EFC_RX_FIFO_HYST_TH_HIGH_SHIFT));

		reg = params->quanta | (params->quanta << 16);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl01_pause_quanta, reg);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl23_pause_quanta, reg);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl45_pause_quanta, reg);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl67_pause_quanta, reg);
		 
		reg = params->quanta_th | (params->quanta_th << 16);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl01_quanta_thresh, reg);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl23_quanta_thresh, reg);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl45_quanta_thresh, reg);
		al_reg_write32(
			&adapter->mac_regs_base->mac_10g.cl67_quanta_thresh, reg);

		reg = al_reg_read32(&adapter->mac_regs_base->mac_10g.cmd_cfg);
		reg |= 1 << 19;
		al_reg_write32(&adapter->mac_regs_base->mac_10g.cmd_cfg, reg);
	break;
	default:
		al_err("[%s]: unsupported flow control type %d\n", adapter->name, params->type);
		return -EINVAL;

	}
	return 0;
}

int al_eth_vlan_mod_config(struct al_hal_eth_adapter *adapter, uint8_t udma_id, uint16_t udma_etype, uint16_t vlan1_data, uint16_t vlan2_data)
{
	al_dbg("[%s]: config vlan modification registers. udma id %d.\n", adapter->name, udma_id);

	al_reg_write32(&adapter->ec_regs_base->tpm_sel[udma_id].etype, udma_etype);
	al_reg_write32(&adapter->ec_regs_base->tpm_udma[udma_id].vlan_data, vlan1_data | (vlan2_data << 16));

	return 0;
}

int al_eth_eee_get(struct al_hal_eth_adapter *adapter, struct al_eth_eee_params *params)
{
	uint32_t reg;

	al_dbg("[%s]: getting eee.\n", adapter->name);

	reg = al_reg_read32(&adapter->ec_regs_base->eee.cfg_e);
	params->enable = (reg & EC_EEE_CFG_E_ENABLE) ? AL_TRUE : AL_FALSE;

	params->tx_eee_timer = al_reg_read32(&adapter->ec_regs_base->eee.pre_cnt);
	params->min_interval = al_reg_read32(&adapter->ec_regs_base->eee.post_cnt);

	return 0;
}

int al_eth_eee_config(struct al_hal_eth_adapter *adapter, struct al_eth_eee_params *params)
{
	uint32_t reg;
	al_dbg("[%s]: config eee.\n", adapter->name);

	if (params->enable == 0) {
		al_dbg("[%s]: disable eee.\n", adapter->name);
		al_reg_write32(&adapter->ec_regs_base->eee.cfg_e, 0);
		return 0;
	}

	al_reg_write32(&adapter->ec_regs_base->eee.pre_cnt, params->tx_eee_timer);
	al_reg_write32(&adapter->ec_regs_base->eee.post_cnt, params->min_interval);
	al_reg_write32(&adapter->ec_regs_base->eee.stop_cnt, AL_ETH_EEE_STOP_CNT);

	reg = EC_EEE_CFG_E_MASK_EC_TMI_STOP | EC_EEE_CFG_E_MASK_MAC_EEE |
	       EC_EEE_CFG_E_ENABLE |
	       EC_EEE_CFG_E_USE_EC_TX_FIFO;
	al_reg_write32(&adapter->ec_regs_base->eee.cfg_e, reg);

	return 0;
}

int al_eth_ts_init(struct al_hal_eth_adapter *adapter)
{
	uint32_t reg;

	reg = al_reg_read32(&adapter->ec_regs_base->gen.en_ext);
	if (AL_ETH_IS_1G_MAC(adapter->mac_mode))
		reg &= ~EC_GEN_EN_EXT_PTH_1_10_SEL;
	else
		reg |= EC_GEN_EN_EXT_PTH_1_10_SEL;
	 
	reg &= ~EC_GEN_EN_EXT_PTH_COMPLETION_BYPASS;
	al_reg_write32(&adapter->ec_regs_base->gen.en_ext, reg);

	return 0;
}

int al_eth_tx_ts_val_get(struct al_hal_eth_adapter *adapter, uint8_t ts_index,
			 uint32_t *timestamp)
{
	al_assert(ts_index < AL_ETH_PTH_TX_SAMPLES_NUM);

	if (AL_ETH_IS_1G_MAC(adapter->mac_mode)) {
		al_assert(ts_index <= 7);
		al_assert(ts_index >= 1);
	}

	*timestamp = al_reg_read32(&adapter->ec_regs_base->pth_db[ts_index].ts);
	return 0;
}

int al_eth_pth_systime_read(struct al_hal_eth_adapter *adapter,
			    struct al_eth_pth_time *systime)
{
	uint32_t reg;

	reg = al_reg_read32(&adapter->ec_regs_base->pth.system_time_subseconds_msb);
	systime->femto = (uint64_t)reg << 18;
	reg = al_reg_read32(&adapter->ec_regs_base->pth.system_time_seconds);
	systime->seconds = reg;

	return 0;
}

int al_eth_pth_clk_period_write(struct al_hal_eth_adapter *adapter,
				uint64_t clk_period)
{
	uint32_t reg;
	 
	reg = (clk_period & AL_BIT_MASK(18)) << EC_PTH_CLOCK_PERIOD_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.clock_period_lsb, reg);
	reg = clk_period >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.clock_period_msb, reg);

	return 0;
}

int al_eth_pth_int_update_config(struct al_hal_eth_adapter *adapter,
				 struct al_eth_pth_int_update_params *params)
{
	uint32_t reg;

	reg = al_reg_read32(&adapter->ec_regs_base->pth.int_update_ctrl);
	if (params->enable == AL_FALSE) {
		reg &= ~EC_PTH_INT_UPDATE_CTRL_INT_TRIG_EN;
	} else {
		reg |= EC_PTH_INT_UPDATE_CTRL_INT_TRIG_EN;
		AL_REG_FIELD_SET(reg, EC_PTH_INT_UPDATE_CTRL_UPDATE_METHOD_MASK,
				 EC_PTH_INT_UPDATE_CTRL_UPDATE_METHOD_SHIFT,
				 params->method);
		if (params->trigger == AL_ETH_PTH_INT_TRIG_REG_WRITE)
			reg |= EC_PTH_INT_UPDATE_CTRL_UPDATE_TRIG;
		else
			reg &= ~EC_PTH_INT_UPDATE_CTRL_UPDATE_TRIG;
	}
	al_reg_write32(&adapter->ec_regs_base->pth.int_update_ctrl, reg);
	return 0;
}
 
int al_eth_pth_int_update_time_set(struct al_hal_eth_adapter *adapter,
				   struct al_eth_pth_time *time)
{
	uint32_t reg;

	al_reg_write32(&adapter->ec_regs_base->pth.int_update_seconds,
		       time->seconds);
	reg = time->femto & AL_BIT_MASK(18);
	reg = reg << EC_PTH_INT_UPDATE_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.int_update_subseconds_lsb,
		       reg);
	reg = time->femto >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.int_update_subseconds_msb,
		       reg);

	return 0;
}

int al_eth_pth_ext_update_config(struct al_hal_eth_adapter *adapter,
				 struct al_eth_pth_ext_update_params * params)
{
	uint32_t reg;

	reg = al_reg_read32(&adapter->ec_regs_base->pth.int_update_ctrl);
	AL_REG_FIELD_SET(reg, EC_PTH_INT_UPDATE_CTRL_UPDATE_METHOD_MASK,
			 EC_PTH_INT_UPDATE_CTRL_UPDATE_METHOD_SHIFT,
			 params->method);

	AL_REG_FIELD_SET(reg, EC_PTH_EXT_UPDATE_CTRL_EXT_TRIG_EN_MASK,
			 EC_PTH_EXT_UPDATE_CTRL_EXT_TRIG_EN_SHIFT,
			 params->triggers);
	al_reg_write32(&adapter->ec_regs_base->pth.int_update_ctrl, reg);
	return 0;
}

int al_eth_pth_ext_update_time_set(struct al_hal_eth_adapter *adapter,
				   struct al_eth_pth_time *time)
{
	uint32_t reg;

	al_reg_write32(&adapter->ec_regs_base->pth.ext_update_seconds,
		       time->seconds);
	reg = time->femto & AL_BIT_MASK(18);
	reg = reg << EC_PTH_EXT_UPDATE_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.ext_update_subseconds_lsb,
		       reg);
	reg = time->femto >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.ext_update_subseconds_msb,
		       reg);

	return 0;
};

int al_eth_pth_read_compensation_set(struct al_hal_eth_adapter *adapter,
				     uint64_t subseconds)
{
	uint32_t reg;

	reg = (subseconds & AL_BIT_MASK(18)) << EC_PTH_READ_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.read_compensation_subseconds_lsb, reg);

	reg = subseconds >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.read_compensation_subseconds_msb, reg);
	return 0;
}

int al_eth_pth_int_write_compensation_set(struct al_hal_eth_adapter *adapter,
					  uint64_t subseconds)
{
	uint32_t reg;

	reg = (subseconds & AL_BIT_MASK(18)) << EC_PTH_INT_WRITE_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.int_write_compensation_subseconds_lsb, reg);

	reg = subseconds >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.int_write_compensation_subseconds_msb, reg);
	return 0;
}

int al_eth_pth_ext_write_compensation_set(struct al_hal_eth_adapter *adapter,
					  uint64_t subseconds)
{
	uint32_t reg;

	reg = (subseconds & AL_BIT_MASK(18)) << EC_PTH_EXT_WRITE_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.ext_write_compensation_subseconds_lsb, reg);

	reg = subseconds >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.ext_write_compensation_subseconds_msb, reg);
	return 0;
}

int al_eth_pth_sync_compensation_set(struct al_hal_eth_adapter *adapter,
				     uint64_t subseconds)
{
	uint32_t reg;

	reg = (subseconds & AL_BIT_MASK(18)) << EC_PTH_SYNC_COMPENSATION_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth.sync_compensation_subseconds_lsb, reg);

	reg = subseconds >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth.sync_compensation_subseconds_msb, reg);
	return 0;
}

int al_eth_pth_pulse_out_config(struct al_hal_eth_adapter *adapter,
				struct al_eth_pth_pulse_out_params *params)
{
	uint32_t reg;

	if (params->index >= AL_ETH_PTH_PULSE_OUT_NUM) {
		al_err("eth [%s] PTH out pulse index out of range\n",
				 adapter->name);
		return -EINVAL;
	}
	reg = al_reg_read32(&adapter->ec_regs_base->pth_egress[params->index].trigger_ctrl);
	if (params->enable == AL_FALSE) {
		reg &= ~EC_PTH_EGRESS_TRIGGER_CTRL_EN;
	} else {
		reg |= EC_PTH_EGRESS_TRIGGER_CTRL_EN;
		if (params->periodic == AL_FALSE)
			reg &= ~EC_PTH_EGRESS_TRIGGER_CTRL_PERIODIC;
		else
			reg |= EC_PTH_EGRESS_TRIGGER_CTRL_PERIODIC;

		AL_REG_FIELD_SET(reg, EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SUBSEC_MASK,
				 EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SUBSEC_SHIFT,
				 params->period_us);
		AL_REG_FIELD_SET(reg, EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SEC_MASK,
				 EC_PTH_EGRESS_TRIGGER_CTRL_PERIOD_SEC_SHIFT,
				 params->period_sec);
	}
	al_reg_write32(&adapter->ec_regs_base->pth_egress[params->index].trigger_ctrl, reg);

	al_reg_write32(&adapter->ec_regs_base->pth_egress[params->index].trigger_seconds,
		       params->start_time.seconds);
	reg = params->start_time.femto & AL_BIT_MASK(18);
	reg = reg << EC_PTH_EGRESS_TRIGGER_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth_egress[params->index].trigger_subseconds_lsb,
		       reg);
	reg = params->start_time.femto >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth_egress[params->index].trigger_subseconds_msb,
		       reg);

	reg = params->pulse_width & AL_BIT_MASK(18);
	reg = reg << EC_PTH_EGRESS_PULSE_WIDTH_SUBSECONDS_LSB_VAL_SHIFT;
	al_reg_write32(&adapter->ec_regs_base->pth_egress[params->index].pulse_width_subseconds_lsb, reg);

	reg = params->pulse_width  >> 18;
	al_reg_write32(&adapter->ec_regs_base->pth_egress[params->index].pulse_width_subseconds_msb, reg);

	return 0;
}

int al_eth_link_status_get(struct al_hal_eth_adapter *adapter,
			   struct al_eth_link_status *status)
{
	uint32_t reg;

	if (adapter->mac_mode == AL_ETH_MAC_MODE_10GbE_Serial) {
		reg = al_reg_read32(&adapter->mac_regs_base->gen.mac_10g_stat);

		status->link_up = AL_TRUE;

		if (reg & (ETH_MAC_GEN_MAC_10G_STAT_LOC_FAULT |
					ETH_MAC_GEN_MAC_10G_STAT_REM_FAULT))
			status->link_up = AL_FALSE;

	} else if (adapter->mac_mode == AL_ETH_MAC_MODE_SGMII) {
		al_reg_write32(&adapter->mac_regs_base->sgmii.reg_addr, 1);
#ifdef CONFIG_SYNO_ALPINE_A0
		 
		reg = al_reg_read32(&adapter->mac_regs_base->sgmii.reg_data);
#endif
		reg = al_reg_read32(&adapter->mac_regs_base->sgmii.reg_data);

		status->link_up = AL_FALSE;

		if (reg & AL_BIT(2))
			status->link_up = AL_TRUE;

	} else {
		 
		return -EPERM;
	}

	return 0;
}

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
 
int al_eth_led_set(struct al_hal_eth_adapter *adapter, al_bool link_is_up)
 {
 	uint32_t reg = 0;
	uint32_t mode  = ETH_MAC_GEN_LED_CFG_SEL_DEFAULT_REG;

	if (link_is_up)
		mode = ETH_MAC_GEN_LED_CFG_SEL_LINK_ACTIVITY;

	AL_REG_FIELD_SET(reg,  ETH_MAC_GEN_LED_CFG_SEL_MASK,
			 ETH_MAC_GEN_LED_CFG_SEL_SHIFT, mode);

	AL_REG_FIELD_SET(reg, ETH_MAC_GEN_LED_CFG_BLINK_TIMER_MASK,
			 ETH_MAC_GEN_LED_CFG_BLINK_TIMER_SHIFT,
			 ETH_MAC_GEN_LED_CFG_BLINK_TIMER_VAL);
 
	AL_REG_FIELD_SET(reg, ETH_MAC_GEN_LED_CFG_ACT_TIMER_MASK,
			 ETH_MAC_GEN_LED_CFG_ACT_TIMER_SHIFT,
			 ETH_MAC_GEN_LED_CFG_ACT_TIMER_VAL);
 
 	al_reg_write32(&adapter->mac_regs_base->gen.led_cfg, reg);

	return 0;
} 
#else
 
int al_eth_led_config(struct al_hal_eth_adapter *adapter, al_bool link_is_up)
{
	uint32_t reg = 0;
	uint32_t mode  = ETH_MAC_GEN_LED_CFG_SEL_DEFAULT_REG;

	if (link_is_up)
		mode = ETH_MAC_GEN_LED_CFG_SEL_LINK_ACTIVITY;

	AL_REG_FIELD_SET(reg,  ETH_MAC_GEN_LED_CFG_SEL_MASK,
			 ETH_MAC_GEN_LED_CFG_SEL_SHIFT, mode);

	AL_REG_FIELD_SET(reg, ETH_MAC_GEN_LED_CFG_BLINK_TIMER_MASK,
			 ETH_MAC_GEN_LED_CFG_BLINK_TIMER_SHIFT,
			 ETH_MAC_GEN_LED_CFG_BLINK_TIMER_VAL);

	AL_REG_FIELD_SET(reg, ETH_MAC_GEN_LED_CFG_ACT_TIMER_MASK,
			 ETH_MAC_GEN_LED_CFG_ACT_TIMER_SHIFT,
			 ETH_MAC_GEN_LED_CFG_ACT_TIMER_VAL);

	al_reg_write32(&adapter->mac_regs_base->gen.led_cfg, reg);

	return 0;
}
#endif

int al_eth_mac_stats_get(struct al_hal_eth_adapter *adapter, struct al_eth_mac_stats *stats)
{
	al_assert(stats);

	if (AL_ETH_IS_1G_MAC(adapter->mac_mode)) {
		void __iomem *mac_1g_regs_base = &adapter->mac_regs_base->mac_1g;

		stats->ifInUcastPkts = al_reg_read32(mac_1g_regs_base + 0x90);
		stats->ifInMulticastPkts = al_reg_read32(mac_1g_regs_base + 0x94);
		stats->ifInBroadcastPkts = al_reg_read32(mac_1g_regs_base + 0x98);
		stats->etherStatsPkts = al_reg_read32(mac_1g_regs_base + 0xb4);
		stats->ifOutUcastPkts = al_reg_read32(mac_1g_regs_base + 0x9c);
		stats->ifOutMulticastPkts = al_reg_read32(mac_1g_regs_base + 0xa0);
		stats->ifOutBroadcastPkts = al_reg_read32(mac_1g_regs_base + 0xa4);
		stats->ifInErrors = al_reg_read32(mac_1g_regs_base + 0x88);
		stats->ifOutErrors = al_reg_read32(mac_1g_regs_base + 0x8c);

		stats->aFramesReceivedOK = al_reg_read32(mac_1g_regs_base + 0x6c);
		stats->aFramesTransmittedOK = al_reg_read32(mac_1g_regs_base + 0x68);

		stats->aOctetsReceivedOK = al_reg_read32(mac_1g_regs_base + 0x7c);
		stats->aOctetsTransmittedOK = al_reg_read32(mac_1g_regs_base + 0x78);

		stats->etherStatsUndersizePkts = al_reg_read32(mac_1g_regs_base + 0xB8);
		stats->etherStatsFragments = al_reg_read32(mac_1g_regs_base + 0xE0);
		stats->etherStatsJabbers = al_reg_read32(mac_1g_regs_base + 0xDC);
		stats->etherStatsOversizePkts = al_reg_read32(mac_1g_regs_base + 0xBC);

		stats->aFrameCheckSequenceErrors = al_reg_read32(mac_1g_regs_base + 0x70);
		stats->aAlignmentErrors = al_reg_read32(mac_1g_regs_base + 0x74);
		stats->etherStatsDropEvents = al_reg_read32(mac_1g_regs_base + 0xAC);
	} else {
		void __iomem *mac_10g_regs_base = &adapter->mac_regs_base->mac_10g;
		uint64_t octets;

		stats->ifInUcastPkts = al_reg_read32(mac_10g_regs_base + 0xE0);
		stats->ifInMulticastPkts = al_reg_read32(mac_10g_regs_base + 0xE8);
		stats->ifInBroadcastPkts = al_reg_read32(mac_10g_regs_base + 0xF0);
		stats->etherStatsPkts = al_reg_read32(mac_10g_regs_base + 0x130);
		stats->ifOutUcastPkts = al_reg_read32(mac_10g_regs_base + 0x108);
		stats->ifOutMulticastPkts = al_reg_read32(mac_10g_regs_base + 0x110);
		stats->ifOutBroadcastPkts = al_reg_read32(mac_10g_regs_base + 0x118);
		stats->ifInErrors = al_reg_read32(mac_10g_regs_base + 0x190);
		stats->ifOutErrors = al_reg_read32(mac_10g_regs_base + 0xf8);

		stats->aFramesReceivedOK = al_reg_read32(mac_10g_regs_base + 0x88);
		stats->aFramesTransmittedOK = al_reg_read32(mac_10g_regs_base + 0x80);
		 
		octets = al_reg_read32(mac_10g_regs_base + 0xD8);
		octets |= (uint64_t)(al_reg_read32(mac_10g_regs_base + 0xDC)) << 32;
		octets -= 18 * stats->aFramesReceivedOK;
		octets -= 4 * al_reg_read32(mac_10g_regs_base + 0xC8);
		stats->aOctetsReceivedOK = octets;

		octets = al_reg_read32(mac_10g_regs_base + 0xD0);
		octets |= (uint64_t)(al_reg_read32(mac_10g_regs_base + 0xD4)) << 32;
		octets -= 18 * stats->aFramesTransmittedOK;
		octets -= 4 * al_reg_read32(mac_10g_regs_base + 0xC0);
		stats->aOctetsTransmittedOK = octets;

		stats->etherStatsUndersizePkts = al_reg_read32(mac_10g_regs_base + 0x138);
		stats->etherStatsFragments = al_reg_read32(mac_10g_regs_base + 0x188);
		stats->etherStatsJabbers = al_reg_read32(mac_10g_regs_base + 0x180);
		stats->etherStatsOversizePkts = al_reg_read32(mac_10g_regs_base  + 0x178);

		stats->aFrameCheckSequenceErrors = al_reg_read32(mac_10g_regs_base + 0x90);
		stats->aAlignmentErrors = al_reg_read32(mac_10g_regs_base + 0x98);
		stats->etherStatsDropEvents = al_reg_read32(mac_10g_regs_base + 0x120);
	}
	stats->eee_in = al_reg_read32(&adapter->mac_regs_base->stat.eee_in);
	stats->eee_out = al_reg_read32(&adapter->mac_regs_base->stat.eee_out);

	return 0;
}

int al_eth_flr_rmn(int (* pci_read_config_u32)(void *handle, int where, uint32_t *val),
		   int (* pci_write_config_u32)(void *handle, int where, uint32_t val),
		   void *handle,
		   void __iomem	*mac_base)
{
	struct al_eth_mac_regs __iomem *mac_regs_base =
		(struct	al_eth_mac_regs __iomem *)mac_base;
	uint32_t cfg_reg_store[6];
	uint32_t reg;
	uint32_t mux_sel;
	int i = 0;

	(*pci_read_config_u32)(handle, AL_ADAPTER_GENERIC_CONTROL_0, &reg);

	AL_REG_MASK_SET(reg, AL_ADAPTER_GENERIC_CONTROL_0_ETH_RESET_1GMAC);
	(*pci_write_config_u32)(handle, AL_ADAPTER_GENERIC_CONTROL_0, reg);
	al_udelay(1000);
	 
	AL_REG_MASK_CLEAR(reg, AL_ADAPTER_GENERIC_CONTROL_0_ETH_RESET_1GMAC);
	 
	AL_REG_MASK_CLEAR(reg, AL_ADAPTER_GENERIC_CONTROL_0_ETH_RESET_1GMAC_ON_FLR);
	 
	(*pci_write_config_u32)(handle, AL_ADAPTER_GENERIC_CONTROL_0, reg);

	mux_sel = al_reg_read32(&mac_regs_base->gen.mux_sel);

	(*pci_read_config_u32)(handle, AL_PCI_COMMAND, &cfg_reg_store[i++]);
	(*pci_read_config_u32)(handle, 0xC, &cfg_reg_store[i++]);
	(*pci_read_config_u32)(handle, 0x10, &cfg_reg_store[i++]);
	(*pci_read_config_u32)(handle, 0x18, &cfg_reg_store[i++]);
	(*pci_read_config_u32)(handle, 0x20, &cfg_reg_store[i++]);
	(*pci_read_config_u32)(handle, 0x110, &cfg_reg_store[i++]);

	(*pci_write_config_u32)(handle, AL_PCI_EXP_CAP_BASE + AL_PCI_EXP_DEVCTL, AL_PCI_EXP_DEVCTL_BCR_FLR);
	al_udelay(1000);
	 
	i = 0;
	(*pci_write_config_u32)(handle, AL_PCI_COMMAND, cfg_reg_store[i++]);
	(*pci_write_config_u32)(handle, 0xC, cfg_reg_store[i++]);
	(*pci_write_config_u32)(handle, 0x10, cfg_reg_store[i++]);
	(*pci_write_config_u32)(handle, 0x18, cfg_reg_store[i++]);
	(*pci_write_config_u32)(handle, 0x20, cfg_reg_store[i++]);
	(*pci_write_config_u32)(handle, 0x110, cfg_reg_store[i++]);

	al_reg_write32_masked(&mac_regs_base->gen.mux_sel, ETH_MAC_GEN_MUX_SEL_KR_IN_MASK, mux_sel);

	al_reg_write32(mac_base + 0xB08, 0x03320501);

	AL_REG_MASK_SET(reg, AL_ADAPTER_GENERIC_CONTROL_0_ETH_RESET_1GMAC);
	(*pci_write_config_u32)(handle, AL_ADAPTER_GENERIC_CONTROL_0, reg);

	al_udelay(1000);

	AL_REG_MASK_CLEAR(reg, AL_ADAPTER_GENERIC_CONTROL_0_ETH_RESET_1GMAC);
	(*pci_write_config_u32)(handle, AL_ADAPTER_GENERIC_CONTROL_0, reg);

	al_reg_write32(mac_base + 0xB08, 0x00320501);
	al_udelay(1000);
	 
	reg = al_reg_read32(mac_base + 0x95c);
	AL_REG_MASK_SET(reg, 0xF0);
	al_reg_write32(mac_base + 0x95c, reg);
	reg = al_reg_read32(mac_base + 0x95c);
	AL_REG_MASK_CLEAR(reg, 0xF0);
	al_reg_write32(mac_base + 0x95c, reg);

	return 0;
}

#define AL_HAL_ETH_MEDIA_TYPE_MASK	(AL_FIELD_MASK(3, 0))
#define AL_HAL_ETH_MEDIA_TYPE_SHIFT	0
#define AL_HAL_ETH_EXT_PHY_SHIFT	4
#define AL_HAL_ETH_PHY_ADDR_MASK	(AL_FIELD_MASK(9, 5))
#define AL_HAL_ETH_PHY_ADDR_SHIFT	5
#define AL_HAL_ETH_SFP_EXIST_SHIFT	10
#define AL_HAL_ETH_AN_ENABLE_SHIFT	11
#define AL_HAL_ETH_KR_LT_ENABLE_SHIFT	12
#define AL_HAL_ETH_KR_FEC_ENABLE_SHIFT	13
#define AL_HAL_ETH_MDIO_FREQ_MASK	(AL_FIELD_MASK(15, 14))
#define AL_HAL_ETH_MDIO_FREQ_SHIFT	14
#define AL_HAL_ETH_I2C_ADAPTER_ID_MASK	(AL_FIELD_MASK(19, 16))
#define AL_HAL_ETH_I2C_ADAPTER_ID_SHIFT	16
#define AL_HAL_ETH_EXT_PHY_IF_MASK	(AL_FIELD_MASK(21, 20))
#define AL_HAL_ETH_EXT_PHY_IF_SHIFT	20
#define AL_HAL_ETH_AUTO_NEG_MODE_SHIFT	22
#define AL_HAL_ETH_SERDES_GRP_MASK	(AL_FIELD_MASK(26, 25))
#define AL_HAL_ETH_SERDES_GRP_SHIFT	25
#define AL_HAL_ETH_SERDES_LANE_MASK	(AL_FIELD_MASK(28, 27))
#define AL_HAL_ETH_SERDES_LANE_SHIFT	27
#define AL_HAL_ETH_REF_CLK_FREQ_MASK	(AL_FIELD_MASK(31, 29))
#define AL_HAL_ETH_REF_CLK_FREQ_SHIFT	29
#ifdef CONFIG_SYNO_ALPINE_A0
 
#define AL_HAL_ETH_DONT_OVERRIDE_SERDES_SHIFT	0
#define AL_HAL_ETH_1000_BASE_X_SHIFT		1
#define AL_HAL_ETH_1G_AN_DISABLE_SHIFT		2
#define AL_HAL_ETH_1G_SPEED_MASK		(AL_FIELD_MASK(4, 3))
#define AL_HAL_ETH_1G_SPEED_SHIFT		3
#define AL_HAL_ETH_1G_HALF_DUPLEX_SHIFT		5
#define AL_HAL_ETH_1G_FC_DISABLE_SHIFT		6
#endif
#ifdef CONFIG_SYNO_ALPINE_V2_5_3
#define AL_HAL_ETH_RETIMER_EXIST_SHIFT		7
#define AL_HAL_ETH_RETIMER_BUS_ID_MASK		(AL_FIELD_MASK(11, 8))
#define AL_HAL_ETH_RETIMER_BUS_ID_SHIFT		8
#define AL_HAL_ETH_RETIMER_I2C_ADDR_MASK	(AL_FIELD_MASK(18, 12))
#define AL_HAL_ETH_RETIMER_I2C_ADDR_SHIFT	12
#define AL_HAL_ETH_RETIMER_CHANNEL_SHIFT	19
#define AL_HAL_ETH_DAC_LENGTH_MASK		(AL_FIELD_MASK(23, 20))
#define AL_HAL_ETH_DAC_LENGTH_SHIFT		20
#define AL_HAL_ETH_DAC_SHIFT			24 
#endif

int al_eth_board_params_set(void * __iomem mac_base, struct al_eth_board_params *params){
	uint32_t	reg = 0;

	AL_REG_FIELD_SET(reg, AL_HAL_ETH_MEDIA_TYPE_MASK,
			 AL_HAL_ETH_MEDIA_TYPE_SHIFT, params->media_type);
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_EXT_PHY_SHIFT, params->phy_exist == AL_TRUE);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_PHY_ADDR_MASK,
			 AL_HAL_ETH_PHY_ADDR_SHIFT, params->phy_mdio_addr);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_SFP_EXIST_SHIFT, params->sfp_plus_module_exist == AL_TRUE);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_AN_ENABLE_SHIFT, params->autoneg_enable == AL_TRUE);
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_KR_LT_ENABLE_SHIFT, params->kr_lt_enable == AL_TRUE);
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_KR_FEC_ENABLE_SHIFT, params->kr_fec_enable == AL_TRUE);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_MDIO_FREQ_MASK,
			 AL_HAL_ETH_MDIO_FREQ_SHIFT, params->mdio_freq);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_I2C_ADAPTER_ID_MASK,
			 AL_HAL_ETH_I2C_ADAPTER_ID_SHIFT, params->i2c_adapter_id);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_EXT_PHY_IF_MASK,
			 AL_HAL_ETH_EXT_PHY_IF_SHIFT, params->phy_if);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_AUTO_NEG_MODE_SHIFT,
			   params->an_mode == AL_ETH_BOARD_AUTONEG_IN_BAND);

	AL_REG_FIELD_SET(reg, AL_HAL_ETH_SERDES_GRP_MASK,
			 AL_HAL_ETH_SERDES_GRP_SHIFT, params->serdes_grp);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_SERDES_LANE_MASK,
			 AL_HAL_ETH_SERDES_LANE_SHIFT, params->serdes_lane);

	AL_REG_FIELD_SET(reg, AL_HAL_ETH_REF_CLK_FREQ_MASK,
			 AL_HAL_ETH_REF_CLK_FREQ_SHIFT, params->ref_clk_freq);
#ifdef CONFIG_SYNO_ALPINE_A0
	al_assert(reg != 0);
#endif

	al_reg_write32(mac_base + 0x4, reg);
#ifdef CONFIG_SYNO_ALPINE_A0
	 
	reg = 0;
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_DONT_OVERRIDE_SERDES_SHIFT,
			   params->dont_override_serdes == AL_TRUE);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_1000_BASE_X_SHIFT,
			   params->force_1000_base_x == AL_TRUE);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_1G_AN_DISABLE_SHIFT,
			   params->an_disable == AL_TRUE);

	AL_REG_FIELD_SET(reg, AL_HAL_ETH_1G_SPEED_MASK,
			 AL_HAL_ETH_1G_SPEED_SHIFT, params->speed);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_1G_HALF_DUPLEX_SHIFT,
			   params->half_duplex == AL_TRUE);

	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_1G_FC_DISABLE_SHIFT,
			   params->fc_disable == AL_TRUE);

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_RETIMER_EXIST_SHIFT, params->retimer_exist == AL_TRUE);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_RETIMER_BUS_ID_MASK,
			 AL_HAL_ETH_RETIMER_BUS_ID_SHIFT, params->retimer_bus_id);
	AL_REG_FIELD_SET(reg, AL_HAL_ETH_RETIMER_I2C_ADDR_MASK,
			 AL_HAL_ETH_RETIMER_I2C_ADDR_SHIFT, params->retimer_i2c_addr);
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_RETIMER_CHANNEL_SHIFT, params->retimer_channel);

	AL_REG_FIELD_SET(reg, AL_HAL_ETH_DAC_LENGTH_MASK,
			 AL_HAL_ETH_DAC_LENGTH_SHIFT, params->dac_len);
	AL_REG_BIT_VAL_SET(reg, AL_HAL_ETH_DAC_SHIFT, params->dac);
#endif

	al_reg_write32(mac_base + 0x404, reg);
#endif
	return 0;
}

int al_eth_board_params_get(void * __iomem mac_base, struct al_eth_board_params *params){
	uint32_t	reg = al_reg_read32(mac_base + 0x4);
#ifdef CONFIG_SYNO_ALPINE_A0
	 
	if (reg == 0)
		return -ENOENT;

#endif
	params->media_type = AL_REG_FIELD_GET(reg, AL_HAL_ETH_MEDIA_TYPE_MASK,
					      AL_HAL_ETH_MEDIA_TYPE_SHIFT);
	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_EXT_PHY_SHIFT))
		params->phy_exist = AL_TRUE;
	else
		params->phy_exist = AL_FALSE;

	params->phy_mdio_addr = AL_REG_FIELD_GET(reg, AL_HAL_ETH_PHY_ADDR_MASK,
						 AL_HAL_ETH_PHY_ADDR_SHIFT);

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_SFP_EXIST_SHIFT))
		params->sfp_plus_module_exist = AL_TRUE;
	else
		params->sfp_plus_module_exist = AL_FALSE;

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_AN_ENABLE_SHIFT))
		params->autoneg_enable = AL_TRUE;
	else
		params->autoneg_enable = AL_FALSE;

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_KR_LT_ENABLE_SHIFT))
		params->kr_lt_enable = AL_TRUE;
	else
		params->kr_lt_enable = AL_FALSE;

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_KR_FEC_ENABLE_SHIFT))
		params->kr_fec_enable = AL_TRUE;
	else
		params->kr_fec_enable = AL_FALSE;

	params->mdio_freq = AL_REG_FIELD_GET(reg,
					     AL_HAL_ETH_MDIO_FREQ_MASK,
					     AL_HAL_ETH_MDIO_FREQ_SHIFT);

	params->i2c_adapter_id = AL_REG_FIELD_GET(reg,
						  AL_HAL_ETH_I2C_ADAPTER_ID_MASK,
						  AL_HAL_ETH_I2C_ADAPTER_ID_SHIFT);

	params->phy_if = AL_REG_FIELD_GET(reg,
					  AL_HAL_ETH_EXT_PHY_IF_MASK,
					  AL_HAL_ETH_EXT_PHY_IF_SHIFT);

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_AUTO_NEG_MODE_SHIFT))
		params->an_mode = AL_TRUE;
	else
		params->an_mode = AL_FALSE;

	params->serdes_grp = AL_REG_FIELD_GET(reg,
					      AL_HAL_ETH_SERDES_GRP_MASK,
					      AL_HAL_ETH_SERDES_GRP_SHIFT);

	params->serdes_lane = AL_REG_FIELD_GET(reg,
					       AL_HAL_ETH_SERDES_LANE_MASK,
					       AL_HAL_ETH_SERDES_LANE_SHIFT);

	params->ref_clk_freq = AL_REG_FIELD_GET(reg,
						AL_HAL_ETH_REF_CLK_FREQ_MASK,
						AL_HAL_ETH_REF_CLK_FREQ_SHIFT);
#ifdef CONFIG_SYNO_ALPINE_A0
	 
	reg = al_reg_read32(mac_base + 0x404);
	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_DONT_OVERRIDE_SERDES_SHIFT))
		params->dont_override_serdes = AL_TRUE;
	else
		params->dont_override_serdes = AL_FALSE;

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_1000_BASE_X_SHIFT))
		params->force_1000_base_x = AL_TRUE;
	else
		params->force_1000_base_x = AL_FALSE;

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_1G_AN_DISABLE_SHIFT))
		params->an_disable = AL_TRUE;
	else
		params->an_disable = AL_FALSE;

	params->speed = AL_REG_FIELD_GET(reg,
					 AL_HAL_ETH_1G_SPEED_MASK,
					 AL_HAL_ETH_1G_SPEED_SHIFT);

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_1G_HALF_DUPLEX_SHIFT))
		params->half_duplex = AL_TRUE;
	else
		params->half_duplex = AL_FALSE;

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_1G_FC_DISABLE_SHIFT))
		params->fc_disable = AL_TRUE;
	else
		params->fc_disable = AL_FALSE;
#endif

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_RETIMER_EXIST_SHIFT))
		params->retimer_exist = AL_TRUE;
	else
		params->retimer_exist = AL_FALSE;

	params->retimer_bus_id = AL_REG_FIELD_GET(reg,
					       AL_HAL_ETH_RETIMER_BUS_ID_MASK,
					       AL_HAL_ETH_RETIMER_BUS_ID_SHIFT);
	params->retimer_i2c_addr = AL_REG_FIELD_GET(reg,
					       AL_HAL_ETH_RETIMER_I2C_ADDR_MASK,
					       AL_HAL_ETH_RETIMER_I2C_ADDR_SHIFT);

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_RETIMER_CHANNEL_SHIFT))
		params->retimer_channel = AL_ETH_RETIMER_CHANNEL_B;
	else
		params->retimer_channel = AL_ETH_RETIMER_CHANNEL_A;

	params->dac_len = AL_REG_FIELD_GET(reg,
					   AL_HAL_ETH_DAC_LENGTH_MASK,
					   AL_HAL_ETH_DAC_LENGTH_SHIFT);

	if (AL_REG_BIT_GET(reg, AL_HAL_ETH_DAC_SHIFT))
		params->dac = AL_TRUE;
	else
		params->dac = AL_FALSE;
#endif

	return 0;
}

static inline void al_eth_byte_arr_to_reg(
		uint32_t *reg, uint8_t *arr, unsigned int num_bytes)
{
	uint32_t mask = 0xff;
	unsigned int i;

	al_assert(num_bytes <= 4);

	*reg = 0;

	for (i = 0 ; i < num_bytes ; i++) {
		AL_REG_FIELD_SET(*reg, mask, (sizeof(uint8_t) * i), arr[i]);
		mask = mask << sizeof(uint8_t);
	}
}

int al_eth_wol_enable(
		struct al_hal_eth_adapter *adapter,
		struct al_eth_wol_params *wol)
{
	uint32_t reg = 0;

	if (wol->int_mask & AL_ETH_WOL_INT_MAGIC_PSWD) {
		al_assert(wol->pswd != NULL);

		al_eth_byte_arr_to_reg(&reg, &wol->pswd[0], 4);
		al_reg_write32(&adapter->ec_regs_base->wol.magic_pswd_l, reg);

		al_eth_byte_arr_to_reg(&reg, &wol->pswd[4], 2);
		al_reg_write32(&adapter->ec_regs_base->wol.magic_pswd_h, reg);
	}

	if (wol->int_mask & AL_ETH_WOL_INT_IPV4) {
		al_assert(wol->ipv4 != NULL);

		al_eth_byte_arr_to_reg(&reg, &wol->ipv4[0], 4);
		al_reg_write32(&adapter->ec_regs_base->wol.ipv4_dip, reg);
	}

	if (wol->int_mask & AL_ETH_WOL_INT_IPV6) {
		al_assert(wol->ipv6 != NULL);

		al_eth_byte_arr_to_reg(&reg, &wol->ipv6[0], 4);
		al_reg_write32(&adapter->ec_regs_base->wol.ipv6_dip_word0, reg);

		al_eth_byte_arr_to_reg(&reg, &wol->ipv6[4], 4);
		al_reg_write32(&adapter->ec_regs_base->wol.ipv6_dip_word1, reg);

		al_eth_byte_arr_to_reg(&reg, &wol->ipv6[8], 4);
		al_reg_write32(&adapter->ec_regs_base->wol.ipv6_dip_word2, reg);

		al_eth_byte_arr_to_reg(&reg, &wol->ipv6[12], 4);
		al_reg_write32(&adapter->ec_regs_base->wol.ipv6_dip_word3, reg);
	}

	if (wol->int_mask &
		(AL_ETH_WOL_INT_ETHERTYPE_BC | AL_ETH_WOL_INT_ETHERTYPE_DA)) {

		reg = ((uint32_t)wol->ethr_type2 << 16);
		reg |= wol->ethr_type1;

		al_reg_write32(&adapter->ec_regs_base->wol.ethertype, reg);
	}

	al_assert((wol->forward_mask | wol->int_mask) == wol->int_mask);

	reg = ((uint32_t)wol->forward_mask << 16);
	reg |= wol->int_mask;
	al_reg_write32(&adapter->ec_regs_base->wol.wol_en, reg);

	return 0;
}

int al_eth_wol_disable(
		struct al_hal_eth_adapter *adapter)
{
	al_reg_write32(&adapter->ec_regs_base->wol.wol_en, 0);

	return 0;
}
