 
#ifndef __AL_CRYPTO_H__
#define __AL_CRYPTO_H__

#include <linux/init.h>
#include <linux/cache.h>
#include <linux/circ_buf.h>
#include <linux/pci_ids.h>
#include <linux/semaphore.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include "linux/scatterlist.h"
#include "crypto/scatterwalk.h"
#include "linux/crypto.h"

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
#include "al_hal_ssm_crypto.h"
#include "al_hal_ssm_crc_memcpy.h"
#else
#include "al_hal_crypto.h"
#endif

#define AL_CRYPTO_VERSION  "0.01"

#ifndef CONFIG_ALPINE_VP_WA
#define AL_CRYPTO_TX_CDESC_SIZE			8
#define AL_CRYPTO_RX_CDESC_SIZE			8
#else
 
#define AL_CRYPTO_TX_CDESC_SIZE			16
#define AL_CRYPTO_RX_CDESC_SIZE			16
#endif

#define AL_CRYPTO_DMA_MAX_CHANNELS		4

#define AL_CRYPTO_MSIX_INTERRUPTS		AL_CRYPTO_DMA_MAX_CHANNELS + 1

#define AL_CRYPTO_SW_RING_MIN_ORDER		4
#define AL_CRYPTO_SW_RING_MAX_ORDER		16

#define AL_CRYPTO_OP_MAX_BUFS			27
#define AL_CRYPTO_HASH_HMAC_IPAD		0x36
#define AL_CRYPTO_HASH_HMAC_OPAD		0x5c

#define AL_CRYPTO_MAX_IV_LENGTH	16  

#ifdef CONFIG_CRYPTO_DEV_AL_CRYPTO_STATS
#define AL_CRYPTO_STATS_INC(var, incval)	(var) += (incval)
#define AL_CRYPTO_STATS_DEC(var, decval)	(var) -= (decval)
#define AL_CRYPTO_STATS_SET(var, val)		(var) = (val)
#define AL_CRYPTO_STATS_LOCK(lock)		\
	spin_lock_bh(lock)
#define AL_CRYPTO_STATS_UNLOCK(lock)	\
	spin_unlock_bh(lock)
#define AL_CRYPTO_STATS_INIT_LOCK(lock)	\
	spin_lock_init(lock)
#else
#define AL_CRYPTO_STATS_INC(var, incval)
#define AL_CRYPTO_STATS_DEC(var, decval)
#define AL_CRYPTO_STATS_SET(var, val)
#define AL_CRYPTO_STATS_LOCK(lock)
#define AL_CRYPTO_STATS_LOCK(lock)
#define AL_CRYPTO_STATS_UNLOCK(lock)
#define AL_CRYPTO_STATS_INIT_LOCK(lock)
#endif

#define AL_CRYPTO_IRQNAME_SIZE			40

#define AL_CRYPTO_INT_MODER_RES			1

#define CHKSUM_BLOCK_SIZE       1
#define CHKSUM_DIGEST_SIZE      4

#define MAX_CACHE_ENTRIES_PER_CHANNEL 	CACHED_SAD_SIZE

enum al_crypto_req_type {
	AL_CRYPTO_REQ_ABLKCIPHER,
	AL_CRYPTO_REQ_AEAD,
	AL_CRYPTO_REQ_AHASH,
	AL_CRYPTO_REQ_CRC,
};

struct al_crypto_sw_desc {
	union {
		struct al_crypto_transaction	hal_xaction;
		struct al_crc_transaction	hal_crc_xaction;
	};

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	struct al_buf	src_bufs[AL_SSM_MAX_SRC_DESCS];
	struct al_buf	dst_bufs[AL_SSM_MAX_SRC_DESCS];
#else
	struct al_buf	src_bufs[AL_CRYPTO_MAX_DESCS];
	struct al_buf	dst_bufs[AL_CRYPTO_MAX_DESCS];
#endif

	void	*req;
	int	req_type;
	int	src_nents;
	int	assoc_nents;
	int	dst_nents;
	bool	src_chained;
	bool	assoc_chained;
	bool	dst_chained;
};

struct al_crypto_cache_lru_entry {
	struct list_head		list;
	struct al_crypto_cache_state	*ctx;
	u32				cache_idx;
};

#ifdef CONFIG_CRYPTO_DEV_AL_CRYPTO_STATS
 
struct al_crypto_chan_stats_gen {
	uint64_t ablkcipher_tfms;
	uint64_t aead_tfms;
	uint64_t ahash_tfms;
	uint64_t crc_tfms;
};

struct al_crypto_chan_stats_prep {
	uint64_t ablkcipher_encrypt_reqs;
	uint64_t ablkcipher_encrypt_bytes;
	uint64_t ablkcipher_decrypt_reqs;
	uint64_t ablkcipher_decrypt_bytes;
	uint64_t aead_encrypt_hash_reqs;
	uint64_t aead_encrypt_bytes;
	uint64_t aead_hash_bytes;
	uint64_t aead_decrypt_validate_reqs;
	uint64_t aead_decrypt_bytes;
	uint64_t aead_validate_bytes;
	uint64_t ahash_reqs;
	uint64_t ahash_bytes;
	uint64_t crc_reqs;
	uint64_t crc_bytes;
	uint64_t cache_misses;
	uint64_t ablkcipher_reqs_le512;
	uint64_t ablkcipher_reqs_512_2048;
	uint64_t ablkcipher_reqs_2048_4096;
	uint64_t ablkcipher_reqs_gt4096;
	uint64_t aead_reqs_le512;
	uint64_t aead_reqs_512_2048;
	uint64_t aead_reqs_2048_4096;
	uint64_t aead_reqs_gt4096;
	uint64_t ahash_reqs_le512;
	uint64_t ahash_reqs_512_2048;
	uint64_t ahash_reqs_2048_4096;
	uint64_t ahash_reqs_gt4096;
	uint64_t crc_reqs_le512;
	uint64_t crc_reqs_512_2048;
	uint64_t crc_reqs_2048_4096;
	uint64_t crc_reqs_gt4096;
};

struct al_crypto_chan_stats_comp {
	uint64_t redundant_int_cnt;
	uint64_t max_active_descs;
};
#endif

struct al_crypto_chan {
#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	struct al_ssm_dma	*hal_crypto;
#else
	struct al_crypto_dma	*hal_crypto;
#endif
	int	idx;
#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	enum al_ssm_q_type type;
#else
	enum al_crypto_q_type type;
#endif
	cpumask_t      affinity_hint_mask;

	int tx_descs_num;	 
	void *tx_dma_desc_virt;  
	dma_addr_t tx_dma_desc;

	int rx_descs_num;	 
	void *rx_dma_desc_virt;  
	dma_addr_t rx_dma_desc;
	void *rx_dma_cdesc_virt;  
	dma_addr_t rx_dma_cdesc;

	u16 alloc_order;
	struct al_crypto_sw_desc **sw_ring;
#ifdef CONFIG_CRYPTO_DEV_AL_CRYPTO_STATS
	struct al_crypto_chan_stats_gen stats_gen;
	spinlock_t stats_gen_lock;  
#endif

	spinlock_t prep_lock ____cacheline_aligned;	 
	u16 head;
	int sw_desc_num_locked;  
	u32 tx_desc_produced;	 
	struct crypto_queue sw_queue;  
#ifdef CONFIG_CRYPTO_DEV_AL_CRYPTO_STATS
	struct al_crypto_chan_stats_prep stats_prep;
#endif

	int cache_entries_num;
	struct list_head cache_lru_list;
	int cache_lru_count;
	struct al_crypto_cache_lru_entry cache_lru_entries[
					MAX_CACHE_ENTRIES_PER_CHANNEL];

	spinlock_t cleanup_lock ____cacheline_aligned_in_smp;  
	u16 tail;
#ifdef CONFIG_CRYPTO_DEV_AL_CRYPTO_STATS
	struct al_crypto_chan_stats_comp stats_comp;
#endif

	struct al_crypto_device *device;
	struct tasklet_struct cleanup_task;
	struct kobject kobj;
};

#define to_dev(al_crypto_chan) (&(al_crypto_chan)->device->pdev->dev)

struct al_crypto_irq {
	char name[AL_CRYPTO_IRQNAME_SIZE];
};

struct al_crypto_device {
	struct pci_dev		*pdev;
#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	struct al_ssm_dma_params	ssm_dma_params;
#else
	struct al_crypto_dma_params	crypto_dma_params;
#endif
	void __iomem	*udma_regs_base;
	void __iomem	*crypto_regs_base;

#ifdef CONFIG_SYNO_ALPINE_V2_5_3
	struct al_ssm_dma	hal_crypto;
#else
	struct al_crypto_dma	hal_crypto;
#endif

	struct msix_entry	msix_entries[AL_CRYPTO_MSIX_INTERRUPTS];
	struct al_crypto_irq	irq_tbl[AL_CRYPTO_MSIX_INTERRUPTS];
	struct al_crypto_chan	*channels[AL_CRYPTO_DMA_MAX_CHANNELS];
	int			num_channels;
	int			max_channels;
	int			crc_channels;
	struct kset 		*channels_kset;
	struct tasklet_struct	cleanup_task;
	int			int_moderation;
	int			num_irq_used;

	struct kmem_cache	*cache;	    
	atomic_t		tfm_count;  
	atomic_t		crc_tfm_count;  
	struct list_head alg_list;   
	struct list_head hash_list;  
	struct list_head crc_list;  
};

struct al_crypto_cache_state {
	bool			cached;
	int			idx;
};

struct al_crypto_ctx {
	struct al_crypto_chan	*chan;
	struct al_crypto_cache_state	cache_state;
	struct al_crypto_sa	sa;
	struct al_crypto_hw_sa	*hw_sa;
	dma_addr_t		hw_sa_dma_addr;
	struct crypto_shash	*sw_hash;	 
	u8 			*iv;
	dma_addr_t		iv_dma_addr;
};

static inline u16 al_crypto_ring_size(struct al_crypto_chan *chan)
{
	return 1 << chan->alloc_order;
}

static inline u16 al_crypto_ring_active(struct al_crypto_chan *chan)
{
	return CIRC_CNT(chan->head, chan->tail, al_crypto_ring_size(chan));
}
static inline u16 al_crypto_ring_space(struct al_crypto_chan *chan)
{
	return CIRC_SPACE(chan->head, chan->tail, al_crypto_ring_size(chan));
}

static inline struct al_crypto_sw_desc  *
al_crypto_get_ring_ent(struct al_crypto_chan *chan, u16 idx)
{
	return chan->sw_ring[idx & (al_crypto_ring_size(chan) - 1)];
}

int al_crypto_get_sw_desc(struct al_crypto_chan *chan, int num);

void al_crypto_tx_submit(struct al_crypto_chan *chan);

#ifdef DEBUG
#define set_desc_id(desc, i) ((desc)->id = (i))
#define desc_id(desc) ((desc)->id)
#else
#define set_desc_id(desc, i)
#define desc_id(desc) (0)
#endif

#ifdef DEBUG
static inline void hexdump(unsigned char *buf, unsigned int len)
{
	print_hex_dump(KERN_DEBUG, "", DUMP_PREFIX_OFFSET,
			16, 1,
			buf, len, false);
}
#else
static inline void hexdump(unsigned char *buf, unsigned int len) {}
#endif

static inline int sg_count(struct scatterlist *sg_list, int nbytes,
			bool *chained)
{
	struct scatterlist *sg = sg_list;
	int sg_nents = 0;

	while (nbytes > 0) {
		sg_nents++;
		nbytes -= sg->length;
		if (!sg_is_last(sg) && (sg + 1)->length == 0)
			*chained = true;
		sg = scatterwalk_sg_next(sg);
	}

	return sg_nents;
}

static inline void sg_copy(u8 *dest, struct scatterlist *sg, unsigned int len)
{
	struct scatterlist *current_sg = sg;
	int cpy_index = 0, next_cpy_index = current_sg->length;
	void* virtaddr;

	while (next_cpy_index < len) {
		virtaddr = kmap_atomic(sg_page(sg)) + sg->offset;
		memcpy(dest + cpy_index, (u8 *) virtaddr,
		       current_sg->length);
		kunmap_atomic(virtaddr);

		current_sg = scatterwalk_sg_next(current_sg);
		cpy_index = next_cpy_index;
		next_cpy_index += current_sg->length;
	}

	if (cpy_index < len) {
		virtaddr = kmap_atomic(sg_page(current_sg)) + current_sg->offset;
		memcpy(dest + cpy_index, (u8 *) virtaddr,
		       len - cpy_index);
		kunmap_atomic(virtaddr);

	}
}

static inline void sg_copy_part(u8 *dest, struct scatterlist *sg,
				      int to_skip, unsigned int end)
{
	struct scatterlist *current_sg = sg;
	uint32_t sg_index, cpy_index;
	void * virtaddr;

	sg_index = current_sg->length;
	while (sg_index <= to_skip) {
		current_sg = scatterwalk_sg_next(current_sg);
		sg_index += current_sg->length;
	}
	cpy_index = sg_index - to_skip;
	virtaddr = kmap_atomic(sg_page(sg)) + sg->offset;
	if (end <= sg_index) {
		memcpy(dest, (u8 *) virtaddr +
				current_sg->length - cpy_index,
				end - to_skip);
	} else {
		memcpy(dest, (u8 *) virtaddr +
				current_sg->length - cpy_index, cpy_index);
		current_sg = scatterwalk_sg_next(current_sg);
		sg_copy(dest + cpy_index, current_sg, end - sg_index);
	}
	kunmap_atomic(virtaddr);
}

static inline void sg_map_to_xaction_buffers(
		struct scatterlist *sg_in,
		struct al_buf* bufs,
		unsigned int length,
		int *buf_idx)
{
	struct scatterlist *sg, *sg_next;
	unsigned int remain;
	bool contig;

	sg = sg_in;
	remain = length;
	if (remain) {
		bufs[*buf_idx].addr = sg_dma_address(sg);
		bufs[*buf_idx].len = 0;
		while (remain > sg_dma_len(sg)) {
			bufs[*buf_idx].len += sg_dma_len(sg);
			remain -= sg_dma_len(sg);
			sg_next = scatterwalk_sg_next(sg);
			contig = (sg_dma_address(sg) + sg_dma_len(sg) ==
				sg_dma_address(sg_next));
			if (!contig) {
				(*buf_idx)++;
				bufs[*buf_idx].addr =
						sg_dma_address(sg_next);
				bufs[*buf_idx].len = 0;
			}
			sg = sg_next;
		}
		 
		bufs[*buf_idx].len += remain;
		(*buf_idx)++;
	}

}

int dma_map_sg_chained(struct device *dev, struct scatterlist *sg,
			unsigned int nents, enum dma_data_direction dir,
			bool chained);

int dma_unmap_sg_chained(struct device *dev, struct scatterlist *sg,
				unsigned int nents, enum dma_data_direction dir,
				bool chained);

void al_crypto_cache_update_lru(struct al_crypto_chan *chan,
		struct al_crypto_cache_state *ctx);

u32 al_crypto_cache_replace_lru(struct al_crypto_chan *chan,
		struct al_crypto_cache_state *ctx,
		struct al_crypto_cache_state **old_ctx);

void al_crypto_cache_remove_lru(struct al_crypto_chan *chan,
		struct al_crypto_cache_state *ctx);

int al_crypto_core_init(
	struct al_crypto_device	*device,
	void __iomem		*iobase_udma,
	void __iomem		*iobase_app);

int al_crypto_core_terminate(
	struct al_crypto_device	*device);

int al_crypto_cleanup_fn(
	struct al_crypto_chan	*chan,
	int			from_tasklet);

void al_crypto_set_int_moderation(
	struct al_crypto_device	*device,
	int			usec);

int al_crypto_get_int_moderation(
	struct al_crypto_device *device);

void al_crypto_cleanup_single_ablkcipher(
	struct al_crypto_chan		*chan,
	struct al_crypto_sw_desc	*desc,
	u32				comp_status);

int ablkcipher_process_queue(struct al_crypto_chan *chan);

void al_crypto_cleanup_single_aead(
	struct al_crypto_chan		*chan,
	struct al_crypto_sw_desc	*desc,
	u32				comp_status);

void al_crypto_cleanup_single_ahash(
	struct al_crypto_chan		*chan,
	struct al_crypto_sw_desc	*desc,
	u32				comp_status);

void al_crypto_cleanup_single_crc(
	struct al_crypto_chan		*chan,
	struct al_crypto_sw_desc	*desc,
	u32				comp_status);

int hmac_setkey(struct al_crypto_ctx *ctx,
	const u8			*key,
	unsigned int			keylen,
	unsigned int			sw_hash_interm_offset,
	unsigned int			sw_hash_interm_size);

void al_crypto_free_channel(struct al_crypto_chan *chan);

int al_crypto_sysfs_init(struct al_crypto_device *device);

void al_crypto_sysfs_terminate(struct al_crypto_device *device);

int al_crypto_alg_init(struct al_crypto_device *device);

void al_crypto_alg_terminate(struct al_crypto_device *device);

int al_crypto_hash_init(struct al_crypto_device *device);

void al_crypto_hash_terminate(struct al_crypto_device *device);

int al_crypto_crc_init(struct al_crypto_device *device);

void al_crypto_crc_terminate(struct al_crypto_device *device);

#endif	 
