#ifndef MY_ABC_HERE
#define MY_ABC_HERE
#endif
 
#ifndef __LINUX_USB_PHY_H
#define __LINUX_USB_PHY_H

#include <linux/notifier.h>
#include <linux/usb.h>

enum usb_phy_events {
	USB_EVENT_NONE,          
	USB_EVENT_VBUS,          
	USB_EVENT_ID,            
	USB_EVENT_CHARGER,       
	USB_EVENT_ENUMERATED,    
};

enum usb_phy_type {
	USB_PHY_TYPE_UNDEFINED,
	USB_PHY_TYPE_USB2,
	USB_PHY_TYPE_USB3,
};

enum usb_otg_state {
	OTG_STATE_UNDEFINED = 0,

	OTG_STATE_B_IDLE,
	OTG_STATE_B_SRP_INIT,
	OTG_STATE_B_PERIPHERAL,

	OTG_STATE_B_WAIT_ACON,
	OTG_STATE_B_HOST,

	OTG_STATE_A_IDLE,
	OTG_STATE_A_WAIT_VRISE,
	OTG_STATE_A_WAIT_BCON,
	OTG_STATE_A_HOST,
	OTG_STATE_A_SUSPEND,
	OTG_STATE_A_PERIPHERAL,
	OTG_STATE_A_WAIT_VFALL,
	OTG_STATE_A_VBUS_ERR,
};

struct usb_phy;
struct usb_otg;

struct usb_phy_io_ops {
	int (*read)(struct usb_phy *x, u32 reg);
	int (*write)(struct usb_phy *x, u32 val, u32 reg);
};

struct usb_phy {
	struct device		*dev;
	const char		*label;
	unsigned int		 flags;

	enum usb_phy_type	type;
	enum usb_otg_state	state;
	enum usb_phy_events	last_event;

	struct usb_otg		*otg;

	struct device		*io_dev;
	struct usb_phy_io_ops	*io_ops;
	void __iomem		*io_priv;

	struct atomic_notifier_head	notifier;

	u16			port_status;
	u16			port_change;

	struct list_head	head;

	int	(*init)(struct usb_phy *x);
	void	(*shutdown)(struct usb_phy *x);

	int	(*set_vbus)(struct usb_phy *x, int on);

	int	(*set_power)(struct usb_phy *x,
				unsigned mA);

	int	(*set_suspend)(struct usb_phy *x,
				int suspend);

#if defined(MY_DEF_HERE)
	int	(*notify_connect)(struct usb_phy *x,
			struct usb_device *udev);
	int	(*notify_disconnect)(struct usb_phy *x,
			struct usb_device *udev);
#else  
	int	(*notify_connect)(struct usb_phy *x,
			enum usb_device_speed speed);
	int	(*notify_disconnect)(struct usb_phy *x,
			enum usb_device_speed speed);
#endif  
};

struct usb_phy_bind {
	const char	*dev_name;
	const char	*phy_dev_name;
	u8		index;
	struct usb_phy	*phy;
	struct list_head list;
};

extern int usb_add_phy(struct usb_phy *, enum usb_phy_type type);
extern int usb_add_phy_dev(struct usb_phy *);
extern void usb_remove_phy(struct usb_phy *);

static inline int usb_phy_io_read(struct usb_phy *x, u32 reg)
{
#if defined (MY_DEF_HERE)
	if (x && x->io_ops && x->io_ops->read)
		return x->io_ops->read(x, reg);
#else  
	if (x->io_ops && x->io_ops->read)
		return x->io_ops->read(x, reg);
#endif  

	return -EINVAL;
}

static inline int usb_phy_io_write(struct usb_phy *x, u32 val, u32 reg)
{
#if defined (MY_DEF_HERE)
	if (x && x->io_ops && x->io_ops->write)
		return x->io_ops->write(x, val, reg);
#else  
	if (x->io_ops && x->io_ops->write)
		return x->io_ops->write(x, val, reg);
#endif  

	return -EINVAL;
}

static inline int
usb_phy_init(struct usb_phy *x)
{
#if defined (MY_DEF_HERE)
	if (x && x->init)
		return x->init(x);
#else  
	if (x->init)
		return x->init(x);
#endif  

	return 0;
}

static inline void
usb_phy_shutdown(struct usb_phy *x)
{
#if defined (MY_DEF_HERE)
	if (x && x->shutdown)
		x->shutdown(x);
#else  
	if (x->shutdown)
		x->shutdown(x);
#endif  
}

static inline int
usb_phy_vbus_on(struct usb_phy *x)
{
#if defined (MY_DEF_HERE)
	if (!x || !x->set_vbus)
		return 0;
#else  
	if (!x->set_vbus)
		return 0;
#endif  

	return x->set_vbus(x, true);
}

static inline int
usb_phy_vbus_off(struct usb_phy *x)
{
#if defined (MY_DEF_HERE)
	if (!x || !x->set_vbus)
		return 0;
#else  
	if (!x->set_vbus)
		return 0;
#endif  

	return x->set_vbus(x, false);
}

#if IS_ENABLED(CONFIG_USB_PHY)
extern struct usb_phy *usb_get_phy(enum usb_phy_type type);
extern struct usb_phy *devm_usb_get_phy(struct device *dev,
	enum usb_phy_type type);
extern struct usb_phy *usb_get_phy_dev(struct device *dev, u8 index);
extern struct usb_phy *devm_usb_get_phy_dev(struct device *dev, u8 index);
extern struct usb_phy *devm_usb_get_phy_by_phandle(struct device *dev,
	const char *phandle, u8 index);
extern void usb_put_phy(struct usb_phy *);
extern void devm_usb_put_phy(struct device *dev, struct usb_phy *x);
extern int usb_bind_phy(const char *dev_name, u8 index,
				const char *phy_dev_name);
#else
static inline struct usb_phy *usb_get_phy(enum usb_phy_type type)
{
	return ERR_PTR(-ENXIO);
}

static inline struct usb_phy *devm_usb_get_phy(struct device *dev,
	enum usb_phy_type type)
{
	return ERR_PTR(-ENXIO);
}

static inline struct usb_phy *usb_get_phy_dev(struct device *dev, u8 index)
{
	return ERR_PTR(-ENXIO);
}

static inline struct usb_phy *devm_usb_get_phy_dev(struct device *dev, u8 index)
{
	return ERR_PTR(-ENXIO);
}

static inline struct usb_phy *devm_usb_get_phy_by_phandle(struct device *dev,
	const char *phandle, u8 index)
{
	return ERR_PTR(-ENXIO);
}

static inline void usb_put_phy(struct usb_phy *x)
{
}

static inline void devm_usb_put_phy(struct device *dev, struct usb_phy *x)
{
}

static inline int usb_bind_phy(const char *dev_name, u8 index,
				const char *phy_dev_name)
{
	return -EOPNOTSUPP;
}
#endif

static inline int
usb_phy_set_power(struct usb_phy *x, unsigned mA)
{
	if (x && x->set_power)
		return x->set_power(x, mA);
	return 0;
}

static inline int
usb_phy_set_suspend(struct usb_phy *x, int suspend)
{
#if defined (MY_DEF_HERE)
	if (x && x->set_suspend != NULL)
		return x->set_suspend(x, suspend);
#else  
	if (x->set_suspend != NULL)
		return x->set_suspend(x, suspend);
#endif  
	else
		return 0;
}

#if defined(MY_DEF_HERE)
static inline int
usb_phy_notify_connect(struct usb_phy *x, struct usb_device *udev)
#else  
static inline int
usb_phy_notify_connect(struct usb_phy *x, enum usb_device_speed speed)
#endif  
{
#if defined (MY_DEF_HERE)
#if defined(MY_DEF_HERE)
	if (x && x->notify_connect && udev)
		return x->notify_connect(x, udev);
#else  
	if (x && x->notify_connect)
		return x->notify_connect(x, speed);
#endif  
#else  
	if (x->notify_connect)
		return x->notify_connect(x, speed);
#endif  
	else
		return 0;
}

#if defined(MY_DEF_HERE)
static inline int
usb_phy_notify_disconnect(struct usb_phy *x, struct usb_device *udev)
#else  
static inline int
usb_phy_notify_disconnect(struct usb_phy *x, enum usb_device_speed speed)
#endif  
{
#if defined (MY_DEF_HERE)
#if defined(MY_DEF_HERE)
	if (x && x->notify_disconnect && udev)
		return x->notify_disconnect(x, udev);
#else  
	if (x && x->notify_disconnect)
		return x->notify_disconnect(x, speed);
#endif  
#else  
	if (x->notify_disconnect)
		return x->notify_disconnect(x, speed);
#endif  
	else
		return 0;
}

static inline int
usb_register_notifier(struct usb_phy *x, struct notifier_block *nb)
{
	return atomic_notifier_chain_register(&x->notifier, nb);
}

static inline void
usb_unregister_notifier(struct usb_phy *x, struct notifier_block *nb)
{
	atomic_notifier_chain_unregister(&x->notifier, nb);
}

static inline const char *usb_phy_type_string(enum usb_phy_type type)
{
	switch (type) {
	case USB_PHY_TYPE_USB2:
		return "USB2 PHY";
	case USB_PHY_TYPE_USB3:
		return "USB3 PHY";
	default:
		return "UNKNOWN PHY TYPE";
	}
}
#endif  
