#ifndef MY_ABC_HERE
#define MY_ABC_HERE
#endif
 
#if defined(CONFIG_SERIAL_ST_ASC_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/module.h>
#include <linux/serial.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/pm_runtime.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/serial_core.h>
#include <linux/clk.h>
#include <linux/pinctrl/consumer.h>
#include <linux/st-asc.h>

#define DRIVER_NAME "st-asc"
#if defined(MY_DEF_HERE)
#define ASC_SERIAL_NAME "ttyS"
#else  
#define ASC_SERIAL_NAME "ttyAS"
#endif  
#define ASC_FIFO_SIZE 16
#define ASC_MAX_PORTS 8

struct asc_port {
	struct uart_port port;
	struct clk *clk;
	unsigned int hw_flow_control:1;
	unsigned int force_m1:1;
	struct pinctrl			*pinctrl;
	struct pinctrl_state		*pins_default;
	struct pinctrl_state		*pins_sleep;
};

static struct asc_port asc_ports[ASC_MAX_PORTS];
static struct uart_driver asc_uart_driver;

static inline struct asc_port *to_asc_port(struct uart_port *port)
{
	return container_of(port, struct asc_port, port);
}

static inline u32 asc_in(struct uart_port *port, u32 offset)
{
	return readl_relaxed(port->membase + offset);
}

static inline void asc_out(struct uart_port *port, u32 offset, u32 value)
{
	writel_relaxed(value, port->membase + offset);
}

static inline void asc_disable_tx_interrupts(struct uart_port *port)
{
	u32 intenable = asc_in(port, ASC_INTEN) & ~ASC_INTEN_THE;
	asc_out(port, ASC_INTEN, intenable);
	(void)asc_in(port, ASC_INTEN);	 
}

static inline void asc_enable_tx_interrupts(struct uart_port *port)
{
	u32 intenable = asc_in(port, ASC_INTEN) | ASC_INTEN_THE;
	asc_out(port, ASC_INTEN, intenable);
}

static inline void asc_disable_rx_interrupts(struct uart_port *port)
{
	u32 intenable = asc_in(port, ASC_INTEN) & ~ASC_INTEN_RBE;
	asc_out(port, ASC_INTEN, intenable);
	(void)asc_in(port, ASC_INTEN);	 
}

static inline void asc_enable_rx_interrupts(struct uart_port *port)
{
	u32 intenable = asc_in(port, ASC_INTEN) | ASC_INTEN_RBE;
	asc_out(port, ASC_INTEN, intenable);
}

static inline u32 asc_txfifo_is_empty(struct uart_port *port)
{
	return asc_in(port, ASC_STA) & ASC_STA_TE;
}

static inline int asc_txfifo_is_full(struct uart_port *port)
{
	return asc_in(port, ASC_STA) & ASC_STA_TF;
}

static inline const char *asc_port_name(struct uart_port *port)
{
	return to_platform_device(port->dev)->name;
}

static inline unsigned asc_hw_txroom(struct uart_port *port)
{
	u32 status = asc_in(port, ASC_STA);

	if (status & ASC_STA_THE)
		return port->fifosize / 2;
	else if (!(status & ASC_STA_TF))
		return 1;

	return 0;
}

static void asc_transmit_chars(struct uart_port *port)
{
	struct circ_buf *xmit = &port->state->xmit;
	int txroom;
	unsigned char c;

	txroom = asc_hw_txroom(port);

	if ((txroom != 0) && port->x_char) {
		c = port->x_char;
		port->x_char = 0;
		asc_out(port, ASC_TXBUF, c);
		port->icount.tx++;
		txroom = asc_hw_txroom(port);
	}

	if (uart_tx_stopped(port)) {
		 
		asc_disable_tx_interrupts(port);
		return;
	}

	if (uart_circ_empty(xmit)) {
		asc_disable_tx_interrupts(port);
		return;
	}

	if (txroom == 0)
		return;

	do {
		c = xmit->buf[xmit->tail];
		xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
		asc_out(port, ASC_TXBUF, c);
		port->icount.tx++;
		txroom--;
	} while ((txroom > 0) && (!uart_circ_empty(xmit)));

	if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS)
		uart_write_wakeup(port);

	if (uart_circ_empty(xmit))
		asc_disable_tx_interrupts(port);
}

static void asc_receive_chars(struct uart_port *port)
{
	struct tty_port *tport = &port->state->port;
	unsigned long status;
	unsigned long c = 0;
	char flag;

	if (port->irq_wake)
		pm_wakeup_event(tport->tty->dev, 0);

	while ((status = asc_in(port, ASC_STA)) & ASC_STA_RBF) {
		c = asc_in(port, ASC_RXBUF) | ASC_RXBUF_DUMMY_RX;
		flag = TTY_NORMAL;
		port->icount.rx++;

		if ((c & (ASC_RXBUF_FE | ASC_RXBUF_PE)) ||
			status & ASC_STA_OE) {

			if (c & ASC_RXBUF_FE) {
				if (c == ASC_RXBUF_FE) {
					port->icount.brk++;
					if (uart_handle_break(port))
						continue;
					c |= ASC_RXBUF_DUMMY_BE;
				} else {
					port->icount.frame++;
				}
			} else if (c & ASC_RXBUF_PE) {
				port->icount.parity++;
			}
			 
			if (status & ASC_STA_OE) {
				port->icount.overrun++;
				c |= ASC_RXBUF_DUMMY_OE;
			}

			c &= (port->read_status_mask | ASC_RXBUF_DATA_MSK);

			if (c & ASC_RXBUF_DUMMY_BE)
				flag = TTY_BREAK;
			else if (c & ASC_RXBUF_PE)
				flag = TTY_PARITY;
			else if (c & ASC_RXBUF_FE)
				flag = TTY_FRAME;
		}

		if (uart_handle_sysrq_char(port, c))
			continue;

		uart_insert_char(port, c, ASC_RXBUF_DUMMY_OE, c & 0xff, flag);
	}

	spin_unlock(&port->lock);
	tty_flip_buffer_push(tport);
	spin_lock(&port->lock);
}

static irqreturn_t asc_interrupt(int irq, void *ptr)
{
	struct uart_port *port = ptr;
	u32 status;

	spin_lock(&port->lock);

	status = asc_in(port, ASC_STA);

	if (status & ASC_STA_RBF) {
		 
		asc_receive_chars(port);
	}

	if ((status & ASC_STA_THE) &&
	    (asc_in(port, ASC_INTEN) & ASC_INTEN_THE)) {
		 
		asc_transmit_chars(port);
	}

	spin_unlock(&port->lock);

	return IRQ_HANDLED;
}

static unsigned int asc_tx_empty(struct uart_port *port)
{
	return asc_txfifo_is_empty(port) ? TIOCSER_TEMT : 0;
}

static void asc_set_mctrl(struct uart_port *port, unsigned int mctrl)
{
	 
}

static unsigned int asc_get_mctrl(struct uart_port *port)
{
	 
	return TIOCM_CAR | TIOCM_DSR | TIOCM_CTS;
}

static void asc_start_tx(struct uart_port *port)
{
	struct circ_buf *xmit = &port->state->xmit;

	if (!uart_circ_empty(xmit))
		asc_enable_tx_interrupts(port);
}

static void asc_stop_tx(struct uart_port *port)
{
	asc_disable_tx_interrupts(port);
}

static void asc_stop_rx(struct uart_port *port)
{
	asc_disable_rx_interrupts(port);
}

static void asc_enable_ms(struct uart_port *port)
{
	 
}

static void asc_break_ctl(struct uart_port *port, int break_state)
{
	 
}

static int asc_startup(struct uart_port *port)
{
	if (request_irq(port->irq, asc_interrupt, IRQF_NO_SUSPEND,
			asc_port_name(port), port)) {
		dev_err(port->dev, "cannot allocate irq.\n");
		return -ENODEV;
	}

	asc_transmit_chars(port);
	asc_enable_rx_interrupts(port);

	return 0;
}

static void asc_shutdown(struct uart_port *port)
{
	asc_disable_tx_interrupts(port);
	asc_disable_rx_interrupts(port);
	free_irq(port->irq, port);
}

static void asc_pm(struct uart_port *port, unsigned int state,
		unsigned int oldstate)
{
	struct asc_port *ascport = to_asc_port(port);
	unsigned long flags = 0;
	u32 ctl;

	switch (state) {
	case UART_PM_STATE_ON:
		clk_prepare_enable(ascport->clk);

		if (!IS_ERR(ascport->pins_default))
			pinctrl_select_state(ascport->pinctrl,
					ascport->pins_default);
		break;
	case UART_PM_STATE_OFF:
		 
		spin_lock_irqsave(&port->lock, flags);
		ctl = asc_in(port, ASC_CTL) & ~ASC_CTL_RUN;
		asc_out(port, ASC_CTL, ctl);
		spin_unlock_irqrestore(&port->lock, flags);
		clk_disable_unprepare(ascport->clk);

		if (!IS_ERR(ascport->pins_sleep))
			pinctrl_select_state(ascport->pinctrl,
					ascport->pins_sleep);
		break;
	}
}

static void asc_set_termios(struct uart_port *port, struct ktermios *termios,
			    struct ktermios *old)
{
	struct asc_port *ascport = to_asc_port(port);
	unsigned int baud;
	u32 ctrl_val;
	tcflag_t cflag;
	unsigned long flags;

	termios->c_cflag &= ~(CMSPAR |
			 (ascport->hw_flow_control ? 0 : CRTSCTS));

	port->uartclk = clk_get_rate(ascport->clk);

	baud = uart_get_baud_rate(port, termios, old, 0, port->uartclk/16);
	cflag = termios->c_cflag;

	spin_lock_irqsave(&port->lock, flags);

	ctrl_val = asc_in(port, ASC_CTL);

	asc_out(port, ASC_CTL, (ctrl_val & ~ASC_CTL_RUN));
	ctrl_val = ASC_CTL_RXENABLE | ASC_CTL_FIFOENABLE;

	asc_out(port, ASC_TXRESET, 1);
	asc_out(port, ASC_RXRESET, 1);

	if ((cflag & CSIZE) == CS7) {
		ctrl_val |= ASC_CTL_MODE_7BIT_PAR;
	} else {
		ctrl_val |= (cflag & PARENB) ?  ASC_CTL_MODE_8BIT_PAR :
						ASC_CTL_MODE_8BIT;
	}

	ctrl_val |= (cflag & CSTOPB) ? ASC_CTL_STOP_2BIT : ASC_CTL_STOP_1BIT;

	if (cflag & PARODD)
		ctrl_val |= ASC_CTL_PARITYODD;

	if ((cflag & CRTSCTS))
		ctrl_val |= ASC_CTL_CTSENABLE;

	if ((baud < 19200) && !ascport->force_m1) {
		asc_out(port, ASC_BAUDRATE, (port->uartclk / (16 * baud)));
	} else {
		 
		u64 dividend = (u64)baud * (1 << 16);
		do_div(dividend, port->uartclk / 16);

		asc_out(port, ASC_BAUDRATE, dividend);
		ctrl_val |= ASC_CTL_BAUDMODE;
	}

	uart_update_timeout(port, cflag, baud);

	ascport->port.read_status_mask = ASC_RXBUF_DUMMY_OE;
	if (termios->c_iflag & INPCK)
		ascport->port.read_status_mask |= ASC_RXBUF_FE | ASC_RXBUF_PE;
	if (termios->c_iflag & (BRKINT | PARMRK))
		ascport->port.read_status_mask |= ASC_RXBUF_DUMMY_BE;

	ascport->port.ignore_status_mask = 0;
	if (termios->c_iflag & IGNPAR)
		ascport->port.ignore_status_mask |= ASC_RXBUF_FE | ASC_RXBUF_PE;
	if (termios->c_iflag & IGNBRK) {
		ascport->port.ignore_status_mask |= ASC_RXBUF_DUMMY_BE;
		 
		if (termios->c_iflag & IGNPAR)
			ascport->port.ignore_status_mask |= ASC_RXBUF_DUMMY_OE;
	}

	if (!(termios->c_cflag & CREAD))
		ascport->port.ignore_status_mask |= ASC_RXBUF_DUMMY_RX;

	asc_out(port, ASC_TIMEOUT, 20);

	asc_out(port, ASC_CTL, (ctrl_val | ASC_CTL_RUN));

	spin_unlock_irqrestore(&port->lock, flags);
}

static const char *asc_type(struct uart_port *port)
{
	return (port->type == PORT_ASC) ? DRIVER_NAME : NULL;
}

static void asc_release_port(struct uart_port *port)
{
}

static int asc_request_port(struct uart_port *port)
{
	return 0;
}

static void asc_config_port(struct uart_port *port, int flags)
{
	if ((flags & UART_CONFIG_TYPE))
		port->type = PORT_ASC;
}

static int
asc_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	 
	return -EINVAL;
}

#ifdef CONFIG_CONSOLE_POLL
 
static int asc_get_poll_char(struct uart_port *port)
{
	if (!(asc_in(port, ASC_STA) & ASC_STA_RBF))
		return NO_POLL_CHAR;

	return asc_in(port, ASC_RXBUF);
}

static void asc_put_poll_char(struct uart_port *port, unsigned char c)
{
	while (asc_txfifo_is_full(port))
		cpu_relax();
	asc_out(port, ASC_TXBUF, c);
}

#endif  

static struct uart_ops asc_uart_ops = {
	.tx_empty	= asc_tx_empty,
	.set_mctrl	= asc_set_mctrl,
	.get_mctrl	= asc_get_mctrl,
	.start_tx	= asc_start_tx,
	.stop_tx	= asc_stop_tx,
	.stop_rx	= asc_stop_rx,
	.enable_ms	= asc_enable_ms,
	.break_ctl	= asc_break_ctl,
	.startup	= asc_startup,
	.shutdown	= asc_shutdown,
	.set_termios	= asc_set_termios,
	.type		= asc_type,
	.release_port	= asc_release_port,
	.request_port	= asc_request_port,
	.config_port	= asc_config_port,
	.verify_port	= asc_verify_port,
	.pm		= asc_pm,
#ifdef CONFIG_CONSOLE_POLL
	.poll_get_char = asc_get_poll_char,
	.poll_put_char = asc_put_poll_char,
#endif  
};

static int asc_init_port(struct asc_port *ascport,
			  struct platform_device *pdev)
{
	struct uart_port *port = &ascport->port;
	struct resource *res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	if (!res) {
		dev_err(&pdev->dev, "Unable to get io resource\n");
		return -ENODEV;
	}

	port->iotype	= UPIO_MEM;
	port->flags	= UPF_BOOT_AUTOCONF;
	port->ops	= &asc_uart_ops;
	port->fifosize	= ASC_FIFO_SIZE;
	port->dev	= &pdev->dev;
	port->mapbase	= res->start;
	port->irq	= platform_get_irq(pdev, 0);

	port->membase = devm_request_and_ioremap(&pdev->dev, res);
	if (!port->membase) {
		dev_err(&pdev->dev, "Unable to request io memory\n");
		return -ENODEV;
	}

	ascport->pinctrl = devm_pinctrl_get(&pdev->dev);
	if (IS_ERR(ascport->pinctrl)) {
		return PTR_ERR(ascport->pinctrl);
	}
	ascport->pins_default = pinctrl_lookup_state(ascport->pinctrl,
						 PINCTRL_STATE_DEFAULT);
	if (IS_ERR(ascport->pins_default))
		dev_err(&pdev->dev, "could not get default pinstate\n");

	ascport->pins_sleep = pinctrl_lookup_state(ascport->pinctrl,
					       PINCTRL_STATE_SLEEP);
	if (IS_ERR(ascport->pins_sleep))
		dev_dbg(&pdev->dev, "could not get sleep pinstate\n");

	spin_lock_init(&port->lock);

	ascport->clk = devm_clk_get(&pdev->dev, NULL);

	if (WARN_ON(IS_ERR(ascport->clk)))
		return -EINVAL;
	 
	clk_prepare_enable(ascport->clk);
	ascport->port.uartclk = clk_get_rate(ascport->clk);
	WARN_ON(ascport->port.uartclk == 0);
	clk_disable_unprepare(ascport->clk);

	return 0;
}

static struct asc_port *asc_of_get_asc_port(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	int id;

	if (!np)
		return NULL;

	id = of_alias_get_id(np, ASC_SERIAL_NAME);

	if (id < 0)
		id = 0;

	if (WARN_ON(id >= ASC_MAX_PORTS))
		return NULL;

	asc_ports[id].hw_flow_control = of_property_read_bool(np,
							"st,hw-flow-control");
	asc_ports[id].force_m1 =  of_property_read_bool(np, "st,force_m1");
	asc_ports[id].port.line = id;
	return &asc_ports[id];
}

static struct of_device_id asc_match[] = {
	{ .compatible = "st,asc", },
	{},
};

MODULE_DEVICE_TABLE(of, asc_match);

static int asc_serial_probe(struct platform_device *pdev)
{
	int ret;
	struct asc_port *ascport;

	ascport = asc_of_get_asc_port(pdev);
	if (!ascport)
		return -ENODEV;

	ret = asc_init_port(ascport, pdev);
	if (ret)
		return ret;

	ret = uart_add_one_port(&asc_uart_driver, &ascport->port);
	if (ret)
		return ret;

	platform_set_drvdata(pdev, &ascport->port);

	return 0;
}

static int asc_serial_remove(struct platform_device *pdev)
{
	struct uart_port *port = platform_get_drvdata(pdev);

	platform_set_drvdata(pdev, NULL);
	return uart_remove_one_port(&asc_uart_driver, port);
}

#ifdef CONFIG_PM_SLEEP
static int asc_serial_suspend(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct uart_port *port = platform_get_drvdata(pdev);

	return uart_suspend_port(&asc_uart_driver, port);
}

static int asc_serial_resume(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct uart_port *port = platform_get_drvdata(pdev);

	return uart_resume_port(&asc_uart_driver, port);
}

#endif  

#ifdef CONFIG_SERIAL_ST_ASC_CONSOLE
static void asc_console_putchar(struct uart_port *port, int ch)
{
	unsigned int timeout = 1000000;

	while (--timeout && asc_txfifo_is_full(port))
		udelay(1);

	asc_out(port, ASC_TXBUF, ch);
}

static void asc_console_write(struct console *co, const char *s, unsigned count)
{
	struct uart_port *port = &asc_ports[co->index].port;
	unsigned long timeout = 1000000;
	int locked = 1;
	u32 intenable;

	if (port->sysrq)
		locked = 0;  
	else if (oops_in_progress)
		locked = spin_trylock(&port->lock);
	else
		spin_lock(&port->lock);

	intenable = asc_in(port, ASC_INTEN);
	asc_out(port, ASC_INTEN, 0);
	(void)asc_in(port, ASC_INTEN);	 

	uart_console_write(port, s, count, asc_console_putchar);

	while (--timeout && !asc_txfifo_is_empty(port))
		udelay(1);

	asc_out(port, ASC_INTEN, intenable);

	if (locked)
		spin_unlock(&port->lock);
}

static int asc_console_setup(struct console *co, char *options)
{
	struct asc_port *ascport;
	int baud = 9600;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	if (co->index >= ASC_MAX_PORTS)
		return -ENODEV;

	ascport = &asc_ports[co->index];

	if (ascport->port.mapbase == 0 || ascport->port.membase == NULL)
		return -ENXIO;

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);

	return uart_set_options(&ascport->port, co, baud, parity, bits, flow);
}

static struct console asc_console = {
	.name		= ASC_SERIAL_NAME,
	.device		= uart_console_device,
	.write		= asc_console_write,
	.setup		= asc_console_setup,
	.flags		= CON_PRINTBUFFER,
	.index		= -1,
	.data		= &asc_uart_driver,
};

#define ASC_SERIAL_CONSOLE (&asc_console)

#else
#define ASC_SERIAL_CONSOLE NULL
#endif  

static struct uart_driver asc_uart_driver = {
	.owner		= THIS_MODULE,
	.driver_name	= DRIVER_NAME,
	.dev_name	= ASC_SERIAL_NAME,
#if defined(MY_DEF_HERE)
	.major		= TTY_MAJOR,
	.minor		= 64,
#else
	.major		= 0,
	.minor		= 0,
#endif
	.nr		= ASC_MAX_PORTS,
	.cons		= ASC_SERIAL_CONSOLE,
};

static const struct dev_pm_ops asc_serial_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(asc_serial_suspend, asc_serial_resume)
};

static struct platform_driver asc_serial_driver = {
	.probe		= asc_serial_probe,
	.remove		= asc_serial_remove,
	.driver	= {
		.name	= DRIVER_NAME,
		.pm	= &asc_serial_pm_ops,
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(asc_match),
	},
};

static int __init asc_init(void)
{
	int ret;
	static char banner[] __initdata =
		KERN_INFO "STMicroelectronics ASC driver initialized\n";

	printk(banner);

	ret = uart_register_driver(&asc_uart_driver);
	if (ret)
		return ret;

	ret = platform_driver_register(&asc_serial_driver);
	if (ret)
		uart_unregister_driver(&asc_uart_driver);

	return ret;
}

static void __exit asc_exit(void)
{
	platform_driver_unregister(&asc_serial_driver);
	uart_unregister_driver(&asc_uart_driver);
}

module_init(asc_init);
module_exit(asc_exit);

MODULE_ALIAS("platform:" DRIVER_NAME);
MODULE_AUTHOR("STMicroelectronics (R&D) Limited");
MODULE_DESCRIPTION("STMicroelectronics ASC serial port driver");
MODULE_LICENSE("GPL");
