#ifndef MY_ABC_HERE
#define MY_ABC_HERE
#endif
 
#ifndef _CRYPTO_TESTMGR_H
#define _CRYPTO_TESTMGR_H

#include <linux/netlink.h>
#include <linux/zlib.h>

#include <crypto/compress.h>

#define MAX_DIGEST_SIZE		64
#define MAX_TAP			8

#define MAX_KEYLEN		56
#define MAX_IVLEN		32

struct hash_testvec {
	 
	char *key;
	char *plaintext;
	char *digest;
	unsigned char tap[MAX_TAP];
	unsigned short psize;
	unsigned char np;
	unsigned char ksize;
};

struct cipher_testvec {
	char *key;
	char *iv;
	char *input;
	char *result;
	unsigned short tap[MAX_TAP];
	int np;
	unsigned char also_non_np;
	unsigned char fail;
	unsigned char wk;  
	unsigned char klen;
	unsigned short ilen;
	unsigned short rlen;
};

struct aead_testvec {
	char *key;
	char *iv;
	char *input;
	char *assoc;
	char *result;
	unsigned char tap[MAX_TAP];
	unsigned char atap[MAX_TAP];
	int np;
	int anp;
	unsigned char fail;
	unsigned char novrfy;	 
	unsigned char wk;  
	unsigned char klen;
	unsigned short ilen;
	unsigned short alen;
	unsigned short rlen;
};

struct cprng_testvec {
	char *key;
	char *dt;
	char *v;
	char *result;
	unsigned char klen;
	unsigned short dtlen;
	unsigned short vlen;
	unsigned short rlen;
	unsigned short loops;
};

static char zeroed_string[48];

#define MD4_TEST_VECTORS	7

static struct hash_testvec md4_tv_template [] = {
	{
		.plaintext = "",
		.digest	= "\x31\xd6\xcf\xe0\xd1\x6a\xe9\x31"
			  "\xb7\x3c\x59\xd7\xe0\xc0\x89\xc0",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\xbd\xe5\x2c\xb3\x1d\xe3\x3e\x46"
			  "\x24\x5e\x05\xfb\xdb\xd6\xfb\x24",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xa4\x48\x01\x7a\xaf\x21\xd8\x52"
			  "\x5f\xc1\x0a\xe8\x7a\xa6\x72\x9d",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\xd9\x13\x0a\x81\x64\x54\x9f\xe8"
			"\x18\x87\x48\x06\xe1\xc7\x01\x4b",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xd7\x9e\x1c\x30\x8a\xa5\xbb\xcd"
			  "\xee\xa8\xed\x63\xdf\x41\x2d\xa9",
		.np	= 2,
		.tap	= { 13, 13 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\x04\x3f\x85\x82\xf2\x41\xdb\x35"
			  "\x1c\xe6\x27\xe1\x53\xe7\xf0\xe4",
	}, {
		.plaintext = "123456789012345678901234567890123456789012345678901234567890123"
			   "45678901234567890",
		.psize	= 80,
		.digest	= "\xe3\x3b\x4d\xdc\x9c\x38\xf2\x19"
			  "\x9c\x3e\x7b\x16\x4f\xcc\x05\x36",
	},
};

#define MD5_TEST_VECTORS	7

static struct hash_testvec md5_tv_template[] = {
	{
		.digest	= "\xd4\x1d\x8c\xd9\x8f\x00\xb2\x04"
			  "\xe9\x80\x09\x98\xec\xf8\x42\x7e",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x0c\xc1\x75\xb9\xc0\xf1\xb6\xa8"
			  "\x31\xc3\x99\xe2\x69\x77\x26\x61",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\x90\x01\x50\x98\x3c\xd2\x4f\xb0"
			  "\xd6\x96\x3f\x7d\x28\xe1\x7f\x72",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\xf9\x6b\x69\x7d\x7c\xb7\x93\x8d"
			  "\x52\x5a\x2f\x31\xaa\xf1\x61\xd0",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xc3\xfc\xd3\xd7\x61\x92\xe4\x00"
			  "\x7d\xfb\x49\x6c\xca\x67\xe1\x3b",
		.np	= 2,
		.tap	= {13, 13}
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xd1\x74\xab\x98\xd2\x77\xd9\xf5"
			  "\xa5\x61\x1c\x2c\x9f\x41\x9d\x9f",
	}, {
		.plaintext = "12345678901234567890123456789012345678901234567890123456789012"
			   "345678901234567890",
		.psize	= 80,
		.digest	= "\x57\xed\xf4\xa2\x2b\xe3\xc9\x55"
			  "\xac\x49\xda\x2e\x21\x07\xb6\x7a",
	}

};

#define RMD128_TEST_VECTORS     10

static struct hash_testvec rmd128_tv_template[] = {
	{
		.digest	= "\xcd\xf2\x62\x13\xa1\x50\xdc\x3e"
			  "\xcb\x61\x0f\x18\xf6\xb3\x8b\x46",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x86\xbe\x7a\xfa\x33\x9d\x0f\xc7"
			  "\xcf\xc7\x85\xe7\x2f\x57\x8d\x33",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xc1\x4a\x12\x19\x9c\x66\xe4\xba"
			  "\x84\x63\x6b\x0f\x69\x14\x4c\x77",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x9e\x32\x7b\x3d\x6e\x52\x30\x62"
			  "\xaf\xc1\x13\x2d\x7d\xf9\xd1\xb8",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xfd\x2a\xa6\x07\xf7\x1d\xc8\xf5"
			  "\x10\x71\x49\x22\xb3\x71\x83\x4e",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcde"
			     "fghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xd1\xe9\x59\xeb\x17\x9c\x91\x1f"
			  "\xae\xa4\x62\x4c\x60\xc5\xc7\x02",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x3f\x45\xef\x19\x47\x32\xc2\xdb"
			  "\xb2\xc4\xa2\xc7\x69\x79\x5f\xa3",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighij"
			     "hijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\xa1\xaa\x06\x89\xd0\xfa\xfa\x2d"
			  "\xdc\x22\xe8\x8b\x49\x13\x3a\x06",
		.np	= 2,
		.tap	= { 28, 28 },
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghi"
			     "jklmghijklmnhijklmnoijklmnopjklmnopqklmnopqr"
			     "lmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= "\xd4\xec\xc9\x13\xe1\xdf\x77\x6b"
			  "\xf4\x8d\xe9\xd5\x5b\x1f\x25\x46",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= "\x13\xfc\x13\xe8\xef\xff\x34\x7d"
			  "\xe1\x93\xff\x46\xdb\xac\xcf\xd4",
	}
};

#define RMD160_TEST_VECTORS     10

static struct hash_testvec rmd160_tv_template[] = {
	{
		.digest	= "\x9c\x11\x85\xa5\xc5\xe9\xfc\x54\x61\x28"
			  "\x08\x97\x7e\xe8\xf5\x48\xb2\x25\x8d\x31",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x0b\xdc\x9d\x2d\x25\x6b\x3e\xe9\xda\xae"
			  "\x34\x7b\xe6\xf4\xdc\x83\x5a\x46\x7f\xfe",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\x8e\xb2\x08\xf7\xe0\x5d\x98\x7a\x9b\x04"
			  "\x4a\x8e\x98\xc6\xb0\x87\xf1\x5a\x0b\xfc",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x5d\x06\x89\xef\x49\xd2\xfa\xe5\x72\xb8"
			  "\x81\xb1\x23\xa8\x5f\xfa\x21\x59\x5f\x36",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xf7\x1c\x27\x10\x9c\x69\x2c\x1b\x56\xbb"
			  "\xdc\xeb\x5b\x9d\x28\x65\xb3\x70\x8d\xbc",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcde"
			     "fghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xb0\xe2\x0b\x6e\x31\x16\x64\x02\x86\xed"
			  "\x3a\x87\xa5\x71\x30\x79\xb2\x1f\x51\x89",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x9b\x75\x2e\x45\x57\x3d\x4b\x39\xf4\xdb"
			  "\xd3\x32\x3c\xab\x82\xbf\x63\x32\x6b\xfb",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighij"
			     "hijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x12\xa0\x53\x38\x4a\x9c\x0c\x88\xe4\x05"
			  "\xa0\x6c\x27\xdc\xf4\x9a\xda\x62\xeb\x2b",
		.np	= 2,
		.tap	= { 28, 28 },
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghi"
			     "jklmghijklmnhijklmnoijklmnopjklmnopqklmnopqr"
			     "lmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= "\x6f\x3f\xa3\x9b\x6b\x50\x3c\x38\x4f\x91"
			  "\x9a\x49\xa7\xaa\x5c\x2c\x08\xbd\xfb\x45",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= "\x94\xc2\x64\x11\x54\x04\xe6\x33\x79\x0d"
			  "\xfc\xc8\x7b\x58\x7d\x36\x77\x06\x7d\x9f",
	}
};

#define RMD256_TEST_VECTORS     8

static struct hash_testvec rmd256_tv_template[] = {
	{
		.digest	= "\x02\xba\x4c\x4e\x5f\x8e\xcd\x18"
			  "\x77\xfc\x52\xd6\x4d\x30\xe3\x7a"
			  "\x2d\x97\x74\xfb\x1e\x5d\x02\x63"
			  "\x80\xae\x01\x68\xe3\xc5\x52\x2d",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\xf9\x33\x3e\x45\xd8\x57\xf5\xd9"
			  "\x0a\x91\xba\xb7\x0a\x1e\xba\x0c"
			  "\xfb\x1b\xe4\xb0\x78\x3c\x9a\xcf"
			  "\xcd\x88\x3a\x91\x34\x69\x29\x25",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xaf\xbd\x6e\x22\x8b\x9d\x8c\xbb"
			  "\xce\xf5\xca\x2d\x03\xe6\xdb\xa1"
			  "\x0a\xc0\xbc\x7d\xcb\xe4\x68\x0e"
			  "\x1e\x42\xd2\xe9\x75\x45\x9b\x65",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x87\xe9\x71\x75\x9a\x1c\xe4\x7a"
			  "\x51\x4d\x5c\x91\x4c\x39\x2c\x90"
			  "\x18\xc7\xc4\x6b\xc1\x44\x65\x55"
			  "\x4a\xfc\xdf\x54\xa5\x07\x0c\x0e",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\x64\x9d\x30\x34\x75\x1e\xa2\x16"
			  "\x77\x6b\xf9\xa1\x8a\xcc\x81\xbc"
			  "\x78\x96\x11\x8a\x51\x97\x96\x87"
			  "\x82\xdd\x1f\xd9\x7d\x8d\x51\x33",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcde"
			     "fghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\x57\x40\xa4\x08\xac\x16\xb7\x20"
			  "\xb8\x44\x24\xae\x93\x1c\xbb\x1f"
			  "\xe3\x63\xd1\xd0\xbf\x40\x17\xf1"
			  "\xa8\x9f\x7e\xa6\xde\x77\xa0\xb8",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x06\xfd\xcc\x7a\x40\x95\x48\xaa"
			  "\xf9\x13\x68\xc0\x6a\x62\x75\xb5"
			  "\x53\xe3\xf0\x99\xbf\x0e\xa4\xed"
			  "\xfd\x67\x78\xdf\x89\xa8\x90\xdd",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighij"
			     "hijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x38\x43\x04\x55\x83\xaa\xc6\xc8"
			  "\xc8\xd9\x12\x85\x73\xe7\xa9\x80"
			  "\x9a\xfb\x2a\x0f\x34\xcc\xc3\x6e"
			  "\xa9\xe7\x2f\x16\xf6\x36\x8e\x3f",
		.np	= 2,
		.tap	= { 28, 28 },
	}
};

#define RMD320_TEST_VECTORS     8

static struct hash_testvec rmd320_tv_template[] = {
	{
		.digest	= "\x22\xd6\x5d\x56\x61\x53\x6c\xdc\x75\xc1"
			  "\xfd\xf5\xc6\xde\x7b\x41\xb9\xf2\x73\x25"
			  "\xeb\xc6\x1e\x85\x57\x17\x7d\x70\x5a\x0e"
			  "\xc8\x80\x15\x1c\x3a\x32\xa0\x08\x99\xb8",
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\xce\x78\x85\x06\x38\xf9\x26\x58\xa5\xa5"
			  "\x85\x09\x75\x79\x92\x6d\xda\x66\x7a\x57"
			  "\x16\x56\x2c\xfc\xf6\xfb\xe7\x7f\x63\x54"
			  "\x2f\x99\xb0\x47\x05\xd6\x97\x0d\xff\x5d",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xde\x4c\x01\xb3\x05\x4f\x89\x30\xa7\x9d"
			  "\x09\xae\x73\x8e\x92\x30\x1e\x5a\x17\x08"
			  "\x5b\xef\xfd\xc1\xb8\xd1\x16\x71\x3e\x74"
			  "\xf8\x2f\xa9\x42\xd6\x4c\xdb\xc4\x68\x2d",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x3a\x8e\x28\x50\x2e\xd4\x5d\x42\x2f\x68"
			  "\x84\x4f\x9d\xd3\x16\xe7\xb9\x85\x33\xfa"
			  "\x3f\x2a\x91\xd2\x9f\x84\xd4\x25\xc8\x8d"
			  "\x6b\x4e\xff\x72\x7d\xf6\x6a\x7c\x01\x97",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xca\xbd\xb1\x81\x0b\x92\x47\x0a\x20\x93"
			  "\xaa\x6b\xce\x05\x95\x2c\x28\x34\x8c\xf4"
			  "\x3f\xf6\x08\x41\x97\x51\x66\xbb\x40\xed"
			  "\x23\x40\x04\xb8\x82\x44\x63\xe6\xb0\x09",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcde"
			     "fghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xed\x54\x49\x40\xc8\x6d\x67\xf2\x50\xd2"
			  "\x32\xc3\x0b\x7b\x3e\x57\x70\xe0\xc6\x0c"
			  "\x8c\xb9\xa4\xca\xfe\x3b\x11\x38\x8a\xf9"
			  "\x92\x0e\x1b\x99\x23\x0b\x84\x3c\x86\xa4",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x55\x78\x88\xaf\x5f\x6d\x8e\xd6\x2a\xb6"
			  "\x69\x45\xc6\xd2\xa0\xa4\x7e\xcd\x53\x41"
			  "\xe9\x15\xeb\x8f\xea\x1d\x05\x24\x95\x5f"
			  "\x82\x5d\xc7\x17\xe4\xa0\x08\xab\x2d\x42",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighij"
			     "hijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\xd0\x34\xa7\x95\x0c\xf7\x22\x02\x1b\xa4"
			  "\xb8\x4d\xf7\x69\xa5\xde\x20\x60\xe2\x59"
			  "\xdf\x4c\x9b\xb4\xa4\x26\x8c\x0e\x93\x5b"
			  "\xbc\x74\x70\xa9\x69\xc9\xd0\x72\xa1\xac",
		.np	= 2,
		.tap	= { 28, 28 },
	}
};

#define CRCT10DIF_TEST_VECTORS	3
static struct hash_testvec crct10dif_tv_template[] = {
	{
		.plaintext = "abc",
		.psize  = 3,
#ifdef __LITTLE_ENDIAN
		.digest = "\x3b\x44",
#else
		.digest = "\x44\x3b",
#endif
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "123456789012345678901234567890123456789",
		.psize	= 79,
#ifdef __LITTLE_ENDIAN
		.digest	= "\x70\x4b",
#else
		.digest	= "\x4b\x70",
#endif
	}, {
		.plaintext =
		"abcddddddddddddddddddddddddddddddddddddddddddddddddddddd",
		.psize  = 56,
#ifdef __LITTLE_ENDIAN
		.digest = "\xe3\x9c",
#else
		.digest = "\x9c\xe3",
#endif
		.np     = 2,
		.tap    = { 28, 28 }
	}
};

#define SHA1_TEST_VECTORS	3

static struct hash_testvec sha1_tv_template[] = {
	{
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xa9\x99\x3e\x36\x47\x06\x81\x6a\xba\x3e"
			  "\x25\x71\x78\x50\xc2\x6c\x9c\xd0\xd8\x9d",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x84\x98\x3e\x44\x1c\x3b\xd2\x6e\xba\xae"
			  "\x4a\xa1\xf9\x51\x29\xe5\xe5\x46\x70\xf1",
		.np	= 2,
		.tap	= { 28, 28 }
	}, {
		.plaintext = "\xec\x29\x56\x12\x44\xed\xe7\x06"
			     "\xb6\xeb\x30\xa1\xc3\x71\xd7\x44"
			     "\x50\xa1\x05\xc3\xf9\x73\x5f\x7f"
			     "\xa9\xfe\x38\xcf\x67\xf3\x04\xa5"
			     "\x73\x6a\x10\x6e\x92\xe1\x71\x39"
			     "\xa6\x81\x3b\x1c\x81\xa4\xf3\xd3"
			     "\xfb\x95\x46\xab\x42\x96\xfa\x9f"
			     "\x72\x28\x26\xc0\x66\x86\x9e\xda"
			     "\xcd\x73\xb2\x54\x80\x35\x18\x58"
			     "\x13\xe2\x26\x34\xa9\xda\x44\x00"
			     "\x0d\x95\xa2\x81\xff\x9f\x26\x4e"
			     "\xcc\xe0\xa9\x31\x22\x21\x62\xd0"
			     "\x21\xcc\xa2\x8d\xb5\xf3\xc2\xaa"
			     "\x24\x94\x5a\xb1\xe3\x1c\xb4\x13"
			     "\xae\x29\x81\x0f\xd7\x94\xca\xd5"
			     "\xdf\xaf\x29\xec\x43\xcb\x38\xd1"
			     "\x98\xfe\x4a\xe1\xda\x23\x59\x78"
			     "\x02\x21\x40\x5b\xd6\x71\x2a\x53"
			     "\x05\xda\x4b\x1b\x73\x7f\xce\x7c"
			     "\xd2\x1c\x0e\xb7\x72\x8d\x08\x23"
			     "\x5a\x90\x11",
		.psize	= 163,
		.digest	= "\x97\x01\x11\xc4\xe7\x7b\xcc\x88\xcc\x20"
			  "\x45\x9c\x02\xb6\x9b\x4a\xa8\xf5\x82\x17",
		.np	= 4,
		.tap	= { 63, 64, 31, 5 }
	}
};

#define SHA224_TEST_VECTORS     2

static struct hash_testvec sha224_tv_template[] = {
	{
		.plaintext = "abc",
		.psize  = 3,
		.digest = "\x23\x09\x7D\x22\x34\x05\xD8\x22"
			  "\x86\x42\xA4\x77\xBD\xA2\x55\xB3"
			  "\x2A\xAD\xBC\xE4\xBD\xA0\xB3\xF7"
			  "\xE3\x6C\x9D\xA7",
	}, {
		.plaintext =
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize  = 56,
		.digest = "\x75\x38\x8B\x16\x51\x27\x76\xCC"
			  "\x5D\xBA\x5D\xA1\xFD\x89\x01\x50"
			  "\xB0\xC6\x45\x5C\xB4\xF5\x8B\x19"
			  "\x52\x52\x25\x25",
		.np     = 2,
		.tap    = { 28, 28 }
	}
};

#define SHA256_TEST_VECTORS	2

static struct hash_testvec sha256_tv_template[] = {
	{
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xba\x78\x16\xbf\x8f\x01\xcf\xea"
			  "\x41\x41\x40\xde\x5d\xae\x22\x23"
			  "\xb0\x03\x61\xa3\x96\x17\x7a\x9c"
			  "\xb4\x10\xff\x61\xf2\x00\x15\xad",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x24\x8d\x6a\x61\xd2\x06\x38\xb8"
			  "\xe5\xc0\x26\x93\x0c\x3e\x60\x39"
			  "\xa3\x3c\xe4\x59\x64\xff\x21\x67"
			  "\xf6\xec\xed\xd4\x19\xdb\x06\xc1",
		.np	= 2,
		.tap	= { 28, 28 }
	},
};

#define SHA384_TEST_VECTORS	4

static struct hash_testvec sha384_tv_template[] = {
	{
		.plaintext= "abc",
		.psize	= 3,
		.digest	= "\xcb\x00\x75\x3f\x45\xa3\x5e\x8b"
			  "\xb5\xa0\x3d\x69\x9a\xc6\x50\x07"
			  "\x27\x2c\x32\xab\x0e\xde\xd1\x63"
			  "\x1a\x8b\x60\x5a\x43\xff\x5b\xed"
			  "\x80\x86\x07\x2b\xa1\xe7\xcc\x23"
			  "\x58\xba\xec\xa1\x34\xc8\x25\xa7",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x33\x91\xfd\xdd\xfc\x8d\xc7\x39"
			  "\x37\x07\xa6\x5b\x1b\x47\x09\x39"
			  "\x7c\xf8\xb1\xd1\x62\xaf\x05\xab"
			  "\xfe\x8f\x45\x0d\xe5\xf3\x6b\xc6"
			  "\xb0\x45\x5a\x85\x20\xbc\x4e\x6f"
			  "\x5f\xe9\x5b\x1f\xe3\xc8\x45\x2b",
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			   "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= "\x09\x33\x0c\x33\xf7\x11\x47\xe8"
			  "\x3d\x19\x2f\xc7\x82\xcd\x1b\x47"
			  "\x53\x11\x1b\x17\x3b\x3b\x05\xd2"
			  "\x2f\xa0\x80\x86\xe3\xb0\xf7\x12"
			  "\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9"
			  "\x66\xc3\xe9\xfa\x91\x74\x60\x39",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			   "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= "\x3d\x20\x89\x73\xab\x35\x08\xdb"
			  "\xbd\x7e\x2c\x28\x62\xba\x29\x0a"
			  "\xd3\x01\x0e\x49\x78\xc1\x98\xdc"
			  "\x4d\x8f\xd0\x14\xe5\x82\x82\x3a"
			  "\x89\xe1\x6f\x9b\x2a\x7b\xbc\x1a"
			  "\xc9\x38\xe2\xd1\x99\xe8\xbe\xa4",
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	},
};

#define SHA512_TEST_VECTORS	4

static struct hash_testvec sha512_tv_template[] = {
	{
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xdd\xaf\x35\xa1\x93\x61\x7a\xba"
			  "\xcc\x41\x73\x49\xae\x20\x41\x31"
			  "\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2"
			  "\x0a\x9e\xee\xe6\x4b\x55\xd3\x9a"
			  "\x21\x92\x99\x2a\x27\x4f\xc1\xa8"
			  "\x36\xba\x3c\x23\xa3\xfe\xeb\xbd"
			  "\x45\x4d\x44\x23\x64\x3c\xe8\x0e"
			  "\x2a\x9a\xc9\x4f\xa5\x4c\xa4\x9f",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x20\x4a\x8f\xc6\xdd\xa8\x2f\x0a"
			  "\x0c\xed\x7b\xeb\x8e\x08\xa4\x16"
			  "\x57\xc1\x6e\xf4\x68\xb2\x28\xa8"
			  "\x27\x9b\xe3\x31\xa7\x03\xc3\x35"
			  "\x96\xfd\x15\xc1\x3b\x1b\x07\xf9"
			  "\xaa\x1d\x3b\xea\x57\x78\x9c\xa0"
			  "\x31\xad\x85\xc7\xa7\x1d\xd7\x03"
			  "\x54\xec\x63\x12\x38\xca\x34\x45",
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			   "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= "\x8e\x95\x9b\x75\xda\xe3\x13\xda"
			  "\x8c\xf4\xf7\x28\x14\xfc\x14\x3f"
			  "\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1"
			  "\x72\x99\xae\xad\xb6\x88\x90\x18"
			  "\x50\x1d\x28\x9e\x49\x00\xf7\xe4"
			  "\x33\x1b\x99\xde\xc4\xb5\x43\x3a"
			  "\xc7\xd3\x29\xee\xb6\xdd\x26\x54"
			  "\x5e\x96\xe5\x5b\x87\x4b\xe9\x09",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			   "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= "\x93\x0d\x0c\xef\xcb\x30\xff\x11"
			  "\x33\xb6\x89\x81\x21\xf1\xcf\x3d"
			  "\x27\x57\x8a\xfc\xaf\xe8\x67\x7c"
			  "\x52\x57\xcf\x06\x99\x11\xf7\x5d"
			  "\x8f\x58\x31\xb5\x6e\xbf\xda\x67"
			  "\xb2\x78\xe6\x6d\xff\x8b\x84\xfe"
			  "\x2b\x28\x70\xf7\x42\xa5\x80\xd8"
			  "\xed\xb4\x19\x87\x23\x28\x50\xc9",
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	},
};

#define WP512_TEST_VECTORS	8

static struct hash_testvec wp512_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\x19\xFA\x61\xD7\x55\x22\xA4\x66"
			  "\x9B\x44\xE3\x9C\x1D\x2E\x17\x26"
			  "\xC5\x30\x23\x21\x30\xD4\x07\xF8"
			  "\x9A\xFE\xE0\x96\x49\x97\xF7\xA7"
			  "\x3E\x83\xBE\x69\x8B\x28\x8F\xEB"
			  "\xCF\x88\xE3\xE0\x3C\x4F\x07\x57"
			  "\xEA\x89\x64\xE5\x9B\x63\xD9\x37"
			  "\x08\xB1\x38\xCC\x42\xA6\x6E\xB3",

	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x8A\xCA\x26\x02\x79\x2A\xEC\x6F"
			  "\x11\xA6\x72\x06\x53\x1F\xB7\xD7"
			  "\xF0\xDF\xF5\x94\x13\x14\x5E\x69"
			  "\x73\xC4\x50\x01\xD0\x08\x7B\x42"
			  "\xD1\x1B\xC6\x45\x41\x3A\xEF\xF6"
			  "\x3A\x42\x39\x1A\x39\x14\x5A\x59"
			  "\x1A\x92\x20\x0D\x56\x01\x95\xE5"
			  "\x3B\x47\x85\x84\xFD\xAE\x23\x1A",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\x4E\x24\x48\xA4\xC6\xF4\x86\xBB"
			  "\x16\xB6\x56\x2C\x73\xB4\x02\x0B"
			  "\xF3\x04\x3E\x3A\x73\x1B\xCE\x72"
			  "\x1A\xE1\xB3\x03\xD9\x7E\x6D\x4C"
			  "\x71\x81\xEE\xBD\xB6\xC5\x7E\x27"
			  "\x7D\x0E\x34\x95\x71\x14\xCB\xD6"
			  "\xC7\x97\xFC\x9D\x95\xD8\xB5\x82"
			  "\xD2\x25\x29\x20\x76\xD4\xEE\xF5",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x37\x8C\x84\xA4\x12\x6E\x2D\xC6"
			  "\xE5\x6D\xCC\x74\x58\x37\x7A\xAC"
			  "\x83\x8D\x00\x03\x22\x30\xF5\x3C"
			  "\xE1\xF5\x70\x0C\x0F\xFB\x4D\x3B"
			  "\x84\x21\x55\x76\x59\xEF\x55\xC1"
			  "\x06\xB4\xB5\x2A\xC5\xA4\xAA\xA6"
			  "\x92\xED\x92\x00\x52\x83\x8F\x33"
			  "\x62\xE8\x6D\xBD\x37\xA8\x90\x3E",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xF1\xD7\x54\x66\x26\x36\xFF\xE9"
			  "\x2C\x82\xEB\xB9\x21\x2A\x48\x4A"
			  "\x8D\x38\x63\x1E\xAD\x42\x38\xF5"
			  "\x44\x2E\xE1\x3B\x80\x54\xE4\x1B"
			  "\x08\xBF\x2A\x92\x51\xC3\x0B\x6A"
			  "\x0B\x8A\xAE\x86\x17\x7A\xB4\xA6"
			  "\xF6\x8F\x67\x3E\x72\x07\x86\x5D"
			  "\x5D\x98\x19\xA3\xDB\xA4\xEB\x3B",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			   "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xDC\x37\xE0\x08\xCF\x9E\xE6\x9B"
			  "\xF1\x1F\x00\xED\x9A\xBA\x26\x90"
			  "\x1D\xD7\xC2\x8C\xDE\xC0\x66\xCC"
			  "\x6A\xF4\x2E\x40\xF8\x2F\x3A\x1E"
			  "\x08\xEB\xA2\x66\x29\x12\x9D\x8F"
			  "\xB7\xCB\x57\x21\x1B\x92\x81\xA6"
			  "\x55\x17\xCC\x87\x9D\x7B\x96\x21"
			  "\x42\xC6\x5F\x5A\x7A\xF0\x14\x67",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			   "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x46\x6E\xF1\x8B\xAB\xB0\x15\x4D"
			  "\x25\xB9\xD3\x8A\x64\x14\xF5\xC0"
			  "\x87\x84\x37\x2B\xCC\xB2\x04\xD6"
			  "\x54\x9C\x4A\xFA\xDB\x60\x14\x29"
			  "\x4D\x5B\xD8\xDF\x2A\x6C\x44\xE5"
			  "\x38\xCD\x04\x7B\x26\x81\xA5\x1A"
			  "\x2C\x60\x48\x1E\x88\xC5\xA2\x0B"
			  "\x2C\x2A\x80\xCF\x3A\x9A\x08\x3B",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= "\x2A\x98\x7E\xA4\x0F\x91\x70\x61"
			  "\xF5\xD6\xF0\xA0\xE4\x64\x4F\x48"
			  "\x8A\x7A\x5A\x52\xDE\xEE\x65\x62"
			  "\x07\xC5\x62\xF9\x88\xE9\x5C\x69"
			  "\x16\xBD\xC8\x03\x1B\xC5\xBE\x1B"
			  "\x7B\x94\x76\x39\xFE\x05\x0B\x56"
			  "\x93\x9B\xAA\xA0\xAD\xFF\x9A\xE6"
			  "\x74\x5B\x7B\x18\x1C\x3B\xE3\xFD",
	},
};

#define WP384_TEST_VECTORS	8

static struct hash_testvec wp384_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\x19\xFA\x61\xD7\x55\x22\xA4\x66"
			  "\x9B\x44\xE3\x9C\x1D\x2E\x17\x26"
			  "\xC5\x30\x23\x21\x30\xD4\x07\xF8"
			  "\x9A\xFE\xE0\x96\x49\x97\xF7\xA7"
			  "\x3E\x83\xBE\x69\x8B\x28\x8F\xEB"
			  "\xCF\x88\xE3\xE0\x3C\x4F\x07\x57",

	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x8A\xCA\x26\x02\x79\x2A\xEC\x6F"
			  "\x11\xA6\x72\x06\x53\x1F\xB7\xD7"
			  "\xF0\xDF\xF5\x94\x13\x14\x5E\x69"
			  "\x73\xC4\x50\x01\xD0\x08\x7B\x42"
			  "\xD1\x1B\xC6\x45\x41\x3A\xEF\xF6"
			  "\x3A\x42\x39\x1A\x39\x14\x5A\x59",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\x4E\x24\x48\xA4\xC6\xF4\x86\xBB"
			  "\x16\xB6\x56\x2C\x73\xB4\x02\x0B"
			  "\xF3\x04\x3E\x3A\x73\x1B\xCE\x72"
			  "\x1A\xE1\xB3\x03\xD9\x7E\x6D\x4C"
			  "\x71\x81\xEE\xBD\xB6\xC5\x7E\x27"
			  "\x7D\x0E\x34\x95\x71\x14\xCB\xD6",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x37\x8C\x84\xA4\x12\x6E\x2D\xC6"
			  "\xE5\x6D\xCC\x74\x58\x37\x7A\xAC"
			  "\x83\x8D\x00\x03\x22\x30\xF5\x3C"
			  "\xE1\xF5\x70\x0C\x0F\xFB\x4D\x3B"
			  "\x84\x21\x55\x76\x59\xEF\x55\xC1"
			  "\x06\xB4\xB5\x2A\xC5\xA4\xAA\xA6",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xF1\xD7\x54\x66\x26\x36\xFF\xE9"
			  "\x2C\x82\xEB\xB9\x21\x2A\x48\x4A"
			  "\x8D\x38\x63\x1E\xAD\x42\x38\xF5"
			  "\x44\x2E\xE1\x3B\x80\x54\xE4\x1B"
			  "\x08\xBF\x2A\x92\x51\xC3\x0B\x6A"
			  "\x0B\x8A\xAE\x86\x17\x7A\xB4\xA6",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			   "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xDC\x37\xE0\x08\xCF\x9E\xE6\x9B"
			  "\xF1\x1F\x00\xED\x9A\xBA\x26\x90"
			  "\x1D\xD7\xC2\x8C\xDE\xC0\x66\xCC"
			  "\x6A\xF4\x2E\x40\xF8\x2F\x3A\x1E"
			  "\x08\xEB\xA2\x66\x29\x12\x9D\x8F"
			  "\xB7\xCB\x57\x21\x1B\x92\x81\xA6",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			   "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x46\x6E\xF1\x8B\xAB\xB0\x15\x4D"
			  "\x25\xB9\xD3\x8A\x64\x14\xF5\xC0"
			  "\x87\x84\x37\x2B\xCC\xB2\x04\xD6"
			  "\x54\x9C\x4A\xFA\xDB\x60\x14\x29"
			  "\x4D\x5B\xD8\xDF\x2A\x6C\x44\xE5"
			  "\x38\xCD\x04\x7B\x26\x81\xA5\x1A",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= "\x2A\x98\x7E\xA4\x0F\x91\x70\x61"
			  "\xF5\xD6\xF0\xA0\xE4\x64\x4F\x48"
			  "\x8A\x7A\x5A\x52\xDE\xEE\x65\x62"
			  "\x07\xC5\x62\xF9\x88\xE9\x5C\x69"
			  "\x16\xBD\xC8\x03\x1B\xC5\xBE\x1B"
			  "\x7B\x94\x76\x39\xFE\x05\x0B\x56",
	},
};

#define WP256_TEST_VECTORS	8

static struct hash_testvec wp256_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= "\x19\xFA\x61\xD7\x55\x22\xA4\x66"
			  "\x9B\x44\xE3\x9C\x1D\x2E\x17\x26"
			  "\xC5\x30\x23\x21\x30\xD4\x07\xF8"
			  "\x9A\xFE\xE0\x96\x49\x97\xF7\xA7",

	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= "\x8A\xCA\x26\x02\x79\x2A\xEC\x6F"
			  "\x11\xA6\x72\x06\x53\x1F\xB7\xD7"
			  "\xF0\xDF\xF5\x94\x13\x14\x5E\x69"
			  "\x73\xC4\x50\x01\xD0\x08\x7B\x42",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\x4E\x24\x48\xA4\xC6\xF4\x86\xBB"
			  "\x16\xB6\x56\x2C\x73\xB4\x02\x0B"
			  "\xF3\x04\x3E\x3A\x73\x1B\xCE\x72"
			  "\x1A\xE1\xB3\x03\xD9\x7E\x6D\x4C",
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= "\x37\x8C\x84\xA4\x12\x6E\x2D\xC6"
			  "\xE5\x6D\xCC\x74\x58\x37\x7A\xAC"
			  "\x83\x8D\x00\x03\x22\x30\xF5\x3C"
			  "\xE1\xF5\x70\x0C\x0F\xFB\x4D\x3B",
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= "\xF1\xD7\x54\x66\x26\x36\xFF\xE9"
			  "\x2C\x82\xEB\xB9\x21\x2A\x48\x4A"
			  "\x8D\x38\x63\x1E\xAD\x42\x38\xF5"
			  "\x44\x2E\xE1\x3B\x80\x54\xE4\x1B",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			   "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= "\xDC\x37\xE0\x08\xCF\x9E\xE6\x9B"
			  "\xF1\x1F\x00\xED\x9A\xBA\x26\x90"
			  "\x1D\xD7\xC2\x8C\xDE\xC0\x66\xCC"
			  "\x6A\xF4\x2E\x40\xF8\x2F\x3A\x1E",
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			   "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= "\x46\x6E\xF1\x8B\xAB\xB0\x15\x4D"
			  "\x25\xB9\xD3\x8A\x64\x14\xF5\xC0"
			  "\x87\x84\x37\x2B\xCC\xB2\x04\xD6"
			  "\x54\x9C\x4A\xFA\xDB\x60\x14\x29",
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= "\x2A\x98\x7E\xA4\x0F\x91\x70\x61"
			  "\xF5\xD6\xF0\xA0\xE4\x64\x4F\x48"
			  "\x8A\x7A\x5A\x52\xDE\xEE\x65\x62"
			  "\x07\xC5\x62\xF9\x88\xE9\x5C\x69",
	},
};

#define TGR192_TEST_VECTORS	6

static struct hash_testvec tgr192_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest = "\x24\xf0\x13\x0c\x63\xac\x93\x32"
			  "\x16\x16\x6e\x76\xb1\xbb\x92\x5f"
			  "\xf3\x73\xde\x2d\x49\x58\x4e\x7a",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest = "\xf2\x58\xc1\xe8\x84\x14\xab\x2a"
			  "\x52\x7a\xb5\x41\xff\xc5\xb8\xbf"
			  "\x93\x5f\x7b\x95\x1c\x13\x29\x51",
	}, {
		.plaintext = "Tiger",
		.psize	= 5,
		.digest = "\x9f\x00\xf5\x99\x07\x23\x00\xdd"
			  "\x27\x6a\xbb\x38\xc8\xeb\x6d\xec"
			  "\x37\x79\x0c\x11\x6f\x9d\x2b\xdf",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = "\x87\xfb\x2a\x90\x83\x85\x1c\xf7"
			  "\x47\x0d\x2c\xf8\x10\xe6\xdf\x9e"
			  "\xb5\x86\x44\x50\x34\xa5\xa3\x86",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ=abcdefghijklmnopqrstuvwxyz+0123456789",
		.psize	= 64,
		.digest = "\x46\x7d\xb8\x08\x63\xeb\xce\x48"
			  "\x8d\xf1\xcd\x12\x61\x65\x5d\xe9"
			  "\x57\x89\x65\x65\x97\x5f\x91\x97",
	}, {
		.plaintext = "Tiger - A Fast New Hash Function, "
			   "by Ross Anderson and Eli Biham, "
			   "proceedings of Fast Software Encryption 3, "
			   "Cambridge, 1996.",
		.psize  = 125,
		.digest = "\x3d\x9a\xeb\x03\xd1\xbd\x1a\x63"
			  "\x57\xb2\x77\x4d\xfd\x6d\x5b\x24"
			  "\xdd\x68\x15\x1d\x50\x39\x74\xfc",
	},
};

#define TGR160_TEST_VECTORS	6

static struct hash_testvec tgr160_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest = "\x24\xf0\x13\x0c\x63\xac\x93\x32"
			  "\x16\x16\x6e\x76\xb1\xbb\x92\x5f"
			  "\xf3\x73\xde\x2d",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest = "\xf2\x58\xc1\xe8\x84\x14\xab\x2a"
			  "\x52\x7a\xb5\x41\xff\xc5\xb8\xbf"
			  "\x93\x5f\x7b\x95",
	}, {
		.plaintext = "Tiger",
		.psize	= 5,
		.digest = "\x9f\x00\xf5\x99\x07\x23\x00\xdd"
			  "\x27\x6a\xbb\x38\xc8\xeb\x6d\xec"
			  "\x37\x79\x0c\x11",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = "\x87\xfb\x2a\x90\x83\x85\x1c\xf7"
			  "\x47\x0d\x2c\xf8\x10\xe6\xdf\x9e"
			  "\xb5\x86\x44\x50",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ=abcdefghijklmnopqrstuvwxyz+0123456789",
		.psize	= 64,
		.digest = "\x46\x7d\xb8\x08\x63\xeb\xce\x48"
			  "\x8d\xf1\xcd\x12\x61\x65\x5d\xe9"
			  "\x57\x89\x65\x65",
	}, {
		.plaintext = "Tiger - A Fast New Hash Function, "
			   "by Ross Anderson and Eli Biham, "
			   "proceedings of Fast Software Encryption 3, "
			   "Cambridge, 1996.",
		.psize  = 125,
		.digest = "\x3d\x9a\xeb\x03\xd1\xbd\x1a\x63"
			  "\x57\xb2\x77\x4d\xfd\x6d\x5b\x24"
			  "\xdd\x68\x15\x1d",
	},
};

#define TGR128_TEST_VECTORS	6

static struct hash_testvec tgr128_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest = "\x24\xf0\x13\x0c\x63\xac\x93\x32"
			  "\x16\x16\x6e\x76\xb1\xbb\x92\x5f",
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest = "\xf2\x58\xc1\xe8\x84\x14\xab\x2a"
			  "\x52\x7a\xb5\x41\xff\xc5\xb8\xbf",
	}, {
		.plaintext = "Tiger",
		.psize	= 5,
		.digest = "\x9f\x00\xf5\x99\x07\x23\x00\xdd"
			  "\x27\x6a\xbb\x38\xc8\xeb\x6d\xec",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = "\x87\xfb\x2a\x90\x83\x85\x1c\xf7"
			  "\x47\x0d\x2c\xf8\x10\xe6\xdf\x9e",
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ=abcdefghijklmnopqrstuvwxyz+0123456789",
		.psize	= 64,
		.digest = "\x46\x7d\xb8\x08\x63\xeb\xce\x48"
			  "\x8d\xf1\xcd\x12\x61\x65\x5d\xe9",
	}, {
		.plaintext = "Tiger - A Fast New Hash Function, "
			   "by Ross Anderson and Eli Biham, "
			   "proceedings of Fast Software Encryption 3, "
			   "Cambridge, 1996.",
		.psize  = 125,
		.digest = "\x3d\x9a\xeb\x03\xd1\xbd\x1a\x63"
			  "\x57\xb2\x77\x4d\xfd\x6d\x5b\x24",
	},
};

#define GHASH_TEST_VECTORS 1

static struct hash_testvec ghash_tv_template[] =
{
	{

		.key	= "\xdf\xa6\xbf\x4d\xed\x81\xdb\x03\xff\xca\xff\x95\xf8\x30\xf0\x61",
		.ksize	= 16,
		.plaintext = "\x95\x2b\x2a\x56\xa5\x60\x04a\xc0\xb3\x2b\x66\x56\xa0\x5b\x40\xb6",
		.psize	= 16,
		.digest	= "\xda\x53\xeb\x0a\xd2\xc5\x5b\xb6"
			  "\x4f\xc4\x80\x2c\xc3\xfe\xda\x60",
	},
};

#define HMAC_MD5_TEST_VECTORS	7

static struct hash_testvec hmac_md5_tv_template[] =
{
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 16,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x92\x94\x72\x7a\x36\x38\xbb\x1c"
			  "\x13\xf4\x8e\xf8\x15\x8b\xfc\x9d",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x75\x0c\x78\x3e\x6a\xb0\xb5\x03"
			  "\xea\xa8\x6e\x31\x0a\x5d\xb7\x38",
		.np	= 2,
		.tap	= {14, 14}
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 16,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\x56\xbe\x34\x52\x1d\x14\x4c\x88"
			  "\xdb\xb8\xc7\x33\xf0\xe8\xb3\xf6",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18\x19",
		.ksize	= 25,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\x69\x7e\xaf\x0a\xca\x3a\x3a\xea"
			  "\x3a\x75\x16\x47\x46\xff\xaa\x79",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 16,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x56\x46\x1e\xf2\x34\x2e\xdc\x00"
			  "\xf9\xba\xb9\x95\x69\x0e\xfd\x4c",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x6b\x1a\xb7\xfe\x4b\xd7\xbf\x8f"
			  "\x0b\x62\xe6\xce\x61\xb9\xd0\xcd",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			   "Block-Size Data",
		.psize	= 73,
		.digest	= "\x6f\x63\x0f\xad\x67\xcd\xa0\xee"
			  "\x1f\xb1\xf5\x62\xdb\x3a\xa5\x3e",
	},
};

#define HMAC_RMD128_TEST_VECTORS	7

static struct hash_testvec hmac_rmd128_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 16,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xfb\xf6\x1f\x94\x92\xaa\x4b\xbf"
			  "\x81\xc1\x72\xe8\x4e\x07\x34\xdb",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x87\x5f\x82\x88\x62\xb6\xb3\x34"
			  "\xb4\x27\xc5\x5f\x9f\x7f\xf0\x9b",
		.np	= 2,
		.tap	= { 14, 14 },
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 16,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\x09\xf0\xb2\x84\x6d\x2f\x54\x3d"
			  "\xa3\x63\xcb\xec\x8d\x62\xa3\x8d",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18\x19",
		.ksize	= 25,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\xbd\xbb\xd7\xcf\x03\xe4\x4b\x5a"
			  "\xa6\x0a\xf8\x15\xbe\x4d\x22\x94",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 16,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\xe7\x98\x08\xf2\x4b\x25\xfd\x03"
			  "\x1c\x15\x5f\x0d\x55\x1d\x9a\x3a",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\xdc\x73\x29\x28\xde\x98\x10\x4a"
			  "\x1f\x59\xd3\x73\xc1\x50\xac\xbb",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			   "Block-Size Data",
		.psize	= 73,
		.digest	= "\x5c\x6b\xec\x96\x79\x3e\x16\xd4"
			  "\x06\x90\xc2\x37\x63\x5f\x30\xc5",
	},
};

#define HMAC_RMD160_TEST_VECTORS	7

static struct hash_testvec hmac_rmd160_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x24\xcb\x4b\xd6\x7d\x20\xfc\x1a\x5d\x2e"
			  "\xd7\x73\x2d\xcc\x39\x37\x7f\x0a\x56\x68",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xdd\xa6\xc0\x21\x3a\x48\x5a\x9e\x24\xf4"
			  "\x74\x20\x64\xa7\xf0\x33\xb4\x3c\x40\x69",
		.np	= 2,
		.tap	= { 14, 14 },
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 20,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\xb0\xb1\x05\x36\x0d\xe7\x59\x96\x0a\xb4"
			  "\xf3\x52\x98\xe1\x16\xe2\x95\xd8\xe7\xc1",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18\x19",
		.ksize	= 25,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\xd5\xca\x86\x2f\x4d\x21\xd5\xe6\x10\xe1"
			  "\x8b\x4c\xf1\xbe\xb9\x7a\x43\x65\xec\xf4",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 20,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x76\x19\x69\x39\x78\xf9\x1d\x90\x53\x9a"
			  "\xe7\x86\x50\x0f\xf3\xd8\xe0\x51\x8e\x39",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x64\x66\xca\x07\xac\x5e\xac\x29\xe1\xbd"
			  "\x52\x3e\x5a\xda\x76\x05\xb7\x91\xfd\x8b",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			   "Block-Size Data",
		.psize	= 73,
		.digest	= "\x69\xea\x60\x79\x8d\x71\x61\x6c\xce\x5f"
			  "\xd0\x87\x1e\x23\x75\x4c\xd7\x5d\x5a\x0a",
	},
};

#define HMAC_SHA1_TEST_VECTORS	7

static struct hash_testvec hmac_sha1_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xb6\x17\x31\x86\x55\x05\x72\x64"
			  "\xe2\x8b\xc0\xb6\xfb\x37\x8c\x8e\xf1"
			  "\x46\xbe",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xef\xfc\xdf\x6a\xe5\xeb\x2f\xa2\xd2\x74"
			  "\x16\xd5\xf1\x84\xdf\x9c\x25\x9a\x7c\x79",
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 20,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\x12\x5d\x73\x42\xb9\xac\x11\xcd\x91\xa3"
			  "\x9a\xf4\x8a\xa1\x7b\x4f\x63\xf1\x75\xd3",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18\x19",
		.ksize	= 25,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\x4c\x90\x07\xf4\x02\x62\x50\xc6\xbc\x84"
			  "\x14\xf9\xbf\x50\xc8\x6c\x2d\x72\x35\xda",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 20,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x4c\x1a\x03\x42\x4b\x55\xe0\x7f\xe7\xf2"
			  "\x7b\xe1\xd5\x8b\xb9\x32\x4a\x9a\x5a\x04",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\xaa\x4a\xe5\xe1\x52\x72\xd0\x0e\x95\x70"
			  "\x56\x37\xce\x8a\x3b\x55\xed\x40\x21\x12",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			   "Block-Size Data",
		.psize	= 73,
		.digest	= "\xe8\xe9\x9d\x0f\x45\x23\x7d\x78\x6d\x6b"
			  "\xba\xa7\x96\x5c\x78\x08\xbb\xff\x1a\x91",
	},
};

#define HMAC_SHA224_TEST_VECTORS    4

static struct hash_testvec hmac_sha224_tv_template[] = {
	{
		.key    = "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b",
		.ksize  = 20,
		 
		.plaintext = "\x48\x69\x20\x54\x68\x65\x72\x65",
		.psize  = 8,
		.digest = "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19"
			"\x68\x32\x10\x7c\xd4\x9d\xf3\x3f"
			"\x47\xb4\xb1\x16\x99\x12\xba\x4f"
			"\x53\x68\x4b\x22",
	}, {
		.key    = "Jefe",
		.ksize  = 4,
		 
		.plaintext = "\x77\x68\x61\x74\x20\x64\x6f\x20"
			"\x79\x61\x20\x77\x61\x6e\x74\x20"
			"\x66\x6f\x72\x20\x6e\x6f\x74\x68"
			"\x69\x6e\x67\x3f",
		.psize  = 28,
		.digest = "\xa3\x0e\x01\x09\x8b\xc6\xdb\xbf"
			"\x45\x69\x0f\x3a\x7e\x9e\x6d\x0f"
			"\x8b\xbe\xa2\xa3\x9e\x61\x48\x00"
			"\x8f\xd0\x5e\x44",
		.np = 4,
		.tap    = { 7, 7, 7, 7 }
	}, {
		.key    = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa",
		.ksize  = 131,
		 
		.plaintext = "\x54\x65\x73\x74\x20\x55\x73\x69"
			"\x6e\x67\x20\x4c\x61\x72\x67\x65"
			"\x72\x20\x54\x68\x61\x6e\x20\x42"
			"\x6c\x6f\x63\x6b\x2d\x53\x69\x7a"
			"\x65\x20\x4b\x65\x79\x20\x2d\x20"
			"\x48\x61\x73\x68\x20\x4b\x65\x79"
			"\x20\x46\x69\x72\x73\x74",
		.psize  = 54,
		.digest = "\x95\xe9\xa0\xdb\x96\x20\x95\xad"
			"\xae\xbe\x9b\x2d\x6f\x0d\xbc\xe2"
			"\xd4\x99\xf1\x12\xf2\xd2\xb7\x27"
			"\x3f\xa6\x87\x0e",
	}, {
		.key    = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa",
		.ksize  = 131,
		 
		.plaintext = "\x54\x68\x69\x73\x20\x69\x73\x20"
			"\x61\x20\x74\x65\x73\x74\x20\x75"
			"\x73\x69\x6e\x67\x20\x61\x20\x6c"
			"\x61\x72\x67\x65\x72\x20\x74\x68"
			"\x61\x6e\x20\x62\x6c\x6f\x63\x6b"
			"\x2d\x73\x69\x7a\x65\x20\x6b\x65"
			"\x79\x20\x61\x6e\x64\x20\x61\x20"
			"\x6c\x61\x72\x67\x65\x72\x20\x74"
			"\x68\x61\x6e\x20\x62\x6c\x6f\x63"
			"\x6b\x2d\x73\x69\x7a\x65\x20\x64"
			"\x61\x74\x61\x2e\x20\x54\x68\x65"
			"\x20\x6b\x65\x79\x20\x6e\x65\x65"
			"\x64\x73\x20\x74\x6f\x20\x62\x65"
			"\x20\x68\x61\x73\x68\x65\x64\x20"
			"\x62\x65\x66\x6f\x72\x65\x20\x62"
			"\x65\x69\x6e\x67\x20\x75\x73\x65"
			"\x64\x20\x62\x79\x20\x74\x68\x65"
			"\x20\x48\x4d\x41\x43\x20\x61\x6c"
			"\x67\x6f\x72\x69\x74\x68\x6d\x2e",
		.psize  = 152,
		.digest = "\x3a\x85\x41\x66\xac\x5d\x9f\x02"
			"\x3f\x54\xd5\x17\xd0\xb3\x9d\xbd"
			"\x94\x67\x70\xdb\x9c\x2b\x95\xc9"
			"\xf6\xf5\x65\xd1",
	},
};

#define HMAC_SHA256_TEST_VECTORS	10

static struct hash_testvec hmac_sha256_tv_template[] = {
	{
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20",
		.ksize	= 32,
		.plaintext = "abc",
		.psize	= 3,
		.digest	= "\xa2\x1b\x1f\x5d\x4c\xf4\xf7\x3a"
			  "\x4d\xd9\x39\x75\x0f\x7a\x06\x6a"
			  "\x7f\x98\xcc\x13\x1c\xb1\x6a\x66"
			  "\x92\x75\x90\x21\xcf\xab\x81\x81",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20",
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= "\x10\x4f\xdc\x12\x57\x32\x8f\x08"
			  "\x18\x4b\xa7\x31\x31\xc5\x3c\xae"
			  "\xe6\x98\xe3\x61\x19\x42\x11\x49"
			  "\xea\x8c\x71\x24\x56\x69\x7d\x30",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20",
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
			   "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 112,
		.digest	= "\x47\x03\x05\xfc\x7e\x40\xfe\x34"
			  "\xd3\xee\xb3\xe7\x73\xd9\x5a\xab"
			  "\x73\xac\xf0\xfd\x06\x04\x47\xa5"
			  "\xeb\x45\x95\xbf\x33\xa9\xd1\xa3",
	}, {
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			"\x0b\x0b\x0b\x0b\x0b\x0b",
		.ksize	= 32,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x19\x8a\x60\x7e\xb4\x4b\xfb\xc6"
			  "\x99\x03\xa0\xf1\xcf\x2b\xbd\xc5"
			  "\xba\x0a\xa3\xf3\xd9\xae\x3c\x1c"
			  "\x7a\x3b\x16\x96\xa0\xb6\x8c\xf7",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x5b\xdc\xc1\x46\xbf\x60\x75\x4e"
			  "\x6a\x04\x24\x26\x08\x95\x75\xc7"
			  "\x5a\x00\x3f\x08\x9d\x27\x39\x83"
			  "\x9d\xec\x58\xb9\x64\xec\x38\x43",
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa",
		.ksize	= 32,
		.plaintext = "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
			"\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",
		.psize	= 50,
		.digest	= "\xcd\xcb\x12\x20\xd1\xec\xcc\xea"
			  "\x91\xe5\x3a\xba\x30\x92\xf9\x62"
			  "\xe5\x49\xfe\x6c\xe9\xed\x7f\xdc"
			  "\x43\x19\x1f\xbd\xe4\x5c\x30\xb0",
	}, {
		.key	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			  "\x11\x12\x13\x14\x15\x16\x17\x18"
			  "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			  "\x21\x22\x23\x24\x25",
		.ksize	= 37,
		.plaintext = "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
			"\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd",
		.psize	= 50,
		.digest	= "\xd4\x63\x3c\x17\xf6\xfb\x8d\x74"
			  "\x4c\x66\xde\xe0\xf8\xf0\x74\x55"
			  "\x6e\xc4\xaf\x55\xef\x07\x99\x85"
			  "\x41\x46\x8e\xb4\x9b\xd2\xe9\x17",
	}, {
		.key	= "\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c"
			"\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c\x0c"
			"\x0c\x0c\x0c\x0c\x0c\x0c",
		.ksize	= 32,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= "\x75\x46\xaf\x01\x84\x1f\xc0\x9b"
			  "\x1a\xb9\xc3\x74\x9a\x5f\x1c\x17"
			  "\xd4\xf5\x89\x66\x8a\x58\x7b\x27"
			  "\x00\xa9\xc9\x7c\x11\x93\xcf\x42",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x69\x53\x02\x5e\xd9\x6f\x0c\x09"
			  "\xf8\x0a\x96\xf7\x8e\x65\x38\xdb"
			  "\xe2\xe7\xb8\x20\xe3\xdd\x97\x0e"
			  "\x7d\xdd\x39\x09\x1b\x32\x35\x2f",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa",
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than "
			   "One Block-Size Data",
		.psize	= 73,
		.digest	= "\x63\x55\xac\x22\xe8\x90\xd0\xa3"
			  "\xc8\x48\x1a\x5c\xa4\x82\x5b\xc8"
			  "\x84\xd3\xe7\xa1\xff\x98\xa2\xfc"
			  "\x2a\xc7\xd8\xe0\x64\xc3\xb2\xe6",
	},
};

#define CMAC_AES_TEST_VECTORS 6

static struct hash_testvec aes_cmac128_tv_template[] = {
	{  
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= zeroed_string,
		.digest		= "\xbb\x1d\x69\x29\xe9\x59\x37\x28"
				  "\x7f\xa3\x7d\x12\x9b\x75\x67\x46",
		.psize		= 0,
		.ksize		= 16,
	}, {
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a",
		.digest		= "\x07\x0a\x16\xb4\x6b\x4d\x41\x44"
				  "\xf7\x9b\xdd\x9d\xd0\x4a\x28\x7c",
		.psize		= 16,
		.ksize		= 16,
	}, {
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
				  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11",
		.digest		= "\xdf\xa6\x67\x47\xde\x9a\xe6\x30"
				  "\x30\xca\x32\x61\x14\x97\xc8\x27",
		.psize		= 40,
		.ksize		= 16,
	}, {
		.key		= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
				  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
				  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
				  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
				  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
				  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.digest		= "\x51\xf0\xbe\xbf\x7e\x3b\x9d\x92"
				  "\xfc\x49\x74\x17\x79\x36\x3c\xfe",
		.psize		= 64,
		.ksize		= 16,
	}, {  
		.key		= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
				  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
				  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
				  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.plaintext	= zeroed_string,
		.digest		= "\x02\x89\x62\xf6\x1b\x7b\xf8\x9e"
				  "\xfc\x6b\x55\x1f\x46\x67\xd9\x83",
		.psize		= 0,
		.ksize		= 32,
	}, {
		.key		= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
				  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
				  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
				  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
				  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
				  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
				  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
				  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.digest		= "\xe1\x99\x21\x90\x54\x9f\x6e\xd5"
				  "\x69\x6a\x2c\x05\x6c\x31\x54\x10",
		.psize		= 64,
		.ksize		= 32,
	}
};

#define CMAC_DES3_EDE_TEST_VECTORS 4

static struct hash_testvec des3_ede_cmac64_tv_template[] = {
 
	{
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= zeroed_string,
		.digest		= "\xb7\xa6\x88\xe1\x22\xff\xaf\x95",
		.psize		= 0,
		.ksize		= 24,
	}, {
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96",
		.digest		= "\x8e\x8f\x29\x31\x36\x28\x37\x97",
		.psize		= 8,
		.ksize		= 24,
	}, {
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57",
		.digest		= "\x74\x3d\xdb\xe0\xce\x2d\xc2\xed",
		.psize		= 20,
		.ksize		= 24,
	}, {
		.key		= "\x8a\xa8\x3b\xf8\xcb\xda\x10\x62"
				  "\x0b\xc1\xbf\x19\xfb\xb6\xcd\x58"
				  "\xbc\x31\x3d\x4a\x37\x1c\xa8\xb5",
		.plaintext	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
				  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
				  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
				  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51",
		.digest		= "\x33\xe6\xb1\x09\x24\x00\xea\xe5",
		.psize		= 32,
		.ksize		= 24,
	}
};

#define XCBC_AES_TEST_VECTORS 6

static struct hash_testvec aes_xcbc128_tv_template[] = {
	{
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = zeroed_string,
		.digest = "\x75\xf0\x25\x1d\x52\x8a\xc0\x1c"
			  "\x45\x73\xdf\xd5\x84\xd7\x9f\x29",
		.psize	= 0,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02",
		.digest	= "\x5b\x37\x65\x80\xae\x2f\x19\xaf"
			  "\xe7\x21\x9c\xee\xf1\x72\x75\x6f",
		.psize	= 3,
		.ksize	= 16,
	} , {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.digest = "\xd2\xa2\x46\xfa\x34\x9b\x68\xa7"
			  "\x99\x98\xa4\x39\x4f\xf7\xa2\x63",
		.psize	= 16,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			     "\x10\x11\x12\x13",
		.digest = "\x47\xf5\x1b\x45\x64\x96\x62\x15"
			  "\xb8\x98\x5c\x63\x05\x5e\xd3\x08",
		.tap	= { 10, 10 },
		.psize	= 20,
		.np	= 2,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			     "\x10\x11\x12\x13\x14\x15\x16\x17"
			     "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.digest = "\xf5\x4f\x0e\xc8\xd2\xb9\xf3\xd3"
			  "\x68\x07\x73\x4b\xd5\x28\x3f\xd4",
		.psize	= 32,
		.ksize	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = "\x00\x01\x02\x03\x04\x05\x06\x07"
			     "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			     "\x10\x11\x12\x13\x14\x15\x16\x17"
			     "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			     "\x20\x21",
		.digest = "\xbe\xcb\xb3\xbc\xcd\xb5\x18\xa3"
			  "\x06\x77\xd5\x48\x1f\xb6\xb4\xd8",
		.tap	= { 17, 17 },
		.psize	= 34,
		.np	= 2,
		.ksize	= 16,
	}
};

#define VMAC_AES_TEST_VECTORS	11
static char vmac_string1[128] = {'\x01', '\x01', '\x01', '\x01',
				'\x02', '\x03', '\x02', '\x02',
				'\x02', '\x04', '\x01', '\x07',
				'\x04', '\x01', '\x04', '\x03',};
static char vmac_string2[128] = {'a', 'b', 'c',};
static char vmac_string3[128] = {'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				'a', 'b', 'c', 'a', 'b', 'c',
				};

static char vmac_string4[17] = {'b', 'c', 'e', 'f',
				'i', 'j', 'l', 'm',
				'o', 'p', 'r', 's',
				't', 'u', 'w', 'x', 'z'};

static char vmac_string5[127] = {'r', 'm', 'b', 't', 'c',
				 'o', 'l', 'k', ']', '%',
				 '9', '2', '7', '!', 'A'};

static char vmac_string6[129] = {'p', 't', '*', '7', 'l',
				 'i', '!', '#', 'w', '0',
				 'z', '/', '4', 'A', 'n'};

static struct hash_testvec aes_vmac128_tv_template[] = {
	{
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = NULL,
		.digest	= "\x07\x58\x80\x35\x77\xa4\x7b\x54",
		.psize	= 0,
		.ksize	= 16,
	}, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = vmac_string1,
		.digest = "\xce\xf5\x3c\xd3\xae\x68\x8c\xa1",
		.psize  = 128,
		.ksize  = 16,
	}, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = vmac_string2,
		.digest = "\xc9\x27\xb0\x73\x81\xbd\x14\x2d",
		.psize  = 128,
		.ksize  = 16,
	}, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.plaintext = vmac_string3,
		.digest = "\x8d\x1a\x95\x8c\x98\x47\x0b\x19",
		.psize  = 128,
		.ksize  = 16,
	}, {
		.key	= "abcdefghijklmnop",
		.plaintext = NULL,
		.digest	= "\x3b\x89\xa1\x26\x9e\x55\x8f\x84",
		.psize	= 0,
		.ksize	= 16,
	}, {
		.key    = "abcdefghijklmnop",
		.plaintext = vmac_string1,
		.digest = "\xab\x5e\xab\xb0\xf6\x8d\x74\xc2",
		.psize  = 128,
		.ksize  = 16,
	}, {
		.key    = "abcdefghijklmnop",
		.plaintext = vmac_string2,
		.digest = "\x11\x15\x68\x42\x3d\x7b\x09\xdf",
		.psize  = 128,
		.ksize  = 16,
	}, {
		.key    = "abcdefghijklmnop",
		.plaintext = vmac_string3,
		.digest = "\x8b\x32\x8f\xe1\xed\x8f\xfa\xd4",
		.psize  = 128,
		.ksize  = 16,
	}, {
		.key = "a09b5cd!f#07K\x00\x00\x00",
		.plaintext = vmac_string4,
		.digest = "\xab\xa5\x0f\xea\x42\x4e\xa1\x5f",
		.psize = sizeof(vmac_string4),
		.ksize = 16,
	}, {
		.key = "a09b5cd!f#07K\x00\x00\x00",
		.plaintext = vmac_string5,
		.digest = "\x25\x31\x98\xbc\x1d\xe8\x67\x60",
		.psize = sizeof(vmac_string5),
		.ksize = 16,
	}, {
		.key = "a09b5cd!f#07K\x00\x00\x00",
		.plaintext = vmac_string6,
		.digest = "\xc4\xae\x9b\x47\x95\x65\xeb\x41",
		.psize = sizeof(vmac_string6),
		.ksize = 16,
	},
};

#define HMAC_SHA384_TEST_VECTORS	4

static struct hash_testvec hmac_sha384_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\xaf\xd0\x39\x44\xd8\x48\x95\x62"
			  "\x6b\x08\x25\xf4\xab\x46\x90\x7f"
			  "\x15\xf9\xda\xdb\xe4\x10\x1e\xc6"
			  "\x82\xaa\x03\x4c\x7c\xeb\xc5\x9c"
			  "\xfa\xea\x9e\xa9\x07\x6e\xde\x7f"
			  "\x4a\xf1\x52\xe8\xb2\xfa\x9c\xb6",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\xaf\x45\xd2\xe3\x76\x48\x40\x31"
			  "\x61\x7f\x78\xd2\xb5\x8a\x6b\x1b"
			  "\x9c\x7e\xf4\x64\xf5\xa0\x1b\x47"
			  "\xe4\x2e\xc3\x73\x63\x22\x44\x5e"
			  "\x8e\x22\x40\xca\x5e\x69\xe2\xc7"
			  "\x8b\x32\x39\xec\xfa\xb2\x16\x49",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Larger Than Block-Siz"
			   "e Key - Hash Key First",
		.psize	= 54,
		.digest	= "\x4e\xce\x08\x44\x85\x81\x3e\x90"
			  "\x88\xd2\xc6\x3a\x04\x1b\xc5\xb4"
			  "\x4f\x9e\xf1\x01\x2a\x2b\x58\x8f"
			  "\x3c\xd1\x1f\x05\x03\x3a\xc4\xc6"
			  "\x0c\x2e\xf6\xab\x40\x30\xfe\x82"
			  "\x96\x24\x8d\xf1\x63\xf4\x49\x52",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "This is a test u"
			   "sing a larger th"
			   "an block-size ke"
			   "y and a larger t"
			   "han block-size d"
			   "ata. The key nee"
			   "ds to be hashed "
			   "before being use"
			   "d by the HMAC al"
			   "gorithm.",
		.psize	= 152,
		.digest	= "\x66\x17\x17\x8e\x94\x1f\x02\x0d"
			  "\x35\x1e\x2f\x25\x4e\x8f\xd3\x2c"
			  "\x60\x24\x20\xfe\xb0\xb8\xfb\x9a"
			  "\xdc\xce\xbb\x82\x46\x1e\x99\xc5"
			  "\xa6\x78\xcc\x31\xe7\x99\x17\x6d"
			  "\x38\x60\xe6\x11\x0c\x46\x52\x3e",
	},
};

#define HMAC_SHA512_TEST_VECTORS	4

static struct hash_testvec hmac_sha512_tv_template[] = {
	{
		.key	= "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
			  "\x0b\x0b\x0b\x0b",
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= "\x87\xaa\x7c\xde\xa5\xef\x61\x9d"
			  "\x4f\xf0\xb4\x24\x1a\x1d\x6c\xb0"
			  "\x23\x79\xf4\xe2\xce\x4e\xc2\x78"
			  "\x7a\xd0\xb3\x05\x45\xe1\x7c\xde"
			  "\xda\xa8\x33\xb7\xd6\xb8\xa7\x02"
			  "\x03\x8b\x27\x4e\xae\xa3\xf4\xe4"
			  "\xbe\x9d\x91\x4e\xeb\x61\xf1\x70"
			  "\x2e\x69\x6c\x20\x3a\x12\x68\x54",
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= "\x16\x4b\x7a\x7b\xfc\xf8\x19\xe2"
			  "\xe3\x95\xfb\xe7\x3b\x56\xe0\xa3"
			  "\x87\xbd\x64\x22\x2e\x83\x1f\xd6"
			  "\x10\x27\x0c\xd7\xea\x25\x05\x54"
			  "\x97\x58\xbf\x75\xc0\x5a\x99\x4a"
			  "\x6d\x03\x4f\x65\xf8\xf0\xe6\xfd"
			  "\xca\xea\xb1\xa3\x4d\x4a\x6b\x4b"
			  "\x63\x6e\x07\x0a\x38\xbc\xe7\x37",
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			  "\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext = "Test Using Large"
			   "r Than Block-Siz"
			   "e Key - Hash Key"
			   " First",
		.psize	= 54,
		.digest	= "\x80\xb2\x42\x63\xc7\xc1\xa3\xeb"
			"\xb7\x14\x93\xc1\xdd\x7b\xe8\xb4"
			"\x9b\x46\xd1\xf4\x1b\x4a\xee\xc1"
			"\x12\x1b\x01\x37\x83\xf8\xf3\x52"
			"\x6b\x56\xd0\x37\xe0\x5f\x25\x98"
			"\xbd\x0f\xd2\x21\x5d\x6a\x1e\x52"
			"\x95\xe6\x4f\x73\xf6\x3f\x0a\xec"
			"\x8b\x91\x5a\x98\x5d\x78\x65\x98",
	}, {
		.key	= "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
			"\xaa\xaa\xaa",
		.ksize	= 131,
		.plaintext =
			  "This is a test u"
			  "sing a larger th"
			  "an block-size ke"
			  "y and a larger t"
			  "han block-size d"
			  "ata. The key nee"
			  "ds to be hashed "
			  "before being use"
			  "d by the HMAC al"
			  "gorithm.",
		.psize	= 152,
		.digest	= "\xe3\x7b\x6a\x77\x5d\xc8\x7d\xba"
			"\xa4\xdf\xa9\xf9\x6e\x5e\x3f\xfd"
			"\xde\xbd\x71\xf8\x86\x72\x89\x86"
			"\x5d\xf5\xa3\x2d\x20\xcd\xc9\x44"
			"\xb6\x02\x2c\xac\x3c\x49\x82\xb1"
			"\x0d\x5e\xeb\x55\xc3\xe4\xde\x15"
			"\x13\x46\x76\xfb\x6d\xe0\x44\x60"
			"\x65\xc9\x74\x40\xfa\x8c\x6a\x58",
	},
};

#define DES_ENC_TEST_VECTORS		11
#define DES_DEC_TEST_VECTORS		5
#define DES_CBC_ENC_TEST_VECTORS	6
#define DES_CBC_DEC_TEST_VECTORS	5
#define DES_CTR_ENC_TEST_VECTORS	2
#define DES_CTR_DEC_TEST_VECTORS	2
#define DES3_EDE_ENC_TEST_VECTORS	4
#define DES3_EDE_DEC_TEST_VECTORS	4
#define DES3_EDE_CBC_ENC_TEST_VECTORS	2
#define DES3_EDE_CBC_DEC_TEST_VECTORS	2
#define DES3_EDE_CTR_ENC_TEST_VECTORS	2
#define DES3_EDE_CTR_DEC_TEST_VECTORS	2

static struct cipher_testvec des_enc_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.ilen	= 8,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.rlen	= 8,
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 8,
		.result	= "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 8,
	}, {  
		.key	= "\x7c\xa1\x10\x45\x4a\x1a\x6e\x57",
		.klen	= 8,
		.input	= "\x01\xa1\xd6\xd0\x39\x77\x67\x42",
		.ilen	= 8,
		.result	= "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.rlen	= 8,
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xca\xfe\xba\xbe\xfe\xed\xbe\xef",
		.ilen	= 24,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b"
			  "\xb4\x99\x26\xf7\x1f\xe1\xd4\x90",
		.rlen	= 24,
	}, {  
		.fail	= 1,
		.wk	= 1,
		.key	= "\x01\x01\x01\x01\x01\x01\x01\x01",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.ilen	= 8,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.rlen	= 8,
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 16,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xca\xfe\xba\xbe\xfe\xed\xbe\xef"
			  "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 32,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b"
			  "\xb4\x99\x26\xf7\x1f\xe1\xd4\x90"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 32,
		.np	= 3,
		.tap	= { 14, 10, 8 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xca\xfe\xba\xbe\xfe\xed\xbe\xef",
		.ilen	= 24,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b"
			  "\xb4\x99\x26\xf7\x1f\xe1\xd4\x90",
		.rlen	= 24,
		.np	= 4,
		.tap	= { 2, 1, 3, 18 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\x22\x33\x44\x55\x66\x77\x88\x99",
		.ilen	= 16,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\xf7\x9c\x89\x2a\x33\x8f\x4a\x8b",
		.rlen	= 16,
		.np	= 5,
		.tap	= { 2, 2, 2, 2, 8 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.ilen	= 8,
		.result	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.rlen	= 8,
		.np	= 8,
		.tap	= { 1, 1, 1, 1, 1, 1, 1, 1 }
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.ilen	= 248,
		.result	= "\x88\xCB\x1F\xAB\x2F\x2A\x49\x57"
			  "\x92\xB9\x77\xFF\x2F\x47\x58\xDD"
			  "\xD7\x8A\x91\x95\x26\x33\x78\xB2"
			  "\x33\xBA\xB2\x3E\x02\xF5\x1F\xEF"
			  "\x98\xC5\xA6\xD2\x7D\x79\xEC\xB3"
			  "\x45\xF3\x4C\x61\xAC\x6C\xC2\x55"
			  "\xE5\xD3\x06\x58\x8A\x42\x3E\xDD"
			  "\x3D\x20\x45\xE9\x6F\x0D\x25\xA8"
			  "\xA5\xC7\x69\xCE\xD5\x3B\x7B\xC9"
			  "\x9E\x65\xE7\xA3\xF2\xE4\x18\x94"
			  "\xD2\x81\xE9\x33\x2B\x2D\x49\xC4"
			  "\xFE\xDA\x7F\xE2\xF2\x8C\x9C\xDC"
			  "\x73\x58\x11\x1F\x81\xD7\x21\x1A"
			  "\x80\xD0\x0D\xE8\x45\xD6\xD8\xD5"
			  "\x2E\x51\x16\xCA\x09\x89\x54\x62"
			  "\xF7\x04\x3D\x75\xB9\xA3\x84\xF4"
			  "\x62\xF0\x02\x58\x83\xAF\x30\x87"
			  "\x85\x3F\x01\xCD\x8E\x58\x42\xC4"
			  "\x41\x73\xE0\x15\x0A\xE6\x2E\x80"
			  "\x94\xF8\x5B\x3A\x4E\xDF\x51\xB2"
			  "\x9D\xE4\xC4\x9D\xF7\x3F\xF8\x8E"
			  "\x37\x22\x4D\x00\x2A\xEF\xC1\x0F"
			  "\x14\xA0\x66\xAB\x79\x39\xD0\x8E"
			  "\xE9\x95\x61\x74\x12\xED\x07\xD7"
			  "\xDD\x95\xDC\x7B\x57\x25\x27\x9C"
			  "\x51\x96\x16\xF7\x94\x61\xB8\x87"
			  "\xF0\x21\x1B\x32\xFB\x07\x0F\x29"
			  "\x56\xBD\x9D\x22\xA2\x9F\xA2\xB9"
			  "\x46\x31\x4C\x5E\x2E\x95\x61\xEF"
			  "\xE1\x58\x39\x09\xB4\x8B\x40\xAC"
			  "\x5F\x62\xC7\x72\xD9\xFC\xCB\x9A",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 248 - 8, 8 },
	},
};

static struct cipher_testvec des_dec_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xe7",
		.rlen	= 8,
	}, {  
		.key	= "\x7c\xa1\x10\x45\x4a\x1a\x6e\x57",
		.klen	= 8,
		.input	= "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.ilen	= 8,
		.result	= "\x01\xa1\xd6\xd0\x39\x77\x67\x42",
		.rlen	= 8,
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\xa3\x99\x7b\xca\xaf\x69\xa0\xf5",
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\xc9\x57\x44\x25\x6a\x5e\xd3\x1d"
			  "\x69\x0f\x5b\x0d\x9a\x26\x93\x9b",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xe7"
			  "\xa3\x99\x7b\xca\xaf\x69\xa0\xf5",
		.rlen	= 16,
		.np	= 3,
		.tap	= { 3, 12, 1 }
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.input	= "\x88\xCB\x1F\xAB\x2F\x2A\x49\x57"
			  "\x92\xB9\x77\xFF\x2F\x47\x58\xDD"
			  "\xD7\x8A\x91\x95\x26\x33\x78\xB2"
			  "\x33\xBA\xB2\x3E\x02\xF5\x1F\xEF"
			  "\x98\xC5\xA6\xD2\x7D\x79\xEC\xB3"
			  "\x45\xF3\x4C\x61\xAC\x6C\xC2\x55"
			  "\xE5\xD3\x06\x58\x8A\x42\x3E\xDD"
			  "\x3D\x20\x45\xE9\x6F\x0D\x25\xA8"
			  "\xA5\xC7\x69\xCE\xD5\x3B\x7B\xC9"
			  "\x9E\x65\xE7\xA3\xF2\xE4\x18\x94"
			  "\xD2\x81\xE9\x33\x2B\x2D\x49\xC4"
			  "\xFE\xDA\x7F\xE2\xF2\x8C\x9C\xDC"
			  "\x73\x58\x11\x1F\x81\xD7\x21\x1A"
			  "\x80\xD0\x0D\xE8\x45\xD6\xD8\xD5"
			  "\x2E\x51\x16\xCA\x09\x89\x54\x62"
			  "\xF7\x04\x3D\x75\xB9\xA3\x84\xF4"
			  "\x62\xF0\x02\x58\x83\xAF\x30\x87"
			  "\x85\x3F\x01\xCD\x8E\x58\x42\xC4"
			  "\x41\x73\xE0\x15\x0A\xE6\x2E\x80"
			  "\x94\xF8\x5B\x3A\x4E\xDF\x51\xB2"
			  "\x9D\xE4\xC4\x9D\xF7\x3F\xF8\x8E"
			  "\x37\x22\x4D\x00\x2A\xEF\xC1\x0F"
			  "\x14\xA0\x66\xAB\x79\x39\xD0\x8E"
			  "\xE9\x95\x61\x74\x12\xED\x07\xD7"
			  "\xDD\x95\xDC\x7B\x57\x25\x27\x9C"
			  "\x51\x96\x16\xF7\x94\x61\xB8\x87"
			  "\xF0\x21\x1B\x32\xFB\x07\x0F\x29"
			  "\x56\xBD\x9D\x22\xA2\x9F\xA2\xB9"
			  "\x46\x31\x4C\x5E\x2E\x95\x61\xEF"
			  "\xE1\x58\x39\x09\xB4\x8B\x40\xAC"
			  "\x5F\x62\xC7\x72\xD9\xFC\xCB\x9A",
		.ilen	= 248,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 248 - 8, 8 },
	},
};

static struct cipher_testvec des_cbc_enc_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x37\x36\x35\x34\x33\x32\x31\x20"
			  "\x4e\x6f\x77\x20\x69\x73\x20\x74"
			  "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.ilen	= 24,
		.result	= "\xcc\xd1\x73\xff\xab\x20\x39\xf4"
			  "\xac\xd8\xae\xfd\xdf\xd8\xa1\xeb"
			  "\x46\x8e\x91\x15\x78\x88\xba\x68",
		.rlen	= 24,
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x12\x34\x56\x78\x90\xab\xcd\xef",
		.input	= "\x4e\x6f\x77\x20\x69\x73\x20\x74",
		.ilen	= 8,
		.result	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.input	= "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.ilen	= 8,
		.result	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.input	= "\x66\x6f\x72\x20\x61\x6c\x6c\x20",
		.ilen	= 8,
		.result	= "\x68\x37\x88\x49\x9a\x7c\x05\xf6",
		.rlen	= 8,
	}, {  
	      
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x37\x36\x35\x34\x33\x32\x31\x20"
			  "\x4e\x6f\x77\x20\x69\x73\x20\x74"
			  "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.ilen	= 24,
		.result	= "\xcc\xd1\x73\xff\xab\x20\x39\xf4"
			  "\xac\xd8\xae\xfd\xdf\xd8\xa1\xeb"
			  "\x46\x8e\x91\x15\x78\x88\xba\x68",
		.rlen	= 24,
		.np	= 2,
		.tap	= { 13, 11 }
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.ilen	= 248,
		.result	= "\x71\xCC\x56\x1C\x87\x2C\x43\x20"
			  "\x1C\x20\x13\x09\xF9\x2B\x40\x47"
			  "\x99\x10\xD1\x1B\x65\x33\x33\xBA"
			  "\x88\x0D\xA2\xD1\x86\xFF\x4D\xF4"
			  "\x5A\x0C\x12\x96\x32\x57\xAA\x26"
			  "\xA7\xF4\x32\x8D\xBC\x10\x31\x9E"
			  "\x81\x72\x74\xDE\x30\x19\x69\x49"
			  "\x54\x9C\xC3\xEB\x0B\x97\xDD\xD1"
			  "\xE8\x6D\x0D\x05\x83\xA5\x12\x08"
			  "\x47\xF8\x88\x03\x86\x51\x3C\xEF"
			  "\xE7\x11\x73\x4D\x44\x2B\xE2\x16"
			  "\xE8\xA5\x06\x50\x66\x70\x0E\x14"
			  "\xBA\x21\x3B\xD5\x23\x5B\xA7\x8F"
			  "\x56\xB6\xA7\x44\xDB\x86\xAB\x69"
			  "\x33\x3C\xBE\x64\xC4\x22\xD3\xFE"
			  "\x49\x90\x88\x6A\x09\x8F\x76\x59"
			  "\xCB\xB7\xA0\x2D\x79\x75\x92\x8A"
			  "\x82\x1D\xC2\xFE\x09\x1F\x78\x6B"
			  "\x2F\xD6\xA4\x87\x1E\xC4\x53\x63"
			  "\x80\x02\x61\x2F\xE3\x46\xB6\xB5"
			  "\xAA\x95\xF4\xEE\xA7\x64\x2B\x4F"
			  "\x20\xCF\xD2\x47\x4E\x39\x65\xB3"
			  "\x11\x87\xA2\x6C\x49\x7E\x36\xC7"
			  "\x62\x8B\x48\x0D\x6A\x64\x00\xBD"
			  "\x71\x91\x8C\xE9\x70\x19\x01\x4F"
			  "\x4E\x68\x23\xBA\xDA\x24\x2E\x45"
			  "\x02\x14\x33\x21\xAE\x58\x4B\xCF"
			  "\x3B\x4B\xE8\xF8\xF6\x4F\x34\x93"
			  "\xD7\x07\x8A\xD7\x18\x92\x36\x8C"
			  "\x82\xA9\xBD\x6A\x31\x91\x39\x11"
			  "\xC6\x4A\xF3\x55\xC7\x29\x2E\x63",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 248 - 8, 8 },
	},
};

static struct cipher_testvec des_cbc_dec_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x12\x34\x56\x78\x90\xab\xcd\xef",
		.input	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.ilen	= 8,
		.result	= "\x4e\x6f\x77\x20\x69\x73\x20\x74",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\xe5\xc7\xcd\xde\x87\x2b\xf2\x7c",
		.input	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.ilen	= 8,
		.result	= "\x68\x65\x20\x74\x69\x6d\x65\x20",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.input	= "\x68\x37\x88\x49\x9a\x7c\x05\xf6",
		.ilen	= 8,
		.result	= "\x66\x6f\x72\x20\x61\x6c\x6c\x20",
		.rlen	= 8,
	}, {  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.iv	= "\x43\xe9\x34\x00\x8c\x38\x9c\x0f",
		.input	= "\x68\x37\x88\x49\x9a\x7c\x05\xf6",
		.ilen	= 8,
		.result	= "\x66\x6f\x72\x20\x61\x6c\x6c\x20",
		.rlen	= 8,
		.np	= 2,
		.tap	= { 4, 4 }
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x71\xCC\x56\x1C\x87\x2C\x43\x20"
			  "\x1C\x20\x13\x09\xF9\x2B\x40\x47"
			  "\x99\x10\xD1\x1B\x65\x33\x33\xBA"
			  "\x88\x0D\xA2\xD1\x86\xFF\x4D\xF4"
			  "\x5A\x0C\x12\x96\x32\x57\xAA\x26"
			  "\xA7\xF4\x32\x8D\xBC\x10\x31\x9E"
			  "\x81\x72\x74\xDE\x30\x19\x69\x49"
			  "\x54\x9C\xC3\xEB\x0B\x97\xDD\xD1"
			  "\xE8\x6D\x0D\x05\x83\xA5\x12\x08"
			  "\x47\xF8\x88\x03\x86\x51\x3C\xEF"
			  "\xE7\x11\x73\x4D\x44\x2B\xE2\x16"
			  "\xE8\xA5\x06\x50\x66\x70\x0E\x14"
			  "\xBA\x21\x3B\xD5\x23\x5B\xA7\x8F"
			  "\x56\xB6\xA7\x44\xDB\x86\xAB\x69"
			  "\x33\x3C\xBE\x64\xC4\x22\xD3\xFE"
			  "\x49\x90\x88\x6A\x09\x8F\x76\x59"
			  "\xCB\xB7\xA0\x2D\x79\x75\x92\x8A"
			  "\x82\x1D\xC2\xFE\x09\x1F\x78\x6B"
			  "\x2F\xD6\xA4\x87\x1E\xC4\x53\x63"
			  "\x80\x02\x61\x2F\xE3\x46\xB6\xB5"
			  "\xAA\x95\xF4\xEE\xA7\x64\x2B\x4F"
			  "\x20\xCF\xD2\x47\x4E\x39\x65\xB3"
			  "\x11\x87\xA2\x6C\x49\x7E\x36\xC7"
			  "\x62\x8B\x48\x0D\x6A\x64\x00\xBD"
			  "\x71\x91\x8C\xE9\x70\x19\x01\x4F"
			  "\x4E\x68\x23\xBA\xDA\x24\x2E\x45"
			  "\x02\x14\x33\x21\xAE\x58\x4B\xCF"
			  "\x3B\x4B\xE8\xF8\xF6\x4F\x34\x93"
			  "\xD7\x07\x8A\xD7\x18\x92\x36\x8C"
			  "\x82\xA9\xBD\x6A\x31\x91\x39\x11"
			  "\xC6\x4A\xF3\x55\xC7\x29\x2E\x63",
		.ilen	= 248,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 248 - 8, 8 },
	},
};

static struct cipher_testvec des_ctr_enc_tv_template[] = {
	{  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.ilen	= 248,
		.result	= "\x2F\x96\x06\x0F\x50\xC9\x68\x03"
			  "\x0F\x31\xD4\x64\xA5\x29\x77\x35"
			  "\xBC\x7A\x9F\x19\xE7\x0D\x33\x3E"
			  "\x12\x0B\x8C\xAE\x48\xAE\xD9\x02"
			  "\x0A\xD4\xB0\xD6\x37\xB2\x65\x1C"
			  "\x4B\x65\xEB\x24\xB5\x8E\xAD\x47"
			  "\x0D\xDA\x79\x77\xA0\x29\xA0\x2B"
			  "\xC8\x0F\x85\xDC\x03\x13\xA9\x04"
			  "\x19\x40\xBE\xBE\x5C\x49\x4A\x69"
			  "\xED\xE8\xE1\x9E\x14\x43\x74\xDE"
			  "\xEC\x6E\x11\x3F\x36\xEF\x7B\xFB"
			  "\xBE\x4C\x91\x43\x22\x65\x72\x48"
			  "\xE2\x12\xED\x88\xAC\xA7\xC9\x91"
			  "\x14\xA2\x36\x1C\x29\xFF\xC8\x4F"
			  "\x72\x5C\x4B\xB0\x1E\x93\xC2\xFA"
			  "\x9D\x53\x86\xA0\xAE\xC6\xB7\x3C"
			  "\x59\x0C\xD0\x8F\xA6\xD8\xA4\x31"
			  "\xB7\x30\x1C\x21\x38\xFB\x68\x8C"
			  "\x2E\xF5\x6E\x73\xC3\x16\x5F\x12"
			  "\x0C\x33\xB9\x1E\x7B\x70\xDE\x86"
			  "\x32\xB3\xC1\x16\xAB\xD9\x49\x0B"
			  "\x96\x28\x72\x6B\xF3\x30\xA9\xEB"
			  "\x69\xE2\x1E\x58\x46\xA2\x8E\xC7"
			  "\xC0\xEF\x07\xB7\x77\x2C\x00\x05"
			  "\x46\xBD\xFE\x53\x81\x8B\xA4\x03"
			  "\x20\x0F\xDB\x78\x0B\x1F\x53\x04"
			  "\x4C\x60\x4C\xC3\x2A\x86\x86\x7E"
			  "\x13\xD2\x26\xED\x5D\x3E\x9C\xF2"
			  "\x5C\xC4\x15\xC9\x9A\x21\xC5\xCD"
			  "\x19\x7F\x99\x19\x53\xCE\x1D\x14"
			  "\x69\x74\xA1\x06\x46\x0F\x4E\x75",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 248 - 8, 8 },
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82",
		.ilen	= 247,
		.result	= "\x62\xE5\xF4\xDC\x99\xE7\x89\xE3"
			  "\xF4\x10\xCC\x21\x99\xEB\xDC\x15"
			  "\x19\x13\x93\x27\x9D\xB6\x6F\x45"
			  "\x17\x55\x61\x72\xC8\xD3\x7F\xA5"
			  "\x32\xD0\xD3\x02\x15\xA4\x05\x23"
			  "\x9C\x23\x61\x60\x77\x7B\x6C\x95"
			  "\x26\x49\x42\x2E\xF3\xC1\x8C\x6D"
			  "\xC8\x47\xD5\x94\xE7\x53\xC8\x23"
			  "\x1B\xA5\x0B\xCB\x12\xD3\x7A\x12"
			  "\xA4\x42\x15\x34\xF7\x5F\xDC\x58"
			  "\x5B\x58\x4C\xAD\xD1\x33\x8E\xE6"
			  "\xE5\xA0\xDA\x4D\x94\x3D\x63\xA8"
			  "\x02\x82\xBB\x16\xB8\xDC\xB5\x58"
			  "\xC3\x2D\x79\xE4\x25\x79\x43\xF9"
			  "\x6D\xD3\xCA\xC0\xE8\x12\xD4\x7E"
			  "\x04\x25\x79\xFD\x27\xFB\xC4\xEA"
			  "\x32\x94\x48\x92\xF3\x68\x1A\x7F"
			  "\x36\x33\x43\x79\xF7\xCA\xC2\x38"
			  "\xC0\x68\xD4\x53\xA9\xCC\x43\x0C"
			  "\x40\x57\x3E\xED\x00\x9F\x22\x6E"
			  "\x80\x99\x0B\xCC\x40\x63\x46\x8A"
			  "\xE8\xC4\x9B\x6D\x7A\x08\x6E\xA9"
			  "\x6F\x84\xBC\xB3\xF4\x95\x0B\x2D"
			  "\x6A\xBA\x37\x50\xC3\xCF\x9F\x7C"
			  "\x59\x5E\xDE\x0B\x30\xFA\x34\x8A"
			  "\xF8\xD1\xA2\xF8\x4E\xBD\x5D\x5E"
			  "\x7D\x71\x99\xE0\xF6\xE5\x7C\xE0"
			  "\x6D\xEE\x82\x89\x92\xD4\xF5\xD7"
			  "\xDF\x85\x2D\xE1\xB2\xD6\xAB\x94"
			  "\xA5\xA6\xE7\xB0\x51\x36\x52\x37"
			  "\x91\x45\x05\x3E\x58\xBF\x32",
		.rlen	= 247,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 247 - 8, 8 },
	},
};

static struct cipher_testvec des_ctr_dec_tv_template[] = {
	{  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x2F\x96\x06\x0F\x50\xC9\x68\x03"
			  "\x0F\x31\xD4\x64\xA5\x29\x77\x35"
			  "\xBC\x7A\x9F\x19\xE7\x0D\x33\x3E"
			  "\x12\x0B\x8C\xAE\x48\xAE\xD9\x02"
			  "\x0A\xD4\xB0\xD6\x37\xB2\x65\x1C"
			  "\x4B\x65\xEB\x24\xB5\x8E\xAD\x47"
			  "\x0D\xDA\x79\x77\xA0\x29\xA0\x2B"
			  "\xC8\x0F\x85\xDC\x03\x13\xA9\x04"
			  "\x19\x40\xBE\xBE\x5C\x49\x4A\x69"
			  "\xED\xE8\xE1\x9E\x14\x43\x74\xDE"
			  "\xEC\x6E\x11\x3F\x36\xEF\x7B\xFB"
			  "\xBE\x4C\x91\x43\x22\x65\x72\x48"
			  "\xE2\x12\xED\x88\xAC\xA7\xC9\x91"
			  "\x14\xA2\x36\x1C\x29\xFF\xC8\x4F"
			  "\x72\x5C\x4B\xB0\x1E\x93\xC2\xFA"
			  "\x9D\x53\x86\xA0\xAE\xC6\xB7\x3C"
			  "\x59\x0C\xD0\x8F\xA6\xD8\xA4\x31"
			  "\xB7\x30\x1C\x21\x38\xFB\x68\x8C"
			  "\x2E\xF5\x6E\x73\xC3\x16\x5F\x12"
			  "\x0C\x33\xB9\x1E\x7B\x70\xDE\x86"
			  "\x32\xB3\xC1\x16\xAB\xD9\x49\x0B"
			  "\x96\x28\x72\x6B\xF3\x30\xA9\xEB"
			  "\x69\xE2\x1E\x58\x46\xA2\x8E\xC7"
			  "\xC0\xEF\x07\xB7\x77\x2C\x00\x05"
			  "\x46\xBD\xFE\x53\x81\x8B\xA4\x03"
			  "\x20\x0F\xDB\x78\x0B\x1F\x53\x04"
			  "\x4C\x60\x4C\xC3\x2A\x86\x86\x7E"
			  "\x13\xD2\x26\xED\x5D\x3E\x9C\xF2"
			  "\x5C\xC4\x15\xC9\x9A\x21\xC5\xCD"
			  "\x19\x7F\x99\x19\x53\xCE\x1D\x14"
			  "\x69\x74\xA1\x06\x46\x0F\x4E\x75",
		.ilen	= 248,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB",
		.rlen	= 248,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 248 - 8, 8 },
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55",
		.klen	= 8,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47",
		.input	= "\x62\xE5\xF4\xDC\x99\xE7\x89\xE3"
			  "\xF4\x10\xCC\x21\x99\xEB\xDC\x15"
			  "\x19\x13\x93\x27\x9D\xB6\x6F\x45"
			  "\x17\x55\x61\x72\xC8\xD3\x7F\xA5"
			  "\x32\xD0\xD3\x02\x15\xA4\x05\x23"
			  "\x9C\x23\x61\x60\x77\x7B\x6C\x95"
			  "\x26\x49\x42\x2E\xF3\xC1\x8C\x6D"
			  "\xC8\x47\xD5\x94\xE7\x53\xC8\x23"
			  "\x1B\xA5\x0B\xCB\x12\xD3\x7A\x12"
			  "\xA4\x42\x15\x34\xF7\x5F\xDC\x58"
			  "\x5B\x58\x4C\xAD\xD1\x33\x8E\xE6"
			  "\xE5\xA0\xDA\x4D\x94\x3D\x63\xA8"
			  "\x02\x82\xBB\x16\xB8\xDC\xB5\x58"
			  "\xC3\x2D\x79\xE4\x25\x79\x43\xF9"
			  "\x6D\xD3\xCA\xC0\xE8\x12\xD4\x7E"
			  "\x04\x25\x79\xFD\x27\xFB\xC4\xEA"
			  "\x32\x94\x48\x92\xF3\x68\x1A\x7F"
			  "\x36\x33\x43\x79\xF7\xCA\xC2\x38"
			  "\xC0\x68\xD4\x53\xA9\xCC\x43\x0C"
			  "\x40\x57\x3E\xED\x00\x9F\x22\x6E"
			  "\x80\x99\x0B\xCC\x40\x63\x46\x8A"
			  "\xE8\xC4\x9B\x6D\x7A\x08\x6E\xA9"
			  "\x6F\x84\xBC\xB3\xF4\x95\x0B\x2D"
			  "\x6A\xBA\x37\x50\xC3\xCF\x9F\x7C"
			  "\x59\x5E\xDE\x0B\x30\xFA\x34\x8A"
			  "\xF8\xD1\xA2\xF8\x4E\xBD\x5D\x5E"
			  "\x7D\x71\x99\xE0\xF6\xE5\x7C\xE0"
			  "\x6D\xEE\x82\x89\x92\xD4\xF5\xD7"
			  "\xDF\x85\x2D\xE1\xB2\xD6\xAB\x94"
			  "\xA5\xA6\xE7\xB0\x51\x36\x52\x37"
			  "\x91\x45\x05\x3E\x58\xBF\x32",
		.ilen	= 247,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82",
		.rlen	= 247,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 247 - 8, 8 },
	},
};

static struct cipher_testvec des3_ede_enc_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\x55\x55\x55\x55\x55\x55\x55\x55"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 24,
		.input	= "\x73\x6f\x6d\x65\x64\x61\x74\x61",
		.ilen	= 8,
		.result	= "\x18\xd7\x48\xe5\x63\x62\x05\x72",
		.rlen	= 8,
	}, {
		.key	= "\x03\x52\x02\x07\x67\x20\x82\x17"
			  "\x86\x02\x87\x66\x59\x08\x21\x98"
			  "\x64\x05\x6a\xbd\xfe\xa9\x34\x57",
		.klen	= 24,
		.input	= "\x73\x71\x75\x69\x67\x67\x6c\x65",
		.ilen	= 8,
		.result	= "\xc0\x7d\x2a\x0f\xa5\x66\xfa\x30",
		.rlen	= 8,
	}, {
		.key	= "\x10\x46\x10\x34\x89\x98\x80\x20"
			  "\x91\x07\xd0\x15\x89\x19\x01\x01"
			  "\x19\x07\x92\x10\x98\x1a\x01\x01",
		.klen	= 24,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\xe1\xef\x62\xc3\x32\xfe\x82\x5b",
		.rlen	= 8,
	}, {  
		.key	= "\xF3\x9C\xD6\xF3\x9C\xB9\x5A\x67"
			  "\x00\x5A\x67\x00\x2D\xCE\xEB\x2D"
			  "\xCE\xEB\xB4\x51\x72\xB4\x51\x72",
		.klen	= 24,
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.ilen	= 496,
		.result	= "\x4E\x9A\x40\x3D\x61\x7D\x17\xFA"
			  "\x16\x86\x88\x0B\xD8\xAE\xF8\xE4"
			  "\x81\x01\x04\x00\x76\xFA\xED\xD3"
			  "\x44\x7E\x21\x9D\xF0\xFB\x2B\x64"
			  "\xCA\x4E\x90\xE0\xC0\x63\x28\x92"
			  "\xF3\x1F\xA4\x53\x2C\x77\xCC\x77"
			  "\x69\x56\xD0\x19\xAD\x00\x2D\x97"
			  "\xBC\xDE\x49\x6A\x82\xBC\x16\xE2"
			  "\x2F\x3E\x72\xEE\xD1\xCE\xFC\x1B"
			  "\xEA\x32\x56\xE4\x0B\xAF\x27\x36"
			  "\xAF\x08\xB9\x61\xB7\x48\x23\x27"
			  "\xEE\x4D\xC8\x79\x56\x06\xEB\xC7"
			  "\x5B\xCA\x0A\xC6\x5E\x5C\xCB\xB6"
			  "\x9D\xDA\x04\x59\xE2\x09\x48\x7E"
			  "\x6B\x37\xC6\xFE\x92\xA9\x1E\x6E"
			  "\x0D\x19\xFA\x33\x0F\xEE\x36\x68"
			  "\x11\xBB\xF9\x5A\x73\xAB\x3A\xEA"
			  "\xAC\x28\xD8\xD5\x27\xE8\x6B\x16"
			  "\x45\x86\x50\x01\x70\x35\x99\x92"
			  "\xDF\x0C\x07\x88\x8B\x7F\x9E\x4B"
			  "\xD2\x04\x84\x90\xC4\x27\xDF\x0A"
			  "\x49\xA8\xA7\x1A\x6D\x78\x16\xCA"
			  "\xB3\x18\x5C\xC3\x93\x63\x5A\x68"
			  "\x77\x02\xBA\xED\x62\x71\xB1\xD9"
			  "\x5E\xE5\x6F\x1A\xCC\x1D\xBE\x2E"
			  "\x11\xF3\xA6\x97\xCA\x8E\xBF\xB4"
			  "\x56\xA1\x36\x6B\xB1\x0A\x3E\x70"
			  "\xEA\xD7\xCD\x72\x7B\x79\xC8\xAD"
			  "\x6B\xFE\xFB\xBA\x64\xAE\x19\xC1"
			  "\x82\xCF\x8A\xA1\x50\x17\x7F\xB2"
			  "\x6F\x7B\x0F\x52\xC5\x3E\x4A\x52"
			  "\x3F\xD9\x3F\x01\xA6\x41\x1A\xB3"
			  "\xB3\x7A\x0E\x8E\x75\xB2\xB1\x5F"
			  "\xDB\xEA\x84\x13\x26\x6C\x85\x4E"
			  "\xAE\x6B\xDC\xE7\xE7\xAD\xB0\x06"
			  "\x5C\xBA\x92\xD0\x30\xBB\x8D\xD2"
			  "\xAE\x4C\x70\x85\xA0\x07\xE3\x2C"
			  "\xD1\x27\x9C\xCF\xDB\x13\xB7\xE5"
			  "\xF9\x6A\x02\xD0\x39\x9D\xB6\xE7"
			  "\xD1\x17\x25\x08\xF9\xA9\xA6\x67"
			  "\x38\x80\xD1\x22\xAB\x1A\xD7\x26"
			  "\xAD\xCA\x19\x1B\xFA\x18\xA7\x57"
			  "\x31\xEC\xC9\xED\xDB\x79\xC0\x48"
			  "\xAC\x31\x9F\x03\x8B\x62\x5B\x7E"
			  "\x0E\xA6\xD0\x64\xEE\xEA\x00\xFC"
			  "\x58\xC8\xDE\x51\x4E\x17\x15\x11"
			  "\x66\x58\xB6\x90\xDC\xDF\xA1\x49"
			  "\xCA\x79\xE9\x31\x31\x42\xDC\x56"
			  "\x0B\xCD\xB6\x0D\xC7\x64\xF7\x19"
			  "\xD9\x42\x05\x7F\xBC\x2F\xFC\x90"
			  "\xAE\x29\x86\xAA\x43\x7A\x4F\x6B"
			  "\xCE\xEA\xBC\x31\x8D\x65\x9D\x46"
			  "\xEA\x77\xB4\xF9\x58\xEA\x5D\x84"
			  "\xE4\xDC\x14\xBB\xBD\x15\x0E\xDA"
			  "\xD8\xE4\xA4\x5D\x61\xF9\x58\x0F"
			  "\xE4\x82\x77\xCE\x87\xC0\x09\xF0"
			  "\xD6\x10\x9E\x34\xE1\x0C\x67\x55"
			  "\x7B\x6D\xD5\x51\x4B\x00\xEE\xBA"
			  "\xF2\x7B\xBE\x75\x07\x42\x9D\x99"
			  "\x12\xE1\x71\x4A\xF9\x2A\xF5\xF6"
			  "\x93\x03\xD7\x51\x09\xFA\xBE\x68"
			  "\xD8\x45\xFF\x33\xBA\xBB\x2B\x63",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec des3_ede_dec_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\x55\x55\x55\x55\x55\x55\x55\x55"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 24,
		.input	= "\x18\xd7\x48\xe5\x63\x62\x05\x72",
		.ilen	= 8,
		.result	= "\x73\x6f\x6d\x65\x64\x61\x74\x61",
		.rlen	= 8,
	}, {
		.key	= "\x03\x52\x02\x07\x67\x20\x82\x17"
			  "\x86\x02\x87\x66\x59\x08\x21\x98"
			  "\x64\x05\x6a\xbd\xfe\xa9\x34\x57",
		.klen	= 24,
		.input	= "\xc0\x7d\x2a\x0f\xa5\x66\xfa\x30",
		.ilen	= 8,
		.result	= "\x73\x71\x75\x69\x67\x67\x6c\x65",
		.rlen	= 8,
	}, {
		.key	= "\x10\x46\x10\x34\x89\x98\x80\x20"
			  "\x91\x07\xd0\x15\x89\x19\x01\x01"
			  "\x19\x07\x92\x10\x98\x1a\x01\x01",
		.klen	= 24,
		.input	= "\xe1\xef\x62\xc3\x32\xfe\x82\x5b",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {  
		.key	= "\xF3\x9C\xD6\xF3\x9C\xB9\x5A\x67"
			  "\x00\x5A\x67\x00\x2D\xCE\xEB\x2D"
			  "\xCE\xEB\xB4\x51\x72\xB4\x51\x72",
		.klen	= 24,
		.input	= "\x4E\x9A\x40\x3D\x61\x7D\x17\xFA"
			  "\x16\x86\x88\x0B\xD8\xAE\xF8\xE4"
			  "\x81\x01\x04\x00\x76\xFA\xED\xD3"
			  "\x44\x7E\x21\x9D\xF0\xFB\x2B\x64"
			  "\xCA\x4E\x90\xE0\xC0\x63\x28\x92"
			  "\xF3\x1F\xA4\x53\x2C\x77\xCC\x77"
			  "\x69\x56\xD0\x19\xAD\x00\x2D\x97"
			  "\xBC\xDE\x49\x6A\x82\xBC\x16\xE2"
			  "\x2F\x3E\x72\xEE\xD1\xCE\xFC\x1B"
			  "\xEA\x32\x56\xE4\x0B\xAF\x27\x36"
			  "\xAF\x08\xB9\x61\xB7\x48\x23\x27"
			  "\xEE\x4D\xC8\x79\x56\x06\xEB\xC7"
			  "\x5B\xCA\x0A\xC6\x5E\x5C\xCB\xB6"
			  "\x9D\xDA\x04\x59\xE2\x09\x48\x7E"
			  "\x6B\x37\xC6\xFE\x92\xA9\x1E\x6E"
			  "\x0D\x19\xFA\x33\x0F\xEE\x36\x68"
			  "\x11\xBB\xF9\x5A\x73\xAB\x3A\xEA"
			  "\xAC\x28\xD8\xD5\x27\xE8\x6B\x16"
			  "\x45\x86\x50\x01\x70\x35\x99\x92"
			  "\xDF\x0C\x07\x88\x8B\x7F\x9E\x4B"
			  "\xD2\x04\x84\x90\xC4\x27\xDF\x0A"
			  "\x49\xA8\xA7\x1A\x6D\x78\x16\xCA"
			  "\xB3\x18\x5C\xC3\x93\x63\x5A\x68"
			  "\x77\x02\xBA\xED\x62\x71\xB1\xD9"
			  "\x5E\xE5\x6F\x1A\xCC\x1D\xBE\x2E"
			  "\x11\xF3\xA6\x97\xCA\x8E\xBF\xB4"
			  "\x56\xA1\x36\x6B\xB1\x0A\x3E\x70"
			  "\xEA\xD7\xCD\x72\x7B\x79\xC8\xAD"
			  "\x6B\xFE\xFB\xBA\x64\xAE\x19\xC1"
			  "\x82\xCF\x8A\xA1\x50\x17\x7F\xB2"
			  "\x6F\x7B\x0F\x52\xC5\x3E\x4A\x52"
			  "\x3F\xD9\x3F\x01\xA6\x41\x1A\xB3"
			  "\xB3\x7A\x0E\x8E\x75\xB2\xB1\x5F"
			  "\xDB\xEA\x84\x13\x26\x6C\x85\x4E"
			  "\xAE\x6B\xDC\xE7\xE7\xAD\xB0\x06"
			  "\x5C\xBA\x92\xD0\x30\xBB\x8D\xD2"
			  "\xAE\x4C\x70\x85\xA0\x07\xE3\x2C"
			  "\xD1\x27\x9C\xCF\xDB\x13\xB7\xE5"
			  "\xF9\x6A\x02\xD0\x39\x9D\xB6\xE7"
			  "\xD1\x17\x25\x08\xF9\xA9\xA6\x67"
			  "\x38\x80\xD1\x22\xAB\x1A\xD7\x26"
			  "\xAD\xCA\x19\x1B\xFA\x18\xA7\x57"
			  "\x31\xEC\xC9\xED\xDB\x79\xC0\x48"
			  "\xAC\x31\x9F\x03\x8B\x62\x5B\x7E"
			  "\x0E\xA6\xD0\x64\xEE\xEA\x00\xFC"
			  "\x58\xC8\xDE\x51\x4E\x17\x15\x11"
			  "\x66\x58\xB6\x90\xDC\xDF\xA1\x49"
			  "\xCA\x79\xE9\x31\x31\x42\xDC\x56"
			  "\x0B\xCD\xB6\x0D\xC7\x64\xF7\x19"
			  "\xD9\x42\x05\x7F\xBC\x2F\xFC\x90"
			  "\xAE\x29\x86\xAA\x43\x7A\x4F\x6B"
			  "\xCE\xEA\xBC\x31\x8D\x65\x9D\x46"
			  "\xEA\x77\xB4\xF9\x58\xEA\x5D\x84"
			  "\xE4\xDC\x14\xBB\xBD\x15\x0E\xDA"
			  "\xD8\xE4\xA4\x5D\x61\xF9\x58\x0F"
			  "\xE4\x82\x77\xCE\x87\xC0\x09\xF0"
			  "\xD6\x10\x9E\x34\xE1\x0C\x67\x55"
			  "\x7B\x6D\xD5\x51\x4B\x00\xEE\xBA"
			  "\xF2\x7B\xBE\x75\x07\x42\x9D\x99"
			  "\x12\xE1\x71\x4A\xF9\x2A\xF5\xF6"
			  "\x93\x03\xD7\x51\x09\xFA\xBE\x68"
			  "\xD8\x45\xFF\x33\xBA\xBB\x2B\x63",
		.ilen	= 496,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec des3_ede_cbc_enc_tv_template[] = {
	{  
		.key	= "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.input	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.ilen	= 128,
		.result	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19",
		.rlen	= 128,
	}, {  
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.ilen	= 496,
		.result	= "\xF8\xF6\xB5\x60\x5C\x5A\x75\x84"
			  "\x87\x81\x53\xBA\xC9\x6F\xEC\xD5"
			  "\x1E\x68\x8E\x85\x12\x86\x1D\x38"
			  "\x1C\x91\x40\xCC\x69\x6A\xD5\x35"
			  "\x0D\x7C\xB5\x07\x7C\x7B\x2A\xAF"
			  "\x32\xBC\xA1\xB3\x84\x31\x1B\x3C"
			  "\x0A\x2B\xFA\xD3\x9F\xB0\x8C\x37"
			  "\x8F\x9D\xA7\x6D\x6C\xFA\xD7\x90"
			  "\xE3\x69\x54\xED\x3A\xC4\xF1\x6B"
			  "\xB1\xCC\xFB\x7D\xD8\x8E\x17\x0B"
			  "\x9C\xF6\x4C\xD6\xFF\x03\x4E\xD9"
			  "\xE6\xA5\xAD\x25\xE6\x17\x69\x63"
			  "\x11\x35\x61\x94\x88\x7B\x1C\x48"
			  "\xF1\x24\x20\x29\x6B\x93\x1A\x8E"
			  "\x43\x03\x89\xD8\xB1\xDA\x47\x7B"
			  "\x79\x3A\x83\x76\xDA\xAE\xC6\xBB"
			  "\x22\xF8\xE8\x3D\x9A\x65\x54\xD8"
			  "\x4C\xE9\xE7\xE4\x63\x2F\x5C\x73"
			  "\x5A\xC3\xAE\x46\xA8\xCD\x57\xE6"
			  "\x67\x88\xA5\x20\x6F\x5F\x97\xC7"
			  "\xCC\x15\xA2\x0A\x93\xEA\x33\xE7"
			  "\x03\x5F\xEC\x64\x30\x6F\xEE\xD7"
			  "\x7E\xDF\xD6\xE9\x6F\x3F\xD6\x1E"
			  "\xBE\x67\x6C\x5B\x97\xA0\x09\xE6"
			  "\xEE\xFE\x55\xA3\x29\x65\xE0\x12"
			  "\xA1\x6A\x8A\x6F\xF2\xE6\xF1\x96"
			  "\x87\xFB\x9C\x05\xDD\x80\xEC\xFF"
			  "\xC5\xED\x50\xFE\xFC\x91\xCD\xCE"
			  "\x25\x2C\x5F\xD9\xAD\x95\x7D\x99"
			  "\xF0\x05\xC4\x71\x46\x5F\xF9\x0D"
			  "\xD2\x63\xDF\x9B\x96\x2E\x2B\xA6"
			  "\x2B\x1C\xD5\xFB\x96\x24\x60\x60"
			  "\x54\x40\xB8\x62\xA4\xF8\x46\x95"
			  "\x73\x28\xA3\xA6\x16\x2B\x17\xE7"
			  "\x7A\xF8\x62\x54\x3B\x64\x69\xE1"
			  "\x71\x34\x29\x5B\x4E\x05\x9B\xFA"
			  "\x5E\xF1\x96\xB7\xCE\x16\x9B\x59"
			  "\xF1\x1A\x4C\x51\x26\xFD\x79\xE2"
			  "\x3B\x8E\x71\x69\x6A\x91\xB6\x65"
			  "\x32\x09\xB8\xE4\x09\x1F\xEA\x39"
			  "\xCE\x20\x65\x9F\xD6\xD1\xC7\xF0"
			  "\x73\x50\x08\x56\x20\x9B\x94\x23"
			  "\x14\x39\xB7\x2B\xB1\x2D\x6D\x6F"
			  "\x41\x5B\xCC\xE2\x18\xAE\x62\x89"
			  "\x78\x8E\x67\x23\xD0\xFB\x2B\xE5"
			  "\x25\xC9\x48\x97\xB5\xD3\x17\xD5"
			  "\x6A\x9F\xA7\x48\x0C\x2B\x73\x3B"
			  "\x57\x08\xAE\x91\xF2\xB7\x57\x89"
			  "\xF4\xD0\xB0\x07\xB0\x42\x6C\xAF"
			  "\x98\x1A\xE7\xD1\xAC\x1E\xB5\x02"
			  "\xD4\x56\x42\x79\x79\x7F\x2A\x77"
			  "\x25\xE9\x7D\xC1\x88\x19\x2B\x49"
			  "\x6F\x46\x59\xAB\x56\x1F\x61\xE0"
			  "\x0C\x24\x9C\xC9\x5B\x63\xA9\x12"
			  "\xCF\x88\x96\xB6\xA8\x24\xC6\xA8"
			  "\x21\x85\x1A\x62\x7E\x34\xBB\xEB"
			  "\xBD\x02\x2A\xC7\xD8\x89\x80\xC5"
			  "\xB1\xBB\x60\xA5\x22\xFC\x6F\x38"
			  "\x02\x80\xA3\x28\x22\x75\xE1\xE9"
			  "\x90\xE9\xFA\x4B\x00\x10\xAC\x58"
			  "\x83\x70\xFF\x86\xE6\xAA\x0F\x1F"
			  "\x95\x63\x73\xA2\x44\xAC\xF8\xA5",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec des3_ede_cbc_dec_tv_template[] = {
	{  
		.key	= "\xE9\xC0\xFF\x2E\x76\x0B\x64\x24"
			  "\x44\x4D\x99\x5A\x12\xD6\x40\xC0"
			  "\xEA\xC2\x84\xE8\x14\x95\xDB\xE8",
		.klen	= 24,
		.iv	= "\x7D\x33\x88\x93\x0F\x93\xB2\x42",
		.input	= "\x0e\x2d\xb6\x97\x3c\x56\x33\xf4"
			  "\x67\x17\x21\xc7\x6e\x8a\xd5\x49"
			  "\x74\xb3\x49\x05\xc5\x1c\xd0\xed"
			  "\x12\x56\x5c\x53\x96\xb6\x00\x7d"
			  "\x90\x48\xfc\xf5\x8d\x29\x39\xcc"
			  "\x8a\xd5\x35\x18\x36\x23\x4e\xd7"
			  "\x76\xd1\xda\x0c\x94\x67\xbb\x04"
			  "\x8b\xf2\x03\x6c\xa8\xcf\xb6\xea"
			  "\x22\x64\x47\xaa\x8f\x75\x13\xbf"
			  "\x9f\xc2\xc3\xf0\xc9\x56\xc5\x7a"
			  "\x71\x63\x2e\x89\x7b\x1e\x12\xca"
			  "\xe2\x5f\xaf\xd8\xa4\xf8\xc9\x7a"
			  "\xd6\xf9\x21\x31\x62\x44\x45\xa6"
			  "\xd6\xbc\x5a\xd3\x2d\x54\x43\xcc"
			  "\x9d\xde\xa5\x70\xe9\x42\x45\x8a"
			  "\x6b\xfa\xb1\x91\x13\xb0\xd9\x19",
		.ilen	= 128,
		.result	= "\x6f\x54\x20\x6f\x61\x4d\x79\x6e"
			  "\x53\x20\x63\x65\x65\x72\x73\x74"
			  "\x54\x20\x6f\x6f\x4d\x20\x6e\x61"
			  "\x20\x79\x65\x53\x72\x63\x74\x65"
			  "\x20\x73\x6f\x54\x20\x6f\x61\x4d"
			  "\x79\x6e\x53\x20\x63\x65\x65\x72"
			  "\x73\x74\x54\x20\x6f\x6f\x4d\x20"
			  "\x6e\x61\x20\x79\x65\x53\x72\x63"
			  "\x74\x65\x20\x73\x6f\x54\x20\x6f"
			  "\x61\x4d\x79\x6e\x53\x20\x63\x65"
			  "\x65\x72\x73\x74\x54\x20\x6f\x6f"
			  "\x4d\x20\x6e\x61\x20\x79\x65\x53"
			  "\x72\x63\x74\x65\x20\x73\x6f\x54"
			  "\x20\x6f\x61\x4d\x79\x6e\x53\x20"
			  "\x63\x65\x65\x72\x73\x74\x54\x20"
			  "\x6f\x6f\x4d\x20\x6e\x61\x0a\x79",
		.rlen	= 128,
	}, {  
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\xF8\xF6\xB5\x60\x5C\x5A\x75\x84"
			  "\x87\x81\x53\xBA\xC9\x6F\xEC\xD5"
			  "\x1E\x68\x8E\x85\x12\x86\x1D\x38"
			  "\x1C\x91\x40\xCC\x69\x6A\xD5\x35"
			  "\x0D\x7C\xB5\x07\x7C\x7B\x2A\xAF"
			  "\x32\xBC\xA1\xB3\x84\x31\x1B\x3C"
			  "\x0A\x2B\xFA\xD3\x9F\xB0\x8C\x37"
			  "\x8F\x9D\xA7\x6D\x6C\xFA\xD7\x90"
			  "\xE3\x69\x54\xED\x3A\xC4\xF1\x6B"
			  "\xB1\xCC\xFB\x7D\xD8\x8E\x17\x0B"
			  "\x9C\xF6\x4C\xD6\xFF\x03\x4E\xD9"
			  "\xE6\xA5\xAD\x25\xE6\x17\x69\x63"
			  "\x11\x35\x61\x94\x88\x7B\x1C\x48"
			  "\xF1\x24\x20\x29\x6B\x93\x1A\x8E"
			  "\x43\x03\x89\xD8\xB1\xDA\x47\x7B"
			  "\x79\x3A\x83\x76\xDA\xAE\xC6\xBB"
			  "\x22\xF8\xE8\x3D\x9A\x65\x54\xD8"
			  "\x4C\xE9\xE7\xE4\x63\x2F\x5C\x73"
			  "\x5A\xC3\xAE\x46\xA8\xCD\x57\xE6"
			  "\x67\x88\xA5\x20\x6F\x5F\x97\xC7"
			  "\xCC\x15\xA2\x0A\x93\xEA\x33\xE7"
			  "\x03\x5F\xEC\x64\x30\x6F\xEE\xD7"
			  "\x7E\xDF\xD6\xE9\x6F\x3F\xD6\x1E"
			  "\xBE\x67\x6C\x5B\x97\xA0\x09\xE6"
			  "\xEE\xFE\x55\xA3\x29\x65\xE0\x12"
			  "\xA1\x6A\x8A\x6F\xF2\xE6\xF1\x96"
			  "\x87\xFB\x9C\x05\xDD\x80\xEC\xFF"
			  "\xC5\xED\x50\xFE\xFC\x91\xCD\xCE"
			  "\x25\x2C\x5F\xD9\xAD\x95\x7D\x99"
			  "\xF0\x05\xC4\x71\x46\x5F\xF9\x0D"
			  "\xD2\x63\xDF\x9B\x96\x2E\x2B\xA6"
			  "\x2B\x1C\xD5\xFB\x96\x24\x60\x60"
			  "\x54\x40\xB8\x62\xA4\xF8\x46\x95"
			  "\x73\x28\xA3\xA6\x16\x2B\x17\xE7"
			  "\x7A\xF8\x62\x54\x3B\x64\x69\xE1"
			  "\x71\x34\x29\x5B\x4E\x05\x9B\xFA"
			  "\x5E\xF1\x96\xB7\xCE\x16\x9B\x59"
			  "\xF1\x1A\x4C\x51\x26\xFD\x79\xE2"
			  "\x3B\x8E\x71\x69\x6A\x91\xB6\x65"
			  "\x32\x09\xB8\xE4\x09\x1F\xEA\x39"
			  "\xCE\x20\x65\x9F\xD6\xD1\xC7\xF0"
			  "\x73\x50\x08\x56\x20\x9B\x94\x23"
			  "\x14\x39\xB7\x2B\xB1\x2D\x6D\x6F"
			  "\x41\x5B\xCC\xE2\x18\xAE\x62\x89"
			  "\x78\x8E\x67\x23\xD0\xFB\x2B\xE5"
			  "\x25\xC9\x48\x97\xB5\xD3\x17\xD5"
			  "\x6A\x9F\xA7\x48\x0C\x2B\x73\x3B"
			  "\x57\x08\xAE\x91\xF2\xB7\x57\x89"
			  "\xF4\xD0\xB0\x07\xB0\x42\x6C\xAF"
			  "\x98\x1A\xE7\xD1\xAC\x1E\xB5\x02"
			  "\xD4\x56\x42\x79\x79\x7F\x2A\x77"
			  "\x25\xE9\x7D\xC1\x88\x19\x2B\x49"
			  "\x6F\x46\x59\xAB\x56\x1F\x61\xE0"
			  "\x0C\x24\x9C\xC9\x5B\x63\xA9\x12"
			  "\xCF\x88\x96\xB6\xA8\x24\xC6\xA8"
			  "\x21\x85\x1A\x62\x7E\x34\xBB\xEB"
			  "\xBD\x02\x2A\xC7\xD8\x89\x80\xC5"
			  "\xB1\xBB\x60\xA5\x22\xFC\x6F\x38"
			  "\x02\x80\xA3\x28\x22\x75\xE1\xE9"
			  "\x90\xE9\xFA\x4B\x00\x10\xAC\x58"
			  "\x83\x70\xFF\x86\xE6\xAA\x0F\x1F"
			  "\x95\x63\x73\xA2\x44\xAC\xF8\xA5",
		.ilen	= 496,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec des3_ede_ctr_enc_tv_template[] = {
	{  
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.ilen	= 496,
		.result	= "\x07\xC2\x08\x20\x72\x1F\x49\xEF"
			  "\x19\xCD\x6F\x32\x53\x05\x22\x15"
			  "\xA2\x85\x2B\xDB\x85\xD2\xD8\xB9"
			  "\xDD\x0D\x1B\x45\xCB\x69\x11\xD4"
			  "\xEA\xBE\xB2\x45\x5D\x0C\xAE\xBE"
			  "\xA0\xC1\x27\xAC\x65\x9F\x53\x7E"
			  "\xAF\xC2\x1B\xB5\xB8\x6D\x36\x0C"
			  "\x25\xC0\xF8\x6D\x0B\x29\x01\xDA"
			  "\x13\x78\xDC\x89\x12\x12\x43\xFA"
			  "\xF6\x12\xEF\x8D\x87\x62\x78\x83"
			  "\xE2\xBE\x41\x20\x4C\x6D\x35\x1B"
			  "\xD1\x0C\x30\xCF\xE2\xDE\x2B\x03"
			  "\xBF\x45\x73\xD4\xE5\x59\x95\xD1"
			  "\xB3\x9B\x27\x62\x97\xBD\xDE\x7F"
			  "\xA4\xD2\x39\x80\xAA\x50\x23\xF0"
			  "\x74\x88\x3D\xA8\x6A\x18\x79\x3B"
			  "\xC4\x96\x6C\x8D\x22\x40\x92\x6E"
			  "\xD6\xAD\x2A\x1F\xDE\x63\xC0\xE7"
			  "\x07\xF7\x2D\xF7\xB5\xF3\xF0\xCC"
			  "\x01\x7C\x2A\x9B\xC2\x10\xCA\xAA"
			  "\xFD\x2B\x3F\xC5\xF3\xF6\xFC\x9B"
			  "\x45\xDB\x53\xE4\x5B\xF3\xC9\x7B"
			  "\x8E\x52\xFF\xC8\x02\xB8\xAC\x9D"
			  "\xA1\x00\x39\xDA\x3D\x2D\x0E\x01"
			  "\x09\x7D\x8D\x5E\xBE\x53\xB9\xB0"
			  "\x8E\xE7\xE2\x96\x6A\xB2\x78\xEA"
			  "\xDE\x23\x8B\xA5\xFA\x5C\xE3\xDA"
			  "\xBF\x8E\x31\x6A\x55\xD1\x6A\xB2"
			  "\xB5\x46\x6F\xA5\xF0\xEE\xBA\x1F"
			  "\x9F\x98\xB0\x66\x4F\xD0\x3F\xA9"
			  "\xDF\x5F\x58\xC4\xF4\xFF\x75\x5C"
			  "\x40\x3A\x09\x7E\x6E\x1C\x97\xD4"
			  "\xCC\xE7\xE7\x71\xCF\x0B\x15\x08"
			  "\x71\xFA\x07\x97\xCD\xE6\xCA\x1D"
			  "\x14\x28\x0C\xCF\x99\x13\x7A\xF1"
			  "\xEB\xFA\xFA\x92\x07\xDE\x1D\xA1"
			  "\xD3\x36\x69\xFE\x51\x4D\x9F\x2E"
			  "\x83\x37\x4F\x1F\x48\x30\xED\x04"
			  "\x4D\xA4\xEF\x3A\xCA\x76\xF4\x1C"
			  "\x41\x8F\x63\x37\x78\x2F\x86\xA6"
			  "\xEF\x41\x7E\xD2\xAF\x88\xAB\x67"
			  "\x52\x71\xC3\x8E\xF8\x26\x93\x72"
			  "\xAA\xD6\x0E\xE7\x0B\x46\xB1\x3A"
			  "\xB4\x08\xA9\xA8\xA0\xCF\x20\x0C"
			  "\x52\xBC\x8B\x05\x56\xB2\xBC\x31"
			  "\x9B\x74\xB9\x29\x29\x96\x9A\x50"
			  "\xDC\x45\xDC\x1A\xEB\x0C\x64\xD4"
			  "\xD3\x05\x7E\x59\x55\xC3\xF4\x90"
			  "\xC2\xAB\xF8\x9B\x8A\xDA\xCE\xA1"
			  "\xC3\xF4\xAD\x77\xDD\x44\xC8\xAC"
			  "\xA3\xF1\xC9\xD2\x19\x5C\xB0\xCA"
			  "\xA2\x34\xC1\xF7\x6C\xFD\xAC\x65"
			  "\x32\xDC\x48\xC4\xF2\x00\x6B\x77"
			  "\xF1\x7D\x76\xAC\xC0\x31\x63\x2A"
			  "\xA5\x3A\x62\xC8\x91\xB1\x03\x65"
			  "\xCB\x43\xD1\x06\xDF\xC3\x67\xBC"
			  "\xDC\xE0\xCD\x35\xCE\x49\x65\xA0"
			  "\x52\x7B\xA7\x0D\x07\xA9\x1B\xB0"
			  "\x40\x77\x72\xC2\xEA\x0E\x3A\x78"
			  "\x46\xB9\x91\xB6\xE7\x3D\x51\x42"
			  "\xFD\x51\xB0\xC6\x2C\x63\x13\x78"
			  "\x5C\xEE\xFC\xCF\xC4\x70\x00\x34",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	}, {  
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47"
			  "\x2E\xB1\x18",
		.ilen	= 499,
		.result	= "\x23\xFF\x5C\x99\x75\xBB\x1F\xD4"
			  "\xBC\x27\x9D\x36\x60\xA9\xC9\xF7"
			  "\x94\x9D\x1B\xFF\x8E\x95\x57\x89"
			  "\x8C\x2E\x33\x70\x43\x61\xE6\xD2"
			  "\x82\x33\x63\xB6\xC4\x34\x5E\xF8"
			  "\x96\x07\xA7\xD2\x3B\x8E\xC9\xAA"
			  "\x7C\xA0\x55\x89\x2E\xE1\x85\x25"
			  "\x14\x04\xDA\x6B\xE0\xEE\x56\xCF"
			  "\x08\x2E\x69\xD4\x54\xDE\x22\x84"
			  "\x69\xA6\xA7\xD3\x3A\x9A\xE8\x05"
			  "\x63\xDB\xBF\x46\x3A\x26\x2E\x0F"
			  "\x58\x5C\x46\xEA\x07\x40\xDA\xE1"
			  "\x14\x1D\xCD\x4F\x06\xC0\xCA\x54"
			  "\x1E\xC9\x45\x85\x67\x7C\xC2\xB5"
			  "\x97\x5D\x61\x78\x2E\x46\xEC\x6A"
			  "\x53\xF4\xD0\xAE\xFA\xB4\x86\x29"
			  "\x9F\x17\x33\x24\xD8\xB9\xB2\x05"
			  "\x93\x88\xEA\xF7\xA0\x70\x69\x49"
			  "\x88\x6B\x73\x40\x41\x8D\xD9\xD9"
			  "\x7E\x78\xE9\xBE\x6C\x14\x22\x7A"
			  "\x66\xE1\xDA\xED\x10\xFF\x69\x1D"
			  "\xB9\xAA\xF2\x56\x72\x1B\x23\xE2"
			  "\x45\x54\x8B\xA3\x70\x23\xB4\x5E"
			  "\x8E\x96\xC9\x05\x00\xB3\xB6\xC2"
			  "\x2A\x02\x43\x7A\x62\xD5\xC8\xD2"
			  "\xC2\xD0\xE4\x78\xA1\x7B\x3E\xE8"
			  "\x9F\x7F\x7D\x40\x54\x30\x3B\xC0"
			  "\xA5\x54\xFD\xCA\x25\xEC\x44\x3E"
			  "\x1A\x54\x7F\x88\xD0\xE1\xFE\x71"
			  "\xCE\x05\x49\x89\xBA\xD6\x72\xE7"
			  "\xD6\x5D\x3F\xA2\xD9\xAB\xC5\x02"
			  "\xD6\x43\x22\xAF\xA2\xE4\x80\x85"
			  "\xD7\x87\xB9\xEA\x43\xDB\xC8\xEF"
			  "\x5C\x82\x2E\x98\x0D\x30\x41\x6B"
			  "\x08\x48\x8D\xF0\xF8\x60\xD7\x9D"
			  "\xE9\xDE\x40\xAD\x0D\xAD\x0D\x58"
			  "\x2A\x98\x35\xFE\xF7\xDD\x4B\x40"
			  "\xDE\xB0\x05\xD9\x7B\x09\x4D\xBC"
			  "\x42\xC0\xF1\x15\x0B\xFA\x26\x6B"
			  "\xC6\x12\x13\x4F\xCB\x35\xBA\x35"
			  "\xDD\x7A\x36\x9C\x12\x57\x55\x83"
			  "\x78\x58\x09\xD0\xB0\xCF\x7C\x5C"
			  "\x38\xCF\xBD\x79\x5B\x13\x4D\x97"
			  "\xC1\x85\x6F\x97\xC9\xE8\xC2\xA4"
			  "\x98\xE2\xBD\x77\x6B\x53\x39\x1A"
			  "\x28\x10\xE7\xE0\xE7\xDE\x9D\x69"
			  "\x78\x6F\x8E\xD2\xD9\x5D\xD2\x15"
			  "\x9E\xB5\x4D\x8C\xC0\x78\x22\x2F"
			  "\x17\x11\x2E\x99\xD7\xE3\xA4\x4F"
			  "\x65\xA5\x6B\x03\x2C\x35\x6F\xDA"
			  "\x8A\x19\x08\xE1\x08\x48\x59\x51"
			  "\x53\x4B\xD1\xDF\xDA\x14\x50\x5F"
			  "\xDF\xB5\x8C\xDF\xC6\xFD\x85\xFA"
			  "\xD4\xF9\x64\x45\x65\x0D\x7D\xF4"
			  "\xC8\xCD\x3F\x32\xAF\xDD\x30\xED"
			  "\x7B\xAA\xAC\xF0\xDA\x7F\xDF\x75"
			  "\x1C\xA4\xF1\xCB\x5E\x4F\x0B\xB4"
			  "\x97\x73\x28\xDE\xCF\xAF\x82\xBD"
			  "\xC4\xBA\xB4\x9C\x0D\x16\x77\x42"
			  "\x42\x39\x7C\x53\xA4\xD4\xDD\x40"
			  "\x5C\x60\x1F\x6E\xA7\xE2\xDC\xE7"
			  "\x32\x0F\x05\x2F\xF2\x4C\x95\x3B"
			  "\xF2\x79\xD9",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec des3_ede_ctr_dec_tv_template[] = {
	{  
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x07\xC2\x08\x20\x72\x1F\x49\xEF"
			  "\x19\xCD\x6F\x32\x53\x05\x22\x15"
			  "\xA2\x85\x2B\xDB\x85\xD2\xD8\xB9"
			  "\xDD\x0D\x1B\x45\xCB\x69\x11\xD4"
			  "\xEA\xBE\xB2\x45\x5D\x0C\xAE\xBE"
			  "\xA0\xC1\x27\xAC\x65\x9F\x53\x7E"
			  "\xAF\xC2\x1B\xB5\xB8\x6D\x36\x0C"
			  "\x25\xC0\xF8\x6D\x0B\x29\x01\xDA"
			  "\x13\x78\xDC\x89\x12\x12\x43\xFA"
			  "\xF6\x12\xEF\x8D\x87\x62\x78\x83"
			  "\xE2\xBE\x41\x20\x4C\x6D\x35\x1B"
			  "\xD1\x0C\x30\xCF\xE2\xDE\x2B\x03"
			  "\xBF\x45\x73\xD4\xE5\x59\x95\xD1"
			  "\xB3\x9B\x27\x62\x97\xBD\xDE\x7F"
			  "\xA4\xD2\x39\x80\xAA\x50\x23\xF0"
			  "\x74\x88\x3D\xA8\x6A\x18\x79\x3B"
			  "\xC4\x96\x6C\x8D\x22\x40\x92\x6E"
			  "\xD6\xAD\x2A\x1F\xDE\x63\xC0\xE7"
			  "\x07\xF7\x2D\xF7\xB5\xF3\xF0\xCC"
			  "\x01\x7C\x2A\x9B\xC2\x10\xCA\xAA"
			  "\xFD\x2B\x3F\xC5\xF3\xF6\xFC\x9B"
			  "\x45\xDB\x53\xE4\x5B\xF3\xC9\x7B"
			  "\x8E\x52\xFF\xC8\x02\xB8\xAC\x9D"
			  "\xA1\x00\x39\xDA\x3D\x2D\x0E\x01"
			  "\x09\x7D\x8D\x5E\xBE\x53\xB9\xB0"
			  "\x8E\xE7\xE2\x96\x6A\xB2\x78\xEA"
			  "\xDE\x23\x8B\xA5\xFA\x5C\xE3\xDA"
			  "\xBF\x8E\x31\x6A\x55\xD1\x6A\xB2"
			  "\xB5\x46\x6F\xA5\xF0\xEE\xBA\x1F"
			  "\x9F\x98\xB0\x66\x4F\xD0\x3F\xA9"
			  "\xDF\x5F\x58\xC4\xF4\xFF\x75\x5C"
			  "\x40\x3A\x09\x7E\x6E\x1C\x97\xD4"
			  "\xCC\xE7\xE7\x71\xCF\x0B\x15\x08"
			  "\x71\xFA\x07\x97\xCD\xE6\xCA\x1D"
			  "\x14\x28\x0C\xCF\x99\x13\x7A\xF1"
			  "\xEB\xFA\xFA\x92\x07\xDE\x1D\xA1"
			  "\xD3\x36\x69\xFE\x51\x4D\x9F\x2E"
			  "\x83\x37\x4F\x1F\x48\x30\xED\x04"
			  "\x4D\xA4\xEF\x3A\xCA\x76\xF4\x1C"
			  "\x41\x8F\x63\x37\x78\x2F\x86\xA6"
			  "\xEF\x41\x7E\xD2\xAF\x88\xAB\x67"
			  "\x52\x71\xC3\x8E\xF8\x26\x93\x72"
			  "\xAA\xD6\x0E\xE7\x0B\x46\xB1\x3A"
			  "\xB4\x08\xA9\xA8\xA0\xCF\x20\x0C"
			  "\x52\xBC\x8B\x05\x56\xB2\xBC\x31"
			  "\x9B\x74\xB9\x29\x29\x96\x9A\x50"
			  "\xDC\x45\xDC\x1A\xEB\x0C\x64\xD4"
			  "\xD3\x05\x7E\x59\x55\xC3\xF4\x90"
			  "\xC2\xAB\xF8\x9B\x8A\xDA\xCE\xA1"
			  "\xC3\xF4\xAD\x77\xDD\x44\xC8\xAC"
			  "\xA3\xF1\xC9\xD2\x19\x5C\xB0\xCA"
			  "\xA2\x34\xC1\xF7\x6C\xFD\xAC\x65"
			  "\x32\xDC\x48\xC4\xF2\x00\x6B\x77"
			  "\xF1\x7D\x76\xAC\xC0\x31\x63\x2A"
			  "\xA5\x3A\x62\xC8\x91\xB1\x03\x65"
			  "\xCB\x43\xD1\x06\xDF\xC3\x67\xBC"
			  "\xDC\xE0\xCD\x35\xCE\x49\x65\xA0"
			  "\x52\x7B\xA7\x0D\x07\xA9\x1B\xB0"
			  "\x40\x77\x72\xC2\xEA\x0E\x3A\x78"
			  "\x46\xB9\x91\xB6\xE7\x3D\x51\x42"
			  "\xFD\x51\xB0\xC6\x2C\x63\x13\x78"
			  "\x5C\xEE\xFC\xCF\xC4\x70\x00\x34",
		.ilen	= 496,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	}, {  
		.key	= "\x9C\xD6\xF3\x9C\xB9\x5A\x67\x00"
			  "\x5A\x67\x00\x2D\xCE\xEB\x2D\xCE"
			  "\xEB\xB4\x51\x72\xB4\x51\x72\x1F",
		.klen	= 24,
		.iv	= "\xB2\xD7\x48\xED\x06\x44\xF9\x12"
			  "\xB7\x28\x4D\x83\x24\x59\xF2\x17",
		.input	= "\x23\xFF\x5C\x99\x75\xBB\x1F\xD4"
			  "\xBC\x27\x9D\x36\x60\xA9\xC9\xF7"
			  "\x94\x9D\x1B\xFF\x8E\x95\x57\x89"
			  "\x8C\x2E\x33\x70\x43\x61\xE6\xD2"
			  "\x82\x33\x63\xB6\xC4\x34\x5E\xF8"
			  "\x96\x07\xA7\xD2\x3B\x8E\xC9\xAA"
			  "\x7C\xA0\x55\x89\x2E\xE1\x85\x25"
			  "\x14\x04\xDA\x6B\xE0\xEE\x56\xCF"
			  "\x08\x2E\x69\xD4\x54\xDE\x22\x84"
			  "\x69\xA6\xA7\xD3\x3A\x9A\xE8\x05"
			  "\x63\xDB\xBF\x46\x3A\x26\x2E\x0F"
			  "\x58\x5C\x46\xEA\x07\x40\xDA\xE1"
			  "\x14\x1D\xCD\x4F\x06\xC0\xCA\x54"
			  "\x1E\xC9\x45\x85\x67\x7C\xC2\xB5"
			  "\x97\x5D\x61\x78\x2E\x46\xEC\x6A"
			  "\x53\xF4\xD0\xAE\xFA\xB4\x86\x29"
			  "\x9F\x17\x33\x24\xD8\xB9\xB2\x05"
			  "\x93\x88\xEA\xF7\xA0\x70\x69\x49"
			  "\x88\x6B\x73\x40\x41\x8D\xD9\xD9"
			  "\x7E\x78\xE9\xBE\x6C\x14\x22\x7A"
			  "\x66\xE1\xDA\xED\x10\xFF\x69\x1D"
			  "\xB9\xAA\xF2\x56\x72\x1B\x23\xE2"
			  "\x45\x54\x8B\xA3\x70\x23\xB4\x5E"
			  "\x8E\x96\xC9\x05\x00\xB3\xB6\xC2"
			  "\x2A\x02\x43\x7A\x62\xD5\xC8\xD2"
			  "\xC2\xD0\xE4\x78\xA1\x7B\x3E\xE8"
			  "\x9F\x7F\x7D\x40\x54\x30\x3B\xC0"
			  "\xA5\x54\xFD\xCA\x25\xEC\x44\x3E"
			  "\x1A\x54\x7F\x88\xD0\xE1\xFE\x71"
			  "\xCE\x05\x49\x89\xBA\xD6\x72\xE7"
			  "\xD6\x5D\x3F\xA2\xD9\xAB\xC5\x02"
			  "\xD6\x43\x22\xAF\xA2\xE4\x80\x85"
			  "\xD7\x87\xB9\xEA\x43\xDB\xC8\xEF"
			  "\x5C\x82\x2E\x98\x0D\x30\x41\x6B"
			  "\x08\x48\x8D\xF0\xF8\x60\xD7\x9D"
			  "\xE9\xDE\x40\xAD\x0D\xAD\x0D\x58"
			  "\x2A\x98\x35\xFE\xF7\xDD\x4B\x40"
			  "\xDE\xB0\x05\xD9\x7B\x09\x4D\xBC"
			  "\x42\xC0\xF1\x15\x0B\xFA\x26\x6B"
			  "\xC6\x12\x13\x4F\xCB\x35\xBA\x35"
			  "\xDD\x7A\x36\x9C\x12\x57\x55\x83"
			  "\x78\x58\x09\xD0\xB0\xCF\x7C\x5C"
			  "\x38\xCF\xBD\x79\x5B\x13\x4D\x97"
			  "\xC1\x85\x6F\x97\xC9\xE8\xC2\xA4"
			  "\x98\xE2\xBD\x77\x6B\x53\x39\x1A"
			  "\x28\x10\xE7\xE0\xE7\xDE\x9D\x69"
			  "\x78\x6F\x8E\xD2\xD9\x5D\xD2\x15"
			  "\x9E\xB5\x4D\x8C\xC0\x78\x22\x2F"
			  "\x17\x11\x2E\x99\xD7\xE3\xA4\x4F"
			  "\x65\xA5\x6B\x03\x2C\x35\x6F\xDA"
			  "\x8A\x19\x08\xE1\x08\x48\x59\x51"
			  "\x53\x4B\xD1\xDF\xDA\x14\x50\x5F"
			  "\xDF\xB5\x8C\xDF\xC6\xFD\x85\xFA"
			  "\xD4\xF9\x64\x45\x65\x0D\x7D\xF4"
			  "\xC8\xCD\x3F\x32\xAF\xDD\x30\xED"
			  "\x7B\xAA\xAC\xF0\xDA\x7F\xDF\x75"
			  "\x1C\xA4\xF1\xCB\x5E\x4F\x0B\xB4"
			  "\x97\x73\x28\xDE\xCF\xAF\x82\xBD"
			  "\xC4\xBA\xB4\x9C\x0D\x16\x77\x42"
			  "\x42\x39\x7C\x53\xA4\xD4\xDD\x40"
			  "\x5C\x60\x1F\x6E\xA7\xE2\xDC\xE7"
			  "\x32\x0F\x05\x2F\xF2\x4C\x95\x3B"
			  "\xF2\x79\xD9",
		.ilen	= 499,
		.result	= "\x05\xEC\x77\xFB\x42\xD5\x59\x20"
			  "\x8B\x12\x86\x69\xF0\x5B\xCF\x56"
			  "\x39\xAD\x34\x9F\x66\xEA\x7D\xC4"
			  "\x48\xD3\xBA\x0D\xB1\x18\xE3\x4A"
			  "\xFE\x41\x28\x5C\x27\x8E\x11\x85"
			  "\x6C\xF7\x5E\xC2\x55\x3C\xA0\x0B"
			  "\x92\x65\xE9\x70\xDB\x4F\xD6\xB9"
			  "\x00\xB4\x1F\xE6\x49\xFD\x44\x2F"
			  "\x53\x3A\x8D\x14\x98\x63\xCA\x5D"
			  "\xC1\xA8\x33\xA7\x0E\x91\x78\xEC"
			  "\x77\xDE\x42\xD5\xBC\x07\x8B\x12"
			  "\xE5\x4C\xF0\x5B\x22\x56\x39\x80"
			  "\x6B\x9F\x66\xC9\x50\xC4\xAF\x36"
			  "\xBA\x0D\x94\x7F\xE3\x4A\xDD\x41"
			  "\x28\xB3\x1A\x8E\x11\xF8\x43\xF7"
			  "\x5E\x21\x55\x3C\x87\x6E\x92\x65"
			  "\xCC\x57\xDB\xA2\x35\xB9\x00\xEB"
			  "\x72\xE6\x49\xD0\x44\x2F\xB6\x19"
			  "\x8D\x14\xFF\x46\xCA\x5D\x24\xA8"
			  "\x33\x9A\x6D\x91\x78\xC3\x77\xDE"
			  "\xA1\x08\xBC\x07\xEE\x71\xE5\x4C"
			  "\xD7\x5B\x22\xB5\x1C\x80\x6B\xF2"
			  "\x45\xC9\x50\x3B\xAF\x36\x99\x60"
			  "\x94\x7F\xC6\x4A\xDD\xA4\x0F\xB3"
			  "\x1A\xED\x74\xF8\x43\x2A\x5E\x21"
			  "\x88\x13\x87\x6E\xF1\x58\xCC\x57"
			  "\x3E\xA2\x35\x9C\x67\xEB\x72\xC5"
			  "\x49\xD0\xBB\x02\xB6\x19\xE0\x4B"
			  "\xFF\x46\x29\x5D\x24\x8F\x16\x9A"
			  "\x6D\xF4\x5F\xC3\xAA\x3D\xA1\x08"
			  "\x93\x7A\xEE\x71\xD8\x4C\xD7\xBE"
			  "\x01\xB5\x1C\xE7\x4E\xF2\x45\x2C"
			  "\x50\x3B\x82\x15\x99\x60\xCB\x52"
			  "\xC6\xA9\x30\xA4\x0F\x96\x79\xED"
			  "\x74\xDF\x43\x2A\xBD\x04\x88\x13"
			  "\xFA\x4D\xF1\x58\x23\x57\x3E\x81"
			  "\x68\x9C\x67\xCE\x51\xC5\xAC\x37"
			  "\xBB\x02\x95\x7C\xE0\x4B\xD2\x46"
			  "\x29\xB0\x1B\x8F\x16\xF9\x40\xF4"
			  "\x5F\x26\xAA\x3D\x84\x6F\x93\x7A"
			  "\xCD\x54\xD8\xA3\x0A\xBE\x01\xE8"
			  "\x73\xE7\x4E\xD1\x45\x2C\xB7\x1E"
			  "\x82\x15\xFC\x47\xCB\x52\x25\xA9"
			  "\x30\x9B\x62\x96\x79\xC0\x74\xDF"
			  "\xA6\x09\xBD\x04\xEF\x76\xFA\x4D"
			  "\xD4\x58\x23\x8A\x1D\x81\x68\xF3"
			  "\x5A\xCE\x51\x38\xAC\x37\x9E\x61"
			  "\x95\x7C\xC7\x4B\xD2\xA5\x0C\xB0"
			  "\x1B\xE2\x75\xF9\x40\x2B\x5F\x26"
			  "\x89\x10\x84\x6F\xF6\x59\xCD\x54"
			  "\x3F\xA3\x0A\x9D\x64\xE8\x73\xDA"
			  "\x4E\xD1\xB8\x03\xB7\x1E\xE1\x48"
			  "\xFC\x47\x2E\x52\x25\x8C\x17\x9B"
			  "\x62\xF5\x5C\xC0\xAB\x32\xA6\x09"
			  "\x90\x7B\xEF\x76\xD9\x4D\xD4\xBF"
			  "\x06\x8A\x1D\xE4\x4F\xF3\x5A\x2D"
			  "\x51\x38\x83\x6A\x9E\x61\xC8\x53"
			  "\xC7\xAE\x31\xA5\x0C\x97\x7E\xE2"
			  "\x75\xDC\x40\x2B\xB2\x05\x89\x10"
			  "\xFB\x42\xF6\x59\x20\x54\x3F\x86"
			  "\x69\x9D\x64\xCF\x56\xDA\xAD\x34"
			  "\xB8\x03\xEA\x7D\xE1\x48\xD3\x47"
			  "\x2E\xB1\x18",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

#define BF_ENC_TEST_VECTORS	7
#define BF_DEC_TEST_VECTORS	7
#define BF_CBC_ENC_TEST_VECTORS	2
#define BF_CBC_DEC_TEST_VECTORS	2
#define BF_CTR_ENC_TEST_VECTORS	2
#define BF_CTR_DEC_TEST_VECTORS	2

static struct cipher_testvec bf_enc_tv_template[] = {
	{  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\x4e\xf9\x97\x45\x61\x98\xdd\x78",
		.rlen	= 8,
	}, {
		.key	= "\x1f\x1f\x1f\x1f\x0e\x0e\x0e\x0e",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.ilen	= 8,
		.result	= "\xa7\x90\x79\x51\x08\xea\x3c\xae",
		.rlen	= 8,
	}, {
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.klen	= 8,
		.input	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 8,
		.result	= "\xe8\x7a\x24\x4e\x2c\xc8\x5e\x82",
		.rlen	= 8,
	}, {  
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87"
			  "\x78\x69\x5a\x4b\x3c\x2d\x1e\x0f",
		.klen	= 16,
		.input	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 8,
		.result	= "\x93\x14\x28\x87\xee\x3b\xe1\x5c",
		.rlen	= 8,
	}, {
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87"
			  "\x78\x69\x5a\x4b\x3c\x2d\x1e\x0f"
			  "\x00\x11\x22\x33\x44",
		.klen	= 21,
		.input	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 8,
		.result	= "\xe6\xf5\x1e\xd7\x9b\x9d\xb2\x1f",
		.rlen	= 8,
	}, {  
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87"
			  "\x78\x69\x5a\x4b\x3c\x2d\x1e\x0f"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x04\x68\x91\x04\xc2\xfd\x3b\x2f"
			  "\x58\x40\x23\x64\x1a\xba\x61\x76"
			  "\x1f\x1f\x1f\x1f\x0e\x0e\x0e\x0e"
			  "\xff\xff\xff\xff\xff\xff\xff\xff",
		.klen	= 56,
		.input	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 8,
		.result	= "\xc0\x45\x04\x01\x2e\x4e\x1f\x53",
		.rlen	= 8,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.ilen	= 504,
		.result	= "\x96\x87\x3D\x0C\x7B\xFB\xBD\x1F"
			  "\xE3\xC1\x99\x6D\x39\xD4\xC2\x7D"
			  "\xD7\x87\xA1\xF2\xDF\x51\x71\x26"
			  "\xC2\xF4\x6D\xFF\xF6\xCD\x6B\x40"
			  "\xE1\xB3\xBF\xD4\x38\x2B\xC8\x3B"
			  "\xD3\xB2\xD4\x61\xC7\x9F\x06\xE9"
			  "\xCD\xF3\x88\x39\x39\x7A\xDF\x19"
			  "\xE8\x03\x2A\x0B\x9E\xA0\x2B\x86"
			  "\x31\xF8\x9D\xB1\xEE\x78\x9D\xB5"
			  "\xCD\x8B\x7C\x2E\xF5\xA2\x2D\x5D"
			  "\x6E\x66\xAF\x38\x6C\xD3\x13\xED"
			  "\x14\xEA\x5D\xD0\x17\x77\x0F\x4A"
			  "\x50\xF2\xD0\x0F\xC8\xF7\x1E\x7B"
			  "\x9D\x5B\x54\x65\x4F\x16\x8A\x97"
			  "\xF3\xF6\xD4\xAA\x87\x36\x77\x72"
			  "\x99\x4A\xB5\x5E\x88\xC3\xCD\x7D"
			  "\x1D\x97\xF9\x11\xBD\xE0\x1F\x1F"
			  "\x96\x3E\x4B\x22\xF4\xC0\xE6\xB8"
			  "\x47\x82\x98\x23\x33\x36\xBC\x1B"
			  "\x36\xE7\xF6\xCF\x97\x37\x16\xC0"
			  "\x87\x31\x8B\xB0\xDB\x19\x42\xA5"
			  "\x1F\x90\x7E\x66\x34\xDD\x5E\xE9"
			  "\x4F\xB2\x2B\x9A\xDE\xB3\x5D\x71"
			  "\x4D\x68\xF0\xDC\xA6\xEA\xE3\x9B"
			  "\x60\x00\x55\x57\x06\x8B\xD5\xB3"
			  "\x86\x30\x78\xDA\x33\x9A\x9D\xCC"
			  "\xBA\x0B\x81\x06\x77\x43\xC7\xC9"
			  "\xDB\x37\x60\x11\x45\x59\x6D\x2D"
			  "\x90\x3D\x65\x3E\xD0\x13\xC6\x3C"
			  "\x0E\x78\x7D\x9A\x00\xD6\x2F\x0B"
			  "\x3B\x53\x19\x1E\xA8\x9B\x11\xD9"
			  "\x98\xE4\x7F\xC3\x6E\x51\x24\x70"
			  "\x9F\x04\x9C\xC2\x9E\x44\x84\xE3"
			  "\xE0\x8A\x44\xA2\x5C\x94\x74\x34"
			  "\x37\x52\x7C\x03\xE8\x8E\x97\xE1"
			  "\x5B\x5C\x0E\xB0\x70\xFE\x54\x3F"
			  "\xD8\x65\xA9\xC5\xCD\xEC\xF4\x45"
			  "\x55\xC5\xA7\xA3\x19\x80\x28\x51"
			  "\xBE\x64\x4A\xC1\xD4\xE1\xBE\xEB"
			  "\x73\x4C\xB6\xF9\x5F\x6D\x82\xBC"
			  "\x3E\x42\x14\x49\x88\x51\xBF\x68"
			  "\x45\x75\x27\x1B\x0A\x72\xED\xAF"
			  "\xDA\xC4\x4D\x67\x0D\xEE\x75\xE3"
			  "\x34\xDD\x91\x19\x42\x3A\xCB\xDA"
			  "\x38\xFA\x3C\x93\x62\xF2\xE3\x81"
			  "\xB3\xE4\xBB\xF6\x0D\x0B\x1D\x09"
			  "\x9C\x52\x0D\x50\x63\xA4\xB2\xD2"
			  "\x82\xA0\x23\x3F\x1F\xB6\xED\x6E"
			  "\xC2\x9C\x1C\xD0\x9A\x40\xB6\xFC"
			  "\x36\x56\x6E\x85\x73\xD7\x52\xBA"
			  "\x35\x5E\x32\x89\x5D\x42\xF5\x36"
			  "\x52\x8D\x46\x7D\xC8\x71\xAD\x33"
			  "\xE1\xAF\x6A\xA8\xEC\xBA\x1C\xDC"
			  "\xFE\x88\xE6\x16\xE4\xC8\x13\x00"
			  "\x3C\xDA\x59\x32\x38\x19\xD5\xEB"
			  "\xB6\x7F\x78\x45\x1B\x8E\x07\x8C"
			  "\x66\x52\x75\xFF\xAF\xCE\x2D\x2B"
			  "\x22\x29\xCA\xB3\x5F\x7F\xE3\x29"
			  "\xB2\xB8\x9D\xEB\x16\xC8\xC5\x1D"
			  "\xC9\x0D\x59\x82\x27\x57\x9D\x42"
			  "\x54\x59\x09\xA5\x3D\xC5\x84\x68"
			  "\x56\xEB\x36\x77\x3D\xAA\xB8\xF5"
			  "\xC9\x1A\xFB\x5D\xDE\xBB\x43\xF4",
		.rlen	= 504,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 504 - 8, 8 },
	},
};

static struct cipher_testvec bf_dec_tv_template[] = {
	{  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.input	= "\x4e\xf9\x97\x45\x61\x98\xdd\x78",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\x1f\x1f\x1f\x1f\x0e\x0e\x0e\x0e",
		.klen	= 8,
		.input	= "\xa7\x90\x79\x51\x08\xea\x3c\xae",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.rlen	= 8,
	}, {
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.klen	= 8,
		.input	= "\xe8\x7a\x24\x4e\x2c\xc8\x5e\x82",
		.ilen	= 8,
		.result	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 8,
	}, {  
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87"
			  "\x78\x69\x5a\x4b\x3c\x2d\x1e\x0f",
		.klen	= 16,
		.input	= "\x93\x14\x28\x87\xee\x3b\xe1\x5c",
		.ilen	= 8,
		.result	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 8,
	}, {
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87"
			  "\x78\x69\x5a\x4b\x3c\x2d\x1e\x0f"
			  "\x00\x11\x22\x33\x44",
		.klen	= 21,
		.input	= "\xe6\xf5\x1e\xd7\x9b\x9d\xb2\x1f",
		.ilen	= 8,
		.result	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 8,
	}, {  
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87"
			  "\x78\x69\x5a\x4b\x3c\x2d\x1e\x0f"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x04\x68\x91\x04\xc2\xfd\x3b\x2f"
			  "\x58\x40\x23\x64\x1a\xba\x61\x76"
			  "\x1f\x1f\x1f\x1f\x0e\x0e\x0e\x0e"
			  "\xff\xff\xff\xff\xff\xff\xff\xff",
		.klen	= 56,
		.input	= "\xc0\x45\x04\x01\x2e\x4e\x1f\x53",
		.ilen	= 8,
		.result	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 8,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.input	= "\x96\x87\x3D\x0C\x7B\xFB\xBD\x1F"
			  "\xE3\xC1\x99\x6D\x39\xD4\xC2\x7D"
			  "\xD7\x87\xA1\xF2\xDF\x51\x71\x26"
			  "\xC2\xF4\x6D\xFF\xF6\xCD\x6B\x40"
			  "\xE1\xB3\xBF\xD4\x38\x2B\xC8\x3B"
			  "\xD3\xB2\xD4\x61\xC7\x9F\x06\xE9"
			  "\xCD\xF3\x88\x39\x39\x7A\xDF\x19"
			  "\xE8\x03\x2A\x0B\x9E\xA0\x2B\x86"
			  "\x31\xF8\x9D\xB1\xEE\x78\x9D\xB5"
			  "\xCD\x8B\x7C\x2E\xF5\xA2\x2D\x5D"
			  "\x6E\x66\xAF\x38\x6C\xD3\x13\xED"
			  "\x14\xEA\x5D\xD0\x17\x77\x0F\x4A"
			  "\x50\xF2\xD0\x0F\xC8\xF7\x1E\x7B"
			  "\x9D\x5B\x54\x65\x4F\x16\x8A\x97"
			  "\xF3\xF6\xD4\xAA\x87\x36\x77\x72"
			  "\x99\x4A\xB5\x5E\x88\xC3\xCD\x7D"
			  "\x1D\x97\xF9\x11\xBD\xE0\x1F\x1F"
			  "\x96\x3E\x4B\x22\xF4\xC0\xE6\xB8"
			  "\x47\x82\x98\x23\x33\x36\xBC\x1B"
			  "\x36\xE7\xF6\xCF\x97\x37\x16\xC0"
			  "\x87\x31\x8B\xB0\xDB\x19\x42\xA5"
			  "\x1F\x90\x7E\x66\x34\xDD\x5E\xE9"
			  "\x4F\xB2\x2B\x9A\xDE\xB3\x5D\x71"
			  "\x4D\x68\xF0\xDC\xA6\xEA\xE3\x9B"
			  "\x60\x00\x55\x57\x06\x8B\xD5\xB3"
			  "\x86\x30\x78\xDA\x33\x9A\x9D\xCC"
			  "\xBA\x0B\x81\x06\x77\x43\xC7\xC9"
			  "\xDB\x37\x60\x11\x45\x59\x6D\x2D"
			  "\x90\x3D\x65\x3E\xD0\x13\xC6\x3C"
			  "\x0E\x78\x7D\x9A\x00\xD6\x2F\x0B"
			  "\x3B\x53\x19\x1E\xA8\x9B\x11\xD9"
			  "\x98\xE4\x7F\xC3\x6E\x51\x24\x70"
			  "\x9F\x04\x9C\xC2\x9E\x44\x84\xE3"
			  "\xE0\x8A\x44\xA2\x5C\x94\x74\x34"
			  "\x37\x52\x7C\x03\xE8\x8E\x97\xE1"
			  "\x5B\x5C\x0E\xB0\x70\xFE\x54\x3F"
			  "\xD8\x65\xA9\xC5\xCD\xEC\xF4\x45"
			  "\x55\xC5\xA7\xA3\x19\x80\x28\x51"
			  "\xBE\x64\x4A\xC1\xD4\xE1\xBE\xEB"
			  "\x73\x4C\xB6\xF9\x5F\x6D\x82\xBC"
			  "\x3E\x42\x14\x49\x88\x51\xBF\x68"
			  "\x45\x75\x27\x1B\x0A\x72\xED\xAF"
			  "\xDA\xC4\x4D\x67\x0D\xEE\x75\xE3"
			  "\x34\xDD\x91\x19\x42\x3A\xCB\xDA"
			  "\x38\xFA\x3C\x93\x62\xF2\xE3\x81"
			  "\xB3\xE4\xBB\xF6\x0D\x0B\x1D\x09"
			  "\x9C\x52\x0D\x50\x63\xA4\xB2\xD2"
			  "\x82\xA0\x23\x3F\x1F\xB6\xED\x6E"
			  "\xC2\x9C\x1C\xD0\x9A\x40\xB6\xFC"
			  "\x36\x56\x6E\x85\x73\xD7\x52\xBA"
			  "\x35\x5E\x32\x89\x5D\x42\xF5\x36"
			  "\x52\x8D\x46\x7D\xC8\x71\xAD\x33"
			  "\xE1\xAF\x6A\xA8\xEC\xBA\x1C\xDC"
			  "\xFE\x88\xE6\x16\xE4\xC8\x13\x00"
			  "\x3C\xDA\x59\x32\x38\x19\xD5\xEB"
			  "\xB6\x7F\x78\x45\x1B\x8E\x07\x8C"
			  "\x66\x52\x75\xFF\xAF\xCE\x2D\x2B"
			  "\x22\x29\xCA\xB3\x5F\x7F\xE3\x29"
			  "\xB2\xB8\x9D\xEB\x16\xC8\xC5\x1D"
			  "\xC9\x0D\x59\x82\x27\x57\x9D\x42"
			  "\x54\x59\x09\xA5\x3D\xC5\x84\x68"
			  "\x56\xEB\x36\x77\x3D\xAA\xB8\xF5"
			  "\xC9\x1A\xFB\x5D\xDE\xBB\x43\xF4",
		.ilen	= 504,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.rlen	= 504,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 504 - 8, 8 },
	},
};

static struct cipher_testvec bf_cbc_enc_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.klen	= 16,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x37\x36\x35\x34\x33\x32\x31\x20"
			  "\x4e\x6f\x77\x20\x69\x73\x20\x74"
			  "\x68\x65\x20\x74\x69\x6d\x65\x20"
			  "\x66\x6f\x72\x20\x00\x00\x00\x00",
		.ilen	= 32,
		.result	= "\x6b\x77\xb4\xd6\x30\x06\xde\xe6"
			  "\x05\xb1\x56\xe2\x74\x03\x97\x93"
			  "\x58\xde\xb9\xe7\x15\x46\x16\xd9"
			  "\x59\xf1\x65\x2b\xd5\xff\x92\xcc",
		.rlen	= 32,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.ilen	= 504,
		.result	= "\xB4\xFE\xA5\xBB\x3D\x2C\x27\x06"
			  "\x06\x2B\x3A\x92\xB2\xF5\x5E\x62"
			  "\x84\xCD\xF7\x66\x7E\x41\x6C\x8E"
			  "\x1B\xD9\x02\xB6\x48\xB0\x87\x25"
			  "\x01\x9C\x93\x63\x51\x60\x82\xD2"
			  "\x4D\xE5\xC2\xB7\xAE\x60\xD8\xAD"
			  "\x9F\xAB\x6C\xFA\x20\x05\xDA\x6F"
			  "\x1F\xD1\xD8\x36\x0F\xB5\x16\x69"
			  "\x3C\xAF\xB3\x30\x18\x33\xE6\xB5"
			  "\x43\x29\x9D\x94\xF4\x2F\x0A\x65"
			  "\x40\xB2\xB2\xB2\x42\x89\xEE\x8A"
			  "\x60\xD3\x52\xA8\xED\x91\xDF\xE1"
			  "\x91\x73\x7C\x28\xA1\x14\xC3\x4C"
			  "\x82\x72\x4B\x7D\x7D\x32\xD5\x19"
			  "\xE8\xB8\x6B\x30\x21\x09\x0E\x27"
			  "\x10\x9D\x2D\x3A\x6A\x4B\x7B\xE6"
			  "\x8D\x4E\x02\x32\xFF\x7F\x8E\x13"
			  "\xB0\x96\xF4\xC2\xA1\x60\x8A\x69"
			  "\xEF\x0F\x86\xD0\x25\x13\x1A\x7C"
			  "\x6E\xF0\x41\xA3\xFB\xB3\xAB\x40"
			  "\x7D\x19\xA0\x11\x4F\x3E\x1D\x43"
			  "\x65\xFE\x15\x40\xD0\x62\x41\x02"
			  "\xEA\x0C\x7A\xC3\x84\xEE\xB0\xBE"
			  "\xBE\xC8\x57\x51\xCD\x4F\xAD\x5C"
			  "\xCC\x79\xBA\x0D\x85\x3A\xED\x6B"
			  "\xAC\x6B\xA3\x4D\xBC\xE8\x02\x6A"
			  "\xC2\x6D\xBD\x5E\x89\x95\x86\x43"
			  "\x2C\x17\x4B\xC6\x40\xA2\xBD\x24"
			  "\x04\xF0\x86\x08\x78\x18\x42\xE0"
			  "\x39\x1B\x22\x9E\x89\x4C\x04\x6B"
			  "\x65\xC5\xB6\x0E\xF6\x63\xFC\xD7"
			  "\xAE\x9E\x87\x13\xCC\xD3\x1A\xEC"
			  "\xF0\x51\xCC\x93\x68\xFC\xE9\x19"
			  "\x7C\x4E\x9B\xCC\x17\xAD\xD2\xFC"
			  "\x97\x18\x92\xFF\x15\x11\xCE\xED"
			  "\x04\x41\x05\xA3\x92\xFF\x3B\xE6"
			  "\xB6\x8C\x90\xC6\xCD\x15\xA0\x04"
			  "\x25\x8B\x5D\x5B\x5F\xDB\xAE\x68"
			  "\xEF\xB3\x61\x18\xDB\x83\x9B\x39"
			  "\xCA\x82\xD1\x88\xF0\xA2\x5C\x02"
			  "\x87\xBD\x8D\x8F\xBB\x62\xF0\x35"
			  "\x75\x6F\x06\x81\x0A\x97\x4D\xF0"
			  "\x43\x12\x73\x77\xDB\x91\x83\x5B"
			  "\xE7\x3A\xA6\x07\x7B\xBF\x2C\x50"
			  "\x94\xDE\x7B\x65\xDA\x1C\xF1\x9F"
			  "\x7E\x12\x40\xB2\x3E\x19\x23\xF1"
			  "\x7C\x1B\x5F\xA8\xF3\xAC\x63\x87"
			  "\xEB\x3E\x0C\xBE\xA3\x63\x97\x88"
			  "\x8D\x27\xC6\x2A\xF8\xF2\x67\x9A"
			  "\x0D\x14\x16\x2B\x6F\xCB\xD4\x76"
			  "\x14\x48\x2E\xDE\x2A\x44\x5E\x45"
			  "\xF1\x97\x82\xEF\xB7\xAE\xED\x3A"
			  "\xED\x73\xD3\x79\xF7\x38\x1D\xD0"
			  "\xC5\xF8\x69\x83\x28\x84\x87\x56"
			  "\x3F\xAE\x81\x04\x79\x1F\xD1\x09"
			  "\xC5\xE5\x05\x0D\x64\x16\xCE\x42"
			  "\xC5\xF8\xDB\x57\x89\x33\x22\xFC"
			  "\xB4\xD7\x94\xB9\xF3\xCC\x02\x90"
			  "\x02\xBA\x55\x1E\x24\x3E\x02\x1D"
			  "\xC6\xCD\x8F\xD9\xBD\xED\xB0\x51"
			  "\xCD\xE9\xD5\x0C\xFE\x12\x39\xA9"
			  "\x93\x9B\xEE\xB5\x97\x41\xD2\xA0"
			  "\xB4\x98\xD8\x6B\x74\xE7\x65\xF4",
		.rlen	= 504,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 504 - 8, 8 },
	},
};

static struct cipher_testvec bf_cbc_dec_tv_template[] = {
	{  
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.klen	= 16,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x6b\x77\xb4\xd6\x30\x06\xde\xe6"
			  "\x05\xb1\x56\xe2\x74\x03\x97\x93"
			  "\x58\xde\xb9\xe7\x15\x46\x16\xd9"
			  "\x59\xf1\x65\x2b\xd5\xff\x92\xcc",
		.ilen	= 32,
		.result	= "\x37\x36\x35\x34\x33\x32\x31\x20"
			  "\x4e\x6f\x77\x20\x69\x73\x20\x74"
			  "\x68\x65\x20\x74\x69\x6d\x65\x20"
			  "\x66\x6f\x72\x20\x00\x00\x00\x00",
		.rlen	= 32,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\xB4\xFE\xA5\xBB\x3D\x2C\x27\x06"
			  "\x06\x2B\x3A\x92\xB2\xF5\x5E\x62"
			  "\x84\xCD\xF7\x66\x7E\x41\x6C\x8E"
			  "\x1B\xD9\x02\xB6\x48\xB0\x87\x25"
			  "\x01\x9C\x93\x63\x51\x60\x82\xD2"
			  "\x4D\xE5\xC2\xB7\xAE\x60\xD8\xAD"
			  "\x9F\xAB\x6C\xFA\x20\x05\xDA\x6F"
			  "\x1F\xD1\xD8\x36\x0F\xB5\x16\x69"
			  "\x3C\xAF\xB3\x30\x18\x33\xE6\xB5"
			  "\x43\x29\x9D\x94\xF4\x2F\x0A\x65"
			  "\x40\xB2\xB2\xB2\x42\x89\xEE\x8A"
			  "\x60\xD3\x52\xA8\xED\x91\xDF\xE1"
			  "\x91\x73\x7C\x28\xA1\x14\xC3\x4C"
			  "\x82\x72\x4B\x7D\x7D\x32\xD5\x19"
			  "\xE8\xB8\x6B\x30\x21\x09\x0E\x27"
			  "\x10\x9D\x2D\x3A\x6A\x4B\x7B\xE6"
			  "\x8D\x4E\x02\x32\xFF\x7F\x8E\x13"
			  "\xB0\x96\xF4\xC2\xA1\x60\x8A\x69"
			  "\xEF\x0F\x86\xD0\x25\x13\x1A\x7C"
			  "\x6E\xF0\x41\xA3\xFB\xB3\xAB\x40"
			  "\x7D\x19\xA0\x11\x4F\x3E\x1D\x43"
			  "\x65\xFE\x15\x40\xD0\x62\x41\x02"
			  "\xEA\x0C\x7A\xC3\x84\xEE\xB0\xBE"
			  "\xBE\xC8\x57\x51\xCD\x4F\xAD\x5C"
			  "\xCC\x79\xBA\x0D\x85\x3A\xED\x6B"
			  "\xAC\x6B\xA3\x4D\xBC\xE8\x02\x6A"
			  "\xC2\x6D\xBD\x5E\x89\x95\x86\x43"
			  "\x2C\x17\x4B\xC6\x40\xA2\xBD\x24"
			  "\x04\xF0\x86\x08\x78\x18\x42\xE0"
			  "\x39\x1B\x22\x9E\x89\x4C\x04\x6B"
			  "\x65\xC5\xB6\x0E\xF6\x63\xFC\xD7"
			  "\xAE\x9E\x87\x13\xCC\xD3\x1A\xEC"
			  "\xF0\x51\xCC\x93\x68\xFC\xE9\x19"
			  "\x7C\x4E\x9B\xCC\x17\xAD\xD2\xFC"
			  "\x97\x18\x92\xFF\x15\x11\xCE\xED"
			  "\x04\x41\x05\xA3\x92\xFF\x3B\xE6"
			  "\xB6\x8C\x90\xC6\xCD\x15\xA0\x04"
			  "\x25\x8B\x5D\x5B\x5F\xDB\xAE\x68"
			  "\xEF\xB3\x61\x18\xDB\x83\x9B\x39"
			  "\xCA\x82\xD1\x88\xF0\xA2\x5C\x02"
			  "\x87\xBD\x8D\x8F\xBB\x62\xF0\x35"
			  "\x75\x6F\x06\x81\x0A\x97\x4D\xF0"
			  "\x43\x12\x73\x77\xDB\x91\x83\x5B"
			  "\xE7\x3A\xA6\x07\x7B\xBF\x2C\x50"
			  "\x94\xDE\x7B\x65\xDA\x1C\xF1\x9F"
			  "\x7E\x12\x40\xB2\x3E\x19\x23\xF1"
			  "\x7C\x1B\x5F\xA8\xF3\xAC\x63\x87"
			  "\xEB\x3E\x0C\xBE\xA3\x63\x97\x88"
			  "\x8D\x27\xC6\x2A\xF8\xF2\x67\x9A"
			  "\x0D\x14\x16\x2B\x6F\xCB\xD4\x76"
			  "\x14\x48\x2E\xDE\x2A\x44\x5E\x45"
			  "\xF1\x97\x82\xEF\xB7\xAE\xED\x3A"
			  "\xED\x73\xD3\x79\xF7\x38\x1D\xD0"
			  "\xC5\xF8\x69\x83\x28\x84\x87\x56"
			  "\x3F\xAE\x81\x04\x79\x1F\xD1\x09"
			  "\xC5\xE5\x05\x0D\x64\x16\xCE\x42"
			  "\xC5\xF8\xDB\x57\x89\x33\x22\xFC"
			  "\xB4\xD7\x94\xB9\xF3\xCC\x02\x90"
			  "\x02\xBA\x55\x1E\x24\x3E\x02\x1D"
			  "\xC6\xCD\x8F\xD9\xBD\xED\xB0\x51"
			  "\xCD\xE9\xD5\x0C\xFE\x12\x39\xA9"
			  "\x93\x9B\xEE\xB5\x97\x41\xD2\xA0"
			  "\xB4\x98\xD8\x6B\x74\xE7\x65\xF4",
		.ilen	= 504,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.rlen	= 504,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 504 - 8, 8 },
	},
};

static struct cipher_testvec bf_ctr_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.ilen	= 504,
		.result	= "\xC7\xA3\xDF\xB9\x05\xF4\x9E\x8D"
			  "\x9E\xDF\x38\x18\x83\x07\xEF\xC1"
			  "\x93\x3C\xAA\xAA\xFE\x06\x42\xCC"
			  "\x0D\x70\x86\x5A\x44\xAD\x85\x17"
			  "\xE4\x1F\x5E\xA5\x89\xAC\x32\xBC"
			  "\x3D\xA7\xE9\x0A\x5C\x70\x4D\xDE"
			  "\x99\x38\x07\xCA\x1D\x21\xC1\x11"
			  "\x97\xEB\x98\x75\xC4\x73\x45\x83"
			  "\x46\x1C\x9C\x91\x87\xC1\xA0\x56"
			  "\x98\xA1\x8B\xDB\x22\x76\xBD\x62"
			  "\xA4\xBC\xE8\x86\xDA\xD2\x51\x13"
			  "\x13\xD2\x96\x68\x69\x10\x67\x0C"
			  "\xD0\x17\x25\x7C\xB2\xAE\x4F\x93"
			  "\xA6\x82\x20\xCF\x0F\xA6\x47\x79"
			  "\x88\x09\x40\x59\xBD\x12\x64\xB5"
			  "\x19\x38\x0D\xFF\x86\xD9\x42\x20"
			  "\x81\x0D\x96\x99\xAF\x22\x1F\x94"
			  "\x5C\x6E\xEC\xEA\xA3\x39\xCB\x09"
			  "\x43\x19\x7F\xD0\xBB\x10\xC2\x49"
			  "\xF7\xE9\xF2\xEE\xBF\xF7\xF8\xB3"
			  "\x0E\x1A\xF1\x8D\x70\x82\x0C\x04"
			  "\xFD\x29\x1A\xAC\xC0\x92\x48\x34"
			  "\x6A\xE3\x1D\x4F\xFC\x1C\x72\x6A"
			  "\x57\xCB\xAD\xD0\x98\xAB\xB1\x01"
			  "\x03\x6A\x45\xDD\x07\x71\x5F\x5B"
			  "\xB5\x4A\xE4\xE5\xB9\xB9\xBC\xAC"
			  "\x44\xF7\x41\xA4\x5F\x2E\xE9\x28"
			  "\xE3\x05\xD2\x94\x78\x4C\x33\x1B"
			  "\xBD\xC1\x6E\x51\xD9\xAD\xD9\x86"
			  "\x15\x4A\x78\xAE\x7B\xAD\x3B\xBC"
			  "\x2F\xE0\x0E\xC5\x7B\x54\x97\x5F"
			  "\x60\x51\x14\x65\xF9\x91\xE9\xDA"
			  "\x9A\xBC\xFC\x19\x29\x67\xAA\x63"
			  "\x5E\xF2\x48\x88\xEB\x79\xE1\xE4"
			  "\xF7\xF6\x4C\xA9\xE2\x8C\x3B\xE0"
			  "\xED\x52\xAE\x90\x8F\x5B\x98\x34"
			  "\x29\x94\x34\x7F\xF9\x6C\x1E\xB6"
			  "\xA4\xE7\x2D\x06\x54\x9D\xC3\x02"
			  "\xC1\x90\xA4\x72\x31\x6B\x24\x51"
			  "\x0B\xB3\x7C\x63\x15\xBA\xAF\x5D"
			  "\x41\xE0\x37\x6D\xBE\x41\x58\xDE"
			  "\xF2\x07\x62\x99\xBE\xC1\x8C\x0F"
			  "\x0F\x28\xFB\x8F\x0E\x1D\x91\xE2"
			  "\xDA\x99\x5C\x49\xBA\x9C\xA8\x86"
			  "\x82\x63\x11\xB3\x54\x49\x00\x08"
			  "\x07\xF2\xE8\x1F\x34\x49\x61\xF4"
			  "\x81\xE9\xF6\xA9\x5A\x28\x60\x1F"
			  "\x66\x99\x08\x06\xF2\xE8\x2D\xD1"
			  "\xD0\x67\xBA\x32\x1F\x02\x86\x7B"
			  "\xFB\x79\x3D\xC5\xB1\x7F\x15\xAF"
			  "\xD7\xBF\x31\x46\x22\x7F\xAE\x5B"
			  "\x8B\x95\x47\xC2\xB1\x62\xA1\xCE"
			  "\x52\xAC\x9C\x8B\xC2\x49\x7F\xBC"
			  "\x9C\x89\xB8\xB6\xCA\xE3\x8F\xEA"
			  "\xAC\xB4\x5D\xE4\x50\xDC\x3A\xB5"
			  "\x91\x04\x94\x99\x03\x3B\x42\x6D"
			  "\x9C\x4A\x02\xF5\xB5\x38\x98\xA8"
			  "\x5C\x97\x2E\x4D\x79\x67\x71\xAF"
			  "\xF0\x70\x77\xFF\x2D\xDA\xA0\x9E"
			  "\x23\x8D\xD6\xA6\x68\x10\x78\x9A"
			  "\x64\xBB\x15\xB8\x56\xCF\xEE\xE5"
			  "\x32\x44\x96\x1C\xD8\xEB\x95\xD2"
			  "\xF3\x71\xEF\xEB\x4E\xBB\x4D\x29",
		.rlen	= 504,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92",
		.ilen	= 503,
		.result	= "\xC7\xA3\xDF\xB9\x05\xF4\x9E\x8D"
			  "\x9E\xDF\x38\x18\x83\x07\xEF\xC1"
			  "\x93\x3C\xAA\xAA\xFE\x06\x42\xCC"
			  "\x0D\x70\x86\x5A\x44\xAD\x85\x17"
			  "\xE4\x1F\x5E\xA5\x89\xAC\x32\xBC"
			  "\x3D\xA7\xE9\x0A\x5C\x70\x4D\xDE"
			  "\x99\x38\x07\xCA\x1D\x21\xC1\x11"
			  "\x97\xEB\x98\x75\xC4\x73\x45\x83"
			  "\x46\x1C\x9C\x91\x87\xC1\xA0\x56"
			  "\x98\xA1\x8B\xDB\x22\x76\xBD\x62"
			  "\xA4\xBC\xE8\x86\xDA\xD2\x51\x13"
			  "\x13\xD2\x96\x68\x69\x10\x67\x0C"
			  "\xD0\x17\x25\x7C\xB2\xAE\x4F\x93"
			  "\xA6\x82\x20\xCF\x0F\xA6\x47\x79"
			  "\x88\x09\x40\x59\xBD\x12\x64\xB5"
			  "\x19\x38\x0D\xFF\x86\xD9\x42\x20"
			  "\x81\x0D\x96\x99\xAF\x22\x1F\x94"
			  "\x5C\x6E\xEC\xEA\xA3\x39\xCB\x09"
			  "\x43\x19\x7F\xD0\xBB\x10\xC2\x49"
			  "\xF7\xE9\xF2\xEE\xBF\xF7\xF8\xB3"
			  "\x0E\x1A\xF1\x8D\x70\x82\x0C\x04"
			  "\xFD\x29\x1A\xAC\xC0\x92\x48\x34"
			  "\x6A\xE3\x1D\x4F\xFC\x1C\x72\x6A"
			  "\x57\xCB\xAD\xD0\x98\xAB\xB1\x01"
			  "\x03\x6A\x45\xDD\x07\x71\x5F\x5B"
			  "\xB5\x4A\xE4\xE5\xB9\xB9\xBC\xAC"
			  "\x44\xF7\x41\xA4\x5F\x2E\xE9\x28"
			  "\xE3\x05\xD2\x94\x78\x4C\x33\x1B"
			  "\xBD\xC1\x6E\x51\xD9\xAD\xD9\x86"
			  "\x15\x4A\x78\xAE\x7B\xAD\x3B\xBC"
			  "\x2F\xE0\x0E\xC5\x7B\x54\x97\x5F"
			  "\x60\x51\x14\x65\xF9\x91\xE9\xDA"
			  "\x9A\xBC\xFC\x19\x29\x67\xAA\x63"
			  "\x5E\xF2\x48\x88\xEB\x79\xE1\xE4"
			  "\xF7\xF6\x4C\xA9\xE2\x8C\x3B\xE0"
			  "\xED\x52\xAE\x90\x8F\x5B\x98\x34"
			  "\x29\x94\x34\x7F\xF9\x6C\x1E\xB6"
			  "\xA4\xE7\x2D\x06\x54\x9D\xC3\x02"
			  "\xC1\x90\xA4\x72\x31\x6B\x24\x51"
			  "\x0B\xB3\x7C\x63\x15\xBA\xAF\x5D"
			  "\x41\xE0\x37\x6D\xBE\x41\x58\xDE"
			  "\xF2\x07\x62\x99\xBE\xC1\x8C\x0F"
			  "\x0F\x28\xFB\x8F\x0E\x1D\x91\xE2"
			  "\xDA\x99\x5C\x49\xBA\x9C\xA8\x86"
			  "\x82\x63\x11\xB3\x54\x49\x00\x08"
			  "\x07\xF2\xE8\x1F\x34\x49\x61\xF4"
			  "\x81\xE9\xF6\xA9\x5A\x28\x60\x1F"
			  "\x66\x99\x08\x06\xF2\xE8\x2D\xD1"
			  "\xD0\x67\xBA\x32\x1F\x02\x86\x7B"
			  "\xFB\x79\x3D\xC5\xB1\x7F\x15\xAF"
			  "\xD7\xBF\x31\x46\x22\x7F\xAE\x5B"
			  "\x8B\x95\x47\xC2\xB1\x62\xA1\xCE"
			  "\x52\xAC\x9C\x8B\xC2\x49\x7F\xBC"
			  "\x9C\x89\xB8\xB6\xCA\xE3\x8F\xEA"
			  "\xAC\xB4\x5D\xE4\x50\xDC\x3A\xB5"
			  "\x91\x04\x94\x99\x03\x3B\x42\x6D"
			  "\x9C\x4A\x02\xF5\xB5\x38\x98\xA8"
			  "\x5C\x97\x2E\x4D\x79\x67\x71\xAF"
			  "\xF0\x70\x77\xFF\x2D\xDA\xA0\x9E"
			  "\x23\x8D\xD6\xA6\x68\x10\x78\x9A"
			  "\x64\xBB\x15\xB8\x56\xCF\xEE\xE5"
			  "\x32\x44\x96\x1C\xD8\xEB\x95\xD2"
			  "\xF3\x71\xEF\xEB\x4E\xBB\x4D",
		.rlen	= 503,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 503 - 8, 8 },
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.ilen	= 504,
		.result	= "\x5F\x58\x6E\x60\x51\x6E\xDC\x3D"
			  "\xD1\xBB\xF7\xB7\xFD\x04\x44\x82"
			  "\xDC\x9F\x4B\x02\xF1\xD2\x5A\x6F"
			  "\x25\xF9\x27\x21\xF2\xD2\x9A\x01"
			  "\xBD\xAD\x3D\x93\x87\xCA\x0D\xFE"
			  "\xB7\x2C\x17\x1F\x42\x8C\x13\xB2"
			  "\x62\x44\x72\xB9\x5D\xC0\xF8\x37"
			  "\xDF\xEA\x78\x81\x8F\xA6\x34\xB2"
			  "\x07\x09\x7C\xB9\x3A\xA0\x2B\x18"
			  "\x34\x6A\x9D\x3D\xA5\xEB\xF4\x60"
			  "\xF8\x98\xA2\x39\x81\x23\x6C\xA9"
			  "\x70\xCA\xCC\x45\xD8\x1F\xDF\x44"
			  "\x2A\x67\x7A\x88\x28\xDC\x36\x83"
			  "\x18\xD7\x48\x43\x17\x2B\x1B\xE6"
			  "\x0B\x82\x59\x14\x26\x67\x08\x09"
			  "\x5B\x5D\x38\xD0\x81\xCE\x54\x2A"
			  "\xCD\x22\x94\x42\xF5\xBA\x74\x7E"
			  "\xD9\x00\x40\xA9\x0D\x0B\xBD\x8E"
			  "\xC4\x8E\x5E\x17\x8F\x48\xE2\xB8"
			  "\xF4\xCC\x19\x76\xAB\x48\x29\xAA"
			  "\x81\xD5\xCE\xD5\x8A\x3B\xC9\x21"
			  "\xEF\x50\x4F\x04\x02\xBF\xE1\x1F"
			  "\x59\x28\x1A\xE4\x18\x16\xA0\x29"
			  "\xBF\x34\xA9\x2D\x28\x83\xC0\x5E"
			  "\xEA\x44\xC4\x6E\xAB\x24\x79\x9D"
			  "\x2D\xA1\xE8\x55\xCA\x74\xFC\xBD"
			  "\xFE\xDD\xDA\xA5\xFB\x34\x90\x31"
			  "\x0E\x62\x28\x9B\xDC\xD7\xA1\xBB"
			  "\xF0\x1A\xB3\xE2\xD0\xFA\xBD\xE8"
			  "\x5C\x5A\x10\x67\xF6\x6A\x17\x3F"
			  "\xC5\xE9\x09\x08\xDD\x22\x77\x42"
			  "\x26\x6A\x6A\x7A\x3F\x87\x80\x0C"
			  "\xF0\xFF\x15\x8E\x84\x86\xC0\x10"
			  "\x0F\x8D\x33\x06\xB8\x72\xA4\x47"
			  "\x6B\xED\x2E\x05\x94\x6C\x5C\x5B"
			  "\x13\xF6\x77\xEE\x3B\x16\xDF\xC2"
			  "\x63\x66\x07\x6D\x3F\x6C\x51\x7C"
			  "\x1C\xAC\x80\xB6\x58\x48\xB7\x9D"
			  "\xB4\x19\xD8\x19\x45\x66\x27\x02"
			  "\xA1\xA9\x99\xF3\x1F\xE5\xA7\x1D"
			  "\x31\xE7\x1B\x0D\xFF\xBB\xB5\xA1"
			  "\xF5\x9C\x45\x1E\x18\x19\xA1\xE7"
			  "\xC2\xF1\xBF\x68\xC3\xEC\xCF\x53"
			  "\x67\xA6\x2B\x7D\x3C\x6D\x24\xC3"
			  "\xE8\xE6\x07\x5A\x09\xE0\x32\xA8"
			  "\x52\xF6\xE9\xED\x0E\xC6\x0A\x6A"
			  "\xFC\x60\x2A\xE0\x93\xCE\xB8\x2E"
			  "\xA2\xA8\x0E\x79\x9E\x34\x5D\x37"
			  "\x6F\x12\xFE\x48\x7B\xE7\xB9\x22"
			  "\x29\xE8\xD7\xBE\x5D\xD1\x8B\xD9"
			  "\x91\x51\x4E\x71\xF2\x98\x85\x16"
			  "\x25\x7A\x76\x8A\x51\x0E\x65\x14"
			  "\x81\xB5\x3A\x37\xFD\xEC\xB5\x8A"
			  "\xE1\xCF\x41\x72\x14\x29\x4C\xF0"
			  "\x20\xD9\x9A\xC5\x66\xA4\x03\x76"
			  "\x5B\xA4\x15\x4F\x0E\x64\x39\x40"
			  "\x25\xF9\x20\x22\xF5\x88\xF5\xBA"
			  "\xE4\xDF\x45\x61\xBF\x8D\x7A\x24"
			  "\x4B\x92\x71\xD9\x2F\x77\xA7\x95"
			  "\xA8\x7F\x61\xD5\xA4\x57\xB0\xFB"
			  "\xB5\x77\xBA\x1C\xEE\x71\xFA\xB0"
			  "\x16\x4C\x18\x6B\xF2\x69\xA0\x07"
			  "\xEF\xBE\xEC\x69\xAC\xA8\x63\x9E",
		.rlen	= 504,
	},
};

static struct cipher_testvec bf_ctr_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\xC7\xA3\xDF\xB9\x05\xF4\x9E\x8D"
			  "\x9E\xDF\x38\x18\x83\x07\xEF\xC1"
			  "\x93\x3C\xAA\xAA\xFE\x06\x42\xCC"
			  "\x0D\x70\x86\x5A\x44\xAD\x85\x17"
			  "\xE4\x1F\x5E\xA5\x89\xAC\x32\xBC"
			  "\x3D\xA7\xE9\x0A\x5C\x70\x4D\xDE"
			  "\x99\x38\x07\xCA\x1D\x21\xC1\x11"
			  "\x97\xEB\x98\x75\xC4\x73\x45\x83"
			  "\x46\x1C\x9C\x91\x87\xC1\xA0\x56"
			  "\x98\xA1\x8B\xDB\x22\x76\xBD\x62"
			  "\xA4\xBC\xE8\x86\xDA\xD2\x51\x13"
			  "\x13\xD2\x96\x68\x69\x10\x67\x0C"
			  "\xD0\x17\x25\x7C\xB2\xAE\x4F\x93"
			  "\xA6\x82\x20\xCF\x0F\xA6\x47\x79"
			  "\x88\x09\x40\x59\xBD\x12\x64\xB5"
			  "\x19\x38\x0D\xFF\x86\xD9\x42\x20"
			  "\x81\x0D\x96\x99\xAF\x22\x1F\x94"
			  "\x5C\x6E\xEC\xEA\xA3\x39\xCB\x09"
			  "\x43\x19\x7F\xD0\xBB\x10\xC2\x49"
			  "\xF7\xE9\xF2\xEE\xBF\xF7\xF8\xB3"
			  "\x0E\x1A\xF1\x8D\x70\x82\x0C\x04"
			  "\xFD\x29\x1A\xAC\xC0\x92\x48\x34"
			  "\x6A\xE3\x1D\x4F\xFC\x1C\x72\x6A"
			  "\x57\xCB\xAD\xD0\x98\xAB\xB1\x01"
			  "\x03\x6A\x45\xDD\x07\x71\x5F\x5B"
			  "\xB5\x4A\xE4\xE5\xB9\xB9\xBC\xAC"
			  "\x44\xF7\x41\xA4\x5F\x2E\xE9\x28"
			  "\xE3\x05\xD2\x94\x78\x4C\x33\x1B"
			  "\xBD\xC1\x6E\x51\xD9\xAD\xD9\x86"
			  "\x15\x4A\x78\xAE\x7B\xAD\x3B\xBC"
			  "\x2F\xE0\x0E\xC5\x7B\x54\x97\x5F"
			  "\x60\x51\x14\x65\xF9\x91\xE9\xDA"
			  "\x9A\xBC\xFC\x19\x29\x67\xAA\x63"
			  "\x5E\xF2\x48\x88\xEB\x79\xE1\xE4"
			  "\xF7\xF6\x4C\xA9\xE2\x8C\x3B\xE0"
			  "\xED\x52\xAE\x90\x8F\x5B\x98\x34"
			  "\x29\x94\x34\x7F\xF9\x6C\x1E\xB6"
			  "\xA4\xE7\x2D\x06\x54\x9D\xC3\x02"
			  "\xC1\x90\xA4\x72\x31\x6B\x24\x51"
			  "\x0B\xB3\x7C\x63\x15\xBA\xAF\x5D"
			  "\x41\xE0\x37\x6D\xBE\x41\x58\xDE"
			  "\xF2\x07\x62\x99\xBE\xC1\x8C\x0F"
			  "\x0F\x28\xFB\x8F\x0E\x1D\x91\xE2"
			  "\xDA\x99\x5C\x49\xBA\x9C\xA8\x86"
			  "\x82\x63\x11\xB3\x54\x49\x00\x08"
			  "\x07\xF2\xE8\x1F\x34\x49\x61\xF4"
			  "\x81\xE9\xF6\xA9\x5A\x28\x60\x1F"
			  "\x66\x99\x08\x06\xF2\xE8\x2D\xD1"
			  "\xD0\x67\xBA\x32\x1F\x02\x86\x7B"
			  "\xFB\x79\x3D\xC5\xB1\x7F\x15\xAF"
			  "\xD7\xBF\x31\x46\x22\x7F\xAE\x5B"
			  "\x8B\x95\x47\xC2\xB1\x62\xA1\xCE"
			  "\x52\xAC\x9C\x8B\xC2\x49\x7F\xBC"
			  "\x9C\x89\xB8\xB6\xCA\xE3\x8F\xEA"
			  "\xAC\xB4\x5D\xE4\x50\xDC\x3A\xB5"
			  "\x91\x04\x94\x99\x03\x3B\x42\x6D"
			  "\x9C\x4A\x02\xF5\xB5\x38\x98\xA8"
			  "\x5C\x97\x2E\x4D\x79\x67\x71\xAF"
			  "\xF0\x70\x77\xFF\x2D\xDA\xA0\x9E"
			  "\x23\x8D\xD6\xA6\x68\x10\x78\x9A"
			  "\x64\xBB\x15\xB8\x56\xCF\xEE\xE5"
			  "\x32\x44\x96\x1C\xD8\xEB\x95\xD2"
			  "\xF3\x71\xEF\xEB\x4E\xBB\x4D\x29",
		.ilen	= 504,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.rlen	= 504,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\xC7\xA3\xDF\xB9\x05\xF4\x9E\x8D"
			  "\x9E\xDF\x38\x18\x83\x07\xEF\xC1"
			  "\x93\x3C\xAA\xAA\xFE\x06\x42\xCC"
			  "\x0D\x70\x86\x5A\x44\xAD\x85\x17"
			  "\xE4\x1F\x5E\xA5\x89\xAC\x32\xBC"
			  "\x3D\xA7\xE9\x0A\x5C\x70\x4D\xDE"
			  "\x99\x38\x07\xCA\x1D\x21\xC1\x11"
			  "\x97\xEB\x98\x75\xC4\x73\x45\x83"
			  "\x46\x1C\x9C\x91\x87\xC1\xA0\x56"
			  "\x98\xA1\x8B\xDB\x22\x76\xBD\x62"
			  "\xA4\xBC\xE8\x86\xDA\xD2\x51\x13"
			  "\x13\xD2\x96\x68\x69\x10\x67\x0C"
			  "\xD0\x17\x25\x7C\xB2\xAE\x4F\x93"
			  "\xA6\x82\x20\xCF\x0F\xA6\x47\x79"
			  "\x88\x09\x40\x59\xBD\x12\x64\xB5"
			  "\x19\x38\x0D\xFF\x86\xD9\x42\x20"
			  "\x81\x0D\x96\x99\xAF\x22\x1F\x94"
			  "\x5C\x6E\xEC\xEA\xA3\x39\xCB\x09"
			  "\x43\x19\x7F\xD0\xBB\x10\xC2\x49"
			  "\xF7\xE9\xF2\xEE\xBF\xF7\xF8\xB3"
			  "\x0E\x1A\xF1\x8D\x70\x82\x0C\x04"
			  "\xFD\x29\x1A\xAC\xC0\x92\x48\x34"
			  "\x6A\xE3\x1D\x4F\xFC\x1C\x72\x6A"
			  "\x57\xCB\xAD\xD0\x98\xAB\xB1\x01"
			  "\x03\x6A\x45\xDD\x07\x71\x5F\x5B"
			  "\xB5\x4A\xE4\xE5\xB9\xB9\xBC\xAC"
			  "\x44\xF7\x41\xA4\x5F\x2E\xE9\x28"
			  "\xE3\x05\xD2\x94\x78\x4C\x33\x1B"
			  "\xBD\xC1\x6E\x51\xD9\xAD\xD9\x86"
			  "\x15\x4A\x78\xAE\x7B\xAD\x3B\xBC"
			  "\x2F\xE0\x0E\xC5\x7B\x54\x97\x5F"
			  "\x60\x51\x14\x65\xF9\x91\xE9\xDA"
			  "\x9A\xBC\xFC\x19\x29\x67\xAA\x63"
			  "\x5E\xF2\x48\x88\xEB\x79\xE1\xE4"
			  "\xF7\xF6\x4C\xA9\xE2\x8C\x3B\xE0"
			  "\xED\x52\xAE\x90\x8F\x5B\x98\x34"
			  "\x29\x94\x34\x7F\xF9\x6C\x1E\xB6"
			  "\xA4\xE7\x2D\x06\x54\x9D\xC3\x02"
			  "\xC1\x90\xA4\x72\x31\x6B\x24\x51"
			  "\x0B\xB3\x7C\x63\x15\xBA\xAF\x5D"
			  "\x41\xE0\x37\x6D\xBE\x41\x58\xDE"
			  "\xF2\x07\x62\x99\xBE\xC1\x8C\x0F"
			  "\x0F\x28\xFB\x8F\x0E\x1D\x91\xE2"
			  "\xDA\x99\x5C\x49\xBA\x9C\xA8\x86"
			  "\x82\x63\x11\xB3\x54\x49\x00\x08"
			  "\x07\xF2\xE8\x1F\x34\x49\x61\xF4"
			  "\x81\xE9\xF6\xA9\x5A\x28\x60\x1F"
			  "\x66\x99\x08\x06\xF2\xE8\x2D\xD1"
			  "\xD0\x67\xBA\x32\x1F\x02\x86\x7B"
			  "\xFB\x79\x3D\xC5\xB1\x7F\x15\xAF"
			  "\xD7\xBF\x31\x46\x22\x7F\xAE\x5B"
			  "\x8B\x95\x47\xC2\xB1\x62\xA1\xCE"
			  "\x52\xAC\x9C\x8B\xC2\x49\x7F\xBC"
			  "\x9C\x89\xB8\xB6\xCA\xE3\x8F\xEA"
			  "\xAC\xB4\x5D\xE4\x50\xDC\x3A\xB5"
			  "\x91\x04\x94\x99\x03\x3B\x42\x6D"
			  "\x9C\x4A\x02\xF5\xB5\x38\x98\xA8"
			  "\x5C\x97\x2E\x4D\x79\x67\x71\xAF"
			  "\xF0\x70\x77\xFF\x2D\xDA\xA0\x9E"
			  "\x23\x8D\xD6\xA6\x68\x10\x78\x9A"
			  "\x64\xBB\x15\xB8\x56\xCF\xEE\xE5"
			  "\x32\x44\x96\x1C\xD8\xEB\x95\xD2"
			  "\xF3\x71\xEF\xEB\x4E\xBB\x4D",
		.ilen	= 503,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92",
		.rlen	= 503,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 503 - 8, 8 },
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x5F\x58\x6E\x60\x51\x6E\xDC\x3D"
			  "\xD1\xBB\xF7\xB7\xFD\x04\x44\x82"
			  "\xDC\x9F\x4B\x02\xF1\xD2\x5A\x6F"
			  "\x25\xF9\x27\x21\xF2\xD2\x9A\x01"
			  "\xBD\xAD\x3D\x93\x87\xCA\x0D\xFE"
			  "\xB7\x2C\x17\x1F\x42\x8C\x13\xB2"
			  "\x62\x44\x72\xB9\x5D\xC0\xF8\x37"
			  "\xDF\xEA\x78\x81\x8F\xA6\x34\xB2"
			  "\x07\x09\x7C\xB9\x3A\xA0\x2B\x18"
			  "\x34\x6A\x9D\x3D\xA5\xEB\xF4\x60"
			  "\xF8\x98\xA2\x39\x81\x23\x6C\xA9"
			  "\x70\xCA\xCC\x45\xD8\x1F\xDF\x44"
			  "\x2A\x67\x7A\x88\x28\xDC\x36\x83"
			  "\x18\xD7\x48\x43\x17\x2B\x1B\xE6"
			  "\x0B\x82\x59\x14\x26\x67\x08\x09"
			  "\x5B\x5D\x38\xD0\x81\xCE\x54\x2A"
			  "\xCD\x22\x94\x42\xF5\xBA\x74\x7E"
			  "\xD9\x00\x40\xA9\x0D\x0B\xBD\x8E"
			  "\xC4\x8E\x5E\x17\x8F\x48\xE2\xB8"
			  "\xF4\xCC\x19\x76\xAB\x48\x29\xAA"
			  "\x81\xD5\xCE\xD5\x8A\x3B\xC9\x21"
			  "\xEF\x50\x4F\x04\x02\xBF\xE1\x1F"
			  "\x59\x28\x1A\xE4\x18\x16\xA0\x29"
			  "\xBF\x34\xA9\x2D\x28\x83\xC0\x5E"
			  "\xEA\x44\xC4\x6E\xAB\x24\x79\x9D"
			  "\x2D\xA1\xE8\x55\xCA\x74\xFC\xBD"
			  "\xFE\xDD\xDA\xA5\xFB\x34\x90\x31"
			  "\x0E\x62\x28\x9B\xDC\xD7\xA1\xBB"
			  "\xF0\x1A\xB3\xE2\xD0\xFA\xBD\xE8"
			  "\x5C\x5A\x10\x67\xF6\x6A\x17\x3F"
			  "\xC5\xE9\x09\x08\xDD\x22\x77\x42"
			  "\x26\x6A\x6A\x7A\x3F\x87\x80\x0C"
			  "\xF0\xFF\x15\x8E\x84\x86\xC0\x10"
			  "\x0F\x8D\x33\x06\xB8\x72\xA4\x47"
			  "\x6B\xED\x2E\x05\x94\x6C\x5C\x5B"
			  "\x13\xF6\x77\xEE\x3B\x16\xDF\xC2"
			  "\x63\x66\x07\x6D\x3F\x6C\x51\x7C"
			  "\x1C\xAC\x80\xB6\x58\x48\xB7\x9D"
			  "\xB4\x19\xD8\x19\x45\x66\x27\x02"
			  "\xA1\xA9\x99\xF3\x1F\xE5\xA7\x1D"
			  "\x31\xE7\x1B\x0D\xFF\xBB\xB5\xA1"
			  "\xF5\x9C\x45\x1E\x18\x19\xA1\xE7"
			  "\xC2\xF1\xBF\x68\xC3\xEC\xCF\x53"
			  "\x67\xA6\x2B\x7D\x3C\x6D\x24\xC3"
			  "\xE8\xE6\x07\x5A\x09\xE0\x32\xA8"
			  "\x52\xF6\xE9\xED\x0E\xC6\x0A\x6A"
			  "\xFC\x60\x2A\xE0\x93\xCE\xB8\x2E"
			  "\xA2\xA8\x0E\x79\x9E\x34\x5D\x37"
			  "\x6F\x12\xFE\x48\x7B\xE7\xB9\x22"
			  "\x29\xE8\xD7\xBE\x5D\xD1\x8B\xD9"
			  "\x91\x51\x4E\x71\xF2\x98\x85\x16"
			  "\x25\x7A\x76\x8A\x51\x0E\x65\x14"
			  "\x81\xB5\x3A\x37\xFD\xEC\xB5\x8A"
			  "\xE1\xCF\x41\x72\x14\x29\x4C\xF0"
			  "\x20\xD9\x9A\xC5\x66\xA4\x03\x76"
			  "\x5B\xA4\x15\x4F\x0E\x64\x39\x40"
			  "\x25\xF9\x20\x22\xF5\x88\xF5\xBA"
			  "\xE4\xDF\x45\x61\xBF\x8D\x7A\x24"
			  "\x4B\x92\x71\xD9\x2F\x77\xA7\x95"
			  "\xA8\x7F\x61\xD5\xA4\x57\xB0\xFB"
			  "\xB5\x77\xBA\x1C\xEE\x71\xFA\xB0"
			  "\x16\x4C\x18\x6B\xF2\x69\xA0\x07"
			  "\xEF\xBE\xEC\x69\xAC\xA8\x63\x9E",
		.ilen	= 504,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06",
		.rlen	= 504,
	},
};

#define TF_ENC_TEST_VECTORS		4
#define TF_DEC_TEST_VECTORS		4
#define TF_CBC_ENC_TEST_VECTORS		5
#define TF_CBC_DEC_TEST_VECTORS		5
#define TF_CTR_ENC_TEST_VECTORS		2
#define TF_CTR_DEC_TEST_VECTORS		2
#define TF_LRW_ENC_TEST_VECTORS		8
#define TF_LRW_DEC_TEST_VECTORS		8
#define TF_XTS_ENC_TEST_VECTORS		5
#define TF_XTS_DEC_TEST_VECTORS		5

static struct cipher_testvec tf_enc_tv_template[] = {
	{
		.key	= zeroed_string,
		.klen	= 16,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a",
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77",
		.klen	= 24,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\xcf\xd1\xd2\xe5\xa9\xbe\x9c\xdf"
			  "\x50\x1f\x13\xb8\x92\xbd\x22\x48",
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.klen	= 32,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\x37\x52\x7b\xe0\x05\x23\x34\xb8"
			  "\x9f\x0c\xfc\xca\xe8\x7c\xfa\x20",
		.rlen	= 16,
	}, {  
		.key	= "\x3F\x85\x62\x3F\x1C\xF9\xD6\x1C"
			  "\xF9\xD6\xB3\x90\x6D\x4A\x90\x6D"
			  "\x4A\x27\x04\xE1\x27\x04\xE1\xBE"
			  "\x9B\x78\xBE\x9B\x78\x55\x32\x0F",
		.klen	= 32,
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x88\xCB\x1E\xC2\xAF\x8A\x97\xFF"
			  "\xF6\x90\x46\x9C\x4A\x0F\x08\xDC"
			  "\xDE\xAB\xAD\xFA\xFC\xA8\xC2\x3D"
			  "\xE0\xE4\x8B\x3F\xD5\xA3\xF7\x14"
			  "\x34\x9E\xB6\x08\xB2\xDD\xA8\xF5"
			  "\xDF\xFA\xC7\xE8\x09\x50\x76\x08"
			  "\xA2\xB6\x6A\x59\xC0\x2B\x6D\x05"
			  "\x89\xF6\x82\xF0\xD3\xDB\x06\x02"
			  "\xB5\x11\x5C\x5E\x79\x1A\xAC\x43"
			  "\x5C\xC0\x30\x4B\x6B\x16\xA1\x40"
			  "\x80\x27\x88\xBA\x2C\x74\x42\xE0"
			  "\x1B\xA5\x85\x08\xB9\xE6\x22\x7A"
			  "\x36\x3B\x0D\x9F\xA0\x22\x6C\x2A"
			  "\x91\x75\x47\xBC\x67\x21\x4E\xF9"
			  "\xEA\xFF\xD9\xD5\xC0\xFC\x9E\x2C"
			  "\x3E\xAD\xC6\x61\x0E\x93\x7A\x22"
			  "\x09\xC8\x8D\xC1\x8E\xB4\x8B\x5C"
			  "\xC6\x24\x42\xB8\x23\x66\x80\xA9"
			  "\x32\x0B\x7A\x29\xBF\xB3\x0B\x63"
			  "\x43\x27\x13\xA9\xBE\xEB\xBD\xF3"
			  "\x33\x62\x70\xE2\x1B\x86\x7A\xA1"
			  "\x51\x4A\x16\xFE\x29\x63\x7E\xD0"
			  "\x7A\xA4\x6E\x2C\xF8\xC1\xDB\xE8"
			  "\xCB\x4D\xD2\x8C\x04\x14\xB4\x66"
			  "\x41\xB7\x3A\x96\x16\x7C\x1D\x5B"
			  "\xB6\x41\x42\x64\x43\xEE\x6E\x7C"
			  "\x8B\xAF\x01\x9C\xA4\x6E\x75\x8F"
			  "\xDE\x10\x9F\xA6\xE7\xD6\x44\x97"
			  "\x66\xA3\x96\x0F\x1C\x25\x60\xF5"
			  "\x3C\x2E\x32\x69\x0E\x82\xFF\x27"
			  "\x0F\xB5\x06\xDA\xD8\x31\x15\x6C"
			  "\xDF\x18\x6C\x87\xF5\x3B\x11\x9A"
			  "\x1B\x42\x1F\x5B\x29\x19\x96\x13"
			  "\x68\x2E\x5E\x08\x1C\x8F\x32\x4B"
			  "\x81\x77\x6D\xF4\xA0\x01\x42\xEC"
			  "\xDD\x5B\xFD\x3A\x8E\x6A\x14\xFB"
			  "\x83\x54\xDF\x0F\x86\xB7\xEA\x40"
			  "\x46\x39\xF7\x2A\x89\x8D\x4E\x96"
			  "\x5F\x5F\x6D\x76\xC6\x13\x9D\x3D"
			  "\x1D\x5F\x0C\x7D\xE2\xBC\xC2\x16"
			  "\x16\xBE\x89\x3E\xB0\x61\xA2\x5D"
			  "\xAF\xD1\x40\x5F\x1A\xB8\x26\x41"
			  "\xC6\xBD\x36\xEF\xED\x29\x50\x6D"
			  "\x10\xEF\x26\xE8\xA8\x93\x11\x3F"
			  "\x2D\x1F\x88\x20\x77\x45\xF5\x66"
			  "\x08\xB9\xF1\xEF\xB1\x93\xA8\x81"
			  "\x65\xC5\xCD\x3E\x8C\x06\x60\x2C"
			  "\xB2\x10\x7A\xCA\x05\x25\x59\xDB"
			  "\xC7\x28\xF5\x20\x35\x52\x9E\x62"
			  "\xF8\x88\x24\x1C\x4D\x84\x12\x39"
			  "\x39\xE4\x2E\xF4\xD4\x9D\x2B\xBC"
			  "\x87\x66\xE6\xC0\x6B\x31\x9A\x66"
			  "\x03\xDC\x95\xD8\x6B\xD0\x30\x8F"
			  "\xDF\x8F\x8D\xFA\xEC\x1F\x08\xBD"
			  "\xA3\x63\xE2\x71\x4F\x03\x94\x87"
			  "\x50\xDF\x15\x1F\xED\x3A\xA3\x7F"
			  "\x1F\x2A\xB5\xA1\x69\xAC\x4B\x0D"
			  "\x84\x9B\x2A\xE9\x55\xDD\x46\x91"
			  "\x15\x33\xF3\x2B\x9B\x46\x97\x00"
			  "\xF0\x29\xD8\x59\x5D\x33\x37\xF9"
			  "\x58\x33\x9B\x78\xC7\x58\x48\x6B"
			  "\x2C\x75\x64\xC4\xCA\xC1\x7E\xD5",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec tf_dec_tv_template[] = {
	{
		.key	= zeroed_string,
		.klen	= 16,
		.input	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77",
		.klen	= 24,
		.input	= "\xcf\xd1\xd2\xe5\xa9\xbe\x9c\xdf"
			  "\x50\x1f\x13\xb8\x92\xbd\x22\x48",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.klen	= 32,
		.input	= "\x37\x52\x7b\xe0\x05\x23\x34\xb8"
			  "\x9f\x0c\xfc\xca\xe8\x7c\xfa\x20",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {  
		.key	= "\x3F\x85\x62\x3F\x1C\xF9\xD6\x1C"
			  "\xF9\xD6\xB3\x90\x6D\x4A\x90\x6D"
			  "\x4A\x27\x04\xE1\x27\x04\xE1\xBE"
			  "\x9B\x78\xBE\x9B\x78\x55\x32\x0F",
		.klen	= 32,
		.input	= "\x88\xCB\x1E\xC2\xAF\x8A\x97\xFF"
			  "\xF6\x90\x46\x9C\x4A\x0F\x08\xDC"
			  "\xDE\xAB\xAD\xFA\xFC\xA8\xC2\x3D"
			  "\xE0\xE4\x8B\x3F\xD5\xA3\xF7\x14"
			  "\x34\x9E\xB6\x08\xB2\xDD\xA8\xF5"
			  "\xDF\xFA\xC7\xE8\x09\x50\x76\x08"
			  "\xA2\xB6\x6A\x59\xC0\x2B\x6D\x05"
			  "\x89\xF6\x82\xF0\xD3\xDB\x06\x02"
			  "\xB5\x11\x5C\x5E\x79\x1A\xAC\x43"
			  "\x5C\xC0\x30\x4B\x6B\x16\xA1\x40"
			  "\x80\x27\x88\xBA\x2C\x74\x42\xE0"
			  "\x1B\xA5\x85\x08\xB9\xE6\x22\x7A"
			  "\x36\x3B\x0D\x9F\xA0\x22\x6C\x2A"
			  "\x91\x75\x47\xBC\x67\x21\x4E\xF9"
			  "\xEA\xFF\xD9\xD5\xC0\xFC\x9E\x2C"
			  "\x3E\xAD\xC6\x61\x0E\x93\x7A\x22"
			  "\x09\xC8\x8D\xC1\x8E\xB4\x8B\x5C"
			  "\xC6\x24\x42\xB8\x23\x66\x80\xA9"
			  "\x32\x0B\x7A\x29\xBF\xB3\x0B\x63"
			  "\x43\x27\x13\xA9\xBE\xEB\xBD\xF3"
			  "\x33\x62\x70\xE2\x1B\x86\x7A\xA1"
			  "\x51\x4A\x16\xFE\x29\x63\x7E\xD0"
			  "\x7A\xA4\x6E\x2C\xF8\xC1\xDB\xE8"
			  "\xCB\x4D\xD2\x8C\x04\x14\xB4\x66"
			  "\x41\xB7\x3A\x96\x16\x7C\x1D\x5B"
			  "\xB6\x41\x42\x64\x43\xEE\x6E\x7C"
			  "\x8B\xAF\x01\x9C\xA4\x6E\x75\x8F"
			  "\xDE\x10\x9F\xA6\xE7\xD6\x44\x97"
			  "\x66\xA3\x96\x0F\x1C\x25\x60\xF5"
			  "\x3C\x2E\x32\x69\x0E\x82\xFF\x27"
			  "\x0F\xB5\x06\xDA\xD8\x31\x15\x6C"
			  "\xDF\x18\x6C\x87\xF5\x3B\x11\x9A"
			  "\x1B\x42\x1F\x5B\x29\x19\x96\x13"
			  "\x68\x2E\x5E\x08\x1C\x8F\x32\x4B"
			  "\x81\x77\x6D\xF4\xA0\x01\x42\xEC"
			  "\xDD\x5B\xFD\x3A\x8E\x6A\x14\xFB"
			  "\x83\x54\xDF\x0F\x86\xB7\xEA\x40"
			  "\x46\x39\xF7\x2A\x89\x8D\x4E\x96"
			  "\x5F\x5F\x6D\x76\xC6\x13\x9D\x3D"
			  "\x1D\x5F\x0C\x7D\xE2\xBC\xC2\x16"
			  "\x16\xBE\x89\x3E\xB0\x61\xA2\x5D"
			  "\xAF\xD1\x40\x5F\x1A\xB8\x26\x41"
			  "\xC6\xBD\x36\xEF\xED\x29\x50\x6D"
			  "\x10\xEF\x26\xE8\xA8\x93\x11\x3F"
			  "\x2D\x1F\x88\x20\x77\x45\xF5\x66"
			  "\x08\xB9\xF1\xEF\xB1\x93\xA8\x81"
			  "\x65\xC5\xCD\x3E\x8C\x06\x60\x2C"
			  "\xB2\x10\x7A\xCA\x05\x25\x59\xDB"
			  "\xC7\x28\xF5\x20\x35\x52\x9E\x62"
			  "\xF8\x88\x24\x1C\x4D\x84\x12\x39"
			  "\x39\xE4\x2E\xF4\xD4\x9D\x2B\xBC"
			  "\x87\x66\xE6\xC0\x6B\x31\x9A\x66"
			  "\x03\xDC\x95\xD8\x6B\xD0\x30\x8F"
			  "\xDF\x8F\x8D\xFA\xEC\x1F\x08\xBD"
			  "\xA3\x63\xE2\x71\x4F\x03\x94\x87"
			  "\x50\xDF\x15\x1F\xED\x3A\xA3\x7F"
			  "\x1F\x2A\xB5\xA1\x69\xAC\x4B\x0D"
			  "\x84\x9B\x2A\xE9\x55\xDD\x46\x91"
			  "\x15\x33\xF3\x2B\x9B\x46\x97\x00"
			  "\xF0\x29\xD8\x59\x5D\x33\x37\xF9"
			  "\x58\x33\x9B\x78\xC7\x58\x48\x6B"
			  "\x2C\x75\x64\xC4\xCA\xC1\x7E\xD5",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec tf_cbc_enc_tv_template[] = {
	{  
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= zeroed_string,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a",
		.rlen	= 16,
	}, {
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a",
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\xd4\x91\xdb\x16\xe7\xb1\xc3\x9e"
			  "\x86\xcb\x08\x6b\x78\x9f\x54\x19",
		.rlen	= 16,
	}, {
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= "\xd4\x91\xdb\x16\xe7\xb1\xc3\x9e"
			  "\x86\xcb\x08\x6b\x78\x9f\x54\x19",
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\x05\xef\x8c\x61\xa8\x11\x58\x26"
			  "\x34\xba\x5c\xb7\x10\x6a\xa6\x41",
		.rlen	= 16,
	}, {
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= zeroed_string,
		.input	= zeroed_string,
		.ilen	= 48,
		.result	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a"
			  "\xd4\x91\xdb\x16\xe7\xb1\xc3\x9e"
			  "\x86\xcb\x08\x6b\x78\x9f\x54\x19"
			  "\x05\xef\x8c\x61\xa8\x11\x58\x26"
			  "\x34\xba\x5c\xb7\x10\x6a\xa6\x41",
		.rlen	= 48,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xC8\xFF\xF2\x53\xA6\x27\x09\xD1"
			  "\x33\x38\xC2\xC0\x0C\x14\x7E\xB5"
			  "\x26\x1B\x05\x0C\x05\x12\x3F\xC0"
			  "\xF9\x1C\x02\x28\x40\x96\x6F\xD0"
			  "\x3D\x32\xDF\xDA\x56\x00\x6E\xEE"
			  "\x5B\x2A\x72\x9D\xC2\x4D\x19\xBC"
			  "\x8C\x53\xFA\x87\x6F\xDD\x81\xA3"
			  "\xB1\xD3\x44\x65\xDF\xE7\x63\x38"
			  "\x4A\xFC\xDC\xEC\x3F\x26\x8E\xB8"
			  "\x43\xFC\xFE\x18\xB5\x11\x6D\x31"
			  "\x81\x8B\x0D\x75\xF6\x80\xEC\x84"
			  "\x04\xB9\xE6\x09\x63\xED\x39\xDB"
			  "\xC3\xF6\x14\xD6\x6E\x5E\x8B\xBD"
			  "\x3E\xFA\xD7\x98\x50\x6F\xD9\x63"
			  "\x02\xCD\x0D\x39\x4B\x0D\xEC\x80"
			  "\xE3\x6A\x17\xF4\xCC\xAD\xFF\x68"
			  "\x45\xDD\xC8\x83\x1D\x41\x96\x0D"
			  "\x91\x2E\x05\xD3\x59\x82\xE0\x43"
			  "\x90\x4F\xB9\xF7\xAD\x6B\x2E\xAF"
			  "\xA7\x84\x00\x53\xCD\x6F\xD1\x0C"
			  "\x4E\xF9\x5A\x23\xFB\xCA\xC7\xD3"
			  "\xA9\xAA\x9D\xB2\x3F\x66\xF1\xAC"
			  "\x25\x21\x8F\xF7\xEF\xF2\x6A\xDF"
			  "\xE8\xDA\x75\x1A\x8A\xF1\xDD\x38"
			  "\x1F\xF9\x3D\x68\x4A\xBB\x9E\x34"
			  "\x1F\x66\x1F\x9C\x2B\x54\xFF\x60"
			  "\x7F\x29\x4B\x55\x80\x8F\x4E\xA7"
			  "\xA6\x9A\x0A\xD9\x0D\x19\x00\xF8"
			  "\x1F\xBC\x0C\x40\x6B\xEC\x99\x25"
			  "\x94\x70\x74\x0E\x1D\xC5\xBC\x12"
			  "\xF3\x42\xBE\x95\xBF\xFB\x4E\x55"
			  "\x9A\xB9\xCE\x14\x16\x5B\xDC\xD3"
			  "\x75\x42\x62\x04\x31\x1F\x95\x7C"
			  "\x66\x1A\x97\xDC\x2F\x40\x5C\x39"
			  "\x78\xE6\x02\xDB\x49\xE1\xC6\x47"
			  "\xC2\x78\x9A\xBB\xF3\xBE\xCB\x93"
			  "\xD8\xB8\xE8\xBB\x8C\xB3\x9B\xA7"
			  "\xC2\x89\xF3\x91\x88\x83\x3D\xF0"
			  "\x29\xA2\xCD\xB5\x79\x16\xC2\x40"
			  "\x11\x03\x8E\x9C\xFD\xC9\x43\xC4"
			  "\xC2\x19\xF0\x4A\x32\xEF\x0C\x2B"
			  "\xD3\x2B\xE9\xD4\x4C\xDE\x95\xCF"
			  "\x04\x03\xD3\x2C\x7F\x82\xC8\xFA"
			  "\x0F\xD8\x7A\x39\x7B\x01\x41\x9C"
			  "\x78\xB6\xC9\xBF\xF9\x78\x57\x88"
			  "\xB1\xA5\xE1\xE0\xD9\x16\xD4\xC8"
			  "\xEE\xC4\xBE\x7B\x55\x59\x00\x48"
			  "\x1B\xBC\x14\xFA\x2A\x9D\xC9\x1C"
			  "\xFB\x28\x3F\x95\xDD\xB7\xD6\xCE"
			  "\x3A\x7F\x09\x0C\x0E\x69\x30\x7D"
			  "\xBC\x68\x9C\x91\x2A\x59\x57\x04"
			  "\xED\x1A\x1E\x00\xB1\x85\x92\x04"
			  "\x28\x8C\x0C\x3C\xC1\xD5\x12\xF7"
			  "\x4C\x3E\xB0\xE7\x86\x62\x68\x91"
			  "\xFC\xC4\xE2\xCE\xA6\xDC\x5E\x93"
			  "\x5D\x8D\x8C\x68\xB3\xB2\xB9\x64"
			  "\x16\xB8\xC8\x6F\xD8\xEE\x21\xBD"
			  "\xAC\x18\x0C\x7D\x0D\x05\xAB\xF1"
			  "\xFA\xDD\xE2\x48\xDF\x4C\x02\x39"
			  "\x69\xA1\x62\xBD\x49\x3A\x9D\x91"
			  "\x30\x70\x56\xA4\x37\xDD\x7C\xC0"
			  "\x0A\xA3\x30\x10\x26\x25\x41\x2C",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec tf_cbc_dec_tv_template[] = {
	{  
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= zeroed_string,
		.input	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a",
		.input	= "\xd4\x91\xdb\x16\xe7\xb1\xc3\x9e"
			  "\x86\xcb\x08\x6b\x78\x9f\x54\x19",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= "\xd4\x91\xdb\x16\xe7\xb1\xc3\x9e"
			  "\x86\xcb\x08\x6b\x78\x9f\x54\x19",
		.input	= "\x05\xef\x8c\x61\xa8\x11\x58\x26"
			  "\x34\xba\x5c\xb7\x10\x6a\xa6\x41",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= zeroed_string,
		.klen	= 16,
		.iv	= zeroed_string,
		.input	= "\x9f\x58\x9f\x5c\xf6\x12\x2c\x32"
			  "\xb6\xbf\xec\x2f\x2a\xe8\xc3\x5a"
			  "\xd4\x91\xdb\x16\xe7\xb1\xc3\x9e"
			  "\x86\xcb\x08\x6b\x78\x9f\x54\x19"
			  "\x05\xef\x8c\x61\xa8\x11\x58\x26"
			  "\x34\xba\x5c\xb7\x10\x6a\xa6\x41",
		.ilen	= 48,
		.result	= zeroed_string,
		.rlen	= 48,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xC8\xFF\xF2\x53\xA6\x27\x09\xD1"
			  "\x33\x38\xC2\xC0\x0C\x14\x7E\xB5"
			  "\x26\x1B\x05\x0C\x05\x12\x3F\xC0"
			  "\xF9\x1C\x02\x28\x40\x96\x6F\xD0"
			  "\x3D\x32\xDF\xDA\x56\x00\x6E\xEE"
			  "\x5B\x2A\x72\x9D\xC2\x4D\x19\xBC"
			  "\x8C\x53\xFA\x87\x6F\xDD\x81\xA3"
			  "\xB1\xD3\x44\x65\xDF\xE7\x63\x38"
			  "\x4A\xFC\xDC\xEC\x3F\x26\x8E\xB8"
			  "\x43\xFC\xFE\x18\xB5\x11\x6D\x31"
			  "\x81\x8B\x0D\x75\xF6\x80\xEC\x84"
			  "\x04\xB9\xE6\x09\x63\xED\x39\xDB"
			  "\xC3\xF6\x14\xD6\x6E\x5E\x8B\xBD"
			  "\x3E\xFA\xD7\x98\x50\x6F\xD9\x63"
			  "\x02\xCD\x0D\x39\x4B\x0D\xEC\x80"
			  "\xE3\x6A\x17\xF4\xCC\xAD\xFF\x68"
			  "\x45\xDD\xC8\x83\x1D\x41\x96\x0D"
			  "\x91\x2E\x05\xD3\x59\x82\xE0\x43"
			  "\x90\x4F\xB9\xF7\xAD\x6B\x2E\xAF"
			  "\xA7\x84\x00\x53\xCD\x6F\xD1\x0C"
			  "\x4E\xF9\x5A\x23\xFB\xCA\xC7\xD3"
			  "\xA9\xAA\x9D\xB2\x3F\x66\xF1\xAC"
			  "\x25\x21\x8F\xF7\xEF\xF2\x6A\xDF"
			  "\xE8\xDA\x75\x1A\x8A\xF1\xDD\x38"
			  "\x1F\xF9\x3D\x68\x4A\xBB\x9E\x34"
			  "\x1F\x66\x1F\x9C\x2B\x54\xFF\x60"
			  "\x7F\x29\x4B\x55\x80\x8F\x4E\xA7"
			  "\xA6\x9A\x0A\xD9\x0D\x19\x00\xF8"
			  "\x1F\xBC\x0C\x40\x6B\xEC\x99\x25"
			  "\x94\x70\x74\x0E\x1D\xC5\xBC\x12"
			  "\xF3\x42\xBE\x95\xBF\xFB\x4E\x55"
			  "\x9A\xB9\xCE\x14\x16\x5B\xDC\xD3"
			  "\x75\x42\x62\x04\x31\x1F\x95\x7C"
			  "\x66\x1A\x97\xDC\x2F\x40\x5C\x39"
			  "\x78\xE6\x02\xDB\x49\xE1\xC6\x47"
			  "\xC2\x78\x9A\xBB\xF3\xBE\xCB\x93"
			  "\xD8\xB8\xE8\xBB\x8C\xB3\x9B\xA7"
			  "\xC2\x89\xF3\x91\x88\x83\x3D\xF0"
			  "\x29\xA2\xCD\xB5\x79\x16\xC2\x40"
			  "\x11\x03\x8E\x9C\xFD\xC9\x43\xC4"
			  "\xC2\x19\xF0\x4A\x32\xEF\x0C\x2B"
			  "\xD3\x2B\xE9\xD4\x4C\xDE\x95\xCF"
			  "\x04\x03\xD3\x2C\x7F\x82\xC8\xFA"
			  "\x0F\xD8\x7A\x39\x7B\x01\x41\x9C"
			  "\x78\xB6\xC9\xBF\xF9\x78\x57\x88"
			  "\xB1\xA5\xE1\xE0\xD9\x16\xD4\xC8"
			  "\xEE\xC4\xBE\x7B\x55\x59\x00\x48"
			  "\x1B\xBC\x14\xFA\x2A\x9D\xC9\x1C"
			  "\xFB\x28\x3F\x95\xDD\xB7\xD6\xCE"
			  "\x3A\x7F\x09\x0C\x0E\x69\x30\x7D"
			  "\xBC\x68\x9C\x91\x2A\x59\x57\x04"
			  "\xED\x1A\x1E\x00\xB1\x85\x92\x04"
			  "\x28\x8C\x0C\x3C\xC1\xD5\x12\xF7"
			  "\x4C\x3E\xB0\xE7\x86\x62\x68\x91"
			  "\xFC\xC4\xE2\xCE\xA6\xDC\x5E\x93"
			  "\x5D\x8D\x8C\x68\xB3\xB2\xB9\x64"
			  "\x16\xB8\xC8\x6F\xD8\xEE\x21\xBD"
			  "\xAC\x18\x0C\x7D\x0D\x05\xAB\xF1"
			  "\xFA\xDD\xE2\x48\xDF\x4C\x02\x39"
			  "\x69\xA1\x62\xBD\x49\x3A\x9D\x91"
			  "\x30\x70\x56\xA4\x37\xDD\x7C\xC0"
			  "\x0A\xA3\x30\x10\x26\x25\x41\x2C",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec tf_ctr_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xDF\xDD\x69\xFA\xB0\x2E\xFD\xFE"
			  "\x70\x9E\xC5\x4B\xC9\xD4\xA1\x30"
			  "\x26\x9B\x89\xA1\xEE\x43\xE0\x52"
			  "\x55\x17\x4E\xC7\x0E\x33\x1F\xF1"
			  "\x9F\x8D\x40\x9F\x24\xFD\x92\xA0"
			  "\xBC\x8F\x35\xDD\x67\x38\xD8\xAA"
			  "\xCF\xF8\x48\xCA\xFB\xE4\x5C\x60"
			  "\x01\x41\x21\x12\x38\xAB\x52\x4F"
			  "\xA8\x57\x20\xE0\x21\x6A\x17\x0D"
			  "\x0E\xF9\x8E\x49\x42\x00\x3C\x94"
			  "\x14\xC0\xD0\x8D\x8A\x98\xEB\x29"
			  "\xEC\xAE\x96\x44\xC0\x3C\x48\xDC"
			  "\x29\x35\x25\x2F\xE7\x11\x6C\x68"
			  "\xC8\x67\x0A\x2F\xF4\x07\xBE\xF9"
			  "\x2C\x31\x87\x40\xAB\xB2\xB6\xFA"
			  "\xD2\xC9\x6D\x5C\x50\xE9\xE6\x7E"
			  "\xE3\x0A\xD2\xD5\x6D\x8D\x64\x9E"
			  "\x70\xCE\x03\x76\xDD\xE0\xF0\x8C"
			  "\x84\x86\x8B\x6A\xFE\xC7\xF9\x69"
			  "\x2E\xFE\xFC\xC2\xC4\x1A\x55\x58"
			  "\xB3\xBE\xE2\x7E\xED\x39\x42\x6C"
			  "\xB4\x42\x97\x9A\xEC\xE1\x0A\x06"
			  "\x02\xC5\x03\x9D\xC4\x48\x15\x66"
			  "\x35\x6A\xC2\xC9\xA2\x26\x30\xBB"
			  "\xDB\x2D\xC8\x08\x2B\xA0\x29\x1A"
			  "\x23\x61\x48\xEA\x80\x04\x27\xAA"
			  "\x69\x49\xE8\xE8\x4A\x83\x6B\x5A"
			  "\xCA\x7C\xD3\xB1\xB5\x0B\xCC\x23"
			  "\x74\x1F\xA9\x87\xCD\xED\xC0\x2D"
			  "\xBF\xEB\xCF\x16\x2D\x2A\x2E\x1D"
			  "\x96\xBA\x36\x11\x45\x41\xDA\xCE"
			  "\xA4\x48\x80\x8B\x06\xF4\x98\x89"
			  "\x8B\x23\x08\x53\xF4\xD4\x5A\x24"
			  "\x8B\xF8\x43\x73\xD1\xEE\xC4\xB0"
			  "\xF8\xFE\x09\x0C\x75\x05\x38\x0B"
			  "\x7C\x81\xDE\x9D\xE4\x61\x37\x63"
			  "\x63\xAD\x12\xD2\x04\xB9\xCE\x45"
			  "\x5A\x1A\x6E\xB3\x78\x2A\xA4\x74"
			  "\x86\xD0\xE3\xFF\xDA\x38\x9C\xB5"
			  "\xB8\xB1\xDB\x38\x2F\xC5\x6A\xB4"
			  "\xEB\x6E\x96\xE8\x43\x80\xB5\x51"
			  "\x61\x2D\x48\xAA\x07\x65\x11\x8C"
			  "\x48\xE3\x90\x7E\x78\x3A\xEC\x97"
			  "\x05\x3D\x84\xE7\x90\x2B\xAA\xBD"
			  "\x83\x29\x0E\x1A\x81\x73\x7B\xE0"
			  "\x7A\x01\x4A\x37\x3B\x77\x7F\x8D"
			  "\x49\xA4\x2F\x6E\xBE\x68\x99\x08"
			  "\x99\xAA\x4C\x12\x04\xAE\x1F\x77"
			  "\x35\x88\xF1\x65\x06\x0A\x0B\x4D"
			  "\x47\xF9\x50\x38\x5D\x71\xF9\x6E"
			  "\xDE\xEC\x61\x35\x2C\x4C\x96\x50"
			  "\xE8\x28\x93\x9C\x7E\x01\xC6\x04"
			  "\xB2\xD6\xBC\x6C\x17\xEB\xC1\x7D"
			  "\x11\xE9\x43\x83\x76\xAA\x53\x37"
			  "\x0C\x1D\x39\x89\x53\x72\x09\x7E"
			  "\xD9\x85\x16\x04\xA5\x2C\x05\x6F"
			  "\x17\x0C\x6E\x66\xAA\x84\xA7\xD9"
			  "\xE2\xD9\xC4\xEB\x43\x3E\xB1\x8D"
			  "\x7C\x36\xC7\x71\x70\x9C\x10\xD8"
			  "\xE8\x47\x2A\x4D\xFD\xA1\xBC\xE3"
			  "\xB9\x32\xE2\xC1\x82\xAC\xFE\xCC"
			  "\xC5\xC9\x7F\x9E\xCF\x33\x7A\xDF",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xEB\x44\xAF\x49\x27\xB8\xFB\x44"
			  "\x4C\xA6\xC3\x0C\x8B\xD0\x01\x0C"
			  "\x53\xC8\x16\x38\xDE\x40\x4F\x91"
			  "\x25\x6D\x4C\xA0\x9A\x87\x1E\xDA"
			  "\x88\x7E\x89\xE9\x67\x2B\x83\xA2"
			  "\x5F\x2E\x23\x3E\x45\xB9\x77\x7B"
			  "\xA6\x7E\x47\x36\x81\x9F\x9B\xF3"
			  "\xE0\xF0\xD7\x47\xA9\xC8\xEF\x33"
			  "\x0C\x43\xFE\x67\x50\x0A\x2C\x3E"
			  "\xA0\xE1\x25\x8E\x80\x07\x4A\xC0"
			  "\x64\x89\x9F\x6A\x27\x96\x07\xA6"
			  "\x9B\xC8\x1B\x21\x60\xAE\x5D\x01"
			  "\xE2\xCD\xC8\xAA\x6C\x9D\x1C\x34"
			  "\x39\x18\x09\xA4\x82\x59\x78\xE7"
			  "\xFC\x59\x65\xF2\x94\xFF\xFB\xE2"
			  "\x3C\xDA\xB1\x90\x95\xBF\x91\xE3"
			  "\xE6\x87\x31\x9E\x16\x85\xAD\xB1"
			  "\x4C\xAE\x43\x4D\x19\x58\xB5\x5E"
			  "\x2E\xF5\x09\xAA\x39\xF4\xC0\xB3"
			  "\xD4\x4D\xDB\x73\x7A\xD4\xF1\xBF"
			  "\x89\x16\x4D\x2D\xA2\x26\x33\x72"
			  "\x18\x33\x7E\xD6\xD2\x16\xA4\x54"
			  "\xF4\x8C\xB3\x52\xDF\x21\x9C\xEB"
			  "\xBF\x49\xD3\xF9\x05\x06\xCB\xD2"
			  "\xA9\xD2\x3B\x6E\x19\x8C\xBC\x19"
			  "\xAB\x89\xD6\xD8\xCD\x56\x89\x5E"
			  "\xAC\x00\xE3\x50\x63\x4A\x80\x9A"
			  "\x05\xBC\x50\x39\xD3\x32\xD9\x0D"
			  "\xE3\x20\x0D\x75\x54\xEC\xE6\x31"
			  "\x14\xB9\x3A\x59\x00\x43\x37\x8E"
			  "\x8C\x5A\x79\x62\x14\x76\x8A\xAE"
			  "\x8F\xCC\xA1\x6C\x38\x78\xDD\x2D"
			  "\x8B\x6D\xEA\xBD\x7B\x25\xFF\x60"
			  "\xC9\x87\xB1\x79\x1E\xA5\x86\x68"
			  "\x81\xB4\xE2\xC1\x05\x7D\x3A\x73"
			  "\xD0\xDA\x75\x77\x9E\x05\x27\xF1"
			  "\x08\xA9\x66\x64\x6C\xBC\x82\x17"
			  "\x2C\x23\x5F\x62\x4D\x02\x1A\x58"
			  "\xE7\xB7\x23\x6D\xE2\x20\xDA\xEF"
			  "\xB4\xB3\x3F\xB2\x2B\x69\x98\x83"
			  "\x95\x87\x13\x57\x60\xD7\xB5\xB1"
			  "\xEE\x0A\x2F\x95\x36\x4C\x76\x5D"
			  "\x5F\xD9\x19\xED\xB9\xA5\x48\xBF"
			  "\xC8\xAB\x0F\x71\xCC\x61\x8E\x0A"
			  "\xD0\x29\x44\xA8\xB9\xC1\xE8\xC8"
			  "\xC9\xA8\x28\x81\xFB\x50\xF2\xF0"
			  "\x26\xAE\x39\xB8\x91\xCD\xA8\xAC"
			  "\xDE\x55\x1B\x50\x14\x53\x44\x17"
			  "\x54\x46\xFC\xB1\xE4\x07\x6B\x9A"
			  "\x01\x14\xF0\x2E\x2E\xDB\x46\x1B"
			  "\x1A\x09\x97\xA9\xB6\x97\x79\x06"
			  "\xFB\xCB\x85\xCF\xDD\xA1\x41\xB1"
			  "\x00\xAA\xF7\xE0\x89\x73\xFB\xE5"
			  "\xBF\x84\xDB\xC9\xCD\xC4\xA2\x0D"
			  "\x3B\xAC\xF9\xDF\x96\xBF\x88\x23"
			  "\x41\x67\xA1\x24\x99\x7E\xCC\x9B"
			  "\x02\x8F\x6A\x49\xF6\x25\xBA\x7A"
			  "\xF4\x78\xFD\x79\x62\x63\x4F\x14"
			  "\xD6\x11\x11\x04\x05\x5F\x7E\xEA"
			  "\x4C\xB6\xF8\xF4\x5F\x48\x52\x54"
			  "\x94\x63\xA8\x4E\xCF\xD2\x1B\x1B"
			  "\x22\x18\x6A\xAF\x6E\x3E\xE1\x0D",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59",
		.ilen	= 499,
		.result	= "\xDF\xDD\x69\xFA\xB0\x2E\xFD\xFE"
			  "\x70\x9E\xC5\x4B\xC9\xD4\xA1\x30"
			  "\x26\x9B\x89\xA1\xEE\x43\xE0\x52"
			  "\x55\x17\x4E\xC7\x0E\x33\x1F\xF1"
			  "\x9F\x8D\x40\x9F\x24\xFD\x92\xA0"
			  "\xBC\x8F\x35\xDD\x67\x38\xD8\xAA"
			  "\xCF\xF8\x48\xCA\xFB\xE4\x5C\x60"
			  "\x01\x41\x21\x12\x38\xAB\x52\x4F"
			  "\xA8\x57\x20\xE0\x21\x6A\x17\x0D"
			  "\x0E\xF9\x8E\x49\x42\x00\x3C\x94"
			  "\x14\xC0\xD0\x8D\x8A\x98\xEB\x29"
			  "\xEC\xAE\x96\x44\xC0\x3C\x48\xDC"
			  "\x29\x35\x25\x2F\xE7\x11\x6C\x68"
			  "\xC8\x67\x0A\x2F\xF4\x07\xBE\xF9"
			  "\x2C\x31\x87\x40\xAB\xB2\xB6\xFA"
			  "\xD2\xC9\x6D\x5C\x50\xE9\xE6\x7E"
			  "\xE3\x0A\xD2\xD5\x6D\x8D\x64\x9E"
			  "\x70\xCE\x03\x76\xDD\xE0\xF0\x8C"
			  "\x84\x86\x8B\x6A\xFE\xC7\xF9\x69"
			  "\x2E\xFE\xFC\xC2\xC4\x1A\x55\x58"
			  "\xB3\xBE\xE2\x7E\xED\x39\x42\x6C"
			  "\xB4\x42\x97\x9A\xEC\xE1\x0A\x06"
			  "\x02\xC5\x03\x9D\xC4\x48\x15\x66"
			  "\x35\x6A\xC2\xC9\xA2\x26\x30\xBB"
			  "\xDB\x2D\xC8\x08\x2B\xA0\x29\x1A"
			  "\x23\x61\x48\xEA\x80\x04\x27\xAA"
			  "\x69\x49\xE8\xE8\x4A\x83\x6B\x5A"
			  "\xCA\x7C\xD3\xB1\xB5\x0B\xCC\x23"
			  "\x74\x1F\xA9\x87\xCD\xED\xC0\x2D"
			  "\xBF\xEB\xCF\x16\x2D\x2A\x2E\x1D"
			  "\x96\xBA\x36\x11\x45\x41\xDA\xCE"
			  "\xA4\x48\x80\x8B\x06\xF4\x98\x89"
			  "\x8B\x23\x08\x53\xF4\xD4\x5A\x24"
			  "\x8B\xF8\x43\x73\xD1\xEE\xC4\xB0"
			  "\xF8\xFE\x09\x0C\x75\x05\x38\x0B"
			  "\x7C\x81\xDE\x9D\xE4\x61\x37\x63"
			  "\x63\xAD\x12\xD2\x04\xB9\xCE\x45"
			  "\x5A\x1A\x6E\xB3\x78\x2A\xA4\x74"
			  "\x86\xD0\xE3\xFF\xDA\x38\x9C\xB5"
			  "\xB8\xB1\xDB\x38\x2F\xC5\x6A\xB4"
			  "\xEB\x6E\x96\xE8\x43\x80\xB5\x51"
			  "\x61\x2D\x48\xAA\x07\x65\x11\x8C"
			  "\x48\xE3\x90\x7E\x78\x3A\xEC\x97"
			  "\x05\x3D\x84\xE7\x90\x2B\xAA\xBD"
			  "\x83\x29\x0E\x1A\x81\x73\x7B\xE0"
			  "\x7A\x01\x4A\x37\x3B\x77\x7F\x8D"
			  "\x49\xA4\x2F\x6E\xBE\x68\x99\x08"
			  "\x99\xAA\x4C\x12\x04\xAE\x1F\x77"
			  "\x35\x88\xF1\x65\x06\x0A\x0B\x4D"
			  "\x47\xF9\x50\x38\x5D\x71\xF9\x6E"
			  "\xDE\xEC\x61\x35\x2C\x4C\x96\x50"
			  "\xE8\x28\x93\x9C\x7E\x01\xC6\x04"
			  "\xB2\xD6\xBC\x6C\x17\xEB\xC1\x7D"
			  "\x11\xE9\x43\x83\x76\xAA\x53\x37"
			  "\x0C\x1D\x39\x89\x53\x72\x09\x7E"
			  "\xD9\x85\x16\x04\xA5\x2C\x05\x6F"
			  "\x17\x0C\x6E\x66\xAA\x84\xA7\xD9"
			  "\xE2\xD9\xC4\xEB\x43\x3E\xB1\x8D"
			  "\x7C\x36\xC7\x71\x70\x9C\x10\xD8"
			  "\xE8\x47\x2A\x4D\xFD\xA1\xBC\xE3"
			  "\xB9\x32\xE2\xC1\x82\xAC\xFE\xCC"
			  "\xC5\xC9\x7F\x9E\xCF\x33\x7A\xDF"
			  "\x6C\x82\x9D",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec tf_ctr_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xDF\xDD\x69\xFA\xB0\x2E\xFD\xFE"
			  "\x70\x9E\xC5\x4B\xC9\xD4\xA1\x30"
			  "\x26\x9B\x89\xA1\xEE\x43\xE0\x52"
			  "\x55\x17\x4E\xC7\x0E\x33\x1F\xF1"
			  "\x9F\x8D\x40\x9F\x24\xFD\x92\xA0"
			  "\xBC\x8F\x35\xDD\x67\x38\xD8\xAA"
			  "\xCF\xF8\x48\xCA\xFB\xE4\x5C\x60"
			  "\x01\x41\x21\x12\x38\xAB\x52\x4F"
			  "\xA8\x57\x20\xE0\x21\x6A\x17\x0D"
			  "\x0E\xF9\x8E\x49\x42\x00\x3C\x94"
			  "\x14\xC0\xD0\x8D\x8A\x98\xEB\x29"
			  "\xEC\xAE\x96\x44\xC0\x3C\x48\xDC"
			  "\x29\x35\x25\x2F\xE7\x11\x6C\x68"
			  "\xC8\x67\x0A\x2F\xF4\x07\xBE\xF9"
			  "\x2C\x31\x87\x40\xAB\xB2\xB6\xFA"
			  "\xD2\xC9\x6D\x5C\x50\xE9\xE6\x7E"
			  "\xE3\x0A\xD2\xD5\x6D\x8D\x64\x9E"
			  "\x70\xCE\x03\x76\xDD\xE0\xF0\x8C"
			  "\x84\x86\x8B\x6A\xFE\xC7\xF9\x69"
			  "\x2E\xFE\xFC\xC2\xC4\x1A\x55\x58"
			  "\xB3\xBE\xE2\x7E\xED\x39\x42\x6C"
			  "\xB4\x42\x97\x9A\xEC\xE1\x0A\x06"
			  "\x02\xC5\x03\x9D\xC4\x48\x15\x66"
			  "\x35\x6A\xC2\xC9\xA2\x26\x30\xBB"
			  "\xDB\x2D\xC8\x08\x2B\xA0\x29\x1A"
			  "\x23\x61\x48\xEA\x80\x04\x27\xAA"
			  "\x69\x49\xE8\xE8\x4A\x83\x6B\x5A"
			  "\xCA\x7C\xD3\xB1\xB5\x0B\xCC\x23"
			  "\x74\x1F\xA9\x87\xCD\xED\xC0\x2D"
			  "\xBF\xEB\xCF\x16\x2D\x2A\x2E\x1D"
			  "\x96\xBA\x36\x11\x45\x41\xDA\xCE"
			  "\xA4\x48\x80\x8B\x06\xF4\x98\x89"
			  "\x8B\x23\x08\x53\xF4\xD4\x5A\x24"
			  "\x8B\xF8\x43\x73\xD1\xEE\xC4\xB0"
			  "\xF8\xFE\x09\x0C\x75\x05\x38\x0B"
			  "\x7C\x81\xDE\x9D\xE4\x61\x37\x63"
			  "\x63\xAD\x12\xD2\x04\xB9\xCE\x45"
			  "\x5A\x1A\x6E\xB3\x78\x2A\xA4\x74"
			  "\x86\xD0\xE3\xFF\xDA\x38\x9C\xB5"
			  "\xB8\xB1\xDB\x38\x2F\xC5\x6A\xB4"
			  "\xEB\x6E\x96\xE8\x43\x80\xB5\x51"
			  "\x61\x2D\x48\xAA\x07\x65\x11\x8C"
			  "\x48\xE3\x90\x7E\x78\x3A\xEC\x97"
			  "\x05\x3D\x84\xE7\x90\x2B\xAA\xBD"
			  "\x83\x29\x0E\x1A\x81\x73\x7B\xE0"
			  "\x7A\x01\x4A\x37\x3B\x77\x7F\x8D"
			  "\x49\xA4\x2F\x6E\xBE\x68\x99\x08"
			  "\x99\xAA\x4C\x12\x04\xAE\x1F\x77"
			  "\x35\x88\xF1\x65\x06\x0A\x0B\x4D"
			  "\x47\xF9\x50\x38\x5D\x71\xF9\x6E"
			  "\xDE\xEC\x61\x35\x2C\x4C\x96\x50"
			  "\xE8\x28\x93\x9C\x7E\x01\xC6\x04"
			  "\xB2\xD6\xBC\x6C\x17\xEB\xC1\x7D"
			  "\x11\xE9\x43\x83\x76\xAA\x53\x37"
			  "\x0C\x1D\x39\x89\x53\x72\x09\x7E"
			  "\xD9\x85\x16\x04\xA5\x2C\x05\x6F"
			  "\x17\x0C\x6E\x66\xAA\x84\xA7\xD9"
			  "\xE2\xD9\xC4\xEB\x43\x3E\xB1\x8D"
			  "\x7C\x36\xC7\x71\x70\x9C\x10\xD8"
			  "\xE8\x47\x2A\x4D\xFD\xA1\xBC\xE3"
			  "\xB9\x32\xE2\xC1\x82\xAC\xFE\xCC"
			  "\xC5\xC9\x7F\x9E\xCF\x33\x7A\xDF",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\xEB\x44\xAF\x49\x27\xB8\xFB\x44"
			  "\x4C\xA6\xC3\x0C\x8B\xD0\x01\x0C"
			  "\x53\xC8\x16\x38\xDE\x40\x4F\x91"
			  "\x25\x6D\x4C\xA0\x9A\x87\x1E\xDA"
			  "\x88\x7E\x89\xE9\x67\x2B\x83\xA2"
			  "\x5F\x2E\x23\x3E\x45\xB9\x77\x7B"
			  "\xA6\x7E\x47\x36\x81\x9F\x9B\xF3"
			  "\xE0\xF0\xD7\x47\xA9\xC8\xEF\x33"
			  "\x0C\x43\xFE\x67\x50\x0A\x2C\x3E"
			  "\xA0\xE1\x25\x8E\x80\x07\x4A\xC0"
			  "\x64\x89\x9F\x6A\x27\x96\x07\xA6"
			  "\x9B\xC8\x1B\x21\x60\xAE\x5D\x01"
			  "\xE2\xCD\xC8\xAA\x6C\x9D\x1C\x34"
			  "\x39\x18\x09\xA4\x82\x59\x78\xE7"
			  "\xFC\x59\x65\xF2\x94\xFF\xFB\xE2"
			  "\x3C\xDA\xB1\x90\x95\xBF\x91\xE3"
			  "\xE6\x87\x31\x9E\x16\x85\xAD\xB1"
			  "\x4C\xAE\x43\x4D\x19\x58\xB5\x5E"
			  "\x2E\xF5\x09\xAA\x39\xF4\xC0\xB3"
			  "\xD4\x4D\xDB\x73\x7A\xD4\xF1\xBF"
			  "\x89\x16\x4D\x2D\xA2\x26\x33\x72"
			  "\x18\x33\x7E\xD6\xD2\x16\xA4\x54"
			  "\xF4\x8C\xB3\x52\xDF\x21\x9C\xEB"
			  "\xBF\x49\xD3\xF9\x05\x06\xCB\xD2"
			  "\xA9\xD2\x3B\x6E\x19\x8C\xBC\x19"
			  "\xAB\x89\xD6\xD8\xCD\x56\x89\x5E"
			  "\xAC\x00\xE3\x50\x63\x4A\x80\x9A"
			  "\x05\xBC\x50\x39\xD3\x32\xD9\x0D"
			  "\xE3\x20\x0D\x75\x54\xEC\xE6\x31"
			  "\x14\xB9\x3A\x59\x00\x43\x37\x8E"
			  "\x8C\x5A\x79\x62\x14\x76\x8A\xAE"
			  "\x8F\xCC\xA1\x6C\x38\x78\xDD\x2D"
			  "\x8B\x6D\xEA\xBD\x7B\x25\xFF\x60"
			  "\xC9\x87\xB1\x79\x1E\xA5\x86\x68"
			  "\x81\xB4\xE2\xC1\x05\x7D\x3A\x73"
			  "\xD0\xDA\x75\x77\x9E\x05\x27\xF1"
			  "\x08\xA9\x66\x64\x6C\xBC\x82\x17"
			  "\x2C\x23\x5F\x62\x4D\x02\x1A\x58"
			  "\xE7\xB7\x23\x6D\xE2\x20\xDA\xEF"
			  "\xB4\xB3\x3F\xB2\x2B\x69\x98\x83"
			  "\x95\x87\x13\x57\x60\xD7\xB5\xB1"
			  "\xEE\x0A\x2F\x95\x36\x4C\x76\x5D"
			  "\x5F\xD9\x19\xED\xB9\xA5\x48\xBF"
			  "\xC8\xAB\x0F\x71\xCC\x61\x8E\x0A"
			  "\xD0\x29\x44\xA8\xB9\xC1\xE8\xC8"
			  "\xC9\xA8\x28\x81\xFB\x50\xF2\xF0"
			  "\x26\xAE\x39\xB8\x91\xCD\xA8\xAC"
			  "\xDE\x55\x1B\x50\x14\x53\x44\x17"
			  "\x54\x46\xFC\xB1\xE4\x07\x6B\x9A"
			  "\x01\x14\xF0\x2E\x2E\xDB\x46\x1B"
			  "\x1A\x09\x97\xA9\xB6\x97\x79\x06"
			  "\xFB\xCB\x85\xCF\xDD\xA1\x41\xB1"
			  "\x00\xAA\xF7\xE0\x89\x73\xFB\xE5"
			  "\xBF\x84\xDB\xC9\xCD\xC4\xA2\x0D"
			  "\x3B\xAC\xF9\xDF\x96\xBF\x88\x23"
			  "\x41\x67\xA1\x24\x99\x7E\xCC\x9B"
			  "\x02\x8F\x6A\x49\xF6\x25\xBA\x7A"
			  "\xF4\x78\xFD\x79\x62\x63\x4F\x14"
			  "\xD6\x11\x11\x04\x05\x5F\x7E\xEA"
			  "\x4C\xB6\xF8\xF4\x5F\x48\x52\x54"
			  "\x94\x63\xA8\x4E\xCF\xD2\x1B\x1B"
			  "\x22\x18\x6A\xAF\x6E\x3E\xE1\x0D",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xDF\xDD\x69\xFA\xB0\x2E\xFD\xFE"
			  "\x70\x9E\xC5\x4B\xC9\xD4\xA1\x30"
			  "\x26\x9B\x89\xA1\xEE\x43\xE0\x52"
			  "\x55\x17\x4E\xC7\x0E\x33\x1F\xF1"
			  "\x9F\x8D\x40\x9F\x24\xFD\x92\xA0"
			  "\xBC\x8F\x35\xDD\x67\x38\xD8\xAA"
			  "\xCF\xF8\x48\xCA\xFB\xE4\x5C\x60"
			  "\x01\x41\x21\x12\x38\xAB\x52\x4F"
			  "\xA8\x57\x20\xE0\x21\x6A\x17\x0D"
			  "\x0E\xF9\x8E\x49\x42\x00\x3C\x94"
			  "\x14\xC0\xD0\x8D\x8A\x98\xEB\x29"
			  "\xEC\xAE\x96\x44\xC0\x3C\x48\xDC"
			  "\x29\x35\x25\x2F\xE7\x11\x6C\x68"
			  "\xC8\x67\x0A\x2F\xF4\x07\xBE\xF9"
			  "\x2C\x31\x87\x40\xAB\xB2\xB6\xFA"
			  "\xD2\xC9\x6D\x5C\x50\xE9\xE6\x7E"
			  "\xE3\x0A\xD2\xD5\x6D\x8D\x64\x9E"
			  "\x70\xCE\x03\x76\xDD\xE0\xF0\x8C"
			  "\x84\x86\x8B\x6A\xFE\xC7\xF9\x69"
			  "\x2E\xFE\xFC\xC2\xC4\x1A\x55\x58"
			  "\xB3\xBE\xE2\x7E\xED\x39\x42\x6C"
			  "\xB4\x42\x97\x9A\xEC\xE1\x0A\x06"
			  "\x02\xC5\x03\x9D\xC4\x48\x15\x66"
			  "\x35\x6A\xC2\xC9\xA2\x26\x30\xBB"
			  "\xDB\x2D\xC8\x08\x2B\xA0\x29\x1A"
			  "\x23\x61\x48\xEA\x80\x04\x27\xAA"
			  "\x69\x49\xE8\xE8\x4A\x83\x6B\x5A"
			  "\xCA\x7C\xD3\xB1\xB5\x0B\xCC\x23"
			  "\x74\x1F\xA9\x87\xCD\xED\xC0\x2D"
			  "\xBF\xEB\xCF\x16\x2D\x2A\x2E\x1D"
			  "\x96\xBA\x36\x11\x45\x41\xDA\xCE"
			  "\xA4\x48\x80\x8B\x06\xF4\x98\x89"
			  "\x8B\x23\x08\x53\xF4\xD4\x5A\x24"
			  "\x8B\xF8\x43\x73\xD1\xEE\xC4\xB0"
			  "\xF8\xFE\x09\x0C\x75\x05\x38\x0B"
			  "\x7C\x81\xDE\x9D\xE4\x61\x37\x63"
			  "\x63\xAD\x12\xD2\x04\xB9\xCE\x45"
			  "\x5A\x1A\x6E\xB3\x78\x2A\xA4\x74"
			  "\x86\xD0\xE3\xFF\xDA\x38\x9C\xB5"
			  "\xB8\xB1\xDB\x38\x2F\xC5\x6A\xB4"
			  "\xEB\x6E\x96\xE8\x43\x80\xB5\x51"
			  "\x61\x2D\x48\xAA\x07\x65\x11\x8C"
			  "\x48\xE3\x90\x7E\x78\x3A\xEC\x97"
			  "\x05\x3D\x84\xE7\x90\x2B\xAA\xBD"
			  "\x83\x29\x0E\x1A\x81\x73\x7B\xE0"
			  "\x7A\x01\x4A\x37\x3B\x77\x7F\x8D"
			  "\x49\xA4\x2F\x6E\xBE\x68\x99\x08"
			  "\x99\xAA\x4C\x12\x04\xAE\x1F\x77"
			  "\x35\x88\xF1\x65\x06\x0A\x0B\x4D"
			  "\x47\xF9\x50\x38\x5D\x71\xF9\x6E"
			  "\xDE\xEC\x61\x35\x2C\x4C\x96\x50"
			  "\xE8\x28\x93\x9C\x7E\x01\xC6\x04"
			  "\xB2\xD6\xBC\x6C\x17\xEB\xC1\x7D"
			  "\x11\xE9\x43\x83\x76\xAA\x53\x37"
			  "\x0C\x1D\x39\x89\x53\x72\x09\x7E"
			  "\xD9\x85\x16\x04\xA5\x2C\x05\x6F"
			  "\x17\x0C\x6E\x66\xAA\x84\xA7\xD9"
			  "\xE2\xD9\xC4\xEB\x43\x3E\xB1\x8D"
			  "\x7C\x36\xC7\x71\x70\x9C\x10\xD8"
			  "\xE8\x47\x2A\x4D\xFD\xA1\xBC\xE3"
			  "\xB9\x32\xE2\xC1\x82\xAC\xFE\xCC"
			  "\xC5\xC9\x7F\x9E\xCF\x33\x7A\xDF"
			  "\x6C\x82\x9D",
		.ilen	= 499,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec tf_lrw_enc_tv_template[] = {
	 
	{
		.key	= "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\xa1\x6c\x50\x69\x26\xa4\xef\x7b"
			  "\x7c\xc6\x91\xeb\x72\xdd\x9b\xee",
		.rlen	= 16,
	}, {
		.key	= "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\xab\x72\x0a\xad\x3b\x0c\xf0\xc9"
			  "\x42\x2f\xf1\xae\xf1\x3c\xb1\xbd",
		.rlen	= 16,
	}, {
		.key	= "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x85\xa7\x56\x67\x08\xfa\x42\xe1"
			  "\x22\xe6\x82\xfc\xd9\xb4\xd7\xd4",
		.rlen	= 16,
	}, {
		.key	= "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\xd2\xaf\x69\x35\x24\x1d\x0e\x1c"
			  "\x84\x8b\x05\xe4\xa2\x2f\x16\xf5",
		.rlen	= 16,
	}, {
		.key	= "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x4a\x23\x56\xd7\xff\x90\xd0\x9a"
			  "\x0d\x7c\x26\xfc\xf0\xf0\xf6\xe4",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x30\xaf\x26\x05\x9d\x5d\x0a\x58"
			  "\xe2\xe7\xce\x8a\xb2\x56\x6d\x76",
		.rlen	= 16,
	}, {
		.key	= "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\xdf\xcf\xdc\xd2\xe1\xcf\x86\x75"
			  "\x17\x66\x5e\x0c\x14\xa1\x3d\x40",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.ilen	= 512,
		.result	= "\x30\x38\xeb\xaf\x12\x43\x1a\x89"
			  "\x62\xa2\x36\xe5\xcf\x77\x1e\xd9"
			  "\x08\xc3\x0d\xdd\x95\xab\x19\x96"
			  "\x27\x52\x41\xc3\xca\xfb\xf6\xee"
			  "\x40\x2d\xdf\xdd\x00\x0c\xb9\x0a"
			  "\x3a\xf0\xc0\xd1\xda\x63\x9e\x45"
			  "\x42\xe9\x29\xc0\xb4\x07\xb4\x31"
			  "\x66\x77\x72\xb5\xb6\xb3\x57\x46"
			  "\x34\x9a\xfe\x03\xaf\x6b\x36\x07"
			  "\x63\x8e\xc2\x5d\xa6\x0f\xb6\x7d"
			  "\xfb\x6d\x82\x51\xb6\x98\xd0\x71"
			  "\xe7\x10\x7a\xdf\xb2\xbd\xf1\x1d"
			  "\x72\x2b\x54\x13\xe3\x6d\x79\x37"
			  "\xa9\x39\x2c\xdf\x21\xab\x87\xd5"
			  "\xee\xef\x9a\x12\x50\x39\x2e\x1b"
			  "\x7d\xe6\x6a\x27\x48\xb9\xe7\xac"
			  "\xaa\xcd\x79\x5f\xf2\xf3\xa0\x08"
			  "\x6f\x2c\xf4\x0e\xd1\xb8\x89\x25"
			  "\x31\x9d\xef\xb1\x1d\x27\x55\x04"
			  "\xc9\x8c\xb7\x68\xdc\xb6\x67\x8a"
			  "\xdb\xcf\x22\xf2\x3b\x6f\xce\xbb"
			  "\x26\xbe\x4f\x27\x04\x42\xd1\x44"
			  "\x4c\x08\xa3\x95\x4c\x7f\x1a\xaf"
			  "\x1d\x28\x14\xfd\xb1\x1a\x34\x18"
			  "\xf5\x1e\x28\x69\x95\x6a\x5a\xba"
			  "\x8e\xb2\x58\x1d\x28\x17\x13\x3d"
			  "\x38\x7d\x14\x8d\xab\x5d\xf9\xe8"
			  "\x3c\x0f\x2b\x0d\x2b\x08\xb4\x4b"
			  "\x6b\x0d\xc8\xa7\x84\xc2\x3a\x1a"
			  "\xb7\xbd\xda\x92\x29\xb8\x5b\x5a"
			  "\x63\xa5\x99\x82\x09\x72\x8f\xc6"
			  "\xa4\x62\x24\x69\x8c\x2d\x26\x00"
			  "\x99\x83\x91\xd6\xc6\xcf\x57\x67"
			  "\x38\xea\xf2\xfc\x29\xe0\x73\x39"
			  "\xf9\x13\x94\x6d\xe2\x58\x28\x75"
			  "\x3e\xae\x71\x90\x07\x70\x1c\x38"
			  "\x5b\x4c\x1e\xb5\xa5\x3b\x20\xef"
			  "\xb1\x4c\x3e\x1a\x72\x62\xbb\x22"
			  "\x82\x09\xe3\x18\x3f\x4f\x48\xfc"
			  "\xdd\xac\xfc\xb6\x09\xdb\xd2\x7b"
			  "\xd6\xb7\x7e\x41\x2f\x14\xf5\x0e"
			  "\xc3\xac\x4a\xed\xe7\x82\xef\x31"
			  "\x1f\x1a\x51\x1e\x29\x60\xc8\x98"
			  "\x93\x51\x1d\x3d\x62\x59\x83\x82"
			  "\x0c\xf1\xd7\x8d\xac\x33\x44\x81"
			  "\x3c\x59\xb7\xd4\x5b\x65\x82\xc4"
			  "\xec\xdc\x24\xfd\x0e\x1a\x79\x94"
			  "\x34\xb0\x62\xfa\x98\x49\x26\x1f"
			  "\xf4\x9e\x40\x44\x5b\x1f\xf8\xbe"
			  "\x36\xff\xc6\xc6\x9d\xf2\xd6\xcc"
			  "\x63\x93\x29\xb9\x0b\x6d\xd7\x6c"
			  "\xdb\xf6\x21\x80\xf7\x5a\x37\x15"
			  "\x0c\xe3\x36\xc8\x74\x75\x20\x91"
			  "\xdf\x52\x2d\x0c\xe7\x45\xff\x46"
			  "\xb3\xf4\xec\xc2\xbd\xd3\x37\xb6"
			  "\x26\xa2\x5d\x7d\x61\xbf\x10\x46"
			  "\x57\x8d\x05\x96\x70\x0b\xd6\x41"
			  "\x5c\xe9\xd3\x54\x81\x39\x3a\xdd"
			  "\x5f\x92\x81\x6e\x35\x03\xd4\x72"
			  "\x3d\x5a\xe7\xb9\x3b\x0c\x84\x23"
			  "\x45\x5d\xec\x72\xc1\x52\xef\x2e"
			  "\x81\x00\xd3\xfe\x4c\x3c\x05\x61"
			  "\x80\x18\xc4\x6c\x03\xd3\xb7\xba"
			  "\x11\xd7\xb8\x6e\xea\xe1\x80\x30",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec tf_lrw_dec_tv_template[] = {
	 
	{
		.key	= "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xa1\x6c\x50\x69\x26\xa4\xef\x7b"
			  "\x7c\xc6\x91\xeb\x72\xdd\x9b\xee",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input	= "\xab\x72\x0a\xad\x3b\x0c\xf0\xc9"
			  "\x42\x2f\xf1\xae\xf1\x3c\xb1\xbd",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x85\xa7\x56\x67\x08\xfa\x42\xe1"
			  "\x22\xe6\x82\xfc\xd9\xb4\xd7\xd4",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xd2\xaf\x69\x35\x24\x1d\x0e\x1c"
			  "\x84\x8b\x05\xe4\xa2\x2f\x16\xf5",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x4a\x23\x56\xd7\xff\x90\xd0\x9a"
			  "\x0d\x7c\x26\xfc\xf0\xf0\xf6\xe4",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\xaf\x26\x05\x9d\x5d\x0a\x58"
			  "\xe2\xe7\xce\x8a\xb2\x56\x6d\x76",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\xdf\xcf\xdc\xd2\xe1\xcf\x86\x75"
			  "\x17\x66\x5e\x0c\x14\xa1\x3d\x40",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x38\xeb\xaf\x12\x43\x1a\x89"
			  "\x62\xa2\x36\xe5\xcf\x77\x1e\xd9"
			  "\x08\xc3\x0d\xdd\x95\xab\x19\x96"
			  "\x27\x52\x41\xc3\xca\xfb\xf6\xee"
			  "\x40\x2d\xdf\xdd\x00\x0c\xb9\x0a"
			  "\x3a\xf0\xc0\xd1\xda\x63\x9e\x45"
			  "\x42\xe9\x29\xc0\xb4\x07\xb4\x31"
			  "\x66\x77\x72\xb5\xb6\xb3\x57\x46"
			  "\x34\x9a\xfe\x03\xaf\x6b\x36\x07"
			  "\x63\x8e\xc2\x5d\xa6\x0f\xb6\x7d"
			  "\xfb\x6d\x82\x51\xb6\x98\xd0\x71"
			  "\xe7\x10\x7a\xdf\xb2\xbd\xf1\x1d"
			  "\x72\x2b\x54\x13\xe3\x6d\x79\x37"
			  "\xa9\x39\x2c\xdf\x21\xab\x87\xd5"
			  "\xee\xef\x9a\x12\x50\x39\x2e\x1b"
			  "\x7d\xe6\x6a\x27\x48\xb9\xe7\xac"
			  "\xaa\xcd\x79\x5f\xf2\xf3\xa0\x08"
			  "\x6f\x2c\xf4\x0e\xd1\xb8\x89\x25"
			  "\x31\x9d\xef\xb1\x1d\x27\x55\x04"
			  "\xc9\x8c\xb7\x68\xdc\xb6\x67\x8a"
			  "\xdb\xcf\x22\xf2\x3b\x6f\xce\xbb"
			  "\x26\xbe\x4f\x27\x04\x42\xd1\x44"
			  "\x4c\x08\xa3\x95\x4c\x7f\x1a\xaf"
			  "\x1d\x28\x14\xfd\xb1\x1a\x34\x18"
			  "\xf5\x1e\x28\x69\x95\x6a\x5a\xba"
			  "\x8e\xb2\x58\x1d\x28\x17\x13\x3d"
			  "\x38\x7d\x14\x8d\xab\x5d\xf9\xe8"
			  "\x3c\x0f\x2b\x0d\x2b\x08\xb4\x4b"
			  "\x6b\x0d\xc8\xa7\x84\xc2\x3a\x1a"
			  "\xb7\xbd\xda\x92\x29\xb8\x5b\x5a"
			  "\x63\xa5\x99\x82\x09\x72\x8f\xc6"
			  "\xa4\x62\x24\x69\x8c\x2d\x26\x00"
			  "\x99\x83\x91\xd6\xc6\xcf\x57\x67"
			  "\x38\xea\xf2\xfc\x29\xe0\x73\x39"
			  "\xf9\x13\x94\x6d\xe2\x58\x28\x75"
			  "\x3e\xae\x71\x90\x07\x70\x1c\x38"
			  "\x5b\x4c\x1e\xb5\xa5\x3b\x20\xef"
			  "\xb1\x4c\x3e\x1a\x72\x62\xbb\x22"
			  "\x82\x09\xe3\x18\x3f\x4f\x48\xfc"
			  "\xdd\xac\xfc\xb6\x09\xdb\xd2\x7b"
			  "\xd6\xb7\x7e\x41\x2f\x14\xf5\x0e"
			  "\xc3\xac\x4a\xed\xe7\x82\xef\x31"
			  "\x1f\x1a\x51\x1e\x29\x60\xc8\x98"
			  "\x93\x51\x1d\x3d\x62\x59\x83\x82"
			  "\x0c\xf1\xd7\x8d\xac\x33\x44\x81"
			  "\x3c\x59\xb7\xd4\x5b\x65\x82\xc4"
			  "\xec\xdc\x24\xfd\x0e\x1a\x79\x94"
			  "\x34\xb0\x62\xfa\x98\x49\x26\x1f"
			  "\xf4\x9e\x40\x44\x5b\x1f\xf8\xbe"
			  "\x36\xff\xc6\xc6\x9d\xf2\xd6\xcc"
			  "\x63\x93\x29\xb9\x0b\x6d\xd7\x6c"
			  "\xdb\xf6\x21\x80\xf7\x5a\x37\x15"
			  "\x0c\xe3\x36\xc8\x74\x75\x20\x91"
			  "\xdf\x52\x2d\x0c\xe7\x45\xff\x46"
			  "\xb3\xf4\xec\xc2\xbd\xd3\x37\xb6"
			  "\x26\xa2\x5d\x7d\x61\xbf\x10\x46"
			  "\x57\x8d\x05\x96\x70\x0b\xd6\x41"
			  "\x5c\xe9\xd3\x54\x81\x39\x3a\xdd"
			  "\x5f\x92\x81\x6e\x35\x03\xd4\x72"
			  "\x3d\x5a\xe7\xb9\x3b\x0c\x84\x23"
			  "\x45\x5d\xec\x72\xc1\x52\xef\x2e"
			  "\x81\x00\xd3\xfe\x4c\x3c\x05\x61"
			  "\x80\x18\xc4\x6c\x03\xd3\xb7\xba"
			  "\x11\xd7\xb8\x6e\xea\xe1\x80\x30",
		.ilen	= 512,
		.result	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec tf_xts_enc_tv_template[] = {
	 
{
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 32,
		.result	= "\x4b\xc9\x44\x4a\x11\xa3\xef\xac"
			  "\x30\x74\xe4\x44\x52\x77\x97\x43"
			  "\xa7\x60\xb2\x45\x2e\xf9\x00\x90"
			  "\x9f\xaa\xfd\x89\x6e\x9d\x4a\xe0",
		.rlen	= 32,
	}, {
		.key	= "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen	= 32,
		.result	= "\x57\x0e\x8f\xe5\x2a\x35\x61\x4f"
			  "\x32\xd3\xbd\x36\x05\x15\x44\x2c"
			  "\x58\x06\xf7\xf8\x00\xa8\xb6\xd5"
			  "\xc6\x28\x92\xdb\xd8\x34\xa2\xe9",
		.rlen	= 32,
	}, {
		.key	= "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen	= 32,
		.result	= "\x96\x45\x8f\x8d\x7a\x75\xb1\xde"
			  "\x40\x0c\x89\x56\xf6\x4d\xa7\x07"
			  "\x38\xbb\x5b\xe9\xcd\x84\xae\xb2"
			  "\x7b\x6a\x62\xf4\x8c\xb5\x37\xea",
		.rlen	= 32,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\xa9\x78\xae\x1e\xea\xa2\x44\x4c"
			  "\xa2\x7a\x64\x1f\xaf\x46\xc1\xe0"
			  "\x6c\xb2\xf3\x92\x9a\xd6\x7d\x58"
			  "\xb8\x2d\xb9\x5d\x58\x07\x66\x50"
			  "\xea\x35\x35\x8c\xb2\x46\x61\x06"
			  "\x5d\x65\xfc\x57\x8f\x69\x74\xab"
			  "\x8a\x06\x69\xb5\x6c\xda\x66\xc7"
			  "\x52\x90\xbb\x8e\x6d\x8b\xb5\xa2"
			  "\x78\x1d\xc2\xa9\xc2\x73\x00\xc3"
			  "\x32\x36\x7c\x97\x6b\x4e\x8a\x50"
			  "\xe4\x91\x83\x96\x8f\xf4\x94\x1a"
			  "\xa6\x27\xe1\x33\xcb\x91\xc6\x5f"
			  "\x94\x75\xbc\xd7\x3e\x3e\x6f\x9e"
			  "\xa9\x31\x80\x5e\xe5\xdb\xc8\x53"
			  "\x01\x73\x68\x32\x25\x19\xfa\xfb"
			  "\xe4\xcf\xb9\x3e\xa2\xa0\x8f\x31"
			  "\xbf\x54\x06\x93\xa8\xb1\x0f\xb6"
			  "\x7c\x3c\xde\x6f\x0f\xfb\x0c\x11"
			  "\x39\x80\x39\x09\x97\x65\xf2\x83"
			  "\xae\xe6\xa1\x6f\x47\xb8\x49\xde"
			  "\x99\x36\x20\x7d\x97\x3b\xec\xfa"
			  "\xb4\x33\x6e\x7a\xc7\x46\x84\x49"
			  "\x91\xcd\xe1\x57\x0d\xed\x40\x08"
			  "\x13\xf1\x4e\x3e\xa4\xa4\x5c\xe6"
			  "\xd2\x0c\x20\x8f\x3e\xdf\x3f\x47"
			  "\x9a\x2f\xde\x6d\x66\xc9\x99\x4a"
			  "\x2d\x9e\x9d\x4b\x1a\x27\xa2\x12"
			  "\x99\xf0\xf8\xb1\xb6\xf6\x57\xc3"
			  "\xca\x1c\xa3\x8e\xed\x39\x28\xb5"
			  "\x10\x1b\x4b\x08\x42\x00\x4a\xd3"
			  "\xad\x5a\xc6\x8e\xc8\xbb\x95\xc4"
			  "\x4b\xaa\xfe\xd5\x42\xa8\xa3\x6d"
			  "\x3c\xf3\x34\x91\x2d\xb4\xdd\x20"
			  "\x0c\x90\x6d\xa3\x9b\x66\x9d\x24"
			  "\x02\xa6\xa9\x3f\x3f\x58\x5d\x47"
			  "\x24\x65\x63\x7e\xbd\x8c\xe6\x52"
			  "\x7d\xef\x33\x53\x63\xec\xaa\x0b"
			  "\x64\x15\xa9\xa6\x1f\x10\x00\x38"
			  "\x35\xa8\xe7\xbe\x23\x70\x22\xe0"
			  "\xd3\xb9\xe6\xfd\xe6\xaa\x03\x50"
			  "\xf3\x3c\x27\x36\x8b\xcc\xfe\x9c"
			  "\x9c\xa3\xb3\xe7\x68\x9b\xa2\x71"
			  "\xe0\x07\xd9\x1f\x68\x1f\xac\x5e"
			  "\x7a\x74\x85\xa9\x6a\x90\xab\x2c"
			  "\x38\x51\xbc\x1f\x43\x4a\x56\x1c"
			  "\xf8\x47\x03\x4e\x67\xa8\x1f\x99"
			  "\x04\x39\x73\x32\xb2\x86\x79\xe7"
			  "\x14\x28\x70\xb8\xe2\x7d\x69\x85"
			  "\xb6\x0f\xc5\xd0\xd0\x01\x5c\xe6"
			  "\x09\x0f\x75\xf7\xb6\x81\xd2\x11"
			  "\x20\x9c\xa1\xee\x11\x44\x79\xd0"
			  "\xb2\x34\x77\xda\x10\x9a\x6f\x6f"
			  "\xef\x7c\xd9\xdc\x35\xb7\x61\xdd"
			  "\xf1\xa4\xc6\x1c\xbf\x05\x22\xac"
			  "\xfe\x2f\x85\x00\x44\xdf\x33\x16"
			  "\x35\xb6\xa3\xd3\x70\xdf\x69\x35"
			  "\x6a\xc7\xb4\x99\x45\x27\xc8\x8e"
			  "\x5a\x14\x30\xd0\x55\x3e\x4f\x64"
			  "\x0d\x38\xe3\xdf\x8b\xa8\x93\x26"
			  "\x75\xae\xf6\xb5\x23\x0b\x17\x31"
			  "\xbf\x27\xb8\xb5\x94\x31\xa7\x8f"
			  "\x43\xc4\x46\x24\x22\x4f\x8f\x7e"
			  "\xe5\xf4\x6d\x1e\x0e\x18\x7a\xbb"
			  "\xa6\x8f\xfb\x49\x49\xd8\x7e\x5a",
		.rlen	= 512,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\xd7\x4b\x93\x7d\x13\xa2\xa2\xe1"
			  "\x35\x39\x71\x88\x76\x1e\xc9\xea"
			  "\x86\xad\xf3\x14\x48\x3d\x5e\xe9"
			  "\xe9\x2d\xb2\x56\x59\x35\x9d\xec"
			  "\x84\xfa\x7e\x9d\x6d\x33\x36\x8f"
			  "\xce\xf4\xa9\x21\x0b\x5f\x96\xec"
			  "\xcb\xf9\x57\x68\x33\x88\x39\xbf"
			  "\x2f\xbb\x59\x03\xbd\x66\x8b\x11"
			  "\x11\x65\x51\x2e\xb8\x67\x05\xd1"
			  "\x27\x11\x5c\xd4\xcc\x97\xc2\xb3"
			  "\xa9\x55\xaf\x07\x56\xd1\xdc\xf5"
			  "\x85\xdc\x46\xe6\xf0\x24\xeb\x93"
			  "\x4d\xf0\x9b\xf5\x73\x1c\xda\x03"
			  "\x22\xc8\x3a\x4f\xb4\x19\x91\x09"
			  "\x54\x0b\xf6\xfe\x17\x3d\x1a\x53"
			  "\x72\x60\x79\xcb\x0e\x32\x8a\x77"
			  "\xd5\xed\xdb\x33\xd7\x62\x16\x69"
			  "\x63\xe0\xab\xb5\xf6\x9c\x5f\x3d"
			  "\x69\x35\x61\x86\xf8\x86\xb9\x89"
			  "\x6e\x59\x35\xac\xf6\x6b\x33\xa0"
			  "\xea\xef\x96\x62\xd8\xa9\xcf\x56"
			  "\xbf\xdb\x8a\xfd\xa1\x82\x77\x73"
			  "\x3d\x94\x4a\x49\x42\x6d\x08\x60"
			  "\xa1\xea\xab\xb6\x88\x13\x94\xb8"
			  "\x51\x98\xdb\x35\x85\xdf\xf6\xb9"
			  "\x8f\xcd\xdf\x80\xd3\x40\x2d\x72"
			  "\xb8\xb2\x6c\x02\x43\x35\x22\x2a"
			  "\x31\xed\xcd\x16\x19\xdf\x62\x0f"
			  "\x29\xcf\x87\x04\xec\x02\x4f\xe4"
			  "\xa2\xed\x73\xc6\x69\xd3\x7e\x89"
			  "\x0b\x76\x10\x7c\xd6\xf9\x6a\x25"
			  "\xed\xcc\x60\x5d\x61\x20\xc1\x97"
			  "\x56\x91\x57\x28\xbe\x71\x0d\xcd"
			  "\xde\xc4\x9e\x55\x91\xbe\xd1\x28"
			  "\x9b\x90\xeb\x73\xf3\x68\x51\xc6"
			  "\xdf\x82\xcc\xd8\x1f\xce\x5b\x27"
			  "\xc0\x60\x5e\x33\xd6\xa7\x20\xea"
			  "\xb2\x54\xc7\x5d\x6a\x3b\x67\x47"
			  "\xcf\xa0\xe3\xab\x86\xaf\xc1\x42"
			  "\xe6\xb0\x23\x4a\xaf\x53\xdf\xa0"
			  "\xad\x12\x32\x31\x03\xf7\x21\xbe"
			  "\x2d\xd5\x82\x42\xb6\x4a\x3d\xcd"
			  "\xd8\x81\x77\xa9\x49\x98\x6c\x09"
			  "\xc5\xa3\x61\x12\x62\x85\x6b\xcd"
			  "\xb3\xf4\x20\x0c\x41\xc4\x05\x37"
			  "\x46\x5f\xeb\x71\x8b\xf1\xaf\x6e"
			  "\xba\xf3\x50\x2e\xfe\xa8\x37\xeb"
			  "\xe8\x8c\x4f\xa4\x0c\xf1\x31\xc8"
			  "\x6e\x71\x4f\xa5\xd7\x97\x73\xe0"
			  "\x93\x4a\x2f\xda\x7b\xe0\x20\x54"
			  "\x1f\x8d\x85\x79\x0b\x7b\x5e\x75"
			  "\xb9\x07\x67\xcc\xc8\xe7\x21\x15"
			  "\xa7\xc8\x98\xff\x4b\x80\x1c\x12"
			  "\xa8\x54\xe1\x38\x52\xe6\x74\x81"
			  "\x97\x47\xa1\x41\x0e\xc0\x50\xe3"
			  "\x55\x0e\xc3\xa7\x70\x77\xce\x07"
			  "\xed\x8c\x88\xe6\xa1\x5b\x14\xec"
			  "\xe6\xde\x06\x6d\x74\xc5\xd9\xfa"
			  "\xe5\x2f\x5a\xff\xc8\x05\xee\x27"
			  "\x35\x61\xbf\x0b\x19\x78\x9b\xd2"
			  "\x04\xc7\x05\xb1\x79\xb4\xff\x5f"
			  "\xf3\xea\x67\x52\x78\xc2\xce\x70"
			  "\xa4\x05\x0b\xb2\xb3\xa8\x30\x97"
			  "\x37\x30\xe1\x91\x8d\xb3\x2a\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec tf_xts_dec_tv_template[] = {
	 
	{
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x4b\xc9\x44\x4a\x11\xa3\xef\xac"
			  "\x30\x74\xe4\x44\x52\x77\x97\x43"
			  "\xa7\x60\xb2\x45\x2e\xf9\x00\x90"
			  "\x9f\xaa\xfd\x89\x6e\x9d\x4a\xe0",
		.ilen	= 32,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 32,
	}, {
		.key	= "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x57\x0e\x8f\xe5\x2a\x35\x61\x4f"
			  "\x32\xd3\xbd\x36\x05\x15\x44\x2c"
			  "\x58\x06\xf7\xf8\x00\xa8\xb6\xd5"
			  "\xc6\x28\x92\xdb\xd8\x34\xa2\xe9",
		.ilen	= 32,
		.result	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen	= 32,
	}, {
		.key	= "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x96\x45\x8f\x8d\x7a\x75\xb1\xde"
			  "\x40\x0c\x89\x56\xf6\x4d\xa7\x07"
			  "\x38\xbb\x5b\xe9\xcd\x84\xae\xb2"
			  "\x7b\x6a\x62\xf4\x8c\xb5\x37\xea",
		.ilen	= 32,
		.result	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen	= 32,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xa9\x78\xae\x1e\xea\xa2\x44\x4c"
			  "\xa2\x7a\x64\x1f\xaf\x46\xc1\xe0"
			  "\x6c\xb2\xf3\x92\x9a\xd6\x7d\x58"
			  "\xb8\x2d\xb9\x5d\x58\x07\x66\x50"
			  "\xea\x35\x35\x8c\xb2\x46\x61\x06"
			  "\x5d\x65\xfc\x57\x8f\x69\x74\xab"
			  "\x8a\x06\x69\xb5\x6c\xda\x66\xc7"
			  "\x52\x90\xbb\x8e\x6d\x8b\xb5\xa2"
			  "\x78\x1d\xc2\xa9\xc2\x73\x00\xc3"
			  "\x32\x36\x7c\x97\x6b\x4e\x8a\x50"
			  "\xe4\x91\x83\x96\x8f\xf4\x94\x1a"
			  "\xa6\x27\xe1\x33\xcb\x91\xc6\x5f"
			  "\x94\x75\xbc\xd7\x3e\x3e\x6f\x9e"
			  "\xa9\x31\x80\x5e\xe5\xdb\xc8\x53"
			  "\x01\x73\x68\x32\x25\x19\xfa\xfb"
			  "\xe4\xcf\xb9\x3e\xa2\xa0\x8f\x31"
			  "\xbf\x54\x06\x93\xa8\xb1\x0f\xb6"
			  "\x7c\x3c\xde\x6f\x0f\xfb\x0c\x11"
			  "\x39\x80\x39\x09\x97\x65\xf2\x83"
			  "\xae\xe6\xa1\x6f\x47\xb8\x49\xde"
			  "\x99\x36\x20\x7d\x97\x3b\xec\xfa"
			  "\xb4\x33\x6e\x7a\xc7\x46\x84\x49"
			  "\x91\xcd\xe1\x57\x0d\xed\x40\x08"
			  "\x13\xf1\x4e\x3e\xa4\xa4\x5c\xe6"
			  "\xd2\x0c\x20\x8f\x3e\xdf\x3f\x47"
			  "\x9a\x2f\xde\x6d\x66\xc9\x99\x4a"
			  "\x2d\x9e\x9d\x4b\x1a\x27\xa2\x12"
			  "\x99\xf0\xf8\xb1\xb6\xf6\x57\xc3"
			  "\xca\x1c\xa3\x8e\xed\x39\x28\xb5"
			  "\x10\x1b\x4b\x08\x42\x00\x4a\xd3"
			  "\xad\x5a\xc6\x8e\xc8\xbb\x95\xc4"
			  "\x4b\xaa\xfe\xd5\x42\xa8\xa3\x6d"
			  "\x3c\xf3\x34\x91\x2d\xb4\xdd\x20"
			  "\x0c\x90\x6d\xa3\x9b\x66\x9d\x24"
			  "\x02\xa6\xa9\x3f\x3f\x58\x5d\x47"
			  "\x24\x65\x63\x7e\xbd\x8c\xe6\x52"
			  "\x7d\xef\x33\x53\x63\xec\xaa\x0b"
			  "\x64\x15\xa9\xa6\x1f\x10\x00\x38"
			  "\x35\xa8\xe7\xbe\x23\x70\x22\xe0"
			  "\xd3\xb9\xe6\xfd\xe6\xaa\x03\x50"
			  "\xf3\x3c\x27\x36\x8b\xcc\xfe\x9c"
			  "\x9c\xa3\xb3\xe7\x68\x9b\xa2\x71"
			  "\xe0\x07\xd9\x1f\x68\x1f\xac\x5e"
			  "\x7a\x74\x85\xa9\x6a\x90\xab\x2c"
			  "\x38\x51\xbc\x1f\x43\x4a\x56\x1c"
			  "\xf8\x47\x03\x4e\x67\xa8\x1f\x99"
			  "\x04\x39\x73\x32\xb2\x86\x79\xe7"
			  "\x14\x28\x70\xb8\xe2\x7d\x69\x85"
			  "\xb6\x0f\xc5\xd0\xd0\x01\x5c\xe6"
			  "\x09\x0f\x75\xf7\xb6\x81\xd2\x11"
			  "\x20\x9c\xa1\xee\x11\x44\x79\xd0"
			  "\xb2\x34\x77\xda\x10\x9a\x6f\x6f"
			  "\xef\x7c\xd9\xdc\x35\xb7\x61\xdd"
			  "\xf1\xa4\xc6\x1c\xbf\x05\x22\xac"
			  "\xfe\x2f\x85\x00\x44\xdf\x33\x16"
			  "\x35\xb6\xa3\xd3\x70\xdf\x69\x35"
			  "\x6a\xc7\xb4\x99\x45\x27\xc8\x8e"
			  "\x5a\x14\x30\xd0\x55\x3e\x4f\x64"
			  "\x0d\x38\xe3\xdf\x8b\xa8\x93\x26"
			  "\x75\xae\xf6\xb5\x23\x0b\x17\x31"
			  "\xbf\x27\xb8\xb5\x94\x31\xa7\x8f"
			  "\x43\xc4\x46\x24\x22\x4f\x8f\x7e"
			  "\xe5\xf4\x6d\x1e\x0e\x18\x7a\xbb"
			  "\xa6\x8f\xfb\x49\x49\xd8\x7e\x5a",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xd7\x4b\x93\x7d\x13\xa2\xa2\xe1"
			  "\x35\x39\x71\x88\x76\x1e\xc9\xea"
			  "\x86\xad\xf3\x14\x48\x3d\x5e\xe9"
			  "\xe9\x2d\xb2\x56\x59\x35\x9d\xec"
			  "\x84\xfa\x7e\x9d\x6d\x33\x36\x8f"
			  "\xce\xf4\xa9\x21\x0b\x5f\x96\xec"
			  "\xcb\xf9\x57\x68\x33\x88\x39\xbf"
			  "\x2f\xbb\x59\x03\xbd\x66\x8b\x11"
			  "\x11\x65\x51\x2e\xb8\x67\x05\xd1"
			  "\x27\x11\x5c\xd4\xcc\x97\xc2\xb3"
			  "\xa9\x55\xaf\x07\x56\xd1\xdc\xf5"
			  "\x85\xdc\x46\xe6\xf0\x24\xeb\x93"
			  "\x4d\xf0\x9b\xf5\x73\x1c\xda\x03"
			  "\x22\xc8\x3a\x4f\xb4\x19\x91\x09"
			  "\x54\x0b\xf6\xfe\x17\x3d\x1a\x53"
			  "\x72\x60\x79\xcb\x0e\x32\x8a\x77"
			  "\xd5\xed\xdb\x33\xd7\x62\x16\x69"
			  "\x63\xe0\xab\xb5\xf6\x9c\x5f\x3d"
			  "\x69\x35\x61\x86\xf8\x86\xb9\x89"
			  "\x6e\x59\x35\xac\xf6\x6b\x33\xa0"
			  "\xea\xef\x96\x62\xd8\xa9\xcf\x56"
			  "\xbf\xdb\x8a\xfd\xa1\x82\x77\x73"
			  "\x3d\x94\x4a\x49\x42\x6d\x08\x60"
			  "\xa1\xea\xab\xb6\x88\x13\x94\xb8"
			  "\x51\x98\xdb\x35\x85\xdf\xf6\xb9"
			  "\x8f\xcd\xdf\x80\xd3\x40\x2d\x72"
			  "\xb8\xb2\x6c\x02\x43\x35\x22\x2a"
			  "\x31\xed\xcd\x16\x19\xdf\x62\x0f"
			  "\x29\xcf\x87\x04\xec\x02\x4f\xe4"
			  "\xa2\xed\x73\xc6\x69\xd3\x7e\x89"
			  "\x0b\x76\x10\x7c\xd6\xf9\x6a\x25"
			  "\xed\xcc\x60\x5d\x61\x20\xc1\x97"
			  "\x56\x91\x57\x28\xbe\x71\x0d\xcd"
			  "\xde\xc4\x9e\x55\x91\xbe\xd1\x28"
			  "\x9b\x90\xeb\x73\xf3\x68\x51\xc6"
			  "\xdf\x82\xcc\xd8\x1f\xce\x5b\x27"
			  "\xc0\x60\x5e\x33\xd6\xa7\x20\xea"
			  "\xb2\x54\xc7\x5d\x6a\x3b\x67\x47"
			  "\xcf\xa0\xe3\xab\x86\xaf\xc1\x42"
			  "\xe6\xb0\x23\x4a\xaf\x53\xdf\xa0"
			  "\xad\x12\x32\x31\x03\xf7\x21\xbe"
			  "\x2d\xd5\x82\x42\xb6\x4a\x3d\xcd"
			  "\xd8\x81\x77\xa9\x49\x98\x6c\x09"
			  "\xc5\xa3\x61\x12\x62\x85\x6b\xcd"
			  "\xb3\xf4\x20\x0c\x41\xc4\x05\x37"
			  "\x46\x5f\xeb\x71\x8b\xf1\xaf\x6e"
			  "\xba\xf3\x50\x2e\xfe\xa8\x37\xeb"
			  "\xe8\x8c\x4f\xa4\x0c\xf1\x31\xc8"
			  "\x6e\x71\x4f\xa5\xd7\x97\x73\xe0"
			  "\x93\x4a\x2f\xda\x7b\xe0\x20\x54"
			  "\x1f\x8d\x85\x79\x0b\x7b\x5e\x75"
			  "\xb9\x07\x67\xcc\xc8\xe7\x21\x15"
			  "\xa7\xc8\x98\xff\x4b\x80\x1c\x12"
			  "\xa8\x54\xe1\x38\x52\xe6\x74\x81"
			  "\x97\x47\xa1\x41\x0e\xc0\x50\xe3"
			  "\x55\x0e\xc3\xa7\x70\x77\xce\x07"
			  "\xed\x8c\x88\xe6\xa1\x5b\x14\xec"
			  "\xe6\xde\x06\x6d\x74\xc5\xd9\xfa"
			  "\xe5\x2f\x5a\xff\xc8\x05\xee\x27"
			  "\x35\x61\xbf\x0b\x19\x78\x9b\xd2"
			  "\x04\xc7\x05\xb1\x79\xb4\xff\x5f"
			  "\xf3\xea\x67\x52\x78\xc2\xce\x70"
			  "\xa4\x05\x0b\xb2\xb3\xa8\x30\x97"
			  "\x37\x30\xe1\x91\x8d\xb3\x2a\xff",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

#define SERPENT_ENC_TEST_VECTORS	5
#define SERPENT_DEC_TEST_VECTORS	5

#define TNEPRES_ENC_TEST_VECTORS	4
#define TNEPRES_DEC_TEST_VECTORS	4

#define SERPENT_CBC_ENC_TEST_VECTORS	1
#define SERPENT_CBC_DEC_TEST_VECTORS	1

#define SERPENT_CTR_ENC_TEST_VECTORS	2
#define SERPENT_CTR_DEC_TEST_VECTORS	2

#define SERPENT_LRW_ENC_TEST_VECTORS	8
#define SERPENT_LRW_DEC_TEST_VECTORS	8

#define SERPENT_XTS_ENC_TEST_VECTORS	5
#define SERPENT_XTS_DEC_TEST_VECTORS	5

static struct cipher_testvec serpent_enc_tv_template[] = {
	{
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.ilen	= 16,
		.result	= "\x12\x07\xfc\xce\x9b\xd0\xd6\x47"
			  "\x6a\xe9\x8f\xbe\xd1\x43\xa0\xe2",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.ilen	= 16,
		.result	= "\x4c\x7d\x8a\x32\x80\x72\xa2\x2c"
			  "\x82\x3e\x4a\x1f\x3a\xcd\xa1\x6d",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.ilen	= 16,
		.result	= "\xde\x26\x9f\xf8\x33\xe4\x32\xb8"
			  "\x5b\x2e\x88\xd2\x70\x1c\xe7\x5c",
		.rlen	= 16,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80",
		.klen	= 16,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\xdd\xd2\x6b\x98\xa5\xff\xd8\x2c"
			  "\x05\x34\x5a\x9d\xad\xbf\xaf\x49",
		.rlen	= 16,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xFB\xB0\x5D\xDE\xC0\xFE\xFC\xEB"
			  "\xB1\x80\x10\x43\xDE\x62\x70\xBD"
			  "\xFA\x8A\x93\xEA\x6B\xF7\xC5\xD7"
			  "\x0C\xD1\xBB\x29\x25\x14\x4C\x22"
			  "\x77\xA6\x38\x00\xDB\xB9\xE2\x07"
			  "\xD1\xAC\x82\xBA\xEA\x67\xAA\x39"
			  "\x99\x34\x89\x5B\x54\xE9\x12\x13"
			  "\x3B\x04\xE5\x12\x42\xC5\x79\xAB"
			  "\x0D\xC7\x3C\x58\x2D\xA3\x98\xF6"
			  "\xE4\x61\x9E\x17\x0B\xCE\xE8\xAA"
			  "\xB5\x6C\x1A\x3A\x67\x52\x81\x6A"
			  "\x04\xFF\x8A\x1B\x96\xFE\xE6\x87"
			  "\x3C\xD4\x39\x7D\x36\x9B\x03\xD5"
			  "\xB6\xA0\x75\x3C\x83\xE6\x1C\x73"
			  "\x9D\x74\x2B\x77\x53\x2D\xE5\xBD"
			  "\x69\xDA\x7A\x01\xF5\x6A\x70\x39"
			  "\x30\xD4\x2C\xF2\x8E\x06\x4B\x39"
			  "\xB3\x12\x1D\xB3\x17\x46\xE6\xD6"
			  "\xB6\x31\x36\x34\x38\x3C\x1D\x69"
			  "\x9F\x47\x28\x9A\x1D\x96\x70\x54"
			  "\x8E\x88\xCB\xE0\xF5\x6A\xAE\x0A"
			  "\x3C\xD5\x93\x1C\x21\xC9\x14\x3A"
			  "\x23\x9C\x9B\x79\xC7\x75\xC8\x39"
			  "\xA6\xAC\x65\x9A\x99\x37\xAF\x6D"
			  "\xBD\xB5\x32\xFD\xD8\x9C\x95\x7B"
			  "\xC6\x6A\x80\x64\xEA\xEF\x6D\x3F"
			  "\xA9\xFE\x5B\x16\xA3\xCF\x32\xC8"
			  "\xEF\x50\x22\x20\x93\x30\xBE\xE2"
			  "\x38\x05\x65\xAF\xBA\xB6\xE4\x72"
			  "\xA9\xEE\x05\x42\x88\xBD\x9D\x49"
			  "\xAD\x93\xCA\x4D\x45\x11\x43\x4D"
			  "\xB8\xF5\x74\x2B\x48\xE7\x21\xE4"
			  "\x4E\x3A\x4C\xDE\x65\x7A\x5A\xAD"
			  "\x86\xE6\x23\xEC\x6B\xA7\x17\xE6"
			  "\xF6\xA1\xAC\x29\xAE\xF9\x9B\x69"
			  "\x73\x65\x65\x51\xD6\x0B\x4E\x8C"
			  "\x17\x15\x9D\xB0\xCF\xB2\x42\x2B"
			  "\x51\xC3\x03\xE8\xB7\x7D\x2D\x39"
			  "\xE8\x10\x93\x16\xC8\x68\x4C\x60"
			  "\x87\x70\x14\xD0\x01\x57\xCB\x42"
			  "\x13\x59\xB1\x7F\x12\x4F\xBB\xC7"
			  "\xBD\x2B\xD4\xA9\x12\x26\x4F\xDE"
			  "\xFD\x72\xEC\xD7\x6F\x97\x14\x90"
			  "\x0E\x37\x13\xE6\x67\x1D\xE5\xFE"
			  "\x9E\x18\x3C\x8F\x3A\x3F\x59\x9B"
			  "\x71\x80\x05\x35\x3F\x40\x0B\x21"
			  "\x76\xE5\xEF\x42\x6C\xDB\x31\x05"
			  "\x5F\x05\xCF\x14\xE3\xF0\x61\xA2"
			  "\x49\x03\x5E\x77\x2E\x20\xBA\xA1"
			  "\xAF\x46\x51\xC0\x2B\xC4\x64\x1E"
			  "\x65\xCC\x51\x58\x0A\xDF\xF0\x5F"
			  "\x75\x9F\x48\xCD\x81\xEC\xC3\xF6"
			  "\xED\xC9\x4B\x7B\x4E\x26\x23\xE1"
			  "\xBB\xE9\x83\x0B\xCF\xE4\xDE\x00"
			  "\x48\xFF\xBF\x6C\xB4\x72\x16\xEF"
			  "\xC7\x46\xEE\x48\x8C\xB8\xAF\x45"
			  "\x91\x76\xE7\x6E\x65\x3D\x15\x86"
			  "\x10\xF8\xDB\x66\x97\x7C\x43\x4D"
			  "\x79\x12\x4E\xCE\x06\xD1\xD1\x6A"
			  "\x34\xC1\xC9\xF2\x28\x4A\xCD\x02"
			  "\x75\x55\x9B\xFF\x36\x73\xAB\x7C"
			  "\xF4\x46\x2E\xEB\xAC\xF3\xD2\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec tnepres_enc_tv_template[] = {
	{  
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.key    = "\x80\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen   = 16,
		.ilen	= 16,
		.result	= "\x49\xaf\xbf\xad\x9d\x5a\x34\x05"
			  "\x2c\xd8\xff\xa5\x98\x6b\xd2\xdd",
		.rlen	= 16,
	}, {  
		.key	= "\x80\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 24,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 16,
		.result	= "\xe7\x8e\x54\x02\xc7\x19\x55\x68"
			  "\xac\x36\x78\xf7\xa3\xf6\x0c\x66",
		.rlen	= 16,
	}, {  
		.key	= "\x80\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 16,
		.result	= "\xab\xed\x96\xe7\x66\xbf\x28\xcb"
			  "\xc0\xeb\xd2\x1a\x82\xef\x08\x19",
		.rlen	= 16,
	}, {  
		.key	= "\x1f\x1e\x1d\x1c\x1b\x1a\x19\x18"
			  "\x17\x16\x15\x14\x13\x12\x11\x10"
			  "\x0f\x0e\x0d\x0c\x0b\x0a\x09\x08"
			  "\x07\x06\x05\x04\x03\x02\x01\x00",
		.klen	= 32,
		.input	= "\x0f\x0e\x0d\x0c\x0b\x0a\x09\x08"
			  "\x07\x06\x05\x04\x03\x02\x01\x00",
		.ilen	= 16,
		.result	= "\x5c\xe7\x1c\x70\xd2\x88\x2e\x5b"
			  "\xb8\x32\xe4\x33\xf8\x9f\x26\xde",
		.rlen	= 16,
	},
};

static struct cipher_testvec serpent_dec_tv_template[] = {
	{
		.input	= "\x12\x07\xfc\xce\x9b\xd0\xd6\x47"
			  "\x6a\xe9\x8f\xbe\xd1\x43\xa0\xe2",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x4c\x7d\x8a\x32\x80\x72\xa2\x2c"
			  "\x82\x3e\x4a\x1f\x3a\xcd\xa1\x6d",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\xde\x26\x9f\xf8\x33\xe4\x32\xb8"
			  "\x5b\x2e\x88\xd2\x70\x1c\xe7\x5c",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80",
		.klen	= 16,
		.input	= "\xdd\xd2\x6b\x98\xa5\xff\xd8\x2c"
			  "\x05\x34\x5a\x9d\xad\xbf\xaf\x49",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.input	= "\xFB\xB0\x5D\xDE\xC0\xFE\xFC\xEB"
			  "\xB1\x80\x10\x43\xDE\x62\x70\xBD"
			  "\xFA\x8A\x93\xEA\x6B\xF7\xC5\xD7"
			  "\x0C\xD1\xBB\x29\x25\x14\x4C\x22"
			  "\x77\xA6\x38\x00\xDB\xB9\xE2\x07"
			  "\xD1\xAC\x82\xBA\xEA\x67\xAA\x39"
			  "\x99\x34\x89\x5B\x54\xE9\x12\x13"
			  "\x3B\x04\xE5\x12\x42\xC5\x79\xAB"
			  "\x0D\xC7\x3C\x58\x2D\xA3\x98\xF6"
			  "\xE4\x61\x9E\x17\x0B\xCE\xE8\xAA"
			  "\xB5\x6C\x1A\x3A\x67\x52\x81\x6A"
			  "\x04\xFF\x8A\x1B\x96\xFE\xE6\x87"
			  "\x3C\xD4\x39\x7D\x36\x9B\x03\xD5"
			  "\xB6\xA0\x75\x3C\x83\xE6\x1C\x73"
			  "\x9D\x74\x2B\x77\x53\x2D\xE5\xBD"
			  "\x69\xDA\x7A\x01\xF5\x6A\x70\x39"
			  "\x30\xD4\x2C\xF2\x8E\x06\x4B\x39"
			  "\xB3\x12\x1D\xB3\x17\x46\xE6\xD6"
			  "\xB6\x31\x36\x34\x38\x3C\x1D\x69"
			  "\x9F\x47\x28\x9A\x1D\x96\x70\x54"
			  "\x8E\x88\xCB\xE0\xF5\x6A\xAE\x0A"
			  "\x3C\xD5\x93\x1C\x21\xC9\x14\x3A"
			  "\x23\x9C\x9B\x79\xC7\x75\xC8\x39"
			  "\xA6\xAC\x65\x9A\x99\x37\xAF\x6D"
			  "\xBD\xB5\x32\xFD\xD8\x9C\x95\x7B"
			  "\xC6\x6A\x80\x64\xEA\xEF\x6D\x3F"
			  "\xA9\xFE\x5B\x16\xA3\xCF\x32\xC8"
			  "\xEF\x50\x22\x20\x93\x30\xBE\xE2"
			  "\x38\x05\x65\xAF\xBA\xB6\xE4\x72"
			  "\xA9\xEE\x05\x42\x88\xBD\x9D\x49"
			  "\xAD\x93\xCA\x4D\x45\x11\x43\x4D"
			  "\xB8\xF5\x74\x2B\x48\xE7\x21\xE4"
			  "\x4E\x3A\x4C\xDE\x65\x7A\x5A\xAD"
			  "\x86\xE6\x23\xEC\x6B\xA7\x17\xE6"
			  "\xF6\xA1\xAC\x29\xAE\xF9\x9B\x69"
			  "\x73\x65\x65\x51\xD6\x0B\x4E\x8C"
			  "\x17\x15\x9D\xB0\xCF\xB2\x42\x2B"
			  "\x51\xC3\x03\xE8\xB7\x7D\x2D\x39"
			  "\xE8\x10\x93\x16\xC8\x68\x4C\x60"
			  "\x87\x70\x14\xD0\x01\x57\xCB\x42"
			  "\x13\x59\xB1\x7F\x12\x4F\xBB\xC7"
			  "\xBD\x2B\xD4\xA9\x12\x26\x4F\xDE"
			  "\xFD\x72\xEC\xD7\x6F\x97\x14\x90"
			  "\x0E\x37\x13\xE6\x67\x1D\xE5\xFE"
			  "\x9E\x18\x3C\x8F\x3A\x3F\x59\x9B"
			  "\x71\x80\x05\x35\x3F\x40\x0B\x21"
			  "\x76\xE5\xEF\x42\x6C\xDB\x31\x05"
			  "\x5F\x05\xCF\x14\xE3\xF0\x61\xA2"
			  "\x49\x03\x5E\x77\x2E\x20\xBA\xA1"
			  "\xAF\x46\x51\xC0\x2B\xC4\x64\x1E"
			  "\x65\xCC\x51\x58\x0A\xDF\xF0\x5F"
			  "\x75\x9F\x48\xCD\x81\xEC\xC3\xF6"
			  "\xED\xC9\x4B\x7B\x4E\x26\x23\xE1"
			  "\xBB\xE9\x83\x0B\xCF\xE4\xDE\x00"
			  "\x48\xFF\xBF\x6C\xB4\x72\x16\xEF"
			  "\xC7\x46\xEE\x48\x8C\xB8\xAF\x45"
			  "\x91\x76\xE7\x6E\x65\x3D\x15\x86"
			  "\x10\xF8\xDB\x66\x97\x7C\x43\x4D"
			  "\x79\x12\x4E\xCE\x06\xD1\xD1\x6A"
			  "\x34\xC1\xC9\xF2\x28\x4A\xCD\x02"
			  "\x75\x55\x9B\xFF\x36\x73\xAB\x7C"
			  "\xF4\x46\x2E\xEB\xAC\xF3\xD2\xB7",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec tnepres_dec_tv_template[] = {
	{
		.input	= "\x41\xcc\x6b\x31\x59\x31\x45\x97"
			  "\x6d\x6f\xbb\x38\x4b\x37\x21\x28",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\xea\xf4\xd7\xfc\xd8\x01\x34\x47"
			  "\x81\x45\x0b\xfa\x0c\xd6\xad\x6e",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x64\xa9\x1a\x37\xed\x9f\xe7\x49"
			  "\xa8\x4e\x76\xd6\xf5\x0d\x78\xee",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80",
		.klen	= 16,
		.input	= "\x3d\xda\xbf\xc0\x06\xda\xab\x06"
			  "\x46\x2a\xf4\xef\x81\x54\x4e\x26",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	},
};

static struct cipher_testvec serpent_cbc_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x80\xCF\x11\x41\x1A\xB9\x4B\x9C"
			  "\xFF\xB7\x6C\xEA\xF0\xAF\x77\x6E"
			  "\x71\x75\x95\x9D\x4E\x1C\xCF\xAD"
			  "\x81\x34\xE9\x8F\xAE\x5A\x91\x1C"
			  "\x38\x63\x35\x7E\x79\x18\x0A\xE8"
			  "\x67\x06\x76\xD5\xFF\x22\x2F\xDA"
			  "\xB6\x2D\x57\x13\xB6\x3C\xBC\x97"
			  "\xFE\x53\x75\x35\x97\x7F\x51\xEA"
			  "\xDF\x5D\xE8\x9D\xCC\xD9\xAE\xE7"
			  "\x62\x67\xFF\x04\xC2\x18\x22\x5F"
			  "\x2E\x06\xC1\xE2\x26\xCD\xC6\x1E"
			  "\xE5\x2C\x4E\x87\x23\xDD\xF0\x41"
			  "\x08\xA5\xB4\x3E\x07\x1E\x0B\xBB"
			  "\x72\x84\xF8\x0A\x3F\x38\x5E\x91"
			  "\x15\x26\xE1\xDB\xA4\x3D\x74\xD2"
			  "\x41\x1E\x3F\xA9\xC6\x7D\x2A\xAB"
			  "\x27\xDF\x89\x1D\x86\x3E\xF7\x5A"
			  "\xF6\xE3\x0F\xC7\x6B\x4C\x96\x7C"
			  "\x2D\x12\xA5\x05\x92\xCB\xD7\x4A"
			  "\x4D\x1E\x88\x21\xE1\x63\xB4\xFC"
			  "\x4A\xF2\xCD\x35\xB9\xD7\x70\x97"
			  "\x5A\x5E\x7E\x96\x52\x20\xDC\x25"
			  "\xE9\x6B\x36\xB4\xE0\x98\x85\x2C"
			  "\x3C\xD2\xF7\x78\x8A\x73\x26\x9B"
			  "\xAF\x0B\x11\xE8\x4D\x67\x23\xE9"
			  "\x77\xDF\x58\xF6\x6F\x9E\xA4\xC5"
			  "\x10\xA1\x82\x0E\x80\xA0\x8F\x4B"
			  "\xA1\xC0\x12\x54\x4E\xC9\x20\x92"
			  "\x11\x00\x10\x4E\xB3\x7C\xCA\x63"
			  "\xE5\x3F\xD3\x41\x37\xCD\x74\xB7"
			  "\xA5\x7C\x61\xB8\x0B\x7A\x7F\x4D"
			  "\xFE\x96\x7D\x1B\xBE\x60\x37\xB7"
			  "\x81\x92\x66\x67\x15\x1E\x39\x98"
			  "\x52\xC0\xF4\x69\xC0\x99\x4F\x5A"
			  "\x2E\x32\xAD\x7C\x8B\xE9\xAD\x05"
			  "\x55\xF9\x0A\x1F\x97\x5C\xFA\x2B"
			  "\xF4\x99\x76\x3A\x6E\x4D\xE1\x4C"
			  "\x14\x4E\x6F\x87\xEE\x1A\x85\xA3"
			  "\x96\xC6\x66\x49\xDA\x0D\x71\xAC"
			  "\x04\x05\x46\xD3\x90\x0F\x64\x64"
			  "\x01\x66\x2C\x62\x5D\x34\xD1\xCB"
			  "\x3A\x24\xCE\x95\xEF\xAE\x2C\x97"
			  "\x0E\x0C\x1D\x36\x49\xEB\xE9\x3D"
			  "\x62\xA6\x19\x28\x9E\x26\xB4\x3F"
			  "\xD7\x55\x42\x3C\xCD\x72\x0A\xF0"
			  "\x7D\xE9\x95\x45\x86\xED\xB1\xE0"
			  "\x8D\xE9\xC5\x86\x13\x24\x28\x7D"
			  "\x74\xEF\xCA\x50\x12\x7E\x64\x8F"
			  "\x1B\xF5\x5B\xFE\xE2\xAC\xFA\xE7"
			  "\xBD\x38\x8C\x11\x20\xEF\xB1\xAA"
			  "\x7B\xE5\xE5\x78\xAD\x9D\x2D\xA2"
			  "\x8E\xDD\x48\xB3\xEF\x18\x92\x7E"
			  "\xE6\x75\x0D\x54\x64\x11\xA3\x3A"
			  "\xDB\x97\x0F\xD3\xDF\x07\xD3\x7E"
			  "\x1E\xD1\x87\xE4\x74\xBB\x46\xF4"
			  "\xBA\x23\x2D\x8D\x29\x07\x12\xCF"
			  "\x34\xCD\x72\x7F\x01\x30\xE7\xA0"
			  "\xF8\xDD\xA8\x08\xF0\xBC\xB1\xA2"
			  "\xCC\xE1\x6B\x5F\xBE\xEA\xF1\xE4"
			  "\x02\xC4\xAF\xFA\xAD\x31\xF4\xBF"
			  "\xFC\x66\xAA\x37\xF2\x37\x39\x6B"
			  "\xBC\x08\x3A\xA2\x29\xB3\xDF\xD1",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec serpent_cbc_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x80\xCF\x11\x41\x1A\xB9\x4B\x9C"
			  "\xFF\xB7\x6C\xEA\xF0\xAF\x77\x6E"
			  "\x71\x75\x95\x9D\x4E\x1C\xCF\xAD"
			  "\x81\x34\xE9\x8F\xAE\x5A\x91\x1C"
			  "\x38\x63\x35\x7E\x79\x18\x0A\xE8"
			  "\x67\x06\x76\xD5\xFF\x22\x2F\xDA"
			  "\xB6\x2D\x57\x13\xB6\x3C\xBC\x97"
			  "\xFE\x53\x75\x35\x97\x7F\x51\xEA"
			  "\xDF\x5D\xE8\x9D\xCC\xD9\xAE\xE7"
			  "\x62\x67\xFF\x04\xC2\x18\x22\x5F"
			  "\x2E\x06\xC1\xE2\x26\xCD\xC6\x1E"
			  "\xE5\x2C\x4E\x87\x23\xDD\xF0\x41"
			  "\x08\xA5\xB4\x3E\x07\x1E\x0B\xBB"
			  "\x72\x84\xF8\x0A\x3F\x38\x5E\x91"
			  "\x15\x26\xE1\xDB\xA4\x3D\x74\xD2"
			  "\x41\x1E\x3F\xA9\xC6\x7D\x2A\xAB"
			  "\x27\xDF\x89\x1D\x86\x3E\xF7\x5A"
			  "\xF6\xE3\x0F\xC7\x6B\x4C\x96\x7C"
			  "\x2D\x12\xA5\x05\x92\xCB\xD7\x4A"
			  "\x4D\x1E\x88\x21\xE1\x63\xB4\xFC"
			  "\x4A\xF2\xCD\x35\xB9\xD7\x70\x97"
			  "\x5A\x5E\x7E\x96\x52\x20\xDC\x25"
			  "\xE9\x6B\x36\xB4\xE0\x98\x85\x2C"
			  "\x3C\xD2\xF7\x78\x8A\x73\x26\x9B"
			  "\xAF\x0B\x11\xE8\x4D\x67\x23\xE9"
			  "\x77\xDF\x58\xF6\x6F\x9E\xA4\xC5"
			  "\x10\xA1\x82\x0E\x80\xA0\x8F\x4B"
			  "\xA1\xC0\x12\x54\x4E\xC9\x20\x92"
			  "\x11\x00\x10\x4E\xB3\x7C\xCA\x63"
			  "\xE5\x3F\xD3\x41\x37\xCD\x74\xB7"
			  "\xA5\x7C\x61\xB8\x0B\x7A\x7F\x4D"
			  "\xFE\x96\x7D\x1B\xBE\x60\x37\xB7"
			  "\x81\x92\x66\x67\x15\x1E\x39\x98"
			  "\x52\xC0\xF4\x69\xC0\x99\x4F\x5A"
			  "\x2E\x32\xAD\x7C\x8B\xE9\xAD\x05"
			  "\x55\xF9\x0A\x1F\x97\x5C\xFA\x2B"
			  "\xF4\x99\x76\x3A\x6E\x4D\xE1\x4C"
			  "\x14\x4E\x6F\x87\xEE\x1A\x85\xA3"
			  "\x96\xC6\x66\x49\xDA\x0D\x71\xAC"
			  "\x04\x05\x46\xD3\x90\x0F\x64\x64"
			  "\x01\x66\x2C\x62\x5D\x34\xD1\xCB"
			  "\x3A\x24\xCE\x95\xEF\xAE\x2C\x97"
			  "\x0E\x0C\x1D\x36\x49\xEB\xE9\x3D"
			  "\x62\xA6\x19\x28\x9E\x26\xB4\x3F"
			  "\xD7\x55\x42\x3C\xCD\x72\x0A\xF0"
			  "\x7D\xE9\x95\x45\x86\xED\xB1\xE0"
			  "\x8D\xE9\xC5\x86\x13\x24\x28\x7D"
			  "\x74\xEF\xCA\x50\x12\x7E\x64\x8F"
			  "\x1B\xF5\x5B\xFE\xE2\xAC\xFA\xE7"
			  "\xBD\x38\x8C\x11\x20\xEF\xB1\xAA"
			  "\x7B\xE5\xE5\x78\xAD\x9D\x2D\xA2"
			  "\x8E\xDD\x48\xB3\xEF\x18\x92\x7E"
			  "\xE6\x75\x0D\x54\x64\x11\xA3\x3A"
			  "\xDB\x97\x0F\xD3\xDF\x07\xD3\x7E"
			  "\x1E\xD1\x87\xE4\x74\xBB\x46\xF4"
			  "\xBA\x23\x2D\x8D\x29\x07\x12\xCF"
			  "\x34\xCD\x72\x7F\x01\x30\xE7\xA0"
			  "\xF8\xDD\xA8\x08\xF0\xBC\xB1\xA2"
			  "\xCC\xE1\x6B\x5F\xBE\xEA\xF1\xE4"
			  "\x02\xC4\xAF\xFA\xAD\x31\xF4\xBF"
			  "\xFC\x66\xAA\x37\xF2\x37\x39\x6B"
			  "\xBC\x08\x3A\xA2\x29\xB3\xDF\xD1",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec serpent_ctr_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x84\x68\xEC\xF2\x1C\x88\x20\xCA"
			  "\x37\x69\xE3\x3A\x22\x85\x48\x46"
			  "\x70\xAA\x25\xB4\xCD\x8B\x04\x4E"
			  "\x8D\x15\x2B\x98\xDF\x7B\x6D\xB9"
			  "\xE0\x4A\x73\x00\x65\xB6\x1A\x0D"
			  "\x5C\x60\xDF\x34\xDC\x60\x4C\xDF"
			  "\xB5\x1F\x26\x8C\xDA\xC1\x11\xA8"
			  "\x80\xFA\x37\x7A\x89\xAA\xAE\x7B"
			  "\x92\x6E\xB9\xDC\xC9\x62\x4F\x88"
			  "\x0A\x5D\x97\x2F\x6B\xAC\x03\x7C"
			  "\x22\xF6\x55\x5A\xFA\x35\xA5\x17"
			  "\xA1\x5C\x5E\x2B\x63\x2D\xB9\x91"
			  "\x3E\x83\x26\x00\x4E\xD5\xBE\xCE"
			  "\x79\xC4\x3D\xFC\x70\xA0\xAD\x96"
			  "\xBA\x58\x2A\x1C\xDF\xC2\x3A\xA5"
			  "\x7C\xB5\x12\x89\xED\xBF\xB6\x09"
			  "\x13\x4F\x7D\x61\x3C\x5C\x27\xFC"
			  "\x5D\xE1\x4F\xA1\xEA\xB3\xCA\xB9"
			  "\xE6\xD0\x97\x81\xDE\xD1\xFB\x8A"
			  "\x30\xDB\xA3\x5D\xEC\x25\x0B\x86"
			  "\x71\xC8\xA7\x67\xE8\xBC\x7D\x4C"
			  "\xAE\x82\xD3\x73\x31\x09\xCB\xB3"
			  "\x4D\xD4\xC0\x8A\x2B\xFA\xA6\x55"
			  "\x39\x0A\xBC\x6E\x75\xAB\xC2\xE2"
			  "\x8A\xF2\x26\xCD\x63\x38\x35\xF7"
			  "\xAE\x12\x83\xCD\x8A\x9E\x7E\x4C"
			  "\xFE\x4D\xD7\xCE\x5C\x6E\x4C\xAF"
			  "\xE3\xCD\x76\xA7\x87\xA1\x54\x7C"
			  "\xEC\x32\xC7\x83\x2A\xFF\xF8\xEA"
			  "\x87\xB2\x47\xA3\x9D\xC2\x9C\xA2"
			  "\xB7\x2C\x7C\x1A\x24\xCB\x88\x61"
			  "\xFF\xA7\x1A\x16\x01\xDD\x4B\xFC"
			  "\x2E\xE0\x48\x67\x09\x42\xCC\x91"
			  "\xBE\x20\x38\xC0\x5E\x3B\x95\x00"
			  "\xA1\x96\x66\x0B\x8A\xE9\x9E\xF7"
			  "\x6B\x34\x0A\x51\xC0\x3B\xEB\x71"
			  "\x07\x97\x38\x4B\x5C\x56\x98\x67"
			  "\x78\x9C\xD0\x0E\x2B\xB5\x67\x90"
			  "\x75\xF8\xFE\x6D\x4E\x85\xCC\x0D"
			  "\x18\x06\x15\x9D\x5A\x10\x13\x37"
			  "\xA3\xD6\x68\xA2\xDF\x7E\xC7\x12"
			  "\xC9\x0D\x4D\x91\xB0\x2A\x55\xFF"
			  "\x6F\x73\x13\xDF\x28\xB5\x2A\x2C"
			  "\xE4\xFC\x20\xD9\xF1\x7A\x82\xB1"
			  "\xCB\x57\xB6\x3D\x8C\xF4\x8E\x27"
			  "\x37\xDC\x35\xF3\x79\x01\x53\xA4"
			  "\x7B\x37\xDE\x7C\x04\xAE\x50\xDB"
			  "\x9B\x1E\x8C\x07\xA7\x52\x49\x50"
			  "\x34\x25\x65\xDD\xA9\x8F\x7E\xBD"
			  "\x7A\xC9\x36\xAE\xDE\x21\x48\x64"
			  "\xC2\x02\xBA\xBE\x11\x1E\x3D\x9C"
			  "\x98\x52\xCC\x04\xBD\x5E\x61\x26"
			  "\x10\xD3\x21\xD9\x6E\x25\x98\x77"
			  "\x8E\x98\x63\xF6\xF6\x52\xFB\x13"
			  "\xAA\x30\xF2\xB9\xA4\x43\x53\x39"
			  "\x1C\x97\x07\x7E\x6B\xFF\x3D\x43"
			  "\xA6\x71\x6B\x66\x8F\x58\x3F\x71"
			  "\x90\x47\x40\x92\xE6\x69\xD1\x96"
			  "\x34\xB3\x3B\xE5\x43\xE4\xD5\x56"
			  "\xB2\xE6\x7E\x86\x7A\x12\x17\x5B"
			  "\x30\xF3\x9B\x0D\xFA\x57\xE4\x50"
			  "\x40\x53\x77\x8C\x15\xF8\x8D\x13",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59",
		.ilen	= 499,
		.result	= "\x84\x68\xEC\xF2\x1C\x88\x20\xCA"
			  "\x37\x69\xE3\x3A\x22\x85\x48\x46"
			  "\x70\xAA\x25\xB4\xCD\x8B\x04\x4E"
			  "\x8D\x15\x2B\x98\xDF\x7B\x6D\xB9"
			  "\xE0\x4A\x73\x00\x65\xB6\x1A\x0D"
			  "\x5C\x60\xDF\x34\xDC\x60\x4C\xDF"
			  "\xB5\x1F\x26\x8C\xDA\xC1\x11\xA8"
			  "\x80\xFA\x37\x7A\x89\xAA\xAE\x7B"
			  "\x92\x6E\xB9\xDC\xC9\x62\x4F\x88"
			  "\x0A\x5D\x97\x2F\x6B\xAC\x03\x7C"
			  "\x22\xF6\x55\x5A\xFA\x35\xA5\x17"
			  "\xA1\x5C\x5E\x2B\x63\x2D\xB9\x91"
			  "\x3E\x83\x26\x00\x4E\xD5\xBE\xCE"
			  "\x79\xC4\x3D\xFC\x70\xA0\xAD\x96"
			  "\xBA\x58\x2A\x1C\xDF\xC2\x3A\xA5"
			  "\x7C\xB5\x12\x89\xED\xBF\xB6\x09"
			  "\x13\x4F\x7D\x61\x3C\x5C\x27\xFC"
			  "\x5D\xE1\x4F\xA1\xEA\xB3\xCA\xB9"
			  "\xE6\xD0\x97\x81\xDE\xD1\xFB\x8A"
			  "\x30\xDB\xA3\x5D\xEC\x25\x0B\x86"
			  "\x71\xC8\xA7\x67\xE8\xBC\x7D\x4C"
			  "\xAE\x82\xD3\x73\x31\x09\xCB\xB3"
			  "\x4D\xD4\xC0\x8A\x2B\xFA\xA6\x55"
			  "\x39\x0A\xBC\x6E\x75\xAB\xC2\xE2"
			  "\x8A\xF2\x26\xCD\x63\x38\x35\xF7"
			  "\xAE\x12\x83\xCD\x8A\x9E\x7E\x4C"
			  "\xFE\x4D\xD7\xCE\x5C\x6E\x4C\xAF"
			  "\xE3\xCD\x76\xA7\x87\xA1\x54\x7C"
			  "\xEC\x32\xC7\x83\x2A\xFF\xF8\xEA"
			  "\x87\xB2\x47\xA3\x9D\xC2\x9C\xA2"
			  "\xB7\x2C\x7C\x1A\x24\xCB\x88\x61"
			  "\xFF\xA7\x1A\x16\x01\xDD\x4B\xFC"
			  "\x2E\xE0\x48\x67\x09\x42\xCC\x91"
			  "\xBE\x20\x38\xC0\x5E\x3B\x95\x00"
			  "\xA1\x96\x66\x0B\x8A\xE9\x9E\xF7"
			  "\x6B\x34\x0A\x51\xC0\x3B\xEB\x71"
			  "\x07\x97\x38\x4B\x5C\x56\x98\x67"
			  "\x78\x9C\xD0\x0E\x2B\xB5\x67\x90"
			  "\x75\xF8\xFE\x6D\x4E\x85\xCC\x0D"
			  "\x18\x06\x15\x9D\x5A\x10\x13\x37"
			  "\xA3\xD6\x68\xA2\xDF\x7E\xC7\x12"
			  "\xC9\x0D\x4D\x91\xB0\x2A\x55\xFF"
			  "\x6F\x73\x13\xDF\x28\xB5\x2A\x2C"
			  "\xE4\xFC\x20\xD9\xF1\x7A\x82\xB1"
			  "\xCB\x57\xB6\x3D\x8C\xF4\x8E\x27"
			  "\x37\xDC\x35\xF3\x79\x01\x53\xA4"
			  "\x7B\x37\xDE\x7C\x04\xAE\x50\xDB"
			  "\x9B\x1E\x8C\x07\xA7\x52\x49\x50"
			  "\x34\x25\x65\xDD\xA9\x8F\x7E\xBD"
			  "\x7A\xC9\x36\xAE\xDE\x21\x48\x64"
			  "\xC2\x02\xBA\xBE\x11\x1E\x3D\x9C"
			  "\x98\x52\xCC\x04\xBD\x5E\x61\x26"
			  "\x10\xD3\x21\xD9\x6E\x25\x98\x77"
			  "\x8E\x98\x63\xF6\xF6\x52\xFB\x13"
			  "\xAA\x30\xF2\xB9\xA4\x43\x53\x39"
			  "\x1C\x97\x07\x7E\x6B\xFF\x3D\x43"
			  "\xA6\x71\x6B\x66\x8F\x58\x3F\x71"
			  "\x90\x47\x40\x92\xE6\x69\xD1\x96"
			  "\x34\xB3\x3B\xE5\x43\xE4\xD5\x56"
			  "\xB2\xE6\x7E\x86\x7A\x12\x17\x5B"
			  "\x30\xF3\x9B\x0D\xFA\x57\xE4\x50"
			  "\x40\x53\x77\x8C\x15\xF8\x8D\x13"
			  "\x38\xE2\xE5",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x06\x9A\xF8\xB4\x53\x88\x62\xFC"
			  "\x68\xB8\x2E\xDF\xC1\x05\x0F\x3D"
			  "\xAF\x4D\x95\xAE\xC4\xE9\x1C\xDC"
			  "\xF6\x2B\x8F\x90\x89\xF6\x7E\x1A"
			  "\xA6\xB9\xE4\xF4\xFA\xCA\xE5\x7E"
			  "\x71\x28\x06\x4F\xE8\x08\x39\xDA"
			  "\xA5\x0E\xC8\xC0\xB8\x16\xE5\x69"
			  "\xE5\xCA\xEC\x4F\x63\x2C\xC0\x9B"
			  "\x9F\x3E\x39\x79\xF0\xCD\x64\x35"
			  "\x4A\xD3\xC8\xA9\x31\xCD\x48\x5B"
			  "\x92\x3D\x8F\x3F\x96\xBD\xB3\x18"
			  "\x74\x2A\x5D\x29\x3F\x57\x8F\xE2"
			  "\x67\x9A\xE0\xE5\xD4\x4A\xE2\x47"
			  "\xBC\xF6\xEB\x14\xF3\x8C\x20\xC2"
			  "\x7D\xE2\x43\x81\x86\x72\x2E\xB1"
			  "\x39\xF6\x95\xE1\x1F\xCB\x76\x33"
			  "\x5B\x7D\x23\x0F\x3A\x67\x2A\x2F"
			  "\xB9\x37\x9D\xDD\x1F\x16\xA1\x3C"
			  "\x70\xFE\x52\xAA\x93\x3C\xC4\x46"
			  "\xB1\xE5\xFF\xDA\xAF\xE2\x84\xFE"
			  "\x25\x92\xB2\x63\xBD\x49\x77\xB4"
			  "\x22\xA4\x6A\xD5\x04\xE0\x45\x58"
			  "\x1C\x34\x96\x7C\x03\x0C\x13\xA2"
			  "\x05\x22\xE2\xCB\x5A\x35\x03\x09"
			  "\x40\xD2\x82\x05\xCA\x58\x73\xF2"
			  "\x29\x5E\x01\x47\x13\x32\x78\xBE"
			  "\x06\xB0\x51\xDB\x6C\x31\xA0\x1C"
			  "\x74\xBC\x8D\x25\xDF\xF8\x65\xD1"
			  "\x38\x35\x11\x26\x4A\xB4\x06\x32"
			  "\xFA\xD2\x07\x77\xB3\x74\x98\x80"
			  "\x61\x59\xA8\x9F\xF3\x6F\x2A\xBF"
			  "\xE6\xA5\x9A\xC4\x6B\xA6\x49\x6F"
			  "\xBC\x47\xD9\xFB\xC6\xEF\x25\x65"
			  "\x96\xAC\x9F\xE4\x81\x4B\xD8\xBA"
			  "\xD6\x9B\xC9\x6D\x58\x40\x81\x02"
			  "\x73\x44\x4E\x43\x6E\x37\xBB\x11"
			  "\xE3\xF9\xB8\x2F\xEC\x76\x34\xEA"
			  "\x90\xCD\xB7\x2E\x0E\x32\x71\xE8"
			  "\xBB\x4E\x0B\x98\xA4\x17\x17\x5B"
			  "\x07\xB5\x82\x3A\xC4\xE8\x42\x51"
			  "\x5A\x4C\x4E\x7D\xBF\xC4\xC0\x4F"
			  "\x68\xB8\xC6\x4A\x32\x6F\x0B\xD7"
			  "\x85\xED\x6B\xFB\x72\xD2\xA5\x8F"
			  "\xBF\xF9\xAC\x59\x50\xA8\x08\x70"
			  "\xEC\xBD\x0A\xBF\xE5\x87\xA1\xC2"
			  "\x92\x14\x78\xAF\xE8\xEA\x2E\xDD"
			  "\xC1\x03\x9A\xAA\x89\x8B\x32\x46"
			  "\x5B\x18\x27\xBA\x46\xAA\x64\xDE"
			  "\xE3\xD5\xA3\xFC\x7B\x5B\x61\xDB"
			  "\x7E\xDA\xEC\x30\x17\x19\xF8\x80"
			  "\xB5\x5E\x27\xB5\x37\x3A\x1F\x28"
			  "\x07\x73\xC3\x63\xCE\xFF\x8C\xFE"
			  "\x81\x4E\xF8\x24\xF3\xB8\xC7\xE8"
			  "\x16\x9A\xCC\x58\x2F\x88\x1C\x4B"
			  "\xBB\x33\xA2\x73\xF0\x1C\x89\x0E"
			  "\xDC\x34\x27\x89\x98\xCE\x1C\xA2"
			  "\xD8\xB8\x90\xBE\xEC\x72\x28\x13"
			  "\xAC\x7B\xF1\xD0\x7F\x7A\x28\x50"
			  "\xB7\x99\x65\x8A\xC9\xC6\x21\x34"
			  "\x7F\x67\x9D\xB7\x2C\xCC\xF5\x17"
			  "\x2B\x89\xAC\xB0\xD7\x1E\x47\xB0"
			  "\x61\xAF\xD4\x63\x6D\xB8\x2D\x20",
		.rlen	= 496,
	},
};

static struct cipher_testvec serpent_ctr_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x84\x68\xEC\xF2\x1C\x88\x20\xCA"
			  "\x37\x69\xE3\x3A\x22\x85\x48\x46"
			  "\x70\xAA\x25\xB4\xCD\x8B\x04\x4E"
			  "\x8D\x15\x2B\x98\xDF\x7B\x6D\xB9"
			  "\xE0\x4A\x73\x00\x65\xB6\x1A\x0D"
			  "\x5C\x60\xDF\x34\xDC\x60\x4C\xDF"
			  "\xB5\x1F\x26\x8C\xDA\xC1\x11\xA8"
			  "\x80\xFA\x37\x7A\x89\xAA\xAE\x7B"
			  "\x92\x6E\xB9\xDC\xC9\x62\x4F\x88"
			  "\x0A\x5D\x97\x2F\x6B\xAC\x03\x7C"
			  "\x22\xF6\x55\x5A\xFA\x35\xA5\x17"
			  "\xA1\x5C\x5E\x2B\x63\x2D\xB9\x91"
			  "\x3E\x83\x26\x00\x4E\xD5\xBE\xCE"
			  "\x79\xC4\x3D\xFC\x70\xA0\xAD\x96"
			  "\xBA\x58\x2A\x1C\xDF\xC2\x3A\xA5"
			  "\x7C\xB5\x12\x89\xED\xBF\xB6\x09"
			  "\x13\x4F\x7D\x61\x3C\x5C\x27\xFC"
			  "\x5D\xE1\x4F\xA1\xEA\xB3\xCA\xB9"
			  "\xE6\xD0\x97\x81\xDE\xD1\xFB\x8A"
			  "\x30\xDB\xA3\x5D\xEC\x25\x0B\x86"
			  "\x71\xC8\xA7\x67\xE8\xBC\x7D\x4C"
			  "\xAE\x82\xD3\x73\x31\x09\xCB\xB3"
			  "\x4D\xD4\xC0\x8A\x2B\xFA\xA6\x55"
			  "\x39\x0A\xBC\x6E\x75\xAB\xC2\xE2"
			  "\x8A\xF2\x26\xCD\x63\x38\x35\xF7"
			  "\xAE\x12\x83\xCD\x8A\x9E\x7E\x4C"
			  "\xFE\x4D\xD7\xCE\x5C\x6E\x4C\xAF"
			  "\xE3\xCD\x76\xA7\x87\xA1\x54\x7C"
			  "\xEC\x32\xC7\x83\x2A\xFF\xF8\xEA"
			  "\x87\xB2\x47\xA3\x9D\xC2\x9C\xA2"
			  "\xB7\x2C\x7C\x1A\x24\xCB\x88\x61"
			  "\xFF\xA7\x1A\x16\x01\xDD\x4B\xFC"
			  "\x2E\xE0\x48\x67\x09\x42\xCC\x91"
			  "\xBE\x20\x38\xC0\x5E\x3B\x95\x00"
			  "\xA1\x96\x66\x0B\x8A\xE9\x9E\xF7"
			  "\x6B\x34\x0A\x51\xC0\x3B\xEB\x71"
			  "\x07\x97\x38\x4B\x5C\x56\x98\x67"
			  "\x78\x9C\xD0\x0E\x2B\xB5\x67\x90"
			  "\x75\xF8\xFE\x6D\x4E\x85\xCC\x0D"
			  "\x18\x06\x15\x9D\x5A\x10\x13\x37"
			  "\xA3\xD6\x68\xA2\xDF\x7E\xC7\x12"
			  "\xC9\x0D\x4D\x91\xB0\x2A\x55\xFF"
			  "\x6F\x73\x13\xDF\x28\xB5\x2A\x2C"
			  "\xE4\xFC\x20\xD9\xF1\x7A\x82\xB1"
			  "\xCB\x57\xB6\x3D\x8C\xF4\x8E\x27"
			  "\x37\xDC\x35\xF3\x79\x01\x53\xA4"
			  "\x7B\x37\xDE\x7C\x04\xAE\x50\xDB"
			  "\x9B\x1E\x8C\x07\xA7\x52\x49\x50"
			  "\x34\x25\x65\xDD\xA9\x8F\x7E\xBD"
			  "\x7A\xC9\x36\xAE\xDE\x21\x48\x64"
			  "\xC2\x02\xBA\xBE\x11\x1E\x3D\x9C"
			  "\x98\x52\xCC\x04\xBD\x5E\x61\x26"
			  "\x10\xD3\x21\xD9\x6E\x25\x98\x77"
			  "\x8E\x98\x63\xF6\xF6\x52\xFB\x13"
			  "\xAA\x30\xF2\xB9\xA4\x43\x53\x39"
			  "\x1C\x97\x07\x7E\x6B\xFF\x3D\x43"
			  "\xA6\x71\x6B\x66\x8F\x58\x3F\x71"
			  "\x90\x47\x40\x92\xE6\x69\xD1\x96"
			  "\x34\xB3\x3B\xE5\x43\xE4\xD5\x56"
			  "\xB2\xE6\x7E\x86\x7A\x12\x17\x5B"
			  "\x30\xF3\x9B\x0D\xFA\x57\xE4\x50"
			  "\x40\x53\x77\x8C\x15\xF8\x8D\x13",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x84\x68\xEC\xF2\x1C\x88\x20\xCA"
			  "\x37\x69\xE3\x3A\x22\x85\x48\x46"
			  "\x70\xAA\x25\xB4\xCD\x8B\x04\x4E"
			  "\x8D\x15\x2B\x98\xDF\x7B\x6D\xB9"
			  "\xE0\x4A\x73\x00\x65\xB6\x1A\x0D"
			  "\x5C\x60\xDF\x34\xDC\x60\x4C\xDF"
			  "\xB5\x1F\x26\x8C\xDA\xC1\x11\xA8"
			  "\x80\xFA\x37\x7A\x89\xAA\xAE\x7B"
			  "\x92\x6E\xB9\xDC\xC9\x62\x4F\x88"
			  "\x0A\x5D\x97\x2F\x6B\xAC\x03\x7C"
			  "\x22\xF6\x55\x5A\xFA\x35\xA5\x17"
			  "\xA1\x5C\x5E\x2B\x63\x2D\xB9\x91"
			  "\x3E\x83\x26\x00\x4E\xD5\xBE\xCE"
			  "\x79\xC4\x3D\xFC\x70\xA0\xAD\x96"
			  "\xBA\x58\x2A\x1C\xDF\xC2\x3A\xA5"
			  "\x7C\xB5\x12\x89\xED\xBF\xB6\x09"
			  "\x13\x4F\x7D\x61\x3C\x5C\x27\xFC"
			  "\x5D\xE1\x4F\xA1\xEA\xB3\xCA\xB9"
			  "\xE6\xD0\x97\x81\xDE\xD1\xFB\x8A"
			  "\x30\xDB\xA3\x5D\xEC\x25\x0B\x86"
			  "\x71\xC8\xA7\x67\xE8\xBC\x7D\x4C"
			  "\xAE\x82\xD3\x73\x31\x09\xCB\xB3"
			  "\x4D\xD4\xC0\x8A\x2B\xFA\xA6\x55"
			  "\x39\x0A\xBC\x6E\x75\xAB\xC2\xE2"
			  "\x8A\xF2\x26\xCD\x63\x38\x35\xF7"
			  "\xAE\x12\x83\xCD\x8A\x9E\x7E\x4C"
			  "\xFE\x4D\xD7\xCE\x5C\x6E\x4C\xAF"
			  "\xE3\xCD\x76\xA7\x87\xA1\x54\x7C"
			  "\xEC\x32\xC7\x83\x2A\xFF\xF8\xEA"
			  "\x87\xB2\x47\xA3\x9D\xC2\x9C\xA2"
			  "\xB7\x2C\x7C\x1A\x24\xCB\x88\x61"
			  "\xFF\xA7\x1A\x16\x01\xDD\x4B\xFC"
			  "\x2E\xE0\x48\x67\x09\x42\xCC\x91"
			  "\xBE\x20\x38\xC0\x5E\x3B\x95\x00"
			  "\xA1\x96\x66\x0B\x8A\xE9\x9E\xF7"
			  "\x6B\x34\x0A\x51\xC0\x3B\xEB\x71"
			  "\x07\x97\x38\x4B\x5C\x56\x98\x67"
			  "\x78\x9C\xD0\x0E\x2B\xB5\x67\x90"
			  "\x75\xF8\xFE\x6D\x4E\x85\xCC\x0D"
			  "\x18\x06\x15\x9D\x5A\x10\x13\x37"
			  "\xA3\xD6\x68\xA2\xDF\x7E\xC7\x12"
			  "\xC9\x0D\x4D\x91\xB0\x2A\x55\xFF"
			  "\x6F\x73\x13\xDF\x28\xB5\x2A\x2C"
			  "\xE4\xFC\x20\xD9\xF1\x7A\x82\xB1"
			  "\xCB\x57\xB6\x3D\x8C\xF4\x8E\x27"
			  "\x37\xDC\x35\xF3\x79\x01\x53\xA4"
			  "\x7B\x37\xDE\x7C\x04\xAE\x50\xDB"
			  "\x9B\x1E\x8C\x07\xA7\x52\x49\x50"
			  "\x34\x25\x65\xDD\xA9\x8F\x7E\xBD"
			  "\x7A\xC9\x36\xAE\xDE\x21\x48\x64"
			  "\xC2\x02\xBA\xBE\x11\x1E\x3D\x9C"
			  "\x98\x52\xCC\x04\xBD\x5E\x61\x26"
			  "\x10\xD3\x21\xD9\x6E\x25\x98\x77"
			  "\x8E\x98\x63\xF6\xF6\x52\xFB\x13"
			  "\xAA\x30\xF2\xB9\xA4\x43\x53\x39"
			  "\x1C\x97\x07\x7E\x6B\xFF\x3D\x43"
			  "\xA6\x71\x6B\x66\x8F\x58\x3F\x71"
			  "\x90\x47\x40\x92\xE6\x69\xD1\x96"
			  "\x34\xB3\x3B\xE5\x43\xE4\xD5\x56"
			  "\xB2\xE6\x7E\x86\x7A\x12\x17\x5B"
			  "\x30\xF3\x9B\x0D\xFA\x57\xE4\x50"
			  "\x40\x53\x77\x8C\x15\xF8\x8D\x13"
			  "\x38\xE2\xE5",
		.ilen	= 499,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x06\x9A\xF8\xB4\x53\x88\x62\xFC"
			  "\x68\xB8\x2E\xDF\xC1\x05\x0F\x3D"
			  "\xAF\x4D\x95\xAE\xC4\xE9\x1C\xDC"
			  "\xF6\x2B\x8F\x90\x89\xF6\x7E\x1A"
			  "\xA6\xB9\xE4\xF4\xFA\xCA\xE5\x7E"
			  "\x71\x28\x06\x4F\xE8\x08\x39\xDA"
			  "\xA5\x0E\xC8\xC0\xB8\x16\xE5\x69"
			  "\xE5\xCA\xEC\x4F\x63\x2C\xC0\x9B"
			  "\x9F\x3E\x39\x79\xF0\xCD\x64\x35"
			  "\x4A\xD3\xC8\xA9\x31\xCD\x48\x5B"
			  "\x92\x3D\x8F\x3F\x96\xBD\xB3\x18"
			  "\x74\x2A\x5D\x29\x3F\x57\x8F\xE2"
			  "\x67\x9A\xE0\xE5\xD4\x4A\xE2\x47"
			  "\xBC\xF6\xEB\x14\xF3\x8C\x20\xC2"
			  "\x7D\xE2\x43\x81\x86\x72\x2E\xB1"
			  "\x39\xF6\x95\xE1\x1F\xCB\x76\x33"
			  "\x5B\x7D\x23\x0F\x3A\x67\x2A\x2F"
			  "\xB9\x37\x9D\xDD\x1F\x16\xA1\x3C"
			  "\x70\xFE\x52\xAA\x93\x3C\xC4\x46"
			  "\xB1\xE5\xFF\xDA\xAF\xE2\x84\xFE"
			  "\x25\x92\xB2\x63\xBD\x49\x77\xB4"
			  "\x22\xA4\x6A\xD5\x04\xE0\x45\x58"
			  "\x1C\x34\x96\x7C\x03\x0C\x13\xA2"
			  "\x05\x22\xE2\xCB\x5A\x35\x03\x09"
			  "\x40\xD2\x82\x05\xCA\x58\x73\xF2"
			  "\x29\x5E\x01\x47\x13\x32\x78\xBE"
			  "\x06\xB0\x51\xDB\x6C\x31\xA0\x1C"
			  "\x74\xBC\x8D\x25\xDF\xF8\x65\xD1"
			  "\x38\x35\x11\x26\x4A\xB4\x06\x32"
			  "\xFA\xD2\x07\x77\xB3\x74\x98\x80"
			  "\x61\x59\xA8\x9F\xF3\x6F\x2A\xBF"
			  "\xE6\xA5\x9A\xC4\x6B\xA6\x49\x6F"
			  "\xBC\x47\xD9\xFB\xC6\xEF\x25\x65"
			  "\x96\xAC\x9F\xE4\x81\x4B\xD8\xBA"
			  "\xD6\x9B\xC9\x6D\x58\x40\x81\x02"
			  "\x73\x44\x4E\x43\x6E\x37\xBB\x11"
			  "\xE3\xF9\xB8\x2F\xEC\x76\x34\xEA"
			  "\x90\xCD\xB7\x2E\x0E\x32\x71\xE8"
			  "\xBB\x4E\x0B\x98\xA4\x17\x17\x5B"
			  "\x07\xB5\x82\x3A\xC4\xE8\x42\x51"
			  "\x5A\x4C\x4E\x7D\xBF\xC4\xC0\x4F"
			  "\x68\xB8\xC6\x4A\x32\x6F\x0B\xD7"
			  "\x85\xED\x6B\xFB\x72\xD2\xA5\x8F"
			  "\xBF\xF9\xAC\x59\x50\xA8\x08\x70"
			  "\xEC\xBD\x0A\xBF\xE5\x87\xA1\xC2"
			  "\x92\x14\x78\xAF\xE8\xEA\x2E\xDD"
			  "\xC1\x03\x9A\xAA\x89\x8B\x32\x46"
			  "\x5B\x18\x27\xBA\x46\xAA\x64\xDE"
			  "\xE3\xD5\xA3\xFC\x7B\x5B\x61\xDB"
			  "\x7E\xDA\xEC\x30\x17\x19\xF8\x80"
			  "\xB5\x5E\x27\xB5\x37\x3A\x1F\x28"
			  "\x07\x73\xC3\x63\xCE\xFF\x8C\xFE"
			  "\x81\x4E\xF8\x24\xF3\xB8\xC7\xE8"
			  "\x16\x9A\xCC\x58\x2F\x88\x1C\x4B"
			  "\xBB\x33\xA2\x73\xF0\x1C\x89\x0E"
			  "\xDC\x34\x27\x89\x98\xCE\x1C\xA2"
			  "\xD8\xB8\x90\xBE\xEC\x72\x28\x13"
			  "\xAC\x7B\xF1\xD0\x7F\x7A\x28\x50"
			  "\xB7\x99\x65\x8A\xC9\xC6\x21\x34"
			  "\x7F\x67\x9D\xB7\x2C\xCC\xF5\x17"
			  "\x2B\x89\xAC\xB0\xD7\x1E\x47\xB0"
			  "\x61\xAF\xD4\x63\x6D\xB8\x2D\x20",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
	},
};

static struct cipher_testvec serpent_lrw_enc_tv_template[] = {
	 
	{
		.key	= "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x6f\xbf\xd4\xa4\x5d\x71\x16\x79"
			  "\x63\x9c\xa6\x8e\x40\xbe\x0d\x8a",
		.rlen	= 16,
	}, {
		.key	= "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\xfd\xb2\x66\x98\x80\x96\x55\xad"
			  "\x08\x94\x54\x9c\x21\x7c\x69\xe3",
		.rlen	= 16,
	}, {
		.key	= "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x14\x5e\x3d\x70\xc0\x6e\x9c\x34"
			  "\x5b\x5e\xcf\x0f\xe4\x8c\x21\x5c",
		.rlen	= 16,
	}, {
		.key	= "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x25\x39\xaa\xa5\xf0\x65\xc8\xdc"
			  "\x5d\x45\x95\x30\x8f\xff\x2f\x1b",
		.rlen	= 16,
	}, {
		.key	= "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x0c\x20\x20\x63\xd6\x8b\xfc\x8f"
			  "\xc0\xe2\x17\xbb\xd2\x59\x6f\x26",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\xc1\x35\x2e\x53\xf0\x96\x4d\x9c"
			  "\x2e\x18\xe6\x99\xcd\xd3\x15\x68",
		.rlen	= 16,
	}, {
		.key	= "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x86\x0a\xc6\xa9\x1a\x9f\xe7\xe6"
			  "\x64\x3b\x33\xd6\xd5\x84\xd6\xdf",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.ilen	= 512,
		.result	= "\xe3\x5a\x38\x0f\x4d\x92\x3a\x74"
			  "\x15\xb1\x50\x8c\x9a\xd8\x99\x1d"
			  "\x82\xec\xf1\x5f\x03\x6d\x02\x58"
			  "\x90\x67\xfc\xdd\x8d\xe1\x38\x08"
			  "\x7b\xc9\x9b\x4b\x04\x09\x50\x15"
			  "\xce\xab\xda\x33\x30\x20\x12\xfa"
			  "\x83\xc4\xa6\x9a\x2e\x7d\x90\xd9"
			  "\xa6\xa6\x67\x43\xb4\xa7\xa8\x5c"
			  "\xbb\x6a\x49\x2b\x8b\xf8\xd0\x22"
			  "\xe5\x9e\xba\xe8\x8c\x67\xb8\x5b"
			  "\x60\xbc\xf5\xa4\x95\x4e\x66\xe5"
			  "\x6d\x8e\xa9\xf6\x65\x2e\x04\xf5"
			  "\xba\xb5\xdb\x88\xc2\xf6\x7a\x4b"
			  "\x89\x58\x7c\x9a\xae\x26\xe8\xb7"
			  "\xb7\x28\xcc\xd6\xcc\xa5\x98\x4d"
			  "\xb9\x91\xcb\xb4\xe4\x8b\x96\x47"
			  "\x5f\x03\x8b\xdd\x94\xd1\xee\x12"
			  "\xa7\x83\x80\xf2\xc1\x15\x74\x4f"
			  "\x49\xf9\xb0\x7e\x6f\xdc\x73\x2f"
			  "\xe2\xcf\xe0\x1b\x34\xa5\xa0\x52"
			  "\xfb\x3c\x5d\x85\x91\xe6\x6d\x98"
			  "\x04\xd6\xdd\x4c\x00\x64\xd9\x54"
			  "\x5c\x3c\x08\x1d\x4c\x06\x9f\xb8"
			  "\x1c\x4d\x8d\xdc\xa4\x3c\xb9\x3b"
			  "\x9e\x85\xce\xc3\xa8\x4a\x0c\xd9"
			  "\x04\xc3\x6f\x17\x66\xa9\x1f\x59"
			  "\xd9\xe2\x19\x36\xa3\x88\xb8\x0b"
			  "\x0f\x4a\x4d\xf8\xc8\x6f\xd5\x43"
			  "\xeb\xa0\xab\x1f\x61\xc0\x06\xeb"
			  "\x93\xb7\xb8\x6f\x0d\xbd\x07\x49"
			  "\xb3\xac\x5d\xcf\x31\xa0\x27\x26"
			  "\x21\xbe\x94\x2e\x19\xea\xf4\xee"
			  "\xb5\x13\x89\xf7\x94\x0b\xef\x59"
			  "\x44\xc5\x78\x8b\x3c\x3b\x71\x20"
			  "\xf9\x35\x0c\x70\x74\xdc\x5b\xc2"
			  "\xb4\x11\x0e\x2c\x61\xa1\x52\x46"
			  "\x18\x11\x16\xc6\x86\x44\xa7\xaf"
			  "\xd5\x0c\x7d\xa6\x9e\x25\x2d\x1b"
			  "\x9a\x8f\x0f\xf8\x6a\x61\xa0\xea"
			  "\x3f\x0e\x90\xd6\x8f\x83\x30\x64"
			  "\xb5\x51\x2d\x08\x3c\xcd\x99\x36"
			  "\x96\xd4\xb1\xb5\x48\x30\xca\x48"
			  "\xf7\x11\xa8\xf5\x97\x8a\x6a\x6d"
			  "\x12\x33\x2f\xc0\xe8\xda\xec\x8a"
			  "\xe1\x88\x72\x63\xde\x20\xa3\xe1"
			  "\x8e\xac\x84\x37\x35\xf5\xf7\x3f"
			  "\x00\x02\x0e\xe4\xc1\x53\x68\x3f"
			  "\xaa\xd5\xac\x52\x3d\x20\x2f\x4d"
			  "\x7c\x83\xd0\xbd\xaa\x97\x35\x36"
			  "\x98\x88\x59\x5d\xe7\x24\xe3\x90"
			  "\x9d\x30\x47\xa7\xc3\x60\x35\xf4"
			  "\xd5\xdb\x0e\x4d\x44\xc1\x81\x8b"
			  "\xfd\xbd\xc3\x2b\xba\x68\xfe\x8d"
			  "\x49\x5a\x3c\x8a\xa3\x01\xae\x25"
			  "\x42\xab\xd2\x87\x1b\x35\xd6\xd2"
			  "\xd7\x70\x1c\x1f\x72\xd1\xe1\x39"
			  "\x1c\x58\xa2\xb4\xd0\x78\x55\x72"
			  "\x76\x59\xea\xd9\xd7\x6e\x63\x8b"
			  "\xcc\x9b\xa7\x74\x89\xfc\xa3\x68"
			  "\x86\x28\xd1\xbb\x54\x8d\x66\xad"
			  "\x2a\x92\xf9\x4e\x04\x3d\xae\xfd"
			  "\x1b\x2b\x7f\xc3\x2f\x1a\x78\x0a"
			  "\x5c\xc6\x84\xfe\x7c\xcb\x26\xfd"
			  "\xd9\x51\x0f\xd7\x94\x2f\xc5\xa7",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec serpent_lrw_dec_tv_template[] = {
	 
	{
		.key	= "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x6f\xbf\xd4\xa4\x5d\x71\x16\x79"
			  "\x63\x9c\xa6\x8e\x40\xbe\x0d\x8a",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input	= "\xfd\xb2\x66\x98\x80\x96\x55\xad"
			  "\x08\x94\x54\x9c\x21\x7c\x69\xe3",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x14\x5e\x3d\x70\xc0\x6e\x9c\x34"
			  "\x5b\x5e\xcf\x0f\xe4\x8c\x21\x5c",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x25\x39\xaa\xa5\xf0\x65\xc8\xdc"
			  "\x5d\x45\x95\x30\x8f\xff\x2f\x1b",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x0c\x20\x20\x63\xd6\x8b\xfc\x8f"
			  "\xc0\xe2\x17\xbb\xd2\x59\x6f\x26",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xc1\x35\x2e\x53\xf0\x96\x4d\x9c"
			  "\x2e\x18\xe6\x99\xcd\xd3\x15\x68",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x86\x0a\xc6\xa9\x1a\x9f\xe7\xe6"
			  "\x64\x3b\x33\xd6\xd5\x84\xd6\xdf",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\xe3\x5a\x38\x0f\x4d\x92\x3a\x74"
			  "\x15\xb1\x50\x8c\x9a\xd8\x99\x1d"
			  "\x82\xec\xf1\x5f\x03\x6d\x02\x58"
			  "\x90\x67\xfc\xdd\x8d\xe1\x38\x08"
			  "\x7b\xc9\x9b\x4b\x04\x09\x50\x15"
			  "\xce\xab\xda\x33\x30\x20\x12\xfa"
			  "\x83\xc4\xa6\x9a\x2e\x7d\x90\xd9"
			  "\xa6\xa6\x67\x43\xb4\xa7\xa8\x5c"
			  "\xbb\x6a\x49\x2b\x8b\xf8\xd0\x22"
			  "\xe5\x9e\xba\xe8\x8c\x67\xb8\x5b"
			  "\x60\xbc\xf5\xa4\x95\x4e\x66\xe5"
			  "\x6d\x8e\xa9\xf6\x65\x2e\x04\xf5"
			  "\xba\xb5\xdb\x88\xc2\xf6\x7a\x4b"
			  "\x89\x58\x7c\x9a\xae\x26\xe8\xb7"
			  "\xb7\x28\xcc\xd6\xcc\xa5\x98\x4d"
			  "\xb9\x91\xcb\xb4\xe4\x8b\x96\x47"
			  "\x5f\x03\x8b\xdd\x94\xd1\xee\x12"
			  "\xa7\x83\x80\xf2\xc1\x15\x74\x4f"
			  "\x49\xf9\xb0\x7e\x6f\xdc\x73\x2f"
			  "\xe2\xcf\xe0\x1b\x34\xa5\xa0\x52"
			  "\xfb\x3c\x5d\x85\x91\xe6\x6d\x98"
			  "\x04\xd6\xdd\x4c\x00\x64\xd9\x54"
			  "\x5c\x3c\x08\x1d\x4c\x06\x9f\xb8"
			  "\x1c\x4d\x8d\xdc\xa4\x3c\xb9\x3b"
			  "\x9e\x85\xce\xc3\xa8\x4a\x0c\xd9"
			  "\x04\xc3\x6f\x17\x66\xa9\x1f\x59"
			  "\xd9\xe2\x19\x36\xa3\x88\xb8\x0b"
			  "\x0f\x4a\x4d\xf8\xc8\x6f\xd5\x43"
			  "\xeb\xa0\xab\x1f\x61\xc0\x06\xeb"
			  "\x93\xb7\xb8\x6f\x0d\xbd\x07\x49"
			  "\xb3\xac\x5d\xcf\x31\xa0\x27\x26"
			  "\x21\xbe\x94\x2e\x19\xea\xf4\xee"
			  "\xb5\x13\x89\xf7\x94\x0b\xef\x59"
			  "\x44\xc5\x78\x8b\x3c\x3b\x71\x20"
			  "\xf9\x35\x0c\x70\x74\xdc\x5b\xc2"
			  "\xb4\x11\x0e\x2c\x61\xa1\x52\x46"
			  "\x18\x11\x16\xc6\x86\x44\xa7\xaf"
			  "\xd5\x0c\x7d\xa6\x9e\x25\x2d\x1b"
			  "\x9a\x8f\x0f\xf8\x6a\x61\xa0\xea"
			  "\x3f\x0e\x90\xd6\x8f\x83\x30\x64"
			  "\xb5\x51\x2d\x08\x3c\xcd\x99\x36"
			  "\x96\xd4\xb1\xb5\x48\x30\xca\x48"
			  "\xf7\x11\xa8\xf5\x97\x8a\x6a\x6d"
			  "\x12\x33\x2f\xc0\xe8\xda\xec\x8a"
			  "\xe1\x88\x72\x63\xde\x20\xa3\xe1"
			  "\x8e\xac\x84\x37\x35\xf5\xf7\x3f"
			  "\x00\x02\x0e\xe4\xc1\x53\x68\x3f"
			  "\xaa\xd5\xac\x52\x3d\x20\x2f\x4d"
			  "\x7c\x83\xd0\xbd\xaa\x97\x35\x36"
			  "\x98\x88\x59\x5d\xe7\x24\xe3\x90"
			  "\x9d\x30\x47\xa7\xc3\x60\x35\xf4"
			  "\xd5\xdb\x0e\x4d\x44\xc1\x81\x8b"
			  "\xfd\xbd\xc3\x2b\xba\x68\xfe\x8d"
			  "\x49\x5a\x3c\x8a\xa3\x01\xae\x25"
			  "\x42\xab\xd2\x87\x1b\x35\xd6\xd2"
			  "\xd7\x70\x1c\x1f\x72\xd1\xe1\x39"
			  "\x1c\x58\xa2\xb4\xd0\x78\x55\x72"
			  "\x76\x59\xea\xd9\xd7\x6e\x63\x8b"
			  "\xcc\x9b\xa7\x74\x89\xfc\xa3\x68"
			  "\x86\x28\xd1\xbb\x54\x8d\x66\xad"
			  "\x2a\x92\xf9\x4e\x04\x3d\xae\xfd"
			  "\x1b\x2b\x7f\xc3\x2f\x1a\x78\x0a"
			  "\x5c\xc6\x84\xfe\x7c\xcb\x26\xfd"
			  "\xd9\x51\x0f\xd7\x94\x2f\xc5\xa7",
		.ilen	= 512,
		.result	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec serpent_xts_enc_tv_template[] = {
	 
	{
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 32,
		.result	= "\xe1\x08\xb8\x1d\x2c\xf5\x33\x64"
			  "\xc8\x12\x04\xc7\xb3\x70\xe8\xc4"
			  "\x6a\x31\xc5\xf3\x00\xca\xb9\x16"
			  "\xde\xe2\x77\x66\xf7\xfe\x62\x08",
		.rlen	= 32,
	}, {
		.key	= "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen	= 32,
		.result	= "\x1a\x0a\x09\x5f\xcd\x07\x07\x98"
			  "\x41\x86\x12\xaf\xb3\xd7\x68\x13"
			  "\xed\x81\xcd\x06\x87\x43\x1a\xbb"
			  "\x13\x3d\xd6\x1e\x2b\xe1\x77\xbe",
		.rlen	= 32,
	}, {
		.key	= "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen	= 32,
		.result	= "\xf9\x9b\x28\xb8\x5c\xaf\x8c\x61"
			  "\xb6\x1c\x81\x8f\x2c\x87\x60\x89"
			  "\x0d\x8d\x7a\xe8\x60\x48\xcc\x86"
			  "\xc1\x68\x45\xaa\x00\xe9\x24\xc5",
		.rlen	= 32,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\xfe\x47\x4a\xc8\x60\x7e\xb4\x8b"
			  "\x0d\x10\xf4\xb0\x0d\xba\xf8\x53"
			  "\x65\x6e\x38\x4b\xdb\xaa\xb1\x9e"
			  "\x28\xca\xb0\x22\xb3\x85\x75\xf4"
			  "\x00\x5c\x75\x14\x06\xd6\x25\x82"
			  "\xe6\xcb\x08\xf7\x29\x90\x23\x8e"
			  "\xa4\x68\x57\xe4\xf0\xd8\x32\xf3"
			  "\x80\x51\x67\xb5\x0b\x85\x69\xe8"
			  "\x19\xfe\xc4\xc7\x3e\xea\x90\xd3"
			  "\x8f\xa3\xf2\x0a\xac\x17\x4b\xa0"
			  "\x63\x5a\x16\x0f\xf0\xce\x66\x1f"
			  "\x2c\x21\x07\xf1\xa4\x03\xa3\x44"
			  "\x41\x61\x87\x5d\x6b\xb3\xef\xd4"
			  "\xfc\xaa\x32\x7e\x55\x58\x04\x41"
			  "\xc9\x07\x33\xc6\xa2\x68\xd6\x5a"
			  "\x55\x79\x4b\x6f\xcf\x89\xb9\x19"
			  "\xe5\x54\x13\x15\xb2\x1a\xfa\x15"
			  "\xc2\xf0\x06\x59\xfa\xa0\x25\x05"
			  "\x58\xfa\x43\x91\x16\x85\x40\xbb"
			  "\x0d\x34\x4d\xc5\x1e\x20\xd5\x08"
			  "\xcd\x22\x22\x41\x11\x9f\x6c\x7c"
			  "\x8d\x57\xc9\xba\x57\xe8\x2c\xf7"
			  "\xa0\x42\xa8\xde\xfc\xa3\xca\x98"
			  "\x4b\x43\xb1\xce\x4b\xbf\x01\x67"
			  "\x6e\x29\x60\xbd\x10\x14\x84\x82"
			  "\x83\x82\x0c\x63\x73\x92\x02\x7c"
			  "\x55\x37\x20\x80\x17\x51\xc8\xbc"
			  "\x46\x02\xcb\x38\x07\x6d\xe2\x85"
			  "\xaa\x29\xaf\x24\x58\x0d\xf0\x75"
			  "\x08\x0a\xa5\x34\x25\x16\xf3\x74"
			  "\xa7\x0b\x97\xbe\xc1\xa9\xdc\x29"
			  "\x1a\x0a\x56\xc1\x1a\x91\x97\x8c"
			  "\x0b\xc7\x16\xed\x5a\x22\xa6\x2e"
			  "\x8c\x2b\x4f\x54\x76\x47\x53\x8e"
			  "\xe8\x00\xec\x92\xb9\x55\xe6\xa2"
			  "\xf3\xe2\x4f\x6a\x66\x60\xd0\x87"
			  "\xe6\xd1\xcc\xe3\x6a\xc5\x2d\x21"
			  "\xcc\x9d\x6a\xb6\x75\xaa\xe2\x19"
			  "\x21\x9f\xa1\x5e\x4c\xfd\x72\xf9"
			  "\x94\x4e\x63\xc7\xae\xfc\xed\x47"
			  "\xe2\xfe\x7a\x63\x77\xfe\x97\x82"
			  "\xb1\x10\x6e\x36\x1d\xe1\xc4\x80"
			  "\xec\x69\x41\xec\xa7\x8a\xe0\x2f"
			  "\xe3\x49\x26\xa2\x41\xb2\x08\x0f"
			  "\x28\xb4\xa7\x39\xa1\x99\x2d\x1e"
			  "\x43\x42\x35\xd0\xcf\xec\x77\x67"
			  "\xb2\x3b\x9e\x1c\x35\xde\x4f\x5e"
			  "\x73\x3f\x5d\x6f\x07\x4b\x2e\x50"
			  "\xab\x6c\x6b\xff\xea\x00\x67\xaa"
			  "\x0e\x82\x32\xdd\x3d\xb5\xe5\x76"
			  "\x2b\x77\x3f\xbe\x12\x75\xfb\x92"
			  "\xc6\x89\x67\x4d\xca\xf7\xd4\x50"
			  "\xc0\x74\x47\xcc\xd9\x0a\xd4\xc6"
			  "\x3b\x17\x2e\xe3\x35\xbb\x53\xb5"
			  "\x86\xad\x51\xcc\xd5\x96\xb8\xdc"
			  "\x03\x57\xe6\x98\x52\x2f\x61\x62"
			  "\xc4\x5c\x9c\x36\x71\x07\xfb\x94"
			  "\xe3\x02\xc4\x2b\x08\x75\xc7\x35"
			  "\xfb\x2e\x88\x7b\xbb\x67\x00\xe1"
			  "\xc9\xdd\x99\xb2\x13\x53\x1a\x4e"
			  "\x76\x87\x19\x04\x1a\x2f\x38\x3e"
			  "\xef\x91\x64\x1d\x18\x07\x4e\x31"
			  "\x88\x21\x7c\xb0\xa5\x12\x4c\x3c"
			  "\xb0\x20\xbd\xda\xdf\xf9\x7c\xdd",
		.rlen	= 512,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\x2b\xc9\xb4\x6b\x10\x94\xa9\x32"
			  "\xaa\xb0\x20\xc6\x44\x3d\x74\x1f"
			  "\x75\x01\xa7\xf6\xf5\xf7\x62\x1b"
			  "\x80\x1b\x82\xcb\x01\x59\x91\x7f"
			  "\x80\x3a\x98\xf0\xd2\xca\xc4\xc3"
			  "\x34\xfd\xe6\x11\xf9\x33\x45\x12"
			  "\x48\xc5\x8c\x25\xf1\xc5\xc5\x23"
			  "\xd3\x44\xb4\x73\xd5\x04\xc0\xb7"
			  "\xca\x2f\xf5\xcd\xc5\xb4\xdd\xb0"
			  "\xf4\x60\xe8\xfb\xc6\x9c\xc5\x78"
			  "\xcd\xec\x7d\xdc\x19\x9c\x72\x64"
			  "\x63\x0b\x38\x2e\x76\xdd\x2d\x36"
			  "\x49\xb0\x1d\xea\x78\x9e\x00\xca"
			  "\x20\xcc\x1b\x1e\x98\x74\xab\xed"
			  "\x79\xf7\xd0\x6c\xd8\x93\x80\x29"
			  "\xac\xa5\x5e\x34\xa9\xab\xa0\x55"
			  "\x9a\xea\xaa\x95\x4d\x7b\xfe\x46"
			  "\x26\x8a\xfd\x88\xa2\xa8\xa6\xae"
			  "\x25\x42\x17\xbf\x76\x8f\x1c\x3d"
			  "\xec\x9a\xda\x64\x96\xb5\x61\xff"
			  "\x99\xeb\x12\x96\x85\x82\x9d\xd5"
			  "\x81\x85\x14\xa8\x59\xac\x8c\x94"
			  "\xbb\x3b\x85\x2b\xdf\xb3\x0c\xba"
			  "\x82\xc6\x4d\xca\x86\xea\x53\x28"
			  "\x4c\xe0\x4e\x31\xe3\x73\x2f\x79"
			  "\x9d\x42\xe1\x03\xe3\x8b\xc4\xff"
			  "\x05\xca\x81\x7b\xda\xa2\xde\x63"
			  "\x3a\x10\xbe\xc2\xac\x32\xc4\x05"
			  "\x47\x7e\xef\x67\xe2\x5f\x5b\xae"
			  "\xed\xf1\x70\x34\x16\x9a\x07\x7b"
			  "\xf2\x25\x2b\xb0\xf8\x3c\x15\x9a"
			  "\xa6\x59\x55\x5f\xc1\xf4\x1e\xcd"
			  "\x93\x1f\x06\xba\xd4\x9a\x22\x69"
			  "\xfa\x8e\x95\x0d\xf3\x23\x59\x2c"
			  "\xfe\x00\xba\xf0\x0e\xbc\x6d\xd6"
			  "\x62\xf0\x7a\x0e\x83\x3e\xdb\x32"
			  "\xfd\x43\x7d\xda\x42\x51\x87\x43"
			  "\x9d\xf9\xef\xf4\x30\x97\xf8\x09"
			  "\x88\xfc\x3f\x93\x70\xc1\x4a\xec"
			  "\x27\x5f\x11\xac\x71\xc7\x48\x46"
			  "\x2f\xf9\xdf\x8d\x9f\xf7\x2e\x56"
			  "\x0d\x4e\xb0\x32\x76\xce\x86\x81"
			  "\xcd\xdf\xe4\x00\xbf\xfd\x5f\x24"
			  "\xaf\xf7\x9a\xde\xff\x18\xac\x14"
			  "\x90\xc5\x01\x39\x34\x0f\x24\xf3"
			  "\x13\x2f\x5e\x4f\x30\x9a\x36\x40"
			  "\xec\xea\xbc\xcd\x9e\x0e\x5b\x23"
			  "\x50\x88\x97\x40\x69\xb1\x37\xf5"
			  "\xc3\x15\xf9\x3f\xb7\x79\x64\xe8"
			  "\x7b\x10\x20\xb9\x2b\x46\x83\x5b"
			  "\xd8\x39\xfc\xe4\xfa\x88\x52\xf2"
			  "\x72\xb0\x97\x4e\x89\xb3\x48\x00"
			  "\xc1\x16\x73\x50\x77\xba\xa6\x65"
			  "\x20\x2d\xb0\x02\x27\x89\xda\x99"
			  "\x45\xfb\xe9\xd3\x1d\x39\x2f\xd6"
			  "\x2a\xda\x09\x12\x11\xaf\xe6\x57"
			  "\x01\x04\x8a\xff\x86\x8b\xac\xf8"
			  "\xee\xe4\x1c\x98\x5b\xcf\x6b\x76"
			  "\xa3\x0e\x33\x74\x40\x18\x39\x72"
			  "\x66\x50\x31\xfd\x70\xdf\xe8\x51"
			  "\x96\x21\x36\xb2\x9b\xfa\x85\xd1"
			  "\x30\x05\xc8\x92\x98\x80\xff\x7a"
			  "\xaf\x43\x0b\xc5\x20\x41\x92\x20"
			  "\xd4\xa0\x91\x98\x11\x5f\x4d\xb1",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec serpent_xts_dec_tv_template[] = {
	 
	{
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xe1\x08\xb8\x1d\x2c\xf5\x33\x64"
			  "\xc8\x12\x04\xc7\xb3\x70\xe8\xc4"
			  "\x6a\x31\xc5\xf3\x00\xca\xb9\x16"
			  "\xde\xe2\x77\x66\xf7\xfe\x62\x08",
		.ilen	= 32,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 32,
	}, {
		.key	= "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x1a\x0a\x09\x5f\xcd\x07\x07\x98"
			  "\x41\x86\x12\xaf\xb3\xd7\x68\x13"
			  "\xed\x81\xcd\x06\x87\x43\x1a\xbb"
			  "\x13\x3d\xd6\x1e\x2b\xe1\x77\xbe",
		.ilen	= 32,
		.result	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen	= 32,
	}, {
		.key	= "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xf9\x9b\x28\xb8\x5c\xaf\x8c\x61"
			  "\xb6\x1c\x81\x8f\x2c\x87\x60\x89"
			  "\x0d\x8d\x7a\xe8\x60\x48\xcc\x86"
			  "\xc1\x68\x45\xaa\x00\xe9\x24\xc5",
		.ilen	= 32,
		.result	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen	= 32,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xfe\x47\x4a\xc8\x60\x7e\xb4\x8b"
			  "\x0d\x10\xf4\xb0\x0d\xba\xf8\x53"
			  "\x65\x6e\x38\x4b\xdb\xaa\xb1\x9e"
			  "\x28\xca\xb0\x22\xb3\x85\x75\xf4"
			  "\x00\x5c\x75\x14\x06\xd6\x25\x82"
			  "\xe6\xcb\x08\xf7\x29\x90\x23\x8e"
			  "\xa4\x68\x57\xe4\xf0\xd8\x32\xf3"
			  "\x80\x51\x67\xb5\x0b\x85\x69\xe8"
			  "\x19\xfe\xc4\xc7\x3e\xea\x90\xd3"
			  "\x8f\xa3\xf2\x0a\xac\x17\x4b\xa0"
			  "\x63\x5a\x16\x0f\xf0\xce\x66\x1f"
			  "\x2c\x21\x07\xf1\xa4\x03\xa3\x44"
			  "\x41\x61\x87\x5d\x6b\xb3\xef\xd4"
			  "\xfc\xaa\x32\x7e\x55\x58\x04\x41"
			  "\xc9\x07\x33\xc6\xa2\x68\xd6\x5a"
			  "\x55\x79\x4b\x6f\xcf\x89\xb9\x19"
			  "\xe5\x54\x13\x15\xb2\x1a\xfa\x15"
			  "\xc2\xf0\x06\x59\xfa\xa0\x25\x05"
			  "\x58\xfa\x43\x91\x16\x85\x40\xbb"
			  "\x0d\x34\x4d\xc5\x1e\x20\xd5\x08"
			  "\xcd\x22\x22\x41\x11\x9f\x6c\x7c"
			  "\x8d\x57\xc9\xba\x57\xe8\x2c\xf7"
			  "\xa0\x42\xa8\xde\xfc\xa3\xca\x98"
			  "\x4b\x43\xb1\xce\x4b\xbf\x01\x67"
			  "\x6e\x29\x60\xbd\x10\x14\x84\x82"
			  "\x83\x82\x0c\x63\x73\x92\x02\x7c"
			  "\x55\x37\x20\x80\x17\x51\xc8\xbc"
			  "\x46\x02\xcb\x38\x07\x6d\xe2\x85"
			  "\xaa\x29\xaf\x24\x58\x0d\xf0\x75"
			  "\x08\x0a\xa5\x34\x25\x16\xf3\x74"
			  "\xa7\x0b\x97\xbe\xc1\xa9\xdc\x29"
			  "\x1a\x0a\x56\xc1\x1a\x91\x97\x8c"
			  "\x0b\xc7\x16\xed\x5a\x22\xa6\x2e"
			  "\x8c\x2b\x4f\x54\x76\x47\x53\x8e"
			  "\xe8\x00\xec\x92\xb9\x55\xe6\xa2"
			  "\xf3\xe2\x4f\x6a\x66\x60\xd0\x87"
			  "\xe6\xd1\xcc\xe3\x6a\xc5\x2d\x21"
			  "\xcc\x9d\x6a\xb6\x75\xaa\xe2\x19"
			  "\x21\x9f\xa1\x5e\x4c\xfd\x72\xf9"
			  "\x94\x4e\x63\xc7\xae\xfc\xed\x47"
			  "\xe2\xfe\x7a\x63\x77\xfe\x97\x82"
			  "\xb1\x10\x6e\x36\x1d\xe1\xc4\x80"
			  "\xec\x69\x41\xec\xa7\x8a\xe0\x2f"
			  "\xe3\x49\x26\xa2\x41\xb2\x08\x0f"
			  "\x28\xb4\xa7\x39\xa1\x99\x2d\x1e"
			  "\x43\x42\x35\xd0\xcf\xec\x77\x67"
			  "\xb2\x3b\x9e\x1c\x35\xde\x4f\x5e"
			  "\x73\x3f\x5d\x6f\x07\x4b\x2e\x50"
			  "\xab\x6c\x6b\xff\xea\x00\x67\xaa"
			  "\x0e\x82\x32\xdd\x3d\xb5\xe5\x76"
			  "\x2b\x77\x3f\xbe\x12\x75\xfb\x92"
			  "\xc6\x89\x67\x4d\xca\xf7\xd4\x50"
			  "\xc0\x74\x47\xcc\xd9\x0a\xd4\xc6"
			  "\x3b\x17\x2e\xe3\x35\xbb\x53\xb5"
			  "\x86\xad\x51\xcc\xd5\x96\xb8\xdc"
			  "\x03\x57\xe6\x98\x52\x2f\x61\x62"
			  "\xc4\x5c\x9c\x36\x71\x07\xfb\x94"
			  "\xe3\x02\xc4\x2b\x08\x75\xc7\x35"
			  "\xfb\x2e\x88\x7b\xbb\x67\x00\xe1"
			  "\xc9\xdd\x99\xb2\x13\x53\x1a\x4e"
			  "\x76\x87\x19\x04\x1a\x2f\x38\x3e"
			  "\xef\x91\x64\x1d\x18\x07\x4e\x31"
			  "\x88\x21\x7c\xb0\xa5\x12\x4c\x3c"
			  "\xb0\x20\xbd\xda\xdf\xf9\x7c\xdd",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x2b\xc9\xb4\x6b\x10\x94\xa9\x32"
			  "\xaa\xb0\x20\xc6\x44\x3d\x74\x1f"
			  "\x75\x01\xa7\xf6\xf5\xf7\x62\x1b"
			  "\x80\x1b\x82\xcb\x01\x59\x91\x7f"
			  "\x80\x3a\x98\xf0\xd2\xca\xc4\xc3"
			  "\x34\xfd\xe6\x11\xf9\x33\x45\x12"
			  "\x48\xc5\x8c\x25\xf1\xc5\xc5\x23"
			  "\xd3\x44\xb4\x73\xd5\x04\xc0\xb7"
			  "\xca\x2f\xf5\xcd\xc5\xb4\xdd\xb0"
			  "\xf4\x60\xe8\xfb\xc6\x9c\xc5\x78"
			  "\xcd\xec\x7d\xdc\x19\x9c\x72\x64"
			  "\x63\x0b\x38\x2e\x76\xdd\x2d\x36"
			  "\x49\xb0\x1d\xea\x78\x9e\x00\xca"
			  "\x20\xcc\x1b\x1e\x98\x74\xab\xed"
			  "\x79\xf7\xd0\x6c\xd8\x93\x80\x29"
			  "\xac\xa5\x5e\x34\xa9\xab\xa0\x55"
			  "\x9a\xea\xaa\x95\x4d\x7b\xfe\x46"
			  "\x26\x8a\xfd\x88\xa2\xa8\xa6\xae"
			  "\x25\x42\x17\xbf\x76\x8f\x1c\x3d"
			  "\xec\x9a\xda\x64\x96\xb5\x61\xff"
			  "\x99\xeb\x12\x96\x85\x82\x9d\xd5"
			  "\x81\x85\x14\xa8\x59\xac\x8c\x94"
			  "\xbb\x3b\x85\x2b\xdf\xb3\x0c\xba"
			  "\x82\xc6\x4d\xca\x86\xea\x53\x28"
			  "\x4c\xe0\x4e\x31\xe3\x73\x2f\x79"
			  "\x9d\x42\xe1\x03\xe3\x8b\xc4\xff"
			  "\x05\xca\x81\x7b\xda\xa2\xde\x63"
			  "\x3a\x10\xbe\xc2\xac\x32\xc4\x05"
			  "\x47\x7e\xef\x67\xe2\x5f\x5b\xae"
			  "\xed\xf1\x70\x34\x16\x9a\x07\x7b"
			  "\xf2\x25\x2b\xb0\xf8\x3c\x15\x9a"
			  "\xa6\x59\x55\x5f\xc1\xf4\x1e\xcd"
			  "\x93\x1f\x06\xba\xd4\x9a\x22\x69"
			  "\xfa\x8e\x95\x0d\xf3\x23\x59\x2c"
			  "\xfe\x00\xba\xf0\x0e\xbc\x6d\xd6"
			  "\x62\xf0\x7a\x0e\x83\x3e\xdb\x32"
			  "\xfd\x43\x7d\xda\x42\x51\x87\x43"
			  "\x9d\xf9\xef\xf4\x30\x97\xf8\x09"
			  "\x88\xfc\x3f\x93\x70\xc1\x4a\xec"
			  "\x27\x5f\x11\xac\x71\xc7\x48\x46"
			  "\x2f\xf9\xdf\x8d\x9f\xf7\x2e\x56"
			  "\x0d\x4e\xb0\x32\x76\xce\x86\x81"
			  "\xcd\xdf\xe4\x00\xbf\xfd\x5f\x24"
			  "\xaf\xf7\x9a\xde\xff\x18\xac\x14"
			  "\x90\xc5\x01\x39\x34\x0f\x24\xf3"
			  "\x13\x2f\x5e\x4f\x30\x9a\x36\x40"
			  "\xec\xea\xbc\xcd\x9e\x0e\x5b\x23"
			  "\x50\x88\x97\x40\x69\xb1\x37\xf5"
			  "\xc3\x15\xf9\x3f\xb7\x79\x64\xe8"
			  "\x7b\x10\x20\xb9\x2b\x46\x83\x5b"
			  "\xd8\x39\xfc\xe4\xfa\x88\x52\xf2"
			  "\x72\xb0\x97\x4e\x89\xb3\x48\x00"
			  "\xc1\x16\x73\x50\x77\xba\xa6\x65"
			  "\x20\x2d\xb0\x02\x27\x89\xda\x99"
			  "\x45\xfb\xe9\xd3\x1d\x39\x2f\xd6"
			  "\x2a\xda\x09\x12\x11\xaf\xe6\x57"
			  "\x01\x04\x8a\xff\x86\x8b\xac\xf8"
			  "\xee\xe4\x1c\x98\x5b\xcf\x6b\x76"
			  "\xa3\x0e\x33\x74\x40\x18\x39\x72"
			  "\x66\x50\x31\xfd\x70\xdf\xe8\x51"
			  "\x96\x21\x36\xb2\x9b\xfa\x85\xd1"
			  "\x30\x05\xc8\x92\x98\x80\xff\x7a"
			  "\xaf\x43\x0b\xc5\x20\x41\x92\x20"
			  "\xd4\xa0\x91\x98\x11\x5f\x4d\xb1",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

#define CAST6_ENC_TEST_VECTORS		4
#define CAST6_DEC_TEST_VECTORS		4
#define CAST6_CBC_ENC_TEST_VECTORS	1
#define CAST6_CBC_DEC_TEST_VECTORS	1
#define CAST6_CTR_ENC_TEST_VECTORS	2
#define CAST6_CTR_DEC_TEST_VECTORS	2
#define CAST6_LRW_ENC_TEST_VECTORS	1
#define CAST6_LRW_DEC_TEST_VECTORS	1
#define CAST6_XTS_ENC_TEST_VECTORS	1
#define CAST6_XTS_DEC_TEST_VECTORS	1

static struct cipher_testvec cast6_enc_tv_template[] = {
	{
		.key	= "\x23\x42\xbb\x9e\xfa\x38\x54\x2c"
			  "\x0a\xf7\x56\x47\xf2\x9f\x61\x5d",
		.klen	= 16,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\xc8\x42\xa0\x89\x72\xb4\x3d\x20"
			  "\x83\x6c\x91\xd1\xb7\x53\x0f\x6b",
		.rlen	= 16,
	}, {
		.key	= "\x23\x42\xbb\x9e\xfa\x38\x54\x2c"
			  "\xbe\xd0\xac\x83\x94\x0a\xc2\x98"
			  "\xba\xc7\x7a\x77\x17\x94\x28\x63",
		.klen	= 24,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\x1b\x38\x6c\x02\x10\xdc\xad\xcb"
			  "\xdd\x0e\x41\xaa\x08\xa7\xa7\xe8",
		.rlen	= 16,
	}, {
		.key	= "\x23\x42\xbb\x9e\xfa\x38\x54\x2c"
			  "\xbe\xd0\xac\x83\x94\x0a\xc2\x98"
			  "\x8d\x7c\x47\xce\x26\x49\x08\x46"
			  "\x1c\xc1\xb5\x13\x7a\xe6\xb6\x04",
		.klen	= 32,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\x4f\x6a\x20\x38\x28\x68\x97\xb9"
			  "\xc9\x87\x01\x36\x55\x33\x17\xfa",
		.rlen	= 16,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xC3\x70\x22\x32\xF5\x80\xCB\x54"
			  "\xFC\x30\xE0\xF6\xEB\x39\x57\xA6"
			  "\xB6\xB9\xC5\xA4\x91\x55\x14\x97"
			  "\xC1\x20\xFF\x6C\x5C\xF0\x67\xEA"
			  "\x2F\xED\xD8\xC9\xFB\x38\x3F\xFE"
			  "\x93\xBE\xDC\x00\xD3\x7F\xAD\x4C"
			  "\x5A\x08\x92\xD1\x47\x0C\xFA\x6C"
			  "\xD0\x6A\x99\x10\x72\xF8\x47\x62"
			  "\x81\x42\xF8\xD8\xF5\xBB\x94\x08"
			  "\xAA\x97\xA2\x8B\x69\xB3\xD2\x7E"
			  "\xBC\xB5\x00\x0C\xE5\x44\x4B\x58"
			  "\xE8\x63\xDC\xB3\xC4\xE5\x23\x12"
			  "\x5A\x72\x85\x47\x8B\xEC\x9F\x26"
			  "\x84\xB6\xED\x10\x33\x63\x9B\x5F"
			  "\x4D\x53\xEE\x94\x45\x8B\x60\x58"
			  "\x86\x20\xF9\x1E\x82\x08\x3E\x58"
			  "\x60\x1B\x34\x19\x02\xBE\x4E\x09"
			  "\xBB\x7C\x15\xCC\x60\x27\x55\x7A"
			  "\x12\xB8\xD8\x08\x89\x3C\xA6\xF3"
			  "\xF1\xDD\xA7\x07\xA3\x12\x85\x28"
			  "\xE9\x57\xAC\x80\x0C\x5C\x0F\x3A"
			  "\x5D\xC2\x91\xC7\x90\xE4\x8C\x43"
			  "\x92\xE4\x7C\x26\x69\x4D\x83\x68"
			  "\x14\x96\x42\x47\xBD\xA9\xE4\x8A"
			  "\x33\x19\xEB\x54\x8E\x0D\x4B\x6E"
			  "\x91\x51\xB5\x36\x08\xDE\x1C\x06"
			  "\x03\xBD\xDE\x81\x26\xF7\x99\xC2"
			  "\xBA\xF7\x6D\x87\x0D\xE4\xA6\xCF"
			  "\xC1\xF5\x27\x05\xB8\x02\x57\x72"
			  "\xE6\x42\x13\x0B\xC6\x47\x05\x74"
			  "\x24\x15\xF7\x0D\xC2\x23\x9D\xB9"
			  "\x3C\x77\x18\x93\xBA\xB4\xFC\x8C"
			  "\x98\x82\x67\x67\xB4\xD7\xD3\x43"
			  "\x23\x08\x02\xB7\x9B\x99\x05\xFB"
			  "\xD3\xB5\x00\x0A\xA9\x9D\x66\xD6"
			  "\x2E\x49\x58\xD0\xA8\x57\x29\x7F"
			  "\x0A\x0E\x7D\xFC\x92\x83\xCC\x67"
			  "\xA2\xB1\x70\x3A\x8F\x87\x4A\x8D"
			  "\x17\xE2\x58\x2B\x88\x0D\x68\x62"
			  "\xBF\x35\xD1\x6F\xC0\xF0\x18\x62"
			  "\xB2\xC7\x2D\x58\xC7\x16\xDE\x08"
			  "\xEB\x84\x1D\x25\xA7\x38\x94\x06"
			  "\x93\x9D\xF8\xFE\x88\x71\xE7\x84"
			  "\x2C\xA0\x38\xA3\x1D\x48\xCF\x29"
			  "\x0B\xBC\xD8\x50\x99\x1A\x26\xFB"
			  "\x8E\x75\x3D\x73\xEB\x6A\xED\x29"
			  "\xE0\x8E\xED\xFC\xFE\x6F\xF6\xBA"
			  "\x41\xE2\x10\x4C\x01\x8B\x69\x2B"
			  "\x25\x3F\x4D\x70\x7B\x92\xD6\x3B"
			  "\xAC\xF9\x77\x18\xD9\x6A\x30\xA6"
			  "\x2E\xFA\x30\xFF\xC8\xD5\x1D\x06"
			  "\x59\x28\x1D\x86\x43\x04\x5D\x3B"
			  "\x99\x4C\x04\x5A\x21\x17\x8B\x76"
			  "\x8F\x72\xCB\xA1\x9C\x29\x4C\xC3"
			  "\x65\xA2\x58\x2A\xC5\x66\x24\xBF"
			  "\xBA\xE6\x0C\xDD\x34\x24\x74\xC8"
			  "\x84\x0A\x66\x2C\xBE\x8F\x32\xA9"
			  "\xE7\xE4\xA1\xD7\xDA\xAB\x23\x1E"
			  "\xEB\xEE\x6C\x94\x6F\x9C\x2E\xD1"
			  "\x49\x2C\xF3\xD4\x90\xCC\x93\x4C"
			  "\x84\x52\x6D\x68\xDE\xC6\x64\xB2"
			  "\x11\x74\x93\x57\xB4\x7E\xC6\x00",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast6_dec_tv_template[] = {
	{
		.key	= "\x23\x42\xbb\x9e\xfa\x38\x54\x2c"
			  "\x0a\xf7\x56\x47\xf2\x9f\x61\x5d",
		.klen	= 16,
		.input	= "\xc8\x42\xa0\x89\x72\xb4\x3d\x20"
			  "\x83\x6c\x91\xd1\xb7\x53\x0f\x6b",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\x23\x42\xbb\x9e\xfa\x38\x54\x2c"
			  "\xbe\xd0\xac\x83\x94\x0a\xc2\x98"
			  "\xba\xc7\x7a\x77\x17\x94\x28\x63",
		.klen	= 24,
		.input	= "\x1b\x38\x6c\x02\x10\xdc\xad\xcb"
			  "\xdd\x0e\x41\xaa\x08\xa7\xa7\xe8",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\x23\x42\xbb\x9e\xfa\x38\x54\x2c"
			  "\xbe\xd0\xac\x83\x94\x0a\xc2\x98"
			  "\x8d\x7c\x47\xce\x26\x49\x08\x46"
			  "\x1c\xc1\xb5\x13\x7a\xe6\xb6\x04",
		.klen	= 32,
		.input	= "\x4f\x6a\x20\x38\x28\x68\x97\xb9"
			  "\xc9\x87\x01\x36\x55\x33\x17\xfa",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xC3\x70\x22\x32\xF5\x80\xCB\x54"
			  "\xFC\x30\xE0\xF6\xEB\x39\x57\xA6"
			  "\xB6\xB9\xC5\xA4\x91\x55\x14\x97"
			  "\xC1\x20\xFF\x6C\x5C\xF0\x67\xEA"
			  "\x2F\xED\xD8\xC9\xFB\x38\x3F\xFE"
			  "\x93\xBE\xDC\x00\xD3\x7F\xAD\x4C"
			  "\x5A\x08\x92\xD1\x47\x0C\xFA\x6C"
			  "\xD0\x6A\x99\x10\x72\xF8\x47\x62"
			  "\x81\x42\xF8\xD8\xF5\xBB\x94\x08"
			  "\xAA\x97\xA2\x8B\x69\xB3\xD2\x7E"
			  "\xBC\xB5\x00\x0C\xE5\x44\x4B\x58"
			  "\xE8\x63\xDC\xB3\xC4\xE5\x23\x12"
			  "\x5A\x72\x85\x47\x8B\xEC\x9F\x26"
			  "\x84\xB6\xED\x10\x33\x63\x9B\x5F"
			  "\x4D\x53\xEE\x94\x45\x8B\x60\x58"
			  "\x86\x20\xF9\x1E\x82\x08\x3E\x58"
			  "\x60\x1B\x34\x19\x02\xBE\x4E\x09"
			  "\xBB\x7C\x15\xCC\x60\x27\x55\x7A"
			  "\x12\xB8\xD8\x08\x89\x3C\xA6\xF3"
			  "\xF1\xDD\xA7\x07\xA3\x12\x85\x28"
			  "\xE9\x57\xAC\x80\x0C\x5C\x0F\x3A"
			  "\x5D\xC2\x91\xC7\x90\xE4\x8C\x43"
			  "\x92\xE4\x7C\x26\x69\x4D\x83\x68"
			  "\x14\x96\x42\x47\xBD\xA9\xE4\x8A"
			  "\x33\x19\xEB\x54\x8E\x0D\x4B\x6E"
			  "\x91\x51\xB5\x36\x08\xDE\x1C\x06"
			  "\x03\xBD\xDE\x81\x26\xF7\x99\xC2"
			  "\xBA\xF7\x6D\x87\x0D\xE4\xA6\xCF"
			  "\xC1\xF5\x27\x05\xB8\x02\x57\x72"
			  "\xE6\x42\x13\x0B\xC6\x47\x05\x74"
			  "\x24\x15\xF7\x0D\xC2\x23\x9D\xB9"
			  "\x3C\x77\x18\x93\xBA\xB4\xFC\x8C"
			  "\x98\x82\x67\x67\xB4\xD7\xD3\x43"
			  "\x23\x08\x02\xB7\x9B\x99\x05\xFB"
			  "\xD3\xB5\x00\x0A\xA9\x9D\x66\xD6"
			  "\x2E\x49\x58\xD0\xA8\x57\x29\x7F"
			  "\x0A\x0E\x7D\xFC\x92\x83\xCC\x67"
			  "\xA2\xB1\x70\x3A\x8F\x87\x4A\x8D"
			  "\x17\xE2\x58\x2B\x88\x0D\x68\x62"
			  "\xBF\x35\xD1\x6F\xC0\xF0\x18\x62"
			  "\xB2\xC7\x2D\x58\xC7\x16\xDE\x08"
			  "\xEB\x84\x1D\x25\xA7\x38\x94\x06"
			  "\x93\x9D\xF8\xFE\x88\x71\xE7\x84"
			  "\x2C\xA0\x38\xA3\x1D\x48\xCF\x29"
			  "\x0B\xBC\xD8\x50\x99\x1A\x26\xFB"
			  "\x8E\x75\x3D\x73\xEB\x6A\xED\x29"
			  "\xE0\x8E\xED\xFC\xFE\x6F\xF6\xBA"
			  "\x41\xE2\x10\x4C\x01\x8B\x69\x2B"
			  "\x25\x3F\x4D\x70\x7B\x92\xD6\x3B"
			  "\xAC\xF9\x77\x18\xD9\x6A\x30\xA6"
			  "\x2E\xFA\x30\xFF\xC8\xD5\x1D\x06"
			  "\x59\x28\x1D\x86\x43\x04\x5D\x3B"
			  "\x99\x4C\x04\x5A\x21\x17\x8B\x76"
			  "\x8F\x72\xCB\xA1\x9C\x29\x4C\xC3"
			  "\x65\xA2\x58\x2A\xC5\x66\x24\xBF"
			  "\xBA\xE6\x0C\xDD\x34\x24\x74\xC8"
			  "\x84\x0A\x66\x2C\xBE\x8F\x32\xA9"
			  "\xE7\xE4\xA1\xD7\xDA\xAB\x23\x1E"
			  "\xEB\xEE\x6C\x94\x6F\x9C\x2E\xD1"
			  "\x49\x2C\xF3\xD4\x90\xCC\x93\x4C"
			  "\x84\x52\x6D\x68\xDE\xC6\x64\xB2"
			  "\x11\x74\x93\x57\xB4\x7E\xC6\x00",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast6_cbc_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xDF\x77\x68\x96\xC7\xBA\xF8\xE2"
			  "\x0E\x24\x99\x1A\xAA\xF3\xC6\x9F"
			  "\xA0\x73\xB3\x70\xC3\x68\x64\x70"
			  "\xAD\x33\x02\xFB\x88\x74\xAA\x78"
			  "\xC7\x47\x1A\x18\x61\x2D\xAC\x9F"
			  "\x7E\x6F\xDF\x05\x13\x76\xA6\x72"
			  "\xB7\x13\x09\x0F\x7D\x38\xDF\x25"
			  "\x4E\xFD\x50\x45\xFA\x35\x6A\xC0"
			  "\x57\x95\xE1\x21\x26\x10\x9A\x21"
			  "\xA1\x8A\x51\x05\xD1\xB1\x78\x35"
			  "\x98\xF5\xAE\xC0\xC1\x8B\x94\xFF"
			  "\xD0\x69\x3F\x42\xC2\x01\xA7\x9B"
			  "\x23\x16\x47\x72\x81\x13\x3A\x72"
			  "\xEC\xD9\x40\x88\x00\x9C\xB0\xA8"
			  "\x9C\xAC\xCE\x11\x73\x7B\x63\x3E"
			  "\xA3\x63\x98\x7D\x35\xE4\xD9\x83"
			  "\xE2\xD0\x52\x87\x0C\x1F\xB0\xB3"
			  "\x41\x1A\x93\x8D\x76\x31\x9F\xF2"
			  "\xFE\x09\xA3\x8F\x22\x6A\x3B\xB9"
			  "\x6C\x9E\xE4\xA1\xA0\xC4\xE7\xA1"
			  "\x21\x9C\x1A\xCA\x65\xDE\x44\x03"
			  "\x99\xF2\xD2\x39\xE3\x3F\x0F\x37"
			  "\x53\x50\x23\xA4\x81\x6E\xDA\xFB"
			  "\xF8\x7B\x01\xD7\xB2\x32\x9C\xB8"
			  "\xB1\x0E\x99\x17\xB5\x38\xF9\xD7"
			  "\x86\x2D\x6E\x94\x5C\x99\x9D\xB3"
			  "\xD3\x63\x4B\x2A\x7D\x44\x6A\xB2"
			  "\xC1\x03\xE6\x5A\x37\xD8\x64\x18"
			  "\xAA\x32\xCE\x29\xED\xC0\xA2\xCB"
			  "\x8D\xAF\xCD\xBE\x8F\xB6\xEC\xB4"
			  "\x89\x05\x81\x6E\x71\x4F\xC3\x28"
			  "\x10\xC1\x62\xC4\x41\xE9\xD2\x39"
			  "\xF3\x22\x39\x12\x2C\xC2\x95\x2D"
			  "\xBF\x93\x58\x4B\x04\xD1\x8D\x57"
			  "\xAE\xEB\x60\x03\x56\x35\xAD\x5A"
			  "\xE9\xC3\xFF\x4E\x31\xE1\x37\xF8"
			  "\x7D\xEE\x65\x8A\xB6\x88\x1A\x3E"
			  "\x07\x09\x82\xBA\xF0\x80\x8A\xD0"
			  "\xA0\x3F\x6A\xE9\x24\x87\x19\x65"
			  "\x73\x3F\x12\x91\x47\x54\xBA\x39"
			  "\x30\x5B\x1E\xE5\xC2\xF9\x3F\xEF"
			  "\xD6\x75\xF9\xB8\x7C\x8B\x05\x76"
			  "\xEE\xB7\x08\x25\x4B\xB6\x7B\x47"
			  "\x72\xC0\x4C\xD4\xDA\xE0\x75\xF1"
			  "\x7C\xE8\x94\x9E\x16\x6E\xB8\x12"
			  "\xA1\xC1\x6E\x3B\x1C\x59\x41\x2D"
			  "\x23\xFA\x7D\x77\xB8\x46\x75\xFE"
			  "\x4F\x10\xD3\x09\x60\xA1\x36\x96"
			  "\x5B\xC2\xDC\x6E\x84\x7D\x9B\x14"
			  "\x80\x21\x83\x58\x3C\x76\xFD\x28"
			  "\x1D\xF9\x93\x13\xD7\x0E\x62\x14"
			  "\x5A\xC5\x4E\x08\xA5\x56\xA4\x3C"
			  "\x68\x93\x44\x70\xDF\xCF\x4A\x51"
			  "\x0B\x81\x29\x41\xE5\x62\x4D\x36"
			  "\xB3\xEA\x94\xA6\xB9\xDD\x3F\x09"
			  "\x62\x34\xA0\x6A\x7E\x7D\xF5\xF6"
			  "\x01\x91\xB4\x27\xDA\x59\xD6\x17"
			  "\x56\x4D\x82\x62\x37\xA3\x48\x01"
			  "\x99\x91\x77\xB2\x08\x6B\x2C\x37"
			  "\xC5\x5C\xAD\xB6\x07\xB6\x84\xF3"
			  "\x4D\x59\x7D\xC5\x28\x69\xFA\x92"
			  "\x22\x46\x89\x2D\x0F\x2B\x08\x24",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast6_cbc_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xDF\x77\x68\x96\xC7\xBA\xF8\xE2"
			  "\x0E\x24\x99\x1A\xAA\xF3\xC6\x9F"
			  "\xA0\x73\xB3\x70\xC3\x68\x64\x70"
			  "\xAD\x33\x02\xFB\x88\x74\xAA\x78"
			  "\xC7\x47\x1A\x18\x61\x2D\xAC\x9F"
			  "\x7E\x6F\xDF\x05\x13\x76\xA6\x72"
			  "\xB7\x13\x09\x0F\x7D\x38\xDF\x25"
			  "\x4E\xFD\x50\x45\xFA\x35\x6A\xC0"
			  "\x57\x95\xE1\x21\x26\x10\x9A\x21"
			  "\xA1\x8A\x51\x05\xD1\xB1\x78\x35"
			  "\x98\xF5\xAE\xC0\xC1\x8B\x94\xFF"
			  "\xD0\x69\x3F\x42\xC2\x01\xA7\x9B"
			  "\x23\x16\x47\x72\x81\x13\x3A\x72"
			  "\xEC\xD9\x40\x88\x00\x9C\xB0\xA8"
			  "\x9C\xAC\xCE\x11\x73\x7B\x63\x3E"
			  "\xA3\x63\x98\x7D\x35\xE4\xD9\x83"
			  "\xE2\xD0\x52\x87\x0C\x1F\xB0\xB3"
			  "\x41\x1A\x93\x8D\x76\x31\x9F\xF2"
			  "\xFE\x09\xA3\x8F\x22\x6A\x3B\xB9"
			  "\x6C\x9E\xE4\xA1\xA0\xC4\xE7\xA1"
			  "\x21\x9C\x1A\xCA\x65\xDE\x44\x03"
			  "\x99\xF2\xD2\x39\xE3\x3F\x0F\x37"
			  "\x53\x50\x23\xA4\x81\x6E\xDA\xFB"
			  "\xF8\x7B\x01\xD7\xB2\x32\x9C\xB8"
			  "\xB1\x0E\x99\x17\xB5\x38\xF9\xD7"
			  "\x86\x2D\x6E\x94\x5C\x99\x9D\xB3"
			  "\xD3\x63\x4B\x2A\x7D\x44\x6A\xB2"
			  "\xC1\x03\xE6\x5A\x37\xD8\x64\x18"
			  "\xAA\x32\xCE\x29\xED\xC0\xA2\xCB"
			  "\x8D\xAF\xCD\xBE\x8F\xB6\xEC\xB4"
			  "\x89\x05\x81\x6E\x71\x4F\xC3\x28"
			  "\x10\xC1\x62\xC4\x41\xE9\xD2\x39"
			  "\xF3\x22\x39\x12\x2C\xC2\x95\x2D"
			  "\xBF\x93\x58\x4B\x04\xD1\x8D\x57"
			  "\xAE\xEB\x60\x03\x56\x35\xAD\x5A"
			  "\xE9\xC3\xFF\x4E\x31\xE1\x37\xF8"
			  "\x7D\xEE\x65\x8A\xB6\x88\x1A\x3E"
			  "\x07\x09\x82\xBA\xF0\x80\x8A\xD0"
			  "\xA0\x3F\x6A\xE9\x24\x87\x19\x65"
			  "\x73\x3F\x12\x91\x47\x54\xBA\x39"
			  "\x30\x5B\x1E\xE5\xC2\xF9\x3F\xEF"
			  "\xD6\x75\xF9\xB8\x7C\x8B\x05\x76"
			  "\xEE\xB7\x08\x25\x4B\xB6\x7B\x47"
			  "\x72\xC0\x4C\xD4\xDA\xE0\x75\xF1"
			  "\x7C\xE8\x94\x9E\x16\x6E\xB8\x12"
			  "\xA1\xC1\x6E\x3B\x1C\x59\x41\x2D"
			  "\x23\xFA\x7D\x77\xB8\x46\x75\xFE"
			  "\x4F\x10\xD3\x09\x60\xA1\x36\x96"
			  "\x5B\xC2\xDC\x6E\x84\x7D\x9B\x14"
			  "\x80\x21\x83\x58\x3C\x76\xFD\x28"
			  "\x1D\xF9\x93\x13\xD7\x0E\x62\x14"
			  "\x5A\xC5\x4E\x08\xA5\x56\xA4\x3C"
			  "\x68\x93\x44\x70\xDF\xCF\x4A\x51"
			  "\x0B\x81\x29\x41\xE5\x62\x4D\x36"
			  "\xB3\xEA\x94\xA6\xB9\xDD\x3F\x09"
			  "\x62\x34\xA0\x6A\x7E\x7D\xF5\xF6"
			  "\x01\x91\xB4\x27\xDA\x59\xD6\x17"
			  "\x56\x4D\x82\x62\x37\xA3\x48\x01"
			  "\x99\x91\x77\xB2\x08\x6B\x2C\x37"
			  "\xC5\x5C\xAD\xB6\x07\xB6\x84\xF3"
			  "\x4D\x59\x7D\xC5\x28\x69\xFA\x92"
			  "\x22\x46\x89\x2D\x0F\x2B\x08\x24",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast6_ctr_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A",
		.ilen	= 17,
		.result	= "\x26\x0A\xF1\xE2\x3F\x8A\xEF\xA3"
			  "\x53\x9A\x5E\x1B\x2A\x1A\xC6\x0A"
			  "\x57",
		.rlen	= 17,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x26\x0A\xF1\xE2\x3F\x8A\xEF\xA3"
			  "\x53\x9A\x5E\x1B\x2A\x1A\xC6\x0A"
			  "\x57\xA3\xEF\x47\x2A\xE8\x88\xA7"
			  "\x3C\xD0\xEC\xB9\x94\x50\x7D\x56"
			  "\xBC\xE1\xC1\xF5\xE1\xEE\x12\xF8"
			  "\x4F\x03\x82\x3A\x93\x6B\x4C\xD3"
			  "\xE3\xF3\xFA\xC2\x23\x55\x98\x20"
			  "\x49\x76\x9B\x6B\xC1\x23\xBF\xE5"
			  "\xD4\xC4\x2F\x61\xE1\x67\x2A\x30"
			  "\x6F\x29\xCA\x54\xF8\x1B\xA6\x7D"
			  "\x66\x45\xEE\xC8\x19\xBE\x50\xF0"
			  "\x5F\x65\xF8\x1E\x4D\x07\x87\xD9"
			  "\xD3\xD9\x1B\x09\x89\xFD\x42\xC5"
			  "\xDB\xEB\x86\xF1\x67\x04\x0F\x5C"
			  "\x81\xDF\x82\x12\xC7\x4C\x1B\x07"
			  "\xDE\xE6\xFA\x29\x86\xD1\xB0\xBA"
			  "\x3D\x6A\x69\x76\xEC\x0F\xB4\xE6"
			  "\xCD\xA7\xF8\xA8\xB8\xE0\x33\xF5"
			  "\x49\x61\x22\x52\x64\x8C\x46\x41"
			  "\x1F\x48\x5F\x4F\xA2\x89\x36\x17"
			  "\x20\xF8\x2F\x8F\x4B\xFA\xF2\xC0"
			  "\x1E\x18\xA2\xF8\xB7\x6D\x98\xE3"
			  "\x00\x14\x15\x59\xC1\x30\x64\xAF"
			  "\xA8\x01\x38\xAB\xD4\x8B\xEC\x7C"
			  "\x44\x9A\xC6\x2C\x2E\x2B\x2B\xF4"
			  "\x02\x37\xC4\x69\xEF\x36\xC1\xF3"
			  "\xA0\xFB\xFE\x29\xAD\x39\xCF\xD0"
			  "\x51\x73\xA3\x22\x42\x41\xAB\xD2"
			  "\x0F\x50\x14\xB9\x54\xD3\xD4\xFA"
			  "\xBF\xC9\xBB\xCE\xC4\x1D\x2D\xAF"
			  "\xC9\x3F\x07\x87\x42\x4B\x3A\x54"
			  "\x34\x8E\x37\xA3\x03\x6F\x65\x66"
			  "\xDB\x44\xC3\xE8\xD7\xDD\x7D\xDD"
			  "\x61\xB4\x2B\x80\xA3\x98\x13\xF5"
			  "\x5A\xD3\x34\x58\xC3\x6E\xF6\xB8"
			  "\x0A\xC6\x50\x01\x8E\xD5\x6C\x7D"
			  "\xFE\x16\xB6\xCF\xFC\x51\x40\xAE"
			  "\xB3\x15\xAC\x90\x6F\x0B\x28\x3A"
			  "\x60\x40\x38\x90\x20\x46\xC7\xB3"
			  "\x0B\x12\x6D\x3B\x15\x14\xF9\xF4"
			  "\x11\x41\x76\x6B\xB3\x60\x82\x3C"
			  "\x84\xFB\x08\x2E\x92\x25\xCB\x79"
			  "\x6F\x58\xC5\x94\x00\x00\x47\xB6"
			  "\x9E\xDC\x0F\x29\x70\x46\x20\x76"
			  "\x65\x75\x66\x5C\x00\x96\xB3\xE1"
			  "\x0B\xA7\x11\x8B\x2E\x61\x4E\x45"
			  "\x73\xFC\x91\xAB\x79\x41\x23\x14"
			  "\x13\xB6\x72\x6C\x46\xB3\x03\x11"
			  "\xE4\xF1\xEE\xC9\x7A\xCF\x96\x32"
			  "\xB6\xF0\x8B\x97\xB4\xCF\x82\xB7"
			  "\x15\x48\x44\x99\x09\xF6\xE0\xD7"
			  "\xBC\xF1\x5B\x91\x4F\x30\x22\xA2"
			  "\x45\xC4\x68\x55\xC2\xBE\xA7\xD2"
			  "\x12\x53\x35\x9C\xF9\xE7\x35\x5D"
			  "\x81\xE4\x86\x42\xC3\x58\xFB\xF0"
			  "\x38\x9B\x8E\x5A\xEF\x83\x33\x0F"
			  "\x00\x4E\x3F\x9F\xF5\x84\x62\xC4"
			  "\x19\x35\x88\x22\x45\x59\x0E\x8F"
			  "\xEC\x27\xDD\x4A\xA4\x1F\xBC\x41"
			  "\x9B\x66\x8D\x32\xBA\x81\x34\x87"
			  "\x0E\x74\x33\x30\x62\xB9\x89\xDF"
			  "\xF9\xC5\xDD\x27\xB3\x39\xCB\xCB",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast6_ctr_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x26\x0A\xF1\xE2\x3F\x8A\xEF\xA3"
			  "\x53\x9A\x5E\x1B\x2A\x1A\xC6\x0A"
			  "\x57",
		.ilen	= 17,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A",
		.rlen	= 17,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x26\x0A\xF1\xE2\x3F\x8A\xEF\xA3"
			  "\x53\x9A\x5E\x1B\x2A\x1A\xC6\x0A"
			  "\x57\xA3\xEF\x47\x2A\xE8\x88\xA7"
			  "\x3C\xD0\xEC\xB9\x94\x50\x7D\x56"
			  "\xBC\xE1\xC1\xF5\xE1\xEE\x12\xF8"
			  "\x4F\x03\x82\x3A\x93\x6B\x4C\xD3"
			  "\xE3\xF3\xFA\xC2\x23\x55\x98\x20"
			  "\x49\x76\x9B\x6B\xC1\x23\xBF\xE5"
			  "\xD4\xC4\x2F\x61\xE1\x67\x2A\x30"
			  "\x6F\x29\xCA\x54\xF8\x1B\xA6\x7D"
			  "\x66\x45\xEE\xC8\x19\xBE\x50\xF0"
			  "\x5F\x65\xF8\x1E\x4D\x07\x87\xD9"
			  "\xD3\xD9\x1B\x09\x89\xFD\x42\xC5"
			  "\xDB\xEB\x86\xF1\x67\x04\x0F\x5C"
			  "\x81\xDF\x82\x12\xC7\x4C\x1B\x07"
			  "\xDE\xE6\xFA\x29\x86\xD1\xB0\xBA"
			  "\x3D\x6A\x69\x76\xEC\x0F\xB4\xE6"
			  "\xCD\xA7\xF8\xA8\xB8\xE0\x33\xF5"
			  "\x49\x61\x22\x52\x64\x8C\x46\x41"
			  "\x1F\x48\x5F\x4F\xA2\x89\x36\x17"
			  "\x20\xF8\x2F\x8F\x4B\xFA\xF2\xC0"
			  "\x1E\x18\xA2\xF8\xB7\x6D\x98\xE3"
			  "\x00\x14\x15\x59\xC1\x30\x64\xAF"
			  "\xA8\x01\x38\xAB\xD4\x8B\xEC\x7C"
			  "\x44\x9A\xC6\x2C\x2E\x2B\x2B\xF4"
			  "\x02\x37\xC4\x69\xEF\x36\xC1\xF3"
			  "\xA0\xFB\xFE\x29\xAD\x39\xCF\xD0"
			  "\x51\x73\xA3\x22\x42\x41\xAB\xD2"
			  "\x0F\x50\x14\xB9\x54\xD3\xD4\xFA"
			  "\xBF\xC9\xBB\xCE\xC4\x1D\x2D\xAF"
			  "\xC9\x3F\x07\x87\x42\x4B\x3A\x54"
			  "\x34\x8E\x37\xA3\x03\x6F\x65\x66"
			  "\xDB\x44\xC3\xE8\xD7\xDD\x7D\xDD"
			  "\x61\xB4\x2B\x80\xA3\x98\x13\xF5"
			  "\x5A\xD3\x34\x58\xC3\x6E\xF6\xB8"
			  "\x0A\xC6\x50\x01\x8E\xD5\x6C\x7D"
			  "\xFE\x16\xB6\xCF\xFC\x51\x40\xAE"
			  "\xB3\x15\xAC\x90\x6F\x0B\x28\x3A"
			  "\x60\x40\x38\x90\x20\x46\xC7\xB3"
			  "\x0B\x12\x6D\x3B\x15\x14\xF9\xF4"
			  "\x11\x41\x76\x6B\xB3\x60\x82\x3C"
			  "\x84\xFB\x08\x2E\x92\x25\xCB\x79"
			  "\x6F\x58\xC5\x94\x00\x00\x47\xB6"
			  "\x9E\xDC\x0F\x29\x70\x46\x20\x76"
			  "\x65\x75\x66\x5C\x00\x96\xB3\xE1"
			  "\x0B\xA7\x11\x8B\x2E\x61\x4E\x45"
			  "\x73\xFC\x91\xAB\x79\x41\x23\x14"
			  "\x13\xB6\x72\x6C\x46\xB3\x03\x11"
			  "\xE4\xF1\xEE\xC9\x7A\xCF\x96\x32"
			  "\xB6\xF0\x8B\x97\xB4\xCF\x82\xB7"
			  "\x15\x48\x44\x99\x09\xF6\xE0\xD7"
			  "\xBC\xF1\x5B\x91\x4F\x30\x22\xA2"
			  "\x45\xC4\x68\x55\xC2\xBE\xA7\xD2"
			  "\x12\x53\x35\x9C\xF9\xE7\x35\x5D"
			  "\x81\xE4\x86\x42\xC3\x58\xFB\xF0"
			  "\x38\x9B\x8E\x5A\xEF\x83\x33\x0F"
			  "\x00\x4E\x3F\x9F\xF5\x84\x62\xC4"
			  "\x19\x35\x88\x22\x45\x59\x0E\x8F"
			  "\xEC\x27\xDD\x4A\xA4\x1F\xBC\x41"
			  "\x9B\x66\x8D\x32\xBA\x81\x34\x87"
			  "\x0E\x74\x33\x30\x62\xB9\x89\xDF"
			  "\xF9\xC5\xDD\x27\xB3\x39\xCB\xCB",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast6_lrw_enc_tv_template[] = {
	{  
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.ilen	= 512,
		.result	= "\x55\x25\x09\x8B\xB5\xD5\xF8\xBF"
			  "\x37\x4A\xFE\x3C\x47\xD8\xE6\xEB"
			  "\xCA\xA4\x9B\xB0\xAB\x6D\x64\xCA"
			  "\x58\xB6\x73\xF0\xD7\x52\x34\xEF"
			  "\xFB\x3E\x96\x81\xB7\x71\x34\xA4"
			  "\x55\x20\xBE\x39\x5A\x2B\xF9\xD1"
			  "\x65\x0B\xDA\xD3\x7E\xB3\xA6\xF7"
			  "\x2E\x0B\x5A\x52\xDB\x39\x8C\x9B"
			  "\x61\x17\x5F\xAF\xB6\x5A\xC8\x08"
			  "\xA7\xB7\x2A\x11\x7C\x97\x38\x9D"
			  "\x59\x0E\x66\x59\x5E\xD8\x8B\xCE"
			  "\x70\xE0\xC3\x42\xB0\x8C\x0F\xBA"
			  "\xB2\x0D\x81\xB6\xBE\x61\x1C\x2D"
			  "\x7E\xEA\x91\x25\xAC\xEC\xF8\x28"
			  "\x80\x1D\xF0\x30\xBA\x62\x77\x7D"
			  "\xDB\x15\x69\xDF\xFA\x2A\x81\x64"
			  "\x95\x5B\xA4\x7F\x3E\x4F\xE3\x30"
			  "\xB0\x5C\xC2\x05\xF8\xF0\x29\xE7"
			  "\x0A\xA0\x66\xB2\x5D\x0F\x39\x2B"
			  "\xB4\xB3\x00\xA9\xD0\xAB\x63\x61"
			  "\x5E\xDB\xFC\x11\x74\x25\x96\x65"
			  "\xE8\xE2\x34\x57\x77\x15\x5E\x70"
			  "\xFF\x10\x90\xC3\x64\xF0\x11\x0A"
			  "\x63\x3A\xD3\x55\x92\x15\x4B\x0C"
			  "\xC7\x08\x89\x17\x3B\x99\xAD\x63"
			  "\xE7\x06\xDF\x52\xBC\x15\x64\x45"
			  "\x9D\x7A\xFB\x69\xBC\x2D\x6E\xA9"
			  "\x35\xD9\xD8\xF5\x0C\xC4\xA2\x23"
			  "\x9C\x18\x8B\xA8\x8C\xFE\xF8\x0E"
			  "\xBD\xAB\x60\x1A\x51\x17\x54\x27"
			  "\xB6\xE8\xBE\x0F\xA9\xA5\x82\x19"
			  "\x2F\x6F\x20\xA7\x47\xED\x74\x6C"
			  "\x4E\xC1\xF8\x8C\x14\xF3\xBB\x1F"
			  "\xED\x4D\x8F\x7C\x37\xEF\x19\xA1"
			  "\x07\x16\xDE\x76\xCC\x5E\x94\x02"
			  "\xFB\xBF\xE4\x81\x50\xCE\xFC\x0F"
			  "\x9E\xCF\x3D\xF6\x67\x00\xBF\xA7"
			  "\x6E\x21\x58\x36\x06\xDE\xB3\xD4"
			  "\xA2\xFA\xD8\x4E\xE0\xB9\x7F\x23"
			  "\x51\x21\x2B\x32\x68\xAA\xF8\xA8"
			  "\x93\x08\xB5\x6D\xE6\x43\x2C\xB7"
			  "\x31\xB2\x0F\xD0\xA2\x51\xC0\x25"
			  "\x30\xC7\x10\x3F\x97\x27\x01\x8E"
			  "\xFA\xD8\x4F\x78\xD8\x2E\x1D\xEB"
			  "\xA1\x37\x52\x0F\x7B\x5E\x87\xA8"
			  "\x22\xE2\xE6\x92\xA7\x5F\x11\x32"
			  "\xCC\x93\x34\xFC\xD1\x7E\xAE\x54"
			  "\xBC\x6A\x1B\x91\xD1\x2E\x21\xEC"
			  "\x5D\xF1\xC4\xF1\x55\x20\xBF\xE5"
			  "\x96\x3D\x69\x91\x20\x4E\xF2\x61"
			  "\xDA\x77\xFE\xEE\xC3\x74\x57\x2A"
			  "\x78\x39\xB0\xE0\xCF\x12\x56\xD6"
			  "\x05\xDC\xF9\x19\x66\x44\x1D\xF9"
			  "\x82\x37\xD4\xC2\x60\xB6\x31\xDF"
			  "\x0C\xAF\xBC\x8B\x55\x9A\xC8\x2D"
			  "\xAB\xA7\x88\x7B\x41\xE8\x29\xC9"
			  "\x9B\x8D\xA7\x00\x86\x25\xB6\x14"
			  "\xF5\x13\x73\xD7\x4B\x6B\x83\xF3"
			  "\xAF\x96\x00\xE4\xB7\x3C\x65\xA6"
			  "\x15\xB7\x94\x7D\x4E\x70\x4C\x75"
			  "\xF3\xB4\x02\xA9\x17\x1C\x7A\x0A"
			  "\xC0\xD5\x33\x11\x56\xDE\xDC\xF5"
			  "\x8D\xD9\xCD\x3B\x22\x67\x18\xC7"
			  "\xC4\xF5\x99\x61\xBC\xBB\x5B\x46",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec cast6_lrw_dec_tv_template[] = {
	{  
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x55\x25\x09\x8B\xB5\xD5\xF8\xBF"
			  "\x37\x4A\xFE\x3C\x47\xD8\xE6\xEB"
			  "\xCA\xA4\x9B\xB0\xAB\x6D\x64\xCA"
			  "\x58\xB6\x73\xF0\xD7\x52\x34\xEF"
			  "\xFB\x3E\x96\x81\xB7\x71\x34\xA4"
			  "\x55\x20\xBE\x39\x5A\x2B\xF9\xD1"
			  "\x65\x0B\xDA\xD3\x7E\xB3\xA6\xF7"
			  "\x2E\x0B\x5A\x52\xDB\x39\x8C\x9B"
			  "\x61\x17\x5F\xAF\xB6\x5A\xC8\x08"
			  "\xA7\xB7\x2A\x11\x7C\x97\x38\x9D"
			  "\x59\x0E\x66\x59\x5E\xD8\x8B\xCE"
			  "\x70\xE0\xC3\x42\xB0\x8C\x0F\xBA"
			  "\xB2\x0D\x81\xB6\xBE\x61\x1C\x2D"
			  "\x7E\xEA\x91\x25\xAC\xEC\xF8\x28"
			  "\x80\x1D\xF0\x30\xBA\x62\x77\x7D"
			  "\xDB\x15\x69\xDF\xFA\x2A\x81\x64"
			  "\x95\x5B\xA4\x7F\x3E\x4F\xE3\x30"
			  "\xB0\x5C\xC2\x05\xF8\xF0\x29\xE7"
			  "\x0A\xA0\x66\xB2\x5D\x0F\x39\x2B"
			  "\xB4\xB3\x00\xA9\xD0\xAB\x63\x61"
			  "\x5E\xDB\xFC\x11\x74\x25\x96\x65"
			  "\xE8\xE2\x34\x57\x77\x15\x5E\x70"
			  "\xFF\x10\x90\xC3\x64\xF0\x11\x0A"
			  "\x63\x3A\xD3\x55\x92\x15\x4B\x0C"
			  "\xC7\x08\x89\x17\x3B\x99\xAD\x63"
			  "\xE7\x06\xDF\x52\xBC\x15\x64\x45"
			  "\x9D\x7A\xFB\x69\xBC\x2D\x6E\xA9"
			  "\x35\xD9\xD8\xF5\x0C\xC4\xA2\x23"
			  "\x9C\x18\x8B\xA8\x8C\xFE\xF8\x0E"
			  "\xBD\xAB\x60\x1A\x51\x17\x54\x27"
			  "\xB6\xE8\xBE\x0F\xA9\xA5\x82\x19"
			  "\x2F\x6F\x20\xA7\x47\xED\x74\x6C"
			  "\x4E\xC1\xF8\x8C\x14\xF3\xBB\x1F"
			  "\xED\x4D\x8F\x7C\x37\xEF\x19\xA1"
			  "\x07\x16\xDE\x76\xCC\x5E\x94\x02"
			  "\xFB\xBF\xE4\x81\x50\xCE\xFC\x0F"
			  "\x9E\xCF\x3D\xF6\x67\x00\xBF\xA7"
			  "\x6E\x21\x58\x36\x06\xDE\xB3\xD4"
			  "\xA2\xFA\xD8\x4E\xE0\xB9\x7F\x23"
			  "\x51\x21\x2B\x32\x68\xAA\xF8\xA8"
			  "\x93\x08\xB5\x6D\xE6\x43\x2C\xB7"
			  "\x31\xB2\x0F\xD0\xA2\x51\xC0\x25"
			  "\x30\xC7\x10\x3F\x97\x27\x01\x8E"
			  "\xFA\xD8\x4F\x78\xD8\x2E\x1D\xEB"
			  "\xA1\x37\x52\x0F\x7B\x5E\x87\xA8"
			  "\x22\xE2\xE6\x92\xA7\x5F\x11\x32"
			  "\xCC\x93\x34\xFC\xD1\x7E\xAE\x54"
			  "\xBC\x6A\x1B\x91\xD1\x2E\x21\xEC"
			  "\x5D\xF1\xC4\xF1\x55\x20\xBF\xE5"
			  "\x96\x3D\x69\x91\x20\x4E\xF2\x61"
			  "\xDA\x77\xFE\xEE\xC3\x74\x57\x2A"
			  "\x78\x39\xB0\xE0\xCF\x12\x56\xD6"
			  "\x05\xDC\xF9\x19\x66\x44\x1D\xF9"
			  "\x82\x37\xD4\xC2\x60\xB6\x31\xDF"
			  "\x0C\xAF\xBC\x8B\x55\x9A\xC8\x2D"
			  "\xAB\xA7\x88\x7B\x41\xE8\x29\xC9"
			  "\x9B\x8D\xA7\x00\x86\x25\xB6\x14"
			  "\xF5\x13\x73\xD7\x4B\x6B\x83\xF3"
			  "\xAF\x96\x00\xE4\xB7\x3C\x65\xA6"
			  "\x15\xB7\x94\x7D\x4E\x70\x4C\x75"
			  "\xF3\xB4\x02\xA9\x17\x1C\x7A\x0A"
			  "\xC0\xD5\x33\x11\x56\xDE\xDC\xF5"
			  "\x8D\xD9\xCD\x3B\x22\x67\x18\xC7"
			  "\xC4\xF5\x99\x61\xBC\xBB\x5B\x46",
		.ilen	= 512,
		.result	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec cast6_xts_enc_tv_template[] = {
	{  
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\xDE\x6F\x22\xA5\xE8\x39\xE8\x78"
			  "\x88\x5A\x4F\x8D\x82\x76\x52\x6D"
			  "\xB2\x41\x16\xF4\x2B\xA6\xEB\xF6"
			  "\xE2\xC5\x62\x8D\x61\xA1\x01\xED"
			  "\xD9\x38\x01\xC1\x43\x63\x4E\x88"
			  "\xC9\x4B\x5A\x88\x80\xB7\x5C\x71"
			  "\x47\xEE\x11\xD8\xB7\x2D\x5D\x13"
			  "\x1A\xB1\x68\x5B\x61\xA7\xA9\x81"
			  "\x8B\x83\xA1\x6A\xAA\x36\xD6\xB6"
			  "\x60\x54\x09\x32\xFE\x6A\x76\x2E"
			  "\x28\xFF\xD5\xD6\xDD\x1D\x45\x7D"
			  "\xF0\x8B\xF3\x32\x4E\x6C\x12\xCB"
			  "\xB8\x25\x70\xF8\x40\xBC\x90\x1B"
			  "\x11\xC3\x59\xAF\xF0\x2F\x92\xDD"
			  "\xD3\x3B\xCF\x60\xA1\x78\x94\x57"
			  "\xAF\x76\xC1\x67\xA6\x3C\xCD\x98"
			  "\xB1\xF7\x27\xB9\xA3\xBD\x10\xEA"
			  "\xCD\x8B\xC2\xF2\x14\xF2\xB2\x67"
			  "\x05\xDD\x1D\x58\x6E\x2F\x95\x08"
			  "\x3A\xF8\x78\x76\x82\x56\xA7\xEC"
			  "\x51\x4B\x85\x77\xC2\x4C\x4A\x34"
			  "\x71\x38\x17\x91\x44\xE8\xFC\x65"
			  "\x99\x0D\x52\x91\xEE\xF8\xEF\x27"
			  "\x2A\x9E\x6E\x78\xC4\x26\x87\xF4"
			  "\x8A\xF0\x2D\x04\xE8\x14\x92\x5D"
			  "\x59\x22\x9B\x29\x5C\x18\xF0\xC3"
			  "\x47\xF3\x76\xD8\xE4\xF3\x1B\xD1"
			  "\x70\xA3\x0D\xB5\x70\x02\x1D\xA3"
			  "\x91\x3B\x49\x73\x18\xAB\xD4\xC9"
			  "\xC3\x1E\xEF\x1F\xFE\xD5\x59\x8A"
			  "\xD7\xF6\xC9\x71\x67\x79\xD7\x0E"
			  "\xBE\x1F\x8E\xEC\x55\x7E\x4F\x24"
			  "\xE6\x87\xEA\xFE\x96\x25\x67\x8E"
			  "\x93\x03\xFA\xFF\xCE\xAF\xB2\x3C"
			  "\x6F\xEB\x57\xFB\xD3\x28\x87\xA9"
			  "\xCE\xC2\xF5\x9C\xC6\x67\xB5\x97"
			  "\x49\xF7\x04\xCB\xEF\x84\x98\x33"
			  "\xAF\x38\xD3\x04\x1C\x24\x71\x38"
			  "\xC7\x71\xDD\x43\x0D\x12\x4A\x18"
			  "\xBA\xC4\xAF\xBA\xB2\x5B\xEB\x95"
			  "\x02\x43\x5D\xCE\x19\xCC\xCD\x66"
			  "\x91\x0B\x8C\x7F\x51\xC4\xBF\x3C"
			  "\x8B\xF1\xCC\xAA\x29\xD7\x87\xCB"
			  "\x3E\xC5\xF3\xC9\x75\xE8\xA3\x5B"
			  "\x30\x45\xA9\xB7\xAF\x80\x64\x6F"
			  "\x75\x4A\xA7\xC0\x6D\x19\x6B\xDE"
			  "\x17\xDE\x6D\xEA\x87\x9F\x95\xAE"
			  "\xF5\x3C\xEE\x54\xB8\x27\x84\xF8"
			  "\x97\xA3\xE1\x6F\x38\x24\x34\x88"
			  "\xCE\xBD\x32\x52\xE0\x00\x6C\x94"
			  "\xC9\xD7\x5D\x37\x81\x33\x2E\x7F"
			  "\x4F\x7E\x2E\x0D\x94\xBD\xEA\x59"
			  "\x34\x39\xA8\x35\x12\xB7\xBC\xAC"
			  "\xEA\x52\x9C\x78\x02\x6D\x92\x36"
			  "\xFB\x59\x2B\xA4\xEA\x7B\x1B\x83"
			  "\xE1\x4D\x5E\x2A\x7E\x92\xB1\x64"
			  "\xDE\xE0\x27\x4B\x0A\x6F\x4C\xE3"
			  "\xB0\xEB\x31\xE4\x69\x95\xAB\x35"
			  "\x8B\x2C\xF5\x6B\x7F\xF1\xA2\x82"
			  "\xF8\xD9\x47\x82\xA9\x82\x03\x91"
			  "\x69\x1F\xBE\x4C\xE7\xC7\x34\x2F"
			  "\x45\x72\x80\x17\x81\xBD\x9D\x62"
			  "\xA1\xAC\xE8\xCF\xC6\x74\xCF\xDC"
			  "\x22\x60\x4E\xE8\xA4\x5D\x85\xB9",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec cast6_xts_dec_tv_template[] = {
	{  
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xDE\x6F\x22\xA5\xE8\x39\xE8\x78"
			  "\x88\x5A\x4F\x8D\x82\x76\x52\x6D"
			  "\xB2\x41\x16\xF4\x2B\xA6\xEB\xF6"
			  "\xE2\xC5\x62\x8D\x61\xA1\x01\xED"
			  "\xD9\x38\x01\xC1\x43\x63\x4E\x88"
			  "\xC9\x4B\x5A\x88\x80\xB7\x5C\x71"
			  "\x47\xEE\x11\xD8\xB7\x2D\x5D\x13"
			  "\x1A\xB1\x68\x5B\x61\xA7\xA9\x81"
			  "\x8B\x83\xA1\x6A\xAA\x36\xD6\xB6"
			  "\x60\x54\x09\x32\xFE\x6A\x76\x2E"
			  "\x28\xFF\xD5\xD6\xDD\x1D\x45\x7D"
			  "\xF0\x8B\xF3\x32\x4E\x6C\x12\xCB"
			  "\xB8\x25\x70\xF8\x40\xBC\x90\x1B"
			  "\x11\xC3\x59\xAF\xF0\x2F\x92\xDD"
			  "\xD3\x3B\xCF\x60\xA1\x78\x94\x57"
			  "\xAF\x76\xC1\x67\xA6\x3C\xCD\x98"
			  "\xB1\xF7\x27\xB9\xA3\xBD\x10\xEA"
			  "\xCD\x8B\xC2\xF2\x14\xF2\xB2\x67"
			  "\x05\xDD\x1D\x58\x6E\x2F\x95\x08"
			  "\x3A\xF8\x78\x76\x82\x56\xA7\xEC"
			  "\x51\x4B\x85\x77\xC2\x4C\x4A\x34"
			  "\x71\x38\x17\x91\x44\xE8\xFC\x65"
			  "\x99\x0D\x52\x91\xEE\xF8\xEF\x27"
			  "\x2A\x9E\x6E\x78\xC4\x26\x87\xF4"
			  "\x8A\xF0\x2D\x04\xE8\x14\x92\x5D"
			  "\x59\x22\x9B\x29\x5C\x18\xF0\xC3"
			  "\x47\xF3\x76\xD8\xE4\xF3\x1B\xD1"
			  "\x70\xA3\x0D\xB5\x70\x02\x1D\xA3"
			  "\x91\x3B\x49\x73\x18\xAB\xD4\xC9"
			  "\xC3\x1E\xEF\x1F\xFE\xD5\x59\x8A"
			  "\xD7\xF6\xC9\x71\x67\x79\xD7\x0E"
			  "\xBE\x1F\x8E\xEC\x55\x7E\x4F\x24"
			  "\xE6\x87\xEA\xFE\x96\x25\x67\x8E"
			  "\x93\x03\xFA\xFF\xCE\xAF\xB2\x3C"
			  "\x6F\xEB\x57\xFB\xD3\x28\x87\xA9"
			  "\xCE\xC2\xF5\x9C\xC6\x67\xB5\x97"
			  "\x49\xF7\x04\xCB\xEF\x84\x98\x33"
			  "\xAF\x38\xD3\x04\x1C\x24\x71\x38"
			  "\xC7\x71\xDD\x43\x0D\x12\x4A\x18"
			  "\xBA\xC4\xAF\xBA\xB2\x5B\xEB\x95"
			  "\x02\x43\x5D\xCE\x19\xCC\xCD\x66"
			  "\x91\x0B\x8C\x7F\x51\xC4\xBF\x3C"
			  "\x8B\xF1\xCC\xAA\x29\xD7\x87\xCB"
			  "\x3E\xC5\xF3\xC9\x75\xE8\xA3\x5B"
			  "\x30\x45\xA9\xB7\xAF\x80\x64\x6F"
			  "\x75\x4A\xA7\xC0\x6D\x19\x6B\xDE"
			  "\x17\xDE\x6D\xEA\x87\x9F\x95\xAE"
			  "\xF5\x3C\xEE\x54\xB8\x27\x84\xF8"
			  "\x97\xA3\xE1\x6F\x38\x24\x34\x88"
			  "\xCE\xBD\x32\x52\xE0\x00\x6C\x94"
			  "\xC9\xD7\x5D\x37\x81\x33\x2E\x7F"
			  "\x4F\x7E\x2E\x0D\x94\xBD\xEA\x59"
			  "\x34\x39\xA8\x35\x12\xB7\xBC\xAC"
			  "\xEA\x52\x9C\x78\x02\x6D\x92\x36"
			  "\xFB\x59\x2B\xA4\xEA\x7B\x1B\x83"
			  "\xE1\x4D\x5E\x2A\x7E\x92\xB1\x64"
			  "\xDE\xE0\x27\x4B\x0A\x6F\x4C\xE3"
			  "\xB0\xEB\x31\xE4\x69\x95\xAB\x35"
			  "\x8B\x2C\xF5\x6B\x7F\xF1\xA2\x82"
			  "\xF8\xD9\x47\x82\xA9\x82\x03\x91"
			  "\x69\x1F\xBE\x4C\xE7\xC7\x34\x2F"
			  "\x45\x72\x80\x17\x81\xBD\x9D\x62"
			  "\xA1\xAC\xE8\xCF\xC6\x74\xCF\xDC"
			  "\x22\x60\x4E\xE8\xA4\x5D\x85\xB9",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

#define AES_ENC_TEST_VECTORS 4
#define AES_DEC_TEST_VECTORS 4
#define AES_CBC_ENC_TEST_VECTORS 5
#define AES_CBC_DEC_TEST_VECTORS 5
#define HMAC_SHA1_AES_CBC_ENC_TEST_VECTORS 7
#define HMAC_SHA256_AES_CBC_ENC_TEST_VECTORS 7
#define HMAC_SHA512_AES_CBC_ENC_TEST_VECTORS 7
#define AES_LRW_ENC_TEST_VECTORS 8
#define AES_LRW_DEC_TEST_VECTORS 8
#define AES_XTS_ENC_TEST_VECTORS 5
#define AES_XTS_DEC_TEST_VECTORS 5
#define AES_CTR_ENC_TEST_VECTORS 5
#define AES_CTR_DEC_TEST_VECTORS 5
#define AES_OFB_ENC_TEST_VECTORS 1
#define AES_OFB_DEC_TEST_VECTORS 1
#define AES_CTR_3686_ENC_TEST_VECTORS 7
#define AES_CTR_3686_DEC_TEST_VECTORS 6
#define AES_GCM_ENC_TEST_VECTORS 9
#define AES_GCM_DEC_TEST_VECTORS 8
#define AES_GCM_4106_ENC_TEST_VECTORS 7
#define AES_GCM_4106_DEC_TEST_VECTORS 7
#define AES_GCM_4543_ENC_TEST_VECTORS 1
#define AES_GCM_4543_DEC_TEST_VECTORS 2
#define AES_CCM_ENC_TEST_VECTORS 7
#define AES_CCM_DEC_TEST_VECTORS 7
#define AES_CCM_4309_ENC_TEST_VECTORS 7
#define AES_CCM_4309_DEC_TEST_VECTORS 10

static struct cipher_testvec aes_enc_tv_template[] = {
	{  
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\x69\xc4\xe0\xd8\x6a\x7b\x04\x30"
			  "\xd8\xcd\xb7\x80\x70\xb4\xc5\x5a",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17",
		.klen	= 24,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\xdd\xa9\x7c\xa4\x86\x4c\xdf\xe0"
			  "\x6e\xaf\x70\xa0\xec\x0d\x71\x91",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\x8e\xa2\xb7\xca\x51\x67\x45\xbf"
			  "\xea\xfc\x49\x90\x4b\x49\x60\x89",
		.rlen	= 16,
	}, {  
		.key	= "\xA6\xC9\x83\xA6\xC9\xEC\x0F\x32"
			  "\x55\x0F\x32\x55\x78\x9B\xBE\x78"
			  "\x9B\xBE\xE1\x04\x27\xE1\x04\x27"
			  "\x4A\x6D\x90\x4A\x6D\x90\xB3\xD6",
		.klen	= 32,
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\x71\x73\xF7\xDB\x24\x93\x21\x6D"
			  "\x61\x1E\xBB\x63\x42\x79\xDB\x64"
			  "\x6F\x82\xC0\xCA\xA3\x9B\xFA\x0B"
			  "\xD9\x08\xC7\x4A\x90\xAE\x8F\x5F"
			  "\x5E\x06\xF0\x5F\x31\x51\x18\x37"
			  "\x45\xD7\xCA\x3A\xFD\x6C\x3F\xE1"
			  "\xDD\x8D\x22\x65\x2B\x00\x50\xCE"
			  "\xBA\x28\x67\xD7\xCE\x0E\x0D\xEA"
			  "\x78\x69\x7F\xAE\x8F\x8B\x69\x37"
			  "\x75\xE0\xDC\x96\xE0\xB7\xF4\x09"
			  "\xCB\x6D\xA2\xFB\xDA\xAF\x09\xF8"
			  "\x81\x82\x27\xFA\x45\x9C\x29\xA4"
			  "\x22\x8B\x78\x69\x5B\x46\xF9\x39"
			  "\x1B\xCC\xF9\x1D\x09\xEB\xBC\x5C"
			  "\x41\x72\x51\x97\x1D\x07\x49\xA0"
			  "\x1B\x8E\x65\x4B\xB2\x6A\x12\x03"
			  "\x6A\x60\x95\xAC\xBD\xAC\x1A\x64"
			  "\xDE\x5A\xA5\xF0\x83\x2F\xCB\xCA"
			  "\x22\x74\xA6\x6C\x9B\x73\xCE\x3F"
			  "\xE1\x8B\x22\x17\x59\x0C\x47\x89"
			  "\x33\xA1\xD6\x47\x03\x19\x4F\xA8"
			  "\x67\x69\xF0\x5B\xF0\x20\xAD\x06"
			  "\x27\x81\x92\xD8\xC5\xBA\x98\x12"
			  "\xBE\x24\xB5\x2F\x75\x02\xC2\xAD"
			  "\x12\x2F\x07\x32\xEE\x39\xAF\x64"
			  "\x05\x8F\xB3\xD4\xEB\x1B\x46\x6E"
			  "\xD9\x21\xF9\xC4\xB7\xC9\x45\x68"
			  "\xB4\xA1\x74\x9F\x82\x47\xEB\xCC"
			  "\xBD\x0A\x14\x95\x0F\x8B\xA8\x2F"
			  "\x4B\x1B\xA7\xBF\x82\xA6\x43\x0C"
			  "\xB9\x39\x4A\xA8\x10\x6F\x50\x7B"
			  "\x25\xFB\x26\x81\xE0\x2F\xF0\x96"
			  "\x8D\x8B\xAC\x92\x0F\xF6\xED\x64"
			  "\x63\x29\x4C\x8E\x18\x13\xC5\xBF"
			  "\xFC\xA0\xD9\xBF\x7C\x3A\x0E\x29"
			  "\x6F\xD1\x6C\x6F\xA5\xDA\xBF\xB1"
			  "\x30\xEA\x44\x2D\xC3\x8F\x16\xE1"
			  "\x66\xFA\xA3\x21\x3E\xFC\x13\xCA"
			  "\xF0\xF6\xF0\x59\xBD\x8F\x38\x50"
			  "\x31\xCB\x69\x3F\x96\x15\xD6\xF5"
			  "\xAE\xFF\xF6\xAA\x41\x85\x4C\x10"
			  "\x58\xE3\xF9\x44\xE6\x28\xDA\x9A"
			  "\xDC\x6A\x80\x34\x73\x97\x1B\xC5"
			  "\xCA\x26\x16\x77\x0E\x60\xAB\x89"
			  "\x0F\x04\x27\xBD\xCE\x3E\x71\xB4"
			  "\xA0\xD7\x22\x7E\xDB\xEB\x24\x70"
			  "\x42\x71\x51\x78\x70\xB3\xE0\x3D"
			  "\x84\x8E\x8D\x7B\xD0\x6D\xEA\x92"
			  "\x11\x08\x42\x4F\xE5\xAD\x26\x92"
			  "\xD2\x00\xAE\xA8\xE3\x4B\x37\x47"
			  "\x22\xC1\x95\xC1\x63\x7F\xCB\x03"
			  "\xF3\xE3\xD7\x9D\x60\xC7\xBC\xEA"
			  "\x35\xA2\xFD\x45\x52\x39\x13\x6F"
			  "\xC1\x53\xF3\x53\xDF\x33\x84\xD7"
			  "\xD2\xC8\x37\xB0\x75\xE3\x41\x46"
			  "\xB3\xC7\x83\x2E\x8A\xBB\xA4\xE5"
			  "\x7F\x3C\xFD\x8B\xEB\xEA\x63\xBD"
			  "\xB7\x46\xE7\xBF\x09\x9C\x0D\x0F"
			  "\x40\x86\x7F\x51\xE1\x11\x9C\xCB"
			  "\x88\xE6\x68\x47\xE3\x2B\xC5\xFF"
			  "\x09\x79\xA0\x43\x5C\x0D\x08\x58"
			  "\x17\xBB\xC0\x6B\x62\x3F\x56\xE9",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec aes_dec_tv_template[] = {
	{  
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x69\xc4\xe0\xd8\x6a\x7b\x04\x30"
			  "\xd8\xcd\xb7\x80\x70\xb4\xc5\x5a",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17",
		.klen	= 24,
		.input	= "\xdd\xa9\x7c\xa4\x86\x4c\xdf\xe0"
			  "\x6e\xaf\x70\xa0\xec\x0d\x71\x91",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x8e\xa2\xb7\xca\x51\x67\x45\xbf"
			  "\xea\xfc\x49\x90\x4b\x49\x60\x89",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {  
		.key	= "\xA6\xC9\x83\xA6\xC9\xEC\x0F\x32"
			  "\x55\x0F\x32\x55\x78\x9B\xBE\x78"
			  "\x9B\xBE\xE1\x04\x27\xE1\x04\x27"
			  "\x4A\x6D\x90\x4A\x6D\x90\xB3\xD6",
		.klen	= 32,
		.input	= "\x71\x73\xF7\xDB\x24\x93\x21\x6D"
			  "\x61\x1E\xBB\x63\x42\x79\xDB\x64"
			  "\x6F\x82\xC0\xCA\xA3\x9B\xFA\x0B"
			  "\xD9\x08\xC7\x4A\x90\xAE\x8F\x5F"
			  "\x5E\x06\xF0\x5F\x31\x51\x18\x37"
			  "\x45\xD7\xCA\x3A\xFD\x6C\x3F\xE1"
			  "\xDD\x8D\x22\x65\x2B\x00\x50\xCE"
			  "\xBA\x28\x67\xD7\xCE\x0E\x0D\xEA"
			  "\x78\x69\x7F\xAE\x8F\x8B\x69\x37"
			  "\x75\xE0\xDC\x96\xE0\xB7\xF4\x09"
			  "\xCB\x6D\xA2\xFB\xDA\xAF\x09\xF8"
			  "\x81\x82\x27\xFA\x45\x9C\x29\xA4"
			  "\x22\x8B\x78\x69\x5B\x46\xF9\x39"
			  "\x1B\xCC\xF9\x1D\x09\xEB\xBC\x5C"
			  "\x41\x72\x51\x97\x1D\x07\x49\xA0"
			  "\x1B\x8E\x65\x4B\xB2\x6A\x12\x03"
			  "\x6A\x60\x95\xAC\xBD\xAC\x1A\x64"
			  "\xDE\x5A\xA5\xF0\x83\x2F\xCB\xCA"
			  "\x22\x74\xA6\x6C\x9B\x73\xCE\x3F"
			  "\xE1\x8B\x22\x17\x59\x0C\x47\x89"
			  "\x33\xA1\xD6\x47\x03\x19\x4F\xA8"
			  "\x67\x69\xF0\x5B\xF0\x20\xAD\x06"
			  "\x27\x81\x92\xD8\xC5\xBA\x98\x12"
			  "\xBE\x24\xB5\x2F\x75\x02\xC2\xAD"
			  "\x12\x2F\x07\x32\xEE\x39\xAF\x64"
			  "\x05\x8F\xB3\xD4\xEB\x1B\x46\x6E"
			  "\xD9\x21\xF9\xC4\xB7\xC9\x45\x68"
			  "\xB4\xA1\x74\x9F\x82\x47\xEB\xCC"
			  "\xBD\x0A\x14\x95\x0F\x8B\xA8\x2F"
			  "\x4B\x1B\xA7\xBF\x82\xA6\x43\x0C"
			  "\xB9\x39\x4A\xA8\x10\x6F\x50\x7B"
			  "\x25\xFB\x26\x81\xE0\x2F\xF0\x96"
			  "\x8D\x8B\xAC\x92\x0F\xF6\xED\x64"
			  "\x63\x29\x4C\x8E\x18\x13\xC5\xBF"
			  "\xFC\xA0\xD9\xBF\x7C\x3A\x0E\x29"
			  "\x6F\xD1\x6C\x6F\xA5\xDA\xBF\xB1"
			  "\x30\xEA\x44\x2D\xC3\x8F\x16\xE1"
			  "\x66\xFA\xA3\x21\x3E\xFC\x13\xCA"
			  "\xF0\xF6\xF0\x59\xBD\x8F\x38\x50"
			  "\x31\xCB\x69\x3F\x96\x15\xD6\xF5"
			  "\xAE\xFF\xF6\xAA\x41\x85\x4C\x10"
			  "\x58\xE3\xF9\x44\xE6\x28\xDA\x9A"
			  "\xDC\x6A\x80\x34\x73\x97\x1B\xC5"
			  "\xCA\x26\x16\x77\x0E\x60\xAB\x89"
			  "\x0F\x04\x27\xBD\xCE\x3E\x71\xB4"
			  "\xA0\xD7\x22\x7E\xDB\xEB\x24\x70"
			  "\x42\x71\x51\x78\x70\xB3\xE0\x3D"
			  "\x84\x8E\x8D\x7B\xD0\x6D\xEA\x92"
			  "\x11\x08\x42\x4F\xE5\xAD\x26\x92"
			  "\xD2\x00\xAE\xA8\xE3\x4B\x37\x47"
			  "\x22\xC1\x95\xC1\x63\x7F\xCB\x03"
			  "\xF3\xE3\xD7\x9D\x60\xC7\xBC\xEA"
			  "\x35\xA2\xFD\x45\x52\x39\x13\x6F"
			  "\xC1\x53\xF3\x53\xDF\x33\x84\xD7"
			  "\xD2\xC8\x37\xB0\x75\xE3\x41\x46"
			  "\xB3\xC7\x83\x2E\x8A\xBB\xA4\xE5"
			  "\x7F\x3C\xFD\x8B\xEB\xEA\x63\xBD"
			  "\xB7\x46\xE7\xBF\x09\x9C\x0D\x0F"
			  "\x40\x86\x7F\x51\xE1\x11\x9C\xCB"
			  "\x88\xE6\x68\x47\xE3\x2B\xC5\xFF"
			  "\x09\x79\xA0\x43\x5C\x0D\x08\x58"
			  "\x17\xBB\xC0\x6B\x62\x3F\x56\xE9",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec aes_cbc_enc_tv_template[] = {
	{  
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input	= "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
		.rlen   = 16,
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1",
		.rlen   = 32,
	}, {  
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b",
		.rlen	= 64,
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\xEA\x65\x8A\x19\xB0\x66\xC1\x3F"
			  "\xCE\xF1\x97\x75\xC1\xFD\xB5\xAF"
			  "\x52\x65\xF7\xFF\xBC\xD8\x2D\x9F"
			  "\x2F\xB9\x26\x9B\x6F\x10\xB7\xB8"
			  "\x26\xA1\x02\x46\xA2\xAD\xC6\xC0"
			  "\x11\x15\xFF\x6D\x1E\x82\x04\xA6"
			  "\xB1\x74\xD1\x08\x13\xFD\x90\x7C"
			  "\xF5\xED\xD3\xDB\x5A\x0A\x0C\x2F"
			  "\x0A\x70\xF1\x88\x07\xCF\x21\x26"
			  "\x40\x40\x8A\xF5\x53\xF7\x24\x4F"
			  "\x83\x38\x43\x5F\x08\x99\xEB\xE3"
			  "\xDC\x02\x64\x67\x50\x6E\x15\xC3"
			  "\x01\x1A\xA0\x81\x13\x65\xA6\x73"
			  "\x71\xA6\x3B\x91\x83\x77\xBE\xFA"
			  "\xDB\x71\x73\xA6\xC1\xAE\x43\xC3"
			  "\x36\xCE\xD6\xEB\xF9\x30\x1C\x4F"
			  "\x80\x38\x5E\x9C\x6E\xAB\x98\x2F"
			  "\x53\xAF\xCF\xC8\x9A\xB8\x86\x43"
			  "\x3E\x86\xE7\xA1\xF4\x2F\x30\x40"
			  "\x03\xA8\x6C\x50\x42\x9F\x77\x59"
			  "\x89\xA0\xC5\xEC\x9A\xB8\xDD\x99"
			  "\x16\x24\x02\x07\x48\xAE\xF2\x31"
			  "\x34\x0E\xC3\x85\xFE\x1C\x95\x99"
			  "\x87\x58\x98\x8B\xE7\xC6\xC5\x70"
			  "\x73\x81\x07\x7C\x56\x2F\xD8\x1B"
			  "\xB7\xB9\x2B\xAB\xE3\x01\x87\x0F"
			  "\xD8\xBB\xC0\x0D\xAC\x2C\x2F\x98"
			  "\x3C\x0B\xA2\x99\x4A\x8C\xF7\x04"
			  "\xE0\xE0\xCF\xD1\x81\x5B\xFE\xF5"
			  "\x24\x04\xFD\xB8\xDF\x13\xD8\xCD"
			  "\xF1\xE3\x3D\x98\x50\x02\x77\x9E"
			  "\xBC\x22\xAB\xFA\xC2\x43\x1F\x66"
			  "\x20\x02\x23\xDA\xDF\xA0\x89\xF6"
			  "\xD8\xF3\x45\x24\x53\x6F\x16\x77"
			  "\x02\x3E\x7B\x36\x5F\xA0\x3B\x78"
			  "\x63\xA2\xBD\xB5\xA4\xCA\x1E\xD3"
			  "\x57\xBC\x0B\x9F\x43\x51\x28\x4F"
			  "\x07\x50\x6C\x68\x12\x07\xCF\xFA"
			  "\x6B\x72\x0B\xEB\xF8\x88\x90\x2C"
			  "\x7E\xF5\x91\xD1\x03\xD8\xD5\xBD"
			  "\x22\x39\x7B\x16\x03\x01\x69\xAF"
			  "\x3D\x38\x66\x28\x0C\xBE\x5B\xC5"
			  "\x03\xB4\x2F\x51\x8A\x56\x17\x2B"
			  "\x88\x42\x6D\x40\x68\x8F\xD0\x11"
			  "\x19\xF9\x1F\x43\x79\x95\x31\xFA"
			  "\x28\x7A\x3D\xF7\x66\xEB\xEF\xAC"
			  "\x06\xB2\x01\xAD\xDB\x68\xDB\xEC"
			  "\x8D\x53\x6E\x72\x68\xA3\xC7\x63"
			  "\x43\x2B\x78\xE0\x04\x29\x8F\x72"
			  "\xB2\x2C\xE6\x84\x03\x30\x6D\xCD"
			  "\x26\x92\x37\xE1\x2F\xBB\x8B\x9D"
			  "\xE4\x4C\xF6\x93\xBC\xD9\xAD\x44"
			  "\x52\x65\xC7\xB0\x0E\x3F\x0E\x61"
			  "\x56\x5D\x1C\x6D\xA7\x05\x2E\xBC"
			  "\x58\x08\x15\xAB\x12\xAB\x17\x4A"
			  "\x5E\x1C\xF2\xCD\xB8\xA2\xAE\xFB"
			  "\x9B\x2E\x0E\x85\x34\x80\x0E\x3F"
			  "\x4C\xB8\xDB\xCE\x1C\x90\xA1\x61"
			  "\x6C\x69\x09\x35\x9E\xD4\xF4\xAD"
			  "\xBC\x06\x41\xE3\x01\xB4\x4E\x0A"
			  "\xE0\x1F\x91\xF8\x82\x96\x2D\x65"
			  "\xA3\xAA\x13\xCC\x50\xFF\x7B\x02",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec aes_cbc_dec_tv_template[] = {
	{  
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
		.ilen   = 16,
		.result = "Single block msg",
		.rlen   = 16,
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1",
		.ilen   = 32,
		.result = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen   = 32,
	}, {  
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\xEA\x65\x8A\x19\xB0\x66\xC1\x3F"
			  "\xCE\xF1\x97\x75\xC1\xFD\xB5\xAF"
			  "\x52\x65\xF7\xFF\xBC\xD8\x2D\x9F"
			  "\x2F\xB9\x26\x9B\x6F\x10\xB7\xB8"
			  "\x26\xA1\x02\x46\xA2\xAD\xC6\xC0"
			  "\x11\x15\xFF\x6D\x1E\x82\x04\xA6"
			  "\xB1\x74\xD1\x08\x13\xFD\x90\x7C"
			  "\xF5\xED\xD3\xDB\x5A\x0A\x0C\x2F"
			  "\x0A\x70\xF1\x88\x07\xCF\x21\x26"
			  "\x40\x40\x8A\xF5\x53\xF7\x24\x4F"
			  "\x83\x38\x43\x5F\x08\x99\xEB\xE3"
			  "\xDC\x02\x64\x67\x50\x6E\x15\xC3"
			  "\x01\x1A\xA0\x81\x13\x65\xA6\x73"
			  "\x71\xA6\x3B\x91\x83\x77\xBE\xFA"
			  "\xDB\x71\x73\xA6\xC1\xAE\x43\xC3"
			  "\x36\xCE\xD6\xEB\xF9\x30\x1C\x4F"
			  "\x80\x38\x5E\x9C\x6E\xAB\x98\x2F"
			  "\x53\xAF\xCF\xC8\x9A\xB8\x86\x43"
			  "\x3E\x86\xE7\xA1\xF4\x2F\x30\x40"
			  "\x03\xA8\x6C\x50\x42\x9F\x77\x59"
			  "\x89\xA0\xC5\xEC\x9A\xB8\xDD\x99"
			  "\x16\x24\x02\x07\x48\xAE\xF2\x31"
			  "\x34\x0E\xC3\x85\xFE\x1C\x95\x99"
			  "\x87\x58\x98\x8B\xE7\xC6\xC5\x70"
			  "\x73\x81\x07\x7C\x56\x2F\xD8\x1B"
			  "\xB7\xB9\x2B\xAB\xE3\x01\x87\x0F"
			  "\xD8\xBB\xC0\x0D\xAC\x2C\x2F\x98"
			  "\x3C\x0B\xA2\x99\x4A\x8C\xF7\x04"
			  "\xE0\xE0\xCF\xD1\x81\x5B\xFE\xF5"
			  "\x24\x04\xFD\xB8\xDF\x13\xD8\xCD"
			  "\xF1\xE3\x3D\x98\x50\x02\x77\x9E"
			  "\xBC\x22\xAB\xFA\xC2\x43\x1F\x66"
			  "\x20\x02\x23\xDA\xDF\xA0\x89\xF6"
			  "\xD8\xF3\x45\x24\x53\x6F\x16\x77"
			  "\x02\x3E\x7B\x36\x5F\xA0\x3B\x78"
			  "\x63\xA2\xBD\xB5\xA4\xCA\x1E\xD3"
			  "\x57\xBC\x0B\x9F\x43\x51\x28\x4F"
			  "\x07\x50\x6C\x68\x12\x07\xCF\xFA"
			  "\x6B\x72\x0B\xEB\xF8\x88\x90\x2C"
			  "\x7E\xF5\x91\xD1\x03\xD8\xD5\xBD"
			  "\x22\x39\x7B\x16\x03\x01\x69\xAF"
			  "\x3D\x38\x66\x28\x0C\xBE\x5B\xC5"
			  "\x03\xB4\x2F\x51\x8A\x56\x17\x2B"
			  "\x88\x42\x6D\x40\x68\x8F\xD0\x11"
			  "\x19\xF9\x1F\x43\x79\x95\x31\xFA"
			  "\x28\x7A\x3D\xF7\x66\xEB\xEF\xAC"
			  "\x06\xB2\x01\xAD\xDB\x68\xDB\xEC"
			  "\x8D\x53\x6E\x72\x68\xA3\xC7\x63"
			  "\x43\x2B\x78\xE0\x04\x29\x8F\x72"
			  "\xB2\x2C\xE6\x84\x03\x30\x6D\xCD"
			  "\x26\x92\x37\xE1\x2F\xBB\x8B\x9D"
			  "\xE4\x4C\xF6\x93\xBC\xD9\xAD\x44"
			  "\x52\x65\xC7\xB0\x0E\x3F\x0E\x61"
			  "\x56\x5D\x1C\x6D\xA7\x05\x2E\xBC"
			  "\x58\x08\x15\xAB\x12\xAB\x17\x4A"
			  "\x5E\x1C\xF2\xCD\xB8\xA2\xAE\xFB"
			  "\x9B\x2E\x0E\x85\x34\x80\x0E\x3F"
			  "\x4C\xB8\xDB\xCE\x1C\x90\xA1\x61"
			  "\x6C\x69\x09\x35\x9E\xD4\xF4\xAD"
			  "\xBC\x06\x41\xE3\x01\xB4\x4E\x0A"
			  "\xE0\x1F\x91\xF8\x82\x96\x2D\x65"
			  "\xA3\xAA\x13\xCC\x50\xFF\x7B\x02",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct aead_testvec hmac_sha1_aes_cbc_enc_tv_template[] = {
	{  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 20 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\x1b\x13\xcb\xaf\x89\x5e\xe1\x2c"
			  "\x13\xc5\x2e\xa3\xcc\xed\xdc\xb5"
			  "\x03\x71\xa2\x06",
		.rlen   = 16 + 20,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 20 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\xad\x9b\x4c\x5c\x85\xe1\xda\xae"
			  "\xee\x81\x4e\xd7\xdb\x74\xcf\x58"
			  "\x65\x39\xf8\xde",
		.rlen   = 32 + 20,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 20 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\xc2\xec\x0c\xf8\x7f\x05\xba\xca"
			  "\xff\xee\x4c\xd0\x93\xe6\x36\x7f"
			  "\x8d\x62\xf2\x1e",
		.rlen   = 48 + 20,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 20 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x1c\x45\x57\xa9\x56\xcb\xa9\x2d"
			  "\x18\xac\xf1\xc7\x5d\xd1\xcd\x0d"
			  "\x1d\xbe\xc6\xe9",
		.rlen   = 64 + 20,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 20 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01",
		.alen   = 8,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\x58\xc6\x84\x75\xe4\xe9\x6b\x0c"
			  "\xe1\xc5\x0b\x73\x4d\x82\x55\xa8"
			  "\x85\xe1\x59\xf7",
		.rlen   = 80 + 20,
       }, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"             
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x18"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 20 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x73\xe3\x19\x3f\x8b\xc9\xc6\xf4"
			  "\x5a\xf1\x5b\xa8\x98\x07\xc5\x36"
			  "\x47\x4c\xfc\x36",
		.rlen   = 64 + 20,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x20"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 20 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\xa3\xe8\x9b\x17\xe3\xf4\x7f\xde"
			  "\x1b\x9f\xc6\x81\x26\x43\x4a\x87"
			  "\x51\xee\xd6\x4e",
		.rlen   = 64 + 20,
	},
};

static struct aead_testvec hmac_sha256_aes_cbc_enc_tv_template[] = {
	{  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 32 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\xcc\xde\x2d\x6a\xae\xf1\x0b\xcc"
			  "\x38\x06\x38\x51\xb4\xb8\xf3\x5b"
			  "\x5c\x34\xa6\xa3\x6e\x0b\x05\xe5"
			  "\x6a\x6d\x44\xaa\x26\xa8\x44\xa5",
		.rlen   = 16 + 32,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 32 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\xf5\x33\x53\xf3\x68\x85\x2a\x99"
			  "\x0e\x06\x58\x8f\xba\xf6\x06\xda"
			  "\x49\x69\x0d\x5b\xd4\x36\x06\x62"
			  "\x35\x5e\x54\x58\x53\x4d\xdf\xbf",
		.rlen   = 32 + 32,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 32 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\x68\xb9\x3e\x90\x38\xa0\x88\x01"
			  "\xe7\xc6\xce\x10\x31\x2f\x9b\x1d"
			  "\x24\x78\xfb\xbe\x02\xe0\x4f\x40"
			  "\x10\xbd\xaa\xc6\xa7\x79\xe0\x1a",
		.rlen   = 48 + 32,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 32 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x7a\x1b\xd4\x3c\xdb\x17\x95\xe2"
			  "\xe0\x93\xec\xc9\x9f\xf7\xce\xd8"
			  "\x3f\x54\xe2\x49\x39\xe3\x71\x25"
			  "\x2b\x6c\xe9\x5d\xec\xec\x2b\x64",
		.rlen   = 64 + 32,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 32 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01",
		.alen   = 8,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\xbb\xd4\x0f\xbe\xa3\x3b\x4c\xb8"
			  "\x3a\xd2\xe1\x03\x86\xa5\x59\xb7"
			  "\x73\xc3\x46\x20\x2c\xb1\xef\x68"
			  "\xbb\x8a\x32\x7e\x12\x8c\x69\xcf",
		.rlen   = 80 + 32,
       }, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"             
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x18"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 32 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x2f\xee\x5f\xdb\x66\xfe\x79\x09"
			  "\x61\x81\x31\xea\x5b\x3d\x8e\xfb"
			  "\xca\x71\x85\x93\xf7\x85\x55\x8b"
			  "\x7a\xe4\x94\xca\x8b\xba\x19\x33",
		.rlen   = 64 + 32,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x20"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 32 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\x24\x29\xed\xc2\x31\x49\xdb\xb1"
			  "\x8f\x74\xbd\x17\x92\x03\xbe\x8f"
			  "\xf3\x61\xde\x1c\xe9\xdb\xcd\xd0"
			  "\xcc\xce\xe9\x85\x57\xcf\x6f\x5f",
		.rlen   = 64 + 32,
	},
};

#if defined(MY_ABC_HERE)
#define HMAC_SHA384_AES_CBC_ENC_TEST_VECTORS 7

static struct aead_testvec hmac_sha384_aes_cbc_enc_tv_template[] = {
	{  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 48 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\x79\x1c\xf1\x22\x95\x80\xe0\x60"
			  "\x7f\xf9\x92\x60\x83\xbd\x60\x9c"
			  "\xf6\x62\x8b\xa9\x7d\x56\xe2\xaf"
			  "\x80\x43\xbc\x41\x4a\x63\x0b\xa0"
			  "\x16\x25\xe2\xfe\x0a\x96\xf6\xa5"
			  "\x6c\x0b\xc2\x53\xb4\x27\xd9\x42",
		.rlen   = 16 + 48,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 48 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\x4e\x5b\xa8\x65\x51\xc6\x58\xaf"
			  "\x31\x57\x50\x3d\x01\xa1\xa4\x3f"
			  "\x42\xd1\xd7\x31\x76\x8d\xf8\xc8"
			  "\xe4\xd2\x7e\xc5\x23\xe7\xc6\x2e"
			  "\x2d\xfd\x9d\xc1\xac\x50\x1e\xcf"
			  "\xa0\x10\xeb\x1a\x9c\xb7\xe1\xca",
		.rlen   = 32 + 48,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 48 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\xa1\x52\xe7\xda\xf7\x05\xb6\xca"
			  "\xad\x0f\x51\xed\x5a\xd3\x0f\xdf"
			  "\xde\xeb\x3f\x31\xed\x3a\x43\x93"
			  "\x3b\xb7\xca\xc8\x1b\xe7\x3b\x61"
			  "\x6a\x05\xfd\x2d\x6a\x5c\xb1\x0d"
			  "\x6e\x7a\xeb\x1c\x84\xec\xdb\xde",
		.rlen   = 48 + 48,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 48 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x85\x7b\x91\xe0\x29\xeb\xd3\x59"
			  "\x7c\xe3\x67\x14\xbe\x71\x2a\xd2"
			  "\x8a\x1a\xd2\x35\x78\x6b\x69\xba"
			  "\x64\xa5\x04\x00\x19\xc3\x4c\xae"
			  "\x71\xff\x76\x9f\xbb\xc3\x29\x22"
			  "\xc2\xc6\x51\xf1\xe6\x29\x5e\xa5",
		.rlen   = 64 + 48,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 48 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01",
		.alen   = 8,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\x57\x5f\xb4\xd7\x74\x6f\x18\x97"
			  "\xb7\xde\xfc\xf3\x4e\x0d\x29\x4d"
			  "\xa0\xff\x39\x9e\x2d\xbf\x27\xac"
			  "\x54\xb9\x8a\x3e\xab\x3b\xac\xd3"
			  "\x36\x43\x74\xfc\xc2\x64\x81\x8a"
			  "\x2c\x15\x72\xdf\x3f\x9d\x5b\xa4",
		.rlen   = 80 + 48,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"             
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x18"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 48 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x29\x9b\x42\x47\x0b\xbf\xf3\x54"
			  "\x54\x95\xb0\x89\xd5\xa0\xc3\x78"
			  "\x60\x6c\x18\x39\x6d\xc9\xfb\x2a"
			  "\x34\x1c\xed\x95\x10\x1e\x43\x0a"
			  "\x72\xce\x26\xbc\x74\xd9\x6f\xa2"
			  "\xf1\xd9\xd0\xb1\xdf\x3d\x93\x14",
		.rlen   = 64 + 48,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x20"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 48 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\x9f\x50\xce\x64\xd9\xa3\xc9\x7a"
			  "\x15\x3a\x3d\x46\x9a\x90\xf3\x06"
			  "\x22\xad\xc5\x24\x77\x50\xb8\xfe"
			  "\xbe\x37\x16\x86\x34\x5f\xaf\x97"
			  "\x00\x9d\x86\xc8\x32\x4f\x72\x2f"
			  "\x48\x97\xad\xb6\xb9\x77\x33\xbc",
		.rlen   = 64 + 48,
	},
};

#endif  

static struct aead_testvec hmac_sha512_aes_cbc_enc_tv_template[] = {
	{  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 8 + 64 + 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a"
			  "\x3f\xdc\xad\x90\x03\x63\x5e\x68"
			  "\xc3\x13\xdd\xa4\x5c\x4d\x54\xa7"
			  "\x19\x6e\x03\x75\x2b\xa1\x62\xce"
			  "\xe0\xc6\x96\x75\xb2\x14\xca\x96"
			  "\xec\xbd\x50\x08\x07\x64\x1a\x49"
			  "\xe8\x9a\x7c\x06\x3d\xcb\xff\xb2"
			  "\xfa\x20\x89\xdd\x9c\xac\x9e\x16"
			  "\x18\x8a\xa0\x6d\x01\x6c\xa3\x3a",
		.rlen   = 16 + 64,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 8 + 64 + 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1"
			  "\xda\xb2\x0c\xb2\x26\xc4\xd5\xef"
			  "\x60\x38\xa4\x5e\x9a\x8c\x1b\x41"
			  "\x03\x9f\xc4\x64\x7f\x01\x42\x9b"
			  "\x0e\x1b\xea\xef\xbc\x88\x19\x5e"
			  "\x31\x7e\xc2\x95\xfc\x09\x32\x0a"
			  "\x46\x32\x7c\x41\x9c\x59\x3e\xe9"
			  "\x8f\x9f\xd4\x31\xd6\x22\xbd\xf8"
			  "\xf7\x0a\x94\xe5\xa9\xc3\xf6\x9d",
		.rlen   = 32 + 64,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"		 
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x6c\x3e\xa0\x47\x76\x30\xce\x21"
			  "\xa2\xce\x33\x4a\xa7\x46\xc2\xcd",
		.klen   = 8 + 64 + 16,
		.iv     = "\xc7\x82\xdc\x4c\x09\x8c\x66\xcb"
			  "\xd9\xcd\x27\xd8\x25\x68\x2c\x81",
		.input  = "This is a 48-byte message (exactly 3 AES blocks)",
		.ilen   = 48,
		.result = "\xd0\xa0\x2b\x38\x36\x45\x17\x53"
			  "\xd4\x93\x66\x5d\x33\xf0\xe8\x86"
			  "\x2d\xea\x54\xcd\xb2\x93\xab\xc7"
			  "\x50\x69\x39\x27\x67\x72\xf8\xd5"
			  "\x02\x1c\x19\x21\x6b\xad\x52\x5c"
			  "\x85\x79\x69\x5d\x83\xba\x26\x84"
			  "\x64\x19\x17\x5b\x57\xe0\x21\x0f"
			  "\xca\xdb\xa1\x26\x38\x14\xa2\x69"
			  "\xdb\x54\x67\x80\xc0\x54\xe0\xfd"
			  "\x3e\x91\xe7\x91\x7f\x13\x38\x44"
			  "\xb7\xb1\xd6\xc8\x7d\x48\x8d\x41"
			  "\x08\xea\x29\x6c\x74\x67\x3f\xb0"
			  "\xac\x7f\x5c\x1d\xf5\xee\x22\x66"
			  "\x27\xa6\xb6\x13\xba\xba\xf0\xc2",
		.rlen   = 48 + 64,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x56\xe4\x7a\x38\xc5\x59\x89\x74"
			  "\xbc\x46\x90\x3d\xba\x29\x03\x49",
		.klen   = 8 + 64 + 16,
		.iv     = "\x8c\xe8\x2e\xef\xbe\xa0\xda\x3c"
			  "\x44\x69\x9e\xd7\xdb\x51\xb7\xd9",
		.input  = "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf",
		.ilen   = 64,
		.result = "\xc3\x0e\x32\xff\xed\xc0\x77\x4e"
			  "\x6a\xff\x6a\xf0\x86\x9f\x71\xaa"
			  "\x0f\x3a\xf0\x7a\x9a\x31\xa9\xc6"
			  "\x84\xdb\x20\x7e\xb0\xef\x8e\x4e"
			  "\x35\x90\x7a\xa6\x32\xc3\xff\xdf"
			  "\x86\x8b\xb7\xb2\x9d\x3d\x46\xad"
			  "\x83\xce\x9f\x9a\x10\x2e\xe9\x9d"
			  "\x49\xa5\x3e\x87\xf4\xc3\xda\x55"
			  "\x82\xcd\x42\x28\x21\x20\x15\xcc"
			  "\xb7\xb2\x48\x40\xc7\x64\x41\x3a"
			  "\x61\x32\x82\x85\xcf\x27\xed\xb4"
			  "\xe4\x68\xa2\xf5\x79\x26\x27\xb2"
			  "\x51\x67\x6a\xc4\xf0\x66\x55\x50"
			  "\xbc\x6f\xed\xd5\x8d\xde\x23\x7c"
			  "\x62\x98\x14\xd7\x2f\x37\x8d\xdf"
			  "\xf4\x33\x80\xeb\x8e\xb4\xa4\xda",
		.rlen   = 64 + 64,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"             
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x10"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x90\xd3\x82\xb4\x10\xee\xba\x7a"
			  "\xd9\x38\xc4\x6c\xec\x1a\x82\xbf",
		.klen   = 8 + 64 + 16,
		.iv     = "\xe9\x6e\x8c\x08\xab\x46\x57\x63"
			  "\xfd\x09\x8d\x45\xdd\x3f\xf8\x93",
		.assoc  = "\x00\x00\x43\x21\x00\x00\x00\x01",
		.alen   = 8,
		.input  = "\x08\x00\x0e\xbd\xa7\x0a\x00\x00"
			  "\x8e\x9c\x08\x3d\xb9\x5b\x07\x00"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0e\x01",
		.ilen   = 80,
		.result = "\xf6\x63\xc2\x5d\x32\x5c\x18\xc6"
			  "\xa9\x45\x3e\x19\x4e\x12\x08\x49"
			  "\xa4\x87\x0b\x66\xcc\x6b\x99\x65"
			  "\x33\x00\x13\xb4\x89\x8d\xc8\x56"
			  "\xa4\x69\x9e\x52\x3a\x55\xdb\x08"
			  "\x0b\x59\xec\x3a\x8e\x4b\x7e\x52"
			  "\x77\x5b\x07\xd1\xdb\x34\xed\x9c"
			  "\x53\x8a\xb5\x0c\x55\x1b\x87\x4a"
			  "\xa2\x69\xad\xd0\x47\xad\x2d\x59"
			  "\x13\xac\x19\xb7\xcf\xba\xd4\xa6"
			  "\x74\x84\x94\xe2\xd7\x7a\xf9\xbf"
			  "\x00\x8a\xa2\xd5\xb7\xf3\x60\xcf"
			  "\xa0\x47\xdf\x4e\x09\xf4\xb1\x7f"
			  "\x14\xd9\x3d\x53\x8e\x12\xb3\x00"
			  "\x4c\x0a\x4e\x32\x40\x43\x88\xce"
			  "\x92\x26\xc1\x76\x20\x11\xeb\xba"
			  "\x62\x4f\x9a\x62\x25\xc3\x75\x80"
			  "\xb7\x0a\x17\xf5\xd7\x94\xb4\x14",
		.rlen   = 80 + 64,
       }, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"             
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x18"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen   = 8 + 64 + 24,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd"
			  "\x77\x4b\x69\x9d\x3a\x0d\xb4\x99"
			  "\x8f\xc6\x8e\x0e\x72\x58\xe3\x56"
			  "\xbb\x21\xd2\x7d\x93\x11\x17\x91"
			  "\xc4\x83\xfd\x0a\xea\x71\xfe\x77"
			  "\xae\x6f\x0a\xa5\xf0\xcf\xe1\x35"
			  "\xba\x03\xd5\x32\xfa\x5f\x41\x58"
			  "\x8d\x43\x98\xa7\x94\x16\x07\x02"
			  "\x0f\xb6\x81\x50\x28\x95\x2e\x75",
		.rlen   = 64 + 64,
	}, {  
#ifdef __LITTLE_ENDIAN
		.key    = "\x08\x00"		 
			  "\x01\x00"		 
#else
		.key    = "\x00\x08"		 
			  "\x00\x01"             
#endif
			  "\x00\x00\x00\x20"	 
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xdd\xee\xff\x11"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xdd\xee\xff\x11\x22"
			  "\x33\x44\x55\x66\x77\x88\x99\xaa"
			  "\xbb\xcc\xdd\xee\xff\x11\x22\x33"
			  "\x44\x55\x66\x77\x88\x99\xaa\xbb"
			  "\xcc\xdd\xee\xff\x11\x22\x33\x44"
			  "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen   = 8 + 64 + 32,
		.iv     = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input  = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen   = 64,
		.result = "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b"
			  "\xb2\x27\x69\x7f\x45\x64\x79\x2b"
			  "\xb7\xb8\x4c\xd4\x75\x94\x68\x40"
			  "\x2a\xea\x91\xc7\x3f\x7c\xed\x7b"
			  "\x95\x2c\x9b\xa8\xf5\xe5\x52\x8d"
			  "\x6b\xe1\xae\xf1\x74\xfa\x0d\x0c"
			  "\xe3\x8d\x64\xc3\x8d\xff\x7c\x8c"
			  "\xdb\xbf\xa0\xb4\x01\xa2\xa8\xa2"
			  "\x2c\xb1\x62\x2c\x10\xca\xf1\x21",
		.rlen   = 64 + 64,
	},
};

static struct cipher_testvec aes_lrw_enc_tv_template[] = {
	 
	{  
		.key    = "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\xf1\xb2\x73\xcd\x65\xa3\xdf\x5f"
			  "\xe9\x5d\x48\x92\x54\x63\x4e\xb8",
		.rlen   = 16,
	}, {  
		.key    = "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\x00\xc8\x2b\xae\x95\xbb\xcd\xe5"
			  "\x27\x4f\x07\x69\xb2\x60\xe1\x36",
		.rlen   = 16,
	}, {  
		.key    = "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\x76\x32\x21\x83\xed\x8f\xf1\x82"
			  "\xf9\x59\x62\x03\x69\x0e\x5e\x01",
		.rlen   = 16,
	}, {  
		.key    = "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen   = 40,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\x9c\x0f\x15\x2f\x55\xa2\xd8\xf0"
			  "\xd6\x7b\x8f\x9e\x28\x22\xbc\x41",
		.rlen   = 16,
	}, {  
		.key    = "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen   = 40,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\xd4\x27\x6a\x7f\x14\x91\x3d\x65"
			  "\xc8\x60\x48\x02\x87\xe3\x34\x06",
		.rlen   = 16,
	}, {  
		.key    = "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen   = 48,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\xbd\x06\xb8\xe1\xdb\x98\x89\x9e"
			  "\xc4\x98\xe4\x91\xcf\x1c\x70\x2b",
		.rlen   = 16,
	}, {  
		.key    = "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen   = 48,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen   = 16,
		.result = "\x5b\x90\x8e\xc1\xab\xdd\x67\x5f"
			  "\x3d\x69\x8a\x95\x53\xc8\x9c\xe5",
		.rlen   = 16,
	}, {
 
		.key    = "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen   = 48,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.ilen   = 512,
		.result = "\x1a\x1d\xa9\x30\xad\xf9\x2f\x9b"
			  "\xb6\x1d\xae\xef\xf0\x2f\xf8\x5a"
			  "\x39\x3c\xbf\x2a\xb2\x45\xb2\x23"
			  "\x1b\x63\x3c\xcf\xaa\xbe\xcf\x4e"
			  "\xfa\xe8\x29\xc2\x20\x68\x2b\x3c"
			  "\x2e\x8b\xf7\x6e\x25\xbd\xe3\x3d"
			  "\x66\x27\xd6\xaf\xd6\x64\x3e\xe3"
			  "\xe8\x58\x46\x97\x39\x51\x07\xde"
			  "\xcb\x37\xbc\xa9\xc0\x5f\x75\xc3"
			  "\x0e\x84\x23\x1d\x16\xd4\x1c\x59"
			  "\x9c\x1a\x02\x55\xab\x3a\x97\x1d"
			  "\xdf\xdd\xc7\x06\x51\xd7\x70\xae"
			  "\x23\xc6\x8c\xf5\x1e\xa0\xe5\x82"
			  "\xb8\xb2\xbf\x04\xa0\x32\x8e\x68"
			  "\xeb\xaf\x6e\x2d\x94\x22\x2f\xce"
			  "\x4c\xb5\x59\xe2\xa2\x2f\xa0\x98"
			  "\x1a\x97\xc6\xd4\xb5\x00\x59\xf2"
			  "\x84\x14\x72\xb1\x9a\x6e\xa3\x7f"
			  "\xea\x20\xe7\xcb\x65\x77\x3a\xdf"
			  "\xc8\x97\x67\x15\xc2\x2a\x27\xcc"
			  "\x18\x55\xa1\x24\x0b\x24\x24\xaf"
			  "\x5b\xec\x68\xb8\xc8\xf5\xba\x63"
			  "\xff\xed\x89\xce\xd5\x3d\x88\xf3"
			  "\x25\xef\x05\x7c\x3a\xef\xeb\xd8"
			  "\x7a\x32\x0d\xd1\x1e\x58\x59\x99"
			  "\x90\x25\xb5\x26\xb0\xe3\x2b\x6c"
			  "\x4c\xa9\x8b\x84\x4f\x5e\x01\x50"
			  "\x41\x30\x58\xc5\x62\x74\x52\x1d"
			  "\x45\x24\x6a\x42\x64\x4f\x97\x1c"
			  "\xa8\x66\xb5\x6d\x79\xd4\x0d\x48"
			  "\xc5\x5f\xf3\x90\x32\xdd\xdd\xe1"
			  "\xe4\xa9\x9f\xfc\xc3\x52\x5a\x46"
			  "\xe4\x81\x84\x95\x36\x59\x7a\x6b"
			  "\xaa\xb3\x60\xad\xce\x9f\x9f\x28"
			  "\xe0\x01\x75\x22\xc4\x4e\xa9\x62"
			  "\x5c\x62\x0d\x00\xcb\x13\xe8\x43"
			  "\x72\xd4\x2d\x53\x46\xb5\xd1\x16"
			  "\x22\x18\xdf\x34\x33\xf5\xd6\x1c"
			  "\xb8\x79\x78\x97\x94\xff\x72\x13"
			  "\x4c\x27\xfc\xcb\xbf\x01\x53\xa6"
			  "\xb4\x50\x6e\xde\xdf\xb5\x43\xa4"
			  "\x59\xdf\x52\xf9\x7c\xe0\x11\x6f"
			  "\x2d\x14\x8e\x24\x61\x2c\xe1\x17"
			  "\xcc\xce\x51\x0c\x19\x8a\x82\x30"
			  "\x94\xd5\x3d\x6a\x53\x06\x5e\xbd"
			  "\xb7\xeb\xfa\xfd\x27\x51\xde\x85"
			  "\x1e\x86\x53\x11\x53\x94\x00\xee"
			  "\x2b\x8c\x08\x2a\xbf\xdd\xae\x11"
			  "\xcb\x1e\xa2\x07\x9a\x80\xcf\x62"
			  "\x9b\x09\xdc\x95\x3c\x96\x8e\xb1"
			  "\x09\xbd\xe4\xeb\xdb\xca\x70\x7a"
			  "\x9e\xfa\x31\x18\x45\x3c\x21\x33"
			  "\xb0\xb3\x2b\xea\xf3\x71\x2d\xe1"
			  "\x03\xad\x1b\x48\xd4\x67\x27\xf0"
			  "\x62\xe4\x3d\xfb\x9b\x08\x76\xe7"
			  "\xdd\x2b\x01\x39\x04\x5a\x58\x7a"
			  "\xf7\x11\x90\xec\xbd\x51\x5c\x32"
			  "\x6b\xd7\x35\x39\x02\x6b\xf2\xa6"
			  "\xd0\x0d\x07\xe1\x06\xc4\x5b\x7d"
			  "\xe4\x6a\xd7\xee\x15\x1f\x83\xb4"
			  "\xa3\xa7\x5e\xc3\x90\xb7\xef\xd3"
			  "\xb7\x4f\xf8\x92\x4c\xb7\x3c\x29"
			  "\xcd\x7e\x2b\x5d\x43\xea\x42\xe7"
			  "\x74\x3f\x7d\x58\x88\x75\xde\x3e",
		.rlen   = 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	}
};

static struct cipher_testvec aes_lrw_dec_tv_template[] = {
	 
	{  
		.key    = "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\xf1\xb2\x73\xcd\x65\xa3\xdf\x5f"
			  "\xe9\x5d\x48\x92\x54\x63\x4e\xb8",
		.ilen   = 16,
		.result = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {  
		.key    = "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input  = "\x00\xc8\x2b\xae\x95\xbb\xcd\xe5"
			  "\x27\x4f\x07\x69\xb2\x60\xe1\x36",
		.ilen   = 16,
		.result  = "\x30\x31\x32\x33\x34\x35\x36\x37"
			   "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {  
		.key    = "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input  = "\x76\x32\x21\x83\xed\x8f\xf1\x82"
			  "\xf9\x59\x62\x03\x69\x0e\x5e\x01",
		.ilen   = 16,
		.result = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {  
		.key    = "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen   = 40,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\x9c\x0f\x15\x2f\x55\xa2\xd8\xf0"
			  "\xd6\x7b\x8f\x9e\x28\x22\xbc\x41",
		.ilen   = 16,
		.result = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {  
		.key    = "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen   = 40,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input  = "\xd4\x27\x6a\x7f\x14\x91\x3d\x65"
			  "\xc8\x60\x48\x02\x87\xe3\x34\x06",
		.ilen   = 16,
		.result = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {  
		.key    = "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen   = 48,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input  = "\xbd\x06\xb8\xe1\xdb\x98\x89\x9e"
			  "\xc4\x98\xe4\x91\xcf\x1c\x70\x2b",
		.ilen   = 16,
		.result = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {  
		.key    = "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen   = 48,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input  = "\x5b\x90\x8e\xc1\xab\xdd\x67\x5f"
			  "\x3d\x69\x8a\x95\x53\xc8\x9c\xe5",
		.ilen   = 16,
		.result = "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen   = 16,
	}, {
 
		.key    = "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen   = 48,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x1a\x1d\xa9\x30\xad\xf9\x2f\x9b"
			  "\xb6\x1d\xae\xef\xf0\x2f\xf8\x5a"
			  "\x39\x3c\xbf\x2a\xb2\x45\xb2\x23"
			  "\x1b\x63\x3c\xcf\xaa\xbe\xcf\x4e"
			  "\xfa\xe8\x29\xc2\x20\x68\x2b\x3c"
			  "\x2e\x8b\xf7\x6e\x25\xbd\xe3\x3d"
			  "\x66\x27\xd6\xaf\xd6\x64\x3e\xe3"
			  "\xe8\x58\x46\x97\x39\x51\x07\xde"
			  "\xcb\x37\xbc\xa9\xc0\x5f\x75\xc3"
			  "\x0e\x84\x23\x1d\x16\xd4\x1c\x59"
			  "\x9c\x1a\x02\x55\xab\x3a\x97\x1d"
			  "\xdf\xdd\xc7\x06\x51\xd7\x70\xae"
			  "\x23\xc6\x8c\xf5\x1e\xa0\xe5\x82"
			  "\xb8\xb2\xbf\x04\xa0\x32\x8e\x68"
			  "\xeb\xaf\x6e\x2d\x94\x22\x2f\xce"
			  "\x4c\xb5\x59\xe2\xa2\x2f\xa0\x98"
			  "\x1a\x97\xc6\xd4\xb5\x00\x59\xf2"
			  "\x84\x14\x72\xb1\x9a\x6e\xa3\x7f"
			  "\xea\x20\xe7\xcb\x65\x77\x3a\xdf"
			  "\xc8\x97\x67\x15\xc2\x2a\x27\xcc"
			  "\x18\x55\xa1\x24\x0b\x24\x24\xaf"
			  "\x5b\xec\x68\xb8\xc8\xf5\xba\x63"
			  "\xff\xed\x89\xce\xd5\x3d\x88\xf3"
			  "\x25\xef\x05\x7c\x3a\xef\xeb\xd8"
			  "\x7a\x32\x0d\xd1\x1e\x58\x59\x99"
			  "\x90\x25\xb5\x26\xb0\xe3\x2b\x6c"
			  "\x4c\xa9\x8b\x84\x4f\x5e\x01\x50"
			  "\x41\x30\x58\xc5\x62\x74\x52\x1d"
			  "\x45\x24\x6a\x42\x64\x4f\x97\x1c"
			  "\xa8\x66\xb5\x6d\x79\xd4\x0d\x48"
			  "\xc5\x5f\xf3\x90\x32\xdd\xdd\xe1"
			  "\xe4\xa9\x9f\xfc\xc3\x52\x5a\x46"
			  "\xe4\x81\x84\x95\x36\x59\x7a\x6b"
			  "\xaa\xb3\x60\xad\xce\x9f\x9f\x28"
			  "\xe0\x01\x75\x22\xc4\x4e\xa9\x62"
			  "\x5c\x62\x0d\x00\xcb\x13\xe8\x43"
			  "\x72\xd4\x2d\x53\x46\xb5\xd1\x16"
			  "\x22\x18\xdf\x34\x33\xf5\xd6\x1c"
			  "\xb8\x79\x78\x97\x94\xff\x72\x13"
			  "\x4c\x27\xfc\xcb\xbf\x01\x53\xa6"
			  "\xb4\x50\x6e\xde\xdf\xb5\x43\xa4"
			  "\x59\xdf\x52\xf9\x7c\xe0\x11\x6f"
			  "\x2d\x14\x8e\x24\x61\x2c\xe1\x17"
			  "\xcc\xce\x51\x0c\x19\x8a\x82\x30"
			  "\x94\xd5\x3d\x6a\x53\x06\x5e\xbd"
			  "\xb7\xeb\xfa\xfd\x27\x51\xde\x85"
			  "\x1e\x86\x53\x11\x53\x94\x00\xee"
			  "\x2b\x8c\x08\x2a\xbf\xdd\xae\x11"
			  "\xcb\x1e\xa2\x07\x9a\x80\xcf\x62"
			  "\x9b\x09\xdc\x95\x3c\x96\x8e\xb1"
			  "\x09\xbd\xe4\xeb\xdb\xca\x70\x7a"
			  "\x9e\xfa\x31\x18\x45\x3c\x21\x33"
			  "\xb0\xb3\x2b\xea\xf3\x71\x2d\xe1"
			  "\x03\xad\x1b\x48\xd4\x67\x27\xf0"
			  "\x62\xe4\x3d\xfb\x9b\x08\x76\xe7"
			  "\xdd\x2b\x01\x39\x04\x5a\x58\x7a"
			  "\xf7\x11\x90\xec\xbd\x51\x5c\x32"
			  "\x6b\xd7\x35\x39\x02\x6b\xf2\xa6"
			  "\xd0\x0d\x07\xe1\x06\xc4\x5b\x7d"
			  "\xe4\x6a\xd7\xee\x15\x1f\x83\xb4"
			  "\xa3\xa7\x5e\xc3\x90\xb7\xef\xd3"
			  "\xb7\x4f\xf8\x92\x4c\xb7\x3c\x29"
			  "\xcd\x7e\x2b\x5d\x43\xea\x42\xe7"
			  "\x74\x3f\x7d\x58\x88\x75\xde\x3e",
		.ilen   = 512,
		.result	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.rlen   = 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	}
};

static struct cipher_testvec aes_xts_enc_tv_template[] = {
	 
	{  
		.key    = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen   = 32,
		.result = "\x91\x7c\xf6\x9e\xbd\x68\xb2\xec"
			  "\x9b\x9f\xe9\xa3\xea\xdd\xa6\x92"
			  "\xcd\x43\xd2\xf5\x95\x98\xed\x85"
			  "\x8c\x02\xc2\x65\x2f\xbf\x92\x2e",
		.rlen   = 32,
	}, {  
		.key    = "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen   = 32,
		.result = "\xc4\x54\x18\x5e\x6a\x16\x93\x6e"
			  "\x39\x33\x40\x38\xac\xef\x83\x8b"
			  "\xfb\x18\x6f\xff\x74\x80\xad\xc4"
			  "\x28\x93\x82\xec\xd6\xd3\x94\xf0",
		.rlen   = 32,
	}, {  
		.key    = "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen   = 32,
		.result = "\xaf\x85\x33\x6b\x59\x7a\xfc\x1a"
			  "\x90\x0b\x2e\xb2\x1e\xc9\x49\xd2"
			  "\x92\xdf\x4c\x04\x7e\x0b\x21\x53"
			  "\x21\x86\xa5\x97\x1a\x22\x7a\x89",
		.rlen   = 32,
	}, {  
		.key    = "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen   = 512,
		.result = "\x27\xa7\x47\x9b\xef\xa1\xd4\x76"
			  "\x48\x9f\x30\x8c\xd4\xcf\xa6\xe2"
			  "\xa9\x6e\x4b\xbe\x32\x08\xff\x25"
			  "\x28\x7d\xd3\x81\x96\x16\xe8\x9c"
			  "\xc7\x8c\xf7\xf5\xe5\x43\x44\x5f"
			  "\x83\x33\xd8\xfa\x7f\x56\x00\x00"
			  "\x05\x27\x9f\xa5\xd8\xb5\xe4\xad"
			  "\x40\xe7\x36\xdd\xb4\xd3\x54\x12"
			  "\x32\x80\x63\xfd\x2a\xab\x53\xe5"
			  "\xea\x1e\x0a\x9f\x33\x25\x00\xa5"
			  "\xdf\x94\x87\xd0\x7a\x5c\x92\xcc"
			  "\x51\x2c\x88\x66\xc7\xe8\x60\xce"
			  "\x93\xfd\xf1\x66\xa2\x49\x12\xb4"
			  "\x22\x97\x61\x46\xae\x20\xce\x84"
			  "\x6b\xb7\xdc\x9b\xa9\x4a\x76\x7a"
			  "\xae\xf2\x0c\x0d\x61\xad\x02\x65"
			  "\x5e\xa9\x2d\xc4\xc4\xe4\x1a\x89"
			  "\x52\xc6\x51\xd3\x31\x74\xbe\x51"
			  "\xa1\x0c\x42\x11\x10\xe6\xd8\x15"
			  "\x88\xed\xe8\x21\x03\xa2\x52\xd8"
			  "\xa7\x50\xe8\x76\x8d\xef\xff\xed"
			  "\x91\x22\x81\x0a\xae\xb9\x9f\x91"
			  "\x72\xaf\x82\xb6\x04\xdc\x4b\x8e"
			  "\x51\xbc\xb0\x82\x35\xa6\xf4\x34"
			  "\x13\x32\xe4\xca\x60\x48\x2a\x4b"
			  "\xa1\xa0\x3b\x3e\x65\x00\x8f\xc5"
			  "\xda\x76\xb7\x0b\xf1\x69\x0d\xb4"
			  "\xea\xe2\x9c\x5f\x1b\xad\xd0\x3c"
			  "\x5c\xcf\x2a\x55\xd7\x05\xdd\xcd"
			  "\x86\xd4\x49\x51\x1c\xeb\x7e\xc3"
			  "\x0b\xf1\x2b\x1f\xa3\x5b\x91\x3f"
			  "\x9f\x74\x7a\x8a\xfd\x1b\x13\x0e"
			  "\x94\xbf\xf9\x4e\xff\xd0\x1a\x91"
			  "\x73\x5c\xa1\x72\x6a\xcd\x0b\x19"
			  "\x7c\x4e\x5b\x03\x39\x36\x97\xe1"
			  "\x26\x82\x6f\xb6\xbb\xde\x8e\xcc"
			  "\x1e\x08\x29\x85\x16\xe2\xc9\xed"
			  "\x03\xff\x3c\x1b\x78\x60\xf6\xde"
			  "\x76\xd4\xce\xcd\x94\xc8\x11\x98"
			  "\x55\xef\x52\x97\xca\x67\xe9\xf3"
			  "\xe7\xff\x72\xb1\xe9\x97\x85\xca"
			  "\x0a\x7e\x77\x20\xc5\xb3\x6d\xc6"
			  "\xd7\x2c\xac\x95\x74\xc8\xcb\xbc"
			  "\x2f\x80\x1e\x23\xe5\x6f\xd3\x44"
			  "\xb0\x7f\x22\x15\x4b\xeb\xa0\xf0"
			  "\x8c\xe8\x89\x1e\x64\x3e\xd9\x95"
			  "\xc9\x4d\x9a\x69\xc9\xf1\xb5\xf4"
			  "\x99\x02\x7a\x78\x57\x2a\xee\xbd"
			  "\x74\xd2\x0c\xc3\x98\x81\xc2\x13"
			  "\xee\x77\x0b\x10\x10\xe4\xbe\xa7"
			  "\x18\x84\x69\x77\xae\x11\x9f\x7a"
			  "\x02\x3a\xb5\x8c\xca\x0a\xd7\x52"
			  "\xaf\xe6\x56\xbb\x3c\x17\x25\x6a"
			  "\x9f\x6e\x9b\xf1\x9f\xdd\x5a\x38"
			  "\xfc\x82\xbb\xe8\x72\xc5\x53\x9e"
			  "\xdb\x60\x9e\xf4\xf7\x9c\x20\x3e"
			  "\xbb\x14\x0f\x2e\x58\x3c\xb2\xad"
			  "\x15\xb4\xaa\x5b\x65\x50\x16\xa8"
			  "\x44\x92\x77\xdb\xd4\x77\xef\x2c"
			  "\x8d\x6c\x01\x7d\xb7\x38\xb1\x8d"
			  "\xeb\x4a\x42\x7d\x19\x23\xce\x3f"
			  "\xf2\x62\x73\x57\x79\xa4\x18\xf2"
			  "\x0a\x28\x2d\xf9\x20\x14\x7b\xea"
			  "\xbe\x42\x1e\xe5\x31\x9d\x05\x68",
		.rlen   = 512,
	}, {  
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\x1c\x3b\x3a\x10\x2f\x77\x03\x86"
			  "\xe4\x83\x6c\x99\xe3\x70\xcf\x9b"
			  "\xea\x00\x80\x3f\x5e\x48\x23\x57"
			  "\xa4\xae\x12\xd4\x14\xa3\xe6\x3b"
			  "\x5d\x31\xe2\x76\xf8\xfe\x4a\x8d"
			  "\x66\xb3\x17\xf9\xac\x68\x3f\x44"
			  "\x68\x0a\x86\xac\x35\xad\xfc\x33"
			  "\x45\xbe\xfe\xcb\x4b\xb1\x88\xfd"
			  "\x57\x76\x92\x6c\x49\xa3\x09\x5e"
			  "\xb1\x08\xfd\x10\x98\xba\xec\x70"
			  "\xaa\xa6\x69\x99\xa7\x2a\x82\xf2"
			  "\x7d\x84\x8b\x21\xd4\xa7\x41\xb0"
			  "\xc5\xcd\x4d\x5f\xff\x9d\xac\x89"
			  "\xae\xba\x12\x29\x61\xd0\x3a\x75"
			  "\x71\x23\xe9\x87\x0f\x8a\xcf\x10"
			  "\x00\x02\x08\x87\x89\x14\x29\xca"
			  "\x2a\x3e\x7a\x7d\x7d\xf7\xb1\x03"
			  "\x55\x16\x5c\x8b\x9a\x6d\x0a\x7d"
			  "\xe8\xb0\x62\xc4\x50\x0d\xc4\xcd"
			  "\x12\x0c\x0f\x74\x18\xda\xe3\xd0"
			  "\xb5\x78\x1c\x34\x80\x3f\xa7\x54"
			  "\x21\xc7\x90\xdf\xe1\xde\x18\x34"
			  "\xf2\x80\xd7\x66\x7b\x32\x7f\x6c"
			  "\x8c\xd7\x55\x7e\x12\xac\x3a\x0f"
			  "\x93\xec\x05\xc5\x2e\x04\x93\xef"
			  "\x31\xa1\x2d\x3d\x92\x60\xf7\x9a"
			  "\x28\x9d\x6a\x37\x9b\xc7\x0c\x50"
			  "\x84\x14\x73\xd1\xa8\xcc\x81\xec"
			  "\x58\x3e\x96\x45\xe0\x7b\x8d\x96"
			  "\x70\x65\x5b\xa5\xbb\xcf\xec\xc6"
			  "\xdc\x39\x66\x38\x0a\xd8\xfe\xcb"
			  "\x17\xb6\xba\x02\x46\x9a\x02\x0a"
			  "\x84\xe1\x8e\x8f\x84\x25\x20\x70"
			  "\xc1\x3e\x9f\x1f\x28\x9b\xe5\x4f"
			  "\xbc\x48\x14\x57\x77\x8f\x61\x60"
			  "\x15\xe1\x32\x7a\x02\xb1\x40\xf1"
			  "\x50\x5e\xb3\x09\x32\x6d\x68\x37"
			  "\x8f\x83\x74\x59\x5c\x84\x9d\x84"
			  "\xf4\xc3\x33\xec\x44\x23\x88\x51"
			  "\x43\xcb\x47\xbd\x71\xc5\xed\xae"
			  "\x9b\xe6\x9a\x2f\xfe\xce\xb1\xbe"
			  "\xc9\xde\x24\x4f\xbe\x15\x99\x2b"
			  "\x11\xb7\x7c\x04\x0f\x12\xbd\x8f"
			  "\x6a\x97\x5a\x44\xa0\xf9\x0c\x29"
			  "\xa9\xab\xc3\xd4\xd8\x93\x92\x72"
			  "\x84\xc5\x87\x54\xcc\xe2\x94\x52"
			  "\x9f\x86\x14\xdc\xd2\xab\xa9\x91"
			  "\x92\x5f\xed\xc4\xae\x74\xff\xac"
			  "\x6e\x33\x3b\x93\xeb\x4a\xff\x04"
			  "\x79\xda\x9a\x41\x0e\x44\x50\xe0"
			  "\xdd\x7a\xe4\xc6\xe2\x91\x09\x00"
			  "\x57\x5d\xa4\x01\xfc\x07\x05\x9f"
			  "\x64\x5e\x8b\x7e\x9b\xfd\xef\x33"
			  "\x94\x30\x54\xff\x84\x01\x14\x93"
			  "\xc2\x7b\x34\x29\xea\xed\xb4\xed"
			  "\x53\x76\x44\x1a\x77\xed\x43\x85"
			  "\x1a\xd7\x7f\x16\xf5\x41\xdf\xd2"
			  "\x69\xd5\x0d\x6a\x5f\x14\xfb\x0a"
			  "\xab\x1c\xbb\x4c\x15\x50\xbe\x97"
			  "\xf7\xab\x40\x66\x19\x3c\x4c\xaa"
			  "\x77\x3d\xad\x38\x01\x4b\xd2\x09"
			  "\x2f\xa7\x55\xc8\x24\xbb\x5e\x54"
			  "\xc4\xf3\x6f\xfd\xa9\xfc\xea\x70"
			  "\xb9\xc6\xe6\x93\xe1\x48\xc1\x51",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	}
};

static struct cipher_testvec aes_xts_dec_tv_template[] = {
	 
	{  
		.key    = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input = "\x91\x7c\xf6\x9e\xbd\x68\xb2\xec"
			 "\x9b\x9f\xe9\xa3\xea\xdd\xa6\x92"
			 "\xcd\x43\xd2\xf5\x95\x98\xed\x85"
			 "\x8c\x02\xc2\x65\x2f\xbf\x92\x2e",
		.ilen   = 32,
		.result  = "\x00\x00\x00\x00\x00\x00\x00\x00"
			   "\x00\x00\x00\x00\x00\x00\x00\x00"
			   "\x00\x00\x00\x00\x00\x00\x00\x00"
			   "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen   = 32,
	}, {  
		.key    = "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\xc4\x54\x18\x5e\x6a\x16\x93\x6e"
			  "\x39\x33\x40\x38\xac\xef\x83\x8b"
			  "\xfb\x18\x6f\xff\x74\x80\xad\xc4"
			  "\x28\x93\x82\xec\xd6\xd3\x94\xf0",
		.ilen   = 32,
		.result = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen   = 32,
	}, {  
		.key    = "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen   = 32,
		.iv     = "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input = "\xaf\x85\x33\x6b\x59\x7a\xfc\x1a"
			  "\x90\x0b\x2e\xb2\x1e\xc9\x49\xd2"
			  "\x92\xdf\x4c\x04\x7e\x0b\x21\x53"
			  "\x21\x86\xa5\x97\x1a\x22\x7a\x89",
		.ilen   = 32,
		.result  = "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen   = 32,
	}, {  
		.key    = "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen   = 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input  = "\x27\xa7\x47\x9b\xef\xa1\xd4\x76"
			  "\x48\x9f\x30\x8c\xd4\xcf\xa6\xe2"
			  "\xa9\x6e\x4b\xbe\x32\x08\xff\x25"
			  "\x28\x7d\xd3\x81\x96\x16\xe8\x9c"
			  "\xc7\x8c\xf7\xf5\xe5\x43\x44\x5f"
			  "\x83\x33\xd8\xfa\x7f\x56\x00\x00"
			  "\x05\x27\x9f\xa5\xd8\xb5\xe4\xad"
			  "\x40\xe7\x36\xdd\xb4\xd3\x54\x12"
			  "\x32\x80\x63\xfd\x2a\xab\x53\xe5"
			  "\xea\x1e\x0a\x9f\x33\x25\x00\xa5"
			  "\xdf\x94\x87\xd0\x7a\x5c\x92\xcc"
			  "\x51\x2c\x88\x66\xc7\xe8\x60\xce"
			  "\x93\xfd\xf1\x66\xa2\x49\x12\xb4"
			  "\x22\x97\x61\x46\xae\x20\xce\x84"
			  "\x6b\xb7\xdc\x9b\xa9\x4a\x76\x7a"
			  "\xae\xf2\x0c\x0d\x61\xad\x02\x65"
			  "\x5e\xa9\x2d\xc4\xc4\xe4\x1a\x89"
			  "\x52\xc6\x51\xd3\x31\x74\xbe\x51"
			  "\xa1\x0c\x42\x11\x10\xe6\xd8\x15"
			  "\x88\xed\xe8\x21\x03\xa2\x52\xd8"
			  "\xa7\x50\xe8\x76\x8d\xef\xff\xed"
			  "\x91\x22\x81\x0a\xae\xb9\x9f\x91"
			  "\x72\xaf\x82\xb6\x04\xdc\x4b\x8e"
			  "\x51\xbc\xb0\x82\x35\xa6\xf4\x34"
			  "\x13\x32\xe4\xca\x60\x48\x2a\x4b"
			  "\xa1\xa0\x3b\x3e\x65\x00\x8f\xc5"
			  "\xda\x76\xb7\x0b\xf1\x69\x0d\xb4"
			  "\xea\xe2\x9c\x5f\x1b\xad\xd0\x3c"
			  "\x5c\xcf\x2a\x55\xd7\x05\xdd\xcd"
			  "\x86\xd4\x49\x51\x1c\xeb\x7e\xc3"
			  "\x0b\xf1\x2b\x1f\xa3\x5b\x91\x3f"
			  "\x9f\x74\x7a\x8a\xfd\x1b\x13\x0e"
			  "\x94\xbf\xf9\x4e\xff\xd0\x1a\x91"
			  "\x73\x5c\xa1\x72\x6a\xcd\x0b\x19"
			  "\x7c\x4e\x5b\x03\x39\x36\x97\xe1"
			  "\x26\x82\x6f\xb6\xbb\xde\x8e\xcc"
			  "\x1e\x08\x29\x85\x16\xe2\xc9\xed"
			  "\x03\xff\x3c\x1b\x78\x60\xf6\xde"
			  "\x76\xd4\xce\xcd\x94\xc8\x11\x98"
			  "\x55\xef\x52\x97\xca\x67\xe9\xf3"
			  "\xe7\xff\x72\xb1\xe9\x97\x85\xca"
			  "\x0a\x7e\x77\x20\xc5\xb3\x6d\xc6"
			  "\xd7\x2c\xac\x95\x74\xc8\xcb\xbc"
			  "\x2f\x80\x1e\x23\xe5\x6f\xd3\x44"
			  "\xb0\x7f\x22\x15\x4b\xeb\xa0\xf0"
			  "\x8c\xe8\x89\x1e\x64\x3e\xd9\x95"
			  "\xc9\x4d\x9a\x69\xc9\xf1\xb5\xf4"
			  "\x99\x02\x7a\x78\x57\x2a\xee\xbd"
			  "\x74\xd2\x0c\xc3\x98\x81\xc2\x13"
			  "\xee\x77\x0b\x10\x10\xe4\xbe\xa7"
			  "\x18\x84\x69\x77\xae\x11\x9f\x7a"
			  "\x02\x3a\xb5\x8c\xca\x0a\xd7\x52"
			  "\xaf\xe6\x56\xbb\x3c\x17\x25\x6a"
			  "\x9f\x6e\x9b\xf1\x9f\xdd\x5a\x38"
			  "\xfc\x82\xbb\xe8\x72\xc5\x53\x9e"
			  "\xdb\x60\x9e\xf4\xf7\x9c\x20\x3e"
			  "\xbb\x14\x0f\x2e\x58\x3c\xb2\xad"
			  "\x15\xb4\xaa\x5b\x65\x50\x16\xa8"
			  "\x44\x92\x77\xdb\xd4\x77\xef\x2c"
			  "\x8d\x6c\x01\x7d\xb7\x38\xb1\x8d"
			  "\xeb\x4a\x42\x7d\x19\x23\xce\x3f"
			  "\xf2\x62\x73\x57\x79\xa4\x18\xf2"
			  "\x0a\x28\x2d\xf9\x20\x14\x7b\xea"
			  "\xbe\x42\x1e\xe5\x31\x9d\x05\x68",
		.ilen   = 512,
		.result = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen   = 512,
	}, {  
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x1c\x3b\x3a\x10\x2f\x77\x03\x86"
			  "\xe4\x83\x6c\x99\xe3\x70\xcf\x9b"
			  "\xea\x00\x80\x3f\x5e\x48\x23\x57"
			  "\xa4\xae\x12\xd4\x14\xa3\xe6\x3b"
			  "\x5d\x31\xe2\x76\xf8\xfe\x4a\x8d"
			  "\x66\xb3\x17\xf9\xac\x68\x3f\x44"
			  "\x68\x0a\x86\xac\x35\xad\xfc\x33"
			  "\x45\xbe\xfe\xcb\x4b\xb1\x88\xfd"
			  "\x57\x76\x92\x6c\x49\xa3\x09\x5e"
			  "\xb1\x08\xfd\x10\x98\xba\xec\x70"
			  "\xaa\xa6\x69\x99\xa7\x2a\x82\xf2"
			  "\x7d\x84\x8b\x21\xd4\xa7\x41\xb0"
			  "\xc5\xcd\x4d\x5f\xff\x9d\xac\x89"
			  "\xae\xba\x12\x29\x61\xd0\x3a\x75"
			  "\x71\x23\xe9\x87\x0f\x8a\xcf\x10"
			  "\x00\x02\x08\x87\x89\x14\x29\xca"
			  "\x2a\x3e\x7a\x7d\x7d\xf7\xb1\x03"
			  "\x55\x16\x5c\x8b\x9a\x6d\x0a\x7d"
			  "\xe8\xb0\x62\xc4\x50\x0d\xc4\xcd"
			  "\x12\x0c\x0f\x74\x18\xda\xe3\xd0"
			  "\xb5\x78\x1c\x34\x80\x3f\xa7\x54"
			  "\x21\xc7\x90\xdf\xe1\xde\x18\x34"
			  "\xf2\x80\xd7\x66\x7b\x32\x7f\x6c"
			  "\x8c\xd7\x55\x7e\x12\xac\x3a\x0f"
			  "\x93\xec\x05\xc5\x2e\x04\x93\xef"
			  "\x31\xa1\x2d\x3d\x92\x60\xf7\x9a"
			  "\x28\x9d\x6a\x37\x9b\xc7\x0c\x50"
			  "\x84\x14\x73\xd1\xa8\xcc\x81\xec"
			  "\x58\x3e\x96\x45\xe0\x7b\x8d\x96"
			  "\x70\x65\x5b\xa5\xbb\xcf\xec\xc6"
			  "\xdc\x39\x66\x38\x0a\xd8\xfe\xcb"
			  "\x17\xb6\xba\x02\x46\x9a\x02\x0a"
			  "\x84\xe1\x8e\x8f\x84\x25\x20\x70"
			  "\xc1\x3e\x9f\x1f\x28\x9b\xe5\x4f"
			  "\xbc\x48\x14\x57\x77\x8f\x61\x60"
			  "\x15\xe1\x32\x7a\x02\xb1\x40\xf1"
			  "\x50\x5e\xb3\x09\x32\x6d\x68\x37"
			  "\x8f\x83\x74\x59\x5c\x84\x9d\x84"
			  "\xf4\xc3\x33\xec\x44\x23\x88\x51"
			  "\x43\xcb\x47\xbd\x71\xc5\xed\xae"
			  "\x9b\xe6\x9a\x2f\xfe\xce\xb1\xbe"
			  "\xc9\xde\x24\x4f\xbe\x15\x99\x2b"
			  "\x11\xb7\x7c\x04\x0f\x12\xbd\x8f"
			  "\x6a\x97\x5a\x44\xa0\xf9\x0c\x29"
			  "\xa9\xab\xc3\xd4\xd8\x93\x92\x72"
			  "\x84\xc5\x87\x54\xcc\xe2\x94\x52"
			  "\x9f\x86\x14\xdc\xd2\xab\xa9\x91"
			  "\x92\x5f\xed\xc4\xae\x74\xff\xac"
			  "\x6e\x33\x3b\x93\xeb\x4a\xff\x04"
			  "\x79\xda\x9a\x41\x0e\x44\x50\xe0"
			  "\xdd\x7a\xe4\xc6\xe2\x91\x09\x00"
			  "\x57\x5d\xa4\x01\xfc\x07\x05\x9f"
			  "\x64\x5e\x8b\x7e\x9b\xfd\xef\x33"
			  "\x94\x30\x54\xff\x84\x01\x14\x93"
			  "\xc2\x7b\x34\x29\xea\xed\xb4\xed"
			  "\x53\x76\x44\x1a\x77\xed\x43\x85"
			  "\x1a\xd7\x7f\x16\xf5\x41\xdf\xd2"
			  "\x69\xd5\x0d\x6a\x5f\x14\xfb\x0a"
			  "\xab\x1c\xbb\x4c\x15\x50\xbe\x97"
			  "\xf7\xab\x40\x66\x19\x3c\x4c\xaa"
			  "\x77\x3d\xad\x38\x01\x4b\xd2\x09"
			  "\x2f\xa7\x55\xc8\x24\xbb\x5e\x54"
			  "\xc4\xf3\x6f\xfd\xa9\xfc\xea\x70"
			  "\xb9\xc6\xe6\x93\xe1\x48\xc1\x51",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	}
};

static struct cipher_testvec aes_ctr_enc_tv_template[] = {
	{  
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x87\x4d\x61\x91\xb6\x20\xe3\x26"
			  "\x1b\xef\x68\x64\x99\x0d\xb6\xce"
			  "\x98\x06\xf6\x6b\x79\x70\xfd\xff"
			  "\x86\x17\x18\x7b\xb9\xff\xfd\xff"
			  "\x5a\xe4\xdf\x3e\xdb\xd5\xd3\x5e"
			  "\x5b\x4f\x09\x02\x0d\xb0\x3e\xab"
			  "\x1e\x03\x1d\xda\x2f\xbe\x03\xd1"
			  "\x79\x21\x70\xa0\xf3\x00\x9c\xee",
		.rlen	= 64,
	}, {
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x1a\xbc\x93\x24\x17\x52\x1c\xa2"
			  "\x4f\x2b\x04\x59\xfe\x7e\x6e\x0b"
			  "\x09\x03\x39\xec\x0a\xa6\xfa\xef"
			  "\xd5\xcc\xc2\xc6\xf4\xce\x8e\x94"
			  "\x1e\x36\xb2\x6b\xd1\xeb\xc6\x70"
			  "\xd1\xbd\x1d\x66\x56\x20\xab\xf7"
			  "\x4f\x78\xa7\xf6\xd2\x98\x09\x58"
			  "\x5a\x97\xda\xec\x58\xc6\xb0\x50",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x60\x1e\xc3\x13\x77\x57\x89\xa5"
			  "\xb7\xa7\xf5\x04\xbb\xf3\xd2\x28"
			  "\xf4\x43\xe3\xca\x4d\x62\xb5\x9a"
			  "\xca\x84\xe9\x90\xca\xca\xf5\xc5"
			  "\x2b\x09\x30\xda\xa2\x3d\xe9\x4c"
			  "\xe8\x70\x17\xba\x2d\x84\x98\x8d"
			  "\xdf\xc9\xc5\x8d\xb6\x7a\xad\xa6"
			  "\x13\xc2\xdd\x08\x45\x79\x41\xa6",
		.rlen	= 64,
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\x04\xF3\xD3\x88\x17\xEF\xDC\xEF"
			  "\x8B\x04\xF8\x3A\x66\x8D\x1A\x53"
			  "\x57\x1F\x4B\x23\xE4\xA0\xAF\xF9"
			  "\x69\x95\x35\x98\x8D\x4D\x8C\xC1"
			  "\xF0\xB2\x7F\x80\xBB\x54\x28\xA2"
			  "\x7A\x1B\x9F\x77\xEC\x0E\x6E\xDE"
			  "\xF0\xEC\xB8\xE4\x20\x62\xEE\xDB"
			  "\x5D\xF5\xDD\xE3\x54\xFC\xDD\xEB"
			  "\x6A\xEE\x65\xA1\x21\xD6\xD7\x81"
			  "\x47\x61\x12\x4D\xC2\x8C\xFA\x78"
			  "\x1F\x28\x02\x01\xC3\xFC\x1F\xEC"
			  "\x0F\x10\x4F\xB3\x12\x45\xC6\x3B"
			  "\x7E\x08\xF9\x5A\xD0\x5D\x73\x2D"
			  "\x58\xA4\xE5\xCB\x1C\xB4\xCE\x74"
			  "\x32\x41\x1F\x31\x9C\x08\xA2\x5D"
			  "\x67\xEB\x72\x1D\xF8\xE7\x70\x54"
			  "\x34\x4B\x31\x69\x84\x66\x96\x44"
			  "\x56\xCC\x1E\xD9\xE6\x13\x6A\xB9"
			  "\x2D\x0A\x05\x45\x2D\x90\xCC\xDF"
			  "\x16\x5C\x5F\x79\x34\x52\x54\xFE"
			  "\xFE\xCD\xAD\x04\x2E\xAD\x86\x06"
			  "\x1F\x37\xE8\x28\xBC\xD3\x8F\x5B"
			  "\x92\x66\x87\x3B\x8A\x0A\x1A\xCC"
			  "\x6E\xAB\x9F\x0B\xFA\x5C\xE6\xFD"
			  "\x3C\x98\x08\x12\xEC\xAA\x9E\x11"
			  "\xCA\xB2\x1F\xCE\x5E\x5B\xB2\x72"
			  "\x9C\xCC\x5D\xC5\xE0\x32\xC0\x56"
			  "\xD5\x45\x16\xD2\xAF\x13\x66\xF7"
			  "\x8C\x67\xAC\x79\xB2\xAF\x56\x27"
			  "\x3F\xCC\xFE\xCB\x1E\xC0\x75\xF1"
			  "\xA7\xC9\xC3\x1D\x8E\xDD\xF9\xD4"
			  "\x42\xC8\x21\x08\x16\xF7\x01\xD7"
			  "\xAC\x8E\x3F\x1D\x56\xC1\x06\xE4"
			  "\x9C\x62\xD6\xA5\x6A\x50\x44\xB3"
			  "\x35\x1C\x82\xB9\x10\xF9\x42\xA1"
			  "\xFC\x74\x9B\x44\x4F\x25\x02\xE3"
			  "\x08\xF5\xD4\x32\x39\x08\x11\xE8"
			  "\xD2\x6B\x50\x53\xD4\x08\xD1\x6B"
			  "\x3A\x4A\x68\x7B\x7C\xCD\x46\x5E"
			  "\x0D\x07\x19\xDB\x67\xD7\x98\x91"
			  "\xD7\x17\x10\x9B\x7B\x8A\x9B\x33"
			  "\xAE\xF3\x00\xA6\xD4\x15\xD9\xEA"
			  "\x85\x99\x22\xE8\x91\x38\x70\x83"
			  "\x93\x01\x24\x6C\xFA\x9A\xB9\x07"
			  "\xEA\x8D\x3B\xD9\x2A\x43\x59\x16"
			  "\x2F\x69\xEE\x84\x36\x44\x76\x98"
			  "\xF3\x04\x2A\x7C\x74\x3D\x29\x2B"
			  "\x0D\xAD\x8F\x44\x82\x9E\x57\x8D"
			  "\xAC\xED\x18\x1F\x50\xA4\xF5\x98"
			  "\x1F\xBD\x92\x91\x1B\x2D\xA6\xD6"
			  "\xD2\xE3\x02\xAA\x92\x3B\xC6\xB3"
			  "\x1B\x39\x72\xD5\x26\xCA\x04\xE0"
			  "\xFC\x58\x78\xBB\xB1\x3F\xA1\x9C"
			  "\x42\x24\x3E\x2E\x22\xBB\x4B\xBA"
			  "\xF4\x52\x0A\xE6\xAE\x47\xB4\x7D"
			  "\x1D\xA8\xBE\x81\x1A\x75\xDA\xAC"
			  "\xA6\x25\x1E\xEF\x3A\xC0\x6C\x63"
			  "\xEF\xDC\xC9\x79\x10\x26\xE8\x61"
			  "\x29\xFC\xA4\x05\xDF\x7D\x5C\x63"
			  "\x10\x09\x9B\x46\x9B\xF2\x2C\x2B"
			  "\xFA\x3A\x05\x4C\xFA\xD1\xFF\xFE"
			  "\xF1\x4C\xE5\xB2\x91\x64\x0C\x51",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12"
			  "\x7B\xE4\x4D",
		.ilen	= 499,
		.result	= "\xDA\x4E\x3F\xBC\xE8\xB6\x3A\xA2"
			  "\xD5\x4D\x84\x4A\xA9\x0C\xE1\xA5"
			  "\xB8\x73\xBC\xF9\xBB\x59\x2F\x44"
			  "\x8B\xAB\x82\x6C\xB4\x32\x9A\xDE"
			  "\x5A\x0B\xDB\x7A\x6B\xF2\x38\x9F"
			  "\x06\xF7\xF7\xFF\xFF\xC0\x8A\x2E"
			  "\x76\xEA\x06\x32\x23\xF3\x59\x2E"
			  "\x75\xDE\x71\x86\x3C\x98\x23\x44"
			  "\x5B\xF2\xFA\x6A\x00\xBB\xC1\xAD"
			  "\x58\xBD\x3E\x6F\x2E\xB4\x19\x04"
			  "\x70\x8B\x92\x55\x23\xE9\x6A\x3A"
			  "\x78\x7A\x1B\x10\x85\x52\x9C\x12"
			  "\xE4\x55\x81\x21\xCE\x53\xD0\x3B"
			  "\x63\x77\x2C\x74\xD1\xF5\x60\xF3"
			  "\xA1\xDE\x44\x3C\x8F\x4D\x2F\xDD"
			  "\x8A\xFE\x3C\x42\x8E\xD3\xF2\x8E"
			  "\xA8\x28\x69\x65\x31\xE1\x45\x83"
			  "\xE4\x49\xC4\x9C\xA7\x28\xAA\x21"
			  "\xCD\x5D\x0F\x15\xB7\x93\x07\x26"
			  "\xB0\x65\x6D\x91\x90\x23\x7A\xC6"
			  "\xDB\x68\xB0\xA1\x8E\xA4\x76\x4E"
			  "\xC6\x91\x83\x20\x92\x4D\x63\x7A"
			  "\x45\x18\x18\x74\x19\xAD\x71\x01"
			  "\x6B\x23\xAD\x9D\x4E\xE4\x6E\x46"
			  "\xC9\x73\x7A\xF9\x02\x95\xF4\x07"
			  "\x0E\x7A\xA6\xC5\xAE\xFA\x15\x2C"
			  "\x51\x71\xF1\xDC\x22\xB6\xAC\xD8"
			  "\x19\x24\x44\xBC\x0C\xFB\x3C\x2D"
			  "\xB1\x50\x47\x15\x0E\xDB\xB6\xD7"
			  "\xE8\x61\xE5\x95\x52\x1E\x3E\x49"
			  "\x70\xE9\x66\x04\x4C\xE1\xAF\xBD"
			  "\xDD\x15\x3B\x20\x59\x24\xFF\xB0"
			  "\x39\xAA\xE7\xBF\x23\xA3\x6E\xD5"
			  "\x15\xF0\x61\x4F\xAE\x89\x10\x58"
			  "\x5A\x33\x95\x52\x2A\xB5\x77\x9C"
			  "\xA5\x43\x80\x40\x27\x2D\xAE\xD9"
			  "\x3F\xE0\x80\x94\x78\x79\xCB\x7E"
			  "\xAD\x12\x44\x4C\xEC\x27\xB0\xEE"
			  "\x0B\x05\x2A\x82\x99\x58\xBB\x7A"
			  "\x8D\x6D\x9D\x8E\xE2\x8E\xE7\x93"
			  "\x2F\xB3\x09\x8D\x06\xD5\xEE\x70"
			  "\x16\xAE\x35\xC5\x52\x0F\x46\x1F"
			  "\x71\xF9\x5E\xF2\x67\xDC\x98\x2F"
			  "\xA3\x23\xAA\xD5\xD0\x49\xF4\xA6"
			  "\xF6\xB8\x32\xCD\xD6\x85\x73\x60"
			  "\x59\x20\xE7\x55\x0E\x91\xE2\x0C"
			  "\x3F\x1C\xEB\x3D\xDF\x52\x64\xF2"
			  "\x7D\x8B\x5D\x63\x16\xB9\xB2\x5D"
			  "\x5E\xAB\xB2\x97\xAB\x78\x44\xE7"
			  "\xC6\x72\x20\xC5\x90\x9B\xDC\x5D"
			  "\xB0\xEF\x44\xEF\x87\x31\x8D\xF4"
			  "\xFB\x81\x5D\xF7\x96\x96\xD4\x50"
			  "\x89\xA7\xF6\xB9\x67\x76\x40\x9E"
			  "\x9D\x40\xD5\x2C\x30\xB8\x01\x8F"
			  "\xE4\x7B\x71\x48\xA9\xA0\xA0\x1D"
			  "\x87\x52\xA4\x91\xA9\xD7\xA9\x51"
			  "\xD9\x59\xF7\xCC\x63\x22\xC1\x8D"
			  "\x84\x7B\xD8\x22\x32\x5C\x6F\x1D"
			  "\x6E\x9F\xFA\xDD\x49\x40\xDC\x37"
			  "\x14\x8C\xE1\x80\x1B\xDD\x36\x2A"
			  "\xD0\xE9\x54\x99\x5D\xBA\x3B\x11"
			  "\xD8\xFE\xC9\x5B\x5C\x25\xE5\x76"
			  "\xFB\xF2\x3F",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec aes_ctr_dec_tv_template[] = {
	{  
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x87\x4d\x61\x91\xb6\x20\xe3\x26"
			  "\x1b\xef\x68\x64\x99\x0d\xb6\xce"
			  "\x98\x06\xf6\x6b\x79\x70\xfd\xff"
			  "\x86\x17\x18\x7b\xb9\xff\xfd\xff"
			  "\x5a\xe4\xdf\x3e\xdb\xd5\xd3\x5e"
			  "\x5b\x4f\x09\x02\x0d\xb0\x3e\xab"
			  "\x1e\x03\x1d\xda\x2f\xbe\x03\xd1"
			  "\x79\x21\x70\xa0\xf3\x00\x9c\xee",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x1a\xbc\x93\x24\x17\x52\x1c\xa2"
			  "\x4f\x2b\x04\x59\xfe\x7e\x6e\x0b"
			  "\x09\x03\x39\xec\x0a\xa6\xfa\xef"
			  "\xd5\xcc\xc2\xc6\xf4\xce\x8e\x94"
			  "\x1e\x36\xb2\x6b\xd1\xeb\xc6\x70"
			  "\xd1\xbd\x1d\x66\x56\x20\xab\xf7"
			  "\x4f\x78\xa7\xf6\xd2\x98\x09\x58"
			  "\x5a\x97\xda\xec\x58\xc6\xb0\x50",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x60\x1e\xc3\x13\x77\x57\x89\xa5"
			  "\xb7\xa7\xf5\x04\xbb\xf3\xd2\x28"
			  "\xf4\x43\xe3\xca\x4d\x62\xb5\x9a"
			  "\xca\x84\xe9\x90\xca\xca\xf5\xc5"
			  "\x2b\x09\x30\xda\xa2\x3d\xe9\x4c"
			  "\xe8\x70\x17\xba\x2d\x84\x98\x8d"
			  "\xdf\xc9\xc5\x8d\xb6\x7a\xad\xa6"
			  "\x13\xc2\xdd\x08\x45\x79\x41\xa6",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x04\xF3\xD3\x88\x17\xEF\xDC\xEF"
			  "\x8B\x04\xF8\x3A\x66\x8D\x1A\x53"
			  "\x57\x1F\x4B\x23\xE4\xA0\xAF\xF9"
			  "\x69\x95\x35\x98\x8D\x4D\x8C\xC1"
			  "\xF0\xB2\x7F\x80\xBB\x54\x28\xA2"
			  "\x7A\x1B\x9F\x77\xEC\x0E\x6E\xDE"
			  "\xF0\xEC\xB8\xE4\x20\x62\xEE\xDB"
			  "\x5D\xF5\xDD\xE3\x54\xFC\xDD\xEB"
			  "\x6A\xEE\x65\xA1\x21\xD6\xD7\x81"
			  "\x47\x61\x12\x4D\xC2\x8C\xFA\x78"
			  "\x1F\x28\x02\x01\xC3\xFC\x1F\xEC"
			  "\x0F\x10\x4F\xB3\x12\x45\xC6\x3B"
			  "\x7E\x08\xF9\x5A\xD0\x5D\x73\x2D"
			  "\x58\xA4\xE5\xCB\x1C\xB4\xCE\x74"
			  "\x32\x41\x1F\x31\x9C\x08\xA2\x5D"
			  "\x67\xEB\x72\x1D\xF8\xE7\x70\x54"
			  "\x34\x4B\x31\x69\x84\x66\x96\x44"
			  "\x56\xCC\x1E\xD9\xE6\x13\x6A\xB9"
			  "\x2D\x0A\x05\x45\x2D\x90\xCC\xDF"
			  "\x16\x5C\x5F\x79\x34\x52\x54\xFE"
			  "\xFE\xCD\xAD\x04\x2E\xAD\x86\x06"
			  "\x1F\x37\xE8\x28\xBC\xD3\x8F\x5B"
			  "\x92\x66\x87\x3B\x8A\x0A\x1A\xCC"
			  "\x6E\xAB\x9F\x0B\xFA\x5C\xE6\xFD"
			  "\x3C\x98\x08\x12\xEC\xAA\x9E\x11"
			  "\xCA\xB2\x1F\xCE\x5E\x5B\xB2\x72"
			  "\x9C\xCC\x5D\xC5\xE0\x32\xC0\x56"
			  "\xD5\x45\x16\xD2\xAF\x13\x66\xF7"
			  "\x8C\x67\xAC\x79\xB2\xAF\x56\x27"
			  "\x3F\xCC\xFE\xCB\x1E\xC0\x75\xF1"
			  "\xA7\xC9\xC3\x1D\x8E\xDD\xF9\xD4"
			  "\x42\xC8\x21\x08\x16\xF7\x01\xD7"
			  "\xAC\x8E\x3F\x1D\x56\xC1\x06\xE4"
			  "\x9C\x62\xD6\xA5\x6A\x50\x44\xB3"
			  "\x35\x1C\x82\xB9\x10\xF9\x42\xA1"
			  "\xFC\x74\x9B\x44\x4F\x25\x02\xE3"
			  "\x08\xF5\xD4\x32\x39\x08\x11\xE8"
			  "\xD2\x6B\x50\x53\xD4\x08\xD1\x6B"
			  "\x3A\x4A\x68\x7B\x7C\xCD\x46\x5E"
			  "\x0D\x07\x19\xDB\x67\xD7\x98\x91"
			  "\xD7\x17\x10\x9B\x7B\x8A\x9B\x33"
			  "\xAE\xF3\x00\xA6\xD4\x15\xD9\xEA"
			  "\x85\x99\x22\xE8\x91\x38\x70\x83"
			  "\x93\x01\x24\x6C\xFA\x9A\xB9\x07"
			  "\xEA\x8D\x3B\xD9\x2A\x43\x59\x16"
			  "\x2F\x69\xEE\x84\x36\x44\x76\x98"
			  "\xF3\x04\x2A\x7C\x74\x3D\x29\x2B"
			  "\x0D\xAD\x8F\x44\x82\x9E\x57\x8D"
			  "\xAC\xED\x18\x1F\x50\xA4\xF5\x98"
			  "\x1F\xBD\x92\x91\x1B\x2D\xA6\xD6"
			  "\xD2\xE3\x02\xAA\x92\x3B\xC6\xB3"
			  "\x1B\x39\x72\xD5\x26\xCA\x04\xE0"
			  "\xFC\x58\x78\xBB\xB1\x3F\xA1\x9C"
			  "\x42\x24\x3E\x2E\x22\xBB\x4B\xBA"
			  "\xF4\x52\x0A\xE6\xAE\x47\xB4\x7D"
			  "\x1D\xA8\xBE\x81\x1A\x75\xDA\xAC"
			  "\xA6\x25\x1E\xEF\x3A\xC0\x6C\x63"
			  "\xEF\xDC\xC9\x79\x10\x26\xE8\x61"
			  "\x29\xFC\xA4\x05\xDF\x7D\x5C\x63"
			  "\x10\x09\x9B\x46\x9B\xF2\x2C\x2B"
			  "\xFA\x3A\x05\x4C\xFA\xD1\xFF\xFE"
			  "\xF1\x4C\xE5\xB2\x91\x64\x0C\x51",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	}, {  
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\xDA\x4E\x3F\xBC\xE8\xB6\x3A\xA2"
			  "\xD5\x4D\x84\x4A\xA9\x0C\xE1\xA5"
			  "\xB8\x73\xBC\xF9\xBB\x59\x2F\x44"
			  "\x8B\xAB\x82\x6C\xB4\x32\x9A\xDE"
			  "\x5A\x0B\xDB\x7A\x6B\xF2\x38\x9F"
			  "\x06\xF7\xF7\xFF\xFF\xC0\x8A\x2E"
			  "\x76\xEA\x06\x32\x23\xF3\x59\x2E"
			  "\x75\xDE\x71\x86\x3C\x98\x23\x44"
			  "\x5B\xF2\xFA\x6A\x00\xBB\xC1\xAD"
			  "\x58\xBD\x3E\x6F\x2E\xB4\x19\x04"
			  "\x70\x8B\x92\x55\x23\xE9\x6A\x3A"
			  "\x78\x7A\x1B\x10\x85\x52\x9C\x12"
			  "\xE4\x55\x81\x21\xCE\x53\xD0\x3B"
			  "\x63\x77\x2C\x74\xD1\xF5\x60\xF3"
			  "\xA1\xDE\x44\x3C\x8F\x4D\x2F\xDD"
			  "\x8A\xFE\x3C\x42\x8E\xD3\xF2\x8E"
			  "\xA8\x28\x69\x65\x31\xE1\x45\x83"
			  "\xE4\x49\xC4\x9C\xA7\x28\xAA\x21"
			  "\xCD\x5D\x0F\x15\xB7\x93\x07\x26"
			  "\xB0\x65\x6D\x91\x90\x23\x7A\xC6"
			  "\xDB\x68\xB0\xA1\x8E\xA4\x76\x4E"
			  "\xC6\x91\x83\x20\x92\x4D\x63\x7A"
			  "\x45\x18\x18\x74\x19\xAD\x71\x01"
			  "\x6B\x23\xAD\x9D\x4E\xE4\x6E\x46"
			  "\xC9\x73\x7A\xF9\x02\x95\xF4\x07"
			  "\x0E\x7A\xA6\xC5\xAE\xFA\x15\x2C"
			  "\x51\x71\xF1\xDC\x22\xB6\xAC\xD8"
			  "\x19\x24\x44\xBC\x0C\xFB\x3C\x2D"
			  "\xB1\x50\x47\x15\x0E\xDB\xB6\xD7"
			  "\xE8\x61\xE5\x95\x52\x1E\x3E\x49"
			  "\x70\xE9\x66\x04\x4C\xE1\xAF\xBD"
			  "\xDD\x15\x3B\x20\x59\x24\xFF\xB0"
			  "\x39\xAA\xE7\xBF\x23\xA3\x6E\xD5"
			  "\x15\xF0\x61\x4F\xAE\x89\x10\x58"
			  "\x5A\x33\x95\x52\x2A\xB5\x77\x9C"
			  "\xA5\x43\x80\x40\x27\x2D\xAE\xD9"
			  "\x3F\xE0\x80\x94\x78\x79\xCB\x7E"
			  "\xAD\x12\x44\x4C\xEC\x27\xB0\xEE"
			  "\x0B\x05\x2A\x82\x99\x58\xBB\x7A"
			  "\x8D\x6D\x9D\x8E\xE2\x8E\xE7\x93"
			  "\x2F\xB3\x09\x8D\x06\xD5\xEE\x70"
			  "\x16\xAE\x35\xC5\x52\x0F\x46\x1F"
			  "\x71\xF9\x5E\xF2\x67\xDC\x98\x2F"
			  "\xA3\x23\xAA\xD5\xD0\x49\xF4\xA6"
			  "\xF6\xB8\x32\xCD\xD6\x85\x73\x60"
			  "\x59\x20\xE7\x55\x0E\x91\xE2\x0C"
			  "\x3F\x1C\xEB\x3D\xDF\x52\x64\xF2"
			  "\x7D\x8B\x5D\x63\x16\xB9\xB2\x5D"
			  "\x5E\xAB\xB2\x97\xAB\x78\x44\xE7"
			  "\xC6\x72\x20\xC5\x90\x9B\xDC\x5D"
			  "\xB0\xEF\x44\xEF\x87\x31\x8D\xF4"
			  "\xFB\x81\x5D\xF7\x96\x96\xD4\x50"
			  "\x89\xA7\xF6\xB9\x67\x76\x40\x9E"
			  "\x9D\x40\xD5\x2C\x30\xB8\x01\x8F"
			  "\xE4\x7B\x71\x48\xA9\xA0\xA0\x1D"
			  "\x87\x52\xA4\x91\xA9\xD7\xA9\x51"
			  "\xD9\x59\xF7\xCC\x63\x22\xC1\x8D"
			  "\x84\x7B\xD8\x22\x32\x5C\x6F\x1D"
			  "\x6E\x9F\xFA\xDD\x49\x40\xDC\x37"
			  "\x14\x8C\xE1\x80\x1B\xDD\x36\x2A"
			  "\xD0\xE9\x54\x99\x5D\xBA\x3B\x11"
			  "\xD8\xFE\xC9\x5B\x5C\x25\xE5\x76"
			  "\xFB\xF2\x3F",
		.ilen	= 499,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12"
			  "\x7B\xE4\x4D",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec aes_ctr_rfc3686_enc_tv_template[] = {
	{  
		.key	= "\xae\x68\x52\xf8\x12\x10\x67\xcc"
			  "\x4b\xf7\xa5\x76\x55\x77\xf3\x9e"
			  "\x00\x00\x00\x30",
		.klen	= 20,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "Single block msg",
		.ilen	= 16,
		.result = "\xe4\x09\x5d\x4f\xb7\xa7\xb3\x79"
			  "\x2d\x61\x75\xa3\x26\x13\x11\xb8",
		.rlen	= 16,
	}, {
		.key	= "\x7e\x24\x06\x78\x17\xfa\xe0\xd7"
			  "\x43\xd6\xce\x1f\x32\x53\x91\x63"
			  "\x00\x6c\xb6\xdb",
		.klen	= 20,
		.iv	= "\xc0\x54\x3b\x59\xda\x48\xd9\x0b",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen	= 32,
		.result = "\x51\x04\xa1\x06\x16\x8a\x72\xd9"
			  "\x79\x0d\x41\xee\x8e\xda\xd3\x88"
			  "\xeb\x2e\x1e\xfc\x46\xda\x57\xc8"
			  "\xfc\xe6\x30\xdf\x91\x41\xbe\x28",
		.rlen	= 32,
	}, {
		.key	= "\x16\xaf\x5b\x14\x5f\xc9\xf5\x79"
			  "\xc1\x75\xf9\x3e\x3b\xfb\x0e\xed"
			  "\x86\x3d\x06\xcc\xfd\xb7\x85\x15"
			  "\x00\x00\x00\x48",
		.klen	= 28,
		.iv	= "\x36\x73\x3c\x14\x7d\x6d\x93\xcb",
		.input	= "Single block msg",
		.ilen	= 16,
		.result	= "\x4b\x55\x38\x4f\xe2\x59\xc9\xc8"
			  "\x4e\x79\x35\xa0\x03\xcb\xe9\x28",
		.rlen	= 16,
	}, {
		.key	= "\x7c\x5c\xb2\x40\x1b\x3d\xc3\x3c"
			  "\x19\xe7\x34\x08\x19\xe0\xf6\x9c"
			  "\x67\x8c\x3d\xb8\xe6\xf6\xa9\x1a"
			  "\x00\x96\xb0\x3b",
		.klen	= 28,
		.iv	= "\x02\x0c\x6e\xad\xc2\xcb\x50\x0d",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen	= 32,
		.result	= "\x45\x32\x43\xfc\x60\x9b\x23\x32"
			  "\x7e\xdf\xaa\xfa\x71\x31\xcd\x9f"
			  "\x84\x90\x70\x1c\x5a\xd4\xa7\x9c"
			  "\xfc\x1f\xe0\xff\x42\xf4\xfb\x00",
		.rlen	= 32,
	}, {
		.key	= "\x77\x6b\xef\xf2\x85\x1d\xb0\x6f"
			  "\x4c\x8a\x05\x42\xc8\x69\x6f\x6c"
			  "\x6a\x81\xaf\x1e\xec\x96\xb4\xd3"
			  "\x7f\xc1\xd6\x89\xe6\xc1\xc1\x04"
			  "\x00\x00\x00\x60",
		.klen	= 36,
		.iv	= "\xdb\x56\x72\xc9\x7a\xa8\xf0\xb2",
		.input	= "Single block msg",
		.ilen	= 16,
		.result = "\x14\x5a\xd0\x1d\xbf\x82\x4e\xc7"
			  "\x56\x08\x63\xdc\x71\xe3\xe0\xc0",
		.rlen	= 16,
	}, {
		.key	= "\xf6\xd6\x6d\x6b\xd5\x2d\x59\xbb"
			  "\x07\x96\x36\x58\x79\xef\xf8\x86"
			  "\xc6\x6d\xd5\x1a\x5b\x6a\x99\x74"
			  "\x4b\x50\x59\x0c\x87\xa2\x38\x84"
			  "\x00\xfa\xac\x24",
		.klen	= 36,
		.iv	= "\xc1\x58\x5e\xf1\x5a\x43\xd8\x75",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen	= 32,
		.result = "\xf0\x5e\x23\x1b\x38\x94\x61\x2c"
			  "\x49\xee\x00\x0b\x80\x4e\xb2\xa9"
			  "\xb8\x30\x6b\x50\x8f\x83\x9d\x6a"
			  "\x55\x30\x83\x1d\x93\x44\xaf\x1c",
		.rlen	= 32,
	}, {
	 
		.key = "\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17"
			"\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			"\x00\x00\x00\x00",
		.klen = 32 + 4,
		.iv = "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input =
			"\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17"
			"\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			"\x20\x21\x22\x23\x24\x25\x26\x27"
			"\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			"\x30\x31\x32\x33\x34\x35\x36\x37"
			"\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			"\x40\x41\x42\x43\x44\x45\x46\x47"
			"\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			"\x50\x51\x52\x53\x54\x55\x56\x57"
			"\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			"\x60\x61\x62\x63\x64\x65\x66\x67"
			"\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			"\x70\x71\x72\x73\x74\x75\x76\x77"
			"\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			"\x80\x81\x82\x83\x84\x85\x86\x87"
			"\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			"\x90\x91\x92\x93\x94\x95\x96\x97"
			"\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			"\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			"\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			"\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			"\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			"\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			"\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			"\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			"\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			"\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			"\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			"\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			"\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			"\x00\x03\x06\x09\x0c\x0f\x12\x15"
			"\x18\x1b\x1e\x21\x24\x27\x2a\x2d"
			"\x30\x33\x36\x39\x3c\x3f\x42\x45"
			"\x48\x4b\x4e\x51\x54\x57\x5a\x5d"
			"\x60\x63\x66\x69\x6c\x6f\x72\x75"
			"\x78\x7b\x7e\x81\x84\x87\x8a\x8d"
			"\x90\x93\x96\x99\x9c\x9f\xa2\xa5"
			"\xa8\xab\xae\xb1\xb4\xb7\xba\xbd"
			"\xc0\xc3\xc6\xc9\xcc\xcf\xd2\xd5"
			"\xd8\xdb\xde\xe1\xe4\xe7\xea\xed"
			"\xf0\xf3\xf6\xf9\xfc\xff\x02\x05"
			"\x08\x0b\x0e\x11\x14\x17\x1a\x1d"
			"\x20\x23\x26\x29\x2c\x2f\x32\x35"
			"\x38\x3b\x3e\x41\x44\x47\x4a\x4d"
			"\x50\x53\x56\x59\x5c\x5f\x62\x65"
			"\x68\x6b\x6e\x71\x74\x77\x7a\x7d"
			"\x80\x83\x86\x89\x8c\x8f\x92\x95"
			"\x98\x9b\x9e\xa1\xa4\xa7\xaa\xad"
			"\xb0\xb3\xb6\xb9\xbc\xbf\xc2\xc5"
			"\xc8\xcb\xce\xd1\xd4\xd7\xda\xdd"
			"\xe0\xe3\xe6\xe9\xec\xef\xf2\xf5"
			"\xf8\xfb\xfe\x01\x04\x07\x0a\x0d"
			"\x10\x13\x16\x19\x1c\x1f\x22\x25"
			"\x28\x2b\x2e\x31\x34\x37\x3a\x3d"
			"\x40\x43\x46\x49\x4c\x4f\x52\x55"
			"\x58\x5b\x5e\x61\x64\x67\x6a\x6d"
			"\x70\x73\x76\x79\x7c\x7f\x82\x85"
			"\x88\x8b\x8e\x91\x94\x97\x9a\x9d"
			"\xa0\xa3\xa6\xa9\xac\xaf\xb2\xb5"
			"\xb8\xbb\xbe\xc1\xc4\xc7\xca\xcd"
			"\xd0\xd3\xd6\xd9\xdc\xdf\xe2\xe5"
			"\xe8\xeb\xee\xf1\xf4\xf7\xfa\xfd"
			"\x00\x05\x0a\x0f\x14\x19\x1e\x23"
			"\x28\x2d\x32\x37\x3c\x41\x46\x4b"
			"\x50\x55\x5a\x5f\x64\x69\x6e\x73"
			"\x78\x7d\x82\x87\x8c\x91\x96\x9b"
			"\xa0\xa5\xaa\xaf\xb4\xb9\xbe\xc3"
			"\xc8\xcd\xd2\xd7\xdc\xe1\xe6\xeb"
			"\xf0\xf5\xfa\xff\x04\x09\x0e\x13"
			"\x18\x1d\x22\x27\x2c\x31\x36\x3b"
			"\x40\x45\x4a\x4f\x54\x59\x5e\x63"
			"\x68\x6d\x72\x77\x7c\x81\x86\x8b"
			"\x90\x95\x9a\x9f\xa4\xa9\xae\xb3"
			"\xb8\xbd\xc2\xc7\xcc\xd1\xd6\xdb"
			"\xe0\xe5\xea\xef\xf4\xf9\xfe\x03"
			"\x08\x0d\x12\x17\x1c\x21\x26\x2b"
			"\x30\x35\x3a\x3f\x44\x49\x4e\x53"
			"\x58\x5d\x62\x67\x6c\x71\x76\x7b"
			"\x80\x85\x8a\x8f\x94\x99\x9e\xa3"
			"\xa8\xad\xb2\xb7\xbc\xc1\xc6\xcb"
			"\xd0\xd5\xda\xdf\xe4\xe9\xee\xf3"
			"\xf8\xfd\x02\x07\x0c\x11\x16\x1b"
			"\x20\x25\x2a\x2f\x34\x39\x3e\x43"
			"\x48\x4d\x52\x57\x5c\x61\x66\x6b"
			"\x70\x75\x7a\x7f\x84\x89\x8e\x93"
			"\x98\x9d\xa2\xa7\xac\xb1\xb6\xbb"
			"\xc0\xc5\xca\xcf\xd4\xd9\xde\xe3"
			"\xe8\xed\xf2\xf7\xfc\x01\x06\x0b"
			"\x10\x15\x1a\x1f\x24\x29\x2e\x33"
			"\x38\x3d\x42\x47\x4c\x51\x56\x5b"
			"\x60\x65\x6a\x6f\x74\x79\x7e\x83"
			"\x88\x8d\x92\x97\x9c\xa1\xa6\xab"
			"\xb0\xb5\xba\xbf\xc4\xc9\xce\xd3"
			"\xd8\xdd\xe2\xe7\xec\xf1\xf6\xfb"
			"\x00\x07\x0e\x15\x1c\x23\x2a\x31"
			"\x38\x3f\x46\x4d\x54\x5b\x62\x69"
			"\x70\x77\x7e\x85\x8c\x93\x9a\xa1"
			"\xa8\xaf\xb6\xbd\xc4\xcb\xd2\xd9"
			"\xe0\xe7\xee\xf5\xfc\x03\x0a\x11"
			"\x18\x1f\x26\x2d\x34\x3b\x42\x49"
			"\x50\x57\x5e\x65\x6c\x73\x7a\x81"
			"\x88\x8f\x96\x9d\xa4\xab\xb2\xb9"
			"\xc0\xc7\xce\xd5\xdc\xe3\xea\xf1"
			"\xf8\xff\x06\x0d\x14\x1b\x22\x29"
			"\x30\x37\x3e\x45\x4c\x53\x5a\x61"
			"\x68\x6f\x76\x7d\x84\x8b\x92\x99"
			"\xa0\xa7\xae\xb5\xbc\xc3\xca\xd1"
			"\xd8\xdf\xe6\xed\xf4\xfb\x02\x09"
			"\x10\x17\x1e\x25\x2c\x33\x3a\x41"
			"\x48\x4f\x56\x5d\x64\x6b\x72\x79"
			"\x80\x87\x8e\x95\x9c\xa3\xaa\xb1"
			"\xb8\xbf\xc6\xcd\xd4\xdb\xe2\xe9"
			"\xf0\xf7\xfe\x05\x0c\x13\x1a\x21"
			"\x28\x2f\x36\x3d\x44\x4b\x52\x59"
			"\x60\x67\x6e\x75\x7c\x83\x8a\x91"
			"\x98\x9f\xa6\xad\xb4\xbb\xc2\xc9"
			"\xd0\xd7\xde\xe5\xec\xf3\xfa\x01"
			"\x08\x0f\x16\x1d\x24\x2b\x32\x39"
			"\x40\x47\x4e\x55\x5c\x63\x6a\x71"
			"\x78\x7f\x86\x8d\x94\x9b\xa2\xa9"
			"\xb0\xb7\xbe\xc5\xcc\xd3\xda\xe1"
			"\xe8\xef\xf6\xfd\x04\x0b\x12\x19"
			"\x20\x27\x2e\x35\x3c\x43\x4a\x51"
			"\x58\x5f\x66\x6d\x74\x7b\x82\x89"
			"\x90\x97\x9e\xa5\xac\xb3\xba\xc1"
			"\xc8\xcf\xd6\xdd\xe4\xeb\xf2\xf9"
			"\x00\x09\x12\x1b\x24\x2d\x36\x3f"
			"\x48\x51\x5a\x63\x6c\x75\x7e\x87"
			"\x90\x99\xa2\xab\xb4\xbd\xc6\xcf"
			"\xd8\xe1\xea\xf3\xfc\x05\x0e\x17"
			"\x20\x29\x32\x3b\x44\x4d\x56\x5f"
			"\x68\x71\x7a\x83\x8c\x95\x9e\xa7"
			"\xb0\xb9\xc2\xcb\xd4\xdd\xe6\xef"
			"\xf8\x01\x0a\x13\x1c\x25\x2e\x37"
			"\x40\x49\x52\x5b\x64\x6d\x76\x7f"
			"\x88\x91\x9a\xa3\xac\xb5\xbe\xc7"
			"\xd0\xd9\xe2\xeb\xf4\xfd\x06\x0f"
			"\x18\x21\x2a\x33\x3c\x45\x4e\x57"
			"\x60\x69\x72\x7b\x84\x8d\x96\x9f"
			"\xa8\xb1\xba\xc3\xcc\xd5\xde\xe7"
			"\xf0\xf9\x02\x0b\x14\x1d\x26\x2f"
			"\x38\x41\x4a\x53\x5c\x65\x6e\x77"
			"\x80\x89\x92\x9b\xa4\xad\xb6\xbf"
			"\xc8\xd1\xda\xe3\xec\xf5\xfe\x07"
			"\x10\x19\x22\x2b\x34\x3d\x46\x4f"
			"\x58\x61\x6a\x73\x7c\x85\x8e\x97"
			"\xa0\xa9\xb2\xbb\xc4\xcd\xd6\xdf"
			"\xe8\xf1\xfa\x03\x0c\x15\x1e\x27"
			"\x30\x39\x42\x4b\x54\x5d\x66\x6f"
			"\x78\x81\x8a\x93\x9c\xa5\xae\xb7"
			"\xc0\xc9\xd2\xdb\xe4\xed\xf6\xff"
			"\x08\x11\x1a\x23\x2c\x35\x3e\x47"
			"\x50\x59\x62\x6b\x74\x7d\x86\x8f"
			"\x98\xa1\xaa\xb3\xbc\xc5\xce\xd7"
			"\xe0\xe9\xf2\xfb\x04\x0d\x16\x1f"
			"\x28\x31\x3a\x43\x4c\x55\x5e\x67"
			"\x70\x79\x82\x8b\x94\x9d\xa6\xaf"
			"\xb8\xc1\xca\xd3\xdc\xe5\xee\xf7"
			"\x00\x0b\x16\x21\x2c\x37\x42\x4d"
			"\x58\x63\x6e\x79\x84\x8f\x9a\xa5"
			"\xb0\xbb\xc6\xd1\xdc\xe7\xf2\xfd"
			"\x08\x13\x1e\x29\x34\x3f\x4a\x55"
			"\x60\x6b\x76\x81\x8c\x97\xa2\xad"
			"\xb8\xc3\xce\xd9\xe4\xef\xfa\x05"
			"\x10\x1b\x26\x31\x3c\x47\x52\x5d"
			"\x68\x73\x7e\x89\x94\x9f\xaa\xb5"
			"\xc0\xcb\xd6\xe1\xec\xf7\x02\x0d"
			"\x18\x23\x2e\x39\x44\x4f\x5a\x65"
			"\x70\x7b\x86\x91\x9c\xa7\xb2\xbd"
			"\xc8\xd3\xde\xe9\xf4\xff\x0a\x15"
			"\x20\x2b\x36\x41\x4c\x57\x62\x6d"
			"\x78\x83\x8e\x99\xa4\xaf\xba\xc5"
			"\xd0\xdb\xe6\xf1\xfc\x07\x12\x1d"
			"\x28\x33\x3e\x49\x54\x5f\x6a\x75"
			"\x80\x8b\x96\xa1\xac\xb7\xc2\xcd"
			"\xd8\xe3\xee\xf9\x04\x0f\x1a\x25"
			"\x30\x3b\x46\x51\x5c\x67\x72\x7d"
			"\x88\x93\x9e\xa9\xb4\xbf\xca\xd5"
			"\xe0\xeb\xf6\x01\x0c\x17\x22\x2d"
			"\x38\x43\x4e\x59\x64\x6f\x7a\x85"
			"\x90\x9b\xa6\xb1\xbc\xc7\xd2\xdd"
			"\xe8\xf3\xfe\x09\x14\x1f\x2a\x35"
			"\x40\x4b\x56\x61\x6c\x77\x82\x8d"
			"\x98\xa3\xae\xb9\xc4\xcf\xda\xe5"
			"\xf0\xfb\x06\x11\x1c\x27\x32\x3d"
			"\x48\x53\x5e\x69\x74\x7f\x8a\x95"
			"\xa0\xab\xb6\xc1\xcc\xd7\xe2\xed"
			"\xf8\x03\x0e\x19\x24\x2f\x3a\x45"
			"\x50\x5b\x66\x71\x7c\x87\x92\x9d"
			"\xa8\xb3\xbe\xc9\xd4\xdf\xea\xf5"
			"\x00\x0d\x1a\x27\x34\x41\x4e\x5b"
			"\x68\x75\x82\x8f\x9c\xa9\xb6\xc3"
			"\xd0\xdd\xea\xf7\x04\x11\x1e\x2b"
			"\x38\x45\x52\x5f\x6c\x79\x86\x93"
			"\xa0\xad\xba\xc7\xd4\xe1\xee\xfb"
			"\x08\x15\x22\x2f\x3c\x49\x56\x63"
			"\x70\x7d\x8a\x97\xa4\xb1\xbe\xcb"
			"\xd8\xe5\xf2\xff\x0c\x19\x26\x33"
			"\x40\x4d\x5a\x67\x74\x81\x8e\x9b"
			"\xa8\xb5\xc2\xcf\xdc\xe9\xf6\x03"
			"\x10\x1d\x2a\x37\x44\x51\x5e\x6b"
			"\x78\x85\x92\x9f\xac\xb9\xc6\xd3"
			"\xe0\xed\xfa\x07\x14\x21\x2e\x3b"
			"\x48\x55\x62\x6f\x7c\x89\x96\xa3"
			"\xb0\xbd\xca\xd7\xe4\xf1\xfe\x0b"
			"\x18\x25\x32\x3f\x4c\x59\x66\x73"
			"\x80\x8d\x9a\xa7\xb4\xc1\xce\xdb"
			"\xe8\xf5\x02\x0f\x1c\x29\x36\x43"
			"\x50\x5d\x6a\x77\x84\x91\x9e\xab"
			"\xb8\xc5\xd2\xdf\xec\xf9\x06\x13"
			"\x20\x2d\x3a\x47\x54\x61\x6e\x7b"
			"\x88\x95\xa2\xaf\xbc\xc9\xd6\xe3"
			"\xf0\xfd\x0a\x17\x24\x31\x3e\x4b"
			"\x58\x65\x72\x7f\x8c\x99\xa6\xb3"
			"\xc0\xcd\xda\xe7\xf4\x01\x0e\x1b"
			"\x28\x35\x42\x4f\x5c\x69\x76\x83"
			"\x90\x9d\xaa\xb7\xc4\xd1\xde\xeb"
			"\xf8\x05\x12\x1f\x2c\x39\x46\x53"
			"\x60\x6d\x7a\x87\x94\xa1\xae\xbb"
			"\xc8\xd5\xe2\xef\xfc\x09\x16\x23"
			"\x30\x3d\x4a\x57\x64\x71\x7e\x8b"
			"\x98\xa5\xb2\xbf\xcc\xd9\xe6\xf3"
			"\x00\x0f\x1e\x2d\x3c\x4b\x5a\x69"
			"\x78\x87\x96\xa5\xb4\xc3\xd2\xe1"
			"\xf0\xff\x0e\x1d\x2c\x3b\x4a\x59"
			"\x68\x77\x86\x95\xa4\xb3\xc2\xd1"
			"\xe0\xef\xfe\x0d\x1c\x2b\x3a\x49"
			"\x58\x67\x76\x85\x94\xa3\xb2\xc1"
			"\xd0\xdf\xee\xfd\x0c\x1b\x2a\x39"
			"\x48\x57\x66\x75\x84\x93\xa2\xb1"
			"\xc0\xcf\xde\xed\xfc\x0b\x1a\x29"
			"\x38\x47\x56\x65\x74\x83\x92\xa1"
			"\xb0\xbf\xce\xdd\xec\xfb\x0a\x19"
			"\x28\x37\x46\x55\x64\x73\x82\x91"
			"\xa0\xaf\xbe\xcd\xdc\xeb\xfa\x09"
			"\x18\x27\x36\x45\x54\x63\x72\x81"
			"\x90\x9f\xae\xbd\xcc\xdb\xea\xf9"
			"\x08\x17\x26\x35\x44\x53\x62\x71"
			"\x80\x8f\x9e\xad\xbc\xcb\xda\xe9"
			"\xf8\x07\x16\x25\x34\x43\x52\x61"
			"\x70\x7f\x8e\x9d\xac\xbb\xca\xd9"
			"\xe8\xf7\x06\x15\x24\x33\x42\x51"
			"\x60\x6f\x7e\x8d\x9c\xab\xba\xc9"
			"\xd8\xe7\xf6\x05\x14\x23\x32\x41"
			"\x50\x5f\x6e\x7d\x8c\x9b\xaa\xb9"
			"\xc8\xd7\xe6\xf5\x04\x13\x22\x31"
			"\x40\x4f\x5e\x6d\x7c\x8b\x9a\xa9"
			"\xb8\xc7\xd6\xe5\xf4\x03\x12\x21"
			"\x30\x3f\x4e\x5d\x6c\x7b\x8a\x99"
			"\xa8\xb7\xc6\xd5\xe4\xf3\x02\x11"
			"\x20\x2f\x3e\x4d\x5c\x6b\x7a\x89"
			"\x98\xa7\xb6\xc5\xd4\xe3\xf2\x01"
			"\x10\x1f\x2e\x3d\x4c\x5b\x6a\x79"
			"\x88\x97\xa6\xb5\xc4\xd3\xe2\xf1"
			"\x00\x11\x22\x33\x44\x55\x66\x77"
			"\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
			"\x10\x21\x32\x43\x54\x65\x76\x87"
			"\x98\xa9\xba\xcb\xdc\xed\xfe\x0f"
			"\x20\x31\x42\x53\x64\x75\x86\x97"
			"\xa8\xb9\xca\xdb\xec\xfd\x0e\x1f"
			"\x30\x41\x52\x63\x74\x85\x96\xa7"
			"\xb8\xc9\xda\xeb\xfc\x0d\x1e\x2f"
			"\x40\x51\x62\x73\x84\x95\xa6\xb7"
			"\xc8\xd9\xea\xfb\x0c\x1d\x2e\x3f"
			"\x50\x61\x72\x83\x94\xa5\xb6\xc7"
			"\xd8\xe9\xfa\x0b\x1c\x2d\x3e\x4f"
			"\x60\x71\x82\x93\xa4\xb5\xc6\xd7"
			"\xe8\xf9\x0a\x1b\x2c\x3d\x4e\x5f"
			"\x70\x81\x92\xa3\xb4\xc5\xd6\xe7"
			"\xf8\x09\x1a\x2b\x3c\x4d\x5e\x6f"
			"\x80\x91\xa2\xb3\xc4\xd5\xe6\xf7"
			"\x08\x19\x2a\x3b\x4c\x5d\x6e\x7f"
			"\x90\xa1\xb2\xc3\xd4\xe5\xf6\x07"
			"\x18\x29\x3a\x4b\x5c\x6d\x7e\x8f"
			"\xa0\xb1\xc2\xd3\xe4\xf5\x06\x17"
			"\x28\x39\x4a\x5b\x6c\x7d\x8e\x9f"
			"\xb0\xc1\xd2\xe3\xf4\x05\x16\x27"
			"\x38\x49\x5a\x6b\x7c\x8d\x9e\xaf"
			"\xc0\xd1\xe2\xf3\x04\x15\x26\x37"
			"\x48\x59\x6a\x7b\x8c\x9d\xae\xbf"
			"\xd0\xe1\xf2\x03\x14\x25\x36\x47"
			"\x58\x69\x7a\x8b\x9c\xad\xbe\xcf"
			"\xe0\xf1\x02\x13\x24\x35\x46\x57"
			"\x68\x79\x8a\x9b\xac\xbd\xce\xdf"
			"\xf0\x01\x12\x23\x34\x45\x56\x67"
			"\x78\x89\x9a\xab\xbc\xcd\xde\xef"
			"\x00\x13\x26\x39\x4c\x5f\x72\x85"
			"\x98\xab\xbe\xd1\xe4\xf7\x0a\x1d"
			"\x30\x43\x56\x69\x7c\x8f\xa2\xb5"
			"\xc8\xdb\xee\x01\x14\x27\x3a\x4d"
			"\x60\x73\x86\x99\xac\xbf\xd2\xe5"
			"\xf8\x0b\x1e\x31\x44\x57\x6a\x7d"
			"\x90\xa3\xb6\xc9\xdc\xef\x02\x15"
			"\x28\x3b\x4e\x61\x74\x87\x9a\xad"
			"\xc0\xd3\xe6\xf9\x0c\x1f\x32\x45"
			"\x58\x6b\x7e\x91\xa4\xb7\xca\xdd"
			"\xf0\x03\x16\x29\x3c\x4f\x62\x75"
			"\x88\x9b\xae\xc1\xd4\xe7\xfa\x0d"
			"\x20\x33\x46\x59\x6c\x7f\x92\xa5"
			"\xb8\xcb\xde\xf1\x04\x17\x2a\x3d"
			"\x50\x63\x76\x89\x9c\xaf\xc2\xd5"
			"\xe8\xfb\x0e\x21\x34\x47\x5a\x6d"
			"\x80\x93\xa6\xb9\xcc\xdf\xf2\x05"
			"\x18\x2b\x3e\x51\x64\x77\x8a\x9d"
			"\xb0\xc3\xd6\xe9\xfc\x0f\x22\x35"
			"\x48\x5b\x6e\x81\x94\xa7\xba\xcd"
			"\xe0\xf3\x06\x19\x2c\x3f\x52\x65"
			"\x78\x8b\x9e\xb1\xc4\xd7\xea\xfd"
			"\x10\x23\x36\x49\x5c\x6f\x82\x95"
			"\xa8\xbb\xce\xe1\xf4\x07\x1a\x2d"
			"\x40\x53\x66\x79\x8c\x9f\xb2\xc5"
			"\xd8\xeb\xfe\x11\x24\x37\x4a\x5d"
			"\x70\x83\x96\xa9\xbc\xcf\xe2\xf5"
			"\x08\x1b\x2e\x41\x54\x67\x7a\x8d"
			"\xa0\xb3\xc6\xd9\xec\xff\x12\x25"
			"\x38\x4b\x5e\x71\x84\x97\xaa\xbd"
			"\xd0\xe3\xf6\x09\x1c\x2f\x42\x55"
			"\x68\x7b\x8e\xa1\xb4\xc7\xda\xed"
			"\x00\x15\x2a\x3f\x54\x69\x7e\x93"
			"\xa8\xbd\xd2\xe7\xfc\x11\x26\x3b"
			"\x50\x65\x7a\x8f\xa4\xb9\xce\xe3"
			"\xf8\x0d\x22\x37\x4c\x61\x76\x8b"
			"\xa0\xb5\xca\xdf\xf4\x09\x1e\x33"
			"\x48\x5d\x72\x87\x9c\xb1\xc6\xdb"
			"\xf0\x05\x1a\x2f\x44\x59\x6e\x83"
			"\x98\xad\xc2\xd7\xec\x01\x16\x2b"
			"\x40\x55\x6a\x7f\x94\xa9\xbe\xd3"
			"\xe8\xfd\x12\x27\x3c\x51\x66\x7b"
			"\x90\xa5\xba\xcf\xe4\xf9\x0e\x23"
			"\x38\x4d\x62\x77\x8c\xa1\xb6\xcb"
			"\xe0\xf5\x0a\x1f\x34\x49\x5e\x73"
			"\x88\x9d\xb2\xc7\xdc\xf1\x06\x1b"
			"\x30\x45\x5a\x6f\x84\x99\xae\xc3"
			"\xd8\xed\x02\x17\x2c\x41\x56\x6b"
			"\x80\x95\xaa\xbf\xd4\xe9\xfe\x13"
			"\x28\x3d\x52\x67\x7c\x91\xa6\xbb"
			"\xd0\xe5\xfa\x0f\x24\x39\x4e\x63"
			"\x78\x8d\xa2\xb7\xcc\xe1\xf6\x0b"
			"\x20\x35\x4a\x5f\x74\x89\x9e\xb3"
			"\xc8\xdd\xf2\x07\x1c\x31\x46\x5b"
			"\x70\x85\x9a\xaf\xc4\xd9\xee\x03"
			"\x18\x2d\x42\x57\x6c\x81\x96\xab"
			"\xc0\xd5\xea\xff\x14\x29\x3e\x53"
			"\x68\x7d\x92\xa7\xbc\xd1\xe6\xfb"
			"\x10\x25\x3a\x4f\x64\x79\x8e\xa3"
			"\xb8\xcd\xe2\xf7\x0c\x21\x36\x4b"
			"\x60\x75\x8a\x9f\xb4\xc9\xde\xf3"
			"\x08\x1d\x32\x47\x5c\x71\x86\x9b"
			"\xb0\xc5\xda\xef\x04\x19\x2e\x43"
			"\x58\x6d\x82\x97\xac\xc1\xd6\xeb"
			"\x00\x17\x2e\x45\x5c\x73\x8a\xa1"
			"\xb8\xcf\xe6\xfd\x14\x2b\x42\x59"
			"\x70\x87\x9e\xb5\xcc\xe3\xfa\x11"
			"\x28\x3f\x56\x6d\x84\x9b\xb2\xc9"
			"\xe0\xf7\x0e\x25\x3c\x53\x6a\x81"
			"\x98\xaf\xc6\xdd\xf4\x0b\x22\x39"
			"\x50\x67\x7e\x95\xac\xc3\xda\xf1"
			"\x08\x1f\x36\x4d\x64\x7b\x92\xa9"
			"\xc0\xd7\xee\x05\x1c\x33\x4a\x61"
			"\x78\x8f\xa6\xbd\xd4\xeb\x02\x19"
			"\x30\x47\x5e\x75\x8c\xa3\xba\xd1"
			"\xe8\xff\x16\x2d\x44\x5b\x72\x89"
			"\xa0\xb7\xce\xe5\xfc\x13\x2a\x41"
			"\x58\x6f\x86\x9d\xb4\xcb\xe2\xf9"
			"\x10\x27\x3e\x55\x6c\x83\x9a\xb1"
			"\xc8\xdf\xf6\x0d\x24\x3b\x52\x69"
			"\x80\x97\xae\xc5\xdc\xf3\x0a\x21"
			"\x38\x4f\x66\x7d\x94\xab\xc2\xd9"
			"\xf0\x07\x1e\x35\x4c\x63\x7a\x91"
			"\xa8\xbf\xd6\xed\x04\x1b\x32\x49"
			"\x60\x77\x8e\xa5\xbc\xd3\xea\x01"
			"\x18\x2f\x46\x5d\x74\x8b\xa2\xb9"
			"\xd0\xe7\xfe\x15\x2c\x43\x5a\x71"
			"\x88\x9f\xb6\xcd\xe4\xfb\x12\x29"
			"\x40\x57\x6e\x85\x9c\xb3\xca\xe1"
			"\xf8\x0f\x26\x3d\x54\x6b\x82\x99"
			"\xb0\xc7\xde\xf5\x0c\x23\x3a\x51"
			"\x68\x7f\x96\xad\xc4\xdb\xf2\x09"
			"\x20\x37\x4e\x65\x7c\x93\xaa\xc1"
			"\xd8\xef\x06\x1d\x34\x4b\x62\x79"
			"\x90\xa7\xbe\xd5\xec\x03\x1a\x31"
			"\x48\x5f\x76\x8d\xa4\xbb\xd2\xe9"
			"\x00\x19\x32\x4b\x64\x7d\x96\xaf"
			"\xc8\xe1\xfa\x13\x2c\x45\x5e\x77"
			"\x90\xa9\xc2\xdb\xf4\x0d\x26\x3f"
			"\x58\x71\x8a\xa3\xbc\xd5\xee\x07"
			"\x20\x39\x52\x6b\x84\x9d\xb6\xcf"
			"\xe8\x01\x1a\x33\x4c\x65\x7e\x97"
			"\xb0\xc9\xe2\xfb\x14\x2d\x46\x5f"
			"\x78\x91\xaa\xc3\xdc\xf5\x0e\x27"
			"\x40\x59\x72\x8b\xa4\xbd\xd6\xef"
			"\x08\x21\x3a\x53\x6c\x85\x9e\xb7"
			"\xd0\xe9\x02\x1b\x34\x4d\x66\x7f"
			"\x98\xb1\xca\xe3\xfc\x15\x2e\x47"
			"\x60\x79\x92\xab\xc4\xdd\xf6\x0f"
			"\x28\x41\x5a\x73\x8c\xa5\xbe\xd7"
			"\xf0\x09\x22\x3b\x54\x6d\x86\x9f"
			"\xb8\xd1\xea\x03\x1c\x35\x4e\x67"
			"\x80\x99\xb2\xcb\xe4\xfd\x16\x2f"
			"\x48\x61\x7a\x93\xac\xc5\xde\xf7"
			"\x10\x29\x42\x5b\x74\x8d\xa6\xbf"
			"\xd8\xf1\x0a\x23\x3c\x55\x6e\x87"
			"\xa0\xb9\xd2\xeb\x04\x1d\x36\x4f"
			"\x68\x81\x9a\xb3\xcc\xe5\xfe\x17"
			"\x30\x49\x62\x7b\x94\xad\xc6\xdf"
			"\xf8\x11\x2a\x43\x5c\x75\x8e\xa7"
			"\xc0\xd9\xf2\x0b\x24\x3d\x56\x6f"
			"\x88\xa1\xba\xd3\xec\x05\x1e\x37"
			"\x50\x69\x82\x9b\xb4\xcd\xe6\xff"
			"\x18\x31\x4a\x63\x7c\x95\xae\xc7"
			"\xe0\xf9\x12\x2b\x44\x5d\x76\x8f"
			"\xa8\xc1\xda\xf3\x0c\x25\x3e\x57"
			"\x70\x89\xa2\xbb\xd4\xed\x06\x1f"
			"\x38\x51\x6a\x83\x9c\xb5\xce\xe7"
			"\x00\x1b\x36\x51\x6c\x87\xa2\xbd"
			"\xd8\xf3\x0e\x29\x44\x5f\x7a\x95"
			"\xb0\xcb\xe6\x01\x1c\x37\x52\x6d"
			"\x88\xa3\xbe\xd9\xf4\x0f\x2a\x45"
			"\x60\x7b\x96\xb1\xcc\xe7\x02\x1d"
			"\x38\x53\x6e\x89\xa4\xbf\xda\xf5"
			"\x10\x2b\x46\x61\x7c\x97\xb2\xcd"
			"\xe8\x03\x1e\x39\x54\x6f\x8a\xa5"
			"\xc0\xdb\xf6\x11\x2c\x47\x62\x7d"
			"\x98\xb3\xce\xe9\x04\x1f\x3a\x55"
			"\x70\x8b\xa6\xc1\xdc\xf7\x12\x2d"
			"\x48\x63\x7e\x99\xb4\xcf\xea\x05"
			"\x20\x3b\x56\x71\x8c\xa7\xc2\xdd"
			"\xf8\x13\x2e\x49\x64\x7f\x9a\xb5"
			"\xd0\xeb\x06\x21\x3c\x57\x72\x8d"
			"\xa8\xc3\xde\xf9\x14\x2f\x4a\x65"
			"\x80\x9b\xb6\xd1\xec\x07\x22\x3d"
			"\x58\x73\x8e\xa9\xc4\xdf\xfa\x15"
			"\x30\x4b\x66\x81\x9c\xb7\xd2\xed"
			"\x08\x23\x3e\x59\x74\x8f\xaa\xc5"
			"\xe0\xfb\x16\x31\x4c\x67\x82\x9d"
			"\xb8\xd3\xee\x09\x24\x3f\x5a\x75"
			"\x90\xab\xc6\xe1\xfc\x17\x32\x4d"
			"\x68\x83\x9e\xb9\xd4\xef\x0a\x25"
			"\x40\x5b\x76\x91\xac\xc7\xe2\xfd"
			"\x18\x33\x4e\x69\x84\x9f\xba\xd5"
			"\xf0\x0b\x26\x41\x5c\x77\x92\xad"
			"\xc8\xe3\xfe\x19\x34\x4f\x6a\x85"
			"\xa0\xbb\xd6\xf1\x0c\x27\x42\x5d"
			"\x78\x93\xae\xc9\xe4\xff\x1a\x35"
			"\x50\x6b\x86\xa1\xbc\xd7\xf2\x0d"
			"\x28\x43\x5e\x79\x94\xaf\xca\xe5"
			"\x00\x1d\x3a\x57\x74\x91\xae\xcb"
			"\xe8\x05\x22\x3f\x5c\x79\x96\xb3"
			"\xd0\xed\x0a\x27\x44\x61\x7e\x9b"
			"\xb8\xd5\xf2\x0f\x2c\x49\x66\x83"
			"\xa0\xbd\xda\xf7\x14\x31\x4e\x6b"
			"\x88\xa5\xc2\xdf\xfc\x19\x36\x53"
			"\x70\x8d\xaa\xc7\xe4\x01\x1e\x3b"
			"\x58\x75\x92\xaf\xcc\xe9\x06\x23"
			"\x40\x5d\x7a\x97\xb4\xd1\xee\x0b"
			"\x28\x45\x62\x7f\x9c\xb9\xd6\xf3"
			"\x10\x2d\x4a\x67\x84\xa1\xbe\xdb"
			"\xf8\x15\x32\x4f\x6c\x89\xa6\xc3"
			"\xe0\xfd\x1a\x37\x54\x71\x8e\xab"
			"\xc8\xe5\x02\x1f\x3c\x59\x76\x93"
			"\xb0\xcd\xea\x07\x24\x41\x5e\x7b"
			"\x98\xb5\xd2\xef\x0c\x29\x46\x63"
			"\x80\x9d\xba\xd7\xf4\x11\x2e\x4b"
			"\x68\x85\xa2\xbf\xdc\xf9\x16\x33"
			"\x50\x6d\x8a\xa7\xc4\xe1\xfe\x1b"
			"\x38\x55\x72\x8f\xac\xc9\xe6\x03"
			"\x20\x3d\x5a\x77\x94\xb1\xce\xeb"
			"\x08\x25\x42\x5f\x7c\x99\xb6\xd3"
			"\xf0\x0d\x2a\x47\x64\x81\x9e\xbb"
			"\xd8\xf5\x12\x2f\x4c\x69\x86\xa3"
			"\xc0\xdd\xfa\x17\x34\x51\x6e\x8b"
			"\xa8\xc5\xe2\xff\x1c\x39\x56\x73"
			"\x90\xad\xca\xe7\x04\x21\x3e\x5b"
			"\x78\x95\xb2\xcf\xec\x09\x26\x43"
			"\x60\x7d\x9a\xb7\xd4\xf1\x0e\x2b"
			"\x48\x65\x82\x9f\xbc\xd9\xf6\x13"
			"\x30\x4d\x6a\x87\xa4\xc1\xde\xfb"
			"\x18\x35\x52\x6f\x8c\xa9\xc6\xe3"
			"\x00\x1f\x3e\x5d\x7c\x9b\xba\xd9"
			"\xf8\x17\x36\x55\x74\x93\xb2\xd1"
			"\xf0\x0f\x2e\x4d\x6c\x8b\xaa\xc9"
			"\xe8\x07\x26\x45\x64\x83\xa2\xc1"
			"\xe0\xff\x1e\x3d\x5c\x7b\x9a\xb9"
			"\xd8\xf7\x16\x35\x54\x73\x92\xb1"
			"\xd0\xef\x0e\x2d\x4c\x6b\x8a\xa9"
			"\xc8\xe7\x06\x25\x44\x63\x82\xa1"
			"\xc0\xdf\xfe\x1d\x3c\x5b\x7a\x99"
			"\xb8\xd7\xf6\x15\x34\x53\x72\x91"
			"\xb0\xcf\xee\x0d\x2c\x4b\x6a\x89"
			"\xa8\xc7\xe6\x05\x24\x43\x62\x81"
			"\xa0\xbf\xde\xfd\x1c\x3b\x5a\x79"
			"\x98\xb7\xd6\xf5\x14\x33\x52\x71"
			"\x90\xaf\xce\xed\x0c\x2b\x4a\x69"
			"\x88\xa7\xc6\xe5\x04\x23\x42\x61"
			"\x80\x9f\xbe\xdd\xfc\x1b\x3a\x59"
			"\x78\x97\xb6\xd5\xf4\x13\x32\x51"
			"\x70\x8f\xae\xcd\xec\x0b\x2a\x49"
			"\x68\x87\xa6\xc5\xe4\x03\x22\x41"
			"\x60\x7f\x9e\xbd\xdc\xfb\x1a\x39"
			"\x58\x77\x96\xb5\xd4\xf3\x12\x31"
			"\x50\x6f\x8e\xad\xcc\xeb\x0a\x29"
			"\x48\x67\x86\xa5\xc4\xe3\x02\x21"
			"\x40\x5f\x7e\x9d\xbc\xdb\xfa\x19"
			"\x38\x57\x76\x95\xb4\xd3\xf2\x11"
			"\x30\x4f\x6e\x8d\xac\xcb\xea\x09"
			"\x28\x47\x66\x85\xa4\xc3\xe2\x01"
			"\x20\x3f\x5e\x7d\x9c\xbb\xda\xf9"
			"\x18\x37\x56\x75\x94\xb3\xd2\xf1"
			"\x10\x2f\x4e\x6d\x8c\xab\xca\xe9"
			"\x08\x27\x46\x65\x84\xa3\xc2\xe1"
			"\x00\x21\x42\x63",
		.ilen = 4100,
		.result =
			"\xf0\x5c\x74\xad\x4e\xbc\x99\xe2"
			"\xae\xff\x91\x3a\x44\xcf\x38\x32"
			"\x1e\xad\xa7\xcd\xa1\x39\x95\xaa"
			"\x10\xb1\xb3\x2e\x04\x31\x8f\x86"
			"\xf2\x62\x74\x70\x0c\xa4\x46\x08"
			"\xa8\xb7\x99\xa8\xe9\xd2\x73\x79"
			"\x7e\x6e\xd4\x8f\x1e\xc7\x8e\x31"
			"\x0b\xfa\x4b\xce\xfd\xf3\x57\x71"
			"\xe9\x46\x03\xa5\x3d\x34\x00\xe2"
			"\x18\xff\x75\x6d\x06\x2d\x00\xab"
			"\xb9\x3e\x6c\x59\xc5\x84\x06\xb5"
			"\x8b\xd0\x89\x9c\x4a\x79\x16\xc6"
			"\x3d\x74\x54\xfa\x44\xcd\x23\x26"
			"\x5c\xcf\x7e\x28\x92\x32\xbf\xdf"
			"\xa7\x20\x3c\x74\x58\x2a\x9a\xde"
			"\x61\x00\x1c\x4f\xff\x59\xc4\x22"
			"\xac\x3c\xd0\xe8\x6c\xf9\x97\x1b"
			"\x58\x9b\xad\x71\xe8\xa9\xb5\x0d"
			"\xee\x2f\x04\x1f\x7f\xbc\x99\xee"
			"\x84\xff\x42\x60\xdc\x3a\x18\xa5"
			"\x81\xf9\xef\xdc\x7a\x0f\x65\x41"
			"\x2f\xa3\xd3\xf9\xc2\xcb\xc0\x4d"
			"\x8f\xd3\x76\x96\xad\x49\x6d\x38"
			"\x3d\x39\x0b\x6c\x80\xb7\x54\x69"
			"\xf0\x2c\x90\x02\x29\x0d\x1c\x12"
			"\xad\x55\xc3\x8b\x68\xd9\xcc\xb3"
			"\xb2\x64\x33\x90\x5e\xca\x4b\xe2"
			"\xfb\x75\xdc\x63\xf7\x9f\x82\x74"
			"\xf0\xc9\xaa\x7f\xe9\x2a\x9b\x33"
			"\xbc\x88\x00\x7f\xca\xb2\x1f\x14"
			"\xdb\xc5\x8e\x7b\x11\x3c\x3e\x08"
			"\xf3\x83\xe8\xe0\x94\x86\x2e\x92"
			"\x78\x6b\x01\xc9\xc7\x83\xba\x21"
			"\x6a\x25\x15\x33\x4e\x45\x08\xec"
			"\x35\xdb\xe0\x6e\x31\x51\x79\xa9"
			"\x42\x44\x65\xc1\xa0\xf1\xf9\x2a"
			"\x70\xd5\xb6\xc6\xc1\x8c\x39\xfc"
			"\x25\xa6\x55\xd9\xdd\x2d\x4c\xec"
			"\x49\xc6\xeb\x0e\xa8\x25\x2a\x16"
			"\x1b\x66\x84\xda\xe2\x92\xe5\xc0"
			"\xc8\x53\x07\xaf\x80\x84\xec\xfd"
			"\xcd\xd1\x6e\xcd\x6f\x6a\xf5\x36"
			"\xc5\x15\xe5\x25\x7d\x77\xd1\x1a"
			"\x93\x36\xa9\xcf\x7c\xa4\x54\x4a"
			"\x06\x51\x48\x4e\xf6\x59\x87\xd2"
			"\x04\x02\xef\xd3\x44\xde\x76\x31"
			"\xb3\x34\x17\x1b\x9d\x66\x11\x9f"
			"\x1e\xcc\x17\xe9\xc7\x3c\x1b\xe7"
			"\xcb\x50\x08\xfc\xdc\x2b\x24\xdb"
			"\x65\x83\xd0\x3b\xe3\x30\xea\x94"
			"\x6c\xe7\xe8\x35\x32\xc7\xdb\x64"
			"\xb4\x01\xab\x36\x2c\x77\x13\xaf"
			"\xf8\x2b\x88\x3f\x54\x39\xc4\x44"
			"\xfe\xef\x6f\x68\x34\xbe\x0f\x05"
			"\x16\x6d\xf6\x0a\x30\xe7\xe3\xed"
			"\xc4\xde\x3c\x1b\x13\xd8\xdb\xfe"
			"\x41\x62\xe5\x28\xd4\x8d\xa3\xc7"
			"\x93\x97\xc6\x48\x45\x1d\x9f\x83"
			"\xdf\x4b\x40\x3e\x42\x25\x87\x80"
			"\x4c\x7d\xa8\xd4\x98\x23\x95\x75"
			"\x41\x8c\xda\x41\x9b\xd4\xa7\x06"
			"\xb5\xf1\x71\x09\x53\xbe\xca\xbf"
			"\x32\x03\xed\xf0\x50\x1c\x56\x39"
			"\x5b\xa4\x75\x18\xf7\x9b\x58\xef"
			"\x53\xfc\x2a\x38\x23\x15\x75\xcd"
			"\x45\xe5\x5a\x82\x55\xba\x21\xfa"
			"\xd4\xbd\xc6\x94\x7c\xc5\x80\x12"
			"\xf7\x4b\x32\xc4\x9a\x82\xd8\x28"
			"\x8f\xd9\xc2\x0f\x60\x03\xbe\x5e"
			"\x21\xd6\x5f\x58\xbf\x5c\xb1\x32"
			"\x82\x8d\xa9\xe5\xf2\x66\x1a\xc0"
			"\xa0\xbc\x58\x2f\x71\xf5\x2f\xed"
			"\xd1\x26\xb9\xd8\x49\x5a\x07\x19"
			"\x01\x7c\x59\xb0\xf8\xa4\xb7\xd3"
			"\x7b\x1a\x8c\x38\xf4\x50\xa4\x59"
			"\xb0\xcc\x41\x0b\x88\x7f\xe5\x31"
			"\xb3\x42\xba\xa2\x7e\xd4\x32\x71"
			"\x45\x87\x48\xa9\xc2\xf2\x89\xb3"
			"\xe4\xa7\x7e\x52\x15\x61\xfa\xfe"
			"\xc9\xdd\x81\xeb\x13\xab\xab\xc3"
			"\x98\x59\xd8\x16\x3d\x14\x7a\x1c"
			"\x3c\x41\x9a\x16\x16\x9b\xd2\xd2"
			"\x69\x3a\x29\x23\xac\x86\x32\xa5"
			"\x48\x9c\x9e\xf3\x47\x77\x81\x70"
			"\x24\xe8\x85\xd2\xf5\xb5\xfa\xff"
			"\x59\x6a\xd3\x50\x59\x43\x59\xde"
			"\xd9\xf1\x55\xa5\x0c\xc3\x1a\x1a"
			"\x18\x34\x0d\x1a\x63\x33\xed\x10"
			"\xe0\x1d\x2a\x18\xd2\xc0\x54\xa8"
			"\xca\xb5\x9a\xd3\xdd\xca\x45\x84"
			"\x50\xe7\x0f\xfe\xa4\x99\x5a\xbe"
			"\x43\x2d\x9a\xcb\x92\x3f\x5a\x1d"
			"\x85\xd8\xc9\xdf\x68\xc9\x12\x80"
			"\x56\x0c\xdc\x00\xdc\x3a\x7d\x9d"
			"\xa3\xa2\xe8\x4d\xbf\xf9\x70\xa0"
			"\xa4\x13\x4f\x6b\xaf\x0a\x89\x7f"
			"\xda\xf0\xbf\x9b\xc8\x1d\xe5\xf8"
			"\x2e\x8b\x07\xb5\x73\x1b\xcc\xa2"
			"\xa6\xad\x30\xbc\x78\x3c\x5b\x10"
			"\xfa\x5e\x62\x2d\x9e\x64\xb3\x33"
			"\xce\xf9\x1f\x86\xe7\x8b\xa2\xb8"
			"\xe8\x99\x57\x8c\x11\xed\x66\xd9"
			"\x3c\x72\xb9\xc3\xe6\x4e\x17\x3a"
			"\x6a\xcb\x42\x24\x06\xed\x3e\x4e"
			"\xa3\xe8\x6a\x94\xda\x0d\x4e\xd5"
			"\x14\x19\xcf\xb6\x26\xd8\x2e\xcc"
			"\x64\x76\x38\x49\x4d\xfe\x30\x6d"
			"\xe4\xc8\x8c\x7b\xc4\xe0\x35\xba"
			"\x22\x6e\x76\xe1\x1a\xf2\x53\xc3"
			"\x28\xa2\x82\x1f\x61\x69\xad\xc1"
			"\x7b\x28\x4b\x1e\x6c\x85\x95\x9b"
			"\x51\xb5\x17\x7f\x12\x69\x8c\x24"
			"\xd5\xc7\x5a\x5a\x11\x54\xff\x5a"
			"\xf7\x16\xc3\x91\xa6\xf0\xdc\x0a"
			"\xb6\xa7\x4a\x0d\x7a\x58\xfe\xa5"
			"\xf5\xcb\x8f\x7b\x0e\xea\x57\xe7"
			"\xbd\x79\xd6\x1c\x88\x23\x6c\xf2"
			"\x4d\x29\x77\x53\x35\x6a\x00\x8d"
			"\xcd\xa3\x58\xbe\x77\x99\x18\xf8"
			"\xe6\xe1\x8f\xe9\x37\x8f\xe3\xe2"
			"\x5a\x8a\x93\x25\xaf\xf3\x78\x80"
			"\xbe\xa6\x1b\xc6\xac\x8b\x1c\x91"
			"\x58\xe1\x9f\x89\x35\x9d\x1d\x21"
			"\x29\x9f\xf4\x99\x02\x27\x0f\xa8"
			"\x4f\x79\x94\x2b\x33\x2c\xda\xa2"
			"\x26\x39\x83\x94\xef\x27\xd8\x53"
			"\x8f\x66\x0d\xe4\x41\x7d\x34\xcd"
			"\x43\x7c\x95\x0a\x53\xef\x66\xda"
			"\x7e\x9b\xf3\x93\xaf\xd0\x73\x71"
			"\xba\x40\x9b\x74\xf8\xd7\xd7\x41"
			"\x6d\xaf\x72\x9c\x8d\x21\x87\x3c"
			"\xfd\x0a\x90\xa9\x47\x96\x9e\xd3"
			"\x88\xee\x73\xcf\x66\x2f\x52\x56"
			"\x6d\xa9\x80\x4c\xe2\x6f\x62\x88"
			"\x3f\x0e\x54\x17\x48\x80\x5d\xd3"
			"\xc3\xda\x25\x3d\xa1\xc8\xcb\x9f"
			"\x9b\x70\xb3\xa1\xeb\x04\x52\xa1"
			"\xf2\x22\x0f\xfc\xc8\x18\xfa\xf9"
			"\x85\x9c\xf1\xac\xeb\x0c\x02\x46"
			"\x75\xd2\xf5\x2c\xe3\xd2\x59\x94"
			"\x12\xf3\x3c\xfc\xd7\x92\xfa\x36"
			"\xba\x61\x34\x38\x7c\xda\x48\x3e"
			"\x08\xc9\x39\x23\x5e\x02\x2c\x1a"
			"\x18\x7e\xb4\xd9\xfd\x9e\x40\x02"
			"\xb1\x33\x37\x32\xe7\xde\xd6\xd0"
			"\x7c\x58\x65\x4b\xf8\x34\x27\x9c"
			"\x44\xb4\xbd\xe9\xe9\x4c\x78\x7d"
			"\x4b\x9f\xce\xb1\xcd\x47\xa5\x37"
			"\xe5\x6d\xbd\xb9\x43\x94\x0a\xd4"
			"\xd6\xf9\x04\x5f\xb5\x66\x6c\x1a"
			"\x35\x12\xe3\x36\x28\x27\x36\x58"
			"\x01\x2b\x79\xe4\xba\x6d\x10\x7d"
			"\x65\xdf\x84\x95\xf4\xd5\xb6\x8f"
			"\x2b\x9f\x96\x00\x86\x60\xf0\x21"
			"\x76\xa8\x6a\x8c\x28\x1c\xb3\x6b"
			"\x97\xd7\xb6\x53\x2a\xcc\xab\x40"
			"\x9d\x62\x79\x58\x52\xe6\x65\xb7"
			"\xab\x55\x67\x9c\x89\x7c\x03\xb0"
			"\x73\x59\xc5\x81\xf5\x18\x17\x5c"
			"\x89\xf3\x78\x35\x44\x62\x78\x72"
			"\xd0\x96\xeb\x31\xe7\x87\x77\x14"
			"\x99\x51\xf2\x59\x26\x9e\xb5\xa6"
			"\x45\xfe\x6e\xbd\x07\x4c\x94\x5a"
			"\xa5\x7d\xfc\xf1\x2b\x77\xe2\xfe"
			"\x17\xd4\x84\xa0\xac\xb5\xc7\xda"
			"\xa9\x1a\xb6\xf3\x74\x11\xb4\x9d"
			"\xfb\x79\x2e\x04\x2d\x50\x28\x83"
			"\xbf\xc6\x52\xd3\x34\xd6\xe8\x7a"
			"\xb6\xea\xe7\xa8\x6c\x15\x1e\x2c"
			"\x57\xbc\x48\x4e\x5f\x5c\xb6\x92"
			"\xd2\x49\x77\x81\x6d\x90\x70\xae"
			"\x98\xa1\x03\x0d\x6b\xb9\x77\x14"
			"\xf1\x4e\x23\xd3\xf8\x68\xbd\xc2"
			"\xfe\x04\xb7\x5c\xc5\x17\x60\x8f"
			"\x65\x54\xa4\x7a\x42\xdc\x18\x0d"
			"\xb5\xcf\x0f\xd3\xc7\x91\x66\x1b"
			"\x45\x42\x27\x75\x50\xe5\xee\xb8"
			"\x7f\x33\x2c\xba\x4a\x92\x4d\x2c"
			"\x3c\xe3\x0d\x80\x01\xba\x0d\x29"
			"\xd8\x3c\xe9\x13\x16\x57\xe6\xea"
			"\x94\x52\xe7\x00\x4d\x30\xb0\x0f"
			"\x35\xb8\xb8\xa7\xb1\xb5\x3b\x44"
			"\xe1\x2f\xfd\x88\xed\x43\xe7\x52"
			"\x10\x93\xb3\x8a\x30\x6b\x0a\xf7"
			"\x23\xc6\x50\x9d\x4a\xb0\xde\xc3"
			"\xdc\x9b\x2f\x01\x56\x36\x09\xc5"
			"\x2f\x6b\xfe\xf1\xd8\x27\x45\x03"
			"\x30\x5e\x5c\x5b\xb4\x62\x0e\x1a"
			"\xa9\x21\x2b\x92\x94\x87\x62\x57"
			"\x4c\x10\x74\x1a\xf1\x0a\xc5\x84"
			"\x3b\x9e\x72\x02\xd7\xcc\x09\x56"
			"\xbd\x54\xc1\xf0\xc3\xe3\xb3\xf8"
			"\xd2\x0d\x61\xcb\xef\xce\x0d\x05"
			"\xb0\x98\xd9\x8e\x4f\xf9\xbc\x93"
			"\xa6\xea\xc8\xcf\x10\x53\x4b\xf1"
			"\xec\xfc\x89\xf9\x64\xb0\x22\xbf"
			"\x9e\x55\x46\x9f\x7c\x50\x8e\x84"
			"\x54\x20\x98\xd7\x6c\x40\x1e\xdb"
			"\x69\x34\x78\x61\x24\x21\x9c\x8a"
			"\xb3\x62\x31\x8b\x6e\xf5\x2a\x35"
			"\x86\x13\xb1\x6c\x64\x2e\x41\xa5"
			"\x05\xf2\x42\xba\xd2\x3a\x0d\x8e"
			"\x8a\x59\x94\x3c\xcf\x36\x27\x82"
			"\xc2\x45\xee\x58\xcd\x88\xb4\xec"
			"\xde\xb2\x96\x0a\xaf\x38\x6f\x88"
			"\xd7\xd8\xe1\xdf\xb9\x96\xa9\x0a"
			"\xb1\x95\x28\x86\x20\xe9\x17\x49"
			"\xa2\x29\x38\xaa\xa5\xe9\x6e\xf1"
			"\x19\x27\xc0\xd5\x2a\x22\xc3\x0b"
			"\xdb\x7c\x73\x10\xb9\xba\x89\x76"
			"\x54\xae\x7d\x71\xb3\x93\xf6\x32"
			"\xe6\x47\x43\x55\xac\xa0\x0d\xc2"
			"\x93\x27\x4a\x8e\x0e\x74\x15\xc7"
			"\x0b\x85\xd9\x0c\xa9\x30\x7a\x3e"
			"\xea\x8f\x85\x6d\x3a\x12\x4f\x72"
			"\x69\x58\x7a\x80\xbb\xb5\x97\xf3"
			"\xcf\x70\xd2\x5d\xdd\x4d\x21\x79"
			"\x54\x4d\xe4\x05\xe8\xbd\xc2\x62"
			"\xb1\x3b\x77\x1c\xd6\x5c\xf3\xa0"
			"\x79\x00\xa8\x6c\x29\xd9\x18\x24"
			"\x36\xa2\x46\xc0\x96\x65\x7f\xbd"
			"\x2a\xed\x36\x16\x0c\xaa\x9f\xf4"
			"\xc5\xb4\xe2\x12\xed\x69\xed\x4f"
			"\x26\x2c\x39\x52\x89\x98\xe7\x2c"
			"\x99\xa4\x9e\xa3\x9b\x99\x46\x7a"
			"\x3a\xdc\xa8\x59\xa3\xdb\xc3\x3b"
			"\x95\x0d\x3b\x09\x6e\xee\x83\x5d"
			"\x32\x4d\xed\xab\xfa\x98\x14\x4e"
			"\xc3\x15\x45\x53\x61\xc4\x93\xbd"
			"\x90\xf4\x99\x95\x4c\xe6\x76\x92"
			"\x29\x90\x46\x30\x92\x69\x7d\x13"
			"\xf2\xa5\xcd\x69\x49\x44\xb2\x0f"
			"\x63\x40\x36\x5f\x09\xe2\x78\xf8"
			"\x91\xe3\xe2\xfa\x10\xf7\xc8\x24"
			"\xa8\x89\x32\x5c\x37\x25\x1d\xb2"
			"\xea\x17\x8a\x0a\xa9\x64\xc3\x7c"
			"\x3c\x7c\xbd\xc6\x79\x34\xe7\xe2"
			"\x85\x8e\xbf\xf8\xde\x92\xa0\xae"
			"\x20\xc4\xf6\xbb\x1f\x38\x19\x0e"
			"\xe8\x79\x9c\xa1\x23\xe9\x54\x7e"
			"\x37\x2f\xe2\x94\x32\xaf\xa0\x23"
			"\x49\xe4\xc0\xb3\xac\x00\x8f\x36"
			"\x05\xc4\xa6\x96\xec\x05\x98\x4f"
			"\x96\x67\x57\x1f\x20\x86\x1b\x2d"
			"\x69\xe4\x29\x93\x66\x5f\xaf\x6b"
			"\x88\x26\x2c\x67\x02\x4b\x52\xd0"
			"\x83\x7a\x43\x1f\xc0\x71\x15\x25"
			"\x77\x65\x08\x60\x11\x76\x4c\x8d"
			"\xed\xa9\x27\xc6\xb1\x2a\x2c\x6a"
			"\x4a\x97\xf5\xc6\xb7\x70\x42\xd3"
			"\x03\xd1\x24\x95\xec\x6d\xab\x38"
			"\x72\xce\xe2\x8b\x33\xd7\x51\x09"
			"\xdc\x45\xe0\x09\x96\x32\xf3\xc4"
			"\x84\xdc\x73\x73\x2d\x1b\x11\x98"
			"\xc5\x0e\x69\x28\x94\xc7\xb5\x4d"
			"\xc8\x8a\xd0\xaa\x13\x2e\x18\x74"
			"\xdd\xd1\x1e\xf3\x90\xe8\xfc\x9a"
			"\x72\x4a\x0e\xd1\xe4\xfb\x0d\x96"
			"\xd1\x0c\x79\x85\x1b\x1c\xfe\xe1"
			"\x62\x8f\x7a\x73\x32\xab\xc8\x18"
			"\x69\xe3\x34\x30\xdf\x13\xa6\xe5"
			"\xe8\x0e\x67\x7f\x81\x11\xb4\x60"
			"\xc7\xbd\x79\x65\x50\xdc\xc4\x5b"
			"\xde\x39\xa4\x01\x72\x63\xf3\xd1"
			"\x64\x4e\xdf\xfc\x27\x92\x37\x0d"
			"\x57\xcd\x11\x4f\x11\x04\x8e\x1d"
			"\x16\xf7\xcd\x92\x9a\x99\x30\x14"
			"\xf1\x7c\x67\x1b\x1f\x41\x0b\xe8"
			"\x32\xe8\xb8\xc1\x4f\x54\x86\x4f"
			"\xe5\x79\x81\x73\xcd\x43\x59\x68"
			"\x73\x02\x3b\x78\x21\x72\x43\x00"
			"\x49\x17\xf7\x00\xaf\x68\x24\x53"
			"\x05\x0a\xc3\x33\xe0\x33\x3f\x69"
			"\xd2\x84\x2f\x0b\xed\xde\x04\xf4"
			"\x11\x94\x13\x69\x51\x09\x28\xde"
			"\x57\x5c\xef\xdc\x9a\x49\x1c\x17"
			"\x97\xf3\x96\xc1\x7f\x5d\x2e\x7d"
			"\x55\xb8\xb3\x02\x09\xb3\x1f\xe7"
			"\xc9\x8d\xa3\x36\x34\x8a\x77\x13"
			"\x30\x63\x4c\xa5\xcd\xc3\xe0\x7e"
			"\x05\xa1\x7b\x0c\xcb\x74\x47\x31"
			"\x62\x03\x43\xf1\x87\xb4\xb0\x85"
			"\x87\x8e\x4b\x25\xc7\xcf\xae\x4b"
			"\x36\x46\x3e\x62\xbc\x6f\xeb\x5f"
			"\x73\xac\xe6\x07\xee\xc1\xa1\xd6"
			"\xc4\xab\xc9\xd6\x89\x45\xe1\xf1"
			"\x04\x4e\x1a\x6f\xbb\x4f\x3a\xa3"
			"\xa0\xcb\xa3\x0a\xd8\x71\x35\x55"
			"\xe4\xbc\x2e\x04\x06\xe6\xff\x5b"
			"\x1c\xc0\x11\x7c\xc5\x17\xf3\x38"
			"\xcf\xe9\xba\x0f\x0e\xef\x02\xc2"
			"\x8d\xc6\xbc\x4b\x67\x20\x95\xd7"
			"\x2c\x45\x5b\x86\x44\x8c\x6f\x2e"
			"\x7e\x9f\x1c\x77\xba\x6b\x0e\xa3"
			"\x69\xdc\xab\x24\x57\x60\x47\xc1"
			"\xd1\xa5\x9d\x23\xe6\xb1\x37\xfe"
			"\x93\xd2\x4c\x46\xf9\x0c\xc6\xfb"
			"\xd6\x9d\x99\x69\xab\x7a\x07\x0c"
			"\x65\xe7\xc4\x08\x96\xe2\xa5\x01"
			"\x3f\x46\x07\x05\x7e\xe8\x9a\x90"
			"\x50\xdc\xe9\x7a\xea\xa1\x39\x6e"
			"\x66\xe4\x6f\xa5\x5f\xb2\xd9\x5b"
			"\xf5\xdb\x2a\x32\xf0\x11\x6f\x7c"
			"\x26\x10\x8f\x3d\x80\xe9\x58\xf7"
			"\xe0\xa8\x57\xf8\xdb\x0e\xce\x99"
			"\x63\x19\x3d\xd5\xec\x1b\x77\x69"
			"\x98\xf6\xe4\x5f\x67\x17\x4b\x09"
			"\x85\x62\x82\x70\x18\xe2\x9a\x78"
			"\xe2\x62\xbd\xb4\xf1\x42\xc6\xfb"
			"\x08\xd0\xbd\xeb\x4e\x09\xf2\xc8"
			"\x1e\xdc\x3d\x32\x21\x56\x9c\x4f"
			"\x35\xf3\x61\x06\x72\x84\xc4\x32"
			"\xf2\xf1\xfa\x0b\x2f\xc3\xdb\x02"
			"\x04\xc2\xde\x57\x64\x60\x8d\xcf"
			"\xcb\x86\x5d\x97\x3e\xb1\x9c\x01"
			"\xd6\x28\x8f\x99\xbc\x46\xeb\x05"
			"\xaf\x7e\xb8\x21\x2a\x56\x85\x1c"
			"\xb3\x71\xa0\xde\xca\x96\xf1\x78"
			"\x49\xa2\x99\x81\x80\x5c\x01\xf5"
			"\xa0\xa2\x56\x63\xe2\x70\x07\xa5"
			"\x95\xd6\x85\xeb\x36\x9e\xa9\x51"
			"\x66\x56\x5f\x1d\x02\x19\xe2\xf6"
			"\x4f\x73\x38\x09\x75\x64\x48\xe0"
			"\xf1\x7e\x0e\xe8\x9d\xf9\xed\x94"
			"\xfe\x16\x26\x62\x49\x74\xf4\xb0"
			"\xd4\xa9\x6c\xb0\xfd\x53\xe9\x81"
			"\xe0\x7a\xbf\xcf\xb5\xc4\x01\x81"
			"\x79\x99\x77\x01\x3b\xe9\xa2\xb6"
			"\xe6\x6a\x8a\x9e\x56\x1c\x8d\x1e"
			"\x8f\x06\x55\x2c\x6c\xdc\x92\x87"
			"\x64\x3b\x4b\x19\xa1\x13\x64\x1d"
			"\x4a\xe9\xc0\x00\xb8\x95\xef\x6b"
			"\x1a\x86\x6d\x37\x52\x02\xc2\xe0"
			"\xc8\xbb\x42\x0c\x02\x21\x4a\xc9"
			"\xef\xa0\x54\xe4\x5e\x16\x53\x81"
			"\x70\x62\x10\xaf\xde\xb8\xb5\xd3"
			"\xe8\x5e\x6c\xc3\x8a\x3e\x18\x07"
			"\xf2\x2f\x7d\xa7\xe1\x3d\x4e\xb4"
			"\x26\xa7\xa3\x93\x86\xb2\x04\x1e"
			"\x53\x5d\x86\xd6\xde\x65\xca\xe3"
			"\x4e\xc1\xcf\xef\xc8\x70\x1b\x83"
			"\x13\xdd\x18\x8b\x0d\x76\xd2\xf6"
			"\x37\x7a\x93\x7a\x50\x11\x9f\x96"
			"\x86\x25\xfd\xac\xdc\xbe\x18\x93"
			"\x19\x6b\xec\x58\x4f\xb9\x75\xa7"
			"\xdd\x3f\x2f\xec\xc8\x5a\x84\xab"
			"\xd5\xe4\x8a\x07\xf6\x4d\x23\xd6"
			"\x03\xfb\x03\x6a\xea\x66\xbf\xd4"
			"\xb1\x34\xfb\x78\xe9\x55\xdc\x7c"
			"\x3d\x9c\xe5\x9a\xac\xc3\x7a\x80"
			"\x24\x6d\xa0\xef\x25\x7c\xb7\xea"
			"\xce\x4d\x5f\x18\x60\xce\x87\x22"
			"\x66\x2f\xd5\xdd\xdd\x02\x21\x75"
			"\x82\xa0\x1f\x58\xc6\xd3\x62\xf7"
			"\x32\xd8\xaf\x1e\x07\x77\x51\x96"
			"\xd5\x6b\x1e\x7e\x80\x02\xe8\x67"
			"\xea\x17\x0b\x10\xd2\x3f\x28\x25"
			"\x4f\x05\x77\x02\x14\x69\xf0\x2c"
			"\xbe\x0c\xf1\x74\x30\xd1\xb9\x9b"
			"\xfc\x8c\xbb\x04\x16\xd9\xba\xc3"
			"\xbc\x91\x8a\xc4\x30\xa4\xb0\x12"
			"\x4c\x21\x87\xcb\xc9\x1d\x16\x96"
			"\x07\x6f\x23\x54\xb9\x6f\x79\xe5"
			"\x64\xc0\x64\xda\xb1\xae\xdd\x60"
			"\x6c\x1a\x9d\xd3\x04\x8e\x45\xb0"
			"\x92\x61\xd0\x48\x81\xed\x5e\x1d"
			"\xa0\xc9\xa4\x33\xc7\x13\x51\x5d"
			"\x7f\x83\x73\xb6\x70\x18\x65\x3e"
			"\x2f\x0e\x7a\x12\x39\x98\xab\xd8"
			"\x7e\x6f\xa3\xd1\xba\x56\xad\xbd"
			"\xf0\x03\x01\x1c\x85\x35\x9f\xeb"
			"\x19\x63\xa1\xaf\xfe\x2d\x35\x50"
			"\x39\xa0\x65\x7c\x95\x7e\x6b\xfe"
			"\xc1\xac\x07\x7c\x98\x4f\xbe\x57"
			"\xa7\x22\xec\xe2\x7e\x29\x09\x53"
			"\xe8\xbf\xb4\x7e\x3f\x8f\xfc\x14"
			"\xce\x54\xf9\x18\x58\xb5\xff\x44"
			"\x05\x9d\xce\x1b\xb6\x82\x23\xc8"
			"\x2e\xbc\x69\xbb\x4a\x29\x0f\x65"
			"\x94\xf0\x63\x06\x0e\xef\x8c\xbd"
			"\xff\xfd\xb0\x21\x6e\x57\x05\x75"
			"\xda\xd5\xc4\xeb\x8d\x32\xf7\x50"
			"\xd3\x6f\x22\xed\x5f\x8e\xa2\x5b"
			"\x80\x8c\xc8\x78\x40\x24\x4b\x89"
			"\x30\xce\x7a\x97\x0e\xc4\xaf\xef"
			"\x9b\xb4\xcd\x66\x74\x14\x04\x2b"
			"\xf7\xce\x0b\x1c\x6e\xc2\x78\x8c"
			"\xca\xc5\xd0\x1c\x95\x4a\x91\x2d"
			"\xa7\x20\xeb\x86\x52\xb7\x67\xd8"
			"\x0c\xd6\x04\x14\xde\x51\x74\x75"
			"\xe7\x11\xb4\x87\xa3\x3d\x2d\xad"
			"\x4f\xef\xa0\x0f\x70\x00\x6d\x13"
			"\x19\x1d\x41\x50\xe9\xd8\xf0\x32"
			"\x71\xbc\xd3\x11\xf2\xac\xbe\xaf"
			"\x75\x46\x65\x4e\x07\x34\x37\xa3"
			"\x89\xfe\x75\xd4\x70\x4c\xc6\x3f"
			"\x69\x24\x0e\x38\x67\x43\x8c\xde"
			"\x06\xb5\xb8\xe7\xc4\xf0\x41\x8f"
			"\xf0\xbd\x2f\x0b\xb9\x18\xf8\xde"
			"\x64\xb1\xdb\xee\x00\x50\x77\xe1"
			"\xc7\xff\xa6\xfa\xdd\x70\xf4\xe3"
			"\x93\xe9\x77\x35\x3d\x4b\x2f\x2b"
			"\x6d\x55\xf0\xfc\x88\x54\x4e\x89"
			"\xc1\x8a\x23\x31\x2d\x14\x2a\xb8"
			"\x1b\x15\xdd\x9e\x6e\x7b\xda\x05"
			"\x91\x7d\x62\x64\x96\x72\xde\xfc"
			"\xc1\xec\xf0\x23\x51\x6f\xdb\x5b"
			"\x1d\x08\x57\xce\x09\xb8\xf6\xcd"
			"\x8d\x95\xf2\x20\xbf\x0f\x20\x57"
			"\x98\x81\x84\x4f\x15\x5c\x76\xe7"
			"\x3e\x0a\x3a\x6c\xc4\x8a\xbe\x78"
			"\x74\x77\xc3\x09\x4b\x5d\x48\xe4"
			"\xc8\xcb\x0b\xea\x17\x28\xcf\xcf"
			"\x31\x32\x44\xa4\xe5\x0e\x1a\x98"
			"\x94\xc4\xf0\xff\xae\x3e\x44\xe8"
			"\xa5\xb3\xb5\x37\x2f\xe8\xaf\x6f"
			"\x28\xc1\x37\x5f\x31\xd2\xb9\x33"
			"\xb1\xb2\x52\x94\x75\x2c\x29\x59"
			"\x06\xc2\x25\xe8\x71\x65\x4e\xed"
			"\xc0\x9c\xb1\xbb\x25\xdc\x6c\xe7"
			"\x4b\xa5\x7a\x54\x7a\x60\xff\x7a"
			"\xe0\x50\x40\x96\x35\x63\xe4\x0b"
			"\x76\xbd\xa4\x65\x00\x1b\x57\x88"
			"\xae\xed\x39\x88\x42\x11\x3c\xed"
			"\x85\x67\x7d\xb9\x68\x82\xe9\x43"
			"\x3c\x47\x53\xfa\xe8\xf8\x9f\x1f"
			"\x9f\xef\x0f\xf7\x30\xd9\x30\x0e"
			"\xb9\x9f\x69\x18\x2f\x7e\xf8\xf8"
			"\xf8\x8c\x0f\xd4\x02\x4d\xea\xcd"
			"\x0a\x9c\x6f\x71\x6d\x5a\x4c\x60"
			"\xce\x20\x56\x32\xc6\xc5\x99\x1f"
			"\x09\xe6\x4e\x18\x1a\x15\x13\xa8"
			"\x7d\xb1\x6b\xc0\xb2\x6d\xf8\x26"
			"\x66\xf8\x3d\x18\x74\x70\x66\x7a"
			"\x34\x17\xde\xba\x47\xf1\x06\x18"
			"\xcb\xaf\xeb\x4a\x1e\x8f\xa7\x77"
			"\xe0\x3b\x78\x62\x66\xc9\x10\xea"
			"\x1f\xb7\x29\x0a\x45\xa1\x1d\x1e"
			"\x1d\xe2\x65\x61\x50\x9c\xd7\x05"
			"\xf2\x0b\x5b\x12\x61\x02\xc8\xe5"
			"\x63\x4f\x20\x0c\x07\x17\x33\x5e"
			"\x03\x9a\x53\x0f\x2e\x55\xfe\x50"
			"\x43\x7d\xd0\xb6\x7e\x5a\xda\xae"
			"\x58\xef\x15\xa9\x83\xd9\x46\xb1"
			"\x42\xaa\xf5\x02\x6c\xce\x92\x06"
			"\x1b\xdb\x66\x45\x91\x79\xc2\x2d"
			"\xe6\x53\xd3\x14\xfd\xbb\x44\x63"
			"\xc6\xd7\x3d\x7a\x0c\x75\x78\x9d"
			"\x5c\xa6\x39\xb3\xe5\x63\xca\x8b"
			"\xfe\xd3\xef\x60\x83\xf6\x8e\x70"
			"\xb6\x67\xc7\x77\xed\x23\xef\x4c"
			"\xf0\xed\x2d\x07\x59\x6f\xc1\x01"
			"\x34\x37\x08\xab\xd9\x1f\x09\xb1"
			"\xce\x5b\x17\xff\x74\xf8\x9c\xd5"
			"\x2c\x56\x39\x79\x0f\x69\x44\x75"
			"\x58\x27\x01\xc4\xbf\xa7\xa1\x1d"
			"\x90\x17\x77\x86\x5a\x3f\xd9\xd1"
			"\x0e\xa0\x10\xf8\xec\x1e\xa5\x7f"
			"\x5e\x36\xd1\xe3\x04\x2c\x70\xf7"
			"\x8e\xc0\x98\x2f\x6c\x94\x2b\x41"
			"\xb7\x60\x00\xb7\x2e\xb8\x02\x8d"
			"\xb8\xb0\xd3\x86\xba\x1d\xd7\x90"
			"\xd6\xb6\xe1\xfc\xd7\xd8\x28\x06"
			"\x63\x9b\xce\x61\x24\x79\xc0\x70"
			"\x52\xd0\xb6\xd4\x28\x95\x24\x87"
			"\x03\x1f\xb7\x9a\xda\xa3\xfb\x52"
			"\x5b\x68\xe7\x4c\x8c\x24\xe1\x42"
			"\xf7\xd5\xfd\xad\x06\x32\x9f\xba"
			"\xc1\xfc\xdd\xc6\xfc\xfc\xb3\x38"
			"\x74\x56\x58\x40\x02\x37\x52\x2c"
			"\x55\xcc\xb3\x9e\x7a\xe9\xd4\x38"
			"\x41\x5e\x0c\x35\xe2\x11\xd1\x13"
			"\xf8\xb7\x8d\x72\x6b\x22\x2a\xb0"
			"\xdb\x08\xba\x35\xb9\x3f\xc8\xd3"
			"\x24\x90\xec\x58\xd2\x09\xc7\x2d"
			"\xed\x38\x80\x36\x72\x43\x27\x49"
			"\x4a\x80\x8a\xa2\xe8\xd3\xda\x30"
			"\x7d\xb6\x82\x37\x86\x92\x86\x3e"
			"\x08\xb2\x28\x5a\x55\x44\x24\x7d"
			"\x40\x48\x8a\xb6\x89\x58\x08\xa0"
			"\xd6\x6d\x3a\x17\xbf\xf6\x54\xa2"
			"\xf5\xd3\x8c\x0f\x78\x12\x57\x8b"
			"\xd5\xc2\xfd\x58\x5b\x7f\x38\xe3"
			"\xcc\xb7\x7c\x48\xb3\x20\xe8\x81"
			"\x14\x32\x45\x05\xe0\xdb\x9f\x75"
			"\x85\xb4\x6a\xfc\x95\xe3\x54\x22"
			"\x12\xee\x30\xfe\xd8\x30\xef\x34"
			"\x50\xab\x46\x30\x98\x2f\xb7\xc0"
			"\x15\xa2\x83\xb6\xf2\x06\x21\xa2"
			"\xc3\x26\x37\x14\xd1\x4d\xb5\x10"
			"\x52\x76\x4d\x6a\xee\xb5\x2b\x15"
			"\xb7\xf9\x51\xe8\x2a\xaf\xc7\xfa"
			"\x77\xaf\xb0\x05\x4d\xd1\x68\x8e"
			"\x74\x05\x9f\x9d\x93\xa5\x3e\x7f"
			"\x4e\x5f\x9d\xcb\x09\xc7\x83\xe3"
			"\x02\x9d\x27\x1f\xef\x85\x05\x8d"
			"\xec\x55\x88\x0f\x0d\x7c\x4c\xe8"
			"\xa1\x75\xa0\xd8\x06\x47\x14\xef"
			"\xaa\x61\xcf\x26\x15\xad\xd8\xa3"
			"\xaa\x75\xf2\x78\x4a\x5a\x61\xdf"
			"\x8b\xc7\x04\xbc\xb2\x32\xd2\x7e"
			"\x42\xee\xb4\x2f\x51\xff\x7b\x2e"
			"\xd3\x02\xe8\xdc\x5d\x0d\x50\xdc"
			"\xae\xb7\x46\xf9\xa8\xe6\xd0\x16"
			"\xcc\xe6\x2c\x81\xc7\xad\xe9\xf0"
			"\x05\x72\x6d\x3d\x0a\x7a\xa9\x02"
			"\xac\x82\x93\x6e\xb6\x1c\x28\xfc"
			"\x44\x12\xfb\x73\x77\xd4\x13\x39"
			"\x29\x88\x8a\xf3\x5c\xa6\x36\xa0"
			"\x2a\xed\x7e\xb1\x1d\xd6\x4c\x6b"
			"\x41\x01\x18\x5d\x5d\x07\x97\xa6"
			"\x4b\xef\x31\x18\xea\xac\xb1\x84"
			"\x21\xed\xda\x86",
		.rlen = 4100,
		.np	= 2,
		.tap	= { 4064, 36 },
	},
};

static struct cipher_testvec aes_ctr_rfc3686_dec_tv_template[] = {
	{  
		.key	= "\xae\x68\x52\xf8\x12\x10\x67\xcc"
			  "\x4b\xf7\xa5\x76\x55\x77\xf3\x9e"
			  "\x00\x00\x00\x30",
		.klen	= 20,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xe4\x09\x5d\x4f\xb7\xa7\xb3\x79"
			  "\x2d\x61\x75\xa3\x26\x13\x11\xb8",
		.ilen	= 16,
		.result	= "Single block msg",
		.rlen	= 16,
	}, {
		.key	= "\x7e\x24\x06\x78\x17\xfa\xe0\xd7"
			  "\x43\xd6\xce\x1f\x32\x53\x91\x63"
			  "\x00\x6c\xb6\xdb",
		.klen	= 20,
		.iv	= "\xc0\x54\x3b\x59\xda\x48\xd9\x0b",
		.input	= "\x51\x04\xa1\x06\x16\x8a\x72\xd9"
			  "\x79\x0d\x41\xee\x8e\xda\xd3\x88"
			  "\xeb\x2e\x1e\xfc\x46\xda\x57\xc8"
			  "\xfc\xe6\x30\xdf\x91\x41\xbe\x28",
		.ilen	= 32,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen	= 32,
	}, {
		.key	= "\x16\xaf\x5b\x14\x5f\xc9\xf5\x79"
			  "\xc1\x75\xf9\x3e\x3b\xfb\x0e\xed"
			  "\x86\x3d\x06\xcc\xfd\xb7\x85\x15"
			  "\x00\x00\x00\x48",
		.klen	= 28,
		.iv	= "\x36\x73\x3c\x14\x7d\x6d\x93\xcb",
		.input	= "\x4b\x55\x38\x4f\xe2\x59\xc9\xc8"
			  "\x4e\x79\x35\xa0\x03\xcb\xe9\x28",
		.ilen	= 16,
		.result	= "Single block msg",
		.rlen	= 16,
	}, {
		.key	= "\x7c\x5c\xb2\x40\x1b\x3d\xc3\x3c"
			  "\x19\xe7\x34\x08\x19\xe0\xf6\x9c"
			  "\x67\x8c\x3d\xb8\xe6\xf6\xa9\x1a"
			  "\x00\x96\xb0\x3b",
		.klen	= 28,
		.iv	= "\x02\x0c\x6e\xad\xc2\xcb\x50\x0d",
		.input	= "\x45\x32\x43\xfc\x60\x9b\x23\x32"
			  "\x7e\xdf\xaa\xfa\x71\x31\xcd\x9f"
			  "\x84\x90\x70\x1c\x5a\xd4\xa7\x9c"
			  "\xfc\x1f\xe0\xff\x42\xf4\xfb\x00",
		.ilen	= 32,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen	= 32,
	}, {
		.key	= "\x77\x6b\xef\xf2\x85\x1d\xb0\x6f"
			  "\x4c\x8a\x05\x42\xc8\x69\x6f\x6c"
			  "\x6a\x81\xaf\x1e\xec\x96\xb4\xd3"
			  "\x7f\xc1\xd6\x89\xe6\xc1\xc1\x04"
			  "\x00\x00\x00\x60",
		.klen	= 36,
		.iv	= "\xdb\x56\x72\xc9\x7a\xa8\xf0\xb2",
		.input	= "\x14\x5a\xd0\x1d\xbf\x82\x4e\xc7"
			  "\x56\x08\x63\xdc\x71\xe3\xe0\xc0",
		.ilen	= 16,
		.result	= "Single block msg",
		.rlen	= 16,
	}, {
		.key	= "\xf6\xd6\x6d\x6b\xd5\x2d\x59\xbb"
			  "\x07\x96\x36\x58\x79\xef\xf8\x86"
			  "\xc6\x6d\xd5\x1a\x5b\x6a\x99\x74"
			  "\x4b\x50\x59\x0c\x87\xa2\x38\x84"
			  "\x00\xfa\xac\x24",
		.klen	= 36,
		.iv	= "\xc1\x58\x5e\xf1\x5a\x43\xd8\x75",
		.input	= "\xf0\x5e\x23\x1b\x38\x94\x61\x2c"
			  "\x49\xee\x00\x0b\x80\x4e\xb2\xa9"
			  "\xb8\x30\x6b\x50\x8f\x83\x9d\x6a"
			  "\x55\x30\x83\x1d\x93\x44\xaf\x1c",
		.ilen	= 32,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen	= 32,
	},
};

static struct cipher_testvec aes_ofb_enc_tv_template[] = {
	  
	{
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result = "\x3b\x3f\xd9\x2e\xb7\x2d\xad\x20"
			  "\x33\x34\x49\xf8\xe8\x3c\xfb\x4a"
			  "\x77\x89\x50\x8d\x16\x91\x8f\x03\xf5"
			  "\x3c\x52\xda\xc5\x4e\xd8\x25"
			  "\x97\x40\x05\x1e\x9c\x5f\xec\xf6\x43"
			  "\x44\xf7\xa8\x22\x60\xed\xcc"
			  "\x30\x4c\x65\x28\xf6\x59\xc7\x78"
			  "\x66\xa5\x10\xd9\xc1\xd6\xae\x5e",
		.rlen	= 64,
	}
};

static struct cipher_testvec aes_ofb_dec_tv_template[] = {
	  
	{
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input = "\x3b\x3f\xd9\x2e\xb7\x2d\xad\x20"
			  "\x33\x34\x49\xf8\xe8\x3c\xfb\x4a"
			  "\x77\x89\x50\x8d\x16\x91\x8f\x03\xf5"
			  "\x3c\x52\xda\xc5\x4e\xd8\x25"
			  "\x97\x40\x05\x1e\x9c\x5f\xec\xf6\x43"
			  "\x44\xf7\xa8\x22\x60\xed\xcc"
			  "\x30\x4c\x65\x28\xf6\x59\xc7\x78"
			  "\x66\xa5\x10\xd9\xc1\xd6\xae\x5e",
		.ilen	= 64,
		.result = "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}
};

static struct aead_testvec aes_gcm_enc_tv_template[] = {
	{  
		.key    = zeroed_string,
		.klen	= 16,
		.result	= "\x58\xe2\xfc\xce\xfa\x7e\x30\x61"
			  "\x36\x7f\x1d\x57\xa4\xe7\x45\x5a",
		.rlen	= 16,
	}, {
		.key    = zeroed_string,
		.klen	= 16,
		.input  = zeroed_string,
		.ilen	= 16,
		.result = "\x03\x88\xda\xce\x60\xb6\xa3\x92"
			  "\xf3\x28\xc2\xb9\x71\xb2\xfe\x78"
			  "\xab\x6e\x47\xd4\x2c\xec\x13\xbd"
			  "\xf5\x3a\x67\xb2\x12\x57\xbd\xdf",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.ilen	= 64,
		.result = "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91\x47\x3f\x59\x85"
			  "\x4d\x5c\x2a\xf3\x27\xcd\x64\xa6"
			  "\x2c\xf3\x5a\xbd\x2b\xa6\xfa\xb4",
		.rlen	= 80,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.ilen	= 60,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91"
			  "\x5b\xc9\x4f\xbc\x32\x21\xa5\xdb"
			  "\x94\xfa\xe9\x5a\xe7\x12\x1a\x47",
		.rlen	= 76,
	}, {
		.key    = zeroed_string,
		.klen	= 24,
		.result	= "\xcd\x33\xb2\x8a\xc7\x73\xf7\x4b"
			  "\xa0\x0e\xd1\xf3\x12\x57\x24\x35",
		.rlen	= 16,
	}, {
		.key    = zeroed_string,
		.klen	= 24,
		.input  = zeroed_string,
		.ilen	= 16,
		.result = "\x98\xe7\x24\x7c\x07\xf0\xfe\x41"
			  "\x1c\x26\x7e\x43\x84\xb0\xf6\x00"
			  "\x2f\xf5\x8d\x80\x03\x39\x27\xab"
			  "\x8e\xf4\xd4\x58\x75\x14\xf0\xfb",
		.rlen	= 32,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.ilen	= 64,
		.result = "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10\xac\xad\xe2\x56"
			  "\x99\x24\xa7\xc8\x58\x73\x36\xbf"
			  "\xb1\x18\x02\x4d\xb8\x67\x4a\x14",
		.rlen	= 80,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.ilen	= 60,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10"
			  "\x25\x19\x49\x8e\x80\xf1\x47\x8f"
			  "\x37\xba\x55\xbd\x6d\x27\x61\x8c",
		.rlen	= 76,
		.np	= 2,
		.tap	= { 32, 28 },
		.anp	= 2,
		.atap	= { 8, 12 }
	}, {
		.key    = zeroed_string,
		.klen	= 32,
		.result	= "\x53\x0f\x8a\xfb\xc7\x45\x36\xb9"
			  "\xa9\x63\xb4\xf1\xc4\xcb\x73\x8b",
		.rlen	= 16,
	}
};

static struct aead_testvec aes_gcm_dec_tv_template[] = {
	{  
		.key    = zeroed_string,
		.klen	= 32,
		.input	= "\xce\xa7\x40\x3d\x4d\x60\x6b\x6e"
			  "\x07\x4e\xc5\xd3\xba\xf3\x9d\x18"
			  "\xd0\xd1\xc8\xa7\x99\x99\x6b\xf0"
			  "\x26\x5b\x98\xb5\xd4\x8a\xb9\x19",
		.ilen	= 32,
		.result  = zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 32,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x52\x2d\xc1\xf0\x99\x56\x7d\x07"
			  "\xf4\x7f\x37\xa3\x2a\x84\x42\x7d"
			  "\x64\x3a\x8c\xdc\xbf\xe5\xc0\xc9"
			  "\x75\x98\xa2\xbd\x25\x55\xd1\xaa"
			  "\x8c\xb0\x8e\x48\x59\x0d\xbb\x3d"
			  "\xa7\xb0\x8b\x10\x56\x82\x88\x38"
			  "\xc5\xf6\x1e\x63\x93\xba\x7a\x0a"
			  "\xbc\xc9\xf6\x62\x89\x80\x15\xad"
			  "\xb0\x94\xda\xc5\xd9\x34\x71\xbd"
			  "\xec\x1a\x50\x22\x70\xe3\xcc\x6c",
		.ilen	= 80,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.rlen	= 64,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 32,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x52\x2d\xc1\xf0\x99\x56\x7d\x07"
			  "\xf4\x7f\x37\xa3\x2a\x84\x42\x7d"
			  "\x64\x3a\x8c\xdc\xbf\xe5\xc0\xc9"
			  "\x75\x98\xa2\xbd\x25\x55\xd1\xaa"
			  "\x8c\xb0\x8e\x48\x59\x0d\xbb\x3d"
			  "\xa7\xb0\x8b\x10\x56\x82\x88\x38"
			  "\xc5\xf6\x1e\x63\x93\xba\x7a\x0a"
			  "\xbc\xc9\xf6\x62"
			  "\x76\xfc\x6e\xce\x0f\x4e\x17\x68"
			  "\xcd\xdf\x88\x53\xbb\x2d\x55\x1b",
		.ilen	= 76,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.rlen	= 60,
		.np     = 2,
		.tap    = { 48, 28 },
		.anp	= 3,
		.atap	= { 8, 8, 4 }
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91\x47\x3f\x59\x85"
			  "\x4d\x5c\x2a\xf3\x27\xcd\x64\xa6"
			  "\x2c\xf3\x5a\xbd\x2b\xa6\xfa\xb4",
		.ilen	= 80,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.rlen	= 64,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08",
		.klen	= 16,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x42\x83\x1e\xc2\x21\x77\x74\x24"
			  "\x4b\x72\x21\xb7\x84\xd0\xd4\x9c"
			  "\xe3\xaa\x21\x2f\x2c\x02\xa4\xe0"
			  "\x35\xc1\x7e\x23\x29\xac\xa1\x2e"
			  "\x21\xd5\x14\xb2\x54\x66\x93\x1c"
			  "\x7d\x8f\x6a\x5a\xac\x84\xaa\x05"
			  "\x1b\xa3\x0b\x39\x6a\x0a\xac\x97"
			  "\x3d\x58\xe0\x91"
			  "\x5b\xc9\x4f\xbc\x32\x21\xa5\xdb"
			  "\x94\xfa\xe9\x5a\xe7\x12\x1a\x47",
		.ilen	= 76,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.rlen	= 60,
	}, {
		.key    = zeroed_string,
		.klen	= 24,
		.input	= "\x98\xe7\x24\x7c\x07\xf0\xfe\x41"
			  "\x1c\x26\x7e\x43\x84\xb0\xf6\x00"
			  "\x2f\xf5\x8d\x80\x03\x39\x27\xab"
			  "\x8e\xf4\xd4\x58\x75\x14\xf0\xfb",
		.ilen	= 32,
		.result  = zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10\xac\xad\xe2\x56"
			  "\x99\x24\xa7\xc8\x58\x73\x36\xbf"
			  "\xb1\x18\x02\x4d\xb8\x67\x4a\x14",
		.ilen	= 80,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39\x1a\xaf\xd2\x55",
		.rlen	= 64,
	}, {
		.key	= "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
			  "\xfe\xff\xe9\x92\x86\x65\x73\x1c",
		.klen	= 24,
		.iv	= "\xca\xfe\xba\xbe\xfa\xce\xdb\xad"
			  "\xde\xca\xf8\x88",
		.input	= "\x39\x80\xca\x0b\x3c\x00\xe8\x41"
			  "\xeb\x06\xfa\xc4\x87\x2a\x27\x57"
			  "\x85\x9e\x1c\xea\xa6\xef\xd9\x84"
			  "\x62\x85\x93\xb4\x0c\xa1\xe1\x9c"
			  "\x7d\x77\x3d\x00\xc1\x44\xc5\x25"
			  "\xac\x61\x9d\x18\xc8\x4a\x3f\x47"
			  "\x18\xe2\x44\x8b\x2f\xe3\x24\xd9"
			  "\xcc\xda\x27\x10"
			  "\x25\x19\x49\x8e\x80\xf1\x47\x8f"
			  "\x37\xba\x55\xbd\x6d\x27\x61\x8c",
		.ilen	= 76,
		.assoc	= "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xfe\xed\xfa\xce\xde\xad\xbe\xef"
			  "\xab\xad\xda\xd2",
		.alen	= 20,
		.result = "\xd9\x31\x32\x25\xf8\x84\x06\xe5"
			  "\xa5\x59\x09\xc5\xaf\xf5\x26\x9a"
			  "\x86\xa7\xa9\x53\x15\x34\xf7\xda"
			  "\x2e\x4c\x30\x3d\x8a\x31\x8a\x72"
			  "\x1c\x3c\x0c\x95\x95\x68\x09\x53"
			  "\x2f\xcf\x0e\x24\x49\xa6\xb5\x25"
			  "\xb1\x6a\xed\xf5\xaa\x0d\xe6\x57"
			  "\xba\x63\x7b\x39",
		.rlen	= 60,
	}
};

static struct aead_testvec aes_gcm_rfc4106_enc_tv_template[] = {
        {  
		.key    = zeroed_string,
		.klen	= 20,
                .iv     = zeroed_string,
                .input  = zeroed_string,
                .ilen   = 16,
                .assoc  = zeroed_string,
                .alen   = 8,
		.result	= "\x03\x88\xDA\xCE\x60\xB6\xA3\x92"
                          "\xF3\x28\xC2\xB9\x71\xB2\xFE\x78"
                          "\x97\xFE\x4C\x23\x37\x42\x01\xE0"
                          "\x81\x9F\x8D\xC5\xD7\x41\xA0\x1B",
		.rlen	= 32,
        },{
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x00\x00\x00\x00\x00\x01"
                          "\x00\x00\x00\x00",
                .input  = zeroed_string,
                .ilen   = 16,
                .assoc  = zeroed_string,
                .alen   = 8,
		.result	= "\xC0\x0D\x8B\x42\x0F\x8F\x34\x18"
                          "\x88\xB1\xC5\xBC\xC5\xB6\xD6\x28"
                          "\x6A\x9D\xDF\x11\x5E\xFE\x5E\x9D"
                          "\x2F\x70\x44\x92\xF7\xF2\xE3\xEF",
		.rlen	= 32,

        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = zeroed_string,
                .input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .ilen   = 16,
                .assoc  = zeroed_string,
                .alen   = 8,
		.result	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
                          "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
                          "\x0B\x8F\x88\x69\x17\xE6\xB4\x3C"
                          "\xB1\x68\xFD\x14\x52\x64\x61\xB2",
		.rlen	= 32,
        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = zeroed_string,
                .input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .ilen   = 16,
                .assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01",
                .alen   = 8,
		.result	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
                          "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
                          "\x90\x92\xB7\xE3\x5F\xA3\x9A\x63"
                          "\x7E\xD7\x1F\xD8\xD3\x7C\x4B\xF5",
		.rlen	= 32,
        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x00\x00\x00\x00\x00\x01"
                          "\x00\x00\x00\x00",
                .input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .ilen   = 16,
                .assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01",
                .alen   = 8,
		.result	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
                          "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
                          "\x64\x50\xF9\x32\x13\xFB\x74\x61"
                          "\xF4\xED\x52\xD3\xC5\x10\x55\x3C",
		.rlen	= 32,
        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x00\x00\x00\x00\x00\x01"
                          "\x00\x00\x00\x00",
                .input  = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .ilen   = 64,
                .assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01",
                .alen   = 8,
		.result	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
                          "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
                          "\x98\x14\xA1\x42\x37\x80\xFD\x90"
                          "\x68\x12\x01\xA8\x91\x89\xB9\x83"
                          "\x5B\x11\x77\x12\x9B\xFF\x24\x89"
                          "\x94\x5F\x18\x12\xBA\x27\x09\x39"
                          "\x99\x96\x76\x42\x15\x1C\xCD\xCB"
                          "\xDC\xD3\xDA\x65\x73\xAF\x80\xCD"
                          "\xD2\xB6\xC2\x4A\x76\xC2\x92\x85"
                          "\xBD\xCF\x62\x98\x58\x14\xE5\xBD",
		.rlen	= 80,
        }, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x45\x67\x89\xab\xcd\xef"
                          "\x00\x00\x00\x00",
                .input  = "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff",
                .ilen   = 192,
                .assoc  = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
                          "\xaa\xaa\xaa\xaa",
                .alen   = 12,
		.result	= "\xC1\x76\x33\x85\xE2\x9B\x5F\xDE"
			  "\xDE\x89\x3D\x42\xE7\xC9\x69\x8A"
			  "\x44\x6D\xC3\x88\x46\x2E\xC2\x01"
			  "\x5E\xF6\x0C\x39\xF0\xC4\xA5\x82"
			  "\xCD\xE8\x31\xCC\x0A\x4C\xE4\x44"
			  "\x41\xA9\x82\x6F\x22\xA1\x23\x1A"
			  "\xA8\xE3\x16\xFD\x31\x5C\x27\x31"
			  "\xF1\x7F\x01\x63\xA3\xAF\x70\xA1"
			  "\xCF\x07\x57\x41\x67\xD0\xC4\x42"
			  "\xDB\x18\xC6\x4C\x4C\xE0\x3D\x9F"
			  "\x05\x07\xFB\x13\x7D\x4A\xCA\x5B"
			  "\xF0\xBF\x64\x7E\x05\xB1\x72\xEE"
			  "\x7C\x3B\xD4\xCD\x14\x03\xB2\x2C"
			  "\xD3\xA9\xEE\xFA\x17\xFC\x9C\xDF"
			  "\xC7\x75\x40\xFF\xAE\xAD\x1E\x59"
			  "\x2F\x30\x24\xFB\xAD\x6B\x10\xFA"
			  "\x6C\x9F\x5B\xE7\x25\xD5\xD0\x25"
			  "\xAC\x4A\x4B\xDA\xFC\x7A\x85\x1B"
			  "\x7E\x13\x06\x82\x08\x17\xA4\x35"
			  "\xEC\xC5\x8D\x63\x96\x81\x0A\x8F"
			  "\xA3\x05\x38\x95\x20\x1A\x47\x04"
			  "\x6F\x6D\xDA\x8F\xEF\xC1\x76\x35"
			  "\x6B\xC7\x4D\x0F\x94\x12\xCA\x3E"
			  "\x2E\xD5\x03\x2E\x86\x7E\xAA\x3B"
			  "\x37\x08\x1C\xCF\xBA\x5D\x71\x46"
			  "\x80\x72\xB0\x4C\x82\x0D\x60\x3C",
		.rlen	= 208,
	}
};

static struct aead_testvec aes_gcm_rfc4106_dec_tv_template[] = {
        {  
		.key    = zeroed_string,
		.klen	= 20,
                .iv     = zeroed_string,
		.input	= "\x03\x88\xDA\xCE\x60\xB6\xA3\x92"
                          "\xF3\x28\xC2\xB9\x71\xB2\xFE\x78"
                          "\x97\xFE\x4C\x23\x37\x42\x01\xE0"
                          "\x81\x9F\x8D\xC5\xD7\x41\xA0\x1B",
		.ilen	= 32,
                .assoc  = zeroed_string,
                .alen   = 8,
                .result = zeroed_string,
                .rlen   = 16,

        },{
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x00\x00\x00\x00\x00\x01"
                          "\x00\x00\x00\x00",
		.input	= "\xC0\x0D\x8B\x42\x0F\x8F\x34\x18"
                          "\x88\xB1\xC5\xBC\xC5\xB6\xD6\x28"
                          "\x6A\x9D\xDF\x11\x5E\xFE\x5E\x9D"
                          "\x2F\x70\x44\x92\xF7\xF2\xE3\xEF",
		.ilen	= 32,
                .assoc  = zeroed_string,
                .alen   = 8,
                .result = zeroed_string,
                .rlen   = 16,
        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = zeroed_string,
		.input	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
                          "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
                          "\x0B\x8F\x88\x69\x17\xE6\xB4\x3C"
                          "\xB1\x68\xFD\x14\x52\x64\x61\xB2",
		.ilen	= 32,
                .assoc  = zeroed_string,
                .alen   = 8,
                .result = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .rlen   = 16,
        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = zeroed_string,
		.input	= "\x4B\xB1\xB5\xE3\x25\x71\x70\xDE"
                          "\x7F\xC9\x9C\xA5\x14\x19\xF2\xAC"
                          "\x90\x92\xB7\xE3\x5F\xA3\x9A\x63"
                          "\x7E\xD7\x1F\xD8\xD3\x7C\x4B\xF5",
		.ilen	= 32,
                .assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01",
                .alen   = 8,
                .result = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .rlen   = 16,

        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x00\x00\x00\x00\x00\x01"
                          "\x00\x00\x00\x00",
		.input	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
                          "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
                          "\x64\x50\xF9\x32\x13\xFB\x74\x61"
                          "\xF4\xED\x52\xD3\xC5\x10\x55\x3C",
		.ilen	= 32,
                .assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01",
                .alen   = 8,
                .result = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .rlen   = 16,
        }, {
		.key    = "\xfe\xff\xe9\x92\x86\x65\x73\x1c"
			  "\x6d\x6a\x8f\x94\x67\x30\x83\x08"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x00\x00\x00\x00\x00\x01"
                          "\x00\x00\x00\x00",
		.input	= "\xC1\x0C\x8A\x43\x0E\x8E\x35\x19"
                          "\x89\xB0\xC4\xBD\xC4\xB7\xD7\x29"
                          "\x98\x14\xA1\x42\x37\x80\xFD\x90"
                          "\x68\x12\x01\xA8\x91\x89\xB9\x83"
                          "\x5B\x11\x77\x12\x9B\xFF\x24\x89"
                          "\x94\x5F\x18\x12\xBA\x27\x09\x39"
                          "\x99\x96\x76\x42\x15\x1C\xCD\xCB"
                          "\xDC\xD3\xDA\x65\x73\xAF\x80\xCD"
                          "\xD2\xB6\xC2\x4A\x76\xC2\x92\x85"
                          "\xBD\xCF\x62\x98\x58\x14\xE5\xBD",
		.ilen	= 80,
                .assoc  = "\x01\x01\x01\x01\x01\x01\x01\x01",
                .alen   = 8,
                .result = "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01"
                          "\x01\x01\x01\x01\x01\x01\x01\x01",
                .rlen   = 64,
        }, {
		.key    = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
                          "\x00\x00\x00\x00",
		.klen	= 20,
                .iv     = "\x00\x00\x45\x67\x89\xab\xcd\xef"
                          "\x00\x00\x00\x00",
		.input	= "\xC1\x76\x33\x85\xE2\x9B\x5F\xDE"
			  "\xDE\x89\x3D\x42\xE7\xC9\x69\x8A"
			  "\x44\x6D\xC3\x88\x46\x2E\xC2\x01"
			  "\x5E\xF6\x0C\x39\xF0\xC4\xA5\x82"
			  "\xCD\xE8\x31\xCC\x0A\x4C\xE4\x44"
			  "\x41\xA9\x82\x6F\x22\xA1\x23\x1A"
			  "\xA8\xE3\x16\xFD\x31\x5C\x27\x31"
			  "\xF1\x7F\x01\x63\xA3\xAF\x70\xA1"
			  "\xCF\x07\x57\x41\x67\xD0\xC4\x42"
			  "\xDB\x18\xC6\x4C\x4C\xE0\x3D\x9F"
			  "\x05\x07\xFB\x13\x7D\x4A\xCA\x5B"
			  "\xF0\xBF\x64\x7E\x05\xB1\x72\xEE"
			  "\x7C\x3B\xD4\xCD\x14\x03\xB2\x2C"
			  "\xD3\xA9\xEE\xFA\x17\xFC\x9C\xDF"
			  "\xC7\x75\x40\xFF\xAE\xAD\x1E\x59"
			  "\x2F\x30\x24\xFB\xAD\x6B\x10\xFA"
			  "\x6C\x9F\x5B\xE7\x25\xD5\xD0\x25"
			  "\xAC\x4A\x4B\xDA\xFC\x7A\x85\x1B"
			  "\x7E\x13\x06\x82\x08\x17\xA4\x35"
			  "\xEC\xC5\x8D\x63\x96\x81\x0A\x8F"
			  "\xA3\x05\x38\x95\x20\x1A\x47\x04"
			  "\x6F\x6D\xDA\x8F\xEF\xC1\x76\x35"
			  "\x6B\xC7\x4D\x0F\x94\x12\xCA\x3E"
			  "\x2E\xD5\x03\x2E\x86\x7E\xAA\x3B"
			  "\x37\x08\x1C\xCF\xBA\x5D\x71\x46"
			  "\x80\x72\xB0\x4C\x82\x0D\x60\x3C",
		.ilen	= 208,
                .assoc  = "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
                          "\xaa\xaa\xaa\xaa",
                .alen   = 12,
                .result = "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff"
                          "\xff\xff\xff\xff\xff\xff\xff\xff",
                .rlen   = 192,

	}
};

static struct aead_testvec aes_gcm_rfc4543_enc_tv_template[] = {
	{  
		.key	= "\x4c\x80\xcd\xef\xbb\x5d\x10\xda"
			  "\x90\x6a\xc7\x3c\x36\x13\xa6\x34"
			  "\x22\x43\x3c\x64",
		.klen	= 20,
		.iv	= zeroed_string,
		.assoc	= "\x00\x00\x43\x21\x00\x00\x00\x07",
		.alen	= 8,
		.input	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.ilen	= 52,
		.result	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01\xf2\xa9\xa8\x36"
			  "\xe1\x55\x10\x6a\xa8\xdc\xd6\x18"
			  "\xe4\x09\x9a\xaa",
		.rlen	= 68,
	}
};

static struct aead_testvec aes_gcm_rfc4543_dec_tv_template[] = {
	{  
		.key	= "\x4c\x80\xcd\xef\xbb\x5d\x10\xda"
			  "\x90\x6a\xc7\x3c\x36\x13\xa6\x34"
			  "\x22\x43\x3c\x64",
		.klen	= 20,
		.iv	= zeroed_string,
		.assoc	= "\x00\x00\x43\x21\x00\x00\x00\x07",
		.alen	= 8,
		.input	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01\xf2\xa9\xa8\x36"
			  "\xe1\x55\x10\x6a\xa8\xdc\xd6\x18"
			  "\xe4\x09\x9a\xaa",
		.ilen	= 68,
		.result	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
	}, {  
		.key	= "\x4c\x80\xcd\xef\xbb\x5d\x10\xda"
			  "\x90\x6a\xc7\x3c\x36\x13\xa6\x34"
			  "\x22\x43\x3c\x64",
		.klen	= 20,
		.iv	= zeroed_string,
		.assoc	= "\x00\x00\x43\x21\x00\x00\x00\x07",
		.alen	= 8,
		.input	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01\xf2\xa9\xa8\x36"
			  "\xe1\x55\x10\x6a\xa8\xdc\xd6\x18"
			  "\x00\x00\x00\x00",
		.ilen	= 68,
		.novrfy = 1,
		.result	= "\x45\x00\x00\x30\xda\x3a\x00\x00"
			  "\x80\x01\xdf\x3b\xc0\xa8\x00\x05"
			  "\xc0\xa8\x00\x01\x08\x00\xc6\xcd"
			  "\x02\x00\x07\x00\x61\x62\x63\x64"
			  "\x65\x66\x67\x68\x69\x6a\x6b\x6c"
			  "\x6d\x6e\x6f\x70\x71\x72\x73\x74"
			  "\x01\x02\x02\x01",
		.rlen	= 52,
	},
};

static struct aead_testvec aes_ccm_enc_tv_template[] = {
	{  
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x03\x02\x01\x00"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e",
		.ilen	= 23,
		.result	= "\x58\x8c\x97\x9a\x61\xc6\x63\xd2"
			  "\xf0\x66\xd0\xc2\xc0\xf9\x89\x80"
			  "\x6d\x5f\x6b\x61\xda\xc3\x84\x17"
			  "\xe8\xd1\x2c\xfd\xf9\x26\xe0",
		.rlen	= 31,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x07\x06\x05\x04"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e\x1f",
		.ilen	= 20,
		.result	= "\xdc\xf1\xfb\x7b\x5d\x9e\x23\xfb"
			  "\x9d\x4e\x13\x12\x53\x65\x8a\xd8"
			  "\x6e\xbd\xca\x3e\x51\xe8\x3f\x07"
			  "\x7d\x9c\x2d\x93",
		.rlen	= 28,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0b\x0a\x09\x08"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20",
		.ilen	= 25,
		.result	= "\x82\x53\x1a\x60\xcc\x24\x94\x5a"
			  "\x4b\x82\x79\x18\x1a\xb5\xc8\x4d"
			  "\xf2\x1c\xe7\xf9\xb7\x3f\x42\xe1"
			  "\x97\xea\x9c\x07\xe5\x6b\x5e\xb1"
			  "\x7e\x5f\x4e",
		.rlen	= 35,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0c\x0b\x0a\x09"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e",
		.ilen	= 19,
		.result	= "\x07\x34\x25\x94\x15\x77\x85\x15"
			  "\x2b\x07\x40\x98\x33\x0a\xbb\x14"
			  "\x1b\x94\x7b\x56\x6a\xa9\x40\x6b"
			  "\x4d\x99\x99\x88\xdd",
		.rlen	= 29,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x33\x56\x8e\xf7\xb2\x63"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\x63\x01\x8f\x76\xdc\x8a\x1b\xcb",
		.alen	= 8,
		.input	= "\x90\x20\xea\x6f\x91\xbd\xd8\x5a"
			  "\xfa\x00\x39\xba\x4b\xaf\xf9\xbf"
			  "\xb7\x9c\x70\x28\x94\x9c\xd0\xec",
		.ilen	= 24,
		.result	= "\x4c\xcb\x1e\x7c\xa9\x81\xbe\xfa"
			  "\xa0\x72\x6c\x55\xd3\x78\x06\x12"
			  "\x98\xc8\x5c\x92\x81\x4a\xbc\x33"
			  "\xc5\x2e\xe8\x1d\x7d\x77\xc0\x8a",
		.rlen	= 32,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\xd5\x60\x91\x2d\x3f\x70"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xcd\x90\x44\xd2\xb7\x1f\xdb\x81"
			  "\x20\xea\x60\xc0",
		.alen	= 12,
		.input	= "\x64\x35\xac\xba\xfb\x11\xa8\x2e"
			  "\x2f\x07\x1d\x7c\xa4\xa5\xeb\xd9"
			  "\x3a\x80\x3b\xa8\x7f",
		.ilen	= 21,
		.result	= "\x00\x97\x69\xec\xab\xdf\x48\x62"
			  "\x55\x94\xc5\x92\x51\xe6\x03\x57"
			  "\x22\x67\x5e\x04\xc8\x47\x09\x9e"
			  "\x5a\xe0\x70\x45\x51",
		.rlen	= 29,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x42\xff\xf8\xf1\x95\x1c"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xd8\x5b\xc7\xe6\x9f\x94\x4f\xb8",
		.alen	= 8,
		.input	= "\x8a\x19\xb9\x50\xbc\xf7\x1a\x01"
			  "\x8e\x5e\x67\x01\xc9\x17\x87\x65"
			  "\x98\x09\xd6\x7d\xbe\xdd\x18",
		.ilen	= 23,
		.result	= "\xbc\x21\x8d\xaa\x94\x74\x27\xb6"
			  "\xdb\x38\x6a\x99\xac\x1a\xef\x23"
			  "\xad\xe0\xb5\x29\x39\xcb\x6a\x63"
			  "\x7c\xf9\xbe\xc2\x40\x88\x97\xc6"
			  "\xba",
		.rlen	= 33,
	},
};

static struct aead_testvec aes_ccm_dec_tv_template[] = {
	{  
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x03\x02\x01\x00"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x58\x8c\x97\x9a\x61\xc6\x63\xd2"
			  "\xf0\x66\xd0\xc2\xc0\xf9\x89\x80"
			  "\x6d\x5f\x6b\x61\xda\xc3\x84\x17"
			  "\xe8\xd1\x2c\xfd\xf9\x26\xe0",
		.ilen	= 31,
		.result	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e",
		.rlen	= 23,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x07\x06\x05\x04"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\xdc\xf1\xfb\x7b\x5d\x9e\x23\xfb"
			  "\x9d\x4e\x13\x12\x53\x65\x8a\xd8"
			  "\x6e\xbd\xca\x3e\x51\xe8\x3f\x07"
			  "\x7d\x9c\x2d\x93",
		.ilen	= 28,
		.result	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e\x1f",
		.rlen	= 20,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0b\x0a\x09\x08"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07",
		.alen	= 8,
		.input	= "\x82\x53\x1a\x60\xcc\x24\x94\x5a"
			  "\x4b\x82\x79\x18\x1a\xb5\xc8\x4d"
			  "\xf2\x1c\xe7\xf9\xb7\x3f\x42\xe1"
			  "\x97\xea\x9c\x07\xe5\x6b\x5e\xb1"
			  "\x7e\x5f\x4e",
		.ilen	= 35,
		.result	= "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20",
		.rlen	= 25,
	}, {
		.key	= "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf",
		.klen	= 16,
		.iv	= "\x01\x00\x00\x00\x0c\x0b\x0a\x09"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\x00\x00",
		.assoc	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b",
		.alen	= 12,
		.input	= "\x07\x34\x25\x94\x15\x77\x85\x15"
			  "\x2b\x07\x40\x98\x33\x0a\xbb\x14"
			  "\x1b\x94\x7b\x56\x6a\xa9\x40\x6b"
			  "\x4d\x99\x99\x88\xdd",
		.ilen	= 29,
		.result	= "\x0c\x0d\x0e\x0f\x10\x11\x12\x13"
			  "\x14\x15\x16\x17\x18\x19\x1a\x1b"
			  "\x1c\x1d\x1e",
		.rlen	= 19,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x33\x56\x8e\xf7\xb2\x63"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\x63\x01\x8f\x76\xdc\x8a\x1b\xcb",
		.alen	= 8,
		.input	= "\x4c\xcb\x1e\x7c\xa9\x81\xbe\xfa"
			  "\xa0\x72\x6c\x55\xd3\x78\x06\x12"
			  "\x98\xc8\x5c\x92\x81\x4a\xbc\x33"
			  "\xc5\x2e\xe8\x1d\x7d\x77\xc0\x8a",
		.ilen	= 32,
		.result	= "\x90\x20\xea\x6f\x91\xbd\xd8\x5a"
			  "\xfa\x00\x39\xba\x4b\xaf\xf9\xbf"
			  "\xb7\x9c\x70\x28\x94\x9c\xd0\xec",
		.rlen	= 24,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\xd5\x60\x91\x2d\x3f\x70"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xcd\x90\x44\xd2\xb7\x1f\xdb\x81"
			  "\x20\xea\x60\xc0",
		.alen	= 12,
		.input	= "\x00\x97\x69\xec\xab\xdf\x48\x62"
			  "\x55\x94\xc5\x92\x51\xe6\x03\x57"
			  "\x22\x67\x5e\x04\xc8\x47\x09\x9e"
			  "\x5a\xe0\x70\x45\x51",
		.ilen	= 29,
		.result	= "\x64\x35\xac\xba\xfb\x11\xa8\x2e"
			  "\x2f\x07\x1d\x7c\xa4\xa5\xeb\xd9"
			  "\x3a\x80\x3b\xa8\x7f",
		.rlen	= 21,
	}, {
		.key	= "\xd7\x82\x8d\x13\xb2\xb0\xbd\xc3"
			  "\x25\xa7\x62\x36\xdf\x93\xcc\x6b",
		.klen	= 16,
		.iv	= "\x01\x00\x42\xff\xf8\xf1\x95\x1c"
			  "\x3c\x96\x96\x76\x6c\xfa\x00\x00",
		.assoc	= "\xd8\x5b\xc7\xe6\x9f\x94\x4f\xb8",
		.alen	= 8,
		.input	= "\xbc\x21\x8d\xaa\x94\x74\x27\xb6"
			  "\xdb\x38\x6a\x99\xac\x1a\xef\x23"
			  "\xad\xe0\xb5\x29\x39\xcb\x6a\x63"
			  "\x7c\xf9\xbe\xc2\x40\x88\x97\xc6"
			  "\xba",
		.ilen	= 33,
		.result	= "\x8a\x19\xb9\x50\xbc\xf7\x1a\x01"
			  "\x8e\x5e\x67\x01\xc9\x17\x87\x65"
			  "\x98\x09\xd6\x7d\xbe\xdd\x18",
		.rlen	= 23,
	},
};

static struct aead_testvec aes_ccm_rfc4309_enc_tv_template[] = {
	{
		.key	= "\x83\xac\x54\x66\xc2\xeb\xe5\x05"
			  "\x2e\x01\xd1\xfc\x5d\x82\x66\x2e"
			  "\x96\xac\x59",
		.klen	= 19,
		.iv	= "\x30\x07\xa1\xe2\xa2\xc7\x55\x24",
		.alen	= 0,
		.input	= "\x19\xc8\x81\xf6\xe9\x86\xff\x93"
			  "\x0b\x78\x67\xe5\xbb\xb7\xfc\x6e"
			  "\x83\x77\xb3\xa6\x0c\x8c\x9f\x9c"
			  "\x35\x2e\xad\xe0\x62\xf9\x91\xa1",
		.ilen	= 32,
		.result	= "\xab\x6f\xe1\x69\x1d\x19\x99\xa8"
			  "\x92\xa0\xc4\x6f\x7e\xe2\x8b\xb1"
			  "\x70\xbb\x8c\xa6\x4c\x6e\x97\x8a"
			  "\x57\x2b\xbe\x5d\x98\xa6\xb1\x32"
			  "\xda\x24\xea\xd9\xa1\x39\x98\xfd"
			  "\xa4\xbe\xd9\xf2\x1a\x6d\x22\xa8",
		.rlen	= 48,
	}, {
		.key	= "\x1e\x2c\x7e\x01\x41\x9a\xef\xc0"
			  "\x0d\x58\x96\x6e\x5c\xa2\x4b\xd3"
			  "\x4f\xa3\x19",
		.klen	= 19,
		.iv	= "\xd3\x01\x5a\xd8\x30\x60\x15\x56",
		.assoc	= "\xda\xe6\x28\x9c\x45\x2d\xfd\x63"
			  "\x5e\xda\x4c\xb6\xe6\xfc\xf9\xb7"
			  "\x0c\x56\xcb\xe4\xe0\x05\x7a\xe1"
			  "\x0a\x63\x09\x78\xbc\x2c\x55\xde",
		.alen	= 32,
		.input	= "\x87\xa3\x36\xfd\x96\xb3\x93\x78"
			  "\xa9\x28\x63\xba\x12\xa3\x14\x85"
			  "\x57\x1e\x06\xc9\x7b\x21\xef\x76"
			  "\x7f\x38\x7e\x8e\x29\xa4\x3e\x7e",
		.ilen	= 32,
		.result	= "\x8a\x1e\x11\xf0\x02\x6b\xe2\x19"
			  "\xfc\x70\xc4\x6d\x8e\xb7\x99\xab"
			  "\xc5\x4b\xa2\xac\xd3\xf3\x48\xff"
			  "\x3b\xb5\xce\x53\xef\xde\xbb\x02"
			  "\xa9\x86\x15\x6c\x13\xfe\xda\x0a"
			  "\x22\xb8\x29\x3d\xd8\x39\x9a\x23",
		.rlen	= 48,
	}, {
		.key	= "\xf4\x6b\xc2\x75\x62\xfe\xb4\xe1"
			  "\xa3\xf0\xff\xdd\x4e\x4b\x12\x75"
			  "\x53\x14\x73\x66\x8d\x88\xf6\x80"
			  "\xa0\x20\x35",
		.klen	= 27,
		.iv	= "\x26\xf2\x21\x8d\x50\x20\xda\xe2",
		.assoc	= "\x5b\x9e\x13\x67\x02\x5e\xef\xc1"
			  "\x6c\xf9\xd7\x1e\x52\x8f\x7a\x47"
			  "\xe9\xd4\xcf\x20\x14\x6e\xf0\x2d"
			  "\xd8\x9e\x2b\x56\x10\x23\x56\xe7",
		.alen	= 32,
		.ilen	= 0,
		.result	= "\x36\xea\x7a\x70\x08\xdc\x6a\xbc"
			  "\xad\x0c\x7a\x63\xf6\x61\xfd\x9b",
		.rlen	= 16,
	}, {
		.key	= "\x56\xdf\x5c\x8f\x26\x3f\x0e\x42"
			  "\xef\x7a\xd3\xce\xfc\x84\x60\x62"
			  "\xca\xb4\x40\xaf\x5f\xc9\xc9\x01"
			  "\xd6\x3c\x8c",
		.klen	= 27,
		.iv	= "\x86\x84\xb6\xcd\xef\x09\x2e\x94",
		.assoc	= "\x02\x65\x78\x3c\xe9\x21\x30\x91"
			  "\xb1\xb9\xda\x76\x9a\x78\x6d\x95"
			  "\xf2\x88\x32\xa3\xf2\x50\xcb\x4c"
			  "\xe3\x00\x73\x69\x84\x69\x87\x79",
		.alen	= 32,
		.input	= "\x9f\xd2\x02\x4b\x52\x49\x31\x3c"
			  "\x43\x69\x3a\x2d\x8e\x70\xad\x7e"
			  "\xe0\xe5\x46\x09\x80\x89\x13\xb2"
			  "\x8c\x8b\xd9\x3f\x86\xfb\xb5\x6b",
		.ilen	= 32,
		.result	= "\x39\xdf\x7c\x3c\x5a\x29\xb9\x62"
			  "\x5d\x51\xc2\x16\xd8\xbd\x06\x9f"
			  "\x9b\x6a\x09\x70\xc1\x51\x83\xc2"
			  "\x66\x88\x1d\x4f\x9a\xda\xe0\x1e"
			  "\xc7\x79\x11\x58\xe5\x6b\x20\x40"
			  "\x7a\xea\x46\x42\x8b\xe4\x6f\xe1",
		.rlen	= 48,
	}, {
		.key	= "\xe0\x8d\x99\x71\x60\xd7\x97\x1a"
			  "\xbd\x01\x99\xd5\x8a\xdf\x71\x3a"
			  "\xd3\xdf\x24\x4b\x5e\x3d\x4b\x4e"
			  "\x30\x7a\xb9\xd8\x53\x0a\x5e\x2b"
			  "\x1e\x29\x91",
		.klen	= 35,
		.iv	= "\xad\x8e\xc1\x53\x0a\xcf\x2d\xbe",
		.assoc	= "\x19\xb6\x1f\x57\xc4\xf3\xf0\x8b"
			  "\x78\x2b\x94\x02\x29\x0f\x42\x27"
			  "\x6b\x75\xcb\x98\x34\x08\x7e\x79"
			  "\xe4\x3e\x49\x0d\x84\x8b\x22\x87",
		.alen	= 32,
		.input	= "\xe1\xd9\xd8\x13\xeb\x3a\x75\x3f"
			  "\x9d\xbd\x5f\x66\xbe\xdc\xbb\x66"
			  "\xbf\x17\x99\x62\x4a\x39\x27\x1f"
			  "\x1d\xdc\x24\xae\x19\x2f\x98\x4c",
		.ilen	= 32,
		.result	= "\x19\xb8\x61\x33\x45\x2b\x43\x96"
			  "\x6f\x51\xd0\x20\x30\x7d\x9b\xc6"
			  "\x26\x3d\xf8\xc9\x65\x16\xa8\x9f"
			  "\xf0\x62\x17\x34\xf2\x1e\x8d\x75"
			  "\x4e\x13\xcc\xc0\xc3\x2a\x54\x2d",
		.rlen	= 40,
	}, {
		.key	= "\x7c\xc8\x18\x3b\x8d\x99\xe0\x7c"
			  "\x45\x41\xb8\xbd\x5c\xa7\xc2\x32"
			  "\x8a\xb8\x02\x59\xa4\xfe\xa9\x2c"
			  "\x09\x75\x9a\x9b\x3c\x9b\x27\x39"
			  "\xf9\xd9\x4e",
		.klen	= 35,
		.iv	= "\x63\xb5\x3d\x9d\x43\xf6\x1e\x50",
		.assoc	= "\x57\xf5\x6b\x8b\x57\x5c\x3d\x3b"
			  "\x13\x02\x01\x0c\x83\x4c\x96\x35"
			  "\x8e\xd6\x39\xcf\x7d\x14\x9b\x94"
			  "\xb0\x39\x36\xe6\x8f\x57\xe0\x13",
		.alen	= 32,
		.input	= "\x3b\x6c\x29\x36\xb6\xef\x07\xa6"
			  "\x83\x72\x07\x4f\xcf\xfa\x66\x89"
			  "\x5f\xca\xb1\xba\xd5\x8f\x2c\x27"
			  "\x30\xdb\x75\x09\x93\xd4\x65\xe4",
		.ilen	= 32,
		.result	= "\xb0\x88\x5a\x33\xaa\xe5\xc7\x1d"
			  "\x85\x23\xc7\xc6\x2f\xf4\x1e\x3d"
			  "\xcc\x63\x44\x25\x07\x78\x4f\x9e"
			  "\x96\xb8\x88\xeb\xbc\x48\x1f\x06"
			  "\x39\xaf\x39\xac\xd8\x4a\x80\x39"
			  "\x7b\x72\x8a\xf7",
		.rlen	= 44,
	}, {
		.key	= "\xab\xd0\xe9\x33\x07\x26\xe5\x83"
			  "\x8c\x76\x95\xd4\xb6\xdc\xf3\x46"
			  "\xf9\x8f\xad\xe3\x02\x13\x83\x77"
			  "\x3f\xb0\xf1\xa1\xa1\x22\x0f\x2b"
			  "\x24\xa7\x8b",
		.klen	= 35,
		.iv	= "\x07\xcb\xcc\x0e\xe6\x33\xbf\xf5",
		.assoc	= "\xd4\xdb\x30\x1d\x03\xfe\xfd\x5f"
			  "\x87\xd4\x8c\xb6\xb6\xf1\x7a\x5d"
			  "\xab\x90\x65\x8d\x8e\xca\x4d\x4f"
			  "\x16\x0c\x40\x90\x4b\xc7\x36\x73",
		.alen	= 32,
		.input	= "\xf5\xc6\x7d\x48\xc1\xb7\xe6\x92"
			  "\x97\x5a\xca\xc4\xa9\x6d\xf9\x3d"
			  "\x6c\xde\xbc\xf1\x90\xea\x6a\xb2"
			  "\x35\x86\x36\xaf\x5c\xfe\x4b\x3a",
		.ilen	= 32,
		.result	= "\x83\x6f\x40\x87\x72\xcf\xc1\x13"
			  "\xef\xbb\x80\x21\x04\x6c\x58\x09"
			  "\x07\x1b\xfc\xdf\xc0\x3f\x5b\xc7"
			  "\xe0\x79\xa8\x6e\x71\x7c\x3f\xcf"
			  "\x5c\xda\xb2\x33\xe5\x13\xe2\x0d"
			  "\x74\xd1\xef\xb5\x0f\x3a\xb5\xf8",
		.rlen	= 48,
	},
};

static struct aead_testvec aes_ccm_rfc4309_dec_tv_template[] = {
	{
		.key	= "\xab\x2f\x8a\x74\xb7\x1c\xd2\xb1"
			  "\xff\x80\x2e\x48\x7d\x82\xf8\xb9"
			  "\xc6\xfb\x7d",
		.klen	= 19,
		.iv	= "\x80\x0d\x13\xab\xd8\xa6\xb2\xd8",
		.alen	= 0,
		.input	= "\xd5\xe8\x93\x9f\xc7\x89\x2e\x2b",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
		.novrfy	= 1,
	}, {
		.key	= "\xab\x2f\x8a\x74\xb7\x1c\xd2\xb1"
			  "\xff\x80\x2e\x48\x7d\x82\xf8\xb9"
			  "\xaf\x94\x87",
		.klen	= 19,
		.iv	= "\x78\x35\x82\x81\x7f\x88\x94\x68",
		.alen	= 0,
		.input	= "\x41\x3c\xb8\x87\x73\xcb\xf3\xf3",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
	}, {
		.key	= "\x61\x0e\x8c\xae\xe3\x23\xb6\x38"
			  "\x76\x1c\xf6\x3a\x67\xa3\x9c\xd8"
			  "\xc6\xfb\x7d",
		.klen	= 19,
		.iv	= "\x80\x0d\x13\xab\xd8\xa6\xb2\xd8",
		.assoc	= "\xf3\x94\x87\x78\x35\x82\x81\x7f"
			  "\x88\x94\x68\xb1\x78\x6b\x2b\xd6"
			  "\x04\x1f\x4e\xed\x78\xd5\x33\x66"
			  "\xd8\x94\x99\x91\x81\x54\x62\x57",
		.alen	= 32,
		.input	= "\xf0\x7c\x29\x02\xae\x1c\x2f\x55"
			  "\xd0\xd1\x3d\x1a\xa3\x6d\xe4\x0a"
			  "\x86\xb0\x87\x6b\x62\x33\x8c\x34"
			  "\xce\xab\x57\xcc\x79\x0b\xe0\x6f"
			  "\x5c\x3e\x48\x1f\x6c\x46\xf7\x51"
			  "\x8b\x84\x83\x2a\xc1\x05\xb8\xc5",
		.ilen	= 48,
		.result	= "\x50\x82\x3e\x07\xe2\x1e\xb6\xfb"
			  "\x33\xe4\x73\xce\xd2\xfb\x95\x79"
			  "\xe8\xb4\xb5\x77\x11\x10\x62\x6f"
			  "\x6a\x82\xd1\x13\xec\xf5\xd0\x48",
		.rlen	= 32,
		.novrfy	= 1,
	}, {
		.key	= "\x61\x0e\x8c\xae\xe3\x23\xb6\x38"
			  "\x76\x1c\xf6\x3a\x67\xa3\x9c\xd8"
			  "\x05\xe0\xc9",
		.klen	= 19,
		.iv	= "\x0f\xed\x34\xea\x97\xd4\x3b\xdf",
		.assoc	= "\x49\x5c\x50\x1f\x1d\x94\xcc\x81"
			  "\xba\xb7\xb6\x03\xaf\xa5\xc1\xa1"
			  "\xd8\x5c\x42\x68\xe0\x6c\xda\x89"
			  "\x05\xac\x56\xac\x1b\x2a\xd3\x86",
		.alen	= 32,
		.input	= "\x39\xbe\x7d\x15\x62\x77\xf3\x3c"
			  "\xad\x83\x52\x6d\x71\x03\x25\x1c"
			  "\xed\x81\x3a\x9a\x16\x7d\x19\x80"
			  "\x72\x04\x72\xd0\xf6\xff\x05\x0f"
			  "\xb7\x14\x30\x00\x32\x9e\xa0\xa6"
			  "\x9e\x5a\x18\xa1\xb8\xfe\xdb\xd3",
		.ilen	= 48,
		.result	= "\x75\x05\xbe\xc2\xd9\x1e\xde\x60"
			  "\x47\x3d\x8c\x7d\xbd\xb5\xd9\xb7"
			  "\xf2\xae\x61\x05\x8f\x82\x24\x3f"
			  "\x9c\x67\x91\xe1\x38\x4f\xe4\x0c",
		.rlen	= 32,
	}, {
		.key	= "\x39\xbb\xa7\xbe\x59\x97\x9e\x73"
			  "\xa2\xbc\x6b\x98\xd7\x75\x7f\xe3"
			  "\xa4\x48\x93\x39\x26\x71\x4a\xc6"
			  "\xee\x49\x83",
		.klen	= 27,
		.iv	= "\xe9\xa9\xff\xe9\x57\xba\xfd\x9e",
		.assoc	= "\x44\xa6\x2c\x05\xe9\xe1\x43\xb1"
			  "\x58\x7c\xf2\x5c\x6d\x39\x0a\x64"
			  "\xa4\xf0\x13\x05\xd1\x77\x99\x67"
			  "\x11\xc4\xc6\xdb\x00\x56\x36\x61",
		.alen	= 32,
		.input	= "\x71\x99\xfa\xf4\x44\x12\x68\x9b",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
	}, {
		.key	= "\x58\x5d\xa0\x96\x65\x1a\x04\xd7"
			  "\x96\xe5\xc5\x68\xaa\x95\x35\xe0"
			  "\x29\xa0\xba\x9e\x48\x78\xd1\xba"
			  "\xee\x49\x83",
		.klen	= 27,
		.iv	= "\xe9\xa9\xff\xe9\x57\xba\xfd\x9e",
		.assoc	= "\x44\xa6\x2c\x05\xe9\xe1\x43\xb1"
			  "\x58\x7c\xf2\x5c\x6d\x39\x0a\x64"
			  "\xa4\xf0\x13\x05\xd1\x77\x99\x67"
			  "\x11\xc4\xc6\xdb\x00\x56\x36\x61",
		.alen	= 32,
		.input	= "\xfb\xe5\x5d\x34\xbe\xe5\xe8\xe7"
			  "\x5a\xef\x2f\xbf\x1f\x7f\xd4\xb2"
			  "\x66\xca\x61\x1e\x96\x7a\x61\xb3"
			  "\x1c\x16\x45\x52\xba\x04\x9c\x9f"
			  "\xb1\xd2\x40\xbc\x52\x7c\x6f\xb1",
		.ilen	= 40,
		.result	= "\x85\x34\x66\x42\xc8\x92\x0f\x36"
			  "\x58\xe0\x6b\x91\x3c\x98\x5c\xbb"
			  "\x0a\x85\xcc\x02\xad\x7a\x96\xe9"
			  "\x65\x43\xa4\xc3\x0f\xdc\x55\x81",
		.rlen	= 32,
	}, {
		.key	= "\x58\x5d\xa0\x96\x65\x1a\x04\xd7"
			  "\x96\xe5\xc5\x68\xaa\x95\x35\xe0"
			  "\x29\xa0\xba\x9e\x48\x78\xd1\xba"
			  "\xd1\xfc\x57",
		.klen	= 27,
		.iv	= "\x9c\xfe\xb8\x9c\xad\x71\xaa\x1f",
		.assoc	= "\x86\x67\xa5\xa9\x14\x5f\x0d\xc6"
			  "\xff\x14\xc7\x44\xbf\x6c\x3a\xc3"
			  "\xff\xb6\x81\xbd\xe2\xd5\x06\xc7"
			  "\x3c\xa1\x52\x13\x03\x8a\x23\x3a",
		.alen	= 32,
		.input	= "\x3f\x66\xb0\x9d\xe5\x4b\x38\x00"
			  "\xc6\x0e\x6e\xe5\xd6\x98\xa6\x37"
			  "\x8c\x26\x33\xc6\xb2\xa2\x17\xfa"
			  "\x64\x19\xc0\x30\xd7\xfc\x14\x6b"
			  "\xe3\x33\xc2\x04\xb0\x37\xbe\x3f"
			  "\xa9\xb4\x2d\x68\x03\xa3\x44\xef",
		.ilen	= 48,
		.result	= "\x02\x87\x4d\x28\x80\x6e\xb2\xed"
			  "\x99\x2a\xa8\xca\x04\x25\x45\x90"
			  "\x1d\xdd\x5a\xd9\xe4\xdb\x9c\x9c"
			  "\x49\xe9\x01\xfe\xa7\x80\x6d\x6b",
		.rlen	= 32,
		.novrfy	= 1,
	}, {
		.key	= "\xa4\x4b\x54\x29\x0a\xb8\x6d\x01"
			  "\x5b\x80\x2a\xcf\x25\xc4\xb7\x5c"
			  "\x20\x2c\xad\x30\xc2\x2b\x41\xfb"
			  "\x0e\x85\xbc\x33\xad\x0f\x2b\xff"
			  "\xee\x49\x83",
		.klen	= 35,
		.iv	= "\xe9\xa9\xff\xe9\x57\xba\xfd\x9e",
		.alen	= 0,
		.input	= "\x1f\xb8\x8f\xa3\xdd\x54\x00\xf2",
		.ilen	= 8,
		.result	= "\x00",
		.rlen	= 0,
	}, {
		.key	= "\x39\xbb\xa7\xbe\x59\x97\x9e\x73"
			  "\xa2\xbc\x6b\x98\xd7\x75\x7f\xe3"
			  "\xa4\x48\x93\x39\x26\x71\x4a\xc6"
			  "\xae\x8f\x11\x4c\xc2\x9c\x4a\xbb"
			  "\x85\x34\x66",
		.klen	= 35,
		.iv	= "\x42\xc8\x92\x0f\x36\x58\xe0\x6b",
		.alen	= 0,
		.input	= "\x48\x01\x5e\x02\x24\x04\x66\x47"
			  "\xa1\xea\x6f\xaf\xe8\xfc\xfb\xdd"
			  "\xa5\xa9\x87\x8d\x84\xee\x2e\x77"
			  "\xbb\x86\xb9\xf5\x5c\x6c\xff\xf6"
			  "\x72\xc3\x8e\xf7\x70\xb1\xb2\x07"
			  "\xbc\xa8\xa3\xbd\x83\x7c\x1d\x2a",
		.ilen	= 48,
		.result	= "\xdc\x56\xf2\x71\xb0\xb1\xa0\x6c"
			  "\xf0\x97\x3a\xfb\x6d\xe7\x32\x99"
			  "\x3e\xaf\x70\x5e\xb2\x4d\xea\x39"
			  "\x89\xd4\x75\x7a\x63\xb1\xda\x93",
		.rlen	= 32,
		.novrfy	= 1,
	}, {
		.key	= "\x58\x5d\xa0\x96\x65\x1a\x04\xd7"
			  "\x96\xe5\xc5\x68\xaa\x95\x35\xe0"
			  "\x29\xa0\xba\x9e\x48\x78\xd1\xba"
			  "\x0d\x1a\x53\x3b\xb5\xe3\xf8\x8b"
			  "\xcf\x76\x3f",
		.klen	= 35,
		.iv	= "\xd9\x95\x75\x8f\x44\x89\x40\x7b",
		.assoc	= "\x8f\x86\x6c\x4d\x1d\xc5\x39\x88"
			  "\xc8\xf3\x5c\x52\x10\x63\x6f\x2b"
			  "\x8a\x2a\xc5\x6f\x30\x23\x58\x7b"
			  "\xfb\x36\x03\x11\xb4\xd9\xf2\xfe",
		.alen	= 32,
		.input	= "\x48\x58\xd6\xf3\xad\x63\x58\xbf"
			  "\xae\xc7\x5e\xae\x83\x8f\x7b\xe4"
			  "\x78\x5c\x4c\x67\x71\x89\x94\xbf"
			  "\x47\xf1\x63\x7e\x1c\x59\xbd\xc5"
			  "\x7f\x44\x0a\x0c\x01\x18\x07\x92"
			  "\xe1\xd3\x51\xce\x32\x6d\x0c\x5b",
		.ilen	= 48,
		.result	= "\xc2\x54\xc8\xde\x78\x87\x77\x40"
			  "\x49\x71\xe4\xb7\xe7\xcb\x76\x61"
			  "\x0a\x41\xb9\xe9\xc0\x76\x54\xab"
			  "\x04\x49\x3b\x19\x93\x57\x25\x5d",
		.rlen	= 32,
	},
};

#define ANSI_CPRNG_AES_TEST_VECTORS	6

static struct cprng_testvec ansi_cprng_aes_tv_template[] = {
	{
		.key	= "\xf3\xb1\x66\x6d\x13\x60\x72\x42"
			  "\xed\x06\x1c\xab\xb8\xd4\x62\x02",
		.klen	= 16,
		.dt	= "\xe6\xb3\xbe\x78\x2a\x23\xfa\x62"
			  "\xd7\x1d\x4a\xfb\xb0\xe9\x22\xf9",
		.dtlen	= 16,
		.v	= "\x80\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.vlen	= 16,
		.result	= "\x59\x53\x1e\xd1\x3b\xb0\xc0\x55"
			  "\x84\x79\x66\x85\xc1\x2f\x76\x41",
		.rlen	= 16,
		.loops	= 1,
	}, {
		.key	= "\xf3\xb1\x66\x6d\x13\x60\x72\x42"
			  "\xed\x06\x1c\xab\xb8\xd4\x62\x02",
		.klen	= 16,
		.dt	= "\xe6\xb3\xbe\x78\x2a\x23\xfa\x62"
			  "\xd7\x1d\x4a\xfb\xb0\xe9\x22\xfa",
		.dtlen	= 16,
		.v	= "\xc0\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.vlen	= 16,
		.result	= "\x7c\x22\x2c\xf4\xca\x8f\xa2\x4c"
			  "\x1c\x9c\xb6\x41\xa9\xf3\x22\x0d",
		.rlen	= 16,
		.loops	= 1,
	}, {
		.key	= "\xf3\xb1\x66\x6d\x13\x60\x72\x42"
			  "\xed\x06\x1c\xab\xb8\xd4\x62\x02",
		.klen	= 16,
		.dt	= "\xe6\xb3\xbe\x78\x2a\x23\xfa\x62"
			  "\xd7\x1d\x4a\xfb\xb0\xe9\x22\xfb",
		.dtlen	= 16,
		.v	= "\xe0\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.vlen	= 16,
		.result	= "\x8a\xaa\x00\x39\x66\x67\x5b\xe5"
			  "\x29\x14\x28\x81\xa9\x4d\x4e\xc7",
		.rlen	= 16,
		.loops	= 1,
	}, {
		.key	= "\xf3\xb1\x66\x6d\x13\x60\x72\x42"
			  "\xed\x06\x1c\xab\xb8\xd4\x62\x02",
		.klen	= 16,
		.dt	= "\xe6\xb3\xbe\x78\x2a\x23\xfa\x62"
			  "\xd7\x1d\x4a\xfb\xb0\xe9\x22\xfc",
		.dtlen	= 16,
		.v	= "\xf0\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.vlen	= 16,
		.result	= "\x88\xdd\xa4\x56\x30\x24\x23\xe5"
			  "\xf6\x9d\xa5\x7e\x7b\x95\xc7\x3a",
		.rlen	= 16,
		.loops	= 1,
	}, {
		.key	= "\xf3\xb1\x66\x6d\x13\x60\x72\x42"
			  "\xed\x06\x1c\xab\xb8\xd4\x62\x02",
		.klen	= 16,
		.dt	= "\xe6\xb3\xbe\x78\x2a\x23\xfa\x62"
			  "\xd7\x1d\x4a\xfb\xb0\xe9\x22\xfd",
		.dtlen	= 16,
		.v	= "\xf8\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.vlen	= 16,
		.result	= "\x05\x25\x92\x46\x61\x79\xd2\xcb"
			  "\x78\xc4\x0b\x14\x0a\x5a\x9a\xc8",
		.rlen	= 16,
		.loops	= 1,
	}, {	 
		.key	= "\x9f\x5b\x51\x20\x0b\xf3\x34\xb5"
			  "\xd8\x2b\xe8\xc3\x72\x55\xc8\x48",
		.klen	= 16,
		.dt	= "\x63\x76\xbb\xe5\x29\x02\xba\x3b"
			  "\x67\xc9\x25\xfa\x70\x1f\x11\xac",
		.dtlen	= 16,
		.v	= "\x57\x2c\x8e\x76\x87\x26\x47\x97"
			  "\x7e\x74\xfb\xdd\xc4\x95\x01\xd1",
		.vlen	= 16,
		.result	= "\x48\xe9\xbd\x0d\x06\xee\x18\xfb"
			  "\xe4\x57\x90\xd5\xc3\xfc\x9b\x73",
		.rlen	= 16,
		.loops	= 10000,
	},
};

#define CAST5_ENC_TEST_VECTORS		4
#define CAST5_DEC_TEST_VECTORS		4
#define CAST5_CBC_ENC_TEST_VECTORS	1
#define CAST5_CBC_DEC_TEST_VECTORS	1
#define CAST5_CTR_ENC_TEST_VECTORS	2
#define CAST5_CTR_DEC_TEST_VECTORS	2

static struct cipher_testvec cast5_enc_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x12\x34\x56\x78"
			  "\x23\x45\x67\x89\x34\x56\x78\x9a",
		.klen	= 16,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.ilen	= 8,
		.result	= "\x23\x8b\x4f\xe5\x84\x7e\x44\xb2",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x12\x34\x56\x78"
			  "\x23\x45",
		.klen	= 10,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.ilen	= 8,
		.result	= "\xeb\x6a\x71\x1a\x2c\x02\x27\x1b",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x12",
		.klen	= 5,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.ilen	= 8,
		.result	= "\x7a\xc8\x16\xd1\x6e\x9b\x30\x2e",
		.rlen	= 8,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x8D\xFC\x81\x9C\xCB\xAA\x5A\x1C"
			  "\x7E\x95\xCF\x40\xAB\x4D\x6F\xEA"
			  "\xD3\xD9\xB0\x9A\xB7\xC7\xE0\x2E"
			  "\xD1\x39\x34\x92\x8F\xFA\x14\xF1"
			  "\xD5\xD2\x7B\x59\x1F\x35\x28\xC2"
			  "\x20\xD9\x42\x06\xC9\x0B\x10\x04"
			  "\xF8\x79\xCD\x32\x86\x75\x4C\xB6"
			  "\x7B\x1C\x52\xB1\x91\x64\x22\x4B"
			  "\x13\xC7\xAE\x98\x0E\xB5\xCF\x6F"
			  "\x3F\xF4\x43\x96\x73\x0D\xA2\x05"
			  "\xDB\xFD\x28\x90\x2C\x56\xB9\x37"
			  "\x5B\x69\x0C\xAD\x84\x67\xFF\x15"
			  "\x4A\xD4\xA7\xD3\xDD\x99\x47\x3A"
			  "\xED\x34\x35\x78\x6B\x91\xC9\x32"
			  "\xE1\xBF\xBC\xB4\x04\x85\x6A\x39"
			  "\xC0\xBA\x51\xD0\x0F\x4E\xD1\xE2"
			  "\x1C\xFD\x0E\x05\x07\xF4\x10\xED"
			  "\xA2\x17\xFF\xF5\x64\xC6\x1A\x22"
			  "\xAD\x78\xE7\xD7\x11\xE9\x99\xB9"
			  "\xAA\xEC\x6F\xF8\x3B\xBF\xCE\x77"
			  "\x93\xE8\xAD\x1D\x50\x6C\xAE\xBC"
			  "\xBA\x5C\x80\xD1\x91\x65\x51\x1B"
			  "\xE8\x0A\xCD\x99\x96\x71\x3D\xB6"
			  "\x78\x75\x37\x55\xC1\xF5\x90\x40"
			  "\x34\xF4\x7E\xC8\xCC\x3A\x5F\x6E"
			  "\x36\xA1\xA1\xC2\x3A\x72\x42\x8E"
			  "\x0E\x37\x88\xE8\xCE\x83\xCB\xAD"
			  "\xE0\x69\x77\x50\xC7\x0C\x99\xCA"
			  "\x19\x5B\x30\x25\x9A\xEF\x9B\x0C"
			  "\xEF\x8F\x74\x4C\xCF\x49\x4E\xB9"
			  "\xC5\xAE\x9E\x2E\x78\x9A\xB9\x48"
			  "\xD5\x81\xE4\x37\x1D\xBF\x27\xD9"
			  "\xC5\xD6\x65\x43\x45\x8C\xBB\xB6"
			  "\x55\xF4\x06\xBB\x49\x53\x8B\x1B"
			  "\x07\xA9\x96\x69\x5B\xCB\x0F\xBC"
			  "\x93\x85\x90\x0F\x0A\x68\x40\x2A"
			  "\x95\xED\x2D\x88\xBF\x71\xD0\xBB"
			  "\xEC\xB0\x77\x6C\x79\xFC\x3C\x05"
			  "\x49\x3F\xB8\x24\xEF\x8E\x09\xA2"
			  "\x1D\xEF\x92\x02\x96\xD4\x7F\xC8"
			  "\x03\xB2\xCA\xDB\x17\x5C\x52\xCF"
			  "\xDD\x70\x37\x63\xAA\xA5\x83\x20"
			  "\x52\x02\xF6\xB9\xE7\x6E\x0A\xB6"
			  "\x79\x03\xA0\xDA\xA3\x79\x21\xBD"
			  "\xE3\x37\x3A\xC0\xF7\x2C\x32\xBE"
			  "\x8B\xE8\xA6\x00\xC7\x32\xD5\x06"
			  "\xBB\xE3\xAB\x06\x21\x82\xB8\x32"
			  "\x31\x34\x2A\xA7\x1F\x64\x99\xBF"
			  "\xFA\xDA\x3D\x75\xF7\x48\xD5\x48"
			  "\x4B\x52\x7E\xF6\x7C\xAB\x67\x59"
			  "\xC5\xDC\xA8\xC6\x63\x85\x4A\xDF"
			  "\xF0\x40\x5F\xCF\xE3\x58\x52\x67"
			  "\x7A\x24\x32\xC5\xEC\x9E\xA9\x6F"
			  "\x58\x56\xDD\x94\x1F\x71\x8D\xF4"
			  "\x6E\xFF\x2C\xA7\xA5\xD8\xBA\xAF"
			  "\x1D\x8B\xA2\x46\xB5\xC4\x9F\x57"
			  "\x8D\xD8\xB3\x3C\x02\x0D\xBB\x84"
			  "\xC7\xBD\xB4\x9A\x6E\xBB\xB1\x37"
			  "\x95\x79\xC4\xA7\xEA\x1D\xDC\x33"
			  "\x5D\x0B\x3F\x03\x8F\x30\xF9\xAE"
			  "\x4F\xFE\x24\x9C\x9A\x02\xE5\x57"
			  "\xF5\xBC\x25\xD6\x02\x56\x57\x1C",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast5_dec_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x12\x34\x56\x78"
			  "\x23\x45\x67\x89\x34\x56\x78\x9a",
		.klen	= 16,
		.input	= "\x23\x8b\x4f\xe5\x84\x7e\x44\xb2",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x12\x34\x56\x78"
			  "\x23\x45",
		.klen	= 10,
		.input	= "\xeb\x6a\x71\x1a\x2c\x02\x27\x1b",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x12",
		.klen	= 5,
		.input	= "\x7a\xc8\x16\xd1\x6e\x9b\x30\x2e",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.rlen	= 8,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x8D\xFC\x81\x9C\xCB\xAA\x5A\x1C"
			  "\x7E\x95\xCF\x40\xAB\x4D\x6F\xEA"
			  "\xD3\xD9\xB0\x9A\xB7\xC7\xE0\x2E"
			  "\xD1\x39\x34\x92\x8F\xFA\x14\xF1"
			  "\xD5\xD2\x7B\x59\x1F\x35\x28\xC2"
			  "\x20\xD9\x42\x06\xC9\x0B\x10\x04"
			  "\xF8\x79\xCD\x32\x86\x75\x4C\xB6"
			  "\x7B\x1C\x52\xB1\x91\x64\x22\x4B"
			  "\x13\xC7\xAE\x98\x0E\xB5\xCF\x6F"
			  "\x3F\xF4\x43\x96\x73\x0D\xA2\x05"
			  "\xDB\xFD\x28\x90\x2C\x56\xB9\x37"
			  "\x5B\x69\x0C\xAD\x84\x67\xFF\x15"
			  "\x4A\xD4\xA7\xD3\xDD\x99\x47\x3A"
			  "\xED\x34\x35\x78\x6B\x91\xC9\x32"
			  "\xE1\xBF\xBC\xB4\x04\x85\x6A\x39"
			  "\xC0\xBA\x51\xD0\x0F\x4E\xD1\xE2"
			  "\x1C\xFD\x0E\x05\x07\xF4\x10\xED"
			  "\xA2\x17\xFF\xF5\x64\xC6\x1A\x22"
			  "\xAD\x78\xE7\xD7\x11\xE9\x99\xB9"
			  "\xAA\xEC\x6F\xF8\x3B\xBF\xCE\x77"
			  "\x93\xE8\xAD\x1D\x50\x6C\xAE\xBC"
			  "\xBA\x5C\x80\xD1\x91\x65\x51\x1B"
			  "\xE8\x0A\xCD\x99\x96\x71\x3D\xB6"
			  "\x78\x75\x37\x55\xC1\xF5\x90\x40"
			  "\x34\xF4\x7E\xC8\xCC\x3A\x5F\x6E"
			  "\x36\xA1\xA1\xC2\x3A\x72\x42\x8E"
			  "\x0E\x37\x88\xE8\xCE\x83\xCB\xAD"
			  "\xE0\x69\x77\x50\xC7\x0C\x99\xCA"
			  "\x19\x5B\x30\x25\x9A\xEF\x9B\x0C"
			  "\xEF\x8F\x74\x4C\xCF\x49\x4E\xB9"
			  "\xC5\xAE\x9E\x2E\x78\x9A\xB9\x48"
			  "\xD5\x81\xE4\x37\x1D\xBF\x27\xD9"
			  "\xC5\xD6\x65\x43\x45\x8C\xBB\xB6"
			  "\x55\xF4\x06\xBB\x49\x53\x8B\x1B"
			  "\x07\xA9\x96\x69\x5B\xCB\x0F\xBC"
			  "\x93\x85\x90\x0F\x0A\x68\x40\x2A"
			  "\x95\xED\x2D\x88\xBF\x71\xD0\xBB"
			  "\xEC\xB0\x77\x6C\x79\xFC\x3C\x05"
			  "\x49\x3F\xB8\x24\xEF\x8E\x09\xA2"
			  "\x1D\xEF\x92\x02\x96\xD4\x7F\xC8"
			  "\x03\xB2\xCA\xDB\x17\x5C\x52\xCF"
			  "\xDD\x70\x37\x63\xAA\xA5\x83\x20"
			  "\x52\x02\xF6\xB9\xE7\x6E\x0A\xB6"
			  "\x79\x03\xA0\xDA\xA3\x79\x21\xBD"
			  "\xE3\x37\x3A\xC0\xF7\x2C\x32\xBE"
			  "\x8B\xE8\xA6\x00\xC7\x32\xD5\x06"
			  "\xBB\xE3\xAB\x06\x21\x82\xB8\x32"
			  "\x31\x34\x2A\xA7\x1F\x64\x99\xBF"
			  "\xFA\xDA\x3D\x75\xF7\x48\xD5\x48"
			  "\x4B\x52\x7E\xF6\x7C\xAB\x67\x59"
			  "\xC5\xDC\xA8\xC6\x63\x85\x4A\xDF"
			  "\xF0\x40\x5F\xCF\xE3\x58\x52\x67"
			  "\x7A\x24\x32\xC5\xEC\x9E\xA9\x6F"
			  "\x58\x56\xDD\x94\x1F\x71\x8D\xF4"
			  "\x6E\xFF\x2C\xA7\xA5\xD8\xBA\xAF"
			  "\x1D\x8B\xA2\x46\xB5\xC4\x9F\x57"
			  "\x8D\xD8\xB3\x3C\x02\x0D\xBB\x84"
			  "\xC7\xBD\xB4\x9A\x6E\xBB\xB1\x37"
			  "\x95\x79\xC4\xA7\xEA\x1D\xDC\x33"
			  "\x5D\x0B\x3F\x03\x8F\x30\xF9\xAE"
			  "\x4F\xFE\x24\x9C\x9A\x02\xE5\x57"
			  "\xF5\xBC\x25\xD6\x02\x56\x57\x1C",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast5_cbc_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\x05\x28\xCE\x61\x90\x80\xE1\x78"
			  "\xB9\x2A\x97\x7C\xB0\x83\xD8\x1A"
			  "\xDE\x58\x7F\xD7\xFD\x72\xB8\xFB"
			  "\xDA\xF0\x6E\x77\x14\x47\x82\xBA"
			  "\x29\x0E\x25\x6E\xB4\x39\xD9\x7F"
			  "\x05\xA7\xA7\x3A\xC1\x5D\x9E\x39"
			  "\xA7\xFB\x0D\x05\x00\xF3\x58\x67"
			  "\x60\xEC\x73\x77\x46\x85\x9B\x6A"
			  "\x08\x3E\xBE\x59\xFB\xE4\x96\x34"
			  "\xB4\x05\x49\x1A\x97\x43\xAD\xA0"
			  "\xA9\x1E\x6E\x74\xF1\x94\xEC\xA8"
			  "\xB5\x8A\x20\xEA\x89\x6B\x19\xAA"
			  "\xA7\xF1\x33\x67\x90\x23\x0D\xEE"
			  "\x81\xD5\x78\x4F\xD3\x63\xEA\x46"
			  "\xB5\xB2\x6E\xBB\xCA\x76\x06\x10"
			  "\x96\x2A\x0A\xBA\xF9\x41\x5A\x1D"
			  "\x36\x7C\x56\x14\x54\x83\xFA\xA1"
			  "\x27\xDD\xBA\x8A\x90\x29\xD6\xA6"
			  "\xFA\x48\x3E\x1E\x23\x6E\x98\xA8"
			  "\xA7\xD9\x67\x92\x5C\x13\xB4\x71"
			  "\xA8\xAA\x89\x4A\xA4\xB3\x49\x7C"
			  "\x7D\x7F\xCE\x6F\x29\x2E\x7E\x37"
			  "\xC8\x52\x60\xD9\xE7\xCA\x60\x98"
			  "\xED\xCD\xE8\x60\x83\xAD\x34\x4D"
			  "\x96\x4A\x99\x2B\xB7\x14\x75\x66"
			  "\x6C\x2C\x1A\xBA\x4B\xBB\x49\x56"
			  "\xE1\x86\xA2\x0E\xD0\xF0\x07\xD3"
			  "\x18\x38\x09\x9C\x0E\x8B\x86\x07"
			  "\x90\x12\x37\x49\x27\x98\x69\x18"
			  "\xB0\xCC\xFB\xD3\xBD\x04\xA0\x85"
			  "\x4B\x22\x97\x07\xB6\x97\xE9\x95"
			  "\x0F\x88\x36\xA9\x44\x00\xC6\xE9"
			  "\x27\x53\x5C\x5B\x1F\xD3\xE2\xEE"
			  "\xD0\xCD\x63\x30\xA9\xC0\xDD\x49"
			  "\xFE\x16\xA4\x07\x0D\xE2\x5D\x97"
			  "\xDE\x89\xBA\x2E\xF3\xA9\x5E\xBE"
			  "\x03\x55\x0E\x02\x41\x4A\x45\x06"
			  "\xBE\xEA\x32\xF2\xDC\x91\x5C\x20"
			  "\x94\x02\x30\xD2\xFC\x29\xFA\x8E"
			  "\x34\xA0\x31\xB8\x34\xBA\xAE\x54"
			  "\xB5\x88\x1F\xDC\x43\xDC\x22\x9F"
			  "\xDC\xCE\xD3\xFA\xA4\xA8\xBC\x8A"
			  "\xC7\x5A\x43\x21\xA5\xB1\xDB\xC3"
			  "\x84\x3B\xB4\x9B\xB5\xA7\xF1\x0A"
			  "\xB6\x37\x21\x19\x55\xC2\xBD\x99"
			  "\x49\x24\xBB\x7C\xB3\x8E\xEF\xD2"
			  "\x3A\xCF\xA0\x31\x28\x0E\x25\xA2"
			  "\x11\xB4\x18\x17\x1A\x65\x92\x56"
			  "\xE8\xE0\x52\x9C\x61\x18\x2A\xB1"
			  "\x1A\x01\x22\x45\x17\x62\x52\x6C"
			  "\x91\x44\xCF\x98\xC7\xC0\x79\x26"
			  "\x32\x66\x6F\x23\x7F\x94\x36\x88"
			  "\x3C\xC9\xD0\xB7\x45\x30\x31\x86"
			  "\x3D\xC6\xA3\x98\x62\x84\x1A\x8B"
			  "\x16\x88\xC7\xA3\xE9\x4F\xE0\x86"
			  "\xA4\x93\xA8\x34\x5A\xCA\xDF\xCA"
			  "\x46\x38\xD2\xF4\xE0\x2D\x1E\xC9"
			  "\x7C\xEF\x53\xB7\x60\x72\x41\xBF"
			  "\x29\x00\x87\x02\xAF\x44\x4C\xB7"
			  "\x8C\xF5\x3F\x19\xF4\x80\x45\xA7"
			  "\x15\x5F\xDB\xE9\xB1\x83\xD2\xE6"
			  "\x1D\x18\x66\x44\x5B\x8F\x14\xEB",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast5_cbc_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x05\x28\xCE\x61\x90\x80\xE1\x78"
			  "\xB9\x2A\x97\x7C\xB0\x83\xD8\x1A"
			  "\xDE\x58\x7F\xD7\xFD\x72\xB8\xFB"
			  "\xDA\xF0\x6E\x77\x14\x47\x82\xBA"
			  "\x29\x0E\x25\x6E\xB4\x39\xD9\x7F"
			  "\x05\xA7\xA7\x3A\xC1\x5D\x9E\x39"
			  "\xA7\xFB\x0D\x05\x00\xF3\x58\x67"
			  "\x60\xEC\x73\x77\x46\x85\x9B\x6A"
			  "\x08\x3E\xBE\x59\xFB\xE4\x96\x34"
			  "\xB4\x05\x49\x1A\x97\x43\xAD\xA0"
			  "\xA9\x1E\x6E\x74\xF1\x94\xEC\xA8"
			  "\xB5\x8A\x20\xEA\x89\x6B\x19\xAA"
			  "\xA7\xF1\x33\x67\x90\x23\x0D\xEE"
			  "\x81\xD5\x78\x4F\xD3\x63\xEA\x46"
			  "\xB5\xB2\x6E\xBB\xCA\x76\x06\x10"
			  "\x96\x2A\x0A\xBA\xF9\x41\x5A\x1D"
			  "\x36\x7C\x56\x14\x54\x83\xFA\xA1"
			  "\x27\xDD\xBA\x8A\x90\x29\xD6\xA6"
			  "\xFA\x48\x3E\x1E\x23\x6E\x98\xA8"
			  "\xA7\xD9\x67\x92\x5C\x13\xB4\x71"
			  "\xA8\xAA\x89\x4A\xA4\xB3\x49\x7C"
			  "\x7D\x7F\xCE\x6F\x29\x2E\x7E\x37"
			  "\xC8\x52\x60\xD9\xE7\xCA\x60\x98"
			  "\xED\xCD\xE8\x60\x83\xAD\x34\x4D"
			  "\x96\x4A\x99\x2B\xB7\x14\x75\x66"
			  "\x6C\x2C\x1A\xBA\x4B\xBB\x49\x56"
			  "\xE1\x86\xA2\x0E\xD0\xF0\x07\xD3"
			  "\x18\x38\x09\x9C\x0E\x8B\x86\x07"
			  "\x90\x12\x37\x49\x27\x98\x69\x18"
			  "\xB0\xCC\xFB\xD3\xBD\x04\xA0\x85"
			  "\x4B\x22\x97\x07\xB6\x97\xE9\x95"
			  "\x0F\x88\x36\xA9\x44\x00\xC6\xE9"
			  "\x27\x53\x5C\x5B\x1F\xD3\xE2\xEE"
			  "\xD0\xCD\x63\x30\xA9\xC0\xDD\x49"
			  "\xFE\x16\xA4\x07\x0D\xE2\x5D\x97"
			  "\xDE\x89\xBA\x2E\xF3\xA9\x5E\xBE"
			  "\x03\x55\x0E\x02\x41\x4A\x45\x06"
			  "\xBE\xEA\x32\xF2\xDC\x91\x5C\x20"
			  "\x94\x02\x30\xD2\xFC\x29\xFA\x8E"
			  "\x34\xA0\x31\xB8\x34\xBA\xAE\x54"
			  "\xB5\x88\x1F\xDC\x43\xDC\x22\x9F"
			  "\xDC\xCE\xD3\xFA\xA4\xA8\xBC\x8A"
			  "\xC7\x5A\x43\x21\xA5\xB1\xDB\xC3"
			  "\x84\x3B\xB4\x9B\xB5\xA7\xF1\x0A"
			  "\xB6\x37\x21\x19\x55\xC2\xBD\x99"
			  "\x49\x24\xBB\x7C\xB3\x8E\xEF\xD2"
			  "\x3A\xCF\xA0\x31\x28\x0E\x25\xA2"
			  "\x11\xB4\x18\x17\x1A\x65\x92\x56"
			  "\xE8\xE0\x52\x9C\x61\x18\x2A\xB1"
			  "\x1A\x01\x22\x45\x17\x62\x52\x6C"
			  "\x91\x44\xCF\x98\xC7\xC0\x79\x26"
			  "\x32\x66\x6F\x23\x7F\x94\x36\x88"
			  "\x3C\xC9\xD0\xB7\x45\x30\x31\x86"
			  "\x3D\xC6\xA3\x98\x62\x84\x1A\x8B"
			  "\x16\x88\xC7\xA3\xE9\x4F\xE0\x86"
			  "\xA4\x93\xA8\x34\x5A\xCA\xDF\xCA"
			  "\x46\x38\xD2\xF4\xE0\x2D\x1E\xC9"
			  "\x7C\xEF\x53\xB7\x60\x72\x41\xBF"
			  "\x29\x00\x87\x02\xAF\x44\x4C\xB7"
			  "\x8C\xF5\x3F\x19\xF4\x80\x45\xA7"
			  "\x15\x5F\xDB\xE9\xB1\x83\xD2\xE6"
			  "\x1D\x18\x66\x44\x5B\x8F\x14\xEB",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast5_ctr_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A",
		.ilen	= 17,
		.result	= "\xFF\xC4\x2E\x82\x3D\xF8\xA8\x39"
			  "\x7C\x52\xC4\xD3\xBB\x62\xC6\xA8"
			  "\x0C",
		.rlen	= 17,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xFF\xC4\x2E\x82\x3D\xF8\xA8\x39"
			  "\x7C\x52\xC4\xD3\xBB\x62\xC6\xA8"
			  "\x0C\x63\xA5\x55\xE3\xF8\x1C\x7F"
			  "\xDC\x59\xF9\xA0\x52\xAD\x83\xDF"
			  "\xD5\x3B\x53\x4A\xAA\x1F\x49\x44"
			  "\xE8\x20\xCC\xF8\x97\xE6\xE0\x3C"
			  "\x5A\xD2\x83\xEC\xEE\x25\x3F\xCF"
			  "\x0D\xC2\x79\x80\x99\x6E\xFF\x7B"
			  "\x64\xB0\x7B\x86\x29\x1D\x9F\x17"
			  "\x10\xA5\xA5\xEB\x16\x55\x9E\xE3"
			  "\x88\x18\x52\x56\x48\x58\xD1\x6B"
			  "\xE8\x74\x6E\x48\xB0\x2E\x69\x63"
			  "\x32\xAA\xAC\x26\x55\x45\x94\xDE"
			  "\x30\x26\x26\xE6\x08\x82\x2F\x5F"
			  "\xA7\x15\x94\x07\x75\x2D\xC6\x3A"
			  "\x1B\xA0\x39\xFB\xBA\xB9\x06\x56"
			  "\xF6\x9F\xF1\x2F\x9B\xF3\x89\x8B"
			  "\x08\xC8\x9D\x5E\x6B\x95\x09\xC7"
			  "\x98\xB7\x62\xA4\x1D\x25\xFA\xC5"
			  "\x62\xC8\x5D\x6B\xB4\x85\x88\x7F"
			  "\x3B\x29\xF9\xB4\x32\x62\x69\xBF"
			  "\x32\xB8\xEB\xFD\x0E\x26\xAA\xA3"
			  "\x44\x67\x90\x20\xAC\x41\xDF\x43"
			  "\xC6\xC7\x19\x9F\x2C\x28\x74\xEB"
			  "\x3E\x7F\x7A\x80\x5B\xE4\x08\x60"
			  "\xC7\xC9\x71\x34\x44\xCE\x05\xFD"
			  "\xA8\x91\xA8\x44\x5E\xD3\x89\x2C"
			  "\xAE\x59\x0F\x07\x88\x79\x53\x26"
			  "\xAF\xAC\xCB\x1D\x6F\x08\x25\x62"
			  "\xD0\x82\x65\x66\xE4\x2A\x29\x1C"
			  "\x9C\x64\x5F\x49\x9D\xF8\x62\xF9"
			  "\xED\xC4\x13\x52\x75\xDC\xE4\xF9"
			  "\x68\x0F\x8A\xCD\xA6\x8D\x75\xAA"
			  "\x49\xA1\x86\x86\x37\x5C\x6B\x3D"
			  "\x56\xE5\x6F\xBE\x27\xC0\x10\xF8"
			  "\x3C\x4D\x17\x35\x14\xDC\x1C\xA0"
			  "\x6E\xAE\xD1\x10\xDD\x83\x06\xC2"
			  "\x23\xD3\xC7\x27\x15\x04\x2C\x27"
			  "\xDD\x1F\x2E\x97\x09\x9C\x33\x7D"
			  "\xAC\x50\x1B\x2E\xC9\x52\x0C\x14"
			  "\x4B\x78\xC4\xDE\x07\x6A\x12\x02"
			  "\x6E\xD7\x4B\x91\xB9\x88\x4D\x02"
			  "\xC3\xB5\x04\xBC\xE0\x67\xCA\x18"
			  "\x22\xA1\xAE\x9A\x21\xEF\xB2\x06"
			  "\x35\xCD\xEC\x37\x70\x2D\xFC\x1E"
			  "\xA8\x31\xE7\xFC\xE5\x8E\x88\x66"
			  "\x16\xB5\xC8\x45\x21\x37\xBD\x24"
			  "\xA9\xD5\x36\x12\x9F\x6E\x67\x80"
			  "\x87\x54\xD5\xAF\x97\xE1\x15\xA7"
			  "\x11\xF0\x63\x7B\xE1\x44\x14\x1C"
			  "\x06\x32\x05\x8C\x6C\xDB\x9B\x36"
			  "\x6A\x6B\xAD\x3A\x27\x55\x20\x4C"
			  "\x76\x36\x43\xE8\x16\x60\xB5\xF3"
			  "\xDF\x5A\xC6\xA5\x69\x78\x59\x51"
			  "\x54\x68\x65\x06\x84\xDE\x3D\xAE"
			  "\x38\x91\xBD\xCC\xA2\x8A\xEC\xE6"
			  "\x9E\x83\xAE\x1E\x8E\x34\x5D\xDE"
			  "\x91\xCE\x8F\xED\x40\xF7\xC8\x8B"
			  "\x9A\x13\x4C\xAD\x89\x97\x9E\xD1"
			  "\x91\x01\xD7\x21\x23\x28\x1E\xCC"
			  "\x8C\x98\xDB\xDE\xFC\x72\x94\xAA"
			  "\xC0\x0D\x96\xAA\x23\xF8\xFE\x13",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec cast5_ctr_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\xFF\xC4\x2E\x82\x3D\xF8\xA8\x39"
			  "\x7C\x52\xC4\xD3\xBB\x62\xC6\xA8"
			  "\x0C",
		.ilen	= 17,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A",
		.rlen	= 17,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A",
		.klen	= 16,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F",
		.input	= "\xFF\xC4\x2E\x82\x3D\xF8\xA8\x39"
			  "\x7C\x52\xC4\xD3\xBB\x62\xC6\xA8"
			  "\x0C\x63\xA5\x55\xE3\xF8\x1C\x7F"
			  "\xDC\x59\xF9\xA0\x52\xAD\x83\xDF"
			  "\xD5\x3B\x53\x4A\xAA\x1F\x49\x44"
			  "\xE8\x20\xCC\xF8\x97\xE6\xE0\x3C"
			  "\x5A\xD2\x83\xEC\xEE\x25\x3F\xCF"
			  "\x0D\xC2\x79\x80\x99\x6E\xFF\x7B"
			  "\x64\xB0\x7B\x86\x29\x1D\x9F\x17"
			  "\x10\xA5\xA5\xEB\x16\x55\x9E\xE3"
			  "\x88\x18\x52\x56\x48\x58\xD1\x6B"
			  "\xE8\x74\x6E\x48\xB0\x2E\x69\x63"
			  "\x32\xAA\xAC\x26\x55\x45\x94\xDE"
			  "\x30\x26\x26\xE6\x08\x82\x2F\x5F"
			  "\xA7\x15\x94\x07\x75\x2D\xC6\x3A"
			  "\x1B\xA0\x39\xFB\xBA\xB9\x06\x56"
			  "\xF6\x9F\xF1\x2F\x9B\xF3\x89\x8B"
			  "\x08\xC8\x9D\x5E\x6B\x95\x09\xC7"
			  "\x98\xB7\x62\xA4\x1D\x25\xFA\xC5"
			  "\x62\xC8\x5D\x6B\xB4\x85\x88\x7F"
			  "\x3B\x29\xF9\xB4\x32\x62\x69\xBF"
			  "\x32\xB8\xEB\xFD\x0E\x26\xAA\xA3"
			  "\x44\x67\x90\x20\xAC\x41\xDF\x43"
			  "\xC6\xC7\x19\x9F\x2C\x28\x74\xEB"
			  "\x3E\x7F\x7A\x80\x5B\xE4\x08\x60"
			  "\xC7\xC9\x71\x34\x44\xCE\x05\xFD"
			  "\xA8\x91\xA8\x44\x5E\xD3\x89\x2C"
			  "\xAE\x59\x0F\x07\x88\x79\x53\x26"
			  "\xAF\xAC\xCB\x1D\x6F\x08\x25\x62"
			  "\xD0\x82\x65\x66\xE4\x2A\x29\x1C"
			  "\x9C\x64\x5F\x49\x9D\xF8\x62\xF9"
			  "\xED\xC4\x13\x52\x75\xDC\xE4\xF9"
			  "\x68\x0F\x8A\xCD\xA6\x8D\x75\xAA"
			  "\x49\xA1\x86\x86\x37\x5C\x6B\x3D"
			  "\x56\xE5\x6F\xBE\x27\xC0\x10\xF8"
			  "\x3C\x4D\x17\x35\x14\xDC\x1C\xA0"
			  "\x6E\xAE\xD1\x10\xDD\x83\x06\xC2"
			  "\x23\xD3\xC7\x27\x15\x04\x2C\x27"
			  "\xDD\x1F\x2E\x97\x09\x9C\x33\x7D"
			  "\xAC\x50\x1B\x2E\xC9\x52\x0C\x14"
			  "\x4B\x78\xC4\xDE\x07\x6A\x12\x02"
			  "\x6E\xD7\x4B\x91\xB9\x88\x4D\x02"
			  "\xC3\xB5\x04\xBC\xE0\x67\xCA\x18"
			  "\x22\xA1\xAE\x9A\x21\xEF\xB2\x06"
			  "\x35\xCD\xEC\x37\x70\x2D\xFC\x1E"
			  "\xA8\x31\xE7\xFC\xE5\x8E\x88\x66"
			  "\x16\xB5\xC8\x45\x21\x37\xBD\x24"
			  "\xA9\xD5\x36\x12\x9F\x6E\x67\x80"
			  "\x87\x54\xD5\xAF\x97\xE1\x15\xA7"
			  "\x11\xF0\x63\x7B\xE1\x44\x14\x1C"
			  "\x06\x32\x05\x8C\x6C\xDB\x9B\x36"
			  "\x6A\x6B\xAD\x3A\x27\x55\x20\x4C"
			  "\x76\x36\x43\xE8\x16\x60\xB5\xF3"
			  "\xDF\x5A\xC6\xA5\x69\x78\x59\x51"
			  "\x54\x68\x65\x06\x84\xDE\x3D\xAE"
			  "\x38\x91\xBD\xCC\xA2\x8A\xEC\xE6"
			  "\x9E\x83\xAE\x1E\x8E\x34\x5D\xDE"
			  "\x91\xCE\x8F\xED\x40\xF7\xC8\x8B"
			  "\x9A\x13\x4C\xAD\x89\x97\x9E\xD1"
			  "\x91\x01\xD7\x21\x23\x28\x1E\xCC"
			  "\x8C\x98\xDB\xDE\xFC\x72\x94\xAA"
			  "\xC0\x0D\x96\xAA\x23\xF8\xFE\x13",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

#define ARC4_ENC_TEST_VECTORS	7
#define ARC4_DEC_TEST_VECTORS	7

static struct cipher_testvec arc4_enc_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.ilen	= 8,
		.result	= "\x75\xb7\x87\x80\x99\xe0\xc5\x96",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\x74\x94\xc2\xe7\x10\x4b\x08\x79",
		.rlen	= 8,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\xde\x18\x89\x41\xa3\x37\x5d\x3a",
		.rlen	= 8,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.ilen	= 20,
		.result	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61\x5a\x11\x62\xe1\xc7\xba"
			  "\x36\xb6\x78\x58",
		.rlen	= 20,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78",
		.ilen	= 28,
		.result	= "\x66\xa0\x94\x9f\x8a\xf7\xd6\x89"
			  "\x1f\x7f\x83\x2b\xa8\x33\xc0\x0c"
			  "\x89\x2e\xbe\x30\x14\x3c\xe2\x87"
			  "\x40\x01\x1e\xcf",
		.rlen	= 28,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00",
		.ilen	= 10,
		.result	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61",
		.rlen	= 10,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.input	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF",
		.ilen	= 8,
		.result	= "\x69\x72\x36\x59\x1B\x52\x42\xB1",
		.rlen	= 8,
	},
};

static struct cipher_testvec arc4_dec_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x75\xb7\x87\x80\x99\xe0\xc5\x96",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.rlen	= 8,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x74\x94\xc2\xe7\x10\x4b\x08\x79",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.input	= "\xde\x18\x89\x41\xa3\x37\x5d\x3a",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61\x5a\x11\x62\xe1\xc7\xba"
			  "\x36\xb6\x78\x58",
		.ilen	= 20,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.rlen	= 20,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 8,
		.input	= "\x66\xa0\x94\x9f\x8a\xf7\xd6\x89"
			  "\x1f\x7f\x83\x2b\xa8\x33\xc0\x0c"
			  "\x89\x2e\xbe\x30\x14\x3c\xe2\x87"
			  "\x40\x01\x1e\xcf",
		.ilen	= 28,
		.result	= "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78\x9A\xBC\xDE\xF0"
			  "\x12\x34\x56\x78",
		.rlen	= 28,
	}, {
		.key	= "\xef\x01\x23\x45",
		.klen	= 4,
		.input	= "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf"
			  "\xbd\x61",
		.ilen	= 10,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00",
		.rlen	= 10,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.input	= "\x69\x72\x36\x59\x1B\x52\x42\xB1",
		.ilen	= 8,
		.result	= "\x01\x23\x45\x67\x89\xAB\xCD\xEF",
		.rlen	= 8,
	},
};

#define TEA_ENC_TEST_VECTORS	4
#define TEA_DEC_TEST_VECTORS	4

static struct cipher_testvec tea_enc_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input  = zeroed_string,
		.ilen	= 8,
		.result	= "\x0a\x3a\xea\x41\x40\xa9\xba\x94",
		.rlen	= 8,
	}, {
		.key	= "\x2b\x02\x05\x68\x06\x14\x49\x76"
			  "\x77\x5d\x0e\x26\x6c\x28\x78\x43",
		.klen	= 16,
		.input	= "\x74\x65\x73\x74\x20\x6d\x65\x2e",
		.ilen	= 8,
		.result	= "\x77\x5d\x2a\x6a\xf6\xce\x92\x09",
		.rlen	= 8,
	}, {
		.key	= "\x09\x65\x43\x11\x66\x44\x39\x25"
			  "\x51\x3a\x16\x10\x0a\x08\x12\x6e",
		.klen	= 16,
		.input	= "\x6c\x6f\x6e\x67\x65\x72\x5f\x74"
			  "\x65\x73\x74\x5f\x76\x65\x63\x74",
		.ilen	= 16,
		.result	= "\xbe\x7a\xbb\x81\x95\x2d\x1f\x1e"
			  "\xdd\x89\xa1\x25\x04\x21\xdf\x95",
		.rlen	= 16,
	}, {
		.key	= "\x4d\x76\x32\x17\x05\x3f\x75\x2c"
			  "\x5d\x04\x16\x36\x15\x72\x63\x2f",
		.klen	= 16,
		.input	= "\x54\x65\x61\x20\x69\x73\x20\x67"
			  "\x6f\x6f\x64\x20\x66\x6f\x72\x20"
			  "\x79\x6f\x75\x21\x21\x21\x20\x72"
			  "\x65\x61\x6c\x6c\x79\x21\x21\x21",
		.ilen	= 32,
		.result	= "\xe0\x4d\x5d\x3c\xb7\x8c\x36\x47"
			  "\x94\x18\x95\x91\xa9\xfc\x49\xf8"
			  "\x44\xd1\x2d\xc2\x99\xb8\x08\x2a"
			  "\x07\x89\x73\xc2\x45\x92\xc6\x90",
		.rlen	= 32,
	}
};

static struct cipher_testvec tea_dec_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input	= "\x0a\x3a\xea\x41\x40\xa9\xba\x94",
		.ilen	= 8,
		.result = zeroed_string,
		.rlen	= 8,
	}, {
		.key	= "\x2b\x02\x05\x68\x06\x14\x49\x76"
			  "\x77\x5d\x0e\x26\x6c\x28\x78\x43",
		.klen	= 16,
		.input	= "\x77\x5d\x2a\x6a\xf6\xce\x92\x09",
		.ilen	= 8,
		.result	= "\x74\x65\x73\x74\x20\x6d\x65\x2e",
		.rlen	= 8,
	}, {
		.key	= "\x09\x65\x43\x11\x66\x44\x39\x25"
			  "\x51\x3a\x16\x10\x0a\x08\x12\x6e",
		.klen	= 16,
		.input	= "\xbe\x7a\xbb\x81\x95\x2d\x1f\x1e"
			  "\xdd\x89\xa1\x25\x04\x21\xdf\x95",
		.ilen   = 16,
		.result	= "\x6c\x6f\x6e\x67\x65\x72\x5f\x74"
			  "\x65\x73\x74\x5f\x76\x65\x63\x74",
		.rlen	= 16,
	}, {
		.key	= "\x4d\x76\x32\x17\x05\x3f\x75\x2c"
			  "\x5d\x04\x16\x36\x15\x72\x63\x2f",
		.klen	= 16,
		.input	= "\xe0\x4d\x5d\x3c\xb7\x8c\x36\x47"
			  "\x94\x18\x95\x91\xa9\xfc\x49\xf8"
			  "\x44\xd1\x2d\xc2\x99\xb8\x08\x2a"
			  "\x07\x89\x73\xc2\x45\x92\xc6\x90",
		.ilen	= 32,
		.result	= "\x54\x65\x61\x20\x69\x73\x20\x67"
			  "\x6f\x6f\x64\x20\x66\x6f\x72\x20"
			  "\x79\x6f\x75\x21\x21\x21\x20\x72"
			  "\x65\x61\x6c\x6c\x79\x21\x21\x21",
		.rlen	= 32,
	}
};

#define XTEA_ENC_TEST_VECTORS	4
#define XTEA_DEC_TEST_VECTORS	4

static struct cipher_testvec xtea_enc_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input  = zeroed_string,
		.ilen	= 8,
		.result	= "\xd8\xd4\xe9\xde\xd9\x1e\x13\xf7",
		.rlen	= 8,
	}, {
		.key	= "\x2b\x02\x05\x68\x06\x14\x49\x76"
			  "\x77\x5d\x0e\x26\x6c\x28\x78\x43",
		.klen	= 16,
		.input	= "\x74\x65\x73\x74\x20\x6d\x65\x2e",
		.ilen	= 8,
		.result	= "\x94\xeb\xc8\x96\x84\x6a\x49\xa8",
		.rlen	= 8,
	}, {
		.key	= "\x09\x65\x43\x11\x66\x44\x39\x25"
			  "\x51\x3a\x16\x10\x0a\x08\x12\x6e",
		.klen	= 16,
		.input	= "\x6c\x6f\x6e\x67\x65\x72\x5f\x74"
			  "\x65\x73\x74\x5f\x76\x65\x63\x74",
		.ilen	= 16,
		.result	= "\x3e\xce\xae\x22\x60\x56\xa8\x9d"
			  "\x77\x4d\xd4\xb4\x87\x24\xe3\x9a",
		.rlen	= 16,
	}, {
		.key	= "\x4d\x76\x32\x17\x05\x3f\x75\x2c"
			  "\x5d\x04\x16\x36\x15\x72\x63\x2f",
		.klen	= 16,
		.input	= "\x54\x65\x61\x20\x69\x73\x20\x67"
			  "\x6f\x6f\x64\x20\x66\x6f\x72\x20"
			  "\x79\x6f\x75\x21\x21\x21\x20\x72"
			  "\x65\x61\x6c\x6c\x79\x21\x21\x21",
		.ilen	= 32,
		.result	= "\x99\x81\x9f\x5d\x6f\x4b\x31\x3a"
			  "\x86\xff\x6f\xd0\xe3\x87\x70\x07"
			  "\x4d\xb8\xcf\xf3\x99\x50\xb3\xd4"
			  "\x73\xa2\xfa\xc9\x16\x59\x5d\x81",
		.rlen	= 32,
	}
};

static struct cipher_testvec xtea_dec_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input	= "\xd8\xd4\xe9\xde\xd9\x1e\x13\xf7",
		.ilen	= 8,
		.result = zeroed_string,
		.rlen	= 8,
	}, {
		.key	= "\x2b\x02\x05\x68\x06\x14\x49\x76"
			  "\x77\x5d\x0e\x26\x6c\x28\x78\x43",
		.klen	= 16,
		.input	= "\x94\xeb\xc8\x96\x84\x6a\x49\xa8",
		.ilen	= 8,
		.result	= "\x74\x65\x73\x74\x20\x6d\x65\x2e",
		.rlen	= 8,
	}, {
		.key	= "\x09\x65\x43\x11\x66\x44\x39\x25"
			  "\x51\x3a\x16\x10\x0a\x08\x12\x6e",
		.klen	= 16,
		.input	= "\x3e\xce\xae\x22\x60\x56\xa8\x9d"
			  "\x77\x4d\xd4\xb4\x87\x24\xe3\x9a",
		.ilen	= 16,
		.result	= "\x6c\x6f\x6e\x67\x65\x72\x5f\x74"
			  "\x65\x73\x74\x5f\x76\x65\x63\x74",
		.rlen	= 16,
	}, {
		.key	= "\x4d\x76\x32\x17\x05\x3f\x75\x2c"
			  "\x5d\x04\x16\x36\x15\x72\x63\x2f",
		.klen	= 16,
		.input	= "\x99\x81\x9f\x5d\x6f\x4b\x31\x3a"
			  "\x86\xff\x6f\xd0\xe3\x87\x70\x07"
			  "\x4d\xb8\xcf\xf3\x99\x50\xb3\xd4"
			  "\x73\xa2\xfa\xc9\x16\x59\x5d\x81",
		.ilen	= 32,
		.result	= "\x54\x65\x61\x20\x69\x73\x20\x67"
			  "\x6f\x6f\x64\x20\x66\x6f\x72\x20"
			  "\x79\x6f\x75\x21\x21\x21\x20\x72"
			  "\x65\x61\x6c\x6c\x79\x21\x21\x21",
		.rlen	= 32,
	}
};

#define KHAZAD_ENC_TEST_VECTORS 5
#define KHAZAD_DEC_TEST_VECTORS 5

static struct cipher_testvec khazad_enc_tv_template[] = {
	{
		.key	= "\x80\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\x49\xa4\xce\x32\xac\x19\x0e\x3f",
		.rlen	= 8,
	}, {
		.key	= "\x38\x38\x38\x38\x38\x38\x38\x38"
			  "\x38\x38\x38\x38\x38\x38\x38\x38",
		.klen	= 16,
		.input	= "\x38\x38\x38\x38\x38\x38\x38\x38",
		.ilen	= 8,
		.result	= "\x7e\x82\x12\xa1\xd9\x5b\xe4\xf9",
		.rlen	= 8,
	}, {
		.key	= "\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2"
			"\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2",
		.klen	= 16,
		.input	= "\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2",
		.ilen	= 8,
		.result	= "\xaa\xbe\xc1\x95\xc5\x94\x1a\x9c",
		.rlen	= 8,
	}, {
		.key	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f"
			"\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.klen	= 16,
		.input	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.ilen	= 8,
		.result = "\x04\x74\xf5\x70\x50\x16\xd3\xb8",
		.rlen	= 8,
	}, {
		.key	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f"
			"\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.klen	= 16,
		.input	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f"
			"\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.ilen	= 16,
		.result = "\x04\x74\xf5\x70\x50\x16\xd3\xb8"
			"\x04\x74\xf5\x70\x50\x16\xd3\xb8",
		.rlen	= 16,
	},
};

static struct cipher_testvec khazad_dec_tv_template[] = {
	{
		.key	= "\x80\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.input	= "\x49\xa4\xce\x32\xac\x19\x0e\x3f",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\x38\x38\x38\x38\x38\x38\x38\x38"
			  "\x38\x38\x38\x38\x38\x38\x38\x38",
		.klen	= 16,
		.input	= "\x7e\x82\x12\xa1\xd9\x5b\xe4\xf9",
		.ilen	= 8,
		.result	= "\x38\x38\x38\x38\x38\x38\x38\x38",
		.rlen	= 8,
	}, {
		.key	= "\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2"
			"\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2",
		.klen	= 16,
		.input	= "\xaa\xbe\xc1\x95\xc5\x94\x1a\x9c",
		.ilen	= 8,
		.result	= "\xa2\xa2\xa2\xa2\xa2\xa2\xa2\xa2",
		.rlen	= 8,
	}, {
		.key	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f"
			"\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.klen	= 16,
		.input  = "\x04\x74\xf5\x70\x50\x16\xd3\xb8",
		.ilen	= 8,
		.result	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.rlen	= 8,
	}, {
		.key	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f"
			"\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.klen	= 16,
		.input  = "\x04\x74\xf5\x70\x50\x16\xd3\xb8"
			"\x04\x74\xf5\x70\x50\x16\xd3\xb8",
		.ilen	= 16,
		.result	= "\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f"
			"\x2f\x2f\x2f\x2f\x2f\x2f\x2f\x2f",
		.rlen	= 16,
	},
};

#define ANUBIS_ENC_TEST_VECTORS			5
#define ANUBIS_DEC_TEST_VECTORS			5
#define ANUBIS_CBC_ENC_TEST_VECTORS		2
#define ANUBIS_CBC_DEC_TEST_VECTORS		2

static struct cipher_testvec anubis_enc_tv_template[] = {
	{
		.key	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.klen	= 16,
		.input	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.ilen	= 16,
		.result	= "\x6d\xc5\xda\xa2\x26\x7d\x62\x6f"
			  "\x08\xb7\x52\x8e\x6e\x6e\x86\x90",
		.rlen	= 16,
	}, {

		.key	= "\x03\x03\x03\x03\x03\x03\x03\x03"
			  "\x03\x03\x03\x03\x03\x03\x03\x03"
			  "\x03\x03\x03\x03",
		.klen	= 20,
		.input	= "\x03\x03\x03\x03\x03\x03\x03\x03"
			  "\x03\x03\x03\x03\x03\x03\x03\x03",
		.ilen	= 16,
		.result	= "\xdb\xf1\x42\xf4\xd1\x8a\xc7\x49"
			  "\x87\x41\x6f\x82\x0a\x98\x64\xae",
		.rlen	= 16,
	}, {
		.key	= "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24",
		.klen	= 28,
		.input	= "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24\x24\x24\x24\x24",
		.ilen	= 16,
		.result	= "\xfd\x1b\x4a\xe3\xbf\xf0\xad\x3d"
			  "\x06\xd3\x61\x27\xfd\x13\x9e\xde",
		.rlen	= 16,
	}, {
		.key	= "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25",
		.klen	= 32,
		.input	= "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25",
		.ilen	= 16,
		.result	= "\x1a\x91\xfb\x2b\xb7\x78\x6b\xc4"
			"\x17\xd9\xff\x40\x3b\x0e\xe5\xfe",
		.rlen	= 16,
	}, {
		.key	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.klen	= 40,
		.input	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.ilen	= 16,
		.result = "\xa5\x2c\x85\x6f\x9c\xba\xa0\x97"
			  "\x9e\xc6\x84\x0f\x17\x21\x07\xee",
		.rlen	= 16,
	},
};

static struct cipher_testvec anubis_dec_tv_template[] = {
	{
		.key	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.klen	= 16,
		.input	= "\x6d\xc5\xda\xa2\x26\x7d\x62\x6f"
			  "\x08\xb7\x52\x8e\x6e\x6e\x86\x90",
		.ilen	= 16,
		.result	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.rlen	= 16,
	}, {

		.key	= "\x03\x03\x03\x03\x03\x03\x03\x03"
			  "\x03\x03\x03\x03\x03\x03\x03\x03"
			  "\x03\x03\x03\x03",
		.klen	= 20,
		.input	= "\xdb\xf1\x42\xf4\xd1\x8a\xc7\x49"
			  "\x87\x41\x6f\x82\x0a\x98\x64\xae",
		.ilen	= 16,
		.result	= "\x03\x03\x03\x03\x03\x03\x03\x03"
			  "\x03\x03\x03\x03\x03\x03\x03\x03",
		.rlen	= 16,
	}, {
		.key	= "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24",
		.klen	= 28,
		.input	= "\xfd\x1b\x4a\xe3\xbf\xf0\xad\x3d"
			  "\x06\xd3\x61\x27\xfd\x13\x9e\xde",
		.ilen	= 16,
		.result	= "\x24\x24\x24\x24\x24\x24\x24\x24"
			  "\x24\x24\x24\x24\x24\x24\x24\x24",
		.rlen	= 16,
	}, {
		.key	= "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25",
		.klen	= 32,
		.input	= "\x1a\x91\xfb\x2b\xb7\x78\x6b\xc4"
			"\x17\xd9\xff\x40\x3b\x0e\xe5\xfe",
		.ilen	= 16,
		.result	= "\x25\x25\x25\x25\x25\x25\x25\x25"
			  "\x25\x25\x25\x25\x25\x25\x25\x25",
		.rlen	= 16,
	}, {
		.key	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.input = "\xa5\x2c\x85\x6f\x9c\xba\xa0\x97"
			 "\x9e\xc6\x84\x0f\x17\x21\x07\xee",
		.klen	= 40,
		.ilen	= 16,
		.result	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.rlen	= 16,
	},
};

static struct cipher_testvec anubis_cbc_enc_tv_template[] = {
	{
		.key	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.klen	= 16,
		.input	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.ilen	= 32,
		.result	= "\x6d\xc5\xda\xa2\x26\x7d\x62\x6f"
			  "\x08\xb7\x52\x8e\x6e\x6e\x86\x90"
			  "\x86\xd8\xb5\x6f\x98\x5e\x8a\x66"
			  "\x4f\x1f\x78\xa1\xbb\x37\xf1\xbe",
		.rlen	= 32,
	}, {
		.key	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.klen	= 40,
		.input	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.ilen	= 32,
		.result = "\xa5\x2c\x85\x6f\x9c\xba\xa0\x97"
			  "\x9e\xc6\x84\x0f\x17\x21\x07\xee"
			  "\xa2\xbc\x06\x98\xc6\x4b\xda\x75"
			  "\x2e\xaa\xbe\x58\xce\x01\x5b\xc7",
		.rlen	= 32,
	},
};

static struct cipher_testvec anubis_cbc_dec_tv_template[] = {
	{
		.key	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.klen	= 16,
		.input	= "\x6d\xc5\xda\xa2\x26\x7d\x62\x6f"
			  "\x08\xb7\x52\x8e\x6e\x6e\x86\x90"
			  "\x86\xd8\xb5\x6f\x98\x5e\x8a\x66"
			  "\x4f\x1f\x78\xa1\xbb\x37\xf1\xbe",
		.ilen	= 32,
		.result	= "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe"
			  "\xfe\xfe\xfe\xfe\xfe\xfe\xfe\xfe",
		.rlen	= 32,
	}, {
		.key	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.klen	= 40,
		.input = "\xa5\x2c\x85\x6f\x9c\xba\xa0\x97"
			  "\x9e\xc6\x84\x0f\x17\x21\x07\xee"
			  "\xa2\xbc\x06\x98\xc6\x4b\xda\x75"
			  "\x2e\xaa\xbe\x58\xce\x01\x5b\xc7",
		.ilen	= 32,
		.result	= "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35"
			  "\x35\x35\x35\x35\x35\x35\x35\x35",
		.rlen	= 32,
	},
};

#define XETA_ENC_TEST_VECTORS	4
#define XETA_DEC_TEST_VECTORS	4

static struct cipher_testvec xeta_enc_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input  = zeroed_string,
		.ilen	= 8,
		.result	= "\xaa\x22\x96\xe5\x6c\x61\xf3\x45",
		.rlen	= 8,
	}, {
		.key	= "\x2b\x02\x05\x68\x06\x14\x49\x76"
			  "\x77\x5d\x0e\x26\x6c\x28\x78\x43",
		.klen	= 16,
		.input	= "\x74\x65\x73\x74\x20\x6d\x65\x2e",
		.ilen	= 8,
		.result	= "\x82\x3e\xeb\x35\xdc\xdd\xd9\xc3",
		.rlen	= 8,
	}, {
		.key	= "\x09\x65\x43\x11\x66\x44\x39\x25"
			  "\x51\x3a\x16\x10\x0a\x08\x12\x6e",
		.klen	= 16,
		.input	= "\x6c\x6f\x6e\x67\x65\x72\x5f\x74"
			  "\x65\x73\x74\x5f\x76\x65\x63\x74",
		.ilen	= 16,
		.result	= "\xe2\x04\xdb\xf2\x89\x85\x9e\xea"
			  "\x61\x35\xaa\xed\xb5\xcb\x71\x2c",
		.rlen	= 16,
	}, {
		.key	= "\x4d\x76\x32\x17\x05\x3f\x75\x2c"
			  "\x5d\x04\x16\x36\x15\x72\x63\x2f",
		.klen	= 16,
		.input	= "\x54\x65\x61\x20\x69\x73\x20\x67"
			  "\x6f\x6f\x64\x20\x66\x6f\x72\x20"
			  "\x79\x6f\x75\x21\x21\x21\x20\x72"
			  "\x65\x61\x6c\x6c\x79\x21\x21\x21",
		.ilen	= 32,
		.result	= "\x0b\x03\xcd\x8a\xbe\x95\xfd\xb1"
			  "\xc1\x44\x91\x0b\xa5\xc9\x1b\xb4"
			  "\xa9\xda\x1e\x9e\xb1\x3e\x2a\x8f"
			  "\xea\xa5\x6a\x85\xd1\xf4\xa8\xa5",
		.rlen	= 32,
	}
};

static struct cipher_testvec xeta_dec_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input	= "\xaa\x22\x96\xe5\x6c\x61\xf3\x45",
		.ilen	= 8,
		.result = zeroed_string,
		.rlen	= 8,
	}, {
		.key	= "\x2b\x02\x05\x68\x06\x14\x49\x76"
			  "\x77\x5d\x0e\x26\x6c\x28\x78\x43",
		.klen	= 16,
		.input	= "\x82\x3e\xeb\x35\xdc\xdd\xd9\xc3",
		.ilen	= 8,
		.result	= "\x74\x65\x73\x74\x20\x6d\x65\x2e",
		.rlen	= 8,
	}, {
		.key	= "\x09\x65\x43\x11\x66\x44\x39\x25"
			  "\x51\x3a\x16\x10\x0a\x08\x12\x6e",
		.klen	= 16,
		.input	= "\xe2\x04\xdb\xf2\x89\x85\x9e\xea"
			  "\x61\x35\xaa\xed\xb5\xcb\x71\x2c",
		.ilen	= 16,
		.result	= "\x6c\x6f\x6e\x67\x65\x72\x5f\x74"
			  "\x65\x73\x74\x5f\x76\x65\x63\x74",
		.rlen	= 16,
	}, {
		.key	= "\x4d\x76\x32\x17\x05\x3f\x75\x2c"
			  "\x5d\x04\x16\x36\x15\x72\x63\x2f",
		.klen	= 16,
		.input	= "\x0b\x03\xcd\x8a\xbe\x95\xfd\xb1"
			  "\xc1\x44\x91\x0b\xa5\xc9\x1b\xb4"
			  "\xa9\xda\x1e\x9e\xb1\x3e\x2a\x8f"
			  "\xea\xa5\x6a\x85\xd1\xf4\xa8\xa5",
		.ilen	= 32,
		.result	= "\x54\x65\x61\x20\x69\x73\x20\x67"
			  "\x6f\x6f\x64\x20\x66\x6f\x72\x20"
			  "\x79\x6f\x75\x21\x21\x21\x20\x72"
			  "\x65\x61\x6c\x6c\x79\x21\x21\x21",
		.rlen	= 32,
	}
};

#define FCRYPT_ENC_TEST_VECTORS	ARRAY_SIZE(fcrypt_pcbc_enc_tv_template)
#define FCRYPT_DEC_TEST_VECTORS	ARRAY_SIZE(fcrypt_pcbc_dec_tv_template)

static struct cipher_testvec fcrypt_pcbc_enc_tv_template[] = {
	{  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 8,
		.result	= "\x0E\x09\x00\xC7\x3E\xF7\xED\x41",
		.rlen	= 8,
	}, {
		.key	= "\x11\x44\x77\xAA\xDD\x00\x33\x66",
		.klen	= 8,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x12\x34\x56\x78\x9A\xBC\xDE\xF0",
		.ilen	= 8,
		.result	= "\xD8\xED\x78\x74\x77\xEC\x06\x80",
		.rlen	= 8,
	}, {  
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.klen	= 8,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.ilen	= 48,
		.result	= "\x00\xf0\x0e\x11\x75\xe6\x23\x82"
			  "\xee\xac\x98\x62\x44\x51\xe4\x84"
			  "\xc3\x59\xd8\xaa\x64\x60\xae\xf7"
			  "\xd2\xd9\x13\x79\x72\xa3\x45\x03"
			  "\x23\xb5\x62\xd7\x0c\xf5\x27\xd1"
			  "\xf8\x91\x3c\xac\x44\x22\x92\xef",
		.rlen	= 48,
	}, {
		.key	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 8,
		.iv	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.input	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.ilen	= 48,
		.result	= "\xca\x90\xf5\x9d\xcb\xd4\xd2\x3c"
			  "\x01\x88\x7f\x3e\x31\x6e\x62\x9d"
			  "\xd8\xe0\x57\xa3\x06\x3a\x42\x58"
			  "\x2a\x28\xfe\x72\x52\x2f\xdd\xe0"
			  "\x19\x89\x09\x1c\x2a\x8e\x8c\x94"
			  "\xfc\xc7\x68\xe4\x88\xaa\xde\x0f",
		.rlen	= 48,
	}, {  
		.key	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 8,
		.iv	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.input	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.ilen	= 48,
		.result	= "\xca\x90\xf5\x9d\xcb\xd4\xd2\x3c"
			  "\x01\x88\x7f\x3e\x31\x6e\x62\x9d"
			  "\xd8\xe0\x57\xa3\x06\x3a\x42\x58"
			  "\x2a\x28\xfe\x72\x52\x2f\xdd\xe0"
			  "\x19\x89\x09\x1c\x2a\x8e\x8c\x94"
			  "\xfc\xc7\x68\xe4\x88\xaa\xde\x0f",
		.rlen	= 48,
		.np	= 2,
		.tap	= { 20, 28 },
	}
};

static struct cipher_testvec fcrypt_pcbc_dec_tv_template[] = {
	{  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 8,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x0E\x09\x00\xC7\x3E\xF7\xED\x41",
		.ilen	= 8,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 8,
	}, {
		.key	= "\x11\x44\x77\xAA\xDD\x00\x33\x66",
		.klen	= 8,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xD8\xED\x78\x74\x77\xEC\x06\x80",
		.ilen	= 8,
		.result	= "\x12\x34\x56\x78\x9A\xBC\xDE\xF0",
		.rlen	= 8,
	}, {  
		.key	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.klen	= 8,
		.iv	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.input	= "\x00\xf0\x0e\x11\x75\xe6\x23\x82"
			  "\xee\xac\x98\x62\x44\x51\xe4\x84"
			  "\xc3\x59\xd8\xaa\x64\x60\xae\xf7"
			  "\xd2\xd9\x13\x79\x72\xa3\x45\x03"
			  "\x23\xb5\x62\xd7\x0c\xf5\x27\xd1"
			  "\xf8\x91\x3c\xac\x44\x22\x92\xef",
		.ilen	= 48,
		.result	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.rlen	= 48,
	}, {
		.key	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 8,
		.iv	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.input	= "\xca\x90\xf5\x9d\xcb\xd4\xd2\x3c"
			  "\x01\x88\x7f\x3e\x31\x6e\x62\x9d"
			  "\xd8\xe0\x57\xa3\x06\x3a\x42\x58"
			  "\x2a\x28\xfe\x72\x52\x2f\xdd\xe0"
			  "\x19\x89\x09\x1c\x2a\x8e\x8c\x94"
			  "\xfc\xc7\x68\xe4\x88\xaa\xde\x0f",
		.ilen	= 48,
		.result	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.rlen	= 48,
	}, {  
		.key	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 8,
		.iv	= "\xf0\xe1\xd2\xc3\xb4\xa5\x96\x87",
		.input	= "\xca\x90\xf5\x9d\xcb\xd4\xd2\x3c"
			  "\x01\x88\x7f\x3e\x31\x6e\x62\x9d"
			  "\xd8\xe0\x57\xa3\x06\x3a\x42\x58"
			  "\x2a\x28\xfe\x72\x52\x2f\xdd\xe0"
			  "\x19\x89\x09\x1c\x2a\x8e\x8c\x94"
			  "\xfc\xc7\x68\xe4\x88\xaa\xde\x0f",
		.ilen	= 48,
		.result	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.rlen	= 48,
		.np	= 2,
		.tap	= { 20, 28 },
	}
};

#define CAMELLIA_ENC_TEST_VECTORS 4
#define CAMELLIA_DEC_TEST_VECTORS 4
#define CAMELLIA_CBC_ENC_TEST_VECTORS 3
#define CAMELLIA_CBC_DEC_TEST_VECTORS 3
#define CAMELLIA_CTR_ENC_TEST_VECTORS 2
#define CAMELLIA_CTR_DEC_TEST_VECTORS 2
#define CAMELLIA_LRW_ENC_TEST_VECTORS 8
#define CAMELLIA_LRW_DEC_TEST_VECTORS 8
#define CAMELLIA_XTS_ENC_TEST_VECTORS 5
#define CAMELLIA_XTS_DEC_TEST_VECTORS 5

static struct cipher_testvec camellia_enc_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 16,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 16,
		.result	= "\x67\x67\x31\x38\x54\x96\x69\x73"
			  "\x08\x57\x06\x56\x48\xea\xbe\x43",
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77",
		.klen	= 24,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 16,
		.result	= "\xb4\x99\x34\x01\xb3\xe9\x96\xf8"
			  "\x4e\xe5\xce\xe7\xd7\x9b\x09\xb9",
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.klen	= 32,
		.input	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ilen	= 16,
		.result	= "\x9a\xcc\x23\x7d\xff\x16\xd7\x6c"
			  "\x20\xef\x7c\x91\x9e\x3a\x75\x09",
		.rlen	= 16,
	}, {  
		.key	= "\x3F\x85\x62\x3F\x1C\xF9\xD6\x1C"
			  "\xF9\xD6\xB3\x90\x6D\x4A\x90\x6D"
			  "\x4A\x27\x04\xE1\x27\x04\xE1\xBE"
			  "\x9B\x78\xBE\x9B\x78\x55\x32\x0F",
		.klen	= 32,
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D",
		.ilen	= 1008,
		.result	= "\xED\xCD\xDB\xB8\x68\xCE\xBD\xEA"
			  "\x9D\x9D\xCD\x9F\x4F\xFC\x4D\xB7"
			  "\xA5\xFF\x6F\x43\x0F\xBA\x32\x04"
			  "\xB3\xC2\xB9\x03\xAA\x91\x56\x29"
			  "\x0D\xD0\xFD\xC4\x65\xA5\x69\xB9"
			  "\xF1\xF6\xB1\xA5\xB2\x75\x4F\x8A"
			  "\x8D\x7D\x1B\x9B\xC7\x68\x72\xF8"
			  "\x01\x9B\x17\x0A\x29\xE7\x61\x28"
			  "\x7F\xA7\x50\xCA\x20\x2C\x96\x3B"
			  "\x6E\x5C\x5D\x3F\xB5\x7F\xF3\x2B"
			  "\x04\xEF\x9D\xD4\xCE\x41\x28\x8E"
			  "\x83\x54\xAE\x7C\x82\x46\x10\xC9"
			  "\xC4\x8A\x1E\x1F\x4C\xA9\xFC\xEC"
			  "\x3C\x8C\x30\xFC\x59\xD2\x54\xC4"
			  "\x6F\x50\xC6\xCA\x8C\x14\x5B\x9C"
			  "\x18\x56\x5B\xF8\x33\x0E\x4A\xDB"
			  "\xEC\xB5\x6E\x5B\x31\xC4\x0E\x98"
			  "\x9F\x32\xBA\xA2\x18\xCF\x55\x43"
			  "\xFE\x80\x8F\x60\xCF\x05\x30\x9B"
			  "\x70\x50\x1E\x9C\x08\x87\xE6\x20"
			  "\xD2\xF3\x27\xF8\x2A\x8D\x12\xB2"
			  "\xBC\x5F\xFE\x52\x52\xF6\x7F\xB6"
			  "\xB8\x30\x86\x3B\x0F\x94\x1E\x79"
			  "\x13\x94\x35\xA2\xB1\x35\x5B\x05"
			  "\x2A\x98\x6B\x96\x4C\xB1\x20\xBE"
			  "\xB6\x14\xC2\x06\xBF\xFD\x5F\x2A"
			  "\xF5\x33\xC8\x19\x45\x14\x44\x5D"
			  "\xFE\x94\x7B\xBB\x63\x13\x57\xC3"
			  "\x2A\x8F\x6C\x11\x2A\x07\xA7\x6A"
			  "\xBF\x20\xD3\x99\xC6\x00\x0B\xBF"
			  "\x83\x46\x25\x3A\xB0\xF6\xC5\xC8"
			  "\x00\xCA\xE5\x28\x4A\x7C\x95\x9C"
			  "\x7B\x43\xAB\xF9\xE4\xF8\x74\xAB"
			  "\xA7\xB8\x9C\x0F\x53\x7B\xB6\x74"
			  "\x60\x64\x0D\x1C\x80\xD1\x20\x9E"
			  "\xDC\x14\x27\x9B\xFC\xBD\x5C\x96"
			  "\xD2\x51\xDC\x96\xEE\xE5\xEA\x2B"
			  "\x02\x7C\xAA\x3C\xDC\x9D\x7B\x01"
			  "\x20\xC3\xE1\x0B\xDD\xAB\xF3\x1E"
			  "\x19\xA8\x84\x29\x5F\xCC\xC3\x5B"
			  "\xE4\x33\x59\xDC\x12\xEB\x2B\x4D"
			  "\x5B\x55\x23\xB7\x40\x31\xDE\xEE"
			  "\x18\xC9\x3C\x4D\xBC\xED\xE0\x42"
			  "\xAD\xDE\xA0\xA3\xC3\xFE\x44\xD3"
			  "\xE1\x9A\xDA\xAB\x32\xFC\x1A\xBF"
			  "\x63\xA9\xF0\x6A\x08\x46\xBD\x48"
			  "\x83\x06\xAB\x82\x99\x01\x16\x1A"
			  "\x03\x36\xC5\x59\x6B\xB8\x8C\x9F"
			  "\xC6\x51\x3D\xE5\x7F\xBF\xAB\xBC"
			  "\xC9\xA1\x88\x34\x5F\xA9\x7C\x3B"
			  "\x9F\x1B\x98\x2B\x4F\xFB\x9B\xF0"
			  "\xCD\xB6\x45\xB2\x29\x2E\x34\x23"
			  "\xA9\x97\xC0\x22\x8C\x42\x9B\x5F"
			  "\x40\xC8\xD7\x3D\x82\x9A\x6F\xAA"
			  "\x74\x83\x29\x05\xE8\xC4\x4D\x01"
			  "\xB5\xE5\x84\x3F\x7F\xD3\xE0\x99"
			  "\xDA\xE7\x6F\x30\xFD\xAA\x92\x30"
			  "\xA5\x46\x8B\xA2\xE6\x58\x62\x7C"
			  "\x2C\x35\x1B\x38\x85\x7D\xE8\xF3"
			  "\x87\x4F\xDA\xD8\x5F\xFC\xB6\x44"
			  "\xD0\xE3\x9B\x8B\xBF\xD6\xB8\xC4"
			  "\x73\xAE\x1D\x8B\x5B\x74\x8B\xCB"
			  "\xA4\xAD\xCF\x5D\xD4\x58\xC9\xCD"
			  "\xF7\x90\x68\xCF\xC9\x11\x52\x3E"
			  "\xE8\xA1\xA3\x78\x8B\xD0\xAC\x0A"
			  "\xD4\xC9\xA3\xA5\x55\x30\xC8\x3E"
			  "\xED\x28\x39\xE9\x63\xED\x41\x70"
			  "\x51\xE3\xC4\xA0\xFC\xD5\x43\xCB"
			  "\x4D\x65\xC8\xFD\x3A\x91\x8F\x60"
			  "\x8A\xA6\x6D\x9D\x3E\x01\x23\x4B"
			  "\x50\x47\xC9\xDC\x9B\xDE\x37\xC5"
			  "\xBF\x67\xB1\x6B\x78\x38\xD5\x7E"
			  "\xB6\xFF\x67\x83\x3B\x6E\xBE\x23"
			  "\x45\xFA\x1D\x69\x44\xFD\xC6\xB9"
			  "\xD0\x4A\x92\xD1\xBE\xF6\x4A\xB7"
			  "\xCA\xA8\xA2\x9E\x13\x87\x57\x92"
			  "\x64\x7C\x85\x0B\xB3\x29\x37\xD8"
			  "\xE6\xAA\xAF\xC4\x03\x67\xA3\xBF"
			  "\x2E\x45\x83\xB6\xD8\x54\x00\x89"
			  "\xF6\xBC\x3A\x7A\x88\x58\x51\xED"
			  "\xF4\x4E\x01\xA5\xC3\x2E\xD9\x42"
			  "\xBD\x6E\x0D\x0B\x21\xB0\x1A\xCC"
			  "\xA4\xD3\x3F\xDC\x9B\x81\xD8\xF1"
			  "\xEA\x7A\x6A\xB7\x07\xC9\x6D\x91"
			  "\x6D\x3A\xF5\x5F\xA6\xFF\x87\x1E"
			  "\x3F\xDD\xC0\x72\xEA\xAC\x08\x15"
			  "\x21\xE6\xC6\xB6\x0D\xD8\x51\x86"
			  "\x2A\x03\x73\xF7\x29\xD4\xC4\xE4"
			  "\x7F\x95\x10\xF7\xAB\x3F\x92\x23"
			  "\xD3\xCE\x9C\x2E\x46\x3B\x63\x43"
			  "\xBB\xC2\x82\x7A\x83\xD5\x55\xE2"
			  "\xE7\x9B\x2F\x92\xAF\xFD\x81\x56"
			  "\x79\xFD\x3E\xF9\x46\xE0\x25\xD4"
			  "\x38\xDE\xBC\x2C\xC4\x7A\x2A\x8F"
			  "\x94\x4F\xD0\xAD\x9B\x37\x18\xD4"
			  "\x0E\x4D\x0F\x02\x3A\xDC\x5A\xA2"
			  "\x39\x25\x55\x20\x5A\xA6\x02\x9F"
			  "\xE6\x77\x21\x77\xE5\x4B\x7B\x0B"
			  "\x30\xF8\x5F\x33\x0F\x49\xCD\xFF"
			  "\xF2\xE4\x35\xF9\xF0\x63\xC3\x7E"
			  "\xF1\xA6\x73\xB4\xDF\xE7\xBB\x78"
			  "\xFF\x21\xA9\xF3\xF3\xCF\x5D\xBA"
			  "\xED\x87\x98\xAC\xFE\x48\x97\x6D"
			  "\xA6\x7F\x69\x31\xB1\xC4\xFF\x14"
			  "\xC6\x76\xD4\x10\xDD\xF6\x49\x2C"
			  "\x9C\xC8\x6D\x76\xC0\x8F\x5F\x55"
			  "\x2F\x3C\x8A\x30\xAA\xC3\x16\x55"
			  "\xC6\xFC\x8D\x8B\xB9\xE5\x80\x6C"
			  "\xC8\x7E\xBD\x65\x58\x36\xD5\xBC"
			  "\xF0\x33\x52\x29\x70\xF9\x5C\xE9"
			  "\xAC\x1F\xB5\x73\x56\x66\x54\xAF"
			  "\x1B\x8F\x7D\xED\xAB\x03\xCE\xE3"
			  "\xAE\x47\xB6\x69\x86\xE9\x01\x31"
			  "\x83\x18\x3D\xF4\x74\x7B\xF9\x42"
			  "\x4C\xFD\x75\x4A\x6D\xF0\x03\xA6"
			  "\x2B\x20\x63\xDA\x49\x65\x5E\x8B"
			  "\xC0\x19\xE3\x8D\xD9\xF3\xB0\x34"
			  "\xD3\x52\xFC\x68\x00\x43\x1B\x37"
			  "\x31\x93\x51\x1C\x63\x97\x70\xB0"
			  "\x99\x78\x83\x13\xFD\xCF\x53\x81"
			  "\x36\x46\xB5\x42\x52\x2F\x32\xEB"
			  "\x4A\x3D\xF1\x8F\x1C\x54\x2E\xFC"
			  "\x41\x75\x5A\x8C\x8E\x6F\xE7\x1A"
			  "\xAE\xEF\x3E\x82\x12\x0B\x74\x72"
			  "\xF8\xB2\xAA\x7A\xD6\xFF\xFA\x55"
			  "\x33\x1A\xBB\xD3\xA2\x7E\x97\x66",
		.rlen	= 1008,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 1008 - 16, 16 },
	},
};

static struct cipher_testvec camellia_dec_tv_template[] = {
	{
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.klen	= 16,
		.input	= "\x67\x67\x31\x38\x54\x96\x69\x73"
			  "\x08\x57\x06\x56\x48\xea\xbe\x43",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77",
		.klen	= 24,
		.input	= "\xb4\x99\x34\x01\xb3\xe9\x96\xf8"
			  "\x4e\xe5\xce\xe7\xd7\x9b\x09\xb9",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 16,
	}, {
		.key	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.klen	= 32,
		.input	= "\x9a\xcc\x23\x7d\xff\x16\xd7\x6c"
			  "\x20\xef\x7c\x91\x9e\x3a\x75\x09",
		.ilen	= 16,
		.result	= "\x01\x23\x45\x67\x89\xab\xcd\xef"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.rlen	= 16,
	}, {  
		.key	= "\x3F\x85\x62\x3F\x1C\xF9\xD6\x1C"
			  "\xF9\xD6\xB3\x90\x6D\x4A\x90\x6D"
			  "\x4A\x27\x04\xE1\x27\x04\xE1\xBE"
			  "\x9B\x78\xBE\x9B\x78\x55\x32\x0F",
		.klen	= 32,
		.input	= "\xED\xCD\xDB\xB8\x68\xCE\xBD\xEA"
			  "\x9D\x9D\xCD\x9F\x4F\xFC\x4D\xB7"
			  "\xA5\xFF\x6F\x43\x0F\xBA\x32\x04"
			  "\xB3\xC2\xB9\x03\xAA\x91\x56\x29"
			  "\x0D\xD0\xFD\xC4\x65\xA5\x69\xB9"
			  "\xF1\xF6\xB1\xA5\xB2\x75\x4F\x8A"
			  "\x8D\x7D\x1B\x9B\xC7\x68\x72\xF8"
			  "\x01\x9B\x17\x0A\x29\xE7\x61\x28"
			  "\x7F\xA7\x50\xCA\x20\x2C\x96\x3B"
			  "\x6E\x5C\x5D\x3F\xB5\x7F\xF3\x2B"
			  "\x04\xEF\x9D\xD4\xCE\x41\x28\x8E"
			  "\x83\x54\xAE\x7C\x82\x46\x10\xC9"
			  "\xC4\x8A\x1E\x1F\x4C\xA9\xFC\xEC"
			  "\x3C\x8C\x30\xFC\x59\xD2\x54\xC4"
			  "\x6F\x50\xC6\xCA\x8C\x14\x5B\x9C"
			  "\x18\x56\x5B\xF8\x33\x0E\x4A\xDB"
			  "\xEC\xB5\x6E\x5B\x31\xC4\x0E\x98"
			  "\x9F\x32\xBA\xA2\x18\xCF\x55\x43"
			  "\xFE\x80\x8F\x60\xCF\x05\x30\x9B"
			  "\x70\x50\x1E\x9C\x08\x87\xE6\x20"
			  "\xD2\xF3\x27\xF8\x2A\x8D\x12\xB2"
			  "\xBC\x5F\xFE\x52\x52\xF6\x7F\xB6"
			  "\xB8\x30\x86\x3B\x0F\x94\x1E\x79"
			  "\x13\x94\x35\xA2\xB1\x35\x5B\x05"
			  "\x2A\x98\x6B\x96\x4C\xB1\x20\xBE"
			  "\xB6\x14\xC2\x06\xBF\xFD\x5F\x2A"
			  "\xF5\x33\xC8\x19\x45\x14\x44\x5D"
			  "\xFE\x94\x7B\xBB\x63\x13\x57\xC3"
			  "\x2A\x8F\x6C\x11\x2A\x07\xA7\x6A"
			  "\xBF\x20\xD3\x99\xC6\x00\x0B\xBF"
			  "\x83\x46\x25\x3A\xB0\xF6\xC5\xC8"
			  "\x00\xCA\xE5\x28\x4A\x7C\x95\x9C"
			  "\x7B\x43\xAB\xF9\xE4\xF8\x74\xAB"
			  "\xA7\xB8\x9C\x0F\x53\x7B\xB6\x74"
			  "\x60\x64\x0D\x1C\x80\xD1\x20\x9E"
			  "\xDC\x14\x27\x9B\xFC\xBD\x5C\x96"
			  "\xD2\x51\xDC\x96\xEE\xE5\xEA\x2B"
			  "\x02\x7C\xAA\x3C\xDC\x9D\x7B\x01"
			  "\x20\xC3\xE1\x0B\xDD\xAB\xF3\x1E"
			  "\x19\xA8\x84\x29\x5F\xCC\xC3\x5B"
			  "\xE4\x33\x59\xDC\x12\xEB\x2B\x4D"
			  "\x5B\x55\x23\xB7\x40\x31\xDE\xEE"
			  "\x18\xC9\x3C\x4D\xBC\xED\xE0\x42"
			  "\xAD\xDE\xA0\xA3\xC3\xFE\x44\xD3"
			  "\xE1\x9A\xDA\xAB\x32\xFC\x1A\xBF"
			  "\x63\xA9\xF0\x6A\x08\x46\xBD\x48"
			  "\x83\x06\xAB\x82\x99\x01\x16\x1A"
			  "\x03\x36\xC5\x59\x6B\xB8\x8C\x9F"
			  "\xC6\x51\x3D\xE5\x7F\xBF\xAB\xBC"
			  "\xC9\xA1\x88\x34\x5F\xA9\x7C\x3B"
			  "\x9F\x1B\x98\x2B\x4F\xFB\x9B\xF0"
			  "\xCD\xB6\x45\xB2\x29\x2E\x34\x23"
			  "\xA9\x97\xC0\x22\x8C\x42\x9B\x5F"
			  "\x40\xC8\xD7\x3D\x82\x9A\x6F\xAA"
			  "\x74\x83\x29\x05\xE8\xC4\x4D\x01"
			  "\xB5\xE5\x84\x3F\x7F\xD3\xE0\x99"
			  "\xDA\xE7\x6F\x30\xFD\xAA\x92\x30"
			  "\xA5\x46\x8B\xA2\xE6\x58\x62\x7C"
			  "\x2C\x35\x1B\x38\x85\x7D\xE8\xF3"
			  "\x87\x4F\xDA\xD8\x5F\xFC\xB6\x44"
			  "\xD0\xE3\x9B\x8B\xBF\xD6\xB8\xC4"
			  "\x73\xAE\x1D\x8B\x5B\x74\x8B\xCB"
			  "\xA4\xAD\xCF\x5D\xD4\x58\xC9\xCD"
			  "\xF7\x90\x68\xCF\xC9\x11\x52\x3E"
			  "\xE8\xA1\xA3\x78\x8B\xD0\xAC\x0A"
			  "\xD4\xC9\xA3\xA5\x55\x30\xC8\x3E"
			  "\xED\x28\x39\xE9\x63\xED\x41\x70"
			  "\x51\xE3\xC4\xA0\xFC\xD5\x43\xCB"
			  "\x4D\x65\xC8\xFD\x3A\x91\x8F\x60"
			  "\x8A\xA6\x6D\x9D\x3E\x01\x23\x4B"
			  "\x50\x47\xC9\xDC\x9B\xDE\x37\xC5"
			  "\xBF\x67\xB1\x6B\x78\x38\xD5\x7E"
			  "\xB6\xFF\x67\x83\x3B\x6E\xBE\x23"
			  "\x45\xFA\x1D\x69\x44\xFD\xC6\xB9"
			  "\xD0\x4A\x92\xD1\xBE\xF6\x4A\xB7"
			  "\xCA\xA8\xA2\x9E\x13\x87\x57\x92"
			  "\x64\x7C\x85\x0B\xB3\x29\x37\xD8"
			  "\xE6\xAA\xAF\xC4\x03\x67\xA3\xBF"
			  "\x2E\x45\x83\xB6\xD8\x54\x00\x89"
			  "\xF6\xBC\x3A\x7A\x88\x58\x51\xED"
			  "\xF4\x4E\x01\xA5\xC3\x2E\xD9\x42"
			  "\xBD\x6E\x0D\x0B\x21\xB0\x1A\xCC"
			  "\xA4\xD3\x3F\xDC\x9B\x81\xD8\xF1"
			  "\xEA\x7A\x6A\xB7\x07\xC9\x6D\x91"
			  "\x6D\x3A\xF5\x5F\xA6\xFF\x87\x1E"
			  "\x3F\xDD\xC0\x72\xEA\xAC\x08\x15"
			  "\x21\xE6\xC6\xB6\x0D\xD8\x51\x86"
			  "\x2A\x03\x73\xF7\x29\xD4\xC4\xE4"
			  "\x7F\x95\x10\xF7\xAB\x3F\x92\x23"
			  "\xD3\xCE\x9C\x2E\x46\x3B\x63\x43"
			  "\xBB\xC2\x82\x7A\x83\xD5\x55\xE2"
			  "\xE7\x9B\x2F\x92\xAF\xFD\x81\x56"
			  "\x79\xFD\x3E\xF9\x46\xE0\x25\xD4"
			  "\x38\xDE\xBC\x2C\xC4\x7A\x2A\x8F"
			  "\x94\x4F\xD0\xAD\x9B\x37\x18\xD4"
			  "\x0E\x4D\x0F\x02\x3A\xDC\x5A\xA2"
			  "\x39\x25\x55\x20\x5A\xA6\x02\x9F"
			  "\xE6\x77\x21\x77\xE5\x4B\x7B\x0B"
			  "\x30\xF8\x5F\x33\x0F\x49\xCD\xFF"
			  "\xF2\xE4\x35\xF9\xF0\x63\xC3\x7E"
			  "\xF1\xA6\x73\xB4\xDF\xE7\xBB\x78"
			  "\xFF\x21\xA9\xF3\xF3\xCF\x5D\xBA"
			  "\xED\x87\x98\xAC\xFE\x48\x97\x6D"
			  "\xA6\x7F\x69\x31\xB1\xC4\xFF\x14"
			  "\xC6\x76\xD4\x10\xDD\xF6\x49\x2C"
			  "\x9C\xC8\x6D\x76\xC0\x8F\x5F\x55"
			  "\x2F\x3C\x8A\x30\xAA\xC3\x16\x55"
			  "\xC6\xFC\x8D\x8B\xB9\xE5\x80\x6C"
			  "\xC8\x7E\xBD\x65\x58\x36\xD5\xBC"
			  "\xF0\x33\x52\x29\x70\xF9\x5C\xE9"
			  "\xAC\x1F\xB5\x73\x56\x66\x54\xAF"
			  "\x1B\x8F\x7D\xED\xAB\x03\xCE\xE3"
			  "\xAE\x47\xB6\x69\x86\xE9\x01\x31"
			  "\x83\x18\x3D\xF4\x74\x7B\xF9\x42"
			  "\x4C\xFD\x75\x4A\x6D\xF0\x03\xA6"
			  "\x2B\x20\x63\xDA\x49\x65\x5E\x8B"
			  "\xC0\x19\xE3\x8D\xD9\xF3\xB0\x34"
			  "\xD3\x52\xFC\x68\x00\x43\x1B\x37"
			  "\x31\x93\x51\x1C\x63\x97\x70\xB0"
			  "\x99\x78\x83\x13\xFD\xCF\x53\x81"
			  "\x36\x46\xB5\x42\x52\x2F\x32\xEB"
			  "\x4A\x3D\xF1\x8F\x1C\x54\x2E\xFC"
			  "\x41\x75\x5A\x8C\x8E\x6F\xE7\x1A"
			  "\xAE\xEF\x3E\x82\x12\x0B\x74\x72"
			  "\xF8\xB2\xAA\x7A\xD6\xFF\xFA\x55"
			  "\x33\x1A\xBB\xD3\xA2\x7E\x97\x66",
		.ilen	= 1008,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D",
		.rlen	= 1008,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 1008 - 16, 16 },
	},
};

static struct cipher_testvec camellia_cbc_enc_tv_template[] = {
	{
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input	= "Single block msg",
		.ilen   = 16,
		.result = "\xea\x32\x12\x76\x3b\x50\x10\xe7"
			  "\x18\xf6\xfd\x5d\xf6\x8f\x13\x51",
		.rlen   = 16,
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xa5\xdf\x6e\x50\xda\x70\x6c\x01"
			  "\x4a\xab\xf3\xf2\xd6\xfc\x6c\xfd"
			  "\x19\xb4\x3e\x57\x1c\x02\x5e\xa0"
			  "\x15\x78\xe0\x5e\xf2\xcb\x87\x16",
		.rlen   = 32,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D",
		.ilen	= 1008,
		.result	= "\xCD\x3E\x2A\x3B\x3E\x94\xC5\x77"
			  "\xBA\xBB\x5B\xB1\xDE\x7B\xA4\x40"
			  "\x88\x39\xE3\xFD\x94\x4B\x25\x58"
			  "\xE1\x4B\xC4\x18\x7A\xFD\x17\x2B"
			  "\xB9\xF9\xC2\x27\x6A\xB6\x31\x27"
			  "\xA6\xAD\xEF\xE5\x5D\xE4\x02\x01"
			  "\x56\x2E\x10\xC2\x2C\xFF\xC6\x83"
			  "\xB5\xDC\x4F\x63\xAD\x0E\x63\x5E"
			  "\x56\xC8\x18\x3D\x79\x86\x97\xEF"
			  "\x57\x0E\x63\xA1\xC1\x41\x48\xB8"
			  "\x98\xB7\x51\x6D\x18\xF6\x19\x82"
			  "\x37\x49\x88\xA4\xEF\x91\x21\x47"
			  "\x03\x28\xEA\x42\xF4\xFB\x7A\x58"
			  "\x28\x90\x77\x46\xD8\xD2\x35\x16"
			  "\x44\xA9\x9E\x49\x52\x2A\xE4\x16"
			  "\x5D\xF7\x65\xEB\x0F\xC9\x29\xE6"
			  "\xCF\x76\x91\x89\x8A\x94\x39\xFA"
			  "\x6B\x5F\x63\x53\x74\x43\x91\xF5"
			  "\x3F\xBC\x88\x53\xB2\x1A\x02\x3F"
			  "\x9D\x32\x84\xEB\x56\x28\xD6\x06"
			  "\xD5\xB2\x20\xA9\xFC\xC3\x76\x62"
			  "\x32\xCC\x86\xC8\x36\x67\x5E\x7E"
			  "\xA4\xAA\x15\x63\x6B\xA9\x86\xAF"
			  "\x1A\x52\x82\x36\x5F\xF4\x3F\x7A"
			  "\x9B\x78\x62\x3B\x02\x28\x60\xB3"
			  "\xBA\x82\xB1\xDD\xC9\x60\x8F\x47"
			  "\xF1\x6B\xFE\xE5\x39\x34\xA0\x28"
			  "\xA4\xB3\xC9\x7E\xED\x28\x8D\x70"
			  "\xB2\x1D\xFD\xC6\x00\xCF\x1A\x94"
			  "\x28\xF8\xC1\x34\xB7\x58\xA5\x6C"
			  "\x1A\x9D\xE4\xE4\xF6\xB9\xB4\xB0"
			  "\x5D\x51\x54\x9A\x53\xA0\xF9\x32"
			  "\xBD\x31\x54\x14\x7B\x33\xEE\x17"
			  "\xD3\xC7\x1F\x48\xBF\x0B\x22\xA2"
			  "\x7D\x0C\xDF\xD0\x2E\x98\xFA\xD2"
			  "\xFA\xCF\x24\x1D\x99\x9B\xD0\x7E"
			  "\xF4\x4F\x88\xFF\x45\x99\x4A\xF4"
			  "\xF2\x0A\x5B\x3B\x21\xAB\x92\xAE"
			  "\x40\x78\x91\x95\xC4\x2F\xA3\xE8"
			  "\x18\xC7\x07\xA6\xC8\xC0\x66\x33"
			  "\x35\xC0\xB4\xA0\xF8\xEE\x1E\xF3"
			  "\x40\xF5\x40\x54\xF1\x84\x8C\xEA"
			  "\x27\x38\x1F\xF8\x77\xC7\xDF\xD8"
			  "\x1D\xE2\xD9\x59\x40\x4F\x59\xD4"
			  "\xF8\x17\x99\x8D\x58\x2D\x72\x44"
			  "\x9D\x1D\x91\x64\xD6\x3F\x0A\x82"
			  "\xC7\x57\x3D\xEF\xD3\x41\xFA\xA7"
			  "\x68\xA3\xB8\xA5\x93\x74\x2E\x85"
			  "\x4C\x9D\x69\x59\xCE\x15\xAE\xBF"
			  "\x9C\x8F\x14\x64\x5D\x7F\xCF\x0B"
			  "\xCE\x43\x5D\x28\xC0\x2F\xFB\x18"
			  "\x79\x9A\xFC\x43\x16\x7C\x6B\x7B"
			  "\x38\xB8\x48\x36\x66\x4E\x20\x43"
			  "\xBA\x76\x13\x9A\xC3\xF2\xEB\x52"
			  "\xD7\xDC\xB2\x67\x63\x14\x25\xCD"
			  "\xB1\x13\x4B\xDE\x8C\x59\x21\x84"
			  "\x81\x8D\x97\x23\x45\x33\x7C\xF3"
			  "\xC5\xBC\x79\x95\xAA\x84\x68\x31"
			  "\x2D\x1A\x68\xFE\xEC\x92\x94\xDA"
			  "\x94\x2A\x6F\xD6\xFE\xE5\x76\x97"
			  "\xF4\x6E\xEE\xCB\x2B\x95\x4E\x36"
			  "\x5F\x74\x8C\x86\x5B\x71\xD0\x20"
			  "\x78\x1A\x7F\x18\x8C\xD9\xCD\xF5"
			  "\x21\x41\x56\x72\x13\xE1\x86\x07"
			  "\x07\x26\xF3\x4F\x7B\xEA\xB5\x18"
			  "\xFE\x94\x2D\x9F\xE0\x72\x18\x65"
			  "\xB2\xA5\x63\x48\xB4\x13\x22\xF7"
			  "\x25\xF1\x80\xA8\x7F\x54\x86\x7B"
			  "\x39\xAE\x95\x0C\x09\x32\x22\x2D"
			  "\x4D\x73\x39\x0C\x09\x2C\x7C\x10"
			  "\xD0\x4B\x53\xF6\x90\xC5\x99\x2F"
			  "\x15\xE1\x7F\xC6\xC5\x7A\x52\x14"
			  "\x65\xEE\x93\x54\xD0\x66\x15\x3C"
			  "\x4C\x68\xFD\x64\x0F\xF9\x10\x39"
			  "\x46\x7A\xDD\x97\x20\xEE\xC7\xD2"
			  "\x98\x4A\xB6\xE6\xF5\xA8\x1F\x4F"
			  "\xDB\xAB\x6D\xD5\x9B\x34\x16\x97"
			  "\x2F\x64\xE5\x37\xEF\x0E\xA1\xE9"
			  "\xBE\x31\x31\x96\x8B\x40\x18\x75"
			  "\x11\x75\x14\x32\xA5\x2D\x1B\x6B"
			  "\xDB\x59\xEB\xFA\x3D\x8E\x7C\xC4"
			  "\xDE\x68\xC8\x9F\xC9\x99\xE3\xC6"
			  "\x71\xB0\x12\x57\x89\x0D\xC0\x2B"
			  "\x9F\x12\x6A\x04\x67\xF1\x95\x31"
			  "\x59\xFD\x84\x95\x2C\x9C\x5B\xEC"
			  "\x09\xB0\x43\x96\x4A\x64\x80\x40"
			  "\xB9\x72\x19\xDD\x70\x42\xFA\xB1"
			  "\x4A\x2C\x0C\x0A\x60\x6E\xE3\x7C"
			  "\x37\x5A\xBE\xA4\x62\xCF\x29\xAB"
			  "\x7F\x4D\xA6\xB3\xE2\xB6\x64\xC6"
			  "\x33\x0B\xF3\xD5\x01\x38\x74\xA4"
			  "\x67\x1E\x75\x68\xC3\xAD\x76\xE9"
			  "\xE9\xBC\xF0\xEB\xD8\xFD\x31\x8A"
			  "\x5F\xC9\x18\x94\x4B\x86\x66\xFC"
			  "\xBD\x0B\x3D\xB3\x9F\xFA\x1F\xD9"
			  "\x78\xC4\xE3\x24\x1C\x67\xA2\xF8"
			  "\x43\xBC\x76\x75\xBF\x6C\x05\xB3"
			  "\x32\xE8\x7C\x80\xDB\xC7\xB6\x61"
			  "\x1A\x3E\x2B\xA7\x25\xED\x8F\xA0"
			  "\x00\x4B\xF8\x90\xCA\xD8\xFB\x12"
			  "\xAC\x1F\x18\xE9\xD2\x5E\xA2\x8E"
			  "\xE4\x84\x6B\x9D\xEB\x1E\x6B\xA3"
			  "\x7B\xDC\xCE\x15\x97\x27\xB2\x65"
			  "\xBC\x0E\x47\xAB\x55\x13\x53\xAB"
			  "\x0E\x34\x55\x02\x5F\x27\xC5\x89"
			  "\xDF\xC5\x70\xC4\xDD\x76\x82\xEE"
			  "\x68\xA6\x09\xB0\xE5\x5E\xF1\x0C"
			  "\xE3\xF3\x09\x9B\xFE\x65\x4B\xB8"
			  "\x30\xEC\xD5\x7C\x6A\xEC\x1D\xD2"
			  "\x93\xB7\xA1\x1A\x02\xD4\xC0\xD6"
			  "\x8D\x4D\x83\x9A\xED\x29\x4E\x14"
			  "\x86\xD5\x3C\x1A\xD5\xB9\x0A\x6A"
			  "\x72\x22\xD5\x92\x38\xF1\xA1\x86"
			  "\xB2\x41\x51\xCA\x4E\xAB\x8F\xD3"
			  "\x80\x56\xC3\xD7\x65\xE1\xB3\x86"
			  "\xCB\xCE\x98\xA1\xD4\x59\x1C\x06"
			  "\x01\xED\xF8\x29\x91\x19\x5C\x9A"
			  "\xEE\x28\x1B\x48\xD7\x32\xEF\x9F"
			  "\x6C\x2B\x66\x4E\x78\xD5\x8B\x72"
			  "\x80\xE7\x29\xDC\x23\x55\x98\x54"
			  "\xB1\xFF\x3E\x95\x56\xA8\x78\x78"
			  "\xEF\xC4\xA5\x11\x2D\x2B\xD8\x93"
			  "\x30\x6E\x7E\x51\xBB\x42\x5F\x03"
			  "\x43\x94\x23\x7E\xEE\xF0\xA5\x79"
			  "\x55\x01\xD4\x58\xB2\xF2\x85\x49"
			  "\x70\xC5\xB9\x0B\x3B\x7A\x6E\x6C",
		.rlen	= 1008,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 1008 - 16, 16 },
	},
};

static struct cipher_testvec camellia_cbc_dec_tv_template[] = {
	{
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input	= "\xea\x32\x12\x76\x3b\x50\x10\xe7"
			  "\x18\xf6\xfd\x5d\xf6\x8f\x13\x51",
		.ilen   = 16,
		.result = "Single block msg",
		.rlen   = 16,
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input = "\xa5\xdf\x6e\x50\xda\x70\x6c\x01"
			  "\x4a\xab\xf3\xf2\xd6\xfc\x6c\xfd"
			  "\x19\xb4\x3e\x57\x1c\x02\x5e\xa0"
			  "\x15\x78\xe0\x5e\xf2\xcb\x87\x16",
		.ilen   = 32,
		.result = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen   = 32,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xCD\x3E\x2A\x3B\x3E\x94\xC5\x77"
			  "\xBA\xBB\x5B\xB1\xDE\x7B\xA4\x40"
			  "\x88\x39\xE3\xFD\x94\x4B\x25\x58"
			  "\xE1\x4B\xC4\x18\x7A\xFD\x17\x2B"
			  "\xB9\xF9\xC2\x27\x6A\xB6\x31\x27"
			  "\xA6\xAD\xEF\xE5\x5D\xE4\x02\x01"
			  "\x56\x2E\x10\xC2\x2C\xFF\xC6\x83"
			  "\xB5\xDC\x4F\x63\xAD\x0E\x63\x5E"
			  "\x56\xC8\x18\x3D\x79\x86\x97\xEF"
			  "\x57\x0E\x63\xA1\xC1\x41\x48\xB8"
			  "\x98\xB7\x51\x6D\x18\xF6\x19\x82"
			  "\x37\x49\x88\xA4\xEF\x91\x21\x47"
			  "\x03\x28\xEA\x42\xF4\xFB\x7A\x58"
			  "\x28\x90\x77\x46\xD8\xD2\x35\x16"
			  "\x44\xA9\x9E\x49\x52\x2A\xE4\x16"
			  "\x5D\xF7\x65\xEB\x0F\xC9\x29\xE6"
			  "\xCF\x76\x91\x89\x8A\x94\x39\xFA"
			  "\x6B\x5F\x63\x53\x74\x43\x91\xF5"
			  "\x3F\xBC\x88\x53\xB2\x1A\x02\x3F"
			  "\x9D\x32\x84\xEB\x56\x28\xD6\x06"
			  "\xD5\xB2\x20\xA9\xFC\xC3\x76\x62"
			  "\x32\xCC\x86\xC8\x36\x67\x5E\x7E"
			  "\xA4\xAA\x15\x63\x6B\xA9\x86\xAF"
			  "\x1A\x52\x82\x36\x5F\xF4\x3F\x7A"
			  "\x9B\x78\x62\x3B\x02\x28\x60\xB3"
			  "\xBA\x82\xB1\xDD\xC9\x60\x8F\x47"
			  "\xF1\x6B\xFE\xE5\x39\x34\xA0\x28"
			  "\xA4\xB3\xC9\x7E\xED\x28\x8D\x70"
			  "\xB2\x1D\xFD\xC6\x00\xCF\x1A\x94"
			  "\x28\xF8\xC1\x34\xB7\x58\xA5\x6C"
			  "\x1A\x9D\xE4\xE4\xF6\xB9\xB4\xB0"
			  "\x5D\x51\x54\x9A\x53\xA0\xF9\x32"
			  "\xBD\x31\x54\x14\x7B\x33\xEE\x17"
			  "\xD3\xC7\x1F\x48\xBF\x0B\x22\xA2"
			  "\x7D\x0C\xDF\xD0\x2E\x98\xFA\xD2"
			  "\xFA\xCF\x24\x1D\x99\x9B\xD0\x7E"
			  "\xF4\x4F\x88\xFF\x45\x99\x4A\xF4"
			  "\xF2\x0A\x5B\x3B\x21\xAB\x92\xAE"
			  "\x40\x78\x91\x95\xC4\x2F\xA3\xE8"
			  "\x18\xC7\x07\xA6\xC8\xC0\x66\x33"
			  "\x35\xC0\xB4\xA0\xF8\xEE\x1E\xF3"
			  "\x40\xF5\x40\x54\xF1\x84\x8C\xEA"
			  "\x27\x38\x1F\xF8\x77\xC7\xDF\xD8"
			  "\x1D\xE2\xD9\x59\x40\x4F\x59\xD4"
			  "\xF8\x17\x99\x8D\x58\x2D\x72\x44"
			  "\x9D\x1D\x91\x64\xD6\x3F\x0A\x82"
			  "\xC7\x57\x3D\xEF\xD3\x41\xFA\xA7"
			  "\x68\xA3\xB8\xA5\x93\x74\x2E\x85"
			  "\x4C\x9D\x69\x59\xCE\x15\xAE\xBF"
			  "\x9C\x8F\x14\x64\x5D\x7F\xCF\x0B"
			  "\xCE\x43\x5D\x28\xC0\x2F\xFB\x18"
			  "\x79\x9A\xFC\x43\x16\x7C\x6B\x7B"
			  "\x38\xB8\x48\x36\x66\x4E\x20\x43"
			  "\xBA\x76\x13\x9A\xC3\xF2\xEB\x52"
			  "\xD7\xDC\xB2\x67\x63\x14\x25\xCD"
			  "\xB1\x13\x4B\xDE\x8C\x59\x21\x84"
			  "\x81\x8D\x97\x23\x45\x33\x7C\xF3"
			  "\xC5\xBC\x79\x95\xAA\x84\x68\x31"
			  "\x2D\x1A\x68\xFE\xEC\x92\x94\xDA"
			  "\x94\x2A\x6F\xD6\xFE\xE5\x76\x97"
			  "\xF4\x6E\xEE\xCB\x2B\x95\x4E\x36"
			  "\x5F\x74\x8C\x86\x5B\x71\xD0\x20"
			  "\x78\x1A\x7F\x18\x8C\xD9\xCD\xF5"
			  "\x21\x41\x56\x72\x13\xE1\x86\x07"
			  "\x07\x26\xF3\x4F\x7B\xEA\xB5\x18"
			  "\xFE\x94\x2D\x9F\xE0\x72\x18\x65"
			  "\xB2\xA5\x63\x48\xB4\x13\x22\xF7"
			  "\x25\xF1\x80\xA8\x7F\x54\x86\x7B"
			  "\x39\xAE\x95\x0C\x09\x32\x22\x2D"
			  "\x4D\x73\x39\x0C\x09\x2C\x7C\x10"
			  "\xD0\x4B\x53\xF6\x90\xC5\x99\x2F"
			  "\x15\xE1\x7F\xC6\xC5\x7A\x52\x14"
			  "\x65\xEE\x93\x54\xD0\x66\x15\x3C"
			  "\x4C\x68\xFD\x64\x0F\xF9\x10\x39"
			  "\x46\x7A\xDD\x97\x20\xEE\xC7\xD2"
			  "\x98\x4A\xB6\xE6\xF5\xA8\x1F\x4F"
			  "\xDB\xAB\x6D\xD5\x9B\x34\x16\x97"
			  "\x2F\x64\xE5\x37\xEF\x0E\xA1\xE9"
			  "\xBE\x31\x31\x96\x8B\x40\x18\x75"
			  "\x11\x75\x14\x32\xA5\x2D\x1B\x6B"
			  "\xDB\x59\xEB\xFA\x3D\x8E\x7C\xC4"
			  "\xDE\x68\xC8\x9F\xC9\x99\xE3\xC6"
			  "\x71\xB0\x12\x57\x89\x0D\xC0\x2B"
			  "\x9F\x12\x6A\x04\x67\xF1\x95\x31"
			  "\x59\xFD\x84\x95\x2C\x9C\x5B\xEC"
			  "\x09\xB0\x43\x96\x4A\x64\x80\x40"
			  "\xB9\x72\x19\xDD\x70\x42\xFA\xB1"
			  "\x4A\x2C\x0C\x0A\x60\x6E\xE3\x7C"
			  "\x37\x5A\xBE\xA4\x62\xCF\x29\xAB"
			  "\x7F\x4D\xA6\xB3\xE2\xB6\x64\xC6"
			  "\x33\x0B\xF3\xD5\x01\x38\x74\xA4"
			  "\x67\x1E\x75\x68\xC3\xAD\x76\xE9"
			  "\xE9\xBC\xF0\xEB\xD8\xFD\x31\x8A"
			  "\x5F\xC9\x18\x94\x4B\x86\x66\xFC"
			  "\xBD\x0B\x3D\xB3\x9F\xFA\x1F\xD9"
			  "\x78\xC4\xE3\x24\x1C\x67\xA2\xF8"
			  "\x43\xBC\x76\x75\xBF\x6C\x05\xB3"
			  "\x32\xE8\x7C\x80\xDB\xC7\xB6\x61"
			  "\x1A\x3E\x2B\xA7\x25\xED\x8F\xA0"
			  "\x00\x4B\xF8\x90\xCA\xD8\xFB\x12"
			  "\xAC\x1F\x18\xE9\xD2\x5E\xA2\x8E"
			  "\xE4\x84\x6B\x9D\xEB\x1E\x6B\xA3"
			  "\x7B\xDC\xCE\x15\x97\x27\xB2\x65"
			  "\xBC\x0E\x47\xAB\x55\x13\x53\xAB"
			  "\x0E\x34\x55\x02\x5F\x27\xC5\x89"
			  "\xDF\xC5\x70\xC4\xDD\x76\x82\xEE"
			  "\x68\xA6\x09\xB0\xE5\x5E\xF1\x0C"
			  "\xE3\xF3\x09\x9B\xFE\x65\x4B\xB8"
			  "\x30\xEC\xD5\x7C\x6A\xEC\x1D\xD2"
			  "\x93\xB7\xA1\x1A\x02\xD4\xC0\xD6"
			  "\x8D\x4D\x83\x9A\xED\x29\x4E\x14"
			  "\x86\xD5\x3C\x1A\xD5\xB9\x0A\x6A"
			  "\x72\x22\xD5\x92\x38\xF1\xA1\x86"
			  "\xB2\x41\x51\xCA\x4E\xAB\x8F\xD3"
			  "\x80\x56\xC3\xD7\x65\xE1\xB3\x86"
			  "\xCB\xCE\x98\xA1\xD4\x59\x1C\x06"
			  "\x01\xED\xF8\x29\x91\x19\x5C\x9A"
			  "\xEE\x28\x1B\x48\xD7\x32\xEF\x9F"
			  "\x6C\x2B\x66\x4E\x78\xD5\x8B\x72"
			  "\x80\xE7\x29\xDC\x23\x55\x98\x54"
			  "\xB1\xFF\x3E\x95\x56\xA8\x78\x78"
			  "\xEF\xC4\xA5\x11\x2D\x2B\xD8\x93"
			  "\x30\x6E\x7E\x51\xBB\x42\x5F\x03"
			  "\x43\x94\x23\x7E\xEE\xF0\xA5\x79"
			  "\x55\x01\xD4\x58\xB2\xF2\x85\x49"
			  "\x70\xC5\xB9\x0B\x3B\x7A\x6E\x6C",
		.ilen	= 1008,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D",
		.rlen	= 1008,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 1008 - 16, 16 },
	},
};

static struct cipher_testvec camellia_ctr_enc_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.ilen	= 496,
		.result	= "\xF3\x06\x3A\x84\xCD\xBA\x8E\x11"
			  "\xB7\x74\x6F\x5C\x97\xFB\x36\xFE"
			  "\xDE\x71\x58\xD4\x15\xD1\xC1\xA4"
			  "\xC9\x28\x74\xA6\x6B\xC7\x95\xA6"
			  "\x6C\x77\xF7\x2F\xDF\xC7\xBB\x85"
			  "\x60\xFC\xE8\x94\xE8\xB5\x09\x2C"
			  "\x1E\x43\xEF\x6C\xE9\x98\xC5\xA0"
			  "\x7B\x13\xE5\x7F\xF8\x49\x9A\x8C"
			  "\xE6\x7B\x08\xC3\x32\x66\x55\x4E"
			  "\xA5\x44\x1D\x2C\x18\xC7\x29\x1F"
			  "\x61\x28\x4A\xE3\xCD\xE5\x47\xB2"
			  "\x82\x2F\x66\x83\x91\x51\xAE\xD7"
			  "\x1C\x91\x3C\x57\xE3\x1D\x5A\xC9"
			  "\xFD\xC5\x58\x58\xEF\xCC\x33\xC9"
			  "\x0F\xEA\x26\x32\xD1\x15\x19\x2D"
			  "\x25\xB4\x7F\xB0\xDF\xFB\x88\x60"
			  "\x4E\x4D\x06\x7D\xCC\x1F\xED\x3B"
			  "\x68\x84\xD5\xB3\x1B\xE7\xB9\xA1"
			  "\x68\x8B\x2C\x1A\x44\xDA\x63\xD3"
			  "\x29\xE9\x59\x32\x1F\x30\x1C\x43"
			  "\xEA\x3A\xA3\x6B\x54\x3C\xAA\x11"
			  "\xAD\x38\x20\xC9\xB9\x8A\x64\x66"
			  "\x5A\x07\x49\xDF\xA1\x9C\xF9\x76"
			  "\x36\x65\xB6\x81\x8F\x76\x09\xE5"
			  "\xEB\xD1\x29\xA4\xE4\xF4\x4C\xCD"
			  "\xAF\xFC\xB9\x16\xD9\xC3\x73\x6A"
			  "\x33\x12\xF8\x7E\xBC\xCC\x7D\x80"
			  "\xBF\x3C\x25\x06\x13\x84\xFA\x35"
			  "\xF7\x40\xFA\xA1\x44\x13\x70\xD8"
			  "\x01\xF9\x85\x15\x63\xEC\x7D\xB9"
			  "\x02\xD8\xBA\x41\x6C\x92\x68\x66"
			  "\x95\xDD\xD6\x42\xE7\xBB\xE1\xFD"
			  "\x28\x3E\x94\xB6\xBD\xA7\xBF\x47"
			  "\x58\x8D\xFF\x19\x30\x75\x0D\x48"
			  "\x94\xE9\xA6\xCD\xB3\x8E\x1E\xCD"
			  "\x59\xBC\x1A\xAC\x3C\x4F\xA9\xEB"
			  "\xF4\xA7\xE4\x75\x4A\x18\x40\xC9"
			  "\x1E\xEC\x06\x9C\x28\x4B\xF7\x2B"
			  "\xE2\xEF\xD6\x42\x2E\xBB\xFC\x0A"
			  "\x79\xA2\x99\x28\x93\x1B\x00\x57"
			  "\x35\x1E\x1A\x93\x90\xA4\x68\x95"
			  "\x5E\x57\x40\xD5\xA9\xAA\x19\x48"
			  "\xEC\xFF\x76\x77\xDC\x78\x89\x76"
			  "\xE5\x3B\x00\xEC\x58\x4D\xD1\xE3"
			  "\xC8\x6C\x2C\x45\x5E\x5F\xD9\x4E"
			  "\x71\xA5\x36\x6D\x03\xF1\xC7\xD5"
			  "\xF3\x63\xC0\xD8\xCB\x2B\xF1\xA8"
			  "\xB9\x2B\xE6\x0B\xB9\x65\x78\xA0"
			  "\xC4\x46\xE6\x9B\x8B\x43\x2D\xAB"
			  "\x70\xA6\xE0\x59\x1E\xAC\x9D\xE0"
			  "\x76\x44\x45\xF3\x24\x11\x57\x98"
			  "\x9A\x86\xB4\x12\x80\x28\x86\x20"
			  "\x23\x9D\x2D\xE9\x38\x32\xB1\xE1"
			  "\xCF\x0A\x23\x73\x7D\xC5\x80\x3D"
			  "\x9F\x6D\xA0\xD0\xEE\x93\x8A\x79"
			  "\x3A\xDD\x1D\xBB\x9E\x26\x5D\x01"
			  "\x44\xD0\xD4\x4E\xC3\xF1\xE4\x38"
			  "\x09\x62\x0A\x1A\x4E\xD2\x63\x0F"
			  "\x6E\x3E\xD2\xA4\x3A\xF4\xF3\xFF"
			  "\x7E\x42\xEC\xB6\x6F\x4D\x6B\x48"
			  "\xE6\xA6\x50\x80\x78\x9E\xF1\xB0"
			  "\x4D\xB2\x0D\x3D\xFC\x40\x25\x4D",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D"
			  "\xE4\x7B\x12",
		.ilen	= 1011,
		.result	= "\xF3\x06\x3A\x84\xCD\xBA\x8E\x11"
			  "\xB7\x74\x6F\x5C\x97\xFB\x36\xFE"
			  "\xDE\x71\x58\xD4\x15\xD1\xC1\xA4"
			  "\xC9\x28\x74\xA6\x6B\xC7\x95\xA6"
			  "\x6C\x77\xF7\x2F\xDF\xC7\xBB\x85"
			  "\x60\xFC\xE8\x94\xE8\xB5\x09\x2C"
			  "\x1E\x43\xEF\x6C\xE9\x98\xC5\xA0"
			  "\x7B\x13\xE5\x7F\xF8\x49\x9A\x8C"
			  "\xE6\x7B\x08\xC3\x32\x66\x55\x4E"
			  "\xA5\x44\x1D\x2C\x18\xC7\x29\x1F"
			  "\x61\x28\x4A\xE3\xCD\xE5\x47\xB2"
			  "\x82\x2F\x66\x83\x91\x51\xAE\xD7"
			  "\x1C\x91\x3C\x57\xE3\x1D\x5A\xC9"
			  "\xFD\xC5\x58\x58\xEF\xCC\x33\xC9"
			  "\x0F\xEA\x26\x32\xD1\x15\x19\x2D"
			  "\x25\xB4\x7F\xB0\xDF\xFB\x88\x60"
			  "\x4E\x4D\x06\x7D\xCC\x1F\xED\x3B"
			  "\x68\x84\xD5\xB3\x1B\xE7\xB9\xA1"
			  "\x68\x8B\x2C\x1A\x44\xDA\x63\xD3"
			  "\x29\xE9\x59\x32\x1F\x30\x1C\x43"
			  "\xEA\x3A\xA3\x6B\x54\x3C\xAA\x11"
			  "\xAD\x38\x20\xC9\xB9\x8A\x64\x66"
			  "\x5A\x07\x49\xDF\xA1\x9C\xF9\x76"
			  "\x36\x65\xB6\x81\x8F\x76\x09\xE5"
			  "\xEB\xD1\x29\xA4\xE4\xF4\x4C\xCD"
			  "\xAF\xFC\xB9\x16\xD9\xC3\x73\x6A"
			  "\x33\x12\xF8\x7E\xBC\xCC\x7D\x80"
			  "\xBF\x3C\x25\x06\x13\x84\xFA\x35"
			  "\xF7\x40\xFA\xA1\x44\x13\x70\xD8"
			  "\x01\xF9\x85\x15\x63\xEC\x7D\xB9"
			  "\x02\xD8\xBA\x41\x6C\x92\x68\x66"
			  "\x95\xDD\xD6\x42\xE7\xBB\xE1\xFD"
			  "\x28\x3E\x94\xB6\xBD\xA7\xBF\x47"
			  "\x58\x8D\xFF\x19\x30\x75\x0D\x48"
			  "\x94\xE9\xA6\xCD\xB3\x8E\x1E\xCD"
			  "\x59\xBC\x1A\xAC\x3C\x4F\xA9\xEB"
			  "\xF4\xA7\xE4\x75\x4A\x18\x40\xC9"
			  "\x1E\xEC\x06\x9C\x28\x4B\xF7\x2B"
			  "\xE2\xEF\xD6\x42\x2E\xBB\xFC\x0A"
			  "\x79\xA2\x99\x28\x93\x1B\x00\x57"
			  "\x35\x1E\x1A\x93\x90\xA4\x68\x95"
			  "\x5E\x57\x40\xD5\xA9\xAA\x19\x48"
			  "\xEC\xFF\x76\x77\xDC\x78\x89\x76"
			  "\xE5\x3B\x00\xEC\x58\x4D\xD1\xE3"
			  "\xC8\x6C\x2C\x45\x5E\x5F\xD9\x4E"
			  "\x71\xA5\x36\x6D\x03\xF1\xC7\xD5"
			  "\xF3\x63\xC0\xD8\xCB\x2B\xF1\xA8"
			  "\xB9\x2B\xE6\x0B\xB9\x65\x78\xA0"
			  "\xC4\x46\xE6\x9B\x8B\x43\x2D\xAB"
			  "\x70\xA6\xE0\x59\x1E\xAC\x9D\xE0"
			  "\x76\x44\x45\xF3\x24\x11\x57\x98"
			  "\x9A\x86\xB4\x12\x80\x28\x86\x20"
			  "\x23\x9D\x2D\xE9\x38\x32\xB1\xE1"
			  "\xCF\x0A\x23\x73\x7D\xC5\x80\x3D"
			  "\x9F\x6D\xA0\xD0\xEE\x93\x8A\x79"
			  "\x3A\xDD\x1D\xBB\x9E\x26\x5D\x01"
			  "\x44\xD0\xD4\x4E\xC3\xF1\xE4\x38"
			  "\x09\x62\x0A\x1A\x4E\xD2\x63\x0F"
			  "\x6E\x3E\xD2\xA4\x3A\xF4\xF3\xFF"
			  "\x7E\x42\xEC\xB6\x6F\x4D\x6B\x48"
			  "\xE6\xA6\x50\x80\x78\x9E\xF1\xB0"
			  "\x4D\xB2\x0D\x3D\xFC\x40\x25\x4D"
			  "\x93\x11\x1C\xE9\xD2\x9F\x6E\x90"
			  "\xE5\x41\x4A\xE2\x3C\x45\x29\x35"
			  "\xEC\xD6\x47\x50\xCB\x7B\xA2\x32"
			  "\xF7\x8B\x62\xF1\xE3\x9A\xFE\xC7"
			  "\x1D\x8C\x02\x72\x68\x09\xE9\xB6"
			  "\x4A\x80\xE6\xB1\x56\xDF\x90\xD4"
			  "\x93\x74\xA4\xCE\x20\x23\xBF\x48"
			  "\xA5\xDE\x1B\xFA\x40\x69\x31\x98"
			  "\x62\x6E\xA5\xC7\xBF\x0C\x62\xE5"
			  "\x6D\xE1\x93\xF1\x83\x10\x1C\xCA"
			  "\xF6\x5C\x19\xF8\x90\x78\xCB\xE4"
			  "\x0B\x3A\xB5\xF8\x43\x86\xD3\x3F"
			  "\xBA\x83\x34\x3C\x42\xCC\x7D\x28"
			  "\x29\x63\x4F\xD8\x02\x17\xC5\x07"
			  "\x2C\xA4\xAC\x79\xCB\xC3\xA9\x09"
			  "\x81\x45\x18\xED\xE4\xCB\x42\x3B"
			  "\x87\x2D\x23\xDC\xC5\xBA\x45\xBD"
			  "\x92\xE5\x02\x97\x96\xCE\xAD\xEC"
			  "\xBA\xD8\x76\xF8\xCA\xC1\x31\xEC"
			  "\x1E\x4F\x3F\x83\xF8\x33\xE8\x6E"
			  "\xCC\xF8\x5F\xDD\x65\x50\x99\x69"
			  "\xAF\x48\xCE\xA5\xBA\xB6\x14\x9F"
			  "\x05\x93\xB2\xE6\x59\xC8\x28\xFE"
			  "\x8F\x37\xF9\x64\xB9\xA5\x56\x8F"
			  "\xF1\x1B\x90\xEF\xAE\xEB\xFC\x09"
			  "\x11\x7A\xF2\x19\x0A\x0A\x9A\x3C"
			  "\xE2\x5E\x29\xFA\x31\x9B\xC1\x74"
			  "\x1E\x10\x3E\x07\xA9\x31\x6D\xF8"
			  "\x81\xF5\xD5\x8A\x04\x23\x51\xAC"
			  "\xA2\xE2\x63\xFD\x27\x1F\x79\x5B"
			  "\x1F\xE8\xDA\x11\x49\x4D\x1C\xBA"
			  "\x54\xCC\x0F\xBA\x92\x69\xE5\xCB"
			  "\x41\x1A\x67\xA6\x40\x82\x70\x8C"
			  "\x19\x79\x08\xA4\x51\x20\x7D\xC9"
			  "\x12\x27\xAE\x20\x0D\x2C\xA1\x6D"
			  "\xF4\x55\xD4\xE7\xE6\xD4\x28\x08"
			  "\x00\x70\x12\x56\x56\x50\xAD\x14"
			  "\x5C\x3E\xA2\xD1\x36\x3F\x36\x48"
			  "\xED\xB1\x57\x3E\x5D\x15\xF6\x1E"
			  "\x53\xE9\xA4\x3E\xED\x7D\xCF\x7D"
			  "\x29\xAF\xF3\x1E\x51\xA8\x9F\x85"
			  "\x8B\xF0\xBB\xCE\xCC\x39\xC3\x64"
			  "\x4B\xF2\xAD\x70\x19\xD4\x44\x8F"
			  "\x91\x76\xE8\x15\x66\x34\x9F\xF6"
			  "\x0F\x15\xA4\xA8\x24\xF8\x58\xB1"
			  "\x38\x46\x47\xC7\x9B\xCA\xE9\x42"
			  "\x44\xAA\xE6\xB5\x9C\x91\xA4\xD3"
			  "\x16\xA0\xED\x42\xBE\xB5\x06\x19"
			  "\xBE\x67\xE8\xBC\x22\x32\xA4\x1E"
			  "\x93\xEB\xBE\xE9\xE1\x93\xE5\x31"
			  "\x3A\xA2\x75\xDF\xE3\x6B\xE7\xCC"
			  "\xB4\x70\x20\xE0\x6D\x82\x7C\xC8"
			  "\x94\x5C\x5E\x37\x18\xAD\xED\x8B"
			  "\x44\x86\xCA\x5E\x07\xB7\x70\x8D"
			  "\x40\x48\x19\x73\x7C\x78\x64\x0B"
			  "\xDB\x01\xCA\xAE\x63\x19\xE9\xD1"
			  "\x6B\x2C\x84\x10\x45\x42\x2E\xC3"
			  "\xDF\x7F\xAA\xE8\x87\x1B\x63\x46"
			  "\x74\x28\x9D\x05\x30\x20\x62\x41"
			  "\xC0\x9F\x2C\x36\x2B\x78\xD7\x26"
			  "\xDF\x58\x51\xED\xFA\xDC\x87\x79"
			  "\xBF\x8C\xBF\xC4\x0F\xE5\x05\xDA"
			  "\x45\xE3\x35\x0D\x69\x91\x54\x1C"
			  "\xE7\x2C\x49\x08\x8B\x72\xFA\x5C"
			  "\xF1\x6B\xD9",
		.rlen	= 1011,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 1011 - 16, 16 },
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D",
		.ilen	= 1008,
		.result	= "\x85\x79\x6C\x8B\x2B\x6D\x14\xF9"
			  "\xA6\x83\xB6\x80\x5B\x3A\xF3\x7E"
			  "\x30\x29\xEB\x1F\xDC\x19\x5F\xEB"
			  "\xF7\xC4\x27\x04\x51\x87\xD7\x6F"
			  "\xB8\x4E\x07\xFB\xAC\x3B\x08\xB4"
			  "\x4D\xCB\xE8\xE1\x71\x7D\x4F\x48"
			  "\xCD\x81\x64\xA5\xC4\x07\x1A\x9A"
			  "\x4B\x62\x90\x0E\xC8\xB3\x2B\x6B"
			  "\x8F\x9C\x6E\x72\x4B\xBA\xEF\x07"
			  "\x2C\x56\x07\x5E\x37\x30\x60\xA9"
			  "\xE3\xEF\xD6\x69\xE1\xA1\x77\x64"
			  "\x93\x75\x7A\xB7\x7A\x3B\xE9\x43"
			  "\x23\x35\x95\x91\x80\x8A\xC7\xCF"
			  "\xC3\xD5\xBF\xE7\xFE\x4C\x06\x6B"
			  "\x05\x19\x48\xE2\x62\xBA\x4F\xF2"
			  "\xFB\xEE\xE4\xCB\x79\x9D\xA3\x10"
			  "\x1D\x29\x8C\x1D\x7A\x88\x5A\xDD"
			  "\x4E\xB6\x18\xAA\xCD\xE6\x33\x96"
			  "\xD9\x0F\x90\x5A\x78\x76\x4D\x77"
			  "\x3C\x20\x89\x3B\xA3\xF9\x07\xFD"
			  "\xE4\xE8\x20\x2D\x15\x0A\x63\x49"
			  "\xF5\x4F\x89\xD8\xDE\xA1\x28\x78"
			  "\x28\x07\x09\x1B\x03\x94\x1D\x4B"
			  "\x82\x28\x1E\x1D\x95\xBA\xAC\x85"
			  "\x71\x6E\x3C\x18\x4B\x77\x74\x79"
			  "\xBF\x67\x0A\x53\x3C\x94\xD9\x60"
			  "\xE9\x6D\x40\x34\xA0\x2A\x53\x5D"
			  "\x27\xD5\x47\xF9\xC3\x4B\x27\x29"
			  "\xE4\x76\x9C\x3F\xA7\x1C\x87\xFC"
			  "\x6E\x0F\xCF\x9B\x60\xF0\xF0\x8B"
			  "\x70\x1C\x84\x81\x72\x4D\xB4\x98"
			  "\x23\x62\xE7\x6A\x2B\xFC\xA5\xB2"
			  "\xFF\xF5\x71\x07\xCD\x90\x23\x13"
			  "\x19\xD7\x79\x36\x6C\x9D\x55\x8B"
			  "\x93\x78\x86\x05\x69\x46\xD0\xC5"
			  "\x39\x09\xEB\x79\xEF\xFA\x9F\xAE"
			  "\xF3\xD5\x44\xC3\xFD\x86\xD2\x7C"
			  "\x83\x4B\xD8\x75\x9C\x18\x04\x7B"
			  "\x73\xAD\x72\xA4\xF6\xAB\xCF\x4B"
			  "\xCC\x01\x45\x90\xA6\x43\x05\x0C"
			  "\x6C\x4F\x62\x77\x57\x97\x9F\xEE"
			  "\x75\xA7\x3C\x38\xD1\x0F\x3D\x0E"
			  "\x2C\x43\x98\xFB\x13\x65\x73\xE4"
			  "\x3C\x1E\xD6\x90\x08\xF7\xE0\x99"
			  "\x3B\xF1\x9D\x6C\x48\xA9\x0E\x32"
			  "\x17\xC2\xCC\x20\xA1\x19\x26\xAA"
			  "\xE0\x75\x2F\xFB\x54\x66\x0A\xDF"
			  "\xB5\xF2\x1F\xC1\x34\x3C\x30\x56"
			  "\xE8\xDC\xF7\x92\x6B\xBF\x17\x24"
			  "\xEC\x94\xB5\x3B\xD6\xCE\xA2\x54"
			  "\x10\x7F\x50\xDE\x69\x77\xD5\x37"
			  "\xFE\x9C\x10\x83\xC5\xEB\xC9\x53"
			  "\xB7\xF3\xC4\x20\xAF\x0A\x7E\x57"
			  "\x3A\xE6\x75\xFE\x89\x00\x6E\x48"
			  "\xFB\x99\x17\x2C\xF6\x64\x40\x95"
			  "\x5E\xDC\x7A\xA6\x70\xC7\xF4\xDD"
			  "\x52\x05\x24\x34\xF9\x0E\xC8\x64"
			  "\x6D\xE2\xD8\x80\x53\x31\x4C\xFE"
			  "\xB4\x3A\x5F\x19\xCF\x42\x1B\x22"
			  "\x0B\x2D\x7B\xF1\xC5\x43\xF7\x5E"
			  "\x12\xA8\x01\x64\x16\x0B\x26\x5A"
			  "\x0C\x95\x0F\x40\xC5\x5A\x06\x7C"
			  "\xCF\xF5\xD5\xB7\x7A\x34\x23\xB6"
			  "\xAA\x9E\xA8\x98\xA2\xF8\x3D\xD3"
			  "\x3F\x23\x69\x63\x56\x96\x45\xD6"
			  "\x74\x23\x1D\x5C\x63\xCC\xD8\x78"
			  "\x16\xE2\x9C\xD2\x80\x02\xF2\x28"
			  "\x69\x2F\xC4\xA8\x15\x15\x24\x3B"
			  "\xCB\xF0\x14\xE4\x62\xC8\xF3\xD1"
			  "\x03\x58\x1B\x33\x77\x74\x1F\xB4"
			  "\x07\x86\xF2\x21\xB7\x41\xAE\xBF"
			  "\x25\xC2\xFF\x51\xEF\xEA\xCE\xC4"
			  "\x5F\xD9\xB8\x18\x6A\xF0\x0F\x0D"
			  "\xF8\x04\xBB\x6D\x62\x33\x87\x26"
			  "\x4F\x2F\x14\x6E\xDC\xDB\x66\x09"
			  "\x2A\xEF\x7D\x84\x10\xAC\x82\x5E"
			  "\xD2\xE4\xAD\x74\x7A\x6D\xCC\x3A"
			  "\x7B\x62\xD8\xD6\x07\x2D\xF7\xDF"
			  "\x9B\xB3\x82\xCF\x9C\x1D\x76\x5C"
			  "\xAC\x7B\xD4\x9B\x45\xA1\x64\x11"
			  "\x66\xF1\xA7\x0B\xF9\xDD\x00\xDD"
			  "\xA4\x45\x3D\x3E\x03\xC9\x2E\xCB"
			  "\xC3\x14\x84\x72\xFD\x41\xDC\xBD"
			  "\x75\xBE\xA8\xE5\x16\x48\x64\x39"
			  "\xCA\xF3\xE6\xDC\x25\x24\xF1\x6D"
			  "\xB2\x8D\xC5\x38\x54\xD3\x5D\x6D"
			  "\x0B\x29\x10\x15\x0E\x13\x3B\xAC"
			  "\x7E\xCC\x9E\x3E\x18\x48\xA6\x02"
			  "\xEF\x03\xB2\x2E\xE3\xD2\x70\x21"
			  "\xB4\x19\x26\xBE\x3A\x3D\x05\xE0"
			  "\xF8\x09\xAF\xE4\x31\x26\x92\x2F"
			  "\x8F\x55\xAC\xED\x0B\xB2\xA5\x34"
			  "\xBE\x50\xB1\x02\x22\x96\xE3\x40"
			  "\x7B\x70\x50\x6E\x3B\xD5\xE5\xA0"
			  "\x8E\xA2\xAD\x14\x60\x5C\x7A\x2B"
			  "\x3D\x1B\x7F\xC1\xC0\x2C\x56\x36"
			  "\xD2\x0A\x32\x06\x97\x34\xB9\xF4"
			  "\x6F\x9F\x7E\x80\xD0\x9D\xF7\x6A"
			  "\x21\xC1\xA2\x6A\xB1\x96\x5B\x4D"
			  "\x7A\x15\x6C\xC4\x4E\xB8\xE0\x9E"
			  "\x6C\x50\xF3\x9C\xC9\xB5\x23\xB7"
			  "\xF1\xD4\x29\x4A\x23\xC4\xAD\x1E"
			  "\x2C\x07\xD2\x43\x5F\x57\x93\xCA"
			  "\x85\xF9\x9F\xAD\x4C\xF1\xE4\xB1"
			  "\x1A\x8E\x28\xA4\xB6\x52\x77\x7E"
			  "\x68\xC6\x47\xB9\x76\xCC\x65\x5F"
			  "\x0B\xF9\x67\x93\xD8\x0E\x9A\x37"
			  "\x5F\x41\xED\x64\x6C\xAD\x5F\xED"
			  "\x3F\x8D\xFB\x8E\x1E\xA0\xE4\x1F"
			  "\xC2\xC7\xED\x18\x43\xE1\x20\x86"
			  "\x5D\xBC\x30\x70\x22\xA1\xDC\x53"
			  "\x10\x3A\x8D\x47\x82\xCD\x7F\x59"
			  "\x03\x2D\x6D\xF5\xE7\x79\xD4\x07"
			  "\x68\x2A\xA5\x42\x19\x4D\xAF\xF5"
			  "\xED\x47\x83\xBC\x5F\x62\x84\xDA"
			  "\xDA\x41\xFF\xB0\x1D\x64\xA3\xC8"
			  "\xBD\x4E\xE0\xB8\x7F\xEE\x55\x0A"
			  "\x4E\x61\xB2\x51\xF6\x9C\x95\xF6"
			  "\x92\xBB\xF6\xC5\xF0\x09\x86\xDE"
			  "\x37\x9E\x29\xF9\x2A\x18\x73\x0D"
			  "\xDC\x7E\x6B\x7B\x1B\x43\x8C\xEA"
			  "\x13\xC8\x1A\x47\x0A\x2D\x6D\x56"
			  "\xCD\xD2\xE7\x53\x1A\xAB\x1C\x3C"
			  "\xC5\x9B\x03\x70\x29\x2A\x49\x09"
			  "\x67\xA1\xEA\xD6\x3A\x5B\xBF\x71"
			  "\x1D\x48\x64\x6C\xFB\xC0\x9E\x36",
		.rlen	= 1008,
	},
};

static struct cipher_testvec camellia_ctr_dec_tv_template[] = {
	{  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xF3\x06\x3A\x84\xCD\xBA\x8E\x11"
			  "\xB7\x74\x6F\x5C\x97\xFB\x36\xFE"
			  "\xDE\x71\x58\xD4\x15\xD1\xC1\xA4"
			  "\xC9\x28\x74\xA6\x6B\xC7\x95\xA6"
			  "\x6C\x77\xF7\x2F\xDF\xC7\xBB\x85"
			  "\x60\xFC\xE8\x94\xE8\xB5\x09\x2C"
			  "\x1E\x43\xEF\x6C\xE9\x98\xC5\xA0"
			  "\x7B\x13\xE5\x7F\xF8\x49\x9A\x8C"
			  "\xE6\x7B\x08\xC3\x32\x66\x55\x4E"
			  "\xA5\x44\x1D\x2C\x18\xC7\x29\x1F"
			  "\x61\x28\x4A\xE3\xCD\xE5\x47\xB2"
			  "\x82\x2F\x66\x83\x91\x51\xAE\xD7"
			  "\x1C\x91\x3C\x57\xE3\x1D\x5A\xC9"
			  "\xFD\xC5\x58\x58\xEF\xCC\x33\xC9"
			  "\x0F\xEA\x26\x32\xD1\x15\x19\x2D"
			  "\x25\xB4\x7F\xB0\xDF\xFB\x88\x60"
			  "\x4E\x4D\x06\x7D\xCC\x1F\xED\x3B"
			  "\x68\x84\xD5\xB3\x1B\xE7\xB9\xA1"
			  "\x68\x8B\x2C\x1A\x44\xDA\x63\xD3"
			  "\x29\xE9\x59\x32\x1F\x30\x1C\x43"
			  "\xEA\x3A\xA3\x6B\x54\x3C\xAA\x11"
			  "\xAD\x38\x20\xC9\xB9\x8A\x64\x66"
			  "\x5A\x07\x49\xDF\xA1\x9C\xF9\x76"
			  "\x36\x65\xB6\x81\x8F\x76\x09\xE5"
			  "\xEB\xD1\x29\xA4\xE4\xF4\x4C\xCD"
			  "\xAF\xFC\xB9\x16\xD9\xC3\x73\x6A"
			  "\x33\x12\xF8\x7E\xBC\xCC\x7D\x80"
			  "\xBF\x3C\x25\x06\x13\x84\xFA\x35"
			  "\xF7\x40\xFA\xA1\x44\x13\x70\xD8"
			  "\x01\xF9\x85\x15\x63\xEC\x7D\xB9"
			  "\x02\xD8\xBA\x41\x6C\x92\x68\x66"
			  "\x95\xDD\xD6\x42\xE7\xBB\xE1\xFD"
			  "\x28\x3E\x94\xB6\xBD\xA7\xBF\x47"
			  "\x58\x8D\xFF\x19\x30\x75\x0D\x48"
			  "\x94\xE9\xA6\xCD\xB3\x8E\x1E\xCD"
			  "\x59\xBC\x1A\xAC\x3C\x4F\xA9\xEB"
			  "\xF4\xA7\xE4\x75\x4A\x18\x40\xC9"
			  "\x1E\xEC\x06\x9C\x28\x4B\xF7\x2B"
			  "\xE2\xEF\xD6\x42\x2E\xBB\xFC\x0A"
			  "\x79\xA2\x99\x28\x93\x1B\x00\x57"
			  "\x35\x1E\x1A\x93\x90\xA4\x68\x95"
			  "\x5E\x57\x40\xD5\xA9\xAA\x19\x48"
			  "\xEC\xFF\x76\x77\xDC\x78\x89\x76"
			  "\xE5\x3B\x00\xEC\x58\x4D\xD1\xE3"
			  "\xC8\x6C\x2C\x45\x5E\x5F\xD9\x4E"
			  "\x71\xA5\x36\x6D\x03\xF1\xC7\xD5"
			  "\xF3\x63\xC0\xD8\xCB\x2B\xF1\xA8"
			  "\xB9\x2B\xE6\x0B\xB9\x65\x78\xA0"
			  "\xC4\x46\xE6\x9B\x8B\x43\x2D\xAB"
			  "\x70\xA6\xE0\x59\x1E\xAC\x9D\xE0"
			  "\x76\x44\x45\xF3\x24\x11\x57\x98"
			  "\x9A\x86\xB4\x12\x80\x28\x86\x20"
			  "\x23\x9D\x2D\xE9\x38\x32\xB1\xE1"
			  "\xCF\x0A\x23\x73\x7D\xC5\x80\x3D"
			  "\x9F\x6D\xA0\xD0\xEE\x93\x8A\x79"
			  "\x3A\xDD\x1D\xBB\x9E\x26\x5D\x01"
			  "\x44\xD0\xD4\x4E\xC3\xF1\xE4\x38"
			  "\x09\x62\x0A\x1A\x4E\xD2\x63\x0F"
			  "\x6E\x3E\xD2\xA4\x3A\xF4\xF3\xFF"
			  "\x7E\x42\xEC\xB6\x6F\x4D\x6B\x48"
			  "\xE6\xA6\x50\x80\x78\x9E\xF1\xB0"
			  "\x4D\xB2\x0D\x3D\xFC\x40\x25\x4D",
		.ilen	= 496,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7",
		.rlen	= 496,
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xE2\x24\x89\xEE\x53\xB8\x1D\x5F"
			  "\xC4\x29\x8E\xF3\x35\x9A\xFF\x64",
		.input	= "\xF3\x06\x3A\x84\xCD\xBA\x8E\x11"
			  "\xB7\x74\x6F\x5C\x97\xFB\x36\xFE"
			  "\xDE\x71\x58\xD4\x15\xD1\xC1\xA4"
			  "\xC9\x28\x74\xA6\x6B\xC7\x95\xA6"
			  "\x6C\x77\xF7\x2F\xDF\xC7\xBB\x85"
			  "\x60\xFC\xE8\x94\xE8\xB5\x09\x2C"
			  "\x1E\x43\xEF\x6C\xE9\x98\xC5\xA0"
			  "\x7B\x13\xE5\x7F\xF8\x49\x9A\x8C"
			  "\xE6\x7B\x08\xC3\x32\x66\x55\x4E"
			  "\xA5\x44\x1D\x2C\x18\xC7\x29\x1F"
			  "\x61\x28\x4A\xE3\xCD\xE5\x47\xB2"
			  "\x82\x2F\x66\x83\x91\x51\xAE\xD7"
			  "\x1C\x91\x3C\x57\xE3\x1D\x5A\xC9"
			  "\xFD\xC5\x58\x58\xEF\xCC\x33\xC9"
			  "\x0F\xEA\x26\x32\xD1\x15\x19\x2D"
			  "\x25\xB4\x7F\xB0\xDF\xFB\x88\x60"
			  "\x4E\x4D\x06\x7D\xCC\x1F\xED\x3B"
			  "\x68\x84\xD5\xB3\x1B\xE7\xB9\xA1"
			  "\x68\x8B\x2C\x1A\x44\xDA\x63\xD3"
			  "\x29\xE9\x59\x32\x1F\x30\x1C\x43"
			  "\xEA\x3A\xA3\x6B\x54\x3C\xAA\x11"
			  "\xAD\x38\x20\xC9\xB9\x8A\x64\x66"
			  "\x5A\x07\x49\xDF\xA1\x9C\xF9\x76"
			  "\x36\x65\xB6\x81\x8F\x76\x09\xE5"
			  "\xEB\xD1\x29\xA4\xE4\xF4\x4C\xCD"
			  "\xAF\xFC\xB9\x16\xD9\xC3\x73\x6A"
			  "\x33\x12\xF8\x7E\xBC\xCC\x7D\x80"
			  "\xBF\x3C\x25\x06\x13\x84\xFA\x35"
			  "\xF7\x40\xFA\xA1\x44\x13\x70\xD8"
			  "\x01\xF9\x85\x15\x63\xEC\x7D\xB9"
			  "\x02\xD8\xBA\x41\x6C\x92\x68\x66"
			  "\x95\xDD\xD6\x42\xE7\xBB\xE1\xFD"
			  "\x28\x3E\x94\xB6\xBD\xA7\xBF\x47"
			  "\x58\x8D\xFF\x19\x30\x75\x0D\x48"
			  "\x94\xE9\xA6\xCD\xB3\x8E\x1E\xCD"
			  "\x59\xBC\x1A\xAC\x3C\x4F\xA9\xEB"
			  "\xF4\xA7\xE4\x75\x4A\x18\x40\xC9"
			  "\x1E\xEC\x06\x9C\x28\x4B\xF7\x2B"
			  "\xE2\xEF\xD6\x42\x2E\xBB\xFC\x0A"
			  "\x79\xA2\x99\x28\x93\x1B\x00\x57"
			  "\x35\x1E\x1A\x93\x90\xA4\x68\x95"
			  "\x5E\x57\x40\xD5\xA9\xAA\x19\x48"
			  "\xEC\xFF\x76\x77\xDC\x78\x89\x76"
			  "\xE5\x3B\x00\xEC\x58\x4D\xD1\xE3"
			  "\xC8\x6C\x2C\x45\x5E\x5F\xD9\x4E"
			  "\x71\xA5\x36\x6D\x03\xF1\xC7\xD5"
			  "\xF3\x63\xC0\xD8\xCB\x2B\xF1\xA8"
			  "\xB9\x2B\xE6\x0B\xB9\x65\x78\xA0"
			  "\xC4\x46\xE6\x9B\x8B\x43\x2D\xAB"
			  "\x70\xA6\xE0\x59\x1E\xAC\x9D\xE0"
			  "\x76\x44\x45\xF3\x24\x11\x57\x98"
			  "\x9A\x86\xB4\x12\x80\x28\x86\x20"
			  "\x23\x9D\x2D\xE9\x38\x32\xB1\xE1"
			  "\xCF\x0A\x23\x73\x7D\xC5\x80\x3D"
			  "\x9F\x6D\xA0\xD0\xEE\x93\x8A\x79"
			  "\x3A\xDD\x1D\xBB\x9E\x26\x5D\x01"
			  "\x44\xD0\xD4\x4E\xC3\xF1\xE4\x38"
			  "\x09\x62\x0A\x1A\x4E\xD2\x63\x0F"
			  "\x6E\x3E\xD2\xA4\x3A\xF4\xF3\xFF"
			  "\x7E\x42\xEC\xB6\x6F\x4D\x6B\x48"
			  "\xE6\xA6\x50\x80\x78\x9E\xF1\xB0"
			  "\x4D\xB2\x0D\x3D\xFC\x40\x25\x4D"
			  "\x93\x11\x1C\xE9\xD2\x9F\x6E\x90"
			  "\xE5\x41\x4A\xE2\x3C\x45\x29\x35"
			  "\xEC\xD6\x47\x50\xCB\x7B\xA2\x32"
			  "\xF7\x8B\x62\xF1\xE3\x9A\xFE\xC7"
			  "\x1D\x8C\x02\x72\x68\x09\xE9\xB6"
			  "\x4A\x80\xE6\xB1\x56\xDF\x90\xD4"
			  "\x93\x74\xA4\xCE\x20\x23\xBF\x48"
			  "\xA5\xDE\x1B\xFA\x40\x69\x31\x98"
			  "\x62\x6E\xA5\xC7\xBF\x0C\x62\xE5"
			  "\x6D\xE1\x93\xF1\x83\x10\x1C\xCA"
			  "\xF6\x5C\x19\xF8\x90\x78\xCB\xE4"
			  "\x0B\x3A\xB5\xF8\x43\x86\xD3\x3F"
			  "\xBA\x83\x34\x3C\x42\xCC\x7D\x28"
			  "\x29\x63\x4F\xD8\x02\x17\xC5\x07"
			  "\x2C\xA4\xAC\x79\xCB\xC3\xA9\x09"
			  "\x81\x45\x18\xED\xE4\xCB\x42\x3B"
			  "\x87\x2D\x23\xDC\xC5\xBA\x45\xBD"
			  "\x92\xE5\x02\x97\x96\xCE\xAD\xEC"
			  "\xBA\xD8\x76\xF8\xCA\xC1\x31\xEC"
			  "\x1E\x4F\x3F\x83\xF8\x33\xE8\x6E"
			  "\xCC\xF8\x5F\xDD\x65\x50\x99\x69"
			  "\xAF\x48\xCE\xA5\xBA\xB6\x14\x9F"
			  "\x05\x93\xB2\xE6\x59\xC8\x28\xFE"
			  "\x8F\x37\xF9\x64\xB9\xA5\x56\x8F"
			  "\xF1\x1B\x90\xEF\xAE\xEB\xFC\x09"
			  "\x11\x7A\xF2\x19\x0A\x0A\x9A\x3C"
			  "\xE2\x5E\x29\xFA\x31\x9B\xC1\x74"
			  "\x1E\x10\x3E\x07\xA9\x31\x6D\xF8"
			  "\x81\xF5\xD5\x8A\x04\x23\x51\xAC"
			  "\xA2\xE2\x63\xFD\x27\x1F\x79\x5B"
			  "\x1F\xE8\xDA\x11\x49\x4D\x1C\xBA"
			  "\x54\xCC\x0F\xBA\x92\x69\xE5\xCB"
			  "\x41\x1A\x67\xA6\x40\x82\x70\x8C"
			  "\x19\x79\x08\xA4\x51\x20\x7D\xC9"
			  "\x12\x27\xAE\x20\x0D\x2C\xA1\x6D"
			  "\xF4\x55\xD4\xE7\xE6\xD4\x28\x08"
			  "\x00\x70\x12\x56\x56\x50\xAD\x14"
			  "\x5C\x3E\xA2\xD1\x36\x3F\x36\x48"
			  "\xED\xB1\x57\x3E\x5D\x15\xF6\x1E"
			  "\x53\xE9\xA4\x3E\xED\x7D\xCF\x7D"
			  "\x29\xAF\xF3\x1E\x51\xA8\x9F\x85"
			  "\x8B\xF0\xBB\xCE\xCC\x39\xC3\x64"
			  "\x4B\xF2\xAD\x70\x19\xD4\x44\x8F"
			  "\x91\x76\xE8\x15\x66\x34\x9F\xF6"
			  "\x0F\x15\xA4\xA8\x24\xF8\x58\xB1"
			  "\x38\x46\x47\xC7\x9B\xCA\xE9\x42"
			  "\x44\xAA\xE6\xB5\x9C\x91\xA4\xD3"
			  "\x16\xA0\xED\x42\xBE\xB5\x06\x19"
			  "\xBE\x67\xE8\xBC\x22\x32\xA4\x1E"
			  "\x93\xEB\xBE\xE9\xE1\x93\xE5\x31"
			  "\x3A\xA2\x75\xDF\xE3\x6B\xE7\xCC"
			  "\xB4\x70\x20\xE0\x6D\x82\x7C\xC8"
			  "\x94\x5C\x5E\x37\x18\xAD\xED\x8B"
			  "\x44\x86\xCA\x5E\x07\xB7\x70\x8D"
			  "\x40\x48\x19\x73\x7C\x78\x64\x0B"
			  "\xDB\x01\xCA\xAE\x63\x19\xE9\xD1"
			  "\x6B\x2C\x84\x10\x45\x42\x2E\xC3"
			  "\xDF\x7F\xAA\xE8\x87\x1B\x63\x46"
			  "\x74\x28\x9D\x05\x30\x20\x62\x41"
			  "\xC0\x9F\x2C\x36\x2B\x78\xD7\x26"
			  "\xDF\x58\x51\xED\xFA\xDC\x87\x79"
			  "\xBF\x8C\xBF\xC4\x0F\xE5\x05\xDA"
			  "\x45\xE3\x35\x0D\x69\x91\x54\x1C"
			  "\xE7\x2C\x49\x08\x8B\x72\xFA\x5C"
			  "\xF1\x6B\xD9",
		.ilen	= 1011,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D"
			  "\xE4\x7B\x12",
		.rlen	= 1011,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 1011 - 16, 16 },
	}, {  
		.key	= "\x85\x62\x3F\x1C\xF9\xD6\x1C\xF9"
			  "\xD6\xB3\x90\x6D\x4A\x90\x6D\x4A"
			  "\x27\x04\xE1\x27\x04\xE1\xBE\x9B"
			  "\x78\xBE\x9B\x78\x55\x32\x0F\x55",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x85\x79\x6C\x8B\x2B\x6D\x14\xF9"
			  "\xA6\x83\xB6\x80\x5B\x3A\xF3\x7E"
			  "\x30\x29\xEB\x1F\xDC\x19\x5F\xEB"
			  "\xF7\xC4\x27\x04\x51\x87\xD7\x6F"
			  "\xB8\x4E\x07\xFB\xAC\x3B\x08\xB4"
			  "\x4D\xCB\xE8\xE1\x71\x7D\x4F\x48"
			  "\xCD\x81\x64\xA5\xC4\x07\x1A\x9A"
			  "\x4B\x62\x90\x0E\xC8\xB3\x2B\x6B"
			  "\x8F\x9C\x6E\x72\x4B\xBA\xEF\x07"
			  "\x2C\x56\x07\x5E\x37\x30\x60\xA9"
			  "\xE3\xEF\xD6\x69\xE1\xA1\x77\x64"
			  "\x93\x75\x7A\xB7\x7A\x3B\xE9\x43"
			  "\x23\x35\x95\x91\x80\x8A\xC7\xCF"
			  "\xC3\xD5\xBF\xE7\xFE\x4C\x06\x6B"
			  "\x05\x19\x48\xE2\x62\xBA\x4F\xF2"
			  "\xFB\xEE\xE4\xCB\x79\x9D\xA3\x10"
			  "\x1D\x29\x8C\x1D\x7A\x88\x5A\xDD"
			  "\x4E\xB6\x18\xAA\xCD\xE6\x33\x96"
			  "\xD9\x0F\x90\x5A\x78\x76\x4D\x77"
			  "\x3C\x20\x89\x3B\xA3\xF9\x07\xFD"
			  "\xE4\xE8\x20\x2D\x15\x0A\x63\x49"
			  "\xF5\x4F\x89\xD8\xDE\xA1\x28\x78"
			  "\x28\x07\x09\x1B\x03\x94\x1D\x4B"
			  "\x82\x28\x1E\x1D\x95\xBA\xAC\x85"
			  "\x71\x6E\x3C\x18\x4B\x77\x74\x79"
			  "\xBF\x67\x0A\x53\x3C\x94\xD9\x60"
			  "\xE9\x6D\x40\x34\xA0\x2A\x53\x5D"
			  "\x27\xD5\x47\xF9\xC3\x4B\x27\x29"
			  "\xE4\x76\x9C\x3F\xA7\x1C\x87\xFC"
			  "\x6E\x0F\xCF\x9B\x60\xF0\xF0\x8B"
			  "\x70\x1C\x84\x81\x72\x4D\xB4\x98"
			  "\x23\x62\xE7\x6A\x2B\xFC\xA5\xB2"
			  "\xFF\xF5\x71\x07\xCD\x90\x23\x13"
			  "\x19\xD7\x79\x36\x6C\x9D\x55\x8B"
			  "\x93\x78\x86\x05\x69\x46\xD0\xC5"
			  "\x39\x09\xEB\x79\xEF\xFA\x9F\xAE"
			  "\xF3\xD5\x44\xC3\xFD\x86\xD2\x7C"
			  "\x83\x4B\xD8\x75\x9C\x18\x04\x7B"
			  "\x73\xAD\x72\xA4\xF6\xAB\xCF\x4B"
			  "\xCC\x01\x45\x90\xA6\x43\x05\x0C"
			  "\x6C\x4F\x62\x77\x57\x97\x9F\xEE"
			  "\x75\xA7\x3C\x38\xD1\x0F\x3D\x0E"
			  "\x2C\x43\x98\xFB\x13\x65\x73\xE4"
			  "\x3C\x1E\xD6\x90\x08\xF7\xE0\x99"
			  "\x3B\xF1\x9D\x6C\x48\xA9\x0E\x32"
			  "\x17\xC2\xCC\x20\xA1\x19\x26\xAA"
			  "\xE0\x75\x2F\xFB\x54\x66\x0A\xDF"
			  "\xB5\xF2\x1F\xC1\x34\x3C\x30\x56"
			  "\xE8\xDC\xF7\x92\x6B\xBF\x17\x24"
			  "\xEC\x94\xB5\x3B\xD6\xCE\xA2\x54"
			  "\x10\x7F\x50\xDE\x69\x77\xD5\x37"
			  "\xFE\x9C\x10\x83\xC5\xEB\xC9\x53"
			  "\xB7\xF3\xC4\x20\xAF\x0A\x7E\x57"
			  "\x3A\xE6\x75\xFE\x89\x00\x6E\x48"
			  "\xFB\x99\x17\x2C\xF6\x64\x40\x95"
			  "\x5E\xDC\x7A\xA6\x70\xC7\xF4\xDD"
			  "\x52\x05\x24\x34\xF9\x0E\xC8\x64"
			  "\x6D\xE2\xD8\x80\x53\x31\x4C\xFE"
			  "\xB4\x3A\x5F\x19\xCF\x42\x1B\x22"
			  "\x0B\x2D\x7B\xF1\xC5\x43\xF7\x5E"
			  "\x12\xA8\x01\x64\x16\x0B\x26\x5A"
			  "\x0C\x95\x0F\x40\xC5\x5A\x06\x7C"
			  "\xCF\xF5\xD5\xB7\x7A\x34\x23\xB6"
			  "\xAA\x9E\xA8\x98\xA2\xF8\x3D\xD3"
			  "\x3F\x23\x69\x63\x56\x96\x45\xD6"
			  "\x74\x23\x1D\x5C\x63\xCC\xD8\x78"
			  "\x16\xE2\x9C\xD2\x80\x02\xF2\x28"
			  "\x69\x2F\xC4\xA8\x15\x15\x24\x3B"
			  "\xCB\xF0\x14\xE4\x62\xC8\xF3\xD1"
			  "\x03\x58\x1B\x33\x77\x74\x1F\xB4"
			  "\x07\x86\xF2\x21\xB7\x41\xAE\xBF"
			  "\x25\xC2\xFF\x51\xEF\xEA\xCE\xC4"
			  "\x5F\xD9\xB8\x18\x6A\xF0\x0F\x0D"
			  "\xF8\x04\xBB\x6D\x62\x33\x87\x26"
			  "\x4F\x2F\x14\x6E\xDC\xDB\x66\x09"
			  "\x2A\xEF\x7D\x84\x10\xAC\x82\x5E"
			  "\xD2\xE4\xAD\x74\x7A\x6D\xCC\x3A"
			  "\x7B\x62\xD8\xD6\x07\x2D\xF7\xDF"
			  "\x9B\xB3\x82\xCF\x9C\x1D\x76\x5C"
			  "\xAC\x7B\xD4\x9B\x45\xA1\x64\x11"
			  "\x66\xF1\xA7\x0B\xF9\xDD\x00\xDD"
			  "\xA4\x45\x3D\x3E\x03\xC9\x2E\xCB"
			  "\xC3\x14\x84\x72\xFD\x41\xDC\xBD"
			  "\x75\xBE\xA8\xE5\x16\x48\x64\x39"
			  "\xCA\xF3\xE6\xDC\x25\x24\xF1\x6D"
			  "\xB2\x8D\xC5\x38\x54\xD3\x5D\x6D"
			  "\x0B\x29\x10\x15\x0E\x13\x3B\xAC"
			  "\x7E\xCC\x9E\x3E\x18\x48\xA6\x02"
			  "\xEF\x03\xB2\x2E\xE3\xD2\x70\x21"
			  "\xB4\x19\x26\xBE\x3A\x3D\x05\xE0"
			  "\xF8\x09\xAF\xE4\x31\x26\x92\x2F"
			  "\x8F\x55\xAC\xED\x0B\xB2\xA5\x34"
			  "\xBE\x50\xB1\x02\x22\x96\xE3\x40"
			  "\x7B\x70\x50\x6E\x3B\xD5\xE5\xA0"
			  "\x8E\xA2\xAD\x14\x60\x5C\x7A\x2B"
			  "\x3D\x1B\x7F\xC1\xC0\x2C\x56\x36"
			  "\xD2\x0A\x32\x06\x97\x34\xB9\xF4"
			  "\x6F\x9F\x7E\x80\xD0\x9D\xF7\x6A"
			  "\x21\xC1\xA2\x6A\xB1\x96\x5B\x4D"
			  "\x7A\x15\x6C\xC4\x4E\xB8\xE0\x9E"
			  "\x6C\x50\xF3\x9C\xC9\xB5\x23\xB7"
			  "\xF1\xD4\x29\x4A\x23\xC4\xAD\x1E"
			  "\x2C\x07\xD2\x43\x5F\x57\x93\xCA"
			  "\x85\xF9\x9F\xAD\x4C\xF1\xE4\xB1"
			  "\x1A\x8E\x28\xA4\xB6\x52\x77\x7E"
			  "\x68\xC6\x47\xB9\x76\xCC\x65\x5F"
			  "\x0B\xF9\x67\x93\xD8\x0E\x9A\x37"
			  "\x5F\x41\xED\x64\x6C\xAD\x5F\xED"
			  "\x3F\x8D\xFB\x8E\x1E\xA0\xE4\x1F"
			  "\xC2\xC7\xED\x18\x43\xE1\x20\x86"
			  "\x5D\xBC\x30\x70\x22\xA1\xDC\x53"
			  "\x10\x3A\x8D\x47\x82\xCD\x7F\x59"
			  "\x03\x2D\x6D\xF5\xE7\x79\xD4\x07"
			  "\x68\x2A\xA5\x42\x19\x4D\xAF\xF5"
			  "\xED\x47\x83\xBC\x5F\x62\x84\xDA"
			  "\xDA\x41\xFF\xB0\x1D\x64\xA3\xC8"
			  "\xBD\x4E\xE0\xB8\x7F\xEE\x55\x0A"
			  "\x4E\x61\xB2\x51\xF6\x9C\x95\xF6"
			  "\x92\xBB\xF6\xC5\xF0\x09\x86\xDE"
			  "\x37\x9E\x29\xF9\x2A\x18\x73\x0D"
			  "\xDC\x7E\x6B\x7B\x1B\x43\x8C\xEA"
			  "\x13\xC8\x1A\x47\x0A\x2D\x6D\x56"
			  "\xCD\xD2\xE7\x53\x1A\xAB\x1C\x3C"
			  "\xC5\x9B\x03\x70\x29\x2A\x49\x09"
			  "\x67\xA1\xEA\xD6\x3A\x5B\xBF\x71"
			  "\x1D\x48\x64\x6C\xFB\xC0\x9E\x36",
		.ilen	= 1008,
		.result	= "\x56\xED\x84\x1B\x8F\x26\xBD\x31"
			  "\xC8\x5F\xF6\x6A\x01\x98\x0C\xA3"
			  "\x3A\xD1\x45\xDC\x73\x0A\x7E\x15"
			  "\xAC\x20\xB7\x4E\xE5\x59\xF0\x87"
			  "\x1E\x92\x29\xC0\x34\xCB\x62\xF9"
			  "\x6D\x04\x9B\x0F\xA6\x3D\xD4\x48"
			  "\xDF\x76\x0D\x81\x18\xAF\x23\xBA"
			  "\x51\xE8\x5C\xF3\x8A\x21\x95\x2C"
			  "\xC3\x37\xCE\x65\xFC\x70\x07\x9E"
			  "\x12\xA9\x40\xD7\x4B\xE2\x79\x10"
			  "\x84\x1B\xB2\x26\xBD\x54\xEB\x5F"
			  "\xF6\x8D\x01\x98\x2F\xC6\x3A\xD1"
			  "\x68\xFF\x73\x0A\xA1\x15\xAC\x43"
			  "\xDA\x4E\xE5\x7C\x13\x87\x1E\xB5"
			  "\x29\xC0\x57\xEE\x62\xF9\x90\x04"
			  "\x9B\x32\xC9\x3D\xD4\x6B\x02\x76"
			  "\x0D\xA4\x18\xAF\x46\xDD\x51\xE8"
			  "\x7F\x16\x8A\x21\xB8\x2C\xC3\x5A"
			  "\xF1\x65\xFC\x93\x07\x9E\x35\xCC"
			  "\x40\xD7\x6E\x05\x79\x10\xA7\x1B"
			  "\xB2\x49\xE0\x54\xEB\x82\x19\x8D"
			  "\x24\xBB\x2F\xC6\x5D\xF4\x68\xFF"
			  "\x96\x0A\xA1\x38\xCF\x43\xDA\x71"
			  "\x08\x7C\x13\xAA\x1E\xB5\x4C\xE3"
			  "\x57\xEE\x85\x1C\x90\x27\xBE\x32"
			  "\xC9\x60\xF7\x6B\x02\x99\x0D\xA4"
			  "\x3B\xD2\x46\xDD\x74\x0B\x7F\x16"
			  "\xAD\x21\xB8\x4F\xE6\x5A\xF1\x88"
			  "\x1F\x93\x2A\xC1\x35\xCC\x63\xFA"
			  "\x6E\x05\x9C\x10\xA7\x3E\xD5\x49"
			  "\xE0\x77\x0E\x82\x19\xB0\x24\xBB"
			  "\x52\xE9\x5D\xF4\x8B\x22\x96\x2D"
			  "\xC4\x38\xCF\x66\xFD\x71\x08\x9F"
			  "\x13\xAA\x41\xD8\x4C\xE3\x7A\x11"
			  "\x85\x1C\xB3\x27\xBE\x55\xEC\x60"
			  "\xF7\x8E\x02\x99\x30\xC7\x3B\xD2"
			  "\x69\x00\x74\x0B\xA2\x16\xAD\x44"
			  "\xDB\x4F\xE6\x7D\x14\x88\x1F\xB6"
			  "\x2A\xC1\x58\xEF\x63\xFA\x91\x05"
			  "\x9C\x33\xCA\x3E\xD5\x6C\x03\x77"
			  "\x0E\xA5\x19\xB0\x47\xDE\x52\xE9"
			  "\x80\x17\x8B\x22\xB9\x2D\xC4\x5B"
			  "\xF2\x66\xFD\x94\x08\x9F\x36\xCD"
			  "\x41\xD8\x6F\x06\x7A\x11\xA8\x1C"
			  "\xB3\x4A\xE1\x55\xEC\x83\x1A\x8E"
			  "\x25\xBC\x30\xC7\x5E\xF5\x69\x00"
			  "\x97\x0B\xA2\x39\xD0\x44\xDB\x72"
			  "\x09\x7D\x14\xAB\x1F\xB6\x4D\xE4"
			  "\x58\xEF\x86\x1D\x91\x28\xBF\x33"
			  "\xCA\x61\xF8\x6C\x03\x9A\x0E\xA5"
			  "\x3C\xD3\x47\xDE\x75\x0C\x80\x17"
			  "\xAE\x22\xB9\x50\xE7\x5B\xF2\x89"
			  "\x20\x94\x2B\xC2\x36\xCD\x64\xFB"
			  "\x6F\x06\x9D\x11\xA8\x3F\xD6\x4A"
			  "\xE1\x78\x0F\x83\x1A\xB1\x25\xBC"
			  "\x53\xEA\x5E\xF5\x8C\x00\x97\x2E"
			  "\xC5\x39\xD0\x67\xFE\x72\x09\xA0"
			  "\x14\xAB\x42\xD9\x4D\xE4\x7B\x12"
			  "\x86\x1D\xB4\x28\xBF\x56\xED\x61"
			  "\xF8\x8F\x03\x9A\x31\xC8\x3C\xD3"
			  "\x6A\x01\x75\x0C\xA3\x17\xAE\x45"
			  "\xDC\x50\xE7\x7E\x15\x89\x20\xB7"
			  "\x2B\xC2\x59\xF0\x64\xFB\x92\x06"
			  "\x9D\x34\xCB\x3F\xD6\x6D\x04\x78"
			  "\x0F\xA6\x1A\xB1\x48\xDF\x53\xEA"
			  "\x81\x18\x8C\x23\xBA\x2E\xC5\x5C"
			  "\xF3\x67\xFE\x95\x09\xA0\x37\xCE"
			  "\x42\xD9\x70\x07\x7B\x12\xA9\x1D"
			  "\xB4\x4B\xE2\x56\xED\x84\x1B\x8F"
			  "\x26\xBD\x31\xC8\x5F\xF6\x6A\x01"
			  "\x98\x0C\xA3\x3A\xD1\x45\xDC\x73"
			  "\x0A\x7E\x15\xAC\x20\xB7\x4E\xE5"
			  "\x59\xF0\x87\x1E\x92\x29\xC0\x34"
			  "\xCB\x62\xF9\x6D\x04\x9B\x0F\xA6"
			  "\x3D\xD4\x48\xDF\x76\x0D\x81\x18"
			  "\xAF\x23\xBA\x51\xE8\x5C\xF3\x8A"
			  "\x21\x95\x2C\xC3\x37\xCE\x65\xFC"
			  "\x70\x07\x9E\x12\xA9\x40\xD7\x4B"
			  "\xE2\x79\x10\x84\x1B\xB2\x26\xBD"
			  "\x54\xEB\x5F\xF6\x8D\x01\x98\x2F"
			  "\xC6\x3A\xD1\x68\xFF\x73\x0A\xA1"
			  "\x15\xAC\x43\xDA\x4E\xE5\x7C\x13"
			  "\x87\x1E\xB5\x29\xC0\x57\xEE\x62"
			  "\xF9\x90\x04\x9B\x32\xC9\x3D\xD4"
			  "\x6B\x02\x76\x0D\xA4\x18\xAF\x46"
			  "\xDD\x51\xE8\x7F\x16\x8A\x21\xB8"
			  "\x2C\xC3\x5A\xF1\x65\xFC\x93\x07"
			  "\x9E\x35\xCC\x40\xD7\x6E\x05\x79"
			  "\x10\xA7\x1B\xB2\x49\xE0\x54\xEB"
			  "\x82\x19\x8D\x24\xBB\x2F\xC6\x5D"
			  "\xF4\x68\xFF\x96\x0A\xA1\x38\xCF"
			  "\x43\xDA\x71\x08\x7C\x13\xAA\x1E"
			  "\xB5\x4C\xE3\x57\xEE\x85\x1C\x90"
			  "\x27\xBE\x32\xC9\x60\xF7\x6B\x02"
			  "\x99\x0D\xA4\x3B\xD2\x46\xDD\x74"
			  "\x0B\x7F\x16\xAD\x21\xB8\x4F\xE6"
			  "\x5A\xF1\x88\x1F\x93\x2A\xC1\x35"
			  "\xCC\x63\xFA\x6E\x05\x9C\x10\xA7"
			  "\x3E\xD5\x49\xE0\x77\x0E\x82\x19"
			  "\xB0\x24\xBB\x52\xE9\x5D\xF4\x8B"
			  "\x22\x96\x2D\xC4\x38\xCF\x66\xFD"
			  "\x71\x08\x9F\x13\xAA\x41\xD8\x4C"
			  "\xE3\x7A\x11\x85\x1C\xB3\x27\xBE"
			  "\x55\xEC\x60\xF7\x8E\x02\x99\x30"
			  "\xC7\x3B\xD2\x69\x00\x74\x0B\xA2"
			  "\x16\xAD\x44\xDB\x4F\xE6\x7D\x14"
			  "\x88\x1F\xB6\x2A\xC1\x58\xEF\x63"
			  "\xFA\x91\x05\x9C\x33\xCA\x3E\xD5"
			  "\x6C\x03\x77\x0E\xA5\x19\xB0\x47"
			  "\xDE\x52\xE9\x80\x17\x8B\x22\xB9"
			  "\x2D\xC4\x5B\xF2\x66\xFD\x94\x08"
			  "\x9F\x36\xCD\x41\xD8\x6F\x06\x7A"
			  "\x11\xA8\x1C\xB3\x4A\xE1\x55\xEC"
			  "\x83\x1A\x8E\x25\xBC\x30\xC7\x5E"
			  "\xF5\x69\x00\x97\x0B\xA2\x39\xD0"
			  "\x44\xDB\x72\x09\x7D\x14\xAB\x1F"
			  "\xB6\x4D\xE4\x58\xEF\x86\x1D\x91"
			  "\x28\xBF\x33\xCA\x61\xF8\x6C\x03"
			  "\x9A\x0E\xA5\x3C\xD3\x47\xDE\x75"
			  "\x0C\x80\x17\xAE\x22\xB9\x50\xE7"
			  "\x5B\xF2\x89\x20\x94\x2B\xC2\x36"
			  "\xCD\x64\xFB\x6F\x06\x9D\x11\xA8"
			  "\x3F\xD6\x4A\xE1\x78\x0F\x83\x1A"
			  "\xB1\x25\xBC\x53\xEA\x5E\xF5\x8C"
			  "\x00\x97\x2E\xC5\x39\xD0\x67\xFE"
			  "\x72\x09\xA0\x14\xAB\x42\xD9\x4D",
		.rlen	= 1008,
	},
};

static struct cipher_testvec camellia_lrw_enc_tv_template[] = {
	 
	{
		.key	= "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x92\x68\x19\xd7\xb7\x5b\x0a\x31"
			  "\x97\xcc\x72\xbe\x99\x17\xeb\x3e",
		.rlen	= 16,
	}, {
		.key	= "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x73\x09\xb7\x50\xb6\x77\x30\x50"
			  "\x5c\x8a\x9c\x26\x77\x9d\xfc\x4a",
		.rlen	= 16,
	}, {
		.key	= "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x90\xae\x83\xe0\x22\xb9\x60\x91"
			  "\xfa\xa9\xb7\x98\xe3\xed\x87\x01",
		.rlen	= 16,
	}, {
		.key	= "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x99\xe9\x6e\xd4\xc9\x21\xa5\xf0"
			  "\xd8\x83\xef\xd9\x07\x16\x5f\x35",
		.rlen	= 16,
	}, {
		.key	= "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x42\x88\xf4\xcb\x21\x11\x6d\x8e"
			  "\xde\x1a\xf2\x29\xf1\x4a\xe0\x15",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x40\xaa\x34\x86\x4a\x8f\x78\xb9"
			  "\xdb\xdb\x0f\x3d\x48\x70\xbe\x8d",
		.rlen	= 16,
	}, {
		.key	= "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.ilen	= 16,
		.result	= "\x04\xab\x28\x37\x31\x7a\x26\xab"
			  "\xa1\x70\x1b\x9c\xe7\xdd\x83\xff",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.ilen	= 512,
		.result	= "\x90\x69\x8e\xf2\x14\x86\x59\xf9"
			  "\xec\xe7\xfa\x3f\x48\x9d\x7f\x96"
			  "\x67\x76\xac\x2c\xd2\x63\x18\x93"
			  "\x13\xf8\xf1\xf6\x71\x77\xb3\xee"
			  "\x93\xb2\xcc\xf3\x26\xc1\x16\x4f"
			  "\xd4\xe8\x43\xc1\x68\xa3\x3e\x06"
			  "\x38\x51\xff\xa8\xb9\xa4\xeb\xb1"
			  "\x62\xdd\x78\x81\xea\x1d\xef\x04"
			  "\x1d\x07\xc1\x67\xc8\xd6\x77\xa1"
			  "\x84\x95\xf4\x9a\xd9\xbc\x2d\xe2"
			  "\xf6\x80\xfc\x91\x2a\xbc\x42\xa0"
			  "\x40\x41\x69\xaa\x71\xc0\x37\xec"
			  "\x39\xf3\xf2\xec\x82\xc3\x88\x79"
			  "\xbc\xc3\xaa\xb7\xcf\x6a\x72\x80"
			  "\x4c\xf4\x84\x8f\x13\x9e\x94\x5c"
			  "\xe5\xb2\x91\xbb\x92\x51\x4d\xf1"
			  "\xd6\x0d\x71\x6b\x7a\xc2\x2f\x12"
			  "\x6f\x75\xc7\x80\x99\x50\x84\xcf"
			  "\xa8\xeb\xd6\xe1\x1c\x59\x81\x7e"
			  "\xb9\xb3\xde\x7a\x93\x14\x12\xa2"
			  "\xf7\x43\xb3\x9d\x1a\x87\x65\x91"
			  "\x42\x08\x40\x82\x06\x1c\x2d\x55"
			  "\x6e\x48\xd5\x74\x07\x6e\x9d\x80"
			  "\xeb\xb4\x97\xa1\x36\xdf\xfa\x74"
			  "\x79\x7f\x5a\x75\xe7\x71\xc8\x8c"
			  "\x7e\xf8\x3a\x77\xcd\x32\x05\xf9"
			  "\x3d\xd4\xe9\xa2\xbb\xc4\x8b\x83"
			  "\x42\x5c\x82\xfa\xe9\x4b\x96\x3b"
			  "\x7f\x89\x8b\xf9\xf1\x87\xda\xf0"
			  "\x87\xef\x13\x5d\xf0\xe2\xc5\xc1"
			  "\xed\x14\xa9\x57\x19\x63\x40\x04"
			  "\x24\xeb\x6e\x19\xd1\x3d\x70\x78"
			  "\xeb\xda\x55\x70\x2c\x4f\x41\x5b"
			  "\x56\x9f\x1a\xd3\xac\xf1\xc0\xc3"
			  "\x21\xec\xd7\xd2\x55\x32\x7c\x2e"
			  "\x3c\x48\x8e\xb4\x85\x35\x47\xfe"
			  "\xe2\x88\x79\x98\x6a\xc9\x8d\xff"
			  "\xe9\x89\x6e\xb8\xe2\x97\x00\xbd"
			  "\xa4\x8f\xba\xd0\x8c\xcb\x79\x99"
			  "\xb3\xb2\xb2\x7a\xc3\xb7\xef\x75"
			  "\x23\x52\x76\xc3\x50\x6e\x66\xf8"
			  "\xa2\xe2\xce\xba\x40\x21\x3f\xc9"
			  "\x0a\x32\x7f\xf7\x08\x8c\x66\xcf"
			  "\xd3\xdf\x57\x59\x83\xb8\xe1\x85"
			  "\xd6\x8f\xfb\x48\x1f\x3a\xc4\x2f"
			  "\xb4\x2d\x58\xab\xd8\x7f\x5e\x3a"
			  "\xbc\x62\x3e\xe2\x6a\x52\x0d\x76"
			  "\x2f\x1c\x1a\x30\xed\x95\x2a\x44"
			  "\x35\xa5\x83\x04\x84\x01\x99\x56"
			  "\xb7\xe3\x10\x96\xfa\xdc\x19\xdd"
			  "\xe2\x7f\xcb\xa0\x49\x1b\xff\x4c"
			  "\x73\xf6\xbb\x94\x00\xe8\xa9\x3d"
			  "\xe2\x20\xe9\x3f\xfa\x07\x5d\x77"
			  "\x06\xd5\x4f\x4d\x02\xb8\x40\x1b"
			  "\x30\xed\x1a\x50\x19\xef\xc4\x2c"
			  "\x02\xd9\xc5\xd3\x11\x33\x37\xe5"
			  "\x2b\xa3\x95\xa6\xee\xd8\x74\x1d"
			  "\x68\xa0\xeb\xbf\xdd\x5e\x99\x96"
			  "\x91\xc3\x94\x24\xa5\x12\xa2\x37"
			  "\xb3\xac\xcf\x2a\xfd\x55\x34\xfe"
			  "\x79\x92\x3e\xe6\x1b\x49\x57\x5d"
			  "\x93\x6c\x01\xf7\xcc\x4e\x20\xd1"
			  "\xb2\x1a\xd8\x4c\xbd\x1d\x10\xe9"
			  "\x5a\xa8\x92\x7f\xba\xe6\x0c\x95",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec camellia_lrw_dec_tv_template[] = {
	 
	{
		.key	= "\x45\x62\xac\x25\xf8\x28\x17\x6d"
			  "\x4c\x26\x84\x14\xb5\x68\x01\x85"
			  "\x25\x8e\x2a\x05\xe7\x3e\x9d\x03"
			  "\xee\x5a\x83\x0c\xcc\x09\x4c\x87",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x92\x68\x19\xd7\xb7\x5b\x0a\x31"
			  "\x97\xcc\x72\xbe\x99\x17\xeb\x3e",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x59\x70\x47\x14\xf5\x57\x47\x8c"
			  "\xd7\x79\xe8\x0f\x54\x88\x79\x44"
			  "\x0d\x48\xf0\xb7\xb1\x5a\x53\xea"
			  "\x1c\xaa\x6b\x29\xc2\xca\xfb\xaf",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x02",
		.input	= "\x73\x09\xb7\x50\xb6\x77\x30\x50"
			  "\x5c\x8a\x9c\x26\x77\x9d\xfc\x4a",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xd8\x2a\x91\x34\xb2\x6a\x56\x50"
			  "\x30\xfe\x69\xe2\x37\x7f\x98\x47"
			  "\xcd\xf9\x0b\x16\x0c\x64\x8f\xb6"
			  "\xb0\x0d\x0d\x1b\xae\x85\x87\x1f",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x90\xae\x83\xe0\x22\xb9\x60\x91"
			  "\xfa\xa9\xb7\x98\xe3\xed\x87\x01",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x0f\x6a\xef\xf8\xd3\xd2\xbb\x15"
			  "\x25\x83\xf7\x3c\x1f\x01\x28\x74"
			  "\xca\xc6\xbc\x35\x4d\x4a\x65\x54"
			  "\x90\xae\x61\xcf\x7b\xae\xbd\xcc"
			  "\xad\xe4\x94\xc5\x4a\x29\xae\x70",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x99\xe9\x6e\xd4\xc9\x21\xa5\xf0"
			  "\xd8\x83\xef\xd9\x07\x16\x5f\x35",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\x8a\xd4\xee\x10\x2f\xbd\x81\xff"
			  "\xf8\x86\xce\xac\x93\xc5\xad\xc6"
			  "\xa0\x19\x07\xc0\x9d\xf7\xbb\xdd"
			  "\x52\x13\xb2\xb7\xf0\xff\x11\xd8"
			  "\xd6\x08\xd0\xcd\x2e\xb1\x17\x6f",
		.klen	= 40,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x42\x88\xf4\xcb\x21\x11\x6d\x8e"
			  "\xde\x1a\xf2\x29\xf1\x4a\xe0\x15",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x40\xaa\x34\x86\x4a\x8f\x78\xb9"
			  "\xdb\xdb\x0f\x3d\x48\x70\xbe\x8d",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xfb\x76\x15\xb2\x3d\x80\x89\x1d"
			  "\xd4\x70\x98\x0b\xc7\x95\x84\xc8"
			  "\xb2\xfb\x64\xce\x60\x97\x87\x8d"
			  "\x17\xfc\xe4\x5a\x49\xe8\x30\xb7"
			  "\x6e\x78\x17\xe7\x2d\x5e\x12\xd4"
			  "\x60\x64\x04\x7a\xf1\x2f\x9e\x0c",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x02\x00\x00\x00\x00",
		.input	= "\x04\xab\x28\x37\x31\x7a\x26\xab"
			  "\xa1\x70\x1b\x9c\xe7\xdd\x83\xff",
		.ilen	= 16,
		.result	= "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x41\x42\x43\x44\x45\x46",
		.rlen	= 16,
	}, {
		.key	= "\xf8\xd4\x76\xff\xd6\x46\xee\x6c"
			  "\x23\x84\xcb\x1c\x77\xd6\x19\x5d"
			  "\xfe\xf1\xa9\xf3\x7b\xbc\x8d\x21"
			  "\xa7\x9c\x21\xf8\xcb\x90\x02\x89"
			  "\xa8\x45\x34\x8e\xc8\xc5\xb5\xf1"
			  "\x26\xf5\x0e\x76\xfe\xfd\x1b\x1e",
		.klen	= 48,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x01",
		.input	= "\x90\x69\x8e\xf2\x14\x86\x59\xf9"
			  "\xec\xe7\xfa\x3f\x48\x9d\x7f\x96"
			  "\x67\x76\xac\x2c\xd2\x63\x18\x93"
			  "\x13\xf8\xf1\xf6\x71\x77\xb3\xee"
			  "\x93\xb2\xcc\xf3\x26\xc1\x16\x4f"
			  "\xd4\xe8\x43\xc1\x68\xa3\x3e\x06"
			  "\x38\x51\xff\xa8\xb9\xa4\xeb\xb1"
			  "\x62\xdd\x78\x81\xea\x1d\xef\x04"
			  "\x1d\x07\xc1\x67\xc8\xd6\x77\xa1"
			  "\x84\x95\xf4\x9a\xd9\xbc\x2d\xe2"
			  "\xf6\x80\xfc\x91\x2a\xbc\x42\xa0"
			  "\x40\x41\x69\xaa\x71\xc0\x37\xec"
			  "\x39\xf3\xf2\xec\x82\xc3\x88\x79"
			  "\xbc\xc3\xaa\xb7\xcf\x6a\x72\x80"
			  "\x4c\xf4\x84\x8f\x13\x9e\x94\x5c"
			  "\xe5\xb2\x91\xbb\x92\x51\x4d\xf1"
			  "\xd6\x0d\x71\x6b\x7a\xc2\x2f\x12"
			  "\x6f\x75\xc7\x80\x99\x50\x84\xcf"
			  "\xa8\xeb\xd6\xe1\x1c\x59\x81\x7e"
			  "\xb9\xb3\xde\x7a\x93\x14\x12\xa2"
			  "\xf7\x43\xb3\x9d\x1a\x87\x65\x91"
			  "\x42\x08\x40\x82\x06\x1c\x2d\x55"
			  "\x6e\x48\xd5\x74\x07\x6e\x9d\x80"
			  "\xeb\xb4\x97\xa1\x36\xdf\xfa\x74"
			  "\x79\x7f\x5a\x75\xe7\x71\xc8\x8c"
			  "\x7e\xf8\x3a\x77\xcd\x32\x05\xf9"
			  "\x3d\xd4\xe9\xa2\xbb\xc4\x8b\x83"
			  "\x42\x5c\x82\xfa\xe9\x4b\x96\x3b"
			  "\x7f\x89\x8b\xf9\xf1\x87\xda\xf0"
			  "\x87\xef\x13\x5d\xf0\xe2\xc5\xc1"
			  "\xed\x14\xa9\x57\x19\x63\x40\x04"
			  "\x24\xeb\x6e\x19\xd1\x3d\x70\x78"
			  "\xeb\xda\x55\x70\x2c\x4f\x41\x5b"
			  "\x56\x9f\x1a\xd3\xac\xf1\xc0\xc3"
			  "\x21\xec\xd7\xd2\x55\x32\x7c\x2e"
			  "\x3c\x48\x8e\xb4\x85\x35\x47\xfe"
			  "\xe2\x88\x79\x98\x6a\xc9\x8d\xff"
			  "\xe9\x89\x6e\xb8\xe2\x97\x00\xbd"
			  "\xa4\x8f\xba\xd0\x8c\xcb\x79\x99"
			  "\xb3\xb2\xb2\x7a\xc3\xb7\xef\x75"
			  "\x23\x52\x76\xc3\x50\x6e\x66\xf8"
			  "\xa2\xe2\xce\xba\x40\x21\x3f\xc9"
			  "\x0a\x32\x7f\xf7\x08\x8c\x66\xcf"
			  "\xd3\xdf\x57\x59\x83\xb8\xe1\x85"
			  "\xd6\x8f\xfb\x48\x1f\x3a\xc4\x2f"
			  "\xb4\x2d\x58\xab\xd8\x7f\x5e\x3a"
			  "\xbc\x62\x3e\xe2\x6a\x52\x0d\x76"
			  "\x2f\x1c\x1a\x30\xed\x95\x2a\x44"
			  "\x35\xa5\x83\x04\x84\x01\x99\x56"
			  "\xb7\xe3\x10\x96\xfa\xdc\x19\xdd"
			  "\xe2\x7f\xcb\xa0\x49\x1b\xff\x4c"
			  "\x73\xf6\xbb\x94\x00\xe8\xa9\x3d"
			  "\xe2\x20\xe9\x3f\xfa\x07\x5d\x77"
			  "\x06\xd5\x4f\x4d\x02\xb8\x40\x1b"
			  "\x30\xed\x1a\x50\x19\xef\xc4\x2c"
			  "\x02\xd9\xc5\xd3\x11\x33\x37\xe5"
			  "\x2b\xa3\x95\xa6\xee\xd8\x74\x1d"
			  "\x68\xa0\xeb\xbf\xdd\x5e\x99\x96"
			  "\x91\xc3\x94\x24\xa5\x12\xa2\x37"
			  "\xb3\xac\xcf\x2a\xfd\x55\x34\xfe"
			  "\x79\x92\x3e\xe6\x1b\x49\x57\x5d"
			  "\x93\x6c\x01\xf7\xcc\x4e\x20\xd1"
			  "\xb2\x1a\xd8\x4c\xbd\x1d\x10\xe9"
			  "\x5a\xa8\x92\x7f\xba\xe6\x0c\x95",
		.ilen	= 512,
		.result	= "\x05\x11\xb7\x18\xab\xc6\x2d\xac"
			  "\x70\x5d\xf6\x22\x94\xcd\xe5\x6c"
			  "\x17\x6b\xf6\x1c\xf0\xf3\x6e\xf8"
			  "\x50\x38\x1f\x71\x49\xb6\x57\xd6"
			  "\x8f\xcb\x8d\x6b\xe3\xa6\x29\x90"
			  "\xfe\x2a\x62\x82\xae\x6d\x8b\xf6"
			  "\xad\x1e\x9e\x20\x5f\x38\xbe\x04"
			  "\xda\x10\x8e\xed\xa2\xa4\x87\xab"
			  "\xda\x6b\xb4\x0c\x75\xba\xd3\x7c"
			  "\xc9\xac\x42\x31\x95\x7c\xc9\x04"
			  "\xeb\xd5\x6e\x32\x69\x8a\xdb\xa6"
			  "\x15\xd7\x3f\x4f\x2f\x66\x69\x03"
			  "\x9c\x1f\x54\x0f\xde\x1f\xf3\x65"
			  "\x4c\x96\x12\xed\x7c\x92\x03\x01"
			  "\x6f\xbc\x35\x93\xac\xf1\x27\xf1"
			  "\xb4\x96\x82\x5a\x5f\xb0\xa0\x50"
			  "\x89\xa4\x8e\x66\x44\x85\xcc\xfd"
			  "\x33\x14\x70\xe3\x96\xb2\xc3\xd3"
			  "\xbb\x54\x5a\x1a\xf9\x74\xa2\xc5"
			  "\x2d\x64\x75\xdd\xb4\x54\xe6\x74"
			  "\x8c\xd3\x9d\x9e\x86\xab\x51\x53"
			  "\xb7\x93\x3e\x6f\xd0\x4e\x2c\x40"
			  "\xf6\xa8\x2e\x3e\x9d\xf4\x66\xa5"
			  "\x76\x12\x73\x44\x1a\x56\xd7\x72"
			  "\x88\xcd\x21\x8c\x4c\x0f\xfe\xda"
			  "\x95\xe0\x3a\xa6\xa5\x84\x46\xcd"
			  "\xd5\x3e\x9d\x3a\xe2\x67\xe6\x60"
			  "\x1a\xe2\x70\x85\x58\xc2\x1b\x09"
			  "\xe1\xd7\x2c\xca\xad\xa8\x8f\xf9"
			  "\xac\xb3\x0e\xdb\xca\x2e\xe2\xb8"
			  "\x51\x71\xd9\x3c\x6c\xf1\x56\xf8"
			  "\xea\x9c\xf1\xfb\x0c\xe6\xb7\x10"
			  "\x1c\xf8\xa9\x7c\xe8\x53\x35\xc1"
			  "\x90\x3e\x76\x4a\x74\xa4\x21\x2c"
			  "\xf6\x2c\x4e\x0f\x94\x3a\x88\x2e"
			  "\x41\x09\x6a\x33\x7d\xf6\xdd\x3f"
			  "\x8d\x23\x31\x74\x84\xeb\x88\x6e"
			  "\xcc\xb9\xbc\x22\x83\x19\x07\x22"
			  "\xa5\x2d\xdf\xa5\xf3\x80\x85\x78"
			  "\x84\x39\x6a\x6d\x6a\x99\x4f\xa5"
			  "\x15\xfe\x46\xb0\xe4\x6c\xa5\x41"
			  "\x3c\xce\x8f\x42\x60\x71\xa7\x75"
			  "\x08\x40\x65\x8a\x82\xbf\xf5\x43"
			  "\x71\x96\xa9\x4d\x44\x8a\x20\xbe"
			  "\xfa\x4d\xbb\xc0\x7d\x31\x96\x65"
			  "\xe7\x75\xe5\x3e\xfd\x92\x3b\xc9"
			  "\x55\xbb\x16\x7e\xf7\xc2\x8c\xa4"
			  "\x40\x1d\xe5\xef\x0e\xdf\xe4\x9a"
			  "\x62\x73\x65\xfd\x46\x63\x25\x3d"
			  "\x2b\xaf\xe5\x64\xfe\xa5\x5c\xcf"
			  "\x24\xf3\xb4\xac\x64\xba\xdf\x4b"
			  "\xc6\x96\x7d\x81\x2d\x8d\x97\xf7"
			  "\xc5\x68\x77\x84\x32\x2b\xcc\x85"
			  "\x74\x96\xf0\x12\x77\x61\xb9\xeb"
			  "\x71\xaa\x82\xcb\x1c\xdb\x89\xc8"
			  "\xc6\xb5\xe3\x5c\x7d\x39\x07\x24"
			  "\xda\x39\x87\x45\xc0\x2b\xbb\x01"
			  "\xac\xbc\x2a\x5c\x7f\xfc\xe8\xce"
			  "\x6d\x9c\x6f\xed\xd3\xc1\xa1\xd6"
			  "\xc5\x55\xa9\x66\x2f\xe1\xc8\x32"
			  "\xa6\x5d\xa4\x3a\x98\x73\xe8\x45"
			  "\xa4\xc7\xa8\xb4\xf6\x13\x03\xf6"
			  "\xe9\x2e\xc4\x29\x0f\x84\xdb\xc4"
			  "\x21\xc4\xc2\x75\x67\x89\x37\x0a",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec camellia_xts_enc_tv_template[] = {
	 
	{
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 32,
		.result	= "\x06\xcb\xa5\xf1\x04\x63\xb2\x41"
			  "\xdc\xca\xfa\x09\xba\x74\xb9\x05"
			  "\x78\xba\xa4\xf8\x67\x4d\x7e\xad"
			  "\x20\x18\xf5\x0c\x41\x16\x2a\x61",
		.rlen	= 32,
	}, {
		.key	= "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen	= 32,
		.result	= "\xc2\xb9\xdc\x44\x1d\xdf\xf2\x86"
			  "\x8d\x35\x42\x0a\xa5\x5e\x3d\x4f"
			  "\xb5\x37\x06\xff\xbd\xd4\x91\x70"
			  "\x80\x1f\xb2\x39\x10\x89\x44\xf5",
		.rlen	= 32,
	}, {
		.key	= "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.ilen	= 32,
		.result	= "\x52\x1f\x9d\xf5\x5a\x58\x5a\x7e"
			  "\x9f\xd0\x8e\x02\x9c\x9a\x6a\xa7"
			  "\xb4\x3b\xce\xe7\x17\xaa\x89\x6a"
			  "\x35\x3c\x6b\xb5\x61\x1c\x79\x38",
		.rlen	= 32,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\xc7\xf9\x0a\xaa\xcb\xb5\x8f\x33"
			  "\x60\xc3\xe9\x47\x90\xb7\x50\x57"
			  "\xa3\xad\x81\x2f\xf5\x22\x96\x02"
			  "\xaa\x7f\xea\xac\x29\x78\xca\x2a"
			  "\x7c\xcd\x31\x1a\x3c\x40\x0a\x73"
			  "\x09\x66\xad\x72\x0e\x4d\x5d\x77"
			  "\xbc\xb8\x76\x80\x37\x59\xa9\x01"
			  "\x9e\xfb\xdb\x6c\x93\xef\xb6\x8d"
			  "\x1e\xc1\x94\xa8\xd4\xb5\xb0\x01"
			  "\xd5\x01\x97\x28\xcd\x7a\x1f\xe8"
			  "\x08\xda\x76\x00\x65\xcf\x7b\x31"
			  "\xc6\xfa\xf2\x3b\x00\xa7\x6a\x9e"
			  "\x6c\x43\x80\x87\xe0\xbb\x4e\xe5"
			  "\xdc\x8a\xdf\xc3\x1d\x1b\x41\x04"
			  "\xfb\x54\xdd\x29\x27\xc2\x65\x17"
			  "\x36\x88\xb0\x85\x8d\x73\x7e\x4b"
			  "\x1d\x16\x8a\x52\xbc\xa6\xbc\xa4"
			  "\x8c\xd1\x04\x16\xbf\x8c\x01\x0f"
			  "\x7e\x6b\x59\x15\x29\xd1\x9b\xd3"
			  "\x6c\xee\xac\xdc\x45\x58\xca\x5b"
			  "\x70\x0e\x6a\x12\x86\x82\x79\x9f"
			  "\x16\xd4\x9d\x67\xcd\x70\x65\x26"
			  "\x21\x72\x1e\xa1\x94\x8a\x83\x0c"
			  "\x92\x42\x58\x5e\xa2\xc5\x31\xf3"
			  "\x7b\xd1\x31\xd4\x15\x80\x31\x61"
			  "\x5c\x53\x10\xdd\xea\xc8\x83\x5c"
			  "\x7d\xa7\x05\x66\xcc\x1e\xbb\x05"
			  "\x47\xae\xb4\x0f\x84\xd8\xf6\xb5"
			  "\xa1\xc6\x52\x00\x52\xe8\xdc\xd9"
			  "\x16\x31\xb2\x47\x91\x67\xaa\x28"
			  "\x2c\x29\x85\xa3\xf7\xf2\x24\x93"
			  "\x23\x80\x1f\xa8\x1b\x82\x8d\xdc"
			  "\x9f\x0b\xcd\xb4\x3c\x20\xbc\xec"
			  "\x4f\xc7\xee\xf8\xfd\xd9\xfb\x7e"
			  "\x3f\x0d\x23\xfa\x3f\xa7\xcc\x66"
			  "\x1c\xfe\xa6\x86\xf6\xf7\x85\xc7"
			  "\x43\xc1\xd4\xfc\xe4\x79\xc9\x1d"
			  "\xf8\x89\xcd\x20\x27\x84\x5d\x5c"
			  "\x8e\x4f\x1f\xeb\x08\x21\x4f\xa3"
			  "\xe0\x7e\x0b\x9c\xe7\x42\xcf\xb7"
			  "\x3f\x43\xcc\x86\x71\x34\x6a\xd9"
			  "\x5e\xec\x8f\x36\xc9\x0a\x03\xfe"
			  "\x18\x41\xdc\x9e\x2e\x75\x20\x3e"
			  "\xcc\x77\xe0\x8f\xe8\x43\x37\x4c"
			  "\xed\x1a\x5a\xb3\xfa\x43\xc9\x71"
			  "\x9f\xc5\xce\xcf\xff\xe7\x77\x1e"
			  "\x35\x93\xde\x6b\xc0\x6a\x7e\xa9"
			  "\x34\xb8\x27\x74\x08\xda\xf2\x4a"
			  "\x23\x5b\x9f\x55\x3a\x57\x82\x52"
			  "\xea\x6d\xc3\xc7\xf2\xc8\xb5\xdc"
			  "\xc5\xb9\xbb\xaa\xf2\x29\x9f\x49"
			  "\x7a\xef\xfe\xdc\x9f\xc9\x28\xe2"
			  "\x96\x0b\x35\x84\x05\x0d\xd6\x2a"
			  "\xea\x5a\xbf\x69\xde\xee\x4f\x8f"
			  "\x84\xb9\xcf\xa7\x57\xea\xe0\xe8"
			  "\x96\xef\x0f\x0e\xec\xc7\xa6\x74"
			  "\xb1\xfe\x7a\x6d\x11\xdd\x0e\x15"
			  "\x4a\x1e\x73\x7f\x55\xea\xf6\xe1"
			  "\x5b\xb6\x71\xda\xb0\x0c\xba\x26"
			  "\x5c\x48\x38\x6d\x1c\x32\xb2\x7d"
			  "\x05\x87\xc2\x1e\x7e\x2d\xd4\x33"
			  "\xcc\x06\xdb\xe7\x82\x29\x63\xd1"
			  "\x52\x84\x4f\xee\x27\xe8\x02\xd4"
			  "\x34\x3c\x69\xc2\xbd\x20\xe6\x7a",
		.rlen	= 512,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.ilen	= 512,
		.result	= "\x49\xcd\xb8\xbf\x2f\x73\x37\x28"
			  "\x9a\x7f\x6e\x57\x55\xb8\x07\x88"
			  "\x4a\x0d\x8b\x55\x60\xed\xb6\x7b"
			  "\xf1\x74\xac\x96\x05\x7b\x32\xca"
			  "\xd1\x4e\xf1\x58\x29\x16\x24\x6c"
			  "\xf2\xb3\xe4\x88\x84\xac\x4d\xee"
			  "\x97\x07\x82\xf0\x07\x12\x38\x0a"
			  "\x67\x62\xaf\xfd\x85\x9f\x0a\x55"
			  "\xa5\x20\xc5\x60\xe4\x68\x53\xa4"
			  "\x0e\x2e\x65\xe3\xe4\x0c\x30\x7c"
			  "\x1c\x01\x4f\x55\xa9\x13\xeb\x25"
			  "\x21\x87\xbc\xd3\xe7\x67\x4f\x38"
			  "\xa8\x14\x25\x71\xe9\x2e\x4c\x21"
			  "\x41\x82\x0c\x45\x39\x35\xa8\x75"
			  "\x03\x29\x01\x84\x8c\xab\x48\xbe"
			  "\x11\x56\x22\x67\xb7\x67\x1a\x09"
			  "\xa1\x72\x25\x41\x3c\x39\x65\x80"
			  "\x7d\x2f\xf8\x2c\x73\x04\x58\x9d"
			  "\xdd\x16\x8b\x63\x70\x4e\xc5\x17"
			  "\x21\xe0\x84\x51\x4b\x6f\x05\x52"
			  "\xe3\x63\x34\xfa\xa4\xaf\x33\x20"
			  "\xc1\xae\x32\xc4\xb8\x2b\xdb\x76"
			  "\xd9\x02\x31\x2f\xa3\xc6\xd0\x7b"
			  "\xaf\x1b\x84\xe3\x9b\xbf\xa6\xe0"
			  "\xb8\x8a\x13\x88\x71\xf4\x11\xa5"
			  "\xe9\xa9\x10\x33\xe0\xbe\x49\x89"
			  "\x41\x22\xf5\x9d\x80\x3e\x3b\x76"
			  "\x01\x16\x50\x6e\x7c\x6a\x81\xe9"
			  "\x13\x2c\xde\xb2\x5f\x79\xba\xb2"
			  "\xb1\x75\xae\xd2\x07\x98\x4b\x69"
			  "\xae\x7d\x5b\x90\xc2\x6c\xe6\x98"
			  "\xd3\x4c\xa1\xa3\x9c\xc9\x33\x6a"
			  "\x0d\x23\xb1\x79\x25\x13\x4b\xe5"
			  "\xaf\x93\x20\x5c\x7f\x06\x7a\x34"
			  "\x0b\x78\xe3\x67\x26\xe0\xad\x95"
			  "\xc5\x4e\x26\x22\xcf\x73\x77\x62"
			  "\x3e\x10\xd7\x90\x4b\x52\x1c\xc9"
			  "\xef\x38\x52\x18\x0e\x29\x7e\xef"
			  "\x34\xfe\x31\x95\xc5\xbc\xa8\xe2"
			  "\xa8\x4e\x9f\xea\xa6\xf0\xfe\x5d"
			  "\xc5\x39\x86\xed\x2f\x6d\xa0\xfe"
			  "\x96\xcd\x41\x10\x78\x4e\x0c\xc9"
			  "\xc3\x6d\x0f\xb7\xe8\xe0\x62\xab"
			  "\x8b\xf1\x21\x89\xa1\x12\xaa\xfa"
			  "\x9d\x70\xbe\x4c\xa8\x98\x89\x01"
			  "\xb9\xe2\x61\xde\x0c\x4a\x0b\xaa"
			  "\x89\xf5\x14\x79\x18\x8f\x3b\x0d"
			  "\x21\x17\xf8\x59\x15\x24\x64\x22"
			  "\x57\x48\x80\xd5\x3d\x92\x30\x07"
			  "\xd9\xa1\x4a\x23\x16\x43\x48\x0e"
			  "\x2b\x2d\x1b\x87\xef\x7e\xbd\xfa"
			  "\x49\xbc\x7e\x68\x6e\xa8\x46\x95"
			  "\xad\x5e\xfe\x0a\xa8\xd3\x1a\x5d"
			  "\x6b\x84\xf3\x00\xba\x52\x05\x02"
			  "\xe3\x96\x4e\xb6\x79\x3f\x43\xd3"
			  "\x4d\x3f\xd6\xab\x0a\xc4\x75\x2d"
			  "\xd1\x08\xc3\x6a\xc8\x37\x29\xa0"
			  "\xcc\x9a\x05\xdd\x5c\xe1\xff\x66"
			  "\xf2\x7a\x1d\xf2\xaf\xa9\x48\x89"
			  "\xf5\x21\x0f\x02\x48\x83\x74\xbf"
			  "\x2e\xe6\x93\x7b\xa0\xf4\xb1\x2b"
			  "\xb1\x02\x0a\x5c\x79\x19\x3b\x75"
			  "\xb7\x16\xd8\x12\x5c\xcd\x7d\x4e"
			  "\xd5\xc6\x99\xcc\x4e\x6c\x94\x95",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

static struct cipher_testvec camellia_xts_dec_tv_template[] = {
	 
	{
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x06\xcb\xa5\xf1\x04\x63\xb2\x41"
			  "\xdc\xca\xfa\x09\xba\x74\xb9\x05"
			  "\x78\xba\xa4\xf8\x67\x4d\x7e\xad"
			  "\x20\x18\xf5\x0c\x41\x16\x2a\x61",
		.ilen	= 32,
		.result	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.rlen	= 32,
	}, {
		.key	= "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x11\x11\x11\x11\x11\x11\x11\x11"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xc2\xb9\xdc\x44\x1d\xdf\xf2\x86"
			  "\x8d\x35\x42\x0a\xa5\x5e\x3d\x4f"
			  "\xb5\x37\x06\xff\xbd\xd4\x91\x70"
			  "\x80\x1f\xb2\x39\x10\x89\x44\xf5",
		.ilen	= 32,
		.result	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen	= 32,
	}, {
		.key	= "\xff\xfe\xfd\xfc\xfb\xfa\xf9\xf8"
			  "\xf7\xf6\xf5\xf4\xf3\xf2\xf1\xf0"
			  "\x22\x22\x22\x22\x22\x22\x22\x22"
			  "\x22\x22\x22\x22\x22\x22\x22\x22",
		.klen	= 32,
		.iv	= "\x33\x33\x33\x33\x33\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x52\x1f\x9d\xf5\x5a\x58\x5a\x7e"
			  "\x9f\xd0\x8e\x02\x9c\x9a\x6a\xa7"
			  "\xb4\x3b\xce\xe7\x17\xaa\x89\x6a"
			  "\x35\x3c\x6b\xb5\x61\x1c\x79\x38",
		.ilen	= 32,
		.result	= "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44"
			  "\x44\x44\x44\x44\x44\x44\x44\x44",
		.rlen	= 32,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95",
		.klen	= 32,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xc7\xf9\x0a\xaa\xcb\xb5\x8f\x33"
			  "\x60\xc3\xe9\x47\x90\xb7\x50\x57"
			  "\xa3\xad\x81\x2f\xf5\x22\x96\x02"
			  "\xaa\x7f\xea\xac\x29\x78\xca\x2a"
			  "\x7c\xcd\x31\x1a\x3c\x40\x0a\x73"
			  "\x09\x66\xad\x72\x0e\x4d\x5d\x77"
			  "\xbc\xb8\x76\x80\x37\x59\xa9\x01"
			  "\x9e\xfb\xdb\x6c\x93\xef\xb6\x8d"
			  "\x1e\xc1\x94\xa8\xd4\xb5\xb0\x01"
			  "\xd5\x01\x97\x28\xcd\x7a\x1f\xe8"
			  "\x08\xda\x76\x00\x65\xcf\x7b\x31"
			  "\xc6\xfa\xf2\x3b\x00\xa7\x6a\x9e"
			  "\x6c\x43\x80\x87\xe0\xbb\x4e\xe5"
			  "\xdc\x8a\xdf\xc3\x1d\x1b\x41\x04"
			  "\xfb\x54\xdd\x29\x27\xc2\x65\x17"
			  "\x36\x88\xb0\x85\x8d\x73\x7e\x4b"
			  "\x1d\x16\x8a\x52\xbc\xa6\xbc\xa4"
			  "\x8c\xd1\x04\x16\xbf\x8c\x01\x0f"
			  "\x7e\x6b\x59\x15\x29\xd1\x9b\xd3"
			  "\x6c\xee\xac\xdc\x45\x58\xca\x5b"
			  "\x70\x0e\x6a\x12\x86\x82\x79\x9f"
			  "\x16\xd4\x9d\x67\xcd\x70\x65\x26"
			  "\x21\x72\x1e\xa1\x94\x8a\x83\x0c"
			  "\x92\x42\x58\x5e\xa2\xc5\x31\xf3"
			  "\x7b\xd1\x31\xd4\x15\x80\x31\x61"
			  "\x5c\x53\x10\xdd\xea\xc8\x83\x5c"
			  "\x7d\xa7\x05\x66\xcc\x1e\xbb\x05"
			  "\x47\xae\xb4\x0f\x84\xd8\xf6\xb5"
			  "\xa1\xc6\x52\x00\x52\xe8\xdc\xd9"
			  "\x16\x31\xb2\x47\x91\x67\xaa\x28"
			  "\x2c\x29\x85\xa3\xf7\xf2\x24\x93"
			  "\x23\x80\x1f\xa8\x1b\x82\x8d\xdc"
			  "\x9f\x0b\xcd\xb4\x3c\x20\xbc\xec"
			  "\x4f\xc7\xee\xf8\xfd\xd9\xfb\x7e"
			  "\x3f\x0d\x23\xfa\x3f\xa7\xcc\x66"
			  "\x1c\xfe\xa6\x86\xf6\xf7\x85\xc7"
			  "\x43\xc1\xd4\xfc\xe4\x79\xc9\x1d"
			  "\xf8\x89\xcd\x20\x27\x84\x5d\x5c"
			  "\x8e\x4f\x1f\xeb\x08\x21\x4f\xa3"
			  "\xe0\x7e\x0b\x9c\xe7\x42\xcf\xb7"
			  "\x3f\x43\xcc\x86\x71\x34\x6a\xd9"
			  "\x5e\xec\x8f\x36\xc9\x0a\x03\xfe"
			  "\x18\x41\xdc\x9e\x2e\x75\x20\x3e"
			  "\xcc\x77\xe0\x8f\xe8\x43\x37\x4c"
			  "\xed\x1a\x5a\xb3\xfa\x43\xc9\x71"
			  "\x9f\xc5\xce\xcf\xff\xe7\x77\x1e"
			  "\x35\x93\xde\x6b\xc0\x6a\x7e\xa9"
			  "\x34\xb8\x27\x74\x08\xda\xf2\x4a"
			  "\x23\x5b\x9f\x55\x3a\x57\x82\x52"
			  "\xea\x6d\xc3\xc7\xf2\xc8\xb5\xdc"
			  "\xc5\xb9\xbb\xaa\xf2\x29\x9f\x49"
			  "\x7a\xef\xfe\xdc\x9f\xc9\x28\xe2"
			  "\x96\x0b\x35\x84\x05\x0d\xd6\x2a"
			  "\xea\x5a\xbf\x69\xde\xee\x4f\x8f"
			  "\x84\xb9\xcf\xa7\x57\xea\xe0\xe8"
			  "\x96\xef\x0f\x0e\xec\xc7\xa6\x74"
			  "\xb1\xfe\x7a\x6d\x11\xdd\x0e\x15"
			  "\x4a\x1e\x73\x7f\x55\xea\xf6\xe1"
			  "\x5b\xb6\x71\xda\xb0\x0c\xba\x26"
			  "\x5c\x48\x38\x6d\x1c\x32\xb2\x7d"
			  "\x05\x87\xc2\x1e\x7e\x2d\xd4\x33"
			  "\xcc\x06\xdb\xe7\x82\x29\x63\xd1"
			  "\x52\x84\x4f\xee\x27\xe8\x02\xd4"
			  "\x34\x3c\x69\xc2\xbd\x20\xe6\x7a",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
	}, {
		.key	= "\x27\x18\x28\x18\x28\x45\x90\x45"
			  "\x23\x53\x60\x28\x74\x71\x35\x26"
			  "\x62\x49\x77\x57\x24\x70\x93\x69"
			  "\x99\x59\x57\x49\x66\x96\x76\x27"
			  "\x31\x41\x59\x26\x53\x58\x97\x93"
			  "\x23\x84\x62\x64\x33\x83\x27\x95"
			  "\x02\x88\x41\x97\x16\x93\x99\x37"
			  "\x51\x05\x82\x09\x74\x94\x45\x92",
		.klen	= 64,
		.iv	= "\xff\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x49\xcd\xb8\xbf\x2f\x73\x37\x28"
			  "\x9a\x7f\x6e\x57\x55\xb8\x07\x88"
			  "\x4a\x0d\x8b\x55\x60\xed\xb6\x7b"
			  "\xf1\x74\xac\x96\x05\x7b\x32\xca"
			  "\xd1\x4e\xf1\x58\x29\x16\x24\x6c"
			  "\xf2\xb3\xe4\x88\x84\xac\x4d\xee"
			  "\x97\x07\x82\xf0\x07\x12\x38\x0a"
			  "\x67\x62\xaf\xfd\x85\x9f\x0a\x55"
			  "\xa5\x20\xc5\x60\xe4\x68\x53\xa4"
			  "\x0e\x2e\x65\xe3\xe4\x0c\x30\x7c"
			  "\x1c\x01\x4f\x55\xa9\x13\xeb\x25"
			  "\x21\x87\xbc\xd3\xe7\x67\x4f\x38"
			  "\xa8\x14\x25\x71\xe9\x2e\x4c\x21"
			  "\x41\x82\x0c\x45\x39\x35\xa8\x75"
			  "\x03\x29\x01\x84\x8c\xab\x48\xbe"
			  "\x11\x56\x22\x67\xb7\x67\x1a\x09"
			  "\xa1\x72\x25\x41\x3c\x39\x65\x80"
			  "\x7d\x2f\xf8\x2c\x73\x04\x58\x9d"
			  "\xdd\x16\x8b\x63\x70\x4e\xc5\x17"
			  "\x21\xe0\x84\x51\x4b\x6f\x05\x52"
			  "\xe3\x63\x34\xfa\xa4\xaf\x33\x20"
			  "\xc1\xae\x32\xc4\xb8\x2b\xdb\x76"
			  "\xd9\x02\x31\x2f\xa3\xc6\xd0\x7b"
			  "\xaf\x1b\x84\xe3\x9b\xbf\xa6\xe0"
			  "\xb8\x8a\x13\x88\x71\xf4\x11\xa5"
			  "\xe9\xa9\x10\x33\xe0\xbe\x49\x89"
			  "\x41\x22\xf5\x9d\x80\x3e\x3b\x76"
			  "\x01\x16\x50\x6e\x7c\x6a\x81\xe9"
			  "\x13\x2c\xde\xb2\x5f\x79\xba\xb2"
			  "\xb1\x75\xae\xd2\x07\x98\x4b\x69"
			  "\xae\x7d\x5b\x90\xc2\x6c\xe6\x98"
			  "\xd3\x4c\xa1\xa3\x9c\xc9\x33\x6a"
			  "\x0d\x23\xb1\x79\x25\x13\x4b\xe5"
			  "\xaf\x93\x20\x5c\x7f\x06\x7a\x34"
			  "\x0b\x78\xe3\x67\x26\xe0\xad\x95"
			  "\xc5\x4e\x26\x22\xcf\x73\x77\x62"
			  "\x3e\x10\xd7\x90\x4b\x52\x1c\xc9"
			  "\xef\x38\x52\x18\x0e\x29\x7e\xef"
			  "\x34\xfe\x31\x95\xc5\xbc\xa8\xe2"
			  "\xa8\x4e\x9f\xea\xa6\xf0\xfe\x5d"
			  "\xc5\x39\x86\xed\x2f\x6d\xa0\xfe"
			  "\x96\xcd\x41\x10\x78\x4e\x0c\xc9"
			  "\xc3\x6d\x0f\xb7\xe8\xe0\x62\xab"
			  "\x8b\xf1\x21\x89\xa1\x12\xaa\xfa"
			  "\x9d\x70\xbe\x4c\xa8\x98\x89\x01"
			  "\xb9\xe2\x61\xde\x0c\x4a\x0b\xaa"
			  "\x89\xf5\x14\x79\x18\x8f\x3b\x0d"
			  "\x21\x17\xf8\x59\x15\x24\x64\x22"
			  "\x57\x48\x80\xd5\x3d\x92\x30\x07"
			  "\xd9\xa1\x4a\x23\x16\x43\x48\x0e"
			  "\x2b\x2d\x1b\x87\xef\x7e\xbd\xfa"
			  "\x49\xbc\x7e\x68\x6e\xa8\x46\x95"
			  "\xad\x5e\xfe\x0a\xa8\xd3\x1a\x5d"
			  "\x6b\x84\xf3\x00\xba\x52\x05\x02"
			  "\xe3\x96\x4e\xb6\x79\x3f\x43\xd3"
			  "\x4d\x3f\xd6\xab\x0a\xc4\x75\x2d"
			  "\xd1\x08\xc3\x6a\xc8\x37\x29\xa0"
			  "\xcc\x9a\x05\xdd\x5c\xe1\xff\x66"
			  "\xf2\x7a\x1d\xf2\xaf\xa9\x48\x89"
			  "\xf5\x21\x0f\x02\x48\x83\x74\xbf"
			  "\x2e\xe6\x93\x7b\xa0\xf4\xb1\x2b"
			  "\xb1\x02\x0a\x5c\x79\x19\x3b\x75"
			  "\xb7\x16\xd8\x12\x5c\xcd\x7d\x4e"
			  "\xd5\xc6\x99\xcc\x4e\x6c\x94\x95",
		.ilen	= 512,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			  "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			  "\x20\x21\x22\x23\x24\x25\x26\x27"
			  "\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			  "\x30\x31\x32\x33\x34\x35\x36\x37"
			  "\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			  "\x40\x41\x42\x43\x44\x45\x46\x47"
			  "\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			  "\x50\x51\x52\x53\x54\x55\x56\x57"
			  "\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			  "\x60\x61\x62\x63\x64\x65\x66\x67"
			  "\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			  "\x70\x71\x72\x73\x74\x75\x76\x77"
			  "\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			  "\x80\x81\x82\x83\x84\x85\x86\x87"
			  "\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			  "\x90\x91\x92\x93\x94\x95\x96\x97"
			  "\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			  "\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			  "\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			  "\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			  "\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			  "\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			  "\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			  "\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			  "\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			  "\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			  "\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.rlen	= 512,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 512 - 16, 16 },
	},
};

#define SEED_ENC_TEST_VECTORS	4
#define SEED_DEC_TEST_VECTORS	4

static struct cipher_testvec seed_enc_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.ilen	= 16,
		.result	= "\x5e\xba\xc6\xe0\x05\x4e\x16\x68"
			  "\x19\xaf\xf1\xcc\x6d\x34\x6c\xdb",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= zeroed_string,
		.ilen	= 16,
		.result	= "\xc1\x1f\x22\xf2\x01\x40\x50\x50"
			  "\x84\x48\x35\x97\xe4\x37\x0f\x43",
		.rlen	= 16,
	}, {
		.key	= "\x47\x06\x48\x08\x51\xe6\x1b\xe8"
			  "\x5d\x74\xbf\xb3\xfd\x95\x61\x85",
		.klen	= 16,
		.input	= "\x83\xa2\xf8\xa2\x88\x64\x1f\xb9"
			  "\xa4\xe9\xa5\xcc\x2f\x13\x1c\x7d",
		.ilen	= 16,
		.result	= "\xee\x54\xd1\x3e\xbc\xae\x70\x6d"
			  "\x22\x6b\xc3\x14\x2c\xd4\x0d\x4a",
		.rlen	= 16,
	}, {
		.key	= "\x28\xdb\xc3\xbc\x49\xff\xd8\x7d"
			  "\xcf\xa5\x09\xb1\x1d\x42\x2b\xe7",
		.klen	= 16,
		.input	= "\xb4\x1e\x6b\xe2\xeb\xa8\x4a\x14"
			  "\x8e\x2e\xed\x84\x59\x3c\x5e\xc7",
		.ilen	= 16,
		.result	= "\x9b\x9b\x7b\xfc\xd1\x81\x3c\xb9"
			  "\x5d\x0b\x36\x18\xf4\x0f\x51\x22",
		.rlen	= 16,
	}
};

static struct cipher_testvec seed_dec_tv_template[] = {
	{
		.key    = zeroed_string,
		.klen	= 16,
		.input	= "\x5e\xba\xc6\xe0\x05\x4e\x16\x68"
			  "\x19\xaf\xf1\xcc\x6d\x34\x6c\xdb",
		.ilen	= 16,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\xc1\x1f\x22\xf2\x01\x40\x50\x50"
			  "\x84\x48\x35\x97\xe4\x37\x0f\x43",
		.ilen	= 16,
		.result	= zeroed_string,
		.rlen	= 16,
	}, {
		.key	= "\x47\x06\x48\x08\x51\xe6\x1b\xe8"
			  "\x5d\x74\xbf\xb3\xfd\x95\x61\x85",
		.klen	= 16,
		.input	= "\xee\x54\xd1\x3e\xbc\xae\x70\x6d"
			  "\x22\x6b\xc3\x14\x2c\xd4\x0d\x4a",
		.ilen	= 16,
		.result	= "\x83\xa2\xf8\xa2\x88\x64\x1f\xb9"
			  "\xa4\xe9\xa5\xcc\x2f\x13\x1c\x7d",
		.rlen	= 16,
	}, {
		.key	= "\x28\xdb\xc3\xbc\x49\xff\xd8\x7d"
			  "\xcf\xa5\x09\xb1\x1d\x42\x2b\xe7",
		.klen	= 16,
		.input	= "\x9b\x9b\x7b\xfc\xd1\x81\x3c\xb9"
			  "\x5d\x0b\x36\x18\xf4\x0f\x51\x22",
		.ilen	= 16,
		.result	= "\xb4\x1e\x6b\xe2\xeb\xa8\x4a\x14"
			  "\x8e\x2e\xed\x84\x59\x3c\x5e\xc7",
		.rlen	= 16,
	}
};

#define SALSA20_STREAM_ENC_TEST_VECTORS 5
static struct cipher_testvec salsa20_stream_enc_tv_template[] = {
	 
	{  
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F",
		.klen	= 16,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 39,
		.result	= "\x2D\xD5\xC3\xF7\xBA\x2B\x20\xF7"
			 "\x68\x02\x41\x0C\x68\x86\x88\x89"
			 "\x5A\xD8\xC1\xBD\x4E\xA6\xC9\xB1"
			 "\x40\xFB\x9B\x90\xE2\x10\x49\xBF"
			 "\x58\x3F\x52\x79\x70\xEB\xC1",
		.rlen	= 39,
	}, {  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 16,
		.iv     = "\x80\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 64,
		.result	= "\xB6\x6C\x1E\x44\x46\xDD\x95\x57"
			 "\xE5\x78\xE2\x23\xB0\xB7\x68\x01"
			 "\x7B\x23\xB2\x67\xBB\x02\x34\xAE"
			 "\x46\x26\xBF\x44\x3F\x21\x97\x76"
			 "\x43\x6F\xB1\x9F\xD0\xE8\x86\x6F"
			 "\xCD\x0D\xE9\xA9\x53\x8F\x4A\x09"
			 "\xCA\x9A\xC0\x73\x2E\x30\xBC\xF9"
			 "\x8E\x4F\x13\xE4\xB9\xE2\x01\xD9",
		.rlen	= 64,
	}, {  
		.key	= "\x1B\x1C\x1D\x1E\x1F\x20\x21\x22"
			"\x23\x24\x25\x26\x27\x28\x29\x2A"
			"\x2B\x2C\x2D\x2E\x2F\x30\x31\x32"
			"\x33\x34\x35\x36\x37\x38\x39\x3A",
		.klen	= 32,
		.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00",
		.ilen	= 111,
		.result	= "\xAE\x39\x50\x8E\xAC\x9A\xEC\xE7"
			 "\xBF\x97\xBB\x20\xB9\xDE\xE4\x1F"
			 "\x87\xD9\x47\xF8\x28\x91\x35\x98"
			 "\xDB\x72\xCC\x23\x29\x48\x56\x5E"
			 "\x83\x7E\x0B\xF3\x7D\x5D\x38\x7B"
			 "\x2D\x71\x02\xB4\x3B\xB5\xD8\x23"
			 "\xB0\x4A\xDF\x3C\xEC\xB6\xD9\x3B"
			 "\x9B\xA7\x52\xBE\xC5\xD4\x50\x59"
			 "\x15\x14\xB4\x0E\x40\xE6\x53\xD1"
			 "\x83\x9C\x5B\xA0\x92\x29\x6B\x5E"
			 "\x96\x5B\x1E\x2F\xD3\xAC\xC1\x92"
			 "\xB1\x41\x3F\x19\x2F\xC4\x3B\xC6"
			 "\x95\x46\x45\x54\xE9\x75\x03\x08"
			 "\x44\xAF\xE5\x8A\x81\x12\x09",
		.rlen	= 111,
	}, {  
		.key	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.klen	= 32,
		.iv     = "\x00\x00\x00\x10\x00\x00\x00\x00",
		.input	= "\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00",
		.ilen	= 129,
		.result	= "\xD2\xDB\x1A\x5C\xF1\xC1\xAC\xDB"
			 "\xE8\x1A\x7A\x43\x40\xEF\x53\x43"
			 "\x5E\x7F\x4B\x1A\x50\x52\x3F\x8D"
			 "\x28\x3D\xCF\x85\x1D\x69\x6E\x60"
			 "\xF2\xDE\x74\x56\x18\x1B\x84\x10"
			 "\xD4\x62\xBA\x60\x50\xF0\x61\xF2"
			 "\x1C\x78\x7F\xC1\x24\x34\xAF\x58"
			 "\xBF\x2C\x59\xCA\x90\x77\xF3\xB0"
			 "\x5B\x4A\xDF\x89\xCE\x2C\x2F\xFC"
			 "\x67\xF0\xE3\x45\xE8\xB3\xB3\x75"
			 "\xA0\x95\x71\xA1\x29\x39\x94\xCA"
			 "\x45\x2F\xBD\xCB\x10\xB6\xBE\x9F"
			 "\x8E\xF9\xB2\x01\x0A\x5A\x0A\xB7"
			 "\x6B\x9D\x70\x8E\x4B\xD6\x2F\xCD"
			 "\x2E\x40\x48\x75\xE9\xE2\x21\x45"
			 "\x0B\xC9\xB6\xB5\x66\xBC\x9A\x59"
			 "\x5A",
		.rlen	= 129,
	}, {  
		.key =  "\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17"
			"\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen = 32,
		.iv =	"\x00\x00\x00\x00\x00\x00\x00\x00"
			"\x00\x00\x00\x00\x00\x00\x00\x00",
		.input =
			"\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17"
			"\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			"\x20\x21\x22\x23\x24\x25\x26\x27"
			"\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			"\x30\x31\x32\x33\x34\x35\x36\x37"
			"\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			"\x40\x41\x42\x43\x44\x45\x46\x47"
			"\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			"\x50\x51\x52\x53\x54\x55\x56\x57"
			"\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			"\x60\x61\x62\x63\x64\x65\x66\x67"
			"\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			"\x70\x71\x72\x73\x74\x75\x76\x77"
			"\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			"\x80\x81\x82\x83\x84\x85\x86\x87"
			"\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			"\x90\x91\x92\x93\x94\x95\x96\x97"
			"\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			"\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			"\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			"\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			"\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			"\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			"\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			"\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			"\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			"\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			"\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			"\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			"\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			"\x00\x03\x06\x09\x0c\x0f\x12\x15"
			"\x18\x1b\x1e\x21\x24\x27\x2a\x2d"
			"\x30\x33\x36\x39\x3c\x3f\x42\x45"
			"\x48\x4b\x4e\x51\x54\x57\x5a\x5d"
			"\x60\x63\x66\x69\x6c\x6f\x72\x75"
			"\x78\x7b\x7e\x81\x84\x87\x8a\x8d"
			"\x90\x93\x96\x99\x9c\x9f\xa2\xa5"
			"\xa8\xab\xae\xb1\xb4\xb7\xba\xbd"
			"\xc0\xc3\xc6\xc9\xcc\xcf\xd2\xd5"
			"\xd8\xdb\xde\xe1\xe4\xe7\xea\xed"
			"\xf0\xf3\xf6\xf9\xfc\xff\x02\x05"
			"\x08\x0b\x0e\x11\x14\x17\x1a\x1d"
			"\x20\x23\x26\x29\x2c\x2f\x32\x35"
			"\x38\x3b\x3e\x41\x44\x47\x4a\x4d"
			"\x50\x53\x56\x59\x5c\x5f\x62\x65"
			"\x68\x6b\x6e\x71\x74\x77\x7a\x7d"
			"\x80\x83\x86\x89\x8c\x8f\x92\x95"
			"\x98\x9b\x9e\xa1\xa4\xa7\xaa\xad"
			"\xb0\xb3\xb6\xb9\xbc\xbf\xc2\xc5"
			"\xc8\xcb\xce\xd1\xd4\xd7\xda\xdd"
			"\xe0\xe3\xe6\xe9\xec\xef\xf2\xf5"
			"\xf8\xfb\xfe\x01\x04\x07\x0a\x0d"
			"\x10\x13\x16\x19\x1c\x1f\x22\x25"
			"\x28\x2b\x2e\x31\x34\x37\x3a\x3d"
			"\x40\x43\x46\x49\x4c\x4f\x52\x55"
			"\x58\x5b\x5e\x61\x64\x67\x6a\x6d"
			"\x70\x73\x76\x79\x7c\x7f\x82\x85"
			"\x88\x8b\x8e\x91\x94\x97\x9a\x9d"
			"\xa0\xa3\xa6\xa9\xac\xaf\xb2\xb5"
			"\xb8\xbb\xbe\xc1\xc4\xc7\xca\xcd"
			"\xd0\xd3\xd6\xd9\xdc\xdf\xe2\xe5"
			"\xe8\xeb\xee\xf1\xf4\xf7\xfa\xfd"
			"\x00\x05\x0a\x0f\x14\x19\x1e\x23"
			"\x28\x2d\x32\x37\x3c\x41\x46\x4b"
			"\x50\x55\x5a\x5f\x64\x69\x6e\x73"
			"\x78\x7d\x82\x87\x8c\x91\x96\x9b"
			"\xa0\xa5\xaa\xaf\xb4\xb9\xbe\xc3"
			"\xc8\xcd\xd2\xd7\xdc\xe1\xe6\xeb"
			"\xf0\xf5\xfa\xff\x04\x09\x0e\x13"
			"\x18\x1d\x22\x27\x2c\x31\x36\x3b"
			"\x40\x45\x4a\x4f\x54\x59\x5e\x63"
			"\x68\x6d\x72\x77\x7c\x81\x86\x8b"
			"\x90\x95\x9a\x9f\xa4\xa9\xae\xb3"
			"\xb8\xbd\xc2\xc7\xcc\xd1\xd6\xdb"
			"\xe0\xe5\xea\xef\xf4\xf9\xfe\x03"
			"\x08\x0d\x12\x17\x1c\x21\x26\x2b"
			"\x30\x35\x3a\x3f\x44\x49\x4e\x53"
			"\x58\x5d\x62\x67\x6c\x71\x76\x7b"
			"\x80\x85\x8a\x8f\x94\x99\x9e\xa3"
			"\xa8\xad\xb2\xb7\xbc\xc1\xc6\xcb"
			"\xd0\xd5\xda\xdf\xe4\xe9\xee\xf3"
			"\xf8\xfd\x02\x07\x0c\x11\x16\x1b"
			"\x20\x25\x2a\x2f\x34\x39\x3e\x43"
			"\x48\x4d\x52\x57\x5c\x61\x66\x6b"
			"\x70\x75\x7a\x7f\x84\x89\x8e\x93"
			"\x98\x9d\xa2\xa7\xac\xb1\xb6\xbb"
			"\xc0\xc5\xca\xcf\xd4\xd9\xde\xe3"
			"\xe8\xed\xf2\xf7\xfc\x01\x06\x0b"
			"\x10\x15\x1a\x1f\x24\x29\x2e\x33"
			"\x38\x3d\x42\x47\x4c\x51\x56\x5b"
			"\x60\x65\x6a\x6f\x74\x79\x7e\x83"
			"\x88\x8d\x92\x97\x9c\xa1\xa6\xab"
			"\xb0\xb5\xba\xbf\xc4\xc9\xce\xd3"
			"\xd8\xdd\xe2\xe7\xec\xf1\xf6\xfb"
			"\x00\x07\x0e\x15\x1c\x23\x2a\x31"
			"\x38\x3f\x46\x4d\x54\x5b\x62\x69"
			"\x70\x77\x7e\x85\x8c\x93\x9a\xa1"
			"\xa8\xaf\xb6\xbd\xc4\xcb\xd2\xd9"
			"\xe0\xe7\xee\xf5\xfc\x03\x0a\x11"
			"\x18\x1f\x26\x2d\x34\x3b\x42\x49"
			"\x50\x57\x5e\x65\x6c\x73\x7a\x81"
			"\x88\x8f\x96\x9d\xa4\xab\xb2\xb9"
			"\xc0\xc7\xce\xd5\xdc\xe3\xea\xf1"
			"\xf8\xff\x06\x0d\x14\x1b\x22\x29"
			"\x30\x37\x3e\x45\x4c\x53\x5a\x61"
			"\x68\x6f\x76\x7d\x84\x8b\x92\x99"
			"\xa0\xa7\xae\xb5\xbc\xc3\xca\xd1"
			"\xd8\xdf\xe6\xed\xf4\xfb\x02\x09"
			"\x10\x17\x1e\x25\x2c\x33\x3a\x41"
			"\x48\x4f\x56\x5d\x64\x6b\x72\x79"
			"\x80\x87\x8e\x95\x9c\xa3\xaa\xb1"
			"\xb8\xbf\xc6\xcd\xd4\xdb\xe2\xe9"
			"\xf0\xf7\xfe\x05\x0c\x13\x1a\x21"
			"\x28\x2f\x36\x3d\x44\x4b\x52\x59"
			"\x60\x67\x6e\x75\x7c\x83\x8a\x91"
			"\x98\x9f\xa6\xad\xb4\xbb\xc2\xc9"
			"\xd0\xd7\xde\xe5\xec\xf3\xfa\x01"
			"\x08\x0f\x16\x1d\x24\x2b\x32\x39"
			"\x40\x47\x4e\x55\x5c\x63\x6a\x71"
			"\x78\x7f\x86\x8d\x94\x9b\xa2\xa9"
			"\xb0\xb7\xbe\xc5\xcc\xd3\xda\xe1"
			"\xe8\xef\xf6\xfd\x04\x0b\x12\x19"
			"\x20\x27\x2e\x35\x3c\x43\x4a\x51"
			"\x58\x5f\x66\x6d\x74\x7b\x82\x89"
			"\x90\x97\x9e\xa5\xac\xb3\xba\xc1"
			"\xc8\xcf\xd6\xdd\xe4\xeb\xf2\xf9"
			"\x00\x09\x12\x1b\x24\x2d\x36\x3f"
			"\x48\x51\x5a\x63\x6c\x75\x7e\x87"
			"\x90\x99\xa2\xab\xb4\xbd\xc6\xcf"
			"\xd8\xe1\xea\xf3\xfc\x05\x0e\x17"
			"\x20\x29\x32\x3b\x44\x4d\x56\x5f"
			"\x68\x71\x7a\x83\x8c\x95\x9e\xa7"
			"\xb0\xb9\xc2\xcb\xd4\xdd\xe6\xef"
			"\xf8\x01\x0a\x13\x1c\x25\x2e\x37"
			"\x40\x49\x52\x5b\x64\x6d\x76\x7f"
			"\x88\x91\x9a\xa3\xac\xb5\xbe\xc7"
			"\xd0\xd9\xe2\xeb\xf4\xfd\x06\x0f"
			"\x18\x21\x2a\x33\x3c\x45\x4e\x57"
			"\x60\x69\x72\x7b\x84\x8d\x96\x9f"
			"\xa8\xb1\xba\xc3\xcc\xd5\xde\xe7"
			"\xf0\xf9\x02\x0b\x14\x1d\x26\x2f"
			"\x38\x41\x4a\x53\x5c\x65\x6e\x77"
			"\x80\x89\x92\x9b\xa4\xad\xb6\xbf"
			"\xc8\xd1\xda\xe3\xec\xf5\xfe\x07"
			"\x10\x19\x22\x2b\x34\x3d\x46\x4f"
			"\x58\x61\x6a\x73\x7c\x85\x8e\x97"
			"\xa0\xa9\xb2\xbb\xc4\xcd\xd6\xdf"
			"\xe8\xf1\xfa\x03\x0c\x15\x1e\x27"
			"\x30\x39\x42\x4b\x54\x5d\x66\x6f"
			"\x78\x81\x8a\x93\x9c\xa5\xae\xb7"
			"\xc0\xc9\xd2\xdb\xe4\xed\xf6\xff"
			"\x08\x11\x1a\x23\x2c\x35\x3e\x47"
			"\x50\x59\x62\x6b\x74\x7d\x86\x8f"
			"\x98\xa1\xaa\xb3\xbc\xc5\xce\xd7"
			"\xe0\xe9\xf2\xfb\x04\x0d\x16\x1f"
			"\x28\x31\x3a\x43\x4c\x55\x5e\x67"
			"\x70\x79\x82\x8b\x94\x9d\xa6\xaf"
			"\xb8\xc1\xca\xd3\xdc\xe5\xee\xf7"
			"\x00\x0b\x16\x21\x2c\x37\x42\x4d"
			"\x58\x63\x6e\x79\x84\x8f\x9a\xa5"
			"\xb0\xbb\xc6\xd1\xdc\xe7\xf2\xfd"
			"\x08\x13\x1e\x29\x34\x3f\x4a\x55"
			"\x60\x6b\x76\x81\x8c\x97\xa2\xad"
			"\xb8\xc3\xce\xd9\xe4\xef\xfa\x05"
			"\x10\x1b\x26\x31\x3c\x47\x52\x5d"
			"\x68\x73\x7e\x89\x94\x9f\xaa\xb5"
			"\xc0\xcb\xd6\xe1\xec\xf7\x02\x0d"
			"\x18\x23\x2e\x39\x44\x4f\x5a\x65"
			"\x70\x7b\x86\x91\x9c\xa7\xb2\xbd"
			"\xc8\xd3\xde\xe9\xf4\xff\x0a\x15"
			"\x20\x2b\x36\x41\x4c\x57\x62\x6d"
			"\x78\x83\x8e\x99\xa4\xaf\xba\xc5"
			"\xd0\xdb\xe6\xf1\xfc\x07\x12\x1d"
			"\x28\x33\x3e\x49\x54\x5f\x6a\x75"
			"\x80\x8b\x96\xa1\xac\xb7\xc2\xcd"
			"\xd8\xe3\xee\xf9\x04\x0f\x1a\x25"
			"\x30\x3b\x46\x51\x5c\x67\x72\x7d"
			"\x88\x93\x9e\xa9\xb4\xbf\xca\xd5"
			"\xe0\xeb\xf6\x01\x0c\x17\x22\x2d"
			"\x38\x43\x4e\x59\x64\x6f\x7a\x85"
			"\x90\x9b\xa6\xb1\xbc\xc7\xd2\xdd"
			"\xe8\xf3\xfe\x09\x14\x1f\x2a\x35"
			"\x40\x4b\x56\x61\x6c\x77\x82\x8d"
			"\x98\xa3\xae\xb9\xc4\xcf\xda\xe5"
			"\xf0\xfb\x06\x11\x1c\x27\x32\x3d"
			"\x48\x53\x5e\x69\x74\x7f\x8a\x95"
			"\xa0\xab\xb6\xc1\xcc\xd7\xe2\xed"
			"\xf8\x03\x0e\x19\x24\x2f\x3a\x45"
			"\x50\x5b\x66\x71\x7c\x87\x92\x9d"
			"\xa8\xb3\xbe\xc9\xd4\xdf\xea\xf5"
			"\x00\x0d\x1a\x27\x34\x41\x4e\x5b"
			"\x68\x75\x82\x8f\x9c\xa9\xb6\xc3"
			"\xd0\xdd\xea\xf7\x04\x11\x1e\x2b"
			"\x38\x45\x52\x5f\x6c\x79\x86\x93"
			"\xa0\xad\xba\xc7\xd4\xe1\xee\xfb"
			"\x08\x15\x22\x2f\x3c\x49\x56\x63"
			"\x70\x7d\x8a\x97\xa4\xb1\xbe\xcb"
			"\xd8\xe5\xf2\xff\x0c\x19\x26\x33"
			"\x40\x4d\x5a\x67\x74\x81\x8e\x9b"
			"\xa8\xb5\xc2\xcf\xdc\xe9\xf6\x03"
			"\x10\x1d\x2a\x37\x44\x51\x5e\x6b"
			"\x78\x85\x92\x9f\xac\xb9\xc6\xd3"
			"\xe0\xed\xfa\x07\x14\x21\x2e\x3b"
			"\x48\x55\x62\x6f\x7c\x89\x96\xa3"
			"\xb0\xbd\xca\xd7\xe4\xf1\xfe\x0b"
			"\x18\x25\x32\x3f\x4c\x59\x66\x73"
			"\x80\x8d\x9a\xa7\xb4\xc1\xce\xdb"
			"\xe8\xf5\x02\x0f\x1c\x29\x36\x43"
			"\x50\x5d\x6a\x77\x84\x91\x9e\xab"
			"\xb8\xc5\xd2\xdf\xec\xf9\x06\x13"
			"\x20\x2d\x3a\x47\x54\x61\x6e\x7b"
			"\x88\x95\xa2\xaf\xbc\xc9\xd6\xe3"
			"\xf0\xfd\x0a\x17\x24\x31\x3e\x4b"
			"\x58\x65\x72\x7f\x8c\x99\xa6\xb3"
			"\xc0\xcd\xda\xe7\xf4\x01\x0e\x1b"
			"\x28\x35\x42\x4f\x5c\x69\x76\x83"
			"\x90\x9d\xaa\xb7\xc4\xd1\xde\xeb"
			"\xf8\x05\x12\x1f\x2c\x39\x46\x53"
			"\x60\x6d\x7a\x87\x94\xa1\xae\xbb"
			"\xc8\xd5\xe2\xef\xfc\x09\x16\x23"
			"\x30\x3d\x4a\x57\x64\x71\x7e\x8b"
			"\x98\xa5\xb2\xbf\xcc\xd9\xe6\xf3"
			"\x00\x0f\x1e\x2d\x3c\x4b\x5a\x69"
			"\x78\x87\x96\xa5\xb4\xc3\xd2\xe1"
			"\xf0\xff\x0e\x1d\x2c\x3b\x4a\x59"
			"\x68\x77\x86\x95\xa4\xb3\xc2\xd1"
			"\xe0\xef\xfe\x0d\x1c\x2b\x3a\x49"
			"\x58\x67\x76\x85\x94\xa3\xb2\xc1"
			"\xd0\xdf\xee\xfd\x0c\x1b\x2a\x39"
			"\x48\x57\x66\x75\x84\x93\xa2\xb1"
			"\xc0\xcf\xde\xed\xfc\x0b\x1a\x29"
			"\x38\x47\x56\x65\x74\x83\x92\xa1"
			"\xb0\xbf\xce\xdd\xec\xfb\x0a\x19"
			"\x28\x37\x46\x55\x64\x73\x82\x91"
			"\xa0\xaf\xbe\xcd\xdc\xeb\xfa\x09"
			"\x18\x27\x36\x45\x54\x63\x72\x81"
			"\x90\x9f\xae\xbd\xcc\xdb\xea\xf9"
			"\x08\x17\x26\x35\x44\x53\x62\x71"
			"\x80\x8f\x9e\xad\xbc\xcb\xda\xe9"
			"\xf8\x07\x16\x25\x34\x43\x52\x61"
			"\x70\x7f\x8e\x9d\xac\xbb\xca\xd9"
			"\xe8\xf7\x06\x15\x24\x33\x42\x51"
			"\x60\x6f\x7e\x8d\x9c\xab\xba\xc9"
			"\xd8\xe7\xf6\x05\x14\x23\x32\x41"
			"\x50\x5f\x6e\x7d\x8c\x9b\xaa\xb9"
			"\xc8\xd7\xe6\xf5\x04\x13\x22\x31"
			"\x40\x4f\x5e\x6d\x7c\x8b\x9a\xa9"
			"\xb8\xc7\xd6\xe5\xf4\x03\x12\x21"
			"\x30\x3f\x4e\x5d\x6c\x7b\x8a\x99"
			"\xa8\xb7\xc6\xd5\xe4\xf3\x02\x11"
			"\x20\x2f\x3e\x4d\x5c\x6b\x7a\x89"
			"\x98\xa7\xb6\xc5\xd4\xe3\xf2\x01"
			"\x10\x1f\x2e\x3d\x4c\x5b\x6a\x79"
			"\x88\x97\xa6\xb5\xc4\xd3\xe2\xf1"
			"\x00\x11\x22\x33\x44\x55\x66\x77"
			"\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
			"\x10\x21\x32\x43\x54\x65\x76\x87"
			"\x98\xa9\xba\xcb\xdc\xed\xfe\x0f"
			"\x20\x31\x42\x53\x64\x75\x86\x97"
			"\xa8\xb9\xca\xdb\xec\xfd\x0e\x1f"
			"\x30\x41\x52\x63\x74\x85\x96\xa7"
			"\xb8\xc9\xda\xeb\xfc\x0d\x1e\x2f"
			"\x40\x51\x62\x73\x84\x95\xa6\xb7"
			"\xc8\xd9\xea\xfb\x0c\x1d\x2e\x3f"
			"\x50\x61\x72\x83\x94\xa5\xb6\xc7"
			"\xd8\xe9\xfa\x0b\x1c\x2d\x3e\x4f"
			"\x60\x71\x82\x93\xa4\xb5\xc6\xd7"
			"\xe8\xf9\x0a\x1b\x2c\x3d\x4e\x5f"
			"\x70\x81\x92\xa3\xb4\xc5\xd6\xe7"
			"\xf8\x09\x1a\x2b\x3c\x4d\x5e\x6f"
			"\x80\x91\xa2\xb3\xc4\xd5\xe6\xf7"
			"\x08\x19\x2a\x3b\x4c\x5d\x6e\x7f"
			"\x90\xa1\xb2\xc3\xd4\xe5\xf6\x07"
			"\x18\x29\x3a\x4b\x5c\x6d\x7e\x8f"
			"\xa0\xb1\xc2\xd3\xe4\xf5\x06\x17"
			"\x28\x39\x4a\x5b\x6c\x7d\x8e\x9f"
			"\xb0\xc1\xd2\xe3\xf4\x05\x16\x27"
			"\x38\x49\x5a\x6b\x7c\x8d\x9e\xaf"
			"\xc0\xd1\xe2\xf3\x04\x15\x26\x37"
			"\x48\x59\x6a\x7b\x8c\x9d\xae\xbf"
			"\xd0\xe1\xf2\x03\x14\x25\x36\x47"
			"\x58\x69\x7a\x8b\x9c\xad\xbe\xcf"
			"\xe0\xf1\x02\x13\x24\x35\x46\x57"
			"\x68\x79\x8a\x9b\xac\xbd\xce\xdf"
			"\xf0\x01\x12\x23\x34\x45\x56\x67"
			"\x78\x89\x9a\xab\xbc\xcd\xde\xef"
			"\x00\x13\x26\x39\x4c\x5f\x72\x85"
			"\x98\xab\xbe\xd1\xe4\xf7\x0a\x1d"
			"\x30\x43\x56\x69\x7c\x8f\xa2\xb5"
			"\xc8\xdb\xee\x01\x14\x27\x3a\x4d"
			"\x60\x73\x86\x99\xac\xbf\xd2\xe5"
			"\xf8\x0b\x1e\x31\x44\x57\x6a\x7d"
			"\x90\xa3\xb6\xc9\xdc\xef\x02\x15"
			"\x28\x3b\x4e\x61\x74\x87\x9a\xad"
			"\xc0\xd3\xe6\xf9\x0c\x1f\x32\x45"
			"\x58\x6b\x7e\x91\xa4\xb7\xca\xdd"
			"\xf0\x03\x16\x29\x3c\x4f\x62\x75"
			"\x88\x9b\xae\xc1\xd4\xe7\xfa\x0d"
			"\x20\x33\x46\x59\x6c\x7f\x92\xa5"
			"\xb8\xcb\xde\xf1\x04\x17\x2a\x3d"
			"\x50\x63\x76\x89\x9c\xaf\xc2\xd5"
			"\xe8\xfb\x0e\x21\x34\x47\x5a\x6d"
			"\x80\x93\xa6\xb9\xcc\xdf\xf2\x05"
			"\x18\x2b\x3e\x51\x64\x77\x8a\x9d"
			"\xb0\xc3\xd6\xe9\xfc\x0f\x22\x35"
			"\x48\x5b\x6e\x81\x94\xa7\xba\xcd"
			"\xe0\xf3\x06\x19\x2c\x3f\x52\x65"
			"\x78\x8b\x9e\xb1\xc4\xd7\xea\xfd"
			"\x10\x23\x36\x49\x5c\x6f\x82\x95"
			"\xa8\xbb\xce\xe1\xf4\x07\x1a\x2d"
			"\x40\x53\x66\x79\x8c\x9f\xb2\xc5"
			"\xd8\xeb\xfe\x11\x24\x37\x4a\x5d"
			"\x70\x83\x96\xa9\xbc\xcf\xe2\xf5"
			"\x08\x1b\x2e\x41\x54\x67\x7a\x8d"
			"\xa0\xb3\xc6\xd9\xec\xff\x12\x25"
			"\x38\x4b\x5e\x71\x84\x97\xaa\xbd"
			"\xd0\xe3\xf6\x09\x1c\x2f\x42\x55"
			"\x68\x7b\x8e\xa1\xb4\xc7\xda\xed"
			"\x00\x15\x2a\x3f\x54\x69\x7e\x93"
			"\xa8\xbd\xd2\xe7\xfc\x11\x26\x3b"
			"\x50\x65\x7a\x8f\xa4\xb9\xce\xe3"
			"\xf8\x0d\x22\x37\x4c\x61\x76\x8b"
			"\xa0\xb5\xca\xdf\xf4\x09\x1e\x33"
			"\x48\x5d\x72\x87\x9c\xb1\xc6\xdb"
			"\xf0\x05\x1a\x2f\x44\x59\x6e\x83"
			"\x98\xad\xc2\xd7\xec\x01\x16\x2b"
			"\x40\x55\x6a\x7f\x94\xa9\xbe\xd3"
			"\xe8\xfd\x12\x27\x3c\x51\x66\x7b"
			"\x90\xa5\xba\xcf\xe4\xf9\x0e\x23"
			"\x38\x4d\x62\x77\x8c\xa1\xb6\xcb"
			"\xe0\xf5\x0a\x1f\x34\x49\x5e\x73"
			"\x88\x9d\xb2\xc7\xdc\xf1\x06\x1b"
			"\x30\x45\x5a\x6f\x84\x99\xae\xc3"
			"\xd8\xed\x02\x17\x2c\x41\x56\x6b"
			"\x80\x95\xaa\xbf\xd4\xe9\xfe\x13"
			"\x28\x3d\x52\x67\x7c\x91\xa6\xbb"
			"\xd0\xe5\xfa\x0f\x24\x39\x4e\x63"
			"\x78\x8d\xa2\xb7\xcc\xe1\xf6\x0b"
			"\x20\x35\x4a\x5f\x74\x89\x9e\xb3"
			"\xc8\xdd\xf2\x07\x1c\x31\x46\x5b"
			"\x70\x85\x9a\xaf\xc4\xd9\xee\x03"
			"\x18\x2d\x42\x57\x6c\x81\x96\xab"
			"\xc0\xd5\xea\xff\x14\x29\x3e\x53"
			"\x68\x7d\x92\xa7\xbc\xd1\xe6\xfb"
			"\x10\x25\x3a\x4f\x64\x79\x8e\xa3"
			"\xb8\xcd\xe2\xf7\x0c\x21\x36\x4b"
			"\x60\x75\x8a\x9f\xb4\xc9\xde\xf3"
			"\x08\x1d\x32\x47\x5c\x71\x86\x9b"
			"\xb0\xc5\xda\xef\x04\x19\x2e\x43"
			"\x58\x6d\x82\x97\xac\xc1\xd6\xeb"
			"\x00\x17\x2e\x45\x5c\x73\x8a\xa1"
			"\xb8\xcf\xe6\xfd\x14\x2b\x42\x59"
			"\x70\x87\x9e\xb5\xcc\xe3\xfa\x11"
			"\x28\x3f\x56\x6d\x84\x9b\xb2\xc9"
			"\xe0\xf7\x0e\x25\x3c\x53\x6a\x81"
			"\x98\xaf\xc6\xdd\xf4\x0b\x22\x39"
			"\x50\x67\x7e\x95\xac\xc3\xda\xf1"
			"\x08\x1f\x36\x4d\x64\x7b\x92\xa9"
			"\xc0\xd7\xee\x05\x1c\x33\x4a\x61"
			"\x78\x8f\xa6\xbd\xd4\xeb\x02\x19"
			"\x30\x47\x5e\x75\x8c\xa3\xba\xd1"
			"\xe8\xff\x16\x2d\x44\x5b\x72\x89"
			"\xa0\xb7\xce\xe5\xfc\x13\x2a\x41"
			"\x58\x6f\x86\x9d\xb4\xcb\xe2\xf9"
			"\x10\x27\x3e\x55\x6c\x83\x9a\xb1"
			"\xc8\xdf\xf6\x0d\x24\x3b\x52\x69"
			"\x80\x97\xae\xc5\xdc\xf3\x0a\x21"
			"\x38\x4f\x66\x7d\x94\xab\xc2\xd9"
			"\xf0\x07\x1e\x35\x4c\x63\x7a\x91"
			"\xa8\xbf\xd6\xed\x04\x1b\x32\x49"
			"\x60\x77\x8e\xa5\xbc\xd3\xea\x01"
			"\x18\x2f\x46\x5d\x74\x8b\xa2\xb9"
			"\xd0\xe7\xfe\x15\x2c\x43\x5a\x71"
			"\x88\x9f\xb6\xcd\xe4\xfb\x12\x29"
			"\x40\x57\x6e\x85\x9c\xb3\xca\xe1"
			"\xf8\x0f\x26\x3d\x54\x6b\x82\x99"
			"\xb0\xc7\xde\xf5\x0c\x23\x3a\x51"
			"\x68\x7f\x96\xad\xc4\xdb\xf2\x09"
			"\x20\x37\x4e\x65\x7c\x93\xaa\xc1"
			"\xd8\xef\x06\x1d\x34\x4b\x62\x79"
			"\x90\xa7\xbe\xd5\xec\x03\x1a\x31"
			"\x48\x5f\x76\x8d\xa4\xbb\xd2\xe9"
			"\x00\x19\x32\x4b\x64\x7d\x96\xaf"
			"\xc8\xe1\xfa\x13\x2c\x45\x5e\x77"
			"\x90\xa9\xc2\xdb\xf4\x0d\x26\x3f"
			"\x58\x71\x8a\xa3\xbc\xd5\xee\x07"
			"\x20\x39\x52\x6b\x84\x9d\xb6\xcf"
			"\xe8\x01\x1a\x33\x4c\x65\x7e\x97"
			"\xb0\xc9\xe2\xfb\x14\x2d\x46\x5f"
			"\x78\x91\xaa\xc3\xdc\xf5\x0e\x27"
			"\x40\x59\x72\x8b\xa4\xbd\xd6\xef"
			"\x08\x21\x3a\x53\x6c\x85\x9e\xb7"
			"\xd0\xe9\x02\x1b\x34\x4d\x66\x7f"
			"\x98\xb1\xca\xe3\xfc\x15\x2e\x47"
			"\x60\x79\x92\xab\xc4\xdd\xf6\x0f"
			"\x28\x41\x5a\x73\x8c\xa5\xbe\xd7"
			"\xf0\x09\x22\x3b\x54\x6d\x86\x9f"
			"\xb8\xd1\xea\x03\x1c\x35\x4e\x67"
			"\x80\x99\xb2\xcb\xe4\xfd\x16\x2f"
			"\x48\x61\x7a\x93\xac\xc5\xde\xf7"
			"\x10\x29\x42\x5b\x74\x8d\xa6\xbf"
			"\xd8\xf1\x0a\x23\x3c\x55\x6e\x87"
			"\xa0\xb9\xd2\xeb\x04\x1d\x36\x4f"
			"\x68\x81\x9a\xb3\xcc\xe5\xfe\x17"
			"\x30\x49\x62\x7b\x94\xad\xc6\xdf"
			"\xf8\x11\x2a\x43\x5c\x75\x8e\xa7"
			"\xc0\xd9\xf2\x0b\x24\x3d\x56\x6f"
			"\x88\xa1\xba\xd3\xec\x05\x1e\x37"
			"\x50\x69\x82\x9b\xb4\xcd\xe6\xff"
			"\x18\x31\x4a\x63\x7c\x95\xae\xc7"
			"\xe0\xf9\x12\x2b\x44\x5d\x76\x8f"
			"\xa8\xc1\xda\xf3\x0c\x25\x3e\x57"
			"\x70\x89\xa2\xbb\xd4\xed\x06\x1f"
			"\x38\x51\x6a\x83\x9c\xb5\xce\xe7"
			"\x00\x1b\x36\x51\x6c\x87\xa2\xbd"
			"\xd8\xf3\x0e\x29\x44\x5f\x7a\x95"
			"\xb0\xcb\xe6\x01\x1c\x37\x52\x6d"
			"\x88\xa3\xbe\xd9\xf4\x0f\x2a\x45"
			"\x60\x7b\x96\xb1\xcc\xe7\x02\x1d"
			"\x38\x53\x6e\x89\xa4\xbf\xda\xf5"
			"\x10\x2b\x46\x61\x7c\x97\xb2\xcd"
			"\xe8\x03\x1e\x39\x54\x6f\x8a\xa5"
			"\xc0\xdb\xf6\x11\x2c\x47\x62\x7d"
			"\x98\xb3\xce\xe9\x04\x1f\x3a\x55"
			"\x70\x8b\xa6\xc1\xdc\xf7\x12\x2d"
			"\x48\x63\x7e\x99\xb4\xcf\xea\x05"
			"\x20\x3b\x56\x71\x8c\xa7\xc2\xdd"
			"\xf8\x13\x2e\x49\x64\x7f\x9a\xb5"
			"\xd0\xeb\x06\x21\x3c\x57\x72\x8d"
			"\xa8\xc3\xde\xf9\x14\x2f\x4a\x65"
			"\x80\x9b\xb6\xd1\xec\x07\x22\x3d"
			"\x58\x73\x8e\xa9\xc4\xdf\xfa\x15"
			"\x30\x4b\x66\x81\x9c\xb7\xd2\xed"
			"\x08\x23\x3e\x59\x74\x8f\xaa\xc5"
			"\xe0\xfb\x16\x31\x4c\x67\x82\x9d"
			"\xb8\xd3\xee\x09\x24\x3f\x5a\x75"
			"\x90\xab\xc6\xe1\xfc\x17\x32\x4d"
			"\x68\x83\x9e\xb9\xd4\xef\x0a\x25"
			"\x40\x5b\x76\x91\xac\xc7\xe2\xfd"
			"\x18\x33\x4e\x69\x84\x9f\xba\xd5"
			"\xf0\x0b\x26\x41\x5c\x77\x92\xad"
			"\xc8\xe3\xfe\x19\x34\x4f\x6a\x85"
			"\xa0\xbb\xd6\xf1\x0c\x27\x42\x5d"
			"\x78\x93\xae\xc9\xe4\xff\x1a\x35"
			"\x50\x6b\x86\xa1\xbc\xd7\xf2\x0d"
			"\x28\x43\x5e\x79\x94\xaf\xca\xe5"
			"\x00\x1d\x3a\x57\x74\x91\xae\xcb"
			"\xe8\x05\x22\x3f\x5c\x79\x96\xb3"
			"\xd0\xed\x0a\x27\x44\x61\x7e\x9b"
			"\xb8\xd5\xf2\x0f\x2c\x49\x66\x83"
			"\xa0\xbd\xda\xf7\x14\x31\x4e\x6b"
			"\x88\xa5\xc2\xdf\xfc\x19\x36\x53"
			"\x70\x8d\xaa\xc7\xe4\x01\x1e\x3b"
			"\x58\x75\x92\xaf\xcc\xe9\x06\x23"
			"\x40\x5d\x7a\x97\xb4\xd1\xee\x0b"
			"\x28\x45\x62\x7f\x9c\xb9\xd6\xf3"
			"\x10\x2d\x4a\x67\x84\xa1\xbe\xdb"
			"\xf8\x15\x32\x4f\x6c\x89\xa6\xc3"
			"\xe0\xfd\x1a\x37\x54\x71\x8e\xab"
			"\xc8\xe5\x02\x1f\x3c\x59\x76\x93"
			"\xb0\xcd\xea\x07\x24\x41\x5e\x7b"
			"\x98\xb5\xd2\xef\x0c\x29\x46\x63"
			"\x80\x9d\xba\xd7\xf4\x11\x2e\x4b"
			"\x68\x85\xa2\xbf\xdc\xf9\x16\x33"
			"\x50\x6d\x8a\xa7\xc4\xe1\xfe\x1b"
			"\x38\x55\x72\x8f\xac\xc9\xe6\x03"
			"\x20\x3d\x5a\x77\x94\xb1\xce\xeb"
			"\x08\x25\x42\x5f\x7c\x99\xb6\xd3"
			"\xf0\x0d\x2a\x47\x64\x81\x9e\xbb"
			"\xd8\xf5\x12\x2f\x4c\x69\x86\xa3"
			"\xc0\xdd\xfa\x17\x34\x51\x6e\x8b"
			"\xa8\xc5\xe2\xff\x1c\x39\x56\x73"
			"\x90\xad\xca\xe7\x04\x21\x3e\x5b"
			"\x78\x95\xb2\xcf\xec\x09\x26\x43"
			"\x60\x7d\x9a\xb7\xd4\xf1\x0e\x2b"
			"\x48\x65\x82\x9f\xbc\xd9\xf6\x13"
			"\x30\x4d\x6a\x87\xa4\xc1\xde\xfb"
			"\x18\x35\x52\x6f\x8c\xa9\xc6\xe3"
			"\x00\x1f\x3e\x5d\x7c\x9b\xba\xd9"
			"\xf8\x17\x36\x55\x74\x93\xb2\xd1"
			"\xf0\x0f\x2e\x4d\x6c\x8b\xaa\xc9"
			"\xe8\x07\x26\x45\x64\x83\xa2\xc1"
			"\xe0\xff\x1e\x3d\x5c\x7b\x9a\xb9"
			"\xd8\xf7\x16\x35\x54\x73\x92\xb1"
			"\xd0\xef\x0e\x2d\x4c\x6b\x8a\xa9"
			"\xc8\xe7\x06\x25\x44\x63\x82\xa1"
			"\xc0\xdf\xfe\x1d\x3c\x5b\x7a\x99"
			"\xb8\xd7\xf6\x15\x34\x53\x72\x91"
			"\xb0\xcf\xee\x0d\x2c\x4b\x6a\x89"
			"\xa8\xc7\xe6\x05\x24\x43\x62\x81"
			"\xa0\xbf\xde\xfd\x1c\x3b\x5a\x79"
			"\x98\xb7\xd6\xf5\x14\x33\x52\x71"
			"\x90\xaf\xce\xed\x0c\x2b\x4a\x69"
			"\x88\xa7\xc6\xe5\x04\x23\x42\x61"
			"\x80\x9f\xbe\xdd\xfc\x1b\x3a\x59"
			"\x78\x97\xb6\xd5\xf4\x13\x32\x51"
			"\x70\x8f\xae\xcd\xec\x0b\x2a\x49"
			"\x68\x87\xa6\xc5\xe4\x03\x22\x41"
			"\x60\x7f\x9e\xbd\xdc\xfb\x1a\x39"
			"\x58\x77\x96\xb5\xd4\xf3\x12\x31"
			"\x50\x6f\x8e\xad\xcc\xeb\x0a\x29"
			"\x48\x67\x86\xa5\xc4\xe3\x02\x21"
			"\x40\x5f\x7e\x9d\xbc\xdb\xfa\x19"
			"\x38\x57\x76\x95\xb4\xd3\xf2\x11"
			"\x30\x4f\x6e\x8d\xac\xcb\xea\x09"
			"\x28\x47\x66\x85\xa4\xc3\xe2\x01"
			"\x20\x3f\x5e\x7d\x9c\xbb\xda\xf9"
			"\x18\x37\x56\x75\x94\xb3\xd2\xf1"
			"\x10\x2f\x4e\x6d\x8c\xab\xca\xe9"
			"\x08\x27\x46\x65\x84\xa3\xc2\xe1"
			"\x00\x21\x42\x63",
		.ilen = 4100,
		.result =
			"\xb5\x81\xf5\x64\x18\x73\xe3\xf0"
			"\x4c\x13\xf2\x77\x18\x60\x65\x5e"
			"\x29\x01\xce\x98\x55\x53\xf9\x0c"
			"\x2a\x08\xd5\x09\xb3\x57\x55\x56"
			"\xc5\xe9\x56\x90\xcb\x6a\xa3\xc0"
			"\xff\xc4\x79\xb4\xd2\x97\x5d\xc4"
			"\x43\xd1\xfe\x94\x7b\x88\x06\x5a"
			"\xb2\x9e\x2c\xfc\x44\x03\xb7\x90"
			"\xa0\xc1\xba\x6a\x33\xb8\xc7\xb2"
			"\x9d\xe1\x12\x4f\xc0\x64\xd4\x01"
			"\xfe\x8c\x7a\x66\xf7\xe6\x5a\x91"
			"\xbb\xde\x56\x86\xab\x65\x21\x30"
			"\x00\x84\x65\x24\xa5\x7d\x85\xb4"
			"\xe3\x17\xed\x3a\xb7\x6f\xb4\x0b"
			"\x0b\xaf\x15\xae\x5a\x8f\xf2\x0c"
			"\x2f\x27\xf4\x09\xd8\xd2\x96\xb7"
			"\x71\xf2\xc5\x99\x4d\x7e\x7f\x75"
			"\x77\x89\x30\x8b\x59\xdb\xa2\xb2"
			"\xa0\xf3\x19\x39\x2b\xc5\x7e\x3f"
			"\x4f\xd9\xd3\x56\x28\x97\x44\xdc"
			"\xc0\x8b\x77\x24\xd9\x52\xe7\xc5"
			"\xaf\xf6\x7d\x59\xb2\x44\x05\x1d"
			"\xb1\xb0\x11\xa5\x0f\xec\x33\xe1"
			"\x6d\x1b\x4e\x1f\xff\x57\x91\xb4"
			"\x5b\x9a\x96\xc5\x53\xbc\xae\x20"
			"\x3c\xbb\x14\xe2\xe8\x22\x33\xc1"
			"\x5e\x76\x9e\x46\x99\xf6\x2a\x15"
			"\xc6\x97\x02\xa0\x66\x43\xd1\xa6"
			"\x31\xa6\x9f\xfb\xf4\xd3\x69\xe5"
			"\xcd\x76\x95\xb8\x7a\x82\x7f\x21"
			"\x45\xff\x3f\xce\x55\xf6\x95\x10"
			"\x08\x77\x10\x43\xc6\xf3\x09\xe5"
			"\x68\xe7\x3c\xad\x00\x52\x45\x0d"
			"\xfe\x2d\xc6\xc2\x94\x8c\x12\x1d"
			"\xe6\x25\xae\x98\x12\x8e\x19\x9c"
			"\x81\x68\xb1\x11\xf6\x69\xda\xe3"
			"\x62\x08\x18\x7a\x25\x49\x28\xac"
			"\xba\x71\x12\x0b\xe4\xa2\xe5\xc7"
			"\x5d\x8e\xec\x49\x40\x21\xbf\x5a"
			"\x98\xf3\x02\x68\x55\x03\x7f\x8a"
			"\xe5\x94\x0c\x32\x5c\x07\x82\x63"
			"\xaf\x6f\x91\x40\x84\x8e\x52\x25"
			"\xd0\xb0\x29\x53\x05\xe2\x50\x7a"
			"\x34\xeb\xc9\x46\x20\xa8\x3d\xde"
			"\x7f\x16\x5f\x36\xc5\x2e\xdc\xd1"
			"\x15\x47\xc7\x50\x40\x6d\x91\xc5"
			"\xe7\x93\x95\x1a\xd3\x57\xbc\x52"
			"\x33\xee\x14\x19\x22\x52\x89\xa7"
			"\x4a\x25\x56\x77\x4b\xca\xcf\x0a"
			"\xe1\xf5\x35\x85\x30\x7e\x59\x4a"
			"\xbd\x14\x5b\xdf\xe3\x46\xcb\xac"
			"\x1f\x6c\x96\x0e\xf4\x81\xd1\x99"
			"\xca\x88\x63\x3d\x02\x58\x6b\xa9"
			"\xe5\x9f\xb3\x00\xb2\x54\xc6\x74"
			"\x1c\xbf\x46\xab\x97\xcc\xf8\x54"
			"\x04\x07\x08\x52\xe6\xc0\xda\x93"
			"\x74\x7d\x93\x99\x5d\x78\x68\xa6"
			"\x2e\x6b\xd3\x6a\x69\xcc\x12\x6b"
			"\xd4\xc7\xa5\xc6\xe7\xf6\x03\x04"
			"\x5d\xcd\x61\x5e\x17\x40\xdc\xd1"
			"\x5c\xf5\x08\xdf\x5c\x90\x85\xa4"
			"\xaf\xf6\x78\xbb\x0d\xf1\xf4\xa4"
			"\x54\x26\x72\x9e\x61\xfa\x86\xcf"
			"\xe8\x9e\xa1\xe0\xc7\x48\x23\xae"
			"\x5a\x90\xae\x75\x0a\x74\x18\x89"
			"\x05\xb1\x92\xb2\x7f\xd0\x1b\xa6"
			"\x62\x07\x25\x01\xc7\xc2\x4f\xf9"
			"\xe8\xfe\x63\x95\x80\x07\xb4\x26"
			"\xcc\xd1\x26\xb6\xc4\x3f\x9e\xcb"
			"\x8e\x3b\x2e\x44\x16\xd3\x10\x9a"
			"\x95\x08\xeb\xc8\xcb\xeb\xbf\x6f"
			"\x0b\xcd\x1f\xc8\xca\x86\xaa\xec"
			"\x33\xe6\x69\xf4\x45\x25\x86\x3a"
			"\x22\x94\x4f\x00\x23\x6a\x44\xc2"
			"\x49\x97\x33\xab\x36\x14\x0a\x70"
			"\x24\xc3\xbe\x04\x3b\x79\xa0\xf9"
			"\xb8\xe7\x76\x29\x22\x83\xd7\xf2"
			"\x94\xf4\x41\x49\xba\x5f\x7b\x07"
			"\xb5\xfb\xdb\x03\x1a\x9f\xb6\x4c"
			"\xc2\x2e\x37\x40\x49\xc3\x38\x16"
			"\xe2\x4f\x77\x82\xb0\x68\x4c\x71"
			"\x1d\x57\x61\x9c\xd9\x4e\x54\x99"
			"\x47\x13\x28\x73\x3c\xbb\x00\x90"
			"\xf3\x4d\xc9\x0e\xfd\xe7\xb1\x71"
			"\xd3\x15\x79\xbf\xcc\x26\x2f\xbd"
			"\xad\x6c\x50\x69\x6c\x3e\x6d\x80"
			"\x9a\xea\x78\xaf\x19\xb2\x0d\x4d"
			"\xad\x04\x07\xae\x22\x90\x4a\x93"
			"\x32\x0e\x36\x9b\x1b\x46\xba\x3b"
			"\xb4\xac\xc6\xd1\xa2\x31\x53\x3b"
			"\x2a\x3d\x45\xfe\x03\x61\x10\x85"
			"\x17\x69\xa6\x78\xcc\x6c\x87\x49"
			"\x53\xf9\x80\x10\xde\x80\xa2\x41"
			"\x6a\xc3\x32\x02\xad\x6d\x3c\x56"
			"\x00\x71\x51\x06\xa7\xbd\xfb\xef"
			"\x3c\xb5\x9f\xfc\x48\x7d\x53\x7c"
			"\x66\xb0\x49\x23\xc4\x47\x10\x0e"
			"\xe5\x6c\x74\x13\xe6\xc5\x3f\xaa"
			"\xde\xff\x07\x44\xdd\x56\x1b\xad"
			"\x09\x77\xfb\x5b\x12\xb8\x0d\x38"
			"\x17\x37\x35\x7b\x9b\xbc\xfe\xd4"
			"\x7e\x8b\xda\x7e\x5b\x04\xa7\x22"
			"\xa7\x31\xa1\x20\x86\xc7\x1b\x99"
			"\xdb\xd1\x89\xf4\x94\xa3\x53\x69"
			"\x8d\xe7\xe8\x74\x11\x8d\x74\xd6"
			"\x07\x37\x91\x9f\xfd\x67\x50\x3a"
			"\xc9\xe1\xf4\x36\xd5\xa0\x47\xd1"
			"\xf9\xe5\x39\xa3\x31\xac\x07\x36"
			"\x23\xf8\x66\x18\x14\x28\x34\x0f"
			"\xb8\xd0\xe7\x29\xb3\x04\x4b\x55"
			"\x01\x41\xb2\x75\x8d\xcb\x96\x85"
			"\x3a\xfb\xab\x2b\x9e\xfa\x58\x20"
			"\x44\x1f\xc0\x14\x22\x75\x61\xe8"
			"\xaa\x19\xcf\xf1\x82\x56\xf4\xd7"
			"\x78\x7b\x3d\x5f\xb3\x9e\x0b\x8a"
			"\x57\x50\xdb\x17\x41\x65\x4d\xa3"
			"\x02\xc9\x9c\x9c\x53\xfb\x39\x39"
			"\x9b\x1d\x72\x24\xda\xb7\x39\xbe"
			"\x13\x3b\xfa\x29\xda\x9e\x54\x64"
			"\x6e\xba\xd8\xa1\xcb\xb3\x36\xfa"
			"\xcb\x47\x85\xe9\x61\x38\xbc\xbe"
			"\xc5\x00\x38\x2a\x54\xf7\xc4\xb9"
			"\xb3\xd3\x7b\xa0\xa0\xf8\x72\x7f"
			"\x8c\x8e\x82\x0e\xc6\x1c\x75\x9d"
			"\xca\x8e\x61\x87\xde\xad\x80\xd2"
			"\xf5\xf9\x80\xef\x15\x75\xaf\xf5"
			"\x80\xfb\xff\x6d\x1e\x25\xb7\x40"
			"\x61\x6a\x39\x5a\x6a\xb5\x31\xab"
			"\x97\x8a\x19\x89\x44\x40\xc0\xa6"
			"\xb4\x4e\x30\x32\x7b\x13\xe7\x67"
			"\xa9\x8b\x57\x04\xc2\x01\xa6\xf4"
			"\x28\x99\xad\x2c\x76\xa3\x78\xc2"
			"\x4a\xe6\xca\x5c\x50\x6a\xc1\xb0"
			"\x62\x4b\x10\x8e\x7c\x17\x43\xb3"
			"\x17\x66\x1c\x3e\x8d\x69\xf0\x5a"
			"\x71\xf5\x97\xdc\xd1\x45\xdd\x28"
			"\xf3\x5d\xdf\x53\x7b\x11\xe5\xbc"
			"\x4c\xdb\x1b\x51\x6b\xe9\xfb\x3d"
			"\xc1\xc3\x2c\xb9\x71\xf5\xb6\xb2"
			"\x13\x36\x79\x80\x53\xe8\xd3\xa6"
			"\x0a\xaf\xfd\x56\x97\xf7\x40\x8e"
			"\x45\xce\xf8\xb0\x9e\x5c\x33\x82"
			"\xb0\x44\x56\xfc\x05\x09\xe9\x2a"
			"\xac\x26\x80\x14\x1d\xc8\x3a\x35"
			"\x4c\x82\x97\xfd\x76\xb7\xa9\x0a"
			"\x35\x58\x79\x8e\x0f\x66\xea\xaf"
			"\x51\x6c\x09\xa9\x6e\x9b\xcb\x9a"
			"\x31\x47\xa0\x2f\x7c\x71\xb4\x4a"
			"\x11\xaa\x8c\x66\xc5\x64\xe6\x3a"
			"\x54\xda\x24\x6a\xc4\x41\x65\x46"
			"\x82\xa0\x0a\x0f\x5f\xfb\x25\xd0"
			"\x2c\x91\xa7\xee\xc4\x81\x07\x86"
			"\x75\x5e\x33\x69\x97\xe4\x2c\xa8"
			"\x9d\x9f\x0b\x6a\xbe\xad\x98\xda"
			"\x6d\x94\x41\xda\x2c\x1e\x89\xc4"
			"\xc2\xaf\x1e\x00\x05\x0b\x83\x60"
			"\xbd\x43\xea\x15\x23\x7f\xb9\xac"
			"\xee\x4f\x2c\xaf\x2a\xf3\xdf\xd0"
			"\xf3\x19\x31\xbb\x4a\x74\x84\x17"
			"\x52\x32\x2c\x7d\x61\xe4\xcb\xeb"
			"\x80\x38\x15\x52\xcb\x6f\xea\xe5"
			"\x73\x9c\xd9\x24\x69\xc6\x95\x32"
			"\x21\xc8\x11\xe4\xdc\x36\xd7\x93"
			"\x38\x66\xfb\xb2\x7f\x3a\xb9\xaf"
			"\x31\xdd\x93\x75\x78\x8a\x2c\x94"
			"\x87\x1a\x58\xec\x9e\x7d\x4d\xba"
			"\xe1\xe5\x4d\xfc\xbc\xa4\x2a\x14"
			"\xef\xcc\xa7\xec\xab\x43\x09\x18"
			"\xd3\xab\x68\xd1\x07\x99\x44\x47"
			"\xd6\x83\x85\x3b\x30\xea\xa9\x6b"
			"\x63\xea\xc4\x07\xfb\x43\x2f\xa4"
			"\xaa\xb0\xab\x03\x89\xce\x3f\x8c"
			"\x02\x7c\x86\x54\xbc\x88\xaf\x75"
			"\xd2\xdc\x63\x17\xd3\x26\xf6\x96"
			"\xa9\x3c\xf1\x61\x8c\x11\x18\xcc"
			"\xd6\xea\x5b\xe2\xcd\xf0\xf1\xb2"
			"\xe5\x35\x90\x1f\x85\x4c\x76\x5b"
			"\x66\xce\x44\xa4\x32\x9f\xe6\x7b"
			"\x71\x6e\x9f\x58\x15\x67\x72\x87"
			"\x64\x8e\x3a\x44\x45\xd4\x76\xfa"
			"\xc2\xf6\xef\x85\x05\x18\x7a\x9b"
			"\xba\x41\x54\xac\xf0\xfc\x59\x12"
			"\x3f\xdf\xa0\xe5\x8a\x65\xfd\x3a"
			"\x62\x8d\x83\x2c\x03\xbe\x05\x76"
			"\x2e\x53\x49\x97\x94\x33\xae\x40"
			"\x81\x15\xdb\x6e\xad\xaa\xf5\x4b"
			"\xe3\x98\x70\xdf\xe0\x7c\xcd\xdb"
			"\x02\xd4\x7d\x2f\xc1\xe6\xb4\xf3"
			"\xd7\x0d\x7a\xd9\x23\x9e\x87\x2d"
			"\xce\x87\xad\xcc\x72\x05\x00\x29"
			"\xdc\x73\x7f\x64\xc1\x15\x0e\xc2"
			"\xdf\xa7\x5f\xeb\x41\xa1\xcd\xef"
			"\x5c\x50\x79\x2a\x56\x56\x71\x8c"
			"\xac\xc0\x79\x50\x69\xca\x59\x32"
			"\x65\xf2\x54\xe4\x52\x38\x76\xd1"
			"\x5e\xde\x26\x9e\xfb\x75\x2e\x11"
			"\xb5\x10\xf4\x17\x73\xf5\x89\xc7"
			"\x4f\x43\x5c\x8e\x7c\xb9\x05\x52"
			"\x24\x40\x99\xfe\x9b\x85\x0b\x6c"
			"\x22\x3e\x8b\xae\x86\xa1\xd2\x79"
			"\x05\x68\x6b\xab\xe3\x41\x49\xed"
			"\x15\xa1\x8d\x40\x2d\x61\xdf\x1a"
			"\x59\xc9\x26\x8b\xef\x30\x4c\x88"
			"\x4b\x10\xf8\x8d\xa6\x92\x9f\x4b"
			"\xf3\xc4\x53\x0b\x89\x5d\x28\x92"
			"\xcf\x78\xb2\xc0\x5d\xed\x7e\xfc"
			"\xc0\x12\x23\x5f\x5a\x78\x86\x43"
			"\x6e\x27\xf7\x5a\xa7\x6a\xed\x19"
			"\x04\xf0\xb3\x12\xd1\xbd\x0e\x89"
			"\x6e\xbc\x96\xa8\xd8\x49\x39\x9f"
			"\x7e\x67\xf0\x2e\x3e\x01\xa9\xba"
			"\xec\x8b\x62\x8e\xcb\x4a\x70\x43"
			"\xc7\xc2\xc4\xca\x82\x03\x73\xe9"
			"\x11\xdf\xcf\x54\xea\xc9\xb0\x95"
			"\x51\xc0\x13\x3d\x92\x05\xfa\xf4"
			"\xa9\x34\xc8\xce\x6c\x3d\x54\xcc"
			"\xc4\xaf\xf1\xdc\x11\x44\x26\xa2"
			"\xaf\xf1\x85\x75\x7d\x03\x61\x68"
			"\x4e\x78\xc6\x92\x7d\x86\x7d\x77"
			"\xdc\x71\x72\xdb\xc6\xae\xa1\xcb"
			"\x70\x9a\x0b\x19\xbe\x4a\x6c\x2a"
			"\xe2\xba\x6c\x64\x9a\x13\x28\xdf"
			"\x85\x75\xe6\x43\xf6\x87\x08\x68"
			"\x6e\xba\x6e\x79\x9f\x04\xbc\x23"
			"\x50\xf6\x33\x5c\x1f\x24\x25\xbe"
			"\x33\x47\x80\x45\x56\xa3\xa7\xd7"
			"\x7a\xb1\x34\x0b\x90\x3c\x9c\xad"
			"\x44\x5f\x9e\x0e\x9d\xd4\xbd\x93"
			"\x5e\xfa\x3c\xe0\xb0\xd9\xed\xf3"
			"\xd6\x2e\xff\x24\xd8\x71\x6c\xed"
			"\xaf\x55\xeb\x22\xac\x93\x68\x32"
			"\x05\x5b\x47\xdd\xc6\x4a\xcb\xc7"
			"\x10\xe1\x3c\x92\x1a\xf3\x23\x78"
			"\x2b\xa1\xd2\x80\xf4\x12\xb1\x20"
			"\x8f\xff\x26\x35\xdd\xfb\xc7\x4e"
			"\x78\xf1\x2d\x50\x12\x77\xa8\x60"
			"\x7c\x0f\xf5\x16\x2f\x63\x70\x2a"
			"\xc0\x96\x80\x4e\x0a\xb4\x93\x35"
			"\x5d\x1d\x3f\x56\xf7\x2f\xbb\x90"
			"\x11\x16\x8f\xa2\xec\x47\xbe\xac"
			"\x56\x01\x26\x56\xb1\x8c\xb2\x10"
			"\xf9\x1a\xca\xf5\xd1\xb7\x39\x20"
			"\x63\xf1\x69\x20\x4f\x13\x12\x1f"
			"\x5b\x65\xfc\x98\xf7\xc4\x7a\xbe"
			"\xf7\x26\x4d\x2b\x84\x7b\x42\xad"
			"\xd8\x7a\x0a\xb4\xd8\x74\xbf\xc1"
			"\xf0\x6e\xb4\x29\xa3\xbb\xca\x46"
			"\x67\x70\x6a\x2d\xce\x0e\xa2\x8a"
			"\xa9\x87\xbf\x05\xc4\xc1\x04\xa3"
			"\xab\xd4\x45\x43\x8c\xb6\x02\xb0"
			"\x41\xc8\xfc\x44\x3d\x59\xaa\x2e"
			"\x44\x21\x2a\x8d\x88\x9d\x57\xf4"
			"\xa0\x02\x77\xb8\xa6\xa0\xe6\x75"
			"\x5c\x82\x65\x3e\x03\x5c\x29\x8f"
			"\x38\x55\xab\x33\x26\xef\x9f\x43"
			"\x52\xfd\x68\xaf\x36\xb4\xbb\x9a"
			"\x58\x09\x09\x1b\xc3\x65\x46\x46"
			"\x1d\xa7\x94\x18\x23\x50\x2c\xca"
			"\x2c\x55\x19\x97\x01\x9d\x93\x3b"
			"\x63\x86\xf2\x03\x67\x45\xd2\x72"
			"\x28\x52\x6c\xf4\xe3\x1c\xb5\x11"
			"\x13\xf1\xeb\x21\xc7\xd9\x56\x82"
			"\x2b\x82\x39\xbd\x69\x54\xed\x62"
			"\xc3\xe2\xde\x73\xd4\x6a\x12\xae"
			"\x13\x21\x7f\x4b\x5b\xfc\xbf\xe8"
			"\x2b\xbe\x56\xba\x68\x8b\x9a\xb1"
			"\x6e\xfa\xbf\x7e\x5a\x4b\xf1\xac"
			"\x98\x65\x85\xd1\x93\x53\xd3\x7b"
			"\x09\xdd\x4b\x10\x6d\x84\xb0\x13"
			"\x65\xbd\xcf\x52\x09\xc4\x85\xe2"
			"\x84\x74\x15\x65\xb7\xf7\x51\xaf"
			"\x55\xad\xa4\xd1\x22\x54\x70\x94"
			"\xa0\x1c\x90\x41\xfd\x99\xd7\x5a"
			"\x31\xef\xaa\x25\xd0\x7f\x4f\xea"
			"\x1d\x55\x42\xe5\x49\xb0\xd0\x46"
			"\x62\x36\x43\xb2\x82\x15\x75\x50"
			"\xa4\x72\xeb\x54\x27\x1f\x8a\xe4"
			"\x7d\xe9\x66\xc5\xf1\x53\xa4\xd1"
			"\x0c\xeb\xb8\xf8\xbc\xd4\xe2\xe7"
			"\xe1\xf8\x4b\xcb\xa9\xa1\xaf\x15"
			"\x83\xcb\x72\xd0\x33\x79\x00\x2d"
			"\x9f\xd7\xf1\x2e\x1e\x10\xe4\x45"
			"\xc0\x75\x3a\x39\xea\x68\xf7\x5d"
			"\x1b\x73\x8f\xe9\x8e\x0f\x72\x47"
			"\xae\x35\x0a\x31\x7a\x14\x4d\x4a"
			"\x6f\x47\xf7\x7e\x91\x6e\x74\x8b"
			"\x26\x47\xf9\xc3\xf9\xde\x70\xf5"
			"\x61\xab\xa9\x27\x9f\x82\xe4\x9c"
			"\x89\x91\x3f\x2e\x6a\xfd\xb5\x49"
			"\xe9\xfd\x59\x14\x36\x49\x40\x6d"
			"\x32\xd8\x85\x42\xf3\xa5\xdf\x0c"
			"\xa8\x27\xd7\x54\xe2\x63\x2f\xf2"
			"\x7e\x8b\x8b\xe7\xf1\x9a\x95\x35"
			"\x43\xdc\x3a\xe4\xb6\xf4\xd0\xdf"
			"\x9c\xcb\x94\xf3\x21\xa0\x77\x50"
			"\xe2\xc6\xc4\xc6\x5f\x09\x64\x5b"
			"\x92\x90\xd8\xe1\xd1\xed\x4b\x42"
			"\xd7\x37\xaf\x65\x3d\x11\x39\xb6"
			"\x24\x8a\x60\xae\xd6\x1e\xbf\x0e"
			"\x0d\xd7\xdc\x96\x0e\x65\x75\x4e"
			"\x29\x06\x9d\xa4\x51\x3a\x10\x63"
			"\x8f\x17\x07\xd5\x8e\x3c\xf4\x28"
			"\x00\x5a\x5b\x05\x19\xd8\xc0\x6c"
			"\xe5\x15\xe4\x9c\x9d\x71\x9d\x5e"
			"\x94\x29\x1a\xa7\x80\xfa\x0e\x33"
			"\x03\xdd\xb7\x3e\x9a\xa9\x26\x18"
			"\x37\xa9\x64\x08\x4d\x94\x5a\x88"
			"\xca\x35\xce\x81\x02\xe3\x1f\x1b"
			"\x89\x1a\x77\x85\xe3\x41\x6d\x32"
			"\x42\x19\x23\x7d\xc8\x73\xee\x25"
			"\x85\x0d\xf8\x31\x25\x79\x1b\x6f"
			"\x79\x25\xd2\xd8\xd4\x23\xfd\xf7"
			"\x82\x36\x6a\x0c\x46\x22\x15\xe9"
			"\xff\x72\x41\x91\x91\x7d\x3a\xb7"
			"\xdd\x65\x99\x70\xf6\x8d\x84\xf8"
			"\x67\x15\x20\x11\xd6\xb2\x55\x7b"
			"\xdb\x87\xee\xef\x55\x89\x2a\x59"
			"\x2b\x07\x8f\x43\x8a\x59\x3c\x01"
			"\x8b\x65\x54\xa1\x66\xd5\x38\xbd"
			"\xc6\x30\xa9\xcc\x49\xb6\xa8\x1b"
			"\xb8\xc0\x0e\xe3\x45\x28\xe2\xff"
			"\x41\x9f\x7e\x7c\xd1\xae\x9e\x25"
			"\x3f\x4c\x7c\x7c\xf4\xa8\x26\x4d"
			"\x5c\xfd\x4b\x27\x18\xf9\x61\x76"
			"\x48\xba\x0c\x6b\xa9\x4d\xfc\xf5"
			"\x3b\x35\x7e\x2f\x4a\xa9\xc2\x9a"
			"\xae\xab\x86\x09\x89\xc9\xc2\x40"
			"\x39\x2c\x81\xb3\xb8\x17\x67\xc2"
			"\x0d\x32\x4a\x3a\x67\x81\xd7\x1a"
			"\x34\x52\xc5\xdb\x0a\xf5\x63\x39"
			"\xea\x1f\xe1\x7c\xa1\x9e\xc1\x35"
			"\xe3\xb1\x18\x45\x67\xf9\x22\x38"
			"\x95\xd9\x34\x34\x86\xc6\x41\x94"
			"\x15\xf9\x5b\x41\xa6\x87\x8b\xf8"
			"\xd5\xe1\x1b\xe2\x5b\xf3\x86\x10"
			"\xff\xe6\xae\x69\x76\xbc\x0d\xb4"
			"\x09\x90\x0c\xa2\x65\x0c\xad\x74"
			"\xf5\xd7\xff\xda\xc1\xce\x85\xbe"
			"\x00\xa7\xff\x4d\x2f\x65\xd3\x8c"
			"\x86\x2d\x05\xe8\xed\x3e\x6b\x8b"
			"\x0f\x3d\x83\x8c\xf1\x1d\x5b\x96"
			"\x2e\xb1\x9c\xc2\x98\xe1\x70\xb9"
			"\xba\x5c\x8a\x43\xd6\x34\xa7\x2d"
			"\xc9\x92\xae\xf2\xa5\x7b\x05\x49"
			"\xa7\x33\x34\x86\xca\xe4\x96\x23"
			"\x76\x5b\xf2\xc6\xf1\x51\x28\x42"
			"\x7b\xcc\x76\x8f\xfa\xa2\xad\x31"
			"\xd4\xd6\x7a\x6d\x25\x25\x54\xe4"
			"\x3f\x50\x59\xe1\x5c\x05\xb7\x27"
			"\x48\xbf\x07\xec\x1b\x13\xbe\x2b"
			"\xa1\x57\x2b\xd5\xab\xd7\xd0\x4c"
			"\x1e\xcb\x71\x9b\xc5\x90\x85\xd3"
			"\xde\x59\xec\x71\xeb\x89\xbb\xd0"
			"\x09\x50\xe1\x16\x3f\xfd\x1c\x34"
			"\xc3\x1c\xa1\x10\x77\x53\x98\xef"
			"\xf2\xfd\xa5\x01\x59\xc2\x9b\x26"
			"\xc7\x42\xd9\x49\xda\x58\x2b\x6e"
			"\x9f\x53\x19\x76\x7e\xd9\xc9\x0e"
			"\x68\xc8\x7f\x51\x22\x42\xef\x49"
			"\xa4\x55\xb6\x36\xac\x09\xc7\x31"
			"\x88\x15\x4b\x2e\x8f\x3a\x08\xf7"
			"\xd8\xf7\xa8\xc5\xa9\x33\xa6\x45"
			"\xe4\xc4\x94\x76\xf3\x0d\x8f\x7e"
			"\xc8\xf6\xbc\x23\x0a\xb6\x4c\xd3"
			"\x6a\xcd\x36\xc2\x90\x5c\x5c\x3c"
			"\x65\x7b\xc2\xd6\xcc\xe6\x0d\x87"
			"\x73\x2e\x71\x79\x16\x06\x63\x28"
			"\x09\x15\xd8\x89\x38\x38\x3d\xb5"
			"\x42\x1c\x08\x24\xf7\x2a\xd2\x9d"
			"\xc8\xca\xef\xf9\x27\xd8\x07\x86"
			"\xf7\x43\x0b\x55\x15\x3f\x9f\x83"
			"\xef\xdc\x49\x9d\x2a\xc1\x54\x62"
			"\xbd\x9b\x66\x55\x9f\xb7\x12\xf3"
			"\x1b\x4d\x9d\x2a\x5c\xed\x87\x75"
			"\x87\x26\xec\x61\x2c\xb4\x0f\x89"
			"\xb0\xfb\x2e\x68\x5d\x15\xc7\x8d"
			"\x2e\xc0\xd9\xec\xaf\x4f\xd2\x25"
			"\x29\xe8\xd2\x26\x2b\x67\xe9\xfc"
			"\x2b\xa8\x67\x96\x12\x1f\x5b\x96"
			"\xc6\x14\x53\xaf\x44\xea\xd6\xe2"
			"\x94\x98\xe4\x12\x93\x4c\x92\xe0"
			"\x18\xa5\x8d\x2d\xe4\x71\x3c\x47"
			"\x4c\xf7\xe6\x47\x9e\xc0\x68\xdf"
			"\xd4\xf5\x5a\x74\xb1\x2b\x29\x03"
			"\x19\x07\xaf\x90\x62\x5c\x68\x98"
			"\x48\x16\x11\x02\x9d\xee\xb4\x9b"
			"\xe5\x42\x7f\x08\xfd\x16\x32\x0b"
			"\xd0\xb3\xfa\x2b\xb7\x99\xf9\x29"
			"\xcd\x20\x45\x9f\xb3\x1a\x5d\xa2"
			"\xaf\x4d\xe0\xbd\x42\x0d\xbc\x74"
			"\x99\x9c\x8e\x53\x1a\xb4\x3e\xbd"
			"\xa2\x9a\x2d\xf7\xf8\x39\x0f\x67"
			"\x63\xfc\x6b\xc0\xaf\xb3\x4b\x4f"
			"\x55\xc4\xcf\xa7\xc8\x04\x11\x3e"
			"\x14\x32\xbb\x1b\x38\x77\xd6\x7f"
			"\x54\x4c\xdf\x75\xf3\x07\x2d\x33"
			"\x9b\xa8\x20\xe1\x7b\x12\xb5\xf3"
			"\xef\x2f\xce\x72\xe5\x24\x60\xc1"
			"\x30\xe2\xab\xa1\x8e\x11\x09\xa8"
			"\x21\x33\x44\xfe\x7f\x35\x32\x93"
			"\x39\xa7\xad\x8b\x79\x06\xb2\xcb"
			"\x4e\xa9\x5f\xc7\xba\x74\x29\xec"
			"\x93\xa0\x4e\x54\x93\xc0\xbc\x55"
			"\x64\xf0\x48\xe5\x57\x99\xee\x75"
			"\xd6\x79\x0f\x66\xb7\xc6\x57\x76"
			"\xf7\xb7\xf3\x9c\xc5\x60\xe8\x7f"
			"\x83\x76\xd6\x0e\xaa\xe6\x90\x39"
			"\x1d\xa6\x32\x6a\x34\xe3\x55\xf8"
			"\x58\xa0\x58\x7d\x33\xe0\x22\x39"
			"\x44\x64\x87\x86\x5a\x2f\xa7\x7e"
			"\x0f\x38\xea\xb0\x30\xcc\x61\xa5"
			"\x6a\x32\xae\x1e\xf7\xe9\xd0\xa9"
			"\x0c\x32\x4b\xb5\x49\x28\xab\x85"
			"\x2f\x8e\x01\x36\x38\x52\xd0\xba"
			"\xd6\x02\x78\xf8\x0e\x3e\x9c\x8b"
			"\x6b\x45\x99\x3f\x5c\xfe\x58\xf1"
			"\x5c\x94\x04\xe1\xf5\x18\x6d\x51"
			"\xb2\x5d\x18\x20\xb6\xc2\x9a\x42"
			"\x1d\xb3\xab\x3c\xb6\x3a\x13\x03"
			"\xb2\x46\x82\x4f\xfc\x64\xbc\x4f"
			"\xca\xfa\x9c\xc0\xd5\xa7\xbd\x11"
			"\xb7\xe4\x5a\xf6\x6f\x4d\x4d\x54"
			"\xea\xa4\x98\x66\xd4\x22\x3b\xd3"
			"\x8f\x34\x47\xd9\x7c\xf4\x72\x3b"
			"\x4d\x02\x77\xf6\xd6\xdd\x08\x0a"
			"\x81\xe1\x86\x89\x3e\x56\x10\x3c"
			"\xba\xd7\x81\x8c\x08\xbc\x8b\xe2"
			"\x53\xec\xa7\x89\xee\xc8\x56\xb5"
			"\x36\x2c\xb2\x03\xba\x99\xdd\x7c"
			"\x48\xa0\xb0\xbc\x91\x33\xe9\xa8"
			"\xcb\xcd\xcf\x59\x5f\x1f\x15\xe2"
			"\x56\xf5\x4e\x01\x35\x27\x45\x77"
			"\x47\xc8\xbc\xcb\x7e\x39\xc1\x97"
			"\x28\xd3\x84\xfc\x2c\x3e\xc8\xad"
			"\x9c\xf8\x8a\x61\x9c\x28\xaa\xc5"
			"\x99\x20\x43\x85\x9d\xa5\xe2\x8b"
			"\xb8\xae\xeb\xd0\x32\x0d\x52\x78"
			"\x09\x56\x3f\xc7\xd8\x7e\x26\xfc"
			"\x37\xfb\x6f\x04\xfc\xfa\x92\x10"
			"\xac\xf8\x3e\x21\xdc\x8c\x21\x16"
			"\x7d\x67\x6e\xf6\xcd\xda\xb6\x98"
			"\x23\xab\x23\x3c\xb2\x10\xa0\x53"
			"\x5a\x56\x9f\xc5\xd0\xff\xbb\xe4"
			"\x98\x3c\x69\x1e\xdb\x38\x8f\x7e"
			"\x0f\xd2\x98\x88\x81\x8b\x45\x67"
			"\xea\x33\xf1\xeb\xe9\x97\x55\x2e"
			"\xd9\xaa\xeb\x5a\xec\xda\xe1\x68"
			"\xa8\x9d\x3c\x84\x7c\x05\x3d\x62"
			"\x87\x8f\x03\x21\x28\x95\x0c\x89"
			"\x25\x22\x4a\xb0\x93\xa9\x50\xa2"
			"\x2f\x57\x6e\x18\x42\x19\x54\x0c"
			"\x55\x67\xc6\x11\x49\xf4\x5c\xd2"
			"\xe9\x3d\xdd\x8b\x48\x71\x21\x00"
			"\xc3\x9a\x6c\x85\x74\x28\x83\x4a"
			"\x1b\x31\x05\xe1\x06\x92\xe7\xda"
			"\x85\x73\x78\x45\x20\x7f\xae\x13"
			"\x7c\x33\x06\x22\xf4\x83\xf9\x35"
			"\x3f\x6c\x71\xa8\x4e\x48\xbe\x9b"
			"\xce\x8a\xba\xda\xbe\x28\x08\xf7"
			"\xe2\x14\x8c\x71\xea\x72\xf9\x33"
			"\xf2\x88\x3f\xd7\xbb\x69\x6c\x29"
			"\x19\xdc\x84\xce\x1f\x12\x4f\xc8"
			"\xaf\xa5\x04\xba\x5a\xab\xb0\xd9"
			"\x14\x1f\x6c\x68\x98\x39\x89\x7a"
			"\xd9\xd8\x2f\xdf\xa8\x47\x4a\x25"
			"\xe2\xfb\x33\xf4\x59\x78\xe1\x68"
			"\x85\xcf\xfe\x59\x20\xd4\x05\x1d"
			"\x80\x99\xae\xbc\xca\xae\x0f\x2f"
			"\x65\x43\x34\x8e\x7e\xac\xd3\x93"
			"\x2f\xac\x6d\x14\x3d\x02\x07\x70"
			"\x9d\xa4\xf3\x1b\x5c\x36\xfc\x01"
			"\x73\x34\x85\x0c\x6c\xd6\xf1\xbd"
			"\x3f\xdf\xee\xf5\xd9\xba\x56\xef"
			"\xf4\x9b\x6b\xee\x9f\x5a\x78\x6d"
			"\x32\x19\xf4\xf7\xf8\x4c\x69\x0b"
			"\x4b\xbc\xbb\xb7\xf2\x85\xaf\x70"
			"\x75\x24\x6c\x54\xa7\x0e\x4d\x1d"
			"\x01\xbf\x08\xac\xcf\x7f\x2c\xe3"
			"\x14\x89\x5e\x70\x5a\x99\x92\xcd"
			"\x01\x84\xc8\xd2\xab\xe5\x4f\x58"
			"\xe7\x0f\x2f\x0e\xff\x68\xea\xfd"
			"\x15\xb3\x17\xe6\xb0\xe7\x85\xd8"
			"\x23\x2e\x05\xc7\xc9\xc4\x46\x1f"
			"\xe1\x9e\x49\x20\x23\x24\x4d\x7e"
			"\x29\x65\xff\xf4\xb6\xfd\x1a\x85"
			"\xc4\x16\xec\xfc\xea\x7b\xd6\x2c"
			"\x43\xf8\xb7\xbf\x79\xc0\x85\xcd"
			"\xef\xe1\x98\xd3\xa5\xf7\x90\x8c"
			"\xe9\x7f\x80\x6b\xd2\xac\x4c\x30"
			"\xa7\xc6\x61\x6c\xd2\xf9\x2c\xff"
			"\x30\xbc\x22\x81\x7d\x93\x12\xe4"
			"\x0a\xcd\xaf\xdd\xe8\xab\x0a\x1e"
			"\x13\xa4\x27\xc3\x5f\xf7\x4b\xbb"
			"\x37\x09\x4b\x91\x6f\x92\x4f\xaf"
			"\x52\xee\xdf\xef\x09\x6f\xf7\x5c"
			"\x6e\x12\x17\x72\x63\x57\xc7\xba"
			"\x3b\x6b\x38\x32\x73\x1b\x9c\x80"
			"\xc1\x7a\xc6\xcf\xcd\x35\xc0\x6b"
			"\x31\x1a\x6b\xe9\xd8\x2c\x29\x3f"
			"\x96\xfb\xb6\xcd\x13\x91\x3b\xc2"
			"\xd2\xa3\x31\x8d\xa4\xcd\x57\xcd"
			"\x13\x3d\x64\xfd\x06\xce\xe6\xdc"
			"\x0c\x24\x43\x31\x40\x57\xf1\x72"
			"\x17\xe3\x3a\x63\x6d\x35\xcf\x5d"
			"\x97\x40\x59\xdd\xf7\x3c\x02\xf7"
			"\x1c\x7e\x05\xbb\xa9\x0d\x01\xb1"
			"\x8e\xc0\x30\xa9\x53\x24\xc9\x89"
			"\x84\x6d\xaa\xd0\xcd\x91\xc2\x4d"
			"\x91\xb0\x89\xe2\xbf\x83\x44\xaa"
			"\x28\x72\x23\xa0\xc2\xad\xad\x1c"
			"\xfc\x3f\x09\x7a\x0b\xdc\xc5\x1b"
			"\x87\x13\xc6\x5b\x59\x8d\xf2\xc8"
			"\xaf\xdf\x11\x95",
		.rlen	= 4100,
		.np	= 2,
		.tap	= { 4064, 36 },
	},
};

#define CTS_MODE_ENC_TEST_VECTORS 6
#define CTS_MODE_DEC_TEST_VECTORS 6
static struct cipher_testvec cts_mode_enc_tv_template[] = {
	{  
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.ilen	= 17,
		.input  = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20",
		.rlen	= 17,
		.result	= "\xc6\x35\x35\x68\xf2\xbf\x8c\xb4"
			  "\xd8\xa5\x80\x36\x2d\xa7\xff\x7f"
			  "\x97",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.ilen   = 31,
		.input  = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20",
		.rlen   = 31,
		.result = "\xfc\x00\x78\x3e\x0e\xfd\xb2\xc1"
			  "\xd4\x45\xd4\xc8\xef\xf7\xed\x22"
			  "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.ilen   = 32,
		.input  = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43",
		.rlen   = 32,
		.result = "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5\xa8"
			  "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.ilen   = 47,
		.input  = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43"
			  "\x68\x69\x63\x6b\x65\x6e\x2c\x20"
			  "\x70\x6c\x65\x61\x73\x65\x2c",
		.rlen   = 47,
		.result = "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84"
			  "\xb3\xff\xfd\x94\x0c\x16\xa1\x8c"
			  "\x1b\x55\x49\xd2\xf8\x38\x02\x9e"
			  "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.ilen   = 48,
		.input  = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43"
			  "\x68\x69\x63\x6b\x65\x6e\x2c\x20"
			  "\x70\x6c\x65\x61\x73\x65\x2c\x20",
		.rlen   = 48,
		.result = "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84"
			  "\x9d\xad\x8b\xbb\x96\xc4\xcd\xc0"
			  "\x3b\xc1\x03\xe1\xa1\x94\xbb\xd8"
			  "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5\xa8",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.ilen   = 64,
		.input  = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43"
			  "\x68\x69\x63\x6b\x65\x6e\x2c\x20"
			  "\x70\x6c\x65\x61\x73\x65\x2c\x20"
			  "\x61\x6e\x64\x20\x77\x6f\x6e\x74"
			  "\x6f\x6e\x20\x73\x6f\x75\x70\x2e",
		.rlen   = 64,
		.result = "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84"
			  "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5\xa8"
			  "\x48\x07\xef\xe8\x36\xee\x89\xa5"
			  "\x26\x73\x0d\xbc\x2f\x7b\xc8\x40"
			  "\x9d\xad\x8b\xbb\x96\xc4\xcd\xc0"
			  "\x3b\xc1\x03\xe1\xa1\x94\xbb\xd8",
	}
};

static struct cipher_testvec cts_mode_dec_tv_template[] = {
	{  
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.rlen	= 17,
		.result = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20",
		.ilen	= 17,
		.input	= "\xc6\x35\x35\x68\xf2\xbf\x8c\xb4"
			  "\xd8\xa5\x80\x36\x2d\xa7\xff\x7f"
			  "\x97",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.rlen   = 31,
		.result = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20",
		.ilen   = 31,
		.input  = "\xfc\x00\x78\x3e\x0e\xfd\xb2\xc1"
			  "\xd4\x45\xd4\xc8\xef\xf7\xed\x22"
			  "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.rlen   = 32,
		.result = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43",
		.ilen   = 32,
		.input  = "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5\xa8"
			  "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.rlen   = 47,
		.result = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43"
			  "\x68\x69\x63\x6b\x65\x6e\x2c\x20"
			  "\x70\x6c\x65\x61\x73\x65\x2c",
		.ilen   = 47,
		.input  = "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84"
			  "\xb3\xff\xfd\x94\x0c\x16\xa1\x8c"
			  "\x1b\x55\x49\xd2\xf8\x38\x02\x9e"
			  "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.rlen   = 48,
		.result = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43"
			  "\x68\x69\x63\x6b\x65\x6e\x2c\x20"
			  "\x70\x6c\x65\x61\x73\x65\x2c\x20",
		.ilen   = 48,
		.input  = "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84"
			  "\x9d\xad\x8b\xbb\x96\xc4\xcd\xc0"
			  "\x3b\xc1\x03\xe1\xa1\x94\xbb\xd8"
			  "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5\xa8",
	}, {
		.klen	= 16,
		.key    = "\x63\x68\x69\x63\x6b\x65\x6e\x20"
			  "\x74\x65\x72\x69\x79\x61\x6b\x69",
		.rlen   = 64,
		.result = "\x49\x20\x77\x6f\x75\x6c\x64\x20"
			  "\x6c\x69\x6b\x65\x20\x74\x68\x65"
			  "\x20\x47\x65\x6e\x65\x72\x61\x6c"
			  "\x20\x47\x61\x75\x27\x73\x20\x43"
			  "\x68\x69\x63\x6b\x65\x6e\x2c\x20"
			  "\x70\x6c\x65\x61\x73\x65\x2c\x20"
			  "\x61\x6e\x64\x20\x77\x6f\x6e\x74"
			  "\x6f\x6e\x20\x73\x6f\x75\x70\x2e",
		.ilen   = 64,
		.input  = "\x97\x68\x72\x68\xd6\xec\xcc\xc0"
			  "\xc0\x7b\x25\xe2\x5e\xcf\xe5\x84"
			  "\x39\x31\x25\x23\xa7\x86\x62\xd5"
			  "\xbe\x7f\xcb\xcc\x98\xeb\xf5\xa8"
			  "\x48\x07\xef\xe8\x36\xee\x89\xa5"
			  "\x26\x73\x0d\xbc\x2f\x7b\xc8\x40"
			  "\x9d\xad\x8b\xbb\x96\xc4\xcd\xc0"
			  "\x3b\xc1\x03\xe1\xa1\x94\xbb\xd8",
	}
};

#define COMP_BUF_SIZE           512

struct comp_testvec {
	int inlen, outlen;
	char input[COMP_BUF_SIZE];
	char output[COMP_BUF_SIZE];
};

struct pcomp_testvec {
	void *params;
	unsigned int paramsize;
	int inlen, outlen;
	char input[COMP_BUF_SIZE];
	char output[COMP_BUF_SIZE];
};

#define DEFLATE_COMP_TEST_VECTORS 2
#define DEFLATE_DECOMP_TEST_VECTORS 2

static struct comp_testvec deflate_comp_tv_template[] = {
	{
		.inlen	= 70,
		.outlen	= 38,
		.input	= "Join us now and share the software "
			"Join us now and share the software ",
		.output	= "\xf3\xca\xcf\xcc\x53\x28\x2d\x56"
			  "\xc8\xcb\x2f\x57\x48\xcc\x4b\x51"
			  "\x28\xce\x48\x2c\x4a\x55\x28\xc9"
			  "\x48\x55\x28\xce\x4f\x2b\x29\x07"
			  "\x71\xbc\x08\x2b\x01\x00",
	}, {
		.inlen	= 191,
		.outlen	= 122,
		.input	= "This document describes a compression method based on the DEFLATE"
			"compression algorithm.  This document defines the application of "
			"the DEFLATE algorithm to the IP Payload Compression Protocol.",
		.output	= "\x5d\x8d\x31\x0e\xc2\x30\x10\x04"
			  "\xbf\xb2\x2f\xc8\x1f\x10\x04\x09"
			  "\x89\xc2\x85\x3f\x70\xb1\x2f\xf8"
			  "\x24\xdb\x67\xd9\x47\xc1\xef\x49"
			  "\x68\x12\x51\xae\x76\x67\xd6\x27"
			  "\x19\x88\x1a\xde\x85\xab\x21\xf2"
			  "\x08\x5d\x16\x1e\x20\x04\x2d\xad"
			  "\xf3\x18\xa2\x15\x85\x2d\x69\xc4"
			  "\x42\x83\x23\xb6\x6c\x89\x71\x9b"
			  "\xef\xcf\x8b\x9f\xcf\x33\xca\x2f"
			  "\xed\x62\xa9\x4c\x80\xff\x13\xaf"
			  "\x52\x37\xed\x0e\x52\x6b\x59\x02"
			  "\xd9\x4e\xe8\x7a\x76\x1d\x02\x98"
			  "\xfe\x8a\x87\x83\xa3\x4f\x56\x8a"
			  "\xb8\x9e\x8e\x5c\x57\xd3\xa0\x79"
			  "\xfa\x02",
	},
};

static struct comp_testvec deflate_decomp_tv_template[] = {
	{
		.inlen	= 122,
		.outlen	= 191,
		.input	= "\x5d\x8d\x31\x0e\xc2\x30\x10\x04"
			  "\xbf\xb2\x2f\xc8\x1f\x10\x04\x09"
			  "\x89\xc2\x85\x3f\x70\xb1\x2f\xf8"
			  "\x24\xdb\x67\xd9\x47\xc1\xef\x49"
			  "\x68\x12\x51\xae\x76\x67\xd6\x27"
			  "\x19\x88\x1a\xde\x85\xab\x21\xf2"
			  "\x08\x5d\x16\x1e\x20\x04\x2d\xad"
			  "\xf3\x18\xa2\x15\x85\x2d\x69\xc4"
			  "\x42\x83\x23\xb6\x6c\x89\x71\x9b"
			  "\xef\xcf\x8b\x9f\xcf\x33\xca\x2f"
			  "\xed\x62\xa9\x4c\x80\xff\x13\xaf"
			  "\x52\x37\xed\x0e\x52\x6b\x59\x02"
			  "\xd9\x4e\xe8\x7a\x76\x1d\x02\x98"
			  "\xfe\x8a\x87\x83\xa3\x4f\x56\x8a"
			  "\xb8\x9e\x8e\x5c\x57\xd3\xa0\x79"
			  "\xfa\x02",
		.output	= "This document describes a compression method based on the DEFLATE"
			"compression algorithm.  This document defines the application of "
			"the DEFLATE algorithm to the IP Payload Compression Protocol.",
	}, {
		.inlen	= 38,
		.outlen	= 70,
		.input	= "\xf3\xca\xcf\xcc\x53\x28\x2d\x56"
			  "\xc8\xcb\x2f\x57\x48\xcc\x4b\x51"
			  "\x28\xce\x48\x2c\x4a\x55\x28\xc9"
			  "\x48\x55\x28\xce\x4f\x2b\x29\x07"
			  "\x71\xbc\x08\x2b\x01\x00",
		.output	= "Join us now and share the software "
			"Join us now and share the software ",
	},
};

#define ZLIB_COMP_TEST_VECTORS 2
#define ZLIB_DECOMP_TEST_VECTORS 2

static const struct {
	struct nlattr nla;
	int val;
} deflate_comp_params[] = {
	{
		.nla = {
			.nla_len	= NLA_HDRLEN + sizeof(int),
			.nla_type	= ZLIB_COMP_LEVEL,
		},
		.val			= Z_DEFAULT_COMPRESSION,
	}, {
		.nla = {
			.nla_len	= NLA_HDRLEN + sizeof(int),
			.nla_type	= ZLIB_COMP_METHOD,
		},
		.val			= Z_DEFLATED,
	}, {
		.nla = {
			.nla_len	= NLA_HDRLEN + sizeof(int),
			.nla_type	= ZLIB_COMP_WINDOWBITS,
		},
		.val			= -11,
	}, {
		.nla = {
			.nla_len	= NLA_HDRLEN + sizeof(int),
			.nla_type	= ZLIB_COMP_MEMLEVEL,
		},
		.val			= MAX_MEM_LEVEL,
	}, {
		.nla = {
			.nla_len	= NLA_HDRLEN + sizeof(int),
			.nla_type	= ZLIB_COMP_STRATEGY,
		},
		.val			= Z_DEFAULT_STRATEGY,
	}
};

static const struct {
	struct nlattr nla;
	int val;
} deflate_decomp_params[] = {
	{
		.nla = {
			.nla_len	= NLA_HDRLEN + sizeof(int),
			.nla_type	= ZLIB_DECOMP_WINDOWBITS,
		},
		.val			= -11,
	}
};

static struct pcomp_testvec zlib_comp_tv_template[] = {
	{
		.params = &deflate_comp_params,
		.paramsize = sizeof(deflate_comp_params),
		.inlen	= 70,
		.outlen	= 38,
		.input	= "Join us now and share the software "
			"Join us now and share the software ",
		.output	= "\xf3\xca\xcf\xcc\x53\x28\x2d\x56"
			  "\xc8\xcb\x2f\x57\x48\xcc\x4b\x51"
			  "\x28\xce\x48\x2c\x4a\x55\x28\xc9"
			  "\x48\x55\x28\xce\x4f\x2b\x29\x07"
			  "\x71\xbc\x08\x2b\x01\x00",
	}, {
		.params = &deflate_comp_params,
		.paramsize = sizeof(deflate_comp_params),
		.inlen	= 191,
		.outlen	= 122,
		.input	= "This document describes a compression method based on the DEFLATE"
			"compression algorithm.  This document defines the application of "
			"the DEFLATE algorithm to the IP Payload Compression Protocol.",
		.output	= "\x5d\x8d\x31\x0e\xc2\x30\x10\x04"
			  "\xbf\xb2\x2f\xc8\x1f\x10\x04\x09"
			  "\x89\xc2\x85\x3f\x70\xb1\x2f\xf8"
			  "\x24\xdb\x67\xd9\x47\xc1\xef\x49"
			  "\x68\x12\x51\xae\x76\x67\xd6\x27"
			  "\x19\x88\x1a\xde\x85\xab\x21\xf2"
			  "\x08\x5d\x16\x1e\x20\x04\x2d\xad"
			  "\xf3\x18\xa2\x15\x85\x2d\x69\xc4"
			  "\x42\x83\x23\xb6\x6c\x89\x71\x9b"
			  "\xef\xcf\x8b\x9f\xcf\x33\xca\x2f"
			  "\xed\x62\xa9\x4c\x80\xff\x13\xaf"
			  "\x52\x37\xed\x0e\x52\x6b\x59\x02"
			  "\xd9\x4e\xe8\x7a\x76\x1d\x02\x98"
			  "\xfe\x8a\x87\x83\xa3\x4f\x56\x8a"
			  "\xb8\x9e\x8e\x5c\x57\xd3\xa0\x79"
			  "\xfa\x02",
	},
};

static struct pcomp_testvec zlib_decomp_tv_template[] = {
	{
		.params = &deflate_decomp_params,
		.paramsize = sizeof(deflate_decomp_params),
		.inlen	= 122,
		.outlen	= 191,
		.input	= "\x5d\x8d\x31\x0e\xc2\x30\x10\x04"
			  "\xbf\xb2\x2f\xc8\x1f\x10\x04\x09"
			  "\x89\xc2\x85\x3f\x70\xb1\x2f\xf8"
			  "\x24\xdb\x67\xd9\x47\xc1\xef\x49"
			  "\x68\x12\x51\xae\x76\x67\xd6\x27"
			  "\x19\x88\x1a\xde\x85\xab\x21\xf2"
			  "\x08\x5d\x16\x1e\x20\x04\x2d\xad"
			  "\xf3\x18\xa2\x15\x85\x2d\x69\xc4"
			  "\x42\x83\x23\xb6\x6c\x89\x71\x9b"
			  "\xef\xcf\x8b\x9f\xcf\x33\xca\x2f"
			  "\xed\x62\xa9\x4c\x80\xff\x13\xaf"
			  "\x52\x37\xed\x0e\x52\x6b\x59\x02"
			  "\xd9\x4e\xe8\x7a\x76\x1d\x02\x98"
			  "\xfe\x8a\x87\x83\xa3\x4f\x56\x8a"
			  "\xb8\x9e\x8e\x5c\x57\xd3\xa0\x79"
			  "\xfa\x02",
		.output	= "This document describes a compression method based on the DEFLATE"
			"compression algorithm.  This document defines the application of "
			"the DEFLATE algorithm to the IP Payload Compression Protocol.",
	}, {
		.params = &deflate_decomp_params,
		.paramsize = sizeof(deflate_decomp_params),
		.inlen	= 38,
		.outlen	= 70,
		.input	= "\xf3\xca\xcf\xcc\x53\x28\x2d\x56"
			  "\xc8\xcb\x2f\x57\x48\xcc\x4b\x51"
			  "\x28\xce\x48\x2c\x4a\x55\x28\xc9"
			  "\x48\x55\x28\xce\x4f\x2b\x29\x07"
			  "\x71\xbc\x08\x2b\x01\x00",
		.output	= "Join us now and share the software "
			"Join us now and share the software ",
	},
};

#define LZO_COMP_TEST_VECTORS 2
#define LZO_DECOMP_TEST_VECTORS 2

static struct comp_testvec lzo_comp_tv_template[] = {
	{
		.inlen	= 70,
		.outlen	= 57,
		.input	= "Join us now and share the software "
			"Join us now and share the software ",
		.output	= "\x00\x0d\x4a\x6f\x69\x6e\x20\x75"
			  "\x73\x20\x6e\x6f\x77\x20\x61\x6e"
			  "\x64\x20\x73\x68\x61\x72\x65\x20"
			  "\x74\x68\x65\x20\x73\x6f\x66\x74"
			  "\x77\x70\x01\x32\x88\x00\x0c\x65"
			  "\x20\x74\x68\x65\x20\x73\x6f\x66"
			  "\x74\x77\x61\x72\x65\x20\x11\x00"
			  "\x00",
	}, {
		.inlen	= 159,
		.outlen	= 131,
		.input	= "This document describes a compression method based on the LZO "
			"compression algorithm.  This document defines the application of "
			"the LZO algorithm used in UBIFS.",
		.output	= "\x00\x2c\x54\x68\x69\x73\x20\x64"
			  "\x6f\x63\x75\x6d\x65\x6e\x74\x20"
			  "\x64\x65\x73\x63\x72\x69\x62\x65"
			  "\x73\x20\x61\x20\x63\x6f\x6d\x70"
			  "\x72\x65\x73\x73\x69\x6f\x6e\x20"
			  "\x6d\x65\x74\x68\x6f\x64\x20\x62"
			  "\x61\x73\x65\x64\x20\x6f\x6e\x20"
			  "\x74\x68\x65\x20\x4c\x5a\x4f\x20"
			  "\x2a\x8c\x00\x09\x61\x6c\x67\x6f"
			  "\x72\x69\x74\x68\x6d\x2e\x20\x20"
			  "\x2e\x54\x01\x03\x66\x69\x6e\x65"
			  "\x73\x20\x74\x06\x05\x61\x70\x70"
			  "\x6c\x69\x63\x61\x74\x76\x0a\x6f"
			  "\x66\x88\x02\x60\x09\x27\xf0\x00"
			  "\x0c\x20\x75\x73\x65\x64\x20\x69"
			  "\x6e\x20\x55\x42\x49\x46\x53\x2e"
			  "\x11\x00\x00",
	},
};

static struct comp_testvec lzo_decomp_tv_template[] = {
	{
		.inlen	= 133,
		.outlen	= 159,
		.input	= "\x00\x2b\x54\x68\x69\x73\x20\x64"
			  "\x6f\x63\x75\x6d\x65\x6e\x74\x20"
			  "\x64\x65\x73\x63\x72\x69\x62\x65"
			  "\x73\x20\x61\x20\x63\x6f\x6d\x70"
			  "\x72\x65\x73\x73\x69\x6f\x6e\x20"
			  "\x6d\x65\x74\x68\x6f\x64\x20\x62"
			  "\x61\x73\x65\x64\x20\x6f\x6e\x20"
			  "\x74\x68\x65\x20\x4c\x5a\x4f\x2b"
			  "\x8c\x00\x0d\x61\x6c\x67\x6f\x72"
			  "\x69\x74\x68\x6d\x2e\x20\x20\x54"
			  "\x68\x69\x73\x2a\x54\x01\x02\x66"
			  "\x69\x6e\x65\x73\x94\x06\x05\x61"
			  "\x70\x70\x6c\x69\x63\x61\x74\x76"
			  "\x0a\x6f\x66\x88\x02\x60\x09\x27"
			  "\xf0\x00\x0c\x20\x75\x73\x65\x64"
			  "\x20\x69\x6e\x20\x55\x42\x49\x46"
			  "\x53\x2e\x11\x00\x00",
		.output	= "This document describes a compression method based on the LZO "
			"compression algorithm.  This document defines the application of "
			"the LZO algorithm used in UBIFS.",
	}, {
		.inlen	= 46,
		.outlen	= 70,
		.input	= "\x00\x0d\x4a\x6f\x69\x6e\x20\x75"
			  "\x73\x20\x6e\x6f\x77\x20\x61\x6e"
			  "\x64\x20\x73\x68\x61\x72\x65\x20"
			  "\x74\x68\x65\x20\x73\x6f\x66\x74"
			  "\x77\x70\x01\x01\x4a\x6f\x69\x6e"
			  "\x3d\x88\x00\x11\x00\x00",
		.output	= "Join us now and share the software "
			"Join us now and share the software ",
	},
};

#define MICHAEL_MIC_TEST_VECTORS 6

static struct hash_testvec michael_mic_tv_template[] = {
	{
		.key = "\x00\x00\x00\x00\x00\x00\x00\x00",
		.ksize = 8,
		.plaintext = zeroed_string,
		.psize = 0,
		.digest = "\x82\x92\x5c\x1c\xa1\xd1\x30\xb8",
	},
	{
		.key = "\x82\x92\x5c\x1c\xa1\xd1\x30\xb8",
		.ksize = 8,
		.plaintext = "M",
		.psize = 1,
		.digest = "\x43\x47\x21\xca\x40\x63\x9b\x3f",
	},
	{
		.key = "\x43\x47\x21\xca\x40\x63\x9b\x3f",
		.ksize = 8,
		.plaintext = "Mi",
		.psize = 2,
		.digest = "\xe8\xf9\xbe\xca\xe9\x7e\x5d\x29",
	},
	{
		.key = "\xe8\xf9\xbe\xca\xe9\x7e\x5d\x29",
		.ksize = 8,
		.plaintext = "Mic",
		.psize = 3,
		.digest = "\x90\x03\x8f\xc6\xcf\x13\xc1\xdb",
	},
	{
		.key = "\x90\x03\x8f\xc6\xcf\x13\xc1\xdb",
		.ksize = 8,
		.plaintext = "Mich",
		.psize = 4,
		.digest = "\xd5\x5e\x10\x05\x10\x12\x89\x86",
	},
	{
		.key = "\xd5\x5e\x10\x05\x10\x12\x89\x86",
		.ksize = 8,
		.plaintext = "Michael",
		.psize = 7,
		.digest = "\x0a\x94\x2b\x12\x4e\xca\xa5\x46",
	}
};

#define CRC32C_TEST_VECTORS 15

static struct hash_testvec crc32c_tv_template[] = {
	{
		.psize = 0,
		.digest = "\x00\x00\x00\x00",
	},
	{
		.key = "\x87\xa9\xcb\xed",
		.ksize = 4,
		.psize = 0,
		.digest = "\x78\x56\x34\x12",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x01\x02\x03\x04\x05\x06\x07\x08"
			     "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			     "\x11\x12\x13\x14\x15\x16\x17\x18"
			     "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			     "\x21\x22\x23\x24\x25\x26\x27\x28",
		.psize = 40,
		.digest = "\x7f\x15\x2c\x0e",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x29\x2a\x2b\x2c\x2d\x2e\x2f\x30"
			     "\x31\x32\x33\x34\x35\x36\x37\x38"
			     "\x39\x3a\x3b\x3c\x3d\x3e\x3f\x40"
			     "\x41\x42\x43\x44\x45\x46\x47\x48"
			     "\x49\x4a\x4b\x4c\x4d\x4e\x4f\x50",
		.psize = 40,
		.digest = "\xf6\xeb\x80\xe9",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x51\x52\x53\x54\x55\x56\x57\x58"
			     "\x59\x5a\x5b\x5c\x5d\x5e\x5f\x60"
			     "\x61\x62\x63\x64\x65\x66\x67\x68"
			     "\x69\x6a\x6b\x6c\x6d\x6e\x6f\x70"
			     "\x71\x72\x73\x74\x75\x76\x77\x78",
		.psize = 40,
		.digest = "\xed\xbd\x74\xde",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x79\x7a\x7b\x7c\x7d\x7e\x7f\x80"
			     "\x81\x82\x83\x84\x85\x86\x87\x88"
			     "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90"
			     "\x91\x92\x93\x94\x95\x96\x97\x98"
			     "\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0",
		.psize = 40,
		.digest = "\x62\xc8\x79\xd5",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8"
			     "\xa9\xaa\xab\xac\xad\xae\xaf\xb0"
			     "\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8"
			     "\xb9\xba\xbb\xbc\xbd\xbe\xbf\xc0"
			     "\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8",
		.psize = 40,
		.digest = "\xd0\x9a\x97\xba",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0"
			     "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			     "\xd9\xda\xdb\xdc\xdd\xde\xdf\xe0"
			     "\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8"
			     "\xe9\xea\xeb\xec\xed\xee\xef\xf0",
		.psize = 40,
		.digest = "\x13\xd9\x29\x2b",
	},
	{
		.key = "\x80\xea\xd3\xf1",
		.ksize = 4,
		.plaintext = "\x29\x2a\x2b\x2c\x2d\x2e\x2f\x30"
			     "\x31\x32\x33\x34\x35\x36\x37\x38"
			     "\x39\x3a\x3b\x3c\x3d\x3e\x3f\x40"
			     "\x41\x42\x43\x44\x45\x46\x47\x48"
			     "\x49\x4a\x4b\x4c\x4d\x4e\x4f\x50",
		.psize = 40,
		.digest = "\x0c\xb5\xe2\xa2",
	},
	{
		.key = "\xf3\x4a\x1d\x5d",
		.ksize = 4,
		.plaintext = "\x51\x52\x53\x54\x55\x56\x57\x58"
			     "\x59\x5a\x5b\x5c\x5d\x5e\x5f\x60"
			     "\x61\x62\x63\x64\x65\x66\x67\x68"
			     "\x69\x6a\x6b\x6c\x6d\x6e\x6f\x70"
			     "\x71\x72\x73\x74\x75\x76\x77\x78",
		.psize = 40,
		.digest = "\xd1\x7f\xfb\xa6",
	},
	{
		.key = "\x2e\x80\x04\x59",
		.ksize = 4,
		.plaintext = "\x79\x7a\x7b\x7c\x7d\x7e\x7f\x80"
			     "\x81\x82\x83\x84\x85\x86\x87\x88"
			     "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90"
			     "\x91\x92\x93\x94\x95\x96\x97\x98"
			     "\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0",
		.psize = 40,
		.digest = "\x59\x33\xe6\x7a",
	},
	{
		.key = "\xa6\xcc\x19\x85",
		.ksize = 4,
		.plaintext = "\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8"
			     "\xa9\xaa\xab\xac\xad\xae\xaf\xb0"
			     "\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8"
			     "\xb9\xba\xbb\xbc\xbd\xbe\xbf\xc0"
			     "\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8",
		.psize = 40,
		.digest = "\xbe\x03\x01\xd2",
	},
	{
		.key = "\x41\xfc\xfe\x2d",
		.ksize = 4,
		.plaintext = "\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0"
			     "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			     "\xd9\xda\xdb\xdc\xdd\xde\xdf\xe0"
			     "\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8"
			     "\xe9\xea\xeb\xec\xed\xee\xef\xf0",
		.psize = 40,
		.digest = "\x75\xd3\xc5\x24",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x01\x02\x03\x04\x05\x06\x07\x08"
			     "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			     "\x11\x12\x13\x14\x15\x16\x17\x18"
			     "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			     "\x21\x22\x23\x24\x25\x26\x27\x28"
			     "\x29\x2a\x2b\x2c\x2d\x2e\x2f\x30"
			     "\x31\x32\x33\x34\x35\x36\x37\x38"
			     "\x39\x3a\x3b\x3c\x3d\x3e\x3f\x40"
			     "\x41\x42\x43\x44\x45\x46\x47\x48"
			     "\x49\x4a\x4b\x4c\x4d\x4e\x4f\x50"
			     "\x51\x52\x53\x54\x55\x56\x57\x58"
			     "\x59\x5a\x5b\x5c\x5d\x5e\x5f\x60"
			     "\x61\x62\x63\x64\x65\x66\x67\x68"
			     "\x69\x6a\x6b\x6c\x6d\x6e\x6f\x70"
			     "\x71\x72\x73\x74\x75\x76\x77\x78"
			     "\x79\x7a\x7b\x7c\x7d\x7e\x7f\x80"
			     "\x81\x82\x83\x84\x85\x86\x87\x88"
			     "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90"
			     "\x91\x92\x93\x94\x95\x96\x97\x98"
			     "\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0"
			     "\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8"
			     "\xa9\xaa\xab\xac\xad\xae\xaf\xb0"
			     "\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8"
			     "\xb9\xba\xbb\xbc\xbd\xbe\xbf\xc0"
			     "\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8"
			     "\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0"
			     "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			     "\xd9\xda\xdb\xdc\xdd\xde\xdf\xe0"
			     "\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8"
			     "\xe9\xea\xeb\xec\xed\xee\xef\xf0",
		.psize = 240,
		.digest = "\x75\xd3\xc5\x24",
		.np = 2,
		.tap = { 31, 209 }
	}, {
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext =	"\x6e\x05\x79\x10\xa7\x1b\xb2\x49"
				"\xe0\x54\xeb\x82\x19\x8d\x24\xbb"
				"\x2f\xc6\x5d\xf4\x68\xff\x96\x0a"
				"\xa1\x38\xcf\x43\xda\x71\x08\x7c"
				"\x13\xaa\x1e\xb5\x4c\xe3\x57\xee"
				"\x85\x1c\x90\x27\xbe\x32\xc9\x60"
				"\xf7\x6b\x02\x99\x0d\xa4\x3b\xd2"
				"\x46\xdd\x74\x0b\x7f\x16\xad\x21"
				"\xb8\x4f\xe6\x5a\xf1\x88\x1f\x93"
				"\x2a\xc1\x35\xcc\x63\xfa\x6e\x05"
				"\x9c\x10\xa7\x3e\xd5\x49\xe0\x77"
				"\x0e\x82\x19\xb0\x24\xbb\x52\xe9"
				"\x5d\xf4\x8b\x22\x96\x2d\xc4\x38"
				"\xcf\x66\xfd\x71\x08\x9f\x13\xaa"
				"\x41\xd8\x4c\xe3\x7a\x11\x85\x1c"
				"\xb3\x27\xbe\x55\xec\x60\xf7\x8e"
				"\x02\x99\x30\xc7\x3b\xd2\x69\x00"
				"\x74\x0b\xa2\x16\xad\x44\xdb\x4f"
				"\xe6\x7d\x14\x88\x1f\xb6\x2a\xc1"
				"\x58\xef\x63\xfa\x91\x05\x9c\x33"
				"\xca\x3e\xd5\x6c\x03\x77\x0e\xa5"
				"\x19\xb0\x47\xde\x52\xe9\x80\x17"
				"\x8b\x22\xb9\x2d\xc4\x5b\xf2\x66"
				"\xfd\x94\x08\x9f\x36\xcd\x41\xd8"
				"\x6f\x06\x7a\x11\xa8\x1c\xb3\x4a"
				"\xe1\x55\xec\x83\x1a\x8e\x25\xbc"
				"\x30\xc7\x5e\xf5\x69\x00\x97\x0b"
				"\xa2\x39\xd0\x44\xdb\x72\x09\x7d"
				"\x14\xab\x1f\xb6\x4d\xe4\x58\xef"
				"\x86\x1d\x91\x28\xbf\x33\xca\x61"
				"\xf8\x6c\x03\x9a\x0e\xa5\x3c\xd3"
				"\x47\xde\x75\x0c\x80\x17\xae\x22"
				"\xb9\x50\xe7\x5b\xf2\x89\x20\x94"
				"\x2b\xc2\x36\xcd\x64\xfb\x6f\x06"
				"\x9d\x11\xa8\x3f\xd6\x4a\xe1\x78"
				"\x0f\x83\x1a\xb1\x25\xbc\x53\xea"
				"\x5e\xf5\x8c\x00\x97\x2e\xc5\x39"
				"\xd0\x67\xfe\x72\x09\xa0\x14\xab"
				"\x42\xd9\x4d\xe4\x7b\x12\x86\x1d"
				"\xb4\x28\xbf\x56\xed\x61\xf8\x8f"
				"\x03\x9a\x31\xc8\x3c\xd3\x6a\x01"
				"\x75\x0c\xa3\x17\xae\x45\xdc\x50"
				"\xe7\x7e\x15\x89\x20\xb7\x2b\xc2"
				"\x59\xf0\x64\xfb\x92\x06\x9d\x34"
				"\xcb\x3f\xd6\x6d\x04\x78\x0f\xa6"
				"\x1a\xb1\x48\xdf\x53\xea\x81\x18"
				"\x8c\x23\xba\x2e\xc5\x5c\xf3\x67"
				"\xfe\x95\x09\xa0\x37\xce\x42\xd9"
				"\x70\x07\x7b\x12\xa9\x1d\xb4\x4b"
				"\xe2\x56\xed\x84\x1b\x8f\x26\xbd"
				"\x31\xc8\x5f\xf6\x6a\x01\x98\x0c"
				"\xa3\x3a\xd1\x45\xdc\x73\x0a\x7e"
				"\x15\xac\x20\xb7\x4e\xe5\x59\xf0"
				"\x87\x1e\x92\x29\xc0\x34\xcb\x62"
				"\xf9\x6d\x04\x9b\x0f\xa6\x3d\xd4"
				"\x48\xdf\x76\x0d\x81\x18\xaf\x23"
				"\xba\x51\xe8\x5c\xf3\x8a\x21\x95"
				"\x2c\xc3\x37\xce\x65\xfc\x70\x07"
				"\x9e\x12\xa9\x40\xd7\x4b\xe2\x79"
				"\x10\x84\x1b\xb2\x26\xbd\x54\xeb"
				"\x5f\xf6\x8d\x01\x98\x2f\xc6\x3a"
				"\xd1\x68\xff\x73\x0a\xa1\x15\xac"
				"\x43\xda\x4e\xe5\x7c\x13\x87\x1e"
				"\xb5\x29\xc0\x57\xee\x62\xf9\x90"
				"\x04\x9b\x32\xc9\x3d\xd4\x6b\x02"
				"\x76\x0d\xa4\x18\xaf\x46\xdd\x51"
				"\xe8\x7f\x16\x8a\x21\xb8\x2c\xc3"
				"\x5a\xf1\x65\xfc\x93\x07\x9e\x35"
				"\xcc\x40\xd7\x6e\x05\x79\x10\xa7"
				"\x1b\xb2\x49\xe0\x54\xeb\x82\x19"
				"\x8d\x24\xbb\x2f\xc6\x5d\xf4\x68"
				"\xff\x96\x0a\xa1\x38\xcf\x43\xda"
				"\x71\x08\x7c\x13\xaa\x1e\xb5\x4c"
				"\xe3\x57\xee\x85\x1c\x90\x27\xbe"
				"\x32\xc9\x60\xf7\x6b\x02\x99\x0d"
				"\xa4\x3b\xd2\x46\xdd\x74\x0b\x7f"
				"\x16\xad\x21\xb8\x4f\xe6\x5a\xf1"
				"\x88\x1f\x93\x2a\xc1\x35\xcc\x63"
				"\xfa\x6e\x05\x9c\x10\xa7\x3e\xd5"
				"\x49\xe0\x77\x0e\x82\x19\xb0\x24"
				"\xbb\x52\xe9\x5d\xf4\x8b\x22\x96"
				"\x2d\xc4\x38\xcf\x66\xfd\x71\x08"
				"\x9f\x13\xaa\x41\xd8\x4c\xe3\x7a"
				"\x11\x85\x1c\xb3\x27\xbe\x55\xec"
				"\x60\xf7\x8e\x02\x99\x30\xc7\x3b"
				"\xd2\x69\x00\x74\x0b\xa2\x16\xad"
				"\x44\xdb\x4f\xe6\x7d\x14\x88\x1f"
				"\xb6\x2a\xc1\x58\xef\x63\xfa\x91"
				"\x05\x9c\x33\xca\x3e\xd5\x6c\x03"
				"\x77\x0e\xa5\x19\xb0\x47\xde\x52"
				"\xe9\x80\x17\x8b\x22\xb9\x2d\xc4"
				"\x5b\xf2\x66\xfd\x94\x08\x9f\x36"
				"\xcd\x41\xd8\x6f\x06\x7a\x11\xa8"
				"\x1c\xb3\x4a\xe1\x55\xec\x83\x1a"
				"\x8e\x25\xbc\x30\xc7\x5e\xf5\x69"
				"\x00\x97\x0b\xa2\x39\xd0\x44\xdb"
				"\x72\x09\x7d\x14\xab\x1f\xb6\x4d"
				"\xe4\x58\xef\x86\x1d\x91\x28\xbf"
				"\x33\xca\x61\xf8\x6c\x03\x9a\x0e"
				"\xa5\x3c\xd3\x47\xde\x75\x0c\x80"
				"\x17\xae\x22\xb9\x50\xe7\x5b\xf2"
				"\x89\x20\x94\x2b\xc2\x36\xcd\x64"
				"\xfb\x6f\x06\x9d\x11\xa8\x3f\xd6"
				"\x4a\xe1\x78\x0f\x83\x1a\xb1\x25"
				"\xbc\x53\xea\x5e\xf5\x8c\x00\x97"
				"\x2e\xc5\x39\xd0\x67\xfe\x72\x09"
				"\xa0\x14\xab\x42\xd9\x4d\xe4\x7b"
				"\x12\x86\x1d\xb4\x28\xbf\x56\xed"
				"\x61\xf8\x8f\x03\x9a\x31\xc8\x3c"
				"\xd3\x6a\x01\x75\x0c\xa3\x17\xae"
				"\x45\xdc\x50\xe7\x7e\x15\x89\x20"
				"\xb7\x2b\xc2\x59\xf0\x64\xfb\x92"
				"\x06\x9d\x34\xcb\x3f\xd6\x6d\x04"
				"\x78\x0f\xa6\x1a\xb1\x48\xdf\x53"
				"\xea\x81\x18\x8c\x23\xba\x2e\xc5"
				"\x5c\xf3\x67\xfe\x95\x09\xa0\x37"
				"\xce\x42\xd9\x70\x07\x7b\x12\xa9"
				"\x1d\xb4\x4b\xe2\x56\xed\x84\x1b"
				"\x8f\x26\xbd\x31\xc8\x5f\xf6\x6a"
				"\x01\x98\x0c\xa3\x3a\xd1\x45\xdc"
				"\x73\x0a\x7e\x15\xac\x20\xb7\x4e"
				"\xe5\x59\xf0\x87\x1e\x92\x29\xc0"
				"\x34\xcb\x62\xf9\x6d\x04\x9b\x0f"
				"\xa6\x3d\xd4\x48\xdf\x76\x0d\x81"
				"\x18\xaf\x23\xba\x51\xe8\x5c\xf3"
				"\x8a\x21\x95\x2c\xc3\x37\xce\x65"
				"\xfc\x70\x07\x9e\x12\xa9\x40\xd7"
				"\x4b\xe2\x79\x10\x84\x1b\xb2\x26"
				"\xbd\x54\xeb\x5f\xf6\x8d\x01\x98"
				"\x2f\xc6\x3a\xd1\x68\xff\x73\x0a"
				"\xa1\x15\xac\x43\xda\x4e\xe5\x7c"
				"\x13\x87\x1e\xb5\x29\xc0\x57\xee"
				"\x62\xf9\x90\x04\x9b\x32\xc9\x3d"
				"\xd4\x6b\x02\x76\x0d\xa4\x18\xaf"
				"\x46\xdd\x51\xe8\x7f\x16\x8a\x21"
				"\xb8\x2c\xc3\x5a\xf1\x65\xfc\x93"
				"\x07\x9e\x35\xcc\x40\xd7\x6e\x05"
				"\x79\x10\xa7\x1b\xb2\x49\xe0\x54"
				"\xeb\x82\x19\x8d\x24\xbb\x2f\xc6"
				"\x5d\xf4\x68\xff\x96\x0a\xa1\x38"
				"\xcf\x43\xda\x71\x08\x7c\x13\xaa"
				"\x1e\xb5\x4c\xe3\x57\xee\x85\x1c"
				"\x90\x27\xbe\x32\xc9\x60\xf7\x6b"
				"\x02\x99\x0d\xa4\x3b\xd2\x46\xdd"
				"\x74\x0b\x7f\x16\xad\x21\xb8\x4f"
				"\xe6\x5a\xf1\x88\x1f\x93\x2a\xc1"
				"\x35\xcc\x63\xfa\x6e\x05\x9c\x10"
				"\xa7\x3e\xd5\x49\xe0\x77\x0e\x82"
				"\x19\xb0\x24\xbb\x52\xe9\x5d\xf4"
				"\x8b\x22\x96\x2d\xc4\x38\xcf\x66"
				"\xfd\x71\x08\x9f\x13\xaa\x41\xd8"
				"\x4c\xe3\x7a\x11\x85\x1c\xb3\x27"
				"\xbe\x55\xec\x60\xf7\x8e\x02\x99"
				"\x30\xc7\x3b\xd2\x69\x00\x74\x0b"
				"\xa2\x16\xad\x44\xdb\x4f\xe6\x7d"
				"\x14\x88\x1f\xb6\x2a\xc1\x58\xef"
				"\x63\xfa\x91\x05\x9c\x33\xca\x3e"
				"\xd5\x6c\x03\x77\x0e\xa5\x19\xb0"
				"\x47\xde\x52\xe9\x80\x17\x8b\x22"
				"\xb9\x2d\xc4\x5b\xf2\x66\xfd\x94"
				"\x08\x9f\x36\xcd\x41\xd8\x6f\x06"
				"\x7a\x11\xa8\x1c\xb3\x4a\xe1\x55"
				"\xec\x83\x1a\x8e\x25\xbc\x30\xc7"
				"\x5e\xf5\x69\x00\x97\x0b\xa2\x39"
				"\xd0\x44\xdb\x72\x09\x7d\x14\xab"
				"\x1f\xb6\x4d\xe4\x58\xef\x86\x1d"
				"\x91\x28\xbf\x33\xca\x61\xf8\x6c"
				"\x03\x9a\x0e\xa5\x3c\xd3\x47\xde"
				"\x75\x0c\x80\x17\xae\x22\xb9\x50"
				"\xe7\x5b\xf2\x89\x20\x94\x2b\xc2"
				"\x36\xcd\x64\xfb\x6f\x06\x9d\x11"
				"\xa8\x3f\xd6\x4a\xe1\x78\x0f\x83"
				"\x1a\xb1\x25\xbc\x53\xea\x5e\xf5"
				"\x8c\x00\x97\x2e\xc5\x39\xd0\x67"
				"\xfe\x72\x09\xa0\x14\xab\x42\xd9"
				"\x4d\xe4\x7b\x12\x86\x1d\xb4\x28"
				"\xbf\x56\xed\x61\xf8\x8f\x03\x9a"
				"\x31\xc8\x3c\xd3\x6a\x01\x75\x0c"
				"\xa3\x17\xae\x45\xdc\x50\xe7\x7e"
				"\x15\x89\x20\xb7\x2b\xc2\x59\xf0"
				"\x64\xfb\x92\x06\x9d\x34\xcb\x3f"
				"\xd6\x6d\x04\x78\x0f\xa6\x1a\xb1"
				"\x48\xdf\x53\xea\x81\x18\x8c\x23"
				"\xba\x2e\xc5\x5c\xf3\x67\xfe\x95"
				"\x09\xa0\x37\xce\x42\xd9\x70\x07"
				"\x7b\x12\xa9\x1d\xb4\x4b\xe2\x56"
				"\xed\x84\x1b\x8f\x26\xbd\x31\xc8"
				"\x5f\xf6\x6a\x01\x98\x0c\xa3\x3a"
				"\xd1\x45\xdc\x73\x0a\x7e\x15\xac"
				"\x20\xb7\x4e\xe5\x59\xf0\x87\x1e"
				"\x92\x29\xc0\x34\xcb\x62\xf9\x6d"
				"\x04\x9b\x0f\xa6\x3d\xd4\x48\xdf"
				"\x76\x0d\x81\x18\xaf\x23\xba\x51"
				"\xe8\x5c\xf3\x8a\x21\x95\x2c\xc3"
				"\x37\xce\x65\xfc\x70\x07\x9e\x12"
				"\xa9\x40\xd7\x4b\xe2\x79\x10\x84"
				"\x1b\xb2\x26\xbd\x54\xeb\x5f\xf6"
				"\x8d\x01\x98\x2f\xc6\x3a\xd1\x68"
				"\xff\x73\x0a\xa1\x15\xac\x43\xda"
				"\x4e\xe5\x7c\x13\x87\x1e\xb5\x29"
				"\xc0\x57\xee\x62\xf9\x90\x04\x9b"
				"\x32\xc9\x3d\xd4\x6b\x02\x76\x0d"
				"\xa4\x18\xaf\x46\xdd\x51\xe8\x7f"
				"\x16\x8a\x21\xb8\x2c\xc3\x5a\xf1"
				"\x65\xfc\x93\x07\x9e\x35\xcc\x40"
				"\xd7\x6e\x05\x79\x10\xa7\x1b\xb2"
				"\x49\xe0\x54\xeb\x82\x19\x8d\x24"
				"\xbb\x2f\xc6\x5d\xf4\x68\xff\x96"
				"\x0a\xa1\x38\xcf\x43\xda\x71\x08"
				"\x7c\x13\xaa\x1e\xb5\x4c\xe3\x57"
				"\xee\x85\x1c\x90\x27\xbe\x32\xc9"
				"\x60\xf7\x6b\x02\x99\x0d\xa4\x3b"
				"\xd2\x46\xdd\x74\x0b\x7f\x16\xad"
				"\x21\xb8\x4f\xe6\x5a\xf1\x88\x1f"
				"\x93\x2a\xc1\x35\xcc\x63\xfa\x6e"
				"\x05\x9c\x10\xa7\x3e\xd5\x49\xe0"
				"\x77\x0e\x82\x19\xb0\x24\xbb\x52"
				"\xe9\x5d\xf4\x8b\x22\x96\x2d\xc4"
				"\x38\xcf\x66\xfd\x71\x08\x9f\x13"
				"\xaa\x41\xd8\x4c\xe3\x7a\x11\x85"
				"\x1c\xb3\x27\xbe\x55\xec\x60\xf7"
				"\x8e\x02\x99\x30\xc7\x3b\xd2\x69"
				"\x00\x74\x0b\xa2\x16\xad\x44\xdb"
				"\x4f\xe6\x7d\x14\x88\x1f\xb6\x2a"
				"\xc1\x58\xef\x63\xfa\x91\x05\x9c"
				"\x33\xca\x3e\xd5\x6c\x03\x77\x0e"
				"\xa5\x19\xb0\x47\xde\x52\xe9\x80"
				"\x17\x8b\x22\xb9\x2d\xc4\x5b\xf2"
				"\x66\xfd\x94\x08\x9f\x36\xcd\x41"
				"\xd8\x6f\x06\x7a\x11\xa8\x1c\xb3"
				"\x4a\xe1\x55\xec\x83\x1a\x8e\x25"
				"\xbc\x30\xc7\x5e\xf5\x69\x00\x97"
				"\x0b\xa2\x39\xd0\x44\xdb\x72\x09"
				"\x7d\x14\xab\x1f\xb6\x4d\xe4\x58"
				"\xef\x86\x1d\x91\x28\xbf\x33\xca"
				"\x61\xf8\x6c\x03\x9a\x0e\xa5\x3c"
				"\xd3\x47\xde\x75\x0c\x80\x17\xae"
				"\x22\xb9\x50\xe7\x5b\xf2\x89\x20"
				"\x94\x2b\xc2\x36\xcd\x64\xfb\x6f"
				"\x06\x9d\x11\xa8\x3f\xd6\x4a\xe1"
				"\x78\x0f\x83\x1a\xb1\x25\xbc\x53"
				"\xea\x5e\xf5\x8c\x00\x97\x2e\xc5"
				"\x39\xd0\x67\xfe\x72\x09\xa0\x14"
				"\xab\x42\xd9\x4d\xe4\x7b\x12\x86"
				"\x1d\xb4\x28\xbf\x56\xed\x61\xf8"
				"\x8f\x03\x9a\x31\xc8\x3c\xd3\x6a"
				"\x01\x75\x0c\xa3\x17\xae\x45\xdc"
				"\x50\xe7\x7e\x15\x89\x20\xb7\x2b"
				"\xc2\x59\xf0\x64\xfb\x92\x06\x9d"
				"\x34\xcb\x3f\xd6\x6d\x04\x78\x0f"
				"\xa6\x1a\xb1\x48\xdf\x53\xea\x81"
				"\x18\x8c\x23\xba\x2e\xc5\x5c\xf3"
				"\x67\xfe\x95\x09\xa0\x37\xce\x42"
				"\xd9\x70\x07\x7b\x12\xa9\x1d\xb4"
				"\x4b\xe2\x56\xed\x84\x1b\x8f\x26"
				"\xbd\x31\xc8\x5f\xf6\x6a\x01\x98",
		.psize = 2048,
		.digest = "\xec\x26\x4d\x95",
	}
};

#define BFIN_CRC_TEST_VECTORS 6

static struct hash_testvec bfin_crc_tv_template[] = {
	{
		.psize = 0,
		.digest = "\x00\x00\x00\x00",
	},
	{
		.key = "\x87\xa9\xcb\xed",
		.ksize = 4,
		.psize = 0,
		.digest = "\x87\xa9\xcb\xed",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x01\x02\x03\x04\x05\x06\x07\x08"
			     "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			     "\x11\x12\x13\x14\x15\x16\x17\x18"
			     "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			     "\x21\x22\x23\x24\x25\x26\x27\x28",
		.psize = 40,
		.digest = "\x84\x0c\x8d\xa2",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x01\x02\x03\x04\x05\x06\x07\x08"
			     "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			     "\x11\x12\x13\x14\x15\x16\x17\x18"
			     "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			     "\x21\x22\x23\x24\x25\x26",
		.psize = 38,
		.digest = "\x8c\x58\xec\xb7",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x01\x02\x03\x04\x05\x06\x07\x08"
			     "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			     "\x11\x12\x13\x14\x15\x16\x17\x18"
			     "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			     "\x21\x22\x23\x24\x25\x26\x27",
		.psize = 39,
		.digest = "\xdc\x50\x28\x7b",
	},
	{
		.key = "\xff\xff\xff\xff",
		.ksize = 4,
		.plaintext = "\x01\x02\x03\x04\x05\x06\x07\x08"
			     "\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10"
			     "\x11\x12\x13\x14\x15\x16\x17\x18"
			     "\x19\x1a\x1b\x1c\x1d\x1e\x1f\x20"
			     "\x21\x22\x23\x24\x25\x26\x27\x28"
			     "\x29\x2a\x2b\x2c\x2d\x2e\x2f\x30"
			     "\x31\x32\x33\x34\x35\x36\x37\x38"
			     "\x39\x3a\x3b\x3c\x3d\x3e\x3f\x40"
			     "\x41\x42\x43\x44\x45\x46\x47\x48"
			     "\x49\x4a\x4b\x4c\x4d\x4e\x4f\x50"
			     "\x51\x52\x53\x54\x55\x56\x57\x58"
			     "\x59\x5a\x5b\x5c\x5d\x5e\x5f\x60"
			     "\x61\x62\x63\x64\x65\x66\x67\x68"
			     "\x69\x6a\x6b\x6c\x6d\x6e\x6f\x70"
			     "\x71\x72\x73\x74\x75\x76\x77\x78"
			     "\x79\x7a\x7b\x7c\x7d\x7e\x7f\x80"
			     "\x81\x82\x83\x84\x85\x86\x87\x88"
			     "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90"
			     "\x91\x92\x93\x94\x95\x96\x97\x98"
			     "\x99\x9a\x9b\x9c\x9d\x9e\x9f\xa0"
			     "\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8"
			     "\xa9\xaa\xab\xac\xad\xae\xaf\xb0"
			     "\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8"
			     "\xb9\xba\xbb\xbc\xbd\xbe\xbf\xc0"
			     "\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8"
			     "\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0"
			     "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			     "\xd9\xda\xdb\xdc\xdd\xde\xdf\xe0"
			     "\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8"
			     "\xe9\xea\xeb\xec\xed\xee\xef\xf0",
		.psize = 240,
		.digest = "\x10\x19\x4a\x5c",
		.np = 2,
		.tap = { 31, 209 }
	},

};

#endif	 
