/*

  Broadcom B43 wireless driver
  IEEE 802.11n PHY and radio device data tables

  Copyright (c) 2008 Michael Buesch <mb@bu3sch.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/

#include "b43.h"
#include "tables_nphy.h"
#include "phy_common.h"
#include "phy_n.h"

struct b2055_inittab_entry {
	/* Value to write if we use the 5GHz band. */
	u16 ghz5;
	/* Value to write if we use the 2.4GHz band. */
	u16 ghz2;
	/* Flags */
	u8 flags;
#define B2055_INITTAB_ENTRY_OK	0x01
#define B2055_INITTAB_UPLOAD	0x02
};
#define UPLOAD		.flags = B2055_INITTAB_ENTRY_OK | B2055_INITTAB_UPLOAD
#define NOUPLOAD	.flags = B2055_INITTAB_ENTRY_OK

static const struct b2055_inittab_entry b2055_inittab [] = {
  [B2055_SP_PINPD]		= { .ghz5 = 0x0080, .ghz2 = 0x0080, NOUPLOAD, },
  [B2055_C1_SP_RSSI]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_SP_PDMISC]		= { .ghz5 = 0x0027, .ghz2 = 0x0027, NOUPLOAD, },
  [B2055_C2_SP_RSSI]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_SP_PDMISC]		= { .ghz5 = 0x0027, .ghz2 = 0x0027, NOUPLOAD, },
  [B2055_C1_SP_RXGC1]		= { .ghz5 = 0x007F, .ghz2 = 0x007F, UPLOAD, },
  [B2055_C1_SP_RXGC2]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, UPLOAD, },
  [B2055_C2_SP_RXGC1]		= { .ghz5 = 0x007F, .ghz2 = 0x007F, UPLOAD, },
  [B2055_C2_SP_RXGC2]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, UPLOAD, },
  [B2055_C1_SP_LPFBWSEL]	= { .ghz5 = 0x0015, .ghz2 = 0x0015, NOUPLOAD, },
  [B2055_C2_SP_LPFBWSEL]	= { .ghz5 = 0x0015, .ghz2 = 0x0015, NOUPLOAD, },
  [B2055_C1_SP_TXGC1]		= { .ghz5 = 0x004F, .ghz2 = 0x004F, UPLOAD, },
  [B2055_C1_SP_TXGC2]		= { .ghz5 = 0x0005, .ghz2 = 0x0005, UPLOAD, },
  [B2055_C2_SP_TXGC1]		= { .ghz5 = 0x004F, .ghz2 = 0x004F, UPLOAD, },
  [B2055_C2_SP_TXGC2]		= { .ghz5 = 0x0005, .ghz2 = 0x0005, UPLOAD, },
  [B2055_MASTER1]		= { .ghz5 = 0x00D0, .ghz2 = 0x00D0, NOUPLOAD, },
  [B2055_MASTER2]		= { .ghz5 = 0x0002, .ghz2 = 0x0002, NOUPLOAD, },
  [B2055_PD_LGEN]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_PD_PLLTS]		= { .ghz5 = 0x0040, .ghz2 = 0x0040, NOUPLOAD, },
  [B2055_C1_PD_LGBUF]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_PD_TX]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_PD_RXTX]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_PD_RSSIMISC]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_PD_LGBUF]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_PD_TX]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_PD_RXTX]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_PD_RSSIMISC]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_PWRDET_LGEN]		= { .ghz5 = 0x00C0, .ghz2 = 0x00C0, NOUPLOAD, },
  [B2055_C1_PWRDET_LGBUF]	= { .ghz5 = 0x00FF, .ghz2 = 0x00FF, NOUPLOAD, },
  [B2055_C1_PWRDET_RXTX]	= { .ghz5 = 0x00C0, .ghz2 = 0x00C0, NOUPLOAD, },
  [B2055_C2_PWRDET_LGBUF]	= { .ghz5 = 0x00FF, .ghz2 = 0x00FF, NOUPLOAD, },
  [B2055_C2_PWRDET_RXTX]	= { .ghz5 = 0x00C0, .ghz2 = 0x00C0, NOUPLOAD, },
  [B2055_RRCCAL_CS]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_RRCCAL_NOPTSEL]	= { .ghz5 = 0x002C, .ghz2 = 0x002C, NOUPLOAD, },
  [B2055_CAL_MISC]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_COUT]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_COUT2]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_CVARCTL]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_RVARCTL]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_LPOCTL]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_TS]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_RCCALRTS]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_CAL_RCALRTS]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_PADDRV]		= { .ghz5 = 0x00A4, .ghz2 = 0x00A4, NOUPLOAD, },
  [B2055_XOCTL1]		= { .ghz5 = 0x0038, .ghz2 = 0x0038, NOUPLOAD, },
  [B2055_XOCTL2]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_XOREGUL]		= { .ghz5 = 0x0004, .ghz2 = 0x0004, UPLOAD, },
  [B2055_XOMISC]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_PLL_LFC1]		= { .ghz5 = 0x000A, .ghz2 = 0x000A, NOUPLOAD, },
  [B2055_PLL_CALVTH]		= { .ghz5 = 0x0087, .ghz2 = 0x0087, NOUPLOAD, },
  [B2055_PLL_LFC2]		= { .ghz5 = 0x0009, .ghz2 = 0x0009, NOUPLOAD, },
  [B2055_PLL_REF]		= { .ghz5 = 0x0070, .ghz2 = 0x0070, NOUPLOAD, },
  [B2055_PLL_LFR1]		= { .ghz5 = 0x0011, .ghz2 = 0x0011, NOUPLOAD, },
  [B2055_PLL_PFDCP]		= { .ghz5 = 0x0018, .ghz2 = 0x0018, UPLOAD, },
  [B2055_PLL_IDAC_CPOPAMP]	= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_PLL_CPREG]		= { .ghz5 = 0x0004, .ghz2 = 0x0004, UPLOAD, },
  [B2055_PLL_RCAL]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_RF_PLLMOD0]		= { .ghz5 = 0x009E, .ghz2 = 0x009E, NOUPLOAD, },
  [B2055_RF_PLLMOD1]		= { .ghz5 = 0x0009, .ghz2 = 0x0009, NOUPLOAD, },
  [B2055_RF_MMDIDAC1]		= { .ghz5 = 0x00C8, .ghz2 = 0x00C8, UPLOAD, },
  [B2055_RF_MMDIDAC0]		= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_RF_MMDSP]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL1]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL2]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL3]		= { .ghz5 = 0x0001, .ghz2 = 0x0001, NOUPLOAD, },
  [B2055_VCO_CAL4]		= { .ghz5 = 0x0002, .ghz2 = 0x0002, NOUPLOAD, },
  [B2055_VCO_CAL5]		= { .ghz5 = 0x0096, .ghz2 = 0x0096, NOUPLOAD, },
  [B2055_VCO_CAL6]		= { .ghz5 = 0x003E, .ghz2 = 0x003E, NOUPLOAD, },
  [B2055_VCO_CAL7]		= { .ghz5 = 0x003E, .ghz2 = 0x003E, NOUPLOAD, },
  [B2055_VCO_CAL8]		= { .ghz5 = 0x0013, .ghz2 = 0x0013, NOUPLOAD, },
  [B2055_VCO_CAL9]		= { .ghz5 = 0x0002, .ghz2 = 0x0002, NOUPLOAD, },
  [B2055_VCO_CAL10]		= { .ghz5 = 0x0015, .ghz2 = 0x0015, NOUPLOAD, },
  [B2055_VCO_CAL11]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, NOUPLOAD, },
  [B2055_VCO_CAL12]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL13]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL14]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL15]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_CAL16]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_VCO_KVCO]		= { .ghz5 = 0x0008, .ghz2 = 0x0008, NOUPLOAD, },
  [B2055_VCO_CAPTAIL]		= { .ghz5 = 0x0008, .ghz2 = 0x0008, NOUPLOAD, },
  [B2055_VCO_IDACVCO]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_VCO_REG]		= { .ghz5 = 0x0084, .ghz2 = 0x0084, UPLOAD, },
  [B2055_PLL_RFVTH]		= { .ghz5 = 0x00C3, .ghz2 = 0x00C3, NOUPLOAD, },
  [B2055_LGBUF_CENBUF]		= { .ghz5 = 0x008F, .ghz2 = 0x008F, NOUPLOAD, },
  [B2055_LGEN_TUNE1]		= { .ghz5 = 0x00FF, .ghz2 = 0x00FF, NOUPLOAD, },
  [B2055_LGEN_TUNE2]		= { .ghz5 = 0x00FF, .ghz2 = 0x00FF, NOUPLOAD, },
  [B2055_LGEN_IDAC1]		= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_LGEN_IDAC2]		= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_LGEN_BIASC]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_LGEN_BIASIDAC]		= { .ghz5 = 0x00CC, .ghz2 = 0x00CC, NOUPLOAD, },
  [B2055_LGEN_RCAL]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_LGEN_DIV]		= { .ghz5 = 0x0080, .ghz2 = 0x0080, NOUPLOAD, },
  [B2055_LGEN_SPARE2]		= { .ghz5 = 0x0080, .ghz2 = 0x0080, NOUPLOAD, },
  [B2055_C1_LGBUF_ATUNE]	= { .ghz5 = 0x00F8, .ghz2 = 0x00F8, NOUPLOAD, },
  [B2055_C1_LGBUF_GTUNE]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C1_LGBUF_DIV]		= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C1_LGBUF_AIDAC]	= { .ghz5 = 0x0088, .ghz2 = 0x0008, UPLOAD, },
  [B2055_C1_LGBUF_GIDAC]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C1_LGBUF_IDACFO]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_LGBUF_SPARE]	= { .ghz5 = 0x0001, .ghz2 = 0x0001, UPLOAD, },
  [B2055_C1_RX_RFSPC1]		= { .ghz5 = 0x008A, .ghz2 = 0x008A, NOUPLOAD, },
  [B2055_C1_RX_RFR1]		= { .ghz5 = 0x0008, .ghz2 = 0x0008, NOUPLOAD, },
  [B2055_C1_RX_RFR2]		= { .ghz5 = 0x0083, .ghz2 = 0x0083, NOUPLOAD, },
  [B2055_C1_RX_RFRCAL]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_C1_RX_BB_BLCMP]	= { .ghz5 = 0x00A0, .ghz2 = 0x00A0, NOUPLOAD, },
  [B2055_C1_RX_BB_LPF]		= { .ghz5 = 0x000A, .ghz2 = 0x000A, NOUPLOAD, },
  [B2055_C1_RX_BB_MIDACHP]	= { .ghz5 = 0x0087, .ghz2 = 0x0087, UPLOAD, },
  [B2055_C1_RX_BB_VGA1IDAC]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C1_RX_BB_VGA2IDAC]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C1_RX_BB_VGA3IDAC]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C1_RX_BB_BUFOCTL]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C1_RX_BB_RCCALCTL]	= { .ghz5 = 0x0018, .ghz2 = 0x0018, NOUPLOAD, },
  [B2055_C1_RX_BB_RSSICTL1]	= { .ghz5 = 0x006A, .ghz2 = 0x006A, UPLOAD, },
  [B2055_C1_RX_BB_RSSICTL2]	= { .ghz5 = 0x00AB, .ghz2 = 0x00AB, UPLOAD, },
  [B2055_C1_RX_BB_RSSICTL3]	= { .ghz5 = 0x0013, .ghz2 = 0x0013, UPLOAD, },
  [B2055_C1_RX_BB_RSSICTL4]	= { .ghz5 = 0x00C1, .ghz2 = 0x00C1, UPLOAD, },
  [B2055_C1_RX_BB_RSSICTL5]	= { .ghz5 = 0x00AA, .ghz2 = 0x00AA, UPLOAD, },
  [B2055_C1_RX_BB_REG]		= { .ghz5 = 0x0087, .ghz2 = 0x0087, UPLOAD, },
  [B2055_C1_RX_BB_SPARE1]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_RX_TXBBRCAL]	= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_C1_TX_RF_SPGA]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, NOUPLOAD, },
  [B2055_C1_TX_RF_SPAD]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, NOUPLOAD, },
  [B2055_C1_TX_RF_CNTPGA1]	= { .ghz5 = 0x0015, .ghz2 = 0x0015, NOUPLOAD, },
  [B2055_C1_TX_RF_CNTPAD1]	= { .ghz5 = 0x0055, .ghz2 = 0x0055, NOUPLOAD, },
  [B2055_C1_TX_RF_PGAIDAC]	= { .ghz5 = 0x0097, .ghz2 = 0x0097, UPLOAD, },
  [B2055_C1_TX_PGAPADTN]	= { .ghz5 = 0x0008, .ghz2 = 0x0008, NOUPLOAD, },
  [B2055_C1_TX_PADIDAC1]	= { .ghz5 = 0x0014, .ghz2 = 0x0014, UPLOAD, },
  [B2055_C1_TX_PADIDAC2]	= { .ghz5 = 0x0033, .ghz2 = 0x0033, NOUPLOAD, },
  [B2055_C1_TX_MXBGTRIM]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C1_TX_RF_RCAL]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_C1_TX_RF_PADTSSI1]	= { .ghz5 = 0x0003, .ghz2 = 0x0003, UPLOAD, },
  [B2055_C1_TX_RF_PADTSSI2]	= { .ghz5 = 0x000A, .ghz2 = 0x000A, NOUPLOAD, },
  [B2055_C1_TX_RF_SPARE]	= { .ghz5 = 0x0003, .ghz2 = 0x0003, UPLOAD, },
  [B2055_C1_TX_RF_IQCAL1]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C1_TX_RF_IQCAL2]	= { .ghz5 = 0x00A4, .ghz2 = 0x00A4, NOUPLOAD, },
  [B2055_C1_TXBB_RCCAL]		= { .ghz5 = 0x0018, .ghz2 = 0x0018, NOUPLOAD, },
  [B2055_C1_TXBB_LPF1]		= { .ghz5 = 0x0028, .ghz2 = 0x0028, NOUPLOAD, },
  [B2055_C1_TX_VOSCNCL]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_TX_LPF_MXGMIDAC]	= { .ghz5 = 0x004A, .ghz2 = 0x004A, NOUPLOAD, },
  [B2055_C1_TX_BB_MXGM]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_LGBUF_ATUNE]	= { .ghz5 = 0x00F8, .ghz2 = 0x00F8, NOUPLOAD, },
  [B2055_C2_LGBUF_GTUNE]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C2_LGBUF_DIV]		= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C2_LGBUF_AIDAC]	= { .ghz5 = 0x0088, .ghz2 = 0x0008, UPLOAD, },
  [B2055_C2_LGBUF_GIDAC]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C2_LGBUF_IDACFO]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_LGBUF_SPARE]	= { .ghz5 = 0x0001, .ghz2 = 0x0001, UPLOAD, },
  [B2055_C2_RX_RFSPC1]		= { .ghz5 = 0x008A, .ghz2 = 0x008A, NOUPLOAD, },
  [B2055_C2_RX_RFR1]		= { .ghz5 = 0x0008, .ghz2 = 0x0008, NOUPLOAD, },
  [B2055_C2_RX_RFR2]		= { .ghz5 = 0x0083, .ghz2 = 0x0083, NOUPLOAD, },
  [B2055_C2_RX_RFRCAL]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_C2_RX_BB_BLCMP]	= { .ghz5 = 0x00A0, .ghz2 = 0x00A0, NOUPLOAD, },
  [B2055_C2_RX_BB_LPF]		= { .ghz5 = 0x000A, .ghz2 = 0x000A, NOUPLOAD, },
  [B2055_C2_RX_BB_MIDACHP]	= { .ghz5 = 0x0087, .ghz2 = 0x0087, UPLOAD, },
  [B2055_C2_RX_BB_VGA1IDAC]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C2_RX_BB_VGA2IDAC]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C2_RX_BB_VGA3IDAC]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C2_RX_BB_BUFOCTL]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C2_RX_BB_RCCALCTL]	= { .ghz5 = 0x0018, .ghz2 = 0x0018, NOUPLOAD, },
  [B2055_C2_RX_BB_RSSICTL1]	= { .ghz5 = 0x006A, .ghz2 = 0x006A, UPLOAD, },
  [B2055_C2_RX_BB_RSSICTL2]	= { .ghz5 = 0x00AB, .ghz2 = 0x00AB, UPLOAD, },
  [B2055_C2_RX_BB_RSSICTL3]	= { .ghz5 = 0x0013, .ghz2 = 0x0013, UPLOAD, },
  [B2055_C2_RX_BB_RSSICTL4]	= { .ghz5 = 0x00C1, .ghz2 = 0x00C1, UPLOAD, },
  [B2055_C2_RX_BB_RSSICTL5]	= { .ghz5 = 0x00AA, .ghz2 = 0x00AA, UPLOAD, },
  [B2055_C2_RX_BB_REG]		= { .ghz5 = 0x0087, .ghz2 = 0x0087, UPLOAD, },
  [B2055_C2_RX_BB_SPARE1]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_RX_TXBBRCAL]	= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_C2_TX_RF_SPGA]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, NOUPLOAD, },
  [B2055_C2_TX_RF_SPAD]		= { .ghz5 = 0x0007, .ghz2 = 0x0007, NOUPLOAD, },
  [B2055_C2_TX_RF_CNTPGA1]	= { .ghz5 = 0x0015, .ghz2 = 0x0015, NOUPLOAD, },
  [B2055_C2_TX_RF_CNTPAD1]	= { .ghz5 = 0x0055, .ghz2 = 0x0055, NOUPLOAD, },
  [B2055_C2_TX_RF_PGAIDAC]	= { .ghz5 = 0x0097, .ghz2 = 0x0097, UPLOAD, },
  [B2055_C2_TX_PGAPADTN]	= { .ghz5 = 0x0008, .ghz2 = 0x0008, NOUPLOAD, },
  [B2055_C2_TX_PADIDAC1]	= { .ghz5 = 0x0014, .ghz2 = 0x0014, UPLOAD, },
  [B2055_C2_TX_PADIDAC2]	= { .ghz5 = 0x0033, .ghz2 = 0x0033, NOUPLOAD, },
  [B2055_C2_TX_MXBGTRIM]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [B2055_C2_TX_RF_RCAL]		= { .ghz5 = 0x0006, .ghz2 = 0x0006, NOUPLOAD, },
  [B2055_C2_TX_RF_PADTSSI1]	= { .ghz5 = 0x0003, .ghz2 = 0x0003, UPLOAD, },
  [B2055_C2_TX_RF_PADTSSI2]	= { .ghz5 = 0x000A, .ghz2 = 0x000A, NOUPLOAD, },
  [B2055_C2_TX_RF_SPARE]	= { .ghz5 = 0x0003, .ghz2 = 0x0003, UPLOAD, },
  [B2055_C2_TX_RF_IQCAL1]	= { .ghz5 = 0x002A, .ghz2 = 0x002A, NOUPLOAD, },
  [B2055_C2_TX_RF_IQCAL2]	= { .ghz5 = 0x00A4, .ghz2 = 0x00A4, NOUPLOAD, },
  [B2055_C2_TXBB_RCCAL]		= { .ghz5 = 0x0018, .ghz2 = 0x0018, NOUPLOAD, },
  [B2055_C2_TXBB_LPF1]		= { .ghz5 = 0x0028, .ghz2 = 0x0028, NOUPLOAD, },
  [B2055_C2_TX_VOSCNCL]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_TX_LPF_MXGMIDAC]	= { .ghz5 = 0x004A, .ghz2 = 0x004A, NOUPLOAD, },
  [B2055_C2_TX_BB_MXGM]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_PRG_GCHP21]		= { .ghz5 = 0x0071, .ghz2 = 0x0071, NOUPLOAD, },
  [B2055_PRG_GCHP22]		= { .ghz5 = 0x0072, .ghz2 = 0x0072, NOUPLOAD, },
  [B2055_PRG_GCHP23]		= { .ghz5 = 0x0073, .ghz2 = 0x0073, NOUPLOAD, },
  [B2055_PRG_GCHP24]		= { .ghz5 = 0x0074, .ghz2 = 0x0074, NOUPLOAD, },
  [B2055_PRG_GCHP25]		= { .ghz5 = 0x0075, .ghz2 = 0x0075, NOUPLOAD, },
  [B2055_PRG_GCHP26]		= { .ghz5 = 0x0076, .ghz2 = 0x0076, NOUPLOAD, },
  [B2055_PRG_GCHP27]		= { .ghz5 = 0x0077, .ghz2 = 0x0077, NOUPLOAD, },
  [B2055_PRG_GCHP28]		= { .ghz5 = 0x0078, .ghz2 = 0x0078, NOUPLOAD, },
  [B2055_PRG_GCHP29]		= { .ghz5 = 0x0079, .ghz2 = 0x0079, NOUPLOAD, },
  [B2055_PRG_GCHP30]		= { .ghz5 = 0x007A, .ghz2 = 0x007A, NOUPLOAD, },
  [0xC7]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xC8]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xC9]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xCA]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xCB]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xCC]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_LNA_GAINBST]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xCE]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xCF]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xD0]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xD1]			= { .ghz5 = 0x0018, .ghz2 = 0x0018, NOUPLOAD, },
  [B2055_C1_B0NB_RSSIVCM]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [0xD3]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xD4]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xD5]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C1_GENSPARE2]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xD7]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xD8]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_LNA_GAINBST]	= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xDA]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xDB]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xDC]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xDD]			= { .ghz5 = 0x0018, .ghz2 = 0x0018, NOUPLOAD, },
  [B2055_C2_B0NB_RSSIVCM]	= { .ghz5 = 0x0088, .ghz2 = 0x0088, NOUPLOAD, },
  [0xDF]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xE0]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [0xE1]			= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
  [B2055_C2_GENSPARE2]		= { .ghz5 = 0x0000, .ghz2 = 0x0000, NOUPLOAD, },
};

void b2055_upload_inittab(struct b43_wldev *dev,
			  bool ghz5, bool ignore_uploadflag)
{
	const struct b2055_inittab_entry *e;
	unsigned int i;
	u16 value;

	for (i = 0; i < ARRAY_SIZE(b2055_inittab); i++) {
		e = &(b2055_inittab[i]);
		if (!(e->flags & B2055_INITTAB_ENTRY_OK))
			continue;
		if ((e->flags & B2055_INITTAB_UPLOAD) || ignore_uploadflag) {
			if (ghz5)
				value = e->ghz5;
			else
				value = e->ghz2;
			b43_radio_write16(dev, i, value);
		}
	}
}

#define RADIOREGS(r0, r1, r2, r3, r4, r5, r6, r7, r8, r9, r10, r11, \
		  r12, r13, r14, r15, r16, r17, r18, r19, r20, r21) \
	.radio_pll_ref		= r0,	\
	.radio_rf_pllmod0	= r1,	\
	.radio_rf_pllmod1	= r2,	\
	.radio_vco_captail	= r3,	\
	.radio_vco_cal1		= r4,	\
	.radio_vco_cal2		= r5,	\
	.radio_pll_lfc1		= r6,	\
	.radio_pll_lfr1		= r7,	\
	.radio_pll_lfc2		= r8,	\
	.radio_lgbuf_cenbuf	= r9,	\
	.radio_lgen_tune1	= r10,	\
	.radio_lgen_tune2	= r11,	\
	.radio_c1_lgbuf_atune	= r12,	\
	.radio_c1_lgbuf_gtune	= r13,	\
	.radio_c1_rx_rfr1	= r14,	\
	.radio_c1_tx_pgapadtn	= r15,	\
	.radio_c1_tx_mxbgtrim	= r16,	\
	.radio_c2_lgbuf_atune	= r17,	\
	.radio_c2_lgbuf_gtune	= r18,	\
	.radio_c2_rx_rfr1	= r19,	\
	.radio_c2_tx_pgapadtn	= r20,	\
	.radio_c2_tx_mxbgtrim	= r21

#define PHYREGS(r0, r1, r2, r3, r4, r5)	\
	.phy_bw1a	= r0,		\
	.phy_bw2	= r1,		\
	.phy_bw3	= r2,		\
	.phy_bw4	= r3,		\
	.phy_bw5	= r4,		\
	.phy_bw6	= r5

static const struct b43_nphy_channeltab_entry b43_nphy_channeltab[] = {
  {	.channel		= 184,
	.freq			= 4920, /* MHz */
	.unk2			= 3280,
	RADIOREGS(0x71, 0x01, 0xEC, 0x0F, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xFF, 0xFF, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xB407, 0xB007, 0xAC07, 0x1402, 0x1502, 0x1602),
  },
  {	.channel		= 186,
	.freq			= 4930, /* MHz */
	.unk2			= 3287,
	RADIOREGS(0x71, 0x01, 0xED, 0x0F, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xFF, 0xFF, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xB807, 0xB407, 0xB007, 0x1302, 0x1402, 0x1502),
  },
  {	.channel		= 188,
	.freq			= 4940, /* MHz */
	.unk2			= 3293,
	RADIOREGS(0x71, 0x01, 0xEE, 0x0F, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xEE, 0xEE, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xBC07, 0xB807, 0xB407, 0x1202, 0x1302, 0x1402),
  },
  {	.channel		= 190,
	.freq			= 4950, /* MHz */
	.unk2			= 3300,
	RADIOREGS(0x71, 0x01, 0xEF, 0x0F, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xEE, 0xEE, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xC007, 0xBC07, 0xB807, 0x1102, 0x1202, 0x1302),
  },
  {	.channel		= 192,
	.freq			= 4960, /* MHz */
	.unk2			= 3307,
	RADIOREGS(0x71, 0x01, 0xF0, 0x0F, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xEE, 0xEE, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xC407, 0xC007, 0xBC07, 0x0F02, 0x1102, 0x1202),
  },
  {	.channel		= 194,
	.freq			= 4970, /* MHz */
	.unk2			= 3313,
	RADIOREGS(0x71, 0x01, 0xF1, 0x0F, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xEE, 0xEE, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xC807, 0xC407, 0xC007, 0x0E02, 0x0F02, 0x1102),
  },
  {	.channel		= 196,
	.freq			= 4980, /* MHz */
	.unk2			= 3320,
	RADIOREGS(0x71, 0x01, 0xF2, 0x0E, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xDD, 0xDD, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xCC07, 0xC807, 0xC407, 0x0D02, 0x0E02, 0x0F02),
  },
  {	.channel		= 198,
	.freq			= 4990, /* MHz */
	.unk2			= 3327,
	RADIOREGS(0x71, 0x01, 0xF3, 0x0E, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xDD, 0xDD, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xD007, 0xCC07, 0xC807, 0x0C02, 0x0D02, 0x0E02),
  },
  {	.channel		= 200,
	.freq			= 5000, /* MHz */
	.unk2			= 3333,
	RADIOREGS(0x71, 0x01, 0xF4, 0x0E, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xDD, 0xDD, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xD407, 0xD007, 0xCC07, 0x0B02, 0x0C02, 0x0D02),
  },
  {	.channel		= 202,
	.freq			= 5010, /* MHz */
	.unk2			= 3340,
	RADIOREGS(0x71, 0x01, 0xF5, 0x0E, 0xFF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xDD, 0xDD, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xD807, 0xD407, 0xD007, 0x0A02, 0x0B02, 0x0C02),
  },
  {	.channel		= 204,
	.freq			= 5020, /* MHz */
	.unk2			= 3347,
	RADIOREGS(0x71, 0x01, 0xF6, 0x0E, 0xF7, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xCC, 0xCC, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xDC07, 0xD807, 0xD407, 0x0902, 0x0A02, 0x0B02),
  },
  {	.channel		= 206,
	.freq			= 5030, /* MHz */
	.unk2			= 3353,
	RADIOREGS(0x71, 0x01, 0xF7, 0x0E, 0xF7, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xCC, 0xCC, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xE007, 0xDC07, 0xD807, 0x0802, 0x0902, 0x0A02),
  },
  {	.channel		= 208,
	.freq			= 5040, /* MHz */
	.unk2			= 3360,
	RADIOREGS(0x71, 0x01, 0xF8, 0x0D, 0xEF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xCC, 0xCC, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xE407, 0xE007, 0xDC07, 0x0702, 0x0802, 0x0902),
  },
  {	.channel		= 210,
	.freq			= 5050, /* MHz */
	.unk2			= 3367,
	RADIOREGS(0x71, 0x01, 0xF9, 0x0D, 0xEF, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xCC, 0xCC, 0xFF, 0x00, 0x0F, 0x0F,
		  0x8F, 0xFF, 0x00, 0x0F, 0x0F, 0x8F),
	PHYREGS(0xE807, 0xE407, 0xE007, 0x0602, 0x0702, 0x0802),
  },
  {	.channel		= 212,
	.freq			= 5060, /* MHz */
	.unk2			= 3373,
	RADIOREGS(0x71, 0x01, 0xFA, 0x0D, 0xE6, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xBB, 0xBB, 0xFF, 0x00, 0x0E, 0x0F,
		  0x8E, 0xFF, 0x00, 0x0E, 0x0F, 0x8E),
	PHYREGS(0xEC07, 0xE807, 0xE407, 0x0502, 0x0602, 0x0702),
  },
  {	.channel		= 214,
	.freq			= 5070, /* MHz */
	.unk2			= 3380,
	RADIOREGS(0x71, 0x01, 0xFB, 0x0D, 0xE6, 0x01, 0x04, 0x0A,
		  0x00, 0x8F, 0xBB, 0xBB, 0xFF, 0x00, 0x0E, 0x0F,
		  0x8E, 0xFF, 0x00, 0x0E, 0x0F, 0x8E),
	PHYREGS(0xF007, 0xEC07, 0xE807, 0x0402, 0x0502, 0x0602),
  },
  {	.channel		= 216,
	.freq			= 5080, /* MHz */
	.unk2			= 3387,
	RADIOREGS(0x71, 0x01, 0xFC, 0x0D, 0xDE, 0x01, 0x04, 0x0A,
		  0x00, 0x8E, 0xBB, 0xBB, 0xEE, 0x00, 0x0E, 0x0F,
		  0x8D, 0xEE, 0x00, 0x0E, 0x0F, 0x8D),
	PHYREGS(0xF407, 0xF007, 0xEC07, 0x0302, 0x0402, 0x0502),
  },
  {	.channel		= 218,
	.freq			= 5090, /* MHz */
	.unk2			= 3393,
	RADIOREGS(0x71, 0x01, 0xFD, 0x0D, 0xDE, 0x01, 0x04, 0x0A,
		  0x00, 0x8E, 0xBB, 0xBB, 0xEE, 0x00, 0x0E, 0x0F,
		  0x8D, 0xEE, 0x00, 0x0E, 0x0F, 0x8D),
	PHYREGS(0xF807, 0xF407, 0xF007, 0x0202, 0x0302, 0x0402),
  },
  {	.channel		= 220,
	.freq			= 5100, /* MHz */
	.unk2			= 3400,
	RADIOREGS(0x71, 0x01, 0xFE, 0x0C, 0xD6, 0x01, 0x04, 0x0A,
		  0x00, 0x8E, 0xAA, 0xAA, 0xEE, 0x00, 0x0D, 0x0F,
		  0x8D, 0xEE, 0x00, 0x0D, 0x0F, 0x8D),
	PHYREGS(0xFC07, 0xF807, 0xF407, 0x0102, 0x0202, 0x0302),
  },
  {	.channel		= 222,
	.freq			= 5110, /* MHz */
	.unk2			= 3407,
	RADIOREGS(0x71, 0x01, 0xFF, 0x0C, 0xD6, 0x01, 0x04, 0x0A,
		  0x00, 0x8E, 0xAA, 0xAA, 0xEE, 0x00, 0x0D, 0x0F,
		  0x8D, 0xEE, 0x00, 0x0D, 0x0F, 0x8D),
	PHYREGS(0x0008, 0xFC07, 0xF807, 0x0002, 0x0102, 0x0202),
  },
  {	.channel		= 224,
	.freq			= 5120, /* MHz */
	.unk2			= 3413,
	RADIOREGS(0x71, 0x02, 0x00, 0x0C, 0xCE, 0x01, 0x04, 0x0A,
		  0x00, 0x8D, 0xAA, 0xAA, 0xDD, 0x00, 0x0D, 0x0F,
		  0x8C, 0xDD, 0x00, 0x0D, 0x0F, 0x8C),
	PHYREGS(0x0408, 0x0008, 0xFC07, 0xFF01, 0x0002, 0x0102),
  },
  {	.channel		= 226,
	.freq			= 5130, /* MHz */
	.unk2			= 3420,
	RADIOREGS(0x71, 0x02, 0x01, 0x0C, 0xCE, 0x01, 0x04, 0x0A,
		  0x00, 0x8D, 0xAA, 0xAA, 0xDD, 0x00, 0x0D, 0x0F,
		  0x8C, 0xDD, 0x00, 0x0D, 0x0F, 0x8C),
	PHYREGS(0x0808, 0x0408, 0x0008, 0xFE01, 0xFF01, 0x0002),
  },
  {	.channel		= 228,
	.freq			= 5140, /* MHz */
	.unk2			= 3427,
	RADIOREGS(0x71, 0x02, 0x02, 0x0C, 0xC6, 0x01, 0x04, 0x0A,
		  0x00, 0x8D, 0x99, 0x99, 0xDD, 0x00, 0x0C, 0x0E,
		  0x8B, 0xDD, 0x00, 0x0C, 0x0E, 0x8B),
	PHYREGS(0x0C08, 0x0808, 0x0408, 0xFD01, 0xFE01, 0xFF01),
  },
  {	.channel		= 32,
	.freq			= 5160, /* MHz */
	.unk2			= 3440,
	RADIOREGS(0x71, 0x02, 0x04, 0x0B, 0xBE, 0x01, 0x04, 0x0A,
		  0x00, 0x8C, 0x99, 0x99, 0xCC, 0x00, 0x0B, 0x0D,
		  0x8A, 0xCC, 0x00, 0x0B, 0x0D, 0x8A),
	PHYREGS(0x1408, 0x1008, 0x0C08, 0xFB01, 0xFC01, 0xFD01),
  },
  {	.channel		= 34,
	.freq			= 5170, /* MHz */
	.unk2			= 3447,
	RADIOREGS(0x71, 0x02, 0x05, 0x0B, 0xBE, 0x01, 0x04, 0x0A,
		  0x00, 0x8C, 0x99, 0x99, 0xCC, 0x00, 0x0B, 0x0D,
		  0x8A, 0xCC, 0x00, 0x0B, 0x0D, 0x8A),
	PHYREGS(0x1808, 0x1408, 0x1008, 0xFA01, 0xFB01, 0xFC01),
  },
  {	.channel		= 36,
	.freq			= 5180, /* MHz */
	.unk2			= 3453,
	RADIOREGS(0x71, 0x02, 0x06, 0x0B, 0xB6, 0x01, 0x04, 0x0A,
		  0x00, 0x8C, 0x88, 0x88, 0xCC, 0x00, 0x0B, 0x0C,
		  0x89, 0xCC, 0x00, 0x0B, 0x0C, 0x89),
	PHYREGS(0x1C08, 0x1808, 0x1408, 0xF901, 0xFA01, 0xFB01),
  },
  {	.channel		= 38,
	.freq			= 5190, /* MHz */
	.unk2			= 3460,
	RADIOREGS(0x71, 0x02, 0x07, 0x0B, 0xB6, 0x01, 0x04, 0x0A,
		  0x00, 0x8C, 0x88, 0x88, 0xCC, 0x00, 0x0B, 0x0C,
		  0x89, 0xCC, 0x00, 0x0B, 0x0C, 0x89),
	PHYREGS(0x2008, 0x1C08, 0x1808, 0xF801, 0xF901, 0xFA01),
  },
  {	.channel		= 40,
	.freq			= 5200, /* MHz */
	.unk2			= 3467,
	RADIOREGS(0x71, 0x02, 0x08, 0x0B, 0xAF, 0x01, 0x04, 0x0A,
		  0x00, 0x8B, 0x88, 0x88, 0xBB, 0x00, 0x0A, 0x0B,
		  0x89, 0xBB, 0x00, 0x0A, 0x0B, 0x89),
	PHYREGS(0x2408, 0x2008, 0x1C08, 0xF701, 0xF801, 0xF901),
  },
  {	.channel		= 42,
	.freq			= 5210, /* MHz */
	.unk2			= 3473,
	RADIOREGS(0x71, 0x02, 0x09, 0x0B, 0xAF, 0x01, 0x04, 0x0A,
		  0x00, 0x8B, 0x88, 0x88, 0xBB, 0x00, 0x0A, 0x0B,
		  0x89, 0xBB, 0x00, 0x0A, 0x0B, 0x89),
	PHYREGS(0x2808, 0x2408, 0x2008, 0xF601, 0xF701, 0xF801),
  },
  {	.channel		= 44,
	.freq			= 5220, /* MHz */
	.unk2			= 3480,
	RADIOREGS(0x71, 0x02, 0x0A, 0x0A, 0xA7, 0x01, 0x04, 0x0A,
		  0x00, 0x8B, 0x77, 0x77, 0xBB, 0x00, 0x09, 0x0A,
		  0x88, 0xBB, 0x00, 0x09, 0x0A, 0x88),
	PHYREGS(0x2C08, 0x2808, 0x2408, 0xF501, 0xF601, 0xF701),
  },
  {	.channel		= 46,
	.freq			= 5230, /* MHz */
	.unk2			= 3487,
	RADIOREGS(0x71, 0x02, 0x0B, 0x0A, 0xA7, 0x01, 0x04, 0x0A,
		  0x00, 0x8B, 0x77, 0x77, 0xBB, 0x00, 0x09, 0x0A,
		  0x88, 0xBB, 0x00, 0x09, 0x0A, 0x88),
	PHYREGS(0x3008, 0x2C08, 0x2808, 0xF401, 0xF501, 0xF601),
  },
  {	.channel		= 48,
	.freq			= 5240, /* MHz */
	.unk2			= 3493,
	RADIOREGS(0x71, 0x02, 0x0C, 0x0A, 0xA0, 0x01, 0x04, 0x0A,
		  0x00, 0x8A, 0x77, 0x77, 0xAA, 0x00, 0x09, 0x0A,
		  0x87, 0xAA, 0x00, 0x09, 0x0A, 0x87),
	PHYREGS(0x3408, 0x3008, 0x2C08, 0xF301, 0xF401, 0xF501),
  },
  {	.channel		= 50,
	.freq			= 5250, /* MHz */
	.unk2			= 3500,
	RADIOREGS(0x71, 0x02, 0x0D, 0x0A, 0xA0, 0x01, 0x04, 0x0A,
		  0x00, 0x8A, 0x77, 0x77, 0xAA, 0x00, 0x09, 0x0A,
		  0x87, 0xAA, 0x00, 0x09, 0x0A, 0x87),
	PHYREGS(0x3808, 0x3408, 0x3008, 0xF201, 0xF301, 0xF401),
  },
  {	.channel		= 52,
	.freq			= 5260, /* MHz */
	.unk2			= 3507,
	RADIOREGS(0x71, 0x02, 0x0E, 0x0A, 0x98, 0x01, 0x04, 0x0A,
		  0x00, 0x8A, 0x66, 0x66, 0xAA, 0x00, 0x08, 0x09,
		  0x87, 0xAA, 0x00, 0x08, 0x09, 0x87),
	PHYREGS(0x3C08, 0x3808, 0x3408, 0xF101, 0xF201, 0xF301),
  },
  {	.channel		= 54,
	.freq			= 5270, /* MHz */
	.unk2			= 3513,
	RADIOREGS(0x71, 0x02, 0x0F, 0x0A, 0x98, 0x01, 0x04, 0x0A,
		  0x00, 0x8A, 0x66, 0x66, 0xAA, 0x00, 0x08, 0x09,
		  0x87, 0xAA, 0x00, 0x08, 0x09, 0x87),
	PHYREGS(0x4008, 0x3C08, 0x3808, 0xF001, 0xF101, 0xF201),
  },
  {	.channel		= 56,
	.freq			= 5280, /* MHz */
	.unk2			= 3520,
	RADIOREGS(0x71, 0x02, 0x10, 0x09, 0x91, 0x01, 0x04, 0x0A,
		  0x00, 0x89, 0x66, 0x66, 0x99, 0x00, 0x08, 0x08,
		  0x86, 0x99, 0x00, 0x08, 0x08, 0x86),
	PHYREGS(0x4408, 0x4008, 0x3C08, 0xF001, 0xF001, 0xF101),
  },
  {	.channel		= 58,
	.freq			= 5290, /* MHz */
	.unk2			= 3527,
	RADIOREGS(0x71, 0x02, 0x11, 0x09, 0x91, 0x01, 0x04, 0x0A,
		  0x00, 0x89, 0x66, 0x66, 0x99, 0x00, 0x08, 0x08,
		  0x86, 0x99, 0x00, 0x08, 0x08, 0x86),
	PHYREGS(0x4808, 0x4408, 0x4008, 0xEF01, 0xF001, 0xF001),
  },
  {	.channel		= 60,
	.freq			= 5300, /* MHz */
	.unk2			= 3533,
	RADIOREGS(0x71, 0x02, 0x12, 0x09, 0x8A, 0x01, 0x04, 0x0A,
		  0x00, 0x89, 0x55, 0x55, 0x99, 0x00, 0x08, 0x07,
		  0x85, 0x99, 0x00, 0x08, 0x07, 0x85),
	PHYREGS(0x4C08, 0x4808, 0x4408, 0xEE01, 0xEF01, 0xF001),
  },
  {	.channel		= 62,
	.freq			= 5310, /* MHz */
	.unk2			= 3540,
	RADIOREGS(0x71, 0x02, 0x13, 0x09, 0x8A, 0x01, 0x04, 0x0A,
		  0x00, 0x89, 0x55, 0x55, 0x99, 0x00, 0x08, 0x07,
		  0x85, 0x99, 0x00, 0x08, 0x07, 0x85),
	PHYREGS(0x5008, 0x4C08, 0x4808, 0xED01, 0xEE01, 0xEF01),
  },
  {	.channel		= 64,
	.freq			= 5320, /* MHz */
	.unk2			= 3547,
	RADIOREGS(0x71, 0x02, 0x14, 0x09, 0x83, 0x01, 0x04, 0x0A,
		  0x00, 0x88, 0x55, 0x55, 0x88, 0x00, 0x07, 0x07,
		  0x84, 0x88, 0x00, 0x07, 0x07, 0x84),
	PHYREGS(0x5408, 0x5008, 0x4C08, 0xEC01, 0xED01, 0xEE01),
  },
  {	.channel		= 66,
	.freq			= 5330, /* MHz */
	.unk2			= 3553,
	RADIOREGS(0x71, 0x02, 0x15, 0x09, 0x83, 0x01, 0x04, 0x0A,
		  0x00, 0x88, 0x55, 0x55, 0x88, 0x00, 0x07, 0x07,
		  0x84, 0x88, 0x00, 0x07, 0x07, 0x84),
	PHYREGS(0x5808, 0x5408, 0x5008, 0xEB01, 0xEC01, 0xED01),
  },
  {	.channel		= 68,
	.freq			= 5340, /* MHz */
	.unk2			= 3560,
	RADIOREGS(0x71, 0x02, 0x16, 0x08, 0x7C, 0x01, 0x04, 0x0A,
		  0x00, 0x88, 0x44, 0x44, 0x88, 0x00, 0x07, 0x06,
		  0x84, 0x88, 0x00, 0x07, 0x06, 0x84),
	PHYREGS(0x5C08, 0x5808, 0x5408, 0xEA01, 0xEB01, 0xEC01),
  },
  {	.channel		= 70,
	.freq			= 5350, /* MHz */
	.unk2			= 3567,
	RADIOREGS(0x71, 0x02, 0x17, 0x08, 0x7C, 0x01, 0x04, 0x0A,
		  0x00, 0x88, 0x44, 0x44, 0x88, 0x00, 0x07, 0x06,
		  0x84, 0x88, 0x00, 0x07, 0x06, 0x84),
	PHYREGS(0x6008, 0x5C08, 0x5808, 0xE901, 0xEA01, 0xEB01),
  },
  {	.channel		= 72,
	.freq			= 5360, /* MHz */
	.unk2			= 3573,
	RADIOREGS(0x71, 0x02, 0x18, 0x08, 0x75, 0x01, 0x04, 0x0A,
		  0x00, 0x87, 0x44, 0x44, 0x77, 0x00, 0x06, 0x05,
		  0x83, 0x77, 0x00, 0x06, 0x05, 0x83),
	PHYREGS(0x6408, 0x6008, 0x5C08, 0xE801, 0xE901, 0xEA01),
  },
  {	.channel		= 74,
	.freq			= 5370, /* MHz */
	.unk2			= 3580,
	RADIOREGS(0x71, 0x02, 0x19, 0x08, 0x75, 0x01, 0x04, 0x0A,
		  0x00, 0x87, 0x44, 0x44, 0x77, 0x00, 0x06, 0x05,
		  0x83, 0x77, 0x00, 0x06, 0x05, 0x83),
	PHYREGS(0x6808, 0x6408, 0x6008, 0xE701, 0xE801, 0xE901),
  },
  {	.channel		= 76,
	.freq			= 5380, /* MHz */
	.unk2			= 3587,
	RADIOREGS(0x71, 0x02, 0x1A, 0x08, 0x6E, 0x01, 0x04, 0x0A,
		  0x00, 0x87, 0x33, 0x33, 0x77, 0x00, 0x06, 0x04,
		  0x82, 0x77, 0x00, 0x06, 0x04, 0x82),
	PHYREGS(0x6C08, 0x6808, 0x6408, 0xE601, 0xE701, 0xE801),
  },
  {	.channel		= 78,
	.freq			= 5390, /* MHz */
	.unk2			= 3593,
	RADIOREGS(0x71, 0x02, 0x1B, 0x08, 0x6E, 0x01, 0x04, 0x0A,
		  0x00, 0x87, 0x33, 0x33, 0x77, 0x00, 0x06, 0x04,
		  0x82, 0x77, 0x00, 0x06, 0x04, 0x82),
	PHYREGS(0x7008, 0x6C08, 0x6808, 0xE501, 0xE601, 0xE701),
  },
  {	.channel		= 80,
	.freq			= 5400, /* MHz */
	.unk2			= 3600,
	RADIOREGS(0x71, 0x02, 0x1C, 0x07, 0x67, 0x01, 0x04, 0x0A,
		  0x00, 0x86, 0x33, 0x33, 0x66, 0x00, 0x05, 0x04,
		  0x81, 0x66, 0x00, 0x05, 0x04, 0x81),
	PHYREGS(0x7408, 0x7008, 0x6C08, 0xE501, 0xE501, 0xE601),
  },
  {	.channel		= 82,
	.freq			= 5410, /* MHz */
	.unk2			= 3607,
	RADIOREGS(0x71, 0x02, 0x1D, 0x07, 0x67, 0x01, 0x04, 0x0A,
		  0x00, 0x86, 0x33, 0x33, 0x66, 0x00, 0x05, 0x04,
		  0x81, 0x66, 0x00, 0x05, 0x04, 0x81),
	PHYREGS(0x7808, 0x7408, 0x7008, 0xE401, 0xE501, 0xE501),
  },
  {	.channel		= 84,
	.freq			= 5420, /* MHz */
	.unk2			= 3613,
	RADIOREGS(0x71, 0x02, 0x1E, 0x07, 0x61, 0x01, 0x04, 0x0A,
		  0x00, 0x86, 0x22, 0x22, 0x66, 0x00, 0x05, 0x03,
		  0x80, 0x66, 0x00, 0x05, 0x03, 0x80),
	PHYREGS(0x7C08, 0x7808, 0x7408, 0xE301, 0xE401, 0xE501),
  },
  {	.channel		= 86,
	.freq			= 5430, /* MHz */
	.unk2			= 3620,
	RADIOREGS(0x71, 0x02, 0x1F, 0x07, 0x61, 0x01, 0x04, 0x0A,
		  0x00, 0x86, 0x22, 0x22, 0x66, 0x00, 0x05, 0x03,
		  0x80, 0x66, 0x00, 0x05, 0x03, 0x80),
	PHYREGS(0x8008, 0x7C08, 0x7808, 0xE201, 0xE301, 0xE401),
  },
  {	.channel		= 88,
	.freq			= 5440, /* MHz */
	.unk2			= 3627,
	RADIOREGS(0x71, 0x02, 0x20, 0x07, 0x5A, 0x01, 0x04, 0x0A,
		  0x00, 0x85, 0x22, 0x22, 0x55, 0x00, 0x04, 0x02,
		  0x80, 0x55, 0x00, 0x04, 0x02, 0x80),
	PHYREGS(0x8408, 0x8008, 0x7C08, 0xE101, 0xE201, 0xE301),
  },
  {	.channel		= 90,
	.freq			= 5450, /* MHz */
	.unk2			= 3633,
	RADIOREGS(0x71, 0x02, 0x21, 0x07, 0x5A, 0x01, 0x04, 0x0A,
		  0x00, 0x85, 0x22, 0x22, 0x55, 0x00, 0x04, 0x02,
		  0x80, 0x55, 0x00, 0x04, 0x02, 0x80),
	PHYREGS(0x8808, 0x8408, 0x8008, 0xE001, 0xE101, 0xE201),
  },
  {	.channel		= 92,
	.freq			= 5460, /* MHz */
	.unk2			= 3640,
	RADIOREGS(0x71, 0x02, 0x22, 0x06, 0x53, 0x01, 0x04, 0x0A,
		  0x00, 0x85, 0x11, 0x11, 0x55, 0x00, 0x04, 0x01,
		  0x80, 0x55, 0x00, 0x04, 0x01, 0x80),
	PHYREGS(0x8C08, 0x8808, 0x8408, 0xDF01, 0xE001, 0xE101),
  },
  {	.channel		= 94,
	.freq			= 5470, /* MHz */
	.unk2			= 3647,
	RADIOREGS(0x71, 0x02, 0x23, 0x06, 0x53, 0x01, 0x04, 0x0A,
		  0x00, 0x85, 0x11, 0x11, 0x55, 0x00, 0x04, 0x01,
		  0x80, 0x55, 0x00, 0x04, 0x01, 0x80),
	PHYREGS(0x9008, 0x8C08, 0x8808, 0xDE01, 0xDF01, 0xE001),
  },
  {	.channel		= 96,
	.freq			= 5480, /* MHz */
	.unk2			= 3653,
	RADIOREGS(0x71, 0x02, 0x24, 0x06, 0x4D, 0x01, 0x04, 0x0A,
		  0x00, 0x84, 0x11, 0x11, 0x44, 0x00, 0x03, 0x00,
		  0x80, 0x44, 0x00, 0x03, 0x00, 0x80),
	PHYREGS(0x9408, 0x9008, 0x8C08, 0xDD01, 0xDE01, 0xDF01),
  },
  {	.channel		= 98,
	.freq			= 5490, /* MHz */
	.unk2			= 3660,
	RADIOREGS(0x71, 0x02, 0x25, 0x06, 0x4D, 0x01, 0x04, 0x0A,
		  0x00, 0x84, 0x11, 0x11, 0x44, 0x00, 0x03, 0x00,
		  0x80, 0x44, 0x00, 0x03, 0x00, 0x80),
	PHYREGS(0x9808, 0x9408, 0x9008, 0xDD01, 0xDD01, 0xDE01),
  },
  {	.channel		= 100,
	.freq			= 5500, /* MHz */
	.unk2			= 3667,
	RADIOREGS(0x71, 0x02, 0x26, 0x06, 0x47, 0x01, 0x04, 0x0A,
		  0x00, 0x84, 0x00, 0x00, 0x44, 0x00, 0x03, 0x00,
		  0x80, 0x44, 0x00, 0x03, 0x00, 0x80),
	PHYREGS(0x9C08, 0x9808, 0x9408, 0xDC01, 0xDD01, 0xDD01),
  },
  {	.channel		= 102,
	.freq			= 5510, /* MHz */
	.unk2			= 3673,
	RADIOREGS(0x71, 0x02, 0x27, 0x06, 0x47, 0x01, 0x04, 0x0A,
		  0x00, 0x84, 0x00, 0x00, 0x44, 0x00, 0x03, 0x00,
		  0x80, 0x44, 0x00, 0x03, 0x00, 0x80),
	PHYREGS(0xA008, 0x9C08, 0x9808, 0xDB01, 0xDC01, 0xDD01),
  },
  {	.channel		= 104,
	.freq			= 5520, /* MHz */
	.unk2			= 3680,
	RADIOREGS(0x71, 0x02, 0x28, 0x05, 0x40, 0x01, 0x04, 0x0A,
		  0x00, 0x83, 0x00, 0x00, 0x33, 0x00, 0x02, 0x00,
		  0x80, 0x33, 0x00, 0x02, 0x00, 0x80),
	PHYREGS(0xA408, 0xA008, 0x9C08, 0xDA01, 0xDB01, 0xDC01),
  },
  {	.channel		= 106,
	.freq			= 5530, /* MHz */
	.unk2			= 3687,
	RADIOREGS(0x71, 0x02, 0x29, 0x05, 0x40, 0x01, 0x04, 0x0A,
		  0x00, 0x83, 0x00, 0x00, 0x33, 0x00, 0x02, 0x00,
		  0x80, 0x33, 0x00, 0x02, 0x00, 0x80),
	PHYREGS(0xA808, 0xA408, 0xA008, 0xD901, 0xDA01, 0xDB01),
  },
  {	.channel		= 108,
	.freq			= 5540, /* MHz */
	.unk2			= 3693,
	RADIOREGS(0x71, 0x02, 0x2A, 0x05, 0x3A, 0x01, 0x04, 0x0A,
		  0x00, 0x83, 0x00, 0x00, 0x33, 0x00, 0x02, 0x00,
		  0x80, 0x33, 0x00, 0x02, 0x00, 0x80),
	PHYREGS(0xAC08, 0xA808, 0xA408, 0xD801, 0xD901, 0xDA01),
  },
  {	.channel		= 110,
	.freq			= 5550, /* MHz */
	.unk2			= 3700,
	RADIOREGS(0x71, 0x02, 0x2B, 0x05, 0x3A, 0x01, 0x04, 0x0A,
		  0x00, 0x83, 0x00, 0x00, 0x33, 0x00, 0x02, 0x00,
		  0x80, 0x33, 0x00, 0x02, 0x00, 0x80),
	PHYREGS(0xB008, 0xAC08, 0xA808, 0xD701, 0xD801, 0xD901),
  },
  {	.channel		= 112,
	.freq			= 5560, /* MHz */
	.unk2			= 3707,
	RADIOREGS(0x71, 0x02, 0x2C, 0x05, 0x34, 0x01, 0x04, 0x0A,
		  0x00, 0x82, 0x00, 0x00, 0x22, 0x00, 0x01, 0x00,
		  0x80, 0x22, 0x00, 0x01, 0x00, 0x80),
	PHYREGS(0xB408, 0xB008, 0xAC08, 0xD701, 0xD701, 0xD801),
  },
  {	.channel		= 114,
	.freq			= 5570, /* MHz */
	.unk2			= 3713,
	RADIOREGS(0x71, 0x02, 0x2D, 0x05, 0x34, 0x01, 0x04, 0x0A,
		  0x00, 0x82, 0x00, 0x00, 0x22, 0x00, 0x01, 0x00,
		  0x80, 0x22, 0x00, 0x01, 0x00, 0x80),
	PHYREGS(0xB808, 0xB408, 0xB008, 0xD601, 0xD701, 0xD701),
  },
  {	.channel		= 116,
	.freq			= 5580, /* MHz */
	.unk2			= 3720,
	RADIOREGS(0x71, 0x02, 0x2E, 0x04, 0x2E, 0x01, 0x04, 0x0A,
		  0x00, 0x82, 0x00, 0x00, 0x22, 0x00, 0x01, 0x00,
		  0x80, 0x22, 0x00, 0x01, 0x00, 0x80),
	PHYREGS(0xBC08, 0xB808, 0xB408, 0xD501, 0xD601, 0xD701),
  },
  {	.channel		= 118,
	.freq			= 5590, /* MHz */
	.unk2			= 3727,
	RADIOREGS(0x71, 0x02, 0x2F, 0x04, 0x2E, 0x01, 0x04, 0x0A,
		  0x00, 0x82, 0x00, 0x00, 0x22, 0x00, 0x01, 0x00,
		  0x80, 0x22, 0x00, 0x01, 0x00, 0x80),
	PHYREGS(0xC008, 0xBC08, 0xB808, 0xD401, 0xD501, 0xD601),
  },
  {	.channel		= 120,
	.freq			= 5600, /* MHz */
	.unk2			= 3733,
	RADIOREGS(0x71, 0x02, 0x30, 0x04, 0x28, 0x01, 0x04, 0x0A,
		  0x00, 0x81, 0x00, 0x00, 0x11, 0x00, 0x01, 0x00,
		  0x80, 0x11, 0x00, 0x01, 0x00, 0x80),
	PHYREGS(0xC408, 0xC008, 0xBC08, 0xD301, 0xD401, 0xD501),
  },
  {	.channel		= 122,
	.freq			= 5610, /* MHz */
	.unk2			= 3740,
	RADIOREGS(0x71, 0x02, 0x31, 0x04, 0x28, 0x01, 0x04, 0x0A,
		  0x00, 0x81, 0x00, 0x00, 0x11, 0x00, 0x01, 0x00,
		  0x80, 0x11, 0x00, 0x01, 0x00, 0x80),
	PHYREGS(0xC808, 0xC408, 0xC008, 0xD201, 0xD301, 0xD401),
  },
  {	.channel		= 124,
	.freq			= 5620, /* MHz */
	.unk2			= 3747,
	RADIOREGS(0x71, 0x02, 0x32, 0x04, 0x21, 0x01, 0x04, 0x0A,
		  0x00, 0x81, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00,
		  0x80, 0x11, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xCC08, 0xC808, 0xC408, 0xD201, 0xD201, 0xD301),
  },
  {	.channel		= 126,
	.freq			= 5630, /* MHz */
	.unk2			= 3753,
	RADIOREGS(0x71, 0x02, 0x33, 0x04, 0x21, 0x01, 0x04, 0x0A,
		  0x00, 0x81, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00,
		  0x80, 0x11, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xD008, 0xCC08, 0xC808, 0xD101, 0xD201, 0xD201),
  },
  {	.channel		= 128,
	.freq			= 5640, /* MHz */
	.unk2			= 3760,
	RADIOREGS(0x71, 0x02, 0x34, 0x03, 0x1C, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xD408, 0xD008, 0xCC08, 0xD001, 0xD101, 0xD201),
  },
  {	.channel		= 130,
	.freq			= 5650, /* MHz */
	.unk2			= 3767,
	RADIOREGS(0x71, 0x02, 0x35, 0x03, 0x1C, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xD808, 0xD408, 0xD008, 0xCF01, 0xD001, 0xD101),
  },
  {	.channel		= 132,
	.freq			= 5660, /* MHz */
	.unk2			= 3773,
	RADIOREGS(0x71, 0x02, 0x36, 0x03, 0x16, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xDC08, 0xD808, 0xD408, 0xCE01, 0xCF01, 0xD001),
  },
  {	.channel		= 134,
	.freq			= 5670, /* MHz */
	.unk2			= 3780,
	RADIOREGS(0x71, 0x02, 0x37, 0x03, 0x16, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xE008, 0xDC08, 0xD808, 0xCE01, 0xCE01, 0xCF01),
  },
  {	.channel		= 136,
	.freq			= 5680, /* MHz */
	.unk2			= 3787,
	RADIOREGS(0x71, 0x02, 0x38, 0x03, 0x10, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xE408, 0xE008, 0xDC08, 0xCD01, 0xCE01, 0xCE01),
  },
  {	.channel		= 138,
	.freq			= 5690, /* MHz */
	.unk2			= 3793,
	RADIOREGS(0x71, 0x02, 0x39, 0x03, 0x10, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xE808, 0xE408, 0xE008, 0xCC01, 0xCD01, 0xCE01),
  },
  {	.channel		= 140,
	.freq			= 5700, /* MHz */
	.unk2			= 3800,
	RADIOREGS(0x71, 0x02, 0x3A, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xEC08, 0xE808, 0xE408, 0xCB01, 0xCC01, 0xCD01),
  },
  {	.channel		= 142,
	.freq			= 5710, /* MHz */
	.unk2			= 3807,
	RADIOREGS(0x71, 0x02, 0x3B, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xF008, 0xEC08, 0xE808, 0xCA01, 0xCB01, 0xCC01),
  },
  {	.channel		= 144,
	.freq			= 5720, /* MHz */
	.unk2			= 3813,
	RADIOREGS(0x71, 0x02, 0x3C, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xF408, 0xF008, 0xEC08, 0xC901, 0xCA01, 0xCB01),
  },
  {	.channel		= 145,
	.freq			= 5725, /* MHz */
	.unk2			= 3817,
	RADIOREGS(0x72, 0x04, 0x79, 0x02, 0x03, 0x01, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xF608, 0xF208, 0xEE08, 0xC901, 0xCA01, 0xCB01),
  },
  {	.channel		= 146,
	.freq			= 5730, /* MHz */
	.unk2			= 3820,
	RADIOREGS(0x71, 0x02, 0x3D, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xF808, 0xF408, 0xF008, 0xC901, 0xC901, 0xCA01),
  },
  {	.channel		= 147,
	.freq			= 5735, /* MHz */
	.unk2			= 3823,
	RADIOREGS(0x72, 0x04, 0x7B, 0x02, 0x03, 0x01, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xFA08, 0xF608, 0xF208, 0xC801, 0xC901, 0xCA01),
  },
  {	.channel		= 148,
	.freq			= 5740, /* MHz */
	.unk2			= 3827,
	RADIOREGS(0x71, 0x02, 0x3E, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xFC08, 0xF808, 0xF408, 0xC801, 0xC901, 0xC901),
  },
  {	.channel		= 149,
	.freq			= 5745, /* MHz */
	.unk2			= 3830,
	RADIOREGS(0x72, 0x04, 0x7D, 0x02, 0xFE, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0xFE08, 0xFA08, 0xF608, 0xC801, 0xC801, 0xC901),
  },
  {	.channel		= 150,
	.freq			= 5750, /* MHz */
	.unk2			= 3833,
	RADIOREGS(0x71, 0x02, 0x3F, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0009, 0xFC08, 0xF808, 0xC701, 0xC801, 0xC901),
  },
  {	.channel		= 151,
	.freq			= 5755, /* MHz */
	.unk2			= 3837,
	RADIOREGS(0x72, 0x04, 0x7F, 0x02, 0xFE, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0209, 0xFE08, 0xFA08, 0xC701, 0xC801, 0xC801),
  },
  {	.channel		= 152,
	.freq			= 5760, /* MHz */
	.unk2			= 3840,
	RADIOREGS(0x71, 0x02, 0x40, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0409, 0x0009, 0xFC08, 0xC601, 0xC701, 0xC801),
  },
  {	.channel		= 153,
	.freq			= 5765, /* MHz */
	.unk2			= 3843,
	RADIOREGS(0x72, 0x04, 0x81, 0x02, 0xF8, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0609, 0x0209, 0xFE08, 0xC601, 0xC701, 0xC801),
  },
  {	.channel		= 154,
	.freq			= 5770, /* MHz */
	.unk2			= 3847,
	RADIOREGS(0x71, 0x02, 0x41, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0809, 0x0409, 0x0009, 0xC601, 0xC601, 0xC701),
  },
  {	.channel		= 155,
	.freq			= 5775, /* MHz */
	.unk2			= 3850,
	RADIOREGS(0x72, 0x04, 0x83, 0x02, 0xF8, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0A09, 0x0609, 0x0209, 0xC501, 0xC601, 0xC701),
  },
  {	.channel		= 156,
	.freq			= 5780, /* MHz */
	.unk2			= 3853,
	RADIOREGS(0x71, 0x02, 0x42, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0C09, 0x0809, 0x0409, 0xC501, 0xC601, 0xC601),
  },
  {	.channel		= 157,
	.freq			= 5785, /* MHz */
	.unk2			= 3857,
	RADIOREGS(0x72, 0x04, 0x85, 0x02, 0xF2, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x0E09, 0x0A09, 0x0609, 0xC401, 0xC501, 0xC601),
  },
  {	.channel		= 158,
	.freq			= 5790, /* MHz */
	.unk2			= 3860,
	RADIOREGS(0x71, 0x02, 0x43, 0x02, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1009, 0x0C09, 0x0809, 0xC401, 0xC501, 0xC601),
  },
  {	.channel		= 159,
	.freq			= 5795, /* MHz */
	.unk2			= 3863,
	RADIOREGS(0x72, 0x04, 0x87, 0x02, 0xF2, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1209, 0x0E09, 0x0A09, 0xC401, 0xC401, 0xC501),
  },
  {	.channel		= 160,
	.freq			= 5800, /* MHz */
	.unk2			= 3867,
	RADIOREGS(0x71, 0x02, 0x44, 0x01, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1409, 0x1009, 0x0C09, 0xC301, 0xC401, 0xC501),
  },
  {	.channel		= 161,
	.freq			= 5805, /* MHz */
	.unk2			= 3870,
	RADIOREGS(0x72, 0x04, 0x89, 0x01, 0xED, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1609, 0x1209, 0x0E09, 0xC301, 0xC401, 0xC401),
  },
  {	.channel		= 162,
	.freq			= 5810, /* MHz */
	.unk2			= 3873,
	RADIOREGS(0x71, 0x02, 0x45, 0x01, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1809, 0x1409, 0x1009, 0xC201, 0xC301, 0xC401),
  },
  {	.channel		= 163,
	.freq			= 5815, /* MHz */
	.unk2			= 3877,
	RADIOREGS(0x72, 0x04, 0x8B, 0x01, 0xED, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1A09, 0x1609, 0x1209, 0xC201, 0xC301, 0xC401),
  },
  {	.channel		= 164,
	.freq			= 5820, /* MHz */
	.unk2			= 3880,
	RADIOREGS(0x71, 0x02, 0x46, 0x01, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1C09, 0x1809, 0x1409, 0xC201, 0xC201, 0xC301),
  },
  {	.channel		= 165,
	.freq			= 5825, /* MHz */
	.unk2			= 3883,
	RADIOREGS(0x72, 0x04, 0x8D, 0x01, 0xED, 0x00, 0x03, 0x14,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x1E09, 0x1A09, 0x1609, 0xC101, 0xC201, 0xC301),
  },
  {	.channel		= 166,
	.freq			= 5830, /* MHz */
	.unk2			= 3887,
	RADIOREGS(0x71, 0x02, 0x47, 0x01, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x2009, 0x1C09, 0x1809, 0xC101, 0xC201, 0xC201),
  },
  {	.channel		= 168,
	.freq			= 5840, /* MHz */
	.unk2			= 3893,
	RADIOREGS(0x71, 0x02, 0x48, 0x01, 0x0A, 0x01, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x2409, 0x2009, 0x1C09, 0xC001, 0xC101, 0xC201),
  },
  {	.channel		= 170,
	.freq			= 5850, /* MHz */
	.unk2			= 3900,
	RADIOREGS(0x71, 0x02, 0x49, 0x01, 0xE0, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x2809, 0x2409, 0x2009, 0xBF01, 0xC001, 0xC101),
  },
  {	.channel		= 172,
	.freq			= 5860, /* MHz */
	.unk2			= 3907,
	RADIOREGS(0x71, 0x02, 0x4A, 0x01, 0xDE, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x2C09, 0x2809, 0x2409, 0xBF01, 0xBF01, 0xC001),
  },
  {	.channel		= 174,
	.freq			= 5870, /* MHz */
	.unk2			= 3913,
	RADIOREGS(0x71, 0x02, 0x4B, 0x00, 0xDB, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x3009, 0x2C09, 0x2809, 0xBE01, 0xBF01, 0xBF01),
  },
  {	.channel		= 176,
	.freq			= 5880, /* MHz */
	.unk2			= 3920,
	RADIOREGS(0x71, 0x02, 0x4C, 0x00, 0xD8, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x3409, 0x3009, 0x2C09, 0xBD01, 0xBE01, 0xBF01),
  },
  {	.channel		= 178,
	.freq			= 5890, /* MHz */
	.unk2			= 3927,
	RADIOREGS(0x71, 0x02, 0x4D, 0x00, 0xD6, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x3809, 0x3409, 0x3009, 0xBC01, 0xBD01, 0xBE01),
  },
  {	.channel		= 180,
	.freq			= 5900, /* MHz */
	.unk2			= 3933,
	RADIOREGS(0x71, 0x02, 0x4E, 0x00, 0xD3, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x3C09, 0x3809, 0x3409, 0xBC01, 0xBC01, 0xBD01),
  },
  {	.channel		= 182,
	.freq			= 5910, /* MHz */
	.unk2			= 3940,
	RADIOREGS(0x71, 0x02, 0x4F, 0x00, 0xD6, 0x00, 0x04, 0x0A,
		  0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		  0x80, 0x00, 0x00, 0x00, 0x00, 0x80),
	PHYREGS(0x4009, 0x3C09, 0x3809, 0xBB01, 0xBC01, 0xBC01),
  },
  {	.channel		= 1,
	.freq			= 2412, /* MHz */
	.unk2			= 3216,
	RADIOREGS(0x73, 0x09, 0x6C, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0D, 0x0C,
		  0x80, 0xFF, 0x88, 0x0D, 0x0C, 0x80),
	PHYREGS(0xC903, 0xC503, 0xC103, 0x3A04, 0x3F04, 0x4304),
  },
  {	.channel		= 2,
	.freq			= 2417, /* MHz */
	.unk2			= 3223,
	RADIOREGS(0x73, 0x09, 0x71, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0C, 0x0B,
		  0x80, 0xFF, 0x88, 0x0C, 0x0B, 0x80),
	PHYREGS(0xCB03, 0xC703, 0xC303, 0x3804, 0x3D04, 0x4104),
  },
  {	.channel		= 3,
	.freq			= 2422, /* MHz */
	.unk2			= 3229,
	RADIOREGS(0x73, 0x09, 0x76, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0C, 0x0A,
		  0x80, 0xFF, 0x88, 0x0C, 0x0A, 0x80),
	PHYREGS(0xCD03, 0xC903, 0xC503, 0x3604, 0x3A04, 0x3F04),
  },
  {	.channel		= 4,
	.freq			= 2427, /* MHz */
	.unk2			= 3236,
	RADIOREGS(0x73, 0x09, 0x7B, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0C, 0x0A,
		  0x80, 0xFF, 0x88, 0x0C, 0x0A, 0x80),
	PHYREGS(0xCF03, 0xCB03, 0xC703, 0x3404, 0x3804, 0x3D04),
  },
  {	.channel		= 5,
	.freq			= 2432, /* MHz */
	.unk2			= 3243,
	RADIOREGS(0x73, 0x09, 0x80, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0C, 0x09,
		  0x80, 0xFF, 0x88, 0x0C, 0x09, 0x80),
	PHYREGS(0xD103, 0xCD03, 0xC903, 0x3104, 0x3604, 0x3A04),
  },
  {	.channel		= 6,
	.freq			= 2437, /* MHz */
	.unk2			= 3249,
	RADIOREGS(0x73, 0x09, 0x85, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0B, 0x08,
		  0x80, 0xFF, 0x88, 0x0B, 0x08, 0x80),
	PHYREGS(0xD303, 0xCF03, 0xCB03, 0x2F04, 0x3404, 0x3804),
  },
  {	.channel		= 7,
	.freq			= 2442, /* MHz */
	.unk2			= 3256,
	RADIOREGS(0x73, 0x09, 0x8A, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0A, 0x07,
		  0x80, 0xFF, 0x88, 0x0A, 0x07, 0x80),
	PHYREGS(0xD503, 0xD103, 0xCD03, 0x2D04, 0x3104, 0x3604),
  },
  {	.channel		= 8,
	.freq			= 2447, /* MHz */
	.unk2			= 3263,
	RADIOREGS(0x73, 0x09, 0x8F, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x0A, 0x06,
		  0x80, 0xFF, 0x88, 0x0A, 0x06, 0x80),
	PHYREGS(0xD703, 0xD303, 0xCF03, 0x2B04, 0x2F04, 0x3404),
  },
  {	.channel		= 9,
	.freq			= 2452, /* MHz */
	.unk2			= 3269,
	RADIOREGS(0x73, 0x09, 0x94, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x09, 0x06,
		  0x80, 0xFF, 0x88, 0x09, 0x06, 0x80),
	PHYREGS(0xD903, 0xD503, 0xD103, 0x2904, 0x2D04, 0x3104),
  },
  {	.channel		= 10,
	.freq			= 2457, /* MHz */
	.unk2			= 3276,
	RADIOREGS(0x73, 0x09, 0x99, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x08, 0x05,
		  0x80, 0xFF, 0x88, 0x08, 0x05, 0x80),
	PHYREGS(0xDB03, 0xD703, 0xD303, 0x2704, 0x2B04, 0x2F04),
  },
  {	.channel		= 11,
	.freq			= 2462, /* MHz */
	.unk2			= 3283,
	RADIOREGS(0x73, 0x09, 0x9E, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x08, 0x04,
		  0x80, 0xFF, 0x88, 0x08, 0x04, 0x80),
	PHYREGS(0xDD03, 0xD903, 0xD503, 0x2404, 0x2904, 0x2D04),
  },
  {	.channel		= 12,
	.freq			= 2467, /* MHz */
	.unk2			= 3289,
	RADIOREGS(0x73, 0x09, 0xA3, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x08, 0x03,
		  0x80, 0xFF, 0x88, 0x08, 0x03, 0x80),
	PHYREGS(0xDF03, 0xDB03, 0xD703, 0x2204, 0x2704, 0x2B04),
  },
  {	.channel		= 13,
	.freq			= 2472, /* MHz */
	.unk2			= 3296,
	RADIOREGS(0x73, 0x09, 0xA8, 0x0F, 0x00, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x07, 0x03,
		  0x80, 0xFF, 0x88, 0x07, 0x03, 0x80),
	PHYREGS(0xE103, 0xDD03, 0xD903, 0x2004, 0x2404, 0x2904),
  },
  {	.channel		= 14,
	.freq			= 2484, /* MHz */
	.unk2			= 3312,
	RADIOREGS(0x73, 0x09, 0xB4, 0x0F, 0xFF, 0x01, 0x07, 0x15,
		  0x01, 0x8F, 0xFF, 0xFF, 0xFF, 0x88, 0x07, 0x01,
		  0x80, 0xFF, 0x88, 0x07, 0x01, 0x80),
	PHYREGS(0xE603, 0xE203, 0xDE03, 0x1B04, 0x1F04, 0x2404),
  },
};

const struct b43_nphy_channeltab_entry *
b43_nphy_get_chantabent(struct b43_wldev *dev, u8 channel)
{
	const struct b43_nphy_channeltab_entry *e;
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(b43_nphy_channeltab); i++) {
		e = &(b43_nphy_channeltab[i]);
		if (e->channel == channel)
			return e;
	}

	return NULL;
}

const u8 b43_ntab_adjustpower0[] = {
	0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x02, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03,
	0x04, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05,
	0x06, 0x06, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x09,
	0x0A, 0x0A, 0x0A, 0x0A, 0x0B, 0x0B, 0x0B, 0x0B,
	0x0C, 0x0C, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D,
	0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F,
	0x10, 0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11,
	0x12, 0x12, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13,
	0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x15,
	0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 0x17,
	0x18, 0x18, 0x18, 0x18, 0x19, 0x19, 0x19, 0x19,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1B,
	0x1C, 0x1C, 0x1C, 0x1C, 0x1D, 0x1D, 0x1D, 0x1D,
	0x1E, 0x1E, 0x1E, 0x1E, 0x1F, 0x1F, 0x1F, 0x1F,
};

const u8 b43_ntab_adjustpower1[] = {
	0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x02, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03,
	0x04, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05,
	0x06, 0x06, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x09,
	0x0A, 0x0A, 0x0A, 0x0A, 0x0B, 0x0B, 0x0B, 0x0B,
	0x0C, 0x0C, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D,
	0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F,
	0x10, 0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11,
	0x12, 0x12, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13,
	0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x15,
	0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 0x17,
	0x18, 0x18, 0x18, 0x18, 0x19, 0x19, 0x19, 0x19,
	0x1A, 0x1A, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1B,
	0x1C, 0x1C, 0x1C, 0x1C, 0x1D, 0x1D, 0x1D, 0x1D,
	0x1E, 0x1E, 0x1E, 0x1E, 0x1F, 0x1F, 0x1F, 0x1F,
};

const u16 b43_ntab_bdi[] = {
	0x0070, 0x0126, 0x012C, 0x0246, 0x048D, 0x04D2,
};

const u32 b43_ntab_channelest[] = {
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x44444444, 0x44444444, 0x44444444, 0x44444444,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
	0x10101010, 0x10101010, 0x10101010, 0x10101010,
};

const u8 b43_ntab_estimatepowerlt0[] = {
	0x50, 0x4F, 0x4E, 0x4D, 0x4C, 0x4B, 0x4A, 0x49,
	0x48, 0x47, 0x46, 0x45, 0x44, 0x43, 0x42, 0x41,
	0x40, 0x3F, 0x3E, 0x3D, 0x3C, 0x3B, 0x3A, 0x39,
	0x38, 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31,
	0x30, 0x2F, 0x2E, 0x2D, 0x2C, 0x2B, 0x2A, 0x29,
	0x28, 0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21,
	0x20, 0x1F, 0x1E, 0x1D, 0x1C, 0x1B, 0x1A, 0x19,
	0x18, 0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11,
};

const u8 b43_ntab_estimatepowerlt1[] = {
	0x50, 0x4F, 0x4E, 0x4D, 0x4C, 0x4B, 0x4A, 0x49,
	0x48, 0x47, 0x46, 0x45, 0x44, 0x43, 0x42, 0x41,
	0x40, 0x3F, 0x3E, 0x3D, 0x3C, 0x3B, 0x3A, 0x39,
	0x38, 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31,
	0x30, 0x2F, 0x2E, 0x2D, 0x2C, 0x2B, 0x2A, 0x29,
	0x28, 0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0x21,
	0x20, 0x1F, 0x1E, 0x1D, 0x1C, 0x1B, 0x1A, 0x19,
	0x18, 0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11,
};

const u8 b43_ntab_framelookup[] = {
	0x02, 0x04, 0x14, 0x14, 0x03, 0x05, 0x16, 0x16,
	0x0A, 0x0C, 0x1C, 0x1C, 0x0B, 0x0D, 0x1E, 0x1E,
	0x06, 0x08, 0x18, 0x18, 0x07, 0x09, 0x1A, 0x1A,
	0x0E, 0x10, 0x20, 0x28, 0x0F, 0x11, 0x22, 0x2A,
};

const u32 b43_ntab_framestruct[] = {
	0x08004A04, 0x00100000, 0x01000A05, 0x00100020,
	0x09804506, 0x00100030, 0x09804507, 0x00100030,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x08004A0C, 0x00100008, 0x01000A0D, 0x00100028,
	0x0980450E, 0x00100038, 0x0980450F, 0x00100038,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000A04, 0x00100000, 0x11008A05, 0x00100020,
	0x1980C506, 0x00100030, 0x21810506, 0x00100030,
	0x21810506, 0x00100030, 0x01800504, 0x00100030,
	0x11808505, 0x00100030, 0x29814507, 0x01100030,
	0x00000A04, 0x00100000, 0x11008A05, 0x00100020,
	0x21810506, 0x00100030, 0x21810506, 0x00100030,
	0x29814507, 0x01100030, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000A0C, 0x00100008, 0x11008A0D, 0x00100028,
	0x1980C50E, 0x00100038, 0x2181050E, 0x00100038,
	0x2181050E, 0x00100038, 0x0180050C, 0x00100038,
	0x1180850D, 0x00100038, 0x2981450F, 0x01100038,
	0x00000A0C, 0x00100008, 0x11008A0D, 0x00100028,
	0x2181050E, 0x00100038, 0x2181050E, 0x00100038,
	0x2981450F, 0x01100038, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x08004A04, 0x00100000, 0x01000A05, 0x00100020,
	0x1980C506, 0x00100030, 0x1980C506, 0x00100030,
	0x11808504, 0x00100030, 0x3981CA05, 0x00100030,
	0x29814507, 0x01100030, 0x00000000, 0x00000000,
	0x10008A04, 0x00100000, 0x3981CA05, 0x00100030,
	0x1980C506, 0x00100030, 0x29814507, 0x01100030,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x08004A0C, 0x00100008, 0x01000A0D, 0x00100028,
	0x1980C50E, 0x00100038, 0x1980C50E, 0x00100038,
	0x1180850C, 0x00100038, 0x3981CA0D, 0x00100038,
	0x2981450F, 0x01100038, 0x00000000, 0x00000000,
	0x10008A0C, 0x00100008, 0x3981CA0D, 0x00100038,
	0x1980C50E, 0x00100038, 0x2981450F, 0x01100038,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x40021404, 0x00100000, 0x02001405, 0x00100040,
	0x0B004A06, 0x01900060, 0x13008A06, 0x01900060,
	0x13008A06, 0x01900060, 0x43020A04, 0x00100060,
	0x1B00CA05, 0x00100060, 0x23010A07, 0x01500060,
	0x40021404, 0x00100000, 0x1A00D405, 0x00100040,
	0x13008A06, 0x01900060, 0x13008A06, 0x01900060,
	0x23010A07, 0x01500060, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4002140C, 0x00100010, 0x0200140D, 0x00100050,
	0x0B004A0E, 0x01900070, 0x13008A0E, 0x01900070,
	0x13008A0E, 0x01900070, 0x43020A0C, 0x00100070,
	0x1B00CA0D, 0x00100070, 0x23010A0F, 0x01500070,
	0x4002140C, 0x00100010, 0x1A00D40D, 0x00100050,
	0x13008A0E, 0x01900070, 0x13008A0E, 0x01900070,
	0x23010A0F, 0x01500070, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x50029404, 0x00100000, 0x32019405, 0x00100040,
	0x0B004A06, 0x01900060, 0x0B004A06, 0x01900060,
	0x5B02CA04, 0x00100060, 0x3B01D405, 0x00100060,
	0x23010A07, 0x01500060, 0x00000000, 0x00000000,
	0x5802D404, 0x00100000, 0x3B01D405, 0x00100060,
	0x0B004A06, 0x01900060, 0x23010A07, 0x01500060,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x5002940C, 0x00100010, 0x3201940D, 0x00100050,
	0x0B004A0E, 0x01900070, 0x0B004A0E, 0x01900070,
	0x5B02CA0C, 0x00100070, 0x3B01D40D, 0x00100070,
	0x23010A0F, 0x01500070, 0x00000000, 0x00000000,
	0x5802D40C, 0x00100010, 0x3B01D40D, 0x00100070,
	0x0B004A0E, 0x01900070, 0x23010A0F, 0x01500070,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x40021404, 0x000F4800, 0x62031405, 0x00100040,
	0x53028A06, 0x01900060, 0x53028A07, 0x01900060,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4002140C, 0x000F4810, 0x6203140D, 0x00100050,
	0x53028A0E, 0x01900070, 0x53028A0F, 0x01900070,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000A0C, 0x00100008, 0x11008A0D, 0x00100028,
	0x1980C50E, 0x00100038, 0x2181050E, 0x00100038,
	0x2181050E, 0x00100038, 0x0180050C, 0x00100038,
	0x1180850D, 0x00100038, 0x1181850D, 0x00100038,
	0x2981450F, 0x01100038, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000A0C, 0x00100008, 0x11008A0D, 0x00100028,
	0x2181050E, 0x00100038, 0x2181050E, 0x00100038,
	0x1181850D, 0x00100038, 0x2981450F, 0x01100038,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x08004A04, 0x00100000, 0x01000A05, 0x00100020,
	0x0180C506, 0x00100030, 0x0180C506, 0x00100030,
	0x2180C50C, 0x00100030, 0x49820A0D, 0x0016A130,
	0x41824A0D, 0x0016A130, 0x2981450F, 0x01100030,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x2000CA0C, 0x00100000, 0x49820A0D, 0x0016A130,
	0x1980C50E, 0x00100030, 0x41824A0D, 0x0016A130,
	0x2981450F, 0x01100030, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4002140C, 0x00100010, 0x0200140D, 0x00100050,
	0x0B004A0E, 0x01900070, 0x13008A0E, 0x01900070,
	0x13008A0E, 0x01900070, 0x43020A0C, 0x00100070,
	0x1B00CA0D, 0x00100070, 0x1B014A0D, 0x00100070,
	0x23010A0F, 0x01500070, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4002140C, 0x00100010, 0x1A00D40D, 0x00100050,
	0x13008A0E, 0x01900070, 0x13008A0E, 0x01900070,
	0x1B014A0D, 0x00100070, 0x23010A0F, 0x01500070,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x50029404, 0x00100000, 0x32019405, 0x00100040,
	0x03004A06, 0x01900060, 0x03004A06, 0x01900060,
	0x6B030A0C, 0x00100060, 0x4B02140D, 0x0016A160,
	0x4302540D, 0x0016A160, 0x23010A0F, 0x01500060,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x6B03140C, 0x00100060, 0x4B02140D, 0x0016A160,
	0x0B004A0E, 0x01900060, 0x4302540D, 0x0016A160,
	0x23010A0F, 0x01500060, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x40021404, 0x00100000, 0x1A00D405, 0x00100040,
	0x53028A06, 0x01900060, 0x5B02CA06, 0x01900060,
	0x5B02CA06, 0x01900060, 0x43020A04, 0x00100060,
	0x1B00CA05, 0x00100060, 0x53028A07, 0x0190C060,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4002140C, 0x00100010, 0x1A00D40D, 0x00100050,
	0x53028A0E, 0x01900070, 0x5B02CA0E, 0x01900070,
	0x5B02CA0E, 0x01900070, 0x43020A0C, 0x00100070,
	0x1B00CA0D, 0x00100070, 0x53028A0F, 0x0190C070,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x40021404, 0x00100000, 0x1A00D405, 0x00100040,
	0x5B02CA06, 0x01900060, 0x5B02CA06, 0x01900060,
	0x53028A07, 0x0190C060, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4002140C, 0x00100010, 0x1A00D40D, 0x00100050,
	0x5B02CA0E, 0x01900070, 0x5B02CA0E, 0x01900070,
	0x53028A0F, 0x0190C070, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
};

const u32 b43_ntab_gainctl0[] = {
	0x007F003F, 0x007E013F, 0x007D023E, 0x007C033E,
	0x007B043D, 0x007A053D, 0x0079063C, 0x0078073C,
	0x0077083B, 0x0076093B, 0x00750A3A, 0x00740B3A,
	0x00730C39, 0x00720D39, 0x00710E38, 0x00700F38,
	0x006F0037, 0x006E0137, 0x006D0236, 0x006C0336,
	0x006B0435, 0x006A0535, 0x00690634, 0x00680734,
	0x00670833, 0x00660933, 0x00650A32, 0x00640B32,
	0x00630C31, 0x00620D31, 0x00610E30, 0x00600F30,
	0x005F002F, 0x005E012F, 0x005D022E, 0x005C032E,
	0x005B042D, 0x005A052D, 0x0059062C, 0x0058072C,
	0x0057082B, 0x0056092B, 0x00550A2A, 0x00540B2A,
	0x00530C29, 0x00520D29, 0x00510E28, 0x00500F28,
	0x004F0027, 0x004E0127, 0x004D0226, 0x004C0326,
	0x004B0425, 0x004A0525, 0x00490624, 0x00480724,
	0x00470823, 0x00460923, 0x00450A22, 0x00440B22,
	0x00430C21, 0x00420D21, 0x00410E20, 0x00400F20,
	0x003F001F, 0x003E011F, 0x003D021E, 0x003C031E,
	0x003B041D, 0x003A051D, 0x0039061C, 0x0038071C,
	0x0037081B, 0x0036091B, 0x00350A1A, 0x00340B1A,
	0x00330C19, 0x00320D19, 0x00310E18, 0x00300F18,
	0x002F0017, 0x002E0117, 0x002D0216, 0x002C0316,
	0x002B0415, 0x002A0515, 0x00290614, 0x00280714,
	0x00270813, 0x00260913, 0x00250A12, 0x00240B12,
	0x00230C11, 0x00220D11, 0x00210E10, 0x00200F10,
	0x001F000F, 0x001E010F, 0x001D020E, 0x001C030E,
	0x001B040D, 0x001A050D, 0x0019060C, 0x0018070C,
	0x0017080B, 0x0016090B, 0x00150A0A, 0x00140B0A,
	0x00130C09, 0x00120D09, 0x00110E08, 0x00100F08,
	0x000F0007, 0x000E0107, 0x000D0206, 0x000C0306,
	0x000B0405, 0x000A0505, 0x00090604, 0x00080704,
	0x00070803, 0x00060903, 0x00050A02, 0x00040B02,
	0x00030C01, 0x00020D01, 0x00010E00, 0x00000F00,
};

const u32 b43_ntab_gainctl1[] = {
	0x007F003F, 0x007E013F, 0x007D023E, 0x007C033E,
	0x007B043D, 0x007A053D, 0x0079063C, 0x0078073C,
	0x0077083B, 0x0076093B, 0x00750A3A, 0x00740B3A,
	0x00730C39, 0x00720D39, 0x00710E38, 0x00700F38,
	0x006F0037, 0x006E0137, 0x006D0236, 0x006C0336,
	0x006B0435, 0x006A0535, 0x00690634, 0x00680734,
	0x00670833, 0x00660933, 0x00650A32, 0x00640B32,
	0x00630C31, 0x00620D31, 0x00610E30, 0x00600F30,
	0x005F002F, 0x005E012F, 0x005D022E, 0x005C032E,
	0x005B042D, 0x005A052D, 0x0059062C, 0x0058072C,
	0x0057082B, 0x0056092B, 0x00550A2A, 0x00540B2A,
	0x00530C29, 0x00520D29, 0x00510E28, 0x00500F28,
	0x004F0027, 0x004E0127, 0x004D0226, 0x004C0326,
	0x004B0425, 0x004A0525, 0x00490624, 0x00480724,
	0x00470823, 0x00460923, 0x00450A22, 0x00440B22,
	0x00430C21, 0x00420D21, 0x00410E20, 0x00400F20,
	0x003F001F, 0x003E011F, 0x003D021E, 0x003C031E,
	0x003B041D, 0x003A051D, 0x0039061C, 0x0038071C,
	0x0037081B, 0x0036091B, 0x00350A1A, 0x00340B1A,
	0x00330C19, 0x00320D19, 0x00310E18, 0x00300F18,
	0x002F0017, 0x002E0117, 0x002D0216, 0x002C0316,
	0x002B0415, 0x002A0515, 0x00290614, 0x00280714,
	0x00270813, 0x00260913, 0x00250A12, 0x00240B12,
	0x00230C11, 0x00220D11, 0x00210E10, 0x00200F10,
	0x001F000F, 0x001E010F, 0x001D020E, 0x001C030E,
	0x001B040D, 0x001A050D, 0x0019060C, 0x0018070C,
	0x0017080B, 0x0016090B, 0x00150A0A, 0x00140B0A,
	0x00130C09, 0x00120D09, 0x00110E08, 0x00100F08,
	0x000F0007, 0x000E0107, 0x000D0206, 0x000C0306,
	0x000B0405, 0x000A0505, 0x00090604, 0x00080704,
	0x00070803, 0x00060903, 0x00050A02, 0x00040B02,
	0x00030C01, 0x00020D01, 0x00010E00, 0x00000F00,
};

const u32 b43_ntab_intlevel[] = {
	0x00802070, 0x0671188D, 0x0A60192C, 0x0A300E46,
	0x00C1188D, 0x080024D2, 0x00000070,
};

const u32 b43_ntab_iqlt0[] = {
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
};

const u32 b43_ntab_iqlt1[] = {
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
	0x0000007F, 0x0000007F, 0x0000007F, 0x0000007F,
};

const u16 b43_ntab_loftlt0[] = {
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103,
};

const u16 b43_ntab_loftlt1[] = {
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103, 0x0000, 0x0101, 0x0002, 0x0103,
	0x0000, 0x0101, 0x0002, 0x0103, 0x0000, 0x0101,
	0x0002, 0x0103,
};

const u8 b43_ntab_mcs[] = {
	0x00, 0x08, 0x0A, 0x10, 0x12, 0x19, 0x1A, 0x1C,
	0x40, 0x48, 0x4A, 0x50, 0x52, 0x59, 0x5A, 0x5C,
	0x80, 0x88, 0x8A, 0x90, 0x92, 0x99, 0x9A, 0x9C,
	0xC0, 0xC8, 0xCA, 0xD0, 0xD2, 0xD9, 0xDA, 0xDC,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x01, 0x02, 0x04, 0x08, 0x09, 0x0A, 0x0C,
	0x10, 0x11, 0x12, 0x14, 0x18, 0x19, 0x1A, 0x1C,
	0x20, 0x21, 0x22, 0x24, 0x40, 0x41, 0x42, 0x44,
	0x48, 0x49, 0x4A, 0x4C, 0x50, 0x51, 0x52, 0x54,
	0x58, 0x59, 0x5A, 0x5C, 0x60, 0x61, 0x62, 0x64,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

const u32 b43_ntab_noisevar10[] = {
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
};

const u32 b43_ntab_noisevar11[] = {
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
	0x020C020C, 0x0000014D, 0x020C020C, 0x0000014D,
};

const u16 b43_ntab_pilot[] = {
	0xFF08, 0xFF08, 0xFF08, 0xFF08, 0xFF08, 0xFF08,
	0xFF08, 0xFF08, 0x80D5, 0x80D5, 0x80D5, 0x80D5,
	0x80D5, 0x80D5, 0x80D5, 0x80D5, 0xFF0A, 0xFF82,
	0xFFA0, 0xFF28, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	0xFF82, 0xFFA0, 0xFF28, 0xFF0A, 0xFFFF, 0xFFFF,
	0xFFFF, 0xFFFF, 0xF83F, 0xFA1F, 0xFA97, 0xFAB5,
	0xF2BD, 0xF0BF, 0xFFFF, 0xFFFF, 0xF017, 0xF815,
	0xF215, 0xF095, 0xF035, 0xF01D, 0xFFFF, 0xFFFF,
	0xFF08, 0xFF02, 0xFF80, 0xFF20, 0xFF08, 0xFF02,
	0xFF80, 0xFF20, 0xF01F, 0xF817, 0xFA15, 0xF295,
	0xF0B5, 0xF03D, 0xFFFF, 0xFFFF, 0xF82A, 0xFA0A,
	0xFA82, 0xFAA0, 0xF2A8, 0xF0AA, 0xFFFF, 0xFFFF,
	0xF002, 0xF800, 0xF200, 0xF080, 0xF020, 0xF008,
	0xFFFF, 0xFFFF, 0xF00A, 0xF802, 0xFA00, 0xF280,
	0xF0A0, 0xF028, 0xFFFF, 0xFFFF,
};

const u32 b43_ntab_pilotlt[] = {
	0x76540123, 0x62407351, 0x76543201, 0x76540213,
	0x76540123, 0x76430521,
};

const u32 b43_ntab_tdi20a0[] = {
	0x00091226, 0x000A1429, 0x000B56AD, 0x000C58B0,
	0x000D5AB3, 0x000E9CB6, 0x000F9EBA, 0x0000C13D,
	0x00020301, 0x00030504, 0x00040708, 0x0005090B,
	0x00064B8E, 0x00095291, 0x000A5494, 0x000B9718,
	0x000C9927, 0x000D9B2A, 0x000EDD2E, 0x000FDF31,
	0x000101B4, 0x000243B7, 0x000345BB, 0x000447BE,
	0x00058982, 0x00068C05, 0x00099309, 0x000A950C,
	0x000BD78F, 0x000CD992, 0x000DDB96, 0x000F1D99,
	0x00005FA8, 0x0001422C, 0x0002842F, 0x00038632,
	0x00048835, 0x0005CA38, 0x0006CCBC, 0x0009D3BF,
	0x000B1603, 0x000C1806, 0x000D1A0A, 0x000E1C0D,
	0x000F5E10, 0x00008093, 0x00018297, 0x0002C49A,
	0x0003C680, 0x0004C880, 0x00060B00, 0x00070D00,
	0x00000000, 0x00000000, 0x00000000,
};

const u32 b43_ntab_tdi20a1[] = {
	0x00014B26, 0x00028D29, 0x000393AD, 0x00049630,
	0x0005D833, 0x0006DA36, 0x00099C3A, 0x000A9E3D,
	0x000BC081, 0x000CC284, 0x000DC488, 0x000F068B,
	0x0000488E, 0x00018B91, 0x0002D214, 0x0003D418,
	0x0004D6A7, 0x000618AA, 0x00071AAE, 0x0009DCB1,
	0x000B1EB4, 0x000C0137, 0x000D033B, 0x000E053E,
	0x000F4702, 0x00008905, 0x00020C09, 0x0003128C,
	0x0004148F, 0x00051712, 0x00065916, 0x00091B19,
	0x000A1D28, 0x000B5F2C, 0x000C41AF, 0x000D43B2,
	0x000E85B5, 0x000F87B8, 0x0000C9BC, 0x00024CBF,
	0x00035303, 0x00045506, 0x0005978A, 0x0006998D,
	0x00095B90, 0x000A5D93, 0x000B9F97, 0x000C821A,
	0x000D8400, 0x000EC600, 0x000FC800, 0x00010A00,
	0x00000000, 0x00000000, 0x00000000,
};

const u32 b43_ntab_tdi40a0[] = {
	0x0011A346, 0x00136CCF, 0x0014F5D9, 0x001641E2,
	0x0017CB6B, 0x00195475, 0x001B2383, 0x001CAD0C,
	0x001E7616, 0x0000821F, 0x00020BA8, 0x0003D4B2,
	0x00056447, 0x00072DD0, 0x0008B6DA, 0x000A02E3,
	0x000B8C6C, 0x000D15F6, 0x0011E484, 0x0013AE0D,
	0x00153717, 0x00168320, 0x00180CA9, 0x00199633,
	0x001B6548, 0x001CEED1, 0x001EB7DB, 0x0000C3E4,
	0x00024D6D, 0x000416F7, 0x0005A585, 0x00076F0F,
	0x0008F818, 0x000A4421, 0x000BCDAB, 0x000D9734,
	0x00122649, 0x0013EFD2, 0x001578DC, 0x0016C4E5,
	0x00184E6E, 0x001A17F8, 0x001BA686, 0x001D3010,
	0x001EF999, 0x00010522, 0x00028EAC, 0x00045835,
	0x0005E74A, 0x0007B0D3, 0x00093A5D, 0x000A85E6,
	0x000C0F6F, 0x000DD8F9, 0x00126787, 0x00143111,
	0x0015BA9A, 0x00170623, 0x00188FAD, 0x001A5936,
	0x001BE84B, 0x001DB1D4, 0x001F3B5E, 0x000146E7,
	0x00031070, 0x000499FA, 0x00062888, 0x0007F212,
	0x00097B9B, 0x000AC7A4, 0x000C50AE, 0x000E1A37,
	0x0012A94C, 0x001472D5, 0x0015FC5F, 0x00174868,
	0x0018D171, 0x001A9AFB, 0x001C2989, 0x001DF313,
	0x001F7C9C, 0x000188A5, 0x000351AF, 0x0004DB38,
	0x0006AA4D, 0x000833D7, 0x0009BD60, 0x000B0969,
	0x000C9273, 0x000E5BFC, 0x00132A8A, 0x0014B414,
	0x00163D9D, 0x001789A6, 0x001912B0, 0x001ADC39,
	0x001C6BCE, 0x001E34D8, 0x001FBE61, 0x0001CA6A,
	0x00039374, 0x00051CFD, 0x0006EC0B, 0x00087515,
	0x0009FE9E, 0x000B4AA7, 0x000CD3B1, 0x000E9D3A,
	0x00000000, 0x00000000,
};

const u32 b43_ntab_tdi40a1[] = {
	0x001EDB36, 0x000129CA, 0x0002B353, 0x00047CDD,
	0x0005C8E6, 0x000791EF, 0x00091BF9, 0x000AAA07,
	0x000C3391, 0x000DFD1A, 0x00120923, 0x0013D22D,
	0x00155C37, 0x0016EACB, 0x00187454, 0x001A3DDE,
	0x001B89E7, 0x001D12F0, 0x001F1CFA, 0x00016B88,
	0x00033492, 0x0004BE1B, 0x00060A24, 0x0007D32E,
	0x00095D38, 0x000AEC4C, 0x000C7555, 0x000E3EDF,
	0x00124AE8, 0x001413F1, 0x0015A37B, 0x00172C89,
	0x0018B593, 0x001A419C, 0x001BCB25, 0x001D942F,
	0x001F63B9, 0x0001AD4D, 0x00037657, 0x0004C260,
	0x00068BE9, 0x000814F3, 0x0009A47C, 0x000B2D8A,
	0x000CB694, 0x000E429D, 0x00128C26, 0x001455B0,
	0x0015E4BA, 0x00176E4E, 0x0018F758, 0x001A8361,
	0x001C0CEA, 0x001DD674, 0x001FA57D, 0x0001EE8B,
	0x0003B795, 0x0005039E, 0x0006CD27, 0x000856B1,
	0x0009E5C6, 0x000B6F4F, 0x000CF859, 0x000E8462,
	0x00130DEB, 0x00149775, 0x00162603, 0x0017AF8C,
	0x00193896, 0x001AC49F, 0x001C4E28, 0x001E17B2,
	0x0000A6C7, 0x00023050, 0x0003F9DA, 0x00054563,
	0x00070EEC, 0x00089876, 0x000A2704, 0x000BB08D,
	0x000D3A17, 0x001185A0, 0x00134F29, 0x0014D8B3,
	0x001667C8, 0x0017F151, 0x00197ADB, 0x001B0664,
	0x001C8FED, 0x001E5977, 0x0000E805, 0x0002718F,
	0x00043B18, 0x000586A1, 0x0007502B, 0x0008D9B4,
	0x000A68C9, 0x000BF252, 0x000DBBDC, 0x0011C7E5,
	0x001390EE, 0x00151A78, 0x0016A906, 0x00183290,
	0x0019BC19, 0x001B4822, 0x001CD12C, 0x001E9AB5,
	0x00000000, 0x00000000,
};

const u32 b43_ntab_tdtrn[] = {
	0x061C061C, 0x0050EE68, 0xF592FE36, 0xFE5212F6,
	0x00000C38, 0xFE5212F6, 0xF592FE36, 0x0050EE68,
	0x061C061C, 0xEE680050, 0xFE36F592, 0x12F6FE52,
	0x0C380000, 0x12F6FE52, 0xFE36F592, 0xEE680050,
	0x061C061C, 0x0050EE68, 0xF592FE36, 0xFE5212F6,
	0x00000C38, 0xFE5212F6, 0xF592FE36, 0x0050EE68,
	0x061C061C, 0xEE680050, 0xFE36F592, 0x12F6FE52,
	0x0C380000, 0x12F6FE52, 0xFE36F592, 0xEE680050,
	0x05E305E3, 0x004DEF0C, 0xF5F3FE47, 0xFE611246,
	0x00000BC7, 0xFE611246, 0xF5F3FE47, 0x004DEF0C,
	0x05E305E3, 0xEF0C004D, 0xFE47F5F3, 0x1246FE61,
	0x0BC70000, 0x1246FE61, 0xFE47F5F3, 0xEF0C004D,
	0x05E305E3, 0x004DEF0C, 0xF5F3FE47, 0xFE611246,
	0x00000BC7, 0xFE611246, 0xF5F3FE47, 0x004DEF0C,
	0x05E305E3, 0xEF0C004D, 0xFE47F5F3, 0x1246FE61,
	0x0BC70000, 0x1246FE61, 0xFE47F5F3, 0xEF0C004D,
	0xFA58FA58, 0xF895043B, 0xFF4C09C0, 0xFBC6FFA8,
	0xFB84F384, 0x0798F6F9, 0x05760122, 0x058409F6,
	0x0B500000, 0x05B7F542, 0x08860432, 0x06DDFEE7,
	0xFB84F384, 0xF9D90664, 0xF7E8025C, 0x00FFF7BD,
	0x05A805A8, 0xF7BD00FF, 0x025CF7E8, 0x0664F9D9,
	0xF384FB84, 0xFEE706DD, 0x04320886, 0xF54205B7,
	0x00000B50, 0x09F60584, 0x01220576, 0xF6F90798,
	0xF384FB84, 0xFFA8FBC6, 0x09C0FF4C, 0x043BF895,
	0x02D402D4, 0x07DE0270, 0xFC96079C, 0xF90AFE94,
	0xFE00FF2C, 0x02D4065D, 0x092A0096, 0x0014FBB8,
	0xFD2CFD2C, 0x076AFB3C, 0x0096F752, 0xF991FD87,
	0xFB2C0200, 0xFEB8F960, 0x08E0FC96, 0x049802A8,
	0xFD2CFD2C, 0x02A80498, 0xFC9608E0, 0xF960FEB8,
	0x0200FB2C, 0xFD87F991, 0xF7520096, 0xFB3C076A,
	0xFD2CFD2C, 0xFBB80014, 0x0096092A, 0x065D02D4,
	0xFF2CFE00, 0xFE94F90A, 0x079CFC96, 0x027007DE,
	0x02D402D4, 0x027007DE, 0x079CFC96, 0xFE94F90A,
	0xFF2CFE00, 0x065D02D4, 0x0096092A, 0xFBB80014,
	0xFD2CFD2C, 0xFB3C076A, 0xF7520096, 0xFD87F991,
	0x0200FB2C, 0xF960FEB8, 0xFC9608E0, 0x02A80498,
	0xFD2CFD2C, 0x049802A8, 0x08E0FC96, 0xFEB8F960,
	0xFB2C0200, 0xF991FD87, 0x0096F752, 0x076AFB3C,
	0xFD2CFD2C, 0x0014FBB8, 0x092A0096, 0x02D4065D,
	0xFE00FF2C, 0xF90AFE94, 0xFC96079C, 0x07DE0270,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x062A0000, 0xFEFA0759, 0x08B80908, 0xF396FC2D,
	0xF9D6045C, 0xFC4EF608, 0xF748F596, 0x07B207BF,
	0x062A062A, 0xF84EF841, 0xF748F596, 0x03B209F8,
	0xF9D6045C, 0x0C6A03D3, 0x08B80908, 0x0106F8A7,
	0x062A0000, 0xFEFAF8A7, 0x08B8F6F8, 0xF39603D3,
	0xF9D6FBA4, 0xFC4E09F8, 0xF7480A6A, 0x07B2F841,
	0x062AF9D6, 0xF84E07BF, 0xF7480A6A, 0x03B2F608,
	0xF9D6FBA4, 0x0C6AFC2D, 0x08B8F6F8, 0x01060759,
	0x062A0000, 0xFEFA0759, 0x08B80908, 0xF396FC2D,
	0xF9D6045C, 0xFC4EF608, 0xF748F596, 0x07B207BF,
	0x062A062A, 0xF84EF841, 0xF748F596, 0x03B209F8,
	0xF9D6045C, 0x0C6A03D3, 0x08B80908, 0x0106F8A7,
	0x062A0000, 0xFEFAF8A7, 0x08B8F6F8, 0xF39603D3,
	0xF9D6FBA4, 0xFC4E09F8, 0xF7480A6A, 0x07B2F841,
	0x062AF9D6, 0xF84E07BF, 0xF7480A6A, 0x03B2F608,
	0xF9D6FBA4, 0x0C6AFC2D, 0x08B8F6F8, 0x01060759,
	0x061C061C, 0xFF30009D, 0xFFB21141, 0xFD87FB54,
	0xF65DFE59, 0x02EEF99E, 0x0166F03C, 0xFFF809B6,
	0x000008A4, 0x000AF42B, 0x00EFF577, 0xFA840BF2,
	0xFC02FF51, 0x08260F67, 0xFFF0036F, 0x0842F9C3,
	0x00000000, 0x063DF7BE, 0xFC910010, 0xF099F7DA,
	0x00AF03FE, 0xF40E057C, 0x0A89FF11, 0x0BD5FFF6,
	0xF75C0000, 0xF64A0008, 0x0FC4FE9A, 0x0662FD12,
	0x01A709A3, 0x04AC0279, 0xEEBF004E, 0xFF6300D0,
	0xF9E4F9E4, 0x00D0FF63, 0x004EEEBF, 0x027904AC,
	0x09A301A7, 0xFD120662, 0xFE9A0FC4, 0x0008F64A,
	0x0000F75C, 0xFFF60BD5, 0xFF110A89, 0x057CF40E,
	0x03FE00AF, 0xF7DAF099, 0x0010FC91, 0xF7BE063D,
	0x00000000, 0xF9C30842, 0x036FFFF0, 0x0F670826,
	0xFF51FC02, 0x0BF2FA84, 0xF57700EF, 0xF42B000A,
	0x08A40000, 0x09B6FFF8, 0xF03C0166, 0xF99E02EE,
	0xFE59F65D, 0xFB54FD87, 0x1141FFB2, 0x009DFF30,
	0x05E30000, 0xFF060705, 0x085408A0, 0xF425FC59,
	0xFA1D042A, 0xFC78F67A, 0xF7ACF60E, 0x075A0766,
	0x05E305E3, 0xF8A6F89A, 0xF7ACF60E, 0x03880986,
	0xFA1D042A, 0x0BDB03A7, 0x085408A0, 0x00FAF8FB,
	0x05E30000, 0xFF06F8FB, 0x0854F760, 0xF42503A7,
	0xFA1DFBD6, 0xFC780986, 0xF7AC09F2, 0x075AF89A,
	0x05E3FA1D, 0xF8A60766, 0xF7AC09F2, 0x0388F67A,
	0xFA1DFBD6, 0x0BDBFC59, 0x0854F760, 0x00FA0705,
	0x05E30000, 0xFF060705, 0x085408A0, 0xF425FC59,
	0xFA1D042A, 0xFC78F67A, 0xF7ACF60E, 0x075A0766,
	0x05E305E3, 0xF8A6F89A, 0xF7ACF60E, 0x03880986,
	0xFA1D042A, 0x0BDB03A7, 0x085408A0, 0x00FAF8FB,
	0x05E30000, 0xFF06F8FB, 0x0854F760, 0xF42503A7,
	0xFA1DFBD6, 0xFC780986, 0xF7AC09F2, 0x075AF89A,
	0x05E3FA1D, 0xF8A60766, 0xF7AC09F2, 0x0388F67A,
	0xFA1DFBD6, 0x0BDBFC59, 0x0854F760, 0x00FA0705,
	0xFA58FA58, 0xF8F0FE00, 0x0448073D, 0xFDC9FE46,
	0xF9910258, 0x089D0407, 0xFD5CF71A, 0x02AFFDE0,
	0x083E0496, 0xFF5A0740, 0xFF7AFD97, 0x00FE01F1,
	0x0009082E, 0xFA94FF75, 0xFECDF8EA, 0xFFB0F693,
	0xFD2CFA58, 0x0433FF16, 0xFBA405DD, 0xFA610341,
	0x06A606CB, 0x0039FD2D, 0x0677FA97, 0x01FA05E0,
	0xF896003E, 0x075A068B, 0x012CFC3E, 0xFA23F98D,
	0xFC7CFD43, 0xFF90FC0D, 0x01C10982, 0x00C601D6,
	0xFD2CFD2C, 0x01D600C6, 0x098201C1, 0xFC0DFF90,
	0xFD43FC7C, 0xF98DFA23, 0xFC3E012C, 0x068B075A,
	0x003EF896, 0x05E001FA, 0xFA970677, 0xFD2D0039,
	0x06CB06A6, 0x0341FA61, 0x05DDFBA4, 0xFF160433,
	0xFA58FD2C, 0xF693FFB0, 0xF8EAFECD, 0xFF75FA94,
	0x082E0009, 0x01F100FE, 0xFD97FF7A, 0x0740FF5A,
	0x0496083E, 0xFDE002AF, 0xF71AFD5C, 0x0407089D,
	0x0258F991, 0xFE46FDC9, 0x073D0448, 0xFE00F8F0,
	0xFD2CFD2C, 0xFCE00500, 0xFC09FDDC, 0xFE680157,
	0x04C70571, 0xFC3AFF21, 0xFCD70228, 0x056D0277,
	0x0200FE00, 0x0022F927, 0xFE3C032B, 0xFC44FF3C,
	0x03E9FBDB, 0x04570313, 0x04C9FF5C, 0x000D03B8,
	0xFA580000, 0xFBE900D2, 0xF9D0FE0B, 0x0125FDF9,
	0x042501BF, 0x0328FA2B, 0xFFA902F0, 0xFA250157,
	0x0200FE00, 0x03740438, 0xFF0405FD, 0x030CFE52,
	0x0037FB39, 0xFF6904C5, 0x04F8FD23, 0xFD31FC1B,
	0xFD2CFD2C, 0xFC1BFD31, 0xFD2304F8, 0x04C5FF69,
	0xFB390037, 0xFE52030C, 0x05FDFF04, 0x04380374,
	0xFE000200, 0x0157FA25, 0x02F0FFA9, 0xFA2B0328,
	0x01BF0425, 0xFDF90125, 0xFE0BF9D0, 0x00D2FBE9,
	0x0000FA58, 0x03B8000D, 0xFF5C04C9, 0x03130457,
	0xFBDB03E9, 0xFF3CFC44, 0x032BFE3C, 0xF9270022,
	0xFE000200, 0x0277056D, 0x0228FCD7, 0xFF21FC3A,
	0x057104C7, 0x0157FE68, 0xFDDCFC09, 0x0500FCE0,
	0xFD2CFD2C, 0x0500FCE0, 0xFDDCFC09, 0x0157FE68,
	0x057104C7, 0xFF21FC3A, 0x0228FCD7, 0x0277056D,
	0xFE000200, 0xF9270022, 0x032BFE3C, 0xFF3CFC44,
	0xFBDB03E9, 0x03130457, 0xFF5C04C9, 0x03B8000D,
	0x0000FA58, 0x00D2FBE9, 0xFE0BF9D0, 0xFDF90125,
	0x01BF0425, 0xFA2B0328, 0x02F0FFA9, 0x0157FA25,
	0xFE000200, 0x04380374, 0x05FDFF04, 0xFE52030C,
	0xFB390037, 0x04C5FF69, 0xFD2304F8, 0xFC1BFD31,
	0xFD2CFD2C, 0xFD31FC1B, 0x04F8FD23, 0xFF6904C5,
	0x0037FB39, 0x030CFE52, 0xFF0405FD, 0x03740438,
	0x0200FE00, 0xFA250157, 0xFFA902F0, 0x0328FA2B,
	0x042501BF, 0x0125FDF9, 0xF9D0FE0B, 0xFBE900D2,
	0xFA580000, 0x000D03B8, 0x04C9FF5C, 0x04570313,
	0x03E9FBDB, 0xFC44FF3C, 0xFE3C032B, 0x0022F927,
	0x0200FE00, 0x056D0277, 0xFCD70228, 0xFC3AFF21,
	0x04C70571, 0xFE680157, 0xFC09FDDC, 0xFCE00500,
	0x05A80000, 0xFF1006BE, 0x0800084A, 0xF49CFC7E,
	0xFA580400, 0xFC9CF6DA, 0xF800F672, 0x0710071C,
	0x05A805A8, 0xF8F0F8E4, 0xF800F672, 0x03640926,
	0xFA580400, 0x0B640382, 0x0800084A, 0x00F0F942,
	0x05A80000, 0xFF10F942, 0x0800F7B6, 0xF49C0382,
	0xFA58FC00, 0xFC9C0926, 0xF800098E, 0x0710F8E4,
	0x05A8FA58, 0xF8F0071C, 0xF800098E, 0x0364F6DA,
	0xFA58FC00, 0x0B64FC7E, 0x0800F7B6, 0x00F006BE,
	0x05A80000, 0xFF1006BE, 0x0800084A, 0xF49CFC7E,
	0xFA580400, 0xFC9CF6DA, 0xF800F672, 0x0710071C,
	0x05A805A8, 0xF8F0F8E4, 0xF800F672, 0x03640926,
	0xFA580400, 0x0B640382, 0x0800084A, 0x00F0F942,
	0x05A80000, 0xFF10F942, 0x0800F7B6, 0xF49C0382,
	0xFA58FC00, 0xFC9C0926, 0xF800098E, 0x0710F8E4,
	0x05A8FA58, 0xF8F0071C, 0xF800098E, 0x0364F6DA,
	0xFA58FC00, 0x0B64FC7E, 0x0800F7B6, 0x00F006BE,
};

const u32 b43_ntab_tmap[] = {
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0xF1111110, 0x11111111, 0x11F11111, 0x00000111,
	0x11000000, 0x1111F111, 0x11111111, 0x111111F1,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x000AA888,
	0x88880000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0xA1111110, 0x11111111, 0x11C11111, 0x00000111,
	0x11000000, 0x1111A111, 0x11111111, 0x111111A1,
	0xA2222220, 0x22222222, 0x22C22222, 0x00000222,
	0x22000000, 0x2222A222, 0x22222222, 0x222222A2,
	0xF1111110, 0x11111111, 0x11F11111, 0x00011111,
	0x11110000, 0x1111F111, 0x11111111, 0x111111F1,
	0xA8AA88A0, 0xA88888A8, 0xA8A8A88A, 0x00088AAA,
	0xAAAA0000, 0xA8A8AA88, 0xA88AAAAA, 0xAAAA8A8A,
	0xAAA8AAA0, 0x8AAA8AAA, 0xAA8A8A8A, 0x000AAA88,
	0x8AAA0000, 0xAAA8A888, 0x8AA88A8A, 0x8A88A888,
	0x08080A00, 0x0A08080A, 0x080A0A08, 0x00080808,
	0x080A0000, 0x080A0808, 0x080A0808, 0x0A0A0A08,
	0xA0A0A0A0, 0x80A0A080, 0x8080A0A0, 0x00008080,
	0x80A00000, 0x80A080A0, 0xA080A0A0, 0x8080A0A0,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x99999000, 0x9B9B99BB, 0x9BB99999, 0x9999B9B9,
	0x9B99BB90, 0x9BBBBB9B, 0x9B9B9BB9, 0x00000999,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00AAA888,
	0x22000000, 0x2222B222, 0x22222222, 0x222222B2,
	0xB2222220, 0x22222222, 0x22D22222, 0x00000222,
	0x11000000, 0x1111A111, 0x11111111, 0x111111A1,
	0xA1111110, 0x11111111, 0x11C11111, 0x00000111,
	0x33000000, 0x3333B333, 0x33333333, 0x333333B3,
	0xB3333330, 0x33333333, 0x33D33333, 0x00000333,
	0x22000000, 0x2222A222, 0x22222222, 0x222222A2,
	0xA2222220, 0x22222222, 0x22C22222, 0x00000222,
	0x99B99B00, 0x9B9B99BB, 0x9BB99999, 0x9999B9B9,
	0x9B99BB99, 0x9BBBBB9B, 0x9B9B9BB9, 0x00000999,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA88, 0x8AAAAA8A, 0x8A8A8AA8, 0x08AAA888,
	0x22222200, 0x2222F222, 0x22222222, 0x222222F2,
	0x22222222, 0x22222222, 0x22F22222, 0x00000222,
	0x11000000, 0x1111F111, 0x11111111, 0x11111111,
	0xF1111111, 0x11111111, 0x11F11111, 0x01111111,
	0xBB9BB900, 0xB9B9BB99, 0xB99BBBBB, 0xBBBB9B9B,
	0xB9BB99BB, 0xB99999B9, 0xB9B9B99B, 0x00000BBB,
	0xAA000000, 0xA8A8AA88, 0xA88AAAAA, 0xAAAA8A8A,
	0xA8AA88AA, 0xA88888A8, 0xA8A8A88A, 0x0A888AAA,
	0xAA000000, 0xA8A8AA88, 0xA88AAAAA, 0xAAAA8A8A,
	0xA8AA88A0, 0xA88888A8, 0xA8A8A88A, 0x00000AAA,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0xBBBBBB00, 0x999BBBBB, 0x9BB99B9B, 0xB9B9B9BB,
	0xB9B99BBB, 0xB9B9B9BB, 0xB9BB9B99, 0x00000999,
	0x8A000000, 0xAA88A888, 0xA88888AA, 0xA88A8A88,
	0xA88AA88A, 0x88A8AAAA, 0xA8AA8AAA, 0x0888A88A,
	0x0B0B0B00, 0x090B0B0B, 0x0B090B0B, 0x0909090B,
	0x09090B0B, 0x09090B0B, 0x09090B09, 0x00000909,
	0x0A000000, 0x0A080808, 0x080A080A, 0x080A0A08,
	0x080A080A, 0x0808080A, 0x0A0A0A08, 0x0808080A,
	0xB0B0B000, 0x9090B0B0, 0x90B09090, 0xB0B0B090,
	0xB0B090B0, 0x90B0B0B0, 0xB0B09090, 0x00000090,
	0x80000000, 0xA080A080, 0xA08080A0, 0xA0808080,
	0xA080A080, 0x80A0A0A0, 0xA0A080A0, 0x00A0A0A0,
	0x22000000, 0x2222F222, 0x22222222, 0x222222F2,
	0xF2222220, 0x22222222, 0x22F22222, 0x00000222,
	0x11000000, 0x1111F111, 0x11111111, 0x111111F1,
	0xF1111110, 0x11111111, 0x11F11111, 0x00000111,
	0x33000000, 0x3333F333, 0x33333333, 0x333333F3,
	0xF3333330, 0x33333333, 0x33F33333, 0x00000333,
	0x22000000, 0x2222F222, 0x22222222, 0x222222F2,
	0xF2222220, 0x22222222, 0x22F22222, 0x00000222,
	0x99000000, 0x9B9B99BB, 0x9BB99999, 0x9999B9B9,
	0x9B99BB90, 0x9BBBBB9B, 0x9B9B9BB9, 0x00000999,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0x88888000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00AAA888,
	0x88A88A00, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA88, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA88, 0x8AAAAA8A, 0x8A8A8AA8, 0x08AAA888,
	0x11000000, 0x1111A111, 0x11111111, 0x111111A1,
	0xA1111110, 0x11111111, 0x11C11111, 0x00000111,
	0x11000000, 0x1111A111, 0x11111111, 0x111111A1,
	0xA1111110, 0x11111111, 0x11C11111, 0x00000111,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0x88000000, 0x8A8A88AA, 0x8AA88888, 0x8888A8A8,
	0x8A88AA80, 0x8AAAAA8A, 0x8A8A8AA8, 0x00000888,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
};

static inline void assert_ntab_array_sizes(void)
{
#undef check
#define check(table, size)	\
	BUILD_BUG_ON(ARRAY_SIZE(b43_ntab_##table) != B43_NTAB_##size##_SIZE)

	check(adjustpower0, C0_ADJPLT);
	check(adjustpower1, C1_ADJPLT);
	check(bdi, BDI);
	check(channelest, CHANEST);
	check(estimatepowerlt0, C0_ESTPLT);
	check(estimatepowerlt1, C1_ESTPLT);
	check(framelookup, FRAMELT);
	check(framestruct, FRAMESTRUCT);
	check(gainctl0, C0_GAINCTL);
	check(gainctl1, C1_GAINCTL);
	check(intlevel, INTLEVEL);
	check(iqlt0, C0_IQLT);
	check(iqlt1, C1_IQLT);
	check(loftlt0, C0_LOFEEDTH);
	check(loftlt1, C1_LOFEEDTH);
	check(mcs, MCS);
	check(noisevar10, NOISEVAR10);
	check(noisevar11, NOISEVAR11);
	check(pilot, PILOT);
	check(pilotlt, PILOTLT);
	check(tdi20a0, TDI20A0);
	check(tdi20a1, TDI20A1);
	check(tdi40a0, TDI40A0);
	check(tdi40a1, TDI40A1);
	check(tdtrn, TDTRN);
	check(tmap, TMAP);

#undef check
}

void b43_ntab_write(struct b43_wldev *dev, u32 offset, u32 value)
{
	u32 type;

	type = offset & B43_NTAB_TYPEMASK;
	offset &= 0xFFFF;

	switch (type) {
	case B43_NTAB_8BIT:
		B43_WARN_ON(value & ~0xFF);
		b43_phy_write(dev, B43_NPHY_TABLE_ADDR, offset);
		b43_phy_write(dev, B43_NPHY_TABLE_DATALO, value);
		break;
	case B43_NTAB_16BIT:
		B43_WARN_ON(value & ~0xFFFF);
		b43_phy_write(dev, B43_NPHY_TABLE_ADDR, offset);
		b43_phy_write(dev, B43_NPHY_TABLE_DATALO, value);
		break;
	case B43_NTAB_32BIT:
		b43_phy_write(dev, B43_NPHY_TABLE_ADDR, offset);
		b43_phy_write(dev, B43_NPHY_TABLE_DATAHI, value >> 16);
		b43_phy_write(dev, B43_NPHY_TABLE_DATALO, value & 0xFFFF);
		break;
	default:
		B43_WARN_ON(1);
	}

	return;

	/* Some compiletime assertions... */
	assert_ntab_array_sizes();
}
