/*
 * linux/include/asm-arm/arch-oxnas/dma.h
 *
 * Copyright (C) 2005,2010 Oxford Semiconductor Ltd
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * We have a generic DMAC that is a two port, memory to memory design, supporting
 * multiple channels, each of which can transfer between any pair of memory
 * regions. This DMAC architecture does not sit well with the std. ARM DMA
 * architecture, thus we define a custom way of acquiring and operating on the
 * available channels of the DMAC
 */

#ifndef __ASM_ARCH_DMA_H
#define __ASM_ARCH_DMA_H

#include <asm/scatterlist.h>
#include <linux/semaphore.h>
#include <linux/ata.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <mach/desc_alloc.h>

#undef DEBUG_PRD_ALLOC

/* All memory DMAable */
#define MAX_DMA_ADDRESS (~0UL)

/* Do not want to use the std. ARM DMA architecure */
#define MAX_DMA_CHANNELS 0

#define MAX_OXNAS_DMA_CHANNELS 5

#define MAX_OXNAS_DMA_TRANSFER_LENGTH ((1 << 21) - 1)

//#define OXNAS_DMA_TEST
//#define OXNAS_DMA_TEST_ITERATIONS 1
//#define OXNAS_DMA_SG_TEST
//#define OXNAS_DMA_SG_TEST_2
//#define OXNAS_DMA_SG_TEST_ITERATIONS 1
//#define OXNAS_DMA_SG_TEST2_ITERATIONS 1
//#define OXNAS_DMA_SG_TEST_DUMP_BUFFERS
//#define OXNAS_DMA_SG_TEST_DUMP_DESCRIPTORS
//#define OXNAS_DMA_TEST_AFTER_SG_ITERATIONS 0
//#define OXNAS_DMA_OVERALL_TEST_LOOPS 1

// All other error codes are generated by the SG engine - refer to its
// documentation for details
typedef enum oxnas_dma_callback_status {
    OXNAS_DMA_ERROR_CODE_NONE
} oxnas_dma_callback_status_t;

typedef enum oxnas_dma_mode {
    OXNAS_DMA_MODE_FIXED,
    OXNAS_DMA_MODE_INC
} oxnas_dma_mode_t;

typedef enum oxnas_dma_direction {
    OXNAS_DMA_TO_DEVICE,
    OXNAS_DMA_FROM_DEVICE
} oxnas_dma_direction_t;

struct oxnas_dma_channel;
typedef struct oxnas_dma_channel oxnas_dma_channel_t;

#define OXNAS_DMA_CHANNEL_NUL ((oxnas_dma_channel_t*)0)

typedef void* oxnas_callback_arg_t;

#define OXNAS_DMA_CALLBACK_ARG_NUL ((oxnas_callback_arg_t)0)

typedef void (*oxnas_dma_callback_t)(oxnas_dma_channel_t*, oxnas_callback_arg_t, oxnas_dma_callback_status_t, int interrupt_count);

#define OXNAS_DMA_CALLBACK_NUL ((oxnas_dma_callback_t)0)

typedef enum oxnas_dma_eot_type {
    OXNAS_DMA_EOT_NONE,
    OXNAS_DMA_EOT_ALL,
    OXNAS_DMA_EOT_FINAL
} oxnas_dma_eot_type_t;

// Will be exchanged with SG DMA controller
typedef struct oxnas_dma_sg_entry {
    dma_addr_t                 addr_;   // The physical address of the buffer described by this descriptor
    unsigned long              length_; // The length of the buffer described by this descriptor
    dma_addr_t                 p_next_; // The physical address of the next descriptor
    struct oxnas_dma_sg_entry *v_next_; // The virtual address of the next descriptor
    dma_addr_t                 paddr_;  // The physical address of this descriptor
    struct oxnas_dma_sg_entry *next_;   // To allow insertion into single-linked list
} oxnas_dma_sg_entry_t;

// Will be exchanged with SG DMA controller
typedef struct oxnas_dma_sg_info {
    unsigned long         qualifer_;
    unsigned long         control_;
    dma_addr_t            p_srcEntries_; // The physical address of the first source SG descriptor
    dma_addr_t            p_dstEntries_; // The physical address of the first destination SG descriptor
    oxnas_dma_sg_entry_t *v_srcEntries_; // The virtual address of the first source SG descriptor
    oxnas_dma_sg_entry_t *v_dstEntries_; // The virtual address of the first destination SG descriptor
} oxnas_dma_sg_info_t;

struct oxnas_dma_channel {
    unsigned                     channel_number_;
    oxnas_dma_callback_t         notification_callback_;
    oxnas_callback_arg_t         notification_arg_;
    dma_addr_t                   p_sg_info_;    // Physical address of sg_info structure
    oxnas_dma_sg_info_t         *v_sg_info_;    // Virtual address of sg_info structure
    oxnas_dma_callback_status_t  error_code_;
    unsigned                     rps_interrupt_;
    struct oxnas_dma_channel    *next_;
    struct semaphore             default_semaphore_;
    atomic_t                     interrupt_count_;
    atomic_t                     active_count_;
	int							  auto_sg_entries_;
};

typedef struct oxnas_dma_controller {
    oxnas_dma_channel_t        channels_[MAX_OXNAS_DMA_CHANNELS];
    unsigned                   numberOfChannels_;
    int                        version_;
    atomic_t                   run_bh_;
    spinlock_t                 spinlock_;
    struct                     tasklet_struct tasklet_;
    dma_addr_t                 p_sg_infos_;		// Physical address of the array of sg_info structures
    oxnas_dma_sg_info_t       *v_sg_infos_;		// Virtual address of the array of sg_info structures
    struct semaphore           csum_engine_sem_;
    spinlock_t                 alloc_spinlock_;	// Sync. for SG management
    spinlock_t                 channel_alloc_spinlock_;	// Sync. for channel management
    oxnas_dma_sg_entry_t      *sg_entry_head_;	// Pointer to head of free list for oxnas_dma_sg_entry_t objects
    struct semaphore           sg_entry_sem_;
    unsigned                   sg_entry_available_;
    oxnas_dma_channel_t       *channel_head_;
    struct semaphore           channel_sem_;
#ifdef CONFIG_OXNAS_ODRB_DMA_SUPPORT
	struct list_head           odrb_sata_sg_list_head_;	// List of free SG entry lists
	unsigned                   odrb_sata_num_free_sg_lists_;

#ifdef CONFIG_ODRB_USE_PRDS
    struct list_head           odrb_prd_array_head_;	// List of free PRD arrays
#ifdef DEBUG_PRD_ALLOC
    struct list_head           odrb_prd_inuse_array_head_;	// List of in-use PRD arrays
#endif // DEBUG_PRD_ALLOC
    unsigned                   odrb_num_free_prd_arrays_;
	wait_queue_head_t          odrb_prd_array_wait_queue_;
	struct list_head           odrb_reader_prd_array_head_;	// List of free PRD arrays
    unsigned                   odrb_reader_num_free_prd_arrays_;
#else // CONFIG_ODRB_USE_PRDS
	struct list_head           odrb_sg_list_head_;	// List of free SG entry lists
    unsigned                   odrb_num_free_sg_lists_;
	wait_queue_head_t          odrb_sg_list_wait_queue_;
	struct list_head           odrb_reader_sg_list_head_;	// List of free SG entry lists
	unsigned                   odrb_reader_num_free_sg_lists_;
#endif // CONFIG_ODRB_USE_PRDS
#endif // CONFIG_OXNAS_ODRB_DMA_SUPPORT
} oxnas_dma_controller_t;

typedef struct oxnas_dma_device_settings {
    unsigned long address_;
    unsigned      fifo_size_;   // Chained transfers must take account of FIFO offset at end of previous transfer
    unsigned char dreq_;
    unsigned      read_eot_policy_:2;
    unsigned      write_eot_policy_:2;
    unsigned      bus_:1;
    unsigned      width_:2;
    unsigned      transfer_mode_:1;
    unsigned      address_mode_:1;
    unsigned      address_really_fixed_:1;
} oxnas_dma_device_settings_t;

/* Pre-defined settings for known DMA devices */
extern oxnas_dma_device_settings_t oxnas_pata_dma_settings;
extern oxnas_dma_device_settings_t oxnas_sata_dma_settings;
extern oxnas_dma_device_settings_t oxnas_dpe_rx_dma_settings;
extern oxnas_dma_device_settings_t oxnas_dpe_tx_dma_settings;

extern void oxnas_dma_init(void);

extern void oxnas_dma_shutdown(void);

extern oxnas_dma_channel_t* oxnas_dma_request(int block);

extern void oxnas_dma_free(oxnas_dma_channel_t*);

extern int oxnas_dma_is_active(oxnas_dma_channel_t*);

extern int oxnas_dma_raw_isactive(oxnas_dma_channel_t*);

extern int oxnas_dma_raw_sg_isactive(oxnas_dma_channel_t*);

extern int oxnas_dma_get_raw_direction(oxnas_dma_channel_t*);

/**
 * @return An int which is zero on success. Non-zero will returned if the length
 *         exceeds that allowed by the hardware
 */
extern int oxnas_dma_set(
    oxnas_dma_channel_t *channel,
    unsigned char       *src_adr,   // Physical address
    unsigned long        length,
    unsigned char       *dst_adr,   // Physical address
    oxnas_dma_mode_t     src_mode,
    oxnas_dma_mode_t     dst_mode,
    int                  paused);

/**
 * @return An int which is zero on success. Non-zero will returned if the length
 *         exceeds that allowed by the hardware
 */
extern int oxnas_dma_device_set(
    oxnas_dma_channel_t         *channel,
    oxnas_dma_direction_t        direction,
    unsigned char               *mem_adr,   // Physical address
    unsigned long                length,
    oxnas_dma_device_settings_t *device_settings,
    oxnas_dma_mode_t             mem_mode,
    int                          paused);

/**
 * @return An int which is zero on success. Non-zero will returned if the length
 *         exceeds that allowed by the hardware
 */
extern int oxnas_dma_device_pair_set(
    oxnas_dma_channel_t         *channel,
    unsigned long                length,
    oxnas_dma_device_settings_t *src_device_settings,
    oxnas_dma_device_settings_t *dst_device_settings,
    int                          paused);

/**
 *  NB This function does not have an error return value, but if it is passed
 *     a transfer segment longer than the maximum allowed by the hardware, that
 *     entry will be zeroed in the SG descriptor list and a kernel warning
 *     message generated
 */
extern int oxnas_dma_set_sg(
    oxnas_dma_channel_t* channel,
    struct scatterlist*  src_sg,
    unsigned             src_sg_count,
    struct scatterlist*  dst_sg,
    unsigned             dst_sg_count,
    oxnas_dma_mode_t     src_mode,
    oxnas_dma_mode_t     dst_mode,
	int                  in_atomic);

/**
 *  NB This function does not have an error return value, but if it is passed
 *     a transfer segment longer than the maximum allowed by the hardware, that
 *     entry will be zeroed in the SG descriptor list and a kernel warning
 *     message generated
 */
extern int oxnas_dma_device_set_sg(
    oxnas_dma_channel_t*         channel,
    oxnas_dma_direction_t        direction,
    struct scatterlist*          mem_sg,
    unsigned                     mem_sg_count,
    oxnas_dma_device_settings_t* device_settings,
    oxnas_dma_mode_t             mem_mode,
	int                          in_atomic);

extern int oxnas_dma_device_set_prd(
    oxnas_dma_channel_t			 *channel,
    oxnas_dma_direction_t        direction,
	struct ata_prd				 *prd,
    oxnas_dma_device_settings_t *device_settings,
    oxnas_dma_mode_t             mem_mode,
	oxnas_dma_sg_entry_t		 *sg_entries);

/**
 * The callback function should complete quickly and must not sleep
 */
extern void oxnas_dma_set_callback(
    oxnas_dma_channel_t*,
    oxnas_dma_callback_t callback,
    oxnas_callback_arg_t callback_arg);

extern void oxnas_dma_abort(oxnas_dma_channel_t*);

extern void oxnas_dma_start(oxnas_dma_channel_t*);

extern void oxnas_dma_dump_registers(void);

extern void oxnas_dma_dump_registers_single(int channel_number);

extern int oxnas_dma_alloc_sg_entries(oxnas_dma_sg_entry_t** entries, unsigned required, int in_atomic);

extern void oxnas_dma_free_sg_entries(oxnas_dma_sg_entry_t* entries);

#ifdef CONFIG_OXNAS_ODRB_DMA_SUPPORT

typedef struct odrb_sg_entry {
    dma_addr_t    addr_;	// Double up as page
    unsigned long length_;
    dma_addr_t    next_;	// Double up of offset
} odrb_sg_entry_t;
/* this defines the size of the structure - needs to be modified if the structure is modified */
#define SIZE_OF_SG_ENTRY (3 * 4)

typedef struct odrb_sg_list {
	struct list_head head;
	dma_addr_t       phys;
	odrb_sg_entry_t *sg_entries;
} odrb_sg_list_t;

#ifdef CONFIG_ODRB_USE_PRDS
typedef struct odrb_dma_prd_list {
	struct list_head   head;
	dma_addr_t         phys;
	prd_table_entry_t *prds;
#ifdef DEBUG_PRD_ALLOC
	const char        *file;
	int                line;
	unsigned long      jiffies;
#endif // DEBUG_PRD_ALLOC
} odrb_prd_list_t;
#endif // CONFIG_ODRB_USE_PRDS

#ifdef CONFIG_ODRB_USE_PRDS_FOR_SATA
#define SATA_LISTS_SIZE 0
#else // CONFIG_ODRB_USE_PRDS_FOR_SATA
#define SATA_LISTS_SIZE (CONFIG_ODRB_NUM_SATA_SG_ENTRIES * CONFIG_ODRB_NUM_SATA_SG_LISTS * SIZE_OF_SG_ENTRY)
#endif // CONFIG_ODRB_USE_PRDS_FOR_SATA

#ifdef CONFIG_ODRB_USE_PRDS
#define PRD_LISTS_SIZE ( \
	((CONFIG_ODRB_WRITER_PRD_ARRAY_SIZE * CONFIG_ODRB_NUM_WRITER_PRD_ARRAYS) + \
	 (CONFIG_ODRB_READER_PRD_ARRAY_SIZE * CONFIG_ODRB_NUM_READER_PRD_ARRAYS)) * \
	SIZE_OF_PRD_ENTRY)

#if ((PRD_LISTS_SIZE + SATA_LISTS_SIZE + SIZE_OF_SG_ENTRY + SIZE_OF_SIMPLE_SG_INFO) > DMA_DESC_ALLOC_SIZE)
#error "Too many PRD and SG lists - descriptor SRAM allocation exceeded"
#endif
#else // CONFIG_ODRB_USE_PRDS
#ifdef OXNAS_FAST_READS_AND_WRITES
#define SG_LISTS_SIZE ( \
	((CONFIG_ODRB_NUM_WRITER_SG_LISTS * CONFIG_ODRB_NUM_WRITER_SG_LISTS) + \
	 (CONFIG_ODRB_NUM_READER_SG_LISTS * CONFIG_ODRB_NUM_READER_SG_LISTS)) * \
	SIZE_OF_SG_ENTRY)
#else // OXNAS_FAST_READS_AND_WRITES
#define SG_LISTS_SIZE 0
#endif // OXNAS_FAST_READS_AND_WRITES

#if ((SG_LISTS_SIZE + SATA_LISTS_SIZE + SIZE_OF_SG_ENTRY + SIZE_OF_SIMPLE_SG_INFO) > DMA_DESC_ALLOC_SIZE)
#error "Too many SG lists - descriptor SRAM allocation exceeded"
#endif
#endif // CONFIG_ODRB_USE_PRDS

extern void odrb_dma_sata_single(
	oxnas_dma_direction_t dir,
	dma_addr_t            adr,
	unsigned long         len);

extern int odrb_dma_isactive(int is_sg);
extern void odrb_dma_abort(int is_sg);

extern void odrb_dma_postop_housekeeping(int is_sg);

extern void odrb_dump_dma_regs(void);
extern void odrb_dump_sg_info(void);
extern void odrb_dma_dump_sg_regs(void);
extern void odrb_decode_sg_error(void);

#if !defined(CONFIG_ODRB_USE_PRDS_FOR_SATA) || !defined(CONFIG_ODRB_USE_PRDS)
/* Support for prepared SG entries */
extern void odrb_dma_sata_sq(oxnas_dma_direction_t dir, unsigned long nsects, dma_addr_t sg_phys, int legacy);
extern void odrb_dma_sata_sq_nogo(oxnas_dma_direction_t dir, unsigned long nsects, dma_addr_t sg_phys);
extern void odrb_dma_sata_sq_go(void);
#endif

#ifdef CONFIG_ODRB_USE_PRDS_FOR_SATA
#else // CONFIG_ODRB_USE_PRDS_FOR_SATA
/* Alloc/free of SG entries for exclusive use of SCSI/SATA stack driver */
extern int odrb_alloc_sata_sg_list(odrb_sg_list_t **lists, int count);
extern void odrb_free_sata_sg_list(odrb_sg_list_t *sg_list);
#endif // CONFIG_ODRB_USE_PRDS_FOR_SATA

#define SECTOR_SHIFT 9
#ifdef CONFIG_ODRB_USE_PRDS
/* Support for prepared PRDs */
#define ODRB_NUM_SATA_PRD_ENTRIES	((MAX_OXNAS_DMA_TRANSFER_LENGTH + 1) / PRD_MAX_LEN)
extern prd_table_entry_t *odrb_sata_prd_entry;
extern dma_addr_t         odrb_sata_prd_entry_phys;

static inline dma_addr_t odrb_dma_prepare_sata_prd_table(unsigned long nsects)
{
	prd_table_entry_t *prd = odrb_sata_prd_entry;
	unsigned long len = nsects << SECTOR_SHIFT;

	BUG_ON(len > MAX_OXNAS_DMA_TRANSFER_LENGTH);

	// Fill sufficient PRD entries to describe the total transfer length to the
	// SATA side of the DMA controller
	while (len > PRD_MAX_LEN) {
		prd++->flags_len = 0;	// 64K encoded in PRD entry as zero
		len -= PRD_MAX_LEN;
	}
	if (len == PRD_MAX_LEN) {
		prd->flags_len = PRD_EOF_MASK;
	} else {
		prd->flags_len = (len | PRD_EOF_MASK);
	}

	return odrb_sata_prd_entry_phys;
}

extern void odrb_dma_sata_prd(oxnas_dma_direction_t dir, unsigned long nsects, dma_addr_t prds_phys, int legacy);
extern void odrb_dma_sata_prd_nogo(oxnas_dma_direction_t dir, unsigned long nsects, dma_addr_t prds_phys);
extern void odrb_dma_sata_prd_go(void);

extern int __odrb_alloc_prd_array(
	odrb_prd_list_t **lists,
	int               count,
	int               may_sleep
#ifdef DEBUG_PRD_ALLOC
	,const char       *file,
	int               line
#endif // DEBUG_PRD_ALLOC
	);

#ifdef DEBUG_PRD_ALLOC
#define odrb_alloc_prd_array(lists, count, may_sleep) __odrb_alloc_prd_array((lists), (count), (may_sleep), __FILE__, __LINE__)
#else // DEBUG_PRD_ALLOC
#define odrb_alloc_prd_array(lists, count, may_sleep) __odrb_alloc_prd_array((lists), (count), (may_sleep))
#endif // DEBUG_PRD_ALLOC

extern void odrb_free_prd_array(odrb_prd_list_t *prd_list);
extern int odrb_reader_alloc_prd_array(odrb_prd_list_t **list);
extern void odrb_reader_free_prd_array(odrb_prd_list_t *prd_list);
#else // CONFIG_ODRB_USE_PRDS
extern int odrb_alloc_sg_list(odrb_sg_list_t **lists, int count, int may_sleep);
extern void odrb_free_sg_list(odrb_sg_list_t *sg_list);
extern int odrb_reader_alloc_sg_list(odrb_sg_list_t **list);
extern void odrb_reader_free_sg_list(odrb_sg_list_t *sg_list);
#endif // CONFIG_ODRB_USE_PRDS

#endif // CONFIG_OXNAS_ODRB_DMA_SUPPORT

#endif // __ASM_ARCH_DMA_H
