#include <Copyright.h>

/*******************************************************************************
* gtMisc.c
*
* DESCRIPTION:
*       API definitions for Ip Mapping Table
*							EEPROM access
*							Scratch and Misc Control
* DEPENDENCIES:
*
* FILE REVISION NUMBER:
*******************************************************************************/

#include <msApi.h>
#include <gtSem.h>
#include <gtHwCntl.h>
#include <gtDrvSwRegs.h>

/*******************************************************************************
* gsysSetUseIpMappingTable
*
* DESCRIPTION:
*       This API set to use IP Frame Priorities from this table. 
*		Set GT_TRUE:  The IP_FPRI data in this table is used as the frames 
*			initial IP_FPRI use Ip Mappingg table priorities.
*		Set GT_FALSE: The IP_FPRI data in this table is ignored. Instead the 
*			frames initial IP_FPRI is generated by using the frames IP_QPRI
*			as the IP_FPRIs upper two bits, and the IP_FPRIs lowest bit comes 
*			from bit 0 of the frames source ports Default PRI (Port offset 0x07).
*
* INPUTS:
*		en	- [GT_TRUE] / [GT_FALSE]
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetUseIpMappingTable
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_BOOL			en
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U16		data;

    DBG_INFO(("gsysSetUseIpMappingTable Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_IP_MAPPING_TABLE))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	data = (en==GT_TRUE)?1:0;

    retVal = hwSetGlobalRegField(dev,QD_REG_IP_MAPPING_TABLE,14,1,data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;
}

/*******************************************************************************
* gsysGetUseIpMappingTable
*
* DESCRIPTION:
*       This API get to use IP Frame Priorities from this table. 
*		Set GT_TRUE:  The IP_FPRI data in this table is used as the frames 
*			initial IP_FPRI use Ip Mappingg table priorities.
*		Set GT_FALSE: The IP_FPRI data in this table is ignored. Instead the 
*			frames initial IP_FPRI is generated by using the frames IP_QPRI
*			as the IP_FPRIs upper two bits, and the IP_FPRIs lowest bit comes 
*			from bit 0 of the frames source ports Default PRI (Port offset 0x07).
*
* INPUTS:
*       None.
*
* OUTPUTS:
*		en	- [GT_TRUE] / [GT_FALSE]
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetUseIpMappingTable
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_BOOL			*en
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U16		data;

    DBG_INFO(("gsysGetUseIpMappingTable Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_IP_MAPPING_TABLE))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

    retVal = hwGetGlobalRegField(dev,QD_REG_IP_MAPPING_TABLE,14,1,&data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	*en= (data==1)?GT_TRUE:GT_FALSE;

	return GT_OK;
}

/*******************************************************************************
* gsysSetIpMappingPrio
*
* DESCRIPTION:
*       Set IPv4 and IPv6 Frame Priority Mapping, and 
*			IPv4 and IPv6 Queue Priority Mapping.
*           The ipFpri value is used as the frames initial FPRI when the frame is 
*			an IPv4 or an IPv6 frame, and the ports InitialPri (Port offset 0x04) 
*			is configured to use IP FPris.
*           The ipQpri value is used as the frames initial QPRI when the frame is 
*			an IPv4 or an IPv6 frame, and the ports InitialPri and TagIfBoth 
*			registers (Port offset 0x04) are configured to use IP QPris.
*
* INPUTS:
*       point - Pointer to the Ip Mapping Table.
*				  0 - 0x3f;
*       ipFpri -  The value is 0 - 7
*       ipQpri -  The value is 0 - 3.
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetIpMappingPrio
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			point,
    IN  GT_U8			ipFpri,
    IN  GT_U8			ipQpri
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U16		data;

    DBG_INFO(("gsysSetIpMappingPrio Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_IP_MAPPING_TABLE))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if ((point>0x3f)||(ipFpri>7)||(ipQpri>3))
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	gtSemTake(dev,dev->tblRegsSem,OS_WAIT_FOREVER);

    /* Wait until the Scratch and Misc control is ready. */
   	data = 1;
    while(data == 1)
   	{
        retVal = hwGetGlobalRegField(dev,QD_REG_IP_MAPPING_TABLE,15,1,&data);
   	    if(retVal != GT_OK)
       	{
			gtSemGive(dev,dev->tblRegsSem);
   	        return retVal;
       	}
    }

	data = (((ipFpri&7)<<4) | (ipQpri&3));
	data |=  ((GT_U16)((1 << 15) | (point << 8)));

	retVal = hwWriteGlobalReg(dev, QD_REG_IP_MAPPING_TABLE, data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
        gtSemGive(dev,dev->tblRegsSem);
   	    return retVal;
    }

	gtSemGive(dev,dev->tblRegsSem);

    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;
}

/*******************************************************************************
* gsysGetIpMappingPrio
*
* DESCRIPTION:
*       get IPv4 and IPv6 Frame Priority Mapping, and 
*			IPv4 and IPv6 Queue Priority Mapping.
*           The ipFpri value is used as the frames initial FPRI when the frame is 
*			an IPv4 or an IPv6 frame, and the ports InitialPri (Port offset 0x04) 
*			is configured to use IP FPris.
*           The ipQpri value is used as the frames initial QPRI when the frame is 
*			an IPv4 or an IPv6 frame, and the ports InitialPri and TagIfBoth 
*			registers (Port offset 0x04) are configured to use IP QPris.
*
* INPUTS:
*       point - Pointer to the Ip Mapping Table.
*				  0 - 0x3f;
*
* OUTPUTS:
*       ipFpri -  The value is 0 - 7
*       ipQpri -  The value is 0 - 3.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetIpMappingPrio
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			point,
    OUT  GT_U8			*ipFpri,
    OUT  GT_U8			*ipQpri
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U16		data;

    DBG_INFO(("gsysGetIpMappingPrio Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_IP_MAPPING_TABLE))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if (point > 0x3f)
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	gtSemTake(dev,dev->tblRegsSem,OS_WAIT_FOREVER);

	do {
		retVal = hwReadGlobalReg(dev, QD_REG_IP_MAPPING_TABLE, &data);
		if(retVal != GT_OK)
   		{
   			DBG_INFO(("Failed.\n"));
			gtSemGive(dev,dev->tblRegsSem);
   			return retVal;
		}
	} while (data&0x8000);

	*ipFpri = (data >> 4) & 7;
	*ipQpri = (data) & 3;

	gtSemGive(dev,dev->tblRegsSem);

    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;
}

/*******************************************************************************
* eepromOperationPerform
*
* DESCRIPTION:
*       This function accesses EEPROM Command Register and Data Register.
*       The device supports the following EEPROM 
*			operations
*			GT_EEPROM_NO_OP = No Operation
*			GT_EEPROM_WRITE_DATA = Write EEPROM at Addr.
*			GT_EEPROM_READ_DATA = Read EEPROM from Addr.
*			GT_EEPROM_RESTART = Restart Register Loader execution at Addr 
*				(eepromData = dont care in this case)
*			GT_EEPROM_HALT = Halt (stop executing the EEPROM if its not already 
*				stopped)
*
* INPUTS:
*       eepromOp      - EEPROM Opcode.
*       eepromData    - Data to be written to the EEPROM 
*
* OUTPUTS:
*       eepromData    - Data that was read back from the EEPROM. 
*
Command register above.
*
* RETURNS:
*       GT_OK on success,
*       GT_FAIL otherwise.
*       GT_BAD_PARAM - if input parameters are beyond range.
*
* COMMENTS:
*
*******************************************************************************/
static GT_STATUS eepromOperationPerform
(
    IN    GT_QD_DEV 			*dev,
    IN    GT_EEPROM_OPERATION	eepromOp,
    INOUT GT_EEPROM_OP_DATA		*opData
)
{
    GT_STATUS       retVal;	/* Functions return value */
    GT_U16          data; 	/* temporary Data storage */
    DBG_INFO(("eepromOperationPerform Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_EEPROM))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if (eepromOp>GT_EEPROM_HALT)
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

    gtSemTake(dev,dev->eepromRegsSem,OS_WAIT_FOREVER);

    /* Wait until the eeprom in ready. */
    data = 1;
    while(data == 1)
    {
        retVal = hwGetGlobal2RegField(dev,QD_REG_EEPROM_COMMAND,15,1,&data);
        if(retVal != GT_OK)
        {
            gtSemGive(dev,dev->eepromRegsSem);
            return retVal;
        }
    }

    /* Set the EEPROM Operation register */
	switch (eepromOp)
	{
		case GT_EEPROM_WRITE_DATA:
		    retVal = hwGetGlobal2RegField(dev,QD_REG_EEPROM_COMMAND,10,1,&data);
		    if(retVal != GT_OK)
		    {
		        gtSemGive(dev,dev->eepromRegsSem);
		        return retVal;
			}
			if (data==0)
		    {
		        gtSemGive(dev,dev->eepromRegsSem);
				DBG_INFO(("EEPROM is not writablen"));
		        return GT_FAIL;
			}

		    retVal = hwGetGlobal2RegField(dev,QD_REG_EEPROM_COMMAND,11,1,&data);
		    if(retVal != GT_OK)
		    {
		        gtSemGive(dev,dev->eepromRegsSem);
		        return retVal;
			}
			if (data==1)
		    {
		        gtSemGive(dev,dev->eepromRegsSem);
				DBG_INFO(("EEPROM Loader is running"));
		        return GT_FAIL;
			}

			data = (GT_U16)opData->eepromData;
			retVal = hwWriteGlobal2Reg(dev,QD_REG_EEPROM_DATA,data);
	        if(retVal != GT_OK)
    	    {
        	    gtSemGive(dev,dev->eepromRegsSem);
            	return retVal;
	        }

			data = (GT_U16)((1 << 15) | (GT_EEPROM_WRITE_DATA << 12) | 
					(opData->eepromAddr & 0xFF));
			retVal = hwWriteGlobal2Reg(dev,QD_REG_EEPROM_COMMAND,data);
	        if(retVal != GT_OK)
    	    {
        	    gtSemGive(dev,dev->eepromRegsSem);
            	return retVal;
	        }
			break;

		case GT_EEPROM_READ_DATA:
		    retVal = hwGetGlobal2RegField(dev,QD_REG_EEPROM_COMMAND,11,1,&data);
		    if(retVal != GT_OK)
		    {
		        gtSemGive(dev,dev->eepromRegsSem);
		        return retVal;
			}
			if (data==1)
		    {
		        gtSemGive(dev,dev->eepromRegsSem);
				DBG_INFO(("EEPROM Loader is running"));
		        return GT_FAIL;
			}

			data = (GT_U16)((1 << 15) | (GT_EEPROM_READ_DATA << 12) | 
					(opData->eepromAddr & 0xFF));
			retVal = hwWriteGlobal2Reg(dev,QD_REG_EEPROM_COMMAND,data);
	        if(retVal != GT_OK)
    	    {
        	    gtSemGive(dev,dev->eepromRegsSem);
            	return retVal;
	        }

		    data = 1;
		    while(data == 1)
		    {
		        retVal = hwGetGlobal2RegField(dev,QD_REG_EEPROM_COMMAND,15,1,&data);
		        if(retVal != GT_OK)
		        {
		            gtSemGive(dev,dev->eepromRegsSem);
		            return retVal;
        		}
		    }
			
			retVal = hwReadGlobal2Reg(dev,QD_REG_EEPROM_DATA,&data);
			opData->eepromData = (GT_U32)data;

			break;

		case GT_EEPROM_RESTART:
			data = (GT_U16)((1 << 15) | (GT_EEPROM_RESTART << 12) | 
					(opData->eepromAddr & 0xFF));
			retVal = hwWriteGlobal2Reg(dev,QD_REG_EEPROM_COMMAND,data);
	        if(retVal != GT_OK)
    	    {
        	    gtSemGive(dev,dev->eepromRegsSem);
            	return retVal;
	        }

			break;

		case GT_EEPROM_HALT:
			data = (GT_U16)((1 << 15) | (GT_EEPROM_HALT << 12) | 
					(opData->eepromAddr & 0xFF));
			retVal = hwWriteGlobal2Reg(dev,QD_REG_EEPROM_COMMAND,data);
	        if(retVal != GT_OK)
    	    {
        	    gtSemGive(dev,dev->eepromRegsSem);
            	return retVal;
	        }

			break;
		default:
			gtSemGive(dev,dev->eepromRegsSem);
			return GT_FAIL;
	}

	gtSemGive(dev,dev->eepromRegsSem);
    return GT_OK;
}

/*******************************************************************************
* gsysReadEeprom
*
* DESCRIPTION:
*       Read EEPROM from EEPROMs address where the EEOp is performed.
*
* INPUTS:
*       addr - EEPROM Address. 
*
* OUTPUTS:
*       data -  Data that was read back from the EEPROM.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysReadEeprom
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			addr,
    OUT  GT_U8			*data
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    GT_EEPROM_OPERATION	eepromOp;
    GT_EEPROM_OP_DATA	opData;
 
	eepromOp = GT_EEPROM_READ_DATA;
	opData.eepromAddr = addr;

	retVal = eepromOperationPerform(dev,eepromOp,&opData);
    if(retVal != GT_OK)
    {
      return retVal;
    }
	*data = (GT_U8)opData.eepromData;
	return GT_OK;

}

/*******************************************************************************
* gsysWriteEeprom
*
* DESCRIPTION:
*       Write EEPROM at the EEPROMs address where the EEOp is performed.

*
* INPUTS:
*       addr - EEPROM Address. 
*       data - Data to be written to the EEPROM
*
* OUTPUTS:
*		None
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysWriteEeprom
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			addr,
    IN  GT_U8			data
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    GT_EEPROM_OPERATION	eepromOp;
    GT_EEPROM_OP_DATA	opData;
 
	eepromOp = GT_EEPROM_WRITE_DATA;
	opData.eepromAddr = addr;
	opData.eepromData = data;

	retVal = eepromOperationPerform(dev,eepromOp,&opData);
    if(retVal != GT_OK)
    {
      return retVal;
    }
	return GT_OK;

}

/*******************************************************************************
* gsysRestartEeprom
*
* DESCRIPTION:
*       Restart Register Loader execution at the EEPROMs address where the EEOp 
*		is performed
*
* INPUTS:
*       addr - EEPROM Address. .
*
* OUTPUTS:
*       none   
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysRestartEeprom
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			addr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    GT_EEPROM_OPERATION	eepromOp;
    GT_EEPROM_OP_DATA	opData;
 
	eepromOp = GT_EEPROM_RESTART;
	opData.eepromAddr = addr;

	retVal = eepromOperationPerform(dev,eepromOp,&opData);
    if(retVal != GT_OK)
    {
      return retVal;
    }
	return GT_OK;

}

/*******************************************************************************
* gsysHaltEeprom
*
* DESCRIPTION:
*       Halt (stop executing the EEPROM if its not already stopped)

*
* INPUTS:
*       none  .
*
* OUTPUTS:
*       none   
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysHaltEeprom
(
    IN  GT_QD_DEV 		*dev
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    GT_EEPROM_OPERATION	eepromOp;
    GT_EEPROM_OP_DATA	opData;
 
	eepromOp = GT_EEPROM_HALT;

	retVal = eepromOperationPerform(dev,eepromOp,  &opData);
    if(retVal != GT_OK)
    {
      return retVal;
    }
	return GT_OK;

}

/*******************************************************************************
* gsysGetStEeprom
*
* DESCRIPTION:
*       Get EEPROM status. They are Register Loader Running status and EEPROM 
*		Write Enable status
*       runSt is GT_TRUE: Register Loader Running, whenever the register loader 
*			is busy executing the instructions contained in the EEPROM.
*       writeEn is GT_TRUE: EEPROM Write Enable, that indicates that writing to 
*			the EEPROM is possible. 
*		writeEn is GT_FALSE: the Write EEPROM EEOp above will not do anything.
*			This reflects the value of the EE_WE configuration pin after Reset.
*
* INPUTS:
*       none  .
*
* OUTPUTS:
*       runSt   -   [GT_TRUE] / [GT_FALSE)
*       writeEn -   [GT_TRUE] / [GT_FALSE)
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetStEeprom
(
    IN  GT_QD_DEV 		*dev,
	OUT GT_BOOL			*writeEn,
	OUT GT_BOOL			*runSt
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    GT_U16      data; 			/* temporary Data storage */
 
	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_EEPROM))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = hwReadGlobal2Reg(dev,QD_REG_EEPROM_COMMAND, &data);
    if(retVal != GT_OK)
    {
       	return retVal;
    }

	*runSt   = (data&GT_EEPROM_OP_ST_RUNNING_MASK)?GT_TRUE:GT_FALSE;
	*writeEn = (data&GT_EEPROM_OP_ST_WRITE_EN_MASK)?GT_TRUE:GT_FALSE;

	return GT_OK;

}

/*******************************************************************************
* gsysSetScratchMiscCtrl
*
* DESCRIPTION:
*       Set Scratch and Misc control data to the Pointer to the Scratch and Misc. 
*		Control register.
*				Scratch Byte 0
*				Scratch Byte 1
*				GPIO Configuration
*				Reserved for future use
*				GPIO Direction
*				GPIO Data
*				CONFIG Data 0
*				CONFIG Data 1
*				CONFIG Data 2
*				CONFIG Data 3
*				SyncE & TAICLK125s Drive
*				P5s & CLK125s Clock Drive
*				P6s Clock Drive
*				EEPROM Pad drive
*
* INPUTS:
*       point - Pointer to the Scratch and Misc. Control register.
*       data - Scratch and Misc. Control data written to the register 
*				pointed to by the point above.
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetScratchMiscCtrl
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			point,
    IN  GT_U8			data
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    IN  GT_U16			tmpData;

	if (point > GT_SCRAT_MISC_REG_MAX)
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	if ((point>0x7f)||(data>0xff))
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	gtSemTake(dev,dev->tblRegsSem,OS_WAIT_FOREVER);

	/* program QoS Weight Table, 4 sequences at a time */

    /* Wait until the Scratch and Misc control is ready. */
   	tmpData = 1;
    while(tmpData == 1)
   	{
        retVal = hwGetGlobal2RegField(dev,QD_REG_SCRATCH_MISC,15,1,&tmpData);
   	    if(retVal != GT_OK)
       	{
			gtSemGive(dev,dev->tblRegsSem);
   	        return retVal;
       	}
    }

	tmpData =  (GT_U16)((1 << 15) | (point << 8) | data);

	retVal = hwWriteGlobal2Reg(dev, QD_REG_SCRATCH_MISC, tmpData);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
           gtSemGive(dev,dev->tblRegsSem);
   	    return retVal;
    }

	gtSemGive(dev,dev->tblRegsSem);

   	return retVal;

}

/*******************************************************************************
* gsysGetScratchMiscCtrl
*
* DESCRIPTION:
*       Get Scratch and Misc control data to the Pointer to the Scratch and Misc. 
*		Control register.
*				Scratch Byte 0
*				Scratch Byte 1
*				GPIO Configuration
*				Reserved for future use
*				GPIO Direction
*				GPIO Data
*				CONFIG Data 0
*				CONFIG Data 1
*				CONFIG Data 2
*				CONFIG Data 3
*				SyncE & TAICLK125s Drive
*				P5s & CLK125s Clock Drive
*				P6s Clock Drive
*				EEPROM Pad drive

*
* INPUTS:
*       point - Pointer to the Scratch and Misc. Control register.
*
* OUTPUTS:
*       data - Scratch and Misc. Control data read from the register 
*				pointed to by the point above.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetScratchMiscCtrl
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			point,
    OUT  GT_U8			*data
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
    OUT  GT_U16			tmpData;

	if (point > GT_SCRAT_MISC_REG_MAX)
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	if (point>0x7f)
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	gtSemTake(dev,dev->tblRegsSem,OS_WAIT_FOREVER);

	/* program QoS Weight Table, 4 sequences at a time */

	do {
		retVal = hwReadGlobal2Reg(dev, QD_REG_SCRATCH_MISC, &tmpData);
		if(retVal != GT_OK)
   		{
   			DBG_INFO(("Failed.\n"));
			gtSemGive(dev,dev->tblRegsSem);
   			return retVal;
		}
	} while (tmpData&0x8000);

    *data = tmpData&0xff;

	gtSemGive(dev,dev->tblRegsSem);

	return retVal;
}

/*******************************************************************************
* gsysSetScratchBits
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <scratch byte 0 and 1> bits.
*		Scratch bits. These bits are 100% available to software for whatever 
*		purpose desired. These bits do not connect to any hardware function.
*
* INPUTS:
*       scritch - written bits.
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetScratchBits
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U16			scratch
)
{
    GT_STATUS	retVal;         /* Functions return value.      */

    DBG_INFO(("gsysSetScratchBits Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_SCRAT_0, (GT_U8)(scratch&0xff));
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_SCRAT_1, (GT_U8)((scratch>>8)&0xff));
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetScratchBits
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <scratch byte 0 and 1> bits.
*		Scratch bits. These bits are 100% available to software for whatever 
*		purpose desired. These bits do not connect to any hardware function.
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       scritch - read bits.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetScratchBits
(
    IN  GT_QD_DEV 		*dev,
    OUT  GT_U16			*scratch
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetScratchBits Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_SCRAT_1, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	*scratch = data;
	*scratch = *scratch<<8;
	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_SCRAT_0, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	
	*scratch |= data;

	return GT_OK;

}

/*******************************************************************************
* gsysSetGpioConfigMod
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <GPIO Configuration>.
*		General Purpose Input Output Bits:
*		Bit 6 - GT_GPIO_BIT_6:	1:GPIO[6]	0:SE_RCLK1
*		Bit 5 - GT_GPIO_BIT_5:	1:GPIO[5]	0:SE_RCLK0
*		Now, folloing bits are read only. 
*		Bit 4 - GT_GPIO_BIT_4:	1:GPIO[4]	0:
*		Bit 3 - GT_GPIO_BIT_3:	1:GPIO[3]	0:
*		Bit 2 - GT_GPIO_BIT_2:	1:GPIO[2]	0:
*		Bit 1 - GT_GPIO_BIT_1:	1:GPIO[1]	0:P6_COL
*		Bit 0 - GT_GPIO_BIT_0:	1:GPIO[0]	0:P6_CRS

*
* INPUTS:
*       mode - OR [GT_GPIO_BIT_x]
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetGpioConfigMod
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			mode
)
{
    GT_STATUS	retVal;         /* Functions return value.      */

    DBG_INFO(("gsysSetGpioConfigMod Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_GPIO_CFG, (GT_U8)(mode&0x7f));
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetGpioConfigMod
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <GPIO Configuration>.
*		General Purpose Input Output Bits:
*		Bit 6 - GT_GPIO_BIT_6:	1:GPIO[6]	0:SE_RCLK1
*		Bit 5 - GT_GPIO_BIT_5:	1:GPIO[5]	0:SE_RCLK0
*		Now, folloing bits are read only. 
*		Bit 4 - GT_GPIO_BIT_4:	1:GPIO[4]	0:
*		Bit 3 - GT_GPIO_BIT_3:	1:GPIO[3]	0:
*		Bit 2 - GT_GPIO_BIT_2:	1:GPIO[2]	0:
*		Bit 1 - GT_GPIO_BIT_1:	1:GPIO[1]	0:P6_COL
*		Bit 0 - GT_GPIO_BIT_0:	1:GPIO[0]	0:P6_CRS

*
* INPUTS:
*       None.
*
* OUTPUTS:
*       mode - OR [GT_GPIO_BIT_x]
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetGpioConfigMod
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			*mode
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetGpioConfigMod Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_GPIO_CFG, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	
	*mode = 0x7f&data;

	return GT_OK;

}

/*******************************************************************************
* gsysSetGpioDirection
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <GPIO Direction>.
*		General Purpose Input Output Bits:
*		Bit 6 - GT_GPIO_BIT_6:	1:GPIO[6]	0:SE_RCLK1
*		Bit 5 - GT_GPIO_BIT_5:	1:GPIO[5]	0:SE_RCLK0
*		Now, folloing bits are read only. 
*		Bit 4 - GT_GPIO_BIT_4:	1:GPIO[4]	0:
*		Bit 3 - GT_GPIO_BIT_3:	1:GPIO[3]	0:
*		Bit 2 - GT_GPIO_BIT_2:	1:GPIO[2]	0:
*		Bit 1 - GT_GPIO_BIT_1:	1:GPIO[1]	0:P6_COL
*		Bit 0 - GT_GPIO_BIT_0:	1:GPIO[0]	0:P6_CRS
*
* INPUTS:
*       dir - OR [GT_GPIO_BIT_x]
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetGpioDirection
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			dir
)
{
    GT_STATUS	retVal;         /* Functions return value.      */

    DBG_INFO(("gsysSetGpioDirection Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_GPIO_DIR, (GT_U8)(dir&0x7f));
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetGpioDirection
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <GPIO Direction>.
*		General Purpose Input Output Bits:
*		Bit 6 - GT_GPIO_BIT_6:	1:GPIO[6]	0:SE_RCLK1
*		Bit 5 - GT_GPIO_BIT_5:	1:GPIO[5]	0:SE_RCLK0
*		Now, folloing bits are read only. 
*		Bit 4 - GT_GPIO_BIT_4:	1:GPIO[4]	0:
*		Bit 3 - GT_GPIO_BIT_3:	1:GPIO[3]	0:
*		Bit 2 - GT_GPIO_BIT_2:	1:GPIO[2]	0:
*		Bit 1 - GT_GPIO_BIT_1:	1:GPIO[1]	0:P6_COL
*		Bit 0 - GT_GPIO_BIT_0:	1:GPIO[0]	0:P6_CRS
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       dir - OR [GT_GPIO_BIT_x]
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetGpioDirection
(
    IN  GT_QD_DEV 		*dev,
    OUT  GT_U32			*dir
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetGpioDirection Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_GPIO_DIR, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	*dir = data;

	return GT_OK;

}

/*******************************************************************************
* gsysSetGpioData
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <GPIO data>.
*		General Purpose Input Output Bits:
*		Bit 6 - GT_GPIO_BIT_6:	1:GPIO[6]	0:SE_RCLK1
*		Bit 5 - GT_GPIO_BIT_5:	1:GPIO[5]	0:SE_RCLK0
*		Now, folloing bits are read only. 
*		Bit 4 - GT_GPIO_BIT_4:	1:GPIO[4]	0:
*		Bit 3 - GT_GPIO_BIT_3:	1:GPIO[3]	0:
*		Bit 2 - GT_GPIO_BIT_2:	1:GPIO[2]	0:
*		Bit 1 - GT_GPIO_BIT_1:	1:GPIO[1]	0:P6_COL
*		Bit 0 - GT_GPIO_BIT_0:	1:GPIO[0]	0:P6_CRS
*
* INPUTS:
*       data - OR [GT_GPIO_BIT_x]
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetGpioData
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			data
)
{
    GT_STATUS	retVal;         /* Functions return value.      */

    DBG_INFO(("gsysSetGpioData Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_GPIO_DAT, (GT_U8)(data&0x7f));
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetGpioData
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <GPIO data>.
*		General Purpose Input Output Bits:
*		Bit 6 - GT_GPIO_BIT_6:	1:GPIO[6]	0:SE_RCLK1
*		Bit 5 - GT_GPIO_BIT_5:	1:GPIO[5]	0:SE_RCLK0
*		Now, folloing bits are read only. 
*		Bit 4 - GT_GPIO_BIT_4:	1:GPIO[4]	0:
*		Bit 3 - GT_GPIO_BIT_3:	1:GPIO[3]	0:
*		Bit 2 - GT_GPIO_BIT_2:	1:GPIO[2]	0:
*		Bit 1 - GT_GPIO_BIT_1:	1:GPIO[1]	0:P6_COL
*		Bit 0 - GT_GPIO_BIT_0:	1:GPIO[0]	0:P6_CRS
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       data - OR [GT_GPIO_BIT_x]
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetGpioData
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U32			*data
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		tmpData;

    DBG_INFO(("gsysGetGpioData Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_GPIO_DAT, &tmpData);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	*data = tmpData;

	return GT_OK;

}

/*******************************************************************************
* gsysGetConfigData
*
* DESCRIPTION:
*       Get Reset Configuration pin data 0-3. 
*		These registers return the values observed after a hardware Reset on the 
*		listed CONFIG data listed below.
*			Config data 0:
*				0 USER[0] P6_OUTD[5]
*				1 USER[1] P6_OUTD[6]
*				2 USER[2] P6_OUTD[7]
*				3 ADDR[0] P5_OUTD[0]
*				4 ADDR[1] P5_OUTD[5]
*				5 ADDR[2] P5_OUTD[6]
*				6 ADDR]3] P5_OUTD[7]
*				7 ADDR[4] P5_OUTD[1]
*			Config data 1:
*				0 LED_SEL[0] P1_LED
*				1 LED_SEL[1] P2_LED
*				2 4COL P3_LED
*				3 NormCx P4_LED
*				4 Jumbo P0_LED
*				5 EE_WE EE_CS/C2_LED
*				6 FD_FLOW EE_CLK/C1_LED
*				7 HD_FLOW EE_DIN/C0_LED
*			Config data 2:
*				0 P5_MODE[0] P5_OUTD[2]
*				1 P5_MODE[1] P5_OUTD[3]
*				2 P5_MODE[2] P5_OUTD[4]
*				3 Reserved for future useP5_HALFDPX P5_OUTEN
*				4 P6_MODE[0] P6_OUTD[2]
*				5 P6_MODE[1] P6_OUTD[3]
*				6 P6_MODE[2] P6_OUTD[4]
*				7 Reserved for future useP6_HALFDPX P6_OUTEN
*			Config data 2:
*				0 RMU_MODE[0] P6_OUTD[0]
*				1 RMU_MODE[1] P6_OUTD[1]
*				2 
*				3
*				4
*				5
*				6
*				7

* INPUTS:
*       none.
*
* OUTPUTS:
*       cfgDat - GT_CONFIG_DTTA
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetConfigData
(
    IN  GT_QD_DEV 			*dev,
    OUT  GT_CONFIG_DATA		*cfgData
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		tmpData;

    DBG_INFO(("gsysGetConfigData Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_CFG_DAT0, &tmpData);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	cfgData->cfgData0.Byte = tmpData;

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_CFG_DAT1, &tmpData);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	cfgData->cfgData1.Byte = tmpData;

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_CFG_DAT2, &tmpData);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	cfgData->cfgData2.Byte = tmpData;

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_CFG_DAT3, &tmpData);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	cfgData->cfgData3.Byte = tmpData;

	return GT_OK;

}

/*******************************************************************************
* gsysSetSyncETai
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <SyncE and TAI pad>.
*
* INPUTS:
*       zpr - ZPR for SyncE and TAI
*       znr - ZNR for SyncE and TAI
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetSyncETai
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U8			zpr,
    IN  GT_U8			znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysSetSyncETai Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if ((zpr>0x7) || (znr>0x7))
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	data = ((zpr&0x7)<<3) | (znr&0x7);

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_SYNCE, data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetSyncETai
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <SyncE and TAI pad>.
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       zpr - ZPR for SyncE and TAI
*       znr - ZNR for SyncE and TAI*
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetSyncETai
(
    IN  GT_QD_DEV 		*dev,
    OUT  GT_U8			*zpr,
    OUT  GT_U8			*znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetSyncETai Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_SYNCE, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	
	*zpr = 0x7 &(data>>3);
	*znr = 0x7 &(data);

	return GT_OK;

}

/*******************************************************************************
* gsysSetP6_Clock
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <P6_Clock pad>.
*
* INPUTS:
*       zpr - ZPR for P6_Clock
*       znr - ZNR for P6_Clock
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetP6_Clock
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U8			zpr,
    IN  GT_U8			znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysSetP6_Clock Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if ((zpr>0x7) || (znr>0x7))
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	data = ((zpr&0x7)<<3) | (znr&0x7);

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_P6_CLK, data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetP6_Clock
*
*       Get the Scratch and Misc. Control register <P6_Clock pad>.
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       zpr - ZPR for P6_Clock
*       znr - ZNR for P6_Clock*
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetP6_Clock
(
    IN  GT_QD_DEV 		*dev,
    OUT  GT_U8			*zpr,
    OUT  GT_U8			*znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetP6_Clock Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_P6_CLK, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	
	*zpr = 0x7 &(data>>3);
	*znr = 0x7 &(data);

	return GT_OK;

}

/*******************************************************************************
* gsysSetP5_Clock
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <P5_Clock pad>.
*
* INPUTS:
*       zpr - ZPR for P5_Clock
*       znr - ZNR for P5_Clock
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetP5_Clock
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U8			zpr,
    IN  GT_U8			znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysSetP5_Clock Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if ((zpr>0x7) || (znr>0x7))
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	data = ((zpr&0x7)<<3) | (znr&0x7);

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_P5_CLK, data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetP5_Clock
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <P5_Clock pad>.
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       zpr - ZPR for P5_Clock
*       znr - ZNR for P5_Clock
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetP5_Clock
(
    IN  GT_QD_DEV 		*dev,
    OUT  GT_U8			*zpr,
    OUT  GT_U8			*znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetP6_Clock Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_P5_CLK, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	
	*zpr = 0x7 &(data>>3);
	*znr = 0x7 &(data);

	return GT_OK;

}

/*******************************************************************************
* gsysSetEEPROM
*
* DESCRIPTION:
*       Set the Scratch and Misc. Control register <EEPROM pad>.
*
* INPUTS:
*       dsm - DSM for EEPROM
*       zpr - ZPR for EEPROM
*       znr - ZNR for EEPROM
*
* OUTPUTS:
*       None.
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*       GT_BAD_PARAM - if input parameters are beyond range.
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysSetEEPROM
(
    IN  GT_QD_DEV 		*dev,
    IN  GT_U8			dsm,
    IN  GT_U8			zpr,
    IN  GT_U8			znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysSetEEPROM Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	if ((zpr>0x7) || (znr>0x7))
	{
        DBG_INFO(("GT_BAD_PARAM\n"));
		return GT_BAD_PARAM;
	}

	data = ((dsm&0x3)<<6) | ((zpr&0x7)<<3) | (znr&0x7);

	retVal = gsysSetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_EEPROM, data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }

	return GT_OK;

}

/*******************************************************************************
* gsysGetEEPROM
*
* DESCRIPTION:
*       Get the Scratch and Misc. Control register <EEPROM pad>.
*
* INPUTS:
*       None.
*
* OUTPUTS:
*       dsm - DSM for EEPROM
*       zpr - ZPR for EEPROM
*       znr - ZNR for EEPROM
*
* RETURNS:
*       GT_OK      - on success
*       GT_FAIL    - on error
*		GT_NOT_SUPPORTED - if current device does not support this feature.
*
* COMMENTS:
*       None
*
*******************************************************************************/
GT_STATUS gsysGetEEPROM
(
    IN  GT_QD_DEV 		*dev,
    OUT  GT_U8			*dsm,
    OUT  GT_U8			*zpr,
    OUT  GT_U8			*znr
)
{
    GT_STATUS	retVal;         /* Functions return value.      */
	GT_U8		data;

    DBG_INFO(("gsysGetEEPROM Called.\n"));

	/* Check if Switch supports this feature. */
	if (!IS_IN_DEV_GROUP(dev,DEV_SCRATCH_MISC_CTRL))
    {
        DBG_INFO(("GT_NOT_SUPPORTED\n"));
		return GT_NOT_SUPPORTED;
    }

	retVal = gsysGetScratchMiscCtrl(dev, GT_SCRAT_MISC_REG_EEPROM, &data);
    if(retVal != GT_OK)
   	{
   	    DBG_INFO(("Failed.\n"));
   	    return retVal;
    }
	
	*dsm = 0x3 &(data>>6);
	*zpr = 0x7 &(data>>3);
	*znr = 0x7 &(data);

	return GT_OK;

}
