/*
 * @(#) prototypes for FreeSWAN functions 
 *
 * Copyright (C) 2001  Richard Guy Briggs  <rgb@freeswan.org>
 *                 and Michael Richardson  <mcr@freeswan.org>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: ipsec_proto.h,v 1.1 2008/05/20 09:52:53 evantu Exp $
 *
 */

#ifndef _IPSEC_PROTO_H_

#include "ipsec_param.h"
#include <net/ip.h>

/* 
 * This file is a kernel only file that declares prototypes for
 * all intra-module function calls and global data structures.
 * 
 * Only put functions here that could take/return opaque structure
 * for which the caller simply doesn't need to know the details
 * (i.e. doesn't need the entire .h file)
 *
 * Include this file last.
 *
 */

/* forward references */
enum ipsec_direction;
enum ipsec_life_type;
struct ipsec_lifetime64;
struct ident;
struct sockaddr_encap;
struct ipsec_sa;

/* ipsec_init.c */
extern struct prng ipsec_prng;

/* ipsec_sa.c */
extern struct ipsec_sa *ipsec_sadb_hash[SADB_HASHMOD];
extern spinlock_t       tdb_lock;
extern int ipsec_sadb_init(void);
extern int ipsec_sadb_cleanup(__u8);

extern struct ipsec_sa *ipsec_sa_alloc(int*error); 

extern /* void */ int ipsec_sa_add(struct ipsec_sa *);

extern int ipsec_sa_init(struct ipsec_sa *ipsp);

/* delayed free of extra skb (because of compression) */
extern void ipsec_skb_gc_enqueue(struct sk_buff *skb);


/* debug declarations */

/* ipsec_proc.c */
extern int  ipsec_proc_init(void);
extern void ipsec_proc_cleanup(void);

/* ipsec_rcv.c */
extern int ipsec_rcv(struct sk_buff *skb);
extern int klips26_rcv_encap(struct sk_buff *skb, __u16 encap_type);

/* ipsec_xmit.c */
struct ipsec_xmit_state;
extern enum ipsec_xmit_value ipsec_xmit_sanity_check_dev(struct ipsec_xmit_state *ixs);
extern enum ipsec_xmit_value ipsec_xmit_sanity_check_skb(struct ipsec_xmit_state *ixs);
extern void ipsec_print_ip(struct iphdr *ip);



/* ipsec_radij.c */
extern int ipsec_makeroute(struct sockaddr_encap *ea,
			   struct sockaddr_encap *em,
			   ip_said said,
			   uint32_t pid,
			   struct sk_buff *skb,
			   struct ident *ident_s,
			   struct ident *ident_d);

extern int ipsec_breakroute(struct sockaddr_encap *ea,
			    struct sockaddr_encap *em,
			    struct sk_buff **first,
			    struct sk_buff **last);

int ipsec_radijinit(void);
int ipsec_cleareroutes(void);
int ipsec_radijcleanup(void);

/* ipsec_life.c */
extern enum ipsec_life_alive ipsec_lifetime_check(struct ipsec_lifetime64 *il64,
						  const char *lifename,
						  const char *saname,
						  enum ipsec_life_type ilt,
						  enum ipsec_direction idir,
						  struct ipsec_sa *ips);


extern int ipsec_lifetime_format(char *buffer,
				 int   buflen,
				 char *lifename,
				 enum ipsec_life_type timebaselife,
				 struct ipsec_lifetime64 *lifetime);

extern void ipsec_lifetime_update_hard(struct ipsec_lifetime64 *lifetime,
				       __u64 newvalue);

extern void ipsec_lifetime_update_soft(struct ipsec_lifetime64 *lifetime,
				       __u64 newvalue);

/* ipsec_snprintf.c */
extern int ipsec_snprintf(char * buf, ssize_t size, const char *fmt, ...);
extern void ipsec_dmp_block(char *s, caddr_t bb, int len);


/* ipsec_alg.c */
extern int ipsec_alg_init(void);


#ifdef CONFIG_KLIPS_DEBUG

extern int debug_xform;
extern int debug_eroute;
extern int debug_spi;
extern int debug_netlink;

#endif /* CONFIG_KLIPS_DEBUG */




#define _IPSEC_PROTO_H
#endif /* _IPSEC_PROTO_H_ */

/*
 * $Log: ipsec_proto.h,v $
 * Revision 1.1  2008/05/20 09:52:53  evantu
 * code from freescale BSP: ltib-mpc8548cds-20071214
 *
 *  Modified Files:
 *  	linux/miscdevice.h linux/rio_drv.h linux/rio.h linux/skbuff.h
 *  Added Files:
 *  	linux/ctswic.h linux/debugfs_circular.h mast.h crypto/aes.h
 *  	crypto/aes_cbc.h crypto/aes_xcbc_mac.h crypto/cbc_generic.h
 *  	crypto/des.h crypto/ocf_assist.h des/des_locl.h des/des_ver.h
 *  	des/podd.h des/sk.h des/spr.h opencrypto/crypto.h
 *  	opencrypto/cryptodev.h opencrypto/cryptoprof.h
 *  	openswan/ipcomp.h openswan/ipsec_ah.h openswan/ipsec_alg.h
 *  	openswan/ipsec_alg_3des.h openswan/ipsec_auth.h
 *  	openswan/ipsec_debug.h openswan/ipsec_encap.h
 *  	openswan/ipsec_eroute.h openswan/ipsec_errs.h
 *  	openswan/ipsec_esp.h openswan/ipsec_ipcomp.h
 *  	openswan/ipsec_ipe4.h openswan/ipsec_ipip.h
 *  	openswan/ipsec_kern24.h openswan/ipsec_kversion.h
 *  	openswan/ipsec_life.h openswan/ipsec_md5h.h
 *  	openswan/ipsec_param.h openswan/ipsec_policy.h
 *  	openswan/ipsec_proto.h openswan/ipsec_radij.h
 *  	openswan/ipsec_rcv.h openswan/ipsec_sa.h openswan/ipsec_sha1.h
 *  	openswan/ipsec_stats.h openswan/ipsec_tunnel.h
 *  	openswan/ipsec_xform.h openswan/ipsec_xmit.h
 *  	openswan/passert.h openswan/pfkey.h openswan/pfkey_debug.h
 *  	openswan/pfkeyv2.h openswan/radij.h
 *
 * Revision 1.14  2005/04/29 04:50:03  mcr
 * 	prototypes for xmit and alg code.
 *
 * Revision 1.13  2005/04/17 03:46:07  mcr
 * 	added prototypes for ipsec_rcv() routines.
 *
 * Revision 1.12  2005/04/14 20:28:37  mcr
 * 	added additional prototypes.
 *
 * Revision 1.11  2005/04/14 01:16:28  mcr
 * 	add prototypes for snprintf.
 *
 * Revision 1.10  2005/04/13 22:47:28  mcr
 * 	make sure that forward references are available.
 *
 * Revision 1.9  2004/07/10 19:08:41  mcr
 * 	CONFIG_IPSEC -> CONFIG_KLIPS.
 *
 * Revision 1.8  2004/04/05 19:55:06  mcr
 * Moved from linux/include/freeswan/ipsec_proto.h,v
 *
 * Revision 1.7  2003/10/31 02:27:05  mcr
 * 	pulled up port-selector patches and sa_id elimination.
 *
 * Revision 1.6.30.1  2003/10/29 01:10:19  mcr
 * 	elimited "struct sa_id"
 *
 * Revision 1.6  2002/05/23 07:13:48  rgb
 * Added ipsec_sa_put() for releasing an ipsec_sa refcount.
 *
 * Revision 1.5  2002/05/14 02:36:40  rgb
 * Converted reference from ipsec_sa_put to ipsec_sa_add to avoid confusion
 * with "put" usage in the kernel.
 *
 * Revision 1.4  2002/04/24 07:36:47  mcr
 * Moved from ./klips/net/ipsec/ipsec_proto.h,v
 *
 * Revision 1.3  2002/04/20 00:12:25  rgb
 * Added esp IV CBC attack fix, disabled.
 *
 * Revision 1.2  2001/11/26 09:16:15  rgb
 * Merge MCR's ipsec_sa, eroute, proc and struct lifetime changes.
 *
 * Revision 1.1.2.1  2001/09/25 02:21:01  mcr
 * 	ipsec_proto.h created to keep prototypes rather than deal with
 * 	cyclic dependancies of structures and prototypes in .h files.
 *
 *
 *
 * Local variables:
 * c-file-style: "linux"
 * End:
 *
 */

