/*
 * Copyright (C) 2007 Freescale Semiconductor, Inc.
 * Author: Zhang Wei, wei.zhang@freescale.com, Jun 2007
 *
 * Copyright 2005 MontaVista Software, Inc.
 * Matt Porter <mporter@kernel.crashing.org>
 *
 * Description:
 * PowerPC RapidIO support
 *
 * Changelog:
 * Jun 2007 Zhang Wei <wei.zhang@freescale.com>
 * - This file is moved from arch/ppc/ppc85xx_rio.c. And the OF-tree support
 *   is added. New silicons such as MPC8548, MPC8641 are all supported.
 *   Memory driver hardware operations are added.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/rio.h>
#include <linux/rio_drv.h>

#include <asm/io.h>
#include <asm/prom.h>
#include <asm/of_device.h>
#include <asm/of_platform.h>
#include "fsl_soc.h"

/* RapidIO definition irq, which read from OF-tree */
#define IRQ_RIO_BELL(m)		(((struct rio_priv *)(m->priv))->bellirq)
#define IRQ_RIO_TX(m)		(((struct rio_priv *)(m->priv))->txirq)
#define IRQ_RIO_RX(m)		(((struct rio_priv *)(m->priv))->rxirq)

#define ERR(fmt, arg...) \
	printk(KERN_ERR "RIO %s: " fmt, __FUNCTION__, ## arg)
#define INFO(fmt...) printk(KERN_INFO "RIO: " fmt)
#define IS_64BIT_RES ((sizeof(resource_size_t) == 8) ? 1 : 0)

#define RIO_ATMU_REGS_OFFSET	0x10c00
#define RIO_P_MSG_REGS_OFFSET	0x11000
#define RIO_S_MSG_REGS_OFFSET	0x13000
#define RIO_ESCSR		0x158
#define RIO_CCSR		0x15c
#define RIO_ISR_AACR		0x10120
#define RIO_ISR_AACR_AA		0x1	/* Accept All ID */
#define RIO_MAINT_WIN_SIZE	0x400000
#define RIO_DBELL_WIN_SIZE	0x1000
#define RIO_MAX_INB_ATMU	4
#define RIO_MAX_OUTB_ATMU	8
#define RIO_INB_ATMU_REGS_OFFSET	0x10de0
#define RIO_ATMU_EN_MASK	0x80000000

#define RIO_NREAD		0x4
#define RIO_NWRITE		0x4
#define RIO_NWRITE_R		0x5
#define RIO_NREAD_R		0x5

#define RIO_MSG_OMR_MUI		0x00000002
#define RIO_MSG_OSR_TE		0x00000080
#define RIO_MSG_OSR_QOI		0x00000020
#define RIO_MSG_OSR_QFI		0x00000010
#define RIO_MSG_OSR_MUB		0x00000004
#define RIO_MSG_OSR_EOMI	0x00000002
#define RIO_MSG_OSR_QEI		0x00000001

#define RIO_MSG_IMR_MI		0x00000002
#define RIO_MSG_ISR_TE		0x00000080
#define RIO_MSG_ISR_QFI		0x00000010
#define RIO_MSG_ISR_DIQI	0x00000001

#define RIO_MSG_DESC_SIZE	32
#define RIO_MSG_BUFFER_SIZE	4096
#define RIO_MIN_TX_RING_SIZE	2
#define RIO_MAX_TX_RING_SIZE	2048
#define RIO_MIN_RX_RING_SIZE	2
#define RIO_MAX_RX_RING_SIZE	2048

#define DOORBELL_DMR_DI		0x00000002
#define DOORBELL_DSR_TE		0x00000080
#define DOORBELL_DSR_QFI	0x00000010
#define DOORBELL_DSR_DIQI	0x00000001
#define DOORBELL_TID_OFFSET	0x02
#define DOORBELL_SID_OFFSET	0x04
#define DOORBELL_INFO_OFFSET	0x06

#define DOORBELL_MESSAGE_SIZE	0x08
#define DBELL_SID(x)		(*(u16 *)(x + DOORBELL_SID_OFFSET))
#define DBELL_TID(x)		(*(u16 *)(x + DOORBELL_TID_OFFSET))
#define DBELL_INF(x)		(*(u16 *)(x + DOORBELL_INFO_OFFSET))

struct rio_atmu_regs {
	u32 rowtar;
	u32 rowtear;
	u32 rowbar;
	u32 pad2;
	u32 rowar;
	u32 pad3[3];
};

struct rio_inb_atmu_regs {
	u32 riwtar;
	u32 pad1;
	u32 riwbar;
	u32 pad2;
	u32 riwar;
	u32 pad3[3];
};

struct rio_msg_regs {
	u32 omr;
	u32 osr;
	u32 pad1;
	u32 odqdpar;
	u32 pad2;
	u32 osar;
	u32 odpr;
	u32 odatr;
	u32 odcr;
	u32 pad3;
	u32 odqepar;
	u32 pad4[13];
	u32 imr;
	u32 isr;
	u32 pad5;
	u32 ifqdpar;
	u32 pad6;
	u32 ifqepar;
	u32 pad7[226];
	u32 odmr;
	u32 odsr;
	u32 res0[4];
	u32 oddpr;
	u32 oddatr;
	u32 res1[3];
	u32 odretcr;
	u32 res2[12];
	u32 dmr;
	u32 dsr;
	u32 pad8;
	u32 dqdpar;
	u32 pad9;
	u32 dqepar;
	u32 pad10[26];
	u32 pwmr;
	u32 pwsr;
	u32 pad11;
	u32 pwqbar;
};

struct rio_tx_desc {
	u32 res1;
	u32 saddr;
	u32 dport;
	u32 dattr;
	u32 res2;
	u32 res3;
	u32 dwcnt;
	u32 res4;
};

struct rio_dbell_ring {
	void *virt;
	dma_addr_t phys;
};

struct rio_msg_tx_ring {
	void *virt;
	dma_addr_t phys;
	void *virt_buffer[RIO_MAX_TX_RING_SIZE];
	dma_addr_t phys_buffer[RIO_MAX_TX_RING_SIZE];
	int tx_slot;
	int size;
	void *dev_id;
};

struct rio_msg_rx_ring {
	void *virt;
	dma_addr_t phys;
	void *virt_buffer[RIO_MAX_RX_RING_SIZE];
	int rx_slot;
	int size;
	void *dev_id;
};

struct rio_priv {
	volatile void __iomem *regs_win;
	volatile struct rio_atmu_regs __iomem *atmu_regs;
	volatile struct rio_atmu_regs __iomem *maint_atmu_regs;
	volatile struct rio_atmu_regs __iomem *dbell_atmu_regs;
	volatile void __iomem *dbell_win;
	volatile void __iomem *maint_win;
	volatile struct rio_msg_regs __iomem *msg_regs;
	struct rio_dbell_ring dbell_ring;
	struct rio_msg_tx_ring msg_tx_ring;
	struct rio_msg_rx_ring msg_rx_ring;
	int bellirq;
	int txirq;
	int rxirq;
};

/**
 * fsl_rio_doorbell_send - Send a PowerPC doorbell message
 * @index: ID of RapidIO interface
 * @destid: Destination ID of target device
 * @data: 16-bit info field of RapidIO doorbell message
 *
 * Sends a PowerPC doorbell message. Returns %0 on success or
 * %-EINVAL on failure.
 */
static int fsl_rio_doorbell_send(struct rio_mport *mport, int index, u16 destid, u16 data)
{
	struct rio_priv *priv = mport->priv;
	pr_debug("fsl_doorbell_send: index %d destid 0x%04x data 0x%04x\n",
			index, destid, data);

	switch (mport->phy_type) {
	case RIO_PHY_SERIAL:
		/* In the later version silicons, such as MPC8548, MPC8641,
		 * below operations is must be.
		 */
		out_be32(&priv->msg_regs->odmr, 0x00000000);
		out_be32(&priv->msg_regs->odretcr, 0x00000004);
		out_be32(&priv->msg_regs->oddpr, destid << 16);
		out_be32(&priv->msg_regs->oddatr,data );
		out_be32(&priv->msg_regs->odmr, 0x00000001);
		break;
	case RIO_PHY_PARALLEL:
		out_be32(&priv->dbell_atmu_regs->rowtar, destid << 22);
		out_be16(priv->dbell_win, data);
		break;
	}

	return 0;
}

/**
 * fsl_local_config_read - Generate a PowerPC local config space read
 * @index: ID of RapdiIO interface
 * @offset: Offset into configuration space
 * @len: Length (in bytes) of the maintenance transaction
 * @data: Value to be read into
 *
 * Generates a PowerPC local configuration space read. Returns %0 on
 * success or %-EINVAL on failure.
 */
static int fsl_local_config_read(struct rio_mport *mport, int index, u32 offset, int len, u32 * data)
{
	struct rio_priv *priv = mport->priv;
	pr_debug("fsl_local_config_read: index %d offset 0x%08x\n", index,
			offset);
	*data = in_be32(priv->regs_win + offset);

	return 0;
}

/**
 * fsl_local_config_write - Generate a PowerPC local config space write
 * @index: ID of RapdiIO interface
 * @offset: Offset into configuration space
 * @len: Length (in bytes) of the maintenance transaction
 * @data: Value to be written
 *
 * Generates a PowerPC local configuration space write. Returns %0 on
 * success or %-EINVAL on failure.
 */
static int fsl_local_config_write(struct rio_mport *mport, int index, u32 offset, int len, u32 data)
{
	struct rio_priv *priv = mport->priv;
	pr_debug("fsl_local_config_write: index %d offset 0x%08x data 0x%08x\n",
			index, offset, data);
	out_be32(priv->regs_win + offset, data);

	return 0;
}

/**
 * fsl_rio_config_read - Generate a PowerPC read maintenance transaction
 * @index: ID of RapdiIO interface
 * @destid: Destination ID of transaction
 * @hopcount: Number of hops to target device
 * @offset: Offset into configuration space
 * @len: Length (in bytes) of the maintenance transaction
 * @val: Location to be read into
 *
 * Generates a PowerPC read maintenance transaction. Returns %0 on
 * success or %-EINVAL on failure.
 */
static int fsl_rio_config_read(struct rio_mport *mport, int index, u16 destid,
		u8 hopcount, u32 offset, int len, u32 * val)
{
	void *data;
	struct rio_priv *priv = mport->priv;

	pr_debug("fsl_rio_config_read: index %d destid %d hopcount %d "
			"offset 0x%08x len %d\n",
			index, destid, hopcount, offset, len);
	out_be32(&priv->maint_atmu_regs->rowtar,
		 ((destid & 0x3ff) << 22) | (hopcount << 12)
		 | ((offset & ~0x3) >> 9));
	out_be32(&priv->maint_atmu_regs->rowtear, (destid & 0xfc00) >> 10);

	data = priv->maint_win + offset;
	switch (len) {
	case 1:
		*val = in_8(data);
		break;
	case 2:
		*val = in_be16(data);
		break;
	default:
		*val = in_be32(data);
		break;
	}

	return 0;
}

/**
 * fsl_rio_config_write - Generate a PowerPC write maintenance transaction
 * @index: ID of RapdiIO interface
 * @destid: Destination ID of transaction
 * @hopcount: Number of hops to target device
 * @offset: Offset into configuration space
 * @len: Length (in bytes) of the maintenance transaction
 * @val: Value to be written
 *
 * Generates an PowerPC write maintenance transaction. Returns %0 on
 * success or %-EINVAL on failure.
 */
static int fsl_rio_config_write(struct rio_mport *mport, int index, u16 destid,
		u8 hopcount, u32 offset, int len, u32 val)
{
	void *data;
	struct rio_priv *priv = mport->priv;
	pr_debug("fsl_rio_config_write: index %d destid %d hopcount %d"
			"offset 0x%08x len %d val 0x%08x\n",
			index, destid, hopcount, offset, len, val);
	out_be32(&priv->maint_atmu_regs->rowtar,
		 ((destid & 0x3ff) << 22) | (hopcount << 12)
		 | ((offset & ~0x3) >> 9));
	out_be32(&priv->maint_atmu_regs->rowtear, (destid & 0xfc00) >> 10);

	data = priv->maint_win + offset;
	switch (len) {
	case 1:
		out_8(data, val);
		break;
	case 2:
		out_be16(data, val);
		break;
	default:
		out_be32(data, val);
		break;
	}

	return 0;
}

/**
 * fsl_rio_map_inb_mem -- Mapping inbound memory region.
 * @lstart: Local memory space start address.
 * @rstart: RapidIO space start address.
 * @size: The mapping region size.
 * @flags: Flags for mapping. 0 for using default flags.
 *
 * Return: 0 -- Success.
 *
 * This function will create the inbound mapping
 * from rstart to lstart.
 */
static int fsl_rio_map_inb_mem(struct rio_mport *mport, resource_size_t lstart,
		resource_size_t rstart,	resource_size_t size,
		u32 flags)
{
	int i;
	struct rio_priv *priv = mport->priv;
	volatile struct rio_inb_atmu_regs __iomem *inbatmu =
				(struct rio_inb_atmu_regs *)
				(priv->regs_win + RIO_INB_ATMU_REGS_OFFSET) - 1;
	int size_ffs;
	resource_size_t align;

	if (flags == 0)
		flags = (RIO_NREAD_R << 4) | RIO_NWRITE_R;

	align = (size < 0x1000) ? 0x1000 : 1 << (__ilog2(size - 1) + 1);

	/* Align the size */
	if ((lstart + size) > (_ALIGN_DOWN(lstart, align) + align)) {
		size_ffs = __ffs(_ALIGN_DOWN(lstart + size - 1, align));
		size = 1 << (size_ffs +	(((_ALIGN_DOWN(lstart, 1 << size_ffs) +
				(1 << size_ffs)) < (lstart + size)) ? 1 : 0));
	} else
		size = align;

	if ((lstart & (size - 1)) != (rstart & (size - 1))) {
		ERR("The local address 0x%x can not be aligned to the same size"
			" 0x%x with the RapidIO space address 0x%x!\n", lstart,
			size, rstart);
		return -EINVAL;
	}

	/* Search for free inbound ATMU */
	for (i = 1;
		(i <= RIO_MAX_INB_ATMU) && (inbatmu->riwar & RIO_ATMU_EN_MASK);
		i++, inbatmu--)
		;

	if (i > RIO_MAX_INB_ATMU) {
		ERR("No free inbound ATMU!\n");
		return -EBUSY;
	}
	out_be32(&inbatmu->riwtar, ((IS_64BIT_RES ? (lstart >> 32)
				& 0xf : 0) << 20) | ((lstart >> 12) & 0xfffff));
	out_be32(&inbatmu->riwbar, ((IS_64BIT_RES ? (rstart >> 32)
				& 0x3 : 0) << 20) | ((rstart >> 12) & 0xfffff));
	out_be32(&inbatmu->riwar, 0x80000000 | (0xf << 20)
				| ((flags & 0xff) << 12)
				| (__ilog2(size) - 1));
	return 0;
}

/**
 * fsl_rio_map_outb_mem -- Mapping outbound memory region.
 * @lstart: Local memory space start address.
 * @rstart: RapidIO space start address.
 * @size: The mapping region size.
 * @tid: The target RapidIO device id.
 * @flags: Flags for mapping. 0 for using default flags.
 *
 * Return: 0 -- Success.
 *
 * This function will create the outbound mapping
 * from lstart to rstart.
 */
static int fsl_rio_map_outb_mem(struct rio_mport *mport, resource_size_t lstart,
		resource_size_t rstart,	resource_size_t size,
		u16 tid, u32 flags)
{
	int i;
	struct rio_priv *priv = mport->priv;
	volatile struct rio_atmu_regs __iomem *outbatmu =
			(struct rio_atmu_regs *)
			(priv->regs_win + RIO_ATMU_REGS_OFFSET) + 1;
	int size_ffs;
	resource_size_t align;

	if (flags == 0)
		flags = (RIO_NREAD << 4) | RIO_NWRITE_R;

	align = (size < 0x1000) ? 0x1000 : 1 << (__ilog2(size - 1) + 1);

	/* Align the size */
	if ((lstart + size) > (_ALIGN_DOWN(lstart, align) + align)) {
		size_ffs = __ffs(_ALIGN_DOWN(lstart + size - 1, align));
		size = 1 << (size_ffs +	(((_ALIGN_DOWN(lstart, 1 << size_ffs) +
				(1 << size_ffs)) < (lstart + size)) ? 1 : 0));
	} else
		size = align;

	if ((lstart & (size - 1)) != (rstart & (size - 1))) {
		ERR("The local address 0x%x can not be aligned to the same size"
			" 0x%x with the RapidIO space address 0x%x!\n", lstart,
			size, rstart);
		return -EINVAL;
	}

	/* Search for free outbound ATMU */
	for (i = 1;
	      (i <= RIO_MAX_OUTB_ATMU) && (outbatmu->rowar & RIO_ATMU_EN_MASK);
	      i++, outbatmu++)
		;

	if (i > RIO_MAX_OUTB_ATMU) {
		ERR("No free outbound ATMU!\n");
		return -EBUSY;
	}
	out_be32(&outbatmu->rowtar, ((tid & 0x3ff) << 22)
			| ((IS_64BIT_RES ? (rstart >> 32) & 0x3 : 0) << 20)
			| ((rstart >> 12) & 0xfffff));
	if (mport->phy_type == RIO_PHY_SERIAL)
		out_be32(&outbatmu->rowtear, tid >> 10);
	out_be32(&outbatmu->rowbar, ((IS_64BIT_RES ?
					(lstart >> 32) & 0xf : 0) << 20)
					| ((lstart >> 12) & 0xfffff));
	out_be32(&outbatmu->rowar, 0x80000000
				| ((flags & 0xff) << 12)
				| (__ilog2(size) - 1));
	return 0;
}

/**
 * fsl_rio_unmap_inb_mem -- Unmapping inbound memory region.
 * @lstart: Local memory space start address.
 */
static void fsl_rio_unmap_inb_mem(struct rio_mport *mport, resource_size_t lstart)
{
	int i;
	struct rio_priv *priv = mport->priv;
	volatile struct rio_inb_atmu_regs __iomem *inbatmu = (struct rio_inb_atmu_regs *)
			(priv->regs_win + RIO_INB_ATMU_REGS_OFFSET) - 1;

	/* Search for inbound ATMU */
	for (i = 1; i <= RIO_MAX_INB_ATMU ; i++, inbatmu--) {
		u32 tar = ((IS_64BIT_RES ? (lstart >> 32) & 0xf : 0) << 20)
			| ((lstart >> 12) & 0xfffff);
		if (inbatmu->riwtar == tar) {
			out_be32(&inbatmu->riwar, ~(RIO_ATMU_EN_MASK));
			return;
		}
	}
}

/**
 * fsl_rio_unmap_inb_mem -- Unmapping outbound memory region.
 * @lstart: Local memory space start address.
 */
static void fsl_rio_unmap_outb_mem(struct rio_mport *mport, resource_size_t lstart)
{
	int i;
	struct rio_priv *priv = mport->priv;
	volatile struct rio_atmu_regs __iomem *outbatmu = (struct rio_atmu_regs *)
			(priv->regs_win + RIO_ATMU_REGS_OFFSET) + 1;

	/* Search for outbound ATMU */
	for (i = 1; i <= RIO_MAX_OUTB_ATMU ; i++, outbatmu++) {
		u32 bar = ((IS_64BIT_RES ? (lstart >> 32) & 0xf : 0) << 20)
			| ((lstart >> 12) & 0xfffff);
		if (outbatmu->rowbar == bar) {
			out_be32(&outbatmu->rowar, ~(RIO_ATMU_EN_MASK));
			return;
		}
	}
}

/**
 * rio_hw_add_outb_message - Add message to the PowerPC outbound message queue
 * @mport: Master port with outbound message queue
 * @rdev: Target of outbound message
 * @mbox: Outbound mailbox
 * @buffer: Message to add to outbound queue
 * @len: Length of message
 *
 * Adds the @buffer message to the PowerPC outbound message queue. Returns
 * %0 on success or %-EINVAL on failure.
 */
int rio_hw_add_outb_message(struct rio_mport *mport, struct rio_dev *rdev,
		int mbox, void *buffer, size_t len)
{
	struct rio_priv *priv = mport->priv;
	struct rio_tx_desc *desc =
	    (struct rio_tx_desc *)priv->msg_tx_ring.virt + priv->msg_tx_ring.tx_slot;
	int ret = 0;

	pr_debug("RIO: rio_hw_add_outb_message(): "
			"destid 0x%04x mbox %d buffer %p len 0x%08x\n",
			rdev->destid, mbox, buffer, len);

	if ((len < 8) || (len > RIO_MAX_MSG_SIZE)) {
		ret = -EINVAL;
		goto out;
	}

	/* Copy and clear rest of buffer */
	memcpy(priv->msg_tx_ring.virt_buffer[priv->msg_tx_ring.tx_slot], buffer, len);
	if (len < (RIO_MAX_MSG_SIZE - 4))
		memset(priv->msg_tx_ring.virt_buffer[priv->msg_tx_ring.tx_slot]
				+ len, 0, RIO_MAX_MSG_SIZE - len);

	switch(mport->phy_type) {
	case RIO_PHY_SERIAL:
		/* Set mbox field for message, and set destid */
		desc->dport = (rdev->destid << 16) | ( mbox & 0x3);

		/* Enable EOMI interrupt and priority */
		desc->dattr = 0x28000000;

		/* Set mbox field for message */
		desc->dport = mbox & 0x3;
		break;
	case RIO_PHY_PARALLEL:
		/* Enable EOMI interrupt, set priority, and set destid */
		desc->dattr = 0x28000000 | (rdev->destid << 2);
		break;
	}

	/* Set transfer size aligned to next power of 2 (in double words) */
	desc->dwcnt = is_power_of_2(len) ? len : 1 << get_bitmask_order(len);

	/* Set snooping and source buffer address */
	desc->saddr = 0x00000004 | priv->msg_tx_ring.phys_buffer[priv->msg_tx_ring.tx_slot];

	/* Increment enqueue pointer */
	setbits32(&priv->msg_regs->omr, RIO_MSG_OMR_MUI);

	/* Go to next descriptor */
	if (++priv->msg_tx_ring.tx_slot == priv->msg_tx_ring.size)
		priv->msg_tx_ring.tx_slot = 0;

      out:
	return ret;
}

EXPORT_SYMBOL_GPL(rio_hw_add_outb_message);

/**
 * fsl_rio_tx_handler - PowerPC outbound message interrupt handler
 * @irq: Linux interrupt number
 * @dev_instance: Pointer to interrupt-specific data
 *
 * Handles outbound message interrupts. Executes a register outbound
 * mailbox event handler and acks the interrupt occurence.
 */
static irqreturn_t fsl_rio_tx_handler(int irq, void *dev_instance)
{
	int osr;
	struct rio_mport *port = (struct rio_mport *)dev_instance;
	struct rio_priv *priv = port->priv;

	osr = in_be32(&priv->msg_regs->osr);

	if (unlikely(osr & RIO_MSG_OSR_TE)) {
		pr_info("RIO: outbound message transmission error\n");
		out_be32(&priv->msg_regs->osr, RIO_MSG_OSR_TE);
		goto out;
	}

	if (unlikely(osr & RIO_MSG_OSR_QOI)) {
		pr_info("RIO: outbound message queue overflow\n");
		out_be32(&priv->msg_regs->osr, RIO_MSG_OSR_QOI);
		goto out;
	}

	if (osr & RIO_MSG_OSR_EOMI) {
		u32 dqp = in_be32(&priv->msg_regs->odqdpar);
		int slot = (dqp - priv->msg_tx_ring.phys) >> 5;
		port->outb_msg[0].mcback(port, priv->msg_tx_ring.dev_id, -1, slot);

		/* Ack the end-of-message interrupt */
		out_be32(&priv->msg_regs->osr, RIO_MSG_OSR_EOMI);
	}

      out:
	return IRQ_HANDLED;
}

/**
 * rio_open_outb_mbox - Initialize PowerPC outbound mailbox
 * @mport: Master port implementing the outbound message unit
 * @dev_id: Device specific pointer to pass on event
 * @mbox: Mailbox to open
 * @entries: Number of entries in the outbound mailbox ring
 *
 * Initializes buffer ring, request the outbound message interrupt,
 * and enables the outbound message unit. Returns %0 on success and
 * %-EINVAL or %-ENOMEM on failure.
 */
int rio_open_outb_mbox(struct rio_mport *mport, void *dev_id, int mbox, int entries)
{
	int i, j, rc = 0;
	struct rio_priv *priv = mport->priv;

	if ((entries < RIO_MIN_TX_RING_SIZE) ||
	    (entries > RIO_MAX_TX_RING_SIZE) || (!is_power_of_2(entries))) {
		rc = -EINVAL;
		goto out;
	}

	/* Initialize shadow copy ring */
	priv->msg_tx_ring.dev_id = dev_id;
	priv->msg_tx_ring.size = entries;

	for (i = 0; i < priv->msg_tx_ring.size; i++) {
		priv->msg_tx_ring.virt_buffer[i] =
		     dma_alloc_coherent(NULL, RIO_MSG_BUFFER_SIZE,
					&priv->msg_tx_ring.phys_buffer[i],
					GFP_KERNEL);
		if (!priv->msg_tx_ring.virt_buffer[i]) {
			rc = -ENOMEM;
			for (j = 0; j < priv->msg_tx_ring.size; j++)
				if (priv->msg_tx_ring.virt_buffer[j])
					dma_free_coherent(NULL,
							  RIO_MSG_BUFFER_SIZE,
							  priv->msg_tx_ring.
							  virt_buffer[j],
							  priv->msg_tx_ring.
							  phys_buffer[j]);
			goto out;
		}
	}

	/* Initialize outbound message descriptor ring */
	priv->msg_tx_ring.virt = dma_alloc_coherent(NULL,
						    priv->msg_tx_ring.size *
						    RIO_MSG_DESC_SIZE,
						    &priv->msg_tx_ring.phys,
						    GFP_KERNEL);
	if (!priv->msg_tx_ring.virt) {
		rc = -ENOMEM;
		goto out_dma;
	}
	priv->msg_tx_ring.tx_slot = 0;

	/* Point dequeue/enqueue pointers at first entry in ring */
	out_be32(&priv->msg_regs->odqdpar, priv->msg_tx_ring.phys);
	out_be32(&priv->msg_regs->odqepar, priv->msg_tx_ring.phys);

	/* Configure for snooping */
	out_be32(&priv->msg_regs->osar, 0x00000004);

	/* Clear interrupt status */
	out_be32(&priv->msg_regs->osr, 0x000000b3);

	/* Hook up outbound message handler */
	rc = request_irq(IRQ_RIO_TX(mport), fsl_rio_tx_handler, 0, "msg_tx",
			mport);
	if (rc < 0)
		goto out_irq;

	/*
	 * Configure outbound message unit
	 *      Snooping
	 *      Interrupts (all enabled, except QEIE)
	 *      Chaining mode
	 *      Disable
	 */
	out_be32(&priv->msg_regs->omr, 0x00100220);

	/* Set number of entries */
	setbits32(&priv->msg_regs->omr,
		 ((get_bitmask_order(entries) - 2) << 12));

	/* Now enable the unit */
	setbits32(&priv->msg_regs->omr, 0x1);

      out:
	return rc;

      out_irq:
	dma_free_coherent(NULL, priv->msg_tx_ring.size * RIO_MSG_DESC_SIZE,
			  priv->msg_tx_ring.virt, priv->msg_tx_ring.phys);

      out_dma:
	for (i = 0; i < priv->msg_tx_ring.size; i++)
		dma_free_coherent(NULL, RIO_MSG_BUFFER_SIZE,
				  priv->msg_tx_ring.virt_buffer[i],
				  priv->msg_tx_ring.phys_buffer[i]);

	return rc;
}

/**
 * rio_close_outb_mbox - Shut down PowerPC outbound mailbox
 * @mport: Master port implementing the outbound message unit
 * @mbox: Mailbox to close
 *
 * Disables the outbound message unit, free all buffers, and
 * frees the outbound message interrupt.
 */
void rio_close_outb_mbox(struct rio_mport *mport, int mbox)
{
	struct rio_priv *priv = mport->priv;
	/* Disable inbound message unit */
	out_be32(&priv->msg_regs->omr, 0);

	/* Free ring */
	dma_free_coherent(NULL, priv->msg_tx_ring.size * RIO_MSG_DESC_SIZE,
			  priv->msg_tx_ring.virt, priv->msg_tx_ring.phys);

	/* Free interrupt */
	free_irq(IRQ_RIO_TX(mport), mport);
}

/**
 * fsl_rio_rx_handler - PowerPC inbound message interrupt handler
 * @irq: Linux interrupt number
 * @dev_instance: Pointer to interrupt-specific data
 *
 * Handles inbound message interrupts. Executes a registered inbound
 * mailbox event handler and acks the interrupt occurence.
 */
static irqreturn_t fsl_rio_rx_handler(int irq, void *dev_instance)
{
	int isr;
	struct rio_mport *port = (struct rio_mport *)dev_instance;
	struct rio_priv *priv = port->priv;

	isr = in_be32(&priv->msg_regs->isr);

	if (unlikely(isr & RIO_MSG_ISR_TE)) {
		pr_info("RIO: inbound message reception error\n");
		out_be32(&priv->msg_regs->isr, RIO_MSG_ISR_TE);
		goto out;
	}

	/* XXX Need to check/dispatch until queue empty */
	if (isr & RIO_MSG_ISR_DIQI) {
		/*
		 * We implement *only* mailbox 0, but can receive messages
		 * for any mailbox/letter to that mailbox destination. So,
		 * make the callback with an unknown/invalid mailbox number
		 * argument.
		 */
		port->inb_msg[0].mcback(port, priv->msg_rx_ring.dev_id, -1, -1);

		/* Ack the queueing interrupt */
		out_be32(&priv->msg_regs->isr, RIO_MSG_ISR_DIQI);
	}

      out:
	return IRQ_HANDLED;
}

/**
 * rio_open_inb_mbox - Initialize PowerPC inbound mailbox
 * @mport: Master port implementing the inbound message unit
 * @dev_id: Device specific pointer to pass on event
 * @mbox: Mailbox to open
 * @entries: Number of entries in the inbound mailbox ring
 *
 * Initializes buffer ring, request the inbound message interrupt,
 * and enables the inbound message unit. Returns %0 on success
 * and %-EINVAL or %-ENOMEM on failure.
 */
int rio_open_inb_mbox(struct rio_mport *mport, void *dev_id, int mbox, int entries)
{
	int i, rc = 0;
	struct rio_priv *priv = mport->priv;

	if ((entries < RIO_MIN_RX_RING_SIZE) ||
	    (entries > RIO_MAX_RX_RING_SIZE) || (!is_power_of_2(entries))) {
		rc = -EINVAL;
		goto out;
	}

	/* Initialize client buffer ring */
	priv->msg_rx_ring.dev_id = dev_id;
	priv->msg_rx_ring.size = entries;
	priv->msg_rx_ring.rx_slot = 0;
	for (i = 0; i < priv->msg_rx_ring.size; i++)
		priv->msg_rx_ring.virt_buffer[i] = NULL;

	/* Initialize inbound message ring */
	priv->msg_rx_ring.virt = dma_alloc_coherent(NULL,
						    priv->msg_rx_ring.size *
						    RIO_MAX_MSG_SIZE,
						    &priv->msg_rx_ring.phys,
						    GFP_KERNEL);
	if (!priv->msg_rx_ring.virt) {
		rc = -ENOMEM;
		goto out;
	}

	/* Point dequeue/enqueue pointers at first entry in ring */
	out_be32(&priv->msg_regs->ifqdpar, (u32) priv->msg_rx_ring.phys);
	out_be32(&priv->msg_regs->ifqepar, (u32) priv->msg_rx_ring.phys);

	/* Clear interrupt status */
	out_be32(&priv->msg_regs->isr, 0x00000091);

	/* Hook up inbound message handler */
	rc = request_irq(IRQ_RIO_RX(mport), fsl_rio_rx_handler, 0,
			 "msg_rx", mport);
	if (rc < 0) {
		dma_free_coherent(NULL, RIO_MSG_BUFFER_SIZE,
				  priv->msg_tx_ring.virt_buffer[i],
				  priv->msg_tx_ring.phys_buffer[i]);
		goto out;
	}

	/*
	 * Configure inbound message unit:
	 *      Snooping
	 *      4KB max message size
	 *      Unmask all interrupt sources
	 *      Disable
	 */
	out_be32(&priv->msg_regs->imr, 0x001b0060);

	/* Set number of queue entries */
	setbits32(&priv->msg_regs->imr,
			((get_bitmask_order(entries) - 2) << 12));

	/* Now enable the unit */
	setbits32(&priv->msg_regs->imr, 0x1);

      out:
	return rc;
}

/**
 * rio_close_inb_mbox - Shut down PowerPC inbound mailbox
 * @mport: Master port implementing the inbound message unit
 * @mbox: Mailbox to close
 *
 * Disables the inbound message unit, free all buffers, and
 * frees the inbound message interrupt.
 */
void rio_close_inb_mbox(struct rio_mport *mport, int mbox)
{
	struct rio_priv *priv = mport->priv;
	/* Disable inbound message unit */
	out_be32(&priv->msg_regs->imr, 0);

	/* Free ring */
	dma_free_coherent(NULL, priv->msg_rx_ring.size * RIO_MAX_MSG_SIZE,
			  priv->msg_rx_ring.virt, priv->msg_rx_ring.phys);

	/* Free interrupt */
	free_irq(IRQ_RIO_RX(mport), mport);
}

/**
 * rio_hw_add_inb_buffer - Add buffer to the PowerPC inbound message queue
 * @mport: Master port implementing the inbound message unit
 * @mbox: Inbound mailbox number
 * @buf: Buffer to add to inbound queue
 *
 * Adds the @buf buffer to the PowerPC inbound message queue. Returns
 * %0 on success or %-EINVAL on failure.
 */
int rio_hw_add_inb_buffer(struct rio_mport *mport, int mbox, void *buf)
{
	int rc = 0;
	struct rio_priv *priv = mport->priv;

	pr_debug("RIO: rio_hw_add_inb_buffer(), msg_rx_ring.rx_slot %d\n",
		 priv->msg_rx_ring.rx_slot);

	if (unlikely(priv->msg_rx_ring.virt_buffer[priv->msg_rx_ring.rx_slot])) {
		ERR("error adding inbound buffer %d, buffer exists\n",
		       priv->msg_rx_ring.rx_slot);
		rc = -EINVAL;
		goto out;
	}

	priv->msg_rx_ring.virt_buffer[priv->msg_rx_ring.rx_slot] = buf;
	if (++priv->msg_rx_ring.rx_slot == priv->msg_rx_ring.size)
		priv->msg_rx_ring.rx_slot = 0;

      out:
	return rc;
}

EXPORT_SYMBOL_GPL(rio_hw_add_inb_buffer);

/**
 * rio_hw_get_inb_message - Fetch inbound message from the PowerPC message unit
 * @mport: Master port implementing the inbound message unit
 * @mbox: Inbound mailbox number
 *
 * Gets the next available inbound message from the inbound message queue.
 * A pointer to the message is returned on success or NULL on failure.
 */
void *rio_hw_get_inb_message(struct rio_mport *mport, int mbox)
{
	u32 phys_buf;
	void *virt_buf = NULL;
	void *buf = NULL;
	int buf_idx;
	struct rio_priv *priv = mport->priv;

	phys_buf = in_be32(&priv->msg_regs->ifqdpar);

	/* If no more messages, then bail out */
	if (phys_buf == in_be32(&priv->msg_regs->ifqepar))
		goto out2;

	virt_buf = priv->msg_rx_ring.virt + (phys_buf - priv->msg_rx_ring.phys);
	buf_idx = (phys_buf - priv->msg_rx_ring.phys) / RIO_MAX_MSG_SIZE;
	buf = priv->msg_rx_ring.virt_buffer[buf_idx];

	if (unlikely(!buf)) {
		ERR("inbound message copy failed, no buffers\n");
		goto out1;
	}

	/* Copy max message size, caller is expected to allocate that big */
	memcpy(buf, virt_buf, RIO_MAX_MSG_SIZE);

	/* Clear the available buffer */
	priv->msg_rx_ring.virt_buffer[buf_idx] = NULL;

      out1:
	setbits32(&priv->msg_regs->imr, RIO_MSG_IMR_MI);

      out2:
	return buf;
}

EXPORT_SYMBOL_GPL(rio_hw_get_inb_message);

/**
 * fsl_rio_dbell_handler - PowerPC doorbell interrupt handler
 * @irq: Linux interrupt number
 * @dev_instance: Pointer to interrupt-specific data
 *
 * Handles doorbell interrupts. Parses a list of registered
 * doorbell event handlers and executes a matching event handler.
 */
static irqreturn_t fsl_rio_dbell_handler(int irq, void *dev_instance)
{
	int dsr;
	struct rio_mport *port = (struct rio_mport *)dev_instance;
	struct rio_priv *priv = port->priv;

	dsr = in_be32(&priv->msg_regs->dsr);

	if (dsr & DOORBELL_DSR_TE) {
		pr_info("RIO: doorbell reception error\n");
		out_be32(&priv->msg_regs->dsr, DOORBELL_DSR_TE);
		goto out;
	}

	if (dsr & DOORBELL_DSR_QFI) {
		pr_info("RIO: doorbell queue full\n");
		out_be32(&priv->msg_regs->dsr, DOORBELL_DSR_QFI);
		goto out;
	}

	/* XXX Need to check/dispatch until queue empty */
	if (dsr & DOORBELL_DSR_DIQI) {
		u32 dmsg =
		    (u32) priv->dbell_ring.virt +
		    (in_be32(&priv->msg_regs->dqdpar) & 0xfff);
		struct rio_dbell *dbell;
		int found = 0;

		pr_debug("RIO: processing doorbell, sid 0x%02x tid 0x%02x "
				"info 0x%04x\n", DBELL_SID(dmsg),
				DBELL_TID(dmsg), DBELL_INF(dmsg));

		list_for_each_entry(dbell, &port->dbells, node) {
			if ((dbell->res->start <= DBELL_INF(dmsg)) &&
			    (dbell->res->end >= DBELL_INF(dmsg))) {
				found = 1;
				break;
			}
		}
		if (found) {
			dbell->dinb(port, dbell->dev_id, DBELL_SID(dmsg), DBELL_TID(dmsg),
				    DBELL_INF(dmsg));
		} else {
			pr_debug("RIO: spurious doorbell, sid 0x%02x "
					"tid 0x%02x info %4.4x\n",
					DBELL_SID(dmsg), DBELL_TID(dmsg),
					DBELL_INF(dmsg));
		}
		setbits32(&priv->msg_regs->dmr, DOORBELL_DMR_DI);
		out_be32(&priv->msg_regs->dsr, DOORBELL_DSR_DIQI);
	}

      out:
	return IRQ_HANDLED;
}

/**
 * fsl_rio_doorbell_init - PowerPC doorbell interface init
 * @mport: Master port implementing the inbound doorbell unit
 *
 * Initializes doorbell unit hardware and inbound DMA buffer
 * ring. Called from fsl_rio_setup(). Returns %0 on success
 * or %-ENOMEM on failure.
 */
static int fsl_rio_doorbell_init(struct rio_mport *mport, phys_addr_t dbaddr)
{
	int rc = 0;
	struct rio_priv *priv = mport->priv;

	/* Map outbound doorbell window immediately after maintenance window */
	priv->dbell_win = ioremap(dbaddr, RIO_DBELL_WIN_SIZE);
	if (!priv->dbell_win) {
		ERR("unable to map outbound doorbell window\n");
		rc = -ENOMEM;
		goto out;
	}

	/* Initialize inbound doorbells */
	priv->dbell_ring.virt = dma_alloc_coherent(NULL,
						   512 * DOORBELL_MESSAGE_SIZE,
						   &priv->dbell_ring.phys,
						   GFP_KERNEL);
	if (!priv->dbell_ring.virt) {
		ERR("unable allocate inbound doorbell ring\n");
		rc = -ENOMEM;
		iounmap(priv->dbell_win);
		goto out;
	}

	/* Point dequeue/enqueue pointers at first entry in ring */
	out_be32(&priv->msg_regs->dqdpar, (u32) priv->dbell_ring.phys);
	out_be32(&priv->msg_regs->dqepar, (u32) priv->dbell_ring.phys);

	/* Clear interrupt status */
	out_be32(&priv->msg_regs->dsr, 0x00000091);

	/* Hook up doorbell handler */
	rc = request_irq(IRQ_RIO_BELL(mport), fsl_rio_dbell_handler, 0,
			 "dbell_rx", mport);
	if (rc < 0) {
		iounmap(priv->dbell_win);
		dma_free_coherent(NULL, 512 * DOORBELL_MESSAGE_SIZE,
				  priv->dbell_ring.virt, priv->dbell_ring.phys);
		ERR("unable to request inbound doorbell irq");
		goto out;
	}

	/* Configure doorbells for snooping, 512 entries, and enable */
	out_be32(&priv->msg_regs->dmr, 0x00108161);

      out:
	return rc;
}

u32 rio_get_mport_id(struct rio_mport *mport)
{
	u32 mport_id = 0;

	rio_local_read_config_32(mport, 0x60, &mport_id);
	mport_id = mport->sys_size ? (mport_id & 0xffff) : ((mport_id >> 16) & 0xff);
	return mport_id;

}

static char *cmdline = NULL;

static int fsl_rio_get_hdid(int index)
{
	/* XXX Need to parse multiple entries in some format */
	if (!cmdline)
		return -1;

	return simple_strtol(cmdline, NULL, 0);
}

static int fsl_rio_get_cmdline(char *s)
{
	if (!s)
		return 0;

	cmdline = s;
	return 1;
}

__setup("riohdid=", fsl_rio_get_cmdline);

static struct rio_mem_ops fsl_mem_ops = {
	.map_inb = fsl_rio_map_inb_mem,
	.map_outb = fsl_rio_map_outb_mem,
	.unmap_inb = fsl_rio_unmap_inb_mem,
	.unmap_outb = fsl_rio_unmap_outb_mem,
};

static inline void fsl_rio_info(u32 ccsr)
{
	if (ccsr & 1) {
		/* Serial phy */
		INFO("Hardware port width: ");
		switch (ccsr >> 30) {
		case 0:
			printk("1\n");
			break;
		case 1:
			printk("4\n");
			break;
		default:
			printk("Unknown\n");
			break;;
		}

		INFO("Training connection status: ");
		switch ((ccsr >> 27) & 7) {
		case 0:
			printk("Single-lane 0\n");
			break;
		case 1:
			printk("Single-lane 2\n");
			break;
		case 2:
			printk("Four-lane\n");
			break;
		default:
			printk("Unknown\n");
		}
	} else {
		/* Parallel phy */
		if (ccsr & 0x80000000)
			INFO("Output port operating in 8-bit mode\n");
		if (ccsr & 0x08000000)
			INFO("Input port operating in 8-bit mode\n");
	}

}

static inline u8 hw_port_width(u32 ccsr)
{
	u8 pw = ccsr >> 30;
	switch (pw) {
	case 0:
		return 1;
	case 1:
		return 4;
	default:
		return 0;
	}
}

static int of_cells_get(struct device_node *np, const char *str)
{
	struct device_node *tmp = NULL;
	const int *var = NULL;

	var = of_get_property(np, str, NULL);
	tmp = of_get_parent(np);

	while (!var && tmp) {
		var = (int *)of_get_property(tmp, str, NULL);
		of_node_put(tmp);
		tmp = of_get_parent(np);
	}

	return (var ? *var : 0);
}

/**
 * fsl_rio_setup - Setup PowerPC RapidIO interface
 *
 * Initializes PowerPC RapidIO hardware interface, configures
 * master port with system-specific info, and registers the
 * master port with the RapidIO subsystem.
 */
int fsl_rio_setup(struct of_device *dev)
{
	struct rio_ops *ops = NULL;
	struct rio_mport *port = NULL;
	const u32 *dt_range;
	int rlen = 0;
	resource_size_t law_start = 0, law_size = 0;
	struct resource regs;
	int rc;
	enum rio_phy_type phy_type;
	volatile void __iomem *regs_win = NULL;
	struct rio_priv *priv = NULL;
	u32 ccsr;
	int paw, aw, psw;
	struct device_node *pa;

	if (!dev->node) {
		ERR("Dev ofnode is NULL\n");
		return -EFAULT;
	}

	dt_range = of_get_property(dev->node, "ranges", &rlen);
	if (!dt_range) {
		ERR("Can't get %s property 'ranges'\n", dev->node->full_name);
		return -EFAULT;
	}

	aw = of_cells_get(dev->node, "#address-cells");
	pa = of_get_parent(dev->node);
	paw = of_cells_get(pa, "#address-cells");
	psw = of_cells_get(pa, "#size-cells");
	of_node_put(pa);

	law_start = of_read_number(dt_range + aw, paw);
	law_size = of_read_number(dt_range + aw + paw, psw);

	rc = of_address_to_resource(dev->node, 0, &regs);
	if (rc) {
		ERR("Can't get %s property 'reg'\n", dev->node->full_name);
		return -EFAULT;
	}
	INFO("Of-device full name %s\n", dev->node->full_name);
	INFO("LAW start 0x%016llx, size 0x%016llx.\n", (u64)law_start,
						(u64)law_size);
	INFO("Regs start 0x%08x size 0x%08x\n",	regs.start,
						regs.end - regs.start + 1);

	regs_win = ioremap(regs.start, regs.end - regs.start + 1);
	if (!regs_win) {
		ERR("Can't remap io for 'regs_win'\n");
		rc = -ENOMEM;
		goto err;
	}

	/* Probe the RapidIO phy type */
	ccsr = in_be32(regs_win + RIO_CCSR);
	if (ccsr & 1)
		phy_type = RIO_PHY_SERIAL;
	else
		phy_type = RIO_PHY_PARALLEL;
	INFO("Phy type: ");
	switch (phy_type) {
	case RIO_PHY_SERIAL:
		printk("serial\n");
		break;
	case RIO_PHY_PARALLEL:
		printk("parallel");
		break;
	default:
		printk("Unknown type %d\n", phy_type);
		rc = -EINVAL;
		goto err;
	};
	fsl_rio_info(ccsr);


	/* Checking the port training status */
	if (in_be32((regs_win + RIO_ESCSR)) & 1) {
		ERR("Port is not ready. Try to restart connection...\n");
		switch (phy_type) {
		case RIO_PHY_SERIAL:
			/* Disable ports */
			out_be32(regs_win + RIO_CCSR, 0);
			/* Set 1x lane */
			setbits32(regs_win + RIO_CCSR, 0x02000000);
			/* Enable ports */
			setbits32(regs_win + RIO_CCSR, 0x00600000);
			break;
		case RIO_PHY_PARALLEL:
			/* Disable ports */
			out_be32(regs_win + RIO_CCSR, 0x22000000);
			/* Enable ports */
			out_be32(regs_win + RIO_CCSR, 0x44000000);
			break;
		}
		if (in_be32((regs_win + RIO_ESCSR)) & 1) {
			ERR("Port restart failed.\n");
			rc = -ENOLINK;
			goto err;
		}
		INFO("Port restart success!");
		ccsr = in_be32(regs_win + RIO_CCSR);
		fsl_rio_info(ccsr);
	}

	ops = kzalloc(sizeof(struct rio_ops), GFP_KERNEL);
	ops->lcread = fsl_local_config_read;
	ops->lcwrite = fsl_local_config_write;
	ops->cread = fsl_rio_config_read;
	ops->cwrite = fsl_rio_config_write;
	ops->dsend = fsl_rio_doorbell_send;

	port = kzalloc(sizeof(struct rio_mport), GFP_KERNEL);
	if (!port) {
		ERR("Can't alloc memory for 'port'\n");
		rc = -ENOMEM;
		goto err;
	}
	port->id = 0;
	port->index = 0;

	port->sys_size = (in_be32((regs_win + RIO_PEF_CAR))
				& RIO_PEF_CTLS) >> 4;
	INFO("RapidIO Common Transport System size: %d\n",
			port->sys_size ? 65536 : 256);

	port->phy_type = phy_type;

	priv = kzalloc(sizeof(struct rio_priv), GFP_KERNEL);
	if (!priv) {
		ERR("Can't alloc memory for 'priv'\n");
		rc = -ENOMEM;
		goto err;
	}
	port->priv = priv;
	priv->regs_win = regs_win;
	INIT_LIST_HEAD(&port->dbells);
	port->iores.start = law_start;
	port->iores.end = law_start + law_size;
	port->iores.flags = IORESOURCE_MEM;
	port->iores.name = "rio_io_win";

	priv->bellirq = irq_of_parse_and_map(dev->node, 2);
	priv->txirq = irq_of_parse_and_map(dev->node, 3);
	priv->rxirq = irq_of_parse_and_map(dev->node, 4);
	INFO("bellirq: %d, txirq: %d, rxirq %d\n", priv->bellirq,
				priv->txirq, priv->rxirq);

	rio_init_dbell_res(&port->riores[RIO_DOORBELL_RESOURCE], 0, 0xffff);
	rio_init_mbox_res(&port->riores[RIO_INB_MBOX_RESOURCE], 0, 0);
	rio_init_mbox_res(&port->riores[RIO_OUTB_MBOX_RESOURCE], 0, 0);

	strcpy(port->name, "RIO0 mport");

	port->ops = ops;
	port->mops = &fsl_mem_ops;
	port->host_deviceid = fsl_rio_get_hdid(port->id);

	rio_register_mport(port);

	priv->atmu_regs = (struct rio_atmu_regs *)(regs_win +
			RIO_ATMU_REGS_OFFSET);
	priv->maint_atmu_regs = priv->atmu_regs + 1;
	priv->dbell_atmu_regs = priv->atmu_regs + 2;
	priv->msg_regs = (struct rio_msg_regs *)(regs_win +
				((port->phy_type == RIO_PHY_SERIAL)
					? RIO_S_MSG_REGS_OFFSET
					: RIO_P_MSG_REGS_OFFSET));

	/* Set to receive any dist ID for serial RapidIO controller. */
	if (port->phy_type == RIO_PHY_SERIAL)
		out_be32((regs_win + RIO_ISR_AACR), RIO_ISR_AACR_AA);

	/* Configure maintenance transaction window */
	if (!rio_request_io_region(port, NULL, law_start, RIO_MAINT_WIN_SIZE,
				"maint_win", RIO_RESOURCE_MAINT, NULL)) {
		rc = -EPERM;
		ERR("request maint win error!\n");
		goto err;
	}
	out_be32(&priv->maint_atmu_regs->rowbar,
				(law_start >> 12) & 0xffffff);
	out_be32(&priv->maint_atmu_regs->rowar, 0x80077000
			| (__ilog2(RIO_MAINT_WIN_SIZE) - 1));

	priv->maint_win = ioremap(law_start, RIO_MAINT_WIN_SIZE);

	/* Configure outbound doorbell window */
	if (!rio_request_io_region(port, NULL, law_start + RIO_MAINT_WIN_SIZE,
			RIO_DBELL_WIN_SIZE,
			"dbell_win", RIO_RESOURCE_DOORBELL, NULL)) {
		rc = -EPERM;
		ERR("request doorbell win error!\n");
		goto err;
	}
	out_be32(&priv->dbell_atmu_regs->rowbar, ((law_start +
					RIO_MAINT_WIN_SIZE) >> 12) & 0xfffff);
	out_be32(&priv->dbell_atmu_regs->rowar, 0x80042000
			| (__ilog2(RIO_DBELL_WIN_SIZE) - 1));
	rc = fsl_rio_doorbell_init(port, law_start + RIO_MAINT_WIN_SIZE);
	if (rc)
		goto err;

	return 0;

err:
	if (regs_win)
		iounmap(regs_win);
	if (ops)
		kfree(ops);
	if (port)
		kfree(port);
	if (priv)
		kfree(priv);
	return rc;
}
