// Copyright (c) 2003-2009 Synology Inc. All rights reserved.
#ifdef MY_ABC_HERE
#include <scsi/scsi_device.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/sd.h>
#include <scsi/scsi.h>
#include <linux/genhd.h>

/**
 * Please modify this when SCSI_DISK* is bigger than 15 when 
 * porting kernel 
 * 
 * @param major_idx 
 * 
 * @return unsigned char 
 */
static unsigned char
blIsScsiDevice(int major)
{
	unsigned char ret = 0;

	switch (major) {
	case SCSI_DISK0_MAJOR:
	case SCSI_DISK1_MAJOR ... SCSI_DISK7_MAJOR:
	case SCSI_DISK8_MAJOR ... SCSI_DISK15_MAJOR:
		ret = 1;
		break;
	default:
		break;
	}

	return ret;
}

/**
 * Set the partition to specify remap mode
 * 
 * @param gd     [IN] general disk. Should not be NULL
 * @param phd    [IN] partition. Should not be NULL
 * @param blAutoRemap
 *               [IN] remap mode
 */
void
PartitionRemapModeSet(struct gendisk *gd,
					  struct hd_struct *phd,
					  unsigned char blAutoRemap)
{
	struct scsi_disk *sdkp;
	struct scsi_device *sdev;

	if (!gd || !phd) {
		goto END;
	}

	phd->auto_remap = blAutoRemap;
	if (!blAutoRemap) {
		if (!blIsScsiDevice(gd->major)) {
			/* Only work for scsi disk */
			printk("This is not a kind of scsi disk %d\n", gd->major);
			goto END;
		}

		sdkp = container_of(gd->private_data, struct scsi_disk, driver);
		if (!sdkp) {
			printk(" sdkp is NULL\n");
			goto END;
		}

		sdev = sdkp->device;
		if(!sdev) {
			printk(" sdev is NULL\n");
			goto END;
		}
		sdev->auto_remap = 0;
	}	
END:
	return;
}

/**
 * Set the scsi device to specift remap mode.
 * 
 * And also set the relative partition to that mode.
 * 
 * @param sdev   [IN] scsi devide. Should not be NULL.
 * @param blAutoRemap
 *               [IN] auto remap mode.
 */
void
ScsiRemapModeSet(struct scsi_device *sdev, 
				 unsigned char blAutoRemap)
{
	struct scsi_disk *sdkp;
	struct gendisk *gd;
	struct hd_struct *phd;
	int i = 0;

	if (!sdev) {
		goto END;
	}
	
	if (TYPE_DISK != sdev->type) {
		printk("Only support scsi disk\n");
		goto END;
	}

	sdev->auto_remap = blAutoRemap;
	sdkp = dev_get_drvdata(&sdev->sdev_gendev);
	if (!sdkp) {
		goto END;
	}

	gd = sdkp->disk;
	if (!gd) {
		goto END;
	}

	/* disk part */
	for (i = 0; i < gd->minors; i++) {
		phd = gd->part[i];		
		if (!phd || !phd->nr_sects)
			continue;

		phd->auto_remap = blAutoRemap;
	}
END:
	return;
}

/**
 * Set the block device to specify remap mode
 * 
 * @param bdev   [IN] block device. Should not be NULL.
 * @param blAutoRemap
 *               [IN] remap mode
 */
void
RaidRemapModeSet(struct block_device *bdev, unsigned char blAutoRemap)
{
	struct gendisk *disk = NULL;
	struct scsi_disk *sdkp;

	if (!bdev) {
		WARN_ON(bdev == NULL);
		return;
	}

	disk = bdev->bd_disk;
	if (!disk) {
		WARN_ON(bdev == NULL);
		return;
	}

	if (!blIsScsiDevice(disk->major)) {
		/* Only work for scsi disk */
		printk("This is not a kind of scsi disk %d\n", disk->major);
		return;
	}
	
	if (bdev->bd_part) {
		/* is a partition of some disks */
		bdev->bd_part->auto_remap = blAutoRemap;
	} else {
		/* is whole disk */
		sdkp = container_of(disk->private_data, struct scsi_disk, driver);
		if (!sdkp) {
			WARN_ON(!sdkp);
			return;
		}
		ScsiRemapModeSet(sdkp->device, blAutoRemap);
	}
}

unsigned char
blSectorNeedAutoRemap(struct scsi_cmnd *scsi_cmd,
					  sector_t lba)
{
	struct scsi_device *sdev;
	struct scsi_disk *sdkp;
	struct gendisk *gd;
	struct hd_struct *phd;
	char szName[BDEVNAME_SIZE];
	sector_t start, end;
	u8 ret = 0;
	int i = 0;

	if (!scsi_cmd) {
		WARN_ON(1);
		goto END;
	}

	sdev = scsi_cmd->device;
	if (!sdev) {
		WARN_ON(1);
		goto END;
	}

	if (TYPE_DISK != sdev->type) {
		printk("Only support scsi disk\n");
		goto END;
	}

	/* global disk auto remap */
	if (sdev->auto_remap) {
		ret = 1;
		printk("%s auto remap is on\n", sdev->sdev_gendev.bus_id);
		goto END;
	}

	sdkp = dev_get_drvdata(&sdev->sdev_gendev);
	if (!sdkp) {
		goto END;
	}

	gd = sdkp->disk;
	if (!gd) {
		goto END;
	}

	/* disk part */
	for (i = 0; i < gd->minors; i++) {
		phd = gd->part[i];		
		if (!phd || !phd->nr_sects)
			continue;

		start = phd->start_sect;
		end = phd->nr_sects + start - 1;

		if (lba >= start && lba <= end) {
			printk("LBA %zd start %zd end %zd\n", lba, start ,end);
			ret = phd->auto_remap;
			printk("%s auto_remap %u\n", disk_name(gd, i+1, szName), phd->auto_remap);
		}
	}
END:
	return ret;
}

EXPORT_SYMBOL(blSectorNeedAutoRemap);
EXPORT_SYMBOL(RaidRemapModeSet);
EXPORT_SYMBOL(ScsiRemapModeSet);
EXPORT_SYMBOL(PartitionRemapModeSet);
#endif /* MY_ABC_HERE */
