/*
 *  linux/fs/stat.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#include <linux/module.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/highuid.h>
#include <linux/fs.h>
#include <linux/namei.h>
#include <linux/security.h>
#include <linux/syscalls.h>
#include <linux/pagemap.h>

#include <asm/uaccess.h>
#include <asm/unistd.h>

#ifdef MY_ABC_HERE
extern int SynoDebugFlag;
#endif
void generic_fillattr(struct inode *inode, struct kstat *stat)
{
	stat->dev = inode->i_sb->s_dev;
	stat->ino = inode->i_ino;
	stat->mode = inode->i_mode;
#ifdef MY_ABC_HERE
	stat->SynoMode = inode->i_mode2;
#endif
#ifdef MY_DEF_HERE
	stat->syno_archive_version = inode->i_archive_version;
#endif
	stat->nlink = inode->i_nlink;
	stat->uid = inode->i_uid;
	stat->gid = inode->i_gid;
	stat->rdev = inode->i_rdev;
	stat->atime = inode->i_atime;
	stat->mtime = inode->i_mtime;
	stat->ctime = inode->i_ctime;
#ifdef MY_ABC_HERE
	stat->SynoCreateTime = inode->i_CreateTime;
#endif
	stat->size = i_size_read(inode);
	stat->blocks = inode->i_blocks;
	stat->blksize = (1 << inode->i_blkbits);
}

EXPORT_SYMBOL(generic_fillattr);

int vfs_getattr(struct vfsmount *mnt, struct dentry *dentry, struct kstat *stat)
{
	struct inode *inode = dentry->d_inode;
	int retval;

	retval = security_inode_getattr(mnt, dentry);
	if (retval)
		return retval;

	if (inode->i_op->getattr)
		return inode->i_op->getattr(mnt, dentry, stat);

	generic_fillattr(inode, stat);
	return 0;
}

EXPORT_SYMBOL(vfs_getattr);

int vfs_stat_fd(int dfd, char __user *name, struct kstat *stat)
{
	struct nameidata nd;
	int error;

#ifdef MY_ABC_HERE
	memset(&nd, 0, sizeof(struct nameidata));
#endif
	error = __user_walk_fd(dfd, name, LOOKUP_FOLLOW, &nd);
	if (!error) {
		error = vfs_getattr(nd.mnt, nd.dentry, stat);
		path_release(&nd);	
	}
	return error;
}

int vfs_stat(char __user *name, struct kstat *stat)
{
	return vfs_stat_fd(AT_FDCWD, name, stat);
}
EXPORT_SYMBOL(vfs_stat);

int vfs_lstat_fd(int dfd, char __user *name, struct kstat *stat)
{
	struct nameidata nd;
	int error;

#ifdef MY_ABC_HERE
	memset(&nd, 0, sizeof(struct nameidata));
#endif
	error = __user_walk_fd(dfd, name, 0, &nd);
	if (!error) {
		error = vfs_getattr(nd.mnt, nd.dentry, stat);
		path_release(&nd);
	}
	return error;
}

int vfs_lstat(char __user *name, struct kstat *stat)
{
	return vfs_lstat_fd(AT_FDCWD, name, stat);
}

EXPORT_SYMBOL(vfs_lstat);

int vfs_fstat(unsigned int fd, struct kstat *stat)
{
	struct file *f = fget(fd);
	int error = -EBADF;

	if (f) {
		error = vfs_getattr(f->f_path.mnt, f->f_path.dentry, stat);
		fput(f);
	}
	return error;
}

EXPORT_SYMBOL(vfs_fstat);

#ifdef MY_ABC_HERE
#include <linux/synolib.h>
extern int syno_hibernation_log_sec;
#endif
#ifdef __ARCH_WANT_OLD_STAT

/*
 * For backward compatibility?  Maybe this should be moved
 * into arch/i386 instead?
 */
static int cp_old_stat(struct kstat *stat, struct __old_kernel_stat __user * statbuf)
{
	static int warncount = 5;
	struct __old_kernel_stat tmp;
	
	if (warncount > 0) {
		warncount--;
		printk(KERN_WARNING "VFS: Warning: %s using old stat() call. Recompile your binary.\n",
			current->comm);
	} else if (warncount < 0) {
		/* it's laughable, but... */
		warncount = 0;
	}

	memset(&tmp, 0, sizeof(struct __old_kernel_stat));
	tmp.st_dev = old_encode_dev(stat->dev);
	tmp.st_ino = stat->ino;
	if (sizeof(tmp.st_ino) < sizeof(stat->ino) && tmp.st_ino != stat->ino)
		return -EOVERFLOW;
	tmp.st_mode = stat->mode;
	tmp.st_nlink = stat->nlink;
	if (tmp.st_nlink != stat->nlink)
		return -EOVERFLOW;
	SET_UID(tmp.st_uid, stat->uid);
	SET_GID(tmp.st_gid, stat->gid);
	tmp.st_rdev = old_encode_dev(stat->rdev);
#if BITS_PER_LONG == 32
	if (stat->size > MAX_NON_LFS)
		return -EOVERFLOW;
#endif	
	tmp.st_size = stat->size;
	tmp.st_atime = stat->atime.tv_sec;
	tmp.st_mtime = stat->mtime.tv_sec;
	tmp.st_ctime = stat->ctime.tv_sec;
	return copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : 0;
}

asmlinkage long sys_stat(char __user * filename, struct __old_kernel_stat __user * statbuf)
{
	struct kstat stat;
	int error = vfs_stat_fd(AT_FDCWD, filename, &stat);

#ifdef MY_ABC_HERE
	if(syno_hibernation_log_sec > 0) {
		syno_do_hibernation_log(filename);
	}
#endif

	if (!error)
		error = cp_old_stat(&stat, statbuf);

	return error;
}
asmlinkage long sys_lstat(char __user * filename, struct __old_kernel_stat __user * statbuf)
{
	struct kstat stat;
	int error = vfs_lstat_fd(AT_FDCWD, filename, &stat);

	if (!error)
		error = cp_old_stat(&stat, statbuf);

	return error;
}
asmlinkage long sys_fstat(unsigned int fd, struct __old_kernel_stat __user * statbuf)
{
	struct kstat stat;
	int error = vfs_fstat(fd, &stat);

	if (!error)
		error = cp_old_stat(&stat, statbuf);

	return error;
}

#endif /* __ARCH_WANT_OLD_STAT */

static int cp_new_stat(struct kstat *stat, struct stat __user *statbuf)
{
	struct stat tmp;

#if BITS_PER_LONG == 32
	if (!old_valid_dev(stat->dev) || !old_valid_dev(stat->rdev))
		return -EOVERFLOW;
#else
	if (!new_valid_dev(stat->dev) || !new_valid_dev(stat->rdev))
		return -EOVERFLOW;
#endif

	memset(&tmp, 0, sizeof(tmp));
#if BITS_PER_LONG == 32
	tmp.st_dev = old_encode_dev(stat->dev);
#else
	tmp.st_dev = new_encode_dev(stat->dev);
#endif
	tmp.st_ino = stat->ino;
	if (sizeof(tmp.st_ino) < sizeof(stat->ino) && tmp.st_ino != stat->ino)
		return -EOVERFLOW;
	tmp.st_mode = stat->mode;
	tmp.st_nlink = stat->nlink;
	if (tmp.st_nlink != stat->nlink)
		return -EOVERFLOW;
	SET_UID(tmp.st_uid, stat->uid);
	SET_GID(tmp.st_gid, stat->gid);
#if BITS_PER_LONG == 32
	tmp.st_rdev = old_encode_dev(stat->rdev);
#else
	tmp.st_rdev = new_encode_dev(stat->rdev);
#endif
#if BITS_PER_LONG == 32
	if (stat->size > MAX_NON_LFS)
		return -EOVERFLOW;
#endif	
	tmp.st_size = stat->size;
	tmp.st_atime = stat->atime.tv_sec;
	tmp.st_mtime = stat->mtime.tv_sec;
	tmp.st_ctime = stat->ctime.tv_sec;
#ifdef STAT_HAVE_NSEC
	tmp.st_atime_nsec = stat->atime.tv_nsec;
	tmp.st_mtime_nsec = stat->mtime.tv_nsec;
	tmp.st_ctime_nsec = stat->ctime.tv_nsec;
#endif
#ifdef MY_ABC_HERE
	tmp.st_SynoCreateTime = stat->SynoCreateTime.tv_sec;
#endif
#ifdef MY_ABC_HERE
	tmp.st_SynoMode = stat->SynoMode;
#endif
#ifdef MY_DEF_HERE
	tmp.st_syno_achv_ver = stat->syno_archive_version;
#endif
	tmp.st_blocks = stat->blocks;
	tmp.st_blksize = stat->blksize;
	return copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : 0;
}

asmlinkage long sys_newstat(char __user *filename, struct stat __user *statbuf)
{
	struct kstat stat;
	int error = vfs_stat_fd(AT_FDCWD, filename, &stat);

	if (!error)
		error = cp_new_stat(&stat, statbuf);

	return error;
}

#ifdef MY_ABC_HERE
#include "../fs/ecryptfs/ecryptfs_kernel.h"
int (*fecryptfs_decode_and_decrypt_filename)(char **plaintext_name,
                                        size_t *plaintext_name_size,
                                        struct dentry *ecryptfs_dir_dentry,
                                        const char *name, size_t name_size) = NULL;
EXPORT_SYMBOL(fecryptfs_decode_and_decrypt_filename);

asmlinkage long sys_SYNOEcryptName(char __user * src, char __user * dst)
{
	int                               err = -1;
	struct qstr                      *lower_path = NULL;
	struct nameidata                  nd;
	struct ecryptfs_dentry_info      *crypt_dentry = NULL;

	if (NULL == src || NULL == dst) {
		return -EINVAL;
	}

	memset(&nd, 0, sizeof(struct nameidata));
	err = __user_walk_fd(AT_FDCWD, src, LOOKUP_FOLLOW, &nd);
	if (err) {
		return -ENOENT;
	}
	if (!nd.dentry->d_inode->i_sb->s_type || 
		strcmp(nd.dentry->d_inode->i_sb->s_type->name, "ecryptfs")) {
		err = -EINVAL;
		goto OUT_RELEASE;
	}
	crypt_dentry = ecryptfs_dentry_to_private(nd.dentry);
	if (!crypt_dentry) {
		err = -EINVAL;
		goto OUT_RELEASE;
	}
	lower_path = &crypt_dentry->lower_path.dentry->d_name;
	err = copy_to_user(dst, lower_path->name, lower_path->len + 1);

OUT_RELEASE:
	path_release(&nd);

	return err;
}

asmlinkage long sys_SYNODecryptName(char __user * root, char __user * src, char __user * dst)
{
	int                           err;
	int                           plaintext_name_size = 0;
	char                         *plaintext_name = NULL;
	char                         *token = NULL;
	char                         *szTarget = NULL;
	char                         *root_name = NULL;
	char                         *src_name = NULL;
	char                         *src_walk = NULL;
	struct nameidata              nd;
	struct ecryptfs_dentry_info  *crypt_dentry = NULL;

	if (NULL == src || NULL == root || NULL == dst) {
		return -EINVAL;
	}
	if (!fecryptfs_decode_and_decrypt_filename) {
		return -EPERM;
	}

	src_name = getname(src);
	if (IS_ERR(src_name)) {
		err = PTR_ERR(src_name);
		goto OUT_RELEASE;
	}
	// strsep() will move src_walk, so we should keep the head for free mem
	src_walk = src_name;
	root_name = getname(root);
	if (IS_ERR(root_name)) {
		err = PTR_ERR(root_name);
		goto OUT_RELEASE;
	}
	szTarget = kmalloc(PATH_MAX, GFP_KERNEL);
	if (!szTarget) {
		err = -ENOMEM;
		goto OUT_RELEASE;
	}
	strncpy(szTarget, root_name, PATH_MAX-1);
	szTarget[PATH_MAX-1] = '\0';

	token = strsep(&src_walk, "/");
	if (*token == '\0') {
		token = strsep(&src_walk, "/");
	}
	while (token) {
		memset(&nd, 0, sizeof(struct nameidata));
		err = path_lookup(szTarget, LOOKUP_FOLLOW, &nd);
		if (err) {
			goto OUT_RELEASE;
		}
		err = fecryptfs_decode_and_decrypt_filename(
			&plaintext_name, &plaintext_name_size, nd.dentry, token, strlen(token));
		if (err) {
			path_release(&nd);
			goto OUT_RELEASE;
		}
		if (PATH_MAX < strlen(szTarget) + plaintext_name_size + 1) {
			path_release(&nd);
			goto OUT_RELEASE;
		}
		strcat(szTarget, "/");
		strcat(szTarget, plaintext_name);

		kfree(plaintext_name);
		plaintext_name = NULL;
		path_release(&nd);

		token = strsep(&src_walk, "/");
	}
	err = copy_to_user(dst, szTarget, strlen(szTarget)+1);

OUT_RELEASE:
	if (plaintext_name) {
		kfree(plaintext_name);
	}
	if (szTarget) {
		kfree(szTarget);
	}
	if (!IS_ERR(src_name)) {
		putname(src_name);
	}
	if (!IS_ERR(root_name)) {
		putname(root_name);
	}

	return err;
}
#endif

#ifdef MY_ABC_HERE
/* This stat is used by caseless protocol.
 * The filename will be convert to real filename and return to user space.
 * In caller, the length of filename must equal or be larger than SYNO_SMB_PSTRING_LEN. 
*/
asmlinkage long sys_SYNOCaselessStat(char __user * filename, struct stat __user *statbuf)
{
	struct kstat stat;	
    	struct stat tmp;
	struct nameidata nd;	
	int error;	

	nd.caseless = SYNO_CASELESS_KEY;
	nd.real_filename = kmalloc(SYNO_SMB_PSTRING_LEN, GFP_KERNEL);
	if (!nd.real_filename) {		
		return -ENOMEM;
	}
	nd.real_filename_len = 0;	

#ifdef MY_ABC_HERE
	if (SynoDebugFlag) {	
		printk("%s(%d) orig name:[%s] len:[%d]\n",__FUNCTION__,__LINE__,filename, strlen(filename));		
	}
#endif
	error = __user_walk_fd(AT_FDCWD, filename, LOOKUP_FOLLOW, &nd);
	if (!error) {
		error = vfs_getattr(nd.mnt, nd.dentry, &stat);
		path_release(&nd);
		if (nd.real_filename_len) {
			copy_to_user(filename, nd.real_filename, nd.real_filename_len);		
#ifdef MY_ABC_HERE
			if (SynoDebugFlag) {			
				printk("%s(%d) convert name:[%s]\n",__FUNCTION__,__LINE__,filename);
			}
#endif
		}		
	}

#ifdef MY_ABC_HERE
	if (error && SynoDebugFlag) {			
		printk("%s(%d) convert name:[%s], error:[%d]\n",__FUNCTION__,__LINE__,filename, error);
	}
#endif

	nd.caseless = 0;
	nd.real_filename_len = 0;
	kfree(nd.real_filename);
	nd.real_filename = NULL;

#ifdef MY_ABC_HERE
	if(syno_hibernation_log_sec > 0) {
		syno_do_hibernation_log(filename);
	}
#endif

	if (!error)
		error = cp_new_stat(&stat, statbuf);

    	if(error == -ENOENT) {
		memset(&tmp, 0, sizeof(tmp));

		if( nd.flags & LOOKUP_TO_LASTCOMPONENT )
			tmp.st_SynoUnicodeStat = 0;
		else
			tmp.st_SynoUnicodeStat = 1;               /* badpath */
		error = copy_to_user(statbuf, &tmp, sizeof(tmp)) ? -EFAULT : error;
	}

	return error;
}

/* This stat is used by caseless protocol.
 * The filename will be convert to real filename and return to user space.
 * In caller, the length of filename must equal or be larger than SYNO_SMB_PSTRING_LEN. 
*/
asmlinkage long sys_SYNOCaselessLStat(char __user * filename, struct stat __user *statbuf)
{
	struct kstat stat;	
    	struct stat tmp;
	struct nameidata nd;	
	int error;	

	nd.caseless = SYNO_CASELESS_KEY;
	nd.real_filename = kmalloc(SYNO_SMB_PSTRING_LEN, GFP_KERNEL);
	if (!nd.real_filename) {		
		return -ENOMEM;
	}
	nd.real_filename_len = 0;	

#ifdef MY_ABC_HERE
	if (SynoDebugFlag) {	
		printk("%s(%d) orig name:[%s] len:[%d]\n",__FUNCTION__,__LINE__,filename, strlen(filename));
	}
#endif
	error = __user_walk_fd(AT_FDCWD, filename, 0, &nd);
	if (!error) {
		error = vfs_getattr(nd.mnt, nd.dentry, &stat);
		path_release(&nd);
		if (nd.real_filename_len) {			
			copy_to_user(filename, nd.real_filename, nd.real_filename_len);	
#ifdef MY_ABC_HERE
			if (SynoDebugFlag) {			
				printk("%s(%d) convert name:[%s]\n",__FUNCTION__,__LINE__,filename);
			}
#endif
		}
	}

#ifdef MY_ABC_HERE
	if (error && SynoDebugFlag) {			
		printk("%s(%d) convert name:[%s], error:[%d]\n",__FUNCTION__,__LINE__,filename, error);
	}
#endif

	nd.caseless = 0;
	nd.real_filename_len = 0;
	kfree(nd.real_filename);
	nd.real_filename = NULL;	

	if (!error)
		error = cp_new_stat(&stat, statbuf);

    	if(error == -ENOENT) {
		memset(&tmp, 0, sizeof(tmp));

		if( nd.flags & LOOKUP_TO_LASTCOMPONENT )
			tmp.st_SynoUnicodeStat = 0;
		else
			tmp.st_SynoUnicodeStat = 1;               /* badpath */
		error = copy_to_user(statbuf, &tmp, sizeof(tmp)) ? -EFAULT : error;
	}

	return error;
}
#endif

asmlinkage long sys_newlstat(char __user *filename, struct stat __user *statbuf)
{
	struct kstat stat;
	int error = vfs_lstat_fd(AT_FDCWD, filename, &stat);

	if (!error)
		error = cp_new_stat(&stat, statbuf);

	return error;
}

#if !defined(__ARCH_WANT_STAT64) || defined(__ARCH_WANT_SYS_NEWFSTATAT)
asmlinkage long sys_newfstatat(int dfd, char __user *filename,
				struct stat __user *statbuf, int flag)
{
	struct kstat stat;
	int error = -EINVAL;

	if ((flag & ~AT_SYMLINK_NOFOLLOW) != 0)
		goto out;

	if (flag & AT_SYMLINK_NOFOLLOW)
		error = vfs_lstat_fd(dfd, filename, &stat);
	else
		error = vfs_stat_fd(dfd, filename, &stat);

	if (!error)
		error = cp_new_stat(&stat, statbuf);

out:
	return error;
}
#endif

asmlinkage long sys_newfstat(unsigned int fd, struct stat __user *statbuf)
{
	struct kstat stat;
	int error = vfs_fstat(fd, &stat);

	if (!error)
		error = cp_new_stat(&stat, statbuf);

	return error;
}

asmlinkage long sys_readlinkat(int dfd, const char __user *path,
				char __user *buf, int bufsiz)
{
	struct nameidata nd;
	int error;

#ifdef MY_ABC_HERE
	memset(&nd, 0, sizeof(struct nameidata));
#endif
	if (bufsiz <= 0)
		return -EINVAL;

	error = __user_walk_fd(dfd, path, 0, &nd);
	if (!error) {
		struct inode * inode = nd.dentry->d_inode;

		error = -EINVAL;
		if (inode->i_op && inode->i_op->readlink) {
			error = security_inode_readlink(nd.dentry);
			if (!error) {
				touch_atime(nd.mnt, nd.dentry);
				error = inode->i_op->readlink(nd.dentry, buf, bufsiz);
			}
		}
		path_release(&nd);
	}
	return error;
}

asmlinkage long sys_readlink(const char __user *path, char __user *buf,
				int bufsiz)
{
	return sys_readlinkat(AT_FDCWD, path, buf, bufsiz);
}


/* ---------- LFS-64 ----------- */
#ifdef __ARCH_WANT_STAT64

static long cp_new_stat64(struct kstat *stat, struct stat64 __user *statbuf)
{
	struct stat64 tmp;

	memset(&tmp, 0, sizeof(struct stat64));
#ifdef CONFIG_MIPS
	/* mips has weird padding, so we don't get 64 bits there */
	if (!new_valid_dev(stat->dev) || !new_valid_dev(stat->rdev))
		return -EOVERFLOW;
	tmp.st_dev = new_encode_dev(stat->dev);
	tmp.st_rdev = new_encode_dev(stat->rdev);
#else
	tmp.st_dev = huge_encode_dev(stat->dev);
	tmp.st_rdev = huge_encode_dev(stat->rdev);
#endif
	tmp.st_ino = stat->ino;
	if (sizeof(tmp.st_ino) < sizeof(stat->ino) && tmp.st_ino != stat->ino)
		return -EOVERFLOW;
#ifdef STAT64_HAS_BROKEN_ST_INO
	tmp.__st_ino = stat->ino;
#endif
	tmp.st_mode = stat->mode;
	tmp.st_nlink = stat->nlink;
	tmp.st_uid = stat->uid;
	tmp.st_gid = stat->gid;
	tmp.st_atime = stat->atime.tv_sec;
	tmp.st_atime_nsec = stat->atime.tv_nsec;
	tmp.st_mtime = stat->mtime.tv_sec;
	tmp.st_mtime_nsec = stat->mtime.tv_nsec;
	tmp.st_ctime = stat->ctime.tv_sec;
	tmp.st_ctime_nsec = stat->ctime.tv_nsec;
#ifdef MY_ABC_HERE
	tmp.st_SynoCreateTime = stat->SynoCreateTime.tv_sec;
#endif
#ifdef	MY_ABC_HERE
	tmp.st_SynoMode = stat->SynoMode;
#endif
#ifdef MY_DEF_HERE
	tmp.st_syno_achv_ver = stat->syno_archive_version;
#endif
	tmp.st_size = stat->size;
	tmp.st_blocks = stat->blocks;
	tmp.st_blksize = stat->blksize;
	return copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : 0;
}

#ifdef MY_ABC_HERE
/* This stat is used by caseless protocol.
 * The filename will be convert to real filename and return to user space.
 * In caller, the length of filename must equal or be larger than SYNO_SMB_PSTRING_LEN. 
*/
asmlinkage long sys_SYNOCaselessStat64(char __user * filename, struct stat64 __user * statbuf)
{
	struct kstat stat;
	struct stat64 tmp;
	struct nameidata nd;	
	int error;	

	nd.caseless = SYNO_CASELESS_KEY;
	nd.real_filename = kmalloc(SYNO_SMB_PSTRING_LEN, GFP_KERNEL);
	if (!nd.real_filename) {		
		return -ENOMEM;
	}
	nd.real_filename_len = 0;

#ifdef MY_ABC_HERE
	if (SynoDebugFlag) {	
		printk("%s(%d) orig name:[%s] len:[%d]\n",__FUNCTION__,__LINE__,filename, strlen(filename));
	}
#endif
	error = __user_walk_fd(AT_FDCWD, filename, LOOKUP_FOLLOW, &nd);
	if (!error) {
		error = vfs_getattr(nd.mnt, nd.dentry, &stat);
		path_release(&nd);
		if (nd.real_filename_len) {			
			copy_to_user(filename, nd.real_filename, nd.real_filename_len);	
#ifdef MY_ABC_HERE
			if (SynoDebugFlag) {			
				printk("%s(%d) convert name:[%s]\n",__FUNCTION__,__LINE__,filename);
			}
#endif
		}		
	}

#ifdef MY_ABC_HERE
	if (error && SynoDebugFlag) {			
		printk("%s(%d) convert name:[%s], error:[%d]\n",__FUNCTION__,__LINE__,filename, error);
	}
#endif

	nd.caseless = 0;
	nd.real_filename_len = 0;
	kfree(nd.real_filename);
	nd.real_filename = NULL;	

#ifdef MY_ABC_HERE
	if(syno_hibernation_log_sec > 0) {
		syno_do_hibernation_log(filename);
	}
#endif

	if (!error)
		error = cp_new_stat64(&stat, statbuf);

	if(error == -ENOENT) {
		memset(&tmp, 0, sizeof(tmp));

		if( nd.flags & LOOKUP_TO_LASTCOMPONENT )
			tmp.st_SynoUnicodeStat = 0;
		else
			tmp.st_SynoUnicodeStat = 1;               /* badpath */
		error = copy_to_user(statbuf, &tmp, sizeof(tmp)) ? -EFAULT : error;
	}

	return error;
}

/* This stat is used by caseless protocol.
 * The filename will be convert to real filename and return to user space.
 * In caller, the length of filename must equal or be larger than SYNO_SMB_PSTRING_LEN. 
*/
asmlinkage long sys_SYNOCaselessLStat64(char __user * filename, struct stat64 __user * statbuf)
{
	struct kstat stat;
	struct stat64 tmp;
	struct nameidata nd;	
	int error;	

	nd.caseless = SYNO_CASELESS_KEY;	
	nd.real_filename = kmalloc(SYNO_SMB_PSTRING_LEN, GFP_KERNEL);
	if (!nd.real_filename) {		
		return -ENOMEM;
	}
	nd.real_filename_len = 0;
#ifdef MY_ABC_HERE
	if (SynoDebugFlag) {	
		printk("%s(%d) orig name:[%s] len:[%d]\n",__FUNCTION__,__LINE__,filename, strlen(filename));
	}
#endif
	error = __user_walk_fd(AT_FDCWD, filename, 0, &nd);
	if (!error) {
		error = vfs_getattr(nd.mnt, nd.dentry, &stat);
		path_release(&nd);
		if (nd.real_filename_len) {			
			copy_to_user(filename, nd.real_filename, nd.real_filename_len);		
#ifdef MY_ABC_HERE
			if (SynoDebugFlag) {			
				printk("%s(%d) convert name:[%s]\n",__FUNCTION__,__LINE__,filename);
			}
#endif
		}
	}

#ifdef MY_ABC_HERE
	if (error && SynoDebugFlag) {			
		printk("%s(%d) convert name:[%s], error:[%d]\n",__FUNCTION__,__LINE__,filename, error);
	}
#endif

	nd.caseless = 0;
	nd.real_filename_len = 0;
	kfree(nd.real_filename);
	nd.real_filename = NULL;	

	if (!error)
		error = cp_new_stat64(&stat, statbuf);

	if(error == -ENOENT) {
		memset(&tmp, 0, sizeof(tmp));

		if( nd.flags & LOOKUP_TO_LASTCOMPONENT )
			tmp.st_SynoUnicodeStat = 0;
		else
			tmp.st_SynoUnicodeStat = 1;               /* badpath */
		error = copy_to_user(statbuf,&tmp,sizeof(tmp)) ? -EFAULT : error;
	}

	return error;
}
#endif

asmlinkage long sys_stat64(char __user * filename, struct stat64 __user * statbuf)
{
	struct kstat stat;
	int error = vfs_stat(filename, &stat);

#ifdef MY_ABC_HERE
	if(syno_hibernation_log_sec > 0) {
		syno_do_hibernation_log(filename);
	}
#endif

	if (!error)
		error = cp_new_stat64(&stat, statbuf);

	return error;
}
asmlinkage long sys_lstat64(char __user * filename, struct stat64 __user * statbuf)
{
	struct kstat stat;
	int error = vfs_lstat(filename, &stat);

	if (!error)
		error = cp_new_stat64(&stat, statbuf);

	return error;
}
asmlinkage long sys_fstat64(unsigned long fd, struct stat64 __user * statbuf)
{
	struct kstat stat;
	int error = vfs_fstat(fd, &stat);

	if (!error)
		error = cp_new_stat64(&stat, statbuf);

	return error;
}

asmlinkage long sys_fstatat64(int dfd, char __user *filename,
			       struct stat64 __user *statbuf, int flag)
{
	struct kstat stat;
	int error = -EINVAL;

	if ((flag & ~AT_SYMLINK_NOFOLLOW) != 0)
		goto out;

	if (flag & AT_SYMLINK_NOFOLLOW)
		error = vfs_lstat_fd(dfd, filename, &stat);
	else
		error = vfs_stat_fd(dfd, filename, &stat);

	if (!error)
		error = cp_new_stat64(&stat, statbuf);

out:
	return error;
}
#endif /* __ARCH_WANT_STAT64 */

void inode_add_bytes(struct inode *inode, loff_t bytes)
{
	spin_lock(&inode->i_lock);
	inode->i_blocks += bytes >> 9;
	bytes &= 511;
	inode->i_bytes += bytes;
	if (inode->i_bytes >= 512) {
		inode->i_blocks++;
		inode->i_bytes -= 512;
	}
	spin_unlock(&inode->i_lock);
}

EXPORT_SYMBOL(inode_add_bytes);

void inode_sub_bytes(struct inode *inode, loff_t bytes)
{
	spin_lock(&inode->i_lock);
	inode->i_blocks -= bytes >> 9;
	bytes &= 511;
	if (inode->i_bytes < bytes) {
		inode->i_blocks--;
		inode->i_bytes += 512;
	}
	inode->i_bytes -= bytes;
	spin_unlock(&inode->i_lock);
}

EXPORT_SYMBOL(inode_sub_bytes);

loff_t inode_get_bytes(struct inode *inode)
{
	loff_t ret;

	spin_lock(&inode->i_lock);
	ret = (((loff_t)inode->i_blocks) << 9) + inode->i_bytes;
	spin_unlock(&inode->i_lock);
	return ret;
}

EXPORT_SYMBOL(inode_get_bytes);

void inode_set_bytes(struct inode *inode, loff_t bytes)
{
	/* Caller is here responsible for sufficient locking
	 * (ie. inode->i_lock) */
	inode->i_blocks = bytes >> 9;
	inode->i_bytes = bytes & 511;
}

EXPORT_SYMBOL(inode_set_bytes);
