// Copyright (c) 2000-2006 Synology Inc. All rights reserved.
#include <linux/syno.h>
#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/errno.h>  /* error codes */
#include <linux/delay.h>
#include <linux/synobios.h>
#include <linux/fs.h>
#include "i2c-mv.h"
#include "mapping.h"
#include "rtc.h"

//===========================================================================
//  CPLD definition
//  Local Bus CS1 : 0xF8000000 -- 0xF800000F    : CPLD
//          offset 0 --> Disk Led Control
//          offset 1 --> Disk power state register
//          offset 2 --> Model Number register
//          offset 3 --> Fan status
//
//  Disk Led Control (00:orange blink, 01: orange solid, 10: green, 11: off)
//          bit 7,6: Disk1 LED
//          bit 5,4: Disk2 LED
//          bit 3,2: Disk3 LED
//          bit 1,0: Disk4 LED
//                   11: LED Off (default)
//                   10: Green LED, allow activity blinking from SATA controller
//                   01: Solid Yellow
//                   00: Yellow blinking 2 Hz
//  
//  Model Number :
//          bit 7, 6:
//                   00: reserved
//                   01:  CS-e class
//                   10:  CS class
//                   11:  RS clasee
//          bit 5,4:
//                   00: reserved
//                   01: 64MB (128Mbits device, RAS=12, CAS=9, CAL=3)
//                   10: 128MB (256Mbits device, RAS=13, CAS=9, CAL=3)
//                   11: 256MB (512Mbits device, RAS=13, CAS=10, CAL=3)
//          bit 3,2
//                   00 --> CS406e (Home Version 64MB @64bits SDRAM)
//                   01 --> CS406  (Business Version 128MB @64bits SDRAM)
//                   10 --> RS406  (128MB @64bits SDRAM)
//                   11 --> 2007 product
//          bit 1,0
//                   00 --> CPLD version 1.0
//                   01 --> CPLD version 1.1
//
//  Fan Status :
//          bit 7 --> FAN ON/OFF (0: off, 1: on)
//          bit 6 --> FAN status (0: running, 1: not running)
//          bit 5 --> FAN speed (0: Low, 1: High)
//          bit 2 --> EXT_FAN1 status
//          bit 1 --> EXT_FAN1 status
//          bit 0 --> EXT_FAN1 status
//===========================================================================

#define DS407_CPLD_BASE     0xF8000000
#define DS407_CPLD_END      0xF8000010
#define	DS407_CPLD_DISK_LED_OFFSET		0x0
#define	DS407_CPLD_MODEL_OFFSET			0x2
#define	DS407_CPLD_VERSION_OFFSET		0x2
#define	DS407_CPLD_FAN_STATUS_OFFSET	0x3

#define	DS407_FAN_ON_OFF_BIT	0x7
#define	DS407_FAN_SPEED_BIT		0x5
#define	DS407_FAN_STATUS_BIT	0x6

#define	DS407_LED_OFF        0x3
#define	DS407_LED_GREEN_SOLID    0x2
#define	DS407_LED_ORANGE_SOLID   0x1
#define	DS407_LED_ORANGE_BLINK   0x0

extern char gszSerialNum[];

static int Uninitialize(void);

static int GetBrand(void)
{
	int Brand = -1;

	//YMXX[Z]SSSSS
	if ( gszSerialNum[4] == 'M' ) {
		Brand = BRAND_LOGITEC;
	} else if ( gszSerialNum[4] == 'U' ) {
		Brand = BRAND_SYNOLOGY_USA;
	} else {
		Brand = BRAND_SYNOLOGY;
	} 

	return Brand;
}

static int GetModel(void)
{
	char *ptr;
	char model;

	ptr = (char *)(DS407_CPLD_BASE + DS407_CPLD_MODEL_OFFSET);
	model = (*ptr >> 2) & 0x3;

	if (model != 0x3) {
		printk("Unknown CPLD model %x\n", *ptr);
		return MODEL_CS407;
	}

	model = (*ptr >> 6) & 0x3;
	switch (model) {
	case 0x3:
		return MODEL_RS407;
	case 0x2:
		return MODEL_CS407;
	case 0x1:
		return MODEL_CS407e;
	default:
		printk("Unknown model: %x\n", model);
	}

	return -1;
}

static int 
InitModuleType(struct synobios_ops *ops)
{
	PRODUCT_MODEL model = ops->get_model();
	module_t type_407 = MODULE_T_CS407;
	module_t type_407e = MODULE_T_CS407e;
	module_t *pType = NULL;

	switch (model) {
	case MODEL_CS407:
		pType = &type_407;
		break;
	case MODEL_CS407e:
		pType = &type_407e;
		break;
	default:
		break;
	}

	module_type_set(pType);
	return 0;
}

static int GetCPLDVersion(void)
{
	char *ptr;
	int Version;

	ptr = (char *)(DS407_CPLD_BASE + DS407_CPLD_VERSION_OFFSET);
	Version = *ptr & 0x3;

	return Version;
}

static int SetDiskLedStatus(int disknum, SYNO_DISK_LED status)
{
	char *ptr;
	int led;

	if (disknum < 1 || disknum > 4) {
		return -EINVAL;
	}

	switch (status) {
	case DISK_LED_OFF:
		led = DS407_LED_OFF;
		break;
	case DISK_LED_GREEN_SOLID:
		led = DS407_LED_GREEN_SOLID;
		break;
	case DISK_LED_ORANGE_SOLID:
		led = DS407_LED_ORANGE_SOLID;
		break;
	case DISK_LED_ORANGE_BLINK:
		led = DS407_LED_ORANGE_BLINK;
		break;
	default:
		printk("%s (%d), error control to LED, status = %d\n", __FILE__, __LINE__, status);
		return -EINVAL;
	}
	
	ptr = (char *)(DS407_CPLD_BASE + DS407_CPLD_DISK_LED_OFFSET);
	switch (disknum) {
	case 1:
		led <<= 0x6;
		*ptr &= ~(0xC0);
		break;
	case 2:
		led <<= 0x4;
		*ptr &= ~(0x30);
		break;
	case 3:
		led <<= 0x2;
		*ptr &= ~(0xC);
		break;
	case 4:
		led <<= 0x0;
		*ptr &= ~(0x3);
		break;
	default:
		return -EINVAL;
	}

	*ptr |= led;

	return 0;
}

static int GetCS407FanStatus(int fanno, FAN_STATUS *pStatus)
{
	char *ptr;
	char FanStatus;              
	int counter = 0;           
	
	if (pStatus == NULL) {
		return -EINVAL;
	}
	
	ptr = (char *)(DS407_CPLD_BASE + DS407_CPLD_FAN_STATUS_OFFSET);
	do {            
		FanStatus = (*ptr >> DS407_FAN_STATUS_BIT) & 0x1;
		if (FanStatus == 0) {
			break;
		}
		udelay(300);   
		counter++;
	} while ( counter < 200 );
	
	*pStatus = (FanStatus) ? FAN_STATUS_STOP : FAN_STATUS_RUNNING;
	
	return 0;
}

static int GetRS407FanStatus(int fanno, FAN_STATUS *pStatus)
{
	char *ptr;
	char FanStatus;
	int shift = 0;
	int rgcVolt[2] = {0, 0};

	if (fanno < 1 || fanno > 3 || pStatus == NULL) {
		return -EINVAL;
	}

	switch (fanno) {
	case 1:
		shift = 2;
		break;
	case 2:
		shift = 1;
		break;
	case 3:
		shift = 0;
		break;
	default:
		return -EINVAL;
	}

	ptr = (char *)(DS407_CPLD_BASE + DS407_CPLD_FAN_STATUS_OFFSET);

	do {
		FanStatus = (*ptr >> shift) & 0x1;
		rgcVolt[(int)FanStatus] ++;

		if (rgcVolt[0] && rgcVolt[1]) {
			break;
		}

		udelay(300);
	} while ( (rgcVolt[0] + rgcVolt[1]) < 200 );

	if ((rgcVolt[0] == 0) || (rgcVolt[1] == 0) ) {
		*pStatus = FAN_STATUS_STOP;
	} else {
		*pStatus = FAN_STATUS_RUNNING;
	}

	return 0;
}

static int GetFanStatus(int fanno, FAN_STATUS *pStatus)
{
	int model = GetModel();

	if (model == MODEL_RS407) {
		return GetRS407FanStatus(fanno, pStatus);
	} else if (model == MODEL_CS407 || model == MODEL_CS407e) {
		return GetCS407FanStatus(fanno, pStatus);
	} else {
		return -EINVAL;
	}
}

//  (status, speed)
//  (0, 0) --> Fan off
//  (0, 1) --> low speed
//  (1, 0) --> low speed
//  (1, 1) --> high speed
static int SetFanStatus(FAN_STATUS status, FAN_SPEED speed)
{
	char *ptr;

	ptr = (char *)(DS407_CPLD_BASE + DS407_CPLD_FAN_STATUS_OFFSET);
	/*
	if (status == FAN_STATUS_RUNNING) {
		if (speed == FAN_SPEED_HIGH) {
			//printk("%s (%d) Set fan speed to high\n", __FILE__, __LINE__);
			*ptr |= 1 << DS407_FAN_ON_OFF_BIT;
			*ptr |= 1 << DS407_FAN_SPEED_BIT;
		} else if (speed == FAN_SPEED_LOW) {
			//printk("%s (%d) Set fan speed to low\n", __FILE__, __LINE__);
			*ptr |= 1<< DS407_FAN_ON_OFF_BIT;
			*ptr = *ptr & ~(1 << DS407_FAN_SPEED_BIT);
		}
	} else {
		//printk("%s (%d) Set fan speed to stop\n", __FILE__, __LINE__);
		*ptr = *ptr & ~(1 << DS407_FAN_ON_OFF_BIT | 1 << DS407_FAN_SPEED_BIT);
	}
	*/
	if (status) {
		*ptr |= 1 << DS407_FAN_ON_OFF_BIT;
	} else {
		*ptr &= ~(1 << DS407_FAN_ON_OFF_BIT);
	}
	if (FAN_SPEED_HIGH == speed) {
		*ptr |= 1 << DS407_FAN_SPEED_BIT;
	} else if(FAN_SPEED_LOW == speed) {
		*ptr &= ~(1 << DS407_FAN_SPEED_BIT);
	} else {
		printk("%s(%d) Set Wrong Fan Speed %d\n",__FILE__,__LINE__,speed);
	}
	return 0;
}

static int GetSysTemperature(int *Temperature)
{
	int ret = 0;
	u16 data = 0;

	ret = mvI2CCharRead(0x48, (u8 *)&data, 2, 0);
	if (ret != 0) {
		printk("Failed to read temperature from i2c. ret: %d, data:0x%x\n", ret, data);
		return -1;
	}
	//printk("Temperature raw data: 0x%x\n", data);

#if defined (__LITTLE_ENDIAN)
	data = __swab16(data);
#endif

	/* The temperature data only 9 bits */
	data = data >> 7;

	if (data >> 8) { /* bit 9 is minus sign */
		/* subzero */
		*Temperature = -1 * (0x100 - ((u8 *)&data)[1]); 
	} else {
		*Temperature = data;
	}
	//printk("Temperature: %d/2\n", *Temperature);

	return 0;
}

static int GetCpldReg(CPLDREG *pCpld)
{
	unsigned char *ptr;
	ptr = (unsigned char *)DS407_CPLD_BASE;
	pCpld->diskledctrl = *ptr++;
	pCpld->diskpowerstate = *ptr++;
	pCpld->modelnumber = *ptr++;
	pCpld->fanstatus = *ptr;
	return 0;

}

static int SetMemByte( MEMORY_BYTE *pMemory )
{
	if ( NULL == pMemory ||
			(DS407_CPLD_END - DS407_CPLD_BASE) <= pMemory->offset ){
		return -1;
	}

	*((unsigned char *)(DS407_CPLD_BASE + pMemory->offset)) = pMemory->value;
	return 0;
}

static int GetMemByte( MEMORY_BYTE *pMemory )
{
	if ( NULL == pMemory ||
			(DS407_CPLD_END - DS407_CPLD_BASE) <= pMemory->offset ){
		return -1;
	}

	pMemory->value = *((unsigned char *)(DS407_CPLD_BASE + pMemory->offset));
	return 0;
}

static struct synobios_ops synobios_ops = {
	.owner               = THIS_MODULE,
	.get_brand           = GetBrand,
	.get_model           = GetModel,
	.get_cpld_version    = GetCPLDVersion,
	.get_rtc_time        = rtc_ricoh_get_time,
	.set_rtc_time        = rtc_ricoh_set_time,
	.get_fan_status      = GetFanStatus,
	.set_fan_status      = SetFanStatus,
	.set_disk_led        = SetDiskLedStatus,
	.get_cpld_reg        = GetCpldReg,
	.set_mem_byte        = SetMemByte,
	.get_mem_byte        = GetMemByte,
	.get_sys_temperature = GetSysTemperature,
	.set_disk_led        = SetDiskLedStatus,
	.get_auto_poweron    = rtc_ricoh_get_auto_poweron,
	.set_auto_poweron    = rtc_ricoh_set_auto_poweron,
	.init_auto_poweron   = rtc_ricoh_auto_poweron_init,
	.uninit_auto_poweron = rtc_ricoh_auto_poweron_uninit,
	.set_alarm_led       = NULL,
	.get_power_status    = NULL,
	.module_type_init	 = InitModuleType,
	.uninitialize		 = Uninitialize,
};

int synobios_model_init(struct file_operations *fops, struct synobios_ops **ops)
{
	*ops = &synobios_ops;

	if( synobios_ops.init_auto_poweron ) {
		synobios_ops.init_auto_poweron();
	}

	return 0;
}

static int Uninitialize(void)
{
	if( synobios_ops.uninit_auto_poweron ) {
		synobios_ops.uninit_auto_poweron();
	}

	return 0;
}

int synobios_model_cleanup(struct file_operations *fops, struct synobios_ops **ops){
	return 0;
}
