// Copyright (c) 2000-2006 Synology Inc. All rights reserved.
#include <linux/syno.h>

#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/errno.h>  /* error codes */
#include <asm/delay.h>
#include <linux/synobios.h>
#include <linux/fs.h>
#include "../i2c/i2c-ppc.h"
#include "../mapping.h"
#include "../rtc/rtc.h"

//===================================================
//  CPLD definition
//  Local Bus CS1 : 0xFF000000 -- 0xFF00000F    : CPLD
//          offset 0 --> Disk Led Control
//          offset 1 --> Disk Power State
//          offset 2 --> Model Number
//          offset 3 --> Fan status
//
//  Disk Led Control (00:orange blink, 01: orange solid, 10: green, 11: off)
//          bit 7,6 --> Disk1 LED
//          bit 5,4 --> Disk2 LED
//          bit 3,2 --> Disk3 LED
//          bit 1,0 --> Disk4 LED
//  
//  Model Number : bit 3,2
//          00 --> CS406e
//          01 --> CS406
//          10 --> RS406
//       bit 1,0
//          00 --> CPLD version 1.0
//          01 --> CPLD version 1.1
//
//  Fan Status 
//          bit 7 --> Turn on/off FAN (0: off, 1: on)
//          bit 6 --> CS FAN1 status  (0:Running, 1:Stop)
//          bit 5 --> Set the FAN speed (1: high, 0: low)
//          bit 2 --> RS FAN1 status  (0-1-0-1:Running, 1:Stop)
//          bit 1 --> RS FAN2 status  (0-1-0-1:Running, 1:Stop)
//          bit 0 --> RS FAN3 status  (0-1-0-1:Running, 1:Stop)
//===================================================
#define DS406_CPLD_BASE     0xFF000000
#define DS406_CPLD_END      0xFF000010
#define DS406_DISK_LED_CTRL             0
#define DS406_DISK_POWER_STATE          1
#define DS406_MODEL_NUMBER              2
#define DS406_FANS_STATUS               3

//------ LED Status ------
#define DISK1_LED_SHIFT     6
#define DISK2_LED_SHIFT     4
#define DISK3_LED_SHIFT     2
#define DISK4_LED_SHIFT     0

#define DISK1_LED_MASK      0xC0
#define DISK2_LED_MASK      0x30
#define DISK3_LED_MASK      0x0C
#define DISK4_LED_MASK      0x03

//------ power state ------

//------ model ------
#define MODEL_SHIFT     2
#define MODEL_MASK      0x0C

//------ fan status ------
#define CS406_FAN_ON_OFF_BIT	7
#define	CS406_FAN_SPEED_BIT	5
#define CS406_FAN1_SHIFT    6
#define CS406_FAN1_MASK     0x40
#define RS406_FAN1_SHIFT    2
#define RS406_FAN2_SHIFT    1
#define RS406_FAN3_SHIFT    0
#define RS406_FAN1_MASK     0x04
#define RS406_FAN2_MASK     0x02
#define RS406_FAN3_MASK     0x01

#define DS406_LED_OFF             0x03
#define DS406_LED_GREEN_SOLID     0x02
#define DS406_LED_ORANGE_SOLID    0x01
#define DS406_LED_ORANGE_BLINK    0x00

//====== end of CPLD definition =====================

static int Uninitialize(void);

static int GetSysTemperature(int *Temperature)
{
	int count = 0;
	u16 data = 0;

	count = mpc_i2c_read(0x48, (u8 *)&data, 2, 0, 0);
	if (count != 2) {
		return -1;
	}
	//printk("Temperature raw data: 0x%x\n", data);

	/* The temperature data only 9 bits */
	data = data >> 7;

	if (data >> 8) { /* bit 9 is minus sign */
		/* subzero */
		*Temperature = -1 * (0x100 - ((u8 *)&data)[1]); 
	} else {
		*Temperature = data;
	}
	//printk("Temperature: %d/2\n", *Temperature);

	return 0;
}

/* Model defined in CPLD (0xFF000000)
 *
 * Model Number register: Offset 0x2
 *
 * Map            Bit 7-6     5-4       Bit 3       Bit 2       Bit 1      Bit 0
 * Description      RSV    Brand name   Model[1]    Model[0]     Rev[1]     Rev[0]
 *
 *Brand name [1:0]
 * 00: Synology
 * 01: Logitec
 *
 *Model[1:0]
 * 00: CS-406e
 * 01: CS-406
 * 10: RS-406
 *
 * Rev[1:0]
 * 00: V1.0
 * 01: V1.1
 * 1x: Reserved
 */
static int GetDS406Model(void)
{
	char *ptr;
	char model;
	
	ptr = (char *)(DS406_CPLD_BASE + DS406_MODEL_NUMBER);
	
	model = *ptr;
	model &= MODEL_MASK;
	model >>= MODEL_SHIFT;

	switch (model) {
	case 0x0:
		return MODEL_CS406e;
	case 0x1:
		return MODEL_CS406;
	case 0x2:
		return MODEL_RS406;
	case 0x3:
		return MODEL_CS407e;
	default:
		printk("Unknown model: %x\n", model);
	}

	return -1;
}

static int 
InitModuleType(struct synobios_ops *ops)
{
	PRODUCT_MODEL model = ops->get_model();
	module_t type_406e = MODULE_T_CS406e;
	module_t type_406 = MODULE_T_CS406;
	module_t type_rs406 = MODULE_T_RS406;
	module_t type_407e = MODULE_T_CS407e;
	module_t *pType= NULL;

	switch (model) {
	case MODEL_CS406e:		
		pType = &type_406e;
		break;
	case MODEL_CS406:
		pType = &type_406;
		break;
	case MODEL_RS406:
		pType = &type_rs406;
		break;
	case MODEL_CS407e:
		pType = &type_407e;
		break;	
	default:
		break;
	}

	module_type_set(pType);
	return 0;
}

extern char gszSerialNum[];

static int GetDS406Brand(void)
{
	char *ptr;
	int Brand = -1;
	int tmp = -1;

	if (GetDS406Model() != MODEL_CS407e) {
		ptr = (char *)(DS406_CPLD_BASE + DS406_MODEL_NUMBER);
		tmp = ((*ptr >> 4) & 0x3);
		switch (tmp) {
		case 1:
			Brand = BRAND_LOGITEC;
		case 2:
			Brand = BRAND_SYNOLOGY_USA;
		case 0:
		default:
			Brand = BRAND_SYNOLOGY;
		}
	} else {
		//YMXX[Z]SSSSS
		if ( gszSerialNum[4] == 'M' ) {
			Brand = BRAND_LOGITEC;
		} else if ( gszSerialNum[4] == 'U' ) {
			Brand = BRAND_SYNOLOGY_USA;
		} else {
			Brand = BRAND_SYNOLOGY;
		}
	}

	return Brand;
}

static int GetCPLDVersion(void)
{
	char *ptr;
	int Version;

	ptr = (char *)(DS406_CPLD_BASE + DS406_MODEL_NUMBER);
	Version = *ptr & 0x3;

	return Version;
}

static int SetDiskLedStatus(int disknum, SYNO_DISK_LED status)
{
	char *ptr;
	int led = DS406_LED_GREEN_SOLID;

	if (disknum < 1 || disknum > 4) {
		return -EINVAL;
	}

	switch (status) {
	case DISK_LED_OFF:
		led = DS406_LED_OFF;
		break;
	case DISK_LED_GREEN_SOLID:
		led = DS406_LED_GREEN_SOLID;
		break;
	case DISK_LED_ORANGE_SOLID:
		led = DS406_LED_ORANGE_SOLID;
		break;
	case DISK_LED_ORANGE_BLINK:
		led = DS406_LED_ORANGE_BLINK;
		break;
	default:
		printk("%s (%d), error control to LED, status = %d\n", __FILE__, __LINE__, status);
	}

	ptr = (char *)(DS406_CPLD_BASE + DS406_DISK_LED_CTRL);

	switch (disknum) {
	case 1:
		led <<= DISK1_LED_SHIFT;
		*ptr &= ~(DISK1_LED_MASK);
		break;
	case 2:
		led <<= DISK2_LED_SHIFT;
		*ptr &= ~(DISK2_LED_MASK);
		break;
	case 3:
		led <<= DISK3_LED_SHIFT;
		*ptr &= ~(DISK3_LED_MASK);
		break;
	case 4:
		led <<= DISK4_LED_SHIFT;
		*ptr &= ~(DISK4_LED_MASK);
		break;
	default:
		return -EINVAL;
	}

	*ptr |= led;
	
	return 0;
}

static int GetCS406FanStatus(int fanno, FAN_STATUS *pStatus)
{
	char *ptr;
	char fanstatus;
	int co = 0;

	if (fanno != 1 || pStatus == NULL) {
		return -EINVAL;
	}

	ptr = (char *)(DS406_CPLD_BASE + DS406_FANS_STATUS);
	do {
		fanstatus = *ptr;
		fanstatus &= CS406_FAN1_MASK;
		fanstatus >>= CS406_FAN1_SHIFT;
		if (fanstatus == 0) {
			break;
		}
		udelay(300);
		co++;
	} while ( co < 200 );

	*pStatus = (fanstatus) ? FAN_STATUS_STOP : FAN_STATUS_RUNNING;

	return 0;
}

static int GetRS406FanStatus(int fanno, FAN_STATUS *pStatus)
{
	char *ptr;
	char status;
	int co = 0;
	int shift = 0;
	char mask = 0;

	if (fanno < 1 || fanno > 3 || pStatus == NULL) {
		return -EINVAL;
	}

	switch (fanno) {
	case 1:
		shift = RS406_FAN1_SHIFT;
		mask = RS406_FAN1_MASK;
		break;
	case 2:
		shift = RS406_FAN2_SHIFT;
		mask = RS406_FAN2_MASK;
		break;
	case 3:
		shift = RS406_FAN3_SHIFT;
		mask = RS406_FAN3_MASK;
		break;
	}

	ptr = (char *)(DS406_CPLD_BASE + DS406_FANS_STATUS);
	do {
		status = *ptr;
		status &= mask;
		status >>= shift;
		if (status == 0) {
			break;
		}
		udelay(300);
		co++;
	} while ( co < 200 );

	*pStatus = (status) ? FAN_STATUS_STOP : FAN_STATUS_RUNNING;
	return 0;
}

static int GetFanStatus(int fanno, FAN_STATUS *pStatus)
{
	int model = GetDS406Model();

	if (model == MODEL_RS406) {
		return GetRS406FanStatus(fanno, pStatus);
	} else if (model == MODEL_CS406 || model == MODEL_CS406e || model == MODEL_CS407e) {
		return GetCS406FanStatus(fanno, pStatus);
	} else {
		return -EINVAL;
	}
}

static int SetFanStatus(FAN_STATUS status, FAN_SPEED speed)
{
	char *ptr;

	ptr = (char *)(DS406_CPLD_BASE + DS406_FANS_STATUS);

	if (status) {
		*ptr |= 1 << CS406_FAN_ON_OFF_BIT;
	} else {
		*ptr &= ~(1 << CS406_FAN_ON_OFF_BIT);
	}
	if(FAN_SPEED_HIGH == speed) {
		*ptr |= 1 << CS406_FAN_SPEED_BIT;
	} else if(FAN_SPEED_LOW == speed){
		*ptr &= ~(1 << CS406_FAN_SPEED_BIT);
	} else {
		printk("%s(%d) Set Wrong Fan Speed %d\n",__FILE__,__LINE__,speed);
	}

	return 0;
}

static int GetAutoPowerOn(SYNO_AUTO_POWERON *pAutoPowerOn)
{
	return -EINVAL;
}

static int SetAutoPowerOn(SYNO_AUTO_POWERON *pAutoPowerOn)
{
	return -EINVAL;
}

static int GetCpldReg(CPLDREG *pCpld)
{
	unsigned char *ptr;
	ptr = (unsigned char *)(DS406_CPLD_BASE);
	pCpld->diskledctrl = *ptr++;
	pCpld->diskpowerstate = *ptr++;
	pCpld->modelnumber = *ptr++;
	pCpld->fanstatus = *ptr;
	return 0;

}

static int SetMemByte( MEMORY_BYTE *pMemory )
{
	if ( NULL == pMemory ||
			(DS406_CPLD_END - DS406_CPLD_BASE) <= pMemory->offset ){
		return -1;
	}

	*((unsigned char *)(DS406_CPLD_BASE + pMemory->offset)) = pMemory->value;
	return 0;
}

static int GetMemByte( MEMORY_BYTE *pMemory )
{
	if ( NULL == pMemory ||
			(DS406_CPLD_END - DS406_CPLD_BASE) <= pMemory->offset ){
		return -1;
	}

	pMemory->value = *((unsigned char *)(DS406_CPLD_BASE + pMemory->offset));
	return 0;
}

static struct synobios_ops synobios_ops = {
	.owner               = THIS_MODULE,
	.get_brand           = GetDS406Brand,
	.get_model           = GetDS406Model,
	.get_cpld_version    = GetCPLDVersion,
	.get_rtc_time        = rtc_ricoh_get_time,
	.set_rtc_time        = rtc_ricoh_set_time,
	.get_fan_status      = GetFanStatus,
	.set_fan_status      = SetFanStatus,
	.set_disk_led        = SetDiskLedStatus,
	.get_cpld_reg        = GetCpldReg,
	.set_mem_byte        = SetMemByte,
	.get_mem_byte        = GetMemByte,
	.get_sys_temperature = GetSysTemperature,
	.set_disk_led        = SetDiskLedStatus,
	.get_auto_poweron    = GetAutoPowerOn,
	.set_auto_poweron    = SetAutoPowerOn,
	.init_auto_poweron   = rtc_ricoh_auto_poweron_init,
	.uninit_auto_poweron = rtc_ricoh_auto_poweron_uninit,
	.set_alarm_led       = NULL,
	.get_power_status    = NULL,
	.module_type_init	 = InitModuleType,
	.uninitialize		 = Uninitialize,
};

int synobios_model_init(struct file_operations *fops, struct synobios_ops **ops)
{
	*ops = &synobios_ops;

	mpc_i2c_init();

	if( synobios_ops.init_auto_poweron ) {
		synobios_ops.init_auto_poweron();
	}

	return 0;
}

static int Uninitialize(void)
{
	if( synobios_ops.uninit_auto_poweron ) {
		synobios_ops.uninit_auto_poweron();
	}

	return 0;
}

int synobios_model_cleanup(struct file_operations *fops, struct synobios_ops **ops){
	return 0;
}
