// Copyright (c) 2000-2006 Synology Inc. All rights reserved.
#include <linux/syno.h>
#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/errno.h>  /* error codes */
#include <linux/delay.h>
#include <linux/synobios.h>
#include <linux/fs.h>
#include "../i2c/i2c-ppc.h"
#include "../mapping.h"
#include "../rtc/rtc.h"

//===========================================================================
//  CPLD definition
//  Local Bus CS1 : 0xFF000000 -- 0xFF00000F    : CPLD
//          offset 0 --> Disk Led Control
//          offset 1 --> Brand/Model Number
//          offset 2 --> CPLD version
//          offset 3 --> Fan status
//
//  ** Notice: **
//          Because 207's CPLD has only 4 bit, the 4 low bits (if big endian) should
//          be ignored.
//
//  Disk Led Control (00:orange blink, 01: orange solid, 10: green, 11: off)
//          bit 3,2: Disk1 LED
//          bit 1,0: Disk2 LED
//                   11: LED Off (default)
//                   10: Green LED, allow activity blinking from SATA controller
//                   01: Solid Yellow
//                   00: Yellow blinking 2 Hz
//  
//  Model Number :
//          bit 3,2: Model
//                   00: reserved
//                   01: DS107e
//                   10: DS107 (either MPC824x or Marvell 5281 use same define)
//                   11: DS207
//          bit 1,0: RAM Size
//                   00: 32MB @32bits SDRAM (128Mbits device, RAS=12, CAS=9, CAL=3)
//                   01: 64MB @64bits SDRAM (128Mbits device, RAS=12, CAS=9, CAL=3)
//                   10: 64MB @32bits SDRAM (256Mbits device, RAS=13, CAS=9, CAL=3)
//                                     (or DDR 256Mbits device, RAS=13, CAS=9, CAL=3)
//                   11: 128MB @64bits SDRAM (256Mbits device, RAS=13, CAS=9, CAL=3)
//  CPLD version:
//          bit 2,1,0: Version
//
//  Fan Status 
//          bit 3 --> Turn on/off FAN (0: off, 1: on)
//          bit 2 --> Set FAN speed (0:Low, 1:High)
//          bit 1 --> FAN Status (0: running, 1: no running)
//===========================================================================

#define DS207_CPLD_BASE     0xFF000000
#define DS207_CPLD_END      0xFF000010
#define	DS207_CPLD_DISK_LED_OFFSET		0x0
#define	DS207_CPLD_BRAND_MODEL_OFFSET	0x1
#define	DS207_CPLD_VERSION_OFFSET		0x2
#define	DS207_CPLD_FAN_STATUS_OFFSET	0x3

#define	DS207_FAN_ON_OFF_BIT	0x7
#define	DS207_FAN_SPEED_BIT		0x6
#define	DS207_FAN_STATUS_BIT	0x5

#define	DS207_LED_OFF        0x3
#define	DS207_LED_GREEN_SOLID    0x2
#define	DS207_LED_ORANGE_SOLID   0x1
#define	DS207_LED_ORANGE_BLINK   0x0

extern char gszSerialNum[];
extern char gszSynoHWVersion[16];

static int Uninitialize(void);

static int GetBrand(void)
{
	int Brand = -1;

	//YMXX[Z]SSSSS
	if ( gszSerialNum[4] == 'M' ) {
		Brand = BRAND_LOGITEC;
	} else if ( gszSerialNum[4] == 'U' ) {
		Brand = BRAND_SYNOLOGY_USA;
	} else {
		Brand = BRAND_SYNOLOGY;
	} 

	return Brand;
}

static int GetModel(void)
{
	char *ptr;
	char model;
	
	ptr = (char *)(DS207_CPLD_BASE + DS207_CPLD_BRAND_MODEL_OFFSET);
	model = (*ptr >> 6) & 0x3;

	switch (model) {
	case 0x3:
		return MODEL_DS207;
	case 0x2:
		return MODEL_DS107;
	case 0x1:
		return MODEL_DS107e;
	default:
		printk("Unknown model: %x\n", model);
	}

	return -1;
}

static int 
InitModuleType(struct synobios_ops *ops)
{
	PRODUCT_MODEL model = ops->get_model();
	module_t type_207 = MODULE_T_DS207;
	module_t type_107e = MODULE_T_DS107e;
	module_t type_107v1 = MODULE_T_DS107v1;
	module_t type_107v2 = MODULE_T_DS107v2;
	module_t type_107v3 = MODULE_T_DS107v3;	
	module_t *pType = NULL;

	switch (model) {
	case MODEL_DS207:		
		pType = &type_207;
		break;
	case MODEL_DS107:
		if (!strncmp(gszSynoHWVersion, HW_DS107v20, strlen(HW_DS107v20))) {
			pType = &type_107v2;
		} else if (!strncmp(gszSynoHWVersion, HW_DS107v30, strlen(HW_DS107v30))) {
			pType = &type_107v3;
		} else {			
			pType = &type_107v1;
		}
		break;
	case MODEL_DS107e:
		pType = &type_107e;
		break;
	default:
		break;
	}

	module_type_set(pType);
	return 0;
}

static int GetCPLDVersion(void)
{
	char *ptr;
	int Version;

	ptr = (char *)(DS207_CPLD_BASE + DS207_CPLD_VERSION_OFFSET);
	Version = (*ptr >> 4) & 0x7;

	return Version;
}

static int SetDiskLedStatus(int disknum, SYNO_DISK_LED status)
{
	char *ptr;
	int led;

	if (disknum < 1 || disknum > 2) {
		return -EINVAL;
	}

	switch (status) {
	case DISK_LED_OFF:
		led = DS207_LED_OFF;
		break;
	case DISK_LED_GREEN_SOLID:
		led = DS207_LED_GREEN_SOLID;
		break;
	case DISK_LED_ORANGE_SOLID:
		led = DS207_LED_ORANGE_SOLID;
		break;
	case DISK_LED_ORANGE_BLINK:
		led = DS207_LED_ORANGE_BLINK;
		break;
	default:
		printk("%s (%d), error control to LED, status = %d\n", __FILE__, __LINE__, status);
		return -EINVAL;
	}

	ptr = (char *)(DS207_CPLD_BASE + DS207_CPLD_DISK_LED_OFFSET);
	switch (disknum) {
	case 1:
		led <<= 0x6;
		*ptr &= ~(0xCF);
		break;
	case 2:
		led <<= 0x4;
		*ptr &= ~(0x3F);
		break;
	default:
		return -EINVAL;
	}

	*ptr |= led;
	return 0;
}

static int GetFanStatus(int fanno, FAN_STATUS *pStatus)
{
	char *ptr;
	char FanStatus;
	int rgcVolt[2] = {0, 0};

	if (pStatus == NULL) {
		return -EINVAL;
	}

	ptr = (char *)(DS207_CPLD_BASE + DS207_CPLD_FAN_STATUS_OFFSET);
	do {
		FanStatus = (*ptr >> DS207_FAN_STATUS_BIT) & 0x1;
		rgcVolt[(int)FanStatus] ++;
		udelay(31);
		if (rgcVolt[0] && rgcVolt[1]) {
			break;
		}
	} while ( (rgcVolt[0] + rgcVolt[1]) < 2000 );

	if ((rgcVolt[0] == 0) || (rgcVolt[1] == 0) ) {
		*pStatus = FAN_STATUS_STOP;
	} else {
		*pStatus = FAN_STATUS_RUNNING;
	}

	return 0;
}

static int SetFanStatus(FAN_STATUS status, FAN_SPEED speed)
{
	char *ptr;

	ptr = (char *)(DS207_CPLD_BASE + DS207_CPLD_FAN_STATUS_OFFSET);
	if (status) {
		*ptr |= 1 << (DS207_FAN_ON_OFF_BIT);
	} else {
		*ptr &= ~(1 << DS207_FAN_ON_OFF_BIT);
	}
	if(FAN_SPEED_HIGH == speed) {
		*ptr |= 1 << DS207_FAN_SPEED_BIT;
	} else if(FAN_SPEED_LOW == speed) {
		*ptr &= ~(1 << DS207_FAN_SPEED_BIT);
	} else {
		printk("%s(%d) Set Wrong Fan Speed %d\n",__FILE__,__LINE__,speed);
	}
	return 0;
}

static int GetSysTemperature(int *Temperature)
{
	int count = 0;
	u16 data = 0;

	count = mpc_i2c_read(0x48, (u8 *)&data, 2, 0, 0);
	if (count != 2) {
		return -1;
	}
	//printk("Temperature raw data: 0x%x\n", data);

	/* The temperature data only 9 bits */
	data = data >> 7;

	if (data >> 8) { /* bit 9 is minus sign */
		/* subzero */
		*Temperature = -1 * (0x100 - ((u8 *)&data)[1]); 
	} else {
		*Temperature = data;
	}
	//printk("Temperature: %d/2\n", *Temperature);

	return 0;
}

static int GetAutoPowerOn(SYNO_AUTO_POWERON *pAutoPowerOn)
{
	return -EINVAL;
}

static int SetAutoPowerOn(SYNO_AUTO_POWERON *pAutoPowerOn)
{
	return -EINVAL;
}

static int GetCpldReg(CPLDREG *pCpld)
{
	unsigned char *ptr;
	ptr = (unsigned char *)(DS207_CPLD_BASE);
	pCpld->diskledctrl = *ptr++;
	pCpld->diskpowerstate = *ptr++;
	pCpld->modelnumber = *ptr++;
	pCpld->fanstatus = *ptr;
	return 0;

}

static int SetMemByte( MEMORY_BYTE *pMemory )
{
	if ( NULL == pMemory ||
		(DS207_CPLD_END - DS207_CPLD_BASE) <= pMemory->offset ){
			return -1;
	}
	
	*((unsigned char *)(DS207_CPLD_BASE + pMemory->offset)) = pMemory->value;
	return 0;
}

static int GetMemByte( MEMORY_BYTE *pMemory )
{
	if ( NULL == pMemory ||
		(DS207_CPLD_END - DS207_CPLD_BASE) <= pMemory->offset ){
			return -1;
	}
	
	pMemory->value = *((unsigned char *)(DS207_CPLD_BASE + pMemory->offset));
	return 0;
}

static struct synobios_ops synobios_ops = {
	.owner               = THIS_MODULE,
	.get_brand           = GetBrand,
	.get_model           = GetModel,
	.get_cpld_version    = GetCPLDVersion,
	.get_rtc_time        = rtc_ricoh_get_time,
	.set_rtc_time        = rtc_ricoh_set_time,
	.get_fan_status      = GetFanStatus,
	.set_fan_status      = SetFanStatus,
	.get_sys_temperature = GetSysTemperature,
	.set_disk_led        = SetDiskLedStatus,
	.get_cpld_reg        = GetCpldReg,
	.set_mem_byte        = SetMemByte,
	.get_mem_byte        = GetMemByte,
	.set_disk_led        = SetDiskLedStatus,
	.get_auto_poweron    = GetAutoPowerOn,
	.set_auto_poweron    = SetAutoPowerOn,
	.init_auto_poweron   = rtc_ricoh_auto_poweron_init,
	.uninit_auto_poweron = rtc_ricoh_auto_poweron_uninit,
	.set_alarm_led       = NULL,
	.get_power_status    = NULL,
	.module_type_init    = InitModuleType,
	.uninitialize        = Uninitialize,
};

int synobios_model_init(struct file_operations *fops, struct synobios_ops **ops)
{
	*ops = &synobios_ops;

	mpc_i2c_init();

	if( synobios_ops.init_auto_poweron ) {
		synobios_ops.init_auto_poweron();
	}

	return 0;
}

static int Uninitialize(void)
{
	if( synobios_ops.uninit_auto_poweron ) {
		synobios_ops.uninit_auto_poweron();
	}

	return 0;
}

int synobios_model_cleanup(struct file_operations *fops, struct synobios_ops **ops){
	return 0;
}
