#include <asm/types.h>
#include "i2c-mv.h"
#include "../rtc/rtc.h"

/* The TWSI interface supports both 7-bit and 10-bit addressing.
 * This enumerator describes addressing type.
 */
typedef enum _mvTwsiAddrType
{
        ADDR7_BIT,                      /* 7 bit address    */
        ADDR10_BIT                      /* 10 bit address   */
}MV_TWSI_ADDR_TYPE;

/* This structure describes TWSI address. */
typedef struct _mvTwsiAddr
{
        u32 address;                    /* address          */
        MV_TWSI_ADDR_TYPE type;       /* Address type     */
}MV_TWSI_ADDR;

/* This structure describes a TWSI slave. */
typedef struct _mvTwsiSlave
{
        MV_TWSI_ADDR slaveAddr;
        int validOffset;    /* whether the slave has offset (i.e. Eeprom  etc.) */
        u32 offset;         /* offset in the slave. */
        int moreThen256;    /* whether the ofset is bigger then 256 */
}MV_TWSI_SLAVE;

/******************************************************************************
 * Marvell 88F5182 i2c control                                                *
 *                                                                            *
 * Those structures are copied from arch/arm/mach-mv88fxx81/Soc/twsi/mvTwsi.h *
 * ***************************************************************************/

/********************************************************************************
 * Marvell 88F6281 i2c control                                                  *
 *                                                                              *
 * Those structures are copied from arch/arm/plat-feroceon/mv_hal/twsi/mvTwsi.h *
 * *****************************************************************************/

#ifdef CONFIG_SYNO_MV88F5x8x
extern int mvTwsiRead (MV_TWSI_SLAVE *twsiSlave, u8 *pBlock, u32 blockSize);
extern int mvTwsiWrite(MV_TWSI_SLAVE *twsiSlave, u8 *pBlock, u32 blockSize);
#else
extern int mvTwsiRead (char chanNum, MV_TWSI_SLAVE *twsiSlave, u8 *pBlock, u32 blockSize);
extern int mvTwsiWrite(char chanNum, MV_TWSI_SLAVE *twsiSlave, u8 *pBlock, u32 blockSize);
#endif

int mvI2CCharRead(int target, u8 *data, int length, int offset)
{
	MV_TWSI_SLAVE   twsiSlave;
	
	twsiSlave.slaveAddr.type = ADDR7_BIT;
	twsiSlave.slaveAddr.address = target;
	twsiSlave.validOffset = (offset >= 0) ? 1 : 0;
	twsiSlave.offset = (offset<<4);
	twsiSlave.moreThen256 = 0;
#ifdef CONFIG_SYNO_MV88F5x8x
	return mvTwsiRead (&twsiSlave, data, length);
#elif CONFIG_SYNO_MV88F6281
	return mvTwsiRead (0, &twsiSlave, data, length);
#endif
}

int mvI2CCharWrite(int target, u8 *data, int length, int offset)
{
	MV_TWSI_SLAVE twsiSlave;
	
	twsiSlave.slaveAddr.type = ADDR7_BIT;
	twsiSlave.slaveAddr.address = target;
	twsiSlave.validOffset = (offset >= 0) ? 1 : 0;
	twsiSlave.offset = (offset<<4);
	twsiSlave.moreThen256 = 0;
#ifdef CONFIG_SYNO_MV88F5x8x
	return mvTwsiWrite(&twsiSlave, data, length);
#elif CONFIG_SYNO_MV88F6281
	return mvTwsiWrite(0, &twsiSlave, data, length);
#endif
}
