#include "mv_include.h"
#include "mv_os.h"

#include "hba_header.h"

#include "linux_main.h"
#include "linux_sense.h"
#include "linux_helper.h"

#ifdef MY_ABC_HERE
#include <linux/hdreg.h>
#endif

void GenerateSGTable(
	IN PHBA_Extension pHBA,
	IN struct scsi_cmnd *SCpnt,
	OUT PMV_SG_Table pSGTable
	);
void HBARequestCallback(
	MV_PVOID This,
	PMV_Request pReq
	);

void __hba_dump_req_info(PMV_Request preq)
{
	unsigned long lba =0;

	switch (preq->Cdb[0]) {
	case SCSI_CMD_READ_10:
	case SCSI_CMD_WRITE_10:
		lba = preq->Cdb[2]<<24 | preq->Cdb[3]<<16 | preq->Cdb[4]<<8 | \
			preq->Cdb[5];
		break;
	default:
		lba = 0;
		break;
	} 

	MV_DBG(DMSG_PROF_FREQ, 
	       "_MV_ req "RED("%p")
	       " dev %d : cmd %2X : lba %lu - %lu : length %d.\n",
	       preq, preq->Device_Id, preq->Cdb[0], lba,
	       lba + preq->Data_Transfer_Length/512,
	       preq->Data_Transfer_Length);
}


#ifdef MY_ABC_HERE
/*Definitions of S.M.A.R.T. command buffer offsets*/
/* Copy from mvSata. */
#define SMART_BUF_COMMAND_OFFSET                0
#define SMART_BUF_LBALOW_OFFSET                 1
#define SMART_BUF_FEATURES_OFFSET               2
#define SMART_BUF_SECTORCOUNT_OFFSET            3
#define SMART_BUF_LBAMID_OFFSET                 4
#define SMART_BUF_LBAHIGH_OFFSET                5
#define SMART_BUF_DEVICE_OFFSET                 6
#define SMART_BUF_ERROR_OFFSET                  7

#define ATA_SECTOR_SIZE							512
unsigned char
TranslateSCSIPassThroughRequest(PMV_Request pReq, struct scsi_cmnd *pSCmd)
{
	unsigned char *pBuff;
	unsigned char ret = -1;
	unsigned char data_in = 0;

	if (!pReq || !pSCmd) {
		goto END;
	}

	pBuff = pReq->Data_Buffer;

	if (!pBuff) {
		goto END;
	}
	

#ifdef MY_ABC_HERE
	if (0x24 == pSCmd->cmnd[4] &&
		0x10 == pSCmd->cmnd[1]) {
		/* please refer SDK/lib/disk/disk_write_cache_set.c for 0x0 and 0x4 */
		if (0x0 != pBuff[6] &&
			0x4 != pBuff[6]) {
			printk("%s not a write cache\n", __FUNCTION__);
			WARN_ON(1);
			goto END;
		}

		if (0x0 == pBuff[6]) {
			pReq->Cdb[2] = CDB_CORE_DISABLE_WRITE_CACHE;
		} else {
			pReq->Cdb[2] = CDB_CORE_ENABLE_WRITE_CACHE;
		}
		ret = 0;
		goto END;
	}
#endif

	switch (pBuff[SMART_BUF_COMMAND_OFFSET]) {
	case WIN_IDENTIFY:
        pReq->Cdb[2] = CDB_CORE_IDENTIFY;
		data_in = 1;
		break;
	case WIN_STANDBYNOW1:
		pReq->Cdb[2] = CDB_CORE_STANDBY_IMMEDIATE;
		break;
	case WIN_CHECKPOWERMODE1:
		pReq->Cdb[2] = CDB_CORE_CHECK_POWER;
		break;
	case WIN_SETIDLE1:
		pReq->Cdb[2] = CDB_CORE_SETIDLE1;
		break;
	case WIN_SMART:
		switch (pBuff[SMART_BUF_FEATURES_OFFSET])
		{
		case SMART_READ_VALUES:        
			pReq->Cdb[2] = CDB_CORE_READ_VALUE;
			data_in = 1;
			break;
		case SMART_READ_LOG_SECTOR:
			pReq->Cdb[2] = CDB_CORE_READ_LOG_SECTOR;
			data_in = 1;
			break;
		case SMART_READ_THRESHOLDS:			
			pReq->Cdb[2] = CDB_CORE_READ_THRESHOLDS;
			data_in = 1;
			break;
		case SMART_ENABLE:
			pReq->Cdb[2] = CDB_CORE_ENABLE_SMART;
			break;
		case SMART_DISABLE:
			pReq->Cdb[2] = CDB_CORE_DISABLE_SMART;        
			break;
		case SMART_STATUS:
			pReq->Cdb[2] = CDB_CORE_SMART_RETURN_STATUS;
			break;
		case SMART_AUTO_OFFLINE:
			pReq->Cdb[2] = CDB_CORE_SMART_AUTO_OFFLINE;
			break;
		case SMART_AUTOSAVE:
			pReq->Cdb[2] = CDB_CORE_SMART_AUTOSAVE;
			break;
		case SMART_IMMEDIATE_OFFLINE:
			pReq->Cdb[2] = CDB_CORE_SMART_IMMEDIATE_OFFLINE;
			break;
		default:
			printk("%s unknow pBuff[SMART_BUF_FEATURES_OFFSET] 0x%x\n", __FUNCTION__, pBuff[SMART_BUF_FEATURES_OFFSET]);
			WARN_ON(1);
            goto END;
        }
		break;
	default:		
		printk("%s unknow pBuff[SMART_BUF_COMMAND_OFFSET] 0x%x\n", __FUNCTION__, pBuff[SMART_BUF_COMMAND_OFFSET]);
		WARN_ON(1);
		goto END;
	}

	if (data_in) {
		if (pReq->Data_Transfer_Length < ATA_SECTOR_SIZE + 6) {
			goto END;
		}
	}

	/* Information technology - SCSI / ATA Translation - 2 (SAT-2), 
	 * T10/1826-D, Revision 05, 22 June 2008,
	 * Table 104 - ATA PASS-THROUGH (16) command, page 143.
	 */
	pReq->Cdb[6] = pBuff[3]; /* sector count */
	pReq->Cdb[8] = pBuff[1]; /* lba low */
	ret = 0;
END:
	return ret;
}
#endif

MV_BOOLEAN TranslateSCSIRequest(PHBA_Extension pHBA, struct scsi_cmnd *pSCmd, PMV_Request pReq)
{

#ifdef MY_ABC_HERE
	unsigned char blIsSCSIPassThrough = 0;
#endif	
#ifdef MY_ABC_HERE
	unsigned char blIsWriteCache = 0;
#endif
	pReq->Device_Id = mv_scmd_target(pSCmd);	

	/* Cmd_Flag */	//TBD: For Linux: Is that possible to set these flags or need read the Cdb
	pReq->Cmd_Flag = 0;

	/*
	 * Set three flags: CMD_FLAG_NON_DATA, CMD_FLAG_DATA_IN and CMD_FLAG_DMA
	 */
	if ( pSCmd->request_bufflen==0 ) //TBD lily
	{
		pReq->Cmd_Flag |= CMD_FLAG_NON_DATA;
	}
	else
	{
		//if ( Srb->SrbFlags&SRB_FLAGS_DATA_IN )
		//	pReq->Cmd_Flag |= CMD_FLAG_DATA_IN; TBD ?? Lily
		/*We need to optimize the flags setting. Lily*/
		if(SCSI_IS_READ(pSCmd->cmnd[0]))
			pReq->Cmd_Flag |= CMD_FLAG_DATA_IN; /*NOTE!possible to result in ERROR */
		if ( SCSI_IS_READ(pSCmd->cmnd[0]) || SCSI_IS_WRITE(pSCmd->cmnd[0]) )
			pReq->Cmd_Flag |= CMD_FLAG_DMA;
	}

	pReq->Sense_Info_Buffer_Length = SCSI_SENSE_BUFFERSIZE;  //TBD
	pReq->Data_Transfer_Length = pSCmd->request_bufflen;

	//To handle some special CMDs,lily
	memset(pReq->Cdb, 0, MAX_CDB_SIZE);
	
	switch (pSCmd->sc_data_direction) {
	case DMA_FROM_DEVICE:
		pReq->Cmd_Flag = CMD_FLAG_DATA_IN | CMD_FLAG_DMA;
		break;
	default:
		break;
	}

	switch(pSCmd->cmnd[0]){
	case READ_TOC:
		pReq->Cdb[0] = READ_TOC;
		pReq->Cdb[1] = pSCmd->cmnd[1];
		pReq->Cdb[2] = pSCmd->cmnd[2];
		pReq->Cdb[6] = pSCmd->cmnd[6];
		pReq->Cdb[7] = pSCmd->cmnd[7];
		pReq->Cdb[8] = pSCmd->cmnd[8];
		break;
	case REQUEST_SENSE:
		break;
	case MODE_SELECT:
		pReq->Cdb[0] = MODE_SELECT_10;
		pReq->Cdb[1] = pSCmd->cmnd[1];
		pReq->Cdb[8] = pSCmd->cmnd[4];
#ifdef MY_ABC_HERE
		/* perhaps the 0x24 is a wrong value in SDK/lib/disk_write_cache_set.c
		 * But it is useful here. Thanks god.
		 */
		if (0x24 == pSCmd->cmnd[4] &&
			0x10 == pSCmd->cmnd[1]) {
			memcpy(pReq->Cdb, pSCmd->cmnd, MAX_CDB_SIZE);
			pReq->Cdb[0] = SCSI_CMD_MARVELL_SPECIFIC;
			pReq->Cdb[1] = CDB_CORE_MODULE_EXTENSION;
			blIsWriteCache = 1;
			blIsSCSIPassThrough = 1;
		}
#endif
		break;
		
	case FORMAT_UNIT:
		pReq->Cdb[0] = 0x24; //ATAPI opcodes
		break;
		
	case READ_CAPACITY: //TBD
		pReq->Cdb[0] = pSCmd->cmnd[0];
		break;

	case TEST_UNIT_READY:                       //TBD
		pReq->Cdb[0] = pSCmd->cmnd[0];
		break;

	case READ_6:
		pReq->Cdb[0] = READ_10;
		pReq->Cdb[3] = pSCmd->cmnd[1]&0x1f;
		pReq->Cdb[4] = pSCmd->cmnd[2];
		pReq->Cdb[5] = pSCmd->cmnd[3];
		pReq->Cdb[8] = pSCmd->cmnd[4];
		pReq->Cdb[9] = pSCmd->cmnd[5];
		break;

	case WRITE_6:
		pReq->Cdb[0] = WRITE_10;
		pReq->Cdb[3] = pSCmd->cmnd[1]&0x1f;
		pReq->Cdb[4] = pSCmd->cmnd[2];
		pReq->Cdb[5] = pSCmd->cmnd[3];
		pReq->Cdb[8] = pSCmd->cmnd[4];
		pReq->Cdb[9] = pSCmd->cmnd[5];
		break;
#if 0
	case READ_12:
		pReq->Cdb[0] = READ_10;
		pReq->Cdb[1] = pSCmd->cmnd[1];
		pReq->Cdb[2] = pSCmd->cmnd[2];
		pReq->Cdb[3] = pSCmd->cmnd[3];
		pReq->Cdb[4] = pSCmd->cmnd[4];
		pReq->Cdb[5] = pSCmd->cmnd[5];
		pReq->Cdb[7] = pSCmd->cmnd[8];
		pReq->Cdb[8] = pSCmd->cmnd[9];
		pReq->Cdb[9] = pSCmd->cmnd[11];
		break;

	case WRITE_12:
		pReq->Cdb[0] = WRITE_10;
		pReq->Cdb[1] = pSCmd->cmnd[1];
		pReq->Cdb[2] = pSCmd->cmnd[2];
		pReq->Cdb[3] = pSCmd->cmnd[3];
		pReq->Cdb[4] = pSCmd->cmnd[4];
		pReq->Cdb[5] = pSCmd->cmnd[5];
		pReq->Cdb[7] = pSCmd->cmnd[8];
		pReq->Cdb[8] = pSCmd->cmnd[9];
		pReq->Cdb[9] = pSCmd->cmnd[11];
		break;
#endif
#ifdef MY_ABC_HERE
	case SCSI_CMD_MARVELL_MVSATA_SPECIFIC:
		memcpy(pReq->Cdb, pSCmd->cmnd, MAX_CDB_SIZE);
		pReq->Cdb[0] = SCSI_CMD_MARVELL_SPECIFIC;
		pReq->Cdb[1] = CDB_CORE_MODULE_EXTENSION;		
		pReq->Cmd_Flag |= CMD_FLAG_SHIFT_RESULT;
		blIsSCSIPassThrough = 1;
		break;
#endif
	default:
		memcpy(pReq->Cdb, pSCmd->cmnd, MAX_CDB_SIZE);
		break;
	}

#ifdef MY_ABC_HERE
	/* opcode length not right */
	if (6 >= pSCmd->request_bufflen &&
		blIsSCSIPassThrough) {
		return MV_FALSE;
	}

	if ((SCSI_IS_INSTANT(pSCmd->cmnd[0]) || blIsSCSIPassThrough )
		&& pSCmd->use_sg) {
		unsigned char *pBuffer;
#else
	if (SCSI_IS_INSTANT(pSCmd->cmnd[0]) && pSCmd->use_sg) {
#endif
		struct scatterlist *sg = (struct scatterlist *) pSCmd->request_buffer;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 0)
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
		if ( pSCmd->use_sg > 1 )
			MV_DBG(DMSG_SCSI, 
			       "_MV_ more than 1 sg list in instant cmd.\n");
		pReq->Data_Buffer = kmalloc(sg->length, GFP_ATOMIC);
		if (pReq->Data_Buffer) {
			memset(pReq->Data_Buffer, 0, sg->length);
		}
#else
		pReq->Data_Buffer = kzalloc(sg->length, GFP_ATOMIC);
#endif /* 2.6.14 */
		if ( NULL == pReq->Data_Buffer )
			return MV_FALSE;

#ifdef MY_ABC_HERE
		if (blIsSCSIPassThrough) {
			pBuffer = kmap_atomic(sg_page(sg), KM_IRQ0) + sg->offset;	
			memcpy(pReq->Data_Buffer, pBuffer , sg->length);
			kunmap_atomic(pBuffer - sg->offset, KM_IRQ0);
		}
#endif

		pReq->Data_Transfer_Length = sg->length;
		MV_SCp(pSCmd)->map_atomic = 1;
//#elif //TBD
#endif /* 2.5.0  */
	} else {
		pReq->Data_Buffer = pSCmd->request_buffer;
	}

#ifdef MY_ABC_HERE
	if (blIsSCSIPassThrough &&
		TranslateSCSIPassThroughRequest(pReq, pSCmd)) {
		if (pReq->Data_Buffer) {
			kfree(pReq->Data_Buffer);
			pReq->Data_Buffer = NULL;
		}
		return MV_FALSE;
	}
#endif

	pReq->Sense_Info_Buffer = pSCmd->sense_buffer;

	/* Init the SG table first no matter it's data command or non-data command. */
	SGTable_Init(&pReq->SG_Table, 0);
	if ( pSCmd->request_bufflen )
	{
		GenerateSGTable(pHBA, pSCmd, &pReq->SG_Table);
	}

	pReq->Org_Req = pSCmd;
	pReq->Context = NULL;

	pReq->LBA.value = 0;
	pReq->Sector_Count = 0;

	pReq->Tag = pSCmd->tag; 
	pReq->Scsi_Status = REQ_STATUS_PENDING;

	pReq->Completion = HBARequestCallback;
	
#ifdef __AC_REQ_TRACE__
	MV_DBG(DMSG_PROF_FREQ, "_MV_ OS REQ : ");
	__hba_dump_req_info(pReq);
#endif /* __AC_REQ_TRACE__ */
	return MV_TRUE;
}

MV_BOOLEAN TranslateOSRequest(
	IN PHBA_Extension pHBA,
	IN struct scsi_cmnd * pSCmd,
	OUT PMV_Request pReq
	)
{
	pReq->Cmd_Initiator = pHBA; //TODO
	pReq->Org_Req = pSCmd;
	pReq->Scsi_Status = REQ_STATUS_INVALID_REQUEST; //TBD??

 	return TranslateSCSIRequest(pHBA, pSCmd, pReq);
}

/* This is the only function that OS request can be returned. */
void HBARequestCallback(
	MV_PVOID This,
	PMV_Request pReq
	)
{
	PHBA_Extension pHBA = (PHBA_Extension)This;
	struct scsi_cmnd *pSCmd = (struct scsi_cmnd *)pReq->Org_Req;	

	/* Return this request to OS. */
	HBA_Translate_Req_Status_To_OS_Status(pHBA, pSCmd, pReq);
	
	List_Add(&pReq->Queue_Pointer, &pHBA->Free_Request);
	pHBA->Io_Count--;
}

void GenerateSGTable(
	IN PHBA_Extension pHBA,
	IN struct scsi_cmnd *SCpnt,
	OUT PMV_SG_Table pSGTable
	)
{
	struct scatterlist *sg = (struct scatterlist *)SCpnt->request_buffer;
	unsigned int sg_count = 0;
	BUS_ADDRESS busaddr = 0;
	int i;

	MV_DBG(DMSG_FREQ,
	       "In GenerateSGTable.\n");

	if (SCpnt->request_bufflen > (mv_scmd_host(SCpnt)->max_sectors << 9)) {
		MV_DBG(DMSG_SCSI, "ERROR: request length exceeds "
		       "the maximum alowed value, %x %x\n",
		       pHBA->Device_Id, pHBA->Revision_Id);
	}

	if (SCpnt->use_sg) {
		unsigned int length;
		sg = (struct scatterlist *) SCpnt->request_buffer;
		if (MV_SCp(SCpnt)->mapped == 0) {
			MV_DBG(DMSG_FREQ,"__MV__ call pci_map_sg.\n");
			sg_count = pci_map_sg(pHBA->pcidev, 
					      sg,
					      SCpnt->use_sg,
					      scsi_to_pci_dma_dir(SCpnt->sc_data_direction));
			if (sg_count != SCpnt->use_sg) {
				MV_PRINT("WARNING sg_count(%d) != "
					 "SCpnt->use_sg(%d)\n",
					 (unsigned int) sg_count, 
					 SCpnt->use_sg);
			}
			MV_SCp(SCpnt)->mapped = 1;
		}

		for (i = 0; i < sg_count; i++) {
			busaddr = sg_dma_address(&sg[i]);
			length = sg_dma_len(&sg[i]);
			
			SGTable_Append( pSGTable, 
					LO_BUSADDR(busaddr), 
					HI_BUSADDR(busaddr),
					length );
		}
	} else {
		if (MV_SCp(SCpnt)->mapped == 0) {
			MV_DBG(DMSG_SCSI_FREQ, 
			       "_MV_ pci_map_single for scmd.\n");

			busaddr = pci_map_single(pHBA->pcidev,
						 SCpnt->request_buffer,
						 SCpnt->request_bufflen,
						 scsi_to_pci_dma_dir(SCpnt->sc_data_direction));
			MV_SCp(SCpnt)->bus_address = busaddr;

			MV_SCp(SCpnt)->mapped = 1;
		}

		SGTable_Append( pSGTable, 
				LO_BUSADDR(busaddr), 
				HI_BUSADDR(busaddr),
				SCpnt->request_bufflen);
	}
}

/*need to be optimized lily*/
void HBA_kunmap_sg(void* pReq)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
	void *buf;	
	struct scsi_cmnd *scmd = NULL;
	struct scatterlist *sg = NULL;
	PMV_Request        req = NULL;

	req  = (PMV_Request) pReq;
	scmd = (struct scsi_cmnd *) req->Org_Req;

	if (scmd)
		sg = (struct scatterlist *) scmd->request_buffer;

	if (NULL == sg) {
		MV_DBG(DMSG_HBA, "no org_req found in the req.\n");
		return;
	}
		
	if (MV_SCp(scmd)->map_atomic) {
		WARN_ON(!irqs_disabled());
#ifdef MY_ABC_HERE
		buf = kmap_atomic(sg_page(sg), KM_IRQ0) + sg->offset;
#else
		buf = kmap_atomic(sg->page, KM_IRQ0) + sg->offset;
#endif
#ifdef MY_ABC_HERE
		if ( SCSI_CMD_MARVELL_SPECIFIC == req->Cdb[0] &&
			 CDB_CORE_MODULE_EXTENSION == req->Cdb[1]) {
			unsigned char *pDes = buf;
			unsigned char *pSrc = req->Data_Buffer;
			int len = sg->length;
			char tmp[1024];

			if (req->Cmd_Flag & CMD_FLAG_NON_DATA) {
				if (req->Cmd_Flag & CMD_FLAG_SHIFT_RESULT) {
					pDes += 6;
				}
				len = 8; /* task file size, please refer 
						  * Definitions of S.M.A.R.T. command buffer offsets 
						  * in mvLinuxIalSmart.h
						  */
			} else {
				if (req->Cmd_Flag & CMD_FLAG_SHIFT_RESULT) {
					memcpy(tmp, buf, sg->length);
					pDes += 6;
					pSrc = tmp;
					len = sg->length-6; /* the last 22 bytes is unused */
				}
			}
			memcpy(pDes, pSrc, len);
		}else {
			memcpy(buf, req->Data_Buffer, sg->length);
		}
#else
		memcpy(buf, req->Data_Buffer, sg->length);
#endif
		kunmap_atomic(buf, KM_IRQ0);
		kfree(req->Data_Buffer);
		/* other process might want access to it ... */
		req->Data_Buffer = scmd->request_buffer;
		MV_SCp(scmd)->map_atomic = 0;
	}
#endif	
}

static void hba_shutdown_req_cb(MV_PVOID this, PMV_Request req)
{
	PHBA_Extension phba = (PHBA_Extension) this;

	List_Add(&req->Queue_Pointer, &phba->Free_Request);
	phba->Io_Count--;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11)
	atomic_set(&phba->hba_sync, 0);
#else
	complete(&phba->cmpl);
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11) */
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11)
/* will wait for atomic value atomic to become zero until timed out */
/* return how much 'timeout' is left or 0 if already timed out */
int __hba_wait_for_atomic_timeout(atomic_t *atomic, unsigned long timeout)
{
	unsigned intv = HZ/20; 

	while (timeout) {
		if ( 0 == atomic_read(atomic) )
			break;

		if ( timeout < intv )
			intv = timeout;
		set_current_state(TASK_INTERRUPTIBLE);
		timeout -= (intv - schedule_timeout(intv));
	}
	return timeout;
}
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11) */

void hba_send_shutdown_req(PHBA_Extension phba)
{
	unsigned long flags;
	PMV_Request pReq;

	/*Send MV_REQUEST to do something.*/	
	pReq = kmalloc(sizeof(MV_Request), GFP_ATOMIC);

	/* should we reserve a req for this ? */
	if ( NULL == pReq ) {
		printk("THOR : cannot allocate memory for req.\n");
		return;
	}

	pReq->Cmd_Initiator = phba;
	pReq->Org_Req = pReq; /*no ideas.*/
	pReq->Scsi_Status = REQ_STATUS_INVALID_REQUEST;
	pReq->Completion = hba_shutdown_req_cb;
	
#ifdef RAID_DRIVER
	pReq->Cdb[0] = APICDB0_LD;
	pReq->Cdb[1] = APICDB1_LD_SHUTDOWN;
#else
	pReq->Device_Id = 0;
	pReq->Cmd_Flag = 0;
	pReq->Cmd_Flag |= CMD_FLAG_NON_DATA;
	pReq->Sense_Info_Buffer_Length = 0;  
	pReq->Data_Transfer_Length = 0;
	pReq->Data_Buffer = NULL;
	pReq->Sense_Info_Buffer = NULL;
	SGTable_Init(&pReq->SG_Table, 0);
	pReq->Cdb[0] = SCSI_CMD_MARVELL_SPECIFIC;
	pReq->Cdb[1] = CDB_CORE_MODULE;
	pReq->Cdb[2] = CDB_CORE_SHUTDOWN;
	pReq->Context = NULL;
	pReq->LBA.value = 0;
	pReq->Sector_Count = 0;
	pReq->Scsi_Status = REQ_STATUS_PENDING;
#endif

	spin_lock_irqsave(&phba->lock, flags);
	List_AddTail(&pReq->Queue_Pointer, &phba->Waiting_Request);
	phba->Io_Count++;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11)
	atomic_set(&phba->hba_sync, 1);
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11) */
	HBA_HandleWaitingList(phba);
	spin_unlock_irqrestore(&phba->lock, flags);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11)
	__hba_wait_for_atomic_timeout(&phba->hba_sync, 10*HZ);
#else
	wait_for_completion_timeout(&phba->cmpl, 10*HZ);
#endif /* LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 11) */
}

MV_BOOLEAN HBA_CanHandleRequest (PMV_Request pReq)
{
	switch ( pReq->Cdb[0] )
	{
		case APICDB0_ADAPTER:
			if ( pReq->Cdb[1] == APICDB1_ADAPTER_GETINFO )
				return MV_TRUE;
			else
				return MV_FALSE;
#ifdef SUPPORT_EVENT
		case APICDB0_EVENT:
			return MV_TRUE;
#endif  /* SUPPORT_EVENT */
		default:
			return MV_FALSE;
	}
}

void HBA_HandleWaitingList(PHBA_Extension pHBA)
{
	PMV_Request pReq = NULL;
	MV_PVOID pNextExtension = NULL;
	MV_VOID (*pNextFunction)(MV_PVOID , PMV_Request) = NULL;

	/* Get the request header */
	while ( !List_Empty(&pHBA->Waiting_Request) ) {
		pReq = (PMV_Request)List_GetFirstEntry(&pHBA->Waiting_Request,
						       MV_Request, 
						       Queue_Pointer);
		MV_DASSERT( pReq != NULL );

		if ( NULL == pReq )
			break;
#if 0
		pCore = pHBA->Module_Manage.resource[MODULE_CORE].module_extension;
		//TBD: To the lower module
		module_set[MODULE_CORE].module_sendrequest(pCore, pReq);
#else
		if ( HBA_CanHandleRequest(pReq) ) {
			HBA_ModuleSendRequest( pHBA, pReq );
		} else {
			//TBD: performance
			HBA_GetNextModuleSendFunction(pHBA, 
							&pNextExtension, 
							&pNextFunction);
			MV_DASSERT( pNextExtension!=NULL );
			MV_DASSERT( pNextFunction!=NULL );
			pNextFunction(pNextExtension, pReq);
		}
#endif
	}
}
