/*
   raid0.c : Multiple Devices driver for Linux
             Copyright (C) 1994-96 Marc ZYNGIER
	     <zyngier@ufr-info-p7.ibp.fr> or
	     <maz@gloups.fdn.fr>
             Copyright (C) 1999, 2000 Ingo Molnar, Red Hat


   RAID-0 management functions.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   You should have received a copy of the GNU General Public License
   (for example /usr/src/linux/COPYING); if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
*/

#include <linux/module.h>
#include <linux/raid/raid0.h>

#define MAJOR_NR MD_MAJOR
#define MD_DRIVER
#define MD_PERSONALITY

static void raid0_unplug(struct request_queue *q)
{
	mddev_t *mddev = q->queuedata;
	raid0_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t **devlist = conf->strip_zone[0].dev;
	int i;

#ifndef MY_ABC_HERE
	for (i=0; i<mddev->raid_disks; i++) {
		struct request_queue *r_queue = bdev_get_queue(devlist[i]->bdev);

		blk_unplug(r_queue);
	}
#else
	rcu_read_lock();
	for (i=0; i<mddev->raid_disks; i++) {		
		/* orignal using in stop md array, they will sync all data to these disks
		 * but when some of them are unpluged by user, we not doing sync here
		 * because it will be a NULL point in devlist[i]->bdev
		 */
		mdk_rdev_t *rdev = rcu_dereference(devlist[i]);
		struct request_queue *r_queue = NULL;

		if(!rdev ||
		   test_bit(Faulty, &rdev->flags) ||
		   !atomic_read(&rdev->nr_pending)) {
			continue;
		}

		r_queue = bdev_get_queue(rdev->bdev);

		atomic_inc(&rdev->nr_pending);
		rcu_read_unlock();

		blk_unplug(r_queue);
		atomic_dec(&rdev->nr_pending);
		rcu_read_lock();
	}
	rcu_read_unlock();
#endif
}

static int raid0_congested(void *data, int bits)
{
	mddev_t *mddev = data;
	raid0_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t **devlist = conf->strip_zone[0].dev;
	int i, ret = 0;

#ifdef MY_ABC_HERE
	/* when raid0 lose one of disks, it is not normally,
	 * So we just do a fake report that it is fine,
	 * Nor will encounter NULL pointer access in devlist[i]->bdev.
	 */
	if(mddev->degraded) {
		/* just report it's fine */
		return ret;
	}
#endif

	for (i = 0; i < mddev->raid_disks && !ret ; i++) {
		struct request_queue *q = bdev_get_queue(devlist[i]->bdev);

		ret |= bdi_congested(&q->backing_dev_info, bits);
	}
	return ret;
}


static int create_strip_zones (mddev_t *mddev)
{
	int i, c, j;
#ifdef MD_BACK_PORTING
	sector_t current_start, curr_zone_start;
#else
	sector_t current_offset, curr_zone_offset;
#endif
	sector_t min_spacing;
	raid0_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t *smallest, *rdev1, *rdev2, *rdev;
	struct list_head *tmp1, *tmp2;
	struct strip_zone *zone;
	int cnt;
	char b[BDEVNAME_SIZE];
 
	/*
	 * The number of 'same size groups'
	 */
	conf->nr_strip_zones = 0;
 
	ITERATE_RDEV(mddev,rdev1,tmp1) {
#ifdef MD_BACK_PORTING
		printk(KERN_INFO "raid0: looking at %s\n",
#else
		printk("raid0: looking at %s\n",
#endif
			bdevname(rdev1->bdev,b));
		c = 0;
		ITERATE_RDEV(mddev,rdev2,tmp2) {
#ifdef MD_BACK_PORTING
			printk(KERN_INFO "raid0:   comparing %s(%llu)",
			
			       bdevname(rdev1->bdev,b),
			       (unsigned long long)rdev1->sectors);
			printk(KERN_INFO " with %s(%llu)\n",
			       bdevname(rdev2->bdev,b),
			       (unsigned long long)rdev2->sectors);
#else
			printk("raid0:   comparing %s(%llu)",
				   bdevname(rdev1->bdev,b),
                   (unsigned long long)rdev1->size);
            printk(" with %s(%llu)\n",
                   bdevname(rdev2->bdev,b),
                   (unsigned long long)rdev2->size);
#endif
			if (rdev2 == rdev1) {
#ifdef MD_BACK_PORTING
				printk(KERN_INFO "raid0:   END\n");
#else
				printk("raid0:   END\n");
#endif
				break;
			}
#ifdef MD_BACK_PORTING
			if (rdev2->sectors == rdev1->sectors) {
#else
			if (rdev2->size == rdev1->size)
			{
#endif
				/*
				 * Not unique, don't count it as a new
				 * group
				 */
#ifdef MD_BACK_PORTING
				printk(KERN_INFO "raid0:   EQUAL\n");
#else
				printk("raid0:   EQUAL\n");
#endif
				c = 1;
				break;
			}
#ifdef MD_BACK_PORTING
			printk(KERN_INFO "raid0:   NOT EQUAL\n");
#else
			printk("raid0:   NOT EQUAL\n");
#endif
		}
		if (!c) {
#ifdef MD_BACK_PORTING
			printk(KERN_INFO "raid0:   ==> UNIQUE\n");
#else
			printk("raid0:   ==> UNIQUE\n");
#endif
			conf->nr_strip_zones++;
#ifdef MD_BACK_PORTING
			printk(KERN_INFO "raid0: %d zones\n",
				conf->nr_strip_zones);
#else
			printk("raid0: %d zones\n", conf->nr_strip_zones);
#endif
		}
	}
#ifdef MD_BACK_PORTING
	printk(KERN_INFO "raid0: FINAL %d zones\n", conf->nr_strip_zones);
#else
	printk("raid0: FINAL %d zones\n", conf->nr_strip_zones);
#endif

	conf->strip_zone = kzalloc(sizeof(struct strip_zone)*
				conf->nr_strip_zones, GFP_KERNEL);
	if (!conf->strip_zone)
		return 1;
	conf->devlist = kzalloc(sizeof(mdk_rdev_t*)*
				conf->nr_strip_zones*mddev->raid_disks,
				GFP_KERNEL);
	if (!conf->devlist)
		return 1;

	/* The first zone must contain all devices, so here we check that
	 * there is a proper alignment of slots to devices and find them all
	 */
	zone = &conf->strip_zone[0];
	cnt = 0;
	smallest = NULL;
	zone->dev = conf->devlist;
	ITERATE_RDEV(mddev, rdev1, tmp1) {
		int j = rdev1->raid_disk;

		if (j < 0 || j >= mddev->raid_disks) {
#ifdef MD_BACK_PORTING
			printk(KERN_ERR "raid0: bad disk number %d - "
				"aborting!\n", j);
#else
			printk("raid0: bad disk number %d - aborting!\n", j);
#endif
			goto abort;
		}
		if (zone->dev[j]) {
#ifdef MD_BACK_PORTING
			printk(KERN_ERR "raid0: multiple devices for %d - "
				"aborting!\n", j);
#else
			printk("raid0: multiple devices for %d - aborting!\n",
				   j);
#endif
			goto abort;
		}
		zone->dev[j] = rdev1;

		blk_queue_stack_limits(mddev->queue,
				       rdev1->bdev->bd_disk->queue);
		/* as we don't honour merge_bvec_fn, we must never risk
		 * violating it, so limit ->max_sector to one PAGE, as
		 * a one page request is never in violation.
		 */

		if (rdev1->bdev->bd_disk->queue->merge_bvec_fn &&
		    mddev->queue->max_sectors > (PAGE_SIZE>>9))
			blk_queue_max_sectors(mddev->queue, PAGE_SIZE>>9);

#ifdef MD_BACK_PORTING
		if (!smallest || (rdev1->sectors < smallest->sectors))
#else
		if (!smallest || (rdev1->size <smallest->size))
#endif
			smallest = rdev1;
		cnt++;
	}
	if (cnt != mddev->raid_disks) {
#ifdef MD_BACK_PORTING
		printk(KERN_ERR "raid0: too few disks (%d of %d) - "
				"aborting!\n", cnt, mddev->raid_disks);
#else
		printk("raid0: too few disks (%d of %d) - aborting!\n",
			   cnt, mddev->raid_disks);
#endif
#ifdef MY_ABC_HERE
		/* for raid0 status consistense to other raid type */
		mddev->degraded = mddev->raid_disks - cnt;
		zone->nb_dev = mddev->raid_disks;
#endif
		goto abort;
	}
	zone->nb_dev = cnt;
#ifdef MD_BACK_PORTING
	zone->sectors = smallest->sectors * cnt;
	zone->zone_start = 0;
#else
	zone->size = smallest->size * cnt;
	zone->zone_offset = 0;
#endif

#ifdef MD_BACK_PORTING
	current_start = smallest->sectors;
	curr_zone_start = zone->sectors;
#else
	current_offset = smallest->size;
	curr_zone_offset = zone->size;
#endif

	/* now do the other zones */
	for (i = 1; i < conf->nr_strip_zones; i++)
	{
		zone = conf->strip_zone + i;
		zone->dev = conf->strip_zone[i-1].dev + mddev->raid_disks;

#ifdef MD_BACK_PORTING
		printk(KERN_INFO "raid0: zone %d\n", i);
		zone->dev_start = current_start;
#else
		printk("raid0: zone %d\n", i);
		zone->dev_offset = current_offset;
#endif
		smallest = NULL;
		c = 0;

		for (j=0; j<cnt; j++) {
			char b[BDEVNAME_SIZE];
			rdev = conf->strip_zone[0].dev[j];
#ifdef MD_BACK_PORTING
			printk(KERN_INFO "raid0: checking %s ...",
				bdevname(rdev->bdev, b));
			if (rdev->sectors <= current_start) {
				printk(KERN_INFO " nope.\n");
				continue;
			}
			printk(KERN_INFO " contained as device %d\n", c);
			zone->dev[c] = rdev;
			c++;
			if (!smallest || rdev->sectors < smallest->sectors) {
				smallest = rdev;
				printk(KERN_INFO "  (%llu) is smallest!.\n",
					(unsigned long long)rdev->sectors);
			}
#else
			printk("raid0: checking %s ...", bdevname(rdev->bdev,b));
			if (rdev->size > current_offset)
			{
				printk(" contained as device %d\n", c);
                zone->dev[c] = rdev;
                c++;
                if (!smallest || (rdev->size <smallest->size)) {
                    smallest = rdev;
					printk("  (%llu) is smallest!.\n",
						   (unsigned long long)rdev->size);
				}
			} else
				printk(" nope.\n");
#endif
		}

		zone->nb_dev = c;
#ifdef MD_BACK_PORTING
		zone->sectors = (smallest->sectors - current_start) * c;
		printk(KERN_INFO "raid0: zone->nb_dev: %d, sectors: %llu\n",
			zone->nb_dev, (unsigned long long)zone->sectors);

		zone->zone_start = curr_zone_start;
		curr_zone_start += zone->sectors;

		current_start = smallest->sectors;
		printk(KERN_INFO "raid0: current zone start: %llu\n",
			(unsigned long long)current_start);
#else
		zone->size = (smallest->size - current_offset) * c;
		printk("raid0: zone->nb_dev: %d, size: %llu\n",
			   zone->nb_dev, (unsigned long long)zone->size);
		
		zone->zone_offset = curr_zone_offset;
		curr_zone_offset += zone->size;
		
		current_offset = smallest->size;
		printk("raid0: current zone offset: %llu\n",
			   (unsigned long long)current_offset);
#endif
	}

	/* Now find appropriate hash spacing.
	 * We want a number which causes most hash entries to cover
	 * at most two strips, but the hash table must be at most
	 * 1 PAGE.  We choose the smallest strip, or contiguous collection
	 * of strips, that has big enough size.  We never consider the last
	 * strip though as it's size has no bearing on the efficacy of the hash
	 * table.
	 */
#ifdef MD_BACK_PORTING
	conf->spacing = curr_zone_start;
	min_spacing = curr_zone_start;
#else
	conf->hash_spacing = curr_zone_offset;
	min_spacing = curr_zone_offset;
#endif
	sector_div(min_spacing, PAGE_SIZE/sizeof(struct strip_zone*));
	for (i=0; i < conf->nr_strip_zones-1; i++) {
#ifdef MD_BACK_PORTING
		sector_t s = 0;
		for (j = i; j < conf->nr_strip_zones - 1 &&
				s < min_spacing; j++)
			s += conf->strip_zone[j].sectors;
		if (s >= min_spacing && s < conf->spacing)
			conf->spacing = s;
#else
		sector_t sz = 0;
		for (j=i; j<conf->nr_strip_zones-1 &&
			  sz < min_spacing ; j++)
			sz += conf->strip_zone[j].size;
		if (sz >= min_spacing && sz < conf->hash_spacing)
			conf->hash_spacing = sz;		
#endif
	}

	mddev->queue->unplug_fn = raid0_unplug;

	mddev->queue->backing_dev_info.congested_fn = raid0_congested;
	mddev->queue->backing_dev_info.congested_data = mddev;

#ifdef MD_BACK_PORTING
	printk(KERN_INFO "raid0: done.\n");
#else
	printk("raid0: done.\n");
#endif
	return 0;
 abort:
	return 1;
}

/**
 *	raid0_mergeable_bvec -- tell bio layer if a two requests can be merged
 *	@q: request queue
 *	@bio: the buffer head that's been built up so far
 *	@biovec: the request that could be merged to it.
 *
 *	Return amount of bytes we can accept at this offset
 */
static int raid0_mergeable_bvec(struct request_queue *q, struct bio *bio, struct bio_vec *biovec)
{
	mddev_t *mddev = q->queuedata;
	sector_t sector = bio->bi_sector + get_start_sect(bio->bi_bdev);
	int max;
	unsigned int chunk_sectors = mddev->chunk_size >> 9;
	unsigned int bio_sectors = bio->bi_size >> 9;

	max =  (chunk_sectors - ((sector & (chunk_sectors - 1)) + bio_sectors)) << 9;
	if (max < 0) max = 0; /* bio_add cannot handle a negative return */
	if (max <= biovec->bv_len && bio_sectors == 0)
		return biovec->bv_len;
	else 
		return max;
}

#ifdef MD_BACK_PORTING
static sector_t raid0_size(mddev_t *mddev, sector_t sectors, int raid_disks)
{
	sector_t array_sectors = 0;
	mdk_rdev_t *rdev;

	WARN_ON(sectors || raid_disks);

	list_for_each_entry(rdev, &mddev->disks, same_set)
		array_sectors += rdev->sectors;

	return array_sectors;
}
#endif

static int raid0_run (mddev_t *mddev)
{
	unsigned  cur=0, i=0, nb_zone;
#ifdef MD_BACK_PORTING
	s64 sectors;
#else
	s64 size;
#endif
	raid0_conf_t *conf;
#ifndef MD_BACK_PORTING
	mdk_rdev_t *rdev;
#endif
	struct list_head *tmp;

#ifdef MY_ABC_HERE
	mddev->degraded = 0;
#endif

	if (mddev->chunk_size == 0) {
		printk(KERN_ERR "md/raid0: non-zero chunk size required.\n");
		return -EINVAL;
	}
	printk(KERN_INFO "%s: setting max_sectors to %d, segment boundary to %d\n",
	       mdname(mddev),
	       mddev->chunk_size >> 9,
	       (mddev->chunk_size>>1)-1);
	blk_queue_max_sectors(mddev->queue, mddev->chunk_size >> 9);
	blk_queue_segment_boundary(mddev->queue, (mddev->chunk_size>>1) - 1);

	conf = kmalloc(sizeof (raid0_conf_t), GFP_KERNEL);
	if (!conf)
		goto out;
	mddev->private = (void *)conf;
 
	conf->strip_zone = NULL;
	conf->devlist = NULL;
#ifdef MY_ABC_HERE
	if (create_strip_zones (mddev)){
#ifdef MY_ABC_HERE
			mddev->nodev_and_crashed = 1;
#endif
			/* seeting to NULL for safe free it in mdstop*/
			conf->hash_table = NULL;

			/* The size must greater than zero, 
			 * otherwise this partition would not present in /proc/partitions
			 */
#ifdef MD_BACK_PORTING
			mddev->array_sectors = raid0_size(mddev, 0, 0);
#else
			ITERATE_RDEV(mddev,rdev,tmp)
				mddev->array_size += rdev->size;			
#endif
			/* pretend success for printing mdstatus otherwise it will not show raid0 status when it fail on boot*/			
			return 0;
	}
#else
	if (create_strip_zones (mddev))
		goto out_free_conf;
#endif

	/* calculate array device size */
#ifdef MD_BACK_PORTING
	md_set_array_sectors(mddev, raid0_size(mddev, 0, 0));
#else
	mddev->array_size = 0;
	ITERATE_RDEV(mddev,rdev,tmp)
		mddev->array_size += rdev->size;	
#endif

#ifdef MD_BACK_PORTING
	printk(KERN_INFO "raid0 : md_size is %llu sectors.\n",
			(unsigned long long)mddev->array_sectors);
	printk(KERN_INFO "raid0 : conf->spacing is %llu sectors.\n",
			(unsigned long long)conf->spacing);
#else
	printk("raid0 : md_size is %llu blocks.\n",
		   (unsigned long long)mddev->array_size);
	printk("raid0 : conf->hash_spacing is %llu blocks.\n",
		   (unsigned long long)conf->hash_spacing);
#endif

	{
#ifdef MD_BACK_PORTING
		sector_t s = raid0_size(mddev, 0, 0);
		sector_t space = conf->spacing;
#else
		sector_t s = mddev->array_size;
		sector_t space = conf->hash_spacing;		
#endif
		int round;
#ifdef MD_BACK_PORTING
		conf->sector_shift = 0;
#else
		conf->preshift = 0;
#endif
		if (sizeof(sector_t) > sizeof(u32)) {
			/*shift down space and s so that sector_div will work */
			while (space > (sector_t) (~(u32)0)) {
				s >>= 1;
				space >>= 1;
				s += 1; /* force round-up */
#ifdef MD_BACK_PORTING
				conf->sector_shift++;
#else
				conf->preshift++;
#endif
			}
		}
		round = sector_div(s, (u32)space) ? 1 : 0;
		nb_zone = s + round;
	}
#ifdef MD_BACK_PORTING
	printk(KERN_INFO "raid0 : nb_zone is %d.\n", nb_zone);
#else
	printk("raid0 : nb_zone is %d.\n", nb_zone);
#endif

#ifdef MD_BACK_PORTING
	printk(KERN_INFO "raid0 : Allocating %zu bytes for hash.\n",				
#else
	printk("raid0 : Allocating %Zd bytes for hash.\n",
#endif
				nb_zone*sizeof(struct strip_zone*));
	conf->hash_table = kmalloc (sizeof (struct strip_zone *)*nb_zone, GFP_KERNEL);
	if (!conf->hash_table)
		goto out_free_conf;
#ifdef MD_BACK_PORTING
	sectors = conf->strip_zone[cur].sectors;
#else
	size = conf->strip_zone[cur].size;
#endif

	conf->hash_table[0] = conf->strip_zone + cur;
	for (i=1; i< nb_zone; i++) {
#ifdef MD_BACK_PORTING
		while (sectors <= conf->spacing) {
#else
		while (size <= conf->hash_spacing) {
#endif
			cur++;
#ifdef MD_BACK_PORTING
			sectors += conf->strip_zone[cur].sectors;
#else
			size += conf->strip_zone[cur].size;
#endif
		}
#ifdef MD_BACK_PORTING
		sectors -= conf->spacing;
#else
		size -= conf->hash_spacing;
#endif
		conf->hash_table[i] = conf->strip_zone + cur;
	}
#ifdef MD_BACK_PORTING
	if (conf->sector_shift) {
		conf->spacing >>= conf->sector_shift;
		/* round spacing up so when we divide by it, we
		 * err on the side of too-low, which is safest
		 */
		conf->spacing++;
	}
#else
	if (conf->preshift) {
		conf->hash_spacing >>= conf->preshift;
	    /* round hash_spacing up so when we divide by it, we
		 * err on the side of too-low, which is safest
		 */
		conf->hash_spacing++;
	}
#endif

	/* calculate the max read-ahead size.
	 * For read-ahead of large files to be effective, we need to
	 * readahead at least twice a whole stripe. i.e. number of devices
	 * multiplied by chunk size times 2.
	 * If an individual device has an ra_pages greater than the
	 * chunk size, then we will not drive that device as hard as it
	 * wants.  We consider this a configuration error: a larger
	 * chunksize should be used in that case.
	 */
	{
		int stripe = mddev->raid_disks * mddev->chunk_size / PAGE_SIZE;
		if (mddev->queue->backing_dev_info.ra_pages < 2* stripe)
			mddev->queue->backing_dev_info.ra_pages = 2* stripe;
	}


	blk_queue_merge_bvec(mddev->queue, raid0_mergeable_bvec);
	return 0;

out_free_conf:
	kfree(conf->strip_zone);
	kfree(conf->devlist);
	kfree(conf);
	mddev->private = NULL;
out:
	return -ENOMEM;
}

static int raid0_stop (mddev_t *mddev)
{
	raid0_conf_t *conf = mddev_to_conf(mddev);

	blk_sync_queue(mddev->queue); /* the unplug fn references 'conf'*/
	kfree(conf->hash_table);
	conf->hash_table = NULL;
	kfree(conf->strip_zone);
	conf->strip_zone = NULL;
#ifdef MY_ABC_HERE
	kfree(conf->devlist);
	conf->devlist = NULL;
#endif
	kfree(conf);
	mddev->private = NULL;

	return 0;
}

#ifdef MY_ABC_HERE
/**
 * This is end_io callback function. 
 * We can use this for bad sector report and device error 
 * handing. Prevent umount panic from file system 
 *  
 * @author \$Author: khchen $
 * @version \$Revision: 1.1 
 *  
 * @param bio    Should not be NULL. Passing from block layer
 * @param error  error number
 */
static void Raid0EndRequest(struct bio *bio, int error)
{
	int uptodate = test_bit(BIO_UPTODATE, &bio->bi_flags);
	mddev_t *mddev;
	mdk_rdev_t *rdev;
	struct bio *data_bio;

	data_bio = bio->bi_private;

	rdev = (mdk_rdev_t *)data_bio->bi_next;
	mddev = rdev->mddev;

	bio->bi_end_io = data_bio->bi_end_io;
	bio->bi_private = data_bio->bi_private;

	if (!uptodate) {		
		if (-ENODEV == error) {
			syno_md_error(mddev, rdev);
		}else{
			/* Let raid0 could keep read.(md_error would let it become read-only) */
#ifdef MY_ABC_HERE
			SynoReportBadSector(bio->bi_sector, bio->bi_rw, mddev->md_minor, bio->bi_bdev, __FUNCTION__);
#endif
			md_error(mddev, rdev);
		}
	}

	atomic_dec(&rdev->nr_pending);
	bio_put(data_bio);
	/* Let mount could successful and bad sector could keep accessing */
	bio_endio(bio, 0);
}
#endif

static int raid0_make_request (struct request_queue *q, struct bio *bio)
{
	mddev_t *mddev = q->queuedata;
#ifdef MD_BACK_PORTING
	unsigned int sect_in_chunk, chunksect_bits, chunk_sects;
#else
	unsigned int sect_in_chunk, chunksize_bits,  chunk_size, chunk_sects;
#endif
	raid0_conf_t *conf = mddev_to_conf(mddev);
	struct strip_zone *zone;
	mdk_rdev_t *tmp_dev;
	sector_t chunk;
#ifdef MD_BACK_PORTING
	sector_t sector, rsect;
#else
    sector_t block, rsect;
#endif
	const int rw = bio_data_dir(bio);
#ifdef MY_ABC_HERE
	struct bio *data_bio;
#endif

	if (unlikely(bio_barrier(bio))) {
		bio_endio(bio, -EOPNOTSUPP);
		return 0;
	}

#ifdef MY_ABC_HERE
	/**
	* if there has any device offline, we don't make any request to
	* our raid0 md array
	*/
#ifdef MY_ABC_HERE
	if (mddev->nodev_and_crashed) {
#else
	if (mddev->degraded) {
#endif
		bio_endio(bio, 0);
		return 0;
	}
#endif

	disk_stat_inc(mddev->gendisk, ios[rw]);
	disk_stat_add(mddev->gendisk, sectors[rw], bio_sectors(bio));

#ifndef MD_BACK_PORTING
	chunk_size = mddev->chunk_size >> 10;
#endif
	chunk_sects = mddev->chunk_size >> 9;
#ifdef MD_BACK_PORTING
	chunksect_bits = ffz(~chunk_sects);
	sector = bio->bi_sector;
#else
	chunksize_bits = ffz(~chunk_size);
	block = bio->bi_sector >> 1;
#endif


	if (unlikely(chunk_sects < (bio->bi_sector & (chunk_sects - 1)) + (bio->bi_size >> 9))) {
		struct bio_pair *bp;
		/* Sanity check -- queue functions should prevent this happening */
		if (bio->bi_vcnt != 1 ||
		    bio->bi_idx != 0)
			goto bad_map;
		/* This is a one page bio that upper layers
		 * refuse to split for us, so we need to split it.
		 */
		bp = bio_split(bio, bio_split_pool, chunk_sects - (bio->bi_sector & (chunk_sects - 1)) );
		if (raid0_make_request(q, &bp->bio1))
			generic_make_request(&bp->bio1);
		if (raid0_make_request(q, &bp->bio2))
			generic_make_request(&bp->bio2);

		bio_pair_release(bp);
		return 0;
	}
 

	{
#ifdef MD_BACK_PORTING
		sector_t x = sector >> conf->sector_shift;
		sector_div(x, (u32)conf->spacing);
#else
		sector_t x = block >> conf->preshift;
		sector_div(x, (u32)conf->hash_spacing);		
#endif
		zone = conf->hash_table[x];
	}

#ifdef MD_BACK_PORTING
	while (sector >= zone->zone_start + zone->sectors)
#else
	while (block >= (zone->zone_offset + zone->size))
#endif
		zone++;

#ifdef MD_BACK_PORTING
	sect_in_chunk = bio->bi_sector & (chunk_sects - 1);
#else
	sect_in_chunk = bio->bi_sector & ((chunk_size<<1) -1);
#endif


	{
#ifdef MD_BACK_PORTING
		sector_t x = (sector - zone->zone_start) >> chunksect_bits;
#else
		sector_t x =  (block - zone->zone_offset) >> chunksize_bits;
#endif

		sector_div(x, zone->nb_dev);
		chunk = x;

#ifdef MD_BACK_PORTING
		x = sector >> chunksect_bits;
#else
		x = block >> chunksize_bits;
#endif
		tmp_dev = zone->dev[sector_div(x, zone->nb_dev)];
	}
#ifdef MD_BACK_PORTING
	rsect = (chunk << chunksect_bits) + zone->dev_start + sect_in_chunk;
#else
	rsect = (((chunk << chunksize_bits) + zone->dev_offset)<<1)
		+ sect_in_chunk;
#endif
 
	bio->bi_bdev = tmp_dev->bdev;
	bio->bi_sector = rsect + tmp_dev->data_offset;

#ifdef MY_ABC_HERE
	data_bio = bio_clone(bio, GFP_NOIO);

	if (data_bio) {
		atomic_inc(&tmp_dev->nr_pending);
		data_bio->bi_end_io = bio->bi_end_io;
		data_bio->bi_private = bio->bi_private;
		data_bio->bi_next = (void *)tmp_dev;

		bio->bi_end_io = Raid0EndRequest;
		bio->bi_private = data_bio;		
	}
#endif

	/*
	 * Let the main block layer submit the IO and resolve recursion:
	 */
	return 1;

bad_map:
	printk("raid0_make_request bug: can't convert block across chunks"
#ifdef MD_BACK_PORTING
		" or bigger than %dk %llu %d\n", chunk_sects / 2,
#else
		" or bigger than %dk %llu %d\n", chunk_size,
#endif
		(unsigned long long)bio->bi_sector, bio->bi_size >> 10);

	bio_io_error(bio);
	return 0;
}

#ifdef MY_ABC_HERE
static void
syno_raid0_status (struct seq_file *seq, mddev_t *mddev)
{
	int k;
	raid0_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t *rdev;
	
	seq_printf(seq, " %dk chunks", mddev->chunk_size/1024);
	seq_printf(seq, " [%d/%d] [", mddev->raid_disks, mddev->raid_disks - mddev->degraded);	
	for (k = 0; k < conf->strip_zone[0].nb_dev; k++) {
		rdev = conf->strip_zone[0].dev[k];
		if(rdev) {
#ifdef MY_ABC_HERE
			seq_printf (seq, "%s", 
						test_bit(In_sync, &rdev->flags) ? 
						(test_bit(DiskError, &rdev->flags) ? "E" : "U") : "_");
#else
			seq_printf (seq, "%s", "U");
#endif
		}else{
			seq_printf (seq, "%s", "_");
		}
	}
	seq_printf (seq, "]");
}
#else
static void raid0_status (struct seq_file *seq, mddev_t *mddev)
{
#undef MD_DEBUG
#ifdef MD_DEBUG
	int j, k, h;
	char b[BDEVNAME_SIZE];
	raid0_conf_t *conf = mddev_to_conf(mddev);

	h = 0;
	for (j = 0; j < conf->nr_strip_zones; j++) {
		seq_printf(seq, "      z%d", j);
		if (conf->hash_table[h] == conf->strip_zone+j)
			seq_printf(seq, "(h%d)", h++);
		seq_printf(seq, "=[");
		for (k = 0; k < conf->strip_zone[j].nb_dev; k++)
			seq_printf(seq, "%s/", bdevname(
				conf->strip_zone[j].dev[k]->bdev,b));

#ifdef MD_BACK_PORTING
		seq_printf(seq, "] zs=%d ds=%d s=%d\n",
				conf->strip_zone[j].zone_start,
				conf->strip_zone[j].dev_start,
				conf->strip_zone[j].sectors);
#else
		seq_printf(seq, "] zo=%d do=%d s=%d\n",
			   conf->strip_zone[j].zone_offset,
			   conf->strip_zone[j].dev_offset,
			   conf->strip_zone[j].size);	
#endif
	}
#endif
	seq_printf(seq, " %dk chunks", mddev->chunk_size/1024);
	return;
}
#endif

#ifdef MY_ABC_HERE
int SynoRaid0RemoveDisk(mddev_t *mddev, int number)
{
	int err = 0;
	char nm[20];
	raid0_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t *rdev;	

	rdev = conf->strip_zone[0].dev[number];
	if (!rdev) {
		goto END;
	}

	if (atomic_read(&rdev->nr_pending)) {
		/* lost the race, try later */
		err = -EBUSY;
		goto END;
	}

	/**
	 * raid0 don't has their own thread, we just remove it's sysfs
	 * when there has no other pending request
	 */ 
	sprintf(nm,"rd%d", rdev->raid_disk);
	sysfs_remove_link(&mddev->kobj, nm);
	rdev->raid_disk = -1;
	conf->strip_zone[0].dev[number] = NULL;
END:
	return err;
}

/**
 * This is our implement for raid handler.
 * It mainly for handling device hotplug. 
 * We let it look like other raid type. 
 * Set it faulty could let SDK know it's status
 *  
 * @author \$Author: khchen $
 * @version \$Revision: 1.1  * 
 * 
 * @param mddev  Should not be NULL. passing from md.c
 * @param rdev   Should not be NULL. passing from md.c
 */
static void SynoRaid0Error(mddev_t *mddev, mdk_rdev_t *rdev)
{
	if (test_and_clear_bit(In_sync, &rdev->flags)) {
		if (mddev->degraded < mddev->raid_disks) {
			mddev->degraded++;
#ifdef MY_ABC_HERE
			mddev->nodev_and_crashed = 1;
#endif
			set_bit(Faulty, &rdev->flags);
			clear_bit(DiskError, &rdev->flags);
			set_bit(MD_CHANGE_DEVS, &mddev->flags);
		}
	}else{
		set_bit(Faulty, &rdev->flags);
	}

	return;
}

/**
 * This is our implement for raid handler.
 * It mainly for mdadm set device faulty. We let it look like 
 * other raid type. Let it become read only (scemd would remount
 * if it find DiskError) 
 *  
 * @author \$Author: khchen $
 * @version \$Revision: 1.1  * 
 * 
 * @param mddev  Should not be NULL. passing from md.c
 * @param rdev   Should not be NULL. passing from md.c
 */
static void SynoRaid0ErrorInternal(mddev_t *mddev, mdk_rdev_t *rdev)
{
#ifdef MY_ABC_HERE
	if (!test_bit(DiskError, &rdev->flags)) {
		SYNO_UPDATE_SB_WORK *update_sb = NULL;

		set_bit(DiskError, &rdev->flags);
		if (NULL == (update_sb = kzalloc(sizeof(SYNO_UPDATE_SB_WORK), GFP_ATOMIC))){
			WARN_ON(!update_sb);
			goto END;
		}

		INIT_WORK(&update_sb->work, SynoUpdateSBTask);
		update_sb->mddev = mddev;
		schedule_work(&update_sb->work);
		set_bit(MD_CHANGE_DEVS, &mddev->flags);
	}

END:
#endif
	return;
}
#endif

static struct mdk_personality raid0_personality=
{
	.name		= "raid0",
	.level		= 0,
	.owner		= THIS_MODULE,
	.make_request	= raid0_make_request,
	.run		= raid0_run,
	.stop		= raid0_stop,
#ifdef MY_ABC_HERE
	.status		= syno_raid0_status,
#else
	.status		= raid0_status,
#endif
#ifdef MD_BACK_PORTING
	.size       = raid0_size,
#endif
#ifdef MY_ABC_HERE
	.hot_remove_disk = SynoRaid0RemoveDisk,
	.error_handler = SynoRaid0ErrorInternal,
	.syno_error_handler = SynoRaid0Error,
#endif
};

static int __init raid0_init (void)
{
	return register_md_personality (&raid0_personality);
}

static void raid0_exit (void)
{
	unregister_md_personality (&raid0_personality);
}

module_init(raid0_init);
module_exit(raid0_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("md-personality-2"); /* RAID0 */
MODULE_ALIAS("md-raid0");
MODULE_ALIAS("md-level-0");
