/*
   linear.c : Multiple Devices driver for Linux
	      Copyright (C) 1994-96 Marc ZYNGIER
	      <zyngier@ufr-info-p7.ibp.fr> or
	      <maz@gloups.fdn.fr>

   Linear mode management functions.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   You should have received a copy of the GNU General Public License
   (for example /usr/src/linux/COPYING); if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
*/

#include <linux/module.h>

#include <linux/raid/md.h>
#include <linux/slab.h>
#include <linux/raid/linear.h>

#define MAJOR_NR MD_MAJOR
#define MD_DRIVER
#define MD_PERSONALITY

/*
 * find which device holds a particular offset 
 */
static inline dev_info_t *which_dev(mddev_t *mddev, sector_t sector)
{
	dev_info_t *hash;
	linear_conf_t *conf = mddev_to_conf(mddev);
	sector_t block = sector >> 1;

	/*
	 * sector_div(a,b) returns the remainer and sets a to a/b
	 */
	block >>= conf->preshift;
	(void)sector_div(block, conf->hash_spacing);
	hash = conf->hash_table[block];

#ifdef MD_BACK_PORTING
	while (sector >= hash->num_sectors + hash->start_sector)
#else
	while ((sector>>1) >= (hash->size + hash->offset))
#endif
		hash++;
	return hash;
}

/**
 *	linear_mergeable_bvec -- tell bio layer if two requests can be merged
 *	@q: request queue
 *	@bio: the buffer head that's been built up so far
 *	@biovec: the request that could be merged to it.
 *
 *	Return amount of bytes we can take at this offset
 */
static int linear_mergeable_bvec(struct request_queue *q, struct bio *bio, struct bio_vec *biovec)
{
	mddev_t *mddev = q->queuedata;
	dev_info_t *dev0;
	unsigned long maxsectors, bio_sectors = bio->bi_size >> 9;
	sector_t sector = bio->bi_sector + get_start_sect(bio->bi_bdev);

	dev0 = which_dev(mddev, sector);
#ifdef MD_BACK_PORTING
	maxsectors = dev0->num_sectors - (sector - dev0->start_sector);
#else
	maxsectors = (dev0->size << 1) - (sector - (dev0->offset<<1));
#endif

	if (maxsectors < bio_sectors)
		maxsectors = 0;
	else
		maxsectors -= bio_sectors;

	if (maxsectors <= (PAGE_SIZE >> 9 ) && bio_sectors == 0)
		return biovec->bv_len;
	/* The bytes available at this offset could be really big,
	 * so we cap at 2^31 to avoid overflow */
	if (maxsectors > (1 << (31-9)))
		return 1<<31;
	return maxsectors << 9;
}

static void linear_unplug(struct request_queue *q)
{
	mddev_t *mddev = q->queuedata;
	linear_conf_t *conf = mddev_to_conf(mddev);
	int i;

#ifdef MY_ABC_HERE
	rcu_read_lock();
	for (i=0; i < mddev->raid_disks; i++) {
		struct request_queue *r_queue;
		mdk_rdev_t *rdev = rcu_dereference(conf->disks[i].rdev);

		if(!rdev ||
		   test_bit(Faulty, &rdev->flags) ||
		   !atomic_read(&rdev->nr_pending)) {
			continue;
		}

		r_queue = bdev_get_queue(rdev->bdev);

		atomic_inc(&rdev->nr_pending);
		rcu_read_unlock();

		blk_unplug(r_queue);
		atomic_dec(&rdev->nr_pending);
		rcu_read_lock();
	}
	rcu_read_unlock();
#else
	for (i=0; i < mddev->raid_disks; i++) {
		struct request_queue *r_queue = bdev_get_queue(conf->disks[i].rdev->bdev);
		blk_unplug(r_queue);
	}
#endif
}

static int linear_congested(void *data, int bits)
{
	mddev_t *mddev = data;
	linear_conf_t *conf = mddev_to_conf(mddev);
	int i, ret = 0;

#ifdef MY_ABC_HERE
	if (mddev->degraded) {
		return ret;
	}

	rcu_read_lock();
	for (i = 0; i < mddev->raid_disks && !ret ; i++) {
		mdk_rdev_t *rdev = rcu_dereference(conf->disks[i].rdev);
		struct request_queue *q = NULL;

		if (!rdev) {
			continue;
		}

		q = bdev_get_queue(rdev->bdev);
		ret |= bdi_congested(&q->backing_dev_info, bits);
	}
	rcu_read_unlock();
#else
	for (i = 0; i < mddev->raid_disks && !ret ; i++) {
		struct request_queue *q = bdev_get_queue(conf->disks[i].rdev->bdev);
		ret |= bdi_congested(&q->backing_dev_info, bits);
	}
#endif
	return ret;
}

#ifdef MD_BACK_PORTING
static sector_t linear_size(mddev_t *mddev, sector_t sectors, int raid_disks)
{
	linear_conf_t *conf = mddev_to_conf(mddev);

	WARN_ON(sectors || raid_disks);

	return conf->array_sectors;
}
#endif

static linear_conf_t *linear_conf(mddev_t *mddev, int raid_disks)
{
	linear_conf_t *conf;
	dev_info_t **table;
	mdk_rdev_t *rdev;
	int i, nb_zone, cnt;
	sector_t min_spacing;
#ifdef MD_BACK_PORTING
	sector_t curr_sector;
#else
	sector_t curr_offset;
#endif
	struct list_head *tmp;

	conf = kzalloc (sizeof (*conf) + raid_disks*sizeof(dev_info_t),
			GFP_KERNEL);
	if (!conf)
		return NULL;

	cnt = 0;
#ifdef MD_BACK_PORTING
	conf->array_sectors = 0;
#else
	conf->array_size = 0;
#endif

	ITERATE_RDEV(mddev,rdev,tmp) {
		int j = rdev->raid_disk;
		dev_info_t *disk = conf->disks + j;

		if (j < 0 || j > raid_disks || disk->rdev) {
			printk("linear: disk numbering problem. Aborting!\n");
			goto out;
		}

		disk->rdev = rdev;

		blk_queue_stack_limits(mddev->queue,
				       rdev->bdev->bd_disk->queue);
		/* as we don't honour merge_bvec_fn, we must never risk
		 * violating it, so limit ->max_sector to one PAGE, as
		 * a one page request is never in violation.
		 */
		if (rdev->bdev->bd_disk->queue->merge_bvec_fn &&
		    mddev->queue->max_sectors > (PAGE_SIZE>>9))
			blk_queue_max_sectors(mddev->queue, PAGE_SIZE>>9);

#ifdef MD_BACK_PORTING
		disk->num_sectors = rdev->sectors;
		conf->array_sectors += rdev->sectors;
#else
		disk->size = rdev->size;
		conf->array_size += rdev->size;		
#endif

		cnt++;
	}
	if (cnt != raid_disks) {
#ifdef MY_ABC_HERE
		/* 
		 * for Linear status consistense to other raid type
		 * Let it can assemble.
		 */
		mddev->degraded = mddev->raid_disks - cnt;		
#ifdef MY_ABC_HERE
		mddev->nodev_and_crashed = 1;
#endif
		printk("linear: not enough drives present.\n");
		return conf;
#else
		printk("linear: not enough drives present. Aborting!\n");
		goto out;
#endif
	}

#ifdef MD_BACK_PORTING
	min_spacing = conf->array_sectors / 2;
#else
	min_spacing = conf->array_size;
#endif
	sector_div(min_spacing, PAGE_SIZE/sizeof(struct dev_info *));

	/* min_spacing is the minimum spacing that will fit the hash
	 * table in one PAGE.  This may be much smaller than needed.
	 * We find the smallest non-terminal set of consecutive devices
	 * that is larger than min_spacing as use the size of that as
	 * the actual spacing
	 */
#ifdef MD_BACK_PORTING
	conf->hash_spacing = conf->array_sectors / 2;
#else
    conf->hash_spacing = conf->array_size;
#endif
	for (i=0; i < cnt-1 ; i++) {
		sector_t sz = 0;
		int j;
		for (j = i; j < cnt - 1 && sz < min_spacing; j++)
#ifdef MD_BACK_PORTING
			sz += conf->disks[j].num_sectors / 2;
#else
			sz += conf->disks[j].size;
#endif
		if (sz >= min_spacing && sz < conf->hash_spacing)
			conf->hash_spacing = sz;
	}

	/* hash_spacing may be too large for sector_div to work with,
	 * so we might need to pre-shift
	 */
	conf->preshift = 0;
	if (sizeof(sector_t) > sizeof(u32)) {
		sector_t space = conf->hash_spacing;
		while (space > (sector_t)(~(u32)0)) {
			space >>= 1;
			conf->preshift++;
		}
	}
	/*
	 * This code was restructured to work around a gcc-2.95.3 internal
	 * compiler error.  Alter it with care.
	 */
	{
		sector_t sz;
		unsigned round;
		unsigned long base;

#ifdef MD_BACK_PORTING
		sz = conf->array_sectors >> (conf->preshift + 1);
#else
		sz = conf->array_size >> conf->preshift;
#endif
		sz += 1; /* force round-up */
		base = conf->hash_spacing >> conf->preshift;
		round = sector_div(sz, base);
		nb_zone = sz + (round ? 1 : 0);
	}
	BUG_ON(nb_zone > PAGE_SIZE / sizeof(struct dev_info *));

	conf->hash_table = kmalloc (sizeof (struct dev_info *) * nb_zone,
					GFP_KERNEL);
	if (!conf->hash_table)
		goto out;

	/*
	 * Here we generate the linear hash table
	 * First calculate the device offsets.
	 */
#ifdef MD_BACK_PORTING
	conf->disks[0].start_sector = 0;
#else
	conf->disks[0].offset = 0;
#endif
	for (i = 1; i < raid_disks; i++)
#ifdef MD_BACK_PORTING
		conf->disks[i].start_sector =
			conf->disks[i-1].start_sector +
			conf->disks[i-1].num_sectors;
#else
		conf->disks[i].offset =
			conf->disks[i-1].offset +
			conf->disks[i-1].size;
	
#endif

	table = conf->hash_table;
#ifndef MD_BACK_PORTING
	curr_offset = 0;
#endif
	i = 0;
#ifdef MD_BACK_PORTING
	for (curr_sector = 0;
	     curr_sector < conf->array_sectors;
	     curr_sector += conf->hash_spacing * 2) {
#else
	for (curr_offset = 0;
		 curr_offset < conf->array_size;
		 curr_offset += conf->hash_spacing) {
#endif

		while (i < raid_disks-1 &&
#ifdef MD_BACK_PORTING
		       curr_sector >= conf->disks[i+1].start_sector)
#else
			   curr_offset >= conf->disks[i+1].offset)
#endif
			i++;

		*table ++ = conf->disks + i;
	}

	if (conf->preshift) {
		conf->hash_spacing >>= conf->preshift;
		/* round hash_spacing up so that when we divide by it,
		 * we err on the side of "too-low", which is safest.
		 */
		conf->hash_spacing++;
	}

	BUG_ON(table - conf->hash_table > nb_zone);

	return conf;

out:
	kfree(conf);
	return NULL;
}

static int linear_run (mddev_t *mddev)
{
	linear_conf_t *conf;

#ifdef MY_ABC_HERE
	mddev->degraded = 0;
#endif
	conf = linear_conf(mddev, mddev->raid_disks);

	if (!conf)
		return 1;
	mddev->private = conf;
#ifdef MD_BACK_PORTING
	md_set_array_sectors(mddev, linear_size(mddev, 0, 0));
#else
	mddev->array_size = conf->array_size;
#endif

	blk_queue_merge_bvec(mddev->queue, linear_mergeable_bvec);
	mddev->queue->unplug_fn = linear_unplug;
	mddev->queue->backing_dev_info.congested_fn = linear_congested;
	mddev->queue->backing_dev_info.congested_data = mddev;
	return 0;
}

static int linear_add(mddev_t *mddev, mdk_rdev_t *rdev)
{
	/* Adding a drive to a linear array allows the array to grow.
	 * It is permitted if the new drive has a matching superblock
	 * already on it, with raid_disk equal to raid_disks.
	 * It is achieved by creating a new linear_private_data structure
	 * and swapping it in in-place of the current one.
	 * The current one is never freed until the array is stopped.
	 * This avoids races.
	 */
	linear_conf_t *newconf;

	if (rdev->saved_raid_disk != mddev->raid_disks)
		return -EINVAL;

	rdev->raid_disk = rdev->saved_raid_disk;

	newconf = linear_conf(mddev,mddev->raid_disks+1);

	if (!newconf)
		return -ENOMEM;

	newconf->prev = mddev_to_conf(mddev);
	mddev->private = newconf;
	mddev->raid_disks++;
#ifdef MD_BACK_PORTING
	md_set_array_sectors(mddev, linear_size(mddev, 0, 0));
	set_capacity(mddev->gendisk, mddev->array_sectors);
#ifdef HAS_THIS_FUNCTION_2_6_24
	revalidate_disk(mddev->gendisk);
#else
	{
		struct block_device *bdev;

		bdev = bdget_disk(mddev->gendisk, 0);
		if (bdev) {
			mutex_lock(&bdev->bd_inode->i_mutex);
			i_size_write(bdev->bd_inode,
						(loff_t)mddev->array_sectors << 9);
			mutex_unlock(&bdev->bd_inode->i_mutex);
			bdput(bdev);
		}
	}
#endif
#else
	mddev->array_size = newconf->array_size;
	set_capacity(mddev->gendisk, mddev->array_size << 1);
#endif
	return 0;
}

static int linear_stop (mddev_t *mddev)
{
	linear_conf_t *conf = mddev_to_conf(mddev);
  
	blk_sync_queue(mddev->queue); /* the unplug fn references 'conf'*/
	do {
		linear_conf_t *t = conf->prev;
		kfree(conf->hash_table);
		kfree(conf);
		conf = t;
	} while (conf);

#ifdef MY_ABC_HERE
	mddev->private = NULL;
#endif
	return 0;
}

#ifdef MY_ABC_HERE
/**
 * This is end_io callback function. 
 * We can use this for bad sector report and device error 
 * handing. Prevent umount panic from file system 
 *  
 * @author \$Author: khchen $
 * @version \$Revision: 1.1 
 *  
 * @param bio    Should not be NULL. Passing from block layer
 * @param error  error number
 */
static void 
SynoLinearEndRequest(struct bio *bio, int error)
{
	int uptodate = test_bit(BIO_UPTODATE, &bio->bi_flags);
	mddev_t *mddev;
	mdk_rdev_t *rdev;
	struct bio *data_bio;

	data_bio = bio->bi_private;

	rdev = (mdk_rdev_t *)data_bio->bi_next;
	mddev = rdev->mddev;

	bio->bi_end_io = data_bio->bi_end_io;
	bio->bi_private = data_bio->bi_private;

	if (!uptodate) {
#ifdef MY_ABC_HERE
		if (-ENODEV == error) {
			syno_md_error(mddev, rdev);
		}else{
#ifdef MY_ABC_HERE
			SynoReportBadSector(bio->bi_sector, bio->bi_rw, mddev->md_minor, bio->bi_bdev, __FUNCTION__);			
#endif
			md_error(mddev, rdev);
		}
#else
		md_error(mddev, rdev);
#endif
	}

	atomic_dec(&rdev->nr_pending);
	bio_put(data_bio);
	/* Let mount could successful and bad sector could keep accessing, no matter it success or not */
	bio_endio(bio, 0);
}
#endif

static int linear_make_request (struct request_queue *q, struct bio *bio)
{
	const int rw = bio_data_dir(bio);
	mddev_t *mddev = q->queuedata;
	dev_info_t *tmp_dev;
#ifndef MD_BACK_PORTING
	sector_t block;
#endif
#ifdef MY_ABC_HERE
	struct bio *data_bio;
#endif

	if (unlikely(bio_barrier(bio))) {
		bio_endio(bio, -EOPNOTSUPP);
		return 0;
	}

#ifdef MY_ABC_HERE
	/**
	* if there has any device offline, we don't make any request to
	* our linear md array
	*/
#ifdef MY_ABC_HERE
	if (mddev->nodev_and_crashed) {
#else
	if (mddev->degraded) {
#endif
		bio_endio(bio, 0);
		return 0;
	}
#endif

	disk_stat_inc(mddev->gendisk, ios[rw]);
	disk_stat_add(mddev->gendisk, sectors[rw], bio_sectors(bio));

	tmp_dev = which_dev(mddev, bio->bi_sector);
#ifdef MD_BACK_PORTING
	if (unlikely(bio->bi_sector >= (tmp_dev->num_sectors +
									tmp_dev->start_sector)
				|| (bio->bi_sector <
					tmp_dev->start_sector))) {    
#else
	block = bio->bi_sector >> 1;

	if (unlikely(block >= (tmp_dev->size + tmp_dev->offset)
				 || block < tmp_dev->offset)) {
#endif
		char b[BDEVNAME_SIZE];

#ifdef MD_BACK_PORTING
		printk("linear_make_request: Sector %llu out of bounds on "
				"dev %s: %llu sectors, offset %llu\n",
				(unsigned long long)bio->bi_sector,				
#else
		printk("linear_make_request: Block %llu out of bounds on "
			   "dev %s size %llu offset %llu\n",
			   (unsigned long long)block,
#endif
			   bdevname(tmp_dev->rdev->bdev, b),
#ifdef MD_BACK_PORTING
				(unsigned long long)tmp_dev->num_sectors,
				(unsigned long long)tmp_dev->start_sector);
#else
				(unsigned long long)tmp_dev->size,
					(unsigned long long)tmp_dev->offset);
#endif

		bio_io_error(bio);
		return 0;
	}
	if (unlikely(bio->bi_sector + (bio->bi_size >> 9) >
#ifdef MD_BACK_PORTING
				 tmp_dev->start_sector + tmp_dev->num_sectors)) {
#else
				 (tmp_dev->offset + tmp_dev->size)<<1)) {
#endif
		/* This bio crosses a device boundary, so we have to
		 * split it.
		 */
		struct bio_pair *bp;
		bp = bio_split(bio, bio_split_pool,
#ifdef MD_BACK_PORTING
					   tmp_dev->start_sector + tmp_dev->num_sectors
					   - bio->bi_sector);
#else
					((tmp_dev->offset + tmp_dev->size)<<1) - bio->bi_sector);
#endif

		if (linear_make_request(q, &bp->bio1))
			generic_make_request(&bp->bio1);
		if (linear_make_request(q, &bp->bio2))
			generic_make_request(&bp->bio2);
		bio_pair_release(bp);
		return 0;
	}
		    
	bio->bi_bdev = tmp_dev->rdev->bdev;
#ifdef MD_BACK_PORTING
	bio->bi_sector = bio->bi_sector - tmp_dev->start_sector
			+ tmp_dev->rdev->data_offset;
#else
	bio->bi_sector = bio->bi_sector - (tmp_dev->offset << 1) + tmp_dev->rdev->data_offset;
#endif
#ifdef MY_ABC_HERE	
	data_bio = bio_clone(bio, GFP_NOIO);

	if (data_bio) {
		atomic_inc(&tmp_dev->rdev->nr_pending);
		data_bio->bi_end_io = bio->bi_end_io;
		data_bio->bi_private = bio->bi_private;
		data_bio->bi_next = (void *)tmp_dev->rdev;

		bio->bi_end_io = SynoLinearEndRequest;
		bio->bi_private = data_bio;		
	}
#endif

	return 1;
}

#ifdef MY_ABC_HERE

static void
syno_linear_status(struct seq_file *seq, mddev_t *mddev)
{	
	linear_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t *rdev;
	int j;

	seq_printf(seq, " %dk rounding", mddev->chunk_size/1024);
	seq_printf(seq, " [%d/%d] [", mddev->raid_disks, mddev->raid_disks - mddev->degraded);
	rcu_read_lock();
	for (j = 0; j < mddev->raid_disks; j++)
	{
		rdev = rcu_dereference(conf->disks[j].rdev);
#ifdef MY_ABC_HERE
		if(rdev &&
		   !test_bit(Faulty, &rdev->flags)) {
#else		
		if(rdev) {
#endif
#ifdef MY_ABC_HERE
			seq_printf (seq, "%s", 
						test_bit(In_sync, &rdev->flags) ? 
						(test_bit(DiskError, &rdev->flags) ? "E" : "U") : "_");
#else
			seq_printf (seq, "%s", "U");
#endif
		}else{
			seq_printf (seq, "%s", "_");
		}
	}
	rcu_read_unlock();
	seq_printf (seq, "]");
}
#else
static void linear_status (struct seq_file *seq, mddev_t *mddev)
{

#undef MD_DEBUG
#ifdef MD_DEBUG
	int j;
	linear_conf_t *conf = mddev_to_conf(mddev);
	sector_t s = 0;
  
	seq_printf(seq, "      ");
	for (j = 0; j < mddev->raid_disks; j++)
	{
		char b[BDEVNAME_SIZE];
		s += conf->smallest_size;
		seq_printf(seq, "[%s",
			   bdevname(conf->hash_table[j][0].rdev->bdev,b));

		while (s > conf->hash_table[j][0].offset +
		           conf->hash_table[j][0].size)
			seq_printf(seq, "/%s] ",
				   bdevname(conf->hash_table[j][1].rdev->bdev,b));
		else
			seq_printf(seq, "] ");
	}
	seq_printf(seq, "\n");
#endif
	seq_printf(seq, " %dk rounding", mddev->chunk_size/1024);
}

#endif

#ifdef MY_ABC_HERE
int 
SynoLinearRemoveDisk(mddev_t *mddev, int number)
{
	int err = 0;
	char nm[20];
	linear_conf_t *conf = mddev_to_conf(mddev);
	mdk_rdev_t *rdev;	

	rdev = conf->disks[number].rdev;
	if (!rdev) {
		goto END;
	}

	if (atomic_read(&rdev->nr_pending)) {
		/* lost the race, try later */
		err = -EBUSY;
		goto END;
	}

	/**
	 * Linear don't has their own thread, we just remove it's sysfs
	 * when there has no other pending request
	 */ 
	sprintf(nm,"rd%d", rdev->raid_disk);
	sysfs_remove_link(&mddev->kobj, nm);
	rdev->raid_disk = -1;
	conf->disks[number].rdev = NULL;
END:
	return err;
}

/**
 * This is our implement for raid handler.
 * It mainly for handling device hotplug. 
 * We let it look like other raid type. 
 * Set it faulty could let SDK know it's status
 *  
 * @author \$Author: khchen $
 * @version \$Revision: 1.1   
 * 
 * @param mddev  Should not be NULL. passing from md.c
 * @param rdev   Should not be NULL. passing from md.c
 */
static void 
SynoLinearError(mddev_t *mddev, mdk_rdev_t *rdev)
{
	if (test_and_clear_bit(In_sync, &rdev->flags)) {
		if (mddev->degraded < mddev->raid_disks) {
			mddev->degraded++;
#ifdef MY_ABC_HERE
			mddev->nodev_and_crashed = 1;
#endif
			set_bit(Faulty, &rdev->flags);			
#ifdef MY_ABC_HERE
			clear_bit(DiskError, &rdev->flags);
#endif
			set_bit(MD_CHANGE_DEVS, &mddev->flags);
		}
	}

	return;
}

/**
 * This is our implement for raid handler.
 * It mainly for mdadm set device faulty. We let it look like 
 * other raid type. Let it become read only (scemd would remount
 * if it find DiskError) 
 *  
 * You should not sync super block in the same thread, otherwise 
 * would panic. 
 *  
 * @author \$Author: khchen $
 * @version \$Revision: 1.1  * 
 * 
 * @param mddev  Should not be NULL. passing from md.c
 * @param rdev   Should not be NULL. passing from md.c
 */
static void
SynoLinearErrorInternal(mddev_t *mddev, mdk_rdev_t *rdev)
{
#ifdef MY_ABC_HERE
	if (!test_bit(DiskError, &rdev->flags)) {
		SYNO_UPDATE_SB_WORK *update_sb = NULL;

		set_bit(DiskError, &rdev->flags);
		if (NULL == (update_sb = kzalloc(sizeof(SYNO_UPDATE_SB_WORK), GFP_ATOMIC))){
			WARN_ON(!update_sb);
			goto END;
		}

		INIT_WORK(&update_sb->work, SynoUpdateSBTask);
		update_sb->mddev = mddev;
		schedule_work(&update_sb->work);
		set_bit(MD_CHANGE_DEVS, &mddev->flags);
	}

END:
#endif
	return;
}
#endif

static struct mdk_personality linear_personality =
{
	.name		= "linear",
	.level		= LEVEL_LINEAR,
	.owner		= THIS_MODULE,
	.make_request	= linear_make_request,
	.run		= linear_run,
	.stop		= linear_stop,
#ifdef MY_ABC_HERE
	.status		= syno_linear_status,
#else
	.status		= linear_status,
#endif
	.hot_add_disk	= linear_add,
#ifdef MY_ABC_HERE
	.hot_remove_disk	= SynoLinearRemoveDisk,
	.error_handler		= SynoLinearErrorInternal,
	.syno_error_handler	= SynoLinearError,
#endif
#ifdef MD_BACK_PORTING
	.size       = linear_size,
#endif
};

static int __init linear_init (void)
{
	return register_md_personality (&linear_personality);
}

static void linear_exit (void)
{
	unregister_md_personality (&linear_personality);
}


module_init(linear_init);
module_exit(linear_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("md-personality-1"); /* LINEAR - deprecated*/
MODULE_ALIAS("md-linear");
MODULE_ALIAS("md-level--1");
