/*
 * Freescale MPC83xx DMA XOR Engine support
 *
 * Copyright (C) 2008 Freescale Semiconductor, Inc. All rights reserved.
 *
 * Author:
 *   Dipen Dudhat <Dipen.Dudhat@freescale.com>, Nov 2008
 *   Surender Kumar <R66464@freescale.com>, Feb 2008
 *   Vishnu Suresh <Vishnu@freescale.com>, Feb  2008
 *
 * Description:
 *   DMA XOR engine driver for Freescale SEC Engine. The XWC framework
 *   is used to expose the SEC Engine as a DMA engine with XOR capability.
 *   The support for ZERO SUM is not implemented at present.
 *   This code is the backend which interfaces with the XWC freamework
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/uaccess.h>
#include <linux/irq.h>
#include <linux/io.h>

#include <asm/atomic.h>
#include <asm/semaphore.h>
#include <asm/pgtable.h>
#include <asm/page.h>

#include "fsl_xor.h"

struct fsl_xor_xwc_device *xor_xwc_dev;

#define lh_to_xwc_xor_mesg(lh) container_of(lh, struct fsl_xor_xwc_msg, node)

/**
 * fsl_xor_xwc_alloc_resource - Allocate set of resources
 */
void fsl_xor_xwc_alloc_resource()
{
	uint16_t i = 0;
	struct fsl_xor_xwc_msg *xwc_mesg;
	struct list_head tmp_list;
	INIT_LIST_HEAD(&tmp_list);

	for (i = 0; i < INITIAL_FSL_DESC_COUNT; i++) {
		xwc_mesg = kzalloc(sizeof(*xwc_mesg), GFP_KERNEL);
		if (!xwc_mesg) {
			printk(KERN_DEBUG "Cannot allocate XWC message desc");
			break;
		}
		list_add_tail(&xwc_mesg->node, &tmp_list);
	}
	spin_lock_bh(&xor_xwc_dev->msg_q_lock);
	list_splice_init(&tmp_list, &xor_xwc_dev->free_list);
	spin_unlock_bh(&xor_xwc_dev->msg_q_lock);
	return;
}

/**
 * fsl_xor_xwc_free_resource - free allocated resources
 */
void fsl_xor_xwc_free_resource()
{
	struct fsl_xor_xwc_msg *desc, *_desc;

	spin_lock_bh(&xor_xwc_dev->msg_q_lock);
	list_for_each_entry_safe(desc, _desc,
				 &xor_xwc_dev->in_process_list, node) {
		list_del(&desc->node);
		kfree(desc);
	}
	list_for_each_entry_safe(desc, _desc, &xor_xwc_dev->pending_list,
				 node) {
		list_del(&desc->node);
		kfree(desc);
	}
	list_for_each_entry_safe(desc, _desc, &xor_xwc_dev->free_list, node) {
		list_del(&desc->node);
		kfree(desc);
	}
	spin_unlock_bh(&xor_xwc_dev->msg_q_lock);
}

/**
 * set_dpd_field - Set the DPD field
 * @desc : The XWC Framework desciptor
 * @field : index of the field to set
 * @ptr : the pointer to data
 * @size : the size of the data
 */
static inline void
set_dpd_field(T2DPD *desc, int32_t field, uint8_t *ptr, uint16_t size)
{
	desc->pair[field].size = size;
	desc->pair[field].ptr = ptr;
	desc->pair[field].eptr = 0;
	desc->pair[field].extent = 0;
}

/**
 * schedule_pending_request - Submit the request to XWC Framework
 */
static void schedule_pending_request(void)
{
	struct fsl_xor_xwc_msg *xor_xwc_msg, *_xor_xwc_msg;
	uint32_t entry_id;
	unsigned long flags;

	spin_lock_irqsave(&xor_xwc_dev->msg_q_lock, flags);
	if (list_empty(&xor_xwc_dev->in_process_list)) {
		list_splice_init(&xor_xwc_dev->pending_list,
				 &xor_xwc_dev->in_process_list);
		list_for_each_entry_safe(xor_xwc_msg, _xor_xwc_msg,
					 &xor_xwc_dev->in_process_list, node) {
			xwcRMqueueRequest(0, xor_xwc_dev->if_ctx,
					  &xor_xwc_msg->exec_msg, &entry_id);
		}
	}
	spin_unlock_irqrestore(&xor_xwc_dev->msg_q_lock, flags);
}

/**
 * fsl_xwc_xor_free_msg - Free up the msg queue
 * @xwc_msg : A message queue hoding a list of descriptors
 */
static void fsl_xwc_xor_free_msg(struct fsl_xor_xwc_msg *xwc_msg)
{
	unsigned long flags;

	spin_lock_irqsave(&xor_xwc_dev->msg_q_lock, flags);
	list_del(&xwc_msg->node);
	list_add_tail(&(xwc_msg->node), &xor_xwc_dev->free_list);
	spin_unlock_irqrestore(&xor_xwc_dev->msg_q_lock, flags);
	return;
}

#ifdef DEBUG
static void fsl_xor_xwc_done(uint32_t flag)
{
	if (!flag)
		printk(KERN_INFO "Successfully XORed\n");
}

static void fsl_xor_xwc_error(uint32_t flag)
{
	if (!flag)
		printk(KERN_ERR "XOR Error\n");
}
#endif

/**
 * fsl_xor_xwc_release - Release handler called by XWC Framework
 * @first : The head of the execution message list
 */
static void fsl_xor_xwc_release(struct fsl_xor_xwc_msg *first)
{
	int16_t i = 0;
	struct fsl_xor_desc *desc;
	int16_t count;

	if (first) {
		for (i = 0; i < first->exec_msg.descCount; i++) {
			desc = first->dpd_desc[i];
			if (desc) {
				if (desc->release_hdl && desc->release_arg)
						desc->release_hdl(desc->release_arg);
			}
		}

		fsl_xwc_xor_free_msg(first);
		schedule_pending_request();
	}
}

/**
 * fsl_xor_xwc_get_msg - Allocate a msg descriptor from free list
 * Return - The allocated msg descriptor
 */
static struct fsl_xor_xwc_msg *fsl_xwc_xor_get_msg(void)
{
	struct fsl_xor_xwc_msg *xwc_msg = NULL;
	unsigned long flags;

	if (!list_empty(&xor_xwc_dev->free_list)) {
		spin_lock_irqsave(&xor_xwc_dev->msg_q_lock, flags);
		xwc_msg = lh_to_xwc_xor_mesg(xor_xwc_dev->free_list.next);
		list_del(&xwc_msg->node);
		spin_unlock_irqrestore(&xor_xwc_dev->msg_q_lock, flags);
	}

	memset(xwc_msg, 0, sizeof(*xwc_msg));
	xwc_msg->exec_msg.descHead = &xwc_msg->dpd;
	xwc_msg->exec_msg.messageReleaseHandler =
	    (void (*)(void *))fsl_xor_xwc_release;
	xwc_msg->exec_msg.releaseArgument = (void *)xwc_msg;
#ifdef DEBUG
	xwc_msg->exec_msg.descriptorDoneHandler = (void (*)(uint32_t))
	    fsl_xor_xwc_done;
	xwc_msg->exec_msg.descriptorErrorHandler = (void (*)(uint32_t))
	    fsl_xor_xwc_error;
#endif
	return xwc_msg;
}

/**
 * fsl_xor_xwc_request - Prepare the descriptor
 * @desc : Freescale XOR ADMA descriptor
 *
 * The descriptors from upper layer are mapped to T2DPD.
 * Multiple DPD's  collated under one execution message
 * and the execution message descriptor sceduled for submitting
 */
int32_t fsl_xor_xwc_request(struct fsl_xor_desc *desc)
{
	int32_t status = 0;
	RMexecMessage *exec_msg = NULL;
	T2DPD *xwcdpd = NULL;
	uint32_t num_dpd = 0;
	uint16_t j = 0, k = 0;
	struct fsl_xor_xwc_msg *xor_xwc_msg;
	unsigned long flags;

	spin_lock_irqsave(&xor_xwc_dev->msg_q_lock, flags);
	if (list_empty(&xor_xwc_dev->pending_list)) {
		xor_xwc_msg = fsl_xwc_xor_get_msg();
	} else {
		xor_xwc_msg =
		    lh_to_xwc_xor_mesg(xor_xwc_dev->pending_list.prev);
		if ((xor_xwc_msg->exec_msg.descCount) >= MAX_DPD_FIFO)
			xor_xwc_msg = fsl_xwc_xor_get_msg();
		else
			list_del(&xor_xwc_msg->node);
	}
	spin_unlock_irqrestore(&xor_xwc_dev->msg_q_lock, flags);

	if (likely(xor_xwc_msg)) {
		exec_msg = &xor_xwc_msg->exec_msg;
		num_dpd = exec_msg->descCount;
		xwcdpd = &xor_xwc_msg->dpd[0];

		xwcdpd[num_dpd].hdr = ((EU_AES | AES_XOR) << EU_SHIFT_PRIMARY)
		    | DESCTYPE_RAIDXOR | HDR_OUTBOUND | HDR_DONE;
		set_dpd_field(&xwcdpd[num_dpd], (int32_t) (TOTAL_PAIRS - 1),
			      (uint8_t *) desc->dest, desc->size);
		for (j = NUM_SRC_DWORD; (k < desc->count) && (j > 0); j--, k++)
			set_dpd_field(&xwcdpd[num_dpd], (j - 1),
				      (uint8_t *) desc->src_list[k],
				      desc->size);

		xor_xwc_msg->dpd_desc[num_dpd] = desc;

		if (num_dpd != 0)
			xwcdpd[num_dpd - 1].hdr =
			    ((EU_AES | AES_XOR) << EU_SHIFT_PRIMARY)
			    | DESCTYPE_RAIDXOR | HDR_OUTBOUND;
		exec_msg->descCount++;

		spin_lock_irqsave(&xor_xwc_dev->msg_q_lock, flags);
		list_add_tail(&xor_xwc_msg->node, &xor_xwc_dev->pending_list);
		spin_unlock_irqrestore(&xor_xwc_dev->msg_q_lock, flags);

		schedule_pending_request();
	}
	return status;
}

int fsl_xor_xwc_init(void)
{
	int status;

	xor_xwc_dev = kzalloc(sizeof(*xor_xwc_dev), GFP_KERNEL);
	if (!xor_xwc_dev)
		return -ENOMEM;
	INIT_LIST_HEAD(&xor_xwc_dev->free_list);
	INIT_LIST_HEAD(&xor_xwc_dev->in_process_list);
	INIT_LIST_HEAD(&xor_xwc_dev->pending_list);
	spin_lock_init(&xor_xwc_dev->msg_q_lock);

	status = xwcRMregisterInterface("fsl_xor_xwc", &xor_xwc_dev->if_ctx);

	if (status != RM_OK) {
		printk(KERN_ERR "fsl_xor_xwc: Cannot Register with XRM\n");
		kfree(xor_xwc_dev);
	} else {
		printk(KERN_INFO "fsl_xor_xwc: Registered with XRM\n");
	}

	return status;
}

void fsl_xor_xwc_exit(void)
{
	RMstatus status;

	status = xwcRMderegisterInterface(xor_xwc_dev->if_ctx);

	if (status != RM_OK)
		printk(KERN_ERR "fsl_xor_xwc: Cannot deregister from XRM\n");
	kfree(xor_xwc_dev);

	return;
}
