/*
 * Freescale MPC83xx DMA XOR Engine support
 *
 * Copyright (C) 2008 Freescale Semiconductor, Inc. All rights reserved.
 *
 * Author:
 *   Dipen Dudhat <Dipen.Dudhat@freescale.com>, Nov 2008
 *   Surender Kumar <R66464@freescale.com>, Feb 2008
 *   Vishnu Suresh <Vishnu@freescale.com>, Feb  2008
 *
 * Description:
 *   DMA XOR engine driver for Freescale SEC Engine. The XWC framework
 *   is used to expose the SEC Engine as a DMA engine with XOR capability.
 *   The support for ZERO SUM is not implemented at present.
 *   This code is the frontend which exposes a DMA device with XOR Capability
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/dmaengine.h>
#include <linux/dma-mapping.h>
#include <linux/dmapool.h>
#include <linux/of_platform.h>

#include "fsl_xor.h"

#define chan_to_fsl_xor_chan(chan) \
	container_of(chan, struct fsl_xor_chan, common)
#define lh_to_fsl_xor_desc(lh) container_of(lh, struct fsl_xor_desc, node)
#define tx_to_fsl_xor_desc(tx) container_of(tx, struct fsl_xor_desc, async_tx)
#define	DESC_POOL_ALIGN (32)

/* Pointer to the initialized DMA device; used for cleanup */
static struct fsl_xor_dma_device *gdev;

static inline int fsl_xor_adma_get_max_xor(void)
{
#ifdef CONFIG_FSL_XOR_ADMA
	return NUM_SRC_DWORD - 3;
#endif
}

/**
 * fsl_xor_adma_cleanup - Cleanup descriptor from Tx descriptor queue.
 * @xor_chan : Freescale XOR ADMA channel
 */
static void fsl_xor_adma_cleanup(struct fsl_xor_chan *xor_chan)
{
	struct fsl_xor_desc *desc, *_desc;
	dma_async_tx_callback callback;
	void *callback_param;

	spin_lock_bh(&xor_chan->desc_lock);
	list_for_each_entry_safe(desc, _desc, &xor_chan->tx_desc_q, node) {
		if (dma_async_is_complete(desc->async_tx.cookie,
					  xor_chan->completed_cookie,
					  xor_chan->common.cookie)
		    == DMA_IN_PROGRESS)
			break;

		callback = desc->async_tx.callback;
		callback_param = desc->async_tx.callback_param;

		if (callback) {
			spin_unlock_bh(&xor_chan->desc_lock);
			callback(callback_param);
			spin_lock_bh(&xor_chan->desc_lock);
		}
		list_del(&desc->node);
		list_add_tail(&desc->node, &xor_chan->free_desc);
	}
	spin_unlock_bh(&xor_chan->desc_lock);
}

/**
 * fsl_xor_adma_dependency_added - Cleanup any pending descriptor.
 * @chan : DMA channel
 */
static void fsl_xor_adma_dependency_added(struct dma_chan *chan)
{
	struct fsl_xor_chan *xor_chan = chan_to_fsl_xor_chan(chan);

	fsl_xor_adma_cleanup(xor_chan);
}

/**
 * fsl_xor_adma_is_complete - Check for status and update the cookies.
 * @chan : DMA channel
 */
static enum dma_status fsl_xor_adma_is_complete(struct dma_chan *chan,
						dma_cookie_t cookie,
						dma_cookie_t *done,
						dma_cookie_t *used)
{
	struct fsl_xor_chan *xor_chan = chan_to_fsl_xor_chan(chan);
	dma_cookie_t last_used;
	dma_cookie_t last_complete;
	enum dma_status ret;

	last_used = chan->cookie;
	last_complete = xor_chan->completed_cookie;

	if (done)
		*done = last_complete;

	if (used)
		*used = last_used;

	ret = dma_async_is_complete(cookie, last_complete, last_used);

	if (ret == DMA_SUCCESS)
		return ret;
	fsl_xor_adma_cleanup(xor_chan);

	last_used = chan->cookie;
	last_complete = xor_chan->completed_cookie;

	if (done)
		*done = last_complete;

	if (used)
		*used = last_used;

	return dma_async_is_complete(cookie, last_complete, last_used);
}

/**
 * fsl_xor_adma_release_handler - Handler to be called upon completion
 * @desc : Freescale XOR ADMA descriptor
 */
void fsl_xor_adma_release_handler(struct fsl_xor_desc *desc)
{
	struct fsl_xor_chan *xor_chan =
	    chan_to_fsl_xor_chan(desc->async_tx.chan);

	spin_lock_bh(&desc->cur_chan->desc_lock);
	xor_chan->completed_cookie = desc->async_tx.cookie;
	spin_unlock_bh(&desc->cur_chan->desc_lock);
	fsl_xor_adma_cleanup(xor_chan);
	return;
}

/**
 * fsl_xor_adma_issue_pending - Request for XOR operation
 * @chan : DMA channel
 */
static void fsl_xor_adma_issue_pending(struct dma_chan *chan)
{
	struct fsl_xor_chan *xor_chan = chan_to_fsl_xor_chan(chan);
	struct fsl_xor_desc *desc, *_desc;
	unsigned long flags;

	spin_lock_irqsave(&xor_chan->desc_lock, flags);
	list_for_each_entry_safe(desc, _desc, &xor_chan->tx_desc_q, node) {
		if ((desc->async_tx.cookie > xor_chan->submitted_cookie) &&
		    (desc->async_tx.cookie <= xor_chan->common.cookie)) {
			fsl_xor_xwc_request(desc);
			xor_chan->submitted_cookie++;
		}
	}
	spin_unlock_irqrestore(&xor_chan->desc_lock, flags);
}

/**
 * fsl_xor_adma_tx_submit - Update the transaction cookie values
 * @tx : DMA Async_tx Transaction descriptor
 */
static dma_cookie_t fsl_xor_adma_tx_submit(struct dma_async_tx_descriptor *tx)
{
	struct fsl_xor_desc *desc = tx_to_fsl_xor_desc(tx);
	struct fsl_xor_chan *xor_chan = chan_to_fsl_xor_chan(tx->chan);
	dma_cookie_t cookie;

	spin_lock_bh(&xor_chan->desc_lock);

	cookie = xor_chan->common.cookie + 1;
	if (cookie < 0)
		cookie = 1;

	desc->async_tx.cookie = cookie;
	xor_chan->common.cookie = desc->async_tx.cookie;
	list_splice_init(&desc->async_tx.tx_list, xor_chan->tx_desc_q.prev);
	spin_unlock_bh(&xor_chan->desc_lock);
	return cookie;
}

/**
 * fsl_xor_adma_set_src - Setup the src address
 * @addr : Source address for the XOR operation
 * @tx : DMA Async_tx Transaction descriptor
 * @index : Index of src page in src page array
 */
static void fsl_xor_adma_set_src(dma_addr_t addr,
				 struct dma_async_tx_descriptor *tx, int index)
{
	struct fsl_xor_desc *desc_node, *desc = tx_to_fsl_xor_desc(tx);

	list_for_each_entry(desc_node, &desc->async_tx.tx_list, node) {
		desc_node->src_list[index] = addr;
		addr += XOR_MAX_PAGE_SIZE;
	}
	desc->count++;
}

/**
 * fsl_xor_adma_set_src - Setup the dest address
 * @addr : Dest address for the XOR operation
 * @tx : DMA Async_tx Transaction descriptor
 * @index : Index of dest page
 */
static void fsl_xor_adma_set_dest(dma_addr_t addr,
				  struct dma_async_tx_descriptor *tx, int index)
{
	struct fsl_xor_desc *desc_node, *desc = tx_to_fsl_xor_desc(tx);

	list_for_each_entry(desc_node, &desc->async_tx.tx_list, node) {
		desc_node->dest = addr;
		addr += XOR_MAX_PAGE_SIZE;
	}
}

/**
 * fsl_xor_adma_alloc_descriptor - Allocate a Freescale XOR ADMA descriptor
 * @xor_chan :  Freescale XOR ADMA channel
 */
static struct fsl_xor_desc *fsl_xor_adma_alloc_descriptor(struct fsl_xor_chan
							  *xor_chan,
							  gfp_t flags)
{
	struct fsl_xor_desc *desc;

	desc = kzalloc(sizeof(*desc), flags);
	if (desc) {
		dma_async_tx_descriptor_init(&desc->async_tx,
					     &xor_chan->common);
		desc->async_tx.tx_set_src = fsl_xor_adma_set_src;
		desc->async_tx.tx_set_dest = fsl_xor_adma_set_dest;
		desc->async_tx.tx_submit = fsl_xor_adma_tx_submit;
		INIT_LIST_HEAD(&desc->async_tx.tx_list);
		desc->cur_chan = xor_chan;
	}
	return desc;
}

/**
 * fsl_xor_adma_free_chan_resources - Free XOR ADMA resources, mainly the queues
 * @chan :  DMA channel
 */
static void fsl_xor_adma_free_chan_resources(struct dma_chan *chan)
{
	struct fsl_xor_chan *xor_chan = chan_to_fsl_xor_chan(chan);
	struct fsl_xor_desc *desc, *_desc;

	spin_lock_bh(&xor_chan->desc_lock);
	list_for_each_entry_safe(desc, _desc, &xor_chan->tx_desc_q, node) {
		list_del(&desc->node);
		kfree(desc);
	}

	list_for_each_entry_safe(desc, _desc, &xor_chan->free_desc, node) {
		list_del(&desc->node);
		kfree(desc);
	}
	spin_unlock_bh(&xor_chan->desc_lock);
	fsl_xor_xwc_free_resource();
}

/**
 * fsl_xor_adma_alloc_chan_resources - Allocate XOR ADMA resources
 * @chan :  DMA channel
 */
static int fsl_xor_adma_alloc_chan_resources(struct dma_chan *chan)
{
	struct fsl_xor_chan *xor_chan = chan_to_fsl_xor_chan(chan);
	struct fsl_xor_desc *desc = NULL;
	int i;
	LIST_HEAD(tmp_list);

	if (!list_empty(&xor_chan->free_desc))
		return INITIAL_FSL_DESC_COUNT;

	for (i = 0; i < INITIAL_FSL_DESC_COUNT; i++) {
		desc = fsl_xor_adma_alloc_descriptor(xor_chan, GFP_KERNEL);
		if (!desc) {
			dev_err(xor_chan->chan_dev,
				"Only %d initial descriptors\n", i);
			break;
		}
		list_add_tail(&desc->node, &tmp_list);
	}
	spin_lock_bh(&xor_chan->desc_lock);
	list_splice_init(&tmp_list, &xor_chan->free_desc);
	spin_unlock_bh(&xor_chan->desc_lock);
	fsl_xor_xwc_alloc_resource();
	return 1;
}

/**
 * fsl_xor_adma_prep_xor - Prepare a Freescale XOR ADMA descriptor for operation
 * @chan :  DMA channel
 * @len :  length of source page
 */
static struct dma_async_tx_descriptor *fsl_xor_adma_prep_xor(struct dma_chan
							     *chan,
							     unsigned int
							     src_cnt,
							     size_t len,
							     int int_en)
{
	struct fsl_xor_chan *xor_chan;
	struct fsl_xor_desc *first = NULL, *prev = NULL, *new;
	size_t copy;

	if ((!chan) || (!len))
		return NULL;

	xor_chan = chan_to_fsl_xor_chan(chan);
	do {
		spin_lock_bh(&xor_chan->desc_lock);
		if (!list_empty(&xor_chan->free_desc)) {
			new = lh_to_fsl_xor_desc(xor_chan->free_desc.next);
			list_del(&new->node);
			if (new) {
				memset(new,0, sizeof(*new));
				dma_async_tx_descriptor_init(&new->async_tx,
							     &xor_chan->common);
				INIT_LIST_HEAD(&new->async_tx.tx_list);
				new->cur_chan = xor_chan;

				new->release_hdl = (void (*)(void *))
				    fsl_xor_adma_release_handler;
				new->async_tx.tx_set_src = fsl_xor_adma_set_src;
				new->async_tx.tx_set_dest = fsl_xor_adma_set_dest;
				new->async_tx.tx_submit = fsl_xor_adma_tx_submit;
				INIT_LIST_HEAD(&new->async_tx.tx_list);
				new->release_arg = new;
			}
		} else {
			new = fsl_xor_adma_alloc_descriptor(xor_chan,
							    GFP_KERNEL);
		}
		spin_unlock_bh(&xor_chan->desc_lock);
		if (!new) {
			dev_err(xor_chan->chan_dev,
				"No free memory for link descriptor\n");
			return NULL;
		}

		copy = min(len, XOR_MAX_PAGE_SIZE);
		new->size = copy;

		if (!first)
			first = new;
		new->async_tx.cookie = 0;
		new->async_tx.ack = 1;

		prev = new;
		len -= copy;

		list_add_tail(&new->node, &first->async_tx.tx_list);
	} while (len);

	new->next_ld_desc = 0;
	new->async_tx.ack = 0;
	new->async_tx.cookie = -EBUSY;

	return (first ? &first->async_tx : NULL);
}

static struct of_platform_driver fsl_xor_adma_driver = {
	.name = "fsl-xor-adma",
};

/**
 * fsl_xor_init - Initialize the Freescale XOR ADMA device
 *
 * The device is initialized along with the channel.
 * It is registered as a DMA device with the capability to perform
 * XOR operation. It is registerd with the Async_tx layer and the
 * backend also initialized.
 * The various queues and channel resources are also allocated.
 */
static int __init fsl_xor_init(void)
{
	int error = 0;
	struct of_device dev;
	struct fsl_xor_dma_device *pdev = NULL;
	struct fsl_xor_chan *xor_chan = NULL;
	struct dma_chan *chan = NULL;

#ifdef MY_DEF_HERE
	struct device_node  *dn = NULL;
	dn = of_find_node_by_type(dn, "crypto");
	if (dn == NULL)
	{
		error = -1;
		goto error;
	}
	/* 
	 * Freescale defines SVR IDs so that bit 44 of SVR is set 
	 * for E version SoC. For example 0x803C_0011 and 0x8034_0011
	 * for MPC8533E and MPC8533 respectively.
	 */
	if (0 == (0x00080000U & mfspr(SPRN_SVR))) {
		printk("fsl_xor: no compatible SEC.\n");
		error = -1;
		goto error;
	}
#endif


	pdev = kzalloc(sizeof(*pdev), GFP_KERNEL);
	if (!pdev) {
		printk(KERN_ERR "No enough memory for xor dma\n");
		error = -ENOMEM;
		goto error;
	}

	pdev->dev = &dev.dev;
	pdev->common.device_alloc_chan_resources =
	    fsl_xor_adma_alloc_chan_resources;
	pdev->common.device_free_chan_resources =
	    fsl_xor_adma_free_chan_resources;
	pdev->common.device_prep_dma_xor = fsl_xor_adma_prep_xor;
	pdev->common.max_xor = fsl_xor_adma_get_max_xor();
	pdev->common.device_is_tx_complete = fsl_xor_adma_is_complete;
	pdev->common.device_issue_pending = fsl_xor_adma_issue_pending;
	pdev->common.device_dependency_added = fsl_xor_adma_dependency_added;
	pdev->common.dev = &dev.dev;

	xor_chan = kzalloc(sizeof(*xor_chan), GFP_KERNEL);

	if (!xor_chan) {
		printk(KERN_ERR "No free memory for allocating "
		       "xor dma channels!\n");
		error = -ENOMEM;
		goto free_dev;
	}

	dma_cap_set(DMA_XOR, xor_chan->common.cap_mask);
	xor_chan->device = pdev;
	xor_chan->common.device = &pdev->common;
	INIT_LIST_HEAD(&pdev->common.channels);
	INIT_LIST_HEAD(&xor_chan->tx_desc_q);
	INIT_LIST_HEAD(&xor_chan->free_desc);
	spin_lock_init(&xor_chan->desc_lock);
	list_add_tail(&xor_chan->common.device_node, &pdev->common.channels);
	pdev->common.chancnt++;

	error = fsl_xor_xwc_init();
	if (error)
		goto free_chan;

	error = dma_async_device_register(&pdev->common);
	if (error)
		goto free_xwc;

	/* Assign fsl dma device to global
	 * for acccess by exit fn
	 */
	gdev = pdev;
	printk(KERN_INFO "Registered FSL_XOR_ADMA\n ");
	return 0;

	dma_async_device_unregister(&pdev->common);
free_xwc:
	fsl_xor_xwc_exit();
free_chan:
	list_for_each_entry(chan, &pdev->common.channels, device_node) {
		list_del(&chan->device_node);
		pdev->common.chancnt--;
	}
	kfree(xor_chan);
free_dev:
	kfree(pdev);
error:
	return error;
}

/**
 * fsl_xor_init - Remove the Freescale XOR ADMA device
 */
static void __exit fsl_xor_exit(void)
{
	struct fsl_xor_dma_device *pdev = gdev;
	struct fsl_xor_chan *xor_chan = NULL;
	struct dma_chan *chan;

	dma_async_device_unregister(&pdev->common);
	fsl_xor_xwc_exit();
	list_for_each_entry(chan, &pdev->common.channels, device_node) {
		xor_chan = chan_to_fsl_xor_chan(chan);
		list_del(&chan->device_node);
		pdev->common.chancnt--;
		kfree(xor_chan);
	}
	kfree(pdev);

	return;
}

module_init(fsl_xor_init);
module_exit(fsl_xor_exit);

MODULE_AUTHOR("Freescale Semiconductor - NCSG/IDC");
MODULE_DESCRIPTION("XOR offload DMA Engine over XWC framework");
MODULE_LICENSE("GPL");
